// queue_test3.sc: test bench for c_queue.sc
//
// author: Rainer Doemer
//
// modifications: (most recent first)
//
// 02/14/02 RD	applied naming convention, integrated with distribution
// 02/05/02 RD	initial version, 1 sender thread and 2 receiver threads


#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <sim.sh>


#define QUEUE_SIZE	100	// length of the queue
#define PACKET_SIZE	42	// length of each data packet
#define N		100	// number of read and write transactions
				// (less than 255!)

import "c_queue";


int Rnd(int Range)		// random number in range [0...Range-1]
{
    return((unsigned int)(drand48() * (double)Range));
}


interface I_SafeCount
{
    unsigned long Count(void);
};

channel SafeCounter implements I_SafeCount
{
    unsigned long c = 0;

    unsigned long Count(void)
    {
	return(++c);
    }
};


behavior Sender(
    i_sender Queue,
    in int   ID)
{
    void SendData(int s)
    {
	unsigned char Data[PACKET_SIZE];
	int           i;

	printf("Time%4s: Sender %d storing packet %d.\n",
		time2str(now()), ID, s);
	for(i=0; i<PACKET_SIZE; i++)
	{
	    Data[i] = (unsigned char) s;
	}
	Queue.send(Data, PACKET_SIZE);
    }

    void Sleep(void)
    {
	int Duration;

	printf("Time%4s: Sender %d done, ",
		time2str(now()), ID);
	Duration = Rnd(10);
	printf("going to sleep for %d time units.\n", Duration);
	if (Duration)
	    waitfor(Duration);
    }

    void main(void)
    {
	int s;

	for(s=1; s<=N; s++)
	{
	    SendData(s);
	    Sleep();
	}
    }
};


behavior Receiver(
    i_receiver  Queue,
    I_SafeCount PacketCounter,
    in int      ID)
{
    void ReceiveData(int r)
    {
	unsigned char Data[PACKET_SIZE];
	int           i;

	printf("Time%4s: Receiver %d loading the %dth packet.\n",
		time2str(now()), ID, r);
	Queue.receive(Data, PACKET_SIZE);
	for(i=1; i<PACKET_SIZE; i++)
	{
	    assert(Data[i] == Data[0]);
	}
    }

    void Sleep(void)
    {
	int Duration;

	printf("Time%4s: Receiver %d done, ",
		time2str(now()), ID);
	Duration = Rnd(20);
	printf("going to sleep for %d time units.\n", Duration);
	if (Duration)
	    waitfor(Duration);
    }

    void main(void)
    {
	int r = 0;

	while(PacketCounter.Count() <= N)
	{
	    ReceiveData(++r);
	    Sleep();
	}
    }
};


behavior Main			// test bench for queue
{
    c_queue     Queue(((const unsigned long)QUEUE_SIZE));
    SafeCounter ReceiveCounter;
    Receiver    r1(Queue, ReceiveCounter, 1),
                r2(Queue, ReceiveCounter, 2);
    Sender      s1(Queue, 1);

    int main(void)
    {
	printf("Time%4s: queue_test3: Starting...\n",
		time2str(now()));
	printf("Time%4s: Running 2 receivers, 1 sender.\n",
		time2str(now()));
	par {	r1.main();
		r2.main();
		s1.main();
		}
	printf("Time%4s: queue_test3: Done.\n",
		time2str(now()));
	return 0;
    }
};


// EOF queue_test3.sc
