// Handshake.sc: example of an interrupted one-way handshake
// original by Haobo Yu, extended by R.D.
// (regression test for bug fix in scheduler)

#include <stdio.h>
#include "../../inc/sim.sh"  // <sim.sh>


// this is the _safe_ one-way handshake channel from the standard lib!
// (plus some debugging statements)

interface i_send
{
    void send(void);
};

interface i_receive
{
    void receive(void);
};

channel c_handshake implements i_send, i_receive
{
    event e;
    bool  f = false,
          w = false;
    void receive(void)
    {
	if (! f)
	{
	    w = true;
		printf("(Time %s) waiting for e\n", time2str(now()));
	    wait e;
		printf("(Time %s) after waiting for e\n", time2str(now()));
	    w = false;
	}
	f = false;
    }

    void send(void)
    {
	if (w)
	{
		printf("(Time %s) notify e\n", time2str(now()));
	    notify e;
	}
	f = true;
    }
};


// B1 and B2 are two concurrent behaviors exchanging a handshake

behavior B1(i_receive e1, i_send e2)
{
  void main(void)
  {           
	printf("(Time %s) B1 waiting\n", time2str(now()));
	e1.receive();
	printf("(Time %s) B1 received\n", time2str(now()));
	waitfor(10);
	e2.send();
	printf("(Time %s) B1 send\n", time2str(now()));
  }
};

behavior B2(i_receive e2)
{
  void main(void)
  {           
	printf("(Time %s) B2 waiting\n", time2str(now()));
	e2.receive();
	printf("(Time %s) B2 received\n", time2str(now()));
  }
};


// interrupt handler

behavior ISR()
{
  void main(void)
  {           
	printf("(Time %s) Interrupted!\n", time2str(now()));
  }
};


// interruptable behavior running B1 and B2

behavior SW()
{
	c_handshake e1,e2;

	B1 b1(e1,e2);
	B2 b2(e2);

  void main(void)
  {           
	e1.send();
	waitfor(10);
    par {
      b2.main();
      b1.main();
    }
  }
};


// design under test

behavior System(in event i)
{
	SW  sw;
	ISR isr;

  void main(void)
  {           
    try {
      sw.main();
    }
	interrupt (i)
	{ isr.main(); }
  }
};


// stimulus behavior

behavior Stimulus(out event e)
{
  void main(void)
  {           
	waitfor(20);
	notify e;
  }
};


// test bench

behavior Main()
{
	event i;
	System   system(i);
	Stimulus stimulus(i);
	
  int main(void)
  {           
	par{
		system.main();
		stimulus.main();
	}
	printf("(Time %s) Done.\n", time2str(now()));
	return 0;
  }
};
