/************************************************************************/
/* Initializer.cc: SpecC Internal Representation, Initializer Class	*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 02/10/98 */
/************************************************************************/

/* last update: 09/26/06 */

/* modifications: (most recent first)
 *
 * 09/26/06 PC  Adjustments for scrc 2.1
 * 06/03/05 RD	reorganized and renamed global type names
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 12/19/02 RD	added SIR_Initializer::IsComposite()
 * 03/29/02 RD	added SIR_Initializer::GetAutoArraySize()
 * 01/17/02 RD	added SIR_Initializer::GetIndex()
 * 01/16/02 RD	bug fix in recursion of SIR_Initializer::Print()
 * 01/15/02 RD	added support for composite annotations
 * 11/21/01 RD	took out default arguments from function definitions
 * 11/13/01 RD	added line wrapping in code generation
 * 11/08/01 RD	switched code generation to use GL_IO layer
 * 10/03/01 RD	improved indentation of generated code
 * 05/31/01 RD	eliminated level 2 of SIR API
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 04/30/01 RD	replaced use of obsolete form() from "stream.h" with own one
 * 01/30/01 RD	fixed a potential FMR problem in iterator
 *		SIR_Initials::DFS_ForAllNodes
 */

#include "IntRep/Initializer.h"

#include <assert.h>


/*** constants and macros ***********************************************/


	/* none */


/*** internal type declarations *****************************************/


	/* (none) */


/*** class implementations **********************************************/


	/***********************/
	/*** SIR_Initializer ***/
	/***********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Initializer::SIR_Initializer(	/* constructor #1 */
	sir_initials	*InitList,
	sir_constant	*Initializer)
{

assert(  ((InitList == NULL) && (Initializer != NULL))
       ||((InitList != NULL) && (Initializer == NULL)));

SIR_Initializer::InitList	= InitList;
SIR_Initializer::Initializer	= Initializer;

} /* end of SIR_Initializer::SIR_Initializer #1 */


SIR_Initializer::SIR_Initializer(	/* constructor #3 (duplicator) */
	sir_initializer	*Original)
{

SIR_Initializer::InitList	= (Original->InitList ?
					new SIR_Initials(Original->InitList) :
					NULL);
SIR_Initializer::Initializer	= (Original->Initializer ?
				new SIR_Constant(Original->Initializer) :
					NULL);

} /* end of SIR_Initializer::SIR_Initializer #3 */


SIR_Initializer::~SIR_Initializer(void)	/* destructor */
{

delete InitList;
delete Initializer;

} /* end of SIR_Initializer::~SIR_Initializer */


unsigned int SIR_Initializer::GetAutoArraySize( /* get implicit array size */
        void)                           /* (must be valid array initializer) */
{

if (InitList)
   { return(InitList->NumElements());
    } /* fi */

assert(Initializer != NULL);
assert(Initializer->Type == SIR_CONST_CHARSTRING);

return(Initializer->CS_Value->length() + 1);

} /* end of SIR_Initializer::GetAutoArraySize */


ERROR SIR_Initializer::DFS_ForAllNodes(	/* iterator for over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{

/* this is not a node, but there are nodes below */

if (InitList)
   { if ((SIR_Error = InitList->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Initializer)
   { if ((SIR_Error = Initializer->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Initializer::DFS_ForAllNodes */

bool SIR_Initializer::IsComposite(      /* checks if initializer is composite */
        bool            TargetIsArray)
{

if (! this)
   { return(false);
    } /* fi */

if (InitList)
   { return(true);
    } /* fi */

assert(Initializer != NULL);

if (  (TargetIsArray)
    &&(Initializer->Type == SIR_CONST_CHARSTRING))
   { return(true);
    } /* fi */

return(false);

} /* end of SIR_Initializer::IsComposite */


ERROR SIR_Initializer::WriteSC(	/* (re-) generates SpecC source code */
	gl_io		*IO,
	bool		CplusplusMode,
	bool		StartNewLine /* = true */)
{
sir_initializer	*Init;

if (InitList)
   { if (StartNewLine)
	{ SIR_LineInfo::WriteVSPACE(IO, true);
	 } /* fi */
     IO->PutS("{ ");
     IO->Add2Tab(2);
     Init = InitList->First();
     while(Init)
	{ if ((SIR_Error = Init->WriteSC(IO, CplusplusMode, Init->Pred())))
	     { return(SIR_Error);
	      } /* fi */
	  Init = Init->Succ();
	  if (Init)
	     { IO->PutC(',');
	       SIR_LineInfo::WrapLine(IO);
	      } /* fi */
	 } /* elihw */
     IO->Add2Tab(-2);
     IO->PutS(" }");
    } /* fi */
else
   { assert(Initializer != NULL);
     if (Initializer->LineInfo)
	{ if ((SIR_Error = Initializer->LineInfo->WriteSC(IO)))
	     { return(SIR_Error);
	      } /* fi */
	 } /* fi */
     IO->PutS(Initializer->Print(CplusplusMode));	/* print constant */
    } /* esle */

if ((SIR_Error = IO->Check()))
   { return(SIR_Error);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Initializer::WriteSC */

const char *SIR_Initializer::Print(	/* create SpecC source code (1 line) */
	gl_string	*Buffer /* = NULL */)	/* (NULL = internal buffer) */
{
static gl_string PrintBuffer;
gl_string	*UseBuffer;
sir_initializer	*Init;

if (Buffer)
   { UseBuffer = Buffer;
    } /* fi */
else
   { PrintBuffer = "";
     UseBuffer = &PrintBuffer;
    } /* esle */

if (InitList)
   { assert(Initializer == NULL);
     *UseBuffer += "{";
     Init = InitList->First();
     while(Init)
        { Init->Print(UseBuffer);       /* recursion */
          Init = Init->Succ();
          if (Init)
             { *UseBuffer += ",";
              } /* fi */
         } /* elihw */
     *UseBuffer += "}";
    } /* fi */
else
   { assert(Initializer != NULL);
     *UseBuffer += Initializer->Print();
    } /* esle */

return(UseBuffer->chars());

} /* end of SIR_Initializer::Print */


	/********************/
	/*** SIR_Initials ***/
	/********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Initials::SIR_Initials(		/* constructor #1 */
	sir_initializer	*NewFirstEntry /* = NULL */) :
		SIR_List<SIR_Initializer>(NewFirstEntry)
{

/* nothing else to do */

} /* end of SIR_Initials::SIR_Initials #1 */


SIR_Initials::SIR_Initials(		/* constructor #2 (duplicator) */
	sir_initials	*Original)
{
sir_initializer	*Init;

assert(Original != NULL);

Init = Original->First();
while(Init)
   { Append(new SIR_Initializer(Init));
     Init = Init->Succ();
    } /* elihw */

} /* end of SIR_Initials::SIR_Initials #2 */


SIR_Initials::~SIR_Initials(void)	/* destructor */
{

/* nothing to do */

} /* end of SIR_Initials::~SIR_Initials */


ERROR SIR_Initials::DFS_ForAllNodes(	/* iterator over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{
sir_initializer	*Initial,
		*Succ;

/* this is not a node, but there are nodes below */

Initial = First();
while(Initial)
   { Succ = Initial->Succ();
     if ((SIR_Error = Initial->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Initial = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Initials::DFS_ForAllNodes */


/*** exported functions *************************************************/


	/* none */


/* EOF Initializer.cc */
