// OnesCounter_M4a_new.sc: OnesCounter, Accellera style 4 (SpecC 2.0)
//
// (last update: 02/12/03, RD)
//
// Model characteristics:
// - Accellera style:	4
// - SpecC language:	2.0
// - Machine type:	Mealy (input-based)
// - Reset type:	asynchronous
// - Output ports:	buffered


import "alu_fct";
import "shift_fct";


behavior OnesCounter(
	in  signal bool			CLK,	// system clock
	in  signal bool			RST,	// system reset
	in  signal unsigned bit[31:0]	Inport,
	in  signal unsigned bit[1]	Start,
	out signal unsigned bit[31:0]	Outport,
	out signal unsigned bit[1]	Done,
	out signal unsigned bit[1]	Busy)	// NEW
{

   bit[31:0] ALU0(bit[31:0] a, bit[31:0] b, bit[3:0] ctrl)
   {
     return alu(a, b, ctrl);
    }

   bit[31:0] SHIFT0(bit[31:0] a, bit[31:0] b, bit[1] ctrl)
   {
     return shift(a, b, ctrl);
    }

   void main(void)
   {
     fsmd(CLK rising; Inport, Start;	// sensitive to input ports (Mealy)
			RST)		// asynchronous reset
	{
	  buffered[CLK]	unsigned bit[31:0]	RF0[4];	// storage
	  		unsigned bit[31:0]	BUS0,	// internal busses
						BUS1,
						BUS2;
	  buffered[CLK;RST] unsigned bit[31:0]	OutportBuffer = 0;
	  buffered[CLK;RST] unsigned bit[1]	DoneBuffer = 0b;

	  if (RST)   {  Busy = 0;		// synchronous reset
			Outport = OutportBuffer;
			Done = DoneBuffer;
			}

		      { OutportBuffer = 0;	// default assignments
			Outport = OutportBuffer;
			DoneBuffer = 0;
			Done = DoneBuffer;
			Busy = 1;
		       }
						// state actions
	  S0 :	      { if (Start)
			     Busy = 1;		// immediate feedback! (Mealy)
			else
			     Busy = 0;
			if (Start)
			     goto S1;
			else
			     goto S0;
		       }
	  S1 :	      {	BUS2 = Inport;
			RF0[0] = BUS2;
			goto S2;
		       }
	  S2 :	      {	BUS2 = 0;
			RF0[1] = BUS2;
			goto S3;
		       }
	  S3 :	      { BUS2 = 1;
			RF0[3] = BUS2;
			goto S4;
		       }
	  S4 :	      { BUS0 = RF0[0];
			BUS1 = RF0[3];
			BUS2 = ALU0(BUS0, BUS1, 0x8);
			RF0[2] = BUS2;
			goto S5;
		       }
	  S5 :	      { BUS0 = RF0[1];
			BUS1 = RF0[2];
			BUS2 = ALU0(BUS0, BUS1, 0x0);
			RF0[1] = BUS2;
			goto S6;
		       }
	  S6 :	      { BUS0 = RF0[0];
			BUS2 = SHIFT0(BUS0, 1, 0);
			RF0[0] = BUS2;
			if (BUS2 == 0)
			     goto S7;
			else
			     goto S4;
		       }
	  S7 :	      { BUS1 = RF0[1];
			OutportBuffer = BUS1;
			DoneBuffer = 1;
			Busy = 0;
			if (Start)
			     goto S7;
			else
			     goto S0;
		       }
	 }
    }
};


// test bench /////////////////////////////////////////////////////////////


#include <stdio.h>
#include <stdlib.h>
#include "../../inc/sim.sh"	// #include <sim.sh>


const sim_time ClockPeriod  = 10;
const unsigned int NumTests = 20;


unsigned int Rnd(unsigned int Range)	// random number in range [0...Range-1]
{
    return((unsigned int)(drand48() * (double)Range));
}


unsigned int Ones(unsigned int Value)
{
    unsigned int n;

    n = 0;
    while(Value)
    {
	n += Value & 1u;
	Value >>= 1;
    }
    return(n);
}


behavior ClockGen(
	out signal bool CLK)
{
    void main(void)
    {
	while(true)
	{
	    waitfor(ClockPeriod/2);
//	    printf("Time%6s: Clock falling!\n",
//			time2str(now()));
            CLK = false;
	    waitfor(ClockPeriod/2);
//	    printf("Time%6s: Clock rising!\n",
//			time2str(now()));
	    CLK = true;
	}
    }
};


behavior TestBench(
	in  signal bool			CLK,
	out signal bool			RST,
	out signal unsigned bit[31:0]	Inport,
	out signal unsigned bit[1]	Start,
	in  signal unsigned bit[31:0]	Outport,
	in  signal unsigned bit[1]	Done,
	in  signal unsigned bit[1]	Busy)
{
    void main(void)
    {
	unsigned int	Test,
			Value;

	printf("Time%6s: Starting...\n",
		time2str(now()));
	wait(CLK rising);

	printf("Time%6s: Resetting the unit...\n",
		time2str(now()));
	RST = 1;
	wait(CLK rising);

	Inport = 0;
	Start = 0;
	wait(CLK rising);

	RST = 0;
	wait(CLK rising);

	for(Test=1; Test<=NumTests; Test++)
	{
	    Value = Rnd(0xffff);
	    printf("Time%6s: Running test %u with value 0x%04x.\n",
			time2str(now()), Test, Value);
	    Inport = Value;
	    Start = 1;
	    waitfor 1;
	    if (! Busy)
	    {
		printf("Busy flag is not up!\n");
		exit(15);
	    }
	    while(!Done)
	    {
		wait(CLK rising);
	    }
	    if (Busy)
	    {
		printf("Busy flag is still up!\n");
		exit(15);
	    }

	    printf("Time%6s: Computed result is %u. ",
			time2str(now()), (unsigned int)Outport);
	    if (Outport == Ones(Value))
	    {
		printf("Correct!\n");
	    }
	    else
	    {
		printf("FALSE! Should be %u!\n", Ones(Value));
		exit(10);
	    }
	    Start = 0;
	    while(Done)
	    {
		wait(CLK rising);
	    }
	}
	printf("Time%6s: Done.\n",
		time2str(now()));
	exit(0); // stop the simulation
    }
};


behavior Main
{
    signal bool             CLK;
    signal bool             RST;
    signal unsigned bit[32] Data1,
                            Data2;
    signal unsigned bit[1]  Flag1,
                            Flag2,
			    BusyFlag;
    ClockGen         cg0(CLK);
    OnesCounter      oc0(CLK, RST, Data1, Flag1, Data2, Flag2, BusyFlag);
    TestBench        tb0(CLK, RST, Data1, Flag1, Data2, Flag2, BusyFlag);

    int main(void)
    {
	par
	{
	    cg0.main();
	    oc0.main();
	    tb0.main();
	}
	return(0);
    }
};


// EOF
