/************************************************************************/
/* Note.cc: SpecC Internal Representation, Note Classes			*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 10/08/97 */
/************************************************************************/

/* last update: 12/15/14 */

/* modifications: (most recent first)
 *
 * 12/15/14 RD	finalized scrc_V22
 * 09/26/06 PC  Adjustments for scrc 2.1
 * 06/03/05 RD	reorganized and renamed global type names
 * 04/13/05 RD	bug fix: WriteIntro() argument was pointer instead of bool
 *		(reported by MS compiler)
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 02/13/03 RD	added support for notes at fsmd states
 * 09/19/02 RD	support for adding source code info during code generation
 * 06/21/02 RD	bug fix: SIR_Notes::Merge() missed some pointer unaliasing
 * 05/20/02 RD	removed redefined default arguments from a constructor
 * 01/16/02 RD	added comment in front of generated global annotations
 * 01/15/02 RD	made SIR_Note inherit from SIR_Node to store location info
 * 01/14/02 RD	added support for composite annotations
 * 11/21/01 RD	took out default arguments from function definitions
 * 11/08/01 RD	switched code generation to use GL_IO layer
 * 05/31/01 RD	eliminated level 2 of SIR API
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 04/30/01 RD	replaced use of obsolete form() from "stream.h" with own one
 * 01/30/01 RD	fixed a potential FMR problem in iterators
 *		SIR_Notes::DFS_ForAllXxxxx
 */

#include "IntRep/Note.h"
#include "IntRep/Symbol.h"

#include <assert.h>


/*** constants and macros ***********************************************/


	/* (none) */


/*** internal type declarations *****************************************/

	/* (none) */

/*** prototypes of internal functions ***********************************/


#ifdef SIR_SORT_CODE_BY_LINEINFO
static ERROR WriteOrderedNoteSC(	/* generate sorted source code */
	sir_notes	*NotesList,
	sir_note	*Current,
	gl_io		*IO,
	bool		CplusplusMode);
#endif /* SIR_SORT_CODE_BY_LINEINFO */


/*** class implementations **********************************************/


	/****************/
	/*** SIR_Note ***/
	/****************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Note::SIR_Note(			/* constructor #1 */
	const char	*Name,
	sir_initializer	*Contents,
	sir_symbol	*Symbol /* = NULL */,	/* max. one */
	sir_usertype	*UserType /* = NULL */,	/* of these */
	sir_label	*Label /* = NULL */,	/* can be   */
	sir_fsmdstate	*State /* = NULL */,	/* not NULL */
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(Name != NULL);
assert(*Name != 0);
assert(Contents != NULL);
assert((Symbol == NULL) ||
	((UserType == NULL) && (Label == NULL) && (State == NULL)));
assert((UserType == NULL) ||
	((Symbol == NULL) && (Label == NULL) && (State == NULL)));
assert((Label == NULL) ||
	((Symbol == NULL) && (UserType == NULL) && (State == NULL)));
assert((State == NULL) ||
	((Symbol == NULL) && (UserType == NULL) && (Label == NULL)));

SIR_Note::Name		= Name;
SIR_Note::Contents	= Contents;
SIR_Note::Symbol	= Symbol;
SIR_Note::UserType	= UserType;
SIR_Note::Label		= Label;
SIR_Note::State		= State;
SIR_Note::Imported	= NULL;

} /* end of SIR_Note::SIR_Note #1 */


SIR_Note::SIR_Note(			/* constructor #3 (duplicator) */
	sir_note	*Original) :
		SIR_Node(Original)
{

SIR_Note::Name		= Original->Name;
SIR_Note::Contents	= Original->Contents ?
				new SIR_Initializer(Original->Contents) :
				NULL;
SIR_Note::Symbol	= Original->Symbol;
SIR_Note::UserType	= Original->UserType;
SIR_Note::Label		= Original->Label;
SIR_Note::State		= Original->State;
SIR_Note::Imported	= Original->Imported;

} /* end of SIR_Note::SIR_Note #3 */


SIR_Note::~SIR_Note(void)		/* destructor */
{

delete Contents;

} /* end of SIR_Note::~SIR_Note */


ERROR SIR_Note::DFS_ForAllNodes(	/* iterator for over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{

if ((SIR_Error = (this->*MemberFct)(MemberFctArg)))	/* process this node */
   { return(SIR_Error);
    } /* fi */

if (Contents)
   { if ((SIR_Error = Contents->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Note::DFS_ForAllNodes */


ERROR SIR_Note::DFS_ForAllNotes(	/* iterator over all notes */
	sir_note_mptr	MemberFct,	/* (depth first) */
	sir_note_marg	MemberFctArg)
{

return((this->*MemberFct)(MemberFctArg));

} /* end of SIR_Note::DFS_ForAllNotes */


ERROR SIR_Note::SetImported(	/* sets the Imported pointer (if unset) */
	sir_note_marg	ImportPtr)	/* or resets it if NULL */
{

if (ImportPtr == NULL)
   { Imported = NULL;
     return(SIR_ERROR_NO_ERROR);
    } /* fi */

if (! Imported)		/* import pointer still unset? */
   { Imported = (sir_import*) ImportPtr;
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Note::SetImported */


ERROR SIR_Note::TakeOverImport(		/* takes over an imported note */
	sir_note_marg	ImportPtr)
{

assert(ImportPtr != NULL);
if (Imported == (sir_import*) ImportPtr)
   { Imported = NULL;
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Note::TakeOverImport */


ERROR SIR_Note::MarkImportEntry(	/* marks its import entry being used */
	sir_note_marg	/* Unused */)
{

if (Imported)
   { Imported->Color = SIR_RED;	/* mark as being used */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Note::MarkImportEntry */


ERROR SIR_Note::RemoveDoubleImport(	/* deletes itself if double-imported */
	sir_note_marg	/* Unused */)
{

if (  (Imported)		/* is imported? */
    &&(Imported->Alias))	/* is double-imported? */
   { Remove();		/* remove myself from the list */
     delete this;	/* and commit suicide! */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Note::RemoveDoubleImport */

const char *SIR_Note::Print(		/* print ASCII definition */
	gl_string		*Buffer,	/* (Buffer must be initialized) */
	bool		CplusplusMode /* = false */)
{

assert(Buffer != NULL);

if (CplusplusMode)
   { *Buffer += "/* ";
    } /* fi */

*Buffer += "note ";

if (Symbol)
   { *Buffer += Symbol->Name;
     *Buffer += '.';
    } /* fi */
if (  (UserType)
    &&(UserType->Name))
   { *Buffer += *UserType->Name;
     *Buffer += '.';
    } /* fi */
if (Label)
   { *Buffer += Label->LabelName;
     *Buffer += '.';
    } /* fi */
// note: since labels and fsmd-states are in independent
//       naming scopes, we attach annotations to labels
//       by using the label prefix; for states, however,
//       we use the current scope for attaching annotations;
//       thus, we don't print a prefix for state annotations
//       (unless in C++ mode where it's more readable)
if (  (State)
    &&(CplusplusMode))
   { *Buffer += State->StateName;
     *Buffer += '.';
    } /* fi */

*Buffer += Name;
*Buffer += " = ";
*Buffer += Contents->Print();
*Buffer += ';';

if (CplusplusMode)
   { *Buffer += " */";
    } /* fi */

return(Buffer->chars());

} /* end of SIR_Note::Print */


ERROR SIR_Note::WriteSC(		/* (re-) generates SpecC source code */
	gl_io		*IO,
	bool		CplusplusMode /* = false */)
{

assert(Contents != NULL);

if (LineInfo)
   { if ((SIR_Error = LineInfo->WriteSC(IO, true)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
else
   { SIR_LineInfo::WriteNewSC(IO, true, false, &LineInfo);
    } /* esle */

if (CplusplusMode)
   { IO->PutS("/* ");	/* in C++ mode include notes as comments */
    } /* fi */
IO->PutS("note ");

if (Symbol)
   { IO->PutS(Symbol->Name);
     IO->PutC('.');
    } /* fi */
if (  (UserType)
    &&(UserType->Name))
   { IO->PutS(*UserType->Name);
     IO->PutC('.');
    } /* fi */
if (Label)
   { IO->PutS(Label->LabelName);
     IO->PutC('.');
    } /* fi */
// note: since labels and fsmd-states are in independent
//       naming scopes, we attach annotations to labels
//       by using the label prefix; for states, however,
//       we use the current scope for attaching annotations;
//       thus, we don't print a prefix for state annotations
//       (unless in C++ mode where it's more readable)
if (  (State)
    &&(CplusplusMode))
   { IO->PutS(State->StateName);
     IO->PutC('.');
    } /* fi */

IO->PutS(Name);
IO->PutS(" = ");
if ((SIR_Error = Contents->WriteSC(IO, CplusplusMode)))
   { return(SIR_Error);
    } /* fi */
IO->PutC(';');

if (CplusplusMode)
   { IO->PutS(" */");
    } /* fi */

if ((SIR_Error = IO->Check()))
   { return(SIR_Error);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Note::WriteSC */


	/*****************/
	/*** SIR_Notes ***/
	/*****************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Notes::SIR_Notes(			/* constructor #1 */
	sir_note	*FirstEntry /* = NULL */) :
		SIR_List<SIR_Note>(FirstEntry)
{

/* nothing else to do */

} /* end of SIR_Notes::SIR_Notes #1 */


SIR_Notes::SIR_Notes(			/* constructor #2 (duplicator) */
	sir_notes	*Original)
{
sir_note	*Note;

Note = Original->First();
while(Note)
   { Append(new SIR_Note(Note));
     Note = Note->Succ();
    } /* elihw */

} /* end of SIR_Notes::SIR_Notes #2 */


SIR_Notes::~SIR_Notes(void)		/* destructor */
{

/* nothing to do */

} /* end of SIR_Notes::~SIR_Notes */


ERROR SIR_Notes::DFS_ForAllNodes(	/* iterator over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{
sir_note	*Note,
		*Succ;

/* this is not a node, but there are nodes below */

Note = First();
while(Note)
   { Succ = Note->Succ();
     if ((SIR_Error = Note->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Note = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Notes::DFS_ForAllNodes */


ERROR SIR_Notes::DFS_ForAllNotes(	/* iterator over all notes */
	sir_note_mptr	MemberFct,	/* (depth first) */
	sir_note_marg	MemberFctArg)
{
sir_note	*Note,
		*Succ;

Note = First();
while(Note)
   { Succ = Note->Succ();
     if ((SIR_Error = Note->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Note = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Notes::DFS_ForAllNotes */


sir_note *SIR_Notes::Find(	/* find an entry */
	const char	*Name)	/* (returns NULL if not found) */
{
int		CmpVal;

First();
while(Curr())
   { if (0 == (CmpVal = strcmp(Name, Curr()->Name.chars())))
	{ return(Curr());
	 } /* fi */
     if (CmpVal < 0)
	{ break;
	 } /* fi */
     Next();
    } /* elihw */

return(NULL);	/* not found */

} /* end of SIR_Notes::Find */


sir_note *SIR_Notes::Insert(		/* inserts a new note */
	sir_note	*NewNote)	/* (may return NULL and SIR_Error) */
{
int		CmpVal;

First();
while(Curr())
   { if (0 == (CmpVal = strcmp(NewNote->Name.chars(), Curr()->Name.chars())))
	{ if (NewNote->Symbol)
	     { SIR_ErrMsg.form("Annotation '%s' at symbol '%s' specified twice",
				NewNote->Name.chars(),
				NewNote->Symbol->Name.chars());
	      } /* fi */
	  else
	     { if (NewNote->UserType)
		  { SIR_ErrMsg.form(
				"Annotation '%s' at %s '%s' specified twice",
				NewNote->Name.chars(),
				NewNote->UserType->ClassName(),
				NewNote->UserType->Name->chars());
		   } /* fi */
	       else
		  { if (NewNote->Label)
		       { SIR_ErrMsg.form(
				"Annotation '%s' at label '%s' specified twice",
				NewNote->Name.chars(),
				NewNote->Label->LabelName.chars());
			} /* fi */
		    else
		       { if (NewNote->State)
			    { SIR_ErrMsg.form(
				"Annotation '%s' at state '%s' specified twice",
				NewNote->Name.chars(),
				NewNote->State->StateName.chars());
			     } /* fi */
			 else
			    { SIR_ErrMsg.form(
				"Annotation '%s' at top level specified twice",
				NewNote->Name.chars());
			     } /* esle */
			} /* esle */
		   } /* esle */
	      } /* esle */
	  SIR_Error = SIR_ERROR_NOTE_SPECIFIED_TWICE;
	  return(NULL);
	 } /* fi */
     if (CmpVal < 0)
	{ break;
	 } /* fi */
     Next();
    } /* elihw */

if (Curr())
   { return(InsertBefore(NewNote));
    } /* fi */
else
   { return(Append(NewNote));
    } /* else */

} /* end of SIR_Notes::Insert */


ERROR SIR_Notes::Merge(			/* merges two note-tables */
	sir_notes	**Notes1,	/* (merges 2 into 1) */
	sir_notes	**Notes2)	/* (supports automatic unaliasing) */
{
sir_note	*Curr,
		*Succ;

assert(Notes1 != NULL);
assert(Notes2 != NULL);

if (*Notes1)
   { if (*Notes2)
	{ Curr = (*Notes2)->First();
	  while(Curr)
	     { Succ = Curr->Succ();
	       if (!((*Notes1)->Insert((*Notes2)->Remove(Curr))))
		  { SIR_ErrMsg.prepend(
			"Error while importing:" GL_ERROR_MSG_NEWLINE);
		    delete Curr;
		    return(SIR_ERROR_NOTE_SPECIFIED_TWICE_2);
		   } /* fi */
	       if (  (Curr->Symbol)
		   &&(Curr->Symbol->Alias))
		  { Curr->Symbol = Curr->Symbol->Alias;
		   } /* fi */
	       if (  (Curr->UserType)
		   &&(Curr->UserType->Alias))
		  { Curr->UserType = Curr->UserType->Alias;
		   } /* fi */
	       Curr = Succ;
	      } /* elihw */
	  delete (*Notes2);
	  *Notes2 = NULL;
	 } /* fi */
     else
	{ /* nothing to do */
	 } /* esle */
    } /* fi */
else
   { if (*Notes2)
	{ *Notes1 = *Notes2;	/* just cut and paste */
	  *Notes2 = NULL;
	  /* bug fix: automatic unaliasing was missing (06/21/02, RD) */
	  Curr = (*Notes1)->First();
	  while(Curr)
	     { if (  (Curr->Symbol)
		   &&(Curr->Symbol->Alias))
		  { Curr->Symbol = Curr->Symbol->Alias;
		   } /* fi */
	       if (  (Curr->UserType)
		   &&(Curr->UserType->Alias))
		  { Curr->UserType = Curr->UserType->Alias;
		   } /* fi */
	       Curr = Curr->Succ();
	      } /* elihw */
	 } /* fi */
     else
	{ /* nothing to do */
	 } /* esle */
    } /* esle */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Notes::Merge */


const char *SIR_Notes::Print(		/* print SpecC code (#1) */
	bool		DoIt,
	bool		CplusplusMode /* = false */)
{
static gl_string Buffer;
sir_note	*Note;

if (! DoIt)
   { Buffer = "";
     return(Buffer.chars());
    } /* fi */

Buffer = "";
Note = First();
while(Note)
   { Note->Print(&Buffer, CplusplusMode);
     Note = Note->Succ();
     if (Note)
	{ Buffer += ' ';
	 } /* fi */
    } /* elihw */

return(Buffer.chars());

} /* end of SIR_Notes::Print (#1) */


ERROR SIR_Notes::WriteSC(		/* (re-) generates SpecC source code */
	gl_io		*IO,
	bool		DoIt,
	bool		CplusplusMode /* = false */)
{
sir_note	*Note;
bool		AtTopLevel;

if (! DoIt)
   { return(SIR_ERROR_NO_ERROR);
    } /* fi */

/* (0) Global annotations */

if (!(Note = First()))
   { return(SIR_ERROR_NO_ERROR);
    } /* fi */
AtTopLevel = (  (Note->Symbol == NULL)
	      &&(Note->UserType == NULL)
	      &&(Note->Label == NULL)
	      &&(Note->State == NULL));
if (AtTopLevel)
   { bool PutIntro = true;
     bool PutSeparator = false;
#ifdef SIR_PUT_COMMENTS_IN_SOURCE
     bool PutComment = true;
#else /* !SIR_PUT_COMMENTS_IN_SOURCE */
     bool PutComment = false;
#endif /* SIR_PUT_COMMENTS_IN_SOURCE */

     SIR_Symbols::WriteIntro(IO, false,
			&PutIntro, &PutSeparator,
			PutComment,	/* bug fix 04/13/05, RD */
			"global annotations");
    } /* fi */

#ifdef SIR_SORT_CODE_BY_LINEINFO	/* output notes in LineInfo order */

#ifndef NDEBUG
while(Note)
   { assert(Note->Color == SIR_WHITE);	/* assure clean start */
     Note = Note->Succ();
    } /* elihw */
Note = First();
#endif /* NDEBUG */
while(Note)
   { if ((SIR_Error = WriteOrderedNoteSC(this, Note, IO, CplusplusMode)))
	{ return(SIR_Error);
	 } /* fi */
     Note = Note->Succ();
    } /* elihw */
Note = First();
while(Note)
   { assert(Note->Color == SIR_RED);	/* assure work is done */
     Note->Color = SIR_WHITE;
     Note = Note->Succ();
    } /* elihw */

#else /* !SIR_SORT_CODE_BY_LINEINFO */	/* output notes in alphabetical order */

while(Note)
   { if ((SIR_Error = Note->WriteSC(IO, CplusplusMode)))
	{ return(SIR_Error);
	 } /* fi */
     Note = Note->Succ();
    } /* elihw */

#endif /* SIR_SORT_CODE_BY_LINEINFO */

if (AtTopLevel)
   { SIR_LineInfo::WriteNL(IO);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Notes::WriteSC */


/*** internal functions *************************************************/


#ifdef SIR_SORT_CODE_BY_LINEINFO

static ERROR WriteOrderedNoteSC(	/* generate sorted source code */
	sir_notes	*NotesList,	/* (recursively) */
	sir_note	*Current,
	gl_io		*IO,
	bool		CplusplusMode)
{
sir_note	*Note;

if (Current->Color == SIR_RED)		/* already written? */
   { return(SIR_ERROR_NO_ERROR);
    } /* fi */
assert(Current->Color != SIR_YELLOW);	/* writing still in progress */

Current->Color = SIR_YELLOW;	/* start working on this one */

Note = NotesList->First();
while(Note)
   { if (  (Note->Color != SIR_RED)
	 &&(0 > SIR_Node::CmpLineInfo(Note, Current)))
	{ if ((SIR_Error = WriteOrderedNoteSC(NotesList, Note,
						IO, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	 } /* fi */
     Note = Note->Succ();
    } /* elihw */

if ((SIR_Error = Current->WriteSC(IO, CplusplusMode)))
   { return(SIR_Error);
    } /* fi */

Current->Color = SIR_RED;	/* work is done for this one */

return(SIR_ERROR_NO_ERROR);

} /* end of WriteOrderedNoteSC */

#endif /* SIR_SORT_CODE_BY_LINEINFO */


/* EOF Note.cc */
