/************************************************************************/
/* Expression.cc: SpecC Internal Representation, Expression Class	*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 09/24/97 */
/************************************************************************/

/* last update: 09/26/06 */

/* modifications: (most recent first)
 *
 * 09/26/06 PC  Adjustments for scrc 2.1
 * 04/24/06 RD	bug fix: pointers/arrays are allowed in ? selection
 * 04/24/06 RD	bug fix: pointer subtraction is allowed (results in 'int')
 * 09/15/05 RD	bug fix for DOS-style EOL ("\r\n")
 * 06/03/05 RD	reorganized and renamed global type names
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 11/13/03 HY  added HAVE_METHOD support
 * 10/28/03 RD	bug fix: pointer to function can be called as if it is function
 * 10/24/03 RD	added SIR_Expression::IsInvalidInScope()
 * 10/03/03 RD	added SIR_Expression::WriteArrayAssignment()
 * 10/02/03 RD	added SIR_Expression::Cmp()
 * 05/29/03 RD	bug fix: made iterator SIR_Expression::DFS_ForAllExpressions()
 *		safe with respect to the deletion of the current leaf object
 *		(since it is used with conventions of API layer 2 iterators)
 * 03/14/03 RD	adjusted code generation for special slice operation
 *		for 'buffered' and 'signal' expressions
 * 01/06/03 RD	bug fix: comparison of pointer with array or function is ok
 * 01/03/03 RD	fixed SIR_Expression::ExplicitTypeConv() so that 'float'
 *		is handled just as 'double' (promoted)
 * 01/03/03 RD	fixed the SIR_RELAXED and EXPLICIT_TYPE macros so that
 *		conversion from floating point types to bit vector and
 *		longlong types is handled explicitly (and correctly)
 * 01/03/03 RD	in C++ code generation, explicitly convert signal/buffered/
 *		piped variables to their rvalues whenever two-stage conversion
 *		is required
 * 12/12/02 RD	added support for member access to signal/buffered/piped
 *		variables; separated method IsSymbol() from method IsLvalue()
 * 10/10/02 RD	bug fix: events must not appear in expressions
 * 09/16/02 RD	bug fix: loosened over-restrictive type checking for incr/decr.
 * 07/19/02 RD	bug fix: explicit type cast will not get lost in SpecC code
 * 07/17/02 RD	fixed port direction handling for signals and buffers
 * 07/16/02 RD	strengthened type checking in expressions (closed holes)
 * 07/16/02 RD	refined SIR_Type::IsLvalue()
 * 07/16/02 RD	added handling of SIR_TYPE_SIGNAL and SIR_TYPE_BUFFER
 * 06/26/02 RD	added SIR_Expression::ExpandBehaviorCall() method
 * 01/09/02 RD	bug fix: port access checking was incomplete
 *		(propagate port direction to result type of expressions)
 * 11/29/01 RD	bugfix with long long to bit vector conversions
 * 11/29/01 RD	bugfix with signed vs. unsigned long long conversions
 * 11/21/01 RD	took out default arguments from function definitions
 * 11/13/01 RD	added line wrapping in code generation
 * 11/08/01 RD	switched code generation to use GL_IO layer
 * 11/01/01 RD	added indentation in code generation for '(' and ')'
 * 05/31/01 RD	eliminated level 2 of SIR API
 * 05/27/01 RD	added semantic checking for port accesses, etc.
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 05/25/01 RD	removed code not needed for the SCRC
 * 05/22/01 RD	added SIR_Expression::FixTypePtr()
 * 05/21/01 RD	added support for (native, non-native) array assignment
 * 05/05/01 RD	added explicit type casts for bitvectors and longlongs
 *		used with question mark operator
 * 04/30/01 RD	replaced sir_bitop::resultLen() and sir_bitop::resultUnsigned()
 *		with new macros _BITLEN_*() and _BITUSGN_*()
 * 04/30/01 RD	adjusted SIR_BIT_SLICE_NAME to changes in bitlib
 * 04/30/01 RD	replaced use of obsolete form() from "stream.h" with own one
 * 04/26/01 RD	added explicit promotion to (long) double for class types
 * 04/25/01 RD	fixed a false assertion about promotion in IntegerEval()
 * 04/25/01 RD	added (renamed) CastNeedsExplicitTypeConv()
 * 04/24/01 RD	extended SIR_Expression::ExplicitTypeConv() for floats;
 *		added ExpectedLength argument to SIR_Expression::WriteSC[2];
 *		added non-native long long support for SIR_RELAXED() macro
 * 04/24/01 RD	fixed type cast with non-native long long
 * 04/23/01 RD	fixed the condition around use of ExplicitTypeConv()
 * 04/20/01 RD	added SIR_Expression::ExplicitTypeConv()
 * 04/17/01 RD	adjusted a couple of issues for non-native 'long long'
 * 04/17/01 RD	added missing length argument to Converted() functions
 * 04/16/01 RD	added support for non-native 'long long' type;
 *		changed use of 'long long int' to TIME type
 * 01/30/01 RD	fixed a potential FMR problem in iterators
 *		SIR_Expressions::DFS_ForAllXxxxx
 */

#include "IntRep/Expression.h"
#include "IntRep/Extern.h"
#include "IntRep/Symbol.h"

#include <stdlib.h>
#include <assert.h>


/*** constants and macros ***********************************************/


	/* constants for interface to bitvectors */

#define SIR_BITOP_UNARY_OP	'~'
#define SIR_BITOP_BINARY_OP	'|'
#define SIR_BITOP_CONCATENATION	'@'
#define SIR_BITOP_BITSLICE	'['


// note:
// the classes implementing the bitvector and longlong types
// provide only limited automatic type promotion and conversion
// (because of limitations in the C++ overloading mechanism);
// therefore, some type promotions/conversions have to be performed
// explicitly by the C++ code generator;
// the following macros implement these cases:

	/* macro for 'relaxed' type conversions */

// the following macros handle assignment operators and conversions
// in function calls (arguments and return values):
// - if a bitvector is expected, then no conversion is necessary,
//   unless a floating point or 'long long' type is given (which then
//   needs explicit promotion to its bitvector equivalent)
// - if a longlong is expected and anything but a bit/float is given,
//   then no conversion is necessary; however, if 'unsigned long long'
//   is expected and a 'long long' is given (or vice versa), then
//   conversion is necessary
// (any rules involving longlongs are not needed if native 'long long'
// support is available)

// note: any change to SIR_RELAXED must also be applied to Statement.cc!

#ifdef HAVE_LLONG

#define SIR_RELAXED(expected, given)				\
	( (  (  ((expected) == SIR_TYPE_BIT)			\
	      ||((expected) == SIR_TYPE_UBIT))			\
	   &&(  ((given) != SIR_TYPE_FLOAT)			\
	      &&((given) != SIR_TYPE_DOUBLE)			\
	      &&((given) != SIR_TYPE_LONGDOUBLE)))		\
	 ? (SIR_TYPE_ANY_TYPE) : (expected))

#else /* !HAVE_LLONG */

#define SIR_RELAXED(expected, given)				\
	(  (  (  ((expected) == SIR_TYPE_BIT)			\
	       ||((expected) == SIR_TYPE_UBIT))			\
	    &&(  ((given) != SIR_TYPE_FLOAT)			\
	       &&((given) != SIR_TYPE_DOUBLE)			\
	       &&((given) != SIR_TYPE_LONGDOUBLE)		\
	       &&((given) != SIR_TYPE_LONGLONG)			\
	       &&((given) != SIR_TYPE_ULONGLONG)))		\
	 ||(  ((expected) == SIR_TYPE_LONGLONG)			\
	    &&(  ((given) != SIR_TYPE_BIT)			\
	       &&((given) != SIR_TYPE_UBIT)			\
	       &&((given) != SIR_TYPE_FLOAT)			\
	       &&((given) != SIR_TYPE_DOUBLE)			\
	       &&((given) != SIR_TYPE_LONGDOUBLE)		\
	       &&((given) != SIR_TYPE_ULONGLONG)))		\
	 ||(  ((expected) == SIR_TYPE_ULONGLONG)		\
	    &&(  ((given) != SIR_TYPE_BIT)			\
	       &&((given) != SIR_TYPE_UBIT)			\
	       &&((given) != SIR_TYPE_FLOAT)			\
	       &&((given) != SIR_TYPE_DOUBLE)			\
	       &&((given) != SIR_TYPE_LONGDOUBLE)		\
	       &&((given) != SIR_TYPE_LONGLONG)))		\
	 ? (SIR_TYPE_ANY_TYPE) : (expected))

#endif /* HAVE_LLONG */


	/* macro for explicit promotion of class types */

// the following macros handle binary operators and implement
// the following cases:
// - if 'float', 'double' or 'long double' is expected, then
//   bitvector and longlong classes need explicit promotion
// - if 'unsigned long long' is expected and a 'long long' is given
//   (or vice versa), then explicit promotion is necessary
// - if a bitvector is expected, then any 'long long' needs
//   explicit promotion to its bitvector equivalent
// (EXPLICIT_TYPE is the general macro, just dependent on whether
// native 'long long' is available; EXPLICIT_TYPE2 is a optimized macro
// for binary operators that don't handle floating point operands)

#ifdef HAVE_LLONG

#define EXPLICIT_TYPE(expected, given)				\
	((  (  ((expected) == SIR_TYPE_FLOAT)			\
	     ||((expected) == SIR_TYPE_DOUBLE)			\
	     ||((expected) == SIR_TYPE_LONGDOUBLE))		\
	  &&(  ((given) == SIR_TYPE_BIT)			\
	     ||((given) == SIR_TYPE_UBIT))) ?			\
	(expected) : SIR_TYPE_ANY_TYPE)
#define EXPLICIT_TYPE2(expected, given)				\
	(SIR_TYPE_ANY_TYPE)

#else /* !HAVE_LLONG */

#define EXPLICIT_TYPE(expected, given)				\
	((  (  (  ((expected) == SIR_TYPE_FLOAT)		\
		||((expected) == SIR_TYPE_DOUBLE)		\
		||((expected) == SIR_TYPE_LONGDOUBLE))		\
	     &&(  ((given) == SIR_TYPE_LONGLONG)		\
		||((given) == SIR_TYPE_ULONGLONG)		\
		||((given) == SIR_TYPE_BIT)			\
		||((given) == SIR_TYPE_UBIT)))			\
	  ||(  ((expected) == SIR_TYPE_ULONGLONG)		\
	     &&((given) == SIR_TYPE_LONGLONG))			\
	  ||(  ((expected) == SIR_TYPE_LONGLONG)		\
	     &&((given) == SIR_TYPE_ULONGLONG))			\
	  ||(  (  ((expected) == SIR_TYPE_BIT)			\
		||((expected) == SIR_TYPE_UBIT))		\
	     &&(  ((given) == SIR_TYPE_LONGLONG)		\
		||((given) == SIR_TYPE_ULONGLONG)))) ?		\
	(expected) : SIR_TYPE_ANY_TYPE)
#define EXPLICIT_TYPE2(expected, given)				\
	((  (  ((expected) == SIR_TYPE_ULONGLONG)		\
	     &&((given) == SIR_TYPE_LONGLONG))			\
	  ||(  ((expected) == SIR_TYPE_LONGLONG)		\
	     &&((given) == SIR_TYPE_ULONGLONG))			\
	  ||(  (  ((expected) == SIR_TYPE_BIT)			\
		||((expected) == SIR_TYPE_UBIT))		\
	     &&(  ((given) == SIR_TYPE_LONGLONG)		\
		||((given) == SIR_TYPE_ULONGLONG)))) ?		\
	(expected) : SIR_TYPE_ANY_TYPE)

#endif /* HAVE_LLONG */


/*** internal type declarations *****************************************/


enum SIR_WhatToDo
{
SIR_DO_LL_COMPUTATION,
SIR_DO_ULL_COMPUTATION,
SIR_DO_LD_COMPUTATION,
SIR_DO_BIT_COMPUTATION,
SIR_DO_ERROR
};

enum SIR_WhichArg
{
SIR_LEFT_ARG,
SIR_RIGHT_ARG
};

typedef enum SIR_WhatToDo		SIR_WHAT_TO_DO;
typedef enum SIR_WhichArg		SIR_WHICH_ARG;


/*** internal function prototypes ***************************************/


static SIR_WHAT_TO_DO SIR_PrepareOperand( /* 'ultra-promote' the constant */
	sir_expression		*Arg,	  /* (fills in all possible values) */
	LONG_LONG		*TmpValueLL,
	UNSIGNED_LONG_LONG	*TmpValueULL,
	long double		*TmpValueLD,
	sir_bit			**TmpValueBIT,
	unsigned int		*Line = NULL,
	sir_fileinfo		**FileInfo = NULL);

static SIR_WHAT_TO_DO SIR_PrepareBinaryOperands(
	sir_expression		*Arg1,
	sir_expression		*Arg2,
	LONG_LONG		*TmpValueLL1,
	UNSIGNED_LONG_LONG	*TmpValueULL1,
	long double		*TmpValueLD1,
	sir_bit			**TmpValueBIT1,
	LONG_LONG		*TmpValueLL2,
	UNSIGNED_LONG_LONG	*TmpValueULL2,
	long double		*TmpValueLD2,
	sir_bit			**TmpValueBIT2,
	unsigned int		*Line = NULL,
	sir_fileinfo		**FileInfo = NULL);

static bool SIR_IsConvertable2Bool(	/* checks for legal bool conversion */
	sir_type	*Type);

static bool SIR_IsConvertable2BitVector(/* virtually creates a bitvector */
	sir_type	*OrigType,
	SIR_TYPETYPE	*ResultType = NULL,
	int		*ResultLen = NULL,
	int		*ResultLeft = NULL,
	int		*ResultRight = NULL);

static sir_constant *SIR_Convert2BitVector( /* converts Const to BIT or UBIT */
	sir_constant	*Const);

static void SIR_PutParenthesis(		/* put parenthesis if needed */
	SIR_EXPRTYPE	ExprType,
	char		Parenthesis,
	sir_expression	*Arg,
	SIR_WHICH_ARG	WhichArg,
	gl_io		*IO);

static bool CastNeedsExplicitTypeConv(	/* explicit type conv. needed? */
	SIR_TYPETYPE	FromType,
	SIR_TYPETYPE	ToType);


/*** internal variables *************************************************/


int	SIR_ExprPrecedence[] =	/* as defined in B. Stroustrup, 3rd edition,  */
{				/* "The C++ programming language", p.120-121) */
0,	// SIR_EXPR_VOID
99,	// SIR_EXPR_CONSTANT
99,	// SIR_EXPR_IDENTIFIER
99,	// SIR_EXPR_PARENTHESES
99,	// SIR_EXPR_THIS
16,	// SIR_EXPR_ARRAY_ACCESS
16,	// SIR_EXPR_FUNCTION_CALL
16,	// SIR_EXPR_MEMBER_ACCESS
16,	// SIR_EXPR_MEMBER_POINTER
16,	// SIR_EXPR_POST_INCREMENT
16,	// SIR_EXPR_POST_DECREMENT
16,	// SIR_EXPR_BITSLICE
15,	// SIR_EXPR_PRE_INCREMENT
15,	// SIR_EXPR_PRE_DECREMENT
15,	// SIR_EXPR_ADDRESS_OF
15,	// SIR_EXPR_CONTENT_OF
15,	// SIR_EXPR_POSITIVE
15,	// SIR_EXPR_NEGATIVE
15,	// SIR_EXPR_NOT
15,	// SIR_EXPR_LOGICAL_NOT
15,	// SIR_EXPR_SIZEOF_EXPR
15,	// SIR_EXPR_SIZEOF_TYPE
15,	// SIR_EXPR_TYPE_CONVERSION
14,	// SIR_EXPR_CONCATENATION
13,	// SIR_EXPR_MULTIPLY
13,	// SIR_EXPR_DIVIDE
13,	// SIR_EXPR_MODULO
12,	// SIR_EXPR_ADD
12,	// SIR_EXPR_SUBTRACT
11,	// SIR_EXPR_SHIFT_LEFT
11,	// SIR_EXPR_SHIFT_RIGHT
10,	// SIR_EXPR_LESS
10,	// SIR_EXPR_GREATER
10,	// SIR_EXPR_LESS_EQUAL
10,	// SIR_EXPR_GREATER_EQUAL
9,	// SIR_EXPR_EQUAL
9,	// SIR_EXPR_NOT_EQUAL
8,	// SIR_EXPR_AND
7,	// SIR_EXPR_EOR
6,	// SIR_EXPR_OR
5,	// SIR_EXPR_LOGICAL_AND
4,	// SIR_EXPR_LOGICAL_OR
2,	// SIR_EXPR_CONDITION
3,	// SIR_EXPR_ASSIGNMENT
3,	// SIR_EXPR_MUL_ASSIGN
3,	// SIR_EXPR_DIV_ASSIGN
3,	// SIR_EXPR_MOD_ASSIGN
3,	// SIR_EXPR_ADD_ASSIGN
3,	// SIR_EXPR_SUB_ASSIGN
3,	// SIR_EXPR_SHL_ASSIGN
3,	// SIR_EXPR_SHR_ASSIGN
3,	// SIR_EXPR_AND_ASSIGN
3,	// SIR_EXPR_EOR_ASSIGN
3,	// SIR_EXPR_OR_ASSIGN
1	// SIR_EXPR_COMMA
};


/*** class implementations **********************************************/


	/**********************/
	/*** SIR_Expression ***/
	/**********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Expression::SIR_Expression(		/* constructor #1 (for constants) */
	SIR_EXPRTYPE	ExprType,
	sir_constant	*Constant,
	sir_type	*Type,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(ExprType == SIR_EXPR_CONSTANT);
assert(Constant != NULL);
assert(Type != NULL);
assert(  (  (Constant->Type != SIR_CONST_CHARSTRING)
	  &&((enum SIR_ConstType)Type->Type == Constant->Type))
       ||(  (Constant->Type == SIR_CONST_CHARSTRING)
	  &&(Type->Type == SIR_TYPE_POINTER)));

SIR_Expression::ExprType	= ExprType;
SIR_Expression::Constant	= Constant;
SIR_Expression::Symbol		= NULL;
SIR_Expression::Arg1		= NULL;
SIR_Expression::Arg2		= NULL;
SIR_Expression::Arg3		= NULL;
SIR_Expression::Args		= NULL;
SIR_Expression::TypeArg		= NULL;
SIR_Expression::Type		= Type;
SIR_Expression::LeftBound	= 0;
SIR_Expression::RightBound	= 0;

} /* end of SIR_Expression::SIR_Expression #1 */


SIR_Expression::SIR_Expression(		/* constructor #2 (for identifiers) */
	SIR_EXPRTYPE	ExprType,
	sir_symbol	*Symbol,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(ExprType == SIR_EXPR_IDENTIFIER);
assert(Symbol != NULL);

SIR_Expression::ExprType	= ExprType;
SIR_Expression::Constant	= NULL;
SIR_Expression::Symbol		= Symbol;
SIR_Expression::Arg1		= NULL;
SIR_Expression::Arg2		= NULL;
SIR_Expression::Arg3		= NULL;
SIR_Expression::Args		= NULL;
SIR_Expression::TypeArg		= NULL;
SIR_Expression::Type		= Symbol->Type;
SIR_Expression::LeftBound	= 0;
SIR_Expression::RightBound	= 0;

} /* end of SIR_Expression::SIR_Expression #2 */


SIR_Expression::SIR_Expression(		/* constructor #3 */
	SIR_EXPRTYPE	ExprType,	/* (for std. expr. with 0-3 args.) */
	sir_type	*Type,
	sir_expression	*Arg1 /* = NULL */,
	sir_expression	*Arg2 /* = NULL */,
	sir_expression	*Arg3 /* = NULL */,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(Type != NULL);

SIR_Expression::ExprType	= ExprType;
SIR_Expression::Constant	= NULL;
SIR_Expression::Symbol		= NULL;
SIR_Expression::Arg1		= Arg1;
SIR_Expression::Arg2		= Arg2;
SIR_Expression::Arg3		= Arg3;
SIR_Expression::Args		= NULL;
SIR_Expression::TypeArg		= NULL;
SIR_Expression::Type		= Type;
SIR_Expression::LeftBound	= 0;
SIR_Expression::RightBound	= 0;

} /* end of SIR_Expression::SIR_Expression #3 */


SIR_Expression::SIR_Expression(		/* constructor #4 */
	SIR_EXPRTYPE	ExprType,	/* (for function calls) */
	sir_type	*Type,
	sir_expression	*Arg1,
	sir_expressions	*Args,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(ExprType == SIR_EXPR_FUNCTION_CALL);
assert(Type != NULL);
assert(Arg1 != NULL);
assert(Args != NULL);

SIR_Expression::ExprType	= ExprType;
SIR_Expression::Constant	= NULL;
SIR_Expression::Symbol		= NULL;
SIR_Expression::Arg1		= Arg1;
SIR_Expression::Arg2		= NULL;
SIR_Expression::Arg3		= NULL;
SIR_Expression::Args		= Args;
SIR_Expression::TypeArg		= NULL;
SIR_Expression::Type		= Type;
SIR_Expression::LeftBound	= 0;
SIR_Expression::RightBound	= 0;

} /* end of SIR_Expression::SIR_Expression #4 */


SIR_Expression::SIR_Expression(		/* constructor #5 */
	SIR_EXPRTYPE	ExprType,	/* (for member access) */
	sir_type	*Type,
	sir_expression	*Arg1,
	sir_symbol	*Symbol,	/* (Symbol==NULL for bhvr. main()) */
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(  (ExprType == SIR_EXPR_MEMBER_ACCESS)
       ||(ExprType == SIR_EXPR_MEMBER_POINTER));
assert(Type != NULL);
assert(Arg1 != NULL);

SIR_Expression::ExprType	= ExprType;
SIR_Expression::Constant	= NULL;
SIR_Expression::Symbol		= Symbol;
SIR_Expression::Arg1		= Arg1;
SIR_Expression::Arg2		= NULL;
SIR_Expression::Arg3		= NULL;
SIR_Expression::Args		= NULL;
SIR_Expression::TypeArg		= NULL;
SIR_Expression::Type		= Type;
SIR_Expression::LeftBound	= 0;
SIR_Expression::RightBound	= 0;

} /* end of SIR_Expression::SIR_Expression #5 */


SIR_Expression::SIR_Expression(		/* constructor #6 */
	SIR_EXPRTYPE	ExprType,	/* (for sizeof(type) or casting) */
	sir_type	*Type,
	sir_type	*TypeArg,
	sir_expression	*Arg1 /* = NULL */,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(  (ExprType == SIR_EXPR_SIZEOF_TYPE)
       ||(ExprType == SIR_EXPR_TYPE_CONVERSION));
assert(Type != NULL);
assert(TypeArg != NULL);

SIR_Expression::ExprType	= ExprType;
SIR_Expression::Constant	= NULL;
SIR_Expression::Symbol		= NULL;
SIR_Expression::Arg1		= Arg1;
SIR_Expression::Arg2		= NULL;
SIR_Expression::Arg3		= NULL;
SIR_Expression::Args		= NULL;
SIR_Expression::TypeArg		= TypeArg;
SIR_Expression::Type		= Type;
SIR_Expression::LeftBound	= 0;
SIR_Expression::RightBound	= 0;

} /* end of SIR_Expression::SIR_Expression #6 */


SIR_Expression::SIR_Expression(		/* constructor #7 */
	SIR_EXPRTYPE	ExprType,	/* (for bit slicing) */
	sir_type	*Type,
	sir_expression	*Arg1,
	int		LeftBound,
	int		RightBound,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(ExprType == SIR_EXPR_BITSLICE);
assert(Type != NULL);
assert(Arg1 != NULL);

SIR_Expression::ExprType	= ExprType;
SIR_Expression::Constant	= NULL;
SIR_Expression::Symbol		= NULL;
SIR_Expression::Arg1		= Arg1;
SIR_Expression::Arg2		= NULL;
SIR_Expression::Arg3		= NULL;
SIR_Expression::Args		= NULL;
SIR_Expression::TypeArg		= NULL;
SIR_Expression::Type		= Type;
SIR_Expression::LeftBound	= LeftBound;
SIR_Expression::RightBound	= RightBound;

} /* end of SIR_Expression::SIR_Expression #7 */


SIR_Expression::SIR_Expression(		/* constructor #9 (duplicator) */
	sir_expression	*Original) :	/* (recursive!) */
		SIR_Node(Original)
{

assert(Original != NULL);

SIR_Expression::ExprType	= Original->ExprType;
SIR_Expression::Constant	= (Original->Constant ?
					new SIR_Constant(Original->Constant) :
					NULL);
SIR_Expression::Symbol		= Original->Symbol;
SIR_Expression::Arg1		= (Original->Arg1 ?
					new SIR_Expression(Original->Arg1) :
					NULL);
SIR_Expression::Arg2		= (Original->Arg2 ?
					new SIR_Expression(Original->Arg2) :
					NULL);
SIR_Expression::Arg3		= (Original->Arg3 ?
					new SIR_Expression(Original->Arg3) :
					NULL);
SIR_Expression::Args		= (Original->Args ?
					new SIR_Expressions(Original->Args) :
					NULL);
SIR_Expression::TypeArg		= Original->TypeArg;
SIR_Expression::Type		= Original->Type;
SIR_Expression::LeftBound	= Original->LeftBound;
SIR_Expression::RightBound	= Original->RightBound;

} /* end of SIR_Expression::SIR_Expression #9 */


SIR_Expression::~SIR_Expression(void)	/* destructor */
{

delete Constant;
delete Arg1;
delete Arg2;
delete Arg3;
delete Args;

} /* end of SIR_Expression::~SIR_Expression */


sir_expression *SIR_Expression::New(	/* creates an identifier expr. (#2b) */
	sir_symbol	*Symbol)	/* (returns NULL if SIR_Error) */
{
sir_expression	*NewExpr;
sir_type	*Type;

assert(Symbol != NULL);

Type = Symbol->Type;
assert(Type != NULL);

if (Type->Type == SIR_TYPE_EVENT)
   { SIR_ErrMsg.form("Event '%s' must not be used in any expression",
				Symbol->Name.chars());
     SIR_Error = SIR_ERROR_EVENT_EXPRESSION;
     return(NULL);
    } /* fi */

NewExpr = new SIR_Expression(SIR_EXPR_IDENTIFIER, Symbol);

if (  (Type->Type == SIR_TYPE_SIGNAL)
    ||(Type->Type == SIR_TYPE_BUFFER))
   { NewExpr->Type = Type->SubType->Modified(
				Type->SubType->Const,
				Type->SubType->Volatile,
				Type->Direction);
    } /* fi */
else
   { assert(NewExpr->Type == Type);
    } /* esle */

return(NewExpr);

} /* end of SIR_Expression::New #2b */


sir_expression *SIR_Expression::New(	/* creates a special expression (#8b) */
	SIR_EXPRTYPE	ExprType,	/* (supports VOID, THIS) */
	sir_types	*TypeTable,	/* (returns NULL if SIR_Error) */
	sir_symbol	*ClassSymbol)	/* class for THIS (or NULL) */
{
sir_expression	*NewExpr;
sir_type	*ResultType = NULL;	/* make compiler quiet */

assert(TypeTable != NULL);

switch(ExprType)
   { case SIR_EXPR_VOID:
	{ ResultType = TypeTable->FindOrInsert(SIR_TYPE_VOID);
	  break;
	 }
     case SIR_EXPR_THIS:
	{ assert(ClassSymbol != NULL);
	  assert(ClassSymbol->Type != NULL);
	  ResultType = ClassSymbol->Type;
	  break;
	 }
     default:
	{ assert(false);	/* illegal expression type */
	 }
    } /* hctiws */

NewExpr = new SIR_Expression(ExprType, ResultType);

return(NewExpr);

} /* end of SIR_Expression::New #8b */


sir_symbol *SIR_Expression::IsSymbol(	/* checks if this is based on symbol */
	void)
{

switch (ExprType)
   { case SIR_EXPR_VOID:
	{ return(NULL);
	 }
     case SIR_EXPR_CONSTANT:
	{ return(NULL);
	 }
     case SIR_EXPR_IDENTIFIER:
	{ return(Symbol);
	 }
     case SIR_EXPR_PARENTHESES:
	{ return(Arg1->IsSymbol());
	 }
#ifdef SIR_DELTA_SUPPORT
     case SIR_EXPR_DELTA:
	{ return(NULL);
	 }
#endif /* SIR_DELTA_SUPPORT */
     case SIR_EXPR_THIS:
	{ return(NULL);
	 }
     case SIR_EXPR_ARRAY_ACCESS:
	{ return(Arg1->IsSymbol());
	 }
     case SIR_EXPR_FUNCTION_CALL:
	{ return(NULL);
	 }
     case SIR_EXPR_MEMBER_ACCESS:
     case SIR_EXPR_MEMBER_POINTER:
	{ return(Arg1->IsSymbol());
	 }
     case SIR_EXPR_POST_INCREMENT:
     case SIR_EXPR_POST_DECREMENT:
	{ return(NULL);
	 }
     case SIR_EXPR_BITSLICE:
     case SIR_EXPR_PRE_INCREMENT:
     case SIR_EXPR_PRE_DECREMENT:
	{ return(Arg1->IsSymbol());
	 }
     case SIR_EXPR_ADDRESS_OF:
	{ return(NULL);
	 }
     case SIR_EXPR_CONTENT_OF:
	{ /* no way to find the symbol being pointed to */
	  return(NULL);
	 }
     case SIR_EXPR_POSITIVE:
     case SIR_EXPR_NEGATIVE:
     case SIR_EXPR_NOT:
     case SIR_EXPR_LOGICAL_NOT:
     case SIR_EXPR_SIZEOF_EXPR:
     case SIR_EXPR_SIZEOF_TYPE:
     case SIR_EXPR_TYPE_CONVERSION:
     case SIR_EXPR_CONCATENATION:
     case SIR_EXPR_MULTIPLY:
     case SIR_EXPR_DIVIDE:
     case SIR_EXPR_MODULO:
     case SIR_EXPR_ADD:
     case SIR_EXPR_SUBTRACT:
     case SIR_EXPR_SHIFT_LEFT:
     case SIR_EXPR_SHIFT_RIGHT:
     case SIR_EXPR_LESS:
     case SIR_EXPR_GREATER:
     case SIR_EXPR_LESS_EQUAL:
     case SIR_EXPR_GREATER_EQUAL:
     case SIR_EXPR_EQUAL:
     case SIR_EXPR_NOT_EQUAL:
     case SIR_EXPR_AND:
     case SIR_EXPR_EOR:
     case SIR_EXPR_OR:
     case SIR_EXPR_LOGICAL_AND:
     case SIR_EXPR_LOGICAL_OR:
     case SIR_EXPR_CONDITION:
	{ return(NULL);
	 }
     case SIR_EXPR_ASSIGNMENT:
     case SIR_EXPR_MUL_ASSIGN:
     case SIR_EXPR_DIV_ASSIGN:
     case SIR_EXPR_MOD_ASSIGN:
     case SIR_EXPR_ADD_ASSIGN:
     case SIR_EXPR_SUB_ASSIGN:
     case SIR_EXPR_SHL_ASSIGN:
     case SIR_EXPR_SHR_ASSIGN:
     case SIR_EXPR_AND_ASSIGN:
     case SIR_EXPR_EOR_ASSIGN:
     case SIR_EXPR_OR_ASSIGN:
	{ return(Arg1->IsSymbol());
	 }
     case SIR_EXPR_COMMA:
	{ return(Arg2->IsSymbol());
	 }
     default:
	{ assert(false);	/* bad expression type */
	 }
    } /* hctiws */

return(NULL);	/* never reached */

} /* end of SIR_Expression::IsSymbol */


bool SIR_Expression::IsLvalue(	/* checks if this expr. is an lvalue */
	void)
{

switch (ExprType)
   { case SIR_EXPR_VOID:
	{ assert(false);	/* should never appear as sub-expression */
	 }
     case SIR_EXPR_CONSTANT:
	{ return(Type->Type == SIR_TYPE_POINTER);
	 }
     case SIR_EXPR_IDENTIFIER:
	{ return(true);
	 }
     case SIR_EXPR_PARENTHESES:
	{ return(Arg1->IsLvalue());
	 }
#ifdef SIR_DELTA_SUPPORT
     case SIR_EXPR_DELTA:
	{ return(false);
	 }
#endif /* SIR_DELTA_SUPPORT */
     case SIR_EXPR_THIS:
	{ return(false);
	 }
     case SIR_EXPR_ARRAY_ACCESS:
	{ return(true);
	 }
     case SIR_EXPR_FUNCTION_CALL:
	{ return(false);
	 }
     case SIR_EXPR_MEMBER_ACCESS:
     case SIR_EXPR_MEMBER_POINTER:
	{ return(true);
	 }
     case SIR_EXPR_POST_INCREMENT:
     case SIR_EXPR_POST_DECREMENT:
	{ return(false);
	 }
     case SIR_EXPR_BITSLICE:
     case SIR_EXPR_PRE_INCREMENT:
     case SIR_EXPR_PRE_DECREMENT:
	{ return(true);
	 }
     case SIR_EXPR_ADDRESS_OF:
	{ return(false);
	 }
     case SIR_EXPR_CONTENT_OF:
	{ return(true);
	 }
     case SIR_EXPR_POSITIVE:
     case SIR_EXPR_NEGATIVE:
     case SIR_EXPR_NOT:
     case SIR_EXPR_LOGICAL_NOT:
     case SIR_EXPR_SIZEOF_EXPR:
     case SIR_EXPR_SIZEOF_TYPE:
     case SIR_EXPR_TYPE_CONVERSION:
     case SIR_EXPR_CONCATENATION:
     case SIR_EXPR_MULTIPLY:
     case SIR_EXPR_DIVIDE:
     case SIR_EXPR_MODULO:
     case SIR_EXPR_ADD:
     case SIR_EXPR_SUBTRACT:
     case SIR_EXPR_SHIFT_LEFT:
     case SIR_EXPR_SHIFT_RIGHT:
     case SIR_EXPR_LESS:
     case SIR_EXPR_GREATER:
     case SIR_EXPR_LESS_EQUAL:
     case SIR_EXPR_GREATER_EQUAL:
     case SIR_EXPR_EQUAL:
     case SIR_EXPR_NOT_EQUAL:
     case SIR_EXPR_AND:
     case SIR_EXPR_EOR:
     case SIR_EXPR_OR:
     case SIR_EXPR_LOGICAL_AND:
     case SIR_EXPR_LOGICAL_OR:
     case SIR_EXPR_CONDITION:
	{ return(false);
	 }
     case SIR_EXPR_ASSIGNMENT:
     case SIR_EXPR_MUL_ASSIGN:
     case SIR_EXPR_DIV_ASSIGN:
     case SIR_EXPR_MOD_ASSIGN:
     case SIR_EXPR_ADD_ASSIGN:
     case SIR_EXPR_SUB_ASSIGN:
     case SIR_EXPR_SHL_ASSIGN:
     case SIR_EXPR_SHR_ASSIGN:
     case SIR_EXPR_AND_ASSIGN:
     case SIR_EXPR_EOR_ASSIGN:
     case SIR_EXPR_OR_ASSIGN:
	{ return(true);
	 }
     case SIR_EXPR_COMMA:
	{ return(Arg2->IsLvalue());
	 }
     default:
	{ assert(false);	/* bad expression type */
	 }
    } /* hctiws */

return(false);	/* never reached */

} /* end of SIR_Expression::IsLvalue */


bool SIR_Expression::IsModifiableLvalue(	/* checks if modifiable lv. */
	void)
{

return(  (Type->Const == false)
       &&(IsLvalue()));

} /* end of SIR_Expression::IsModifiableLvalue */


bool SIR_Expression::IsWritable(	/* checks if allows write access */
	void)
{

return(Type->Direction != SIR_PORT_IN);

} /* end of SIR_Expression::IsWritable */


bool SIR_Expression::IsReadable(	/* checks if allows read access */
	void)
{

return(Type->Direction != SIR_PORT_OUT);

} /* end of SIR_Expression::IsReadable */


ERROR SIR_Expression::CheckWriteAccess(	/* checks error for write access */
	void)
{

if (!(IsModifiableLvalue()))
   { if (ExprType == SIR_EXPR_IDENTIFIER)
	{ SIR_ErrMsg.form("Modifiable lvalue expected for '%s'",
				Symbol->Name.chars());
	 } /* fi */
     else
	{ SIR_ErrMsg = "Modifiable lvalue expected";
	 } /* esle */
     return(SIR_Error = SIR_ERROR_MODIFIABLE_LVALUE);
    } /* fi */

if (!(IsWritable()))
   { if (ExprType == SIR_EXPR_IDENTIFIER)
	{ SIR_ErrMsg.form("Write access to input port '%s'",
				Symbol->Name.chars());
	 } /* fi */
     else
	{ SIR_ErrMsg = "Write access to input port";
	 } /* esle */
     return(SIR_Error = SIR_ERROR_PORT_WRITE_ACCESS);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Expression::CheckWriteAccess */


ERROR SIR_Expression::CheckReadAccess(	/* checks error for read access */
	void)
{

if (!(IsReadable()))
   { if (ExprType == SIR_EXPR_IDENTIFIER)
	{ SIR_ErrMsg.form("Read access to output port '%s'",
				Symbol->Name.chars());
	 } /* fi */
     else
	{ SIR_ErrMsg = "Read access to output port";
	 } /* esle */
     return(SIR_Error = SIR_ERROR_PORT_READ_ACCESS);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Expression::CheckReadAccess */


bool SIR_Expression::IsDependant(	/* checks if expr. depends on symbol */
	sir_symbol	*ThatSymbol)
{

if (Symbol == ThatSymbol)
   { return(true);	/* references that symbol */
    } /* fi */

return(false);	/* independent! */

} /* end of SIR_Expression::IsDependant */


ERROR SIR_Expression::IsInvalidInScope( /* checks if this is valid in scope */
        sir_symbols     *Scope)         /* (i.e. if all symbols are visible) */
{
sir_expression  *Arg;
ERROR           Error;

assert(Scope != NULL);  // must be given

if (ExprType == SIR_EXPR_IDENTIFIER)
   { assert(Symbol != NULL);
     if (Symbol != Scope->Find(Symbol->Name.chars()))
        { SIR_ErrMsg.form("symbol %s is not visible in scope",
                                        Symbol->Name.chars());
          return(SIR_ERROR_SYMBOL_NOT_VISIBLE_IN_SCOPE);
         } /* fi */
     return(SIR_ERROR_NO_ERROR);
    } /* fi */

if (  (TypeArg)
    &&(TypeArg->UserType)
    &&(TypeArg->UserType->Name))
   { if (TypeArg->UserType != Scope->UserTypes->Find(
                                        TypeArg->UserType->Name->chars()))
        { SIR_ErrMsg.form("type %s %s is not visible in scope",
                                        TypeArg->UserType->ClassName(),
                                        TypeArg->UserType->Name->chars());
          return(SIR_ERROR_USERTYPE_NOT_VISIBLE_IN_SCOPE);
         } /* fi */
    } /* fi */

if (  (Arg1)
    &&(Error = Arg1->IsInvalidInScope(Scope)))
   { return(Error);
    } /* fi */

if (  (Arg2)
    &&(Error = Arg2->IsInvalidInScope(Scope)))
   { return(Error);
    } /* fi */

if (  (Arg3)
    &&(Error = Arg3->IsInvalidInScope(Scope)))
   { return(Error);
    } /* fi */

if (Args)
   { Arg = Args->First();
     while(Arg)
        { if ((Error = Arg->IsInvalidInScope(Scope)))
             { return(Error);
              } /* fi */
          Arg = Arg->Succ();
         } /* elihw */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Expression::IsInvalidInScope */


ERROR SIR_Expression::DFS_ForAllNodes(	/* iterator over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{

if ((SIR_Error = (this->*MemberFct)(MemberFctArg)))	/* process this node */
   { return(SIR_Error);
    } /* fi */

if (Constant)
   { if ((SIR_Error = Constant->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Arg1)
   { if ((SIR_Error = Arg1->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Arg2)
   { if ((SIR_Error = Arg2->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Arg3)
   { if ((SIR_Error = Arg3->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Args)
   { if ((SIR_Error = Args->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Expression::DFS_ForAllNodes */


ERROR SIR_Expression::DFS_ForAllExpressions(	/* iterator over all exprs. */
	sir_expr_mptr	MemberFct,		/* (depth first) */
	sir_expr_marg	MemberFctArg)
{
bool		IsLeaf;

IsLeaf = !(Arg1 || Arg2 || Arg3 || Args);

if ((SIR_Error = (this->*MemberFct)(MemberFctArg)))	/* process this node */
   { return(SIR_Error);
    } /* fi */

if (IsLeaf)
   { // it is allowed to delete leaf objects within this iterator;
     // in this case, we must not access any members any more,
     // because that would be a free-memory-access violation
     return(SIR_ERROR_NO_ERROR);
    } /* fi */

/* there are no expressions in Constant */
if (Arg1)
   { if ((SIR_Error = Arg1->DFS_ForAllExpressions(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Arg2)
   { if ((SIR_Error = Arg2->DFS_ForAllExpressions(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Arg3)
   { if ((SIR_Error = Arg3->DFS_ForAllExpressions(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Args)
   { if ((SIR_Error = Args->DFS_ForAllExpressions(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Expression::DFS_ForAllExpressions */


bool SIR_Expression::DFS_FindDependant(	/* searches for dependants (DFS) */
	sir_symbol	*ThatSymbol,
	sir_expression	**DepExpr)
{
sir_expression	*Arg;

assert(ThatSymbol != NULL);

if (IsDependant(ThatSymbol))		/* check this expression */
   { if (DepExpr)
	{ *DepExpr = this;
	 } /* fi */
     return(true);
    } /* fi */

/* there are no expressions in Constant */
if (Arg1)
   { if (Arg1->DFS_FindDependant(ThatSymbol, DepExpr))
	{ return(true);
	 } /* fi */
    } /* fi */
if (Arg2)
   { if (Arg2->DFS_FindDependant(ThatSymbol, DepExpr))
	{ return(true);
	 } /* fi */
    } /* fi */
if (Arg3)
   { if (Arg3->DFS_FindDependant(ThatSymbol, DepExpr))
	{ return(true);
	 } /* fi */
    } /* fi */
if (Args)
   { Arg = Args->First();
     while(Arg)
	{ if (Arg->DFS_FindDependant(ThatSymbol, DepExpr))
	     { return(true);
	      } /* fi */
	  Arg = Arg->Succ();
	 } /* elihw */
    } /* fi */

return(false);

} /* end of SIR_Expression::DFS_FindDependant */


ERROR SIR_Expression::MarkUsedTypes(	/* marks the type entries used here */
	sir_expr_marg	/* Unused */)
{

Type->MarkUsed();

if (TypeArg)
   { TypeArg->MarkUsed();
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Expression::MarkUsedTypes */


sir_expression *SIR_Expression::ExpandBehaviorCall( /* expands a short-cut */
        void)                                       /* behavior call       */
{
sir_types       *TypeTable;
sir_type        *VoidType,
                *MethodType;
sir_expression  *MemberAccess,
                *MethodCall;
unsigned int    Line;
sir_fileinfo    *FileInfo;

if (ExprType == SIR_EXPR_IDENTIFIER)
   { assert(Symbol != NULL);
     if (Symbol->IsBehaviorInstance())
        { if (LineInfo)
             { Line = LineInfo->Line;
               FileInfo = LineInfo->File;
              } /* fi */
          else
             { Line = 0;
               FileInfo = NULL;
              } /* fi */
          TypeTable = Type->GetTable();
          VoidType = TypeTable->FindOrInsert(SIR_TYPE_VOID);
          MethodType = TypeTable->FindOrInsert(SIR_TYPE_METHOD, VoidType,
                                new SIR_TypePtrs(new SIR_TypePtr(VoidType)),
                                0, false, false, SIR_PORT_NONE,
                                Symbol->Type->ClassSymbol);
          MemberAccess = new SIR_Expression(SIR_EXPR_MEMBER_ACCESS, MethodType,
                                        this, (sir_symbol*) NULL, /* "main" */
                                        Line, FileInfo);
          MethodCall = new SIR_Expression(SIR_EXPR_FUNCTION_CALL,
                                        VoidType, MemberAccess,
                                        new SIR_Expressions(),
                                        Line, FileInfo);
          return(MethodCall);
         } /* fi */
    } /* fi */

return(this);

} /* end of SIR_Expression::ExpandBehaviorCall */

void SIR_Expression::UnAlias(void)/* unalias all type, usertype, symbol links */
{
sir_expression	*Arg;

/* there is nothing to unalias in Constant */
if (  (Symbol)
    &&(Symbol->Alias))
   { Symbol = Symbol->Alias;
    } /* fi */
if (Arg1)
   { Arg1->UnAlias();
    } /* fi */
if (Arg2)
   { Arg2->UnAlias();
    } /* fi */
if (Arg3)
   { Arg3->UnAlias();
    } /* fi */
if (Args)
   { Arg = Args->First();
     while(Arg)
	{ Arg->UnAlias();
	  Arg = Arg->Succ();
	 } /* elihw */
    } /* fi */
if (  (TypeArg)
    &&(TypeArg->Alias))
   { TypeArg = TypeArg->Alias;
    } /* fi */
if (Type->Alias)
   { Type = Type->Alias;
    } /* fi */

} /* end of SIR_Expression::UnAlias */


void SIR_Expression::ExplicitTypeConv(	/* perform explicit type conversion */
	SIR_CONSTTYPE	FromType,
	SIR_CONSTTYPE	ToType,
	const char	**Prefix,
	const char	**Suffix)	/* (caller must substitute length!) */
{

assert(Prefix != NULL);
assert(Suffix != NULL);

*Prefix = NULL;	/* default: no conversion, or implicit conversion */
*Suffix = NULL;

if (  (FromType == SIR_CONST_FLOAT)
    ||(FromType == SIR_CONST_DOUBLE)
    ||(FromType == SIR_CONST_LONGDOUBLE))
   { if (  (ToType == SIR_CONST_BIT)
	 ||(ToType == SIR_CONST_UBIT))
	{ *Prefix = SIR_BIT_BASE_CLASS_NAME "(";
	  if (ToType == SIR_CONST_BIT)
	     { *Suffix = ", %d, "	/* caller supplies length! */
				SIR_BIT_SIGNED_FLAG ")";
	      } /* fi */
	  else
	     { *Suffix = ", %d,	"	/* caller supplies length! */
				SIR_BIT_UNSIGNED_FLAG ")";
	      } /* esle */
	 } /* fi */
#ifndef HAVE_LLONG
     else if (  (ToType == SIR_CONST_LONGLONG)
	      ||(ToType == SIR_CONST_ULONGLONG))
	{ if (ToType == SIR_CONST_LONGLONG)
	     { *Prefix = SIR_LLONG_CLASS_NAME "<" SIR_LLONG_SIGNED_FLAG ">(";
	       *Suffix = ", " SIR_LLONG_SIGNED_FLAG ")";
	      } /* fi */
	  else
	     { *Prefix = SIR_LLONG_CLASS_NAME "<" SIR_LLONG_UNSIGNED_FLAG ">(";
	       *Suffix = ", " SIR_LLONG_UNSIGNED_FLAG ")";
	      } /* esle */
	 } /* fi else */
#endif /* HAVE_LLONG */
    } /* fi */
else
   { if (  (ToType == SIR_CONST_BIT)
	 ||(ToType == SIR_CONST_UBIT))
	{ if (  (FromType != SIR_CONST_BIT)
	      &&(FromType != SIR_CONST_UBIT))
	     { /* convert integral type to bitvector */
	       *Prefix = SIR_BIT_BASE_CLASS_NAME "(";
	       *Suffix = ")";
	      } /* fi */
	 } /* fi */
     else
	{ if (  (FromType == SIR_CONST_BIT)
	      ||(FromType == SIR_CONST_UBIT))
	     { /* convert bitvector to integral or floating type */
	       switch(ToType)
		  { case SIR_CONST_BOOL:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_BIT_2_BOOL_NAME "())";
			 break;
			}
		    case SIR_CONST_CHAR:
		       { *Prefix = "((char)(";
			 *Suffix = ")." SIR_BIT_2_INT_NAME "())";
			 break;
			}
		    case SIR_CONST_UCHAR:
		       { *Prefix = "((unsigned char)(";
			 *Suffix = ")." SIR_BIT_2_INT_NAME "())";
			 break;
			}
		    case SIR_CONST_SHORT:
		       { *Prefix = "((short)(";
			 *Suffix = ")." SIR_BIT_2_INT_NAME "())";
			 break;
			}
		    case SIR_CONST_USHORT:
		       { *Prefix = "((unsigned short)(";
			 *Suffix = ")." SIR_BIT_2_INT_NAME "())";
			 break;
			}
		    case SIR_CONST_INT:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_BIT_2_INT_NAME "())";
			 break;
			}
		    case SIR_CONST_UINT:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_BIT_2_UINT_NAME "())";
			 break;
			}
		    case SIR_CONST_LONG:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_BIT_2_LONG_NAME "())";
			 break;
			}
		    case SIR_CONST_ULONG:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_BIT_2_ULONG_NAME "())";
			 break;
			}
		    case SIR_CONST_LONGLONG:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_BIT_2_LLONG_NAME "())";
			 break;
			}
		    case SIR_CONST_ULONGLONG:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_BIT_2_ULLONG_NAME "())";
			 break;
			}
		    case SIR_CONST_FLOAT:
		    case SIR_CONST_DOUBLE:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_BIT_2_DOUBLE_NAME "())";
			 break;
			}
		    case SIR_CONST_LONGDOUBLE:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_BIT_2_LDOUBLE_NAME "())";
			 break;
			}
		    case SIR_CONST_BIT:
		    case SIR_CONST_UBIT:
		       { assert(false);	/* handled earlier */
			}
		    default:
		       { assert(false);	/* unsupported conversion */
			}
		   } /* hctiws */
	      } /* fi */
#ifndef HAVE_LLONG
	  if (  (FromType == SIR_CONST_LONGLONG)
	      ||(FromType == SIR_CONST_ULONGLONG))
	     { /* convert 'long long' to integral or floating type */
	       switch(ToType)
		  { case SIR_CONST_BOOL:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_LLONG2_BOOL_NAME "())";
			 break;
			}
		    case SIR_CONST_CHAR:
		       { *Prefix = "((char)(";
			 *Suffix = ")." SIR_LLONG2_INT_NAME "())";
			 break;
			}
		    case SIR_CONST_UCHAR:
		       { *Prefix = "((unsigned char)(";
			 *Suffix = ")." SIR_LLONG2_INT_NAME "())";
			 break;
			}
		    case SIR_CONST_SHORT:
		       { *Prefix = "((short)(";
			 *Suffix = ")." SIR_LLONG2_INT_NAME "())";
			 break;
			}
		    case SIR_CONST_USHORT:
		       { *Prefix = "((unsigned short)(";
			 *Suffix = ")." SIR_LLONG2_INT_NAME "())";
			 break;
			}
		    case SIR_CONST_INT:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_LLONG2_INT_NAME "())";
			 break;
			}
		    case SIR_CONST_UINT:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_LLONG2_UINT_NAME "())";
			 break;
			}
		    case SIR_CONST_LONG:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_LLONG2_LONG_NAME "())";
			 break;
			}
		    case SIR_CONST_ULONG:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_LLONG2_ULONG_NAME "())";
			 break;
			}
		    case SIR_CONST_LONGLONG:
		       { *Prefix = "((" SIR_LLONG_CLASS_NAME
					"<" SIR_LLONG_SIGNED_FLAG ">)(";
			 *Suffix = "))";
			 break;
			}
		    case SIR_CONST_ULONGLONG:
		       { *Prefix = "((" SIR_LLONG_CLASS_NAME
					"<" SIR_LLONG_UNSIGNED_FLAG ">)(";
			 *Suffix = "))";
			 break;
			}
		    case SIR_CONST_FLOAT:
		    case SIR_CONST_DOUBLE:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_LLONG2_DOUBLE_NAME "())";
			 break;
			}
		    case SIR_CONST_LONGDOUBLE:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_LLONG2_LDOUBLE_NAME "())";
			 break;
			}
		    case SIR_CONST_BIT:
		    case SIR_CONST_UBIT:
		       { assert(false);	/* handled earlier */
			}
		    default:
		       { assert(false);	/* unsupported conversion */
			}
		   } /* hctiws */
	      } /* fi */
#endif /* !HAVE_LLONG */
	 } /* esle */
    } /* esle */

} /* end of SIR_Expression::ExplicitTypeConv */


ERROR SIR_Expression::WriteSC(	/* (re-) generates SpecC source code */
	gl_io		*IO,
	bool		WriteNotes,
	bool		CplusplusMode /* = false */,
	SIR_TYPETYPE	ExpectedType /* = SIR_TYPE_ANY_TYPE */,
	int		ExpectedLength /* = 0 */, /* for expected bitvectors */
	bool		RValueExpected /* = false */)
{
sir_expression	*Arg;
sir_constant	*TmpConst;
gl_string	Buffer;
const char	*Prefix,
		*Suffix;
sir_symbol	*BasedOnSymbol;

if (LineInfo)
   { if ((SIR_Error = LineInfo->WriteSC(IO)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

Prefix = NULL;
Suffix = NULL;
if (CplusplusMode)		/* explicitly create automatic conversions */
   { if (  (ExpectedType < SIR_TYPE_VOID) /* (exploit smart type enumeration) */
	 &&(Type->Type < SIR_TYPE_VOID))
	{ ExplicitTypeConv(SIR_Type::ConstType(Type->Type),
				SIR_Type::ConstType(ExpectedType),
				&Prefix, &Suffix);
	 } /* fi */
     if (  (RValueExpected)
	 &&(!Prefix && !Suffix)
	 &&(  (BasedOnSymbol = IsSymbol())
	    &&(  (BasedOnSymbol->Type->Type == SIR_TYPE_SIGNAL)
	       ||(BasedOnSymbol->Type->Type == SIR_TYPE_BUFFER)
	       ||(BasedOnSymbol->StorageClass == SIR_STORAGE_PIPED)))
	 &&(Type->Type != SIR_TYPE_ARRAY)) // never cast to an array type!
	 // there could be further conditions here, which suppress
	 // the explicit conversion if it is not needed (i.e. if the
	 // C++ compiler can figure it out itself); this would definitely
	 // yield better readable code; however, other than beauty of the
	 // code, there is no real reason for doing that, so we can save
	 // ourselves the additional work of figuring out exactly which
	 // conditions would apply here... (RD, 01/03/03)
	{ Buffer = "";
	  Type->PrettyString(&Buffer, false, true);
	  Buffer.prepend("((");
	  Buffer += ")";
	  Prefix = Buffer.chars();
	  Suffix = ")";
	 } /* fi */
     if (Prefix)
	{ IO->PutS(Prefix);	/* start explicit conversion */
	 } /* fi */
    } /* fi */

switch (ExprType)
   { case SIR_EXPR_VOID:
	{ /* nothing */
	  break;
	 }
     case SIR_EXPR_CONSTANT:
	{ IO->PutS(Constant->Print(CplusplusMode));
	  break;
	 }
     case SIR_EXPR_IDENTIFIER:
	{				/* SpecC: m_ptr = method */
	if(  (CplusplusMode)		/* C++:   m_ptr = &B::method */
	   &&(Symbol->Type->Type == SIR_TYPE_METHOD)
	   &&(ExpectedType == SIR_TYPE_POINTER))
	   { assert(Symbol->Type->ClassSymbol != NULL);
	     IO->PutS("&"); 
	     IO->PutS(Symbol->Type->ClassSymbol->Name);
	     IO->PutS("::");
	     IO->PutS(Symbol->Name);
            } /* fi */
	else
	   { IO->PutS(Symbol->Name);
	    } /* else */
	  break;
	 }
     case SIR_EXPR_PARENTHESES:	/* we keep redundant parentheses */
	{ IO->PutC('(');	/* for readability purposes	 */
	  IO->TabStepUp();
	  if ((SIR_Error = Arg1->WriteSC(IO, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->TabStepDown();
	  IO->PutC(')');
	  break;
	 }
     case SIR_EXPR_THIS:
	{ if (CplusplusMode)
	     { IO->PutS("(*this)");	/* in SpecC 'this' is _not_ a pointer */
	      } /* fi */		/* (it is expected as a reference) */
	  else
	     { IO->PutS("this");
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_ARRAY_ACCESS:
	{ SIR_PutParenthesis(ExprType, '(', Arg1, SIR_LEFT_ARG, IO);
	  if ((SIR_Error = Arg1->WriteSC(IO, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_PutParenthesis(ExprType, ')', Arg1, SIR_LEFT_ARG, IO);
	  if (  (CplusplusMode)
	      &&(  (Arg1->Type->Type == SIR_TYPE_BIT)
		 ||(Arg1->Type->Type == SIR_TYPE_UBIT)))
	     { BasedOnSymbol = Arg1->IsSymbol();
	       if (  (BasedOnSymbol)
		   &&(BasedOnSymbol->Type->Type == SIR_TYPE_SIGNAL))
		  { IO->PutS(SIR_SIM_SIG_SLICE_NAME);
		   } /* fi */
	       else
		  { if (  (BasedOnSymbol)
			&&(BasedOnSymbol->Type->Type == SIR_TYPE_BUFFER))
		       { IO->PutS(SIR_SIM_BUF_SLICE_NAME);
			} /* fi */
		    else
		       { IO->PutS(SIR_BIT_SLICE_NAME);
			} /* esle */
		   } /* esle */
	       IO->PrintF("(%d,%d,(",
				Arg1->Type->LeftBound, Arg1->Type->RightBound);
	       if ((SIR_Error = Arg2->WriteSC(IO, WriteNotes, CplusplusMode,
							SIR_TYPE_INT, 0, true)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->PutS("))");
	      } /* fi */
	  else
	     { IO->PutC('[');
	       if ((SIR_Error = Arg2->WriteSC(IO, WriteNotes, CplusplusMode,
							SIR_TYPE_INT, 0, true)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->PutC(']');
	      } /* esle */
	  break;
	 }
     case SIR_EXPR_FUNCTION_CALL:
	{ sir_type_ptr	*ExpectTypePtr;
	  sir_type	*FctType;
	  SIR_TYPETYPE	ExpectType;
	  int		ExpectLen;

          if (  (CplusplusMode)
	      &&(Arg1->Type->Type == SIR_TYPE_POINTER)
	      &&(Arg1->Type->SubType->Type == SIR_TYPE_METHOD))
             { IO->PutS("( this->*");
              } /* fi */
	  SIR_PutParenthesis(ExprType, '(', Arg1, SIR_LEFT_ARG, IO);
	  if ((SIR_Error = Arg1->WriteSC(IO, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_PutParenthesis(ExprType, ')', Arg1, SIR_LEFT_ARG, IO);
          if (  (CplusplusMode)
	      &&(Arg1->Type->Type == SIR_TYPE_POINTER)
	      &&(Arg1->Type->SubType->Type == SIR_TYPE_METHOD))
             { IO->PutC(')');
              } /* fi */
	  IO->PutC('(');
	  IO->TabStepUp();
	  if (Args)
	     { assert(  (Arg1->Type->Type == SIR_TYPE_FUNCTION)
		      ||(Arg1->Type->Type == SIR_TYPE_METHOD)
		      ||(  (Arg1->Type->Type == SIR_TYPE_POINTER)
			 && (  (Arg1->Type->SubType->Type ==SIR_TYPE_FUNCTION)
			     ||(Arg1->Type->SubType->Type ==SIR_TYPE_METHOD))));
	       if (  (Arg1->Type->Type == SIR_TYPE_FUNCTION)
		   ||(Arg1->Type->Type == SIR_TYPE_METHOD))
		  { FctType = Arg1->Type;
		   } /* fi */
	       else
		  { FctType = Arg1->Type->SubType;
		   } /* esle */
	       ExpectTypePtr = NULL;
	       if (FctType->Parameters)
		  { ExpectTypePtr = FctType->Parameters->First();
		   } /* fi */
	       Arg = Args->First();
	       while(Arg)
		  { if (ExpectTypePtr)
		       { ExpectType = ExpectTypePtr->Type->Type;
			 ExpectLen = _BITLEN(ExpectTypePtr->Type->LeftBound,
					     ExpectTypePtr->Type->RightBound);
			} /* fi */
		    else
		       { ExpectType = SIR_TYPE_ANY_TYPE;
			 ExpectLen = 0;
			} /* else */
		    SIR_PutParenthesis(SIR_EXPR_COMMA,
					'(', Arg1, SIR_RIGHT_ARG, IO);
		    if ((SIR_Error = Arg->WriteSC(IO, WriteNotes,
						CplusplusMode,
						SIR_RELAXED(ExpectType,
							Arg->Type->Type),
						ExpectLen, true)))
		       { return(SIR_Error);
			} /* fi */
		    SIR_PutParenthesis(SIR_EXPR_COMMA,
					')', Arg1, SIR_RIGHT_ARG, IO);
		    if (ExpectTypePtr)
		       { ExpectTypePtr = ExpectTypePtr->Succ();
			} /* fi */
		    Arg = Arg->Succ();
		    if (Arg)
		       { IO->PutS(", ");
			 SIR_LineInfo::WrapLine(IO);
			} /* fi */
		   } /* elihw */
	      } /* fi */
	  IO->PutC(')');
	  IO->TabStepDown();
	  break;
	 }
     case SIR_EXPR_MEMBER_ACCESS:
	{ sir_symbol	*BasedOnSymbol;

	  if (  (CplusplusMode)
	      &&(BasedOnSymbol = Arg1->IsSymbol())
	      &&(  (BasedOnSymbol->Type->Type == SIR_TYPE_SIGNAL)
		 ||(BasedOnSymbol->Type->Type == SIR_TYPE_BUFFER)
		 ||(BasedOnSymbol->StorageClass == SIR_STORAGE_PIPED)))
	     { SIR_PutParenthesis(ExprType, '(', Arg1, SIR_LEFT_ARG, IO);
	       if ((SIR_Error = Arg1->WriteSC(IO, WriteNotes, CplusplusMode)))
		  { return(SIR_Error);
		   } /* fi */
	       SIR_PutParenthesis(ExprType, ')', Arg1, SIR_LEFT_ARG, IO);
	       Buffer = "";
	       Type->PrettyString(&Buffer, false, true);
	       if (Buffer[Buffer.length()-1] == '>')
		  { Buffer += " ";	/* avoid ">>" clash! */
		   } /* fi */
	       IO->PrintF(SIR_SIM_MEMBER_ACCS_FMT, Buffer.chars(),
						Arg1->Type->Offset(Symbol));
	      } /* fi */
	  else
	     { SIR_PutParenthesis(ExprType, '(', Arg1, SIR_LEFT_ARG, IO);
	       if ((SIR_Error = Arg1->WriteSC(IO, WriteNotes, CplusplusMode)))
		  { return(SIR_Error);
		   } /* fi */
	       SIR_PutParenthesis(ExprType, ')', Arg1, SIR_LEFT_ARG, IO);
	       IO->PutC('.');
	       if (Symbol)
		  { IO->PutS(Symbol->Name);
		   } /* fi */
	       else
		  { IO->PutS(GL_MAIN_METHOD_NAME);
		   } /* esle */
	      } /* esle */
	  break;
	 }
     case SIR_EXPR_MEMBER_POINTER:
	{ SIR_PutParenthesis(ExprType, '(', Arg1, SIR_LEFT_ARG, IO);
	  if ((SIR_Error = Arg1->WriteSC(IO, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_PutParenthesis(ExprType, ')', Arg1, SIR_LEFT_ARG, IO);
	  IO->PutS("->");
	  if (Symbol)
	     { IO->PutS(Symbol->Name);
	      } /* fi */
	  else
	     { IO->PutS(GL_MAIN_METHOD_NAME);
	      } /* esle */
	  break;
	 }
     case SIR_EXPR_POST_INCREMENT:
	{ SIR_PutParenthesis(ExprType, '(', Arg1, SIR_LEFT_ARG, IO);
	  if ((SIR_Error = Arg1->WriteSC(IO, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_PutParenthesis(ExprType, ')', Arg1, SIR_LEFT_ARG, IO);
	  IO->PutS("++ ");
	  break;
	 }
     case SIR_EXPR_POST_DECREMENT:
	{ SIR_PutParenthesis(ExprType, '(', Arg1, SIR_LEFT_ARG, IO);
	  if ((SIR_Error = Arg1->WriteSC(IO, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_PutParenthesis(ExprType, ')', Arg1, SIR_LEFT_ARG, IO);
	  IO->PutS("-- ");
	  break;
	 }
     case SIR_EXPR_BITSLICE:
	{ if (CplusplusMode)
	     { IO->PutS("(");
	       SIR_PutParenthesis(SIR_EXPR_MEMBER_ACCESS,
					'(', Arg1, SIR_LEFT_ARG, IO);
	       if ((SIR_Error = Arg1->WriteSC(IO, WriteNotes, CplusplusMode,
					SIR_TYPE_BIT,	/* not relaxed! */
					-1 /* unused in this case! */)))
		  { return(SIR_Error);
		   } /* fi */
	       SIR_PutParenthesis(SIR_EXPR_MEMBER_ACCESS,
					')', Arg1, SIR_LEFT_ARG, IO);
	       BasedOnSymbol = Arg1->IsSymbol();
	       if (  (BasedOnSymbol)
		   &&(BasedOnSymbol->Type->Type == SIR_TYPE_SIGNAL))
		  { IO->PutS(SIR_SIM_SIG_SLICE_NAME);
		   } /* fi */
	       else
		  { if (  (BasedOnSymbol)
			&&(BasedOnSymbol->Type->Type == SIR_TYPE_BUFFER))
		       { IO->PutS(SIR_SIM_BUF_SLICE_NAME);
			} /* fi */
		    else
		       { IO->PutS(SIR_BIT_SLICE_NAME);
			} /* esle */
		   } /* esle */
	       if (  (Arg1->Type->Type == SIR_TYPE_BIT)
		   ||(Arg1->Type->Type == SIR_TYPE_UBIT))
		  { /* Arg1 is possibly a non-normalized bitvector */
		    IO->PrintF("(%d,%d, %d,%d))",
				Arg1->Type->LeftBound, Arg1->Type->RightBound,
				LeftBound, RightBound);
		   } /* fi */
	       else
		  { /* Arg1 has been converted to a normalized bitvector */
		    IO->PrintF("(%d,%d))",
				LeftBound, RightBound);
		   } /* esle */
	      } /* fi */
	  else
	     { SIR_PutParenthesis(ExprType, '(', Arg1, SIR_LEFT_ARG, IO);
	       if ((SIR_Error = Arg1->WriteSC(IO, WriteNotes, CplusplusMode,
					SIR_TYPE_BIT,	/* not relaxed! */
					-1 /* unused in this case! */)))
		  { return(SIR_Error);
		   } /* fi */
	       SIR_PutParenthesis(ExprType, ')', Arg1, SIR_LEFT_ARG, IO);
	       IO->PrintF("[%d:%d]", LeftBound, RightBound);
	      } /* esle */
	  break;
	 }
     case SIR_EXPR_PRE_INCREMENT:
	{ IO->PutS(" ++");
	  SIR_PutParenthesis(ExprType, '(', Arg1, SIR_RIGHT_ARG, IO);
	  if ((SIR_Error = Arg1->WriteSC(IO, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_PutParenthesis(ExprType, ')', Arg1, SIR_RIGHT_ARG, IO);
	  break;
	 }
     case SIR_EXPR_PRE_DECREMENT:
	{ IO->PutS(" --");
	  SIR_PutParenthesis(ExprType, '(', Arg1, SIR_RIGHT_ARG, IO);
	  if ((SIR_Error = Arg1->WriteSC(IO, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_PutParenthesis(ExprType, ')', Arg1, SIR_RIGHT_ARG, IO);
	  break;
	 }
     case SIR_EXPR_ADDRESS_OF:
	{ IO->PutS(" &");
	  SIR_PutParenthesis(ExprType, '(', Arg1, SIR_RIGHT_ARG, IO);
	  if ((CplusplusMode) && (Arg1->Type->Type == SIR_TYPE_METHOD))
	     { assert(Arg1->Type->ClassSymbol != NULL);
	       IO->PutS(Arg1->Type->ClassSymbol->Name);
	       IO->PutS("::");
	      } /* fi */
	  if ((SIR_Error = Arg1->WriteSC(IO, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_PutParenthesis(ExprType, ')', Arg1, SIR_RIGHT_ARG, IO);
	  break;
	 }
     case SIR_EXPR_CONTENT_OF:
	{ if (  (CplusplusMode)
	      &&(Arg1->Type->SubType->Type == SIR_TYPE_METHOD))
		  IO->PutS(" this->*");
	  else
		  IO->PutS(" *");
	  SIR_PutParenthesis(ExprType, '(', Arg1, SIR_RIGHT_ARG, IO);
	  if ((SIR_Error = Arg1->WriteSC(IO, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_PutParenthesis(ExprType, ')', Arg1, SIR_RIGHT_ARG, IO);
	  break;
	 }
     case SIR_EXPR_POSITIVE:
	{ IO->PutS(" +");
	  SIR_PutParenthesis(ExprType, '(', Arg1, SIR_RIGHT_ARG, IO);
	  if ((SIR_Error = Arg1->WriteSC(IO, WriteNotes, CplusplusMode,
						SIR_TYPE_ANY_TYPE, 0, true)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_PutParenthesis(ExprType, ')', Arg1, SIR_RIGHT_ARG, IO);
	  break;
	 }
     case SIR_EXPR_NEGATIVE:
	{ IO->PutS(" -");
	  SIR_PutParenthesis(ExprType, '(', Arg1, SIR_RIGHT_ARG, IO);
	  if ((SIR_Error = Arg1->WriteSC(IO, WriteNotes, CplusplusMode,
						SIR_TYPE_ANY_TYPE, 0, true)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_PutParenthesis(ExprType, ')', Arg1, SIR_RIGHT_ARG, IO);
	  break;
	 }
     case SIR_EXPR_NOT:
	{ IO->PutS(" ~");
	  SIR_PutParenthesis(ExprType, '(', Arg1, SIR_RIGHT_ARG, IO);
	  if ((SIR_Error = Arg1->WriteSC(IO, WriteNotes, CplusplusMode,
						SIR_TYPE_ANY_TYPE, 0, true)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_PutParenthesis(ExprType, ')', Arg1, SIR_RIGHT_ARG, IO);
	  break;
	 }
     case SIR_EXPR_LOGICAL_NOT:
	{ IO->PutS(" !");
	  SIR_PutParenthesis(ExprType, '(', Arg1, SIR_RIGHT_ARG, IO);
	  if ((SIR_Error = Arg1->WriteSC(IO, WriteNotes, CplusplusMode,
						SIR_TYPE_BOOL, 0, true)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_PutParenthesis(ExprType, ')', Arg1, SIR_RIGHT_ARG, IO);
	  break;
	 }
     case SIR_EXPR_SIZEOF_EXPR:
	{ if (  (CplusplusMode)
	      &&(Arg1->Type->TypeClass() == SIR_TYPECLASS_BITVECTOR))
	     { /* sizeof(bit[:]) cannot be overloaded, so we compute it here */
	       TmpConst = new SIR_Constant(SIR_CONST_UINT,
				(UNSIGNED_LONG_LONG) Arg1->Type->SizeOf());
	       IO->PutS(TmpConst->Print(CplusplusMode));
	       delete TmpConst;
	      } /* fi */
	  else
	     { IO->PutS("sizeof ");
	       SIR_PutParenthesis(ExprType, '(', Arg1, SIR_RIGHT_ARG, IO);
	       if ((SIR_Error = Arg1->WriteSC(IO, WriteNotes, CplusplusMode)))
		  { return(SIR_Error);
		   } /* fi */
	       SIR_PutParenthesis(ExprType, ')', Arg1, SIR_RIGHT_ARG, IO);
	      } /* esle */
	  break;
	 }
     case SIR_EXPR_SIZEOF_TYPE:
	{ if (  (CplusplusMode)
	      &&(TypeArg->TypeClass() == SIR_TYPECLASS_BITVECTOR))
	     { /* sizeof(bit[:]) cannot be overloaded, so we compute it here */
	       TmpConst = new SIR_Constant(SIR_CONST_UINT,
				(UNSIGNED_LONG_LONG) TypeArg->SizeOf());
	       IO->PutS(TmpConst->Print(CplusplusMode));
	       delete TmpConst;
	      } /* fi */
	  else
	     { IO->PutS("sizeof(");
	       Buffer = "";
	       IO->PutS(TypeArg->PrettyString(&Buffer,
						WriteNotes, CplusplusMode));
	       IO->PutC(')');
	      } /* esle */
	  break;
	 }
     case SIR_EXPR_TYPE_CONVERSION:
	{ if (  (CplusplusMode)
	      &&(CastNeedsExplicitTypeConv(Arg1->Type->Type, TypeArg->Type)))
	     { if ((SIR_Error = Arg1->WriteSC(IO, WriteNotes, CplusplusMode,
						TypeArg->Type,
			_BITLEN(TypeArg->LeftBound, TypeArg->RightBound))))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  else	/* SpecC mode, or: C++ compiler knows how to handle the rest */
	     { IO->PutC('(');
	       Buffer = "";
	       IO->PutS(TypeArg->PrettyString(&Buffer,
						WriteNotes, CplusplusMode));
	       IO->PutC(')');
	       SIR_PutParenthesis(ExprType, '(', Arg1, SIR_RIGHT_ARG, IO);
	       if ((SIR_Error = Arg1->WriteSC(IO, WriteNotes, CplusplusMode,
						SIR_TYPE_ANY_TYPE, 0, true)))
		  { return(SIR_Error);
		   } /* fi */
	       SIR_PutParenthesis(ExprType, ')', Arg1, SIR_RIGHT_ARG, IO);
	      } /* else */
	  break;
	 }
     case SIR_EXPR_CONCATENATION:
	{ if (CplusplusMode)
	     { IO->PutS(SIR_BIT_CONCAT_NAME "(");
	       SIR_PutParenthesis(SIR_EXPR_COMMA,
					'(', Arg1, SIR_RIGHT_ARG, IO);
	       if ((SIR_Error = Arg1->WriteSC(IO, WriteNotes, true,
					SIR_TYPE_BIT,	/* not relaxed! */
					-1 /* unused in this case! */,
					true)))
		  { return(SIR_Error);
		   } /* fi */
	       SIR_PutParenthesis(SIR_EXPR_COMMA,
					')', Arg1, SIR_RIGHT_ARG, IO);
	       IO->PutC(',');
	       SIR_LineInfo::WrapLine(IO);
	       SIR_PutParenthesis(SIR_EXPR_COMMA,
					'(', Arg1, SIR_RIGHT_ARG, IO);
	       if ((SIR_Error = Arg2->WriteSC(IO, WriteNotes, true,
					SIR_TYPE_BIT,	/* not relaxed! */
					-1 /* unused in this case! */,
					true)))
		  { return(SIR_Error);
		   } /* fi */
	       SIR_PutParenthesis(SIR_EXPR_COMMA,
					')', Arg1, SIR_RIGHT_ARG, IO);
	       IO->PutC(')');
	      } /* fi */
	  else
	     { if ((SIR_Error = WriteSC2("@", IO, WriteNotes, false)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* esle */
	  break;
	 }
     case SIR_EXPR_MULTIPLY:
	{ if ((SIR_Error = WriteSC2("*", IO, WriteNotes, CplusplusMode,
				EXPLICIT_TYPE(Type->Type, Arg1->Type->Type),
				EXPLICIT_TYPE(Type->Type, Arg2->Type->Type),
				_BITLEN(Type->LeftBound, Type->RightBound),
				_BITLEN(Type->LeftBound, Type->RightBound))))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_DIVIDE:
	{ if ((SIR_Error = WriteSC2("/", IO, WriteNotes, CplusplusMode,
				EXPLICIT_TYPE(Type->Type, Arg1->Type->Type),
				EXPLICIT_TYPE(Type->Type, Arg2->Type->Type),
				_BITLEN(Type->LeftBound, Type->RightBound),
				_BITLEN(Type->LeftBound, Type->RightBound))))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_MODULO:
	{ if ((SIR_Error = WriteSC2("%", IO, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_ADD:
	{ if (  (Arg1->Type->Type == SIR_TYPE_POINTER)
	      ||(Arg1->Type->Type == SIR_TYPE_ARRAY))
	     { if ((SIR_Error = WriteSC2("+", IO,
					WriteNotes, CplusplusMode,
					SIR_TYPE_ANY_TYPE, SIR_TYPE_INT)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  else
	     { if (  (Arg2->Type->Type == SIR_TYPE_POINTER)
		   ||(Arg2->Type->Type == SIR_TYPE_ARRAY))
		  { if ((SIR_Error = WriteSC2("+", IO,
					WriteNotes, CplusplusMode,
					SIR_TYPE_INT, SIR_TYPE_ANY_TYPE)))
		       { return(SIR_Error);
			} /* fi */
		   } /* fi */
	       else
		  { if ((SIR_Error = WriteSC2("+", IO,
					WriteNotes, CplusplusMode,
				EXPLICIT_TYPE(Type->Type, Arg1->Type->Type),
				EXPLICIT_TYPE(Type->Type, Arg2->Type->Type),
				_BITLEN(Type->LeftBound, Type->RightBound),
				_BITLEN(Type->LeftBound, Type->RightBound))))
		       { return(SIR_Error);
			} /* fi */
		   } /* esle */
	      } /* esle */
	  break;
	 }
     case SIR_EXPR_SUBTRACT:
	{ if (  (Arg1->Type->Type == SIR_TYPE_POINTER)
	      ||(Arg1->Type->Type == SIR_TYPE_ARRAY))
	     { if ((SIR_Error = WriteSC2("-", IO,
					WriteNotes, CplusplusMode,
					SIR_TYPE_ANY_TYPE, SIR_TYPE_INT)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  else
	     { if (  (Arg2->Type->Type == SIR_TYPE_POINTER)
		   ||(Arg2->Type->Type == SIR_TYPE_ARRAY))
		  { if ((SIR_Error = WriteSC2("-", IO,
					WriteNotes, CplusplusMode,
					SIR_TYPE_INT, SIR_TYPE_ANY_TYPE)))
		       { return(SIR_Error);
			} /* fi */
		   } /* fi */
	       else
		  { if ((SIR_Error = WriteSC2("-", IO,
					WriteNotes, CplusplusMode,
				EXPLICIT_TYPE(Type->Type, Arg1->Type->Type),
				EXPLICIT_TYPE(Type->Type, Arg2->Type->Type),
				_BITLEN(Type->LeftBound, Type->RightBound),
				_BITLEN(Type->LeftBound, Type->RightBound))))
		       { return(SIR_Error);
			} /* fi */
		   } /* esle */
	      } /* esle */
	  break;
	 }
     case SIR_EXPR_SHIFT_LEFT:
	{ if ((SIR_Error = WriteSC2("<<", IO, WriteNotes, CplusplusMode,
					SIR_TYPE_ANY_TYPE, SIR_TYPE_INT)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_SHIFT_RIGHT:
	{ if ((SIR_Error = WriteSC2(">>", IO, WriteNotes, CplusplusMode,
					SIR_TYPE_ANY_TYPE, SIR_TYPE_INT)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_LESS:
	{ if ((SIR_Error = WriteSC2("<", IO, WriteNotes, CplusplusMode,
			EXPLICIT_TYPE(Arg2->Type->Type, Arg1->Type->Type),
			EXPLICIT_TYPE(Arg1->Type->Type, Arg2->Type->Type),
			_BITLEN(Arg2->Type->LeftBound,Arg2->Type->RightBound),
			_BITLEN(Arg1->Type->LeftBound,Arg1->Type->RightBound))))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_GREATER:
	{ if ((SIR_Error = WriteSC2(">", IO, WriteNotes, CplusplusMode,
			EXPLICIT_TYPE(Arg2->Type->Type, Arg1->Type->Type),
			EXPLICIT_TYPE(Arg1->Type->Type, Arg2->Type->Type),
			_BITLEN(Arg2->Type->LeftBound,Arg2->Type->RightBound),
			_BITLEN(Arg1->Type->LeftBound,Arg1->Type->RightBound))))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_LESS_EQUAL:
	{ if ((SIR_Error = WriteSC2("<=", IO, WriteNotes, CplusplusMode,
			EXPLICIT_TYPE(Arg2->Type->Type, Arg1->Type->Type),
			EXPLICIT_TYPE(Arg1->Type->Type, Arg2->Type->Type),
			_BITLEN(Arg2->Type->LeftBound,Arg2->Type->RightBound),
			_BITLEN(Arg1->Type->LeftBound,Arg1->Type->RightBound))))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_GREATER_EQUAL:
	{ if ((SIR_Error = WriteSC2(">=", IO, WriteNotes, CplusplusMode,
			EXPLICIT_TYPE(Arg2->Type->Type, Arg1->Type->Type),
			EXPLICIT_TYPE(Arg1->Type->Type, Arg2->Type->Type),
			_BITLEN(Arg2->Type->LeftBound,Arg2->Type->RightBound),
			_BITLEN(Arg1->Type->LeftBound,Arg1->Type->RightBound))))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_EQUAL:
	{ if ((SIR_Error = WriteSC2("==", IO, WriteNotes, CplusplusMode,
			EXPLICIT_TYPE(Arg2->Type->Type, Arg1->Type->Type),
			EXPLICIT_TYPE(Arg1->Type->Type, Arg2->Type->Type),
			_BITLEN(Arg2->Type->LeftBound,Arg2->Type->RightBound),
			_BITLEN(Arg1->Type->LeftBound,Arg1->Type->RightBound))))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_NOT_EQUAL:
	{ if ((SIR_Error = WriteSC2("!=", IO, WriteNotes, CplusplusMode,
			EXPLICIT_TYPE(Arg2->Type->Type, Arg1->Type->Type),
			EXPLICIT_TYPE(Arg1->Type->Type, Arg2->Type->Type),
			_BITLEN(Arg2->Type->LeftBound,Arg2->Type->RightBound),
			_BITLEN(Arg1->Type->LeftBound,Arg1->Type->RightBound))))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_AND:
	{ if ((SIR_Error = WriteSC2("&", IO, WriteNotes, CplusplusMode,
				EXPLICIT_TYPE2(Type->Type, Arg1->Type->Type),
				EXPLICIT_TYPE2(Type->Type, Arg2->Type->Type),
				_BITLEN(Type->LeftBound, Type->RightBound),
				_BITLEN(Type->LeftBound, Type->RightBound))))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_EOR:
	{ if ((SIR_Error = WriteSC2("^", IO, WriteNotes, CplusplusMode,
				EXPLICIT_TYPE2(Type->Type, Arg1->Type->Type),
				EXPLICIT_TYPE2(Type->Type, Arg2->Type->Type),
				_BITLEN(Type->LeftBound, Type->RightBound),
				_BITLEN(Type->LeftBound, Type->RightBound))))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_OR:
	{ if ((SIR_Error = WriteSC2("|", IO, WriteNotes, CplusplusMode,
				EXPLICIT_TYPE2(Type->Type, Arg1->Type->Type),
				EXPLICIT_TYPE2(Type->Type, Arg2->Type->Type),
				_BITLEN(Type->LeftBound, Type->RightBound),
				_BITLEN(Type->LeftBound, Type->RightBound))))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_LOGICAL_AND:
	{ if ((SIR_Error = WriteSC2("&&", IO, WriteNotes, CplusplusMode,
						SIR_TYPE_BOOL, SIR_TYPE_BOOL)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_LOGICAL_OR:
	{ if ((SIR_Error = WriteSC2("||", IO, WriteNotes, CplusplusMode,
						SIR_TYPE_BOOL, SIR_TYPE_BOOL)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_CONDITION:
	{ SIR_PutParenthesis(ExprType, '(', Arg1, SIR_LEFT_ARG, IO);
	  if ((SIR_Error = Arg1->WriteSC(IO, WriteNotes, CplusplusMode,
						SIR_TYPE_BOOL, 0, true)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_PutParenthesis(ExprType, ')', Arg1, SIR_LEFT_ARG, IO);
	  IO->PutS(" ? ");
	  SIR_LineInfo::WrapLine(IO);
	  if (  (CplusplusMode)
	      &&(  (Type->Type == SIR_TYPE_BIT)
		 ||(Type->Type == SIR_TYPE_UBIT)))
	     { IO->PrintF("(" SIR_BIT_CLASS_NAME "<%d,%s>(",
				_BITLEN(Type->LeftBound, Type->RightBound),
				Type->Type == SIR_TYPE_BIT ?
				SIR_BIT_SIGNED_FLAG : SIR_BIT_UNSIGNED_FLAG);
	       if ((SIR_Error = Arg2->WriteSC(IO, WriteNotes, CplusplusMode,
					SIR_TYPE_ANY_TYPE, 0, true)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->PutS("))");
	      } /* fi */
#ifndef HAVE_LLONG
	  else if (  (CplusplusMode)
	      &&(  (Type->Type == SIR_TYPE_LONGLONG)
		 ||(Type->Type == SIR_TYPE_ULONGLONG)))
	     { IO->PrintF("(" SIR_LLONG_CLASS_NAME "<%s>(",
				Type->Type == SIR_TYPE_LONGLONG ?
			SIR_LLONG_SIGNED_FLAG : SIR_LLONG_UNSIGNED_FLAG);
	       if ((SIR_Error = Arg2->WriteSC(IO, WriteNotes, CplusplusMode,
					SIR_TYPE_ANY_TYPE, 0, true)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->PutS("))");
	      } /* fi esle */
#endif /* HAVE_LLONG */
	  else
	     { SIR_PutParenthesis(ExprType, '(', Arg2, SIR_LEFT_ARG, IO);
	       if ((SIR_Error = Arg2->WriteSC(IO, WriteNotes, CplusplusMode,
				EXPLICIT_TYPE(Type->Type, Arg2->Type->Type),
				_BITLEN(Type->LeftBound, Type->RightBound),
				true)))
		  { return(SIR_Error);
		   } /* fi */
	       SIR_PutParenthesis(ExprType, ')', Arg2, SIR_LEFT_ARG, IO);
	      } /* esle */
	  IO->PutS(" : ");
	  SIR_LineInfo::WrapLine(IO);
	  if (  (CplusplusMode)
	      &&(  (Type->Type == SIR_TYPE_BIT)
		 ||(Type->Type == SIR_TYPE_UBIT)))
	     { IO->PrintF("(" SIR_BIT_CLASS_NAME "<%d,%s>(",
				_BITLEN(Type->LeftBound, Type->RightBound),
				Type->Type == SIR_TYPE_BIT ?
				SIR_BIT_SIGNED_FLAG : SIR_BIT_UNSIGNED_FLAG);
	       if ((SIR_Error = Arg3->WriteSC(IO, WriteNotes, CplusplusMode,
					SIR_TYPE_ANY_TYPE, 0, true)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->PutS("))");
	      } /* fi */
#ifndef HAVE_LLONG
	  else if (  (CplusplusMode)
	      &&(  (Type->Type == SIR_TYPE_LONGLONG)
		 ||(Type->Type == SIR_TYPE_ULONGLONG)))
	     { IO->PrintF("(" SIR_LLONG_CLASS_NAME "<%s>(",
				Type->Type == SIR_TYPE_LONGLONG ?
			SIR_LLONG_SIGNED_FLAG : SIR_LLONG_UNSIGNED_FLAG);
	       if ((SIR_Error = Arg3->WriteSC(IO, WriteNotes, CplusplusMode,
					SIR_TYPE_ANY_TYPE, 0, true)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->PutS("))");
	      } /* fi esle */
#endif /* HAVE_LLONG */
	  else
	     { SIR_PutParenthesis(ExprType, '(', Arg3, SIR_LEFT_ARG, IO);
	       if ((SIR_Error = Arg3->WriteSC(IO, WriteNotes, CplusplusMode,
				EXPLICIT_TYPE(Type->Type, Arg3->Type->Type),
				_BITLEN(Type->LeftBound, Type->RightBound),
				true)))
		  { return(SIR_Error);
		   } /* fi */
	       SIR_PutParenthesis(ExprType, ')', Arg3, SIR_LEFT_ARG, IO);
	      } /* esle */
	  break;
	 }
     case SIR_EXPR_ASSIGNMENT:
	{ if ((SIR_Error = WriteSC2("=", IO, WriteNotes, CplusplusMode,
				SIR_TYPE_ANY_TYPE,
				SIR_RELAXED(Arg1->Type->Type, Arg2->Type->Type),
				0, _BITLEN(Arg1->Type->LeftBound,
						Arg1->Type->RightBound),
				false, true)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_MUL_ASSIGN:
	{ if ((SIR_Error = WriteSC2("*=", IO, WriteNotes, CplusplusMode,
				SIR_TYPE_ANY_TYPE,
				SIR_RELAXED(Arg1->Type->Type, Arg2->Type->Type),
				0, _BITLEN(Arg1->Type->LeftBound,
						Arg1->Type->RightBound),
				false, true)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_DIV_ASSIGN:
	{ if ((SIR_Error = WriteSC2("/=", IO, WriteNotes, CplusplusMode,
				SIR_TYPE_ANY_TYPE,
				SIR_RELAXED(Arg1->Type->Type, Arg2->Type->Type),
				0, _BITLEN(Arg1->Type->LeftBound,
						Arg1->Type->RightBound),
				false, true)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_MOD_ASSIGN:
	{ if ((SIR_Error = WriteSC2("%=", IO, WriteNotes, CplusplusMode,
				SIR_TYPE_ANY_TYPE,
				SIR_RELAXED(Arg1->Type->Type, Arg2->Type->Type),
				0, _BITLEN(Arg1->Type->LeftBound,
						Arg1->Type->RightBound),
				false, true)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_ADD_ASSIGN:
	{ if (Arg1->Type->Type == SIR_TYPE_POINTER)
	     /* note: SIR_TYPE_ARRAY not allowed */
	     { if ((SIR_Error = WriteSC2("+=", IO, WriteNotes, CplusplusMode,
					SIR_TYPE_ANY_TYPE, SIR_TYPE_INT,
					0, 0, false, true)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  else
	     { if ((SIR_Error = WriteSC2("+=", IO, WriteNotes, CplusplusMode,
				SIR_TYPE_ANY_TYPE,
				SIR_RELAXED(Arg1->Type->Type, Arg2->Type->Type),
				0, _BITLEN(Arg1->Type->LeftBound,
						Arg1->Type->RightBound),
				false, true)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* esle */
	  break;
	 }
     case SIR_EXPR_SUB_ASSIGN:
	{ if (Arg1->Type->Type == SIR_TYPE_POINTER)
	     /* note: SIR_TYPE_ARRAY not allowed */
	     { if ((SIR_Error = WriteSC2("-=", IO, WriteNotes, CplusplusMode,
					SIR_TYPE_ANY_TYPE, SIR_TYPE_INT,
					0, 0, false, true)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  else
	     { if ((SIR_Error = WriteSC2("-=", IO, WriteNotes, CplusplusMode,
				SIR_TYPE_ANY_TYPE,
				SIR_RELAXED(Arg1->Type->Type, Arg2->Type->Type),
				0, _BITLEN(Arg1->Type->LeftBound,
						Arg1->Type->RightBound),
				false, true)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* esle */
	  break;
	 }
     case SIR_EXPR_SHL_ASSIGN:
	{ if ((SIR_Error = WriteSC2("<<=", IO, WriteNotes, CplusplusMode,
					SIR_TYPE_ANY_TYPE, SIR_TYPE_INT,
					0, 0, false, true)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_SHR_ASSIGN:
	{ if ((SIR_Error = WriteSC2(">>=", IO, WriteNotes, CplusplusMode,
					SIR_TYPE_ANY_TYPE, SIR_TYPE_INT,
					0, 0, false, true)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_AND_ASSIGN:
	{ if ((SIR_Error = WriteSC2("&=", IO, WriteNotes, CplusplusMode,
				SIR_TYPE_ANY_TYPE,
				SIR_RELAXED(Arg1->Type->Type, Arg2->Type->Type),
				0, _BITLEN(Arg1->Type->LeftBound,
						Arg1->Type->RightBound),
				false, true)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_EOR_ASSIGN:
	{ if ((SIR_Error = WriteSC2("^=", IO, WriteNotes, CplusplusMode,
				SIR_TYPE_ANY_TYPE,
				SIR_RELAXED(Arg1->Type->Type, Arg2->Type->Type),
				0, _BITLEN(Arg1->Type->LeftBound,
						Arg1->Type->RightBound),
				false, true)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_OR_ASSIGN:
	{ if ((SIR_Error = WriteSC2("|=", IO, WriteNotes, CplusplusMode,
				SIR_TYPE_ANY_TYPE,
				SIR_RELAXED(Arg1->Type->Type, Arg2->Type->Type),
				0, _BITLEN(Arg1->Type->LeftBound,
						Arg1->Type->RightBound),
				false, true)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_COMMA:
	{ if ((SIR_Error = WriteSC2(",", IO, WriteNotes, CplusplusMode,
					SIR_TYPE_ANY_TYPE, SIR_TYPE_ANY_TYPE,
					0, 0, false, false)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     default:
	{ assert(false);	/* bad expression type */
	 }
    } /* hctiws */

if (Suffix)
   { IO->PrintF(Suffix, ExpectedLength);	/* finish explicit conversion */
    } /* fi */

if ((SIR_Error = IO->Check()))
   { return(SIR_Error);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Expression::WriteSC */


ERROR SIR_Expression::WriteSC2(	/* (re-) generates SpecC source code */
	const char	*Op,	/* (for binary ops only) */
	gl_io		*IO,
	bool		WriteNotes,
	bool		CplusplusMode /* = false */,
	SIR_TYPETYPE	ExpectedType1 /* = SIR_TYPE_ANY_TYPE */,
	SIR_TYPETYPE	ExpectedType2 /* = SIR_TYPE_ANY_TYPE */,
	int		ExpectedLength1 /* = 0 */, /* for expected bitvectors */
	int		ExpectedLength2 /* = 0 */, /* for expected bitvectors */
	bool		RValueExpected1 /* = true */,
	bool		RValueExpected2 /* = true */)
{

SIR_PutParenthesis(ExprType, '(', Arg1, SIR_LEFT_ARG, IO);
if ((SIR_Error = Arg1->WriteSC(IO, WriteNotes, CplusplusMode,
				ExpectedType1, ExpectedLength1,
				RValueExpected1)))
   { return(SIR_Error);
    } /* fi */
SIR_PutParenthesis(ExprType, ')', Arg1, SIR_LEFT_ARG, IO);

IO->PutC(' ');
IO->PutS(Op);
IO->PutC(' ');
SIR_LineInfo::WrapLine(IO);

SIR_PutParenthesis(ExprType, '(', Arg2, SIR_RIGHT_ARG, IO);
if ((SIR_Error = Arg2->WriteSC(IO, WriteNotes, CplusplusMode,
				ExpectedType2, ExpectedLength2,
				RValueExpected2)))
   { return(SIR_Error);
    } /* fi */
SIR_PutParenthesis(ExprType, ')', Arg2, SIR_RIGHT_ARG, IO);

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Expression::WriteSC2 */


#ifndef HAVE_ARYASGN

ERROR SIR_Expression::WriteArrayAssignment(     /* gen. array assignments */
        gl_io       *IO,                    /* (in C++ code generation) */
        bool            WriteNotes)
{
sir_type        *ArrayType;
unsigned int    Depth,
                Index;

assert(ExprType == SIR_EXPR_ASSIGNMENT);        // must be assignment
assert(Arg1->Type->Type == SIR_TYPE_ARRAY);     // must be an array
assert(Arg2->Type->Type == SIR_TYPE_ARRAY);     // must be an array

Depth = 0;
ArrayType = Arg1->Type;
while(ArrayType->Type == SIR_TYPE_ARRAY)
   { assert(ArrayType->Size > 0);       /* type checked */
     IO->PrintF("{ unsigned int " SIR_CXX_INDEX_NAME "%u; "
                        "for(" SIR_CXX_INDEX_NAME "%u=0;"
                        SIR_CXX_INDEX_NAME "%u<%d;"
                        SIR_CXX_INDEX_NAME "%u++) ",
                        Depth, Depth, Depth, ArrayType->Size, Depth);
     Depth++;
     ArrayType = ArrayType->SubType;
    } /* elihw */

IO->PutC('(');
IO->TabStepUp();
if ((SIR_Error = Arg1->WriteSC(IO, WriteNotes, true)))
   { return(SIR_Error);
    } /* fi */
IO->TabStepDown();
IO->PutC(')');
for(Index=0; Index<Depth; Index++)
   { IO->PrintF("[" SIR_CXX_INDEX_NAME "%u]", Index);
    } /* rof */

IO->PutS(" = ");

IO->PutC('(');
IO->TabStepUp();
if ((SIR_Error = Arg2->WriteSC(IO, WriteNotes, true)))
   { return(SIR_Error);
    } /* fi */
IO->TabStepDown();
IO->PutC(')');
for(Index=0; Index<Depth; Index++)
   { IO->PrintF("[" SIR_CXX_INDEX_NAME "%u]", Index);
    } /* rof */

IO->PutC(';');
for(Index=0; Index<Depth; Index++)
   { IO->PutS(" }");
    } /* rof */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Expression::WriteArrayAssignment */

#endif /* HAVE_ARYASGN */

sir_constant *SIR_Expression::Eval(void)	/* evaluate a constant expr. */
{						/* (returns a new Constant */
sir_constant		*EvalArg1,		/*  or NULL and SIR_Error) */
			*EvalArg2,
			*TmpResult;
LONG_LONG		TmpValueLL1,
			TmpValueLL2;
UNSIGNED_LONG_LONG	TmpValueULL1,
			TmpValueULL2;
long double		TmpValueLD1,
			TmpValueLD2;
sir_bit			*TmpValueBIT1,
			*TmpValueBIT2;
unsigned int		Line;
sir_fileinfo		*File;

if (  (  (Type->TypeClass() == SIR_TYPECLASS_OTHER)
       ||(Type->TypeClass() == SIR_TYPECLASS_CLASS))
    &&(  (Type->Type != SIR_TYPE_POINTER)
       ||(Type->SubType->Type != SIR_TYPE_CHAR)))
   { SIR_Error = SIR_ERROR_EXPRESSION_NOT_CONSTANT;
     return(NULL);
    } /* fi */

switch (ExprType)
   { case SIR_EXPR_VOID:
	{ assert(false);	/* bad expression type */
	 }
     case SIR_EXPR_CONSTANT:
	{ TmpResult = new SIR_Constant(Constant);
	  return(TmpResult->Converted(Type->ConstType(),
				_BITLEN(Type->LeftBound, Type->RightBound)));
	 }
     case SIR_EXPR_IDENTIFIER:
	{ if (  (Symbol->Type->Const)		/* constant variable */
	      &&(Symbol->Initializer)
	      &&(Symbol->Initializer->Initializer))
	     { TmpResult = new SIR_Constant(Symbol->Initializer->Initializer);
	       TmpResult->SetLineInfo(this);
	       return(TmpResult->Converted(Type->ConstType(),
				_BITLEN(Type->LeftBound, Type->RightBound)));
	      } /* fi */
	  if (Symbol->IsEnumMember())		/* enum constant */
	     { TmpResult = new SIR_Constant(SIR_CONST_INT, Symbol->EnumValue);
	       TmpResult->SetLineInfo(this);
	       return(TmpResult->Converted(Type->ConstType(),
				_BITLEN(Type->LeftBound, Type->RightBound)));
	      } /* fi */
	  SIR_Error = SIR_ERROR_EXPRESSION_NOT_CONSTANT;
	  return(NULL);
	 }
     case SIR_EXPR_PARENTHESES:
	{ return(Arg1->Eval());		/* simple recursion */
	 }
     case SIR_EXPR_THIS:
     case SIR_EXPR_ARRAY_ACCESS:
     case SIR_EXPR_FUNCTION_CALL:
     case SIR_EXPR_MEMBER_ACCESS:
     case SIR_EXPR_MEMBER_POINTER:
     case SIR_EXPR_POST_INCREMENT:
     case SIR_EXPR_POST_DECREMENT:
	{ SIR_Error = SIR_ERROR_EXPRESSION_NOT_CONSTANT;
	  return(NULL);
	 }
     case SIR_EXPR_BITSLICE:
	{ if (!(EvalArg1 = Arg1->Eval()))
	     { return(NULL);
	      } /* fi */
	  SIR_Convert2BitVector(EvalArg1);
	  TmpResult = new SIR_Constant((*(EvalArg1->BIT_Value))(LeftBound,
								RightBound));
	  TmpResult->SetLineInfo(EvalArg1);
	  delete EvalArg1;
	  assert(TmpResult->Type == Type->ConstType());	/* need no conversion */
	  return(TmpResult);
	 }
     case SIR_EXPR_PRE_INCREMENT:
     case SIR_EXPR_PRE_DECREMENT:
     case SIR_EXPR_ADDRESS_OF:
     case SIR_EXPR_CONTENT_OF:
	{ SIR_Error = SIR_ERROR_EXPRESSION_NOT_CONSTANT;
	  return(NULL);
	 }
     case SIR_EXPR_POSITIVE:
	{ return(Arg1->Eval()			/* nothing to compute */
		->Converted(Type->ConstType(),	/* (but maybe to convert) */
			_BITLEN(Type->LeftBound, Type->RightBound)));
	 }
     case SIR_EXPR_NEGATIVE:
	{ switch (SIR_PrepareOperand(Arg1,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
						- TmpValueLL1, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
						- TmpValueULL1, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
						- TmpValueLD1, Line, File);
		    break;
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(-(*TmpValueBIT1), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1;
	  return(TmpResult);
	 }
     case SIR_EXPR_NOT:
	{ switch (SIR_PrepareOperand(Arg1,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
						~ TmpValueLL1, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
						~ TmpValueULL1, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { assert(false);	/* invalid operand */
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(~(*TmpValueBIT1), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1;
	  return(TmpResult);
	 }
     case SIR_EXPR_LOGICAL_NOT:
	{ if (!(EvalArg1 = Arg1->Eval()))
	     { return(NULL);
	      } /* fi */
	  EvalArg1->Converted(SIR_CONST_BOOL);
	  EvalArg1->B_Value = ! EvalArg1->B_Value;
	  assert(EvalArg1->Type == Type->ConstType());	/* already converted */
	  return(EvalArg1);
	 }
     case SIR_EXPR_SIZEOF_EXPR:
	{ TmpResult = new SIR_Constant(SIR_CONST_UINT,
				(UNSIGNED_LONG_LONG) Arg1->Type->SizeOf());
	  TmpResult->SetLineInfo(this);
	  assert(SIR_CONST_UINT == Type->ConstType());	/* need no conversion */
	  return(TmpResult);
	 }
     case SIR_EXPR_SIZEOF_TYPE:
	{ TmpResult = new SIR_Constant(SIR_CONST_UINT,
				(UNSIGNED_LONG_LONG) TypeArg->SizeOf());
	  TmpResult->SetLineInfo(this);
	  assert(SIR_CONST_UINT == Type->ConstType());	/* need no conversion */
	  return(TmpResult);
	 }
     case SIR_EXPR_TYPE_CONVERSION:
	{ if (!(EvalArg1 = Arg1->Eval()))
	     { return(NULL);
	      } /* fi */
	  switch(TypeArg->Type)
	     { case SIR_TYPE_BOOL:
	       case SIR_TYPE_CHAR:
	       case SIR_TYPE_UCHAR:
	       case SIR_TYPE_SHORT:
	       case SIR_TYPE_USHORT:
	       case SIR_TYPE_INT:
	       case SIR_TYPE_UINT:
	       case SIR_TYPE_LONG:
	       case SIR_TYPE_ULONG:
	       case SIR_TYPE_LONGLONG:
	       case SIR_TYPE_ULONGLONG:
	       case SIR_TYPE_FLOAT:
	       case SIR_TYPE_DOUBLE:
	       case SIR_TYPE_LONGDOUBLE:
	       case SIR_TYPE_BIT:
	       case SIR_TYPE_UBIT:
		  { EvalArg1->Converted(	/* exploit smart enumerations */
			(SIR_CONSTTYPE) TypeArg->Type,
			_BITLEN(TypeArg->LeftBound, TypeArg->RightBound));
		    break;
		   }
	       case SIR_TYPE_VOID:
	       case SIR_TYPE_EVENT:
		  { assert(false);	/* type checking has to catch this */
		   }
	       case SIR_TYPE_POINTER:
		  { SIR_Error = SIR_ERROR_EXPRESSION_NOT_CONSTANT;
		    delete EvalArg1;
		    return(NULL);
		   }
	       case SIR_TYPE_STRUCT:
	       case SIR_TYPE_UNION:
		  { assert(false);	/* type checking has to catch this */
		   }
	       case SIR_TYPE_ENUM:
		  { EvalArg1->Converted(SIR_CONST_INT);
		    break;
		   }
	       case SIR_TYPE_ARRAY:
	       case SIR_TYPE_METHOD:
	       case SIR_TYPE_FUNCTION:
		  { SIR_Error = SIR_ERROR_EXPRESSION_NOT_CONSTANT;
		    delete EvalArg1;
		    return(NULL);
		   }
	       case SIR_TYPE_ANY_TYPE:
	       case SIR_TYPE_BEHAVIOR:
	       case SIR_TYPE_CHANNEL:
	       case SIR_TYPE_INTERFACE:
		  { assert(false);	/* type checking has to catch this */
		   }
	       case SIR_TYPE_SIGNAL:
	       case SIR_TYPE_BUFFER:
		  { assert(false);	/* type checking has to catch this */
		   }
	       default:
		  { assert(false);	/* bad Type */
		   }
	      } /* hctiws */
	  assert(EvalArg1->Type == Type->ConstType());	/* already converted */
	  return(EvalArg1);
	 }
     case SIR_EXPR_CONCATENATION:
	{ if (!(EvalArg1 = Arg1->Eval()))
	     { return(NULL);
	      } /* fi */
	  if (!(EvalArg2 = Arg2->Eval()))
	     { return(NULL);
	      } /* fi */
	  SIR_Convert2BitVector(EvalArg1);
	  SIR_Convert2BitVector(EvalArg2);
	  TmpResult = new SIR_Constant(
				sir_bit::concat(*EvalArg1->BIT_Value,
						*EvalArg2->BIT_Value));
	  if (EvalArg1->LineInfo)
	     { TmpResult->SetLineInfo(EvalArg1);
	      } /* fi */
	  else
	     { TmpResult->SetLineInfo(EvalArg2);
	      } /* esle */
	  delete EvalArg1;
	  delete EvalArg2;
	  assert(TmpResult->Type == Type->ConstType());	/* need no conversion */
	  return(TmpResult);
	 }
     case SIR_EXPR_MULTIPLY:
	{ switch (SIR_PrepareBinaryOperands(Arg1, Arg2,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&TmpValueLL2, &TmpValueULL2,
				&TmpValueLD2, &TmpValueBIT2,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLL1 * TmpValueLL2, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueULL1 * TmpValueULL2, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLD1 * TmpValueLD2, Line, File);
		    break;
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(
				(*TmpValueBIT1) * (*TmpValueBIT2), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_DIVIDE:
	{ switch (SIR_PrepareBinaryOperands(Arg1, Arg2,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&TmpValueLL2, &TmpValueULL2,
				&TmpValueLD2, &TmpValueBIT2,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { if (TmpValueLL2 == 0)
		       { SIR_Error = SIR_ERROR_DIVISION_BY_ZERO;
			 delete TmpValueBIT1; delete TmpValueBIT2;
			 return(NULL);
			} /* fi */
		    TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLL1 / TmpValueLL2, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { if (TmpValueULL2 == 0)
		       { SIR_Error = SIR_ERROR_DIVISION_BY_ZERO;
			 delete TmpValueBIT1; delete TmpValueBIT2;
			 return(NULL);
			} /* fi */
		    TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueULL1 / TmpValueULL2, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { if (TmpValueLD2 == 0.0l)
		       { SIR_Error = SIR_ERROR_DIVISION_BY_ZERO;
			 delete TmpValueBIT1; delete TmpValueBIT2;
			 return(NULL);
			} /* fi */
		    TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLD1 / TmpValueLD2, Line, File);
		    break;
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { if (*TmpValueBIT2 == 0)
		       { SIR_Error = SIR_ERROR_DIVISION_BY_ZERO;
			 delete TmpValueBIT1; delete TmpValueBIT2;
			 return(NULL);
			} /* fi */
		    TmpResult = new SIR_Constant(
				(*TmpValueBIT1) / (*TmpValueBIT2), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_MODULO:
	{ switch (SIR_PrepareBinaryOperands(Arg1, Arg2,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&TmpValueLL2, &TmpValueULL2,
				&TmpValueLD2, &TmpValueBIT2,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { if (TmpValueLL2 == 0)
		       { SIR_Error = SIR_ERROR_DIVISION_BY_ZERO;
			 delete TmpValueBIT1; delete TmpValueBIT2;
			 return(NULL);
			} /* fi */
		    TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLL1 % TmpValueLL2, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { if (TmpValueULL2 == 0)
		       { SIR_Error = SIR_ERROR_DIVISION_BY_ZERO;
			 delete TmpValueBIT1; delete TmpValueBIT2;
			 return(NULL);
			} /* fi */
		    TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueULL1 % TmpValueULL2, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { assert(false);	/* type checking has to catch this */
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { if (*TmpValueBIT2 == 0)
		       { SIR_Error = SIR_ERROR_DIVISION_BY_ZERO;
			 delete TmpValueBIT1; delete TmpValueBIT2;
			 return(NULL);
			} /* fi */
		    TmpResult = new SIR_Constant(
				(*TmpValueBIT1) % (*TmpValueBIT2), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_ADD:
	{ switch (SIR_PrepareBinaryOperands(Arg1, Arg2,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&TmpValueLL2, &TmpValueULL2,
				&TmpValueLD2, &TmpValueBIT2,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLL1 + TmpValueLL2, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueULL1 + TmpValueULL2, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLD1 + TmpValueLD2, Line, File);
		    break;
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(
				(*TmpValueBIT1) + (*TmpValueBIT2), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_SUBTRACT:
	{ switch (SIR_PrepareBinaryOperands(Arg1, Arg2,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&TmpValueLL2, &TmpValueULL2,
				&TmpValueLD2, &TmpValueBIT2,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLL1 - TmpValueLL2, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueULL1 - TmpValueULL2, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLD1 - TmpValueLD2, Line, File);
		    break;
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(
				(*TmpValueBIT1) - (*TmpValueBIT2), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_SHIFT_LEFT:
	{ if (SIR_PrepareOperand(Arg2,
			&TmpValueLL2, &TmpValueULL2,
			&TmpValueLD2, &TmpValueBIT2,
					&Line, &File) == SIR_DO_ERROR)
	     { return(NULL);
	      } /* fi */
	  switch (SIR_PrepareOperand(Arg1,
			&TmpValueLL1, &TmpValueULL1,
			&TmpValueLD1, &TmpValueBIT1))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
					TmpValueLL1 << TmpValueBIT2->toInt(),
					Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
					TmpValueULL1 << TmpValueBIT2->toInt(),
					Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { assert(false);	/* type checking has to catch this */
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(
				(*TmpValueBIT1) << TmpValueBIT2->toInt(),
					Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_SHIFT_RIGHT:
	{ if (SIR_PrepareOperand(Arg2,
			&TmpValueLL2, &TmpValueULL2,
			&TmpValueLD2, &TmpValueBIT2,
					&Line, &File) == SIR_DO_ERROR)
	     { return(NULL);
	      } /* fi */
	  switch (SIR_PrepareOperand(Arg1,
			&TmpValueLL1, &TmpValueULL1,
			&TmpValueLD1, &TmpValueBIT1))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
					TmpValueLL1 >> TmpValueBIT2->toInt(),
					Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
					TmpValueULL1 >> TmpValueBIT2->toInt(),
					Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { assert(false);	/* type checking has to catch this */
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(
				(*TmpValueBIT1) >> TmpValueBIT2->toInt(),
					Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_LESS:
	{ switch (SIR_PrepareBinaryOperands(Arg1, Arg2,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&TmpValueLL2, &TmpValueULL2,
				&TmpValueLD2, &TmpValueBIT2,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLL1 < TmpValueLL2, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueULL1 < TmpValueULL2, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLD1 < TmpValueLD2, Line, File);
		    break;
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				(*TmpValueBIT1) < (*TmpValueBIT2), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_GREATER:
	{ switch (SIR_PrepareBinaryOperands(Arg1, Arg2,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&TmpValueLL2, &TmpValueULL2,
				&TmpValueLD2, &TmpValueBIT2,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLL1 > TmpValueLL2, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueULL1 > TmpValueULL2, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLD1 > TmpValueLD2, Line, File);
		    break;
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				(*TmpValueBIT1) > (*TmpValueBIT2), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_LESS_EQUAL:
	{ switch (SIR_PrepareBinaryOperands(Arg1, Arg2,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&TmpValueLL2, &TmpValueULL2,
				&TmpValueLD2, &TmpValueBIT2,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLL1 <= TmpValueLL2, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueULL1 <= TmpValueULL2, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLD1 <= TmpValueLD2, Line, File);
		    break;
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				(*TmpValueBIT1) <= (*TmpValueBIT2), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_GREATER_EQUAL:
	{ switch (SIR_PrepareBinaryOperands(Arg1, Arg2,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&TmpValueLL2, &TmpValueULL2,
				&TmpValueLD2, &TmpValueBIT2,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLL1 >= TmpValueLL2, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueULL1 >= TmpValueULL2, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLD1 >= TmpValueLD2, Line, File);
		    break;
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				(*TmpValueBIT1) >= (*TmpValueBIT2), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_EQUAL:
	{ switch (SIR_PrepareBinaryOperands(Arg1, Arg2,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&TmpValueLL2, &TmpValueULL2,
				&TmpValueLD2, &TmpValueBIT2,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLL1 == TmpValueLL2, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueULL1 == TmpValueULL2, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLD1 == TmpValueLD2, Line, File);
		    break;
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				(*TmpValueBIT1) == (*TmpValueBIT2), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_NOT_EQUAL:
	{ switch (SIR_PrepareBinaryOperands(Arg1, Arg2,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&TmpValueLL2, &TmpValueULL2,
				&TmpValueLD2, &TmpValueBIT2,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLL1 != TmpValueLL2, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueULL1 != TmpValueULL2, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLD1 != TmpValueLD2, Line, File);
		    break;
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				(*TmpValueBIT1) != (*TmpValueBIT2), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_AND:
	{ switch (SIR_PrepareBinaryOperands(Arg1, Arg2,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&TmpValueLL2, &TmpValueULL2,
				&TmpValueLD2, &TmpValueBIT2,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLL1 & TmpValueLL2, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueULL1 & TmpValueULL2, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { assert(false);	/* type checking has to catch this */
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(
				(*TmpValueBIT1) & (*TmpValueBIT2), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_EOR:
	{ switch (SIR_PrepareBinaryOperands(Arg1, Arg2,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&TmpValueLL2, &TmpValueULL2,
				&TmpValueLD2, &TmpValueBIT2,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLL1 ^ TmpValueLL2, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueULL1 ^ TmpValueULL2, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { assert(false);	/* type checking has to catch this */
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(
				(*TmpValueBIT1) ^ (*TmpValueBIT2), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_OR:
	{ switch (SIR_PrepareBinaryOperands(Arg1, Arg2,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&TmpValueLL2, &TmpValueULL2,
				&TmpValueLD2, &TmpValueBIT2,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLL1 | TmpValueLL2, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueULL1 | TmpValueULL2, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { assert(false);	/* type checking has to catch this */
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(
				(*TmpValueBIT1) | (*TmpValueBIT2), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_LOGICAL_AND:
	{ if (!(EvalArg1 = Arg1->Eval()))
	     { return(NULL);
	      } /* fi */
	  assert(Type->ConstType() == SIR_CONST_BOOL);	/* assert bool result */
	  EvalArg1->Converted(SIR_CONST_BOOL);
	  assert(EvalArg1->Type == SIR_CONST_BOOL);
	  if (! EvalArg1->B_Value)	/* short-cut evaluation! */
	     { return(EvalArg1);
	      } /* fi */
	  delete EvalArg1;
	  if (!(EvalArg2 = Arg2->Eval()))
	     { return(NULL);
	      } /* fi */
	  return(EvalArg2->Converted(SIR_CONST_BOOL));
	 }
     case SIR_EXPR_LOGICAL_OR:
	{ if (!(EvalArg1 = Arg1->Eval()))
	     { return(NULL);
	      } /* fi */
	  assert(Type->ConstType() == SIR_CONST_BOOL);	/* assert bool result */
	  EvalArg1->Converted(SIR_CONST_BOOL);
	  assert(EvalArg1->Type == SIR_CONST_BOOL);
	  if (EvalArg1->B_Value)	/* short-cut evaluation! */
	     { return(EvalArg1);
	      } /* fi */
	  delete EvalArg1;
	  if (!(EvalArg2 = Arg2->Eval()))
	     { return(NULL);
	      } /* fi */
	  return(EvalArg2->Converted(SIR_CONST_BOOL));
	 }
     case SIR_EXPR_CONDITION:
	{ bool		Test;

	  if (!(EvalArg1 = Arg1->Eval()))
	     { return(NULL);
	      } /* fi */
	  EvalArg1->Converted(SIR_CONST_BOOL);
	  assert(EvalArg1->Type == SIR_CONST_BOOL);
	  Test = EvalArg1->B_Value;
	  delete EvalArg1;
	  if (Test)	/* evaluate either Arg2 or Arg3 (only one of them) */
	     { if (!(EvalArg2 = Arg2->Eval()))
		  { return(NULL);
		   } /* fi */
	      } /* fi */
	  else
	     { if (!(EvalArg2 = Arg3->Eval()))
		  { return(NULL);
		   } /* fi */
	      } /* esle */
	  if ((int)Type->Type >= (int)SIR_CONST_CHARSTRING)
	     { SIR_Error = SIR_ERROR_EXPRESSION_NOT_CONSTANT;
	       delete EvalArg2;
	       return(NULL);
	      } /* fi */
	  return(EvalArg2->Converted(Type->ConstType(),
			_BITLEN(Type->LeftBound, Type->RightBound)));
	 }
     case SIR_EXPR_ASSIGNMENT:
     case SIR_EXPR_MUL_ASSIGN:
     case SIR_EXPR_DIV_ASSIGN:
     case SIR_EXPR_MOD_ASSIGN:
     case SIR_EXPR_ADD_ASSIGN:
     case SIR_EXPR_SUB_ASSIGN:
     case SIR_EXPR_SHL_ASSIGN:
     case SIR_EXPR_SHR_ASSIGN:
     case SIR_EXPR_AND_ASSIGN:
     case SIR_EXPR_EOR_ASSIGN:
     case SIR_EXPR_OR_ASSIGN:
     case SIR_EXPR_COMMA:
	{ SIR_Error = SIR_ERROR_EXPRESSION_NOT_CONSTANT;
	  return(NULL);
	 }
     default:
	{ assert(false);	/* bad expression type */
	 }
    } /* hctiws */

return(NULL);	/* should never be executed */

} /* end of SIR_Expression::Eval */


int SIR_Expression::IntegerEval(void)	/* evaluate a constant expr. to int */
{					/* (may set SIR_Error!) */
sir_constant	*Result;
int		IntValue;

if (!(Result = this->Eval()))	/* evaluate myself */
   { return(0);	/* SIR_Error is set */
    } /* fi */

IntValue = 0;	/* default in case of errors */

switch(Result->Type)
   { case SIR_CONST_BOOL:
	{ IntValue = Result->B_Value;
	  break;
	 }
     case SIR_CONST_CHAR:
	{ IntValue = Result->C_Value;
	  break;
	 }
     case SIR_CONST_UCHAR:
	{ IntValue = Result->UC_Value;
	  break;
	 }
     case SIR_CONST_SHORT:
	{ IntValue = Result->S_Value;
	  break;
	 }
     case SIR_CONST_USHORT:
	{ IntValue = Result->US_Value;
	  break;
	 }
     case SIR_CONST_INT:
	{ IntValue = Result->I_Value;
	  break;
	 }
     case SIR_CONST_UINT:
	{ IntValue = Result->UI_Value;
	  break;
	 }
     case SIR_CONST_LONG:
	{ IntValue = Result->L_Value;
	  break;
	 }
     case SIR_CONST_ULONG:
	{ IntValue = Result->UL_Value;
	  break;
	 }
     case SIR_CONST_LONGLONG:
	{ IntValue = SIR_LLONG2I(Result->LL_Value);
	  break;
	 }
     case SIR_CONST_ULONGLONG:
	{ IntValue = SIR_LLONG2I(Result->ULL_Value);
	  break;
	 }
     case SIR_CONST_FLOAT:
	{ assert(false);	/* should have been promoted */
	 }
     case SIR_CONST_DOUBLE:
	{ IntValue = (int)Result->D_Value;
	  break;
	 }
     case SIR_CONST_LONGDOUBLE:
	{ IntValue = (int)Result->LD_Value;
	  break;
	 }
     case SIR_CONST_BIT:
     case SIR_CONST_UBIT:
	{ IntValue = Result->BIT_Value->toInt();
	  break;
	 }
     case SIR_CONST_CHARSTRING:
	{ SIR_Error = SIR_ERROR_CONST_EXPR_NOT_INTEGER;
	  break;
	 }
     default:
	{ assert(false);	/* bad Type */
	 }
    } /* hctiws */

delete Result;	/* not needed any more */

return(IntValue);

} /* end of SIR_Expression::IntegerEval */


sir_lineinfo *SIR_Expression::GetFirstLineInfo(	/* obtain first line info */
	void)
{
sir_lineinfo	*Line;
sir_expression	*Arg;

if (LineInfo)
   { return(LineInfo);
    } /* fi */

if (  (Constant)
    &&(Constant->LineInfo))
   { return(Constant->LineInfo);
    } /* fi */
if (  (Arg1)
    &&(Line = Arg1->GetFirstLineInfo()))
   { return(Line);
    } /* fi */
if (  (Arg2)
    &&(Line = Arg2->GetFirstLineInfo()))
   { return(Line);
    } /* fi */
if (  (Arg3)
    &&(Line = Arg3->GetFirstLineInfo()))
   { return(Line);
    } /* fi */
if (Args)
   { Arg = Args->First();
     while(Arg)
	{ if ((Line = Arg->GetFirstLineInfo()))
	     { return(Line);
	      } /* fi */
	  Arg = Arg->Succ();
	 } /* elihw */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Expression::GetFirstLineInfo */


int SIR_Expression::Cmp(                /* compare two exprs. (like strcmp) */
        sir_expression  *Expr1,         /* (recursive!) */
        sir_expression  *Expr2)
{
int             CmpVal;
sir_expression  *SubExpr1,
                *SubExpr2;

assert(Expr1 != NULL);
assert(Expr2 != NULL);

if (Expr1 == Expr2)     /* check for identity first */
   { return(0);
    } /* fi */

if (0 != (CmpVal = Expr1->ExprType - Expr2->ExprType))
   { return(CmpVal);
    } /* fi */

if (Expr1->Constant)
   { assert(Expr2->Constant != NULL);
     if (0 != (CmpVal = Expr1->Constant->Type - Expr2->Constant->Type))
        { return(CmpVal);
         } /* fi */
     else
        { return(SIR_Constant::Compare(Expr1->Constant, Expr2->Constant));
         } /* esle */
    } /* fi */

if (0 != (CmpVal = (long)Expr1->Symbol - (long)Expr2->Symbol))
   { return(CmpVal);
    } /* fi */

if (Expr1->Arg1)
   { assert(Expr2->Arg1 != NULL);
     if (0 != (CmpVal = Cmp(Expr1->Arg1, Expr2->Arg1)))
        { return(CmpVal);
         } /* fi */
    } /* fi */

if (Expr1->Arg2)
   { assert(Expr2->Arg2 != NULL);
     if (0 != (CmpVal = Cmp(Expr1->Arg2, Expr2->Arg2)))
        { return(CmpVal);
         } /* fi */
    } /* fi */

if (Expr1->Arg3)
   { assert(Expr2->Arg3 != NULL);
     if (0 != (CmpVal = Cmp(Expr1->Arg3, Expr2->Arg3)))
        { return(CmpVal);
         } /* fi */
    } /* fi */

if (Expr1->Args)
   { assert(Expr2->Args != NULL);
     if (0 != (CmpVal = Expr1->Args->NumElements() -
                                Expr2->Args->NumElements()))
        { return(CmpVal);
         } /* fi */
     SubExpr1 = Expr1->Args->First();
     SubExpr2 = Expr2->Args->First();
     while(SubExpr1)
        { assert(SubExpr2 != NULL);
          if (0 != (CmpVal = Cmp(SubExpr1, SubExpr2)))
             { return(CmpVal);
              } /* fi */
          SubExpr1 = SubExpr1->Succ();
          SubExpr2 = SubExpr2->Succ();
         } /* elihw */
    } /* fi */

if (Expr1->TypeArg)
   { assert(Expr2->TypeArg != NULL);
     if (0 != (CmpVal = SIR_Types::TypeCmp(Expr1->TypeArg, Expr2->TypeArg)))
        { return(CmpVal);
         } /* fi */
    } /* fi */

// no need to compare result type

if (0 != (CmpVal = Expr1->LeftBound - Expr2->LeftBound))
   { return(CmpVal);
    } /* fi */

if (0 != (CmpVal = Expr1->RightBound - Expr2->RightBound))
   { return(CmpVal);
    } /* fi */

return(0);      // the two expressions are the same

} /* end of SIR_Expression::Cmp */


sir_expression *SIR_Expression::New(	/* creates a constant expression (#1) */
	sir_constant	*NewConst,	/* (returns NULL if SIR_Error) */
	sir_types	*TypeTable)
{
sir_expression	*NewExpr;

assert(TypeTable != NULL);
assert(NewConst != NULL);

NewExpr = new SIR_Expression(SIR_EXPR_CONSTANT, NewConst,
			TypeTable->FindOrInsert(NewConst));

return(NewExpr);

} /* end of SIR_Expression::New #1 */


sir_expression *SIR_Expression::New(	/* creates expr. with 1-3 args. (#3) */
	SIR_EXPRTYPE	ExprType,	/* (returns NULL if SIR_Error) */
	sir_expression	* /*New*/ Arg1,
	sir_expression	* /*New*/ Arg2 /* = NULL */,
	sir_expression	* /*New*/ Arg3 /* = NULL */)
{
sir_types	*TypeTable;
sir_expression	*NewExpr;
sir_type	*ResultType = NULL;	/* keep the compiler quiet */

assert(Arg1 != NULL);
// Arg2 may be NULL
// Arg3 may be NULL

assert(Arg1->Type != NULL);
TypeTable = Arg1->Type->GetTable();
assert(TypeTable != NULL);

switch(ExprType)
   { case SIR_EXPR_VOID:
     case SIR_EXPR_CONSTANT:
     case SIR_EXPR_IDENTIFIER:
	{ assert(false);	/* not supported/handled here */
	 }
     case SIR_EXPR_PARENTHESES:
	{ ResultType = Arg1->Type;	/* (nothing changes) */
	  break;
	 }
     case SIR_EXPR_THIS:
	{ assert(false);	/* not supported/handled here */
	 }
     case SIR_EXPR_ARRAY_ACCESS:
	{ if (  (Arg2->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
	      &&(Arg2->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR))
	     { SIR_Error = SIR_ERROR_INVALID_ARRAY_INDEX;
	       return(NULL);
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_ARRAY)
	      ||(Arg1->Type->Type == SIR_TYPE_POINTER))
	     { ResultType = Arg1->Type->SubType;
	       ResultType = TypeTable->FindOrInsertModifiedType(
				ResultType,
				ResultType->Const,
				ResultType->Volatile,
				/* propagate direction to result type */
				Arg1->Type->Direction);
	      } /* fi */
	  else
	     { if (  (Arg1->Type->Type == SIR_TYPE_BIT)
		   ||(Arg1->Type->Type == SIR_TYPE_UBIT))
		  { ResultType = TypeTable->FindOrInsert(
				SIR_TYPE_UBIT,		/* [] operator on    */
				Arg1->Type->Const,	/* bitvectors yields */
				Arg1->Type->Volatile,	/* unsigned bit[0:0] */
				0, 0,
				/* propagate direction to result type */
				Arg1->Type->Direction);
		   } /* fi */
	       else
		  { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_ARRAY_ACC;
		    return(NULL);
		   } /* esle */
	      } /* esle */
	  break;
	 }
     case SIR_EXPR_FUNCTION_CALL:
	{ assert(false);	/* handled in SIR_Expression::New() #4 */
	 }
     case SIR_EXPR_MEMBER_ACCESS:
     case SIR_EXPR_MEMBER_POINTER:
	{ assert(false);	/* handled in SIR_Expression::New() #5 */
	 }
     case SIR_EXPR_POST_INCREMENT:
     case SIR_EXPR_POST_DECREMENT:
	{ if (  (Arg1->CheckReadAccess())
	      ||(Arg1->CheckWriteAccess()))
	     { return(NULL);
	      } /* fi */
	  if (  (Arg1->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
	      &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_FLOATING)
	      &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR)
	      &&(Arg1->Type->Type != SIR_TYPE_POINTER)
	      &&(Arg1->Type->Type != SIR_TYPE_ARRAY))	// allowed as fct. arg.
							// (fixed 09/16/02, RD)
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_INCR_DECR;
	       return(NULL);
	      } /* fi */
	  ResultType = Arg1->Type;	/* (not promoted!) */
	  break;
	 }
     case SIR_EXPR_BITSLICE:
	{ assert(false);	/* handled in SIR_Expression::New() #7 */
	 }
     case SIR_EXPR_PRE_INCREMENT:
     case SIR_EXPR_PRE_DECREMENT:
	{ if (  (Arg1->CheckReadAccess())
	      ||(Arg1->CheckWriteAccess()))
	     { return(NULL);
	      } /* fi */
	  if (  (Arg1->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
	      &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_FLOATING)
	      &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR)
	      &&(Arg1->Type->Type != SIR_TYPE_POINTER))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_INCR_DECR;
	       return(NULL);
	      } /* fi */
	  ResultType = Arg1->Type;	/* (not promoted!) */
	  break;
	 }
     case SIR_EXPR_ADDRESS_OF:
	{ sir_symbol	*LvalueSymbol;

	  if (! Arg1->IsLvalue())
	     { SIR_Error = SIR_ERROR_LVALUE_REQUIRED_FOR_ADDR_OP;
	       return(NULL);
	      } /* fi */
	  if ((LvalueSymbol = Arg1->IsSymbol()))
	     { if (  (LvalueSymbol->StorageClass == SIR_STORAGE_REGISTER)
		   ||(LvalueSymbol->StorageClass == SIR_STORAGE_PIPED)
		   ||(LvalueSymbol->StorageClass == SIR_STORAGE_TYPEDEF)
		   ||(LvalueSymbol->Type->Type == SIR_TYPE_SIGNAL)
		   ||(LvalueSymbol->Type->Type == SIR_TYPE_BUFFER))
		  { SIR_Error = SIR_ERROR_INVALID_ARGUMENT_FOR_ADDR_OP;
		    return(NULL);
		   } /* fi */
	      } /* fi */
	  ResultType = TypeTable->FindOrInsert(SIR_TYPE_POINTER, Arg1->Type);
	  break;
	 }
     case SIR_EXPR_CONTENT_OF:
	{ if (Arg1->CheckReadAccess())
	     { return(NULL);
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_POINTER)
	      ||(Arg1->Type->Type == SIR_TYPE_ARRAY))
	     { ResultType = Arg1->Type->SubType;
	      } /* fi */
	  else
	     { SIR_Error = SIR_ERROR_OPERAND_NOT_POINTER_OR_ARRAY;
	       return(NULL);
	      } /* esle */
	  break;
	 }
     case SIR_EXPR_POSITIVE:
     case SIR_EXPR_NEGATIVE:
	{ SIR_TYPETYPE	Arg1Type,
			ResType;
	  int		Arg1Len,
			ResLen;

	  if (Arg1->CheckReadAccess())
	     { return(NULL);
	      } /* fi */
	  if (  (Arg1->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
	      &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_FLOATING)
	      &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_ARITH_OP;
	       return(NULL);
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_LONGDOUBLE)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONGDOUBLE);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_DOUBLE)
	      ||(Arg1->Type->Type == SIR_TYPE_FLOAT))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_DOUBLE);
	       break;
	      } /* fi */
	  if (Arg1->Type->TypeClass() == SIR_TYPECLASS_BITVECTOR)
	     { if (! SIR_IsConvertable2BitVector(Arg1->Type,
						&Arg1Type, &Arg1Len))
		  { assert(false);	/* must be convertable */
		   } /* fi */
	       ResType = (_BITUSGN_UNARY(Arg1Len, (Arg1Type == SIR_TYPE_UBIT))
					? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	       ResLen = _BITLEN_UNARY(Arg1Len, (Arg1Type == SIR_TYPE_UBIT));
	       ResultType = TypeTable->FindOrInsert(ResType, false, false,
							ResLen-1, 0);
	       break;
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_ULONGLONG)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_ULONGLONG);
	       break;
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_LONGLONG)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONGLONG);
	       break;
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_ULONG)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_ULONG);
	       break;
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_LONG)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONG);
	       break;
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_UINT)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_UINT);
	       break;
	      } /* fi */
	  ResultType = TypeTable->FindOrInsert(SIR_TYPE_INT);
	  break;
	 }
     case SIR_EXPR_NOT:
	{ SIR_TYPETYPE	Arg1Type,
			ResType;
	  int		Arg1Len,
			ResLen;

	  if (Arg1->CheckReadAccess())
	     { return(NULL);
	      } /* fi */
	  if (  (Arg1->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
	      &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_BIT_OP;
	       return(NULL);
	      } /* fi */
	  if (Arg1->Type->TypeClass() == SIR_TYPECLASS_BITVECTOR)
	     { if (! SIR_IsConvertable2BitVector(Arg1->Type,
						&Arg1Type, &Arg1Len))
		  { assert(false);	/* must be convertable */
		   } /* fi */
	       ResType = (_BITUSGN_UNARY(Arg1Len, (Arg1Type == SIR_TYPE_UBIT))
					? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	       ResLen = _BITLEN_UNARY(Arg1Len, (Arg1Type == SIR_TYPE_UBIT));
	       ResultType = TypeTable->FindOrInsert(ResType, false, false,
							ResLen-1, 0);
	       break;
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_ULONGLONG)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_ULONGLONG);
	       break;
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_LONGLONG)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONGLONG);
	       break;
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_ULONG)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_ULONG);
	       break;
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_LONG)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONG);
	       break;
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_UINT)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_UINT);
	       break;
	      } /* fi */
	  ResultType = TypeTable->FindOrInsert(SIR_TYPE_INT);
	  break;
	 }
     case SIR_EXPR_LOGICAL_NOT:
	{ if (Arg1->CheckReadAccess())
	     { return(NULL);
	      } /* fi */
	  if (! SIR_IsConvertable2Bool(Arg1->Type))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_BOOL_OP;
	       return(NULL);
	      } /* fi */
	  ResultType = TypeTable->FindOrInsert(SIR_TYPE_BOOL);
	  break;
	 }
     case SIR_EXPR_SIZEOF_EXPR:
	{ switch(Arg1->Type->Type)
	     { case SIR_TYPE_VOID:
		  { SIR_Error = SIR_ERROR_ILLEGAL_SIZEOF_VOID;
		    return(NULL);
		   }
	       case SIR_TYPE_EVENT:
		  { SIR_Error = SIR_ERROR_ILLEGAL_SIZEOF_EVENT;
		    return(NULL);
		   }
	       case SIR_TYPE_METHOD:
	       case SIR_TYPE_FUNCTION:
		  { SIR_Error = SIR_ERROR_ILLEGAL_SIZEOF_FUNCTION;
		    return(NULL);
		   }
	       case SIR_TYPE_ANY_TYPE:
	       case SIR_TYPE_BEHAVIOR:
	       case SIR_TYPE_CHANNEL:
	       case SIR_TYPE_INTERFACE:
		  { SIR_Error = SIR_ERROR_ILLEGAL_SIZEOF_CLASS;
		    return(NULL);
		   }
	       case SIR_TYPE_SIGNAL:
		  { SIR_Error = SIR_ERROR_ILLEGAL_SIZEOF_SIGNAL;
		    return(NULL);
		   }
	       case SIR_TYPE_BUFFER:
		  { SIR_Error = SIR_ERROR_ILLEGAL_SIZEOF_BUFFER;
		    return(NULL);
		   }
	       default:	/* any other type */
		  { /* type is legal for sizeof() operator */
		    break;
		   }
	      } /* hctiws */
	  ResultType = TypeTable->FindOrInsert(SIR_TYPE_UINT);
	  break;
	 }
     case SIR_EXPR_SIZEOF_TYPE:
     case SIR_EXPR_TYPE_CONVERSION:
	{ assert(false);	/* handled in SIR_Expression::New() #6 */
	 }
     case SIR_EXPR_CONCATENATION:
	{ SIR_TYPETYPE	Arg1Type,
			Arg2Type,
			ResType;
	  int		Arg1Len,
			Arg2Len,
			ResLen;

	  if (  (Arg1->CheckReadAccess())
	      ||(Arg2->CheckReadAccess()))
	     { return(NULL);
	      } /* fi */
	  if (! SIR_IsConvertable2BitVector(Arg1->Type, &Arg1Type, &Arg1Len))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_CONCAT;
	       return(NULL);
	      } /* fi */
	  if (! SIR_IsConvertable2BitVector(Arg2->Type, &Arg2Type, &Arg2Len))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_CONCAT;
	       return(NULL);
	      } /* fi */
	  ResType = (_BITUSGN_CONCAT(Arg1Len, (Arg1Type == SIR_TYPE_UBIT),
					Arg2Len, (Arg2Type == SIR_TYPE_UBIT))
					? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	  ResLen = _BITLEN_CONCAT(Arg1Len, (Arg1Type == SIR_TYPE_UBIT),
					Arg2Len, (Arg2Type == SIR_TYPE_UBIT));
	  ResultType = TypeTable->FindOrInsert(ResType, false, false,
						ResLen-1, 0);
	  break;
	 }
     case SIR_EXPR_ADD:
     case SIR_EXPR_SUBTRACT:
	{ /* handle pointer arithmetic first */
	  if (  (Arg1->Type->Type == SIR_TYPE_POINTER)
	      ||(Arg1->Type->Type == SIR_TYPE_ARRAY))
	     { if (  (Arg1->CheckReadAccess())
		   ||(Arg2->CheckReadAccess()))
		  { return(NULL);
		   } /* fi */
	       // bug fix 04/24/06, RD:
	       // pointer subtraction is allowed (see K&R, page 103),
	       // if both arguments point to the same subtype (in the same array)
	       if (  (ExprType == SIR_EXPR_SUBTRACT)
		   &&(  (Arg2->Type->Type == SIR_TYPE_POINTER)
		      ||(Arg2->Type->Type == SIR_TYPE_ARRAY))
		   &&(0 == SIR_Types::TypeCmp(Arg1->Type->SubType, Arg2->Type->SubType,true)))
		  { ResultType = TypeTable->FindOrInsert(SIR_TYPE_INT);
		    break;
		   } /* fi */
	       if (  (Arg2->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		   &&(Arg2->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR))
		  { SIR_Error = SIR_ERROR_INVALID_POINTER_ARITHMETIC;
		    return(NULL);
		   } /* fi */
	       ResultType = Arg1->Type;
	       break;
	      } /* fi */
	  if (  (Arg2->Type->Type == SIR_TYPE_POINTER)
	      ||(Arg2->Type->Type == SIR_TYPE_ARRAY))
	     { if (  (Arg1->CheckReadAccess())
		   ||(Arg2->CheckReadAccess()))
		  { return(NULL);
		   } /* fi */
	       if (  (Arg1->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		   &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR))
		  { SIR_Error = SIR_ERROR_INVALID_POINTER_ARITHMETIC;
		    return(NULL);
		   } /* fi */
	       ResultType = Arg2->Type;
	       break;
	      } /* fi */
	  /* now go on as with multiplication, division, ... */
	 }
/*   case SIR_EXPR_ADD:		fall-through case! */
/*   case SIR_EXPR_SUBTRACT:	fall-through case! */
     case SIR_EXPR_MULTIPLY:
     case SIR_EXPR_DIVIDE:
	{ SIR_TYPETYPE	Arg1Type,
			Arg2Type,
			ResType;
	  int		Arg1Len,
			Arg2Len,
			ResLen;

	  if (  (Arg1->CheckReadAccess())
	      ||(Arg2->CheckReadAccess()))
	     { return(NULL);
	      } /* fi */
	  if (  (  (Arg1->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		 &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_FLOATING)
		 &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR))
	      ||(  (Arg2->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		 &&(Arg2->Type->TypeClass() != SIR_TYPECLASS_FLOATING)
		 &&(Arg2->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR)))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_ARITH_OP;
	       return(NULL);
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_LONGDOUBLE)
	      ||(Arg2->Type->Type == SIR_TYPE_LONGDOUBLE))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONGDOUBLE);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_DOUBLE)
	      ||(Arg2->Type->Type == SIR_TYPE_DOUBLE)
	      ||(Arg1->Type->Type == SIR_TYPE_FLOAT)
	      ||(Arg2->Type->Type == SIR_TYPE_FLOAT))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_DOUBLE);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->TypeClass() == SIR_TYPECLASS_BITVECTOR)
	      ||(Arg2->Type->TypeClass() == SIR_TYPECLASS_BITVECTOR))
	     { if (! SIR_IsConvertable2BitVector(Arg1->Type,
						&Arg1Type, &Arg1Len))
		  { assert(false);	/* must be convertable */
		   } /* fi */
	       if (! SIR_IsConvertable2BitVector(Arg2->Type,
						&Arg2Type, &Arg2Len))
		  { assert(false);	/* must be convertable */
		   } /* fi */
	       ResType = (_BITUSGN_BINARY(Arg1Len, (Arg1Type == SIR_TYPE_UBIT),
					Arg2Len, (Arg2Type == SIR_TYPE_UBIT))
					? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	       ResLen = _BITLEN_BINARY(Arg1Len, (Arg1Type == SIR_TYPE_UBIT),
					Arg2Len, (Arg2Type == SIR_TYPE_UBIT));
	       ResultType = TypeTable->FindOrInsert(ResType, false, false,
							ResLen-1, 0);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_ULONGLONG)
	      ||(Arg2->Type->Type == SIR_TYPE_ULONGLONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_ULONGLONG);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_LONGLONG)
	      ||(Arg2->Type->Type == SIR_TYPE_LONGLONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONGLONG);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_ULONG)
	      ||(Arg2->Type->Type == SIR_TYPE_ULONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_ULONG);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_LONG)
	      ||(Arg2->Type->Type == SIR_TYPE_LONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONG);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_UINT)
	      ||(Arg2->Type->Type == SIR_TYPE_UINT))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_UINT);
	       break;
	      } /* fi */
	  ResultType = TypeTable->FindOrInsert(SIR_TYPE_INT);
	  break;
	 }
     case SIR_EXPR_MODULO:
	{ SIR_TYPETYPE	Arg1Type,
			Arg2Type,
			ResType;
	  int		Arg1Len,
			Arg2Len,
			ResLen;

	  if (  (Arg1->CheckReadAccess())
	      ||(Arg2->CheckReadAccess()))
	     { return(NULL);
	      } /* fi */
	  if (  (  (Arg1->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		 &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR))
	      ||(  (Arg2->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		 &&(Arg2->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR)))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_MODULO_OP;
	       return(NULL);
	      } /* fi */
	  if (  (Arg1->Type->TypeClass() == SIR_TYPECLASS_BITVECTOR)
	      ||(Arg2->Type->TypeClass() == SIR_TYPECLASS_BITVECTOR))
	     { if (! SIR_IsConvertable2BitVector(Arg1->Type,
						&Arg1Type, &Arg1Len))
		  { assert(false);	/* must be convertable */
		   } /* fi */
	       if (! SIR_IsConvertable2BitVector(Arg2->Type,
						&Arg2Type, &Arg2Len))
		  { assert(false);	/* must be convertable */
		   } /* fi */
	       ResType = (_BITUSGN_BINARY(Arg1Len, (Arg1Type == SIR_TYPE_UBIT),
					Arg2Len, (Arg2Type == SIR_TYPE_UBIT))
					? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	       ResLen = _BITLEN_BINARY(Arg1Len, (Arg1Type == SIR_TYPE_UBIT),
					Arg2Len, (Arg2Type == SIR_TYPE_UBIT));
	       ResultType = TypeTable->FindOrInsert(ResType, false, false,
							ResLen-1, 0);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_ULONGLONG)
	      ||(Arg2->Type->Type == SIR_TYPE_ULONGLONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_ULONGLONG);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_LONGLONG)
	      ||(Arg2->Type->Type == SIR_TYPE_LONGLONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONGLONG);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_ULONG)
	      ||(Arg2->Type->Type == SIR_TYPE_ULONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_ULONG);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_LONG)
	      ||(Arg2->Type->Type == SIR_TYPE_LONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONG);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_UINT)
	      ||(Arg2->Type->Type == SIR_TYPE_UINT))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_UINT);
	       break;
	      } /* fi */
	  ResultType = TypeTable->FindOrInsert(SIR_TYPE_INT);
	  break;
	 }
     case SIR_EXPR_SHIFT_LEFT:
     case SIR_EXPR_SHIFT_RIGHT:
	{ SIR_TYPETYPE	Arg1Type,
			ResType;
	  int		Arg1Len,
			ResLen;

	  if (  (Arg1->CheckReadAccess())
	      ||(Arg2->CheckReadAccess()))
	     { return(NULL);
	      } /* fi */
	  if (  (  (Arg1->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		 &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR))
	      ||(  (Arg2->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		 &&(Arg2->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR)))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_SHIFT_OP;
	       return(NULL);
	      } /* fi */
	  /* result type is promoted type of left argument */
	  if (Arg1->Type->TypeClass() == SIR_TYPECLASS_BITVECTOR)
	     { if (! SIR_IsConvertable2BitVector(Arg1->Type,
						&Arg1Type, &Arg1Len))
		  { assert(false);	/* must be convertable */
		   } /* fi */
	       ResType = (_BITUSGN_UNARY(Arg1Len, (Arg1Type == SIR_TYPE_UBIT))
					? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	       ResLen = _BITLEN_UNARY(Arg1Len, (Arg1Type == SIR_TYPE_UBIT));
	       ResultType = TypeTable->FindOrInsert(ResType, false, false,
							ResLen-1, 0);
	       break;
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_ULONGLONG)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_ULONGLONG);
	       break;
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_LONGLONG)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONGLONG);
	       break;
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_ULONG)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_ULONG);
	       break;
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_LONG)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONG);
	       break;
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_UINT)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_UINT);
	       break;
	      } /* fi */
	  ResultType = TypeTable->FindOrInsert(SIR_TYPE_INT);
	  break;
	 }
     case SIR_EXPR_LESS:
     case SIR_EXPR_GREATER:
     case SIR_EXPR_LESS_EQUAL:
     case SIR_EXPR_GREATER_EQUAL:
     case SIR_EXPR_EQUAL:
     case SIR_EXPR_NOT_EQUAL:
	{ if (  (Arg1->CheckReadAccess())
	      ||(Arg2->CheckReadAccess()))
	     { return(NULL);
	      } /* fi */
	  if (  (  (Arg1->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		 &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_FLOATING)
		 &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR)
		 &&(Arg1->Type->Type != SIR_TYPE_POINTER)
		 &&(Arg1->Type->Type != SIR_TYPE_ARRAY)
		 &&(Arg1->Type->Type != SIR_TYPE_METHOD)
		 &&(Arg1->Type->Type != SIR_TYPE_FUNCTION))
	      ||(  (Arg2->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		 &&(Arg2->Type->TypeClass() != SIR_TYPECLASS_FLOATING)
		 &&(Arg2->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR)
		 &&(Arg2->Type->Type != SIR_TYPE_POINTER)
		 &&(Arg2->Type->Type != SIR_TYPE_ARRAY)
		 &&(Arg2->Type->Type != SIR_TYPE_METHOD)
		 &&(Arg2->Type->Type != SIR_TYPE_FUNCTION)))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_COMP_OP;
	       return(NULL);
	      } /* fi */
	  ResultType = TypeTable->FindOrInsert(SIR_TYPE_BOOL);
	  break;
	 }
     case SIR_EXPR_AND:
     case SIR_EXPR_EOR:
     case SIR_EXPR_OR:
	{ SIR_TYPETYPE	Arg1Type,
			Arg2Type,
			ResType;
	  int		Arg1Len,
			Arg2Len,
			ResLen;

	  if (  (Arg1->CheckReadAccess())
	      ||(Arg2->CheckReadAccess()))
	     { return(NULL);
	      } /* fi */
	  if (  (  (Arg1->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		 &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR))
	      ||(  (Arg2->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		 &&(Arg2->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR)))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_BIT_OP;
	       return(NULL);
	      } /* fi */
	  if (  (Arg1->Type->TypeClass() == SIR_TYPECLASS_BITVECTOR)
	      ||(Arg2->Type->TypeClass() == SIR_TYPECLASS_BITVECTOR))
	     { if (! SIR_IsConvertable2BitVector(Arg1->Type,
						&Arg1Type, &Arg1Len))
		  { assert(false);	/* must be convertable */
		   } /* fi */
	       if (! SIR_IsConvertable2BitVector(Arg2->Type,
						&Arg2Type, &Arg2Len))
		  { assert(false);	/* must be convertable */
		   } /* fi */
	       ResType = (_BITUSGN_BINARY(Arg1Len, (Arg1Type == SIR_TYPE_UBIT),
					Arg2Len, (Arg2Type == SIR_TYPE_UBIT))
					? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	       ResLen = _BITLEN_BINARY(Arg1Len, (Arg1Type == SIR_TYPE_UBIT),
					Arg2Len, (Arg2Type == SIR_TYPE_UBIT));
	       ResultType = TypeTable->FindOrInsert(ResType, false, false,
							ResLen-1, 0);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_ULONGLONG)
	      ||(Arg2->Type->Type == SIR_TYPE_ULONGLONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_ULONGLONG);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_LONGLONG)
	      ||(Arg2->Type->Type == SIR_TYPE_LONGLONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONGLONG);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_ULONG)
	      ||(Arg2->Type->Type == SIR_TYPE_ULONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_ULONG);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_LONG)
	      ||(Arg2->Type->Type == SIR_TYPE_LONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONG);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_UINT)
	      ||(Arg2->Type->Type == SIR_TYPE_UINT))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_UINT);
	       break;
	      } /* fi */
	  ResultType = TypeTable->FindOrInsert(SIR_TYPE_INT);
	  break;
	 }
     case SIR_EXPR_LOGICAL_AND:
     case SIR_EXPR_LOGICAL_OR:
	{ if (  (Arg1->CheckReadAccess())
	      ||(Arg2->CheckReadAccess()))
	     { return(NULL);
	      } /* fi */
	  if (  (! SIR_IsConvertable2Bool(Arg1->Type))
	      ||(! SIR_IsConvertable2Bool(Arg2->Type)))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_BOOL_OP;
	       return(NULL);
	      } /* fi */
	  ResultType = TypeTable->FindOrInsert(SIR_TYPE_BOOL);
	  break;
	 }
     case SIR_EXPR_CONDITION:
	{ SIR_TYPETYPE	Arg2Type,
			Arg3Type,
			ResType;
	  int		Arg2Len,
			Arg3Len,
			ResLen;

	  if (  (Arg1->CheckReadAccess())
	      ||(Arg2->CheckReadAccess())
	      ||(Arg3->CheckReadAccess()))
	     { return(NULL);
	      } /* fi */
	  if (! SIR_IsConvertable2Bool(Arg1->Type))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_BOOL_OP;
	       return(NULL);
	      } /* fi */
	  /* check for short-cut, if Arg2 and Arg3 are of the same type */
	  if (Arg2->Type == Arg3->Type)	/* types identical? */
	     { ResultType = Arg2->Type;
	       break;
	      } /* fi */
	  // bug fix 04/24/06, RD:
	  // pointer arguments may also be selected
	  if (  (  (Arg2->Type->Type == SIR_TYPE_POINTER)
		 ||(Arg2->Type->Type == SIR_TYPE_ARRAY))
	      &&(  (Arg3->Type->Type == SIR_TYPE_POINTER)
		 ||(Arg3->Type->Type == SIR_TYPE_ARRAY))
	      &&(0 == SIR_Types::TypeCmp(Arg2->Type->SubType, Arg3->Type->SubType, true)))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_POINTER,
				TypeTable->FindOrInsertModifiedType(Arg2->Type->SubType));
	       break;
	      } /* fi */
	  /* otherwise, we apply the usual conversion rules */
	  if (  (  (Arg2->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		 &&(Arg2->Type->TypeClass() != SIR_TYPECLASS_FLOATING)
		 &&(Arg2->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR))
	      ||(  (Arg3->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		 &&(Arg3->Type->TypeClass() != SIR_TYPECLASS_FLOATING)
		 &&(Arg3->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR)))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_COND_EXPR;
	       return(NULL);
	      } /* fi */
	  if (  (Arg2->Type->Type == SIR_TYPE_LONGDOUBLE)
	      ||(Arg3->Type->Type == SIR_TYPE_LONGDOUBLE))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONGDOUBLE);
	       break;
	      } /* fi */
	  if (  (Arg2->Type->Type == SIR_TYPE_DOUBLE)
	      ||(Arg3->Type->Type == SIR_TYPE_DOUBLE)
	      ||(Arg2->Type->Type == SIR_TYPE_FLOAT)
	      ||(Arg3->Type->Type == SIR_TYPE_FLOAT))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_DOUBLE);
	       break;
	      } /* fi */
	  if (  (Arg2->Type->TypeClass() == SIR_TYPECLASS_BITVECTOR)
	      ||(Arg3->Type->TypeClass() == SIR_TYPECLASS_BITVECTOR))
	     { if (! SIR_IsConvertable2BitVector(Arg2->Type,
						&Arg2Type, &Arg2Len))
		  { assert(false);	/* must be convertable */
		   } /* fi */
	       if (! SIR_IsConvertable2BitVector(Arg3->Type,
						&Arg3Type, &Arg3Len))
		  { assert(false);	/* must be convertable */
		   } /* fi */
	       ResType = (_BITUSGN_BINARY(Arg2Len, (Arg2Type == SIR_TYPE_UBIT),
					Arg3Len, (Arg3Type == SIR_TYPE_UBIT))
					? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	       ResLen = _BITLEN_BINARY(Arg2Len, (Arg2Type == SIR_TYPE_UBIT),
					Arg3Len, (Arg3Type == SIR_TYPE_UBIT));
	       ResultType = TypeTable->FindOrInsert(ResType, false, false,
							ResLen-1, 0);
	       break;
	      } /* fi */
	  if (  (Arg2->Type->Type == SIR_TYPE_ULONGLONG)
	      ||(Arg3->Type->Type == SIR_TYPE_ULONGLONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_ULONGLONG);
	       break;
	      } /* fi */
	  if (  (Arg2->Type->Type == SIR_TYPE_LONGLONG)
	      ||(Arg3->Type->Type == SIR_TYPE_LONGLONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONGLONG);
	       break;
	      } /* fi */
	  if (  (Arg2->Type->Type == SIR_TYPE_ULONG)
	      ||(Arg3->Type->Type == SIR_TYPE_ULONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_ULONG);
	       break;
	      } /* fi */
	  if (  (Arg2->Type->Type == SIR_TYPE_LONG)
	      ||(Arg3->Type->Type == SIR_TYPE_LONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONG);
	       break;
	      } /* fi */
	  if (  (Arg2->Type->Type == SIR_TYPE_UINT)
	      ||(Arg3->Type->Type == SIR_TYPE_UINT))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_UINT);
	       break;
	      } /* fi */
	  ResultType = TypeTable->FindOrInsert(SIR_TYPE_INT);
	  break;
	 }
     case SIR_EXPR_ASSIGNMENT:
	{ if (  (Arg1->CheckWriteAccess())
	      ||(Arg2->CheckReadAccess()))
	     { return(NULL);
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_ARRAY)	/* array assignment? */
	      &&(Arg2->Type->Type == SIR_TYPE_ARRAY))
	     { if (  (Arg1->Type->Size <= 0)
		   ||(Arg2->Type->Size <= 0)
		   ||(Arg1->Type->Size != Arg2->Type->Size)
		   ||(Arg1->Type->SubType->Modified() !=
					Arg2->Type->SubType->Modified()))
		  { SIR_Error = SIR_ERROR_INVALID_ARRAY_ASSIGNMENT;
		    return(NULL);
		   } /* fi */
	       ResultType = TypeTable->FindOrInsert(SIR_TYPE_VOID);
	       break;
	      } /* fi */
	  if (! Arg2->Type->IsAssignableTo(Arg1->Type))
	     { SIR_Error = SIR_ERROR_INVALID_TYPE_FOR_ASSIGN_OP;
	       return(NULL);
	      } /* fi */
	  ResultType = Arg1->Type;	/* (not promoted!) */
	  break;
	 }
     case SIR_EXPR_MUL_ASSIGN:
     case SIR_EXPR_DIV_ASSIGN:
	{ sir_type	*TmpResultType;

	  if (  (Arg1->CheckWriteAccess())
	      ||(Arg1->CheckReadAccess())
	      ||(Arg2->CheckReadAccess()))
	     { return(NULL);
	      } /* fi */
	  if (  (  (Arg1->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		 &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_FLOATING)
		 &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR))
	      ||(  (Arg2->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		 &&(Arg2->Type->TypeClass() != SIR_TYPECLASS_FLOATING)
		 &&(Arg2->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR)))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_ARITH_OP;
	       return(NULL);
	      } /* fi */
	  TmpResultType = Arg2->Type;	/* sufficiently accurate here */
	  if (! TmpResultType->IsAssignableTo(Arg1->Type))
	     { SIR_Error = SIR_ERROR_INVALID_TYPE_FOR_ASSIGN_OP;
	       return(NULL);
	      } /* fi */
	  ResultType = Arg1->Type;	/* (not promoted!) */
	  break;
	 }
     case SIR_EXPR_MOD_ASSIGN:
	{ sir_type	*TmpResultType;

	  if (  (Arg1->CheckWriteAccess())
	      ||(Arg1->CheckReadAccess())
	      ||(Arg2->CheckReadAccess()))
	     { return(NULL);
	      } /* fi */
	  if (  (  (Arg1->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		 &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR))
	      ||(  (Arg2->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		 &&(Arg2->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR)))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_MODULO_OP;
	       return(NULL);
	      } /* fi */
	  TmpResultType = Arg2->Type;	/* sufficiently accurate here */
	  if (! TmpResultType->IsAssignableTo(Arg1->Type))
	     { SIR_Error = SIR_ERROR_INVALID_TYPE_FOR_ASSIGN_OP;
	       return(NULL);
	      } /* fi */
	  ResultType = Arg1->Type;	/* (not promoted!) */
	  break;
	 }
     case SIR_EXPR_ADD_ASSIGN:
     case SIR_EXPR_SUB_ASSIGN:
	{ sir_type	*TmpResultType;

	  if (  (Arg1->CheckWriteAccess())
	      ||(Arg1->CheckReadAccess())
	      ||(Arg2->CheckReadAccess()))
	     { return(NULL);
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_POINTER)
	      ||(Arg1->Type->Type == SIR_TYPE_ARRAY))
	     { if (  (Arg2->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		   &&(Arg2->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR))
		  { SIR_Error = SIR_ERROR_INVALID_POINTER_ARITHMETIC;
		    return(NULL);
		   } /* fi */
	      } /* fi */
	  else if (  (Arg2->Type->Type == SIR_TYPE_POINTER)
		   ||(Arg2->Type->Type == SIR_TYPE_ARRAY))
		  { if (  (Arg1->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
			&&(Arg1->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR))
		       { SIR_Error = SIR_ERROR_INVALID_POINTER_ARITHMETIC;
			 return(NULL);
			} /* fi */
		   } /* fi */
	  else if (  (  (Arg1->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		      &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_FLOATING)
		      &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR))
		   ||(  (Arg2->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		      &&(Arg2->Type->TypeClass() != SIR_TYPECLASS_FLOATING)
		      &&(Arg2->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR)))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_ARITH_OP;
	       return(NULL);
	      } /* fi esle esle */
	  TmpResultType = Arg2->Type;	/* sufficiently accurate here */
	  if (! TmpResultType->IsAssignableTo(Arg1->Type))
	     { SIR_Error = SIR_ERROR_INVALID_TYPE_FOR_ASSIGN_OP;
	       return(NULL);
	      } /* fi */
	  ResultType = Arg1->Type;	/* (not promoted!) */
	  break;
	 }
     case SIR_EXPR_SHL_ASSIGN:
     case SIR_EXPR_SHR_ASSIGN:
	{ sir_type	*TmpResultType;

	  if (  (Arg1->CheckWriteAccess())
	      ||(Arg1->CheckReadAccess())
	      ||(Arg2->CheckReadAccess()))
	     { return(NULL);
	      } /* fi */
	  if (  (  (Arg1->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		 &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR))
	      ||(  (Arg2->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		 &&(Arg2->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR)))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_SHIFT_OP;
	       return(NULL);
	      } /* fi */
	  TmpResultType = Arg1->Type;	/* sufficiently accurate here */
	  if (! TmpResultType->IsAssignableTo(Arg1->Type))
	     { SIR_Error = SIR_ERROR_INVALID_TYPE_FOR_ASSIGN_OP;
	       return(NULL);
	      } /* fi */
	  ResultType = Arg1->Type;	/* (not promoted!) */
	  break;
	 }
     case SIR_EXPR_AND_ASSIGN:
     case SIR_EXPR_EOR_ASSIGN:
     case SIR_EXPR_OR_ASSIGN:
	{ sir_type	*TmpResultType;

	  if (  (Arg1->CheckWriteAccess())
	      ||(Arg1->CheckReadAccess())
	      ||(Arg2->CheckReadAccess()))
	     { return(NULL);
	      } /* fi */
	  if (  (  (Arg1->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		 &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR))
	      ||(  (Arg2->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		 &&(Arg2->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR)))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_BIT_OP;
	       return(NULL);
	      } /* fi */
	  TmpResultType = Arg2->Type;	/* sufficiently accurate here */
	  if (! TmpResultType->IsAssignableTo(Arg1->Type))
	     { SIR_Error = SIR_ERROR_INVALID_TYPE_FOR_ASSIGN_OP;
	       return(NULL);
	      } /* fi */
	  ResultType = Arg1->Type;	/* (not promoted!) */
	  break;
	 }
     case SIR_EXPR_COMMA:
	{ ResultType = Arg2->Type;	/* (not promoted!) */
	  break;
	 }
     default:
	{ assert(false);	/* bad expression type */
	 }
    } /* hctiws */

assert(ResultType != NULL);	/* must have been determined */

NewExpr = new SIR_Expression(ExprType, ResultType, Arg1, Arg2, Arg3);

return(NewExpr);

} /* end of SIR_Expression::New #3 */


sir_expression *SIR_Expression::New(	/* creates a function call expr. (#4) */
	sir_expression	* /*New*/ Arg1,	/* (returns NULL if SIR_Error) */
	sir_expressions	* /*New*/ Args)
{
sir_types	*TypeTable;
sir_expression	*NewExpr;
sir_type	*FctType,
		*ResultType;
sir_type_ptr	*ExpectedArg;
sir_expression	*SuppliedArg;

//PC Check with this fix
assert(Arg1 != NULL);
assert(Args != NULL);

FctType = Arg1->Type;
assert(FctType != NULL);
TypeTable = FctType->GetTable();
assert(TypeTable != NULL);
if (  (FctType->Type != SIR_TYPE_FUNCTION)
    &&(FctType->Type != SIR_TYPE_METHOD))
   { if (  (FctType->Type == SIR_TYPE_POINTER)
	 &&(  (FctType->SubType->Type == SIR_TYPE_FUNCTION)
	    ||(FctType->SubType->Type == SIR_TYPE_METHOD)))
	{ FctType = FctType->SubType;	/* bug fix 10/28/03, RD:      */
					/* pointer to function can be */
					/* used without dereferencing */
					/* to call that function      */
	 } /* fi */
     else
	{ SIR_Error = SIR_ERROR_CALL_OF_NON_FUNCTION;
	  return(NULL);
	 } /* esle */
    } /* fi */

if (FctType->VoidParameters())
   { if (Args->NumElements() > 0)
	{ SIR_Error = SIR_ERROR_TOO_MANY_ARGUMENTS;
	  return(NULL);
	 } /* fi */
    } /* fi */
else
   { ExpectedArg = FctType->Parameters->First();
     SuppliedArg = Args->First();
     while(ExpectedArg)
	{ if (ExpectedArg->Type->Type == SIR_TYPE_ANY_TYPE)
	     { while(SuppliedArg)
		  { if (SuppliedArg->CheckReadAccess())
		       { return(NULL);
			} /* fi */
		    SuppliedArg = SuppliedArg->Succ();
		   } /* elihw */
	       break;	/* stop checking */
	      } /* fi */
	  if (!(SuppliedArg))
	     { SIR_Error = SIR_ERROR_TOO_FEW_ARGUMENTS;
	       return(NULL);
	      } /* fi */
	  if (SuppliedArg->CheckReadAccess())
	     { return(NULL);
	      } /* fi */
	  if (!(SuppliedArg->Type->IsConvertableTo(
					ExpectedArg->Type)))
	     { SIR_Error = SIR_ERROR_ARGUMENT_TYPE_MISMATCH;
	       return(NULL);
	      } /* fi */
	  ExpectedArg = ExpectedArg->Succ();
	  SuppliedArg = SuppliedArg->Succ();
	 } /* elihw */
     if (SuppliedArg)
	{ SIR_Error = SIR_ERROR_TOO_MANY_ARGUMENTS;
	  return(NULL);
	 } /* fi */
    } /* esle */

ResultType = FctType->SubType;

NewExpr = new SIR_Expression(SIR_EXPR_FUNCTION_CALL, ResultType, Arg1, Args);

return(NewExpr);

} /* end of SIR_Expression::New #4 */


sir_expression *SIR_Expression::New(	/* creates a member access expr. (#5) */
	SIR_EXPRTYPE	ExprType,	/* (returns NULL if SIR_Error) */
	sir_expression	* /*New*/ Arg1,
	const char	*MemberName)
{
sir_types	*TypeTable;
sir_expression	*NewExpr;
sir_type	*TypeEntry,
		*ResultType,
		*VoidType;
sir_symbol	*Member;
sir_symbol_ptr	*Interface;

assert(Arg1 != NULL);
assert(Arg1->Type != NULL);
TypeTable = Arg1->Type->GetTable();
assert(TypeTable != NULL);

TypeEntry = NULL;	/* keep compiler quiet */
switch(ExprType)
   { case SIR_EXPR_MEMBER_ACCESS:
	{ TypeEntry = Arg1->Type;
	  break;
	 } /* esac */
     case SIR_EXPR_MEMBER_POINTER:
	{ if (Arg1->Type->Type != SIR_TYPE_POINTER)
	     { SIR_Error = SIR_ERROR_LEFT_ARG_NOT_A_POINTER;
	       return(NULL);
	      } /* fi */
	  TypeEntry = Arg1->Type->SubType;
	  break;
	 } /* esac */
     default:
	{ assert(false);	/* wrong expression type */
	 } /* tluafed */
    } /* hctiws */

assert(TypeEntry != NULL);
switch(TypeEntry->Type)
   { case SIR_TYPE_STRUCT:	/* structs/unions: every member is public */
     case SIR_TYPE_UNION:
	{ assert(TypeEntry->UserType != NULL);	/* must exist */
	  if (! TypeEntry->UserType->Scope)
	     { SIR_Error = SIR_ERROR_ACCESS_TO_INCOMPLETE_TYPE;
	       return(NULL);
	      } /* fi */
	  if (!(Member = TypeEntry->UserType->Scope->FindLocally(MemberName)))
	     { SIR_ErrMsg.form(
			"There is no member '%s' in this aggregate type",
			MemberName);
	       SIR_Error = SIR_ERROR_NO_SUCH_MEMBER_IN_USERTYPE;
	       return(NULL);
	      } /* fi */
	  ResultType = TypeTable->FindOrInsertModifiedType(
				Member->Type,
				Member->Type->Const,
				Member->Type->Volatile,
				/* propagate direction to result type */
				TypeEntry->Direction);
	  break;
	 } /* esac */
     case SIR_TYPE_BEHAVIOR:	/* main method and impl. interf. are public */
	{ if (0 == strcmp(MemberName, GL_MAIN_METHOD_NAME))
	     { Member = NULL;	/* means: "main" */
	       VoidType = TypeTable->FindOrInsert(SIR_TYPE_VOID);
	       assert(TypeEntry->ClassSymbol != NULL);
	       ResultType = TypeTable->FindOrInsert(
				SIR_TYPE_METHOD,
				VoidType,
				new SIR_TypePtrs(
					new SIR_TypePtr(VoidType)),
				0, false, false, SIR_PORT_NONE,
				TypeEntry->ClassSymbol);
	       break;
	      } /* fi */
	/* note: while in the following code the list of implemented	*/
	/* interfaces is searched in left-to-right order, the order	*/
	/* actually doesn't matter; the only thing that counts is	*/
	/* that we find one member (which must be suitable);		*/
	/* later, during modifications to implemented interfaces or	*/
	/* their members, the same search could yield a different	*/
	/* (but compatible) member (R.D., Dec 12, 98)			*/
	  assert(TypeEntry->ClassSymbol != NULL);	/* must exist */
	  assert(TypeEntry->ClassSymbol->Interfaces != NULL);
	  Interface = TypeEntry->ClassSymbol->Interfaces->First();
	  Member = NULL;
	  while(Interface)
	     { assert(Interface->Symbol != NULL);
	       if (! Interface->Symbol->IsInterfaceDefinition())
		  { SIR_ErrMsg.form(
			"Access to incomplete interface '%s'"
							GL_ERROR_MSG_NEWLINE
			"of behavior '%s'",
				Interface->Symbol->Name.chars(),
				TypeEntry->ClassSymbol->Name.chars());
		    SIR_Error = SIR_ERROR_ACCESS_TO_INCOMPLETE_IF_1;
		    return(NULL);
		   } /* fi */
	       if ((Member = Interface->Symbol->ClassScope->FindLocally(
								MemberName)))
		  { break;
		   } /* fi */
	       Interface = Interface->Succ();
	      } /* elihw */
	  if (Member == NULL)
	     { SIR_ErrMsg.form(
		"There is no method '%s' in the list" GL_ERROR_MSG_NEWLINE
		"of implemented interfaces for behavior '%s'",
			MemberName, TypeEntry->ClassSymbol->Name.chars());
	       SIR_Error = SIR_ERROR_NO_SUCH_METHOD_IMPLEMENTED_1;
	       return(NULL);
	      } /* fi */
	  ResultType = Member->Type;
	  break;
	 } /* esac */
     case SIR_TYPE_CHANNEL:	/* members of impl. interfaces are public */
	{ assert(TypeEntry->ClassSymbol != NULL);	/* must exist */
	  assert(TypeEntry->ClassSymbol->Interfaces != NULL);
	/* note: (see above) */
	  Interface = TypeEntry->ClassSymbol->Interfaces->First();
	  Member = NULL;
	  while(Interface)
	     { assert(Interface->Symbol != NULL);
	       if (! Interface->Symbol->IsInterfaceDefinition())
		  { SIR_ErrMsg.form(
			"Access to incomplete interface '%s'"
							GL_ERROR_MSG_NEWLINE
			"of channel '%s'",
				Interface->Symbol->Name.chars(),
				TypeEntry->ClassSymbol->Name.chars());
		    SIR_Error = SIR_ERROR_ACCESS_TO_INCOMPLETE_IF_2;
		    return(NULL);
		   } /* fi */
	       if ((Member = Interface->Symbol->ClassScope->FindLocally(
								MemberName)))
		  { break;
		   } /* fi */
	       Interface = Interface->Succ();
	      } /* elihw */
	  if (Member == NULL)
	     { SIR_ErrMsg.form(
		"There is no method '%s' in the list" GL_ERROR_MSG_NEWLINE
		"of implemented interfaces for channel '%s'",
			MemberName, TypeEntry->ClassSymbol->Name.chars());
	       SIR_Error = SIR_ERROR_NO_SUCH_METHOD_IMPLEMENTED_2;
	       return(NULL);
	      } /* fi */
	  ResultType = Member->Type;
	  break;
	 } /* esac */
     case SIR_TYPE_INTERFACE:	/* interface: every method decl. is public */
	{ assert(TypeEntry->ClassSymbol != NULL);	/* must exist */
	  if (! TypeEntry->ClassSymbol->ClassScope)
	     { SIR_ErrMsg.form("Access to incomplete interface '%s'",
				TypeEntry->ClassSymbol->Name.chars());
	       SIR_Error = SIR_ERROR_ACCESS_TO_INCOMPLETE_INTERFC;
	       return(NULL);
	      } /* fi */
	  if (!(Member = TypeEntry->ClassSymbol->ClassScope->FindLocally(
								MemberName)))
	     { SIR_ErrMsg.form(
		"There is no method '%s' in interface '%s'",
			MemberName, TypeEntry->ClassSymbol->Name.chars());
	       SIR_Error = SIR_ERROR_NO_SUCH_METHOD_IN_INTERFACE;
	       return(NULL);
	      } /* fi */
	  ResultType = Member->Type;
	  break;
	 } /* esac */
     default:
	{ SIR_Error = SIR_ERROR_NOT_AN_AGGREGATE_OR_CLASS;
	  return(NULL);
	 } /* tluafed */
    } /* hctiws */

assert(ResultType != NULL);	/* must have been found */

NewExpr = new SIR_Expression(ExprType, ResultType, Arg1, Member);

return(NewExpr);

} /* end of SIR_Expression::New #5 */


sir_expression *SIR_Expression::New(	/* creates sizeof(type)/cast expr.(#6)*/
	SIR_EXPRTYPE	ExprType,	/* (returns NULL if SIR_Error) */
	sir_type	*TypeArg,
	sir_expression	* /*New*/ Arg1 /* = NULL */)
{
sir_types	*TypeTable;
sir_expression	*NewExpr;
sir_type	*ResultType;

assert(  (ExprType == SIR_EXPR_SIZEOF_TYPE)
       ||(ExprType == SIR_EXPR_TYPE_CONVERSION));
assert(TypeArg != NULL);
TypeTable = TypeArg->GetTable();
assert(TypeTable != NULL);

if (ExprType == SIR_EXPR_SIZEOF_TYPE)
   { switch(TypeArg->Type)
	{ case SIR_TYPE_VOID:
	     { SIR_Error = SIR_ERROR_ILLEGAL_SIZEOF_VOID;
	       return(NULL);
	      }
	  case SIR_TYPE_EVENT:
	     { SIR_Error = SIR_ERROR_ILLEGAL_SIZEOF_EVENT;
	       return(NULL);
	      }
	  case SIR_TYPE_METHOD:
	  case SIR_TYPE_FUNCTION:
	     { SIR_Error = SIR_ERROR_ILLEGAL_SIZEOF_FUNCTION;
	       return(NULL);
	      }
	  case SIR_TYPE_ANY_TYPE:
	  case SIR_TYPE_BEHAVIOR:
	  case SIR_TYPE_CHANNEL:
	  case SIR_TYPE_INTERFACE:
	     { SIR_Error = SIR_ERROR_ILLEGAL_SIZEOF_CLASS;
	       return(NULL);
	      }
	  case SIR_TYPE_SIGNAL:
	     { SIR_Error = SIR_ERROR_ILLEGAL_SIZEOF_SIGNAL;
	       return(NULL);
	      }
	  case SIR_TYPE_BUFFER:
	     { SIR_Error = SIR_ERROR_ILLEGAL_SIZEOF_BUFFER;
	       return(NULL);
	      }
	  default:	/* any other type */
	     { /* type is legal for sizeof() operator */
	       break;
	      }
	 } /* hctiws */
     ResultType = TypeTable->FindOrInsert(SIR_TYPE_UINT);
    } /* fi */
else	/* ExprType == SIR_EXPR_TYPE_CONVERSION */
   { 
     /* note: it is the users responsibility to make sure that */
     /* explicit type casting makes sense; therefore, we will  */
     /* only ensure that no totally crazy casts are performed! */
     if (  (TypeArg->Type == SIR_TYPE_EVENT)
	 ||(TypeArg->Type == SIR_TYPE_METHOD)
	 ||(TypeArg->Type == SIR_TYPE_FUNCTION)
	 ||(TypeArg->Type == SIR_TYPE_ANY_TYPE)
	 ||(TypeArg->Type == SIR_TYPE_BEHAVIOR)
	 ||(TypeArg->Type == SIR_TYPE_CHANNEL)
	 ||(TypeArg->Type == SIR_TYPE_INTERFACE)
	 ||(TypeArg->Type == SIR_TYPE_SIGNAL)
	 ||(TypeArg->Type == SIR_TYPE_BUFFER))
	{ SIR_Error = SIR_ERROR_INVALID_TYPE_CAST;
	  return(NULL);
	 } /* fi */
     ResultType = TypeArg;
    } /* esle */

NewExpr = new SIR_Expression(ExprType, ResultType, TypeArg, Arg1);

return(NewExpr);

} /* end of SIR_Expression::New #6 */


sir_expression *SIR_Expression::New(	/* creates a bit slice expr. (#7) */
	sir_expression	* /*New*/ Arg1,	/* (returns NULL if SIR_Error) */
	int		LeftBound,
	int		RightBound)
{
SIR_TYPETYPE	ArgTypeType,
		ResultTypeType;
int		ArgLeftBound,
		ArgRightBound;
int		ArgBitlen,
		ResultBitlen;
sir_types	*TypeTable;
sir_expression	*NewExpr;
sir_type	*ResultType;

assert(Arg1 != NULL);
assert(Arg1->Type != NULL);
TypeTable = Arg1->Type->GetTable();
assert(TypeTable != NULL);

if (! SIR_IsConvertable2BitVector(Arg1->Type,
		&ArgTypeType, &ArgBitlen, &ArgLeftBound, &ArgRightBound))
   { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_BITSLICE;
     return(NULL);
    } /* fi */

if (  (  (ArgLeftBound >= ArgRightBound)
       &&(  (LeftBound > ArgLeftBound)  || (RightBound > ArgLeftBound)
	  ||(LeftBound < ArgRightBound) || (RightBound < ArgRightBound)))
    ||(  (ArgLeftBound <= ArgRightBound)
       &&(  (LeftBound < ArgLeftBound)  || (RightBound < ArgLeftBound)
	  ||(LeftBound > ArgRightBound) || (RightBound > ArgRightBound))))
   { SIR_Error = SIR_ERROR_INVALID_SLICE_FOR_BITSLICE;
     SIR_ErrMsg.form("Invalid slice [%d:%d] for argument type bit[%d:%d]",
			LeftBound, RightBound, ArgLeftBound, ArgRightBound);
     return(NULL);
    } /* fi */

ResultTypeType = (_BITUSGN_SLICE(ArgBitlen, (ArgTypeType == SIR_TYPE_UBIT),
					LeftBound, RightBound) ?
					SIR_TYPE_UBIT : SIR_TYPE_BIT);
ResultBitlen = _BITLEN_SLICE(ArgBitlen, (ArgTypeType == SIR_TYPE_UBIT),
					LeftBound, RightBound);
ResultType = TypeTable->FindOrInsert(ResultTypeType,
					Arg1->Type->Const,
					Arg1->Type->Volatile,
					ResultBitlen-1, 0,
					/* propagate direction to result type */
					Arg1->Type->Direction);

NewExpr = new SIR_Expression(SIR_EXPR_BITSLICE, ResultType,
				Arg1, LeftBound, RightBound);

return(NewExpr);

} /* end of SIR_Expression::New #7 */

sir_expression *SIR_Expression::Copy(   /* creates a copy of the tree */
        bool            Strip /* = true */)     /* (NULL if SIR_Error) */
{
sir_expression  *Expr;

Expr = new SIR_Expression(this);

return(Expr);

} /* end of SIR_Expression::Copy */


void SIR_Expression::Delete(void)       /* deletes the expression tree */
{

delete this;

} /* end of SIR_Expression::Delete */


const char *SIR_Expression::Print(      /* return text string of this expr. */
        bool            CppNotation /* = false */,      /* default: SpecC */
        unsigned int    LineWrap /*=GL_LINE_WRAP_DEFAULT*/, /* 0 turns off */
        int             TabStep /*= GL_TAB_STEP_DEFAULT */, /* 0 turns off */
        int             SysTabStep /*=GL_TAB_STEP_SYSTEM*/) /* 0 turns off */
{
gl_io       *IO;
unsigned int    LeadingNL;
static gl_string   PrintBuffer;

if (!(IO = GL_IO::OpenString(&PrintBuffer, GL_IO_WRITE)))
   { SIR_Error = GL_Error;
     return(NULL);
    } /* fi */

SIR_LineInfo::InitWriteSC(false);	/* no line infos */
IO->SetLineWrap(LineWrap);
IO->SetTabStep(TabStep);
IO->SetSystemTabStep(SysTabStep);

if ((SIR_Error = WriteSC(IO, false,	/* no annotations */
			CppNotation)))
   { IO->Close();
     return(NULL);
    } /* fi */

if ((SIR_Error = IO->CloseWithCheck()))
   { return(NULL);
    } /* fi */

LeadingNL = 0;				/* no leading newlines */
assert(  (0 == strcmp(EOL, "\n"))	/* (bug fix RD, 09/15/05) */
       ||(0 == strcmp(EOL, "\r\n")));
if (0 == strcmp(EOL, "\n"))
   { while(PrintBuffer[LeadingNL] == '\n')
	{ LeadingNL++;
	 } /* elihw */
    } /* fi */
else
   { while(PrintBuffer.at((int)LeadingNL, 2) == "\r\n")
	{ LeadingNL += 2;
	 } /* elihw */
    } /* esle */
return(&PrintBuffer[LeadingNL]);

} /* end of SIR_Expression::Print */


	/***********************/
	/*** SIR_Expressions ***/
	/***********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Expressions::SIR_Expressions(	/* constructor #1 */
	sir_expression	*FirstEntry /* = NULL */) :
		SIR_List<SIR_Expression>(FirstEntry)
{

/* nothing else to do */

} /* end of SIR_Expressions::SIR_Expressions #1 */


SIR_Expressions::SIR_Expressions(	/* constructor #2 (duplicator) */
	sir_expressions	*Original)	/* (recursive!) */
{
sir_expression	*Expr;

assert(Original != NULL);

Expr = Original->First();
while(Expr)
   { Append(new SIR_Expression(Expr));
     Expr = Expr->Succ();
    } /* elihw */

} /* end of SIR_Expressions::SIR_Expressions #2 */


SIR_Expressions::~SIR_Expressions(void)	/* destructor */
{

/* nothing to do */

} /* end of SIR_Expressions::~SIR_Expressions */


ERROR SIR_Expressions::DFS_ForAllNodes(	/* iterator over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{
sir_expression	*Expr,
		*Succ;

Expr = First();
while (Expr)
   { Succ = Expr->Succ();
     if ((SIR_Error = Expr->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Expr = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Expressions::DFS_ForAllNodes */


ERROR SIR_Expressions::DFS_ForAllExpressions(	/* iterator over all exprs. */
	sir_expr_mptr	MemberFct,		/* (depth first) */
	sir_expr_marg	MemberFctArg)
{
sir_expression	*Expr,
		*Succ;

Expr = First();
while (Expr)
   { Succ = Expr->Succ();
     if ((SIR_Error = Expr->DFS_ForAllExpressions(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Expr = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Expressions::DFS_ForAllExpressions */


/*** exported functions *************************************************/


	/* none */


/*** internal functions *************************************************/


SIR_WHAT_TO_DO SIR_PrepareOperand(	/* 'ultra-promote' the constant */
	sir_expression		*Arg,	/* (fills in all possible values) */
	LONG_LONG		*TmpValueLL,
	UNSIGNED_LONG_LONG	*TmpValueULL,
	long double		*TmpValueLD,
	sir_bit			**TmpValueBIT,
	unsigned int		*Line /* = NULL */,
	sir_fileinfo		**FileInfo /* = NULL */)
{
SIR_WHAT_TO_DO	DoWhat;
sir_constant	*EvalArg;

if (!(EvalArg = Arg->Eval()))
   { return(SIR_DO_ERROR);
    } /* fi */

DoWhat = SIR_DO_ERROR;	/* to keep the compiler quiet */
switch(EvalArg->Type)
   { case SIR_CONST_BOOL:
	{ *TmpValueLL  = EvalArg->B_Value;
	  *TmpValueULL = EvalArg->B_Value;
	  *TmpValueLD  = EvalArg->B_Value;
	  *TmpValueBIT = new sir_bit(EvalArg->B_Value);
	  DoWhat = SIR_DO_LL_COMPUTATION;
	  break;
	 }
     case SIR_CONST_CHAR:
	{ *TmpValueLL  = EvalArg->C_Value;
	  *TmpValueULL = EvalArg->C_Value;
	  *TmpValueLD  = EvalArg->C_Value;
	  *TmpValueBIT = new sir_bit(EvalArg->C_Value);
	  DoWhat = SIR_DO_LL_COMPUTATION;
	  break;
	 }
     case SIR_CONST_UCHAR:
	{ *TmpValueLL  = EvalArg->UC_Value;
	  *TmpValueULL = EvalArg->UC_Value;
	  *TmpValueLD  = EvalArg->UC_Value;
	  *TmpValueBIT = new sir_bit(EvalArg->UC_Value);
	  DoWhat = SIR_DO_LL_COMPUTATION;
	  break;
	 }
     case SIR_CONST_SHORT:
	{ *TmpValueLL  = EvalArg->S_Value;
	  *TmpValueULL = EvalArg->S_Value;
	  *TmpValueLD  = EvalArg->S_Value;
	  *TmpValueBIT = new sir_bit(EvalArg->S_Value);
	  DoWhat = SIR_DO_LL_COMPUTATION;
	  break;
	 }
     case SIR_CONST_USHORT:
	{ *TmpValueLL  = EvalArg->US_Value;
	  *TmpValueULL = EvalArg->US_Value;
	  *TmpValueLD  = EvalArg->US_Value;
	  *TmpValueBIT = new sir_bit(EvalArg->US_Value);
	  DoWhat = SIR_DO_LL_COMPUTATION;
	  break;
	 }
     case SIR_CONST_INT:
	{ *TmpValueLL  = EvalArg->I_Value;
	  *TmpValueULL = EvalArg->I_Value;
	  *TmpValueLD  = EvalArg->I_Value;
	  *TmpValueBIT = new sir_bit(EvalArg->I_Value);
	  DoWhat = SIR_DO_LL_COMPUTATION;
	  break;
	 }
     case SIR_CONST_UINT:
	{ *TmpValueLL  = EvalArg->UI_Value;
	  *TmpValueULL = EvalArg->UI_Value;
	  *TmpValueLD  = EvalArg->UI_Value;
	  *TmpValueBIT = new sir_bit(EvalArg->UI_Value);
	  DoWhat = SIR_DO_LL_COMPUTATION;
	  break;
	 }
     case SIR_CONST_LONG:
	{ *TmpValueLL  = EvalArg->L_Value;
	  *TmpValueULL = EvalArg->L_Value;
	  *TmpValueLD  = EvalArg->L_Value;
	  *TmpValueBIT = new sir_bit(EvalArg->L_Value);
	  DoWhat = SIR_DO_LL_COMPUTATION;
	  break;
	 }
     case SIR_CONST_ULONG:
	{ *TmpValueLL  = EvalArg->UL_Value;
	  *TmpValueULL = EvalArg->UL_Value;
	  *TmpValueLD  = EvalArg->UL_Value;
	  *TmpValueBIT = new sir_bit(EvalArg->UL_Value);
	  DoWhat = SIR_DO_LL_COMPUTATION;
	  break;
	 }
     case SIR_CONST_LONGLONG:
	{ *TmpValueLL  = EvalArg->LL_Value;
	  *TmpValueULL = EvalArg->LL_Value;
	  *TmpValueLD  = SIR_LLONG2LD(EvalArg->LL_Value);
	  *TmpValueBIT = new sir_bit(EvalArg->LL_Value);
	  DoWhat = SIR_DO_LL_COMPUTATION;
	  break;
	 }
     case SIR_CONST_ULONGLONG:
	{ *TmpValueLL  = 0;
	  *TmpValueULL = EvalArg->ULL_Value;
	  *TmpValueLD  = SIR_LLONG2LD(EvalArg->ULL_Value);
	  *TmpValueBIT = new sir_bit(EvalArg->ULL_Value);
	  DoWhat = SIR_DO_ULL_COMPUTATION;
	  break;
	 }
     case SIR_CONST_BIT:
     case SIR_CONST_UBIT:
	{ *TmpValueLL  = 0;
	  *TmpValueULL = 0;
	  *TmpValueLD  = EvalArg->BIT_Value->toLDouble();
	  *TmpValueBIT = new sir_bit(*EvalArg->BIT_Value);
	  DoWhat = SIR_DO_BIT_COMPUTATION;
	  break;
	 }
     case SIR_CONST_FLOAT:
	{ *TmpValueLL  = 0;
	  *TmpValueULL = 0;
	  *TmpValueLD  = EvalArg->F_Value;
	  *TmpValueBIT = NULL;
	  DoWhat = SIR_DO_LD_COMPUTATION;
	  break;
	 }
     case SIR_CONST_DOUBLE:
	{ *TmpValueLL  = 0;
	  *TmpValueULL = 0;
	  *TmpValueLD  = EvalArg->D_Value;
	  *TmpValueBIT = NULL;
	  DoWhat = SIR_DO_LD_COMPUTATION;
	  break;
	 }
     case SIR_CONST_LONGDOUBLE:
	{ *TmpValueLL  = 0;
	  *TmpValueULL = 0;
	  *TmpValueLD  = EvalArg->LD_Value;
	  *TmpValueBIT = NULL;
	  DoWhat = SIR_DO_LD_COMPUTATION;
	  break;
	 }
     case SIR_CONST_CHARSTRING:
	{ SIR_Error = SIR_ERROR_EXPRESSION_NOT_CONSTANT;
	  DoWhat = SIR_DO_ERROR;
	  break;
	 }
     default:
	{ assert(false);	/* bad type */
	 }
    } /* hctiws */

if (Line)	/* if requested, pass line infos (if any) */
   { assert(FileInfo != NULL);
     if (EvalArg->LineInfo)
	{ *Line = EvalArg->LineInfo->Line;
	  *FileInfo = EvalArg->LineInfo->File;
	 } /* fi */
     else
	{ *Line = 0;
	  *FileInfo = NULL;
	 } /* fi */
    } /* fi */

delete EvalArg;

return(DoWhat);

} /* end of SIR_PrepareOperand */


SIR_WHAT_TO_DO SIR_PrepareBinaryOperands(
	sir_expression		*Arg1,
	sir_expression		*Arg2,
	LONG_LONG		*TmpValueLL1,
	UNSIGNED_LONG_LONG	*TmpValueULL1,
	long double		*TmpValueLD1,
	sir_bit			**TmpValueBIT1,
	LONG_LONG		*TmpValueLL2,
	UNSIGNED_LONG_LONG	*TmpValueULL2,
	long double		*TmpValueLD2,
	sir_bit			**TmpValueBIT2,
	unsigned int		*Line /* = NULL */,
	sir_fileinfo		**FileInfo /* = NULL */)
{
SIR_WHAT_TO_DO	DoWhat1,
		DoWhat2,
		DoWhat;

if (Line)
   { assert(FileInfo != NULL);
     if ((DoWhat1 = SIR_PrepareOperand(Arg1,
		TmpValueLL1, TmpValueULL1, TmpValueLD1, TmpValueBIT1,
						Line, FileInfo))
			== SIR_DO_ERROR)
	{ return(SIR_DO_ERROR);
	 } /* fi */
     if (*Line == 0)
	{ if ((DoWhat2 = SIR_PrepareOperand(Arg2,
		TmpValueLL2, TmpValueULL2, TmpValueLD2, TmpValueBIT2,
						Line, FileInfo))
			== SIR_DO_ERROR)
	     { delete *TmpValueBIT1;
	       return(SIR_DO_ERROR);
	      } /* fi */
	 } /* fi */
     else
	{ if ((DoWhat2 = SIR_PrepareOperand(Arg2,
		TmpValueLL2, TmpValueULL2, TmpValueLD2, TmpValueBIT2))
			== SIR_DO_ERROR)
	     { delete *TmpValueBIT1;
	       return(SIR_DO_ERROR);
	      } /* fi */
	 } /* esle */
    } /* fi */
else
   { if ((DoWhat1 = SIR_PrepareOperand(Arg1,
		TmpValueLL1, TmpValueULL1, TmpValueLD1, TmpValueBIT1))
			== SIR_DO_ERROR)
	{ return(SIR_DO_ERROR);
	 } /* fi */
     if ((DoWhat2 = SIR_PrepareOperand(Arg2,
		TmpValueLL2, TmpValueULL2, TmpValueLD2, TmpValueBIT2))
			== SIR_DO_ERROR)
	{ delete *TmpValueBIT1;
	  return(SIR_DO_ERROR);
	 } /* fi */
    } /* esle */

DoWhat = SIR_DO_ERROR;	/* keep the compiler quiet */
switch(DoWhat1)
   { case SIR_DO_LL_COMPUTATION:
	{ DoWhat = DoWhat2;
	  break;
	 }
     case SIR_DO_ULL_COMPUTATION:
	{ switch(DoWhat2)
	     { case SIR_DO_LL_COMPUTATION:
	       case SIR_DO_ULL_COMPUTATION:
		  { DoWhat = SIR_DO_ULL_COMPUTATION;
		    break;
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { DoWhat = SIR_DO_BIT_COMPUTATION;
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { DoWhat = SIR_DO_LD_COMPUTATION;
		    break;
		   }
	       default:
		  { assert(false);	/* bad DoWhat2 */
		   }
	      } /* hctiws */
	  break;
	 }
     case SIR_DO_BIT_COMPUTATION:
	{ switch(DoWhat2)
	     { case SIR_DO_LL_COMPUTATION:
	       case SIR_DO_ULL_COMPUTATION:
	       case SIR_DO_BIT_COMPUTATION:
		  { DoWhat = SIR_DO_BIT_COMPUTATION;
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { DoWhat = SIR_DO_LD_COMPUTATION;
		    break;
		   }
	       default:
		  { assert(false);	/* bad DoWhat2 */
		   }
	      } /* hctiws */
	  break;
	 }
     case SIR_DO_LD_COMPUTATION:
	{ DoWhat = SIR_DO_LD_COMPUTATION;
	  break;
	 }
     default:
	{ assert(false);	/* bad DoWhat1 */
	 }
    } /* hctiws */

return(DoWhat);

} /* end of SIR_PrepareBinaryOperands */


bool SIR_IsConvertable2Bool(		/* checks for legal bool conversion */
	sir_type	*Type)
{
sir_types	*TypeTable;
sir_type	*BoolType;

assert(Type != NULL);

TypeTable = Type->GetTable();
BoolType = TypeTable->FindOrInsert(SIR_TYPE_BOOL);

return(Type->IsConvertableTo(BoolType));

} /* end of SIR_IsConvertable2Bool */


bool SIR_IsConvertable2BitVector(	/* virtually creates a bitvector */
	sir_type	*OrigType,
	SIR_TYPETYPE	*ResultType /* = NULL */,
	int		*ResultLen /* = NULL */,
	int		*ResultLeft /* = NULL */,
	int		*ResultRight /* = NULL */)
{
SIR_TYPETYPE	TypeType	= SIR_TYPE_ANY_TYPE;
int		BitLength	= 0,	/* to keep the compiler quiet */
		LeftBound	= 0,
		RightBound	= 0;

switch(OrigType->Type)
   { case SIR_TYPE_BOOL:
	{ TypeType = (_BITUSGN_BOOL ? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	  BitLength = _BITLEN_BOOL;
	  LeftBound = BitLength - 1;
	  RightBound = 0;
	  break;
	 }
     case SIR_TYPE_CHAR:
	{ TypeType = (_BITUSGN_CHAR ? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	  BitLength = _BITLEN_CHAR;
	  LeftBound = BitLength - 1;
	  RightBound = 0;
	  break;
	 }
     case SIR_TYPE_UCHAR:
	{ TypeType = (_BITUSGN_UCHAR ? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	  BitLength = _BITLEN_UCHAR;
	  LeftBound = BitLength - 1;
	  RightBound = 0;
	  break;
	 }
     case SIR_TYPE_SHORT:
	{ TypeType = (_BITUSGN_SHORT ? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	  BitLength = _BITLEN_SHORT;
	  LeftBound = BitLength - 1;
	  RightBound = 0;
	  break;
	 }
     case SIR_TYPE_USHORT:
	{ TypeType = (_BITUSGN_USHORT ? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	  BitLength = _BITLEN_USHORT;
	  LeftBound = BitLength - 1;
	  RightBound = 0;
	  break;
	 }
     case SIR_TYPE_INT:
	{ TypeType = (_BITUSGN_INT ? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	  BitLength = _BITLEN_INT;
	  LeftBound = BitLength - 1;
	  RightBound = 0;
	  break;
	 }
     case SIR_TYPE_UINT:
	{ TypeType = (_BITUSGN_UINT ? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	  BitLength = _BITLEN_UINT;
	  LeftBound = BitLength - 1;
	  RightBound = 0;
	  break;
	 }
     case SIR_TYPE_LONG:
	{ TypeType = (_BITUSGN_LONG ? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	  BitLength = _BITLEN_LONG;
	  LeftBound = BitLength - 1;
	  RightBound = 0;
	  break;
	 }
     case SIR_TYPE_ULONG:
	{ TypeType = (_BITUSGN_ULONG ? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	  BitLength = _BITLEN_ULONG;
	  LeftBound = BitLength - 1;
	  RightBound = 0;
	  break;
	 }
     case SIR_TYPE_LONGLONG:
	{ TypeType = (_BITUSGN_LLONG ? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	  BitLength = _BITLEN_LLONG;
	  LeftBound = BitLength - 1;
	  RightBound = 0;
	  break;
	 }
     case SIR_TYPE_ULONGLONG:
	{ TypeType = (_BITUSGN_ULLONG ? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	  BitLength = _BITLEN_ULLONG;
	  LeftBound = BitLength - 1;
	  RightBound = 0;
	  break;
	 }
     case SIR_TYPE_FLOAT:
     case SIR_TYPE_DOUBLE:
     case SIR_TYPE_LONGDOUBLE:
	{ return(false);	/* not convertable without specified bounds */
	 }
     case SIR_TYPE_BIT:
     case SIR_TYPE_UBIT:
	{ TypeType = OrigType->Type;
	  LeftBound = OrigType->LeftBound;
	  RightBound = OrigType->RightBound;
	  BitLength = _BITLEN(LeftBound, RightBound);
	  break;
	 }
     case SIR_TYPE_VOID:
     case SIR_TYPE_EVENT:
     case SIR_TYPE_POINTER:
     case SIR_TYPE_STRUCT:
     case SIR_TYPE_UNION:
     case SIR_TYPE_ENUM:
     case SIR_TYPE_ARRAY:
     case SIR_TYPE_METHOD:
     case SIR_TYPE_FUNCTION:
	{ return(false);	/* not convertable */
	 }
     case SIR_TYPE_ANY_TYPE:
	{ assert(false);	/* type '...' must not appear here */
	 }
     case SIR_TYPE_BEHAVIOR:
     case SIR_TYPE_CHANNEL:
     case SIR_TYPE_INTERFACE:
	{ return(false);	/* not convertable */
	 }
     case SIR_TYPE_SIGNAL:
     case SIR_TYPE_BUFFER:
	{ return(SIR_IsConvertable2BitVector(OrigType,
						ResultType, ResultLen,
						ResultLeft, ResultRight));
	 }
     default:
	{ assert(false);	/* bad type */
	 }
    } /* hctiws */

if (ResultType)
   { *ResultType = TypeType;
    } /* fi */
if (ResultLen)
   { *ResultLen = BitLength;
    } /* fi */
if (ResultLeft)
   { *ResultLeft = LeftBound;
    } /* fi */
if (ResultRight)
   { *ResultRight = RightBound;
    } /* fi */

return(true);

} /* end of SIR_IsConvertable2BitVector */


sir_constant *SIR_Convert2BitVector(	/* converts Const to BIT or UBIT */
	sir_constant	*Const)
{
bool	WillBeUnsigned = false;	/* keep compiler quiet */

switch(Const->Type)
   { case SIR_CONST_BOOL:
	{ WillBeUnsigned = _BITUSGN_BOOL;
	  break;
	 }
     case SIR_CONST_CHAR:
	{ WillBeUnsigned = _BITUSGN_CHAR;
	  break;
	 }
     case SIR_CONST_UCHAR:
	{ WillBeUnsigned = _BITUSGN_UCHAR;
	  break;
	 }
     case SIR_CONST_SHORT:
	{ WillBeUnsigned = _BITUSGN_SHORT;
	  break;
	 }
     case SIR_CONST_USHORT:
	{ WillBeUnsigned = _BITUSGN_USHORT;
	  break;
	 }
     case SIR_CONST_INT:
	{ WillBeUnsigned = _BITUSGN_INT;
	  break;
	 }
     case SIR_CONST_UINT:
	{ WillBeUnsigned = _BITUSGN_UINT;
	  break;
	 }
     case SIR_CONST_LONG:
	{ WillBeUnsigned = _BITUSGN_LONG;
	  break;
	 }
     case SIR_CONST_ULONG:
	{ WillBeUnsigned = _BITUSGN_ULONG;
	  break;
	 }
     case SIR_CONST_LONGLONG:
	{ WillBeUnsigned = _BITUSGN_LLONG;
	  break;
	 }
     case SIR_CONST_ULONGLONG:
	{ WillBeUnsigned = _BITUSGN_ULLONG;
	  break;
	 }
     case SIR_CONST_FLOAT:
     case SIR_CONST_DOUBLE:
     case SIR_CONST_LONGDOUBLE:
	{ assert(false);	/* not convertable to bitvector */
	 }			/* without specified bounds     */
     case SIR_CONST_BIT:
     case SIR_CONST_UBIT:
	{ /* nothing to do */
	  return(Const);
	 }
     case SIR_CONST_CHARSTRING:
	{ assert(false);	/* not convertable to bitvector */
	 }
     default:
	{ assert(false);	/* bad constant type */
	 }
    } /* hctiws */

return(Const->Converted(WillBeUnsigned ? SIR_CONST_UBIT : SIR_CONST_BIT));

} /* end of SIR_Convert2BitVector */


static void SIR_PutParenthesis(		/* put parenthesis if needed */
	SIR_EXPRTYPE	ExprType,
	char		Parenthesis,
	sir_expression	*Arg,
	SIR_WHICH_ARG	WhichArg,
	gl_io		*IO)
{
int	ThisPrec,
	ArgPrec;

ThisPrec = SIR_ExprPrecedence[ExprType];
ArgPrec = SIR_ExprPrecedence[Arg->ExprType];

if (ArgPrec < ThisPrec)
   { IO->PutC(Parenthesis);	/* needed because of lower precedence */
     if (Parenthesis == '(')
	{ IO->TabStepUp();
	 } /* fi */
     else
	{ assert(Parenthesis == ')');
	  IO->TabStepDown();
	 } /* esle */
     return;
    } /* fi */
if (ArgPrec == ThisPrec)
   { if (  (  ((ThisPrec == 3) || (ThisPrec == 15)) /* right-associative? */
	    &&(WhichArg == SIR_LEFT_ARG))
	 ||(  ((ThisPrec != 3) && (ThisPrec != 15)) /* left-associative? */
	    &&(WhichArg == SIR_RIGHT_ARG)))
	{ IO->PutC(Parenthesis);	/* needed because of ordering */
	  if (Parenthesis == '(')
	     { IO->TabStepUp();
	      } /* fi */
	  else
	     { assert(Parenthesis == ')');
	       IO->TabStepDown();
	      } /* esle */
	  return;
	 } /* fi */
    } /* fi */

/* no parenthesis needed */

} /* end of SIR_PutParenthesis */


static bool CastNeedsExplicitTypeConv(	/* explicit type conv. needed? */
	SIR_TYPETYPE	FromType,
	SIR_TYPETYPE	ToType)
{

if (  (FromType == SIR_TYPE_FLOAT)
    ||(FromType == SIR_TYPE_DOUBLE)
    ||(FromType == SIR_TYPE_LONGDOUBLE))
   { if (  (ToType == SIR_TYPE_BIT)
	 ||(ToType == SIR_TYPE_UBIT))
	{ return(true);
	 } /* fi */
#ifndef HAVE_LLONG
     if (  (ToType == SIR_TYPE_LONGLONG)
	 ||(ToType == SIR_TYPE_ULONGLONG))
	{ return(true);
	 } /* fi */
#endif /* !HAVE_LLONG */
    } /* fi */
else
   { if (  (ToType != SIR_TYPE_BIT)
	 &&(ToType != SIR_TYPE_UBIT))
	{ if (  (FromType == SIR_TYPE_BIT)
	      ||(FromType == SIR_TYPE_UBIT))
	     { return(true);
	      } /* fi */
#ifndef HAVE_LLONG
	  if (  (FromType == SIR_TYPE_LONGLONG)
	      ||(FromType == SIR_TYPE_ULONGLONG))
	     { return(true);
	      } /* fi */
#endif /* !HAVE_LLONG */
	 } /* fi */
    } /* esle */

return(false);	/* implicit, automatic, or forbidden conversion */

} /* end of CastNeedsExplicitTypeConv */


/* EOF Expression.cc */
