/************************************************************************/
/* Exception.cc: SpecC Internal Representation, Exception Class		*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 10/20/97 */
/************************************************************************/

/* last update: 09/26/06 */

/* modifications: (most recent first)
 *
 * 09/26/06 PC  Adjustments for scrc 2.1
 * 06/03/05 RD	reorganized and renamed global type names
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 10/28/03 RD	adjusted for new member ParentStmnt in SIR_Statement
 * 11/22/02 RD	switched event list from SIR_SymbolPtrs to SIR_Events
 * 07/16/02 RD	allow signals to be used for triggering exceptions
 * 11/21/01 RD	took out default arguments from function definitions
 * 05/31/01 RD	eliminated level 2 of SIR API
 * 05/27/01 RD	added semantic checking for event port accesses
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 04/30/01 RD	replaced use of obsolete form() from "stream.h" with own one
 * 01/30/01 RD	fixed a potential FMR problem in iterators
 *		SIR_Exceptions::DFS_ForAllXxxxx
 */

#include "IntRep/Exception.h"
#include "IntRep/Design.h"

#include <assert.h>


/*** constants and macros ***********************************************/


	/* none */


/*** internal type declarations *****************************************/


	/* none */


/*** class implementations **********************************************/


	/*********************/
	/*** SIR_Exception ***/
	/*********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Exception::SIR_Exception(		/* constructor #1 */
	SIR_EXCEPT_TYPE	ExceptionType,
	sir_events	*Events,
	sir_statement	*Handler,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(  (ExceptionType == SIR_EXCEPTION_TRAP)
       ||(ExceptionType == SIR_EXCEPTION_INTERRUPT));
assert(Events != NULL);
assert(Handler != NULL);

SIR_Exception::ExceptionType	= ExceptionType;
SIR_Exception::Events		= Events;
SIR_Exception::Handler		= Handler;

} /* end of SIR_Exception::SIR_Exception #1 */


SIR_Exception::SIR_Exception(		/* constructor #3 (duplicator) */
	sir_exception	*Original,
	sir_statement	*ParentStmnt /* = NULL */) :
		SIR_Node(Original)
{

SIR_Exception::ExceptionType	= Original->ExceptionType;
SIR_Exception::Events		= Original->Events ?
					new SIR_Events(Original->Events) :
					NULL;
SIR_Exception::Handler		= Original->Handler ?
					new SIR_Statement(Original->Handler) :
					NULL;

if (SIR_Exception::Handler)
   { SIR_Exception::Handler->ParentStmnt = ParentStmnt;
    } /* fi */

} /* end of SIR_Exception::SIR_Exception #3 */


SIR_Exception::~SIR_Exception(void)	/* destructor */
{

delete Events;
delete Handler;

} /* end of SIR_Exception::~SIR_Exception */


sir_exceptions *SIR_Exception::GetList(	/* determines the list of this except.*/
	void)
{

return((sir_exceptions*) Head());

} /* end of SIR_Exception::GetList */


ERROR SIR_Exception::DFS_ForAllNodes(	/* iterator over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{

if ((SIR_Error = (this->*MemberFct)(MemberFctArg)))	/* process this node */
   { return(SIR_Error);
    } /* fi */

/* there are no nodes in Events */
if (Handler)
   { if ((SIR_Error = Handler->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Exception::DFS_ForAllNodes */


ERROR SIR_Exception::DFS_ForAllSymbols(	/* iterator over all symbols */
	sir_symbol_mptr	MemberFct,	/* (depth first) */
	sir_symbol_marg	MemberFctArg)
{

/* this is not a symbol */

/* there are no symbols in Events */
if (Handler)
   { if ((SIR_Error = Handler->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Exception::DFS_ForAllSymbols */


ERROR SIR_Exception::DFS_ForAllUserTypes( /* iterator over all usertypes */
	sir_usertp_mptr	MemberFct,	/* (depth first) */
	sir_usertp_marg	MemberFctArg)
{

/* this is not a usertype */

/* there are no usertypes in Events */
if (Handler)
   { if ((SIR_Error = Handler->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Exception::DFS_ForAllUserTypes */


ERROR SIR_Exception::DFS_ForAllNotes(	/* iterator over all notes */
	sir_note_mptr	MemberFct,	/* (depth first) */
	sir_note_marg	MemberFctArg)
{

/* this is not a note */

/* there are no notes in Events */
if (Handler)
   { if ((SIR_Error = Handler->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Exception::DFS_ForAllNotes */


ERROR SIR_Exception::DFS_ForAllStatements(	/* iterator over statements */
	sir_stmnt_mptr	MemberFct,
	sir_stmnt_marg	MemberFctArg)
{

/* this is not a statement */

/* there are no statements in Events */
if (Handler)
   { if ((SIR_Error = Handler->DFS_ForAllStatements(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Exception::DFS_ForAllStatements */


ERROR SIR_Exception::DFS_ForAllExpressions(	/* iterator over expressions */
	sir_expr_mptr	MemberFct,
	sir_expr_marg	MemberFctArg)
{

/* this is not an expression */

/* there are no expressions in Events */
if (Handler)
   { if ((SIR_Error = Handler->DFS_ForAllExpressions(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Exception::DFS_ForAllExpressions */


void SIR_Exception::SetAlias(	/* sets all type, usertype, symbol alias' */
	sir_exception	*Alias)	/* (iterates over symbols and usertypes) */
{

assert(Alias != NULL);

/* no need to process this node */

/* there are no symbols or usertypes in Events */
if (Handler)
   { Handler->SetAlias(Alias->Handler);
    } /* fi */

} /* end of SIR_Exception::SetAlias */


void SIR_Exception::UnAlias(	/* unalias all type, usertype, symbol links */
	sir_symbols	*GlobalSymbols)
{

if (Events)
   { Events->UnAlias();
    } /* fi */
if (Handler)
   { Handler->UnAlias(GlobalSymbols);
    } /* fi */

} /* end of SIR_Exception::UnAlias */


	/**********************/
	/*** SIR_Exceptions ***/
	/**********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Exceptions::SIR_Exceptions(		/* constructor #1 */
	sir_exception	*FirstEntry /* = NULL */) :
		SIR_List<SIR_Exception>(FirstEntry)
{

/* nothing to do */

} /* end of SIR_Exceptions::SIR_Exceptions #1 */


SIR_Exceptions::SIR_Exceptions(		/* constructor #2 (duplicator) */
	sir_exceptions	*Original,
	sir_statement	*ParentStmnt /* = NULL */)
{
sir_exception	*Curr;

Curr = Original->First();
while(Curr)
   { Append(new SIR_Exception(Curr, ParentStmnt));
     Curr = Curr->Succ();
    } /* elihw */

} /* end of SIR_Exceptions::SIR_Exceptions #2 */


SIR_Exceptions::~SIR_Exceptions(void)	/* destructor */
{

/* nothing to do */

} /* end of SIR_Exceptions::~SIR_Exceptions */


ERROR SIR_Exceptions::DFS_ForAllNodes(	/* iterator over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{
sir_exception	*Exception,
		*Succ;

/* this is not a node, but there are nodes below */

Exception = First();
while(Exception)
   { Succ = Exception->Succ();
     if ((SIR_Error = Exception->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Exception = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Exceptions::DFS_ForAllNodes */


ERROR SIR_Exceptions::DFS_ForAllSymbols( /* iterator over all symbols */
	sir_symbol_mptr	MemberFct,	/* (depth first) */
	sir_symbol_marg	MemberFctArg)
{
sir_exception	*Exception,
		*Succ;

/* this is not a symbol, but there are symbols below */

Exception = First();
while(Exception)
   { Succ = Exception->Succ();
     if ((SIR_Error = Exception->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Exception = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Exceptions::DFS_ForAllSymbols */


ERROR SIR_Exceptions::DFS_ForAllUserTypes( /* iterator over all usertypes */
	sir_usertp_mptr	MemberFct,	/* (depth first) */
	sir_usertp_marg	MemberFctArg)
{
sir_exception	*Exception,
		*Succ;

/* this is not a usertype, but there are usertypes below */

Exception = First();
while(Exception)
   { Succ = Exception->Succ();
     if ((SIR_Error = Exception->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Exception = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Exceptions::DFS_ForAllUserTypes */


ERROR SIR_Exceptions::DFS_ForAllNotes(	/* iterator over all notes */
	sir_note_mptr	MemberFct,	/* (depth first) */
	sir_note_marg	MemberFctArg)
{
sir_exception	*Exception,
		*Succ;

/* this is not a note, but there are notes below */

Exception = First();
while(Exception)
   { Succ = Exception->Succ();
     if ((SIR_Error = Exception->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Exception = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Exceptions::DFS_ForAllNotes */


ERROR SIR_Exceptions::DFS_ForAllStatements(	/* iterator over statements */
	sir_stmnt_mptr	MemberFct,
	sir_stmnt_marg	MemberFctArg)
{
sir_exception	*Exception,
		*Succ;

/* this is not a statement, but there are statements below */

Exception = First();
while(Exception)
   { Succ = Exception->Succ();
     if ((SIR_Error = Exception->DFS_ForAllStatements(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Exception = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Exceptions::DFS_ForAllStatements */


ERROR SIR_Exceptions::DFS_ForAllExpressions(	/* iterator over expressions */
	sir_expr_mptr	MemberFct,
	sir_expr_marg	MemberFctArg)
{
sir_exception	*Exception,
		*Succ;

/* this is not an expression, but there are expressions below */

Exception = First();
while(Exception)
   { Succ = Exception->Succ();
     if ((SIR_Error = Exception->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Exception = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Exceptions::DFS_ForAllExpressions */


void SIR_Exceptions::SetAlias(	/* sets all type, usertype, symbol alias' */
	sir_exceptions	*Alias)	/* (iterates over symbols and usertypes) */
{
sir_exception	*Exception,
		*ExceptionAlias;

assert(Alias != NULL);

/* no need to process this node */

Exception = First();
ExceptionAlias = Alias->First();
while(Exception)
   { Exception->SetAlias(ExceptionAlias);
     Exception = Exception->Succ();
     ExceptionAlias = ExceptionAlias->Succ();
    } /* elihw */

} /* end of SIR_Exceptions::SetAlias */


void SIR_Exceptions::UnAlias(	/* unalias all type, usertype, symbol links */
	sir_symbols	*GlobalSymbols)
{
sir_exception	*Except;

Except = First();
while(Except)
   { Except->UnAlias(GlobalSymbols);
     Except = Except->Succ();
    } /* elihw */

} /* end of SIR_Exceptions::UnAlias */


/*** exported functions *************************************************/


	/* none */


/* EOF Exception.cc */
