/************************************************************************/
/* Event.cc: SpecC Internal Representation, Event Classes		*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 11/21/02 */
/************************************************************************/

/* last update: 09/26/06 */

/* modifications: (most recent first)
 *
 * 09/26/06 PC  Adjustments for scrc 2.1
 * 06/03/05 RD	reorganized and renamed global type names
 * 01/14/05 RD	added inclusion of IntRep/Extern.h
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 12/18/02 RD	added SIR_Events::WriteCC() method (writes recursive list)
 * 12/05/02 RD	added stricter checking to use of 'rising' and 'falling'
 * 11/26/02 RD	added C++ code generation for edges of signals
 * 11/25/02 RD	added support for triggering edges of signals
 * 11/21/02 RD	initial version (separated from Symbol.cc)
 */


#include "IntRep/Event.h"
#include "IntRep/Symbol.h"
#include "IntRep/Extern.h"

#include <assert.h>

/*** constants and macros ***********************************************/


/*** internal type declarations *****************************************/



/*** internal function prototypes ***************************************/


	/* (none) */


/*** class implementations **********************************************/


	/*****************/
	/*** SIR_Event ***/
	/*****************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Event::SIR_Event(			/* constructor #1 */
	sir_symbol	*Symbol,
	SIR_EDGE	Edge /* = SIR_EDGE_ANY */)
{

assert(Symbol != NULL);
assert(Symbol->Type != NULL);
assert(  (Symbol->Type->Type == SIR_TYPE_EVENT)
       ||(Symbol->Type->Type == SIR_TYPE_SIGNAL));
assert(  (Symbol->Type->Type == SIR_TYPE_SIGNAL)
       ||(Edge == SIR_EDGE_ANY));

SIR_Event::Symbol	= Symbol;
SIR_Event::Edge		= Edge;

} /* end of SIR_Event::SIR_Event #1 */


SIR_Event::SIR_Event(			/* constructor #3 (duplicator) */
	sir_event	*Original)
{

SIR_Event::Symbol	= Original->Symbol;
SIR_Event::Edge		= Original->Edge;

} /* end of SIR_Event::SIR_Event #3 */


SIR_Event::~SIR_Event(void)		/* destructor */
{

/* nothing to do */

} /* end of SIR_Event::~SIR_Event */


ERROR SIR_Event::WriteSC(		/* (re-) generates SpecC source code */
	gl_io		*IO,
	bool		CplusplusMode /* = false */)
{

assert(IO != NULL);

assert(Symbol != NULL);
assert(  (Edge == SIR_EDGE_ANY)
       ||(Edge == SIR_EDGE_RISING)
       ||(Edge == SIR_EDGE_FALLING));

if (CplusplusMode)
   { switch(Edge)
	{ case SIR_EDGE_ANY:
	     { IO->PrintF(SIR_SIM_ANY_SIG_FMT, Symbol->Name.chars());
	       break;
	      }
	  case SIR_EDGE_RISING:
	     { IO->PrintF(SIR_SIM_RISING_SIG_FMT, Symbol->Name.chars());
	       break;
	      }
	  case SIR_EDGE_FALLING:
	     { IO->PrintF(SIR_SIM_FALLING_SIG_FMT, Symbol->Name.chars());
	       break;
	      }
	  default:
	     { assert(false);
	      }
	 } /* hctiws */
    } /* fi */
else
   { IO->PutS(Symbol->Name);
     if (Edge == SIR_EDGE_RISING)
	{ IO->PutS(" rising");
	 } /* fi */
     else
	{ if (Edge == SIR_EDGE_FALLING)
	     { IO->PutS(" falling");
	      } /* fi */
	 } /* esle */
   } /* esle */

if ((SIR_Error = IO->Check()))
   { return(SIR_Error);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Event::WriteSC */


bool SIR_Event::IsSignal(void)		/* is a signal or pure event */
{

assert(Symbol != NULL);
assert(Symbol->Type != NULL);

return(Symbol->Type->Type == SIR_TYPE_SIGNAL);

} /* end of SIR_Event::IsSignal */


sir_symbol *SIR_Event::GetSymbol(	/* obtain the symbol */
	void)
{

return(Symbol);

} /* end of SIR_Event::GetSymbol */


SIR_EDGE SIR_Event::GetEdge(		/* obtain the triggering edge */
	void)
{

return(Edge);

} /* end of SIR_Event::GetEdge */


	/******************/
	/*** SIR_Events ***/
	/******************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Events::SIR_Events(			/* constructor #1 */
	sir_event	*FirstEntry /* = NULL */) :
		SIR_List<SIR_Event>(FirstEntry)
{

/* nothing else to do */

} /* end of SIR_Events::SIR_Events #1 */


SIR_Events::SIR_Events(			/* constructor #2 (duplicator) */
	sir_events	*Original)
{
sir_event	*Event;

assert(Original != NULL);

Event = Original->First();
while(Event)
   { Append(new SIR_Event(Event));
     Event = Event->Succ();
    } /* elihw */

} /* end of SIR_Events::SIR_Events #2 */


SIR_Events::~SIR_Events(void)		/* destructor */
{

/* nothing to do */

} /* end of SIR_Events::~SIR_Events */


void SIR_Events::UnAlias(void)	/* unalias all type, usertype, symbol links */
{
sir_event	*Event;

Event = First();
while(Event)
   { if (  (Event->Symbol)
	 &&(Event->Symbol->Alias))
	{ Event->Symbol = Event->Symbol->Alias;
	 } /* fi */
     Event = Event->Succ();
    } /* elihw */

} /* end of SIR_Events::UnAlias */


ERROR SIR_Events::WriteSC(		/* (re-) generates SpecC source code */
	gl_io		*IO,
	const char	*Separator /* = "," */,
	bool		CplusplusMode /* = false */)
{
sir_event	*Event;
ERROR		Error;

assert(IO != NULL);
assert(Separator != NULL);

Event = First();
while(Event)
   { if ((Error = Event->WriteSC(IO, CplusplusMode)))
	{ return(Error);
	 } /* fi */
     Event = Event->Succ();
     if (Event)
	{ IO->PutS(Separator);
	  IO->PutC(' ');
	 } /* fi */
    } /* elihw */

if ((SIR_Error = IO->Check()))
   { return(SIR_Error);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Events::WriteSC */


ERROR SIR_Events::WriteCC(		/* generates (recursive) C++ code */
	gl_io		*IO)
{
sir_event	*Event;
unsigned int	i;

assert(IO != NULL);

Event = First();
while(Event)
   { switch(Event->Edge)
	{ case SIR_EDGE_ANY:
	     { IO->PutS(SIR_SIM_ANY_SIG_PFX);
	       break;
	      }
	  case SIR_EDGE_RISING:
	     { IO->PutS(SIR_SIM_RISING_SIG_PFX);
	       break;
	      }
	  case SIR_EDGE_FALLING:
	     { IO->PutS(SIR_SIM_FALLING_SIG_PFX);
	       break;
	      }
	  default:
	     { assert(false);
	      }
	 } /* hctiws */
     IO->PutS(Event->Symbol->Name);
     Event = Event->Succ();
     if (Event)
	{ IO->PutS(", ");
	 } /* fi */
    } /* elihw */

for(i=0; i<NumElements(); i++)
   { IO->PutC(')');
    } /* rof */

if ((SIR_Error = IO->Check()))
   { return(SIR_Error);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Events::WriteCC */


sir_event *SIR_Events::Find(		/* searches for a specific entry */
	sir_symbol	*Symbol)	/* (returns NULL if not found) */
{
sir_event	*Ptr;

Ptr = First();
while(Ptr)
   { if (Ptr->Symbol == Symbol)
	{ break;	/* found! */
	 } /* fi */
     Ptr = Ptr->Succ();
    } /* elihw */

return(Ptr);

} /* end of SIR_Events::Find */

/*** internal functions *************************************************/


	/* (none) */


/* EOF Event.cc */
