/************************************************************************/
/* specc.h: SpecC run-time simulation library, compiler-level API	*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 03/21/01 */
/************************************************************************/

/* last update: 12/15/14 */

/* modifications: (most recent first)
 *
 * 12/15/14 RD	finalized scrc_V22
 * 12/10/14 SD	Adjustments for modern Linux support
 * 06/21/04 PC  Bug fix: (gcc-2.95.2)Fixed signal keyword used without specc scope.
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 05/16/04 PC  buffered and piped class was included into '_specc' scope
 * 05/14/04 PC  signal class was included into '_specc' scope
 * 05/19/03 RD	bug fix: added missing copy constructor for _specc::pipslice
 * 05/15/03 RD	arranged layout of _specc::event to satisfy sizeof() limitation
 * 03/20/03 RD	added support for signal/buffered bit vectors in port mapping
 * 03/07/03 RD	introduced template specialization of buffered class
 *		for bitvectors (which needs put() and get() methods)
 * 03/04/03 RD	fixed right operand for shifts, should be 'unsigned int'
 * 01/08/03 RD	added support for self-triggering events (_specc::auto_event)
 * 01/03/03 RD	added copy constructors to fix post-in/de-crement operators
 * 12/13/02 RD	added support for asynchronous reset of 'buffered' variables
 * 12/12/02 RD	extended event list helper functions
 * 12/06/02 RD	distinguish 'signal'/'buffered' variables in separate lists
 * 11/29/02 RD	class redesign: signal _is_ an event (not, _has_ an event);
 *		also, added support for signal edges in event lists
 * 06/27/02 RD	refined _specc::event_list::Triggers() to return the match
 * 06/26/02 RD	added 'wait' statement with AND semantics (specc::wait_and())
 * 05/20/02 RD	fixed some constructors to use _specc::event explicitly
 * 05/10/02 RD	added methods 'member' for struct/union member access
 * 04/10/02 RD	added handling of bitvector slices (_specc::bufslice,
 *		_specc::sigslice, _specc::pipslice<d>)
 * 04/09/02 RD	extended template classes 'piped', 'buffered', 'signal'
 *		(and their specializations) with special functions for
 *		bitvectors and longlongs
 * 04/02/02 RD	added conversion ops for 'piped', 'buffered', 'signal' arrays
 * 03/28/02 RD	allow 'piped' variables to be initialized
 * 03/27/02 RD	added template specialization for 'piped' arrays (bug fix)
 * 03/26/02 RD	reverted 'private' class members to 'public'
 * 03/26/02 RD	added template classes for 'signal' array accesses
 * 03/26/02 RD	added missing array-assignment operators
 * 03/22/02 RD	added template classes for 'buffered' array accesses
 * 03/08/02 RD	added template specialization for array types
 * 03/07/02 RD	changed 'buffered' constructors to fixed number of parameters
 *		in order to avoid ambiguities
 * 03/06/02 RD	made 'signal' and 'buffered' members private
 * 03/05/02 RD	refined 'signal' and 'buffered' templates (take out auto-
 *		conversion to event pointer; add explicit constructors for
 *		initialization)
 * 02/27/02 RD	added support for 'signal' and 'buffered' variables
 * 11/19/01 RD	bug fix: made channel destructor virtual
 * 09/07/01 RD	bug fix: modified 'piped' template to use static memory
 * 05/25/01 RD	removed code not needed for the SCRC
 * 05/16/01 RD	resolved name space competition with user by moving more
 *		definitions into _specc name space and adding prefixes
 * 05/16/01 RD	separated common thread code (in thread.[h|cc]) from
 *		platform-specific thread code (in PosixThread.[h|cc])
 * 05/15/01 RD	cleaned up the source code
 * 05/14/01 RD	added _specc::exception to separate data on heap and stack
 * 05/09/01 RD	added support for exception handling
 * 04/25/01 RD	added copy-assignment to piped template (bug fix)
 * 04/16/01 RD	renamed "SIM_Time" to "sim_time"
 * 04/05/01 RD	added support for new 'pipe' syntax with termination
 * 04/04/01 RD	added support for 'pipe', 'piped'
 * 04/02/01 RD	added _specc::event_list
 * 04/02/01 RD	added _specc::event_ptr
 * 04/02/01 RD	replaced event_queue and ready_queue with generic queue
 * 03/30/01 RD	added generic classes _specc::queue and _specc::queue_elem
 * 03/30/01 RD	added _specc::event_queue
 * 03/29/01 RD	added _specc::ready_queue
 * 03/28/01 RD	added inclusion of "piped.h"
 * 03/21/01 RD	initial version
 */


#ifndef __SPECC_H
#define __SPECC_H

#include <sim.h>
#include <bit.h>
#include <longlong.h>


/*** constants and macros ***********************************************/


#ifndef NULL
#define NULL	0
#endif


/*** class declarations *************************************************/


	/**************/
	/*** _specc ***/
	/**************/


class _specc			/* wrapper class serving as name space */
{
public:

	/* wrapped types */

enum SignalEdge			/* type of edge for a signal event */
{
EDGE_ANY,
EDGE_FALLING,
EDGE_RISING
};
typedef enum SignalEdge EDGE;

	/* wrapped sub-classes */

class event;			/* class for event types */
class auto_event;		/* class for self-triggering events */
class event_ptr;		/* class for a list of events */
class event_list;		/* class for a list of event lists */
class behavior;			/* base class for behaviors */
class channel;			/* base class for channels */
class fork;			/* class for parallel behaviors */
class try_block;		/* class for exception-enabled behaviors */
class exception_block;		/* class for exception handler behaviors */
class exception;		/* class for exception handlers (dynamic) */
class queue;			/* class for queues of threads */
class queue_elem;		/* class for elements in thread queues */
class thread_base;		/* base class for threads (native threads) */
class thread;			/* class for threads */
class piped_base;		/* base class for 'piped' variables */

class buffered_base;		/* base class for 'buffered' variables */
template <class T, unsigned int d>
		   class pip_elem;	/* template for 'piped' arrays */
template <class T> class buf_elem;	/* template for 'buffered' arrays */
template <class T> class sig_elem;	/* template for 'signal' arrays */
template <unsigned int d>
	class pipslice;		/* template for 'piped' bitvector slices */
class bufslice;			/* class for 'buffered' bitvector slices */
class sigslice;			/* class for 'signal' bitvector slices */
template <_bit::len_t len, bool usign>
	class bufbus;		/* class for sliced 'buffered' port mapping */
template <_bit::len_t len, bool usign>
	class sigbus;		/* class for sliced 'signal' port mapping */
class bufbus_elem;		/* class for sliced 'buffered' port map */
class sigbus_elem;		/* class for sliced 'signal' port map */
//PC 05/14/04
template<class T>		/* class for signal */
	class signal;
//PC 05/16/04
template<class T>
	class buffered ;       /* template class for 'buffered' types */
template<class T, unsigned int d>
	class piped ;          /* template class for 'piped' variables */

  

	/* wrapped variables */

static sim_time CurrentTime;		/* current simulation time */
#ifdef GL_DELTA_SUPPORT
static sim_time ScheduleTime;		/* scheduler real time stamp */
#endif /* GL_DELTA_SUPPORT */

static thread	*RootThread;		/* the root thread */

static thread	*CurrentThread;		/* the current thread */

static queue	ReadyQueue;		/* the ready queue */

static queue	WaitforQueue;		/* the event queue for 'waitfor' */

static queue	WaitQueue;		/* the queue for 'wait'ing threads */

static queue	NotifiedQueue;		/* the queue for 'notify'ed threads */

static queue	TryQueue;		/* the queue of 'try'ing threads */

static queue	SuspendQueue;		/* the queue of suspended threads */

static event	*NotifiedEvents;	/* list of notified events */

static event_list *Notify1List;		/* list of 'notifyone'ed events */

static buffered_base *BufferedVars;	/* list of 'buffered' variables */

static buffered_base *SignalVars;	/* list of 'signal' variables */

static auto_event *FirstAutoEvent;	/* first self-triggering event */

static auto_event *LastAutoEvent;	/* last self-triggering event */


	/* wrapped functions */

static void start(void);	/* initialize the simulation engine */

static void end(void);		/* clean up the simulation engine */

static void abort(		/* cleanly abort with a message */
	const char	*Format,	/* (arguments as for printf) */
	...		);

static void par(		/* SpecC 'par' replacement */
	fork		*First,		/* (NULL-terminated list of forks) */
	...		);

static void pipe(		/* SpecC 'pipe' replacement (infinite, old) */
	fork		*First,		/* (NULL-terminated list of forks, */
	...		);		/* NULL-term. list of piped vars.) */

static void pipe(		/* SpecC 'pipe' replacement (finite, new) */
	unsigned int	NumStages,	/* total number of pipeline stages */
	unsigned int	FirstStage,	/* first active stage */
	unsigned int	LastStage,	/* last active stage */
	...		);		/* (list of forks (length NumStages), */
					/* NULL-termin. list of piped vars.)  */

static void tryTrapInterrupt(	/* SpecC 'try-trap-interrupt' replacement */
	try_block	*TryBlock,
	exception_block	*First,
	...		);

static void waitfor(		/* SpecC 'waitfor' replacement */
	sim_time	Delay);

static void wait(		/* SpecC 'wait' replacement (OR semantics) */
	event_ptr	*First,		/* (NULL-terminated list of events) */
	...		);

static void wait_and(		/* SpecC 'wait' replacement (AND semantics) */
	event_ptr	*First,		/* (NULL-terminated list of events) */
	...		);

static void notify(		/* SpecC 'notify' replacement */
	event_ptr	*First,		/* (NULL-terminated list of events) */
	...		);

static void notifyone(		/* SpecC 'notifyone' replacement */
	event_ptr	*First,		/* (NULL-terminated list of events) */
	...		);

static sim_time getCurrentTime(	/* obtain current simulation time */
	void);

static void fatalError(		/* fatal error, abort simulation */
	const char	*Msg,
	int		ErrorNo);
};


	/*********************/
	/*** _specc::event ***/
	/*********************/


class _specc::event		/* class for event types */
{
public:
event		*Next;		/* pointer to next notified event (or NULL) */
bool		Notified;	/* flag whether this event has been notified */
bool		Triggered;	/* flag whether this event has been triggered */


event(void);			/* constructor #1 */

virtual ~event(void);		/* destructor */


virtual bool IsNotified(void);	/* check whether this event has been notified */

virtual void Notify(void);	/* notify this event */

static void UnNotifyAll(void);	/* un-notify all notified events */


virtual bool IsValidEdge(	/* signal edge event filter */
	EDGE	Edge);		/* (dummy to be overloaded) */

virtual bool ResetIsActive(	/* asynchronous reset signal check */
	bool	ResetActiveHi);	/* (dummy to be overloaded) */

virtual event_ptr *MappedEvents(/* get list of port-mapped events (or NULL) */
	void);			/* (dummy to be overloaded) */
};


	/**************************/
	/*** _specc::auto_event ***/
	/**************************/


class _specc::auto_event	/* class for self-triggering events */
	: public _specc::event	/* (every auto-event is also an event) */
{
public:
sim_time	Period;		/* cycle time of the self-trigger */
sim_time	TriggerTime;	/* time of the next triggering */
auto_event	*Pred;		/* predecessor in auto-event list */
auto_event	*Succ;		/* successor in auto-event list */


auto_event(			/* constructor #1 */
	sim_time	Period);

~auto_event(void);		/* destructor */


void Insert(void);		/* insert into the sorted list */

void Delete(void);		/* take out of the sorted list */

void TriggerAndUpdate(void);	/* trigger, update and re-order this event */
};


	/*************************/
	/*** _specc::event_ptr ***/
	/*************************/


class _specc::event_ptr		/* class for a list of events */
{
public:
event_ptr	*Next;		/* pointer to the next element (or NULL) */
event		*Event;		/* pointer to the event */
EDGE		Edge;		/* signal edge */


event_ptr(			/* constructor #1 */
	event		*Event,
	EDGE		Edge = EDGE_ANY,
	event_ptr	*Next = NULL);

~event_ptr(void);		/* destructor */
};


	/**************************/
	/*** _specc::event_list ***/
	/**************************/


class _specc::event_list	/* class for a list of event lists */
{
public:
event_list	*Next;		/* pointer to next event list (or NULL) */
event_ptr	*EventList;	/* pointer to the first element of this list */


event_list(			/* constructor #1 */
	event_ptr	*EventList);

~event_list(void);		/* destructor */


event *Triggers(		/* check whether any event matches this list */
	event_ptr	*EventList2);	/* (returns first match, or NULL) */
};


	/************************/
	/*** _specc::behavior ***/
	/************************/


class _specc::behavior		/* base class for behaviors */
{
public:
behavior(void);			/* constructor #1 */

virtual ~behavior(void);	/* destructor */

virtual void main(void) = 0;	/* mandatory main method */
};


	/***********************/
	/*** _specc::channel ***/
	/***********************/


class _specc::channel		/* base class for channels */
{
public:
channel(void);			/* constructor #1 */

virtual ~channel(void);		/* destructor */
};


	/********************/
	/*** _specc::fork ***/
	/********************/


class _specc::fork		/* class for parallel behaviors */
{
public:
behavior	*Behavior;	/* the concurrent behavior */


fork(				/* constructor #1 */
	behavior	*Behavior);

~fork(void);			/* destructor */
};


	/*************************/
	/*** _specc::try_block ***/
	/*************************/


class _specc::try_block		/* class for exception-enabled behaviors */
{
public:
behavior	*Behavior;	/* the sensitive behavior (NULL if empty) */


try_block(			/* constructor #1 */
	behavior	*Behavior);

~try_block(void);		/* destructor */
};


	/*******************************/
	/*** _specc::exception_block ***/
	/*******************************/


class _specc::exception_block	/* class for exception handler behaviors */
{
public:
bool		IsTrap;		/* flag for trap (true) or interrupt (false) */
behavior	*Behavior;	/* the exception handler (NULL if empty) */
event_ptr	*EventList;	/* list of triggering events */


exception_block(		/* constructor #1 */
	bool		IsTrap,
	behavior	*Behavior,
	event_ptr	*First,
	...);

~exception_block(void);		/* destructor */
};


	/*************************/
	/*** _specc::exception ***/
	/*************************/


class _specc::exception		/* class for exception handlers (dynamic) */
{
public:
exception	*Next;		/* next exception in list (or NULL) */
bool		IsTrap;		/* flag for trap (true) or interrupt (false) */
behavior	*Behavior;	/* the exception handler (NULL if empty) */
event_ptr	*EventList;	/* list of triggering events */


exception(			/* constructor #1 */
	exception_block	*ExceptionBlock);

~exception(void);		/* destructor */


exception *FirstMatch(		/* obtain first match with this event list */
	event_ptr	*EventList2);	/* (or NULL if not found) */

exception *FirstMatch(		/* obtain first match with notified events */
	void);				/* (or NULL if not found) */
};


	/**************************/
	/*** _specc::piped_base ***/
	/**************************/


class _specc::piped_base	/* base class for 'piped' variables */
{
public:
piped_base(void);		/* constructor #1 */

virtual ~piped_base(void);	/* destructor */

virtual void update(void) = 0;	/* mandatory update method */
};


	/*****************************/
	/*** _specc::buffered_base ***/
	/*****************************/


class _specc::buffered_base	/* base class for 'buffered' variables */
{
public:
buffered_base	*Next;		/* next 'buffered' variable (or NULL) */
buffered_base	*Prev;		/* previous 'buffered' variable (or NULL) */
event_ptr	*UpdateEvents;	/* list of events that update the buffer */
event		*ResetSignal;	/* asynchronous reset signal (or NULL) */
bool		ResetActiveHi;	/* flag for reset active lo/hi (false/true) */


buffered_base(			/* constructor #1 (for 'buffered' variables) */
	event_ptr	*UpdateEvents,
	event		*ResetSignal = NULL,
	bool		ResetActiveHi = false);

buffered_base(			/* constructor #2 (for 'signal' variables) */
	event		*SignalEvent);

virtual ~buffered_base(void);	/* destructor */


virtual void update(void) = 0;	/* mandatory update method */

virtual void reset(void) = 0;	/* mandatory reset method */
};


	/****************************************/
	/*** piped template (general version) ***/
	/****************************************/


template<class T, unsigned int d>
class _specc::piped :			/* template class for 'piped' variables */
	public _specc::piped_base	/* is based on piped_base */
{
public:
T		Value[d+1];	/* static, FIFO-type storage */


inline piped(void);		/* constructor #1 */

explicit inline piped(		/* constructor #2 (with init) */
	const T		Init);	/* initializer */

inline piped(			/* copy constructor (#3) */
	const piped<T,d> &Orig);

inline ~piped(void);		/* destructor */


inline operator T() const;	/* conversion operator (read access) */


inline _specc::piped<T,d> &operator=(	/* assignment operator #1 */
	T		Var);

inline _specc::piped<T,d> &operator=(	/* assignment operator #1b */
	const _specc::piped<T,d> &PipedVar);

inline _specc::piped<T,d> &operator+=(	/* assignment operator #2 */
	T		Var);

inline _specc::piped<T,d> &operator-=(	/* assignment operator #3 */
	T		Var);

inline _specc::piped<T,d> &operator*=(	/* assignment operator #4 */
	T		Var);

inline _specc::piped<T,d> &operator/=(	/* assignment operator #5 */
	T		Var);

inline _specc::piped<T,d> &operator%=(	/* assignment operator #6 */
	T		Var);

inline _specc::piped<T,d> &operator^=(	/* assignment operator #7 */
	T		Var);

inline _specc::piped<T,d> &operator&=(	/* assignment operator #8 */
	T		Var);

inline _specc::piped<T,d> &operator|=(	/* assignment operator #9 */
	T		Var);

inline _specc::piped<T,d> &operator<<=(	/* assignment operator #10 */
	unsigned int	Var);

inline _specc::piped<T,d> &operator>>=(	/* assignment operator #11 */
	unsigned int	Var);


inline _specc::piped<T,d> &operator++();/* increment operator #1 (pre-) */

inline _specc::piped<T,d> operator++(	/* increment operator #2 (post-) */
	int);

inline _specc::piped<T,d> &operator--();/* decrement operator #1 (pre-) */

inline _specc::piped<T,d> operator--(	/* decrement operator #2 (post-) */
	int);


void update(void);		/* shift the data in the pipeline */


// special pass-through functions for bitvectors and longlongs

inline bool test(void) const;			/* read access to bool */

inline int toInt(void) const;			/* read access to int */

inline unsigned int toUInt(void) const;		/* read access to u. int */

inline long toLong(void) const;			/* read access to long */

inline unsigned long toULong(void) const;	/* read access to u. long */

inline _LONGLONG toLLong(void) const;		/* read access to long long */

inline _ULONGLONG toULLong(void) const;		/* read access to u.long long */

inline double toDouble(void) const;		/* read access to double */

inline long double toLDouble(void) const;	/* read access to long double */

// bitvector slices

inline _specc::pipslice<d> operator()(		/* single bit of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	i);

inline _specc::pipslice<d> operator()(		/* slice of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr);

inline const _specc::pipslice<d> operator()(	/* const bit of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	i) const;

inline const _specc::pipslice<d> operator()(	/* const slice of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr) const;

// struct/union member access

template <class MT>
inline _specc::pip_elem<MT,d> member(		/* member access "operator" */
	int		CharOffset);

template <class MT>
inline const _specc::pip_elem<MT,d> member(	/* const member access "op." */
	int		CharOffset) const;
};


	/*******************************************************/
	/*** piped template (specialization for array types) ***/
	/*******************************************************/


template<int i, class T, unsigned int d>
class _specc::piped<T[i],d> :		/* special template for 'piped' arrays */
	public _specc::piped_base	/* is based on piped_base */
{
public:
T		Value[d+1][i];	/* static, FIFO-type storage */


inline piped(void);			/* constructor #1 */

explicit inline piped(			/* constructor #2 (with init) */
	const T		Init[i]);	/* initializer */

inline piped(				/* copy constructor (#3) */
	const _specc::piped<T[i],d> &Orig);

inline ~piped(void);			/* destructor */


inline operator const T*() const;	/* conversion operator (read access) */


inline _specc::piped<T[i],d> &operator=(	/* assignment operator #1 */
	T		Var[i]);

inline _specc::piped<T[i],d> &operator=(	/* assignment operator #1b */
	const _specc::piped<T[i],d> &PipedVar);

inline _specc::piped<T[i],d> &operator=(	/* assignment operator #1c */
	const _specc::pip_elem<T[i],d> &PipElemVar);


inline _specc::pip_elem<T,d> operator[](/* array access operator */
	int		Index);


void update(void);		/* make the next value the current value */
};


	/*******************************************/
	/*** pip_elem template (general version) ***/
	/*******************************************/


template <class T, unsigned int d>
class _specc::pip_elem		/* template class for 'piped' arrays */
{
public:
T		*ValuePtrs[d+1];/* pointer array to FIFO type storage */


inline pip_elem(		/* constructor #1 */
	void	*ValuePtrs[d+1]);

// default copy constructor is just fine

inline ~pip_elem(void);		/* destructor */


inline operator T() const;	/* conversion operator (read access) */


inline pip_elem<T,d> &operator=(	/* assignment operator #1 */
	T		Var);

inline pip_elem<T,d> &operator=(	/* assignment operator #1b */
	const pip_elem<T,d> &PipElem);

inline pip_elem<T,d> &operator+=(	/* assignment operator #2 */
	T		Var);

inline pip_elem<T,d> &operator-=(	/* assignment operator #3 */
	T		Var);

inline pip_elem<T,d> &operator*=(	/* assignment operator #4 */
	T		Var);

inline pip_elem<T,d> &operator/=(	/* assignment operator #5 */
	T		Var);

inline pip_elem<T,d> &operator%=(	/* assignment operator #6 */
	T		Var);

inline pip_elem<T,d> &operator^=(	/* assignment operator #7 */
	T		Var);

inline pip_elem<T,d> &operator&=(	/* assignment operator #8 */
	T		Var);

inline pip_elem<T,d> &operator|=(	/* assignment operator #9 */
	T		Var);

inline pip_elem<T,d> &operator<<=(	/* assignment operator #10 */
	unsigned int	Var);

inline pip_elem<T,d> &operator>>=(	/* assignment operator #11 */
	unsigned int	Var);


inline pip_elem<T,d> &operator++();	/* increment operator #1 (pre-) */

inline pip_elem<T,d> operator++(	/* increment operator #2 (post-) */
	int);

inline pip_elem<T,d> &operator--();	/* decrement operator #1 (pre-) */

inline pip_elem<T,d> operator--(	/* decrement operator #2 (post-) */
	int);


// special pass-through functions for bitvectors and longlongs

inline bool test(void) const;			/* read access to bool */

inline int toInt(void) const;			/* read access to int */

inline unsigned int toUInt(void) const;		/* read access to u. int */

inline long toLong(void) const;			/* read access to long */

inline unsigned long toULong(void) const;	/* read access to u. long */

inline _LONGLONG toLLong(void) const;		/* read access to long long */

inline _ULONGLONG toULLong(void) const;		/* read access to u.long long */

inline double toDouble(void) const;		/* read access to double */

inline long double toLDouble(void) const;	/* read access to long double */

// bitvector slices

inline _specc::pipslice<d> operator()(		/* single bit of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	i);

inline _specc::pipslice<d> operator()(		/* slice of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr);

inline const _specc::pipslice<d> operator()(	/* const bit of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	i) const;

inline const _specc::pipslice<d> operator()(	/* const slice of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr) const;

// struct/union member access

template <class MT>
inline _specc::pip_elem<MT,d> member(		/* member access "operator" */
	int		CharOffset);

template <class MT>
inline const _specc::pip_elem<MT,d> member(	/* const member access "op." */
	int		CharOffset) const;
};


	/**********************************************************/
	/*** pip_elem template (specialization for array types) ***/
	/**********************************************************/


template<int i, class T, unsigned int d>
class _specc::pip_elem<T[i],d>	/* special template for multi-dim. arrays */
{
public:
T		*ValuePtrs[d+1];/* pointer array to FIFO type storage */


inline pip_elem(		/* constructor #1 */
	void	*ValuePtrs[d+1]);

// default copy constructor is just fine

inline ~pip_elem(void);		/* destructor */


inline operator const T*() const;	/* conversion operator (read access) */


inline pip_elem<T[i],d> &operator=(	/* assignment operator #1 */
	T		Var[i]);

inline pip_elem<T[i],d> &operator=(	/* assignment operator #1b */
	const pip_elem<T[i],d> &PipElem);

inline pip_elem<T[i],d> &operator=(	/* assignment operator #1c */
	const _specc::piped<T[i],d> &PipedVar);


inline pip_elem<T,d> operator[](	/* array access operator */
	int		Index);
};


	/*********************************/
	/*** _specc::pipslice template ***/
	/*********************************/


template<unsigned int d>
class _specc::pipslice		/* template for 'piped' bitvector slices */
{
public:
_bitslice	*Slices[d+1];	/* slice array of FIFO type */


inline pipslice(		/* constructor #1 */
	_bit::chunk	*VecAddr[d+1],
	bool		VecUnsigned,
	_bit::len_t	VecLen,
	_bit::len_t	Left,
	_bit::len_t	Right,
	bool		Unsigned);

inline pipslice(		/* copy constructor */
	const pipslice<d> &Orig);

inline ~pipslice(void);		/* destructor */


inline operator _bit() const;	/* conversion operator (read access) */


inline pipslice<d> &operator=(	/* assignment operator #1 */
	_bit		Var);

inline pipslice<d> &operator=(	/* assignment operator #1b */
	const pipslice<d> &PipSlice);

inline pipslice<d> &operator+=(	/* assignment operator #2 */
	_bit		Var);

inline pipslice<d> &operator-=(	/* assignment operator #3 */
	_bit		Var);

inline pipslice<d> &operator*=(	/* assignment operator #4 */
	_bit		Var);

inline pipslice<d> &operator/=(	/* assignment operator #5 */
	_bit		Var);

inline pipslice<d> &operator%=(	/* assignment operator #6 */
	_bit		Var);

inline pipslice<d> &operator^=(	/* assignment operator #7 */
	_bit		Var);

inline pipslice<d> &operator&=(	/* assignment operator #8 */
	_bit		Var);

inline pipslice<d> &operator|=(	/* assignment operator #9 */
	_bit		Var);

inline pipslice<d> &operator<<=(/* assignment operator #10 */
	unsigned int	Var);

inline pipslice<d> &operator>>=(/* assignment operator #11 */
	unsigned int	Var);


inline pipslice<d> &operator++();	/* increment operator #1 (pre-) */

inline pipslice<d> operator++(		/* increment operator #2 (post-) */
	int);

inline pipslice<d> &operator--();	/* decrement operator #1 (pre-) */

inline pipslice<d> operator--(		/* decrement operator #2 (post-) */
	int);


inline bool test(void) const;			/* read access to bool */

inline int toInt(void) const;			/* read access to int */

inline unsigned int toUInt(void) const;		/* read access to u. int */

inline long toLong(void) const;			/* read access to long */

inline unsigned long toULong(void) const;	/* read access to u. long */

inline _LONGLONG toLLong(void) const;		/* read access to long long */

inline _ULONGLONG toULLong(void) const;		/* read access to u.long long */

inline double toDouble(void) const;		/* read access to double */

inline long double toLDouble(void) const;	/* read access to long double */


inline _specc::pipslice<d> operator()(		/* single bit of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	i);

inline _specc::pipslice<d> operator()(		/* slice of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr);

inline const _specc::pipslice<d> operator()(	/* const bit of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	i) const;

inline const _specc::pipslice<d> operator()(	/* const slice of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr) const;
};


	/*******************************************/
	/*** buffered template (general version) ***/
	/*******************************************/

//PC 05/16/04
template<class T>
class _specc::buffered :	/* template class for 'buffered' types */
	public _specc::buffered_base	/* is based on buffered_base */
{
public:
T		InitValue;	/* initial value (for reset) */
T		CurrValue;	/* current value (for read operations) */
T		NextValue;	/* next value (for write operations) */


explicit inline buffered(	/* constructor #1 */
	_specc::event_ptr *UpdateEvents); /* list of updating events */

inline buffered(		/* constructor #2 */
	_specc::event_ptr *UpdateEvents,/* list of updating events */
	_specc::event	*ResetSignal,	/* async. reset specifier */
	bool		ResetActiveHi);

inline buffered(		/* constructor #3 */
	_specc::event_ptr *UpdateEvents,/* list of updating events */
	const T		Init);		/* initializer */

inline buffered(		/* constructor #4 */
	_specc::event_ptr *UpdateEvents,/* list of updating events */
	_specc::event	*ResetSignal,	/* async. reset specifier */
	bool		ResetActiveHi,
	const T		Init);		/* initializer */

explicit inline buffered(	/* constructor #5 (for signals) */
	_specc::event	*SignalEvent);	/* signal updating event */

inline buffered(		/* constructor #6 (for signals) */
	_specc::event	*SignalEvent,	/* signal updating event */
	const T		Init);		/* initializer */

inline buffered(		/* copy constructor (#7) */
	const _specc::buffered<T> &Orig);

inline ~buffered(void);		/* destructor */


inline operator T() const;	/* conversion operator (read access) */


inline _specc::buffered<T> &operator=(	/* assignment operator #1 */
	T		Var);

inline _specc::buffered<T> &operator=(	/* assignment operator #1b */
	const buffered<T> &BufferedVar);

inline _specc::buffered<T> &operator+=(	/* assignment operator #2 */
	T		Var);

inline _specc::buffered<T> &operator-=(	/* assignment operator #3 */
	T		Var);

inline _specc::buffered<T> &operator*=(	/* assignment operator #4 */
	T		Var);

inline _specc::buffered<T> &operator/=(	/* assignment operator #5 */
	T		Var);

inline _specc::buffered<T> &operator%=(	/* assignment operator #6 */
	T		Var);

inline _specc::buffered<T> &operator^=(	/* assignment operator #7 */
	T		Var);

inline _specc::buffered<T> &operator&=(	/* assignment operator #8 */
	T		Var);

inline _specc::buffered<T> &operator|=(	/* assignment operator #9 */
	T		Var);

inline _specc::buffered<T> &operator<<=(/* assignment operator #10 */
	unsigned int	Var);

inline _specc::buffered<T> &operator>>=(/* assignment operator #11 */
	unsigned int	Var);


inline _specc::buffered<T> &operator++();/* increment operator #1 (pre-) */

inline _specc::buffered<T> operator++(	/* increment operator #2 (post-) */
	int);

inline _specc::buffered<T> &operator--();/* decrement operator #1 (pre-) */

inline _specc::buffered<T> operator--(	/* decrement operator #2 (post-) */
	int);


void update(void);		/* make the next value the current value */

void reset(void);		/* make the next value the initial value */


// special pass-through functions for longlong type

inline bool test(void) const;			/* read access to bool */

inline int toInt(void) const;			/* read access to int */

inline unsigned int toUInt(void) const;		/* read access to u. int */

inline long toLong(void) const;			/* read access to long */

inline unsigned long toULong(void) const;	/* read access to u. long */

inline double toDouble(void) const;		/* read access to double */

inline long double toLDouble(void) const;	/* read access to long double */

// struct/union member access

template <class MT>
inline _specc::buf_elem<MT> member(		/* member access "operator" */
	int		CharOffset);

template <class MT>
inline const _specc::buf_elem<MT> member(	/* const member access "op." */
	int		CharOffset) const;
};


	/*********************************************************/
	/*** buffered template (specialization for bitvectors) ***/
	/*********************************************************/


template<_bit::len_t l, bool s>
class _specc::buffered<bit<l,s> > :
                           /* special template for 'buffered' bitvectors */
	public _specc::buffered_base	/* is based on buffered_base */
{
public:
bit<l,s>	InitValue;	/* initial value (for reset) */
bit<l,s>	CurrValue;	/* current value (for read operations) */
bit<l,s>	NextValue;	/* next value (for write operations) */


explicit inline buffered(	/* constructor #1 */
	_specc::event_ptr *UpdateEvents); /* list of updating events */

inline buffered(		/* constructor #2 */
	_specc::event_ptr *UpdateEvents,/* list of updating events */
	_specc::event	*ResetSignal,	/* async. reset specifier */
	bool		ResetActiveHi);

inline buffered(		/* constructor #3 */
	_specc::event_ptr *UpdateEvents,/* list of updating events */
	const bit<l,s>	Init);		/* initializer */

inline buffered(		/* constructor #4 */
	_specc::event_ptr *UpdateEvents,/* list of updating events */
	_specc::event	*ResetSignal,	/* async. reset specifier */
	bool		ResetActiveHi,
	const bit<l,s>	Init);		/* initializer */

explicit inline buffered(	/* constructor #5 (for signals) */
	_specc::event	*SignalEvent);	/* signal updating event */

inline buffered(		/* constructor #6 (for signals) */
	_specc::event	*SignalEvent,	/* signal updating event */
	const bit<l,s>	Init);		/* initializer */

inline buffered(		/* copy constructor (#7) */
	const _specc::buffered<bit<l,s> > &Orig);

inline ~buffered(void);		/* destructor */


inline operator bit<l,s>() const;	/* conversion operator (read access) */


inline _specc::buffered<bit<l,s> > &operator=(	/* assignment operator #1 */
	bit<l,s>	Var);

inline _specc::buffered<bit<l,s> > &operator=(	/* assignment operator #1b */
	const _specc::buffered<bit<l,s> > &BufferedVar);

inline _specc::buffered<bit<l,s> > &operator+=(	/* assignment operator #2 */
	bit<l,s>	Var);

inline _specc::buffered<bit<l,s> > &operator-=(	/* assignment operator #3 */
	bit<l,s>	Var);

inline _specc::buffered<bit<l,s> > &operator*=(	/* assignment operator #4 */
	bit<l,s>	Var);

inline _specc::buffered<bit<l,s> > &operator/=(	/* assignment operator #5 */
	bit<l,s>	Var);

inline _specc::buffered<bit<l,s> > &operator%=(	/* assignment operator #6 */
	bit<l,s>	Var);

inline _specc::buffered<bit<l,s> > &operator^=(	/* assignment operator #7 */
	bit<l,s>	Var);

inline _specc::buffered<bit<l,s> > &operator&=(	/* assignment operator #8 */
	bit<l,s>	Var);

inline _specc::buffered<bit<l,s> > &operator|=(	/* assignment operator #9 */
	bit<l,s>	Var);

inline _specc::buffered<bit<l,s> > &operator<<=(/* assignment operator #10 */
	unsigned int	Var);

inline _specc::buffered<bit<l,s> > &operator>>=(/* assignment operator #11 */
	unsigned int	Var);


inline _specc::buffered<bit<l,s> > &operator++();
                                    /* increment operator #1 (pre-) */

inline _specc::buffered<bit<l,s> > operator++(	
                      /* increment operator #2 (post-) */
	int);

inline _specc::buffered<bit<l,s> > &operator--();
                      /* decrement operator #1 (pre-) */

inline _specc::buffered<bit<l,s> > operator--(	
                     /* decrement operator #2 (post-) */
	int);


void update(void);		/* make the next value the current value */

void reset(void);		/* make the next value the initial value */


// special pass-through functions for bitvectors

inline bool test(void) const;			/* read access to bool */

inline int toInt(void) const;			/* read access to int */

inline unsigned int toUInt(void) const;		/* read access to u. int */

inline long toLong(void) const;			/* read access to long */

inline unsigned long toULong(void) const;	/* read access to u. long */

inline _LONGLONG toLLong(void) const;		/* read access to long long */

inline _ULONGLONG toULLong(void) const;		/* read access to u.long long */

inline double toDouble(void) const;		/* read access to double */

inline long double toLDouble(void) const;	/* read access to long double */

// bitvector slices

virtual inline _specc::bufslice buf_slice(	/* single bit of a bitvector */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	i);

virtual inline _specc::bufslice buf_slice(	/* slice of a bitvector */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr);

virtual inline const _specc::bufslice buf_slice( /* const bit of a bitvec. */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	i) const;

virtual inline const _specc::bufslice buf_slice( /* const slice of a bitvec.*/
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr) const;

// bitvector bus slices

virtual inline _specc::bufbus_elem* bufbus_slice(/* bus slice in a port map #1*/
	_bit::len_t	sl,
	_bit::len_t	sr);

inline _specc::bufbus_elem* bufbus_slice(	/* bus slice in a port map #2 */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr);


virtual inline void Put(void) const;		/* store value for bufbus */
						/* (dummy to be overloaded) */

virtual inline void Get(void) const;		/* load value for bufbus */
						/* (dummy to be overloaded) */
};


	/**********************************************************/
	/*** buffered template (specialization for array types) ***/
	/**********************************************************/


template<int i, class T>
class _specc::buffered<T[i]> :	/* special template for 'buffered' arrays */
	public _specc::buffered_base	/* is based on buffered_base */
{
public:
T		InitValue[i];	/* initial value (for reset) */
T		CurrValue[i];	/* current value (for read operations) */
T		NextValue[i];	/* next value (for write operations) */


explicit inline buffered(	/* constructor #1 */
	_specc::event_ptr *UpdateEvents); /* list of updating events */

inline buffered(		/* constructor #2 */
	_specc::event_ptr *UpdateEvents,/* list of updating events */
	_specc::event	*ResetSignal,	/* async. reset specifier */
	bool		ResetActiveHi);

inline buffered(		/* constructor #3 */
	_specc::event_ptr *UpdateEvents,/* list of updating events */
	T		Init[i]);	/* initializer */

inline buffered(		/* constructor #4 */
	_specc::event_ptr *UpdateEvents,/* list of updating events */
	_specc::event	*ResetSignal,	/* async. reset specifier */
	bool		ResetActiveHi,
	T		Init[i]);	/* initializer */

explicit inline buffered(	/* constructor #5 (for signals) */
	_specc::event	*SignalEvent);	/* signal updating event */

inline buffered(		/* constructor #6 (for signals) */
	_specc::event	*SignalEvent,	/* signal updating event */
	T		Init[i]);	/* initializer */

inline buffered(		/* copy constructor (#7) */
	const _specc::buffered<T[i]> &Orig);

inline ~buffered(void);		/* destructor */


inline operator const T*() const;	/* conversion operator (read access) */


inline _specc::buffered<T[i]> &operator=(	/* assignment operator #1 */
	T		Var[i]);

inline _specc::buffered<T[i]> &operator=(	/* assignment operator #1b */
	const _specc::buffered<T[i]> &BufferedVar);

inline _specc::buffered<T[i]> &operator=(	/* assignment operator #1c */
	const _specc::buf_elem<T[i]> &BufElemVar);


inline _specc::buf_elem<T> operator[](	/* array access operator */
	int		Index);


void update(void);		/* make the next value the current value */

void reset(void);		/* make the next value the initial value */
};


	/*******************************************/
	/*** buf_elem template (general version) ***/
	/*******************************************/


template <class T>
class _specc::buf_elem		/* template class for 'buffered' arrays */
{
public:
T		*CurrPtr;	/* pointer to current value (for read ops.) */
T		*NextPtr;	/* pointer to next value (for write ops.) */


inline buf_elem(		/* constructor #1 */
	void	*CurrPtr,
	void	*NextPtr);

// default copy constructor is just fine

inline ~buf_elem(void);		/* destructor */


inline operator T() const;	/* conversion operator (read access) */


inline buf_elem<T> &operator=(	/* assignment operator #1 */
	T		Var);

inline buf_elem<T> &operator=(	/* assignment operator #1b */
	const buf_elem<T> &BufElem);

inline buf_elem<T> &operator+=(	/* assignment operator #2 */
	T		Var);

inline buf_elem<T> &operator-=(	/* assignment operator #3 */
	T		Var);

inline buf_elem<T> &operator*=(	/* assignment operator #4 */
	T		Var);

inline buf_elem<T> &operator/=(	/* assignment operator #5 */
	T		Var);

inline buf_elem<T> &operator%=(	/* assignment operator #6 */
	T		Var);

inline buf_elem<T> &operator^=(	/* assignment operator #7 */
	T		Var);

inline buf_elem<T> &operator&=(	/* assignment operator #8 */
	T		Var);

inline buf_elem<T> &operator|=(	/* assignment operator #9 */
	T		Var);

inline buf_elem<T> &operator<<=(/* assignment operator #10 */
	unsigned int	Var);

inline buf_elem<T> &operator>>=(/* assignment operator #11 */
	unsigned int	Var);


inline buf_elem<T> &operator++();/* increment operator #1 (pre-) */

inline buf_elem<T> operator++(	/* increment operator #2 (post-) */
	int);

inline buf_elem<T> &operator--();/* decrement operator #1 (pre-) */

inline buf_elem<T> operator--(	/* decrement operator #2 (post-) */
	int);


// special pass-through functions for bitvectors and longlongs

inline bool test(void) const;			/* read access to bool */

inline int toInt(void) const;			/* read access to int */

inline unsigned int toUInt(void) const;		/* read access to u. int */

inline long toLong(void) const;			/* read access to long */

inline unsigned long toULong(void) const;	/* read access to u. long */

inline _LONGLONG toLLong(void) const;		/* read access to long long */

inline _ULONGLONG toULLong(void) const;		/* read access to u.long long */

inline double toDouble(void) const;		/* read access to double */

inline long double toLDouble(void) const;	/* read access to long double */

// bitvector slices

inline _specc::bufslice buf_slice(		/* single bit of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	i);

inline _specc::bufslice buf_slice(		/* slice of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr);

inline const _specc::bufslice buf_slice(	/* const bit of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	i) const;

inline const _specc::bufslice buf_slice(	/* const slice of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr) const;

// struct/union member access

template <class MT>
inline _specc::buf_elem<MT> member(		/* member access "operator" */
	int		CharOffset);

template <class MT>
inline const _specc::buf_elem<MT> member(	/* const member access "op." */
	int		CharOffset) const;
};


	/**********************************************************/
	/*** buf_elem template (specialization for array types) ***/
	/**********************************************************/


template<int i, class T>
class _specc::buf_elem<T[i]>	/* special template for multi-dim. arrays */
{
public:
T		*CurrPtr;	/* pointer to current value (for read ops.) */
T		*NextPtr;	/* pointer to next value (for write ops.) */


inline buf_elem(		/* constructor #1 */
	void	*CurrPtr,
	void	*NextPtr);

// default copy constructor is just fine

inline ~buf_elem(void);		/* destructor */


inline operator const T*() const;	/* conversion operator (read access) */


inline buf_elem<T[i]> &operator=(	/* assignment operator #1 */
	T		Var[i]);

inline buf_elem<T[i]> &operator=(	/* assignment operator #1b */
	const buf_elem<T[i]> &BufElem);

inline buf_elem<T[i]> &operator=(	/* assignment operator #1c */
	const _specc::buffered<T[i]> &BufferedVar);


inline buf_elem<T> operator[](		/* array access operator */
	int		Index);
};


	/************************/
	/*** _specc::bufslice ***/
	/************************/


class _specc::bufslice		/* class for 'buffered' bitvector slices */
{
public:
_bitslice	CurrSlice;	/* handle for current slice (read ops.) */
_bitslice	NextSlice;	/* handle for next slice (write ops.) */
bufbus_elem	*BusSlices;	/* list of bus slices (or NULL) */


inline bufslice(		/* constructor #1 */
	const _bit	&CurrVec,
	const _bit	&NextVec,
	_bit::len_t	Left,
	_bit::len_t	Right,
	bool		Unsigned,
	bufbus_elem	*Slices = NULL); /* only for slice of bufbus */

inline bufslice(		/* copy constructor */
	const bufslice	&Orig);

inline ~bufslice(void);		/* destructor */


inline operator _bit() const;	/* conversion operator (read access) */


inline bufslice &operator=(	/* assignment operator #1 */
	_bit		Var);

inline bufslice &operator=(	/* assignment operator #1b */
	const bufslice &BufSlice);

inline bufslice &operator+=(	/* assignment operator #2 */
	_bit		Var);

inline bufslice &operator-=(	/* assignment operator #3 */
	_bit		Var);

inline bufslice &operator*=(	/* assignment operator #4 */
	_bit		Var);

inline bufslice &operator/=(	/* assignment operator #5 */
	_bit		Var);

inline bufslice &operator%=(	/* assignment operator #6 */
	_bit		Var);

inline bufslice &operator^=(	/* assignment operator #7 */
	_bit		Var);

inline bufslice &operator&=(	/* assignment operator #8 */
	_bit		Var);

inline bufslice &operator|=(	/* assignment operator #9 */
	_bit		Var);

inline bufslice &operator<<=(	/* assignment operator #10 */
	unsigned int	Var);

inline bufslice &operator>>=(	/* assignment operator #11 */
	unsigned int	Var);


inline bufslice &operator++();	/* increment operator #1 (pre-) */

inline bufslice operator++(	/* increment operator #2 (post-) */
	int);

inline bufslice &operator--();	/* decrement operator #1 (pre-) */

inline bufslice operator--(	/* decrement operator #2 (post-) */
	int);


inline bool test(void) const;			/* read access to bool */

inline int toInt(void) const;			/* read access to int */

inline unsigned int toUInt(void) const;		/* read access to u. int */

inline long toLong(void) const;			/* read access to long */

inline unsigned long toULong(void) const;	/* read access to u. long */

inline _LONGLONG toLLong(void) const;		/* read access to long long */

inline _ULONGLONG toULLong(void) const;		/* read access to u.long long */

inline double toDouble(void) const;		/* read access to double */

inline long double toLDouble(void) const;	/* read access to long double */


inline _specc::bufslice buf_slice(		/* single bit of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	i);

inline _specc::bufslice buf_slice(		/* slice of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr);

inline const _specc::bufslice buf_slice(	/* const bit of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	i) const;

inline const _specc::bufslice buf_slice(	/* const slice of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr) const;


inline void Put(void) const;			/* store value for bufbus */

inline void Get(void) const;			/* load value for bufbus */
};


	/**********************/
	/*** _specc::bufbus ***/
	/**********************/


template <_bit::len_t len, bool usign>
class _specc::bufbus :		/* class for sliced 'buffered' port mapping */
	public _specc::buffered<bit<len,usign> > 
                                   /* is based on 'buffered' bitvectors */
{
public:
bufbus_elem	*Slices;	/* list of bus slices */


inline bufbus(			/* constructor #1 */
	bufbus_elem	*Slices);

inline ~bufbus(void);		/* destructor */


// bitvector slices

virtual inline _specc::bufslice buf_slice(	/* single bit of a bufbus */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	i);

virtual inline _specc::bufslice buf_slice(	/* slice of a bufbus */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr);

virtual inline const _specc::bufslice buf_slice( /* const bit of a bufbus */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	i) const;

virtual inline const _specc::bufslice buf_slice( /* const slice of a bufbus */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr) const;

// bitvector bus slices

virtual inline _specc::bufbus_elem* bufbus_slice(/* bus slice in a port map #1*/
	_bit::len_t	sl,
	_bit::len_t	sr);

inline _specc::bufbus_elem* bufbus_slice(	/* bus slice in a port map #2 */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr);


virtual inline void Put(void) const;		/* store value for bufbus */
						/* (override dummy method) */

virtual inline void Get(void) const;		/* load value for bufbus */
						/* (override dummy method) */
};


	/***************************/
	/*** _specc::bufbus_elem ***/
	/***************************/


class _specc::bufbus_elem	/* class for sliced 'buffered' port map */
{
public:
bufbus_elem	*Next;		/* next slice (or NULL) */
__bitslice	CurrSlice;	/* reference to this slice with current value */
__bitslice	NextSlice;	/* reference to this slice with future value */


inline bufbus_elem(		/* constructor #1 */
	const _bit	&CurrVal,
	const _bit	&NextVal,
	_bit::len_t	sl,
	_bit::len_t	sr,
	bufbus_elem	*Next = NULL);

inline ~bufbus_elem(void);	/* destructor */


inline bufbus_elem *concat(	/* concatenation */
	bufbus_elem	*List);

inline bufbus_elem *slice(	/* slicing */
	_bit::len_t	l,
	_bit::len_t	r);

inline void Put(		/* synchronization with original */
	const _bit	&Buf);	/* (store tmp. value into next value) */

inline void Get(		/* synchronization with original */
	const _bit	&Buf);	/* (load tmp. value from curr. value) */

inline void GetNextValue(	/* synchronization with original */
	const _bit	&Buf);	/* (load tmp. value from next value) */
};


	/*****************************************/
	/*** signal template (general version) ***/
	/*****************************************/


template<class T>
class _specc::signal :			/* template class for 'signal' types */
	public _specc::buffered<T>,	/* is based on 'buffered' template */
	public _specc::event	/* and _is_ an event */
{
public:


inline signal(void);		/* constructor #1 */

explicit inline signal(		/* constructor #2 */
	const T		Init);		/* initializer */

inline signal(			/* copy constructor (#3) */
	const signal<T> &Orig);

inline ~signal(void);		/* destructor */


inline bool IsValidEdge(	/* signal edge event filter */
	_specc::EDGE	Edge);	/* (overrides dummy in event class) */

inline bool ResetIsActive(	/* asynchronous reset signal check */
	bool	ResetActiveHi);	/* (overrides dummy in event class) */


inline signal<T> &operator=(	/* assignment operator #1 */
	T		Var);

inline signal<T> &operator=(	/* assignment operator #1b */
	const signal<T> &SignalVar);

inline signal<T> &operator+=(	/* assignment operator #2 */
	T		Var);

inline signal<T> &operator-=(	/* assignment operator #3 */
	T		Var);

inline signal<T> &operator*=(	/* assignment operator #4 */
	T		Var);

inline signal<T> &operator/=(	/* assignment operator #5 */
	T		Var);

inline signal<T> &operator%=(	/* assignment operator #6 */
	T		Var);

inline signal<T> &operator^=(	/* assignment operator #7 */
	T		Var);

inline signal<T> &operator&=(	/* assignment operator #8 */
	T		Var);

inline signal<T> &operator|=(	/* assignment operator #9 */
	T		Var);

inline signal<T> &operator<<=(	/* assignment operator #10 */
	unsigned int	Var);

inline signal<T> &operator>>=(	/* assignment operator #11 */
	unsigned int	Var);


inline signal<T> &operator++();	/* increment operator #1 (pre-) */

inline signal<T> operator++(	/* increment operator #2 (post-) */
	int);

inline signal<T> &operator--();	/* decrement operator #1 (pre-) */

inline signal<T> operator--(	/* decrement operator #2 (post-) */
	int);

// struct/union member access

template <class MT>
inline _specc::sig_elem<MT> member(		/* member access "operator" */
	int		CharOffset);

template <class MT>
inline const _specc::sig_elem<MT> member(	/* const member access "op." */
	int		CharOffset) const;
};


	/*******************************************************/
	/*** signal template (specialization for bitvectors) ***/
	/*******************************************************/


template<_bit::len_t l, bool s>
class _specc::signal<bit<l,s> > :/* special template for 'signal' bitvectors */
	public _specc::buffered<bit<l,s> >,/* is based on 'buffered' template */
	public _specc::event		/* and _is_ an event */
{
public:


inline signal(void);		/* constructor #1 */

explicit inline signal(		/* constructor #2 */
	const bit<l,s>	Init);		/* initializer */

inline signal(			/* copy constructor (#3) */
	const signal<bit<l,s> > &Orig);

inline ~signal(void);		/* destructor */


inline bool IsValidEdge(	/* signal edge event filter */
	_specc::EDGE	Edge);	/* (overrides dummy in event class) */

inline bool ResetIsActive(	/* asynchronous reset signal check */
	bool	ResetActiveHi);	/* (overrides dummy in event class) */


inline signal<bit<l,s> > &operator=(	/* assignment operator #1 */
	bit<l,s>	Var);

inline signal<bit<l,s> > &operator=(	/* assignment operator #1b */
	const signal<bit<l,s> > &SignalVar);

inline signal<bit<l,s> > &operator+=(	/* assignment operator #2 */
	bit<l,s>	Var);

inline signal<bit<l,s> > &operator-=(	/* assignment operator #3 */
	bit<l,s>	Var);

inline signal<bit<l,s> > &operator*=(	/* assignment operator #4 */
	bit<l,s>	Var);

inline signal<bit<l,s> > &operator/=(	/* assignment operator #5 */
	bit<l,s>	Var);

inline signal<bit<l,s> > &operator%=(	/* assignment operator #6 */
	bit<l,s>	Var);

inline signal<bit<l,s> > &operator^=(	/* assignment operator #7 */
	bit<l,s>	Var);

inline signal<bit<l,s> > &operator&=(	/* assignment operator #8 */
	bit<l,s>	Var);

inline signal<bit<l,s> > &operator|=(	/* assignment operator #9 */
	bit<l,s>	Var);

inline signal<bit<l,s> > &operator<<=(	/* assignment operator #10 */
	unsigned int	Var);

inline signal<bit<l,s> > &operator>>=(	/* assignment operator #11 */
	unsigned int	Var);


inline signal<bit<l,s> > &operator++();	/* increment operator #1 (pre-) */

inline signal<bit<l,s> > operator++(	/* increment operator #2 (post-) */
	int);

inline signal<bit<l,s> > &operator--();	/* decrement operator #1 (pre-) */

inline signal<bit<l,s> > operator--(	/* decrement operator #2 (post-) */
	int);


// bitvector slices

virtual inline _specc::sigslice sig_slice(	/* single bit of a bitvector */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	i);

virtual inline _specc::sigslice sig_slice(	/* slice of a bitvector */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr);

virtual inline const _specc::sigslice sig_slice(/* const bit of a bitvector */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	i) const;

virtual inline const _specc::sigslice sig_slice(/* const slice of a bitvector */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr) const;

// bitvector bus slices

virtual inline _specc::sigbus_elem* sigbus_slice(/* bus slice in a port map #1*/
	_bit::len_t	sl,
	_bit::len_t	sr);

inline _specc::sigbus_elem* sigbus_slice(	/* bus slice in a port map #2 */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr);
};


	/********************************************************/
	/*** signal template (specialization for array types) ***/
	/********************************************************/


template<int i, class T>
class _specc::signal<T[i]> :	/* special template class for 'signal' arrays */
	public _specc::buffered<T[i]>,	/* is based on 'buffered' template */
	public _specc::event	/* and _is_ an event */
{
public:


inline signal(void);		/* constructor #1 */

explicit inline signal(		/* constructor #2 */
	T		Init[i]);	/* initializer */

inline signal(			/* copy constructor (#3) */
	const signal<T[i]> &Orig);

inline ~signal(void);		/* destructor */


inline bool IsValidEdge(	/* signal edge event filter */
	_specc::EDGE	Edge);	/* (overrides dummy in event class) */

inline bool ResetIsActive(	/* asynchronous reset signal check */
	bool	ResetActiveHi);	/* (overrides dummy in event class) */


inline signal<T[i]> &operator=(		/* assignment operator #1 */
	T		Var[i]);

inline signal<T[i]> &operator=(		/* assignment operator #1b */
	const signal<T[i]> &SignalVar);

inline signal<T[i]> &operator=(		/* assignment operator #1c */
	const _specc::sig_elem<T[i]> &SigElemVar);


inline _specc::sig_elem<T> operator[](	/* array access operator */
	int		Index);
};


	/*******************************************/
	/*** sig_elem template (general version) ***/
	/*******************************************/


template <class T>
class _specc::sig_elem :	/* template class for 'signal' arrays */
	public _specc::buf_elem<T>	/* is based on 'buf_elem' template */
{
public:
_specc::event	*SigEventPtr;	/* pointer to event of this signal */


inline sig_elem(		/* constructor #1 */
	_specc::event	*SigEventPtr,
	void		*CurrPtr,
	void		*NextPtr);

// default copy constructor is just fine

inline ~sig_elem(void);		/* destructor */


inline sig_elem<T> &operator=(	/* assignment operator #1 */
	T		Var);

inline sig_elem<T> &operator=(	/* assignment operator #1b */
	const sig_elem<T> &SigElem);

inline sig_elem<T> &operator+=(	/* assignment operator #2 */
	T		Var);

inline sig_elem<T> &operator-=(	/* assignment operator #3 */
	T		Var);

inline sig_elem<T> &operator*=(	/* assignment operator #4 */
	T		Var);

inline sig_elem<T> &operator/=(	/* assignment operator #5 */
	T		Var);

inline sig_elem<T> &operator%=(	/* assignment operator #6 */
	T		Var);

inline sig_elem<T> &operator^=(	/* assignment operator #7 */
	T		Var);

inline sig_elem<T> &operator&=(	/* assignment operator #8 */
	T		Var);

inline sig_elem<T> &operator|=(	/* assignment operator #9 */
	T		Var);

inline sig_elem<T> &operator<<=(/* assignment operator #10 */
	unsigned int	Var);

inline sig_elem<T> &operator>>=(/* assignment operator #11 */
	unsigned int	Var);


inline sig_elem<T> &operator++();/* increment operator #1 (pre-) */

inline sig_elem<T> operator++(	/* increment operator #2 (post-) */
	int);

inline sig_elem<T> &operator--();/* decrement operator #1 (pre-) */

inline sig_elem<T> operator--(	/* decrement operator #2 (post-) */
	int);


// bitvector slices

inline _specc::sigslice sig_slice(		/* single bit of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	i);

inline _specc::sigslice sig_slice(		/* slice of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr);

inline const _specc::sigslice sig_slice(	/* const bit of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	i) const;

inline const _specc::sigslice sig_slice(	/* const slice of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr) const;

// struct/union member access

template <class MT>
inline _specc::sig_elem<MT> member(		/* member access "operator" */
	int		CharOffset);

template <class MT>
inline const _specc::sig_elem<MT> member(	/* const member access "op." */
	int		CharOffset) const;
};


	/**********************************************************/
	/*** sig_elem template (specialization for array types) ***/
	/**********************************************************/


template<int i, class T>
class _specc::sig_elem<T[i]> :	/* special template for multi-dim. arrays */
	public _specc::buf_elem<T[i]>	/* is based on 'buf_elem' template */
{
public:
_specc::event	*SigEventPtr;	/* pointer to event of this signal */


inline sig_elem(		/* constructor #1 */
	_specc::event	*SigEventPtr,
	void		*CurrPtr,
	void		*NextPtr);

// default copy constructor is just fine

inline ~sig_elem(void);		/* destructor */


inline sig_elem<T[i]> &operator=(	/* assignment operator #1 */
	T		Var[i]);

inline sig_elem<T[i]> &operator=(	/* assignment operator #1b */
	const sig_elem<T[i]> &SigElem);

//PC 06/21/04
inline sig_elem<T[i]> &operator=(	/* assignment operator #1c */
	const _specc::signal<T[i]> &SignalVar);


inline sig_elem<T> operator[](		/* array access operator */
	int		Index);
};


	/************************/
	/*** _specc::sigslice ***/
	/************************/


class _specc::sigslice :	/* class for 'signal' bitvector slices */
	public _specc::bufslice		/* is based on 'buffered' slices */
{
public:
_specc::event	*SigEventPtr;	/* pointer to event of this signal */


inline sigslice(		/* constructor #1 */
	_specc::event	*SigEventPtr,
	const _bit	&CurrVec,
	const _bit	&NextVec,
	_bit::len_t	Left,
	_bit::len_t	Right,
	bool		Unsigned,
	sigbus_elem	*Slices = NULL); /* only for slice of sigbus */

inline sigslice(		/* copy constructor */
	const sigslice	&Orig);

inline ~sigslice(void);		/* destructor */


inline sigslice &operator=(	/* assignment operator #1 */
	_bit		Var);

inline sigslice &operator=(	/* assignment operator #1b */
	const sigslice &SigSlice);

inline sigslice &operator+=(	/* assignment operator #2 */
	_bit		Var);

inline sigslice &operator-=(	/* assignment operator #3 */
	_bit		Var);

inline sigslice &operator*=(	/* assignment operator #4 */
	_bit		Var);

inline sigslice &operator/=(	/* assignment operator #5 */
	_bit		Var);

inline sigslice &operator%=(	/* assignment operator #6 */
	_bit		Var);

inline sigslice &operator^=(	/* assignment operator #7 */
	_bit		Var);

inline sigslice &operator&=(	/* assignment operator #8 */
	_bit		Var);

inline sigslice &operator|=(	/* assignment operator #9 */
	_bit		Var);

inline sigslice &operator<<=(	/* assignment operator #10 */
	unsigned int	Var);

inline sigslice &operator>>=(	/* assignment operator #11 */
	unsigned int	Var);


inline sigslice &operator++();	/* increment operator #1 (pre-) */

inline sigslice operator++(	/* increment operator #2 (post-) */
	int);

inline sigslice &operator--();	/* decrement operator #1 (pre-) */

inline sigslice operator--(	/* decrement operator #2 (post-) */
	int);


inline _specc::sigslice sig_slice(		/* single bit of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	i);

inline _specc::sigslice sig_slice(		/* slice of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr);

inline const _specc::sigslice sig_slice(	/* const bit of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	i) const;

inline const _specc::sigslice sig_slice(	/* const slice of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr) const;


inline void Notify(void) const;			/* notify a signal slice */
};


	/**********************/
	/*** _specc::sigbus ***/
	/**********************/


template <_bit::len_t len, bool usign>
class _specc::sigbus :		/* class for sliced 'signal' port mapping */
	public _specc::signal<bit<len,usign> > 
                                /* is based on 'signal' bitvectors */
{
public:
sigbus_elem	*Slices;	/* list of bus slices */


inline sigbus(			/* constructor #1 */
	sigbus_elem	*Slices);

inline ~sigbus(void);		/* destructor */


inline bool IsValidEdge(	/* signal edge event filter */
	_specc::EDGE	Edge);	/* (overrides event and signal classes) */

inline bool ResetIsActive(	/* asynchronous reset signal check */
	bool	ResetActiveHi);	/* (overrides event and signal classes) */


// bitvector slices

virtual inline _specc::sigslice sig_slice(	/* single bit of a sigbus */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	i);

virtual inline _specc::sigslice sig_slice(	/* slice of a sigbus */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr);

virtual inline const _specc::sigslice sig_slice( /* const bit of a sigbus */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	i) const;

virtual inline const _specc::sigslice sig_slice( /* const slice of a sigbus */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr) const;

// bitvector bus slices

virtual inline _specc::sigbus_elem* sigbus_slice(/* bus slice in a port map #1*/
	_bit::len_t	sl,
	_bit::len_t	sr);

inline _specc::sigbus_elem* sigbus_slice(	/* bus slice in a port map #2 */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr);


virtual inline void Put(void) const;	/* store value for sigbus */
					/* (override dummy method) */

virtual inline void Get(void) const;	/* load value for sigbus */
					/* (override dummy method) */

virtual inline void Notify(void);	/* notify all bus events */
					/* (override event method) */

virtual inline bool IsNotified(void);	/* check if this has been notified */

virtual inline _specc::event_ptr *MappedEvents(	/* get port-mapped events */
	void);					/* (override event method) */
};


	/***************************/
	/*** _specc::sigbus_elem ***/
	/***************************/


class _specc::sigbus_elem :	/* class for sliced 'signal' port map */
	public _specc::bufbus_elem
{
public:
_specc::event	*EventPtr;	/* reference to event for this slice */

inline sigbus_elem(		/* constructor #1 */
	const _bit	&CurrVal,
	const _bit	&NextVal,
	_bit::len_t	sl,
	_bit::len_t	sr,
	_specc::event	*EventPtr,
	sigbus_elem	*Next = NULL);

inline ~sigbus_elem(void);	/* destructor */


inline sigbus_elem *concat(	/* concatenation */
	sigbus_elem	*List);

inline sigbus_elem *slice(	/* slicing */
	_bit::len_t	l,
	_bit::len_t	r);

inline void Notify(void);	/* notify all events in this list */
};


/*** exported variables *************************************************/


	/* (none) */


/************************************************************************/
/*** prototype definitions for exported functions		      ***/
/************************************************************************/


	/*** event lists ***/


inline _specc::event_ptr *event(	/* any event (for 'wait', 'notify'...)*/
	_specc::event	*Event,
	_specc::event_ptr *Next = NULL)
{

return(new _specc::event_ptr(Event, _specc::EDGE_ANY, Next));

} /* end of event */


inline _specc::event_ptr *rising(	/* rising signal (for 'wait'...) */
	_specc::event	*Event,
	_specc::event_ptr *Next = NULL)
{

return(new _specc::event_ptr(Event, _specc::EDGE_RISING, Next));

} /* end of rising */


inline _specc::event_ptr *falling(	/* falling signal (for 'wait'...) */
	_specc::event	*Event,
	_specc::event_ptr *Next = NULL)
{

return(new _specc::event_ptr(Event, _specc::EDGE_FALLING, Next));

} /* end of falling */


/*** "implementation" ***************************************************/


#include <piped.h>	/* insert 'piped' template implementation */
#include <signals.h>	/* insert 'signal' template implementation */


#endif /* __SPECC_H */

/* EOF specc.h */
