// c_typed_double_handshake.sh:	template (macro) for a typed 2-way handshake
//				channel between a sender and a receiver
// (rendezvous for data transfer between a sender and a receiver)
//
// author: Rainer Doemer
//
// modifications: (most recent first)
//
// 10/21/04 RD	made receiver wait for its own 'ack' such that both
//		receiver and sender exit in the same synch. cycle
//		(otherwise a tranceiver may receive its own 'ack'
//		when calling send immediately after receive)
// 04/25/03 RD	added support of interface i_typed_tranceiver
// 04/25/03 RD	changed receiver interface to support array types
// 04/22/03 RD	bug fix: receiver must get the data before the acknowledge
// 04/22/03 RD	added comment about non-support of array types
// 10/04/02 RD	added rule about safety of exceptions
// 02/18/02 RD	applied naming convention, integrated with distribution
// 02/07/02 RD	initial version (based on double_handshake.sc)
//
//
// template instantiation parameters (macro arguments):
//
// - type:     the SpecC type of the data to be transferred (i.e. bit[64]);
//             valid types are all basic and composite types including arrays
// - typename: identifier describing the data type (i.e. bit64)
//
// interface rules:
//
// - see files i_typed_sender.sh, i_typed_receiver.sh, i_typed_tranceiver.sh
// - appropriate interfaces must be defined before the channel instantiation
//
// channel rules:
//
// - this double-handshake channel can be seen as a queue (see c_queue.sc)
//   with size 0; because no data can be stored in the channel, the
//   sender and receiver must meet in a rendezvous to exchange the data
// - exactly one receiver and one sender thread may use the same channel
//   instance at the same time; if used by more than one sender or receiver,
//   the behavior of the channel is undefined
// - the same channel instance may be used multiple times in order to
//   transfer multiple data packets from the sender to the receiver
// - using the tranceiver interface, the channel may be used bidirectionally
// - the sender calls send() to send a packet of data to the receiver
// - the receiver calls receive() to receive a packet of data from the sender
// - data packets must be of the specified type
// - the channel operates in rendezvous fashion; a call to send() will
//   suspend the sender until the receiver calls receive(), and vice versa;
//   when both are ready, data is transferred from the sender to the receiver
//   and both can resume their execution
// - calling send() or receive() may suspend the calling thread indefinitely
// - this channel is only safe with respect to exceptions, if any exceptions
//   are guaranteed to occur only for all communicating threads simultaneously;
//   the behavior is undefined, if any exceptions occur only for a subset
//   of the communicating threads
// - no restrictions exist for use of 'waitfor'
// - no restrictions exist for use of 'wait', 'notify', 'notifyone'


#ifndef C_TYPED_DOUBLE_HANDSHAKE_SH
#define C_TYPED_DOUBLE_HANDSHAKE_SH


#include <i_typed_sender.sh>
#include <i_typed_receiver.sh>
#include <i_typed_tranceiver.sh>


#define DEFINE_C_TYPED_DOUBLE_HANDSHAKE(typename, type)			\
									\
channel c_ ## typename ## _double_handshake				\
	implements i_ ## typename ## _sender,				\
		i_ ## typename ## _receiver,				\
		i_ ## typename ## _tranceiver				\
{									\
    event req,								\
          ack;								\
    bool  v = false,							\
          w = false;							\
    type  tmpd;								\
									\
    void receive(type *d)						\
    {									\
	if (!v)								\
	{								\
	    w = true;							\
	    wait req;							\
	    w = false;							\
	}								\
	*d = tmpd;							\
	v = false;							\
	notify ack;							\
	wait ack;							\
    }									\
									\
    void send(type d)							\
    {									\
	tmpd = d;							\
	v = true;							\
	if (w)								\
	{								\
	    notify req;							\
	}								\
	wait ack;							\
    }									\
};


#endif /* C_TYPED_DOUBLE_HANDSHAKE_SH */


// EOF c_typed_double_handshake.sh
