// typed_queue_test4.sc: test bench for c_typed_queue.sh
//
// author: Rainer Doemer
//
// modifications: (most recent first)
//
// 04/25/03 RD	changed receiver interface to support array types
// 02/14/02 RD	applied naming convention, integrated with distribution
// 02/06/02 RD	initial version (type struct, 2 senders, 2 receivers)


#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <sim.sh>


#define QUEUE_SIZE	42	// length of the queue
#define N		100	// number of read and write transactions


import "c_packet_queue";
//import "c_packet_queue_dbg";	// use this to debug the channel!


int Rnd(int Range)		// random number in range [0...Range-1]
{
    return((unsigned int)(drand48() * (double)Range));
}


interface I_SafeCount
{
    unsigned long Count(void);
};

channel SafeCounter implements I_SafeCount
{
    unsigned long c = 0;

    unsigned long Count(void)
    {
	return(++c);
    }
};


behavior Sender(
    i_packet_sender Queue,
    I_SafeCount     PacketCounter,
    in int          ID)
{
    void SendData(int s)
    {
	struct packet Data;
	int           i;

	printf("Time%4s: Sender %d storing the %dth packet.\n",
		time2str(now()), ID, s);
	Data.i = s;
	Data.b = s % 2;
	Data.c = s;
	Data.f = s / 10;
	Data.v = s @ s[9:0];
	for(i=0; i<10; i++)
	{
	    Data.a[i] = s;
	}
	Queue.send(Data);
    }

    void Sleep(void)
    {
	int Duration;

	printf("Time%4s: Sender %d done, ",
		time2str(now()), ID);
	Duration = Rnd(10);
	printf("going to sleep for %d time units.\n", Duration);
	if (Duration)
	    waitfor(Duration);
    }

    void main(void)
    {
	int s = 0;

	while(PacketCounter.Count() <= N)
	{
	    SendData(++s);
	    Sleep();
	}
    }
};


behavior Receiver(
    i_packet_receiver Queue,
    I_SafeCount       PacketCounter,
    in int            ID)
{
    void ReceiveData(int r)
    {
	struct packet Data;
	int           i;

	printf("Time%4s: Receiver %d loading the %dth packet.\n",
		time2str(now()), ID, r);
	Queue.receive(&Data);
	assert(Data.b == Data.i % 2);
	assert(Data.c == Data.i);
	assert(Data.f == Data.i / 10);
	assert(Data.v == Data.i @ Data.i[9:0]);
	for(i=0; i<10; i++)
	{
	    assert(Data.a[i] == Data.i);
	}
    }

    void Sleep(void)
    {
	int Duration;

	printf("Time%4s: Receiver %d done, ",
		time2str(now()), ID);
	Duration = Rnd(20);
	printf("going to sleep for %d time units.\n", Duration);
	if (Duration)
	    waitfor(Duration);
    }

    void main(void)
    {
	int r = 0;

	while(PacketCounter.Count() <= N)
	{
	    ReceiveData(++r);
	    Sleep();
	}
    }
};


behavior Main			// test bench for typed_queue
{
    c_packet_queue Queue(((const unsigned long)QUEUE_SIZE));
    SafeCounter    ReceiveCounter,
		   SendCounter;
    Receiver       r1(Queue, ReceiveCounter, 1),
                   r2(Queue, ReceiveCounter, 2);
    Sender         s1(Queue, SendCounter, 1),
		   s2(Queue, SendCounter, 2);

    int main(void)
    {
	printf("Time%4s: typed_queue_test4: Starting...\n",
		time2str(now()));
	printf("Time%4s: Running 2 receivers, 2 senders, "
			"using data type 'struct packet'.\n",
		time2str(now()));
	par {	r1.main();
		r2.main();
		s1.main();
		s2.main();
		}
	printf("Time%4s: typed_queue_test4: Done.\n",
		time2str(now()));
	return 0;
    }
};


// EOF typed_queue_test4.sc
