/************************************************************************/
/* specc.cc: SpecC run-time simulation library, compiler-level API	*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 03/22/01 */
/************************************************************************/

/* last update: 06/15/04 */

/* modifications: (most recent first)
 *
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 05/15/03 RD	arranged layout of _specc::event to satisfy sizeof() limitation
 * 03/20/03 RD	added support for signal bitvector ports which are mapped
 *		onto concatenated slices of bitvector signals
 *		(i.e. _specc::event::IsNotified(), MappedEvents())
 * 01/08/03 RD	added support for self-triggering events (_specc::auto_event)
 * 12/13/02 RD	added support for asynchronous reset of 'buffered' variables
 * 12/06/02 RD	distinguish 'signal'/'buffered' variables in separate lists
 * 11/27/02 RD	class redesign: signal _is_ an event (not, _has_ an event);
 *		also, added support for signal edges in event lists
 * 06/27/02 RD	refined _specc::event_list::Triggers() to return the match
 * 06/26/02 RD	added 'wait' statement with AND semantics (specc::wait_and())
 * 03/28/02 RD	bug fix: removed bad left-over code in _specc::pipe #2
 * 03/07/02 RD	extended the constructor of 'event_ptr'
 * 02/27/02 RD	added support for 'signal' and 'buffered' variables
 * 02/27/02 RD	bug fix: update event lists in event destructor
 * 06/01/01 RD	two bug fixes for bugs showing up on linux
 * 05/25/01 RD	removed code not needed for the SCRC
 * 05/16/01 RD	resolved name space competition with user by moving more
 *		definitions into _specc name space and adding prefixes
 * 05/15/01 RD	cleaned up the source code
 * 05/14/01 RD	added _specc::exception to separate data on heap and stack
 * 05/12/01 RD	bug fixes for empty constructs
 * 05/11/01 RD	added support for exception handling
 * 04/16/01 RD	renamed "SIM_Time" to "sim_time"
 * 04/04/01 RD	added support for 'pipe' and 'piped'
 * 04/02/01 RD	added class _specc::event_ptr
 * 04/02/01 RD	added support for 'wait' and 'notify'
 * 04/02/01 RD	adjustments for use of generic queues
 * 03/30/01 RD	added code for 'waitfor'
 * 03/28/01 RD	completed setup, filled in initial functions
 * 03/22/01 RD	initial version
 */


#include "specc.h"
#include "thread.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>


/*** constants and macros ***********************************************/


	/* (none) */


/*** internal type declarations *****************************************/


	/* (none) */


/*** internal function prototypes ***************************************/


	/* (none) */


/*** internal variables *************************************************/


	/* (none) */


/*** external variables *************************************************/


	/* (none) */


/************************************************************************/
/*** implementation of internal functions			      ***/
/************************************************************************/


	/* (none) */


/*** class implementations **********************************************/


	/**************/
	/*** _specc ***/
	/**************/


	/*** wrapped variables ***/


sim_time	_specc::CurrentTime	/* current simulation time */
			= 0;
_specc::thread	*_specc::RootThread	/* the root thread */
			= NULL;
_specc::thread	*_specc::CurrentThread	/* the current thread */
			= NULL;
_specc::queue	_specc::ReadyQueue;	/* the ready queue */
					/* (threads in SIM_STATE_READY, */
					/* in planned execution order)  */

_specc::queue	_specc::WaitforQueue;	/* the event queue for 'waitfor' */
					/* (threads in SIM_STATE_WAITFOR,     */
					/* ordered by increasing time stamps) */

_specc::queue	_specc::WaitQueue;	/* the queue for 'wait'ing threads */
					/* (threads in SIM_STATE_WAIT[AND], */
					/* in no particular order)          */

_specc::queue	_specc::NotifiedQueue;	/* the queue for 'notify'ed threads */
					/* (threads in SIM_STATE_NOTIFIED, */
					/* in no particular order)         */

_specc::queue	_specc::TryQueue;	/* the queue of 'try'ing threads */
					/* (threads in SIM_STATE_TRYING, */
					/* in order of creation)         */

_specc::queue	_specc::SuspendQueue;	/* the queue of suspended threads */
					/* (threads in SIM_STATE_SUSPENDED or */
					/* SIM_STATE_SUSPENDED2, any order) */

_specc::event	*_specc::NotifiedEvents	/* list of notified events */
			= NULL;		/* (for 'wait' and 'notify') */

_specc::event_list *_specc::Notify1List	/* list of 'notifyone'ed events */
			= NULL;


// note that there are two lists in order to distinguish 'signal'
// and 'buffered' variables; 'buffered' variables are listed
// in BufferedVars, whereas 'signal' variables are listed in SignalVars;
// this is important for the update-step in the scheduler,
// where 'signal' variables must be updated only after the
// general 'buffered' variables, because 'buffered' variables
// can depend on 'rising' or 'falling' 'signal' variables
// (but not vice-versa!)

_specc::buffered_base *_specc::BufferedVars /* list of 'buffered' variables */
			= NULL;

_specc::buffered_base *_specc::SignalVars   /* list of 'signal' variables */
			= NULL;


_specc::auto_event *_specc::FirstAutoEvent	/* list of self-triggering */
			= NULL;
_specc::auto_event *_specc::LastAutoEvent	/* events                  */
			= NULL;


	/*** wrapped functions ***/


void _specc::start(void)	/* initialize the simulation engine */
{

_specc::thread::Start();		/* initialize the thread package */

_specc::thread::Create(NULL, NULL);	/* create the root thread */

} /* end of _specc::start */


void _specc::end(void)		/* clean up the simulation engine */
{

assert(RootThread != NULL);		/* root thread must exist */
assert(RootThread == CurrentThread);	/* and must be running    */

RootThread->Delete();			/* delete the root thread */

_specc::thread::End();			/* clean up the thread pkg. */

} /* end of _specc::end */


void _specc::abort(		/* cleanly abort with a message */
	const char	*Format,	/* (arguments as for printf) */
	...		)
{
va_list         Args;

va_start(Args, Format);
vfprintf(stderr, Format, Args);		/* print message */
va_end(Args);

::abort();				/* abort the program */

} /* end of _specc::abort */


void _specc::par(		/* SpecC 'par' replacement */
	fork		*First,		/* (NULL-terminated list of forks) */
	...		)
{
va_list		Args;
fork		*Fork;
thread		*ThisThread;

ThisThread = CurrentThread;
assert(ThisThread->Status == SIM_STATE_RUNNING);

if (! First)
   { ThisThread->Schedule();	/* empty 'par' construct is    */
     return;			/* just a possible task switch */
    } /* fi */

va_start(Args, First);
Fork = First;
while(Fork)
   { assert(Fork->Behavior != NULL);	/* must be initialized */
     _specc::thread::Create(Fork->Behavior,	/* create a new child */
				ThisThread);
     Fork = va_arg(Args, fork*);		/* for every argument */
    } /* elihw */
va_end(Args);

ThisThread->Status = SIM_STATE_JOIN;
ThisThread->Schedule();				/* call scheduler */

assert(ThisThread->Status == SIM_STATE_RUNNING);

ThisThread->Join(NULL);				/* join with all children */
ThisThread->DeleteChildren();			/* delete the children */

} /* end of _specc::par */


void _specc::pipe(		/* SpecC 'pipe' replacement (infinite, old) */
	fork		*First,		/* (NULL-terminated list of forks, */
	...		)		/* NULL-term. list of piped vars.) */
{
va_list		Args;
fork		*Fork;
thread		*ThisThread;
piped_base	*PipedVar;
unsigned int	PipeStages,
		ActiveStages,
		i;

ThisThread = CurrentThread;
assert(ThisThread->Status == SIM_STATE_RUNNING);

if (! First)
   { ThisThread->Status = SIM_STATE_JOIN;
     while(true)			/* empty 'pipe' construct is  */
	{ ThisThread->Schedule();	/* a dead end for this thread */
	 } /* elihw */			/* (i.e. should never happen) */
    } /* fi */

PipeStages = 0;
va_start(Args, First);		/* determine number of pipeline stages */
Fork = First;
while(Fork)
   { PipeStages++;
     Fork = va_arg(Args, fork*);
    } /* elihw */
va_end(Args);

ActiveStages = 1;
while(true)	/* endless loop */
   { va_start(Args, First);
     Fork = First;
     for(i=0; i<ActiveStages; i++)
	{ assert(Fork != NULL);			/* must exist */
	  assert(Fork->Behavior != NULL);	/* must be initialized */
	  _specc::thread::Create(Fork->Behavior,
				ThisThread);	/* create a new child     */
	  Fork = va_arg(Args, fork*);		/* for every active stage */
	 } /* elihw */
     while(Fork)
	{ Fork = va_arg(Args, fork*);	/* skip inactive stages */
	 } /* elihw */

     ThisThread->Status = SIM_STATE_JOIN;
     ThisThread->Schedule();			/* call scheduler */

     assert(ThisThread->Status == SIM_STATE_RUNNING);

     ThisThread->Join(NULL);			/* join with all children */
     ThisThread->DeleteChildren();		/* delete the children */

     PipedVar = va_arg(Args, piped_base*);
     while(PipedVar)
	{ PipedVar->update();			/* shift all piped data */
	  PipedVar = va_arg(Args, piped_base*);
	 } /* elihw */
     va_end(Args);

     if (ActiveStages < PipeStages)		/* repeat for next stage */
	{ ActiveStages++;
	 } /* fi */
    } /* elihw */

/* never reached */

} /* end of _specc::pipe #1 */


// note:
//	the following function only specifies the execution of
//	one pipeline iteration; in particular, only the specified
//	range of stages is actually executed, exactly once;
//	therefore, this function must be wrapped in a controlling
//	loop for the pipeline; the initialization, condition and
//	iterator of the pipeline loop are given with the SpecC
//	'pipe' construct and thus can be easily integrated into
//	the loop by the SpecC compiler; also, an appropriate pipeline
//	flushing phase should be specified;
//
//	for example, the SpecC construct
//
//		pipe(i=0; i<10; i++)
//		   { a.main(); b.main(); c.main(); }
//
//	should be translated as follows:
//
//		{
//		unsigned int _First, _Last;
//
//		_First = _Last = 1;
//		for(i=0; i<10; i++)
//		   { pipe(3, _First, _Last, a_fork, b_fork, c_fork, NULL);
//		     if (_Last < 3)
//			  _Last++;
//		    }
//		while (_First++ < _Last)
//		   { pipe(3, _First, _Last, a_fork, b_fork, c_fork, NULL);
//		     if (_Last < 3)
//			  _Last++;
//		    }
//		}


void _specc::pipe(		/* SpecC 'pipe' replacement (finite, new) */
	unsigned int	NumStages,	/* total number of pipeline stages */
	unsigned int	FirstStage,	/* first active stage */
	unsigned int	LastStage,	/* last active stage */
	...		)		/* (list of forks (length NumStages), */
{					/* NULL-termin. list of piped vars.)  */
va_list		Args;
fork		*Fork;
thread		*ThisThread;
piped_base	*PipedVar;
unsigned int	i;

ThisThread = CurrentThread;
assert(ThisThread->Status == SIM_STATE_RUNNING);

if (NumStages == 0)		/* empty 'pipe' construct is    */
   { ThisThread->Schedule();	/* just a possible task switch  */
     va_start(Args, LastStage);	/* plus one shift of piped data */
     PipedVar = va_arg(Args, piped_base*);
     while(PipedVar)
	{ PipedVar->update();
	  PipedVar = va_arg(Args, piped_base*);
	 } /* elihw */
     va_end(Args);
     return;
    } /* fi */

assert(FirstStage <= LastStage);	/* don't allow stupid values */
assert(LastStage <= NumStages);

va_start(Args, LastStage);
for(i=1; i<=NumStages; i++)
   { Fork = va_arg(Args, fork*);
     assert(Fork != NULL);		/* must exist */
     assert(Fork->Behavior != NULL);	/* must be initialized */
     if ((i>=FirstStage) && (i<=LastStage))	 /* for active stages, */
	{ _specc::thread::Create(Fork->Behavior, /* create a new child */
				ThisThread);
	 } /* fi */
    } /* rof */

ThisThread->Status = SIM_STATE_JOIN;
ThisThread->Schedule();			/* call scheduler */

assert(ThisThread->Status == SIM_STATE_RUNNING);

ThisThread->Join(NULL);				/* join with all children */
ThisThread->DeleteChildren();			/* delete the children */

PipedVar = va_arg(Args, piped_base*);
while(PipedVar)
   { PipedVar->update();			/* shift all piped data */
     PipedVar = va_arg(Args, piped_base*);
    } /* elihw */
va_end(Args);

} /* end of _specc::pipe #2 */


void _specc::tryTrapInterrupt(	/* SpecC 'try-trap-interrupt' replacement */
	try_block	*TryBlock,
	exception_block	*First,
	...		)
{
va_list		Args;
exception_block	*ExceptionBlock;
exception	**ExceptionPtr;
thread		*ThisThread,
		*SuspendedChild;
bool		HasBeenInterrupted;
sim_time	InterruptDelay;

ThisThread = CurrentThread;
assert(ThisThread->Status == SIM_STATE_RUNNING);

if (  (! TryBlock)
    ||(! TryBlock->Behavior))
   { ThisThread->Schedule();	/* empty 'try' construct is    */
     return;			/* just a possible task switch */
    } /* fi */

_specc::thread::Create(TryBlock->Behavior,	/* create a new child */
				ThisThread);

va_start(Args, First);
ExceptionBlock = First;
ExceptionPtr = &ThisThread->Exceptions;
assert(ThisThread->Exceptions == NULL);		/* must be fresh */
while(ExceptionBlock)
   { *ExceptionPtr = new exception(ExceptionBlock);
     assert(*ExceptionPtr != NULL);
     ExceptionPtr = &(*ExceptionPtr)->Next;
     ExceptionBlock = va_arg(Args, exception_block*);
    } /* elihw */
va_end(Args);

TryQueue.Append(ThisThread);	/* append entry in the 'try' queue */

do { ThisThread->Status = SIM_STATE_TRYING;
     ThisThread->Schedule();			/* call scheduler */

     assert(ThisThread->Status == SIM_STATE_RUNNING);

     HasBeenInterrupted = (ThisThread->NumChildren == 2);
     if (HasBeenInterrupted)
	{ assert(ThisThread->ExceptionChild != NULL);
	  InterruptDelay = CurrentTime -
		ThisThread->ExceptionChild->CreationTime;
	  ThisThread->Join(		/* join with the completed interrupt */
		ThisThread->ExceptionChild);
	  ThisThread->ExceptionChild->Delete();
	  ThisThread->ExceptionChild = NULL;
	  TryQueue.Append(ThisThread);	/* re-enter in the 'try' queue */
	  assert(ThisThread->NumChildren == 1);
	  SuspendedChild = ThisThread->FirstChild;
	  assert(SuspendedChild != NULL);
	  SuspendedChild->Resume(	/* resume all interrupted children,   */
		InterruptDelay);	/* taking interrupt time into account */
	 } /* fi */
    } while(HasBeenInterrupted);

assert(ThisThread->NumChildren == 1);
ThisThread->Join(ThisThread->FirstChild); /* join and delete the child */
ThisThread->FirstChild->Delete();	  /* (a 'try' or 'abort' behavior) */

delete ThisThread->Exceptions;	/* remove the exception list */
ThisThread->Exceptions = NULL;
ThisThread->ExceptionChild = NULL;

} /* end of _specc::tryTrapInterrupt */


void _specc::waitfor(		/* SpecC 'waitfor' replacement */
	sim_time	Delay)
{
thread		*ThisThread;

ThisThread = CurrentThread;
assert(ThisThread->Status == SIM_STATE_RUNNING);

ThisThread->TimeStamp = CurrentTime + Delay;
WaitforQueue.Insert(ThisThread);	/* insert entry in the waitfor queue */

ThisThread->Status = SIM_STATE_WAITFOR;	/* change state to WAITFOR */
ThisThread->Schedule();			/* call the scheduler */

assert(ThisThread->Status == SIM_STATE_RUNNING);

} /* end of _specc::waitfor */


void _specc::wait(		/* SpecC 'wait' replacement (OR semantics) */
	event_ptr	*First,		/* (NULL-terminated list of events) */
	...		)
{
va_list		Args;
thread		*ThisThread;
event_ptr	*EventPtr;
event_ptr	**EventPtrPtr;

ThisThread = CurrentThread;
assert(ThisThread->Status == SIM_STATE_RUNNING);

assert(ThisThread->EventList == NULL);	/* event list not in use */

// note:
//	the caller is assumed to have properly allocated new event_ptr
//	objects for the argument list; therefore, we just have to
//	concatenate the arguments and store the list for the time it is used;
//	after its use, we will delete the list

va_start(Args, First);		/* traverse the argument list */
EventPtr = First;
EventPtrPtr = &ThisThread->EventList;
while(EventPtr)
   { *EventPtrPtr = EventPtr;
     EventPtrPtr = &EventPtr->Next;
     EventPtr = va_arg(Args, event_ptr*);
    } /* elihw */
va_end(Args);

WaitQueue.Append(ThisThread);	/* create entry in the wait queue */

ThisThread->Status = SIM_STATE_WAIT;	/* change state to WAIT */
ThisThread->Schedule();			/* call the scheduler */

delete ThisThread->EventList;	/* delete the supplied list */
ThisThread->EventList = NULL;

assert(ThisThread->Status == SIM_STATE_RUNNING);

} /* end of _specc::wait */


void _specc::wait_and(		/* SpecC 'wait' replacement (AND semantics) */
	event_ptr	*First,		/* (NULL-terminated list of events) */
	...		)
{
va_list		Args;
thread		*ThisThread;
event_ptr	*EventPtr;
event_ptr	**EventPtrPtr;

ThisThread = CurrentThread;
assert(ThisThread->Status == SIM_STATE_RUNNING);

assert(ThisThread->EventList == NULL);	/* event list not in use */

// note:
//	the caller is assumed to have properly allocated new event_ptr
//	objects for the argument list; therefore, we just have to
//	concatenate the arguments and store the list for the time it is used;
//	the list will be deleted element by element whenever
//	a listed event is notified (until the list is empty)

va_start(Args, First);		/* copy argument list */
EventPtr = First;
EventPtrPtr = &ThisThread->EventList;
while(EventPtr)
   { *EventPtrPtr = EventPtr;
     EventPtrPtr = &EventPtr->Next;
     EventPtr = va_arg(Args, event_ptr*);
    } /* elihw */
va_end(Args);

WaitQueue.Append(ThisThread);	/* create entry in the wait queue */

ThisThread->Status = SIM_STATE_WAITAND;	/* change state to WAITAND */
ThisThread->Schedule();			/* call the scheduler */

assert(ThisThread->EventList == NULL);	/* must be empty now */

assert(ThisThread->Status == SIM_STATE_RUNNING);

} /* end of _specc::wait_and */


void _specc::notify(		/* SpecC 'notify' replacement */
	event_ptr	*First,		/* (NULL-terminated list of events) */
	...		)
{
va_list		Args;
thread		*ThisThread;
event_ptr	*EventPtr;

ThisThread = CurrentThread;
assert(ThisThread->Status == SIM_STATE_RUNNING);

// note:
//	the caller is assumed to have properly allocated new event_ptr
//	objects for the argument list; because we only need the events
//	inside this function, we will delete the arguments immediately
//	after their use

va_start(Args, First);		/* include all arguments into the */
EventPtr = First;		/* list of notified events	  */
while(EventPtr)
   { EventPtr->Event->Notify();
     delete EventPtr;
     EventPtr = va_arg(Args, event_ptr*);
    } /* elihw */
va_end(Args);

/* no scheduler call here */

// note:
//	since we may be in a channel method here,
//	no task switching is allowed at this point!
//
// quote from rule 2.3.2(k), LRM v1.0:
//	"Methods encapsulated in channels are executed in non-preemptive
//	(atomic) manner. A thread executing a channel method is guaranteed
//	not to be interrupted by other threads, unless it is waiting for an
//	event (at a 'wait' statement) or waiting for simulation time increase
//	(at a 'waitfor' statement). [...]"

} /* end of _specc::notify */


void _specc::notifyone(		/* SpecC 'notifyone' replacement */
	event_ptr	*First,		/* (NULL-terminated list of events) */
	...		)
{
va_list		Args;
thread		*ThisThread;
event_ptr	*EventPtr;
event_ptr	*EventList,
		**EventPtrPtr,
		*MappedEvents;
event_list	*NewEventList;

ThisThread = CurrentThread;
assert(ThisThread->Status == SIM_STATE_RUNNING);

// note:
//	the caller is assumed to have properly allocated new event_ptr
//	objects for the argument list; therefore, we just have to
//	concatenate the arguments and store the list for the time it is used;
//	after its use, we will delete the list

va_start(Args, First);		/* store all arguments into the */
EventPtr = First;		/* list of 'notifyone'ed events */
EventList = NULL;
EventPtrPtr = &EventList;
while(EventPtr)
   { assert(EventPtr->Event != NULL);	// must be valid event
     assert(EventPtr->Next == NULL);	// must be single element
     if ((MappedEvents = EventPtr->Event->MappedEvents()))
	{ /* event is actually a set of mapped bitvector signals */
	  *EventPtrPtr = MappedEvents;
	  while(*EventPtrPtr)
	     { EventPtrPtr = &(*EventPtrPtr)->Next;
	      } /* elihw */
	  delete EventPtr;
	 } /* fi */
     else
	{ /* event is an ordinary event or signal */
	  *EventPtrPtr = EventPtr;
	  EventPtrPtr = &EventPtr->Next;
	 } /* esle */
     EventPtr = va_arg(Args, event_ptr*);
    } /* elihw */
va_end(Args);
if (!(NewEventList = new _specc::event_list(EventList)))
   { fatalError("Cannot build event list", ENOMEM);
    } /* fi */
NewEventList->Next = Notify1List;
Notify1List = NewEventList;

/* no scheduler call here */

// note: (see above)

} /* end of _specc::notifyone */


sim_time _specc::getCurrentTime(void)	/* obtain current simulation time */
{

return(CurrentTime);

} /* end of _specc::getCurrentTime */


void _specc::fatalError(		/* fatal error, abort simulation */
	const char	*Msg,
	int		ErrorNo)
{

fprintf(stderr,
	"\n"
	"libsim: FATAL ERROR %d: %s!\n"
	"libsim: Aborted.\n",
	ErrorNo, Msg);

exit(RESULT_FATAL);

} /* end of _specc::fatalError */


	/*********************/
	/*** _specc::event ***/
	/*********************/


_specc::event::event(void)	/* constructor #1 */
{

_specc::event::Next		= NULL;
_specc::event::Notified		= false;
_specc::event::Triggered	= false;

} /* end of _specc::event::event */


_specc::event::~event(void)	/* destructor */
{
event		*Pred;
event_list	*PrevList,
		*List;
event_ptr	*PrevEvent,
		*Event;

/* bug fix 02/27/02, RD: remove yourself from the global event lists */
if (Notified)
   { if (NotifiedEvents == this)
	{ NotifiedEvents = Next;
	 } /* fi */
     else
	{ Pred = NotifiedEvents;
	  while(Pred)
	     { if (Pred->Next == this)
		  { Pred->Next = Next;
		    break;
		   } /* fi */
	       Pred = Pred->Next;
	      } /* elihw */
	 } /* esle */
    } /* fi */
if (Notify1List)
   { List = Notify1List;
     PrevList = NULL;
     while(List)
	{ Event = List->EventList;
	  PrevEvent = NULL;
	  while(Event)
	     { if (Event->Event == this)
		  { if (PrevEvent)
		       { PrevEvent->Next = Event->Next;
			 Event->Next = NULL;
			 delete Event;
			 Event = PrevEvent->Next;
			} /* fi */
		    else
		       { List->EventList = Event->Next;
			 Event->Next = NULL;
			 delete Event;
			 Event = List->EventList;
			} /* esle */
		   } /* fi */
	       else
		  { PrevEvent = Event;
		    Event = Event->Next;
		   } /* esle */
	      } /* elihw */
	  if (! List->EventList)
	     { if (PrevList)
		  { PrevList->Next = List->Next;
		    List->Next = NULL;
		    delete List;
		    List = PrevList->Next;
		   } /* fi */
	       else
		  { Notify1List = List->Next;
		    List->Next = NULL;
		    delete List;
		    List = Notify1List;
		   } /* esle */
	      } /* fi */
	  else
	     { PrevList = List;
	       List = List->Next;
	      } /* esle */
	 } /* elihw */
    } /* fi */

} /* end of _specc::event::~event */


bool _specc::event::IsNotified(void)	/* check if this has been notified */
{					/* (to override by signal busses) */

return(Notified);

} /* end of _specc::event::IsNotified */


void _specc::event::Notify(		/* notify this event */
	void)
{

if (! Notified)		/* not notified yet? */
   { Notified = true;
     assert(Next == NULL);
     Next = NotifiedEvents;	/* insert in list (at the beginning) */
     NotifiedEvents = this;
    } /* fi */

} /* end of _specc::event::Notify */


void _specc::event::UnNotifyAll(	/* un-notify all notified events */
	void)
{
event		*Event,
		*Succ;

Event = NotifiedEvents;
while(Event)
   { Succ = Event->Next;
     Event->Notified = false;
     Event->Next = NULL;
     Event = Succ;
    } /* elihw */

NotifiedEvents = NULL;

} /* end of _specc::event::UnNotifyAll */


bool _specc::event::IsValidEdge(	/* signal edge event filter */
	EDGE		Edge)		/* (dummy to be overloaded) */
{

assert(Edge == EDGE_ANY);	// cannot apply 'rising' or 'falling'
				// to a pure event (type checked)!

return(true);	/* pure event doesn't have edges, every event is valid */

} /* end of _specc::event::IsValidEdge */


bool _specc::event::ResetIsActive(	/* asynchronous reset signal check */
	bool	/* ResetActiveHi */)	/* (dummy to be overloaded) */
{

assert(false);	// pure event must not be used as asynchronous reset!

return(false);	// should never be executed!
		// (only overloaded function will!)

} /* end of _specc::event::ResetIsActive */


_specc::event_ptr *_specc::event::MappedEvents(	/* get port-mapped events */
	void)				/* (dummy to be overloaded) */
{

return(NULL);	// this is an ordinary event

} /* end of _specc::event::MappedEvents */


	/**************************/
	/*** _specc::auto_event ***/
	/**************************/


_specc::auto_event::auto_event(		/* constructor #1 */
	sim_time	Period)
{

assert(Period > 0);	/* must be positive! */

_specc::auto_event::Period	= Period;
_specc::auto_event::TriggerTime	= CurrentTime + Period;
_specc::auto_event::Pred	= NULL;
_specc::auto_event::Succ	= NULL;

Insert();	/* insert this into the list of self-triggering events */

} /* end of _specc::auto_event::auto_event */


_specc::auto_event::~auto_event(	/* destructor */
	void)
{

Delete();

} /* end of _specc::auto_event::~auto_event */


void _specc::auto_event::Insert(	/* insert into the sorted list */
	void)
{
auto_event	*Curr;

assert(Pred == NULL);
assert(Succ == NULL);

Curr = FirstAutoEvent;
while(Curr)
   { if (TriggerTime <= Curr->TriggerTime)
	{ /* insert before Curr */
	  Pred = Curr->Pred;
	  Succ = Curr;
	  if (Pred)
	     { Pred->Succ = this;
	      } /* fi */
	  else
	     { assert(FirstAutoEvent == Curr);
	       FirstAutoEvent = this;
	      } /* esle */
	  Succ->Pred = this;
	  return;
	 } /* fi */
     Curr = Curr->Succ;
    } /* elihw */

/* append this at the end */

Pred = LastAutoEvent;
Succ = NULL;
if (Pred)
   { Pred->Succ = this;
    } /* fi */
else
   { assert(FirstAutoEvent == NULL);
     FirstAutoEvent = this;
    } /* esle */
LastAutoEvent = this;

} /* end of _specc::auto_event::Insert */


void _specc::auto_event::Delete(	/* take out of the sorted list */
	void)
{

if (Pred)
   { Pred->Succ = Succ;
    } /* fi */
else
   { assert(FirstAutoEvent == this);
     FirstAutoEvent = Succ;
    } /* esle */

if (Succ)
   { Succ->Pred = Pred;
    } /* fi */
else
   { assert(LastAutoEvent == this);
     LastAutoEvent = Pred;
    } /* esle */

Pred = NULL;
Succ = NULL;

} /* end of _specc::auto_event::Delete */


void _specc::auto_event::TriggerAndUpdate(	/* trigger, update, re-order */
	void)
{

Notify();			/* notify this event */

assert(TriggerTime == CurrentTime);
TriggerTime += Period;		/* update the triggering time for next time */

if (  (Succ)
    &&(Succ->TriggerTime < TriggerTime))
   { Delete();			/* re-insert if necessary */
     Insert();
    } /* fi */

} /* end of _specc::auto_event::TriggerAndUpdate */


	/*************************/
	/*** _specc::event_ptr ***/
	/*************************/


_specc::event_ptr::event_ptr(		/* constructor #1 */
	event		*Event,
	EDGE		Edge /* = EDGE_ANY */,
	event_ptr	*Next /* = NULL */)
{

_specc::event_ptr::Next		= Next;
_specc::event_ptr::Event	= Event;
_specc::event_ptr::Edge		= Edge;

} /* end of _specc::event_ptr::event_ptr */


_specc::event_ptr::~event_ptr(void)	/* destructor */
{

delete Next;

} /* end of _specc::event_ptr::~event_ptr */


	/**************************/
	/*** _specc::event_list ***/
	/**************************/


_specc::event_list::event_list(		/* constructor #1 */
	event_ptr	*EventList)
{

_specc::event_list::Next	= NULL;
_specc::event_list::EventList	= EventList;

} /* end of _specc::event_list::event_list */


_specc::event_list::~event_list(void)	/* destructor */
{

delete Next;
delete EventList;

} /* end of _specc::event_list::~event_list */


_specc::event *_specc::event_list::Triggers(	/* check if any event matches */
	event_ptr	*EventList2)	/* (returns first match, or NULL) */
{
event_ptr	*Event1,
		*Event2;
_specc::event	*Result;

// note:
//	this function is only used for the Notify1List;
//	therefore, _this_ list does not have any 'falling'
//	or 'rising' filters

if ((!this) || (!EventList2))
   { return(NULL);
    } /* fi */

Event1 = EventList;	/* set flags up */
while(Event1)
   { assert(Event1->Event->Triggered == false);	/* must be unused */
     assert(Event1->Edge == EDGE_ANY);	// see note above!
     Event1->Event->Triggered = true;
     Event1 = Event1->Next;
    } /* elihw */

Result = NULL;
Event2 = EventList2;	/* check for match */
while(Event2)
   { if (  (Event2->Event->Triggered)
	 &&(Event2->Event->IsValidEdge(Event2->Edge)))
	{ Result = Event2->Event;
	  break;
	 } /* fi */
     Event2 = Event2->Next;
    } /* elihw */

Event1 = EventList;	/* reset the flags */
while(Event1)
   { Event1->Event->Triggered = false;
     Event1 = Event1->Next;
    } /* elihw */

return(Result);

} /* end of _specc::event_list::Triggers */


	/************************/
	/*** _specc::behavior ***/
	/************************/


_specc::behavior::behavior(	/* constructor #1 */
	void)
{

/* nothing to do */

} /* end of _specc::behavior::behavior */


_specc::behavior::~behavior(	/* destructor #1 */
	void)
{

/* nothing to do */

} /* end of _specc::behavior::~behavior */


	/***********************/
	/*** _specc::channel ***/
	/***********************/


_specc::channel::channel(	/* constructor #1 */
	void)
{

/* nothing to do */

} /* end of _specc::channel::channel */


_specc::channel::~channel(	/* destructor */
	void)
{

/* nothing to do */

} /* end of _specc::channel::~channel */


	/********************/
	/*** _specc::fork ***/
	/********************/


_specc::fork::fork(		/* constructor #1 */
	behavior	*Behavior)
{

assert(Behavior != NULL);

_specc::fork::Behavior	= Behavior;

} /* end of _specc::fork::fork */


_specc::fork::~fork(void)	/* destructor */
{

/* nothing to do */

} /* end of _specc::fork::~fork */


	/*************************/
	/*** _specc::try_block ***/
	/*************************/


_specc::try_block::try_block(	/* constructor #1 */
	behavior	*Behavior)
{

_specc::try_block::Behavior	= Behavior;

} /* end of _specc::try_block::try_block */


_specc::try_block::~try_block(		/* destructor */
	void)
{

/* nothing to do */

} /* end of _specc::try_block::~try_block */


	/*******************************/
	/*** _specc::exception_block ***/
	/*******************************/


_specc::exception_block::exception_block(	/* constructor #1 */
	bool		IsTrap,
	behavior	*Behavior,
	event_ptr	*First,
	...)
{
va_list		Args;
event_ptr	*EventPtr;
event_ptr	**EventPtrPtr;

// note: Behavior may be NULL indicating an empty exception handler

_specc::exception_block::IsTrap		= IsTrap;
_specc::exception_block::Behavior	= Behavior;
_specc::exception_block::EventList	= NULL;

// note:
//	the caller is assumed to have properly allocated new event_ptr
//	objects for the argument list; therefore, we just have to
//	concatenate the arguments and store the list for the time it is used;
//	after its use, we will delete the list

va_start(Args, First);		/* traverse event list */
EventPtr = First;
EventPtrPtr = &EventList;
while(EventPtr)
   { *EventPtrPtr = EventPtr;
     EventPtrPtr = &EventPtr->Next;
     EventPtr = va_arg(Args, event_ptr*);
    } /* elihw */
va_end(Args);

} /* end of _specc::exception_block::exception_block */


_specc::exception_block::~exception_block(	/* destructor */
	void)
{

delete EventList;

} /* end of _specc::exception_block::~exception_block */


	/*************************/
	/*** _specc::exception ***/
	/*************************/


_specc::exception::exception(		/* constructor #1 */
	exception_block	*ExceptionBlock)
{

assert(ExceptionBlock != NULL);

_specc::exception::Next		= NULL;
_specc::exception::IsTrap	= ExceptionBlock->IsTrap;
_specc::exception::Behavior	= ExceptionBlock->Behavior;
_specc::exception::EventList	= ExceptionBlock->EventList;

ExceptionBlock->EventList = NULL;	/* we take over this list */

} /* end of _specc::exception::exception */


_specc::exception::~exception(		/* destructor */
	void)
{

delete Next;
delete EventList;

} /* end of _specc::exception::~exception */


_specc::exception *_specc::exception::FirstMatch( /* obtain first match (#1) */
	event_ptr	*EventList2)		/* (or NULL if not found) */
{
event_ptr	*Event1,
		*Event2;
exception	*Exception,
		*Result;

// note:
//	this function is only used for the Notify1List;
//	therefore, EventList2 does not have any 'falling'
//	or 'rising' filters

if ((!this) || (!EventList2))
   { return(NULL);
    } /* fi */

Event2 = EventList2;	/* set flags up */
while(Event2)
   { assert(Event2->Event->Triggered == false);	/* must be unused */
     assert(Event2->Edge == EDGE_ANY);	// see note above!
     Event2->Event->Triggered = true;
     Event2 = Event2->Next;
    } /* elihw */

Result = NULL;		/* search for match */
Exception = this;
while(Exception)
   { Event1 = Exception->EventList;
     while(Event1)
	{ if (  (Event1->Event->Triggered)
	      &&(Event1->Event->IsValidEdge(Event1->Edge)))
	     { Result = Exception;
	       break;
	      } /* fi */
	  Event1 = Event1->Next;
	 } /* elihw */
     if (Result)
	{ break;
	 } /* fi */
     Exception = Exception->Next;
    } /* elihw */

Event2 = EventList2;	/* reset the flags */
while(Event2)
   { Event2->Event->Triggered = false;
     Event2 = Event2->Next;
    } /* elihw */

return(Result);

} /* end of _specc::exception::FirstMatch #1 */


_specc::exception *_specc::exception::FirstMatch( /* obtain first match (#2) */
	void)					/* (or NULL if not found) */
{
event_ptr	*Event;
exception	*Exception;

Exception = this;	/* search for match */
while(Exception)
   { Event = Exception->EventList;
     while(Event)
	{ if (  (Event->Event->IsNotified())
	      &&(Event->Event->IsValidEdge(Event->Edge)))
	     { return(Exception);
	      } /* fi */
	  Event = Event->Next;
	 } /* elihw */
     Exception = Exception->Next;
    } /* elihw */

return(NULL);

} /* end of _specc::exception::FirstMatch #2 */


	/**************************/
	/*** _specc::piped_base ***/
	/**************************/


_specc::piped_base::piped_base(void)	/* constructor #1 */
{

/* nothing to do */

} /* end of _specc::piped_base::piped_base */


_specc::piped_base::~piped_base(	/* destructor */
	void)
{

/* nothing to do */

} /* end of _specc::piped_base::~piped_base */


	/*****************************/
	/*** _specc::buffered_base ***/
	/*****************************/


_specc::buffered_base::buffered_base(	/* constructor #1 (for 'buffered') */
	event_ptr	*UpdateEvents,
	event		*ResetSignal /* = NULL */,
	bool		ResetActiveHi /* = false */)
{

// register this with the global list of 'buffered' variables
// (insert at the beginning of BufferedVars)

buffered_base::Next = BufferedVars;
buffered_base::Prev = NULL;

if (BufferedVars)
   { assert(BufferedVars->Prev == NULL);
     BufferedVars->Prev = this;
    } /* fi */
BufferedVars = this;

buffered_base::UpdateEvents	= UpdateEvents;
buffered_base::ResetSignal	= ResetSignal;
buffered_base::ResetActiveHi	= ResetActiveHi;

} /* end of _specc::buffered_base::buffered_base #1 */


_specc::buffered_base::buffered_base(	/* constructor #2 (for 'signal') */
	event		*SignalEvent)
{

// register this with the global list of 'signal' variables
// (insert at the beginning of SignalVars)

buffered_base::Next = SignalVars;
buffered_base::Prev = NULL;

if (SignalVars)
   { assert(SignalVars->Prev == NULL);
     SignalVars->Prev = this;
    } /* fi */
SignalVars = this;

buffered_base::UpdateEvents	= new _specc::event_ptr(SignalEvent);
buffered_base::ResetSignal	= NULL;		// no reset for signals
buffered_base::ResetActiveHi	= false;	// no reset for signals

} /* end of _specc::buffered_base::buffered_base #2 */


_specc::buffered_base::~buffered_base(	/* destructor */
	void)
{

// un-register this with the proper global list of 'buffered'
// or 'signal' variables

if (Prev)
   { Prev->Next = Next;
    } /* fi */
else
   { assert(  ((BufferedVars == this)&&(SignalVars != this))
	    ||((BufferedVars != this)&&(SignalVars == this)));
     if (BufferedVars == this)
	{ BufferedVars = Next;
	 } /* fi */
     else
	{ SignalVars = Next;
	 } /* esle */
    } /* esle */
if (Next)
   { Next->Prev = Prev;
    } /* fi */

delete UpdateEvents;

} /* end of _specc::buffered_base::~buffered_base */


	/**********************/
	/*** piped template ***/
	/**********************/


/* (see header file "piped.h") */


	/*************************/
	/*** buffered template ***/
	/*************************/


/* (see header file "signals.h") */


	/***********************/
	/*** signal template ***/
	/***********************/


/* (see header file "signals.h") */


/************************************************************************/
/*** implementation of exported functions			      ***/
/************************************************************************/


	/* (none) */


/* EOF specc.cc */
