/* Symbol.cc: SpecC Internal Representation, Symbol Classes		*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 10/08/97 */
/************************************************************************/

/* last update: 09/26/06 */

/* modifications: (most recent first)
 *
 * 09/26/06 PC  Adjustments for scrc 2.1
 * 01/12/06 RD	added empty virtual destructor to interface classes
 * 06/03/05 RD	reorganized and renamed global type names
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 11/13/03 HY  added HAVE_METHOD support
 * 10/30/03 RD	added some control-flow checking (i.e. CheckControlFlow())
 * 10/02/03 RD	added SIR_Symbols::CreateNewName()
 * 05/29/03 RD	bug fix: made iterator SIR_Symbol::DFS_ForAllSymbols()
 *		safe with respect to the deletion of the current leaf object
 *		(since it is used with conventions of API layer 2 iterators)
 * 04/02/03 RD	bug fix: const array members (see FixConstArrayTypeInClass())
 * 04/01/03 RD	bug fix: piped array of const type is invalid
 * 02/21/03 RD	fixed the dependency analysis among symbols with respect to
 *		the now supported local variables in functions
 * 02/13/03 RD	added ParentFsmdState link to each scope
 * 02/13/03 RD	added support for notes at fsmd states
 * 02/03/03 RD	added support for local scope in fsmd statements
 * 01/10/03 RD	added support for implicit clock (auto-event)
 * 12/27/02 RD	added support for new dependency of buffers on clock/reset
 * 12/19/02 RD	added support for signals and buffers in code generation
 * 12/18/02 RD	removed unused method SIR_Symbol::WriteSC4ClassBody()
 * 12/17/02 RD	added support for asynchronous reset of 'buffered' variables
 * 12/05/02 RD	added stricter type check for piped variables
 * 11/21/02 RD	separated SIR_Event from SIR_SymbolPtr, see file Event.cc
 * 09/19/02 RD	support for adding source code info during code generation
 * 08/16/02 RD	allow behaviors to be mapped onto suitable interface-type ports
 * 07/22/02 RD	bug fix in IsRedundantMethodDecl(), annotations got lost
 * 07/18/02 RD	disallow uninitialized piped arrays of unknown size
 * 07/16/02 RD	extended storage class checking for 'signal'/'buffered' vars.
 * 07/15/02 RD	extended SIR_Symbols::DefineVariable()
 * 07/10/02 RD	added SIR_Symbols::DeclareLocalState(), DefineLocalState()
 * 06/20/02 RD	bug fix: merge annotations at symbols during Integrate()
 * 04/26/02 RD	added iterator SIR_Symbol::MatchName()
 * 04/17/02 RD	added SIR_Symbols::NumTypeUsers() (bug fix)
 * 04/02/02 RD	compute unspecified array size from initializer where needed
 * 03/29/02 RD	add support for complex initializers for class members
 *		(eliminating SIR_ERROR_COMPLEX_MEMBER_INITIALIZER (#2073))
 * 03/28/02 RD	allow 'piped' variables to be initialized
 * 02/20/02 RD	allow 'import' to sort out recursive structures and enums
 * 02/15/02 RD	eliminated sizeof() difference between SpecC and C++
 *		(otherwise bitvectors cannot be malloc()ed)
 * 01/14/02 RD	added support for composite annotations
 * 11/29/01 RD	inserted a missing 'const' qualifier
 * 11/21/01 RD	took out default arguments from function definitions
 * 11/19/01 RD	skip typedef 'wchar_t' in generated C++ code
 * 11/13/01 RD	added line wrapping in code generation
 * 11/08/01 RD	switched code generation to use GL_IO layer
 * 11/01/01 RD	added line breaks in generated code for class ports and
 *		function/method definitions
 * 10/18/01 RD	SpecC code generator optimizes away redundant method
 *		declarations (added SIR_Symbol::IsRedundantMethodDecl())
 * 10/16/01 RD	implemented code generation by order of source line info
 * 10/11/01 RD	improved overall formatting of generated code
 * 10/04/01 RD	improved indentation of generated code
 * 10/01/01 RD	bug fix: string array initialization in classes not supported
 * 09/07/01 RD	bug fix: code generation adjusted for modified 'piped' template
 * 08/28/01 RD	bug fix: bad C++ code was generated for unused class ports
 *		of array type (array of references)
 * 05/31/01 RD	eliminated level 2 of SIR API
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 05/25/01 RD	eliminated support for automatic IP wrapping (scc -ip option)
 * 05/23/01 RD	added implicit initialization with zero for class members
 *		which are not explicitly initialized by the user
 *		(see SIR_STORE_MAIN_BEHAVIOR_AS_STATIC)
 * 05/19/01 RD	fixed condition under which 'extern "C"' is generated
 * 04/30/01 RD	replaced use of obsolete form() from "stream.h" with own one
 * 04/27/01 RD	inserted ConvertInitializer() to perform explicit type
 *		conversion to expected type when entering into SIR
 *		(this avoids cumbersome conversion later);
 *		took out explicit type conversion during C++ output
 * 04/24/01 RD	adjusted explicit type conversion for initializers
 * 04/20/01 RD	added explicit type conversion for initializers
 * 02/14/01 RD	accomodate traversal routines for new constant node in
 *		port mapping tree (new constant/open mappings); adjusted
 *		C++ code generation
 * 01/30/01 RD	fixed a potential FMR problem in iterators
 *		SIR_Symbols::DFS_ForAllXxxxx
 * RD 06/27/00	added code for 12 and 16 byte alignment
 * RD 06/21/00	bug fix: another false warning about hidden symbols disabled
 * RD 06/21/00	bug fix: self-referencing enums crash (see DataTypes2.sc)
 * RD 06/18/00	Disabled warnings about function-pointer arguments
 *		hiding members of user-defined types (see DataTypes2.sc)
 * AG 05/26/00	Changed concatenation operator in port mappings to '@'
 */


#include "IntRep/Symbol.h"
#include "IntRep/Declarator.h"
#include "IntRep/Extern.h"
#include "Alignment.h"

#include <assert.h>
#include <ctype.h>


/*** constants and macros ***********************************************/


	/* optional code */

#define SIR_WARN_ABOUT_SHADOWED_SYMBOLS		/* during parsing only */


	/* location of the instantiation hierarchy */

/* Note:
 * SpecC LRM 1.0 requires that data at class level
 * is initialized with zero unless explicitly initialized
 * otherwise by the user. This requirement is automatically met
 * if the instantiation tree is allocated in static memory
 * (C++ guarantees zero-initialization of static data).
 * Unless there is other special initialization (not in the
 * current version), the following definition must be set
 * to SIR_STORE_MAIN_BEHAVIOR_AS_STATIC.
 */
//#define SIR_STORE_MAIN_BEHAVIOR_ON_STACK	/* instantiate as local var. */
#define SIR_STORE_MAIN_BEHAVIOR_AS_STATIC	/* instantiate as global var. */


	/* 'wchar_t' type */

/* Note:
 * The following is a temporary trick to suppress a 'typedef xxx wchar_t;'
 * declaration from the "C" header files to appear in the generated "C++" code.
 * The real solution to this issue would be to use our own, clean header
 * files... Until then, we will have to rely on this trick (see also below).
 * (RD, 11/19/01)
 */
#define SIR_SUPPRESS_TYPEDEF_WCHAR_T	/* suppress 'wchar_t' typedef in C++ */


/*** internal type declarations *****************************************/


	/* (none) */

/*** internal function prototypes ***************************************/


static void MarkImplementedMethods(	/* mark all methods with this name */
	const char	*Name,
	sir_symbol_ptr	*InterfaceList);

static sir_type *FixConstArrayTypeInClass( /* bug fix 04/02/03, RD */
	sir_type	*Type,
	SIR_SCOPE	Scope);


/*** class implementations **********************************************/


	/******************/
	/*** SIR_Symbol ***/
	/******************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Symbol::SIR_Symbol(			/* constructor #1 (general) */
	SIR_SYMBOLCLASS	Class,
	const char	*Name,
	sir_type	*Type /* = NULL */,
	SIR_STORAGE	StorageClass /* = SIR_STORAGE_NONE */,
	int		PipeStages /* = 0 */,
	sir_initializer	*Initializer /* = NULL */,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(Name != NULL);
assert(*Name != 0);

SIR_Symbol::Class		= Class;
SIR_Symbol::Name		= Name;
SIR_Symbol::StorageClass	= StorageClass;
SIR_Symbol::PipeStages		= PipeStages;
SIR_Symbol::Type		= Type;
SIR_Symbol::Notes		= NULL;
SIR_Symbol::Initializer		= Initializer;
SIR_Symbol::EnumValue		= 0;
SIR_Symbol::Parameters		= NULL;
SIR_Symbol::ParamScope		= NULL;
SIR_Symbol::ClassScope		= NULL;
SIR_Symbol::Labels		= NULL;
SIR_Symbol::FctBody		= NULL;
SIR_Symbol::Interfaces		= NULL;
SIR_Symbol::Events		= NULL;
SIR_Symbol::Period		= NULL;
SIR_Symbol::ResetSignal		= NULL;
SIR_Symbol::ResetActiveHi	= false;
SIR_Symbol::PortMappings	= NULL;
SIR_Symbol::Imported		= NULL;
SIR_Symbol::Alias		= NULL;

} /* end of SIR_Symbol::SIR_Symbol #1 */


SIR_Symbol::SIR_Symbol(			/* constructor #2 (functions) */
	const char	*Name,
	sir_type	*Type,
	SIR_STORAGE	StorageClass,
	sir_parameters	*Parameters,
	sir_symbols	*ParamScope,
	sir_labels	*Labels,
	sir_statement	*FctBody,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(Name != NULL);
assert(*Name != 0);

SIR_Symbol::Class		= SIR_SYMBOL_IDENTIFIER;
SIR_Symbol::Name		= Name;
SIR_Symbol::StorageClass	= StorageClass;
SIR_Symbol::PipeStages		= 0;
SIR_Symbol::Type		= Type;
SIR_Symbol::Notes		= NULL;
SIR_Symbol::Initializer		= NULL;
SIR_Symbol::EnumValue		= 0;
SIR_Symbol::Parameters		= Parameters;
SIR_Symbol::ParamScope		= ParamScope;
SIR_Symbol::ClassScope		= NULL;
SIR_Symbol::Labels		= Labels;
SIR_Symbol::FctBody		= FctBody;
SIR_Symbol::Interfaces		= NULL;
SIR_Symbol::Events		= NULL;
SIR_Symbol::Period		= NULL;
SIR_Symbol::ResetSignal		= NULL;
SIR_Symbol::ResetActiveHi	= false;
SIR_Symbol::PortMappings	= NULL;
SIR_Symbol::Imported		= NULL;
SIR_Symbol::Alias		= NULL;

if (ParamScope)
   { ParamScope->ParentSymbol = this;	/* automatically fill in back pointer */
    } /* fi */

} /* end of SIR_Symbol::SIR_Symbol #2 */


SIR_Symbol::SIR_Symbol(			/* constructor #3 (classes) */
	SIR_SYMBOLCLASS	Class,
	const char	*Name,
	sir_type	*Type,
	sir_parameters	*Parameters,
	sir_symbols	*ClassScope,
	sir_symbol_ptrs	*Interfaces,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(Name != NULL);
assert(*Name != 0);

SIR_Symbol::Class		= Class;
SIR_Symbol::Name		= Name;
SIR_Symbol::StorageClass	= SIR_STORAGE_NONE;
SIR_Symbol::PipeStages		= 0;
SIR_Symbol::Type		= Type;
SIR_Symbol::Notes		= NULL;
SIR_Symbol::Initializer		= NULL;
SIR_Symbol::EnumValue		= 0;
SIR_Symbol::Parameters		= Parameters;
SIR_Symbol::ParamScope		= NULL;
SIR_Symbol::ClassScope		= ClassScope;
SIR_Symbol::Labels		= NULL;
SIR_Symbol::FctBody		= NULL;
SIR_Symbol::Interfaces		= Interfaces;
SIR_Symbol::Events		= NULL;
SIR_Symbol::Period		= NULL;
SIR_Symbol::ResetSignal		= NULL;
SIR_Symbol::ResetActiveHi	= false;
SIR_Symbol::PortMappings	= NULL;
SIR_Symbol::Imported		= NULL;
SIR_Symbol::Alias		= NULL;

if (ClassScope)
   { ClassScope->ParentSymbol = this;	/* automatically fill in back pointer */
    } /* fi */

} /* end of SIR_Symbol::SIR_Symbol #3 */


SIR_Symbol::SIR_Symbol(			/* constructor #4 (instantiations) */
	const char	*Name,
	sir_type	*Type,
	sir_portmaps	*PortMappings,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(Name != NULL);
assert(Type != NULL);
assert(  (Type->Type == SIR_TYPE_BEHAVIOR)
       ||(Type->Type == SIR_TYPE_CHANNEL));
assert(PortMappings != NULL);

SIR_Symbol::Class		= SIR_SYMBOL_IDENTIFIER;
SIR_Symbol::Name		= Name;
SIR_Symbol::StorageClass	= SIR_STORAGE_NONE;
SIR_Symbol::PipeStages		= 0;
SIR_Symbol::Type		= Type;
SIR_Symbol::Notes		= NULL;
SIR_Symbol::Initializer		= NULL;
SIR_Symbol::EnumValue		= 0;
SIR_Symbol::Parameters		= NULL;
SIR_Symbol::ParamScope		= NULL;
SIR_Symbol::ClassScope		= NULL;
SIR_Symbol::Labels		= NULL;
SIR_Symbol::FctBody		= NULL;
SIR_Symbol::Interfaces		= NULL;
SIR_Symbol::Events		= NULL;
SIR_Symbol::Period		= NULL;
SIR_Symbol::ResetSignal		= NULL;
SIR_Symbol::ResetActiveHi	= false;
SIR_Symbol::PortMappings	= PortMappings;
SIR_Symbol::Imported		= NULL;
SIR_Symbol::Alias		= NULL;

} /* end of SIR_Symbol::SIR_Symbol #4 */


SIR_Symbol::SIR_Symbol(			/* constructor #6 (special f. parser) */
	sir_declspec	*DeclSpec,
	sir_declarator	*Declarator,
	sir_type	*Type,
	bool		StoreParameters /* = false */) :
		SIR_Node(Declarator->Line, Declarator->FileInfo)
{

assert(DeclSpec != NULL);
assert(Declarator != NULL);
assert(Declarator->SymbolName != NULL);
assert(Type != NULL);

SIR_Symbol::Class		= (DeclSpec->StorageClass ==
						SIR_STORAGE_TYPEDEF) ?
					SIR_SYMBOL_TYPEDEF :
					SIR_SYMBOL_IDENTIFIER;
SIR_Symbol::Name		= Declarator->SymbolName->chars();
SIR_Symbol::StorageClass	= DeclSpec->StorageClass;
SIR_Symbol::PipeStages		= DeclSpec->PipeStages;
SIR_Symbol::Type		= Type;
SIR_Symbol::Notes		= NULL;
SIR_Symbol::Initializer		= NULL;
SIR_Symbol::EnumValue		= 0;
if (StoreParameters)
   { assert(Declarator->Parameters != NULL);
     assert(Declarator->ParamScope != NULL);
     SIR_Symbol::Parameters	= Declarator->Parameters;
     SIR_Symbol::ParamScope	= Declarator->ParamScope;
     Declarator->Parameters = NULL;
     Declarator->ParamScope = NULL;
     ParamScope->ParentSymbol = this;	/* automatically fill in back pointer */
    } /* fi */
else
   { SIR_Symbol::Parameters	= NULL;
     SIR_Symbol::ParamScope	= NULL;
    } /* esle */
SIR_Symbol::ClassScope		= NULL;
if (StoreParameters)
   { SIR_Symbol::Labels		= new SIR_Labels();	/* create label list */
    } /* fi */
else
   { SIR_Symbol::Labels		= NULL;
    } /* esle */
SIR_Symbol::FctBody		= NULL;
SIR_Symbol::Interfaces		= NULL;
SIR_Symbol::Events		= NULL;
SIR_Symbol::Period		= NULL;
SIR_Symbol::ResetSignal		= NULL;
SIR_Symbol::ResetActiveHi	= false;
SIR_Symbol::PortMappings	= NULL;
SIR_Symbol::Imported		= NULL;
SIR_Symbol::Alias		= NULL;

} /* end of SIR_Symbol::SIR_Symbol #6 */


SIR_Symbol::SIR_Symbol(			/* constructor #7 (special f. parser) */
	SIR_SYMBOLCLASS	Class,
	sir_declarator	*Declarator,
	sir_type	*Type,
	sir_symbol_ptrs	*Interfaces,
	bool		StoreParameters /* = false */) :
		SIR_Node(Declarator->Line, Declarator->FileInfo)
{

assert(  (Class == SIR_SYMBOL_BEHAVIOR)
       ||(Class == SIR_SYMBOL_CHANNEL)
       ||(Class == SIR_SYMBOL_INTERFACE));
assert(Declarator != NULL);
assert(Declarator->SymbolName != NULL);
assert(Type != NULL);

SIR_Symbol::Class		= Class;
SIR_Symbol::Name		= Declarator->SymbolName->chars();
SIR_Symbol::StorageClass	= SIR_STORAGE_NONE;
SIR_Symbol::PipeStages		= 0;
SIR_Symbol::Type		= Type;
SIR_Symbol::Notes		= NULL;
SIR_Symbol::Initializer		= NULL;
SIR_Symbol::EnumValue		= 0;
if (StoreParameters)
   { /* Declarator->Parameters is NULL for interfaces */
     assert(Declarator->ParamScope != NULL);
     SIR_Symbol::Parameters	= Declarator->Parameters;
     SIR_Symbol::ClassScope	= Declarator->ParamScope;
     Declarator->Parameters = NULL;
     Declarator->ParamScope = NULL;
     SIR_Symbol::ClassScope->ParentSymbol = this; /* fill in back pointer */
    } /* fi */
else
   { SIR_Symbol::Parameters	= NULL;
     SIR_Symbol::ClassScope	= NULL;
    } /* esle */
SIR_Symbol::ParamScope		= NULL;
SIR_Symbol::Labels		= NULL;
SIR_Symbol::FctBody		= NULL;
SIR_Symbol::Interfaces		= Interfaces;
SIR_Symbol::Events		= NULL;
SIR_Symbol::Period		= NULL;
SIR_Symbol::ResetSignal		= NULL;
SIR_Symbol::ResetActiveHi	= false;
SIR_Symbol::PortMappings	= NULL;
SIR_Symbol::Imported		= NULL;
SIR_Symbol::Alias		= NULL;

} /* end of SIR_Symbol::SIR_Symbol #7 */


SIR_Symbol::SIR_Symbol(			/* constructor #8 (duplicator) */
	sir_symbol	*Original) :
		SIR_Node(Original)
{

SIR_Symbol::Class		= Original->Class;
SIR_Symbol::Name		= Original->Name;
SIR_Symbol::StorageClass	= Original->StorageClass;
SIR_Symbol::PipeStages		= Original->PipeStages;
SIR_Symbol::Type		= Original->Type;
SIR_Symbol::Notes		= Original->Notes ?
					new SIR_Notes(
						Original->Notes) :
					NULL;
SIR_Symbol::Initializer		= Original->Initializer ?
					new SIR_Initializer(
						Original->Initializer) :
					NULL;
SIR_Symbol::EnumValue		= Original->EnumValue;
SIR_Symbol::Parameters		= Original->Parameters ?
					new SIR_Parameters(
						Original->Parameters) :
					NULL;
SIR_Symbol::ParamScope		= Original->ParamScope ?
					new SIR_Symbols(
						Original->ParamScope) :
					NULL;
SIR_Symbol::ClassScope		= Original->ClassScope ?
					new SIR_Symbols(
						Original->ClassScope) :
					NULL;
SIR_Symbol::Labels		= Original->Labels ?
					new SIR_Labels(
						Original->Labels) :
					NULL;
SIR_Symbol::FctBody		= Original->FctBody ?
					new SIR_Statement(
						Original->FctBody) :
					NULL;
SIR_Symbol::Interfaces		= Original->Interfaces ?
					new SIR_SymbolPtrs(
						Original->Interfaces) :
					NULL;
SIR_Symbol::Events		= Original->Events ?
					new SIR_Events(
						Original->Events) :
					NULL;
SIR_Symbol::Period		= Original->Period ?
					new SIR_Constant(
						Original->Period) :
					NULL;
SIR_Symbol::ResetSignal		= Original->ResetSignal;
SIR_Symbol::ResetActiveHi	= Original->ResetActiveHi;
SIR_Symbol::PortMappings	= Original->PortMappings ?
					new SIR_PortMaps(
						Original->PortMappings) :
					NULL;
SIR_Symbol::Imported		= Original->Imported;
SIR_Symbol::Alias		= NULL;

} /* end of SIR_Symbol::SIR_Symbol #8 */


SIR_Symbol::~SIR_Symbol(void)		/* destructor */
{

delete Notes;
delete Initializer;
delete Parameters;
delete ParamScope;
delete ClassScope;
delete Labels;
delete FctBody;
delete Interfaces;
delete Events;
delete Period;
delete PortMappings;

} /* end of SIR_Symbol::~SIR_Symbol */


bool SIR_Symbol::IsVariable(void)		/* check if IDENTIFIER */
{						/* and !TYPE_FUNCTION  */
						/* and !TYPE_BEHAVIOR  */
return(  (Class == SIR_SYMBOL_IDENTIFIER)	/* and !TYPE_CHANNEL   */
       &&(Type->Type != SIR_TYPE_FUNCTION)	/* and !TYPE_INTERFACE */
       &&(Type->Type != SIR_TYPE_METHOD)	/* and !TYPE_METHOD    */
       &&(Type->Type != SIR_TYPE_BEHAVIOR)	/* and Direction==NONE */
       &&(Type->Type != SIR_TYPE_CHANNEL)	/* and !IsEnumMember   */
       &&(Type->Type != SIR_TYPE_INTERFACE)
       &&(Type->Direction == SIR_PORT_NONE)
       &&(!(IsEnumMember())));

} /* end of SIR_Symbol::IsVariable */


bool SIR_Symbol::IsVariableDefinition(void)	/* check if IsVariable */
{						/* and !STORAGE_EXTERN */

return(  (IsVariable())
       &&(StorageClass != SIR_STORAGE_EXTERN));

} /* end of SIR_Symbol::IsVariableDefinition */


bool SIR_Symbol::IsFunction(void)		/* check if IDENTIFIER */
{						/* and TYPE_FUNCTION   */

return(  (Class == SIR_SYMBOL_IDENTIFIER)
       &&(  (Type->Type == SIR_TYPE_FUNCTION)
	  ||(Type->Type == SIR_TYPE_METHOD)));

} /* end of SIR_Symbol::IsFunction */


bool SIR_Symbol::IsFunctionDefinition(void)	/* check if IsFunction */
{						/* and FctBody!=NULL   */

return(  (IsFunction())
       &&(FctBody != NULL));

} /* end of SIR_Symbol::IsFunctionDefinition */


bool SIR_Symbol::IsBehaviorInstance(void)	/* check if IDENTIFIER */
{						/* and TYPE_BEHAVIOR   */
bool	Result;

Result = (  (Class == SIR_SYMBOL_IDENTIFIER)
	  &&(Type->Type == SIR_TYPE_BEHAVIOR));

if (Result)
   { assert(PortMappings != NULL);	/* no ports of type behavior */
    } /* fi */				/* (see below for channels) */

return(Result);

} /* end of SIR_Symbol::IsBehaviorInstance */


bool SIR_Symbol::IsChannelInstance(void)	/* check if IDENTIFIER */
{						/* and TYPE_CHANNEL    */

return(  (Class == SIR_SYMBOL_IDENTIFIER)
       &&(Type->Type == SIR_TYPE_CHANNEL));

} /* end of SIR_Symbol::IsChannelInstance */


bool SIR_Symbol::IsTypeDef(void)		/* check if TYPEDEF */
{

return(Class == SIR_SYMBOL_TYPEDEF);

} /* end of SIR_Symbol::IsTypeDef */


bool SIR_Symbol::IsBehavior(void)		/* check if BEHAVIOR */
{

return(Class == SIR_SYMBOL_BEHAVIOR);

} /* end of SIR_Symbol::IsBehavior */


bool SIR_Symbol::IsBehaviorDefinition(void)	/* check if IsBehavior  */
{						/* and ClassScope!=NULL */

return(  (IsBehavior())
       &&(ClassScope != NULL));

} /* end of SIR_Symbol::IsBehaviorDefinition */


bool SIR_Symbol::IsChannel(void)		/* check if CHANNEL */
{

return(Class == SIR_SYMBOL_CHANNEL);

} /* end of SIR_Symbol::IsChannel */


bool SIR_Symbol::IsChannelDefinition(void)	/* check if IsChannel   */
{						/* and ClassScope!=NULL */

return(  (IsChannel())
       &&(ClassScope != NULL));

} /* end of SIR_Symbol::IsChannelDefinition */


bool SIR_Symbol::IsInterface(void)		/* check if INTERFACE */
{

return(Class == SIR_SYMBOL_INTERFACE);

} /* end of SIR_Symbol::IsInterface */


bool SIR_Symbol::IsInterfaceDefinition(void)	/* check if IsInterface */
{						/* and ClassScope!=NULL */

return(  (IsInterface())
       &&(ClassScope != NULL));

} /* end of SIR_Symbol::IsInterfaceDefinition */


bool SIR_Symbol::IsClass(void)			/* check if BEHAVIOR or */
{						/* CHANNEL or INTERFACE */

return(  (Class == SIR_SYMBOL_BEHAVIOR)
       ||(Class == SIR_SYMBOL_CHANNEL)
       ||(Class == SIR_SYMBOL_INTERFACE));

} /* end of SIR_Symbol::IsClass */


bool SIR_Symbol::IsPort(void)			/* check if IDENTIFIER and */
{						/* TYPE_INTERFACE or       */
						/* Direction != PORT_NONE  */
return(  (Class == SIR_SYMBOL_IDENTIFIER)
       &&(Type->IsPort()));

} /* end of SIR_Symbol::IsPort */


bool SIR_Symbol::IsParameter(void)		/* check if defined in */
{						/* SCOPE_PARAMETER */

return(GetScope()->ScopeInfo == SIR_SCOPE_PARAMETER);

} /* end of SIR_Symbol::IsParameter */


bool SIR_Symbol::IsEnumMember(void)		/* check if this is an enum */
{						/* member (a constant) */

return(  (Class == SIR_SYMBOL_IDENTIFIER)
       &&(Type->Type == SIR_TYPE_ENUM)
       &&(  (EnumValue != 0)			// variables don't have value
	  ||(Type->UserType->Members == NULL)	// still under construction
	  ||(Type->UserType->Members->Find(this))));	// inefficient, but
							// the only way with
							// this data structure

} /* end of SIR_Symbol::IsEnumMember */


bool SIR_Symbol::IsDirectDependant(	/* checks if this depends on that */
	sir_symbol	*ThatSymbol,	/* (directly) */
	SIR_DEPENDENCY	*Reason /* = NULL */)
{
sir_symbol_ptr	*Interface;
sir_event	*Event;
sir_portmap	*PortMap;
sir_bitslice	*BitSlice;

assert(ThatSymbol != NULL);

if (this == ThatSymbol)	/* nothing depends on itself */
   { return(false);
    } /* fi */

assert(Type != NULL);
if (  (Type->ClassSymbol)
    &&(Type->ClassSymbol == ThatSymbol))
   { assert(  (IsBehaviorInstance())
	    ||(IsChannelInstance()));
     assert(ThatSymbol->IsClass());
     if (Reason)
	{ *Reason = SIR_DEP_INSTANTIATION;
	 } /* fi */
     return(true);	/* is an instance of that class */
    } /* fi */

/* FctBody may contain _indirect_ dependency */

if (Interfaces)
   { Interface = Interfaces->First();
     while(Interface)
	{ if (Interface->Symbol == ThatSymbol)
	     { if (Reason)
		  { *Reason = SIR_DEP_IMPLEMENTATION;
		   } /* fi */
	       return(true);		/* is implementing that interface */
	      } /* fi */
	  Interface = Interface->Succ();
	 } /* elihw */
    } /* fi */

if (Events)
   { Event = Events->First();
     while(Event)
	{ if (Event->Symbol == ThatSymbol)
	     { if (Reason)
		  { *Reason = SIR_DEP_CLOCK;
		   } /* fi */
	       return(true);	/* is buffered variable clocked by that event */
	      } /* fi */
	  Event = Event->Succ();
	 } /* elihw */
    } /* fi */

if (ResetSignal == ThatSymbol)
   { if (Reason)
	{ *Reason = SIR_DEP_RESET;
	 } /* fi */
     return(true);	/* is reset by that signal */
    } /* fi */

if (PortMappings)
   { PortMap = PortMappings->First();
     while(PortMap)
	{ assert(PortMap->BitSlices != NULL);
	  BitSlice = PortMap->BitSlices->First();
	  while(BitSlice)
	     { if (BitSlice->Symbol == ThatSymbol)
		  { if (Reason)
		       { *Reason = SIR_DEP_CONNECTION;
			} /* fi */
		    return(true);	/* is mapped to that symbol */
		   } /* fi */
	       BitSlice = BitSlice->Succ();
	      } /* elihw */
	  PortMap = PortMap->Succ();
	 } /* elihw */
    } /* fi */

return(false);	/* is independent */

} /* end of SIR_Symbol::IsDirectDependant */


bool SIR_Symbol::IsIndirectDependant(	/* checks if this depends on that */
	sir_symbol	*ThatSymbol,	/* (indirectly) */
	SIR_DEPENDENCY	*Reason /* = NULL */,
	sir_statement	**DepStmnt /* = NULL */,
	sir_fsmdstmnt	**DepFsmdStmnt /* = NULL */,
	sir_expression	**DepExpr /* = NULL */)
{

// note: "IsIndirectDependant" means
//       "is a function whose statements and/or expressions
//        depend on that symbol"

assert(ThatSymbol != NULL);

if (this == ThatSymbol)	/* nothing depends on itself */
   { return(false);
    } /* fi */

if (FctBody)
   { if (FctBody->DFS_FindIndirectDependant(ThatSymbol,
					DepStmnt, DepFsmdStmnt,
					DepExpr, Reason))
	{ return(true);
	 } /* fi */
    } /* fi */

return(false);	/* is independent */

} /* end of SIR_Symbol::IsIndirectDependant */


bool SIR_Symbol::MapsPortTo(		/* checks for port-instantiation */
	sir_symbol	*Symbol)	/* (instantiations only) */
{
sir_portmap	*PortMap;
sir_bitslice	*BitSlice;

assert(PortMappings != NULL);	/* must have a port map list */
PortMap = PortMappings->First();
while(PortMap)
   { assert(PortMap->BitSlices != NULL);	/* must have a mapping */
     BitSlice = PortMap->BitSlices->First();
     while(BitSlice)
	{ if (BitSlice->Symbol == Symbol)
	     { return(true);		/* port instance found */
	      } /* fi */
	  BitSlice = BitSlice->Succ();
	 } /* elihw */
     PortMap = PortMap->Succ();
    } /* elihw */

return(false);	/* not found */

} /* end of SIR_Symbol::MapsPortTo */


bool SIR_Symbol::Instantiates(		/* checks for instantiation */
	sir_symbol	*ClassSymbol)	/* (behaviors and channels only) */
{
sir_symbol	*Symbol;

if (! ClassScope)
   { return(false);	/* declaration only (black box) */
    } /* fi */

Symbol = ClassScope->First();
while(Symbol)
   { if (  (  (Symbol->IsBehaviorInstance())
	    ||(Symbol->IsChannelInstance()))
	 &&(Symbol->Type->ClassSymbol == ClassSymbol))
	{ return(true);		/* instantiation found */
	 } /* fi */
     Symbol = Symbol->Succ();
    } /* elihw */

return(false);	/* not found */

} /* end of SIR_Symbol::Instantiates */


bool SIR_Symbol::Contains(		/* checks for containment */
	sir_symbol	*ClassSymbol)	/* (behaviors and channels only) */
{
sir_symbol	*Symbol;

if (! ClassScope)
   { return(false);	/* declaration only (black box) */
    } /* fi */

Symbol = ClassScope->First();
while(Symbol)
   { if (  (Symbol->IsBehaviorInstance())
	 ||(Symbol->IsChannelInstance()))
	{ if (Symbol->Type->ClassSymbol == ClassSymbol)
	     { return(true);	/* instantiation found */
	      } /* fi */
	  if (Symbol->Type->ClassSymbol->Contains(ClassSymbol))	/* recursion! */
	     { return(true);	/* nested instantiation found */
	      } /* fi */
	 } /* fi */
     Symbol = Symbol->Succ();
    } /* elihw */

return(false);	/* not found */

} /* end of SIR_Symbol::Contains */


bool SIR_Symbol::IsTriggeredBy(		/* checks for clock/reset dependency */
	sir_symbol	*Symbol)
{

assert(Symbol != NULL);

if (Events)
   { if (Events->Find(Symbol))
	{ return(true);	// 'buffered' variable is clocked by Symbol
	 } /* fi */
    } /* fi */

if (ResetSignal == Symbol)
   { return(true);	// 'buffered' variable is reset by Symbol
    } /* fi */

return(false);

} /* end of SIR_Symbol::IsTriggeredBy */


bool SIR_Symbol::IsRedundantMethodDecl(	/* checks if method needs a decl. */
	sir_symbol_ptrs	*DefList,
	bool		WriteNotes)
{
sir_symbol_ptr	*OtherMethod;

assert(DefList != NULL);
assert(GetScope() != NULL);
assert(IsFunction());

if (GetScope()->ScopeInfo != SIR_SCOPE_CLASS)
   { return(false);	/* optimization makes only sense for class methods */
    } /* fi */

if (!IsFunctionDefinition())	/* is a declaration only */
   { return(false);	/* pure declarations are not redundant */
    } /* fi */

if (  (WriteNotes)
    &&(Notes)
    &&(! Notes->Empty()))
   { return(false);	/* declaration required to attach annotations to */
    } /* fi */

OtherMethod = DefList->First();	/* check all function definitions   */
while(OtherMethod)		/* for any dependencies on this one */
   { if (OtherMethod->Symbol == this)
	{ break;	/* reached this methods definition, stop here */
	 } /* fi */
     assert(OtherMethod->Symbol->IsDirectDependant(this)
				== false); /* cannot happen */
     if (OtherMethod->Symbol->IsIndirectDependant(this))
	{ return(false);	/* dependency found, declaration necessary! */
	 } /* fi */
     OtherMethod = OtherMethod->Succ();
    } /* elihw */

return(true);	/* no dependent method found that      */
		/* will be defined earlier as this one */

} /* end of SIR_Symbol::IsRedundantMethodDecl */


sir_symbols *SIR_Symbol::GetScope(void)	/* determines scope of this symbol */
{					/* (returns NULL if not in any table) */

return((sir_symbols*) Head());

} /* end of SIR_Symbol::GetScope */


sir_design *SIR_Symbol::GetDesign(	/* determines the design */
	void)			/* (returns NULL if type is not in design) */
{
sir_types	*TypeTable;

if (  (Type)
    &&(TypeTable = Type->GetTable()))
   { return(TypeTable->ThisDesign);
    } /* fi */

return(NULL);

} /* end of SIR_Symbol::GetDesign */


ERROR SIR_Symbol::DFS_ForAllNodes(	/* iterator over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{

if ((SIR_Error = (this->*MemberFct)(MemberFctArg)))	/* process this node */
   { return(SIR_Error);
    } /* fi */

if (Notes)
   { if ((SIR_Error = Notes->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Initializer)
   { if ((SIR_Error = Initializer->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (ParamScope)
   { if ((SIR_Error = ParamScope->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (ClassScope)
   { if ((SIR_Error = ClassScope->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Labels)
   { if ((SIR_Error = Labels->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (FctBody)
   { if ((SIR_Error = FctBody->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Period)
   { if ((SIR_Error = Period->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (PortMappings)
   { if ((SIR_Error = PortMappings->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbol::DFS_ForAllNodes */


ERROR SIR_Symbol::DFS_ForAllSymbols(	/* iterator over all symbols */
	sir_symbol_mptr	MemberFct,	/* (depth first) */
	sir_symbol_marg	MemberFctArg)
{
bool		IsLeaf;

IsLeaf = !(ParamScope || ClassScope || FctBody);

if ((SIR_Error = (this->*MemberFct)(MemberFctArg)))	/* process this node */
   { return(SIR_Error);
    } /* fi */

if (IsLeaf)
   { // it is allowed to delete leaf objects within this iterator;
     // in this case, we must not access any members any more,
     // because that would be a free-memory-access violation
     return(SIR_ERROR_NO_ERROR);
    } /* fi */

/* there are no symbols in Notes */
/* there are no symbols in Initializer */
/* there are no symbols in Parameters */
if (ParamScope)
   { if ((SIR_Error = ParamScope->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (ClassScope)
   { if ((SIR_Error = ClassScope->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no symbols in Labels */
if (FctBody)
   { if ((SIR_Error = FctBody->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no symbols in Interfaces */
/* there are no symbols in Events */
/* there are no symbols in Period */
/* there are no symbols in PortMappings */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbol::DFS_ForAllSymbols */


ERROR SIR_Symbol::DFS_ForAllUserTypes(	/* iterator over all usertypes */
	sir_usertp_mptr	MemberFct,	/* (depth first) */
	sir_usertp_marg	MemberFctArg)
{

/* this is not a usertype */

/* there are no usertypes in Notes */
/* there are no usertypes in Initializer */
/* there are no usertypes in Parameters */
if (ParamScope)
   { if ((SIR_Error = ParamScope->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (ClassScope)
   { if ((SIR_Error = ClassScope->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no usertypes in Labels */
if (FctBody)
   { if ((SIR_Error = FctBody->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no usertypes in Interfaces */
/* there are no usertypes in Events */
/* there are no usertypes in Period */
/* there are no usertypes in PortMappings */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbol::DFS_ForAllUserTypes */


ERROR SIR_Symbol::DFS_ForAllNotes(	/* iterator over all notes */
	sir_note_mptr	MemberFct,	/* (depth first) */
	sir_note_marg	MemberFctArg)
{

/* this is not a note */

if (Notes)
   { if ((SIR_Error = Notes->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no notes in Initializer */
/* there are no notes in Parameters */
if (ParamScope)
   { if ((SIR_Error = ParamScope->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (ClassScope)
   { if ((SIR_Error = ClassScope->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Labels)
   { if ((SIR_Error = Labels->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (FctBody)
   { if ((SIR_Error = FctBody->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no notes in Interfaces */
/* there are no notes in Events */
/* there are no notes in Period */
/* there are no notes in PortMappings */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbol::DFS_ForAllNotes */


ERROR SIR_Symbol::SetImported(	/* sets the Imported pointer (if unset) */
	sir_symbol_marg	ImportPtr)	/* or resets it if NULL */
{

if (ImportPtr == NULL)
   { Imported = NULL;
     return(SIR_ERROR_NO_ERROR);
    } /* fi */

if (! Imported)		/* import pointer still unset? */
   { Imported = (sir_import*) ImportPtr;
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbol::SetImported */


ERROR SIR_Symbol::TakeOverImport(	/* takes over an imported symbol */
	sir_symbol_marg	ImportPtr)
{

assert(ImportPtr != NULL);
if (Imported == (sir_import*) ImportPtr)
   { Imported = NULL;
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbol::TakeOverImport */


ERROR SIR_Symbol::MarkImportEntry(	/* marks its import entry being used */
	sir_symbol_marg	/* Unused */)
{

if (Imported)
   { Imported->Color = SIR_RED;	/* mark as being used */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbol::MarkImportEntry */


ERROR SIR_Symbol::MarkUsedTypes( /* marks the type entries used here */
	sir_symbol_marg	/* Unused */)
{
sir_parameter	*Para;

Type->MarkUsed();

if (Parameters)
   { Para = Parameters->First();
     while(Para)
	{ Para->Type->MarkUsed();
	  Para = Para->Succ();
	 } /* elihw */
    } /* fi */

if (FctBody)
   { FctBody->DFS_ForAllExpressions(
		&SIR_Expression::MarkUsedTypes, NULL);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbol::MarkUsedTypes */


bool SIR_Symbol::DFS_FindDependant(	/* searches for dependants (DFS) */
	sir_symbol	**SkipSymbol,	/* (internal helper function) */
	sir_symbol	*ThatSymbol,
	sir_symbol	**DepSymbol,
	sir_statement	**DepStmnt,
	sir_fsmdstmnt	**DepFsmdStmnt,
	sir_expression	**DepExpr,
	SIR_DEPENDENCY	*Reason)
{

assert(SkipSymbol != NULL);
assert(ThatSymbol != NULL);

if (*SkipSymbol)
   { if (this == *SkipSymbol)
	{ *SkipSymbol = NULL;	/* no more skipping */
	 } /* fi */
    } /* fi */
else
   { if (IsDirectDependant(ThatSymbol, Reason))	/* check this symbol */
	{ if (DepSymbol)			/* (for direct dep.) */
	     { *DepSymbol = this;
	      } /* fi */
	  if (DepStmnt)
	     { *DepStmnt = NULL;
	      } /* fi */
	  if (DepFsmdStmnt)
	     { *DepFsmdStmnt = NULL;
	      } /* fi */
	  if (DepExpr)
	     { *DepExpr = NULL;
	      } /* fi */
	  return(true);
	 } /* fi */

     if (IsIndirectDependant(ThatSymbol, Reason,/* check this symbol */
			DepStmnt, DepFsmdStmnt,	/* (for indirect dep.) */
			DepExpr))
	{ if (DepSymbol)
	     { *DepSymbol = this; /* report the indirect dep. symbol */
	      } /* fi */
	  return(true);
	 } /* fi */
    } /* esle */

/* there are no symbols in Notes */
/* there are no symbols in Initializer */
/* there are no symbols in Parameters */
/* symbols in ParamScope cannot depend on other symbols */

if (ClassScope)
   { if (ClassScope->DFS_FindDependant(SkipSymbol, ThatSymbol,
					DepSymbol, DepStmnt, DepFsmdStmnt,
					DepExpr, Reason))
	{ return(true);
	 } /* fi */
    } /* fi */

/* there are no symbols in Labels */

if (FctBody)
   { if (FctBody->DFS_FindDirectDependant(SkipSymbol, ThatSymbol,
						DepSymbol, Reason))
	{ if (DepStmnt)
	     { *DepStmnt = NULL;
	      } /* fi */
	  if (DepFsmdStmnt)
	     { *DepFsmdStmnt = NULL;
	      } /* fi */
	  if (DepExpr)
	     { *DepExpr = NULL;
	      } /* fi */
	  return(true);
	 } /* fi */
    } /* fi */

/* there are no symbols in Interfaces */
/* there are no symbols in Events */
/* there are no symbols in Period */
/* there are no symbols in PortMappings */

return(false);

} /* end of SIR_Symbol::DFS_FindDependant */


bool SIR_Symbol::FindDependant(	/* finds symbols that depend on this one #1 */
	sir_symbols	*SearchScope,		/* search in this scope */
	sir_symbol	**DepSymbol /* = NULL */, /* return symbol found */
	sir_statement	**DepStmnt /* = NULL */,  /* return statement found */
	sir_fsmdstmnt	**DepFsmdStmnt /* = NULL */,/* return fsmdstmnt.found */
	sir_expression	**DepExpr /* = NULL */,   /* return expression found */
	SIR_DEPENDENCY	*Reason /* = NULL */,    /* return type of dependency */
	sir_symbol	*SearchAfter /* = NULL */) /* continue search here */
{
sir_symbols	*RootScope;
sir_symbol	*SkipSymbol;

if (SearchScope)			/* determine where to search */
   { assert(SearchScope->ScopeInfo
		!= SIR_SCOPE_USERTYPE);	/* we don't go there! */
     RootScope = SearchScope;
    } /* fi */
else
   { RootScope = GetScope();
     while(RootScope->Parent)
	{ RootScope = RootScope->Parent;
	 } /* elihw */
    } /* esle */

if (SearchAfter)			/* determine starting point */
   { assert(SearchAfter->GetScope()->ScopeInfo
		!= SIR_SCOPE_USERTYPE);	/* we don't go there! */
     SkipSymbol = SearchAfter;
    } /* fi */
else
   { SkipSymbol = NULL;	/* don't skip */
    } /* esle */

if (RootScope->DFS_FindDependant(	/* recursive search */
		&SkipSymbol, this,
		DepSymbol, DepStmnt,
		DepFsmdStmnt, DepExpr, Reason))
   { return(true);	/* found one! */
    } /* fi */

/* no dependant found! */

if (DepSymbol)
   { *DepSymbol = NULL;
    } /* fi */
if (DepStmnt)
   { *DepStmnt = NULL;
    } /* fi */
if (DepFsmdStmnt)
   { *DepFsmdStmnt = NULL;
    } /* fi */
if (DepExpr)
   { *DepExpr = NULL;
    } /* fi */
if (Reason)
   { *Reason = SIR_DEP_NONE;
    } /* fi */

return(false);

} /* end of SIR_Symbol::FindDependant #1 */


bool SIR_Symbol::FindDependant(	/* finds symbols that depend on this one #2 */
	sir_symbol	*RootSymbol,		/* search in this sub-tree */
	sir_symbol	**DepSymbol /* = NULL */, /* return symbol found */
	sir_statement	**DepStmnt /* = NULL */,  /* return statement found */
	sir_fsmdstmnt	**DepFsmdStmnt /* = NULL */,/* return fsmdstmnt.found */
	sir_expression	**DepExpr /* = NULL */,   /* return expression found */
	SIR_DEPENDENCY	*Reason /* = NULL */,    /* return type of dependency */
	sir_symbol	*SearchAfter /* = NULL */) /* continue search here */
{
sir_symbol	*SkipSymbol;

assert(RootSymbol != NULL);
assert(RootSymbol->GetScope()->ScopeInfo
		!= SIR_SCOPE_USERTYPE);	/* we don't go there! */

if (SearchAfter)			/* determine starting point */
   { assert(SearchAfter->GetScope()->ScopeInfo
		!= SIR_SCOPE_USERTYPE);	/* we don't go there! */
     SkipSymbol = SearchAfter;
    } /* fi */
else
   { SkipSymbol = NULL;	/* don't skip */
    } /* esle */

if (RootSymbol->DFS_FindDependant(	/* recursive search */
		&SkipSymbol, this,
		DepSymbol, DepStmnt,
		DepFsmdStmnt, DepExpr, Reason))
   { return(true);	/* found one! */
    } /* fi */

/* no dependant found! */

if (DepSymbol)
   { *DepSymbol = NULL;
    } /* fi */
if (DepStmnt)
   { *DepStmnt = NULL;
    } /* fi */
if (DepFsmdStmnt)
   { *DepFsmdStmnt = NULL;
    } /* fi */
if (DepExpr)
   { *DepExpr = NULL;
    } /* fi */
if (Reason)
   { *Reason = SIR_DEP_NONE;
    } /* fi */

return(false);

} /* end of SIR_Symbol::FindDependant #2 */


void SIR_Symbol::SetAlias(	/* sets all type, usertype, symbol alias' */
	sir_symbol	*Alias)	/* (iterates over symbols and usertypes) */
{

assert(Alias != NULL);

this->Alias = Alias;	/* process this node */

/* there are no symbols or usertypes in Notes */
/* there are no symbols or usertypes in Initializer */
/* there are no symbols or usertypes in Parameters */
if (ParamScope)
   { ParamScope->SetAlias(Alias->ParamScope);
    } /* fi */
if (ClassScope)
   { ClassScope->SetAlias(Alias->ClassScope);
    } /* fi */
/* there are no symbols or usertypes in Labels */
if (FctBody)
   { FctBody->SetAlias(Alias->FctBody);
    } /* fi */
/* there are no symbols or usertypes in Interfaces */
/* there are no symbols or usertypes in PortMappings */

} /* end of SIR_Symbol::SetAlias */


void SIR_Symbol::UnAlias(	/* unalias all type, usertype, symbol links */
	sir_symbols	*GlobalSymbols)
{

if (Type->Alias)
   { Type = Type->Alias;
    } /* fi */
/* there is nothing to unalias in Notes (it's built-in) */
/* there is nothing to unalias in Initializer */
if (Parameters)
   { Parameters->UnAlias();
    } /* fi */
if (ParamScope)
   { ParamScope->UnAlias(GlobalSymbols);
    } /* fi */
if (ClassScope)
   { ClassScope->UnAlias(GlobalSymbols);
    } /* fi */
/* there is nothing to unalias in Labels */
if (FctBody)
   { FctBody->UnAlias(GlobalSymbols);
    } /* fi */
if (Interfaces)
   { Interfaces->UnAlias();
    } /* fi */
if (PortMappings)
   { PortMappings->UnAlias();
    } /* fi */

} /* end of SIR_Symbol::UnAlias */


const char *SIR_Symbol::PrintStorageClass(	/* prints storage class #1 */
	bool		AppendSpace /* = false */)  /* "extern" or "extern " */
{

return(PrintStorageClass(StorageClass, PipeStages, AppendSpace));

} /* end of SIR_Symbol::PrintStorageClass */


const char *SIR_Symbol::PrintStorageClass(	/* prints storage class #2 */
	SIR_STORAGE	StorageClass,	      /* (eg. "extern" or "extern ") */
	int		PipeStages /* = 0 */,
	bool		AppendSpace /* = false */)
{
static gl_string	Buffer;
int		i;

switch(StorageClass)
   { case SIR_STORAGE_NONE:
	{ return("");
	 }
     case SIR_STORAGE_TYPEDEF:
	{ if (AppendSpace)
	     { return("typedef ");
	      } /* fi */
	  else
	     { return("typedef");
	      } /* esle */
	 }
     case SIR_STORAGE_EXTERN:
	{ if (AppendSpace)
	     { return("extern ");
	      } /* fi */
	  else
	     { return("extern");
	      } /* esle */
	 }
     case SIR_STORAGE_STATIC:
	{ if (AppendSpace)
	     { return("static ");
	      } /* fi */
	  else
	     { return("static");
	      } /* esle */
	 }
     case SIR_STORAGE_AUTO:
	{ if (AppendSpace)
	     { return("auto ");
	      } /* fi */
	  else
	     { return("auto");
	      } /* esle */
	 }
     case SIR_STORAGE_REGISTER:
	{ if (AppendSpace)
	     { return("register ");
	      } /* fi */
	  else
	     { return("register");
	      } /* esle */
	 }
     case SIR_STORAGE_PIPED:
	{ Buffer = "";
	  for(i=0; i<PipeStages-1; i++)
	     { Buffer += "piped ";
	      } /* rof */
	  if (AppendSpace)	/* even if PipeStages = default = 0 print one */
	     { Buffer += "piped ";
	      } /* fi */
	  else
	     { Buffer += "piped";
	      } /* esle */
	  break;
	 }
     default:
	{ assert(false);	/* bad storage class */
	 }
    } /* hctiws */

return(Buffer.chars());

} /* end of SIR_Symbol::PrintStorageClass */


ERROR SIR_Symbol::WriteSC(	/* (re-) generates SpecC source code */
	gl_io	*IO,	/* (for non-class symbols) */
	bool		WriteNotes,
	bool		AsDeclaration /* = false */,
	bool		CplusplusMode /* = false */,
	bool		AsReference /* = false */)
{
gl_string		Buffer;
bool		PutLineInfoHere = false,
		PutNotesHere = false;
bool		MakePureVirtual;
bool		InitByConstructor = false,
		ConstructorParensOpen = false;

assert(! IsPort());	/* ports are written with the class */

switch(Class)
   { case SIR_SYMBOL_IDENTIFIER:
	{ switch(Type->Type)
	     { case SIR_TYPE_FUNCTION:
	       case SIR_TYPE_METHOD:
		  { assert(AsDeclaration == true); /* fct.def. not supported */
		    if (IsFunctionDefinition())
		       { PutLineInfoHere	= false;
			 PutNotesHere		= true;
			} /* fi */
		    else
		       { PutLineInfoHere	= true;
			 PutNotesHere		= true;
			} /* esle */
		    break;
		   }
	       case SIR_TYPE_BEHAVIOR:
	       case SIR_TYPE_CHANNEL:
		  { assert(AsDeclaration == false); /* no instance decl. */
		    PutLineInfoHere	= true;
		    PutNotesHere	= true;
		    break;
		   }
	       default:	/* any other TYPE (standard variables) */
		  { assert(  ((AsDeclaration) && (!IsVariableDefinition()))
			   ||((!AsDeclaration) && (IsVariableDefinition())));
		    PutLineInfoHere	= true;
		    PutNotesHere	= true;
		    break;
		   }
	      } /* hctiws */
	  break;
	 }
     case SIR_SYMBOL_TYPEDEF:
	{ PutLineInfoHere	= true;
	  PutNotesHere		= true;
	  break;
	 }
     case SIR_SYMBOL_BEHAVIOR:
     case SIR_SYMBOL_CHANNEL:
     case SIR_SYMBOL_INTERFACE:
	{ assert(false);	/* classes not supported */
 	 }
     default:
	{ assert(false);	/* bad symbol class */
	 }
    } /* hctiws */

MakePureVirtual = (  (CplusplusMode)
		   &&(GetScope())
		   &&(GetScope()->ScopeInfo == SIR_SCOPE_CLASS)
		   &&(GetScope()->ParentSymbol->Class == SIR_SYMBOL_INTERFACE));

if (PutLineInfoHere)
   { if (LineInfo)
	{ if ((SIR_Error = LineInfo->WriteSC(IO, true)))	/* old stuff */
	     { return(SIR_Error);				/* (by user) */
	      } /* fi */
	 } /* fi */
     else
	{ SIR_LineInfo::WriteNewSC(IO, true, false,		/* new stuff */
					&LineInfo);		/* (by tools) */
	 } /* esle */
    } /* fi */
else
   { SIR_LineInfo::WriteNewSC(IO, true);	/* new or old stuff (decls) */
    } /* esle */

if (CplusplusMode)
   { if (  (  (StorageClass == SIR_STORAGE_EXTERN)
	    ||(StorageClass == SIR_STORAGE_NONE))
	 &&(IsFunction())
	 &&(FctBody==NULL)
	 &&(GetScope()->ScopeInfo == SIR_SCOPE_GLOBAL))
	{ /*** C++: function declarations ***/
	  assert(AsReference == false);
	  IO->PutS("extern \"C\" ");	/* C linkage! */
	  Buffer = Name;
	  IO->PutS(Type->PrettyString(&Buffer, WriteNotes, CplusplusMode));
	 } /* fi */
     else
	{ /*** C++: piped variables ***/
	  if (StorageClass == SIR_STORAGE_PIPED)
	     { assert(AsReference == false);
	       if (Initializer)
		  { InitByConstructor = true;
		    if (Initializer->IsComposite(Type->Type == SIR_TYPE_ARRAY))
		       { /* initialization by constructor call with dummy */
			 if (GetScope()->ScopeInfo == SIR_SCOPE_CLASS)
			    { IO->PutS("static ");
			      Buffer = SIR_CXX_ARRAY_INIT_NAME;
			      Buffer += Name;
			      IO->PutS(Type->PrettyString(&Buffer, false,
							CplusplusMode));
			     } /* fi */
			 else
			    { Buffer = SIR_CXX_ARRAY_INIT_NAME;
			      Buffer += Name;
			      IO->PutS(Type->PrettyString(&Buffer, false,
							CplusplusMode));
			      IO->PutS(" = ");
			      if ((SIR_Error = Initializer->WriteSC(IO,
							CplusplusMode)))
				 { return(SIR_Error);
				  } /* fi */
			     } /* esle */
			 IO->PutS("; ");
			 SIR_LineInfo::WriteVSPACE(IO, true);
			} /* fi */
		   } /* fi */
	       IO->PutS(SIR_SIM_PIPED_NAME "<");
	       Buffer = "";
	       if (  (Type->Type == SIR_TYPE_ARRAY)
		   &&(Type->Size == SIR_UNKNOWN_ARRAY_SIZE)
		   &&(Initializer))
		  { /* substitute unknown array size with size from init. */
		    IO->PutS(Type->PrettyString(&Buffer, WriteNotes,
					CplusplusMode, false, false,
					Initializer->GetAutoArraySize()));
		   } /* fi */
	       else
		  { IO->PutS(Type->PrettyString(&Buffer, WriteNotes,
							CplusplusMode));
		   } /* esle */
	       assert(PipeStages >= 1);
	       IO->PrintF(",%d> ", PipeStages);
	       IO->PutS(Name);
	      } /* fi */
	  else
	     { if (  (Type->Type == SIR_TYPE_SIGNAL)
		   ||(Type->Type == SIR_TYPE_BUFFER))
		  { /*** C++: 'signal' and 'buffered' variables ***/
		    if (Initializer)
		       { InitByConstructor = true;
			 if (Initializer->IsComposite(
				Type->SubType->Type == SIR_TYPE_ARRAY))
			    { /* initialization by constructor call with dummy*/
			      if (GetScope()->ScopeInfo == SIR_SCOPE_CLASS)
				 { IO->PutS("static ");
				   Buffer = SIR_CXX_ARRAY_INIT_NAME;
				   Buffer += Name;
				   IO->PutS(Type->SubType->PrettyString(
						&Buffer, false, CplusplusMode));
				  } /* fi */
			      else
				 { Buffer = SIR_CXX_ARRAY_INIT_NAME;
				   Buffer += Name;
				   IO->PutS(Type->SubType->PrettyString(
						&Buffer, false, CplusplusMode));
				   IO->PutS(" = ");
				   if ((SIR_Error = Initializer->WriteSC(IO,
								CplusplusMode)))
				      { return(SIR_Error);
				       } /* fi */
				  } /* esle */
			      IO->PutS("; ");
			      SIR_LineInfo::WriteVSPACE(IO, true);
			     } /* fi */
			} /* fi */
		    if (Period)
		       { IO->PutS(SIR_SIM_AUTO_EVENT_NAME " "
					SIR_CXX_AUTO_EVENT_NAME);
			 IO->PutS(Name);
			 if (GetScope()->ScopeInfo != SIR_SCOPE_CLASS)
			    { IO->PutC('(');
			      IO->PutS(Period->Print(true));
			      IO->PutC(')');
			     } /* fi */
			 IO->PutS("; ");
			 SIR_LineInfo::WriteVSPACE(IO, true);
			} /* fi */
		    IO->PutS(PrintStorageClass(true));
		    assert(MakePureVirtual == false);
		    if (AsReference)
		       { Buffer = "(&";
			 Buffer += Name;
			 Buffer += ")";
			} /* fi */
		    else
		       { Buffer = Name;
			} /* esle */
		    IO->PutS(Type->PrettyString(&Buffer, WriteNotes,
							CplusplusMode));
		    if (  (  (Events)
			   ||(Period))
			&&(GetScope()->ScopeInfo != SIR_SCOPE_CLASS))
		       { IO->PutC('(');
			 if (Events)
			    { if ((SIR_Error = Events->WriteCC(IO)))
				 { return(SIR_Error);
				  } /* fi */
			     } /* fi */
			 else
			    { IO->PutS(SIR_SIM_ANY_SIG_PFX
					SIR_CXX_AUTO_EVENT_NAME);
			      IO->PutS(Name);
			      IO->PutS(")");
			     } /* esle */
			 if (ResetSignal)
			    { IO->PutS(", &");
			      IO->PutS(ResetSignal->Name);
			      IO->PutS(", ");
			      if (ResetActiveHi)
				 { IO->PutS("true");
				  } /* fi */
			      else
				 { IO->PutS("false");
				  } /* esle */
			     } /* fi */
			 if (Initializer)
			    { IO->PutS(", ");
			      ConstructorParensOpen = true;
			     } /* fi */
			 else
			    { IO->PutC(')');
			     } /* esle */
			} /* fi */
		   } /* fi */
	       else
		  { /*** C++: other symbols ***/
		    sir_type	*FixedType;

		    IO->PutS(PrintStorageClass(true));
		    if (MakePureVirtual)
		       { IO->PutS("virtual ");
			} /* fi */
		    if (AsReference)
		       { Buffer = "(&";
			 Buffer += Name;
			 Buffer += ")";
			} /* fi */
		    else
		       { Buffer = Name;
			} /* esle */
		    FixedType = FixConstArrayTypeInClass( /* 04/02/03, RD */
					Type, GetScope()->ScopeInfo);
		    if (  (FixedType->Type == SIR_TYPE_ARRAY)
			&&(FixedType->Size == SIR_UNKNOWN_ARRAY_SIZE)
			&&(Initializer)
			&&(GetScope()->ScopeInfo == SIR_SCOPE_CLASS))
		       { /* substitute unknown array size with size from init.*/
			 IO->PutS(FixedType->PrettyString(&Buffer, WriteNotes,
					CplusplusMode, false, false,
					Initializer->GetAutoArraySize()));
			} /* fi */
		    else
		       { IO->PutS(FixedType->PrettyString(&Buffer, WriteNotes,
							CplusplusMode));
			} /* esle */
		   } /* esle */
	      } /* fi */
	 } /* fi */
    } /* fi */
else
   { /*** SpecC: all symbols ***/
     assert(AsReference == false);
     IO->PutS(PrintStorageClass(true));
     Buffer = Name;
     if (  (Events) /* 'buffered' parameters are printed at the type in SpecC */
	 ||(Period))
	{ assert(Type->Type == SIR_TYPE_BUFFER);
	  assert(0 == strcmp(Type->Name, "buffered"));
	  IO->PutS(Type->Name);
	  IO->PutC('[');
	  if (Events)
	     { if ((SIR_Error = Events->WriteSC(IO)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  else
	     { IO->PutS(Period->Print());
	      } /* esle */
	  if (ResetSignal)
	     { IO->PutS("; ");
	       if (! ResetActiveHi)
		  { IO->PutC('!');
		   } /* fi */
	       IO->PutS(ResetSignal->Name);
	      } /* fi */
	  IO->PutS("] ");
	  assert(Type->SubType != NULL);
	  IO->PutS(Type->SubType->PrettyString(&Buffer, WriteNotes));
	 } /* fi */
     else
	{ IO->PutS(Type->PrettyString(&Buffer, WriteNotes));
	 } /* esle */
    } /* esle */

if (MakePureVirtual)	/*** C++: pure virtual methods ***/
   { assert(Initializer == NULL); /* would collide */
     IO->PutS(" = 0");
    } /* fi */

if (  (Initializer)	/*** C++ and SpecC: initializer ***/
    &&(! AsDeclaration)
    &&(  (! CplusplusMode)
       ||((CplusplusMode)&&(GetScope()->ScopeInfo != SIR_SCOPE_CLASS))))
   { if (InitByConstructor)
	{ if (!ConstructorParensOpen)
	     { IO->PutC('(');
	      } /* fi */
	  if (Initializer->InitList)
	     { IO->PutS(SIR_CXX_ARRAY_INIT_NAME);
	       IO->PutS(Name);
	      } /* fi */
	  else
	     { if ((SIR_Error = Initializer->WriteSC(IO, CplusplusMode)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* esle */
	  IO->PutC(')');
	 } /* fi */
     else
	{ IO->PutS(" = ");
	  if ((SIR_Error = Initializer->WriteSC(IO, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	 } /* else */
    } /* fi */

if (  (! CplusplusMode)	/*** SpecC: port mappings ***/
    &&(  (IsBehaviorInstance())
       ||(IsChannelInstance())))
   { assert(PortMappings != NULL);	/* port mapping */
     PortMappings->WriteSC(IO);
    } /* fi */

IO->PutC(';');
if ((SIR_Error = IO->Check()))
   { return(SIR_Error);
    } /* fi */

if (PutNotesHere)
   { if (Notes)
	{ if ((SIR_Error = Notes->WriteSC(IO, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbol::WriteSC */


ERROR SIR_Symbol::WriteSC2(	/* (re-) generates SpecC source code */
	gl_io		*IO,	/* (for class symbols) */
	bool		WriteNotes,
	bool		AsDeclaration /* = false */)
{
const char	*ClassName = NULL;
gl_string	Buffer;
sir_parameter	*Port;
sir_type_ptr	*PortType;
sir_symbol_ptr	*Interface;
sir_symbol	*ParamSymbol;

if (  ((AsDeclaration) && (ClassScope == NULL))
    ||((! AsDeclaration) && (ClassScope != NULL)))
   { if (LineInfo)
	{ if ((SIR_Error = LineInfo->WriteSC(IO, true)))	/* old stuff */
	     { return(SIR_Error);
	      } /* fi */
	 } /* fi */
     else
	{ SIR_LineInfo::WriteNewSC(IO, true, false, &LineInfo);	/* new stuff */
	 } /* esle */
    } /* fi */
else
   { SIR_LineInfo::WriteNewSC(IO, true);	/* old or new stuff */
    } /* esle */

switch(Class)
   { case SIR_SYMBOL_IDENTIFIER:
     case SIR_SYMBOL_TYPEDEF:
	{ assert(false);	/* this is for classes only */
	 }
     case SIR_SYMBOL_BEHAVIOR:
	{ ClassName = "behavior ";
	  break;
	 }
     case SIR_SYMBOL_CHANNEL:
	{ ClassName = "channel ";
	  break;
	 }
     case SIR_SYMBOL_INTERFACE:
	{ ClassName = "interface ";
	  break;
	 }
     default:
	{ assert(false);	/* bad symbol class */
	 }
    } /* hctiws */
IO->PutS(ClassName);
IO->PutS(Name);

if (Class != SIR_SYMBOL_INTERFACE)
   { IO->PutC('(');
     IO->TabStepUp();
     if (AsDeclaration)
	{ PortType = Type->Parameters->First();
	  while(PortType)
	     { SIR_LineInfo::WriteNL(IO);
	       Buffer = "";
	       IO->PutS(PortType->Type->PrettyString(&Buffer, WriteNotes));
	       PortType = PortType->Succ();
	       if (PortType)
		  { IO->PutC(',');
		   } /* fi */
	      } /* elihw */
	 } /* fi */
     else
	{ assert(Parameters != NULL);
	  Port = Parameters->First();
	  while(Port)
	     { if (Port->Symbol)
		  { if (Port->Symbol->LineInfo)
		       { if ((SIR_Error = Port->Symbol->LineInfo->WriteSC(
								IO, true)))
			    { return(SIR_Error);
			     } /* fi */
			} /* fi */
		    else
		       { SIR_LineInfo::WriteSPACE(IO);
			} /* esle */
		    Buffer = Port->Symbol->Name;
		   } /* fi */
	       else
		  { SIR_LineInfo::WriteSPACE(IO);
		    Buffer = "";
		   } /* esle */
	       IO->PutS(Port->Type->PrettyString(&Buffer, WriteNotes));
	       Port = Port->Succ();
	       if (Port)
		  { IO->PutC(',');
		   } /* fi */
	      } /* elihw */
	 } /* esle */
     IO->TabStepDown();
     IO->PutC(')');
    } /* fi */

if (  (Interfaces)
    &&(! Interfaces->Empty()))
   { IO->PutS(" implements ");
     Interface = Interfaces->First();
     while(Interface)
	{ IO->PutS(Interface->Symbol->Name);
	  Interface = Interface->Succ();
	  if (Interface)
	     { IO->PutS(", ");
	      } /* fi */
	 } /* elihw */
    } /* fi */

if (  (! AsDeclaration)
    &&(ClassScope))
   { SIR_LineInfo::WriteNL(IO);
     IO->PutC('{');
     IO->TabStepUp();
     ParamSymbol = ClassScope->First();
     while(ParamSymbol)
	{ if (  (ParamSymbol->IsPort())
	      &&(ParamSymbol->Notes))
	     { if ((SIR_Error = ParamSymbol->Notes->WriteSC(IO, WriteNotes)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  ParamSymbol = ParamSymbol->Succ();
	 } /* elihw */
     if ((SIR_Error = ClassScope->WriteSC(IO, WriteNotes)))
	{ return(SIR_Error);
	 } /* fi */
     IO->TabStepDown();
     SIR_LineInfo::WriteNL(IO);
     IO->PutC('}');
    } /* fi */

IO->PutC(';');

if ((SIR_Error = IO->Check()))
   { return(SIR_Error);
    } /* fi */

if (AsDeclaration)
   { if (Notes)
	{ if ((SIR_Error = Notes->WriteSC(IO, WriteNotes)))
	     { return(SIR_Error);
	      } /* fi */
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbol::WriteSC2 */


ERROR SIR_Symbol::WriteSC3(	/* (re-) generates SpecC source code */
	gl_io		*IO,	/* (for function definitions) */
	bool		WriteNotes,
	bool		CplusplusMode /* = false */)
{
gl_string	Prefix,
		Postfix,
		ParamBuffer;
sir_parameter	*Parameter;

assert(IsFunctionDefinition());	/* doesn't support anything else */

if (LineInfo)
   { if ((SIR_Error = LineInfo->WriteSC(IO, true)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
else
   { SIR_LineInfo::WriteNewSC(IO, true, false, &LineInfo);
    } /* esle */

IO->PutS(PrintStorageClass(true));
Prefix = "";
Postfix = "";
Type->SubType->PrettyString(&Prefix, &Postfix, WriteNotes, CplusplusMode);
IO->PutS(Prefix);
if (  (CplusplusMode)
    &&(GetScope()->ScopeInfo == SIR_SCOPE_CLASS))
   { IO->PutS(GetScope()->ParentSymbol->Name);
     IO->PutS("::");
    } /* fi */
IO->PutS(Name);
IO->PutC('(');
IO->TabStepUp();
Parameter = Parameters->First();
while(Parameter)
   { if (Parameter->Symbol)
	{ if (Parameter->Symbol->LineInfo)
	     { if ((SIR_Error = Parameter->Symbol->LineInfo->WriteSC(IO)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  IO->PutS(Parameter->Symbol->PrintStorageClass(true));
	  ParamBuffer = Parameter->Symbol->Name;
	 } /* fi */
     else
	{ ParamBuffer = "";
	 } /* esle */
     if (  (CplusplusMode)
	 &&(Parameter->Type->Type == SIR_TYPE_INTERFACE))
	{ ParamBuffer.prepend("(&");	/* interface parameter as reference */
	  ParamBuffer += ")";
	 } /* fi */
     IO->PutS(Parameter->Type->PrettyString(&ParamBuffer, WriteNotes,
						CplusplusMode));
     Parameter = Parameter->Succ();
     if (Parameter)
	{ IO->PutS(", ");
	  SIR_LineInfo::WrapLine(IO);
	 } /* fi */
    } /* elihw */
IO->TabStepDown();
IO->PutC(')');
IO->PutS(Postfix);

if ((SIR_Error = IO->Check()))
   { return(SIR_Error);
    } /* fi */

IO->TabStepUp();
if ((SIR_Error = FctBody->WriteSC(IO, WriteNotes, CplusplusMode)))
   { return(SIR_Error);
    } /* fi */
IO->TabStepDown();

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbol::WriteSC3 */


ERROR SIR_Symbol::WriteCC2(		/* generates C++ source code */
	gl_io		*IO,		/* (for class symbols) */
	bool		WriteNotes,
	bool		AsDeclaration /* = false */)
{
gl_string	Buffer,
		ParamBuffer;
sir_parameter	*Port;
sir_type_ptr	*DeclaredPort;
sir_symbol_ptr	*Interface;
sir_symbol	*ParamSymbol,
		*Instance,
		*Symbol;
sir_portmap	*PortMap;
int		BitBusNum,
		OpenPortNum,
		ConstPortNum;
bool		PutSeparator = false;

assert(IsClass());	/* nothing else supported */

if (  ((AsDeclaration) && (ClassScope == NULL))
    ||((! AsDeclaration) && (ClassScope != NULL)))
   { if (LineInfo)
	{ if ((SIR_Error = LineInfo->WriteSC(IO, true)))	/* old stuff */
	     { return(SIR_Error);
	      } /* fi */
	 } /* fi */
     else
	{ SIR_LineInfo::WriteNewSC(IO, true, false, &LineInfo);	/* new stuff */
	 } /* esle */
    } /* fi */
else
   { SIR_LineInfo::WriteNewSC(IO, true);	/* old or new stuff */
    } /* esle */

IO->PutS("class ");
IO->PutS(Name);

if (  (! AsDeclaration)
    &&(ClassScope))
   { switch(Class)				/* base classes */
	{ case SIR_SYMBOL_IDENTIFIER:
	  case SIR_SYMBOL_TYPEDEF:
	     { assert(false);	/* this is for classes only */
	      }
	  case SIR_SYMBOL_BEHAVIOR:
	     { if (0 != strcmp(Name.chars(), GL_MAIN_BEHAVIOR_NAME))
		  { IO->PutS(" : public " SIR_SIM_BEHAVIOR_NAME);
		   } /* fi */
	       assert(Interfaces != NULL);	/* is a behavior */
	       if (! Interfaces->Empty())
		  { Interface = Interfaces->First();
		    while(Interface)
		       { IO->PutS(", public ");
			 IO->PutS(Interface->Symbol->Name);
			 Interface = Interface->Succ();
			} /* elihw */
		   } /* fi */
	       break;
	      }
	  case SIR_SYMBOL_CHANNEL:
	     { IO->PutS(" : public " SIR_SIM_CHANNEL_NAME);
	       assert(Interfaces != NULL);	/* is a channel */
	       if (! Interfaces->Empty())
		  { Interface = Interfaces->First();
		    while(Interface)
		       { IO->PutS(", public ");
			 IO->PutS(Interface->Symbol->Name);
			 Interface = Interface->Succ();
			} /* elihw */
		   } /* fi */
	       break;
	      }
	  case SIR_SYMBOL_INTERFACE:
	     { /* nothing to inherit from */
	       break;
	      }
	  default:
	     { assert(false);	/* bad symbol class */
	      }
	 } /* hctiws */
     SIR_LineInfo::WriteNL(IO);		/* body of the class */
     IO->PutC('{');
     SIR_LineInfo::WriteNL(IO);
     IO->PutS("private:");	/* everything except methods is private */
     IO->TabStepUp();
     if (  (Parameters)
	 &&(! Parameters->IsVoid()))
	{ Port = Parameters->First();
	  while(Port)
	     { if (Port->Symbol)
		  { if (Port->Symbol->LineInfo)
		       { if ((SIR_Error = Port->Symbol->LineInfo->WriteSC(
								IO, true)))
			    { return(SIR_Error);
			     } /* fi */
			} /* fi */
		    else
		       { SIR_LineInfo::WriteNewSC(IO, true, false,
						&Port->Symbol->LineInfo);
			} /* esle */
		    Buffer = "(&";
		    Buffer += Port->Symbol->Name.chars();
		    Buffer += ")";
		    IO->PutS(Port->Symbol->Type->PrettyString(&Buffer,
						WriteNotes, true));
		    IO->PutC(';');
		   } /* fi */
	       Port = Port->Succ();
	      } /* elihw */
	 } /* fi */
     ParamSymbol = ClassScope->First();	/* notes for ports */
     while(ParamSymbol)
	{ if (  (ParamSymbol->IsPort())
	      &&(ParamSymbol->Notes))
	     { if ((SIR_Error = ParamSymbol->Notes->WriteSC(IO,
							WriteNotes, true)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  ParamSymbol = ParamSymbol->Succ();
	 } /* elihw */
						/* body */
     /* (1)-(4) User-defined types, declarations and definitions */
     if (ClassScope->UserTypes)
	{ if ((SIR_Error = ClassScope->UserTypes->WriteSC(IO,
					WriteNotes, true, &PutSeparator)))
	     { return(SIR_Error);
	      } /* fi */
	 } /* fi */
     /* (5) Type definitions (alias types) */
     Symbol = ClassScope->First();
     while(Symbol)
	{ if (  (Symbol->IsTypeDef())
	      &&((  (Symbol->Type->UserType == NULL)
		  ||(Symbol->Type->UserType->Name != NULL))))
	     { if ((SIR_Error = Symbol->WriteSC(IO, WriteNotes, false, true)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  Symbol = Symbol->Succ();
	 } /* elihw */
     /* (6)-(8) NO interface, channel or behavior declarations */
#ifndef NDEBUG
     Symbol = ClassScope->First();
     while(Symbol)
	{ assert((  (Symbol->IsInterface())
		  ||(Symbol->IsChannel())
		  ||(Symbol->IsBehavior())) == false);
	  Symbol = Symbol->Succ();
	 } /* elihw */
#endif /* NDEBUG */

     IO->TabStepDown();
     SIR_LineInfo::WriteNL(IO);
     IO->PutS("public:");	/* everything except methods is private */
     IO->TabStepUp();
     if (IsInterface())
	{ // interface should have empty virtual destructor (RD, 01/12/06)
	  SIR_LineInfo::WriteNewSC(IO, true);
	  IO->PutS("virtual ~");
	  IO->PutS(Name);
	  IO->PutS("(void){};");
	 } /* fi */
     else	// is a behavior or channel
	{ /* (9a) Declaration of constructor */
	  SIR_LineInfo::WriteNewSC(IO, true);
	  IO->PutS(Name);
	  if (  (Parameters)		/* ports */
	      &&(! Parameters->IsVoid()))
	     { IO->PutC('(');
	       Port = Parameters->First();
	       while(Port)
		  { if (Port->Symbol)
		       { Buffer = "(&";	/* as reference */
			 Buffer += Port->Symbol->Name;
			 Buffer += ")";
			} /* fi */
		    else
		       { Buffer = "(&)"; /* (bug fix 08/28/01, RD) */
			} /* esle */
		    IO->PutS(Port->Type->PrettyString(
						&Buffer, WriteNotes, true));
		    Port = Port->Succ();
		    if (Port)
		       { IO->PutS(", ");
			} /* fi */
		   } /* elihw */
	       IO->PutS(");");
	      } /* fi */
	  else
	     { IO->PutS("(void);");
	      } /* esle */
	  /* (9b) Declaration of destructor */
	  SIR_LineInfo::WriteNewSC(IO, true);
	  IO->PutS("virtual ~");
	  IO->PutS(Name);
	  IO->PutS("(void);");
	 } /* esle */
     /* (9c) Declarations of (variables and) other methods */
     Symbol = ClassScope->First();
     while(Symbol)
	{ if (  (  (Symbol->IsVariable())	/* IsVariableDeclaration() */
		 &&(Symbol->StorageClass == SIR_STORAGE_EXTERN))
	      ||(Symbol->IsFunction()))
	     { if ((SIR_Error = Symbol->WriteSC(IO, WriteNotes, true, true)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  Symbol = Symbol->Succ();
	 } /* elihw */
     IO->TabStepDown();
     SIR_LineInfo::WriteNL(IO);
     IO->PutS("private:");	/* everything except methods is private */
     IO->TabStepUp();
     /* (10) Variable definitions and class instantiations */
     Symbol = ClassScope->First();		/* variable definitions */
     while(Symbol)
	{ if (Symbol->IsVariableDefinition())
	     { if ((SIR_Error = Symbol->WriteSC(IO, WriteNotes, false, true)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  Symbol = Symbol->Succ();
	 } /* elihw */
     Instance = ClassScope->First();		/* sliced bus declarations */
     BitBusNum = 0;			/* (must appear after variable defs. */
     OpenPortNum = 0;			/* and before class instantiations!) */
     ConstPortNum = 0;
     while(Instance)
	{ if (  (Instance->IsBehaviorInstance())
	      ||(Instance->IsChannelInstance()))
	     { assert(Instance->PortMappings != NULL);	/* must exist */
	       PortMap = Instance->PortMappings->First();
	       DeclaredPort = Instance->Type->Parameters->First();
	       while(PortMap)
		  { assert(DeclaredPort != NULL);	/* must exist */
		    PortMap->WriteCC_Decl(IO, DeclaredPort->Type,
				&BitBusNum, &OpenPortNum, &ConstPortNum);
		    PortMap = PortMap->Succ();
		    DeclaredPort = DeclaredPort->Succ();
		   } /* elihw */
	      } /* fi */
	  Instance = Instance->Succ();
	 } /* elihw */
     Symbol = ClassScope->First();		/* class instantiations */
     while(Symbol)
	{ if (  (Symbol->IsBehaviorInstance())
	      ||(Symbol->IsChannelInstance()))
	     { if ((SIR_Error = Symbol->WriteSC(IO, WriteNotes, false, true)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  Symbol = Symbol->Succ();
	 } /* elihw */
     IO->TabStepDown();
     SIR_LineInfo::WriteNL(IO);
     IO->PutC('}');
    } /* fi */

IO->PutC(';');

if ((SIR_Error = IO->Check()))
   { return(SIR_Error);
    } /* fi */

if (AsDeclaration)
   { if (Notes)
	{ if ((SIR_Error = Notes->WriteSC(IO, WriteNotes, true)))
	     { return(SIR_Error);
	      } /* fi */
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbol::WriteCC2 */


ERROR SIR_Symbol::WriteCC2b(	/* generates C++ source code */
	gl_io		*IO,	/* (for classes, method implementations) */
	bool		WriteNotes)
{
gl_string	Buffer,
		ParamBuffer;
sir_parameter	*Port;
sir_type_ptr	*DeclaredPort;
sir_parameter	*Parameter;
sir_symbol	*Instance,
		*MainMethod,
		*Symbol;
sir_portmap	*PortMap;
const char	*TmpString;
bool		PutColon,
		PutNewLine;
int		ParamNum;
int		BitBusNum,
		OpenPortNum,
		ConstPortNum;
bool		ConstructorEmpty;
#ifdef SIR_PUT_SIZE_CHECK_IN_IP_CONSTRUCTOR
sir_note	*Note;
#endif /* SIR_PUT_SIZE_CHECK_IN_IP_CONSTRUCTOR */

assert(IsClass());	/* nothing else supported */
assert(ClassScope != NULL);

     /* (-) static members (initializion by constructor call with dummy) */

     Symbol = ClassScope->First();	/* dummies as member initializer */
     while(Symbol)
	{ if (  (Symbol->IsVariableDefinition())
	      &&(Symbol->StorageClass == SIR_STORAGE_PIPED)
	      &&(Symbol->Initializer->IsComposite(
				Symbol->Type->Type == SIR_TYPE_ARRAY)))
	     { SIR_LineInfo::WriteNewSC(IO, true);
	       Buffer = Name;
	       Buffer += "::" SIR_CXX_ARRAY_INIT_NAME;
	       Buffer += Symbol->Name;
	       IO->PutS(Symbol->Type->PrettyString(&Buffer, false, true));
	       IO->PutS(" = ");
	       if ((SIR_Error = Symbol->Initializer->WriteSC(IO, true)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->PutC(';');
	       SIR_LineInfo::WriteNL(IO);
	      } /* fi */
	  if (  (Symbol->IsVariableDefinition())
	      &&(  (Symbol->Type->Type == SIR_TYPE_SIGNAL)
		 ||(Symbol->Type->Type == SIR_TYPE_BUFFER))
	      &&(Symbol->Initializer->IsComposite(
			Symbol->Type->SubType->Type == SIR_TYPE_ARRAY)))
	     { SIR_LineInfo::WriteNewSC(IO, true);
	       Buffer = Name;
	       Buffer += "::" SIR_CXX_ARRAY_INIT_NAME;
	       Buffer += Symbol->Name;
	       IO->PutS(Symbol->Type->SubType->PrettyString(
						&Buffer, false, true));
	       IO->PutS(" = ");
	       if ((SIR_Error = Symbol->Initializer->WriteSC(IO, true)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->PutC(';');
	       SIR_LineInfo::WriteNL(IO);
	      } /* fi */
	  Symbol = Symbol->Succ();
	 } /* elihw */

     /* (a) constructor implementation */

     SIR_LineInfo::WriteNewSC(IO, true);	/* constructor */
     IO->PutS(Name);
     IO->PutS("::");
     IO->PutS(Name);
     if (  (Parameters)			/* ports */
	 &&(! Parameters->IsVoid()))
	{ IO->PutC('(');
	  Port = Parameters->First();
	  while(Port)
	     { if (Port->Symbol)
		  { Buffer = "(&";	/* as reference */
		    Buffer += Port->Symbol->Name;
		    Buffer += ")";
		   } /* fi */
	       else
		  { Buffer = "(&)"; /* (bug fix 08/28/01, RD) */
		   } /* esle */
	       IO->PutS(Port->Type->PrettyString(&Buffer, WriteNotes, true));
	       Port = Port->Succ();
	       if (Port)
		  { IO->PutS(", ");
		   } /* fi */
	      } /* elihw */
	  IO->PutC(')');
	 } /* fi */
     else
	{ IO->PutS("(void)");
	 } /* esle */
     IO->TabStepUp();
     PutColon = true;
     PutNewLine = true;
     if (Parameters)			/* port mappings */
	{ Port = Parameters->First();
	  while(Port)
	     { if (Port->Symbol)
		  { if (PutNewLine)
		       { SIR_LineInfo::WriteNL(IO);
			 PutNewLine = false;
			} /* fi */
		    if (PutColon)
		       { IO->PutS(": ");
			 PutColon = false;
			} /* fi */
		    else
		       { IO->PutS(", ");
			} /* esle */
		    IO->PutS(Port->Symbol->Name);
		    IO->PutC('(');
		    IO->PutS(Port->Symbol->Name);
		    IO->PutC(')');
		   } /* fi */
	       Port = Port->Succ();
	      } /* elihw */
	 } /* fi */
     Symbol = ClassScope->First();	/* member initializer */
     while(Symbol)
	{ if (  (Symbol->IsVariableDefinition())
	      &&(  (Symbol->Initializer)
		 ||(Symbol->Events)
		 ||(Symbol->Period)))
	     { sir_type		*TargetType;

	       if (  (Symbol->Type->Type == SIR_TYPE_SIGNAL)
		   ||(Symbol->Type->Type == SIR_TYPE_BUFFER))
		  { /* initialize signals/buffers here! */
		    TargetType = Symbol->Type->SubType;
		   } /* fi */
	       else
		  { if (Symbol->StorageClass == SIR_STORAGE_PIPED)
		       { /* initialize piped variables here! */
			 TargetType = Symbol->Type;
			} /* fi */
		    else
		       { if (! Symbol->Initializer->IsComposite(
				Symbol->Type->Type == SIR_TYPE_ARRAY))
			    { /* initialize simple plain class members here! */
			      TargetType = Symbol->Type;
			     } /* fi */
			 else
			    { /* initialize others in constructor body */
			      /* skip these here! */
			      Symbol = Symbol->Succ();
			      continue;
			     } /* esle */
			} /* esle */
		   } /* esle */
	       if (PutColon)
		  { SIR_LineInfo::WriteNL(IO);
		    IO->PutS(": ");
		    PutColon = false;
		   } /* fi */
	       else
		  { IO->PutC(',');
		    SIR_LineInfo::WriteNL(IO);
		   } /* esle */
	       if (Symbol->Period) /* squeeze in the auto-event for buffers */
		  { IO->PutS(SIR_CXX_AUTO_EVENT_NAME);
		    IO->PutS(Symbol->Name);
		    IO->PutC('(');
		    IO->PutS(Symbol->Period->Print(true));
		    IO->PutS("),");
		    SIR_LineInfo::WriteNL(IO);
		   } /* fi */
	       IO->PutS(Symbol->Name);
	       IO->PutC('(');
	       if (  (Symbol->Events) /* buffered variables need extra args! */
		   ||(Symbol->Period))
		  { if (Symbol->Events)
		       { if ((SIR_Error = Symbol->Events->WriteCC(IO)))
			    { return(SIR_Error);
			     } /* fi */
			} /* fi */
		    else
		       { IO->PutS(SIR_SIM_ANY_SIG_PFX
					SIR_CXX_AUTO_EVENT_NAME);
			 IO->PutS(Symbol->Name);
			 IO->PutS(")");
			} /* esle */
		    if (Symbol->ResetSignal)
		       { IO->PutS(", &");
			 IO->PutS(Symbol->ResetSignal->Name);
			 IO->PutS(", ");
			 if (Symbol->ResetActiveHi)
			    { IO->PutS("true");
			     } /* fi */
			 else
			    { IO->PutS("false");
			     } /* esle */
			} /* fi */
		    if (Symbol->Initializer)
		       { IO->PutS(", ");
			} /* fi */
		   } /* fi */
	       if (Symbol->Initializer->IsComposite(
				TargetType->Type == SIR_TYPE_ARRAY))
		  { /* initializion by constructor call with dummy */
		    IO->PutS(SIR_CXX_ARRAY_INIT_NAME);
		    IO->PutS(Symbol->Name);
		   } /* fi */
	       else
		  { if (Symbol->Initializer)
		       { /* initializion by constructor call with constant */
			 assert(Symbol->Initializer->InitList == NULL);
			 IO->PutS(Symbol->Initializer->Initializer->Print(
									true));
			} /* fi */
		   } /* esle */
	       IO->PutC(')');
	      } /* fi */
	  Symbol = Symbol->Succ();
	 } /* elihw */
     Instance = ClassScope->First();	/* sliced bus initializations */
     BitBusNum = 0;
     OpenPortNum = 0;
     ConstPortNum = 0;
     while(Instance)
	{ if (  (Instance->IsBehaviorInstance())
	      ||(Instance->IsChannelInstance()))
	     { assert(Instance->PortMappings != NULL);	/* must exist */
	       PortMap = Instance->PortMappings->First();
	       DeclaredPort = Instance->Type->Parameters->First();
	       while(PortMap)
		  { assert(DeclaredPort != NULL);
		    if ((TmpString = PortMap->WriteCC_Init(DeclaredPort->Type,
				&BitBusNum, &OpenPortNum, &ConstPortNum)))
		       { if (PutColon)
			    { SIR_LineInfo::WriteNL(IO);
			      IO->PutS(": ");
			      PutColon = false;
			     } /* fi */
			 else
			    { IO->PutC(',');
			      SIR_LineInfo::WriteNL(IO);
			     } /* esle */
			 IO->PutS(TmpString);
			} /* fi */
		    PortMap = PortMap->Succ();
		    DeclaredPort = DeclaredPort->Succ();
		   } /* elihw */
	      } /* fi */
	  Instance = Instance->Succ();
	 } /* elihw */
     Instance = ClassScope->First();	/* instance port mappings */
     BitBusNum = 0;
     OpenPortNum = 0;
     ConstPortNum = 0;
     while(Instance)
	{ if (  (Instance->IsBehaviorInstance())
	      ||(Instance->IsChannelInstance()))
	     { assert(Instance->PortMappings != NULL);	/* must exist */
	       if (! Instance->PortMappings->Empty())	/* but can be empty */
		  { if (PutColon)
		       { SIR_LineInfo::WriteNL(IO);
			 IO->PutS(": ");
			 PutColon = false;
			} /* fi */
		    else
		       { IO->PutC(',');
			 SIR_LineInfo::WriteNL(IO);
			} /* esle */
		    IO->PutS(Instance->Name);
		    IO->PutC('(');
		    PortMap = Instance->PortMappings->First();
		    DeclaredPort = Instance->Type->Parameters->First();
		    while(PortMap)
		       { assert(DeclaredPort != NULL);
			 PortMap->WriteCC(IO, DeclaredPort->Type,
				&BitBusNum, &OpenPortNum, &ConstPortNum);
			 PortMap = PortMap->Succ();
			 DeclaredPort = DeclaredPort->Succ();
			 if (PortMap)
			    { IO->PutS(", ");
			     } /* fi */
			} /* elihw */
		    IO->PutC(')');
		   } /* fi */
	      } /* fi */
	  Instance = Instance->Succ();
	 } /* elihw */
     SIR_LineInfo::WriteNL(IO, true, false);	// IO->PutC('{');
     ConstructorEmpty = true;
     Symbol = ClassScope->First();	/* member initializer in constr. body */
     while(Symbol)
	{ if (  (Symbol->IsVariableDefinition())
	      &&(Symbol->Initializer)
	      &&(  (Symbol->Type->Type != SIR_TYPE_SIGNAL)
		 &&(Symbol->Type->Type != SIR_TYPE_BUFFER))
	      &&(Symbol->StorageClass != SIR_STORAGE_PIPED)
	      &&(Symbol->Initializer->IsComposite(
				Symbol->Type->Type == SIR_TYPE_ARRAY)))
	     { /* member initialization in constructor body */
#ifndef HAVE_ARYASGN
	       sir_type		*ArrayType;
	       unsigned int	Depth,
				Index,
				ArraySize;
#endif /* HAVE_ARYASGN */

	       ConstructorEmpty = false;
	       SIR_LineInfo::WriteNL(IO);
	       Buffer = SIR_CXX_ARRAY_INIT_NAME;
	       Buffer += Symbol->Name;
	       IO->PutS(Symbol->Type->PrettyString(&Buffer, false, true));
	       IO->PutS(" = ");
	       if ((SIR_Error = Symbol->Initializer->WriteSC(IO, true)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->PutC(';');
	       SIR_LineInfo::WriteNL(IO);
#ifdef HAVE_ARYASGN
	       IO->PutS(Symbol->Name);
	       IO->PutS(" = ");
	       IO->PutS(SIR_CXX_ARRAY_INIT_NAME);
	       IO->PutS(Symbol->Name);
	       IO->PutC(';');
#else /* !HAVE_ARYASGN */
	       Depth = 0;
	       ArrayType = Symbol->Type;
	       while(ArrayType->Type == SIR_TYPE_ARRAY)
		  { IO->TabStepUp();
		    SIR_LineInfo::WriteNL(IO, true, false); // IO->PutC('{');
		    if (ArrayType->Size == SIR_UNKNOWN_ARRAY_SIZE)
		       { ArraySize = Symbol->Initializer->GetAutoArraySize();
			 assert(ArraySize > 0);		/* must be valid */
			} /* fi */
		    else
		       { assert(ArrayType->Size > 0);	/* must be valid */
			 ArraySize = ArrayType->Size;
			} /* esle */
		    IO->PrintF("unsigned int " SIR_CXX_INDEX_NAME "%u;",
				Depth);
		    SIR_LineInfo::WriteNL(IO);
		    IO->PrintF("for(" SIR_CXX_INDEX_NAME "%u = 0; "
					SIR_CXX_INDEX_NAME "%u < %u; "
					SIR_CXX_INDEX_NAME "%u++)",
					Depth, Depth, ArraySize, Depth);
		    Depth++;
		    ArrayType = ArrayType->SubType;
		   } /* elihw */
	       IO->TabStepUp();
	       SIR_LineInfo::WriteNL(IO);
	       IO->PutS(Symbol->Name);
	       for(Index=0; Index<Depth; Index++)
		  { IO->PrintF("[" SIR_CXX_INDEX_NAME "%u]", Index);
		   } /* rof */
	       IO->PutS(" = ");
	       IO->PutS(SIR_CXX_ARRAY_INIT_NAME);
	       IO->PutS(Symbol->Name);
	       for(Index=0; Index<Depth; Index++)
		  { IO->PrintF("[" SIR_CXX_INDEX_NAME "%u]", Index);
		   } /* rof */
	       IO->PutC(';');
	       IO->TabStepDown();
	       for(Index=0; Index<Depth; Index++)
		  { SIR_LineInfo::WriteNL(IO, false, true); // IO->PutC('}');
		    IO->TabStepDown();
		   } /* rof */
#endif /* HAVE_ARYASGN */
	      } /* fi */
	  Symbol = Symbol->Succ();
	 } /* elihw */
     if (ConstructorEmpty)
	{ IO->PutS("/* nothing */");
	 } /* fi */
     SIR_LineInfo::WriteNL(IO, false, true);	// IO->PutC('}');
     IO->TabStepDown();

     /* (b) destructor implementation */

     SIR_LineInfo::WriteNL(IO);		/* destructor */
     SIR_LineInfo::WriteNewSC(IO, true);
     IO->PutS(Name);
     IO->PutS("::~");
     IO->PutS(Name);
     IO->PutS("(void)");
     IO->TabStepUp();
     SIR_LineInfo::WriteNL(IO, true, false);	// IO->PutC('{');
     IO->PutS("/* nothing */");
     SIR_LineInfo::WriteNL(IO, false, true);	// IO->PutC('}');
     IO->TabStepDown();

     /* (c) other methods implementations */

     Symbol = ClassScope->First();
     while(Symbol)
	{ if (Symbol->IsFunctionDefinition())
	     { SIR_LineInfo::WriteVSPACE(IO);
	       if ((SIR_Error = Symbol->WriteSC3(IO, WriteNotes, true)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  Symbol = Symbol->Succ();
	 } /* elihw */


if (  (IsBehaviorDefinition())			/* main function */
    &&(0 == strcmp(Name.chars(), GL_MAIN_BEHAVIOR_NAME)))
   { assert(GetScope()->ScopeInfo == SIR_SCOPE_GLOBAL);
     SIR_LineInfo::WriteVSPACE(IO);
#ifdef SIR_STORE_MAIN_BEHAVIOR_AS_STATIC
     SIR_LineInfo::WriteNewSC(IO, true);
     IO->PutS(GL_MAIN_BEHAVIOR_NAME " " SIR_CXX_MAIN_INST_NAME ";");
     SIR_LineInfo::WriteNL(IO);
#endif /* SIR_STORE_MAIN_BEHAVIOR_AS_STATIC */
     if (GetScope()->FindLocally(GL_MAIN_FUNCTION_NAME))
	{ GL_PrintWarning(GL_WARN_IMPORTANT,
		"Skipping generation of function '" GL_MAIN_FUNCTION_NAME "';"
							GL_WARN_MSG_NEWLINE
		"global symbol '" GL_MAIN_FUNCTION_NAME "' already exists");
	 } /* fi */
     else
	{ MainMethod = ClassScope->FindLocally(GL_MAIN_METHOD_NAME);
	  assert(  (MainMethod != NULL)	/* semantic checks assure this */
		 &&(MainMethod->IsFunctionDefinition()));
	  SIR_LineInfo::WriteNewSC(IO, true);
	  Buffer = GL_MAIN_FUNCTION_NAME "(";
	  Parameter = MainMethod->Parameters->First();
	  ParamNum = 0;
	  while(Parameter)
	     { if (Parameter->Symbol)
		  { Buffer += Parameter->Symbol->PrintStorageClass(true);
		    ParamBuffer = Parameter->Symbol->Name;
		   } /* fi */
	       else
		  { if (Parameter->Type->Type == SIR_TYPE_VOID)
		       { ParamBuffer = "";
			} /* fi */
		    else
		       { ParamBuffer.form(SIR_CXX_MAIN_ARG_NAME, ParamNum);
			} /* esle */
		   } /* esle */
	       Buffer += Parameter->Type->PrettyString(&ParamBuffer,
							false, true);
	       Parameter = Parameter->Succ();
	       ParamNum++;
	       if (Parameter)
		  { Buffer += ", ";
		   } /* fi */
	      } /* elihw */
	  Buffer += ")";
	  IO->PutS(MainMethod->Type->SubType->Modified(false,
				MainMethod->Type->Volatile)->PrettyString(
							&Buffer, false, true));
	  IO->TabStepUp();
	  SIR_LineInfo::WriteNL(IO, true, false);	// IO->PutC('{');
#ifdef SIR_STORE_MAIN_BEHAVIOR_ON_STACK
	  SIR_LineInfo::WriteNL(IO);
	  IO->PutS(GL_MAIN_BEHAVIOR_NAME " " SIR_CXX_MAIN_INST_NAME ";");
#endif /* SIR_STORE_MAIN_BEHAVIOR_ON_STACK */
	  SIR_LineInfo::WriteNL(IO);
	  if (MainMethod->Type->SubType->Type != SIR_TYPE_VOID)
	     { Buffer = SIR_CXX_MAIN_RET_NAME;
	       IO->PutS(MainMethod->Type->SubType->Modified(false,
				MainMethod->Type->Volatile)->PrettyString(
							&Buffer, false, true));
	       IO->PutC(';');
	       SIR_LineInfo::WriteNL(IO);
	      } /* fi */
	  SIR_LineInfo::WriteNL(IO);
	  IO->PutS(SIR_SIM_INIT "();");
	  SIR_LineInfo::WriteNL(IO);
	  if (MainMethod->Type->SubType->Type != SIR_TYPE_VOID)
	     { IO->PutS(SIR_CXX_MAIN_RET_NAME " = ");
	      } /* fi */
	  IO->PutS(SIR_CXX_MAIN_INST_NAME "." GL_MAIN_METHOD_NAME "(");
	  Parameter = MainMethod->Parameters->First();
	  ParamNum = 0;
	  while(Parameter)
	     { if (Parameter->Symbol)
		  { IO->PutS(Parameter->Symbol->Name);
		   } /* fi */
	       else
		  { if (Parameter->Type->Type != SIR_TYPE_VOID)
		       { IO->PrintF(SIR_CXX_MAIN_ARG_NAME, ParamNum);
			} /* fi */
		   } /* esle */
	       Parameter = Parameter->Succ();
	       ParamNum++;
	       if (Parameter)
		  { IO->PutS(", ");
		   } /* fi */
	      } /* elihw */
	  IO->PutS(");");
	  SIR_LineInfo::WriteNL(IO);
	  IO->PutS(SIR_SIM_EXIT "();");
	  if (MainMethod->Type->SubType->Type != SIR_TYPE_VOID)
	     { SIR_LineInfo::WriteNL(IO);
	       IO->PutS("return(" SIR_CXX_MAIN_RET_NAME ");");
	      } /* fi */
	  SIR_LineInfo::WriteNL(IO, false, true);	// IO->PutC('}');
	  IO->TabStepDown();
	 } /* esle */
    } /* fi */

if ((SIR_Error = IO->Check()))
   { return(SIR_Error);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbol::WriteCC2b */


ERROR SIR_Symbol::WriteCC2IP(	/* generates C++ source code */
	gl_io		*IO)	/* (for IP classes) */
{
gl_string	Buffer,
		ParamBuffer;
sir_type_ptr	*PortTypePtr;
sir_symbol_ptr	*Interface;
sir_symbol	*Symbol,
		*Method;
int		PortNum;

assert(IsBehavior() || IsChannel());	/* nothing else supported */
assert(ClassScope == NULL);	/* otherwise it's not an IP */
assert(Interfaces != NULL);	/* is a behavior or channel */

Symbol = GetScope()->First();
while(Symbol)
   { if (  (  (Symbol->IsBehavior())
	    ||(Symbol->IsChannel()))
	 &&(Symbol->Instantiates(this)))
	{ break;	/* there's at least one instantiation */
	 } /* fi */
     Symbol = Symbol->Succ();
    } /* elihw */
if (! Symbol)		/* no instance found */
   { return(SIR_ERROR_NO_ERROR);	/* unused */
    } /* fi */

if (LineInfo)
   { if ((SIR_Error = LineInfo->WriteSC(IO, true)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
else
   { SIR_LineInfo::WriteNewSC(IO, true, false, &LineInfo);
    } /* esle */

IO->PutS("class ");
IO->PutS(Name);

if (Class == SIR_SYMBOL_BEHAVIOR)	/* base classes */
   { if (0 != strcmp(Name.chars(), GL_MAIN_BEHAVIOR_NAME))
	{ IO->PutS(" : public " SIR_SIM_BEHAVIOR_NAME);
	 } /* fi */
    } /* fi */
else
   { IO->PutS(" : public " SIR_SIM_CHANNEL_NAME);
    } /* esle */
if (! Interfaces->Empty())
   { Interface = Interfaces->First();
     while(Interface)
	{ IO->PutS(", public ");
	  IO->PutS(Interface->Symbol->Name);
	  Interface = Interface->Succ();
	 } /* elihw */
    } /* fi */

SIR_LineInfo::WriteNL(IO);		/* _fake_ body of the class */
IO->PutC('{');
SIR_LineInfo::WriteNL(IO);
IO->PutS("private:");	/* everything except methods is private */
IO->TabStepUp();
if (! Type->VoidParameters())		/* dummy ports */
   { PortNum = 0;
     PortTypePtr = Type->Parameters->First();
     while(PortTypePtr)
	{ SIR_LineInfo::WriteNewSC(IO, true);
	  Buffer.form("(&" SIR_CXX_DUMMY_PORT_NAME ")", PortNum++);
	  IO->PutS(PortTypePtr->Type->PrettyString(&Buffer, false, true));
	  IO->PutC(';');
	  PortTypePtr = PortTypePtr->Succ();
	 } /* elihw */
    } /* fi */
IO->TabStepDown();
SIR_LineInfo::WriteNL(IO);
IO->PutS("public:");		/* everything except methods is private */
IO->TabStepUp();
SIR_LineInfo::WriteNewSC(IO, true);	/* constructor declaration */
IO->PutS(Name);
if (! Type->VoidParameters())		/* dummy ports */
   { IO->PutC('(');
     PortTypePtr = Type->Parameters->First();
     while(PortTypePtr)
	{ Buffer = "(&)";	/* as reference */ /* (bug fix 08/28/01, RD) */
	  IO->PutS(PortTypePtr->Type->PrettyString(&Buffer, false, true));
	  PortTypePtr = PortTypePtr->Succ();
	  if (PortTypePtr)
	     { IO->PutS(", ");
	      } /* fi */
	 } /* elihw */
     IO->PutS(");");
    } /* fi */
else
   { IO->PutS("(void);");
    } /* esle */

SIR_LineInfo::WriteNewSC(IO, true);	/* destructor declaration */
IO->PutS("virtual ~");
IO->PutS(Name);
IO->PutS("(void);");

#ifndef NDEBUG
Interface = Interfaces->First();	/* implemented methods declarations */
while(Interface)
   { if (Interface->Symbol->ClassScope)
	{ Method = Interface->Symbol->ClassScope->First();
	  while(Method)
	     { assert(Method->Color == SIR_WHITE);	/* unmarked */
	       Method = Method->Succ();
	      } /* elihw */
	 } /* fi */
     Interface = Interface->Succ();
    } /* elihw */
#endif /* NDEBUG */
if (IsBehavior())
   { SIR_LineInfo::WriteNewSC(IO, true);
     IO->PutS("void " GL_MAIN_METHOD_NAME "(void);");
     MarkImplementedMethods(GL_MAIN_METHOD_NAME, Interfaces->First());
    } /* fi */
Interface = Interfaces->First();
while(Interface)
   { if (Interface->Symbol->ClassScope)
	{ Method = Interface->Symbol->ClassScope->First();
	  while(Method)
	     { if (Method->Color == SIR_WHITE)	/* not already declared */
		  { assert(Method->IsFunction());
		    SIR_LineInfo::WriteNewSC(IO, true);
		    Buffer = Method->Name;
		    IO->PutS(Method->Type->PrettyString(&Buffer, false, true));
		    IO->PutC(';');
		    Method->Color = SIR_RED;
		    MarkImplementedMethods(Method->Name.chars(),
						Interface->Succ());
		   } /* fi */
	       Method = Method->Succ();
	      } /* elihw */
	 } /* fi */
     Interface = Interface->Succ();
    } /* elihw */
Interface = Interfaces->First();	/* clean up */
while(Interface)
   { if (Interface->Symbol->ClassScope)
	{ Method = Interface->Symbol->ClassScope->First();
	  while(Method)
	     { assert(Method->Color == SIR_RED);/* must have been declared */
	       Method->Color = SIR_WHITE;	/* remove markers */
	       Method = Method->Succ();
	      } /* elihw */
	 } /* fi */
     Interface = Interface->Succ();
    } /* elihw */

IO->TabStepDown();
SIR_LineInfo::WriteNL(IO);
IO->PutS("};");

if ((SIR_Error = IO->Check()))
   { return(SIR_Error);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbol::WriteCC2IP */


ERROR SIR_Symbol::CheckInitializer(	/* checks whether initializer is legal*/
	sir_initializer	*Initializer /* = NULL */,	/* (if not specified, */
	sir_type	*Type /* = NULL */,		/*  own data is used) */
	SIR_STORAGE	*StorageClass /* = NULL */,
	int		*PipeStages /* = NULL */)
{
sir_initializer	*CheckInit;
sir_type	*CheckType;
SIR_STORAGE	CheckStorage;
int		CheckPipes;

if (Initializer)
   { CheckInit = Initializer;
    } /* fi */
else
   { CheckInit = this->Initializer;
    } /* esle */
if (Type)
   { CheckType = Type;
    } /* fi */
else
   { CheckType = this->Type;
    } /* esle */
if (StorageClass)
   { CheckStorage = *StorageClass;
    } /* fi */
else
   { CheckStorage = this->StorageClass;
    } /* esle */
if (PipeStages)
   { CheckPipes = *PipeStages;
    } /* fi */
else
   { CheckPipes = this->PipeStages;
    } /* esle */

if (! Initializer)
   { return(SIR_ERROR_NO_ERROR);	/* empty is always ok */
    } /* fi */

if (  (CheckStorage == SIR_STORAGE_TYPEDEF)
    ||(CheckStorage == SIR_STORAGE_EXTERN))
   { SIR_ErrMsg.form("Storage class '%s' forbids initialization"
							GL_ERROR_MSG_NEWLINE
			"for variable '%s'",
			SIR_Symbol::PrintStorageClass(CheckStorage),
			Name.chars());
     return(SIR_ERROR_FORBIDDEN_INITIALIZER);
    } /* fi */

if ((SIR_Error = CheckType->CheckInitializer(CheckInit)))
   { return(SIR_Error);
    } /* fi */

return(SIR_ERROR_NO_ERROR);	/* is acceptable */

} /* end of SIR_Symbol::CheckInitializer */


ERROR SIR_Symbol::CheckStorageClass(	/* checks whether storage class legal */
	SIR_STORAGE	*StorageClass /* = NULL */,	/* (if not specified, */
	int		*PipeStages /* = NULL */,	/*  own data is used) */
	SIR_SYMBOLCLASS	*Class /* = NULL */,
	SIR_SCOPE	*ScopeInfo /* = NULL */)
{
SIR_STORAGE	CheckStorage;
int		CheckPipes;
SIR_SYMBOLCLASS	CheckClass;
SIR_SCOPE	CheckScope;

if (StorageClass)
   { CheckStorage = *StorageClass;
    } /* fi */
else
   { CheckStorage = this->StorageClass;
    } /* esle */
if (PipeStages)
   { CheckPipes = *PipeStages;
    } /* fi */
else
   { CheckPipes = this->PipeStages;
    } /* esle */
if (Class)
   { CheckClass = *Class;
    } /* fi */
else
   { CheckClass = this->Class;
    } /* esle */
if (ScopeInfo)
   { CheckScope = *ScopeInfo;
    } /* fi */
else
   { CheckScope = GetScope()->ScopeInfo;
    } /* esle */

switch(CheckClass)
   { case SIR_SYMBOL_IDENTIFIER:
	{ switch(CheckScope)
	     { case SIR_SCOPE_GLOBAL:
		  { if (  (CheckStorage == SIR_STORAGE_AUTO)
			||(CheckStorage == SIR_STORAGE_REGISTER)
			||(CheckStorage == SIR_STORAGE_PIPED))
		       { SIR_ErrMsg.form(
			"Storage class '%s' not allowed in global scope",
				PrintStorageClass(CheckStorage, CheckPipes));
			 return(SIR_ERROR_ILLEGAL_STORAGE_CLASS);
			} /* fi */
		    if (  (IsFunction())
			&&(CheckStorage == SIR_STORAGE_TYPEDEF))
		       { SIR_ErrMsg.form(
			"Storage class '%s' not allowed for functions",
				PrintStorageClass(CheckStorage, CheckPipes));
			 return(SIR_ERROR_ILLEGAL_STORAGE_CLASS);
			} /* fi */
		    break;
		   }
	       case SIR_SCOPE_CLASS:
		  { if (  (IsPort())
			&&(CheckStorage != SIR_STORAGE_NONE))
		       { SIR_ErrMsg.form(
			"Storage class '%s' not allowed for ports",
				PrintStorageClass(CheckStorage, CheckPipes));
			 return(SIR_ERROR_ILLEGAL_STORAGE_CLASS);
			} /* fi */
		    if (  (CheckStorage == SIR_STORAGE_STATIC)
			||(CheckStorage == SIR_STORAGE_EXTERN)
			||(CheckStorage == SIR_STORAGE_AUTO)
			||(CheckStorage == SIR_STORAGE_REGISTER))
		       { SIR_ErrMsg.form(
			"Storage class '%s' not allowed in class scope",
				PrintStorageClass(CheckStorage, CheckPipes));
			 return(SIR_ERROR_ILLEGAL_STORAGE_CLASS);
			} /* fi */
		    if (  (IsFunction())
			&&(  (CheckStorage == SIR_STORAGE_PIPED)
			   ||(CheckStorage == SIR_STORAGE_TYPEDEF)))
		       { SIR_ErrMsg.form(
			"Storage class '%s' not allowed for functions",
				PrintStorageClass(CheckStorage, CheckPipes));
			 return(SIR_ERROR_ILLEGAL_STORAGE_CLASS);
			} /* fi */
		    if (  (  (Type->Type == SIR_TYPE_SIGNAL)
			   ||(Type->Type == SIR_TYPE_BUFFER))
			&&(CheckStorage == SIR_STORAGE_PIPED))
		       { return(SIR_ERROR_PIPED_SIGNAL_OR_BUFFER);
			} /* fi */
		    break;
		   }
	       case SIR_SCOPE_PARAMETER:
		  { if (  (CheckStorage == SIR_STORAGE_TYPEDEF)
			||(CheckStorage == SIR_STORAGE_EXTERN)
			||(CheckStorage == SIR_STORAGE_STATIC)
			||(CheckStorage == SIR_STORAGE_PIPED))
		       { SIR_ErrMsg.form(
			"Storage class '%s' not allowed for parameters",
				PrintStorageClass(CheckStorage, CheckPipes));
			 return(SIR_ERROR_ILLEGAL_STORAGE_CLASS);
			} /* fi */
		    break;
		   }
	       case SIR_SCOPE_STATEMENT:
	       case SIR_SCOPE_FSMDSTMNT:
		  { if (  (  (Type->Type == SIR_TYPE_SIGNAL)
			   ||(Type->Type == SIR_TYPE_BUFFER))
			&&(CheckStorage == SIR_STORAGE_PIPED))
		       { return(SIR_ERROR_PIPED_SIGNAL_OR_BUFFER);
			} /* fi */
		    break;
		   }
	       case SIR_SCOPE_USERTYPE:
		  { assert(CheckStorage == SIR_STORAGE_NONE);
		    break;
		   }
	       default:
		  { assert(false);	/* illegal scope info */
		   }
	      } /* hctiws */
	  if (CheckStorage == SIR_STORAGE_PIPED)
	     { sir_type	*ItemType;

	       ItemType = Type;
	       while(ItemType->Type == SIR_TYPE_ARRAY)
		  { ItemType = ItemType->SubType;
		   } /* elihw */
	       if (ItemType->Const || ItemType->Volatile)
		  { return(SIR_ERROR_QUALIFIED_PIPED_TYPE);
		   } /* fi */
	      } /* fi */
	  break;
	 }
     case SIR_SYMBOL_TYPEDEF:
	{ assert(CheckStorage == SIR_STORAGE_TYPEDEF);
	  break;
	 }
     case SIR_SYMBOL_BEHAVIOR:
	{ assert(CheckStorage == SIR_STORAGE_NONE);
	  break;
	 }
     case SIR_SYMBOL_CHANNEL:
	{ assert(CheckStorage == SIR_STORAGE_NONE);
	  break;
	 }
     case SIR_SYMBOL_INTERFACE:
	{ assert(CheckStorage == SIR_STORAGE_NONE);
	  break;
	 }
     default:
	{ assert(false);	/* bad symbol class */
	 }
    } /* hctiws */

return(SIR_ERROR_NO_ERROR);	/* looks ok */

} /* end of SIR_Symbol::CheckStorageClass */


ERROR SIR_Symbol::Annotate(	/* attach a note to the symbol */
	sir_note	*NewNote)	/* (consumes NewNote) */
{

assert(NewNote != NULL);

NewNote->Symbol = this;		/* symbol note */
NewNote->UserType = NULL;
NewNote->Label = NULL;
NewNote->State = NULL;

if (Notes)
   { if (! Notes->Insert(NewNote))
	{ delete NewNote;
	  return(SIR_Error);
	 } /* fi */
    } /* fi */
else
   { Notes = new SIR_Notes(NewNote);
    } /* esle */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbol::Annotate */


void SIR_Symbol::Touch(		/* modification update (remove import info) */
	bool	ImportOnly /* = false */) /* (dflt: remove source info too) */
{

if (Imported)		/* remove import information */
   { Imported->TakeOverAndRemove(GetDesign());
    } /* fi */

if (! ImportOnly)	/* remove source code info */
   { SIR_Node::Strip();
    } /* fi */

} /* end of SIR_Symbol::Touch */


void SIR_Symbol::Strip(		/* remove source location for the tree below */
	bool		LocalOnly /* = false */) /* (dflt: work recursively) */
{

if (LocalOnly)
   { SIR_Node::Strip();
    } /* fi */
else
   { DFS_ForAllNodes(&SIR_Node::Strip, NULL);
    } /* esle */

} /* end of SIR_Symbol::Strip */


	/*******************/
	/*** SIR_Symbols ***/
	/*******************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Symbols::SIR_Symbols(		/* constructor #1 */
	sir_symbols	*Parent,	/* (creates UserTypes automatically) */
	SIR_SCOPE	ScopeInfo,
	sir_symbol	*ParentSymbol /* = NULL */,
	sir_usertype	*ParentUType /* = NULL */,
	sir_statement	*ParentStmnt /* = NULL */,
	sir_fsmdstate	*ParentFsmdState /* = NULL */,
	sir_fsmdstmnt	*ParentFsmdStmnt /* = NULL */)
{

if (Parent)
   { SIR_Symbols::Parent	= Parent;
     SIR_Symbols::UserTypes	= new SIR_UserTypes(Parent->UserTypes, this);
    } /* fi */
else
   { SIR_Symbols::Parent	= NULL;
     SIR_Symbols::UserTypes	= new SIR_UserTypes(NULL, this);
    } /* esle */
SIR_Symbols::ScopeInfo		= ScopeInfo;
SIR_Symbols::ParentSymbol	= ParentSymbol;
SIR_Symbols::ParentUType	= ParentUType;
SIR_Symbols::ParentStmnt	= ParentStmnt;
SIR_Symbols::ParentFsmdState	= ParentFsmdState;
SIR_Symbols::ParentFsmdStmnt	= ParentFsmdStmnt;

} /* end of SIR_Symbols::SIR_Symbols #1 */


SIR_Symbols::SIR_Symbols(		/* constructor #3 (duplicator) */
	sir_symbols	*Original)
{
sir_symbol	*Curr;

Curr = Original->First();
while(Curr)
   { Append(new SIR_Symbol(Curr));
     Curr = Curr->Succ();
    } /* elihw */

SIR_Symbols::Parent		= Original->Parent;
SIR_Symbols::UserTypes		= Original->UserTypes ?
					new SIR_UserTypes(
						Original->UserTypes) :
					NULL;
SIR_Symbols::ScopeInfo		= Original->ScopeInfo;
SIR_Symbols::ParentSymbol	= Original->ParentSymbol;
SIR_Symbols::ParentUType	= Original->ParentUType;
SIR_Symbols::ParentStmnt	= Original->ParentStmnt;
SIR_Symbols::ParentFsmdState	= Original->ParentFsmdState;
SIR_Symbols::ParentFsmdStmnt	= Original->ParentFsmdStmnt;

} /* end of SIR_Symbols::SIR_Symbols #3 */


SIR_Symbols::~SIR_Symbols(void)		/* destructor */
{

delete UserTypes;

} /* end of SIR_Symbols::~SIR_Symbols */


bool SIR_Symbols::IsAncestorOf(	/* checks if the symbol can see this scope */
	sir_symbol	*Symbol)
{
sir_symbols	*Scope;

assert(Symbol != NULL);

Scope = Symbol->GetScope();
while(Scope)
   { if (Scope == this)
	{ return(true);
	 } /* fi */
     Scope = Scope->Parent;
    } /* elihw */

return(false);

} /* end of SIR_Symbols::IsAncestorOf */


unsigned int SIR_Symbols::NumTypeUsers(	/* counts the symbols using the type */
	sir_type	*Type)
{
sir_symbol	*Symbol;
unsigned int	Count;

Count = 0;
Symbol = First();
while(Symbol)
   { if (  (Symbol->Type == Type)
	 ||(Symbol->Type->SubTypeTreeContains(Type)))
	{ if (! Symbol->IsEnumMember())
	     { Count++;
	      } /* fi */
	 } /* fi */
     Symbol = Symbol->Succ();
    } /* elihw */

return(Count);

} /* end of SIR_Symbols::NumTypeUsers */


ERROR SIR_Symbols::DFS_ForAllNodes(	/* iterator for over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{
sir_symbol	*Symbol,
		*Succ;

/* this is not a node, but there are nodes below */

if (UserTypes)
   { if ((SIR_Error = UserTypes->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

Symbol = First();
while(Symbol)
   { Succ = Symbol->Succ();
     if ((SIR_Error = Symbol->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Symbol = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbols::DFS_ForAllNodes */


ERROR SIR_Symbols::DFS_ForAllSymbols(	/* iterator over all symbols */
	sir_symbol_mptr	MemberFct,	/* (depth first) */
	sir_symbol_marg	MemberFctArg)
{
sir_symbol	*Symbol,
		*Succ;

/* this is not a symbol, but there are symbols below */

if (UserTypes)
   { if ((SIR_Error = UserTypes->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

Symbol = First();
while(Symbol)
   { Succ = Symbol->Succ();
     if ((SIR_Error = Symbol->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Symbol = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbols::DFS_ForAllSymbols */


ERROR SIR_Symbols::DFS_ForAllUserTypes(	/* iterator over all usertypes */
	sir_usertp_mptr	MemberFct,	/* (depth first) */
	sir_usertp_marg	MemberFctArg)
{
sir_symbol	*Symbol,
		*Succ;

/* this is not a usertype, but there are usertypes below */

if (UserTypes)
   { if ((SIR_Error = UserTypes->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

Symbol = First();
while(Symbol)
   { Succ = Symbol->Succ();
     if ((SIR_Error = Symbol->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Symbol = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbols::DFS_ForAllUserTypes */


ERROR SIR_Symbols::DFS_ForAllNotes(	/* iterator over all notes */
	sir_note_mptr	MemberFct,	/* (depth first) */
	sir_note_marg	MemberFctArg)
{
sir_symbol	*Symbol,
		*Succ;

/* this is not a note, but there are notes below */

if (UserTypes)
   { if ((SIR_Error = UserTypes->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

Symbol = First();
while(Symbol)
   { Succ = Symbol->Succ();
     if ((SIR_Error = Symbol->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Symbol = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbols::DFS_ForAllNotes */


bool SIR_Symbols::DFS_FindDependant(	/* searches for dependants (DFS) */
	sir_symbol	**SkipSymbol,
	sir_symbol	*ThatSymbol,
	sir_symbol	**DepSymbol,
	sir_statement	**DepStmnt,
	sir_fsmdstmnt	**DepFsmdStmnt,
	sir_expression	**DepExpr,
	SIR_DEPENDENCY	*Reason)
{
sir_symbol	*Symbol;

assert(SkipSymbol != NULL);
assert(ThatSymbol != NULL);

if (  (*SkipSymbol)
    &&(! IsAncestorOf(*SkipSymbol)))
   { return(false);	/* skip this symbol table (prune the subtree) */
    } /* fi */

/* we don't need to go into UserTypes,   */
/* those symbols cannot depend on others */

Symbol = First();
while(Symbol)
   { if (Symbol->DFS_FindDependant(SkipSymbol, ThatSymbol,
					DepSymbol, DepStmnt, DepFsmdStmnt,
					DepExpr, Reason))
	{ return(true);
	 } /* fi */
     Symbol = Symbol->Succ();
    } /* elihw */

return(false);

} /* end of SIR_Symbols::DFS_FindDependant */


void SIR_Symbols::SetAlias(	/* sets all type, usertype, symbol alias' */
	sir_symbols	*Alias)	/* (iterates over symbols and usertypes) */
{
sir_symbol	*Symbol,
		*SymbolAlias;

assert(Alias != NULL);

/* no need to process this node */

if (UserTypes)
   { UserTypes->SetAlias(Alias->UserTypes);
    } /* fi */

Symbol = First();
SymbolAlias = Alias->First();
while(Symbol)
   { Symbol->SetAlias(SymbolAlias);
     Symbol = Symbol->Succ();
     SymbolAlias = SymbolAlias->Succ();
    } /* elihw */

} /* end of SIR_Symbols::SetAlias */


void SIR_Symbols::UnAlias(	/* unalias all type, usertype, symbol links */
	sir_symbols	*GlobalSymbols)
{
sir_symbol	*Symbol;

if (  (Parent)
    &&(Parent->Parent == NULL))
   { Parent = GlobalSymbols;
    } /* fi */
if (  (ParentSymbol)
    &&(ParentSymbol->Alias))
   { ParentSymbol = ParentSymbol->Alias;
    } /* fi */
if (  (ParentUType)
    &&(ParentUType->Alias))
   { ParentUType = ParentUType->Alias;
    } /* fi */

UserTypes->UnAlias(GlobalSymbols);

Symbol = First();
while(Symbol)
   { Symbol->UnAlias(GlobalSymbols);
     Symbol = Symbol->Succ();
    } /* elihw */

} /* end of SIR_Symbols::UnAlias */


ERROR SIR_Symbols::Integrate(		/* integrates imported symbol table */
	sir_symbols	*Imported)
{
sir_symbol	*ImpCurr,
		*ImpSucc,
		*Known;

ImpCurr = Imported->First();	/* for all imported symbols */
while(ImpCurr)
   { ImpSucc = ImpCurr->Succ();
     if ((Known = FindLocally(ImpCurr->Name.chars())))	/* already known? */
	{ /* type and class must match */
	  if (ImpCurr->Class != Known->Class)
	     { SIR_ErrMsg.form("Error while importing:" GL_ERROR_MSG_NEWLINE
				"Redefinition of symbol '%s'",
				ImpCurr->Name.chars());
	       return(SIR_ERROR_REDEFINITION_OF_SYMBOL);
	      } /* fi */
	  if (0 != SIR_Types::TypeCmp(	/* check if the types match */
			ImpCurr->Type, Known->Type, false, true))
	     { SIR_ErrMsg.form(
		"Error while importing:" GL_ERROR_MSG_NEWLINE
		"Declaration of symbol '%s'" GL_ERROR_MSG_NEWLINE
		"does not match former declaration:" GL_ERROR_MSG_NEWLINE
			"type mismatch",
				ImpCurr->Name.chars());
	       return(SIR_ERROR_DECLARATION_MISMATCH_1);
	      } /* fi */
	  /* separate for each class:				*/
	  /* - check matching (eg. storage class)		*/
	  /* - check for redefinition: if yes, complain		*/
	  /* - if no, extend former declaration to a definition	*/
	  switch(ImpCurr->Class)
	     { case SIR_SYMBOL_IDENTIFIER:
		  { if (ImpCurr->IsFunction())	/* is it a function? */
		       { if (ImpCurr->IsFunctionDefinition())
			    { if (Known->IsFunctionDefinition())
				 { SIR_ErrMsg.form(
				"Error while importing:" GL_ERROR_MSG_NEWLINE
				"Redefinition of function '%s'",
				ImpCurr->Name.chars());
				   return(SIR_ERROR_REDEFINITION_OF_FUNCTION);
				  } /* fi */
			      /* check storage class */
			      if (  (ImpCurr->StorageClass ==
							SIR_STORAGE_EXTERN)
				  ||(  (ImpCurr->StorageClass !=
						Known->StorageClass)
				     &&(  (ImpCurr->StorageClass !=
							SIR_STORAGE_NONE)
					||(Known->StorageClass !=
							SIR_STORAGE_EXTERN))))
				 { SIR_ErrMsg.form(
				"Error while importing:" GL_ERROR_MSG_NEWLINE
				"Definition of function '%s'"
							GL_ERROR_MSG_NEWLINE
				"does not match former declaration:"
							GL_ERROR_MSG_NEWLINE
				"storage class mismatch",
					ImpCurr->Name.chars());
				   return(SIR_ERROR_SYMBOL_STORAGE_MISMATCH_2);
				  } /* fi */
			      /* make declaration a definition */
			      Known->StorageClass = ImpCurr->StorageClass;
			      assert(Known->Parameters == NULL);
			      Known->Parameters = ImpCurr->Parameters;
			      ImpCurr->Parameters = NULL;
			      assert(Known->ParamScope == NULL);
			      Known->ParamScope = ImpCurr->ParamScope;
			      ImpCurr->ParamScope = NULL;
			      Known->Labels = ImpCurr->Labels;
			      ImpCurr->Labels = NULL;
			      Known->FctBody = ImpCurr->FctBody;
			      ImpCurr->FctBody = NULL;
			      Known->Imported = ImpCurr->Imported;
			      Known->SetLineInfo(ImpCurr);
			     } /* fi */
			 else
			    { /* re-declaration, just check storage class */
			      if (  (ImpCurr->StorageClass !=
						Known->StorageClass)
				  &&(  (ImpCurr->StorageClass !=
							SIR_STORAGE_EXTERN)
				     ||(Known->StorageClass !=
							SIR_STORAGE_NONE)))
				 { SIR_ErrMsg.form(
				"Error while importing:" GL_ERROR_MSG_NEWLINE
				"Declaration of function '%s'"
							GL_ERROR_MSG_NEWLINE
				"does not match former declaration:"
							GL_ERROR_MSG_NEWLINE
				"storage class mismatch",
					ImpCurr->Name.chars());
				   return(SIR_ERROR_SYMBOL_STORAGE_MISMATCH_2);
				  } /* fi */
			     } /* esle */
			} /* fi */
		    else	/* it's a variable or enum constant */
				/* (we are on top-level here, so it */
				/* cannot be a port or instance)    */
		       { assert(  (ImpCurr->IsVariable())
				||(ImpCurr->IsEnumMember()));
			 if (ImpCurr->IsEnumMember())
			    { /* in 'import', we allow redefinition of enum */
			      /* members if their type is the same (checked */
			      /* above) and their enum value matches        */
			      if (ImpCurr->EnumValue != Known->EnumValue)
				 { SIR_ErrMsg.form(
				"Error while importing:" GL_ERROR_MSG_NEWLINE
				"Enum constant '%s' defined twice with"
					" different values (%d, %d)",
					ImpCurr->Name.chars(),
					ImpCurr->EnumValue,
					Known->EnumValue);
				   return(SIR_ERROR_ENUM_DEFINED_TWICE);
				  } /* fi */
			      /* now we can handle it as a re-declaration */
			     } /* fi */
			 if (ImpCurr->IsVariableDefinition())
			    { if (Known->IsVariableDefinition())
				 { SIR_ErrMsg.form(
				"Error while importing:" GL_ERROR_MSG_NEWLINE
				"Symbol '%s' defined twice",
				ImpCurr->Name.chars());
				   return(SIR_ERROR_SYMBOL_DEFINED_TWICE_1);
				  } /* fi */
			      /* check storage class */
			      if (  (ImpCurr->StorageClass !=
						Known->StorageClass)
				  &&(  (ImpCurr->StorageClass !=
							SIR_STORAGE_NONE)
				     ||(Known->StorageClass !=
							SIR_STORAGE_EXTERN)))
				 { SIR_ErrMsg.form(
				"Error while importing:" GL_ERROR_MSG_NEWLINE
				"Definition of symbol '%s'" GL_ERROR_MSG_NEWLINE
				"does not match former declaration:"
							GL_ERROR_MSG_NEWLINE
				"storage class mismatch",
					ImpCurr->Name.chars());
				   return(SIR_ERROR_SYMBOL_STORAGE_MISMATCH_2);
				  } /* fi */
			      /* make declaration a definition */
			      Known->StorageClass = ImpCurr->StorageClass;
			      assert(ImpCurr->PipeStages == 0);
			      assert(Known->Initializer == NULL);
			      Known->Initializer = ImpCurr->Initializer;
			      ImpCurr->Initializer = NULL;
			      Known->Imported = ImpCurr->Imported;
			      Known->SetLineInfo(ImpCurr);
			     } /* fi */
			 else
			    { /* re-declaration, just check storage class */
			      if (  (ImpCurr->StorageClass !=
						Known->StorageClass)
				  &&(  (ImpCurr->StorageClass !=
							SIR_STORAGE_EXTERN)
				     ||(Known->StorageClass !=
							SIR_STORAGE_NONE)))
				 { SIR_ErrMsg.form(
				"Error while importing:" GL_ERROR_MSG_NEWLINE
				"Declaration of symbol '%s'"
							GL_ERROR_MSG_NEWLINE
				"does not match former declaration:"
							GL_ERROR_MSG_NEWLINE
				"storage class mismatch",
					ImpCurr->Name.chars());
				   return(SIR_ERROR_SYMBOL_STORAGE_MISMATCH_2);
				  } /* fi */
			     } /* esle */
			} /* esle */
		    break;
		   }
	       case SIR_SYMBOL_TYPEDEF:
		  { break;	/* nothing else to check or to extend */
		   }
	       case SIR_SYMBOL_BEHAVIOR:
		  { /* implemented interfaces must match (with names) */
		    assert(Known->Interfaces != NULL);
		    assert(ImpCurr->Interfaces != NULL);
		    if (0 != Known->Interfaces->CmpSymbolNames(
						ImpCurr->Interfaces))
		       { SIR_ErrMsg.form(
				"Error while importing:" GL_ERROR_MSG_NEWLINE
				"Declaration of behavior '%s'"
							GL_ERROR_MSG_NEWLINE
				"does not match former declaration:"
							GL_ERROR_MSG_NEWLINE
				"mismatch of implemented interfaces",
				ImpCurr->Name.chars());
			 return(SIR_ERROR_DECLARATION_MISMATCH_2);
			} /* fi */
		    if (ImpCurr->IsBehaviorDefinition())
		       { if (Known->IsBehaviorDefinition())
			    { SIR_ErrMsg.form(
				"Error while importing:" GL_ERROR_MSG_NEWLINE
				"Redefinition of behavior '%s'",
				ImpCurr->Name.chars());
			      return(SIR_ERROR_REDEFINITION_OF_BEHAVIOR);
			     } /* fi */
			 /* make declaration a definition */
			 assert(Known->Parameters == NULL);
			 Known->Parameters = ImpCurr->Parameters;
			 ImpCurr->Parameters = NULL;
			 assert(Known->ParamScope == NULL);
			 Known->ParamScope = ImpCurr->ParamScope;
			 ImpCurr->ParamScope = NULL;
			 Known->ClassScope = ImpCurr->ClassScope;
			 ImpCurr->ClassScope = NULL;
			 Known->Imported = ImpCurr->Imported;
			 Known->SetLineInfo(ImpCurr);
			} /* fi */
		    else
		       { /* re-declaration, nothing to do */
			} /* esle */
		    break;
		   }
	       case SIR_SYMBOL_CHANNEL:
		  { /* implemented interfaces must match (with names) */
		    assert(Known->Interfaces != NULL);
		    assert(ImpCurr->Interfaces != NULL);
		    if (0 != Known->Interfaces->CmpSymbolNames(
						ImpCurr->Interfaces))
		       { SIR_ErrMsg.form(
				"Error while importing:" GL_ERROR_MSG_NEWLINE
				"Declaration of channel '%s'"
							GL_ERROR_MSG_NEWLINE
				"does not match former declaration:"
							GL_ERROR_MSG_NEWLINE
				"mismatch of implemented interfaces",
				ImpCurr->Name.chars());
			 return(SIR_ERROR_DECLARATION_MISMATCH_2);
			} /* fi */
		    if (ImpCurr->IsChannelDefinition())
		       { if (Known->IsChannelDefinition())
			    { SIR_ErrMsg.form(
				"Error while importing:" GL_ERROR_MSG_NEWLINE
				"Redefinition of channel '%s'",
				ImpCurr->Name.chars());
			      return(SIR_ERROR_REDEFINITION_OF_CHANNEL);
			     } /* fi */
			 /* make declaration a definition */
			 assert(Known->Parameters == NULL);
			 Known->Parameters = ImpCurr->Parameters;
			 ImpCurr->Parameters = NULL;
			 assert(Known->ParamScope == NULL);
			 Known->ParamScope = ImpCurr->ParamScope;
			 ImpCurr->ParamScope = NULL;
			 Known->ClassScope = ImpCurr->ClassScope;
			 ImpCurr->ClassScope = NULL;
			 Known->Imported = ImpCurr->Imported;
			 Known->SetLineInfo(ImpCurr);
			} /* fi */
		    else
		       { /* re-declaration, nothing to do */
			} /* esle */
		    break;
		   }
	       case SIR_SYMBOL_INTERFACE:
		  { if (ImpCurr->IsInterfaceDefinition())
		       { if (Known->IsInterfaceDefinition())
			    { SIR_ErrMsg.form(
				"Error while importing:" GL_ERROR_MSG_NEWLINE
				"Redefinition of interface '%s'",
				ImpCurr->Name.chars());
			      return(SIR_ERROR_REDEFINITION_OF_INTERFACE);
			     } /* fi */
			 /* make declaration a definition */
			 Known->ClassScope = ImpCurr->ClassScope;
			 ImpCurr->ClassScope = NULL;
			 Known->Imported = ImpCurr->Imported;
			 Known->SetLineInfo(ImpCurr);
			} /* fi */
		    else
		       { /* re-declaration, nothing to do */
			} /* esle */
		    break;
		   }
	       default:
		  { assert(false);	/* bad symbol class */
		   }
	      } /* hctiws */
	  /* set the alias pointer (in all these cases) */
	  ImpCurr->Alias = Known;
	  /* also, merge the annotations (bug fix 06/20/02, RD) */
	  if ((SIR_Error = SIR_Notes::Merge(&Known->Notes, &ImpCurr->Notes)))
	     { return(SIR_Error);
	      } /* fi */
	 } /* fi */
     else	/* symbol is not known yet */
	{ /* move the complete symbol into the actual symbol table */
	  if (Curr())
	     { InsertBefore(Imported->Remove(ImpCurr));
	      } /* fi */
	  else
	     { Append(Imported->Remove(ImpCurr));
	      } /* esle */
	 } /* esle */
     ImpCurr = ImpSucc;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbols::Integrate */


ERROR SIR_Symbols::CheckClassContainment(void)	/* checks class hierarchy */
{						/* for cycles */
sir_symbol	*Symbol;

Symbol = First();
while(Symbol)
   { if (  (Symbol->IsBehaviorDefinition())
	 ||(Symbol->IsChannelDefinition()))
	{ if (Symbol->Contains(Symbol))	/* contains itself? */
	     { SIR_ErrMsg.form(
		"Error while importing:" GL_ERROR_MSG_NEWLINE
		"class containment cycle detected;" GL_ERROR_MSG_NEWLINE
		"class '%s' contains itself",
			Symbol->Name.chars());
	       return(SIR_ERROR_CLASS_CONTAINMENT_CYCLE);
	      } /* fi */
	 } /* fi */
     Symbol = Symbol->Succ();
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbols::CheckClassContainment */


sir_symbol *SIR_Symbols::Find(	/* find an entry globally (here and above) */
	const char	*Name)	/* (returns NULL if not found) */
{
sir_symbol	*Symbol;

if ((Symbol = FindLocally(Name)))
   { return(Symbol);
    } /* fi */

if (Parent)
   { return(Parent->Find(Name));
    } /* fi */
else
   { return(NULL);	/* not found */
    } /* esle */

} /* end of SIR_Symbols::Find */


sir_symbol *SIR_Symbols::FindLocally(	/* find an entry locally */
	const char	*Name)		/* (returns NULL if not found) */
{
int		CmpVal;

First();
while(Curr())
   { if (0 == (CmpVal = strcmp(Name, Curr()->Name.chars())))
	{ return(Curr());
	 } /* fi */
     if (CmpVal < 0)
	{ break;
	 } /* fi */
     Next();
    } /* elihw */

return(NULL);	/* not found */

} /* end of SIR_Symbols::FindLocally */


sir_symbol *SIR_Symbols::FindLocally(	/* find a symbol with scope locally */
	sir_symbols	*Scope)		/* (returns NULL if not found) */
{

First();
while(Curr())
   { if (Scope == Curr()->ParamScope)
	{ return(Curr());
	 } /* fi */
     if (Scope == Curr()->ClassScope)
	{ return(Curr());
	 } /* fi */
     Next();
    } /* elihw */

return(NULL);	/* not found */

} /* end of SIR_Symbols::FindLocally */


sir_symbol *SIR_Symbols::Insert(	/* inserts a new symbol */
	sir_symbol	*NewSymbol)
{
int		CmpVal;

First();
while(Curr())
   { if (0 == (CmpVal = strcmp(NewSymbol->Name.chars(),
				Curr()->Name.chars())))
	{ assert(false);	/* already exists! */
	 } /* fi */
     if (CmpVal < 0)
	{ break;
	 } /* fi */
     Next();
    } /* elihw */

if (Curr())
   { return(InsertBefore(NewSymbol));
    } /* fi */
else
   { return(Append(NewSymbol));
    } /* esle */

} /* end of SIR_Symbols::Insert */


void SIR_Symbols::FindName(		/* find a symbol or usertype globally */
	const char	*Name,
	sir_symbol	**SymbolFound,		/* the symbol found (or NULL) */
	sir_usertype	**UserTypeFound)	/* user-type found (or NULL) */
{

FindNameLocally(Name, SymbolFound, UserTypeFound);

if ((*SymbolFound) || (*UserTypeFound))
   { return;
    } /* fi */

if (Parent)
   { Parent->FindName(Name, SymbolFound, UserTypeFound);
    } /* fi */

} /* end of SIR_Symbols::FindName */


void SIR_Symbols::FindNameLocally(	/* find a symbol or usertype locally */
	const char	*Name,
	sir_symbol	**SymbolFound,		/* the symbol found (or NULL) */
	sir_usertype	**UserTypeFound)	/* user-type found (or NULL) */
{

if ((*SymbolFound = FindLocally(Name)))
   { *UserTypeFound = NULL;
     return;
    } /* fi */

*UserTypeFound = UserTypes->FindLocally(Name);

} /* end of SIR_Symbols::FindNameLocally */


const char *SIR_Symbols::CreateNewName(	/* create a new name in this scope */
	const char	*InitialName)	/* (not used anywhere in this scope) */
{
static gl_string Buffer;
gl_string	Suffix;
unsigned int	Count;
int		Pos;

assert(InitialName != NULL);

Buffer = InitialName;
assert(Buffer.length() >= 1);

while(Find(Buffer.chars()))
   { if (isdigit(Buffer[Pos = Buffer.length()-1]))
	{ /* increment given suffix */
	  while(  (Pos > 0)
		&&(isdigit(Buffer[Pos-1])))
	     { Pos--;
	      } /* elihw */
	  Count = atoi(Buffer.after(Pos-1));
	  Buffer = Buffer.before(Pos);
	  Suffix.form("%u", Count+1);
	  Buffer += Suffix;
	 } /* fi */
     else
	{ /* no suffix given, append one */
	  if (Buffer[Buffer.length()-1] != '_')
	     { Buffer += '_';
	      } /* fi */
	  Buffer += '0';
	 } /* esle */
    } /* elihw */

return(Buffer.chars());

} /* end of SIR_Symbols::CreateNewName */


sir_symbol *SIR_Symbols::Declare(/* declares a new symbol (and prepares def.) */
	sir_symbol	*NewSymbol)	/* (the argument is consumed!) */
{					/* (may return NULL and SIR_Error) */
sir_symbol	*OldSymbol,
		*HiddenSymbol,
		*Dependant;

assert(NewSymbol != NULL);		/* note: this method is called */
assert(NewSymbol->LineInfo != NULL);	/* exclusively from the parser */

/* first we check the StorageClass depending on symbol class and this scope */

if ((SIR_Error = NewSymbol->CheckStorageClass(
					&NewSymbol->StorageClass,
					&NewSymbol->PipeStages,
					&NewSymbol->Class,
					&ScopeInfo)))
   { delete NewSymbol;
     return(NULL);
    } /* fi */

/* then we check special requirements for behaviors and channels */

switch(NewSymbol->Class)
   { case SIR_SYMBOL_IDENTIFIER:
     case SIR_SYMBOL_TYPEDEF:
     case SIR_SYMBOL_INTERFACE:
	{ break;
	 }
     case SIR_SYMBOL_BEHAVIOR:
	{ if (0 == strcmp(NewSymbol->Name.chars(), GL_MAIN_BEHAVIOR_NAME))
	     { if (! NewSymbol->Type->VoidParameters())
		  { SIR_Error = SIR_ERROR_NO_PORTS_ALLOWED_FOR_MAIN;
		    delete NewSymbol;
		    return(NULL);
		   } /* fi */
	       assert(NewSymbol->Interfaces != NULL);	/* is a behavior */
	       if (! NewSymbol->Interfaces->Empty())
		  { SIR_Error = SIR_ERROR_NO_INTERFACES_ALLOWED_FR_MAIN;
		    delete NewSymbol;
		    return(NULL);
		   } /* fi */
	      } /* fi */
	  break;
	 }
     case SIR_SYMBOL_CHANNEL:
	{ assert(NewSymbol->Type->Parameters != NULL);
	  assert(NewSymbol->Interfaces != NULL);
	  if (  (NewSymbol->Type->VoidParameters())
	      &&(NewSymbol->Interfaces->Empty()))
	     { GL_PrintWarningFmt(GL_WARN_STANDARD,
		"Channel '%s' is useless because it has" GL_WARN_MSG_NEWLINE
		"no ports and no implemented interfaces",
			NewSymbol->Name.chars());
	      } /* fi */
	  break;
	 }
     default:
	{ assert(false);	/* bad symbol class */
	 }
    } /* hctiws */

#ifdef SIR_WARN_ABOUT_SHADOWED_SYMBOLS
if (  (ScopeInfo != SIR_SCOPE_USERTYPE)
    &&(Parent)
    &&(!(  (ScopeInfo == SIR_SCOPE_PARAMETER)		// bug fix 06/18/00, RD
	 &&(  (Parent->ScopeInfo == SIR_SCOPE_USERTYPE)	// bug fix 06/18/00, RD
	    ||(Parent->ScopeInfo == SIR_SCOPE_PARAMETER))))	// 06/21/00, RD
    &&(HiddenSymbol = Parent->Find(NewSymbol->Name.chars())))
   { if (HiddenSymbol->LineInfo)
	{ GL_PrintWarningFmt(GL_WARN_STANDARD,
		"In line %u, file \"%s\"," GL_WARN_MSG_NEWLINE
		"declaration of symbol '%s' hides symbol '%s'"
							GL_WARN_MSG_NEWLINE
		"declared in line %u, file \"%s\"",
			NewSymbol->LineInfo->Line,
			NewSymbol->LineInfo->File->Filename.chars(),
			NewSymbol->Name.chars(),
			HiddenSymbol->Name.chars(),
			HiddenSymbol->LineInfo->Line,
			HiddenSymbol->LineInfo->File->Filename.chars());
	 } /* fi */
     else
	{ GL_PrintWarningFmt(GL_WARN_STANDARD,
		"In line %u, file \"%s\"," GL_WARN_MSG_NEWLINE
		"declaration of symbol '%s' hides symbol '%s'"
							GL_WARN_MSG_NEWLINE
		"declared at higher level",
			NewSymbol->LineInfo->Line,
			NewSymbol->LineInfo->File->Filename.chars(),
			NewSymbol->Name.chars(),
			HiddenSymbol->Name.chars());
	 } /* esle */
    } /* fi */
#endif /* SIR_WARN_ABOUT_SHADOWED_SYMBOLS */

/* then we check for scoping problems in classes */

if (ScopeInfo == SIR_SCOPE_CLASS)
   { assert(Parent != NULL);
     if ((HiddenSymbol = Parent->Find(NewSymbol->Name.chars())))
	{ if (HiddenSymbol->FindDependant(this, &Dependant))
	     { SIR_ErrMsg.form(
		"Declaration of class member/method '%s'" GL_ERROR_MSG_NEWLINE
		"hides global symbol '%s' from being" GL_ERROR_MSG_NEWLINE
		"referenced by local symbol '%s'",
			NewSymbol->Name.chars(),
			HiddenSymbol->Name.chars(),
			Dependant->Name.chars());
	       SIR_Error = SIR_ERROR_MEMBER_HIDES_USED_GLOBAL_SYMB;
	       delete NewSymbol;
	       return(NULL);
	      } /* fi */
	 } /* fi */
    } /* fi */

/* then we check class instantiations and their connections */

if (  (NewSymbol->IsBehaviorInstance())	/* check class instantiations */
    ||(NewSymbol->IsChannelInstance()))
   {
     /* first, make sure that the hierarchy will not have any cycles */

     assert(ParentSymbol != NULL);
     if (  (NewSymbol->Type->ClassSymbol == ParentSymbol) /* self-inst.? */
	 ||(NewSymbol->Type->ClassSymbol->Contains(ParentSymbol)))
	{ SIR_ErrMsg.form(
		"Cyclic instantiation: Class '%s' contains" GL_ERROR_MSG_NEWLINE
		"instances of class '%s'",
		NewSymbol->Type->ClassSymbol->Name.chars(),
		ParentSymbol->Name.chars());
	  SIR_Error = SIR_ERROR_CYCLIC_INSTANTIATION;
	  delete NewSymbol;
	  return(NULL);
	 } /* fi */

     /* next we check the port mapping list semantically */
     /* (note: this solves BUGS_V200/BUG5 and partially BUGS_V200/BUG4) */

     assert(NewSymbol->PortMappings != NULL);
     assert(NewSymbol->Type->Parameters != NULL);
     if ((SIR_Error = NewSymbol->PortMappings->Check(
				NewSymbol->Type->Parameters,
				NewSymbol->LineInfo->File->Filename.chars(),
				NewSymbol->LineInfo->Line)))
	{ delete NewSymbol;
	  return(NULL);
	 } /* fi */
    } /* fi */

/* everything looks ok, now we can declare the symbol */

if ((OldSymbol = FindLocally(NewSymbol->Name.chars())))	/* already there? */
   { /* type and class must match */
     if (NewSymbol->Class != OldSymbol->Class)
	{ SIR_ErrMsg.form("Redefinition of symbol '%s'",
				NewSymbol->Name.chars());
	  SIR_Error = SIR_ERROR_REDEFINITION_OF_SYMBOL_2;
	  delete NewSymbol;
	  return(NULL);
	 } /* fi */
     if (NewSymbol->Type != OldSymbol->Type)	/* types identical? */
	{ SIR_ErrMsg.form(
		"Declaration of symbol '%s'" GL_ERROR_MSG_NEWLINE
		"does not match former declaration:" GL_ERROR_MSG_NEWLINE
		"type mismatch",
		NewSymbol->Name.chars());
	  SIR_Error = SIR_ERROR_DECLARATION_MISMATCH_3;
	  delete NewSymbol;
	  return(NULL);
	 } /* fi */
     /* separate for each class:				*/
     /* - check matching (eg. storage class), else complain	*/
     /* - check for redefinition: if yes, complain		*/
     /* - if no, prepare former declaration for definition	*/
     /*   with DefineXxxx() methods				*/
     switch(NewSymbol->Class)
	{ case SIR_SYMBOL_IDENTIFIER:
	     { if (NewSymbol->IsFunction())	/* is it a function? */
		  { if (NewSymbol->Parameters)	/* prepare definition? */
		       { if (OldSymbol->IsFunctionDefinition())
			    { SIR_ErrMsg.form(
				"Redefinition of function '%s'",
				NewSymbol->Name.chars());
			      SIR_Error = SIR_ERROR_REDEFINITION_OF_FUNCTION_2;
			      delete NewSymbol;
			      return(NULL);
			     } /* fi */
			 /* check storage class */
			 if (  (NewSymbol->StorageClass == SIR_STORAGE_EXTERN)
			     ||(  (NewSymbol->StorageClass !=
						OldSymbol->StorageClass)
				&&(  (NewSymbol->StorageClass !=
							SIR_STORAGE_NONE)
				   ||(OldSymbol->StorageClass !=
							SIR_STORAGE_EXTERN))))
			    { SIR_ErrMsg.form(
				"Definition of function '%s'"
							GL_ERROR_MSG_NEWLINE
				"does not match former declaration:"
							GL_ERROR_MSG_NEWLINE
				"storage class mismatch",
				NewSymbol->Name.chars());
			      SIR_Error = SIR_ERROR_SYMBOL_STORAGE_MISMATCH_1;
			      delete NewSymbol;
			      return(NULL);
			     } /* fi */
			 /* make declaration a prepared definition */
			 OldSymbol->StorageClass = NewSymbol->StorageClass;
			 assert(OldSymbol->PipeStages == 0);
			 assert(NewSymbol->PipeStages == 0);
			 assert(OldSymbol->Parameters == NULL);
			 OldSymbol->Parameters = NewSymbol->Parameters;
			 NewSymbol->Parameters = NULL;
			 assert(OldSymbol->ParamScope == NULL);
			 assert(NewSymbol->ParamScope != NULL);
			 OldSymbol->ParamScope = NewSymbol->ParamScope;
			 NewSymbol->ParamScope = NULL;
			 OldSymbol->ParamScope->ParentSymbol = OldSymbol;
			 assert(OldSymbol->Labels == NULL);
			 assert(NewSymbol->Labels != NULL);
			 OldSymbol->Labels = NewSymbol->Labels;
			 NewSymbol->Labels = NULL;
			 /* FctBody is handled by DefineFunction() */
			 OldSymbol->Imported = NewSymbol->Imported;
			 OldSymbol->SetLineInfo(NewSymbol);
			} /* fi */
		    else
		       { /* re-declaration, just check storage class */
			 if (  (NewSymbol->StorageClass !=
						OldSymbol->StorageClass)
			     &&(  (NewSymbol->StorageClass !=
							SIR_STORAGE_EXTERN)
				||(OldSymbol->StorageClass !=
							SIR_STORAGE_NONE)))
			    { SIR_ErrMsg.form(
				"Declaration of function '%s'"
							GL_ERROR_MSG_NEWLINE
				"does not match former declaration:"
							GL_ERROR_MSG_NEWLINE
				"storage class mismatch",
				NewSymbol->Name.chars());
			      SIR_Error = SIR_ERROR_SYMBOL_STORAGE_MISMATCH_1;
			      delete NewSymbol;
			      return(NULL);
			     } /* fi */
			} /* esle */
		   } /* fi */
	       else	/* it's a variable, enum constant, instance, or port */
		  { assert(  (NewSymbol->IsVariable())
			   ||(NewSymbol->IsEnumMember())
			   ||(NewSymbol->IsBehaviorInstance())
			   ||(NewSymbol->IsChannelInstance())
			   ||(NewSymbol->IsPort()));
		    if (  (NewSymbol->IsVariableDefinition())
			||(NewSymbol->IsEnumMember())
			||(NewSymbol->IsBehaviorInstance())
			||(NewSymbol->IsChannelInstance())
			||(NewSymbol->IsPort()))
		       { if (  (OldSymbol->IsVariableDefinition())
			     ||(OldSymbol->IsEnumMember())
			     ||(OldSymbol->IsBehaviorInstance())
			     ||(OldSymbol->IsChannelInstance())
			     ||(OldSymbol->IsPort()))
			    { SIR_ErrMsg.form("Symbol '%s' defined twice",
						NewSymbol->Name.chars());
			      SIR_Error = SIR_ERROR_SYMBOL_DEFINED_TWICE_2;
			      delete NewSymbol;
			      return(NULL);
			     } /* fi */
			 /* check storage class */
			 if (  (NewSymbol->StorageClass !=
						OldSymbol->StorageClass)
			     &&(  (NewSymbol->StorageClass !=
							SIR_STORAGE_NONE)
				||(OldSymbol->StorageClass !=
							SIR_STORAGE_EXTERN)))
			    { SIR_ErrMsg.form(
				"Definition of symbol '%s'"
							GL_ERROR_MSG_NEWLINE
				"does not match former declaration:"
							GL_ERROR_MSG_NEWLINE
				"storage class mismatch",
				NewSymbol->Name.chars());
			      SIR_Error = SIR_ERROR_SYMBOL_STORAGE_MISMATCH_1;
			      delete NewSymbol;
			      return(NULL);
			     } /* fi */
			 /* make declaration a prepared definition */
			 OldSymbol->StorageClass = NewSymbol->StorageClass;
			 assert(OldSymbol->PipeStages == 0);
			 assert(NewSymbol->PipeStages == 0);
			 /* Initializer is handled by DefineVariable() */
			 OldSymbol->Imported = NewSymbol->Imported;
			 OldSymbol->SetLineInfo(NewSymbol);
			} /* fi */
		    else
		       { /* re-declaration, just check storage class */
			 if (  (NewSymbol->StorageClass !=
						OldSymbol->StorageClass)
			     &&(  (NewSymbol->StorageClass !=
							SIR_STORAGE_EXTERN)
				||(OldSymbol->StorageClass !=
							SIR_STORAGE_NONE)))
			    { SIR_ErrMsg.form(
				"Declaration of symbol '%s'"
							GL_ERROR_MSG_NEWLINE
				"does not match former declaration:"
							GL_ERROR_MSG_NEWLINE
				"storage class mismatch",
				NewSymbol->Name.chars());
			      SIR_Error = SIR_ERROR_SYMBOL_STORAGE_MISMATCH_1;
			      delete NewSymbol;
			      return(NULL);
			     } /* fi */
			} /* esle */
		   } /* esle */
	       delete NewSymbol;
	       return(OldSymbol);	/* return (updated) old symbol */
	      }
	  case SIR_SYMBOL_TYPEDEF:
	     { /* this is always a declaration, */
	       /* but we may have to set the symbol link for usertypes */
	       if (OldSymbol->Type->UserType)	/* is a usertype? */
		  { if (OldSymbol->Type->UserType->TypeDef == NULL)
		       { OldSymbol->Type->UserType->TypeDef = OldSymbol;
			} /* fi */
		   } /* fi */
	       delete NewSymbol;
	       return(OldSymbol);	/* return old symbol */
	      }
	  case SIR_SYMBOL_BEHAVIOR:
	     { /* implemented interfaces must match (with names) */
	       assert(OldSymbol->Interfaces != NULL);
	       assert(NewSymbol->Interfaces != NULL);
	       if (0 != OldSymbol->Interfaces->CmpSymbolNames(
						NewSymbol->Interfaces))
		  { SIR_ErrMsg.form(
			"Declaration of behavior '%s'" GL_ERROR_MSG_NEWLINE
			"does not match former declaration:"
							GL_ERROR_MSG_NEWLINE
			"mismatch of implemented interfaces",
			NewSymbol->Name.chars());
		    SIR_Error = SIR_ERROR_DECLARATION_MISMATCH_4;
		    delete NewSymbol;
		    return(NULL);
		   } /* fi */
	       if (NewSymbol->Parameters)	/* prepare definition? */
		  { if (OldSymbol->IsBehaviorDefinition())
		       { SIR_ErrMsg.form("Redefinition of behavior '%s'",
						NewSymbol->Name.chars());
			 SIR_Error = SIR_ERROR_REDEFINITION_OF_BEHAVIOR_2;
			 delete NewSymbol;
			 return(NULL);
			} /* fi */
		    /* make declaration a prepared definition */
		    assert(OldSymbol->Parameters == NULL);
		    OldSymbol->Parameters = NewSymbol->Parameters;
		    NewSymbol->Parameters = NULL;
		    assert(OldSymbol->ClassScope == NULL);
		    assert(NewSymbol->ClassScope != NULL);
		    OldSymbol->ClassScope = NewSymbol->ClassScope;
		    NewSymbol->ClassScope = NULL;
		    OldSymbol->ClassScope->ParentSymbol = OldSymbol;
		    OldSymbol->Imported = NewSymbol->Imported;
		    OldSymbol->SetLineInfo(NewSymbol);
		   } /* fi */
	       else
		  { /* re-declaration, nothing to do */
		   } /* esle */
	       delete NewSymbol;
	       return(OldSymbol);	/* return (updated) old behavior */
	      }
	  case SIR_SYMBOL_CHANNEL:
	     { /* implemented interfaces must match (with names) */
	       assert(OldSymbol->Interfaces != NULL);
	       assert(NewSymbol->Interfaces != NULL);
	       if (0 != OldSymbol->Interfaces->CmpSymbolNames(
						NewSymbol->Interfaces))
		  { SIR_ErrMsg.form(
			"Declaration of channel '%s'" GL_ERROR_MSG_NEWLINE
			"does not match former declaration:"
							GL_ERROR_MSG_NEWLINE
			"mismatch of implemented interfaces",
			NewSymbol->Name.chars());
		    SIR_Error = SIR_ERROR_DECLARATION_MISMATCH_4;
		    delete NewSymbol;
		    return(NULL);
		   } /* fi */
	       if (NewSymbol->Parameters)	/* prepare definition? */
		  { if (OldSymbol->IsChannelDefinition())
		       { SIR_ErrMsg.form("Redefinition of channel '%s'",
						NewSymbol->Name.chars());
			 SIR_Error = SIR_ERROR_REDEFINITION_OF_CHANNEL_2;
			 delete NewSymbol;
			 return(NULL);
			} /* fi */
		    /* make declaration a prepared definition */
		    assert(OldSymbol->Parameters == NULL);
		    OldSymbol->Parameters = NewSymbol->Parameters;
		    NewSymbol->Parameters = NULL;
		    assert(OldSymbol->ClassScope == NULL);
		    assert(NewSymbol->ClassScope != NULL);
		    OldSymbol->ClassScope = NewSymbol->ClassScope;
		    NewSymbol->ClassScope = NULL;
		    OldSymbol->ClassScope->ParentSymbol = OldSymbol;
		    OldSymbol->Imported = NewSymbol->Imported;
		    OldSymbol->SetLineInfo(NewSymbol);
		   } /* fi */
	       else
		  { /* re-declaration, nothing to do */
		   } /* esle */
	       delete NewSymbol;
	       return(OldSymbol);	/* return (updated) old channel */
	      }
	  case SIR_SYMBOL_INTERFACE:
	     { if (NewSymbol->ClassScope)	/* prepare definition? */
		  { if (OldSymbol->IsInterfaceDefinition())
		       { SIR_ErrMsg.form("Redefinition of interface '%s'",
						NewSymbol->Name.chars());
			 SIR_Error = SIR_ERROR_REDEFINITION_OF_INTERFACE_2;
			 delete NewSymbol;
			 return(NULL);
			} /* fi */
		    /* make declaration a prepared definition */
		    assert(OldSymbol->ClassScope == NULL);
		    assert(NewSymbol->ClassScope != NULL);
		    OldSymbol->ClassScope = NewSymbol->ClassScope;
		    NewSymbol->ClassScope = NULL;
		    OldSymbol->ClassScope->ParentSymbol = OldSymbol;
		    OldSymbol->Imported = NewSymbol->Imported;
		    OldSymbol->SetLineInfo(NewSymbol);
		   } /* fi */
	       else
		  { /* re-declaration, nothing to do */
		   } /* esle */
	       delete NewSymbol;
	       return(OldSymbol);	/* return old interface */
	      }
	  default:
	     { assert(false);	/* bad symbol class */
	      }
	 } /* hctiws */
    } /* fi */

/* not found, so we just insert it */
assert(OldSymbol == NULL);

/* but before we may have to set the typedef link for (unnamed) usertypes */
if (  (NewSymbol->Type->UserType)			/* is a usertype? */
    &&(NewSymbol->StorageClass == SIR_STORAGE_TYPEDEF))	/* and a typedef? */
   { if (NewSymbol->Type->UserType->TypeDef == NULL)	/* is still unset? */
	{ NewSymbol->Type->UserType->TypeDef = NewSymbol;
	 } /* fi */
    } /* fi */

/* and for classes we have to fix pointers from the symbols type */
if (  (NewSymbol->Class == SIR_SYMBOL_BEHAVIOR)
    ||(NewSymbol->Class == SIR_SYMBOL_CHANNEL)
    ||(NewSymbol->Class == SIR_SYMBOL_INTERFACE))
   { NewSymbol->Type->ClassSymbol = NewSymbol;
     NewSymbol->Type->Name = NewSymbol->Name.chars();
    } /* fi */

/* FindLocally() already located the insertion spot */
if (Curr())
   { return(InsertBefore(NewSymbol));
    } /* fi */
else
   { return(Append(NewSymbol));
    } /* esle */

} /* end of SIR_Symbols::Declare */


ERROR SIR_Symbols::DefineVariable(	/* makes variable decl. a definition */
	sir_symbol	*VariableDecl,	/* (consumes Initializer) */
	SIR_STORAGE	DeclStorageClass,
	sir_initializer	*Initializer,
	sir_events	*BufferEvents /* = NULL */, /* (copied, not consumed)*/
	sir_constant	*ClockPeriod /* = NULL */, /* (copied, not consumed) */
	sir_symbol	*ResetSignal /* = NULL */,
	bool		ResetActiveHi /* = false */)
{

assert(VariableDecl != NULL);

if (Initializer)
   { assert(VariableDecl->Initializer == NULL);	/* not defined yet */
     if ((SIR_Error = VariableDecl->CheckInitializer(
					Initializer,
					NULL,
					&DeclStorageClass)))
	{ delete Initializer;
	  return(SIR_Error);
	 } /* fi */
     VariableDecl->Initializer = Initializer;	/* store initializer */
     VariableDecl->Type->ConvertInitializer(	/* convert to expected type */
				Initializer);
    } /* fi */
else
   { /* note: piped arrays of unknown size are only allowed      */
     /*       if the size can be determined from the initializer */
     if (  (DeclStorageClass == SIR_STORAGE_PIPED)
	 &&(VariableDecl->Type->Type == SIR_TYPE_ARRAY)
	 &&(VariableDecl->Type->Size == SIR_UNKNOWN_ARRAY_SIZE))
	{ return(SIR_ERROR_PIPED_ARRAY_OF_UNKNOWN_SIZE);
	 } /* fi */
    } /* esle */

if (  (BufferEvents)
    ||(ClockPeriod))
   { assert((!BufferEvents)||(!ClockPeriod));
     assert(VariableDecl->Type->Type == SIR_TYPE_BUFFER);
     assert(VariableDecl->Events == NULL);	/* not defined yet */
     assert(VariableDecl->Period == NULL);	/* not defined yet */
     if (  (DeclStorageClass == SIR_STORAGE_TYPEDEF)
	 ||(DeclStorageClass == SIR_STORAGE_EXTERN))
	{ SIR_ErrMsg.form("Storage class '%s' forbids definition"
							GL_ERROR_MSG_NEWLINE
			"of clock specifier for buffered variable '%s'",
			SIR_Symbol::PrintStorageClass(DeclStorageClass),
			VariableDecl->Name.chars());
	  return(SIR_ERROR_ILLEGAL_BUFFER_CLOCK_SPEC);
	 } /* fi */
     if (BufferEvents)
	{ VariableDecl->Events = new SIR_Events(BufferEvents);
	 } /* fi */
     else
	{ VariableDecl->Period = new SIR_Constant(ClockPeriod);
	 } /* esle */
     VariableDecl->ResetSignal = ResetSignal;
     VariableDecl->ResetActiveHi = ResetActiveHi;
    } /* fi */
else
   { assert(ResetSignal == NULL);	/* only with BufferEvents! */
     if (  (VariableDecl->Type->Type == SIR_TYPE_BUFFER)
	 &&(DeclStorageClass != SIR_STORAGE_TYPEDEF)
	 &&(DeclStorageClass != SIR_STORAGE_EXTERN))
	{ SIR_ErrMsg.form("Clock specifier missing for buffered variable '%s'",
				VariableDecl->Name.chars());
	  return(SIR_ERROR_MISSING_BUFFER_CLOCK_SPEC);
	 } /* fi */
    } /* esle */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbols::DefineVariable */


ERROR SIR_Symbols::DefineFunction(	/* makes function decl. a definition */
	sir_symbol	*FunctionDecl,	/* (consumes FctBody) */
	sir_statement	*FctBody)
{

assert(FunctionDecl != NULL);
assert(FunctionDecl->FctBody == NULL);	/* not defined yet */
assert(FctBody != NULL);
assert(FunctionDecl->Labels != NULL);

FunctionDecl->FctBody = FctBody;	/* store function body */

if ((SIR_Error = FunctionDecl->Labels->CheckDefStatements())) /* check labels */
   { return(SIR_Error);
    } /* fi */

if ((SIR_Error = FctBody->DFS_ForAllStatements(		/* check control flow */
				&SIR_Statement::CheckControlFlow,
				NULL)))
   { return(SIR_Error);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbols::DefineFunction */


ERROR SIR_Symbols::DefineClass(		/* makes class decl. a definition */
	sir_symbol	*ClassDecl)
{
sir_symbol	*MainMethod,
		*Symbol;
sir_symbol_ptr	*Interface;
sir_symbol	*Declared,
		*Implemented;

assert(ClassDecl != NULL);
assert(ClassDecl->ClassScope != NULL);

/* first check if the name reserved for constructor/destructor is used */

if (ClassDecl->ClassScope->FindLocally(ClassDecl->Name.chars()))
   { SIR_ErrMsg.form("Name '%s' is reserved in its class" GL_ERROR_MSG_NEWLINE
		"and cannot be redeclared in its class scope",
			ClassDecl->Name.chars());
     return(SIR_ERROR_RESERVED_SYMBOL_NAME);
    } /* fi */

if (ClassDecl->IsBehavior())	/* behaviors must implement a main method */
   { if (!(MainMethod = ClassDecl->ClassScope->FindLocally(
						GL_MAIN_METHOD_NAME)))
	{ SIR_ErrMsg.form(
		"No " GL_MAIN_METHOD_NAME " method declared in behavior '%s'",
			ClassDecl->Name.chars());
	  return(SIR_ERROR_NO_MAIN_METHOD);
	 } /* fi */
     if (! MainMethod->IsFunction())
	{ SIR_ErrMsg.form(
		"Type mismatch in behavior '%s':" GL_ERROR_MSG_NEWLINE
		"member '" GL_MAIN_METHOD_NAME "' is not a method",
			ClassDecl->Name.chars());
	  return(SIR_ERROR_MEMBER_MAIN_NOT_A_METHOD);
	 } /* fi */
     if (  (0 != strcmp(ClassDecl->Name.chars(), GL_MAIN_BEHAVIOR_NAME))
	 &&(  (MainMethod->Type->SubType->Type != SIR_TYPE_VOID)
	    ||(! MainMethod->Type->VoidParameters())))
	{ SIR_ErrMsg.form(
		"Type mismatch in behavior '%s':" GL_ERROR_MSG_NEWLINE
		"method '" GL_MAIN_METHOD_NAME "' is not 'void main(void)'",
			ClassDecl->Name.chars());
	  return(SIR_ERROR_MAIN_METHOD_TYPE_MISMATCH);
	 } /* fi */
    } /* fi */

if (  (ClassDecl->IsBehavior())	/* behaviors and channels must implement */
    ||(ClassDecl->IsChannel()))	/* the listed interfaces */
   { assert(ClassDecl->Interfaces != NULL);
     Interface = ClassDecl->Interfaces->First();
     while(Interface)	/* in each listed interface */
	{ if (! Interface->Symbol->IsInterfaceDefinition())
	     { SIR_ErrMsg.form(
			"Class '%s' implements incomplete interface '%s'",
					ClassDecl->Name.chars(),
					Interface->Symbol->Name.chars());
	       return(SIR_ERROR_IMPLEMENTS_INCOMPLETE_IF);
	      } /* fi */
	  Declared = Interface->Symbol->ClassScope->First();
	  while(Declared)	/* check each declared method */
	     { assert(Declared->IsFunction());	/* nothing else allowed */
	       if (!(Implemented = ClassDecl->ClassScope->FindLocally(
						Declared->Name.chars())))
		  { SIR_ErrMsg.form(
			"Class '%s' lacks implementation of method"
							GL_ERROR_MSG_NEWLINE
			"'%s' declared in interface '%s'",
					ClassDecl->Name.chars(),
					Declared->Name.chars(),
					Interface->Symbol->Name.chars());
		    return(SIR_ERROR_CLASS_LACKS_IMPLEMENTATION);
		   } /* fi */
	       assert(  (Declared->Type->Type == SIR_TYPE_METHOD)  
		      &&(Implemented->Type->Type == SIR_TYPE_METHOD));
	       if (!(0 == SIR_Types::MethodTypeCmp(Declared->Type,
							Implemented->Type)))
		  { SIR_ErrMsg.form(
			"Type mismatch in class '%s':" GL_ERROR_MSG_NEWLINE
			"method '%s' declared in interface '%s'"
							GL_ERROR_MSG_NEWLINE
			"does not match the implementation",
					ClassDecl->Name.chars(),
					Declared->Name.chars(),
					Interface->Symbol->Name.chars());
		    return(SIR_ERROR_IMPLEMENTATION_TYPE_MISMATCH);
		   } /* fi */
	       Declared = Declared->Succ();
	      } /* elihw */
	  Interface = Interface->Succ();
	 } /* elihw */
    } /* fi */

if (ClassDecl->IsInterface())	/* interfaces have restrictions */
   { Symbol = ClassDecl->ClassScope->First();
     while(Symbol)
	{ if (! Symbol->IsFunction())
	     { SIR_ErrMsg.form(
			"Interface member '%s' is not a function declaration",
				Symbol->Name.chars());
	       return(SIR_ERROR_ILLEGAL_INTERFACE_MEMBER);
	      } /* fi */
	  assert(! Symbol->IsFunctionDefinition()); /* grammar assures this */
	  Symbol = Symbol->Succ();
	 } /* elihw */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbols::DefineClass */

sir_symbol *SIR_Symbols::DeclareLocalState(     /* declares a local state */
        const char      *Name,
        sir_types       *TypeTable)
{
sir_symbol      *State;

assert(Name != NULL);
assert(TypeTable != NULL);

if ((State = FindLocally(Name)))        /* already declared? */
   { assert(State->Type->Type == SIR_TYPE_VOID); /* dummy state type! */
     return(State);
    } /* fi */

return(Insert(new SIR_Symbol(SIR_SYMBOL_IDENTIFIER, Name,
                TypeTable->FindOrInsert(SIR_TYPE_VOID)))); /* state type! */

} /* end of SIR_Symbols::DeclareLocalState */


sir_symbol *SIR_Symbols::DefineLocalState(      /* defines a local state */
        const char      *Name,          /* (may return NULL and SIR_Error) */
        sir_types       *TypeTable,
        sir_statement   *StateBody,
        unsigned int    Line /* = 0 */,
        sir_fileinfo    *FileInfo /* = NULL */)
{
sir_symbol      *State;

assert(Name != NULL);
assert(TypeTable != NULL);
assert(StateBody != NULL);
assert(StateBody->StmntType == SIR_STMNT_COMPOUND);

if ((State = FindLocally(Name)))        /* already declared? */
   { assert(State->Type->Type == SIR_TYPE_VOID); /* dummy state type! */
     if (State->FctBody)                /* already defined? */
        { SIR_ErrMsg.form("Redefinition of local FSM state '%s'", Name);
          SIR_Error = SIR_ERROR_REDEFINITION_OF_STATE;
          return(NULL);
         } /* fi */
     State->FctBody = StateBody;
     State->UpdateLineInfo(Line, FileInfo);
     return(State);
    } /* fi */

State = new SIR_Symbol(SIR_SYMBOL_IDENTIFIER, Name,
                TypeTable->FindOrInsert(SIR_TYPE_VOID)); /* state type! */
State->FctBody = StateBody;
State->UpdateLineInfo(Line, FileInfo);

return(Insert(State));

} /* end of SIR_Symbols::DefineLocalState */


ERROR SIR_Symbols::WriteSC(	/* (re-) generates SpecC source code */
	gl_io		*IO,
	bool		WriteNotes)
{
sir_symbol	*Symbol;
sir_symbol_ptrs	*SortedList;
#ifdef SIR_OPTIMIZE_METHOD_DECLS
sir_symbol_ptrs	*SortedMethodDefs;
#endif /* SIR_OPTIMIZE_METHOD_DECLS */
sir_symbol_ptr	*Ptr;
bool		PutIntro,
		PutIntroInit,
		PutSeparator,
		PutComments;

if (  (ScopeInfo == SIR_SCOPE_GLOBAL)
    ||(ScopeInfo == SIR_SCOPE_CLASS))
   { PutIntroInit = true;	/* separate code blocks */
    } /* fi */
else
   { PutIntroInit = false;	/* combine code blocks */
    } /* esle */
PutSeparator = false;
#ifdef SIR_PUT_COMMENTS_IN_SOURCE
if (ScopeInfo == SIR_SCOPE_GLOBAL)
   { PutComments = true;
    } /* fi */
else
   { PutComments = false;
    } /* esle */
#else /* !SIR_PUT_COMMENTS_IN_SOURCE */
PutComments = false;
#endif /* SIR_PUT_COMMENTS_IN_SOURCE */

/* (1)-(4) User-defined types, declarations and definitions */

if (UserTypes)
   { if ((SIR_Error = UserTypes->WriteSC(IO, WriteNotes, false,
							&PutSeparator)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

/* (5) Type definitions (alias types) */

SortedList = new SIR_SymbolPtrs();
PutIntro = PutIntroInit;
Symbol = First();
while(Symbol)
   { if (  (Symbol->IsTypeDef())
	 &&((  (Symbol->Type->UserType == NULL)	/* except unnamed usertypes */
	     ||(Symbol->Type->UserType->Name != NULL))))
#ifdef SIR_SORT_CODE_BY_LINEINFO
	{ SortedList->InsertByLineInfo(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#else /* !SIR_SORT_CODE_BY_LINEINFO */
	{ SortedList->Append(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#endif /* SIR_SORT_CODE_BY_LINEINFO */
     Symbol = Symbol->Succ();
    } /* elihw */
Ptr = SortedList->First();
while(Ptr)
   { SIR_Symbols::WriteIntro(IO, false,
			&PutIntro, &PutSeparator, PutComments,
			"type synonym definitions");
     if ((SIR_Error = Ptr->Symbol->WriteSC(IO, WriteNotes)))
	{ delete SortedList;
	  return(SIR_Error);
	 } /* fi */
     Ptr = Ptr->Succ();
    } /* elihw */
delete SortedList;

/* (6) Interface declarations */

SortedList = new SIR_SymbolPtrs();
PutIntro = PutIntroInit;
Symbol = First();
while(Symbol)
   { if (Symbol->IsInterface())
#ifdef SIR_SORT_CODE_BY_LINEINFO
	{ SortedList->InsertByLineInfo(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#else /* !SIR_SORT_CODE_BY_LINEINFO */
	{ SortedList->Append(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#endif /* SIR_SORT_CODE_BY_LINEINFO */
     Symbol = Symbol->Succ();
    } /* elihw */
Ptr = SortedList->First();
while(Ptr)
   { SIR_Symbols::WriteIntro(IO, false,
			&PutIntro, &PutSeparator, PutComments,
			"interface declarations");
     if ((SIR_Error = Ptr->Symbol->WriteSC2(IO, WriteNotes, true)))
	{ delete SortedList;
	  return(SIR_Error);
	 } /* fi */
     Ptr = Ptr->Succ();
    } /* elihw */
delete SortedList;

/* (7) Channel declarations */

SortedList = new SIR_SymbolPtrs();
PutIntro = PutIntroInit;
Symbol = First();
while(Symbol)
   { if (Symbol->IsChannel())
#ifdef SIR_SORT_CODE_BY_LINEINFO
	{ SortedList->InsertByLineInfo(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#else /* !SIR_SORT_CODE_BY_LINEINFO */
	{ SortedList->Append(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#endif /* SIR_SORT_CODE_BY_LINEINFO */
     Symbol = Symbol->Succ();
    } /* elihw */
Ptr = SortedList->First();
while(Ptr)
   { SIR_Symbols::WriteIntro(IO, true,
			&PutIntro, &PutSeparator, PutComments,
			"channel declarations");
     if ((SIR_Error = Ptr->Symbol->WriteSC2(IO, WriteNotes, true)))
	{ delete SortedList;
	  return(SIR_Error);
	 } /* fi */
     Ptr = Ptr->Succ();
    } /* elihw */
delete SortedList;

/* (8) Behavior declarations */

SortedList = new SIR_SymbolPtrs();
PutIntro = PutIntroInit;
Symbol = First();
while(Symbol)
   { if (Symbol->IsBehavior())
#ifdef SIR_SORT_CODE_BY_LINEINFO
	{ SortedList->InsertByLineInfo(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#else /* !SIR_SORT_CODE_BY_LINEINFO */
	{ SortedList->Append(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#endif /* SIR_SORT_CODE_BY_LINEINFO */
     Symbol = Symbol->Succ();
    } /* elihw */
Ptr = SortedList->First();
while(Ptr)
   { SIR_Symbols::WriteIntro(IO, true,
			&PutIntro, &PutSeparator, PutComments,
			"behavior declarations");
     if ((SIR_Error = Ptr->Symbol->WriteSC2(IO, WriteNotes, true)))
	{ delete SortedList;
	  return(SIR_Error);
	 } /* fi */
     Ptr = Ptr->Succ();
    } /* elihw */
delete SortedList;

/* (9) Declarations of variables and functions/methods */

#ifdef SIR_OPTIMIZE_METHOD_DECLS
SortedMethodDefs = new SIR_SymbolPtrs(); /* compute this list early! */
Symbol = First();
while(Symbol)
   { if (Symbol->IsFunctionDefinition())
#ifdef SIR_SORT_CODE_BY_LINEINFO
	{ SortedMethodDefs->InsertByLineInfo(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#else /* !SIR_SORT_CODE_BY_LINEINFO */
	{ SortedMethodDefs->Append(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#endif /* SIR_SORT_CODE_BY_LINEINFO */
     Symbol = Symbol->Succ();
    } /* elihw */
#endif /* SIR_OPTIMIZE_METHOD_DECLS */

SortedList = new SIR_SymbolPtrs();
PutIntro = PutIntroInit;
Symbol = First();
while(Symbol)
   { if (  (  (Symbol->IsVariable())	/* IsVariableDeclaration() */
	    &&(Symbol->StorageClass == SIR_STORAGE_EXTERN))
#ifdef SIR_OPTIMIZE_METHOD_DECLS
	 ||(  (Symbol->IsFunction())
	    &&(!Symbol->IsRedundantMethodDecl(SortedMethodDefs, WriteNotes))))
#else /* !SIR_OPTIMIZE_METHOD_DECLS */
	 ||(Symbol->IsFunction()))
#endif /* SIR_OPTIMIZE_METHOD_DECLS */
#ifdef SIR_SORT_CODE_BY_LINEINFO
	{ SortedList->InsertByLineInfo(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#else /* !SIR_SORT_CODE_BY_LINEINFO */
	{ SortedList->Append(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#endif /* SIR_SORT_CODE_BY_LINEINFO */
     Symbol = Symbol->Succ();
    } /* elihw */
Ptr = SortedList->First();
while(Ptr)
   { SIR_Symbols::WriteIntro(IO, false,
			&PutIntro, &PutSeparator, PutComments,
			"variable and function declarations");
     if ((SIR_Error = Ptr->Symbol->WriteSC(IO, WriteNotes, true)))
	{ delete SortedList;
#ifdef SIR_OPTIMIZE_METHOD_DECLS
	  delete SortedMethodDefs;
#endif /* SIR_OPTIMIZE_METHOD_DECLS */
	  return(SIR_Error);
	 } /* fi */
     Ptr = Ptr->Succ();
    } /* elihw */
delete SortedList;

/* (10a) Variable definitions (SORTED) */

#ifndef NDEBUG
Symbol = First();
while(Symbol)
   { assert(Symbol->Color == SIR_WHITE);	/* assure clean start */
     Symbol = Symbol->Succ();
    } /* elihw */
#endif /* NDEBUG */

SortedList = new SIR_SymbolPtrs();
PutIntro = PutIntroInit;
Symbol = First();
while(Symbol)
   { if (Symbol->IsVariableDefinition())
#ifdef SIR_SORT_CODE_BY_LINEINFO
	{ SortedList->InsertByLineInfo(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#else /* !SIR_SORT_CODE_BY_LINEINFO */
	{ SortedList->Append(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#endif /* SIR_SORT_CODE_BY_LINEINFO */
     Symbol = Symbol->Succ();
    } /* elihw */
Ptr = SortedList->First();
while(Ptr)
   { if ((SIR_Error = WriteOrderedSC0(Ptr->Symbol, IO, WriteNotes,
					&PutSeparator, &PutIntro, PutComments)))
	{ delete SortedList;
#ifdef SIR_OPTIMIZE_METHOD_DECLS
	  delete SortedMethodDefs;
#endif /* SIR_OPTIMIZE_METHOD_DECLS */
	  return(SIR_Error);
	 } /* fi */
     Ptr = Ptr->Succ();
    } /* elihw */

Ptr = SortedList->First();
while(Ptr)
   { assert(Ptr->Symbol->Color == SIR_RED);	/* assure work is done */
     Ptr->Symbol->Color = SIR_WHITE;
     Ptr = Ptr->Succ();
    } /* elihw */
delete SortedList;

/* (10b) Class instantiations (SORTED) */

#ifndef NDEBUG
Symbol = First();
while(Symbol)
   { assert(Symbol->Color == SIR_WHITE);	/* assure clean start */
     Symbol = Symbol->Succ();
    } /* elihw */
#endif /* NDEBUG */

SortedList = new SIR_SymbolPtrs();
PutIntro = PutIntroInit;
Symbol = First();
while(Symbol)
   { if (  (Symbol->IsBehaviorInstance())
	 ||(Symbol->IsChannelInstance()))
#ifdef SIR_SORT_CODE_BY_LINEINFO
	{ SortedList->InsertByLineInfo(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#else /* !SIR_SORT_CODE_BY_LINEINFO */
	{ SortedList->Append(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#endif /* SIR_SORT_CODE_BY_LINEINFO */
     Symbol = Symbol->Succ();
    } /* elihw */
Ptr = SortedList->First();
while(Ptr)
   { if ((SIR_Error = WriteOrderedSC1(Ptr->Symbol, IO, WriteNotes,
					&PutSeparator, &PutIntro, PutComments)))
	{ delete SortedList;
#ifdef SIR_OPTIMIZE_METHOD_DECLS
	  delete SortedMethodDefs;
#endif /* SIR_OPTIMIZE_METHOD_DECLS */
	  return(SIR_Error);
	 } /* fi */
     Ptr = Ptr->Succ();
    } /* elihw */

Ptr = SortedList->First();
while(Ptr)
   { assert(Ptr->Symbol->Color == SIR_RED);	/* assure work is done */
     Ptr->Symbol->Color = SIR_WHITE;
     Ptr = Ptr->Succ();
    } /* elihw */
delete SortedList;

/* (11) Interface definitions */

SortedList = new SIR_SymbolPtrs();
PutIntro = PutIntroInit;
Symbol = First();
while(Symbol)
   { if (Symbol->IsInterfaceDefinition())
#ifdef SIR_SORT_CODE_BY_LINEINFO
	{ SortedList->InsertByLineInfo(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#else /* !SIR_SORT_CODE_BY_LINEINFO */
	{ SortedList->Append(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#endif /* SIR_SORT_CODE_BY_LINEINFO */
     Symbol = Symbol->Succ();
    } /* elihw */
Ptr = SortedList->First();
while(Ptr)
   { SIR_Symbols::WriteIntro(IO, true,
			&PutIntro, &PutSeparator, PutComments,
			"interface definitions");
     if ((SIR_Error = Ptr->Symbol->WriteSC2(IO, WriteNotes)))
	{ delete SortedList;
#ifdef SIR_OPTIMIZE_METHOD_DECLS
	  delete SortedMethodDefs;
#endif /* SIR_OPTIMIZE_METHOD_DECLS */
	  return(SIR_Error);
	 } /* fi */
     Ptr = Ptr->Succ();
    } /* elihw */
delete SortedList;

/* (12) Behavior and channel definitions (SORTED) */

#ifndef NDEBUG
Symbol = First();
while(Symbol)
   { assert(Symbol->Color == SIR_WHITE);	/* assure clean start */
     Symbol = Symbol->Succ();
    } /* elihw */
#endif /* NDEBUG */

SortedList = new SIR_SymbolPtrs();
PutIntro = PutIntroInit;
Symbol = First();
while(Symbol)
   { if (  (Symbol->IsBehaviorDefinition())
	 ||(Symbol->IsChannelDefinition()))
#ifdef SIR_SORT_CODE_BY_LINEINFO
	{ SortedList->InsertByLineInfo(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#else /* !SIR_SORT_CODE_BY_LINEINFO */
	{ SortedList->Append(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#endif /* SIR_SORT_CODE_BY_LINEINFO */
     Symbol = Symbol->Succ();
    } /* elihw */
Ptr = SortedList->First();
while(Ptr)
   { if ((SIR_Error = WriteOrderedSC2(Ptr->Symbol, IO, WriteNotes,
					&PutSeparator, &PutIntro, PutComments)))
	{ delete SortedList;
#ifdef SIR_OPTIMIZE_METHOD_DECLS
	  delete SortedMethodDefs;
#endif /* SIR_OPTIMIZE_METHOD_DECLS */
	  return(SIR_Error);
	 } /* fi */
     Ptr = Ptr->Succ();
    } /* elihw */

Ptr = SortedList->First();
while(Ptr)
   { assert(Ptr->Symbol->Color == SIR_RED);	/* assure work is done */
     Ptr->Symbol->Color = SIR_WHITE;
     Ptr = Ptr->Succ();
    } /* elihw */
delete SortedList;

/* (13) Function/Method definitions */

#ifdef SIR_OPTIMIZE_METHOD_DECLS
SortedList = SortedMethodDefs;	/* list computed earlier already */
#else /* !SIR_OPTIMIZE_METHOD_DECLS */
SortedList = new SIR_SymbolPtrs();
Symbol = First();
while(Symbol)
   { if (Symbol->IsFunctionDefinition())
#ifdef SIR_SORT_CODE_BY_LINEINFO
	{ SortedList->InsertByLineInfo(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#else /* !SIR_SORT_CODE_BY_LINEINFO */
	{ SortedList->Append(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#endif /* SIR_SORT_CODE_BY_LINEINFO */
     Symbol = Symbol->Succ();
    } /* elihw */
#endif /* SIR_OPTIMIZE_METHOD_DECLS */
PutIntro = PutIntroInit;
Ptr = SortedList->First();
while(Ptr)
   { SIR_Symbols::WriteIntro(IO, true,
			&PutIntro, &PutSeparator, PutComments,
			"function definitions");
     if ((SIR_Error = Ptr->Symbol->WriteSC3(IO, WriteNotes)))
	{ delete SortedList;
	  return(SIR_Error);
	 } /* fi */
     Ptr = Ptr->Succ();
    } /* elihw */
delete SortedList;

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbols::WriteSC */


ERROR SIR_Symbols::WriteOrderedSC0(	/* writes ordered variables defs. */
	sir_symbol	*Current,	/* (recursively) */
	gl_io		*IO,
	bool		WriteNotes,
	bool		*PutSeparator,
	bool		*PutIntro,
	bool		PutComments)
{
sir_symbol	*Symbol;

if (Current->Color == SIR_RED)		/* already written? */
   { return(SIR_ERROR_NO_ERROR);
    } /* fi */
if (Current->Color == SIR_YELLOW)	/* writing in progress? */
   { SIR_ErrMsg.form(
		"Cyclic dependency in variable definition '%s' (internal)",
			Current->Name.chars());
     return(SIR_ERROR_VARIABLE_CYCLIC_NESTING);
    } /* fi */
Current->Color = SIR_YELLOW;	/* start working on this one */

Symbol = First();
while(Symbol)
   { if (Symbol->IsVariableDefinition())
	{ if (Current->IsTriggeredBy(Symbol))
	     { if ((SIR_Error = WriteOrderedSC0(Symbol, IO, WriteNotes,
					PutSeparator, PutIntro, PutComments)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	 } /* fi */
     Symbol = Symbol->Succ();
    } /* elihw */

SIR_Symbols::WriteIntro(IO, false,
			PutIntro, PutSeparator, PutComments,
			"variable definitions");
if ((SIR_Error = Current->WriteSC(IO, WriteNotes)))
   { return(SIR_Error);
    } /* fi */

Current->Color = SIR_RED;	/* work is done for this one */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbols::WriteOrderedSC0 */


ERROR SIR_Symbols::WriteOrderedSC1(	/* writes ordered instantiations */
	sir_symbol	*Current,	/* (recursively) */
	gl_io		*IO,
	bool		WriteNotes,
	bool		*PutSeparator,
	bool		*PutIntro,
	bool		PutComments)
{
sir_symbol	*Symbol;

if (Current->Color == SIR_RED)		/* already written? */
   { return(SIR_ERROR_NO_ERROR);
    } /* fi */
if (Current->Color == SIR_YELLOW)	/* writing in progress? */
   { SIR_ErrMsg.form(
		"Cyclic dependency in class instantiation '%s' (internal)",
			Current->Name.chars());
     return(SIR_ERROR_INSTANCE_CYCLIC_NESTING);
    } /* fi */
Current->Color = SIR_YELLOW;	/* start working on this one */

Symbol = First();
while(Symbol)
   { if (  (Symbol->IsChannelInstance())
	 ||(Symbol->IsBehaviorInstance()))	/* 08/16/02, RD */
	{ if (Current->MapsPortTo(Symbol))
	     { if ((SIR_Error = WriteOrderedSC1(Symbol, IO, WriteNotes,
					PutSeparator, PutIntro, PutComments)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	 } /* fi */
     Symbol = Symbol->Succ();
    } /* elihw */

SIR_Symbols::WriteIntro(IO, false,
			PutIntro, PutSeparator, PutComments,
			"instance definitions");
if ((SIR_Error = Current->WriteSC(IO, WriteNotes)))
   { return(SIR_Error);
    } /* fi */

Current->Color = SIR_RED;	/* work is done for this one */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbols::WriteOrderedSC1 */


ERROR SIR_Symbols::WriteOrderedSC2(	/* writes ordered class definitions */
	sir_symbol	*Current,	/* (recursively) */
	gl_io		*IO,
	bool		WriteNotes,
	bool		*PutSeparator,
	bool		*PutIntro,
	bool		PutComments)
{
sir_symbol	*Symbol;

if (Current->Color == SIR_RED)		/* already written? */
   { return(SIR_ERROR_NO_ERROR);
    } /* fi */
if (Current->Color == SIR_YELLOW)	/* writing in progress? */
   { SIR_ErrMsg.form(
		"Cyclic nesting in class definition '%s' (internal)",
			Current->Name.chars());
     return(SIR_ERROR_CLASS_CYCLIC_NESTING);
    } /* fi */
Current->Color = SIR_YELLOW;	/* start working on this one */

Symbol = First();
while(Symbol)
   { if (  (Symbol->IsBehaviorDefinition())
	 ||(Symbol->IsChannelDefinition()))
	{ if (Current->Instantiates(Symbol))
	     { if ((SIR_Error = WriteOrderedSC2(Symbol, IO, WriteNotes,
					PutSeparator, PutIntro, PutComments)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	 } /* fi */
     Symbol = Symbol->Succ();
    } /* elihw */

SIR_Symbols::WriteIntro(IO, true,
			PutIntro, PutSeparator, PutComments,
			"behavior and channel definitions");
if ((SIR_Error = Current->WriteSC2(IO, WriteNotes)))
   { return(SIR_Error);
    } /* fi */

Current->Color = SIR_RED;	/* work is done for this one */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbols::WriteOrderedSC2 */


ERROR SIR_Symbols::WriteH(	/* generates C++ header code */
	gl_io		*IO,
	bool		WriteNotes)
{
sir_symbol	*Symbol;
sir_symbol_ptrs	*SortedList;
sir_symbol_ptr	*Ptr;
bool		PutIntro,
		PutIntroInit,
		PutSeparator,
		PutComments;

if (  (ScopeInfo == SIR_SCOPE_GLOBAL)
    ||(ScopeInfo == SIR_SCOPE_CLASS))
   { PutIntroInit = true;	/* separate code blocks */
    } /* fi */
else
   { PutIntroInit = false;	/* combine code blocks */
    } /* esle */
PutSeparator = false;
#ifdef SIR_PUT_COMMENTS_IN_SOURCE
if (ScopeInfo == SIR_SCOPE_GLOBAL)
   { PutComments = true;
    } /* fi */
else
   { PutComments = false;
    } /* esle */
#else /* !SIR_PUT_COMMENTS_IN_SOURCE */
PutComments = false;
#endif /* SIR_PUT_COMMENTS_IN_SOURCE */

/* (1)-(4) User-defined types, declarations and definitions */

if (UserTypes)
   { if ((SIR_Error = UserTypes->WriteSC(IO, WriteNotes, true,
							&PutSeparator)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

/* (5) Type definitions (alias types) */

SortedList = new SIR_SymbolPtrs();
PutIntro = PutIntroInit;
Symbol = First();
while(Symbol)
   { if (  (Symbol->IsTypeDef())
	 &&((  (Symbol->Type->UserType == NULL)	/* except unnamed usertypes */
	     ||(Symbol->Type->UserType->Name != NULL))))
#ifdef SIR_SORT_CODE_BY_LINEINFO
	{ SortedList->InsertByLineInfo(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#else /* !SIR_SORT_CODE_BY_LINEINFO */
	{ SortedList->Append(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#endif /* SIR_SORT_CODE_BY_LINEINFO */
     Symbol = Symbol->Succ();
    } /* elihw */
Ptr = SortedList->First();
while(Ptr)
   {
#ifdef SIR_SUPPRESS_TYPEDEF_WCHAR_T
     /* note: in many standard header files, 'wchar_t' is defined as a */
     /*       'typedef' if the compiler runs in "C" language mode;     */
     /*       in "C++" mode, 'wchar_t' is a keyword, however, and no   */
     /*       such 'typedef' is allowed; so, since we still depend on  */
     /*       the standard header files and those are compiled in "C"  */
     /*       mode, we must not output this 'typedef' definition for   */
     /*       the generated C++ code;                                  */
     /*       thus, skipping this symbol in the C++ code generation is */
     /*       a trick that works in such cases; in a perfect world,    */
     /*       however, we should have our own, clean header files      */
     /*       where such problems don't exist; (RD, 11/19/01)          */
     if (0 == strcmp(Ptr->Symbol->Name.chars(), "wchar_t"))
	{ Ptr = Ptr->Succ();
	  continue;	/* skip this typedef */
	 } /* fi */
#endif /* SIR_SUPPRESS_TYPEDEF_WCHAR_T */
     SIR_Symbols::WriteIntro(IO, false,
			&PutIntro, &PutSeparator, PutComments,
			"type synonym definitions");
     if ((SIR_Error = Ptr->Symbol->WriteSC(IO, WriteNotes, false, true)))
	{ delete SortedList;
	  return(SIR_Error);
	 } /* fi */
     Ptr = Ptr->Succ();
    } /* elihw */
delete SortedList;

/* (6) Class declarations */

SortedList = new SIR_SymbolPtrs();
PutIntro = PutIntroInit;
Symbol = First();
while(Symbol)
   { if (Symbol->IsClass())
#ifdef SIR_SORT_CODE_BY_LINEINFO
	{ SortedList->InsertByLineInfo(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#else /* !SIR_SORT_CODE_BY_LINEINFO */
	{ SortedList->Append(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#endif /* SIR_SORT_CODE_BY_LINEINFO */
     Symbol = Symbol->Succ();
    } /* elihw */
Ptr = SortedList->First();
while(Ptr)
   { SIR_Symbols::WriteIntro(IO, false,
			&PutIntro, &PutSeparator, PutComments,
			"class declarations");
     if ((SIR_Error = Ptr->Symbol->WriteCC2(IO, WriteNotes, true)))
	{ delete SortedList;
	  return(SIR_Error);
	 } /* fi */
     Ptr = Ptr->Succ();
    } /* elihw */
delete SortedList;

/* (7) Interface class definitions */

SortedList = new SIR_SymbolPtrs();
PutIntro = PutIntroInit;
Symbol = First();
while(Symbol)
   { if (Symbol->IsInterfaceDefinition())
#ifdef SIR_SORT_CODE_BY_LINEINFO
	{ SortedList->InsertByLineInfo(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#else /* !SIR_SORT_CODE_BY_LINEINFO */
	{ SortedList->Append(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#endif /* SIR_SORT_CODE_BY_LINEINFO */
     Symbol = Symbol->Succ();
    } /* elihw */
Ptr = SortedList->First();
while(Ptr)
   { SIR_Symbols::WriteIntro(IO, true,
			&PutIntro, &PutSeparator, PutComments,
			"interface class definitions");
     if ((SIR_Error = Ptr->Symbol->WriteCC2(IO, WriteNotes)))
	{ delete SortedList;
	  return(SIR_Error);
	 } /* fi */
     Ptr = Ptr->Succ();
    } /* elihw */
delete SortedList;

/* (8) IP class definitions */

SortedList = new SIR_SymbolPtrs();
PutIntro = PutIntroInit;
Symbol = First();
while(Symbol)
   { if (  (  (Symbol->IsBehavior())	/* behavior and channel */
	    ||(Symbol->IsChannel()))	/* _declarations_ only  */
	 &&(! Symbol->ClassScope))
#ifdef SIR_SORT_CODE_BY_LINEINFO
	{ SortedList->InsertByLineInfo(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#else /* !SIR_SORT_CODE_BY_LINEINFO */
	{ SortedList->Append(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#endif /* SIR_SORT_CODE_BY_LINEINFO */
     Symbol = Symbol->Succ();
    } /* elihw */
Ptr = SortedList->First();
while(Ptr)
   { SIR_Symbols::WriteIntro(IO, true,
			&PutIntro, &PutSeparator, PutComments,
			"IP class definitions");
     if ((SIR_Error = Ptr->Symbol->WriteCC2IP(IO)))
	{ delete SortedList;
	  return(SIR_Error);
	 } /* fi */
     Ptr = Ptr->Succ();
    } /* elihw */
delete SortedList;

/* (9) Behavior and channel class definitions (SORTED) */

#ifndef NDEBUG
Symbol = First();
while(Symbol)
   { assert(Symbol->Color == SIR_WHITE);	/* assure clean start */
     Symbol = Symbol->Succ();
    } /* elihw */
#endif /* NDEBUG */

SortedList = new SIR_SymbolPtrs();
PutIntro = PutIntroInit;
Symbol = First();
while(Symbol)
   { if (  (Symbol->IsBehaviorDefinition())
	 ||(Symbol->IsChannelDefinition()))
#ifdef SIR_SORT_CODE_BY_LINEINFO
	{ SortedList->InsertByLineInfo(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#else /* !SIR_SORT_CODE_BY_LINEINFO */
	{ SortedList->Append(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#endif /* SIR_SORT_CODE_BY_LINEINFO */
     Symbol = Symbol->Succ();
    } /* elihw */
Ptr = SortedList->First();
while(Ptr)
   { if ((SIR_Error = WriteOrderedCC2(Ptr->Symbol, IO, WriteNotes,
					&PutSeparator, &PutIntro, PutComments)))
	{ delete SortedList;
	  return(SIR_Error);
	 } /* fi */
     Ptr = Ptr->Succ();
    } /* elihw */

Ptr = SortedList->First();
while(Ptr)
   { assert(Ptr->Symbol->Color == SIR_RED);	/* assure work is done */
     Ptr->Symbol->Color = SIR_WHITE;
     Ptr = Ptr->Succ();
    } /* elihw */
delete SortedList;

/* (10) Declarations of variables and functions */

SortedList = new SIR_SymbolPtrs();
PutIntro = PutIntroInit;
Symbol = First();
while(Symbol)
   { if (  (  (Symbol->IsVariable())	/* IsVariableDeclaration() */
	    &&(Symbol->StorageClass == SIR_STORAGE_EXTERN))
	 ||(Symbol->IsFunction()))
#ifdef SIR_SORT_CODE_BY_LINEINFO
	{ SortedList->InsertByLineInfo(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#else /* !SIR_SORT_CODE_BY_LINEINFO */
	{ SortedList->Append(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#endif /* SIR_SORT_CODE_BY_LINEINFO */
     Symbol = Symbol->Succ();
    } /* elihw */
Ptr = SortedList->First();
while(Ptr)
   { SIR_Symbols::WriteIntro(IO, false,
			&PutIntro, &PutSeparator, PutComments,
			"variable and function declarations");
     if ((SIR_Error = Ptr->Symbol->WriteSC(IO, WriteNotes, true, true)))
	{ delete SortedList;
	  return(SIR_Error);
	 } /* fi */
     Ptr = Ptr->Succ();
    } /* elihw */
delete SortedList;

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbols::WriteH */


ERROR SIR_Symbols::WriteCC(	/* generates C++ main code */
	gl_io		*IO,
	bool		WriteNotes)
{
sir_symbol	*Symbol;
sir_symbol_ptrs	*SortedList;
sir_symbol_ptr	*Ptr;
bool		PutIntro,
		PutIntroInit,
		PutSeparator,
		PutComments;

if (  (ScopeInfo == SIR_SCOPE_GLOBAL)
    ||(ScopeInfo == SIR_SCOPE_CLASS))
   { PutIntroInit = true;	/* separate code blocks */
    } /* fi */
else
   { PutIntroInit = false;	/* combine code blocks */
    } /* esle */
PutSeparator = false;
#ifdef SIR_PUT_COMMENTS_IN_SOURCE
if (ScopeInfo == SIR_SCOPE_GLOBAL)
   { PutComments = true;
    } /* fi */
else
   { PutComments = false;
    } /* esle */
#else /* !SIR_PUT_COMMENTS_IN_SOURCE */
PutComments = false;
#endif /* SIR_PUT_COMMENTS_IN_SOURCE */

/* (1) Variable definitions (SORTED) */

#ifndef NDEBUG
Symbol = First();
while(Symbol)
   { assert(Symbol->Color == SIR_WHITE);	/* assure clean start */
     Symbol = Symbol->Succ();
    } /* elihw */
#endif /* NDEBUG */

SortedList = new SIR_SymbolPtrs();
PutIntro = PutIntroInit;
Symbol = First();
while(Symbol)
   { if (Symbol->IsVariableDefinition())
#ifdef SIR_SORT_CODE_BY_LINEINFO
	{ SortedList->InsertByLineInfo(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#else /* !SIR_SORT_CODE_BY_LINEINFO */
	{ SortedList->Append(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#endif /* SIR_SORT_CODE_BY_LINEINFO */
     Symbol = Symbol->Succ();
    } /* elihw */
Ptr = SortedList->First();
while(Ptr)
   { if ((SIR_Error = WriteOrderedCC0(Ptr->Symbol, IO, WriteNotes,
					&PutSeparator, &PutIntro, PutComments)))
	{ delete SortedList;
	  return(SIR_Error);
	 } /* fi */
     Ptr = Ptr->Succ();
    } /* elihw */

Ptr = SortedList->First();
while(Ptr)
   { assert(Ptr->Symbol->Color == SIR_RED);	/* assure work is done */
     Ptr->Symbol->Color = SIR_WHITE;
     Ptr = Ptr->Succ();
    } /* elihw */
delete SortedList;

/* (2) Channel implementations */

SortedList = new SIR_SymbolPtrs();
PutIntro = PutIntroInit;
Symbol = First();
while(Symbol)
   { if (Symbol->IsChannelDefinition())
#ifdef SIR_SORT_CODE_BY_LINEINFO
	{ SortedList->InsertByLineInfo(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#else /* !SIR_SORT_CODE_BY_LINEINFO */
	{ SortedList->Append(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#endif /* SIR_SORT_CODE_BY_LINEINFO */
     Symbol = Symbol->Succ();
    } /* elihw */
Ptr = SortedList->First();
while(Ptr)
   { SIR_Symbols::WriteIntro(IO, true,
			&PutIntro, &PutSeparator, PutComments,
			"channel class definitions");
     if ((SIR_Error = Ptr->Symbol->WriteCC2b(IO, WriteNotes)))
	{ delete SortedList;
	  return(SIR_Error);
	 } /* fi */
     Ptr = Ptr->Succ();
    } /* elihw */
delete SortedList;

/* (3) Behavior implementations */

SortedList = new SIR_SymbolPtrs();
PutIntro = PutIntroInit;
Symbol = First();
while(Symbol)
   { if (Symbol->IsBehaviorDefinition())
#ifdef SIR_SORT_CODE_BY_LINEINFO
	{ SortedList->InsertByLineInfo(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#else /* !SIR_SORT_CODE_BY_LINEINFO */
	{ SortedList->Append(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#endif /* SIR_SORT_CODE_BY_LINEINFO */
     Symbol = Symbol->Succ();
    } /* elihw */
Ptr = SortedList->First();
while(Ptr)
   { SIR_Symbols::WriteIntro(IO, true,
			&PutIntro, &PutSeparator, PutComments,
			"behavior class definitions");
     if ((SIR_Error = Ptr->Symbol->WriteCC2b(IO, WriteNotes)))
	{ delete SortedList;
	  return(SIR_Error);
	 } /* fi */
     Ptr = Ptr->Succ();
    } /* elihw */
delete SortedList;

/* (4) Function definitions */

SortedList = new SIR_SymbolPtrs();
PutIntro = PutIntroInit;
Symbol = First();
while(Symbol)
   { if (Symbol->IsFunctionDefinition())
#ifdef SIR_SORT_CODE_BY_LINEINFO
	{ SortedList->InsertByLineInfo(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#else /* !SIR_SORT_CODE_BY_LINEINFO */
	{ SortedList->Append(new SIR_SymbolPtr(Symbol));
	 } /* fi */
#endif /* SIR_SORT_CODE_BY_LINEINFO */
     Symbol = Symbol->Succ();
    } /* elihw */
Ptr = SortedList->First();
while(Ptr)
   { SIR_Symbols::WriteIntro(IO, true,
			&PutIntro, &PutSeparator, PutComments,
			"function definitions");
     if ((SIR_Error = Ptr->Symbol->WriteSC3(IO, WriteNotes, true)))
	{ delete SortedList;
	  return(SIR_Error);
	 } /* fi */
     Ptr = Ptr->Succ();
    } /* elihw */
delete SortedList;

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbols::WriteCC */


ERROR SIR_Symbols::WriteOrderedCC0(	/* writes ordered C++ variable defs. */
	sir_symbol	*Current,	/* (recursively) */
	gl_io		*IO,
	bool		WriteNotes,
	bool		*PutSeparator,
	bool		*PutIntro,
	bool		PutComments)
{
sir_symbol	*Symbol;

if (Current->Color == SIR_RED)		/* already written? */
   { return(SIR_ERROR_NO_ERROR);
    } /* fi */
if (Current->Color == SIR_YELLOW)	/* writing in progress? */
   { SIR_ErrMsg.form("Cyclic nesting in variable definition '%s' (internal)",
			Current->Name.chars());
     return(SIR_ERROR_VARIABLE_CYCLIC_NESTING);
    } /* fi */
Current->Color = SIR_YELLOW;	/* start working on this one */

Symbol = First();
while(Symbol)
   { if (Symbol->IsVariableDefinition())
	{ if (Current->IsTriggeredBy(Symbol))
	     { if ((SIR_Error = WriteOrderedCC0(Symbol, IO, WriteNotes,
					PutSeparator, PutIntro, PutComments)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	 } /* fi */
     Symbol = Symbol->Succ();
    } /* elihw */

SIR_Symbols::WriteIntro(IO, false,
			PutIntro, PutSeparator, PutComments,
			"variable definitions");
if ((SIR_Error = Current->WriteSC(IO, WriteNotes, false, true)))
   { return(SIR_Error);
    } /* fi */

Current->Color = SIR_RED;	/* work is done for this one */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbols::WriteOrderedCC0 */


ERROR SIR_Symbols::WriteOrderedCC2(	/* writes ordered C++ class defs. */
	sir_symbol	*Current,	/* (recursively) */
	gl_io		*IO,
	bool		WriteNotes,
	bool		*PutSeparator,
	bool		*PutIntro,
	bool		PutComments)
{
sir_symbol	*Symbol;

if (Current->Color == SIR_RED)		/* already written? */
   { return(SIR_ERROR_NO_ERROR);
    } /* fi */
if (Current->Color == SIR_YELLOW)	/* writing in progress? */
   { SIR_ErrMsg.form("Cyclic nesting in class definition '%s' (internal)",
			Current->Name.chars());
     return(SIR_ERROR_CLASS_CYCLIC_NESTING);
    } /* fi */
Current->Color = SIR_YELLOW;	/* start working on this one */

Symbol = First();
while(Symbol)
   { if (  (Symbol->IsBehaviorDefinition())
	 ||(Symbol->IsChannelDefinition()))
	{ if (Current->Instantiates(Symbol))
	     { if ((SIR_Error = WriteOrderedCC2(Symbol, IO, WriteNotes,
					PutSeparator, PutIntro, PutComments)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	 } /* fi */
     Symbol = Symbol->Succ();
    } /* elihw */

SIR_Symbols::WriteIntro(IO, true,
			PutIntro, PutSeparator, PutComments,
			"behavior and channel class definitions");
if ((SIR_Error = Current->WriteCC2(IO, WriteNotes)))
   { return(SIR_Error);
    } /* fi */

Current->Color = SIR_RED;	/* work is done for this one */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Symbols::WriteOrderedCC2 */


void SIR_Symbols::WriteIntro(	/* create spacing, etc. in front of items */
	gl_io		*IO,
	bool		SeparateItems,
	bool		*PutIntro,
	bool		*PutSeparator,
	bool		PutComment,
	const char	*Comment)
{
unsigned int	i;

if (SeparateItems)
   { if (PutComment)
	{ if (*PutSeparator)
	     { if (*PutIntro)
		  { SIR_LineInfo::WriteNL(IO);
		   } /* fi */
	       else
		  { SIR_LineInfo::WriteVSPACE(IO);
		   } /* esle */
	      } /* fi */
	  *PutSeparator = true;
	  if (*PutIntro)
	     { *PutIntro = false;
	       SIR_LineInfo::WriteNL(IO);
	       IO->PutS("// ");
	       IO->PutS(Comment);
	       IO->PutC(' ');
	       for(i=3+strlen(Comment)+1; i<IO->GetLineWrap(); i++)
		  { IO->PutC('/');
		   } /* rof */
	       SIR_LineInfo::WriteNL(IO);
	      } /* fi */
	 } /* fi */
     else
	{ if (*PutSeparator)
	     { SIR_LineInfo::WriteVSPACE(IO);
	      } /* fi */
	  *PutSeparator = true;
//	  if (*PutIntro)
//	     { *PutIntro = false;
//	      } /* fi */
	 } /* esle */
    } /* fi */
else
   { if (*PutIntro)
	{ if (PutComment)
	     { if (*PutSeparator)
		  { SIR_LineInfo::WriteNL(IO);
		   } /* fi */
	       SIR_LineInfo::WriteNL(IO);
	       IO->PutS("// ");
	       IO->PutS(Comment);
	       IO->PutC(' ');
	       for(i=3+strlen(Comment)+1; i<IO->GetLineWrap(); i++)
		  { IO->PutC('/');
		   } /* rof */
	       SIR_LineInfo::WriteNL(IO);
	      } /* fi */
	  else
	     { if (*PutSeparator)
		  { SIR_LineInfo::WriteVSPACE(IO);
		   } /* fi */
	      } /* esle */
	  *PutIntro = false;
	  *PutSeparator = true;
	 } /* fi */
    } /* esle */

} /* end of SIR_Symbols::WriteIntro */


	/*********************/
	/*** SIR_SymbolPtr ***/
	/*********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_SymbolPtr::SIR_SymbolPtr(			/* constructor #1 */
	sir_symbol	*Symbol)
{

assert(Symbol != NULL);

SIR_SymbolPtr::Symbol	= Symbol;

} /* end of SIR_SymbolPtr::SIR_SymbolPtr #1 */


SIR_SymbolPtr::SIR_SymbolPtr(		/* constructor #3 (duplicator) */
	sir_symbol_ptr	*Original)
{

SIR_SymbolPtr::Symbol	= Original->Symbol;

} /* end of SIR_SymbolPtr::SIR_SymbolPtr #3 */


SIR_SymbolPtr::~SIR_SymbolPtr(void)		/* destructor */
{

/* nothing to do */

} /* end of SIR_SymbolPtr::~SIR_SymbolPtr */


	/**********************/
	/*** SIR_SymbolPtrs ***/
	/**********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_SymbolPtrs::SIR_SymbolPtrs(		/* constructor #1 */
	sir_symbol_ptr	*FirstEntry /* = NULL */) :
		SIR_List<SIR_SymbolPtr>(FirstEntry)
{

/* nothing else to do */

} /* end of SIR_SymbolPtrs::SIR_SymbolPtrs #1 */


SIR_SymbolPtrs::SIR_SymbolPtrs(		/* constructor #2 (duplicator) */
	sir_symbol_ptrs	*Original)
{
sir_symbol_ptr	*SymbolPtr;

assert(Original != NULL);

SymbolPtr = Original->First();
while(SymbolPtr)
   { Append(new SIR_SymbolPtr(SymbolPtr));
     SymbolPtr = SymbolPtr->Succ();
    } /* elihw */

} /* end of SIR_SymbolPtrs::SIR_SymbolPtrs #2 */


SIR_SymbolPtrs::~SIR_SymbolPtrs(void)	/* destructor */
{

/* nothing to do */

} /* end of SIR_SymbolPtrs::~SIR_SymbolPtrs */


void SIR_SymbolPtrs::UnAlias(void)/* unalias all type, usertype, symbol links */
{
sir_symbol_ptr	*SymbolPtr;

SymbolPtr = First();
while(SymbolPtr)
   { if (  (SymbolPtr->Symbol)
	 &&(SymbolPtr->Symbol->Alias))
	{ SymbolPtr->Symbol = SymbolPtr->Symbol->Alias;
	 } /* fi */
     SymbolPtr = SymbolPtr->Succ();
    } /* elihw */

} /* end of SIR_SymbolPtrs::UnAlias */


int SIR_SymbolPtrs::CmpSymbolNames(		/* compare all symbol names */
	sir_symbol_ptrs	*Others)		/* (similar to strcmp) */
{
sir_symbol_ptr	*ThisSymbPtr;
sir_symbol_ptr	*OtherSymbPtr;
int		CmpVal;

ThisSymbPtr = First();
OtherSymbPtr = Others->First();
while(ThisSymbPtr)
   { if (!(OtherSymbPtr))
	{ return(1);	/* this list has more entries */
	 } /* fi */
     if ((CmpVal = strcmp(ThisSymbPtr->Symbol->Name.chars(),
			OtherSymbPtr->Symbol->Name.chars())))
	{ return(CmpVal);
	 } /* fi */
     ThisSymbPtr = ThisSymbPtr->Succ();
     OtherSymbPtr = OtherSymbPtr->Succ();
    } /* elihw */
if (OtherSymbPtr)
   { return(-1);	/* this list has less entries */
    } /* fi */

return(0);	/* the lists are equal */

} /* end of SIR_SymbolPtrs::CmpSymbolNames */


void SIR_SymbolPtrs::InsertByLineInfo(	/* sort symbols by source line info */
	sir_symbol_ptr	*SymbolPtr)
{
sir_symbol_ptr	*Ptr;

assert(SymbolPtr != NULL);

Ptr = Last();
while(Ptr)
   { if (SIR_Node::CmpLineInfo(Ptr->Symbol, SymbolPtr->Symbol) <= 0)
	{ break;
	 } /* fi */
     Ptr = Ptr->Pred();
    } /* elihw */
if (Ptr)
   { InsertAfter(SymbolPtr, Ptr);
    } /* fi */
else
   { Prepend(SymbolPtr);
    } /* esle */

} /* end of SIR_SymbolPtrs::InsertByLineInfo */


sir_symbol_ptr *SIR_SymbolPtrs::Find(	/* searches for a specific entry */
	sir_symbol	*Symbol)	/* (returns NULL if not found) */
{
sir_symbol_ptr	*Ptr;

Ptr = First();
while(Ptr)
   { if (Ptr->Symbol == Symbol)
	{ break;	/* found! */
	 } /* fi */
     Ptr = Ptr->Succ();
    } /* elihw */

return(Ptr);

} /* end of SIR_SymbolPtrs::Find */


/*** internal functions *************************************************/


static void MarkImplementedMethods(	/* mark all methods with this name */
	const char	*Name,
	sir_symbol_ptr	*InterfaceList)
{
sir_symbol_ptr	*Interface;
sir_symbol	*Method;

Interface = InterfaceList;
while(Interface)
   { if (Interface->Symbol->ClassScope)
	{ Method = Interface->Symbol->ClassScope->First();
	  while(Method)
	     { if (0 == strcmp(Method->Name.chars(), Name))
		  { Method->Color = SIR_RED;	/* put marker */
		    break;
		   } /* fi */
	       Method = Method->Succ();
	      } /* elihw */
	 } /* fi */
     Interface = Interface->Succ();
    } /* elihw */

} /* end of MarkImplementedMethods */


static sir_type *FixConstArrayTypeInClass( /* bug fix 04/02/03, RD */
	sir_type	*Type,			/* (recursive!) */
	SIR_SCOPE	Scope)
{

assert(Type != NULL);

// note: C++ does not provide any way to initialize an array of
//       const elements in a class (unless it is static);
//       in order to allow such members in behaviors or channels,
//       we cheat, that is, we suppress the 'const' qualifier of
//       such types when generating C++ code

if (  (Scope != SIR_SCOPE_CLASS)
    ||(Type->Type != SIR_TYPE_ARRAY))
   { return(Type);
    } /* fi */

if (Type->SubType->Type == SIR_TYPE_ARRAY)
   { return(Type->GetTable()->FindOrInsert(
				SIR_TYPE_ARRAY,
				FixConstArrayTypeInClass(Type->SubType, Scope),
				NULL, Type->Size));
    } /* fi */
else
   { if (Type->SubType->Const)
	{ /* use copy without the 'const' flag */
	  return(Type->GetTable()->FindOrInsert(
				SIR_TYPE_ARRAY,
				Type->SubType->Modified(false), // no const!
				NULL, Type->Size));
	 } /* fi */
     else
	{ return(Type);	// element type is not const, no change
	 } /* esle */
    } /* esle */

} /* end of FixConstArrayTypeInClass */


/* EOF Symbol.cc */
