/************************************************************************/
/* Import.cc: SpecC Internal Representation, Imported File Class	*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 02/10/98 */
/************************************************************************/

/* last update: 09/26/06 */

/* modifications: (most recent first)
 *
 * 09/26/06 PC  Adjustments for scrc 2.1
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 11/21/01 RD	took out default arguments from function definitions
 * 05/31/01 RD	eliminated level 2 of SIR API
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 04/30/01 RD	replaced use of obsolete form() from "stream.h" with own one
 * 04/30/01 RD	created this header (last change was 12/16/98)
 */


#include "IntRep/Import.h"
#include "IntRep/Design.h"

#include <stdlib.h>
#include <assert.h>


/*** constants and macros ***********************************************/


//#define SIR_DEBUG_IMPORT_GARBAGE_COLLECTOR	/* debug garbage collector */


/*** internal type declarations *****************************************/


	/* (none) */


/*** class implementations **********************************************/


	/******************/
	/*** SIR_Import ***/
	/******************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Import::SIR_Import(			/* constructor #1 */
	const char	*ImportName)
{

assert(ImportName != NULL);

SIR_Import::ImportName	= ImportName;
SIR_Import::Alias	= NULL;

} /* end of SIR_Import::SIR_Import #1 */


SIR_Import::SIR_Import(			/* constructor #3 (duplicator) */
	sir_import	*Original)
{

SIR_Import::ImportName	= Original->ImportName;
SIR_Import::Alias	= NULL;

} /* end of SIR_Import::SIR_Import #3 */


SIR_Import::~SIR_Import(void)		/* destructor */
{

/* nothing to do */

} /* end of SIR_Import::~SIR_Import */


void SIR_Import::TakeOverAndRemove(	/* takes over these imports */
	sir_design	*Design)	/* and removes this import entry */
{

Design->DFS_ForAllSymbols(&SIR_Symbol::TakeOverImport, this);
Design->DFS_ForAllUserTypes(&SIR_UserType::TakeOverImport, this);
Design->DFS_ForAllNotes(&SIR_Note::TakeOverImport, this);

Remove();	/* take me out of the list */
delete this;	/* and commit suicide */

} /* end of SIR_Import::TakeOverAndRemove */


	/**********************/
	/*** SIR_ImportList ***/
	/**********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_ImportList::SIR_ImportList(		/* constructor #1 */
	sir_import	*FirstEntry /* = NULL */) :
		SIR_List<SIR_Import>(FirstEntry)
{

/* nothing else to do */

} /* end of SIR_ImportList::SIR_ImportList #1 */


SIR_ImportList::SIR_ImportList(		/* constructor #2 (duplicator) */
	sir_import_list	*Original)
{
sir_import	*Curr;

Curr = Original->First();
while(Curr)
   { Append(new SIR_Import(Curr));
     Curr = Curr->Succ();
    } /* elihw */

} /* end of SIR_ImportList::SIR_ImportList #2 */


SIR_ImportList::~SIR_ImportList(void)	/* destructor */
{

/* nothing to do */

} /* end of SIR_ImportList::~SIR_ImportList */


ERROR SIR_ImportList::Integrate(	/* merges with imported import list */
	sir_import_list	*ImportedList)
{
sir_import	*NewImport,
		*NewImportSucc;

/* note: in a perfect world this would be a (efficient) parallel traversal */

NewImport = ImportedList->First();
while(NewImport)
   { NewImportSucc = NewImport->Succ();
     if (!(NewImport->Alias = Find(NewImport->ImportName.chars())))
	{ if (Curr())
	     { InsertBefore(ImportedList->Remove(NewImport));
	      } /* fi */
	  else
	     { Append(ImportedList->Remove(NewImport));
	      } /* esle */
	 } /* fi */
     NewImport = NewImportSucc;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_ImportList::Integrate */


void SIR_ImportList::GarbageCollection(		/* garbage collector */
	sir_design	*Design)
{
sir_import	*Curr,
		*Succ;

#ifdef SIR_DEBUG_IMPORT_GARBAGE_COLLECTOR
fprintf(stderr, "GARBAGE_COLLECTOR: Starting with %d import entries...\n",
			NumElements());
#endif /* SIR_DEBUG_IMPORT_GARBAGE_COLLECTOR */

#ifndef NDEBUG
Curr = First();		/* make sure every entry is unmarked */
while(Curr)
   { assert(Curr->Color == SIR_WHITE);
     Curr = Curr->Succ();
    } /* elihw */
#endif /* NDEBUG */

Design->DFS_ForAllSymbols(&SIR_Symbol::MarkImportEntry, NULL);
Design->DFS_ForAllUserTypes(&SIR_UserType::MarkImportEntry, NULL);
Design->DFS_ForAllNotes(&SIR_Note::MarkImportEntry, NULL);

Curr = First();		/* remove every unused entry (unmarked) */
while(Curr)
   { Succ = Curr->Succ();
     if (Curr->Color != SIR_WHITE)
	{ assert(Curr->Color == SIR_RED);	/* marked = used */
	  Curr->Color = SIR_WHITE;
	 } /* fi */
     else					/* unmarked = not used */
	{
#ifdef SIR_DEBUG_IMPORT_GARBAGE_COLLECTOR
	  fprintf(stderr, "GARBAGE_COLLECTOR: Detected unused import: \"%s\"\n",
				Curr->ImportName.chars());
#endif /* SIR_DEBUG_IMPORT_GARBAGE_COLLECTOR */
	  Remove(Curr);
	  delete Curr;
	 } /* esle */
     Curr = Succ;
    } /* elihw */

#ifdef SIR_DEBUG_IMPORT_GARBAGE_COLLECTOR
fprintf(stderr, "GARBAGE_COLLECTOR: Ending with %d import entries...\n",
			NumElements());
#endif /* SIR_DEBUG_IMPORT_GARBAGE_COLLECTOR */

} /* end of SIR_ImportList::GarbageCollection */


sir_import *SIR_ImportList::FindOrInsert(	/* find an entry or insert it */
	const char	*ImportName)		/* if it not exists           */
{
sir_import	*Entry;

if ((Entry = Find(ImportName)))
   { return(Entry);
    } /* fi */

if (Curr())
   { return(InsertBefore(new SIR_Import(ImportName)));
    } /* fi */
else
   { return(Append(new SIR_Import(ImportName)));
    } /* esle */

} /* end of SIR_ImportList::FindOrInsert */


sir_import *SIR_ImportList::Insert(		/* insert a new entry */
	const char	*ImportName)
{

if (Find(ImportName))
   { assert(false);	/* already exists! */
    } /* fi */

if (Curr())
   { return(InsertBefore(new SIR_Import(ImportName)));
    } /* fi */
else
   { return(Append(new SIR_Import(ImportName)));
    } /* esle */

} /* end of SIR_ImportList::Insert */


sir_import *SIR_ImportList::Find(		/* find an entry */
	const char	*ImportName)
{
int		CmpVal;

First();	/* search for the entry */
while(Curr())
   { if (0 == (CmpVal = strcmp(ImportName, Curr()->ImportName.chars())))
	{ return(Curr());	/* found the entry */
	 } /* fi */
     if (CmpVal < 0)
	{ break;	/* not found (insert here) */
	 } /* fi */
     Next();
    } /* elihw */

return(NULL);	/* not found */

} /* end of SIR_ImportList::Find */


/* EOF Import.cc */
