/************************************************************************/
/* FsmdStmnt.cc: SpecC Internal Representation, FsmdStmnt Class		*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 01/30/03 */
/************************************************************************/

/* last update: 09/26/06 */

/* modifications: (most recent first)
 *
 * 09/26/06 PC  Adjustments for scrc 2.1
 * 06/03/05 RD	reorganized and renamed global type names
 * 01/14/05 RD	added inclusion of IntRep/DelydAssgn.h
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 11/12/03 RD	bug fix: ParentStmnt pointer invalid in WrapInCompound()
 * 11/05/03 RD	added method SIR_FsmdStmnt::GetScope()
 * 10/31/03 RD	reworked state reachability analysis (i.e. ReachableStates())
 * 10/30/03 RD	completed 'piped' and 'after' support under 'switch' statements
 * 10/30/03 RD	added method GetSwitchStmnt() and some control-flow analysis
 * 10/29/03 RD	added member ParentStmnt to SIR_FsmdStmnt
 * 10/24/03 RD	use SIR_Expression::IsInvalidInScope() to validate
 *		attachments of expressions to the statement tree
 * 10/24/03 RD	completed support for PIPED and AFTER in fsmd statements
 * 10/21/03 RD	added TmpVar member and comment about possible NULL value
 *		returned by GetStatement()
 * 10/03/03 RD	more support for PIPED and AFTER in fsmd statements
 * 09/12/03 RD	added support for PIPED and AFTER in fsmd statements
 * 05/29/03 RD	bug fix: made iterator SIR_FsmdStmnt::DFS_ForAllFsmdStmnts()
 *		safe with respect to the deletion of the current leaf object
 *		(since it is used with conventions of API layer 2 iterators)
 * 03/03/03 RD	extended FindAccess(), ReplaceAccess(), DeleteAccess()
 *		to cover 'buffered' variables and 'fsmd' statements
 * 02/26/03 RD	refined SIR_Statement::Print() for use with the GUI
 * 02/22/03 RD	extended SIR_FsmdStmnt::Print() for use at API layer 2
 * 02/21/03 RD	fixed the dependency analysis among symbols with respect to
 *		the now supported local variables in functions
 * 02/13/03 RD	added support for annotations at states
 * 02/13/03 RD	added ParentFsmdState link to each scope
 * 02/11/03 RD	adjusted C++ code generation
 * 02/07/03 RD	refined/extended/added methods
 * 02/05/03 RD	adjustments for SIR_FsmdState and SIR_Statement extensions
 * 02/04/03 RD	initial version (based on Statement.cc)
 */


#include "IntRep/FsmdStmnt.h"
#include "IntRep/Design.h"
#include "IntRep/Extern.h"

#include <assert.h>


/*** constants and macros ***********************************************/


/*** internal type declarations *****************************************/



/*** prototypes of internal functions ***********************************/


static sir_expression *NewTmpAssign(	/* create a new temp. assignment */
	sir_tmpvar	*TmpVar,
	int		Index,
	sir_expression	*SourceExpr);


/*** class implementations **********************************************/


	/*********************/
	/*** SIR_FsmdStmnt ***/
	/*********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_FsmdStmnt::SIR_FsmdStmnt(		/* constructor #1 */
	SIR_FSMDSTMNT	StmntType,
	sir_symbol	*Function,
	sir_expression	*Expression /* = NULL */,
	sir_fsmdstmnt	*FsmdStmnt1 /* = NULL */,
	sir_fsmdstmnt	*FsmdStmnt2 /* = NULL */,
	sir_label	*Label /* = NULL */,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */,
	int		Cycles /* = 0 */) :
		SIR_Node(Line, FileInfo)
{

assert(Function != NULL);

switch(StmntType)
   { case SIR_FSMDSTMNT_COMPOUND:
	{ assert(false);	/* wrong constructor */
	 }
     case SIR_FSMDSTMNT_EXPRESSION:
	{ assert(Expression != NULL);
	  assert(FsmdStmnt1 == NULL);
	  assert(FsmdStmnt2 == NULL);
	  assert(Cycles == 0);
	  break;
	 }
     case SIR_FSMDSTMNT_AFTER_EXPR:
     case SIR_FSMDSTMNT_PIPED_EXPR:
	{ assert(Expression != NULL);
	  assert(FsmdStmnt1 == NULL);
	  assert(FsmdStmnt2 == NULL);
	  assert(Cycles >= 0);
	  break;
	 }
     case SIR_FSMDSTMNT_IF:
	{ assert(Expression != NULL);
	  assert(FsmdStmnt1 != NULL);
	  assert(FsmdStmnt2 == NULL);
	  assert(Cycles == 0);
	  break;
	 }
     case SIR_FSMDSTMNT_IF_ELSE:
	{ assert(Expression != NULL);
	  assert(FsmdStmnt1 != NULL);
	  assert(FsmdStmnt2 != NULL);
	  assert(Cycles == 0);
	  break;
	 }
     case SIR_FSMDSTMNT_SWITCH:
	{ assert(Expression != NULL);
	  assert(FsmdStmnt1 != NULL);
	  assert(FsmdStmnt2 == NULL);
	  assert(Cycles == 0);
	  break;
	 }
     case SIR_FSMDSTMNT_CASE:
	{ assert(false);	/* wrong constructor */
	 }
     case SIR_FSMDSTMNT_DEFAULT:
	{ assert(Expression == NULL);
	  assert(FsmdStmnt1 != NULL);
	  assert(FsmdStmnt2 == NULL);
	  assert(Cycles == 0);
	  break;
	 }
     case SIR_FSMDSTMNT_GOTO:
	{ assert(false);	/* wrong constructor */
	 }
     case SIR_FSMDSTMNT_BREAK:
	{ assert(Expression == NULL);
	  assert(FsmdStmnt1 == NULL);
	  assert(FsmdStmnt2 == NULL);
	  assert(Cycles == 0);
	  break;
	 }
     default:
	{ assert(false);	/* bad StmntType */
	 }
    } /* hctiws */

SIR_FsmdStmnt::StmntType	= StmntType;
SIR_FsmdStmnt::Function		= Function;
SIR_FsmdStmnt::ParentStmnt	= NULL;
SIR_FsmdStmnt::Expression	= Expression;
SIR_FsmdStmnt::Cycles		= Cycles;
SIR_FsmdStmnt::Constant		= NULL;
SIR_FsmdStmnt::FsmdStmnt1	= FsmdStmnt1;
SIR_FsmdStmnt::FsmdStmnt2	= FsmdStmnt2;
SIR_FsmdStmnt::Label		= Label;
SIR_FsmdStmnt::Scope		= NULL;
SIR_FsmdStmnt::FsmdStmnts	= NULL;
SIR_FsmdStmnt::NextState	= NULL;
SIR_FsmdStmnt::TmpVar		= NULL;

if (FsmdStmnt1)
   { FsmdStmnt1->ParentStmnt = this;	// automatically set the parent link
    } /* fi */
if (FsmdStmnt2)
   { FsmdStmnt2->ParentStmnt = this;	// automatically set the parent link
    } /* fi */

} /* end of SIR_FsmdStmnt::SIR_FsmdStmnt #1 */


SIR_FsmdStmnt::SIR_FsmdStmnt(		/* constructor #2 */
	SIR_FSMDSTMNT	StmntType,	/* (for compound statements) */
	sir_symbol	*Function,
	sir_symbols	*Scope,
	sir_fsmdstmnts	*FsmdStmnts,
	sir_label	*Label /* = NULL */,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{
sir_fsmdstmnt	*Stmnt;

assert(StmntType == SIR_FSMDSTMNT_COMPOUND);
assert(Function != NULL);
assert(Scope != NULL);
assert(Scope->ParentSymbol != NULL);	/* we always want a parent function */
/* FsmdStmnts is temorary NULL in Parser */

SIR_FsmdStmnt::StmntType	= StmntType;
SIR_FsmdStmnt::Function		= Function;
SIR_FsmdStmnt::ParentStmnt	= NULL;
SIR_FsmdStmnt::Expression	= NULL;
SIR_FsmdStmnt::Cycles		= 0;
SIR_FsmdStmnt::Constant		= NULL;
SIR_FsmdStmnt::FsmdStmnt1	= NULL;
SIR_FsmdStmnt::FsmdStmnt2	= NULL;
SIR_FsmdStmnt::Label		= Label;
SIR_FsmdStmnt::Scope		= Scope;
SIR_FsmdStmnt::FsmdStmnts	= FsmdStmnts;
SIR_FsmdStmnt::NextState	= NULL;
SIR_FsmdStmnt::TmpVar		= NULL;

Scope->ParentFsmdStmnt = this;	/* automatically fill in the back pointer */

if (FsmdStmnts)
   { FsmdStmnts->CmpndScope = Scope;	/* fill in compound scope link */
     Stmnt = FsmdStmnts->First();
     while(Stmnt)
	{ Stmnt->ParentStmnt = this;	// automatically set the parent link
	  Stmnt = Stmnt->Succ();
	 } /* elihw */
    } /* fi */

} /* end of SIR_FsmdStmnt::SIR_FsmdStmnt #2 */


SIR_FsmdStmnt::SIR_FsmdStmnt(		/* constructor #3 */
	SIR_FSMDSTMNT	StmntType,	/* (for case statements) */
	sir_symbol	*Function,
	sir_constant	*Constant,
	sir_fsmdstmnt	*FsmdStmnt1 /* = NULL */,
	sir_label	*Label /* = NULL */,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(StmntType == SIR_FSMDSTMNT_CASE);
assert(Function != NULL);
assert(Constant != NULL);

SIR_FsmdStmnt::StmntType	= StmntType;
SIR_FsmdStmnt::Function		= Function;
SIR_FsmdStmnt::ParentStmnt	= NULL;
SIR_FsmdStmnt::Expression	= NULL;
SIR_FsmdStmnt::Cycles		= 0;
SIR_FsmdStmnt::Constant		= Constant;
SIR_FsmdStmnt::FsmdStmnt1	= FsmdStmnt1;
SIR_FsmdStmnt::FsmdStmnt2	= NULL;
SIR_FsmdStmnt::Label		= Label;
SIR_FsmdStmnt::Scope		= NULL;
SIR_FsmdStmnt::FsmdStmnts	= NULL;
SIR_FsmdStmnt::NextState	= NULL;
SIR_FsmdStmnt::TmpVar		= NULL;

if (FsmdStmnt1)
   { FsmdStmnt1->ParentStmnt = this;	// automatically set the parent link
    } /* fi */

} /* end of SIR_FsmdStmnt::SIR_FsmdStmnt #3 */


SIR_FsmdStmnt::SIR_FsmdStmnt(		/* constructor #4 */
	SIR_FSMDSTMNT	StmntType,	/* (for goto statements) */
	sir_symbol	*Function,
	sir_fsmdstate	*NextState,
	sir_label	*Label /* = NULL */,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(StmntType == SIR_FSMDSTMNT_GOTO);
assert(Function != NULL);
assert(NextState != NULL);

SIR_FsmdStmnt::StmntType	= StmntType;
SIR_FsmdStmnt::Function		= Function;
SIR_FsmdStmnt::ParentStmnt	= NULL;
SIR_FsmdStmnt::Expression	= NULL;
SIR_FsmdStmnt::Cycles		= 0;
SIR_FsmdStmnt::Constant		= NULL;
SIR_FsmdStmnt::FsmdStmnt1	= NULL;
SIR_FsmdStmnt::FsmdStmnt2	= NULL;
SIR_FsmdStmnt::Label		= Label;
SIR_FsmdStmnt::Scope		= NULL;
SIR_FsmdStmnt::FsmdStmnts	= NULL;
SIR_FsmdStmnt::NextState	= NextState;
SIR_FsmdStmnt::TmpVar		= NULL;

} /* end of SIR_FsmdStmnt::SIR_FsmdStmnt #4 */



SIR_FsmdStmnt::SIR_FsmdStmnt(		/* constructor #6 (duplicator) */
	sir_fsmdstmnt	*Original,
	sir_fsmdstmnt	*ParentStmnt /* = NULL */) :
		SIR_Node(Original)
{

SIR_FsmdStmnt::StmntType	= Original->StmntType;
SIR_FsmdStmnt::Function		= Original->Function;
SIR_FsmdStmnt::ParentStmnt	= ParentStmnt;
SIR_FsmdStmnt::Expression	= Original->Expression ?
					new SIR_Expression(
						Original->Expression) :
					NULL;
SIR_FsmdStmnt::Cycles		= Original->Cycles;
SIR_FsmdStmnt::Constant		= Original->Constant ?
					new SIR_Constant(
						Original->Constant) :
					NULL;
SIR_FsmdStmnt::FsmdStmnt1	= Original->FsmdStmnt1 ?
					new SIR_FsmdStmnt(
						Original->FsmdStmnt1, this) :
					NULL;
SIR_FsmdStmnt::FsmdStmnt2	= Original->FsmdStmnt2 ?
					new SIR_FsmdStmnt(
						Original->FsmdStmnt2, this) :
					NULL;
SIR_FsmdStmnt::Label		= Original->Label;
SIR_FsmdStmnt::Scope		= Original->Scope ?
					new SIR_Symbols(
						Original->Scope) :
					NULL;
SIR_FsmdStmnt::FsmdStmnts	= Original->FsmdStmnts ?
					new SIR_FsmdStmnts(
						Original->FsmdStmnts, this) :
					NULL;
SIR_FsmdStmnt::NextState	= Original->NextState;
SIR_FsmdStmnt::TmpVar		= NULL;


} /* end of SIR_FsmdStmnt::SIR_FsmdStmnt #6 */


SIR_FsmdStmnt::~SIR_FsmdStmnt(void)	/* destructor */
{

delete Expression;
delete Constant;
delete FsmdStmnt1;
delete FsmdStmnt2;
delete Scope;
delete FsmdStmnts;

} /* end of SIR_FsmdStmnt::~SIR_FsmdStmnt */


// note: SIR_FsmdStmnt::IsDirectDependant() is always false


bool SIR_FsmdStmnt::IsIndirectDependant(/* checks if stmnt. depends on symbol */
	sir_symbol	*ThatSymbol,	/* (indirectly, through expression) */
	SIR_DEPENDENCY	*Reason /* = NULL */,
	sir_expression	**DepExpr /* = NULL */)
{

if (Reason)
   { *Reason = SIR_DEP_NONE;
    } /* fi */
if (DepExpr)
   { *DepExpr = NULL;
    } /* fi */

if (Expression)
   { if (Expression->DFS_FindDependant(ThatSymbol, DepExpr))
	{ if (Reason)
	     { *Reason = SIR_DEP_EXPRESSION;
	      } /* fi */
	  return(true);
	 } /* fi */
    } /* fi */

return(false);	/* independent! */

} /* end of SIR_FsmdStmnt::IsIndirectDependant */


sir_fsmdstmnts *SIR_FsmdStmnt::GetList(	/* determines the list of this stmnt. */
	void)
{

return((sir_fsmdstmnts*) Head());

} /* end of SIR_FsmdStmnt::GetList */


ERROR SIR_FsmdStmnt::DFS_ForAllNodes(	/* iterator over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{

if ((SIR_Error = (this->*MemberFct)(MemberFctArg)))	/* process this node */
   { return(SIR_Error);
    } /* fi */

if (Expression)
   { if ((SIR_Error = Expression->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Constant)
   { if ((SIR_Error = Constant->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (FsmdStmnt1)
   { if ((SIR_Error = FsmdStmnt1->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (FsmdStmnt2)
   { if ((SIR_Error = FsmdStmnt2->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Scope)
   { if ((SIR_Error = Scope->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (FsmdStmnts)
   { if ((SIR_Error = FsmdStmnts->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdStmnt::DFS_ForAllNodes */


ERROR SIR_FsmdStmnt::DFS_ForAllSymbols(	/* iterator over all symbols */
	sir_symbol_mptr	MemberFct,	/* (depth first) */
	sir_symbol_marg	MemberFctArg)
{

/* this is not a symbol */

/* there are no symbols in Expression */
/* there are no symbols in Constant */
if (FsmdStmnt1)
   { if ((SIR_Error = FsmdStmnt1->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (FsmdStmnt2)
   { if ((SIR_Error = FsmdStmnt2->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Scope)
   { if ((SIR_Error = Scope->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (FsmdStmnts)
   { if ((SIR_Error = FsmdStmnts->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdStmnt::DFS_ForAllSymbols */


ERROR SIR_FsmdStmnt::DFS_ForAllUserTypes( /* iterator over all usertypes */
	sir_usertp_mptr	MemberFct,	/* (depth first) */
	sir_usertp_marg	MemberFctArg)
{

/* this is not a usertype */

/* there are no usertypes in Expression */
/* there are no usertypes in Constant */
if (FsmdStmnt1)
   { if ((SIR_Error = FsmdStmnt1->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (FsmdStmnt2)
   { if ((SIR_Error = FsmdStmnt2->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Scope)
   { if ((SIR_Error = Scope->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (FsmdStmnts)
   { if ((SIR_Error = FsmdStmnts->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdStmnt::DFS_ForAllUserTypes */


ERROR SIR_FsmdStmnt::DFS_ForAllNotes( /* iterator over all notes */
	sir_note_mptr	MemberFct,	/* (depth first) */
	sir_note_marg	MemberFctArg)
{

/* this is not a note */

/* there are no notes in Expression */
/* there are no notes in Constant */
if (FsmdStmnt1)
   { if ((SIR_Error = FsmdStmnt1->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (FsmdStmnt2)
   { if ((SIR_Error = FsmdStmnt2->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Scope)
   { if ((SIR_Error = Scope->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (FsmdStmnts)
   { if ((SIR_Error = FsmdStmnts->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdStmnt::DFS_ForAllNotes */


ERROR SIR_FsmdStmnt::DFS_ForAllFsmdStmnts( /* iterator over all fsmd stmnts. */
	sir_fsmdstmnt_mptr	MemberFct,	/* (depth first) */
	sir_fsmdstmnt_marg	MemberFctArg)
{
bool		IsLeaf;

IsLeaf = !(FsmdStmnt1 || FsmdStmnt2 || FsmdStmnts);

if ((SIR_Error = (this->*MemberFct)(MemberFctArg)))	/* process this node */
   { return(SIR_Error);
    } /* fi */

if (IsLeaf)
   { // it is allowed to delete leaf objects within this iterator;
     // in this case, we must not access any members any more,
     // because that would be a free-memory-access violation
     return(SIR_ERROR_NO_ERROR);
    } /* fi */

/* there are no fsmd statements in Expression */
/* there are no fsmd statements in Constant */
if (FsmdStmnt1)
   { if ((SIR_Error = FsmdStmnt1->DFS_ForAllFsmdStmnts(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (FsmdStmnt2)
   { if ((SIR_Error = FsmdStmnt2->DFS_ForAllFsmdStmnts(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no fsmd statements in Scope */
if (FsmdStmnts)
   { if ((SIR_Error = FsmdStmnts->DFS_ForAllFsmdStmnts(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdStmnt::DFS_ForAllFsmdStmnts */


ERROR SIR_FsmdStmnt::DFS_ForAllExpressions( /* iterator over all expressions */
	sir_expr_mptr	MemberFct,		/* (depth first) */
	sir_expr_marg	MemberFctArg)
{

/* this is not an expression */

if (Expression)
   { if ((SIR_Error = Expression->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no expressions in Constant */
if (FsmdStmnt1)
   { if ((SIR_Error = FsmdStmnt1->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (FsmdStmnt2)
   { if ((SIR_Error = FsmdStmnt2->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no expressions in Scope */
if (FsmdStmnts)
   { if ((SIR_Error = FsmdStmnts->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdStmnt::DFS_ForAllExpressions */


bool SIR_FsmdStmnt::DFS_FindDirectDependant(	/* searches for direct deps. */
	sir_symbol	**SkipSymbol,
	sir_symbol	*ThatSymbol,
	sir_symbol	**DepSymbol,
	SIR_DEPENDENCY	*Reason)
{
sir_symbol	*Symbol;

assert(SkipSymbol != NULL);
assert(ThatSymbol != NULL);

if (  (Scope)
    &&(*SkipSymbol)
    &&(! Scope->IsAncestorOf(*SkipSymbol)))
   { return(false);	/* skip this symbol table (prune the subtree) */
    } /* fi */

if (Scope)
   { /* check dependency of 'buffered' variables in local scopes */
     Symbol = Scope->First();
     while(Symbol)
	{ if (*SkipSymbol)
	     { if (Symbol == *SkipSymbol)
		  { *SkipSymbol = NULL;	/* no more skipping */
		   } /* fi */
	      } /* fi */
	  else
	     { if (Symbol->IsDirectDependant(ThatSymbol, Reason))
		  { assert(  (! Reason)
			   ||(*Reason == SIR_DEP_CLOCK)
			   ||(*Reason == SIR_DEP_RESET));
		    if (DepSymbol)
		       { *DepSymbol = Symbol;
			} /* fi */
		    return(true);
		   } /* fi */
	      } /* esle */
	  Symbol = Symbol->Succ();
	 } /* elihw */
    } /* fi */

/* there are no fsmd statements in Expression */
/* there are no fsmd statements in Constant */
if (FsmdStmnt1)
   { if (FsmdStmnt1->DFS_FindDirectDependant(SkipSymbol, ThatSymbol,
						DepSymbol, Reason))
	{ return(true);
	 } /* fi */
    } /* fi */
if (FsmdStmnt2)
   { if (FsmdStmnt2->DFS_FindDirectDependant(SkipSymbol, ThatSymbol,
						DepSymbol, Reason))
	{ return(true);
	 } /* fi */
    } /* fi */
/* there are no fsmd statements in Scope */
if (FsmdStmnts)
   { if (FsmdStmnts->DFS_FindDirectDependant(SkipSymbol, ThatSymbol,
						DepSymbol, Reason))
	{ return(true);
	 } /* fi */
    } /* fi */

return(false);

} /* end of SIR_FsmdStmnt::DFS_FindDirectDependant */


bool SIR_FsmdStmnt::DFS_FindIndirectDependant(	/* searches for indirect deps.*/
	sir_symbol	*ThatSymbol,
	sir_fsmdstmnt	**DepStmnt,
	sir_expression	**DepExpr,
	SIR_DEPENDENCY	*Reason)
{

assert(ThatSymbol != NULL);

// note: IsDirectDependant(ThatSymbol) is always false for fsmd statements

if (IsIndirectDependant(ThatSymbol, Reason,	/* check this statement */
			DepExpr))		/* (for expression dep.) */
   { if (DepStmnt)
	{ *DepStmnt = this;
	 } /* fi */
     return(true);
    } /* fi */

/* there are no fsmd statements in Expression */
/* there are no fsmd statements in Constant */
if (FsmdStmnt1)
   { if (FsmdStmnt1->DFS_FindIndirectDependant(ThatSymbol, DepStmnt, DepExpr,
						Reason))
	{ return(true);
	 } /* fi */
    } /* fi */
if (FsmdStmnt2)
   { if (FsmdStmnt2->DFS_FindIndirectDependant(ThatSymbol, DepStmnt, DepExpr,
						Reason))
	{ return(true);
	 } /* fi */
    } /* fi */
/* there are no fsmd statements in Scope */
if (FsmdStmnts)
   { if (FsmdStmnts->DFS_FindIndirectDependant(ThatSymbol, DepStmnt, DepExpr,
						Reason))
	{ return(true);
	 } /* fi */
    } /* fi */

return(false);

} /* end of SIR_FsmdStmnt::DFS_FindIndirectDependant */


ERROR SIR_FsmdStmnt::MarkYourLabel(	/* iterator for color marking */
	sir_fsmdstmnt_marg	ColorArg)
{
SIR_COLOR	MyColor;

MyColor = *((SIR_COLOR*) ColorArg);

if (Label)
   { Label->Color = MyColor;	/* mark the label */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdStmnt::MarkYourLabel */


ERROR SIR_FsmdStmnt::CheckLabeledStmnt(	/* iterator for label check */
	sir_stmnt_marg	/* Unused */)
{

if (Label)
   { SIR_ErrMsg.form("Double definition of label '%s'",
				Label->LabelName.chars());
     return(SIR_ERROR_DOUBLE_DEFINITION_OF_LABEL);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdStmnt::CheckLabeledStmnt */


ERROR SIR_FsmdStmnt::CheckControlFlow(	/* iterator for control flow checking */
	sir_stmnt_marg	/* Unused */)
{
sir_fsmdstmnt	*SwitchStmnt,
		*Case;
bool		DefaultSeen;

// note:
//
// there is two types of restrictions on the control flow inside 'fsmd':
//
// (a) hard restrictions: imposed by the language
// (b) weak restrictions: imposed by the current implementation
//
// hard restrictions will always be there, weak restrictions
// will hopefully be eliminated in future implementations;
// note that the weak restrictions are only checked here
// (after parsing), not when manipulating the SIR via the API
// (not even in API 2);
//
// note also that this control flow checking is still incomplete
// as, for example, multiple cases with the same value are not
// caught as an error;
//
// (RD, 10/30/03)

if (StmntType == SIR_FSMDSTMNT_CASE)
   { if (!(SwitchStmnt = GetSwitchStmnt()))
	{ if (LineInfo)
	     { SIR_ErrMsg.form("'case' statement outside"
				" 'switch' statement" GL_ERROR_MSG_NEWLINE
				"(line %u, file \"%s\")",
				LineInfo->Line,
				LineInfo->File->Filename.chars());
	      } /* fi */
	  else
	     { SIR_ErrMsg = "'case' statement outside"
				" 'switch' statement";
	      } /* esle */
	  return(SIR_ERROR_CASE_OUTSIDE_SWITCH);	// hard restriction (b)
	 } /* fi */
     if (  (ParentStmnt == SwitchStmnt)
	 ||(ParentStmnt->ParentStmnt != SwitchStmnt))
	{ // 'case' must be exactly 2 levels below its 'switch'
	  if (LineInfo)
	     { SIR_ErrMsg.form("'case' statement not in compound statement"
				" below 'switch' statement" GL_ERROR_MSG_NEWLINE
				"(line %u, file \"%s\")",
				LineInfo->Line,
				LineInfo->File->Filename.chars());
	      } /* fi */
	  else
	     { SIR_ErrMsg = "'case' statement not in compound statement"
				" below 'switch' statement";
	      } /* esle */
	  return(SIR_ERROR_CASE_NOT_2_BELOW_SWITCH); // weak restriction (a)
	 } /* fi */
     return(SIR_ERROR_NO_ERROR);
    } /* fi */

if (StmntType == SIR_FSMDSTMNT_DEFAULT)
   { if (!(SwitchStmnt = GetSwitchStmnt()))
	{ if (LineInfo)
	     { SIR_ErrMsg.form("'default' statement outside"
				" 'switch' statement" GL_ERROR_MSG_NEWLINE
				"(line %u, file \"%s\")",
				LineInfo->Line,
				LineInfo->File->Filename.chars());
	      } /* fi */
	  else
	     { SIR_ErrMsg = "'default' statement outside"
				" 'switch' statement";
	      } /* esle */
	  return(SIR_ERROR_DEFAULT_OUTSIDE_SWITCH);	// hard restriction (b)
	 } /* fi */
     if (  (ParentStmnt == SwitchStmnt)
	 ||(ParentStmnt->ParentStmnt != SwitchStmnt))
	{ // 'default' must be exactly 2 levels below its 'switch'
	  if (LineInfo)
	     { SIR_ErrMsg.form("'default' statement not in compound statement"
				" below 'switch' statement" GL_ERROR_MSG_NEWLINE
				"(line %u, file \"%s\")",
				LineInfo->Line,
				LineInfo->File->Filename.chars());
	      } /* fi */
	  else
	     { SIR_ErrMsg = "'default' statement not in compound statement"
				" below 'switch' statement";
	      } /* esle */
	  return(SIR_ERROR_DFLT_NOT_2_BELOW_SWITCH); // weak restriction (a)
	 } /* fi */
     return(SIR_ERROR_NO_ERROR);
    } /* fi */

if (StmntType == SIR_FSMDSTMNT_BREAK)
   { if (!(SwitchStmnt = GetSwitchStmnt()))
	{ return(SIR_ERROR_NO_ERROR);	/* 'break' exits 'fsmd', OK */
	 } /* fi */
     if (  (ParentStmnt == SwitchStmnt)
	 ||(ParentStmnt->ParentStmnt == SwitchStmnt)
	 ||(ParentStmnt->ParentStmnt->ParentStmnt == SwitchStmnt)
	 ||(ParentStmnt->ParentStmnt->ParentStmnt->ParentStmnt != SwitchStmnt))
	{ // 'break' must be exactly 4 levels below its 'switch'
	  if (LineInfo)
	     { SIR_ErrMsg.form("'break' statement not at fourth level"
				" below 'switch' statement" GL_ERROR_MSG_NEWLINE
				"(line %u, file \"%s\")",
				LineInfo->Line,
				LineInfo->File->Filename.chars());
	      } /* fi */
	  else
	     { SIR_ErrMsg = "'break' statement not at fourth level"
				" below 'switch' statement";
	      } /* esle */
	  return(SIR_ERROR_BREAK_NOT_4_BELOW_SWITCH); // weak restriction (a)
	 } /* fi */
     if (  (ParentStmnt->StmntType != SIR_FSMDSTMNT_COMPOUND)
	 ||(ParentStmnt->FsmdStmnts->Last() != this))
	{ // 'break' must be last statement in its compound statement
	  if (LineInfo)
	     { SIR_ErrMsg.form("'break' statement not last"
				" statement in case" GL_ERROR_MSG_NEWLINE
				"(line %u, file \"%s\")",
				LineInfo->Line,
				LineInfo->File->Filename.chars());
	      } /* fi */
	  else
	     { SIR_ErrMsg = "'break' statement not last"
				" statement in case";
	      } /* esle */
	  return(SIR_ERROR_BREAK_NOT_LAST_IN_CASE); // weak restriction (a)
	 } /* fi */
     return(SIR_ERROR_NO_ERROR);
    } /* fi */

if (StmntType == SIR_FSMDSTMNT_SWITCH)
   { if (FsmdStmnt1->StmntType != SIR_FSMDSTMNT_COMPOUND)
	{ // 'switch' must have compound statement as body
	  if (LineInfo)
	     { SIR_ErrMsg.form("'switch' statement body not"
				" a compound statement" GL_ERROR_MSG_NEWLINE
				"(line %u, file \"%s\")",
				LineInfo->Line,
				LineInfo->File->Filename.chars());
	      } /* fi */
	  else
	     { SIR_ErrMsg = "'switch' statement body not"
				" a compound statement";
	      } /* esle */
	  return(SIR_ERROR_SWITCH_BODY_NOT_COMPOUND); // weak restriction (a)
	 } /* fi */
     Case = FsmdStmnt1->FsmdStmnts->First();
     DefaultSeen = false;
     while(Case)
	{ if (  (Case->StmntType != SIR_FSMDSTMNT_CASE)
	      &&(Case->StmntType != SIR_FSMDSTMNT_DEFAULT))
	     { // only 'case' and 'default' allowed in 'switch' body
	       if (Case->LineInfo)
		  { SIR_ErrMsg.form("only 'case' and 'default' statements"
				" allowed in 'switch' body" GL_ERROR_MSG_NEWLINE
				"(line %u, file \"%s\")",
				Case->LineInfo->Line,
				Case->LineInfo->File->Filename.chars());
		   } /* fi */
	       else
		  { SIR_ErrMsg = "only 'case' and 'default' statements"
				" allowed in 'switch' body";
		   } /* esle */
	       return(SIR_ERROR_NOT_CASE_DEFAULT_IN_SWITCH); // weak restr. (a)
	      } /* fi */
	  if (  (Case->FsmdStmnt1->StmntType != SIR_FSMDSTMNT_COMPOUND)
	      ||(Case->FsmdStmnt1->FsmdStmnts->Empty())
	      ||(Case->FsmdStmnt1->FsmdStmnts->Last()->StmntType !=
						SIR_FSMDSTMNT_BREAK))
	     { // case body must be compound statement ending with 'break'
	       if (Case->LineInfo)
		  { SIR_ErrMsg.form("case not a compound statement"
				" ending with 'break'" GL_ERROR_MSG_NEWLINE
				"(line %u, file \"%s\")",
				Case->LineInfo->Line,
				Case->LineInfo->File->Filename.chars());
		   } /* fi */
	       else
		  { SIR_ErrMsg = "case not a compound statement"
				" ending with 'break'";
		   } /* esle */
	       return(SIR_ERROR_CASE_NOT_ENDING_WITH_BREAK); // weak restr. (a)
	      } /* fi */
	  if (Case->StmntType == SIR_FSMDSTMNT_DEFAULT)
	     { if (DefaultSeen)
		  { // multiple 'default' in 'switch'
		    if (Case->LineInfo)
		       { SIR_ErrMsg.form("multiple 'default' statements"
				" in 'switch' body" GL_ERROR_MSG_NEWLINE
				"(line %u, file \"%s\")",
				Case->LineInfo->Line,
				Case->LineInfo->File->Filename.chars());
			} /* fi */
		    else
		       { SIR_ErrMsg = "multiple 'default' statements"
					" in 'switch' body";
			} /* esle */
		    return(SIR_ERROR_MULTIPLE_DEFAULT_IN_SWITCH); // hard! (b)
		   } /* fi */
	       DefaultSeen = true;
	      } /* fi */
	  Case = Case->Succ();
	 } /* elihw */
     return(SIR_ERROR_NO_ERROR);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdStmnt::CheckControlFlow */


ERROR SIR_FsmdStmnt::TargetsState(	/* iterator for finding transitions */
	sir_fsmdstmnt_marg	TargetArg)
{
sir_fsmdstate	*TargetState;

TargetState = (sir_fsmdstate*) TargetArg;
assert(TargetState != NULL);

if (NextState == TargetState)
   { return(FLAG_YES);
    } /* fi */

return(FLAG_NO);

} /* end of SIR_FsmdStmnt::TargetsState */


void SIR_FsmdStmnt::SetAlias(	/* sets all type, usertype, symbol alias' */
	sir_fsmdstmnt	*Alias)	/* (iterates over symbols and usertypes) */
{

assert(Alias != NULL);

/* no need to process this node */

/* there are no symbols or usertypes in Expression */
/* there are no symbols or usertypes in Constant */
if (FsmdStmnt1)
   { FsmdStmnt1->SetAlias(Alias->FsmdStmnt1);
    } /* fi */
if (FsmdStmnt2)
   { FsmdStmnt2->SetAlias(Alias->FsmdStmnt2);
    } /* fi */
if (Scope)
   { Scope->SetAlias(Alias->Scope);
    } /* fi */
if (FsmdStmnts)
   { FsmdStmnts->SetAlias(Alias->FsmdStmnts);
    } /* fi */

} /* end of SIR_FsmdStmnt::SetAlias */


void SIR_FsmdStmnt::UnAlias(	/* unalias all type, usertype, symbol links */
	sir_symbols	*GlobalSymbols)
{

if (  (Function)
    &&(Function->Alias))
   { Function = Function->Alias;
    } /* fi */
if (Expression)
   { Expression->UnAlias();
    } /* fi */
/* there is nothing to unalias in Constant */
if (FsmdStmnt1)
   { FsmdStmnt1->UnAlias(GlobalSymbols);
    } /* fi */
if (FsmdStmnt2)
   { FsmdStmnt2->UnAlias(GlobalSymbols);
    } /* fi */
if (Scope)
   { Scope->UnAlias(GlobalSymbols);
    } /* fi */
if (FsmdStmnts)
   { FsmdStmnts->UnAlias(GlobalSymbols);
    } /* fi */

} /* end of SIR_FsmdStmnt::UnAlias */


ERROR SIR_FsmdStmnt::WriteSC(	/* (re-) generates SpecC source code */
	gl_io	*IO,
	bool		WriteNotes,
	bool		CplusplusMode /* = false */,
	bool		PutNewLine /* = true */,
	bool		IsFsmdState /* = false */,
	bool		BreakExitsFsmd /* = true */)
{
sir_fsmdstmnt	*Stmnt;
gl_string		TmpString;
bool		PutBraces;

PutBraces = (StmntType == SIR_FSMDSTMNT_COMPOUND);

if (LineInfo)
   { if ((SIR_Error = LineInfo->WriteSC(IO, true, PutBraces)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
else
   { if (PutNewLine)
	{ SIR_LineInfo::WriteNewSC(IO, true, PutBraces, &LineInfo);
	 } /* fi */
     else
	{ SIR_LineInfo::WriteSPACE(IO, PutBraces);
	 } /* fi */
    } /* esle */

if (Label)
   { IO->PutS(Label->LabelName);
     IO->PutS(": ");
     IO->TabStepUp();
    } /* fi */

switch(StmntType)
   { case SIR_FSMDSTMNT_COMPOUND:
	{ assert(PutBraces);	// IO->PutC('{');
	  assert(Scope != NULL);	/* must have local scope */
	  if (IsFsmdState)
	     { assert(Scope->ParentFsmdState != NULL);
	       if (Scope->ParentFsmdState->Notes)
		  { if ((SIR_Error = Scope->ParentFsmdState->Notes->WriteSC(
						IO, WriteNotes, CplusplusMode)))
		       { return(SIR_Error);
			} /* fi */
		   } /* fi */
	      } /* fi */
	  if (  (! Scope->Empty())
	      ||(! Scope->UserTypes->Empty()))
	     { if (CplusplusMode)
		  { if ((SIR_Error = Scope->WriteH(IO, WriteNotes)))
		       { return(SIR_Error);
			} /* fi */
		    if ((SIR_Error = Scope->WriteCC(IO, WriteNotes)))
		       { return(SIR_Error);
			} /* fi */
		   } /* fi */
	       else
		  { if ((SIR_Error = Scope->WriteSC(IO, WriteNotes)))
		       { return(SIR_Error);
			} /* fi */
		   } /* esle */
	       if (  (IsFsmdState && CplusplusMode)
		   ||(  (FsmdStmnts)
		      &&(! FsmdStmnts->Empty())))
		  { SIR_LineInfo::WriteVSPACE(IO);
		   } /* fi */
	      } /* fi */
	  if (FsmdStmnts)
	     { Stmnt = FsmdStmnts->First();
	       while(Stmnt)
		  { if ((SIR_Error = Stmnt->WriteSC(IO, WriteNotes,
						CplusplusMode, true,
						false, BreakExitsFsmd)))
		       { return(SIR_Error);
			} /* fi */
		    Stmnt = Stmnt->Succ();
		   } /* elihw */
	      } /* fi */
	  if (IsFsmdState && CplusplusMode)
	     { SIR_LineInfo::WriteSPACE(IO);
	       IO->PutS("break;");
	      } /* fi */
	  SIR_LineInfo::WriteNL(IO, false, true);	// IO->PutC('}');
	  break;
	 }
     case SIR_FSMDSTMNT_EXPRESSION:
	{
#ifdef HAVE_ARYASGN
	  // case (1): print expression
	  if ((SIR_Error = Expression->WriteSC(IO, WriteNotes,
						CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->PutC(';');
#else /* ! HAVE_ARYASGN */
	  if (  (CplusplusMode)		/* array assignment handling */
	      &&(Expression->ExprType == SIR_EXPR_ASSIGNMENT)
	      &&(Expression->Arg1->Type->Type == SIR_TYPE_ARRAY)
	      &&(Expression->Arg2->Type->Type == SIR_TYPE_ARRAY))
	     { // case (3): print array assignment
	       if ((SIR_Error = Expression->WriteArrayAssignment(IO,
								WriteNotes)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  else
	     { // case (1): print expression
	       if ((SIR_Error = Expression->WriteSC(IO, WriteNotes,
							CplusplusMode)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->PutC(';');
	      } /* esle */
#endif /* HAVE_ARYASGN */
	  break;
	 }
     case SIR_FSMDSTMNT_AFTER_EXPR:
	{ assert(Expression->ExprType == SIR_EXPR_ASSIGNMENT);
	  if (CplusplusMode)
	     {
#ifdef HAVE_ARYASGN
	       if (Cycles > 0)
		  { // case (2): print assignment to tmp. variable
		    sir_expression	*TmpAssign;

		    assert(TmpVar != NULL); /* must have been assigned */
		    if (!(TmpAssign = NewTmpAssign(TmpVar, 0,
							Expression->Arg2)))
		       { return(SIR_Error);
			} /* fi */
		    if ((SIR_Error = TmpAssign->WriteSC(IO, WriteNotes,
								CplusplusMode)))
		       { delete TmpAssign;
			 return(SIR_Error);
			} /* fi */
		    delete TmpAssign;
		   } /* fi */
	       else
		  { // case (1): print expression
		    if ((SIR_Error = Expression->WriteSC(IO, WriteNotes,
							CplusplusMode)))
		       { return(SIR_Error);
			} /* fi */
		   } /* esle */
	       IO->PutC(';');
#else /* ! HAVE_ARYASGN */
	       if (  (Expression->Arg1->Type->Type == SIR_TYPE_ARRAY)
		   &&(Expression->Arg2->Type->Type == SIR_TYPE_ARRAY))
		  { if (Cycles > 0)
		       { // case (4): print array assignment to tmp. variable
			 sir_expression	*TmpAssign;

			 assert(TmpVar != NULL); /* must have been assigned */
			 if (!(TmpAssign = NewTmpAssign(TmpVar, 0,
							Expression->Arg2)))
			    { return(SIR_Error);
			     } /* fi */
			 if ((SIR_Error = TmpAssign->WriteArrayAssignment(IO,
								WriteNotes)))
			    { delete TmpAssign;
			      return(SIR_Error);
			     } /* fi */
			 delete TmpAssign;
			} /* fi */
		    else
		       { // case (3): print array assignment
			 if ((SIR_Error = Expression->WriteArrayAssignment(IO,
								WriteNotes)))
			    { return(SIR_Error);
			     } /* fi */
			} /* esle */
		   } /* fi */
	       else
		  { if (Cycles > 0)
		       { // case (2): print assignment to tmp. variable
			 sir_expression	*TmpAssign;

			 assert(TmpVar != NULL); /* must have been assigned */
			 if (!(TmpAssign = NewTmpAssign(TmpVar, 0,
							Expression->Arg2)))
			    { return(SIR_Error);
			     } /* fi */
			 if ((SIR_Error = TmpAssign->WriteSC(IO, WriteNotes,
								CplusplusMode)))
			    { delete TmpAssign;
			      return(SIR_Error);
			     } /* fi */
			 delete TmpAssign;
			} /* fi */
		    else
		       { // case (1): print expression
			 if ((SIR_Error = Expression->WriteSC(IO, WriteNotes,
								CplusplusMode)))
			    { return(SIR_Error);
			     } /* fi */
			} /* esle */
		    IO->PutC(';');
		   } /* esle */
#endif /* HAVE_ARYASGN */
	      } /* fi */
	  else /* ! CplusplusMode */
	     { // case (1): print expression (plus suffix)
	       if ((SIR_Error = Expression->WriteSC(IO, WriteNotes,
							CplusplusMode)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->PrintF(" after %d;", Cycles);
	      } /* esle */
	  break;
	 }
     case SIR_FSMDSTMNT_PIPED_EXPR:
	{ assert(Expression->ExprType == SIR_EXPR_ASSIGNMENT);
	  if (CplusplusMode)
	     {
#ifdef HAVE_ARYASGN
	       if (Cycles > 0)
		  { // case (2): print assignment to tmp. variable
		    sir_expression	*TmpAssign;

		    assert(TmpVar != NULL); /* must have been assigned */
		    if (!(TmpAssign = NewTmpAssign(TmpVar, Cycles-1,
							Expression->Arg2)))
		       { return(SIR_Error);
			} /* fi */
		    if ((SIR_Error = TmpAssign->WriteSC(IO, WriteNotes,
								CplusplusMode)))
		       { delete TmpAssign;
			 return(SIR_Error);
			} /* fi */
		    delete TmpAssign;
		   } /* fi */
	       else
		  { // case (1): print expression
		    if ((SIR_Error = Expression->WriteSC(IO, WriteNotes,
							CplusplusMode)))
		       { return(SIR_Error);
			} /* fi */
		   } /* esle */
	       IO->PutC(';');
#else /* ! HAVE_ARYASGN */
	       if (  (Expression->Arg1->Type->Type == SIR_TYPE_ARRAY)
		   &&(Expression->Arg2->Type->Type == SIR_TYPE_ARRAY))
		  { if (Cycles > 0)
		       { // case (4): print array assignment to tmp. variable
			 sir_expression	*TmpAssign;

			 assert(TmpVar != NULL); /* must have been assigned */
			 if (!(TmpAssign = NewTmpAssign(TmpVar, Cycles-1,
							Expression->Arg2)))
			    { return(SIR_Error);
			     } /* fi */
			 if ((SIR_Error = TmpAssign->WriteArrayAssignment(IO,
								WriteNotes)))
			    { delete TmpAssign;
			      return(SIR_Error);
			     } /* fi */
			 delete TmpAssign;
			} /* fi */
		    else
		       { // case (3): print array assignment
			 if ((SIR_Error = Expression->WriteArrayAssignment(IO,
								WriteNotes)))
			    { return(SIR_Error);
			     } /* fi */
			} /* esle */
		   } /* fi */
	       else
		  { if (Cycles > 0)
		       { // case (2): print assignment to tmp. variable
			 sir_expression	*TmpAssign;

			 assert(TmpVar != NULL); /* must have been assigned */
			 if (!(TmpAssign = NewTmpAssign(TmpVar, Cycles-1,
							Expression->Arg2)))
			    { return(SIR_Error);
			     } /* fi */
			 if ((SIR_Error = TmpAssign->WriteSC(IO, WriteNotes,
								CplusplusMode)))
			    { delete TmpAssign;
			      return(SIR_Error);
			     } /* fi */
			 delete TmpAssign;
			} /* fi */
		    else
		       { // case (1): print expression
			 if ((SIR_Error = Expression->WriteSC(IO, WriteNotes,
								CplusplusMode)))
			    { return(SIR_Error);
			     } /* fi */
			} /* esle */
		    IO->PutC(';');
		   } /* esle */
#endif /* HAVE_ARYASGN */
	      } /* fi */
	  else /* ! CplusplusMode */
	     { // case (1): print expression (plus suffix)
	       if ((SIR_Error = Expression->WriteSC(IO, WriteNotes,
							CplusplusMode)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->PrintF(" piped %d;", Cycles);
	      } /* esle */
	  break;
	 }
     case SIR_FSMDSTMNT_IF:
	{ IO->PutS("if (");
	  IO->TabStepUp();
	  if ((SIR_Error = Expression->WriteSC(IO, WriteNotes, CplusplusMode,
						SIR_TYPE_BOOL, 0, true)))
	     { return(SIR_Error);
	      } /* fi */
	//IO->TabStepDown();
	  IO->PutC(')');
	//IO->TabStepUp();
	  if ((SIR_Error = FsmdStmnt1->WriteSC(IO, WriteNotes,
						CplusplusMode, true,
						false, BreakExitsFsmd)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->TabStepDown();
	  break;
	 }
     case SIR_FSMDSTMNT_IF_ELSE:
	{ IO->PutS("if (");
	  IO->TabStepUp();
	  if ((SIR_Error = Expression->WriteSC(IO, WriteNotes, CplusplusMode,
						SIR_TYPE_BOOL, 0, true)))
	     { return(SIR_Error);
	      } /* fi */
	//IO->TabStepDown();
	  IO->PutC(')');
	//IO->TabStepUp();
	  if ((SIR_Error = FsmdStmnt1->WriteSC(IO, WriteNotes,
						CplusplusMode, true,
						false, BreakExitsFsmd)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->TabStepDown();
	  SIR_LineInfo::WriteNL(IO);
	  IO->PutS("else ");
	  IO->TabStepUp();
	  if ((SIR_Error = FsmdStmnt2->WriteSC(IO, WriteNotes,
						CplusplusMode, true,
						false, BreakExitsFsmd)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->TabStepDown();
	  break;
	 }
     case SIR_FSMDSTMNT_SWITCH:
	{ IO->PutS("switch(");
	  IO->TabStepUp();
	  if ((SIR_Error = Expression->WriteSC(IO, WriteNotes, CplusplusMode,
						SIR_TYPE_INT, 0, true)))
	     { return(SIR_Error);
	      } /* fi */
	//IO->TabStepDown();
	  IO->PutC(')');
	//IO->TabStepUp();
	  if ((SIR_Error = FsmdStmnt1->WriteSC(IO, WriteNotes,
						CplusplusMode, true,
						false,
						false /* enter switch body */)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->TabStepDown();
	  break;
	 }
     case SIR_FSMDSTMNT_CASE:
	{ IO->PutS("case ");
	  IO->PutS(Constant->Print());
	  IO->PutC(':');
	  IO->TabStepUp();
	  if ((SIR_Error = FsmdStmnt1->WriteSC(IO, WriteNotes,
						CplusplusMode, false,
						false, BreakExitsFsmd)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->TabStepDown();
	  break;
	 }
     case SIR_FSMDSTMNT_DEFAULT:
	{ IO->PutS("default:");
	  IO->TabStepUp();
	  if ((SIR_Error = FsmdStmnt1->WriteSC(IO, WriteNotes,
						CplusplusMode, false,
						false, BreakExitsFsmd)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->TabStepDown();
	  break;
	 }
     case SIR_FSMDSTMNT_GOTO:
	{ if (CplusplusMode)
	     { IO->PutS(SIR_CXX_FSMD_STATE_REG " = " SIR_CXX_FSMD_STATE_PFX);
	       IO->PutS(NextState->StateName);
	       IO->PutC(';');
	      } /* fi */
	  else
	     { IO->PutS("goto ");
	       IO->PutS(NextState->StateName);
	       IO->PutC(';');
	      } /* esle */
	  break;
	 }
     case SIR_FSMDSTMNT_BREAK:
	{ if (  (BreakExitsFsmd)
	      &&(CplusplusMode))
	     { IO->PutS(SIR_CXX_FSMD_STATE_REG " = "
				SIR_CXX_FSMD_BREAKSTATE ";");
	      } /* fi */
	  else
	     { IO->PutS("break;");
	      } /* fi */
	  break;
	 }
     default:
	{ assert(false);	/* bad fsmd StmntType */
	 }
    } /* hctiws */

if (Label)
   { IO->TabStepDown();
     // note: notes at labels are written at the end of each function
    } /* fi */

if ((SIR_Error = IO->Check()))
   { return(SIR_Error);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdStmnt::WriteSC */


sir_fsmdstmnt *SIR_FsmdStmnt::WrapInCompound(	/* wrap in a compound stmnt.*/
	sir_symbols	*ParentScope)	/* (caller must fix father pointers) */
{
sir_fsmdstmnt	*NewCompound;

assert(this != NULL);		/* must exist */
assert(Function != NULL);	/* must know its function */
assert(GetList() == NULL);	/* must not be in a list */
assert(ParentScope != NULL);	/* must be given */

NewCompound = new SIR_FsmdStmnt(SIR_FSMDSTMNT_COMPOUND,
				Function,
				new SIR_Symbols(ParentScope,
						SIR_SCOPE_FSMDSTMNT,
						Function, NULL,
						ParentScope->ParentStmnt,
						ParentScope->ParentFsmdState),
				new SIR_FsmdStmnts((sir_symbols*)NULL, this));
ParentStmnt = NewCompound;

/* caller must fix all scopes below to take on this new father */

return(NewCompound);

} /* end of SIR_FsmdStmnt::WrapInCompound */


void SIR_FsmdStmnt::CompoundSubBlocks(	/* compound all blocks below */
	void)				/* (insert missing compound stmnts.) */
{
sir_fsmdstmnt	*Stmnt,
		*NewCompound;

assert(this != NULL);			/* must exist */
assert(StmntType == SIR_FSMDSTMNT_COMPOUND); /* must start from compound */
assert(FsmdStmnts != NULL);		/* must have statement list */
assert(Scope != NULL);			/* must have a scope to start with */

Stmnt = FsmdStmnts->First();
while(Stmnt)
   { assert(Stmnt->ParentStmnt == this);
     if (Stmnt->FsmdStmnt1)
	{ if (Stmnt->FsmdStmnt1->StmntType != SIR_FSMDSTMNT_COMPOUND)
	     { NewCompound = Stmnt->FsmdStmnt1->WrapInCompound(Scope);
	       Stmnt->FsmdStmnt1 = NewCompound;		/* plug it in */
	       NewCompound->ParentStmnt = Stmnt;
	      } /* fi */
	  else
	     { Stmnt->FsmdStmnt1->Scope->Parent = Scope;	/* fix scope */
	       Stmnt->FsmdStmnt1->Scope->UserTypes->Parent =	/* hierarchy */
					Scope->UserTypes;
	       assert(Stmnt->FsmdStmnt1->ParentStmnt == Stmnt);
	      } /* esle */
	  Stmnt->FsmdStmnt1->CompoundSubBlocks();	/* recursive! */
	 } /* fi */
     if (Stmnt->FsmdStmnt2)
	{ if (Stmnt->FsmdStmnt2->StmntType != SIR_FSMDSTMNT_COMPOUND)
	     { NewCompound = Stmnt->FsmdStmnt2->WrapInCompound(Scope);
	       Stmnt->FsmdStmnt2 = NewCompound;		/* plug it in */
	       NewCompound->ParentStmnt = Stmnt;
	      } /* fi */
	  else
	     { Stmnt->FsmdStmnt2->Scope->Parent = Scope;	/* fix scope */
	       Stmnt->FsmdStmnt2->Scope->UserTypes->Parent =	/* hierarchy */
					Scope->UserTypes;
	       assert(Stmnt->FsmdStmnt2->ParentStmnt == Stmnt);
	      } /* esle */
	  Stmnt->FsmdStmnt2->CompoundSubBlocks();	/* recursive! */
	 } /* fi */
     if (Stmnt->FsmdStmnts)
	{ Stmnt->CompoundSubBlocks();	/* recursive! */
	 } /* fi */
     Stmnt = Stmnt->Succ();
    } /* elihw */

} /* end of SIR_FsmdStmnt::CompoundSubBlocks */


ERROR SIR_FsmdStmnt::CreateDelayedAssignments(	/* create 'after'/'piped' tmp.*/
	sir_tmpvars	*TmpVars,
	sir_state_ptrs	*NextStates,
	const char	*StateName,
	sir_expression	*Condition /* = NULL */)
{
sir_expression	*NewCond;
sir_delydassgn	*Assignment;
sir_fsmdstmnt	*SwitchStmnt,
		*Case;
ERROR		Error;

// note:
// the conditions computed here are simply collected and combined;
// there is certainly a lot room for improvement and optimization,
// for example, removing redundancy in the conditions and combining
// them in more efficient ways;
// however, this optimization is left for the future right now,
// simply because of lack of time		(RD, 10/24/03)

assert(TmpVars != NULL);
assert(NextStates != NULL);

switch(StmntType)
   { case SIR_FSMDSTMNT_COMPOUND:
	{ sir_fsmdstmnt	*Stmnt;

	  Stmnt = FsmdStmnts->First();
	  while(Stmnt)
	     { if ((Error = Stmnt->CreateDelayedAssignments(TmpVars, NextStates,
							StateName, Condition)))
		  { return(Error);
		   } /* fi */
	       Stmnt = Stmnt->Succ();
	      } /* elihw */
	  break;
	 }
     case SIR_FSMDSTMNT_EXPRESSION:
	{ // nothing to do
	  break;
	 }
     case SIR_FSMDSTMNT_AFTER_EXPR:
	{ if (Cycles > 0)
	     { assert(Expression->ExprType == SIR_EXPR_ASSIGNMENT);
	       if (!(TmpVar = TmpVars->FindOrInsert(Expression->Arg1,
							1 /* one buffer */)))
		  { return(SIR_Error);
		   } /* fi */
	       if (Condition)
		  { if (!(NewCond = Condition->Copy()))
		       { return(SIR_Error);
			} /* fi */
		   } /* fi */
	       else
		  { NewCond = NULL;
		   } /* esle */
	       Assignment = new SIR_DelydAssgn(SIR_DELYDASSGN_AFTER, TmpVar,
							NewCond, Cycles-1,
							this, StateName);
	       if ((Error = NextStates->AttachDelayedAssignments(Assignment)))
		  { delete Assignment;
		    return(Error);
		   } /* fi */
	       delete Assignment;
	      } /* fi */
	  // else nothing to do
	  break;
	 }
     case SIR_FSMDSTMNT_PIPED_EXPR:
	{ if (Cycles > 0)
	     { assert(Expression->ExprType == SIR_EXPR_ASSIGNMENT);
	       if (!(TmpVar = TmpVars->FindOrInsert(Expression->Arg1,
						Cycles /* N buffers */)))
		  { return(SIR_Error);
		   } /* fi */
	       if (Condition)
		  { if (!(NewCond = Condition->Copy()))
		       { return(SIR_Error);
			} /* fi */
		   } /* fi */
	       else
		  { NewCond = NULL;
		   } /* esle */
	       Assignment = new SIR_DelydAssgn(SIR_DELYDASSGN_PIPED, TmpVar,
							NewCond, Cycles-1,
							this, StateName);
	       if ((Error = NextStates->AttachDelayedAssignments(Assignment)))
		  { delete Assignment;
		    return(Error);
		   } /* fi */
	       delete Assignment;
	      } /* fi */
	  // else nothing to do
	  break;
	 }
     case SIR_FSMDSTMNT_IF:
	{ if (!(NewCond = Expression->Copy()))
	     { return(SIR_Error);
	      } /* fi */
	  if (Condition)
	     { if (!(NewCond = SIR_Expression::New(SIR_EXPR_LOGICAL_AND,
						Condition->Copy(), NewCond)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  if ((Error = FsmdStmnt1->CreateDelayedAssignments(TmpVars, NextStates,
							StateName, NewCond)))
	     { NewCond->Delete();
	       return(Error);
	      } /* fi */
	  NewCond->Delete();
	  break;
	 }
     case SIR_FSMDSTMNT_IF_ELSE:
	{ if (!(NewCond = Expression->Copy()))
	     { return(SIR_Error);
	      } /* fi */
	  if (Condition)
	     { if (!(NewCond = SIR_Expression::New(SIR_EXPR_LOGICAL_AND,
						Condition->Copy(), NewCond)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  if ((Error = FsmdStmnt1->CreateDelayedAssignments(TmpVars, NextStates,
							StateName, NewCond)))
	     { NewCond->Delete();
	       return(Error);
	      } /* fi */
	  NewCond->Delete();
	  if (!(NewCond = SIR_Expression::New(SIR_EXPR_LOGICAL_NOT,
						Expression->Copy())))
	     { return(SIR_Error);
	      } /* fi */
	  if (Condition)
	     { if (!(NewCond = SIR_Expression::New(SIR_EXPR_LOGICAL_AND,
						Condition->Copy(), NewCond)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  if ((Error = FsmdStmnt2->CreateDelayedAssignments(TmpVars, NextStates,
							StateName, NewCond)))
	     { NewCond->Delete();
	       return(Error);
	      } /* fi */
	  NewCond->Delete();
	  break;
	 }
     case SIR_FSMDSTMNT_SWITCH:
	{ /* nothing really to do here, except recursion into the body */
	  if ((Error = FsmdStmnt1->CreateDelayedAssignments(TmpVars, NextStates,
							StateName, Condition)))
	     { return(Error);
	      } /* fi */
	  break;
	 }
     case SIR_FSMDSTMNT_CASE:
	{ // note that the 'switch' statement in the 'fsmd'
	  // is quite restricted (for simplicity reasons),
	  // so this handling here becomes quite straightforward
	  SwitchStmnt = GetSwitchStmnt();
	  assert(SwitchStmnt != NULL);	// semantic checking ensures this!
	  if (!(NewCond = SIR_Expression::New(SIR_EXPR_EQUAL,
				SwitchStmnt->Expression->Copy(),
				SIR_Expression::New(Constant->Copy(),
				    SwitchStmnt->Expression->Type->GetTable()))))
	     { return(SIR_Error);
	      } /* fi */
	  if (Condition)
	     { if (!(NewCond = SIR_Expression::New(SIR_EXPR_LOGICAL_AND,
						Condition->Copy(), NewCond)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  if ((Error = FsmdStmnt1->CreateDelayedAssignments(TmpVars, NextStates,
							StateName, NewCond)))
	     { NewCond->Delete();
	       return(Error);
	      } /* fi */
	  NewCond->Delete();
	  break;
	 }
     case SIR_FSMDSTMNT_DEFAULT:
	{ // note that the 'switch' statement in the 'fsmd'
	  // is quite restricted (for simplicity reasons),
	  // so this handling here becomes quite straightforward
	  SwitchStmnt = GetSwitchStmnt();
	  assert(SwitchStmnt != NULL);	// semantic checking ensures this!
	  assert(SwitchStmnt->FsmdStmnt1->StmntType ==	// restrictions ensure
				SIR_FSMDSTMNT_COMPOUND);// this!
	  Case = SwitchStmnt->FsmdStmnt1->FsmdStmnts->First();
	  NewCond = NULL;
	  while(Case)
	     { assert(  (Case->StmntType == SIR_FSMDSTMNT_CASE) // restricted!
		      ||(Case->StmntType == SIR_FSMDSTMNT_DEFAULT));
	       if (Case->StmntType == SIR_FSMDSTMNT_CASE)
		  { if (NewCond)
		       { if (!(NewCond = SIR_Expression::New(
						SIR_EXPR_LOGICAL_AND, NewCond,
			    SIR_Expression::New(SIR_EXPR_NOT_EQUAL,
				SwitchStmnt->Expression->Copy(),
				SIR_Expression::New(Case->Constant->Copy(),
				    SwitchStmnt->Expression->Type->GetTable())))))
			    { return(SIR_Error);
			     } /* fi */
			} /* fi */
		    else
		       { if (!(NewCond = SIR_Expression::New(SIR_EXPR_NOT_EQUAL,
				SwitchStmnt->Expression->Copy(),
				SIR_Expression::New(Case->Constant->Copy(),
				    SwitchStmnt->Expression->Type->GetTable()))))
			    { return(SIR_Error);
			     } /* fi */
			} /* esle */
		   } /* fi */
	       else
		  { assert(Case == this);	// this is part of that list!
		   } /* esle */
	       Case = Case->Succ();
	      } /* elihw */
	  if (Condition)
	     { if (!(NewCond = SIR_Expression::New(SIR_EXPR_LOGICAL_AND,
						Condition->Copy(), NewCond)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  if ((Error = FsmdStmnt1->CreateDelayedAssignments(TmpVars, NextStates,
							StateName, NewCond)))
	     { NewCond->Delete();
	       return(Error);
	      } /* fi */
	  NewCond->Delete();
	  break;
	 }
     case SIR_FSMDSTMNT_GOTO:
	{ // nothing to do
	  break;
	 }
     case SIR_FSMDSTMNT_BREAK:
	{ // nothing to do
	  break;
	 }
     default:
	{ assert(false);	// invalid StmntType
	 }
    } /* hctiws */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdStmnt::CreateDelayedAssignments */


void SIR_FsmdStmnt::DeleteDelayedAssignments(	/* delete 'after'/'piped' tmp.*/
	void)
{

switch(StmntType)
   { case SIR_FSMDSTMNT_COMPOUND:
	{ sir_fsmdstmnt	*Stmnt;

	  Stmnt = FsmdStmnts->First();
	  while(Stmnt)
	     { Stmnt->DeleteDelayedAssignments();
	       Stmnt = Stmnt->Succ();
	      } /* elihw */
	  break;
	 }
     case SIR_FSMDSTMNT_EXPRESSION:
     case SIR_FSMDSTMNT_GOTO:
     case SIR_FSMDSTMNT_BREAK:
	{ // nothing to do
	  break;
	 }
     case SIR_FSMDSTMNT_AFTER_EXPR:
     case SIR_FSMDSTMNT_PIPED_EXPR:
	{ TmpVar = NULL;
	  break;
	 }
     case SIR_FSMDSTMNT_IF:
     case SIR_FSMDSTMNT_SWITCH:
     case SIR_FSMDSTMNT_CASE:
     case SIR_FSMDSTMNT_DEFAULT:
	{ FsmdStmnt1->DeleteDelayedAssignments();
	  break;
	 }
     case SIR_FSMDSTMNT_IF_ELSE:
	{ FsmdStmnt1->DeleteDelayedAssignments();
	  FsmdStmnt2->DeleteDelayedAssignments();
	  break;
	 }
     default:
	{ assert(false);	// invalid StmntType
	 }
    } /* hctiws */

} /* end of SIR_FsmdStmnt::DeleteDelayedAssignments */


sir_state_ptrs *SIR_FsmdStmnt::ReachableStates(	/* collect all reachable states */
	sir_state_ptrs	*StateList)	/* (consumes StateList, returns new) */
{
sir_state_ptrs	*NewStateList = NULL;	// keep compiler quiet

assert(StateList != NULL);	// reachable states up to this point

switch(StmntType)
   { case SIR_FSMDSTMNT_COMPOUND:
	{ sir_fsmdstmnt	*Stmnt;

	  NewStateList = StateList;	// adjust reachable states while
	  Stmnt = FsmdStmnts->First();	// walking over the sequence
	  while(Stmnt)
	     { NewStateList = Stmnt->ReachableStates(NewStateList);
	       Stmnt = Stmnt->Succ();
	      } /* elihw */
	  break;
	 }
     case SIR_FSMDSTMNT_EXPRESSION:
     case SIR_FSMDSTMNT_AFTER_EXPR:
     case SIR_FSMDSTMNT_PIPED_EXPR:
	{ NewStateList = StateList;	// no change in reachable states
	  break;
	 }
     case SIR_FSMDSTMNT_IF:		// states on both paths reachable
	{ sir_state_ptrs	*ReachableIfBranch,
				*ReachableElseBranch;

	  ReachableIfBranch = FsmdStmnt1->ReachableStates(
					new SIR_StatePtrs(StateList));
	  ReachableElseBranch = StateList;
	  NewStateList = ReachableIfBranch->Join(ReachableElseBranch);
	  break;
	 }
     case SIR_FSMDSTMNT_IF_ELSE:	// states on both paths reachable
	{ sir_state_ptrs	*ReachableIfBranch,
				*ReachableElseBranch;

	  ReachableIfBranch = FsmdStmnt1->ReachableStates(
					new SIR_StatePtrs(StateList));
	  ReachableElseBranch = FsmdStmnt2->ReachableStates(
							StateList);
	  NewStateList = ReachableIfBranch->Join(ReachableElseBranch);
	  break;
	 }
     case SIR_FSMDSTMNT_SWITCH:		// states on all paths reachable
	{ sir_fsmdstmnt		*Case;
	  sir_state_ptrs	*ReachableByCases,
				*ReachableByDefault;

	  // note that the 'switch'-'case'-'default' is clean here
	  // (guaranteed by semantic restrictions!), so the handling
	  // is quite straightforward

	  assert(FsmdStmnt1->StmntType == SIR_FSMDSTMNT_COMPOUND);
	  Case = FsmdStmnt1->FsmdStmnts->First();
	  ReachableByCases = new SIR_StatePtrs(/* empty */);
	  ReachableByDefault = new SIR_StatePtrs(StateList);
	  while(Case)
	     { assert(  (Case->StmntType == SIR_FSMDSTMNT_CASE)
		      ||(Case->StmntType == SIR_FSMDSTMNT_DEFAULT));
	       if (Case->StmntType == SIR_FSMDSTMNT_CASE)
		  { ReachableByCases = ReachableByCases->Join(
					Case->FsmdStmnt1->ReachableStates(
						new SIR_StatePtrs(StateList)));
		   } /* fi */
	       else	/* SIR_FSMD_STMNT_DEFAULT */
		  { ReachableByDefault = Case->FsmdStmnt1->ReachableStates(
							ReachableByDefault);
		   } /* esle */
	       Case = Case->Succ();
	      } /* elihw */
	  delete StateList;
	  NewStateList = ReachableByCases->Join(ReachableByDefault);
	  break;
	 }
     case SIR_FSMDSTMNT_CASE:
     case SIR_FSMDSTMNT_DEFAULT:
	{ // must never come here, these are handled within the 'switch'!
	  assert(false);
	 }
     case SIR_FSMDSTMNT_GOTO:		// only NextState reachable now
	{ delete StateList;
	  NewStateList = new SIR_StatePtrs(
				new SIR_StatePtr(NextState));
	  break;
	 }
     case SIR_FSMDSTMNT_BREAK:
	{ if (BreakExitsFsmd())		// only 'break' reachable now
	     { delete StateList;
	       NewStateList = new SIR_StatePtrs(new SIR_StatePtr(
					(sir_fsmdstate*)NULL /* 'break' */));
	      } /* fi */
	  else
	     { NewStateList = StateList;// no change in reachable states
	      } /* esle */
	  break;
	 }
     default:
	{ assert(false);	// invalid StmntType
	 }
    } /* hctiws */

assert(NewStateList != NULL);

return(NewStateList);		// reachable states after this point

} /* end of SIR_FsmdStmnt::ReachableStates */


bool SIR_FsmdStmnt::BreakExitsFsmd(     /* check if 'break' exits 'fsmd' */
        void)
{

assert(StmntType == SIR_FSMDSTMNT_BREAK);       // for 'break' only!

return(!(GetSwitchStmnt()));

} /* end of SIR_FsmdStmnt::BreakExitsFsmd */


sir_statement *SIR_FsmdStmnt::GetStatement(/* obtain the 'fsmd' statement */
        void)
{

if (Scope)
   { return(Scope->ParentStmnt);
    } /* fi */

assert(ParentStmnt != NULL);            // must exist

return(ParentStmnt->GetStatement());    // recursion ends at compound statement

} /* end of SIR_FsmdStmnt::GetStatement */


sir_fsmdstmnt *SIR_FsmdStmnt::GetSwitchStmnt(   /* obtain control-flow parent */
        void)                                   /* (for BREAK, CASE, DEFAULT)*/
{
sir_fsmdstmnt   *Ancestor;

Ancestor = ParentStmnt;
while(Ancestor)
   { if (Ancestor->StmntType == SIR_FSMDSTMNT_SWITCH)
        { return(Ancestor);             // return innermost 'switch' statement
         } /* fi */
     Ancestor = Ancestor->ParentStmnt;
    } /* elihw */

return(NULL);                           // no ancestor 'switch' statement!

} /* end of SIR_FsmdStmnt::GetSwitchStmnt */


	/**********************/
	/*** SIR_FsmdStmnts ***/
	/**********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_FsmdStmnts::SIR_FsmdStmnts(		/* constructor #1 */
	sir_symbols	*CmpndScope /* = NULL */,
	sir_fsmdstmnt	*FirstEntry /* = NULL */) :
		SIR_List<SIR_FsmdStmnt>(FirstEntry)
{

SIR_FsmdStmnts::CmpndScope = CmpndScope;

} /* end of SIR_FsmdStmnts::SIR_FsmdStmnts #1 */


SIR_FsmdStmnts::SIR_FsmdStmnts(		/* constructor #3 (duplicator) */
	sir_fsmdstmnts	*Original,
	sir_fsmdstmnt	*ParentStmnt /* = NULL */)
{
sir_fsmdstmnt	*Curr;

Curr = Original->First();
while(Curr)
   { Append(new SIR_FsmdStmnt(Curr, ParentStmnt));
     Curr = Curr->Succ();
    } /* elihw */

SIR_FsmdStmnts::CmpndScope	= Original->CmpndScope;

} /* end of SIR_FsmdStmnts::SIR_FsmdStmnts #3 */


SIR_FsmdStmnts::~SIR_FsmdStmnts(void)	/* destructor */
{

/* nothing to do */

} /* end of SIR_FsmdStmnts::~SIR_FsmdStmnts */


ERROR SIR_FsmdStmnts::DFS_ForAllNodes(	/* iterator over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{
sir_fsmdstmnt	*Stmnt,
		*Succ;

/* this is not a node, but there are nodes below */

Stmnt = First();
while(Stmnt)
   { Succ = Stmnt->Succ();
     if ((SIR_Error = Stmnt->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Stmnt = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdStmnts::DFS_ForAllNodes */


ERROR SIR_FsmdStmnts::DFS_ForAllSymbols( /* iterator over all symbols */
	sir_symbol_mptr	MemberFct,	/* (depth first) */
	sir_symbol_marg	MemberFctArg)
{
sir_fsmdstmnt	*Stmnt,
		*Succ;

/* this is not a symbol, but there are symbols below */

Stmnt = First();
while(Stmnt)
   { Succ = Stmnt->Succ();
     if ((SIR_Error = Stmnt->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Stmnt = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdStmnts::DFS_ForAllSymbols */


ERROR SIR_FsmdStmnts::DFS_ForAllUserTypes( /* iterator over all usertypes */
	sir_usertp_mptr	MemberFct,	/* (depth first) */
	sir_usertp_marg	MemberFctArg)
{
sir_fsmdstmnt	*Stmnt,
		*Succ;

/* this is not a usertype, but there are usertypes below */

Stmnt = First();
while(Stmnt)
   { Succ = Stmnt->Succ();
     if ((SIR_Error = Stmnt->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Stmnt = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdStmnts::DFS_ForAllUserTypes */


ERROR SIR_FsmdStmnts::DFS_ForAllNotes(	/* iterator over all notes */
	sir_note_mptr	MemberFct,	/* (depth first) */
	sir_note_marg	MemberFctArg)
{
sir_fsmdstmnt	*Stmnt,
		*Succ;

/* this is not a note, but there are notes below */

Stmnt = First();
while(Stmnt)
   { Succ = Stmnt->Succ();
     if ((SIR_Error = Stmnt->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Stmnt = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdStmnts::DFS_ForAllNotes */


ERROR SIR_FsmdStmnts::DFS_ForAllFsmdStmnts(	/* iterator over all stmnts. */
	sir_fsmdstmnt_mptr	MemberFct,	/* (depth first) */
	sir_fsmdstmnt_marg	MemberFctArg)
{
sir_fsmdstmnt	*Stmnt,
		*Succ;

/* this is not a statement, but there are statements below */

Stmnt = First();
while(Stmnt)
   { Succ = Stmnt->Succ();
     if ((SIR_Error = Stmnt->DFS_ForAllFsmdStmnts(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Stmnt = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdStmnts::DFS_ForAllFsmdStmnts */


ERROR SIR_FsmdStmnts::DFS_ForAllExpressions( /* iterator over all expressions */
	sir_expr_mptr	MemberFct,		/* (depth first) */
	sir_expr_marg	MemberFctArg)
{
sir_fsmdstmnt	*Stmnt,
		*Succ;

/* this is not an expression, but there are expressions below */

Stmnt = First();
while(Stmnt)
   { Succ = Stmnt->Succ();
     if ((SIR_Error = Stmnt->DFS_ForAllExpressions(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Stmnt = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdStmnts::DFS_ForAllExpressions */


bool SIR_FsmdStmnts::DFS_FindDirectDependant(	/* searches direct deps. */
	sir_symbol	**SkipSymbol,
	sir_symbol	*ThatSymbol,
	sir_symbol	**DepSymbol,
	SIR_DEPENDENCY	*Reason)
{
sir_fsmdstmnt	*Stmnt;

assert(ThatSymbol != NULL);

/* this is not a statement, but there are statements below */

Stmnt = First();
while(Stmnt)
   { if (Stmnt->DFS_FindDirectDependant(SkipSymbol, ThatSymbol,
					DepSymbol, Reason))
	{ return(true);
	 } /* fi */
     Stmnt = Stmnt->Succ();
    } /* elihw */

return(false);

} /* end of SIR_FsmdStmnts::DFS_FindDirectDependant */


bool SIR_FsmdStmnts::DFS_FindIndirectDependant(	/* searches indirect deps. */
	sir_symbol	*ThatSymbol,
	sir_fsmdstmnt	**DepStmnt,
	sir_expression	**DepExpr,
	SIR_DEPENDENCY	*Reason)
{
sir_fsmdstmnt	*Stmnt;

assert(ThatSymbol != NULL);

/* this is not a statement, but there are statements below */

Stmnt = First();
while(Stmnt)
   { if (Stmnt->DFS_FindIndirectDependant(ThatSymbol, DepStmnt, DepExpr,
						Reason))
	{ return(true);
	 } /* fi */
     Stmnt = Stmnt->Succ();
    } /* elihw */

return(false);

} /* end of SIR_FsmdStmnts::DFS_FindIndirectDependant */


void SIR_FsmdStmnts::SetAlias(	/* sets all type, usertype, symbol alias' */
	sir_fsmdstmnts	*Alias)	/* (iterates over symbols and usertypes) */
{
sir_fsmdstmnt	*FsmdStmnt,
		*FsmdStmntAlias;

assert(Alias != NULL);

/* no need to process this node */

FsmdStmnt = First();
FsmdStmntAlias = Alias->First();
while(FsmdStmnt)
   { FsmdStmnt->SetAlias(FsmdStmntAlias);
     FsmdStmnt = FsmdStmnt->Succ();
     FsmdStmntAlias = FsmdStmntAlias->Succ();
    } /* elihw */

} /* end of SIR_FsmdStmnts::SetAlias */


void SIR_FsmdStmnts::UnAlias(	/* unalias all type, usertype, symbol links */
	sir_symbols	*GlobalSymbols)
{
sir_fsmdstmnt	*Stmnt;

Stmnt = First();
while(Stmnt)
   { Stmnt->UnAlias(GlobalSymbols);
     Stmnt = Stmnt->Succ();
    } /* elihw */

} /* end of SIR_FsmdStmnts::UnAlias */


/*** exported functions *************************************************/


	/* none */


/************************************************************************/
/*** implementation of internal functions			      ***/
/************************************************************************/


static sir_expression *NewTmpAssign(	/* create a new temp. assignment */
	sir_tmpvar	*TmpVar,
	int		Index,
	sir_expression	*SourceExpr)
{
sir_expression	*LHS,
		*RHS;

assert(TmpVar != NULL);
assert(TmpVar->Symbol != NULL);
assert(SourceExpr != NULL);

if (!(LHS = SIR_Expression::New(SIR_EXPR_ARRAY_ACCESS,
			SIR_Expression::New(TmpVar->Symbol),
			SIR_Expression::New(
				SIR_Constant::New((LONG_LONG) Index),
				TmpVar->Symbol->Type->GetTable()))))
   { return(NULL);
    } /* fi */

RHS = new SIR_Expression(SourceExpr);	// copy right-hand side
assert(RHS != NULL);

return(SIR_Expression::New(SIR_EXPR_ASSIGNMENT, LHS, RHS));

} /* end of NewTmpAssign */


/* EOF FsmdStmnt.cc */
