/************************************************************************/
/* FsmdState.cc: SpecC Internal Representation, FsmdState Class		*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 02/03/03 */
/************************************************************************/

/* last update: 09/26/06 */

/* modifications: (most recent first)
 *
 * 09/26/06 PC  Adjustments for scrc 2.1
 * 06/03/05 RD	reorganized and renamed global type names
 * 01/14/05 RD	added inclusion of IntRep/DelydAssgn.h
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 11/04/03 RD	added SIR_FsmdState::CreateReachableTree()
 * 11/03/03 RD	added method CollectReachableStates(), member NextStates,
 *		and methods AttachDelayedAssignments()
 * 10/31/03 RD	added classes SIR_StatePtr and SIR_StatePtrs
 * 10/06/03 RD	added code generation for delayed assignments
 * 10/02/03 RD	added member DelayedAssgns to SIR_FsmdState
 * 04/08/03 RD	include missing <ctype.h>
 * 04/07/03 RD	added SIR_FsmdStates::CreateNewStateName()
 * 02/20/03 RD	added support for local variables at API level 2 (Variables)
 * 02/18/03 RD	added API layer 2 support
 * 02/13/03 RD	added support for annotations at states
 * 02/11/03 RD	adjusted C++ code generation
 * 02/07/03 RD	added methods Declare(), Define(), CheckAllDefined()
 * 02/06/03 RD	refined/extended/added methods
 * 02/05/03 RD	implementation of basic methods
 * 02/03/03 RD	initial version
 */


#include "IntRep/FsmdState.h"
#include "IntRep/Design.h"
#include "IntRep/Extern.h"

#include <ctype.h>
#include <assert.h>


/*** constants and macros ***********************************************/


/*** internal type declarations *****************************************/




/*** class implementations **********************************************/


	/*********************/
	/*** SIR_FsmdStmnt ***/
	/*********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_FsmdState::SIR_FsmdState(		/* constructor #1 */
	const char	*StateName,
	sir_fsmdstmnt	*CmpndStmnt /* = NULL */,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(StateName != NULL);
assert(*StateName != 0);

SIR_FsmdState::StateName	= StateName;
SIR_FsmdState::CmpndStmnt	= CmpndStmnt;
SIR_FsmdState::Notes		= NULL;
SIR_FsmdState::DelayedAssgns	= NULL;
SIR_FsmdState::NextStates	= NULL;

} /* end of SIR_FsmdState::SIR_FsmdState #1 */


SIR_FsmdState::SIR_FsmdState(		/* constructor #3 (duplicator) */
	sir_fsmdstate	*Original)
{


SIR_FsmdState::StateName	= Original->StateName;
SIR_FsmdState::CmpndStmnt	= Original->CmpndStmnt ?
					new SIR_FsmdStmnt(Original->CmpndStmnt):
					NULL;
SIR_FsmdState::Notes		= Original->Notes ?
					new SIR_Notes(Original->Notes) :
					NULL;
SIR_FsmdState::DelayedAssgns	= NULL;
SIR_FsmdState::NextStates	= NULL;

} /* end of SIR_FsmdState::SIR_FsmdState #3 */


SIR_FsmdState::~SIR_FsmdState(void)	/* destructor */
{

delete CmpndStmnt;
delete Notes;

} /* end of SIR_FsmdState::~SIR_FsmdState */


ERROR SIR_FsmdState::DFS_ForAllNodes(	/* iterator over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{

if ((SIR_Error = (this->*MemberFct)(MemberFctArg)))	/* process this node */
   { return(SIR_Error);
    } /* fi */

if (CmpndStmnt)
   { if ((SIR_Error = CmpndStmnt->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Notes)
   { if ((SIR_Error = Notes->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdState::DFS_ForAllNodes */


ERROR SIR_FsmdState::DFS_ForAllSymbols(	/* iterator over all symbols */
	sir_symbol_mptr	MemberFct,	/* (depth first) */
	sir_symbol_marg	MemberFctArg)
{

/* this is not a symbol */

if (CmpndStmnt)
   { if ((SIR_Error = CmpndStmnt->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no symbols in Notes */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdState::DFS_ForAllSymbols */


ERROR SIR_FsmdState::DFS_ForAllUserTypes( /* iterator over all usertypes */
	sir_usertp_mptr	MemberFct,	/* (depth first) */
	sir_usertp_marg	MemberFctArg)
{

/* this is not a usertype */

if (CmpndStmnt)
   { if ((SIR_Error = CmpndStmnt->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no usertypes in Notes */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdState::DFS_ForAllUserTypes */


ERROR SIR_FsmdState::DFS_ForAllNotes( /* iterator over all notes */
	sir_note_mptr	MemberFct,	/* (depth first) */
	sir_note_marg	MemberFctArg)
{

/* this is not a note */

if (CmpndStmnt)
   { if ((SIR_Error = CmpndStmnt->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Notes)
   { if ((SIR_Error = Notes->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdState::DFS_ForAllNotes */


ERROR SIR_FsmdState::DFS_ForAllFsmdStmnts( /* iterator over all fsmd stmnts. */
	sir_fsmdstmnt_mptr	MemberFct,	/* (depth first) */
	sir_fsmdstmnt_marg	MemberFctArg)
{

/* this is not a fsmd statement */

if (CmpndStmnt)
   { if ((SIR_Error = CmpndStmnt->DFS_ForAllFsmdStmnts(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no fsmd statements in Notes */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdState::DFS_ForAllFsmdStmnts */


ERROR SIR_FsmdState::DFS_ForAllExpressions( /* iterator over all expressions */
	sir_expr_mptr	MemberFct,		/* (depth first) */
	sir_expr_marg	MemberFctArg)
{

/* this is not an expression */

if (CmpndStmnt)
   { if ((SIR_Error = CmpndStmnt->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no expressions in Notes */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdState::DFS_ForAllExpressions */


void SIR_FsmdState::SetAlias(	/* sets all type, usertype, symbol alias' */
	sir_fsmdstate	*Alias)	/* (iterates over symbols and usertypes) */
{

assert(Alias != NULL);

/* no need to process this node */

if (CmpndStmnt)
   { CmpndStmnt->SetAlias(Alias->CmpndStmnt);
    } /* fi */
/* there are no symbols or usertypes in Notes */

} /* end of SIR_FsmdState::SetAlias */


void SIR_FsmdState::UnAlias(	/* unalias all type, usertype, symbol links */
	sir_symbols	*GlobalSymbols)
{

if (CmpndStmnt)
   { CmpndStmnt->UnAlias(GlobalSymbols);
    } /* fi */
/* there is nothing to unalias in Notes */

} /* end of SIR_FsmdState::UnAlias */


ERROR SIR_FsmdState::WriteSC(	/* (re-) generates SpecC source code */
	gl_io		*IO,
	bool		WriteNotes,
	bool		CplusplusMode /* = false */,
	bool		PutNewLine /* = true */)
{

if (LineInfo)
   { if ((SIR_Error = LineInfo->WriteSC(IO, true, false)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
else
   { if (PutNewLine)
	{ SIR_LineInfo::WriteNewSC(IO, true, false, &LineInfo);
	 } /* fi */
     else
	{ SIR_LineInfo::WriteSPACE(IO, false);
	 } /* fi */
    } /* esle */

if (CplusplusMode)
   { IO->PutS("case " SIR_CXX_FSMD_STATE_PFX);
     IO->PutS(StateName);
     IO->PutC(':');
    } /* fi */
else
   { IO->PutS(StateName);
     IO->PutC(':');
    } /* esle */

IO->TabStepUp();
if (DelayedAssgns)	/* if any, output delayed assignments first */
   { assert(CplusplusMode == true);
     SIR_LineInfo::WriteNewSC(IO, true, true);	// IO->PutC('{');
     if ((SIR_Error = DelayedAssgns->WriteSC(IO, WriteNotes)))
	{ return(SIR_Error);
	 } /* fi */
     SIR_LineInfo::WriteNL(IO, false, true);	// IO->PutC('}');
    } /* fi */
if ((SIR_Error = CmpndStmnt->WriteSC(IO, WriteNotes, CplusplusMode, true,
					true /* is actually fsmd state! */)))
   { return(SIR_Error);
    } /* fi */
IO->TabStepDown();

if ((SIR_Error = IO->Check()))
   { return(SIR_Error);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdState::WriteSC */


#ifndef NDEBUG

const char *SIR_FsmdState::Print(	/* print SpecC code */
	void)
{
gl_io		*IO;
static gl_string PrintBuffer;

if (!(IO = GL_IO::OpenString(&PrintBuffer, GL_IO_WRITE)))
   { SIR_Error = GL_Error;
     return(NULL);
    } /* fi */
if ((SIR_Error = WriteSC(IO, true, false)))
   { IO->Close();
     return(NULL);
    } /* fi */
if ((SIR_Error = IO->CloseWithCheck()))
   { return(NULL);
    } /* fi */

return(PrintBuffer.chars());

} /* end of SIR_FsmdState::Print */

#endif /* NDEBUG */


ERROR SIR_FsmdState::Annotate(		/* attach a note to the state (#1) */
	sir_note	*NewNote)	/* (consumes NewNote) */
{

assert(NewNote != NULL);

NewNote->Symbol = NULL;		/* state note */
NewNote->UserType = NULL;
NewNote->Label = NULL;
NewNote->State = this;

if (Notes)
   { if (! Notes->Insert(NewNote))
	{ delete NewNote;
	  return(SIR_Error);
	 } /* fi */
    } /* fi */
else
   { Notes = new SIR_Notes(NewNote);
    } /* esle */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdState::Annotate (#1) */


ERROR SIR_FsmdState::AttachDelayedAssignments(	/* attach delayed assignment */
	sir_delydassgn	*Assignment)		/* (here and to next states) */
{
sir_delydassgn	*InsertedAssignment;
ERROR		Error;

assert(Assignment != NULL);

// assignment condition must be evaluatable in this state!

assert(CmpndStmnt->Scope->Parent != NULL); // fsmd scope!
if (  (Assignment->Condition)
    &&(Error = Assignment->Condition->IsInvalidInScope(
					CmpndStmnt->Scope->Parent)))
   { sir_lineinfo	*LineInfo2;
     gl_string		TmpString,
			AssignLocation;

     assert(  (Error == SIR_ERROR_SYMBOL_NOT_VISIBLE_IN_SCOPE)
	    ||(Error == SIR_ERROR_USERTYPE_NOT_VISIBLE_IN_SCOPE));
     TmpString = SIR_ErrMsg;	// save the exact problem
     if ((LineInfo2 = Assignment->Origin->LineInfo))
	{ AssignLocation.form("line %u, file \"%s\"",
				LineInfo2->Line,
				LineInfo2->File->Filename.chars());
	 } /* fi */
     else
	{ AssignLocation = "line <unknown>, file <unknown>";
	 } /* esle */
     SIR_ErrMsg.form("Condition %s" GL_ERROR_MSG_NEWLINE
			"of '%s' assignment in state %s" GL_ERROR_MSG_NEWLINE
			"(%s)" GL_ERROR_MSG_NEWLINE
			"cannot be evaluated in state %s" GL_ERROR_MSG_NEWLINE
			"(%s)",
			Assignment->Condition->Print(false,0,0,0),
			Assignment->Type == SIR_DELYDASSGN_AFTER ?
				"after" : "piped",
			Assignment->OriginState,
			AssignLocation.chars(),
			StateName.chars(),
			TmpString.chars());
     return(SIR_ERROR_INVALID_DELAYED_ASSIGNMENT_1);
    } /* fi */

// target expression must be evaluatable in NextState (if it is the last)

if (  (Assignment->Cycles == 0)
    &&(Error = Assignment->TmpVar->Target->IsInvalidInScope(
					CmpndStmnt->Scope->Parent)))
   { sir_lineinfo	*LineInfo2;
     gl_string		TmpString,
			AssignLocation;

     assert(  (Error == SIR_ERROR_SYMBOL_NOT_VISIBLE_IN_SCOPE)
	    ||(Error == SIR_ERROR_USERTYPE_NOT_VISIBLE_IN_SCOPE));
     TmpString = SIR_ErrMsg;	// save the exact problem
     if ((LineInfo2 = Assignment->Origin->LineInfo))
	{ AssignLocation.form("line %u, file \"%s\"",
				LineInfo2->Line,
				LineInfo2->File->Filename.chars());
	 } /* fi */
     else
	{ AssignLocation = "line <unknown>, file <unknown>";
	 } /* esle */
     SIR_ErrMsg.form("Target expression %s" GL_ERROR_MSG_NEWLINE
			"of '%s' assignment in state %s" GL_ERROR_MSG_NEWLINE
			"(%s)" GL_ERROR_MSG_NEWLINE
			"cannot be evaluated in state %s" GL_ERROR_MSG_NEWLINE
			"(%s)",
			Assignment->TmpVar->Target->Print(false,0,0,0),
			Assignment->Type == SIR_DELYDASSGN_AFTER ?
				"after" : "piped",
			Assignment->OriginState,
			AssignLocation.chars(),
			StateName.chars(),
			TmpString.chars());
     return(SIR_ERROR_INVALID_DELAYED_ASSIGNMENT_2);
    } /* fi */

// OK, now attach (a copy of) Assignment to this state

if (!(DelayedAssgns))
   { DelayedAssgns = new SIR_DlydAssgns();
    } /* fi */
if (!(InsertedAssignment = DelayedAssgns->Insert(Assignment)))
   { /* assignment already exists,     */
     /* prune the insertion tree here! */
     return(SIR_ERROR_NO_ERROR);
    } /* fi */
InsertedAssignment->State = this;	// fill in the state link
InsertedAssignment->CheckCollision();	// warn about collisions!

// finally, add delayed assignments to the states following this state

if (Assignment->Cycles > 0)
   { Assignment->Cycles--;
     assert(NextStates != NULL); // we have stored these here
     if ((Error = NextStates->AttachDelayedAssignments(Assignment)))
	{ return(Error);
	 } /* fi */
     Assignment->Cycles++;
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdState::AttachDelayedAssignments */


void SIR_FsmdState::CreateReachableTree(	/* create reachable tree */
	void)					/* (DFS recursion) */
{
sir_state_ptr	*NextState;

if (NextStates)
   { return;	// we have been here before, nothing to do anymore
    } /* fi */

NextStates = CollectReachableStates();

NextState = NextStates->First();
while(NextState)
   { if (NextState->State)
	{ NextState->State->CreateReachableTree();
	 } /* fi */
     NextState = NextState->Succ();
    } /* elihw */

} /* end of SIR_FsmdState::CreateReachableTree */


sir_fsmdstates *SIR_FsmdState::GetList( /* determines the list of this state */
        void)
{

return((sir_fsmdstates*) Head());

} /* end of SIR_FsmdState::GetList */


sir_state_ptrs *SIR_FsmdState::CollectReachableStates(  /* collect reachable */
        void)                       /* states from here (returns new list!) */
{
sir_state_ptrs  *NextStates;
sir_fsmdstmnt   *DefaultStmnt;

NextStates = new SIR_StatePtrs(                 // default: stay in this state
                new SIR_StatePtr(this));

DefaultStmnt = GetStatement()->DefaultStmnts;
if (DefaultStmnt)
   { NextStates = DefaultStmnt->ReachableStates(// walk through default stmnts.
                        NextStates);            // collect user-spec. defaults
    } /* fi */

NextStates = CmpndStmnt->ReachableStates(       // finally, collect 'goto' and
                        NextStates);            // 'break' stmnts. in this state
return(NextStates);

} /* end of SIR_FsmdState::CollectReachableStates */


sir_statement *SIR_FsmdState::GetStatement( /* obtain pointer to 'fsmd' stmnt.*/
        void)
{
assert(CmpndStmnt != NULL);
assert(CmpndStmnt->Scope != NULL);
assert(CmpndStmnt->Scope->ParentStmnt != NULL);

return(CmpndStmnt->Scope->ParentStmnt);

} /* end of SIR_FsmdState::GetStatement */


	/**********************/
	/*** SIR_FsmdStates ***/
	/**********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_FsmdStates::SIR_FsmdStates(		/* constructor #1 */
	sir_fsmdstate	*FirstEntry /* = NULL */) :
		SIR_List<SIR_FsmdState>(FirstEntry)
{

/* nothing else to do */

} /* end of SIR_FsmdStates::SIR_FsmdStates #1 */



SIR_FsmdStates::SIR_FsmdStates(		/* constructor #3 (duplicator) */
	sir_fsmdstates	*Original)
{
sir_fsmdstate	*Curr;

Curr = Original->First();
while(Curr)
   { Append(new SIR_FsmdState(Curr));
     Curr = Curr->Succ();
    } /* elihw */

} /* end of SIR_FsmdStates::SIR_FsmdStates #3 */


SIR_FsmdStates::~SIR_FsmdStates(void)	/* destructor */
{

/* nothing to do */

} /* end of SIR_FsmdStates::~SIR_FsmdStates */


sir_fsmdstate *SIR_FsmdStates::Declare(	/* declare a (possibly new) state */
	const char	*StateName)	/* (always succeeds) */
{
sir_fsmdstate	*State;

if ((State = Find(StateName)))
   { return(State);
    } /* fi */

State = new SIR_FsmdState(StateName);

return(Prepend(State));	/* pure declarations are at the beginning of the list */

} /* end of SIR_FsmdStates::Declare */

sir_fsmdstate *SIR_FsmdStates::Define(	/* define a (possibly new) state */
	const char	*StateName,	/* (may return NULL and SIR_Error) */
	sir_fsmdstmnt	*CmpndStmnt,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */)
{
sir_fsmdstate	*State;

assert(CmpndStmnt != NULL);

if ((State = Find(StateName)))
   { if (State->CmpndStmnt)	/* alread defined? */
	{ SIR_ErrMsg.form("Redefinition of fsmd state '%s'",
				State->StateName.chars());
	  SIR_Error = SIR_ERROR_REDEFINITION_OF_FSMD_STATE;
	  return(NULL);
	 } /* fi */
     /* make previous declaration into a complete definition */
     State->CmpndStmnt = CmpndStmnt;
     State->UpdateLineInfo(Line, FileInfo);
     /* move to the end of the list as the last state fully defined */
     return(Append(Remove(State)));
    } /* fi */

State = new SIR_FsmdState(StateName, CmpndStmnt, Line, FileInfo);

return(Append(State));	/* definitions are at the end of the list */

} /* end of SIR_FsmdStates::Define */


ERROR SIR_FsmdStates::CheckAllDefined(	/* check whether all states are def'd.*/
	void)
{

// note: the methods Declare() and Define() (see above) maintain
//       the list of states such that all declarations are kept at
//       the beginning of the list, and all definitions are kept at
//       the end of the list (in order of their definition);
//       thus, if the first state in the list is a definition,
//       then every state is fully defined; otherwise, report the
//       first state as undefined

if (Empty())
   { return(SIR_ERROR_NO_ERROR);
    } /* fi */

if (First()->CmpndStmnt)
   { return(SIR_ERROR_NO_ERROR);
    } /* fi */

SIR_ErrMsg.form("Undefined fsmd state '%s'",
		First()->StateName.chars());
return(SIR_ERROR_UNDEFINED_FSMD_STATE);

} /* end of SIR_FsmdStates::CheckAllDefined */


ERROR SIR_FsmdStates::DFS_ForAllNodes(	/* iterator over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{
sir_fsmdstate	*State,
		*Succ;

/* this is not a node, but there are nodes below */

State = First();
while(State)
   { Succ = State->Succ();
     if ((SIR_Error = State->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     State = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdStates::DFS_ForAllNodes */


ERROR SIR_FsmdStates::DFS_ForAllSymbols( /* iterator over all symbols */
	sir_symbol_mptr	MemberFct,	/* (depth first) */
	sir_symbol_marg	MemberFctArg)
{
sir_fsmdstate	*State,
		*Succ;

/* this is not a symbol, but there are symbols below */

State = First();
while(State)
   { Succ = State->Succ();
     if ((SIR_Error = State->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     State = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdStates::DFS_ForAllSymbols */


ERROR SIR_FsmdStates::DFS_ForAllUserTypes( /* iterator over all usertypes */
	sir_usertp_mptr	MemberFct,	/* (depth first) */
	sir_usertp_marg	MemberFctArg)
{
sir_fsmdstate	*State,
		*Succ;

/* this is not a usertype, but there are usertypes below */

State = First();
while(State)
   { Succ = State->Succ();
     if ((SIR_Error = State->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     State = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdStates::DFS_ForAllUserTypes */


ERROR SIR_FsmdStates::DFS_ForAllNotes(	/* iterator over all notes */
	sir_note_mptr	MemberFct,	/* (depth first) */
	sir_note_marg	MemberFctArg)
{
sir_fsmdstate	*State,
		*Succ;

/* this is not a note, but there are notes below */

State = First();
while(State)
   { Succ = State->Succ();
     if ((SIR_Error = State->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     State = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdStates::DFS_ForAllNotes */


ERROR SIR_FsmdStates::DFS_ForAllFsmdStmnts(	/* iterator over all stmnts. */
	sir_fsmdstmnt_mptr	MemberFct,	/* (depth first) */
	sir_fsmdstmnt_marg	MemberFctArg)
{
sir_fsmdstate	*State,
		*Succ;

/* this is not a statement, but there are statements below */

State = First();
while(State)
   { Succ = State->Succ();
     if ((SIR_Error = State->DFS_ForAllFsmdStmnts(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     State = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdStates::DFS_ForAllFsmdStmnts */


ERROR SIR_FsmdStates::DFS_ForAllExpressions( /* iterator over all expressions */
	sir_expr_mptr	MemberFct,		/* (depth first) */
	sir_expr_marg	MemberFctArg)
{
sir_fsmdstate	*State,
		*Succ;

/* this is not an expression, but there are expressions below */

State = First();
while(State)
   { Succ = State->Succ();
     if ((SIR_Error = State->DFS_ForAllExpressions(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     State = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FsmdStates::DFS_ForAllExpressions */


void SIR_FsmdStates::SetAlias(	/* sets all type, usertype, symbol alias' */
	sir_fsmdstates	*Alias)	/* (iterates over symbols and usertypes) */
{
sir_fsmdstate	*FsmdState,
		*FsmdStateAlias;

assert(Alias != NULL);

/* no need to process this node */

FsmdState = First();
FsmdStateAlias = Alias->First();
while(FsmdState)
   { FsmdState->SetAlias(FsmdStateAlias);
     FsmdState = FsmdState->Succ();
     FsmdStateAlias = FsmdStateAlias->Succ();
    } /* elihw */

} /* end of SIR_FsmdStates::SetAlias */


void SIR_FsmdStates::UnAlias(	/* unalias all type, usertype, symbol links */
	sir_symbols	*GlobalSymbols)
{
sir_fsmdstate	*State;

State = First();
while(State)
   { State->UnAlias(GlobalSymbols);
     State = State->Succ();
    } /* elihw */

} /* end of SIR_FsmdStates::UnAlias */


sir_fsmdstate *SIR_FsmdStates::Find(    /* find a fsmd state */
        const char      *StateName)     /* (returns NULL if not found) */
{
sir_fsmdstate   *State;

assert(StateName != NULL);

State = First();
while(State)
   { if (0 == strcmp(State->StateName.chars(), StateName))
        { break;
         } /* fi */
     State = State->Succ();
    } /* elihw */

return(State);

} /* end of SIR_FsmdStates::Find */


	/********************/
	/*** SIR_StatePtr ***/
	/********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_StatePtr::SIR_StatePtr(		/* constructor #1 */
	sir_fsmdstate	*State)
{

SIR_StatePtr::State	= State;

} /* end of SIR_StatePtr::SIR_StatePtr #1 */


SIR_StatePtr::SIR_StatePtr(		/* constructor #2 (duplicator) */
	sir_state_ptr	*Original)
{

SIR_StatePtr::State	= Original->State;

} /* end of SIR_StatePtr::SIR_StatePtr #2 */


SIR_StatePtr::~SIR_StatePtr(void)		/* destructor */
{

/* nothing to do */

} /* end of SIR_StatePtr::~SIR_StatePtr */


	/*********************/
	/*** SIR_StatePtrs ***/
	/*********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_StatePtrs::SIR_StatePtrs(		/* constructor #1 */
	sir_state_ptr	*FirstEntry /* = NULL */) :
		SIR_List<SIR_StatePtr>(FirstEntry)
{

/* nothing else to do */

} /* end of SIR_StatePtrs::SIR_StatePtrs #1 */


SIR_StatePtrs::SIR_StatePtrs(		/* constructor #2 (duplicator) */
	sir_state_ptrs	*Original)
{
sir_state_ptr	*StatePtr;

assert(Original != NULL);

StatePtr = Original->First();
while(StatePtr)
   { Append(new SIR_StatePtr(StatePtr));
     StatePtr = StatePtr->Succ();
    } /* elihw */

} /* end of SIR_StatePtrs::SIR_StatePtrs #2 */


SIR_StatePtrs::~SIR_StatePtrs(void)	/* destructor */
{

/* nothing to do */

} /* end of SIR_StatePtrs::~SIR_StatePtrs */


int SIR_StatePtrs::Cmp(			/* comparison for list ordering */
	sir_fsmdstate	*State1,	/* (just as strcmp()) */
	sir_fsmdstate	*State2)
{

if (  (! State1)
    ||(! State2))
   { return((unsigned long)State1 - (unsigned long)State2);
    } /* fi */

// now we're sure no state is NULL (which means the exit state),
// so we can compare the state names

return(strcmp(State1->StateName.chars(), State2->StateName.chars()));

} /* end of SIR_StatePtrs::Cmp */


ERROR SIR_StatePtrs::AttachDelayedAssignments(	/* attach delayed assignments */
	sir_delydassgn	*Assignment)
{
sir_state_ptr	*StatePtr;
sir_lineinfo	*LineInfo;
ERROR		Error;
gl_string	AssignLocation;

assert(Assignment != NULL);

StatePtr = First();
while(StatePtr)
   { if (StatePtr->State)
	{ if ((Error = StatePtr->State->AttachDelayedAssignments(Assignment)))
	     { return(Error);
	      } /* fi */
	 } /* fi */
     else
	{ if ((LineInfo = Assignment->Origin->LineInfo))
	     { AssignLocation.form("line %u, file \"%s\"",
					LineInfo->Line,
					LineInfo->File->Filename.chars());
	      } /* fi */
	  else
	     { AssignLocation = "line <unknown>, file <unknown>";
	      } /* esle */
	  GL_PrintWarningFmt(GL_WARN_STANDARD,
			"'%s' assignment to %s in state %s" GL_WARN_MSG_NEWLINE
			"terminates early due to 'break' statement"
							GL_WARN_MSG_NEWLINE
			"(%s)",
				Assignment->Type == SIR_DELYDASSGN_AFTER ?
					"after" : "piped",
				Assignment->TmpVar->Target->Print(false,0,0,0),
				Assignment->OriginState,
				AssignLocation.chars());
	 } /* esle */
     StatePtr = StatePtr->Succ();
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_StatePtrs::AttachDelayedAssignments */


sir_state_ptrs *SIR_StatePtrs::Join(    /* join two lists of state pointers */
        sir_state_ptrs  *Other)         /* (consumes Other list) */
{
sir_state_ptr   *State,
                *InsertionPoint;
int             CmpVal = 0;     // keep compiler quiet

assert(Other != NULL);  // must be given!

InsertionPoint = First();
while((State = Other->First()))
   { while(  (InsertionPoint)
           &&((CmpVal = Cmp(InsertionPoint->State, State->State)) < 0))
        { InsertionPoint = InsertionPoint->Succ();
         } /* elihw */
     if (InsertionPoint)
        { if (CmpVal > 0)
             { InsertBefore(Other->Remove(State), InsertionPoint);
              } /* fi */
          else
             { delete Other->Remove(State);
              } /* esle */
         } /* fi */
     else
        { Append(Other->Remove(State));
         } /* esle */
    } /* elihw */

delete Other;   // delete the empty other list

return(this);   // return this combined list

} /* end of SIR_StatePtrs::Join */


sir_state_ptrs *SIR_StatePtrs::Add(  /* add a state to the state pointer set */
        sir_fsmdstate   *State)
{
sir_state_ptr   *InsertionPoint;
int             CmpVal = 0;     // keep compiler quiet

InsertionPoint = First();
while(  (InsertionPoint)
      &&((CmpVal = Cmp(InsertionPoint->State, State)) < 0))
   { InsertionPoint = InsertionPoint->Succ();
    } /* elihw */
if (InsertionPoint)
   { if (CmpVal > 0)
        { InsertBefore(new SIR_StatePtr(State), InsertionPoint);
         } /* fi */
     /* else already exists, nothing to do */
    } /* fi */
else
   { Append(new SIR_StatePtr(State));
    } /* esle */

return(this);   // return this extended list

} /* end of SIR_StatePtrs::Add */


/*** exported functions *************************************************/


	/* none */


/* EOF FsmdState.cc */
