/************************************************************************/
/* CcDriver.c: driver for external C compiler				*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 08/31/97 */
/************************************************************************/

/* last update: 09/26/06 */

/* modifications: (most recent first)
 *
 * 09/26/06 PC  Adjustments for scrc 2.1
 * 06/03/05 RD	reorganized and renamed global type names
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 11/21/02 RD	modified call format strings to have exactly one option marker
 * 11/19/02 RD	added quoting of user-supplied arguments (see Quote())
 * 11/21/01 RD	took out default arguments from function definitions
 * 05/25/01 RD	eliminated support for automatic IP wrapping (scc -ip option)
 * 04/30/01 RD	replaced use of obsolete form() from "stream.h" with own one
 * 04/30/01 RD	added this header (last change was 06/18/00)
 */

#include "CcDriver.h"

#include <stdlib.h>
#include <strings.h>
#include <assert.h>
#include <ctype.h>


/*** constants and macros ***********************************************/


#ifdef DEBUG
#ifndef CCD_DEBUG
#define CCD_DEBUG
#endif /* CCD_DEBUG */
#endif /* DEBUG */


/*** internal type declarations *****************************************/


typedef struct CCD_OptList	CCD_OPTLIST;

struct CCD_OptList
{
CCD_OPTLIST	*Succ;
const char	*Option;
};


/*** prototypes of internal functions ***********************************/


static ERROR CheckCallFormat(
	const char	*FormatString,
	const char	*OptionMarker,
	const char	*Tool);

static const char *Quote(
        const char      *Argument);

static const char *BuildCommandLine(
	const char	*Call,
	const char	*InputFilename,
	const char	*OutputFilename,
	const char	*OptionMarker,
	const char	*Tool,
	CCD_OPTLIST	*OptionList,
	const char	*Flags0 = NULL,
	const char	*List1Flag = NULL,
	gl_stringlist	*List1 = NULL,
	const char	*Flags1 = NULL,
	const char	*List2Flag = NULL,
	gl_stringlist	*List2 = NULL,
	const char	*Flags2 = NULL,
	const char	*List3Flag = NULL,
	gl_stringlist	*List3 = NULL,
	const char	*Flags3 = NULL);

static ERROR ExecuteCommand(
	const char	*CommandLine,
	const char	*Meaning);


/*** exported variables *************************************************/


ERROR	CCD_Error = CCD_ERROR_NO_ERROR;	/* last error of this module */


/*** internal variables *************************************************/


static gl_string	ErrorMessage;


static const char	*CallPP = CCD_DEFAULT_PP_CALL;
static const char	*CallCC = CCD_DEFAULT_CC_CALL;
static const char	*CallLD = CCD_DEFAULT_LD_CALL;

static CCD_OPTLIST DefaultOptionsPP[] =
{
//	{ &DefaultOptionsPP[1],	CCD_DEFAULT_PP_OPT_1 },
	{ NULL,			CCD_DEFAULT_PP_OPTS }
};
static CCD_OPTLIST DefaultOptionsCC[] =
{
//	{ &DefaultOptionsCC[1],	CCD_DEFAULT_CC_OPT_1 },
	{ NULL,			CCD_DEFAULT_CC_OPTS }
};
static CCD_OPTLIST DefaultOptionsLD[] =
{
//	{ &DefaultOptionsLD[1],	CCD_DEFAULT_LD_OPT_1 },
	{ NULL,			CCD_DEFAULT_LD_OPTS }
};

static CCD_OPTLIST	*OptionsPP = &DefaultOptionsPP[0];
static CCD_OPTLIST	*OptionsCC = &DefaultOptionsCC[0];
static CCD_OPTLIST	*OptionsLD = &DefaultOptionsLD[0];

static CCD_MODE		CompilerMode = CCD_MODE_STANDARD;
static CCD_TARGET       CompilerTarget = CCD_TARGET_EXECUTABLE;

static gl_stringlist	*Definitions = NULL;
static gl_stringlist	*UnDefs = NULL;
static gl_stringlist	*IncludePath = NULL;
static gl_stringlist	*LibraryPath = NULL;
static gl_stringlist	*LibraryList = NULL;

static const char	*StdMacroDefs	= CCD_STD_DEFINES;
static const char	*StdUnDefs	= CCD_STD_UNDEFINES;
static const char	*StdIncludePath	= CCD_STD_INCLUDE_PATH;
static const char	*StdLibraryPath	= CCD_STD_LIBRARY_PATH;
static const char	*StdLibraries	= CCD_STD_LIBRARIES;


/************************************************************************/
/*** implementation of exported functions			      ***/
/************************************************************************/


	/* error handling */

void CCD_Reset(void)		/* recover from error status */
{

GL_Reset();

CCD_SetDefaultOptions();

CCD_Error = CCD_ERROR_NO_ERROR;

} /* end of CCD_Reset */


const char *CCD_ErrorText(	/* returns message of error code */
	ERROR	ErrorNumber)
{

if (  (ErrorNumber >= GL_ERROR_BASE)
    &&(ErrorNumber <= GL_ERROR_BASE + GL_ERROR_RANGE))
   { return(GL_ErrorText(ErrorNumber));
    } /* fi */

switch (ErrorNumber)
   { case CCD_ERROR_NO_ERROR:
	  return("No error (internal)");
     case CCD_ERROR_OUT_OF_MEMORY:
	  return("Out of memory");
     case CCD_ERROR_SUBSHELL_NOT_AVAILABLE:
	  return("Subshell (sh) not available");
     case CCD_ERROR_CANNOT_EXECUTE_SUBSHELL:
	  return(ErrorMessage.chars()
		/* "Cannot execute subshell" + GL_SystemErrorMessage() */);
     case CCD_ERROR_COMMAND_RETURNED_ERROR:
	  return(ErrorMessage.chars()
		/* "Preprocessing failed (subshell returned error code %d)" */);
     case CCD_ERROR_ILLEGAL_MODE:
	  return("Illegal compiler mode (internal)");
     case CCD_ERROR_ILLEGAL_TARGET:
	  return("Illegal compiler target (internal)");
     case CCD_ERROR_EXACTLY_ONE_I_REQUIRED:
	  return(ErrorMessage.chars()
		/* "Bad format string for %s call:" GL_ERROR_MSG_NEWLINE
		"exactly one %%i marker required" */);
     case CCD_ERROR_EXACTLY_ONE_O_REQUIRED:
	  return(ErrorMessage.chars()
		/* "Bad format string for %s call:" GL_ERROR_MSG_NEWLINE
		"exactly one %%o escape sequence required" */);
     case CCD_ERROR_EXACTLY_ONE_FLAG_REQUIRED:
	  return(ErrorMessage.chars()
		/* "Bad format string for %s call:" GL_ERROR_MSG_NEWLINE
		"exactly one %s marker required" */);
     case CCD_ERROR_PERCENT_CHAR_NOT_ALLOWED:
	  return("Percent character (%) not allowed in file name");
    } /* hctiws */

return("Unknown error number (internal)");

} /* end of CCD_ErrorText */


/*** option settings ****************************************************/


void CCD_SetDefaultOptions(void)
{
CCD_OPTLIST	*Opt,
		*Succ;

CallPP = CCD_DEFAULT_PP_CALL;
CallCC = CCD_DEFAULT_CC_CALL;
CallLD = CCD_DEFAULT_LD_CALL;

if (OptionsPP != &DefaultOptionsPP[0])
   { Opt = OptionsPP;
     while(Opt)
	{ Succ = Opt->Succ;
	  delete Opt;
	  Opt = Succ;
	 } /* elihw */
     OptionsPP = &DefaultOptionsPP[0];
    } /* fi */
if (OptionsCC != &DefaultOptionsCC[0])
   { Opt = OptionsCC;
     while(Opt)
	{ Succ = Opt->Succ;
	  delete Opt;
	  Opt = Succ;
	 } /* elihw */
     OptionsCC = &DefaultOptionsCC[0];
    } /* fi */
if (OptionsLD != &DefaultOptionsLD[0])
   { Opt = OptionsLD;
     while(Opt)
	{ Succ = Opt->Succ;
	  delete Opt;
	  Opt = Succ;
	 } /* elihw */
     OptionsLD = &DefaultOptionsLD[0];
    } /* fi */

CompilerMode = CCD_MODE_STANDARD;
CompilerTarget = CCD_TARGET_EXECUTABLE;

Definitions = NULL;
UnDefs = NULL;
IncludePath = NULL;
LibraryPath = NULL;
LibraryList = NULL;

StdMacroDefs	= CCD_STD_DEFINES;
StdUnDefs	= CCD_STD_UNDEFINES;
StdIncludePath	= CCD_STD_INCLUDE_PATH;
StdLibraryPath	= CCD_STD_LIBRARY_PATH;
StdLibraries	= CCD_STD_LIBRARIES;

} /* end of CCD_SetDefaultOptions */


ERROR CCD_SetCompilerMode(CCD_MODE Mode)
{

switch(Mode)
   { case CCD_MODE_STANDARD:
     case CCD_MODE_DEBUG:
     case CCD_MODE_OPTIMIZE:
	{ CompilerMode = Mode;
	  break;
	 }
     default:
	{ return(CCD_ERROR_ILLEGAL_MODE);
	 }
    } /* hctiws */

return(CCD_ERROR_NO_ERROR);

} /* end of CCD_SetCompilerMode */

ERROR CCD_SetTarget(CCD_TARGET Target)
{

switch(Target)
   { case CCD_TARGET_EXECUTABLE:
     case CCD_TARGET_SHARED_LIB:
        { CompilerTarget = Target;
          break;
         }
     default:
        { return(CCD_ERROR_ILLEGAL_TARGET);
         }
    } /* hctiws */

return(CCD_ERROR_NO_ERROR);

} /* end of CCD_SetTarget */


ERROR CCD_SetPreprocessorCall(const char *FormatString)
{
ERROR	Error;

if ((Error = CheckCallFormat(FormatString, "%p", "preprocessor")))
   { return(Error);
    } /* fi */

CallPP = FormatString;

return(CCD_ERROR_NO_ERROR);

} /* end of CCD_SetPreprocessorCall */


ERROR CCD_SetCompilerCall(const char *FormatString)
{
ERROR	Error;

if ((Error = CheckCallFormat(FormatString, "%c", "compiler")))
   { return(Error);
    } /* fi */

CallCC = FormatString;

return(CCD_ERROR_NO_ERROR);

} /* end of CCD_SetCompilerCall */


ERROR CCD_SetLinkerCall(const char *FormatString)
{
ERROR	Error;

if ((Error = CheckCallFormat(FormatString, "%l", "linker")))
   { return(Error);
    } /* fi */

CallLD = FormatString;

return(CCD_ERROR_NO_ERROR);

} /* end of CCD_SetLinkerCall */


ERROR CCD_SetPreprocessorOption(const char *Option)
{
CCD_OPTLIST	*NewOpt,
		*OptPred;

if (!(NewOpt = new(CCD_OPTLIST)))
   { return(CCD_ERROR_OUT_OF_MEMORY);
    } /* fi */
NewOpt->Succ = NULL;
NewOpt->Option = Quote(Option);         /* pre-quote the option */

if (OptionsPP == &DefaultOptionsPP[0])	/* still default options? */
   { OptionsPP = NewOpt;		/* start user options list */
    } /* fi */
else
   { OptPred = (CCD_OPTLIST*)&OptionsPP;
     while(OptPred->Succ)
	{ OptPred = OptPred->Succ;	/* append to user options list */
	 } /* elihw */
     OptPred->Succ = NewOpt;
    } /* esle */

return(CCD_ERROR_NO_ERROR);

} /* end of CCD_SetPreprocessorOption */


ERROR CCD_SetCompilerOption(const char *Option)
{
CCD_OPTLIST	*NewOpt,
		*OptPred;

if (!(NewOpt = new(CCD_OPTLIST)))
   { return(CCD_ERROR_OUT_OF_MEMORY);
    } /* fi */
NewOpt->Succ = NULL;
NewOpt->Option = Quote(Option);         /* pre-quote the option */

if (OptionsCC == &DefaultOptionsCC[0])	/* still default options? */
   { OptionsCC = NewOpt;		/* start user options list */
    } /* fi */
else
   { OptPred = (CCD_OPTLIST*)&OptionsCC;
     while(OptPred->Succ)
	{ OptPred = OptPred->Succ;	/* append to user options list */
	 } /* elihw */
     OptPred->Succ = NewOpt;
    } /* esle */

return(CCD_ERROR_NO_ERROR);

} /* end of CCD_SetCompilerOption */


ERROR CCD_SetLinkerOption(const char *Option)
{
CCD_OPTLIST	*NewOpt,
		*OptPred;

if (!(NewOpt = new(CCD_OPTLIST)))
   { return(CCD_ERROR_OUT_OF_MEMORY);
    } /* fi */
NewOpt->Succ = NULL;
NewOpt->Option = Quote(Option);         /* pre-quote the option */

if (OptionsLD == &DefaultOptionsLD[0])	/* still default options? */
   { OptionsLD = NewOpt;		/* start user options list */
    } /* fi */
else
   { OptPred = (CCD_OPTLIST*)&OptionsLD;
     while(OptPred->Succ)
	{ OptPred = OptPred->Succ;	/* append to user options list */
	 } /* elihw */
     OptPred->Succ = NewOpt;
    } /* esle */

return(CCD_ERROR_NO_ERROR);

} /* end of CCD_SetLinkerOption */


ERROR CCD_SetMacroDefs(			/* specify the macro definitions */
	gl_stringlist	*DList)
{

Definitions = DList;

return(CCD_ERROR_NO_ERROR);

} /* end of CCD_SetMacroDefs */


ERROR CCD_SetUnDefs(			/* specify the un-definitions */
	gl_stringlist	*UList)
{

UnDefs = UList;

return(CCD_ERROR_NO_ERROR);

} /* end of CCD_SetUnDefs */


ERROR CCD_SetIncludePath(		/* specify the include path */
	gl_stringlist	*IList)
{

IncludePath = IList;

return(CCD_ERROR_NO_ERROR);

} /* end of CCD_SetIncludePath */


ERROR CCD_SetLibraryPath(		/* specify the library path */
	gl_stringlist	*LList)
{

LibraryPath = LList;

return(CCD_ERROR_NO_ERROR);

} /* end of CCD_SetLibraryPath */


ERROR CCD_SetLibraries(			/* specify the library list */
	gl_stringlist	*lList)
{

LibraryList = lList;

return(CCD_ERROR_NO_ERROR);

} /* end of CCD_SetLibraries */


ERROR CCD_NoStdMacroDefs(void)
{

StdMacroDefs = NULL;

return(CCD_ERROR_NO_ERROR);

} /* end of CCD_NoStdMacroDefs */


ERROR CCD_NoStdUnDefs(void)
{

StdUnDefs = NULL;

return(CCD_ERROR_NO_ERROR);

} /* end of CCD_NoStdUnDefs */


ERROR CCD_NoStdIncludePath(void)
{

StdIncludePath = NULL;

return(CCD_ERROR_NO_ERROR);

} /* end of CCD_NoStdIncludePath */


ERROR CCD_NoStdLibraryPath(void)
{

StdLibraryPath = NULL;

return(CCD_ERROR_NO_ERROR);

} /* end of CCD_NoStdLibraryPath */


ERROR CCD_NoStdLibraries(void)
{

StdLibraries = NULL;

return(CCD_ERROR_NO_ERROR);

} /* end of CCD_NoStdLibraries */


/*** main functionality *************************************************/


ERROR CCD_Preprocessor(
	const char	*InputFilename,
	const char	*OutputFilename)
{
const char	*CommandLine;

assert(InputFilename != NULL);
assert(OutputFilename != NULL);

if (index(InputFilename, '%'))
   { return(CCD_ERROR_PERCENT_CHAR_NOT_ALLOWED);
    } /* fi */
if (index(OutputFilename, '%'))
   { return(CCD_ERROR_PERCENT_CHAR_NOT_ALLOWED);
    } /* fi */

CommandLine = BuildCommandLine(CallPP,
			InputFilename, OutputFilename,
			"%p", "preprocessor", OptionsPP,
			NULL,
			CCD_DEFINITION_OPT, Definitions, StdMacroDefs,
			CCD_UNDEF_OPT, UnDefs, StdUnDefs,
			CCD_INCLUDE_PATH_OPT, IncludePath, StdIncludePath);

if ((CCD_Error = GL_FileIsReadable(InputFilename)))
   { return(CCD_Error);
    } /* fi */
if ((CCD_Error = GL_FileIsCreatable(OutputFilename)))
   { return(CCD_Error);
    } /* fi */

if ((CCD_Error = ExecuteCommand(CommandLine, "Preprocessing")))
   { return(CCD_ERROR_CANNOT_EXECUTE_SUBSHELL);
    } /* fi */

if ((CCD_Error = GL_FileIsReadable(OutputFilename)))
   { return(CCD_Error);
    } /* fi */

return(CCD_ERROR_NO_ERROR);

} /* end of CCD_Preprocessor */


ERROR CCD_Compiler(
	const char	*InputFilename,
	const char	*OutputFilename)
{
const char	*CommandLine;

assert(InputFilename != NULL);
assert(OutputFilename != NULL);

if (index(InputFilename, '%'))
   { return(CCD_ERROR_PERCENT_CHAR_NOT_ALLOWED);
    } /* fi */
if (index(OutputFilename, '%'))
   { return(CCD_ERROR_PERCENT_CHAR_NOT_ALLOWED);
    } /* fi */

CommandLine = BuildCommandLine(CallCC,
		InputFilename, OutputFilename,
		"%c", "compiler", OptionsCC,
		NULL,
		CCD_DEFINITION_OPT, Definitions, StdMacroDefs,
		CCD_UNDEF_OPT, UnDefs, StdUnDefs,
		CCD_INCLUDE_PATH_OPT, IncludePath, StdIncludePath);

if ((CCD_Error = GL_FileIsReadable(InputFilename)))
   { return(CCD_Error);
    } /* fi */
if ((CCD_Error = GL_FileIsCreatable(OutputFilename)))
   { return(CCD_Error);
    } /* fi */

if ((CCD_Error = ExecuteCommand(CommandLine, "Compiling")))
   { return(CCD_ERROR_CANNOT_EXECUTE_SUBSHELL);
    } /* fi */

if ((CCD_Error = GL_FileIsReadable(OutputFilename)))
   { return(CCD_Error);
    } /* fi */

return(CCD_ERROR_NO_ERROR);

} /* end of CCD_Compiler */


ERROR CCD_Linker(
	const char	*InputFilename,
	const char	*OutputFilename)
{
const char	*CommandLine;

assert(InputFilename != NULL);
assert(OutputFilename != NULL);

if (index(InputFilename, '%'))
   { return(CCD_ERROR_PERCENT_CHAR_NOT_ALLOWED);
    } /* fi */
if (index(OutputFilename, '%'))
   { return(CCD_ERROR_PERCENT_CHAR_NOT_ALLOWED);
    } /* fi */

CommandLine = BuildCommandLine(CallLD,
		InputFilename, OutputFilename,
		"%l", "linker", OptionsLD,
		NULL,
		CCD_LIBRARY_PATH_OPT, LibraryPath, StdLibraryPath,
		CCD_LIBRARY_LINK_OPT, LibraryList, StdLibraries);

if ((CCD_Error = GL_FileIsReadable(InputFilename)))
   { return(CCD_Error);
    } /* fi */
if ((CCD_Error = GL_FileIsCreatable(OutputFilename)))
   { return(CCD_Error);
    } /* fi */

if ((CCD_Error = ExecuteCommand(CommandLine, "Linking")))
   { return(CCD_ERROR_CANNOT_EXECUTE_SUBSHELL);
    } /* fi */

if ((CCD_Error = GL_FileIsReadable(OutputFilename)))
   { return(CCD_Error);
    } /* fi */

return(CCD_ERROR_NO_ERROR);

} /* end of CCD_Linker */


/************************************************************************/
/*** implementation of internal functions			      ***/
/************************************************************************/


static ERROR CheckCallFormat(
	const char	*FormatString,
	const char	*OptionMarker,
	const char	*Tool)
{
gl_string	TmpString;

TmpString = FormatString;
if (TmpString.freq("%i") != 1)
   { ErrorMessage.form(
	"Bad format string for %s call:" GL_ERROR_MSG_NEWLINE
	"exactly one %%i marker required",
	Tool);
     return(CCD_ERROR_EXACTLY_ONE_I_REQUIRED);
    } /* fi */
if (TmpString.freq("%o") != 1)
   { ErrorMessage.form(
	"Bad format string for %s call:" GL_ERROR_MSG_NEWLINE
	"exactly one %%o marker required",
	Tool);
     return(CCD_ERROR_EXACTLY_ONE_O_REQUIRED);
    } /* fi */
if (TmpString.freq(OptionMarker) < 1)
   { ErrorMessage.form(
	"Bad format string for %s call:" GL_ERROR_MSG_NEWLINE
	"exactly one %s marker required",
	Tool, OptionMarker);
     return(CCD_ERROR_EXACTLY_ONE_FLAG_REQUIRED);
    } /* fi */

return(CCD_ERROR_NO_ERROR);

} /* end of CheckCallFormat */


static const char *Quote(
        const char      *Argument)
{
const char	*CharPtr;
bool		QuotingNeeded;
static gl_string QuotedArg;

assert(Argument != NULL);

if (*Argument == 0)
   { return("\"\"");    /* explicit empty argument */
    } /* fi */

QuotingNeeded = false;
CharPtr = Argument;
while(*CharPtr)
   { if (  (!(isalnum(*CharPtr)))
	 &&(*CharPtr != '+')
	 &&(*CharPtr != '-')
	 &&(*CharPtr != '.')
	 &&(*CharPtr != '_')
	 &&(*CharPtr != '=')
	 &&(*CharPtr != '/'))
	{ QuotingNeeded = true;
	  break;
	 } /* fi */
     CharPtr++;
    } /* elihw */

if (! QuotingNeeded)
   { return(Argument);  /* no quoting needed */
    } /* fi */

QuotedArg = '"';
CharPtr = Argument;
while(*CharPtr)
   { if (  (*CharPtr == '\\')   // escape the backslash
         ||(*CharPtr == '`')    // escape the left-tick
         ||(*CharPtr == '"')    // escape the double-quote
         ||(*CharPtr == '$'))   // escape the dollar
        { QuotedArg += '\\';
         } /* fi */
     QuotedArg += *CharPtr++;
    } /* elihw */
QuotedArg += '"';

return(QuotedArg.chars());

} /* end of Quote */


static const char *BuildCommandLine(
	const char	*Call,
	const char	*InputFilename,
	const char	*OutputFilename,
	const char	*OptionMarker,
	const char	*Tool,
	CCD_OPTLIST	*OptionList,
	const char	*Flags0 /* = NULL */,
	const char	*List1Flag /* = NULL */,
	gl_stringlist	*List1 /* = NULL */,
	const char	*Flags1 /* = NULL */,
	const char	*List2Flag /* = NULL */,
	gl_stringlist	*List2 /* = NULL */,
	const char	*Flags2 /* = NULL */,
	const char	*List3Flag /* = NULL */,
	gl_stringlist	*List3 /* = NULL */,
	const char	*Flags3 /* = NULL */)
{
static gl_string CommandLine;
CCD_OPTLIST	*Option;
gl_stringlist	*StringPtr;
gl_string	Flags,
		Flag,
		Options;

#ifdef CCD_DEBUG
printf("CCD_DEBUG: Call   = \"%s\"\n", Call);
printf("CCD_DEBUG: Input  = \"%s\"\n", InputFilename);
printf("CCD_DEBUG: Output = \"%s\"\n", OutputFilename);
#endif /* CCD_DEBUG */

CommandLine = Call;			/* start with the format string */

assert(CommandLine.freq("%i") == 1);	/* replace the '%i' with input */
CommandLine.at("%i") = Quote(InputFilename);

assert(CommandLine.freq("%o") == 1);	/* replace the '%o' with output */
CommandLine.at("%o") = Quote(OutputFilename);

Flags = "";				/* handle the flags */
switch(CompilerMode)
   { case CCD_MODE_STANDARD:
        { /* no flag */
          break;
         }
     case CCD_MODE_DEBUG:
        { Flags += " " CCD_DEBUG_OPT;           // hard-quoted
          break;
         }
     case CCD_MODE_OPTIMIZE:
        { Flags += " " CCD_OPTIMIZE_OPT;        // hard-quoted
          break;
         }
     default:
	{ assert(false);	/* unknown compiler mode */
	 }
    } /* hctiws */

switch(GL_WarningLevel)
   { case GL_WARN_ALWAYS:
        { Flags += " " CCD_NO_WARN_OPT;         // hard-quoted
          break;
         }
     case GL_WARN_IMPORTANT:
        { Flags += " " CCD_IMPORTANT_WARN_OPT;  // hard-quoted
          break;
         }
     case GL_WARN_STANDARD:
     case GL_WARN_INFORMATIVE:
        { Flags += " " CCD_STANDARD_WARN_OPT;   // hard-quoted
          break;
         }
     case GL_WARN_PEDANTIC:
     default:
        { Flags += " " CCD_PEDANTIC_WARN_OPT;   // hard-quoted
          break;
         }
    } /* hctiws */

if (Flags0)
   { Flags += " ";
     Flags += Flags0;   // hard-quoted
    } /* fi */

StringPtr = List1;
while(StringPtr)
   { if (List1Flag)
        { Flag = List1Flag;
          Flag += StringPtr->String;
         } /* fi */
     else
        { Flag = StringPtr->String;
         } /* esle */
     Flags += " ";
     Flags += Quote(Flag.chars());
     StringPtr = StringPtr->Succ;
    } /* elihw */

if (Flags1)
   { Flags += " ";
     Flags += Flags1;   // hard-quoted
    } /* fi */

StringPtr = List2;
while(StringPtr)
   { if (List2Flag)
        { Flag = List2Flag;
          Flag += StringPtr->String;
         } /* fi */
     else
        { Flag = StringPtr->String;
         } /* esle */
     Flags += " ";
     Flags += Quote(Flag.chars());
     StringPtr = StringPtr->Succ;
    } /* elihw */

if (Flags2)
   { Flags += " ";
     Flags += Flags2;   // hard-quoted
    } /* fi */

StringPtr = List3;
while(StringPtr)
   { if (List3Flag)
        { Flag = List3Flag;
          Flag += StringPtr->String;
         } /* fi */
     else
        { Flag = StringPtr->String;
         } /* esle */
     Flags += " ";
     Flags += Quote(Flag.chars());
     StringPtr = StringPtr->Succ;
    } /* elihw */

if (Flags3)
   { Flags += " ";
     Flags += Flags3;   // hard-quoted
    } /* fi */

#ifdef CCD_DEBUG
printf("CCD_DEBUG: Flags = \"%s\"\n", Flags.chars());
#endif /* CCD_DEBUG */

Options = "";                                   /* handle the options */
Option = OptionList;
while(Option)
   { if (  (Option->Option)
         &&(*Option->Option))
        {
#ifdef CCD_DEBUG
          printf("CCD_DEBUG: Option = \"%s\"\n", Option->Option);
#endif /* CCD_DEBUG */
          Options += " ";
          Options += Option->Option;    // hard- or pre-quoted
         } /* fi */
     Option = Option->Succ;
    } /* elihw */

#ifdef CCD_DEBUG
printf("CCD_DEBUG: Options = \"%s\"\n", Options.chars());
#endif /* CCD_DEBUG */

assert(CommandLine.freq(OptionMarker) == 1);    /* replace the option marker */
Flags += Options;                               /* with flags and options    */
CommandLine.at(OptionMarker) = Flags;

#ifdef CCD_DEBUG
printf("CCD_DEBUG: CommandLine = \"%s\"\n", CommandLine.chars());
#endif /* CCD_DEBUG */

return(CommandLine.chars());    /* return the composed command line */

} /* end of BuildCommandLine */

static ERROR ExecuteCommand(
	const char	*CommandLine,
	const char	*Meaning)
{
int	RetVal;

#ifndef SUNOS4	/* not supported on SunOS 4 systems */
if (!(system(NULL)))	/* check availability of subshell (sh) */
   { return(CCD_ERROR_SUBSHELL_NOT_AVAILABLE);
    } /* fi */
#endif /* SUNOS4 */

#ifdef CCD_DEBUG
printf("CCD_DEBUG: system(\"%s\")\n", CommandLine);
#endif /* CCD_DEBUG */

if (GL_VerbosityLevel >= 3)
   { fprintf(stderr, "      %s\n", CommandLine);
    } /* fi */

RetVal = system(CommandLine);	/* execute command line in subshell */

#ifdef CCD_DEBUG
printf("CCD_DEBUG: system() returned %d.\n", RetVal);
#endif /* CCD_DEBUG */

if (RetVal < 0)	/* fork() or exec() failed */
   { ErrorMessage = "Cannot execute subshell";
     ErrorMessage += GL_SystemErrorMessage();
     return(CCD_ERROR_CANNOT_EXECUTE_SUBSHELL);
    } /* fi */
if (RetVal > 0)	/* shell returns error value from command */
   { ErrorMessage.form("%s failed (subshell returned error code %d)",
			Meaning, RetVal);
     return(CCD_ERROR_COMMAND_RETURNED_ERROR);
    } /* fi */

return(CCD_ERROR_NO_ERROR);

} /* end of ExecuteCommand */


/************************************************************************/
/*** main (for debugging only)					      ***/
/************************************************************************/


#ifdef DEBUG	/* module self test */

int main(int argc, char **argv)
{
ERROR	Error;

printf("DEBUGGING: CcDriver ('%s')\n\n", argv[0]);
if (argc != 1)
   { puts("WARNING: Arguments will be ignored!\n");
    } /* fi */

puts("Testing internal functions...");
puts("ExecuteCommand(\"echo Test\", \"Testing\")...");
if ((Error = ExecuteCommand("echo Test", "Testing")))
   { GL_PrintError(Error, CCD_ErrorText(Error));
     exit(RESULT_ERROR);
    } /* fi */

puts("\nTesting compiler steps on HelloWorld.c...");

if ((Error = CCD_NoStdMacroDefs()))	/* disable these, they're not needed */
   { GL_PrintError(Error, CCD_ErrorText(Error));
     exit(RESULT_ERROR);
    } /* fi */
if ((Error = CCD_NoStdUnDefs()))
   { GL_PrintError(Error, CCD_ErrorText(Error));
     exit(RESULT_ERROR);
    } /* fi */
if ((Error = CCD_NoStdIncludePath()))
   { GL_PrintError(Error, CCD_ErrorText(Error));
     exit(RESULT_ERROR);
    } /* fi */
if ((Error = CCD_NoStdLibraryPath()))
   { GL_PrintError(Error, CCD_ErrorText(Error));
     exit(RESULT_ERROR);
    } /* fi */
if ((Error = CCD_NoStdLibraries()))
   { GL_PrintError(Error, CCD_ErrorText(Error));
     exit(RESULT_ERROR);
    } /* fi */

puts("\nPreprocessor(\"HelloWorld.c\", \"HelloWorld.i\")...");
if ((Error = CCD_Preprocessor("HelloWorld.c", "HelloWorld.i")))
   { GL_PrintError(Error, CCD_ErrorText(Error));
     exit(RESULT_ERROR);
    } /* fi */

if ((Error = CCD_SetCompilerCall("gcc -c %c %i -o %o")))
   { GL_PrintError(Error, CCD_ErrorText(Error));
     exit(RESULT_ERROR);
    } /* fi */

puts("\nCompiler(\"HelloWorld.i\", \"HelloWorld.o\")...");
if ((Error = CCD_Compiler("HelloWorld.i", "HelloWorld.o")))
   { GL_PrintError(Error, CCD_ErrorText(Error));
     exit(RESULT_ERROR);
    } /* fi */

if ((Error = CCD_SetLinkerCall("gcc %i -o %o %l")))
   { GL_PrintError(Error, CCD_ErrorText(Error));
     exit(RESULT_ERROR);
    } /* fi */

puts("\nLinker(\"HelloWorld.o\", \"HelloWorld\")...");
#ifdef GNUWIN32
if ((Error = CCD_Linker("HelloWorld.o", "HelloWorld.exe")))
#else
if ((Error = CCD_Linker("HelloWorld.o", "HelloWorld")))
#endif
   { GL_PrintError(Error, CCD_ErrorText(Error));
     exit(RESULT_ERROR);
    } /* fi */

puts("\nExecuteCommand(\"./HelloWorld\", \"Testing\")...");
if ((Error = ExecuteCommand("./HelloWorld", "Testing")))
   { GL_PrintError(Error, CCD_ErrorText(Error));
     exit(RESULT_ERROR);
    } /* fi */

puts("\nDone.");
return(RESULT_OK);	/* success! */

} /* end of main */

#endif /* DEBUG */


/* EOF CcDriver.c */
