/************************************************************************/
/* Compiler.c: The SpecC Compiler					*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 08/31/97 */
/************************************************************************/

/* last update: 09/26/06 */

/* modifications: (most recent first)
 *
 * 09/26/06 PC  Adjustments for scrc 2.1
 * 06/06/05 RD	reorganized and renamed global type names
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 05/29/03 RD	get definitions for standard import directory from CcDriver.h
 * 11/03/02 RD	fixed clearing of settings lists (after code review by AG)
 * 02/13/02 RD	added standard import directory for default import search path
 * 11/13/01 RD	added command line options for code generation parameters
 * 11/08/01 RD	switched code generation to use GL_IO layer
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 05/25/01 RD	eliminated support for automatic IP wrapping (scc -ip option)
 * 05/25/01 RD	first version adjusted for SCRC
 */


#include "Compiler.h"
#include "CcDriver.h"
#include "IntRep.h"
#include "Parser.h"

#include <stdlib.h>
#include <strings.h>


/*** constants and macros ***********************************************/


#ifdef DEBUG
#ifndef SCC_DEBUG
#define SCC_DEBUG
#endif /* SCC_DEBUG */
#endif /* DEBUG */


#define SCC_SPARE_MEMORY_SIZE	256*1024	/* 256 KB */


/*** internal type declarations *****************************************/


	/* none */


/*** prototypes of internal functions ***********************************/


static const char *SCC_ErrorText(	/* returns message of error code */
	ERROR		ErrorNumber);

static void PrintHeader(		/* print program header */
	FILE		*File);

static void PrintUsage(			/* print usage info */
	FILE		*File);

static ERROR GetArguments(		/* parse command line arguments */
	int		argc,
	char		**argv);

static ERROR GetCommand(		/* parse the command argument */
	char		*SuffixCmd);


/*** exported variables *************************************************/


	/* none */


/*** internal variables *************************************************/


static ERROR	SCC_Error = SCC_ERROR_NO_ERROR;	/* last error of this module */
static gl_string ErrorMessage;


	/*** command line options and derivations (and their defaults) ***/

static gl_string	DesignName;	/* no default */

static gl_string	FilenameSCin;	/* these will be build */
static gl_string	FilenameSCout;	/* based on DesignName */
static gl_string	FilenameSI;
static gl_string	FilenameCC;
static gl_string	FilenameH;
static gl_string	FilenameO;
static gl_string	FilenameOUT;

static bool		CallPreprocessor	= true;	/* based on command */
static bool		CallParser		= true;
static bool		CallDeparser		= false;
static bool		CallTranslator		= true;
static bool		CallCompiler		= true;
static bool		CallLinker		= true;

static gl_string	*InputFilename  = &FilenameSCin;/* based on command */
static gl_string	*OutputFilename = &FilenameOUT;

static gl_stringlist	*PDirList = NULL,
			*PDirListInsert = NULL;
static gl_stringlist	*DDefList = NULL,
			*DDefListLast = NULL;
static gl_stringlist	*UDefList = NULL,
			*UDefListLast = NULL;
static gl_stringlist	*IDirList = NULL,
			*IDirListLast = NULL;
static gl_stringlist	*LDirList = NULL,
			*LDirListLast = NULL;
static gl_stringlist	*lLibList = NULL,
			*lLibListLast = NULL;

static bool		UsageInfoOnly	= false;
static int		VerbosityLevel	= 0;
static int		WarningLevel	= 0;
static bool		DebugMode	= false;
static bool		OptimizeMode	= false;
static bool		GenerateLines	= true;
static bool		GenerateNotes	= true;
static int		TabStep		= GL_TAB_STEP_DEFAULT;
static int		SysTabStep	= GL_TAB_STEP_SYSTEM;
static int		LineWrapping	= GL_LINE_WRAP_DEFAULT;

static bool		OptionIorOrequired = false;
static bool		GenerateH	= true;
static bool		GenerateCC	= true;


	/*** the design ***/

sir_design		*TheDesign = NULL;


/************************************************************************/
/*** implementation of exported functions			      ***/
/************************************************************************/


	/* none */


/************************************************************************/
/*** implementation of internal functions			      ***/
/************************************************************************/


	/*** error handling ***/


static const char *SCC_ErrorText(	/* returns message of error code */
	ERROR	ErrorNumber)
{

if (  (ErrorNumber >= GL_ERROR_BASE)
    &&(ErrorNumber <= GL_ERROR_BASE + GL_ERROR_RANGE))
   { return(GL_ErrorText(ErrorNumber));
    } /* fi */
if (  (ErrorNumber >= CCD_ERROR_BASE)
    &&(ErrorNumber <= CCD_ERROR_BASE + CCD_ERROR_RANGE))
   { return(CCD_ErrorText(ErrorNumber));
    } /* fi */
if (  (ErrorNumber >= SIR_ERROR_BASE)
    &&(ErrorNumber <= SIR_ERROR_BASE + SIR_ERROR_RANGE))
   { return(SIR_ErrorText(ErrorNumber));
    } /* fi */
if (  (ErrorNumber >= PRS_ERROR_BASE)
    &&(ErrorNumber <= PRS_ERROR_BASE + PRS_ERROR_RANGE))
   { return(PRS_ErrorText(ErrorNumber));
    } /* fi */

switch (ErrorNumber)
   { case SCC_ERROR_NO_ERROR:
	  return("No error (internal)");
     case SCC_ERROR_NO_DESIGN_NAME:
	  return("Bad arguments: design name missing");
     case SCC_ERROR_UNKNOWN_COMMAND:
	  return(ErrorMessage.chars()
		/* "Unknown command -%s" */);
     case SCC_ERROR_BAD_DESIGN_NAME:
	  return(ErrorMessage.chars()
		/* "'%s' is not a legal design name" */);
     case SCC_ERROR_BAD_ARGUMENT:
	  return(ErrorMessage.chars()
		/* "Bad argument '%s'" */);
     case SCC_ERROR_UNKNOWN_OPTION:
	  return(ErrorMessage.chars()
		/* "Unknown option %s" */);
     case SCC_ERROR_OPTION_NEEDS_ARGUMENT:
	  return(ErrorMessage.chars()
		/* "Option %s requires an argument" */);
     case SCC_ERROR_EMPTY_ARGUMENT:
	  return(ErrorMessage.chars()
		/* "Empty argument for option -%s" */);
     case SCC_ERROR_OUTPUT_FILENAME_AMBIGUOUS:
	  return("Specification of output filename is ambiguous");
     case SCC_ERROR_OPTION_IO_REQUIRED:
	  return("Explicit input or output filename required"
							GL_ERROR_MSG_NEWLINE
		"(use option -i or -o to avoid overriding the input file)");
     case SCC_ERROR_INVALID_VALUE_FOR_OPTION:
	  return(ErrorMessage.chars()
		/* "Invalid value (%d) for option -%s" */);
    } /* hctiws */

return("Unknown error number (internal)");

} /* end of SCC_ErrorText */


	/*** header and usage info ***/


static void PrintHeader(		/* print program header */
	FILE		*File)
{

fputs(	"scrc: SpecC Reference Compiler V " GL_SPECC_VERSION_STRING "\n"
	GL_SPECC_COPYRIGHT_STRING "\n\n",
		File);

} /* end of PrintHeader */


static void PrintUsage(			/* print usage info */
	FILE		*File)
{

fputs(
"Usage: scrc <design> [<command>] [<option>...]\n"
"   or: scrc -h  (print version and usage info and quit)\n"
"\n"
"Command:\n"
"-<suffix>2<suffix>  where <suffix> = sc|si|cc|h|cch|o|out\n"
"                    specifies the compilation steps to be performed;\n"
"e.g. -sc2out        compiles SpecC source code into an executable file\n"
"                    (this is the default command)\n"
"\n"
"Options:         Description:                   Default:\n"
"-v|-vv|-vvv      increase verbosity level       silent\n"
"-w|-ww|-www      increase warning level         no warnings\n"
"-g|-O            enable debugging/optimization  none\n"
"-sl              suppress gen. of line infos    enabled\n"
"-sn              suppress gen. of annotations   enabled\n", File);
fprintf(File,
"-st <tab step>   set tabulator stepping         %d\n", GL_TAB_STEP_DEFAULT);
fprintf(File,
"-sT <sys tab>    set system tabulator stepping  %d\n", GL_TAB_STEP_SYSTEM);
fprintf(File,
"-sw <line wrap>  set column for line wrapping   %d\n", GL_LINE_WRAP_DEFAULT);
fputs(
"-i <infile|->    specify input filename (%i)    <design>.<suffix>\n"
"-o <outfile|->   specify output filename (%o)   <design>.<suffix>\n"
"-D               do not define standard macros  " CCD_USAGE_DEFINES "\n"
"-D<macrodef>     define a preprocessor macro    none\n"
"-U               do not use standard undefines  " CCD_USAGE_UNDEFINES "\n"
"-U<undef>        undefine a preprocessor macro  none\n"
"-I               clear standard include path    " CCD_USAGE_STD_INCLUDE "\n"
"-I<dir>          insert dir in include path     none\n"
"-L               clear standard library path    " CCD_USAGE_STD_LIBRARY "\n"
"-L<dir>          insert dir in library path     none\n"
"-l               do not use standard libraries  " CCD_USAGE_STD_LIBRARIES "\n"
"-l<lib>          link against library lib       none\n"
"-P               clear standard import path     " CCD_SPECC_STD_IMPORT "\n"
"-P<dir>          insert dir in import path      none\n"
"-xpp <pp call>   override preprocessor call     " CCD_USAGE_PP_CALL "\n"
"-xcc <cc call>   override compiler call         " CCD_USAGE_CC_CALL "\n"
"-xld <ld call>   override linker call           " CCD_USAGE_LD_CALL "\n"
"-xp <pp option>  pass preprocessor opt. (%p)    " CCD_USAGE_PP_OPTS "\n"
"-xc <cc option>  pass compiler option (%c)      " CCD_USAGE_CC_OPTS "\n"
"-xl <ld option>  pass linker option (%l)        " CCD_USAGE_LD_OPTS "\n"
	"\n", File);

} /* end of PrintUsage */


	/*** command line argument handling ***/


static ERROR GetArguments(		/* parse command line arguments */
	int	argc,
	char	**argv)
{
gl_stringlist	*Succ,
		*TmpDirList;
static gl_string StdImportDir;
const char	*TmpStr;
ERROR		Error;
int		Arg;

if (argc < 2)
   { return(SCC_ERROR_NO_DESIGN_NAME);
    } /* fi */

if (  (argc == 2)
    &&(0 == strcmp("-h", argv[1])))
   { UsageInfoOnly = true;
     return(SCC_ERROR_NO_ERROR);
    } /* fi */

Arg = 1;
DesignName = argv[Arg];
if (GL_CheckIdentifier(DesignName))
   { GL_Reset();
     ErrorMessage.form("'%s' is not a legal design name", argv[Arg]);
     return(SCC_ERROR_BAD_DESIGN_NAME);
    } /* elihw */

FilenameSCin	= DesignName + ".sc";	/* default file names */
FilenameSCout	= DesignName + ".sc";
FilenameSI	= DesignName + ".si";
FilenameCC	= DesignName + ".cc";
FilenameH	= DesignName + ".h";
FilenameO	= DesignName + ".o";
#ifdef GNUWIN32
FilenameOUT	= DesignName + ".exe";
#else /* not GNUWIN32 */
FilenameOUT	= DesignName;
#endif /* GNUWIN32 */

assert(PDirList == NULL);		/* default import path */
assert(PDirListInsert == NULL);
PDirList = new gl_stringlist();
PDirList->Succ = NULL;
PDirList->String = CCD_SPECC_STD_IMPORT1;
if ((TmpStr = getenv(CCD_SPECC_ENV_VAR)))
   { StdImportDir = TmpStr;
     StdImportDir += "/";
     StdImportDir += CCD_SPECC_IMPORT_DIR;
     PDirList->Succ = new gl_stringlist();
     PDirList->Succ->Succ = NULL;
     PDirList->Succ->String = StdImportDir.chars();
    } /* fi */

Arg++;
if (Arg >= argc)
   { return(SCC_ERROR_NO_ERROR);	/* no more arguments */
    } /* fi */

if (argv[Arg][0] == '-')
   { if (index(&argv[Arg][1], '2'))	/* is this a command option? */
	{ if ((Error = GetCommand(&argv[Arg][1])))
	     { return(Error);
	      } /* fi */
	  Arg++;
	 } /* fi */
    } /* fi */

while(Arg < argc)			/* process all options */
   { if (argv[Arg][0] != '-')
	{ ErrorMessage.form("Bad argument '%s'", argv[Arg]);
	  return(SCC_ERROR_BAD_ARGUMENT);
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "v"))
	{ VerbosityLevel++;
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "vv"))
	{ VerbosityLevel += 2;
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "vvv"))
	{ VerbosityLevel += 3;
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "w"))
	{ WarningLevel++;
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "ww"))
	{ WarningLevel += 2;
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "www"))
	{ WarningLevel += 3;
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "g"))
	{ DebugMode = true;
	  if (OptimizeMode)
	     { GL_PrintWarning(GL_WARN_STANDARD,
				"Option -g disables option -O");
	       OptimizeMode = false;
	      } /* fi */
	  if ((Error = CCD_SetCompilerMode(CCD_MODE_DEBUG)))
	     { return(Error);
	      } /* fi */
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "O"))
	{ OptimizeMode = true;
	  if (DebugMode)
	     { GL_PrintWarning(GL_WARN_STANDARD,
				"Option -O disables option -g");
	       DebugMode = false;
	      } /* fi */
	  if ((Error = CCD_SetCompilerMode(CCD_MODE_OPTIMIZE)))
	     { return(Error);
	      } /* fi */
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "sl"))
	{ GenerateLines = false;
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "sn"))
	{ GenerateNotes = false;
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "st"))
	{ if (++Arg >= argc)
	     { ErrorMessage = "Option -st requires an integer argument";
	       return(SCC_ERROR_OPTION_NEEDS_ARGUMENT);
	      } /* fi */
	  if (argv[Arg][0] == 0)
	     { ErrorMessage = "Empty argument for option -st";
	       return(SCC_ERROR_EMPTY_ARGUMENT);
	      } /* fi */
	  TabStep = atoi(argv[Arg++]);
	  if (TabStep < 0)
	     { ErrorMessage.form("Invalid value (%d) for option -st",
							TabStep);
	       return(SCC_ERROR_INVALID_VALUE_FOR_OPTION);
	      } /* fi */
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "sT"))
	{ if (++Arg >= argc)
	     { ErrorMessage = "Option -sT requires an integer argument";
	       return(SCC_ERROR_OPTION_NEEDS_ARGUMENT);
	      } /* fi */
	  if (argv[Arg][0] == 0)
	     { ErrorMessage = "Empty argument for option -sT";
	       return(SCC_ERROR_EMPTY_ARGUMENT);
	      } /* fi */
	  SysTabStep = atoi(argv[Arg++]);
	  if (SysTabStep < 0)
	     { ErrorMessage.form("Invalid value (%d) for option -sT",
							SysTabStep);
	       return(SCC_ERROR_INVALID_VALUE_FOR_OPTION);
	      } /* fi */
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "sw"))
	{ if (++Arg >= argc)
	     { ErrorMessage = "Option -sw requires an integer argument";
	       return(SCC_ERROR_OPTION_NEEDS_ARGUMENT);
	      } /* fi */
	  if (argv[Arg][0] == 0)
	     { ErrorMessage = "Empty argument for option -sw";
	       return(SCC_ERROR_EMPTY_ARGUMENT);
	      } /* fi */
	  LineWrapping = atoi(argv[Arg++]);
	  if (LineWrapping < 0)
	     { ErrorMessage.form("Invalid value (%d) for option -sw",
							LineWrapping);
	       return(SCC_ERROR_INVALID_VALUE_FOR_OPTION);
	      } /* fi */
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "i"))
	{ if (++Arg >= argc)
	     { ErrorMessage = "Option -i requires an argument";
	       return(SCC_ERROR_OPTION_NEEDS_ARGUMENT);
	      } /* fi */
	  if (argv[Arg][0] == 0)
	     { ErrorMessage = "Empty argument for option -i";
	       return(SCC_ERROR_EMPTY_ARGUMENT);
	      } /* fi */
	  *InputFilename = argv[Arg++];
	  if ((SCC_Error = GL_CheckPathFilename(*InputFilename)))
	     { return(SCC_Error);
	      } /* fi */
	  if (OptionIorOrequired)
	     { OptionIorOrequired = false;
	      } /* fi */
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "o"))
	{ if (++Arg >= argc)
	     { ErrorMessage = "Option -o requires an argument";
	       return(SCC_ERROR_OPTION_NEEDS_ARGUMENT);
	      } /* fi */
	  if (! OutputFilename)
	     { return(SCC_ERROR_OUTPUT_FILENAME_AMBIGUOUS);
	      } /* fi */
	  if (argv[Arg][0] == 0)
	     { ErrorMessage = "Empty argument for option -o";
	       return(SCC_ERROR_EMPTY_ARGUMENT);
	      } /* fi */
	  *OutputFilename = argv[Arg++];
	  if ((SCC_Error = GL_CheckPathFilename(*OutputFilename)))
	     { return(SCC_Error);
	      } /* fi */
	  if (OptionIorOrequired)
	     { OptionIorOrequired = false;
	      } /* fi */
	  continue;
	 } /* fi */
     if (argv[Arg][1] == 'D')
	{ if (argv[Arg][2] == 0)
	     { while(DDefList)
		  { Succ = DDefList->Succ;
		    delete DDefList;
		    DDefList = Succ;
		   } /* eliwh */
	       DDefList = NULL;	/* empty definition list */
	       DDefListLast = NULL;
	       if ((SCC_Error = CCD_NoStdMacroDefs()))
		  { return(SCC_Error);
		   } /* fi */
	      } /* fi */
	  else
	     { if (DDefListLast)
		  { DDefListLast->Succ = new gl_stringlist();
		    DDefListLast = DDefListLast->Succ;
		   } /* fi */
	       else
		  { assert(DDefList == NULL);
		    DDefList = new gl_stringlist();
		    DDefListLast = DDefList;
		   } /* esle */
	       DDefListLast->Succ = NULL;
	       DDefListLast->String = &argv[Arg][2];
	      } /* esle */
	  Arg++;
	  continue;
	 } /* fi */
     if (argv[Arg][1] == 'U')
	{ if (argv[Arg][2] == 0)
	     { while(UDefList)
		  { Succ = UDefList->Succ;
		    delete UDefList;
		    UDefList = Succ;
		   } /* eliwh */
	       UDefList = NULL;	/* empty un-definition list */
	       UDefListLast = NULL;
	       if ((SCC_Error = CCD_NoStdUnDefs()))
		  { return(SCC_Error);
		   } /* fi */
	      } /* fi */
	  else
	     { if (UDefListLast)
		  { UDefListLast->Succ = new gl_stringlist();
		    UDefListLast = UDefListLast->Succ;
		   } /* fi */
	       else
		  { assert(UDefList == NULL);
		    UDefList = new gl_stringlist();
		    UDefListLast = UDefList;
		   } /* esle */
	       UDefListLast->Succ = NULL;
	       UDefListLast->String = &argv[Arg][2];
	      } /* esle */
	  Arg++;
	  continue;
	 } /* fi */
     if (argv[Arg][1] == 'I')
	{ if (argv[Arg][2] == 0)
	     { while(IDirList)
		  { Succ = IDirList->Succ;
		    delete IDirList;
		    IDirList = Succ;
		   } /* eliwh */
	       IDirList = NULL;	/* empty include directory list */
	       IDirListLast = NULL;
	       if ((SCC_Error = CCD_NoStdIncludePath()))
		  { return(SCC_Error);
		   } /* fi */
	      } /* fi */
	  else
	     { if (IDirListLast)
		  { IDirListLast->Succ = new gl_stringlist();
		    IDirListLast = IDirListLast->Succ;
		   } /* fi */
	       else
		  { assert(IDirList == NULL);
		    IDirList = new gl_stringlist();
		    IDirListLast = IDirList;
		   } /* esle */
	       IDirListLast->Succ = NULL;
	       IDirListLast->String = &argv[Arg][2];
	      } /* esle */
	  Arg++;
	  continue;
	 } /* fi */
     if (argv[Arg][1] == 'L')
	{ if (argv[Arg][2] == 0)
	     { while(LDirList)
		  { Succ = LDirList->Succ;
		    delete LDirList;
		    LDirList = Succ;
		   } /* eliwh */
	       LDirList = NULL;	/* empty library directory list */
	       LDirListLast = NULL;
	       if ((SCC_Error = CCD_NoStdLibraryPath()))
		  { return(SCC_Error);
		   } /* fi */
	      } /* fi */
	  else
	     { if (LDirListLast)
		  { LDirListLast->Succ = new gl_stringlist();
		    LDirListLast = LDirListLast->Succ;
		   } /* fi */
	       else
		  { assert(LDirList == NULL);
		    LDirList = new gl_stringlist();
		    LDirListLast = LDirList;
		   } /* esle */
	       LDirListLast->Succ = NULL;
	       LDirListLast->String = &argv[Arg][2];
	      } /* esle */
	  Arg++;
	  continue;
	 } /* fi */
     if (argv[Arg][1] == 'l')
	{ if (argv[Arg][2] == 0)
	     { while(lLibList)
		  { Succ = lLibList->Succ;
		    delete lLibList;
		    lLibList = Succ;
		   } /* eliwh */
	       lLibList = NULL;	/* empty libraries list */
	       lLibListLast = NULL;
	       if ((SCC_Error = CCD_NoStdLibraries()))
		  { return(SCC_Error);
		   } /* fi */
	      } /* fi */
	  else
	     { if (lLibListLast)
		  { lLibListLast->Succ = new gl_stringlist();
		    lLibListLast = lLibListLast->Succ;
		   } /* fi */
	       else
		  { assert(lLibList == NULL);
		    lLibList = new gl_stringlist();
		    lLibListLast = lLibList;
		   } /* esle */
	       lLibListLast->Succ = NULL;
	       lLibListLast->String = &argv[Arg][2];
	      } /* esle */
	  Arg++;
	  continue;
	 } /* fi */
     if (argv[Arg][1] == 'P')
	{ if (argv[Arg][2] == 0)
	     { while(PDirList)
		  { Succ = PDirList->Succ;
		    delete PDirList;
		    PDirList = Succ;
		   } /* eliwh */
	       PDirList = NULL;	/* empty import dir list */
	       PDirListInsert = NULL;
	      } /* fi */
	  else
	     { TmpDirList = new gl_stringlist();
	       TmpDirList->Succ = NULL;
	       TmpDirList->String = &argv[Arg][2];
	       if (PDirList)
		  { if (PDirListInsert)
		       { TmpDirList->Succ = PDirListInsert->Succ;
			 PDirListInsert->Succ = TmpDirList;
			 PDirListInsert = TmpDirList;
			} /* fi */
		    else
		       { TmpDirList->Succ = PDirList;
			 PDirList = TmpDirList;
			 PDirListInsert = TmpDirList;
			} /* esle */
		   } /* fi */
	       else
		  { PDirList = TmpDirList;
		    assert(PDirListInsert == NULL);
		   } /* esle */
	      } /* esle */
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "xpp"))
	{ if (++Arg >= argc)
	     { ErrorMessage = "Option -xpp requires an argument";
	       return(SCC_ERROR_OPTION_NEEDS_ARGUMENT);
	      } /* fi */
	  if (! CallPreprocessor)
	     { GL_PrintWarning(GL_WARN_STANDARD,
			"Option -xpp ignored because preprocessing not done");
	      } /* fi */
	  else
	     { if (argv[Arg][0] == 0)
		  { ErrorMessage = "Empty argument for option -xpp";
		    return(SCC_ERROR_EMPTY_ARGUMENT);
		   } /* fi */
	       if ((Error = CCD_SetPreprocessorCall(argv[Arg])))
		  { return(Error);
		   } /* fi */
	      } /* esle */
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "xcc"))
	{ if (++Arg >= argc)
	     { ErrorMessage = "Option -xcc requires an argument";
	       return(SCC_ERROR_OPTION_NEEDS_ARGUMENT);
	      } /* fi */
	  if (! CallCompiler)
	     { GL_PrintWarning(GL_WARN_STANDARD,
			"Option -xcc ignored because compiling not done");
	      } /* fi */
	  else
	     { if (argv[Arg][0] == 0)
		  { ErrorMessage = "Empty argument for option -xcc";
		    return(SCC_ERROR_EMPTY_ARGUMENT);
		   } /* fi */
	       if ((Error = CCD_SetCompilerCall(argv[Arg])))
		  { return(Error);
		   } /* fi */
	      } /* esle */
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "xld"))
	{ if (++Arg >= argc)
	     { ErrorMessage = "Option -xld requires an argument";
	       return(SCC_ERROR_OPTION_NEEDS_ARGUMENT);
	      } /* fi */
	  if (! CallLinker)
	     { GL_PrintWarning(GL_WARN_STANDARD,
			"Option -xld ignored because linking not done");
	      } /* fi */
	  else
	     { if (argv[Arg][0] == 0)
		  { ErrorMessage = "Empty argument for option -xld";
		    return(SCC_ERROR_EMPTY_ARGUMENT);
		   } /* fi */
	       if ((Error = CCD_SetLinkerCall(argv[Arg])))
		  { return(Error);
		   } /* fi */
	      } /* esle */
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "xp"))
	{ if (++Arg >= argc)
	     { ErrorMessage = "Option -xp requires an argument";
	       return(SCC_ERROR_OPTION_NEEDS_ARGUMENT);
	      } /* fi */
	  if (! CallPreprocessor)
	     { GL_PrintWarning(GL_WARN_STANDARD,
			"Option -xp ignored because preprocessing not done");
	      } /* fi */
	  else
	     { if ((Error = CCD_SetPreprocessorOption(argv[Arg])))
		  { return(Error);
		   } /* fi */
	      } /* esle */
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "xc"))
	{ if (++Arg >= argc)
	     { ErrorMessage = "Option -xc requires an argument";
	       return(SCC_ERROR_OPTION_NEEDS_ARGUMENT);
	      } /* fi */
	  if (! CallCompiler)
	     { GL_PrintWarning(GL_WARN_STANDARD,
			"Option -xc ignored because compiling not done");
	      } /* fi */
	  else
	     { if ((Error = CCD_SetCompilerOption(argv[Arg])))
		  { return(Error);
		   } /* fi */
	      } /* esle */
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "xl"))
	{ if (++Arg >= argc)
	     { ErrorMessage = "Option -xl requires an argument";
	       return(SCC_ERROR_OPTION_NEEDS_ARGUMENT);
	      } /* fi */
	  if (! CallLinker)
	     { GL_PrintWarning(GL_WARN_STANDARD,
			"Option -xl ignored because linking not done");
	      } /* fi */
	  else
	     { if ((Error = CCD_SetLinkerOption(argv[Arg])))
		  { return(Error);
		   } /* fi */
	      } /* esle */
	  Arg++;
	  continue;
	 } /* fi */

     ErrorMessage.form("Unknown option %s", argv[Arg]);
     return(SCC_ERROR_UNKNOWN_OPTION);
    } /* elihw */

if (OptionIorOrequired)
   { return(SCC_ERROR_OPTION_IO_REQUIRED);
    } /* fi */

if ((Error = CCD_SetMacroDefs(DDefList)))	/* set lists in driver */
   { return(Error);
    } /* fi */
if ((Error = CCD_SetUnDefs(UDefList)))
   { return(Error);
    } /* fi */
if ((Error = CCD_SetIncludePath(IDirList)))
   { return(Error);
    } /* fi */
if ((Error = CCD_SetLibraryPath(LDirList)))
   { return(Error);
    } /* fi */
if ((Error = CCD_SetLibraries(lLibList)))
   { return(Error);
    } /* fi */

return(SCC_ERROR_NO_ERROR);	/* all arguments are ok */

} /* end of GetArguments */


static ERROR GetCommand(		/* parse the command argument */
	char	*SuffixCmd)
{

	/* ok, first we assume doing nothing */
CallPreprocessor	= false;
CallParser		= false;
CallDeparser		= false;
CallTranslator		= false;
CallCompiler		= false;
CallLinker		= false;

if (0 == strcmp(SuffixCmd, "sc2si"))	/* start from a .sc file */
   { CallPreprocessor	= true;
     InputFilename  = &FilenameSCin;
     OutputFilename = &FilenameSI;
     return(SCC_ERROR_NO_ERROR);
    } /* fi */
if (0 == strcmp(SuffixCmd, "sc2cc"))
   { CallPreprocessor	= true;
     CallParser		= true;
     CallTranslator	= true;
     InputFilename  = &FilenameSCin;
     OutputFilename = &FilenameCC;
     GenerateH		= false;	/* no .h file */
     return(SCC_ERROR_NO_ERROR);
    } /* fi */
if (0 == strcmp(SuffixCmd, "sc2h"))
   { CallPreprocessor	= true;
     CallParser		= true;
     CallTranslator	= true;
     InputFilename  = &FilenameSCin;
     OutputFilename = &FilenameH;
     GenerateCC		= false;	/* no .cc file */
     return(SCC_ERROR_NO_ERROR);
    } /* fi */
if (0 == strcmp(SuffixCmd, "sc2cch"))
   { CallPreprocessor	= true;
     CallParser		= true;
     CallTranslator	= true;
     InputFilename  = &FilenameSCin;
     OutputFilename = NULL;	/* no -o option allowed */
     return(SCC_ERROR_NO_ERROR);
    } /* fi */
if (0 == strcmp(SuffixCmd, "sc2o"))
   { CallPreprocessor	= true;
     CallParser		= true;
     CallTranslator	= true;
     CallCompiler	= true;
     InputFilename  = &FilenameSCin;
     OutputFilename = &FilenameO;
     return(SCC_ERROR_NO_ERROR);
    } /* fi */
if (0 == strcmp(SuffixCmd, "sc2out"))
   { CallPreprocessor	= true;
     CallParser		= true;
     CallTranslator	= true;
     CallCompiler	= true;
     CallLinker		= true;
     InputFilename  = &FilenameSCin;
     OutputFilename = &FilenameOUT;
     return(SCC_ERROR_NO_ERROR);
    } /* fi */
if (0 == strcmp(SuffixCmd, "sc2sc"))
   { OptionIorOrequired = true;
     CallPreprocessor	= true;
     CallParser		= true;
     CallDeparser	= true;
     InputFilename  = &FilenameSCin;
     OutputFilename = &FilenameSCout;
     return(SCC_ERROR_NO_ERROR);
    } /* fi */
if (0 == strcmp(SuffixCmd, "si2cc"))
   { CallParser		= true;
     CallTranslator	= true;
     InputFilename  = &FilenameSI;
     OutputFilename = &FilenameCC;
     GenerateH		= false;	/* no .h file */
     return(SCC_ERROR_NO_ERROR);
    } /* fi */
if (0 == strcmp(SuffixCmd, "si2h"))
   { CallParser		= true;
     CallTranslator	= true;
     InputFilename  = &FilenameSI;
     OutputFilename = &FilenameH;
     GenerateCC		= false;	/* no .cc file */
     return(SCC_ERROR_NO_ERROR);
    } /* fi */
if (0 == strcmp(SuffixCmd, "si2cch"))
   { CallParser		= true;
     CallTranslator	= true;
     InputFilename  = &FilenameSI;
     OutputFilename = NULL;	/* no -o option allowed */
     return(SCC_ERROR_NO_ERROR);
    } /* fi */
if (0 == strcmp(SuffixCmd, "si2o"))
   { CallParser		= true;
     CallTranslator	= true;
     CallCompiler	= true;
     InputFilename  = &FilenameSI;
     OutputFilename = &FilenameO;
     return(SCC_ERROR_NO_ERROR);
    } /* fi */
if (0 == strcmp(SuffixCmd, "si2out"))
   { CallParser		= true;
     CallTranslator	= true;
     CallCompiler	= true;
     CallLinker		= true;
     InputFilename  = &FilenameSI;
     OutputFilename = &FilenameOUT;
     return(SCC_ERROR_NO_ERROR);
    } /* fi */
if (0 == strcmp(SuffixCmd, "si2sc"))
   { CallParser		= true;
     CallDeparser	= true;
     InputFilename  = &FilenameSI;
     OutputFilename = &FilenameSCout;
     return(SCC_ERROR_NO_ERROR);
    } /* fi */

if (0 == strcmp(SuffixCmd, "cc2o"))	/* start from a .cc file */
   { CallCompiler	= true;
     InputFilename  = &FilenameCC;
     OutputFilename = &FilenameO;
     return(SCC_ERROR_NO_ERROR);
    } /* fi */
if (0 == strcmp(SuffixCmd, "cc2out"))
   { CallCompiler	= true;
     CallLinker		= true;
     InputFilename  = &FilenameCC;
     OutputFilename = &FilenameOUT;
     return(SCC_ERROR_NO_ERROR);
    } /* fi */

if (0 == strcmp(SuffixCmd, "o2out"))	/* start from a .o file */
   { CallLinker		= true;
     InputFilename  = &FilenameO;
     OutputFilename = &FilenameOUT;
     return(SCC_ERROR_NO_ERROR);
    } /* fi */

ErrorMessage = "Unknown command -";
ErrorMessage += SuffixCmd;
return(SCC_ERROR_UNKNOWN_COMMAND);

} /* end of GetCommand */


	/*** clean up ***/


void CleanUp(void)
{
gl_stringlist	*Succ;

while(PDirList)			/* clean up the import path */
   { Succ = PDirList->Succ;
     delete PDirList;
     PDirList = Succ;
    } /* elihw */
PDirListInsert = NULL;

while(DDefList)			/* clean up the macro definitions */
   { Succ = DDefList->Succ;
     delete DDefList;
     DDefList = Succ;
    } /* elihw */
DDefListLast = NULL;

while(UDefList)			/* clean up the macro un-definitions */
   { Succ = UDefList->Succ;
     delete UDefList;
     UDefList = Succ;
    } /* elihw */
UDefListLast = NULL;

while(IDirList)			/* clean up the include path */
   { Succ = IDirList->Succ;
     delete IDirList;
     IDirList = Succ;
    } /* elihw */
IDirListLast = NULL;

while(LDirList)			/* clean up the library path */
   { Succ = LDirList->Succ;
     delete LDirList;
     LDirList = Succ;
    } /* elihw */
LDirListLast = NULL;

while(lLibList)			/* clean up the library list */
   { Succ = lLibList->Succ;
     delete lLibList;
     lLibList = Succ;
    } /* elihw */
lLibListLast = NULL;

delete TheDesign;		/* delete the design data structure */

CCD_SetDefaultOptions();	/* clean up compiler driver */

GL_FreeSpareMemory();

} /* end of CleanUp */


/************************************************************************/
/*** main							      ***/
/************************************************************************/


int main(int argc, char **argv)
{
int		i;
ERROR		Error;
FILE		*InFile;
gl_io		*IO;
gl_string	Command;
#ifdef SCC_DEBUG
gl_stringlist	*StringPtr;
#endif /* SCC_DEBUG */

	/* initialization */

CCD_SetDefaultOptions();	/* initialize CCD module options */

	/* parse the arguments */

SCC_Error = GetArguments(argc, argv);
if (UsageInfoOnly)
   { PrintHeader(stdout);
     PrintUsage(stdout);
     CleanUp();
     return(RESULT_OK);
    } /* fi */

if ((VerbosityLevel >= 1) || (SCC_Error))
   { PrintHeader(stderr);
    } /* fi */

if ((VerbosityLevel >= 3) || (SCC_Error))
   { fprintf(stderr, "      %s", argv[0]);
     for(i=1; i<argc; i++)
	{ if (index(argv[i], ' '))
	     { fprintf(stderr, " \"%s\"", argv[i]);
	      } /* fi */
	  else
	     { fprintf(stderr, " %s", argv[i]);
	      } /* esle */
	 } /* rof */
     fputs("\n\n", stderr);
    } /* fi */

#ifdef SCC_DEBUG	/* dump the evaluated arguments and options */

fprintf(stderr, "SCC_DEBUG: DesignName     = \"%s\"\n", DesignName.chars());
fprintf(stderr, "SCC_DEBUG: FilenameSCin   = \"%s\"\n", FilenameSCin.chars());
fprintf(stderr, "SCC_DEBUG: FilenameSCout  = \"%s\"\n", FilenameSCout.chars());
fprintf(stderr, "SCC_DEBUG: FilenameSI     = \"%s\"\n", FilenameSI.chars());
fprintf(stderr, "SCC_DEBUG: FilenameCC     = \"%s\"\n", FilenameCC.chars());
fprintf(stderr, "SCC_DEBUG: FilenameH      = \"%s\"\n", FilenameH.chars());
fprintf(stderr, "SCC_DEBUG: FilenameO      = \"%s\"\n", FilenameO.chars());
fprintf(stderr, "SCC_DEBUG: FilenameOUT    = \"%s\"\n", FilenameOUT.chars());
fprintf(stderr, "SCC_DEBUG: Tasks are %s%s%s%s%s%s\b\b.\n",
	(CallPreprocessor ? "Preprocessor, " : ""),
	(CallParser ? "Parser, " : ""),
	(CallDeparser ? "Deparser, " : ""),
	(CallTranslator ? "Translator, " : ""),
	(CallCompiler ? "Compiler, " : ""),
	(CallLinker ? "Linker, " : ""));
fprintf(stderr, "SCC_DEBUG: VerbosityLevel = %d\n", VerbosityLevel);
fprintf(stderr, "SCC_DEBUG: WarningLevel   = %d\n", WarningLevel);
fprintf(stderr, "SCC_DEBUG: DebugMode      = %d\n", DebugMode);
fprintf(stderr, "SCC_DEBUG: OptimizeMode   = %d\n", OptimizeMode);
fprintf(stderr, "SCC_DEBUG: GenerateLines  = %d\n", GenerateLines);
fprintf(stderr, "SCC_DEBUG: GenerateNotes  = %d\n", GenerateNotes);
fprintf(stderr, "SCC_DEBUG: TabStep        = %d\n", TabStep);
fprintf(stderr, "SCC_DEBUG: SysTabStep     = %d\n", SysTabStep);
fprintf(stderr, "SCC_DEBUG: LineWrapping   = %d\n", LineWrapping);
fprintf(stderr, "SCC_DEBUG: GenerateH      = %d\n", GenerateH);
fprintf(stderr, "SCC_DEBUG: GenerateCC     = %d\n", GenerateCC);
fprintf(stderr, "SCC_DEBUG: ImportPath     = ");
StringPtr = PDirList;
while(StringPtr)
   { fprintf(stderr, "\"%s\"%s",
		StringPtr->String,
		(StringPtr->Succ ? ", " : ""));
     StringPtr = StringPtr->Succ;
    } /* elihw */
fputs("\n", stderr);
fprintf(stderr, "SCC_DEBUG: IncludePath    = ");
StringPtr = IDirList;
while(StringPtr)
   { fprintf(stderr, "\"%s\"%s",
		StringPtr->String,
		(StringPtr->Succ ? ", " : ""));
     StringPtr = StringPtr->Succ;
    } /* elihw */
fputs("\n", stderr);
fprintf(stderr, "SCC_DEBUG: LibraryPath    = ");
StringPtr = LDirList;
while(StringPtr)
   { fprintf(stderr, "\"%s\"%s",
		StringPtr->String,
		(StringPtr->Succ ? ", " : ""));
     StringPtr = StringPtr->Succ;
    } /* elihw */
fputs("\n", stderr);
fprintf(stderr, "SCC_DEBUG: LibraryList    = ");
StringPtr = lLibList;
while(StringPtr)
   { fprintf(stderr, "\"%s\"%s",
		StringPtr->String,
		(StringPtr->Succ ? ", " : ""));
     StringPtr = StringPtr->Succ;
    } /* elihw */
fputs("\n", stderr);
#endif /* SCC_DEBUG */

if (SCC_Error)
   { GL_PrintError(SCC_Error, SCC_ErrorText(SCC_Error));
     PrintUsage(stderr);
     fputs("Aborted.\n", stderr);
     CleanUp();
     return(RESULT_ERROR);
    } /* fi */

GL_VerbosityLevel = VerbosityLevel;	/* publish the VerbosityLevel */
GL_WarningLevel = WarningLevel;		/* and the WarningLevel */

	/* enable smart memory management */

if ((Error = GL_ReserveSpareMemory(SCC_SPARE_MEMORY_SIZE)))
   { GL_PrintError(Error, GL_ErrorText(Error));
     fputs("Aborted.\n", stderr);
     CleanUp();
     return(RESULT_ERROR);
    } /* fi */


	/* call the preprocessor */

if (CallPreprocessor)
   { if (VerbosityLevel >= 1)
	{ fputs("Preprocessing...\n", stderr);
	 } /* fi */
     if (VerbosityLevel >= 2)
	{ fprintf(stderr, "   Input:  \"%s\"\n", FilenameSCin.chars());
	  fprintf(stderr, "   Output: \"%s\"\n", FilenameSI.chars());
	 } /* fi */
     if ((Error = CCD_Preprocessor(FilenameSCin.chars(),
					FilenameSI.chars())))
	{ GL_PrintError(Error, CCD_ErrorText(Error));
	  fputs("Aborted.\n", stderr);
	  CleanUp();
	  return(RESULT_ERROR);
	 } /* fi */
    } /* fi */

	/* call the parser */

if (CallParser)
   { if (VerbosityLevel >= 1)
	{ fputs("Parsing...\n", stderr);
	 } /* fi */
     if (VerbosityLevel >= 2)
	{ fprintf(stderr, "   Input:  \"%s\"\n", FilenameSI.chars());
	  fprintf(stderr, "   Output: <internal representation>\n");
	 } /* fi */
     if (!(InFile = GL_OpenFileForReading(FilenameSI.chars())))
	{ GL_PrintError(GL_Error, GL_ErrorText(GL_Error));
	  fputs("Aborted.\n", stderr);
	  CleanUp();
	  return(RESULT_ERROR);
	 } /* fi */
     assert(TheDesign == NULL);
     if (!(TheDesign = PRS_ParseDesign(InFile, DesignName.chars(),
				FilenameSI.chars(), PDirList)))
	{ GL_PrintError(PRS_Error, PRS_ErrorText(PRS_Error));
	  fputs("Aborted.\n", stderr);
	  GL_CloseFile(InFile);
	  CleanUp();
	  return(RESULT_ERROR);
	 } /* fi */
     if ((Error = GL_CloseFileWithCheck(InFile)))
	{ GL_PrintError(Error, GL_ErrorText(Error));
	  fputs("Aborted.\n", stderr);
	  CleanUp();
	  return(RESULT_ERROR);
	 } /* fi */
    } /* fi */

	/* call the translator */

if (CallTranslator)
   { if (VerbosityLevel >= 1)
	{ fputs("Translating...\n", stderr);
	 } /* fi */
     if (VerbosityLevel >= 2)
	{ fprintf(stderr, "   Input:  <internal representation>\n");
	 } /* fi */
     if (GenerateH)
	{ if (VerbosityLevel >= 2)
	     { fprintf(stderr, "   Output: \"%s\"\n", FilenameH.chars());
	      } /* fi */
	  if (!(IO = GL_IO::OpenFile(FilenameH.chars(), GL_IO_WRITE)))
	     { GL_PrintError(GL_Error, GL_ErrorText(GL_Error));
	       fputs("Aborted.\n", stderr);
	       CleanUp();
	       return(RESULT_ERROR);
	      } /* fi */
	  IO->SetTabStep(TabStep);
	  assert(GL_Error == GL_ERROR_NO_ERROR);
	  IO->SetSystemTabStep(SysTabStep);
	  assert(GL_Error == GL_ERROR_NO_ERROR);
	  IO->SetLineWrap(LineWrapping);
	  assert(GL_Error == GL_ERROR_NO_ERROR);
	  assert(TheDesign != NULL);
	  if ((Error = TheDesign->WriteH(IO,
			FilenameH.chars(),
			GenerateNotes, GenerateLines)))
	     { GL_PrintError(Error, SIR_ErrorText(Error));
	       fputs("Aborted.\n", stderr);
	       IO->Close();
	       CleanUp();
	       return(RESULT_ERROR);
	      } /* fi */
	  if ((Error = IO->CloseWithCheck()))
	     { GL_PrintError(Error, GL_ErrorText(Error));
	       fputs("Aborted.\n", stderr);
	       CleanUp();
	       return(RESULT_ERROR);
	      } /* fi */
	 } /* fi */
     if (GenerateCC)
	{ if (VerbosityLevel >= 2)
	     { fprintf(stderr, "   Output: \"%s\"\n", FilenameCC.chars());
	      } /* fi */
	  if (!(IO = GL_IO::OpenFile(FilenameCC.chars(), GL_IO_WRITE)))
	     { GL_PrintError(GL_Error, GL_ErrorText(GL_Error));
	       fputs("Aborted.\n", stderr);
	       CleanUp();
	       return(RESULT_ERROR);
	      } /* fi */
	  IO->SetTabStep(TabStep);
	  assert(GL_Error == GL_ERROR_NO_ERROR);
	  IO->SetSystemTabStep(SysTabStep);
	  assert(GL_Error == GL_ERROR_NO_ERROR);
	  IO->SetLineWrap(LineWrapping);
	  assert(GL_Error == GL_ERROR_NO_ERROR);
	  assert(TheDesign != NULL);
	  if ((Error = TheDesign->WriteCC(IO,
			FilenameCC.chars(), FilenameH.chars(),
			GenerateNotes, GenerateLines)))
	     { GL_PrintError(Error, SIR_ErrorText(Error));
	       fputs("Aborted.\n", stderr);
	       IO->Close();
	       CleanUp();
	       return(RESULT_ERROR);
	      } /* fi */
	  if ((Error = IO->CloseWithCheck()))
	     { GL_PrintError(Error, GL_ErrorText(Error));
	       fputs("Aborted.\n", stderr);
	       CleanUp();
	       return(RESULT_ERROR);
	      } /* fi */
	 } /* fi */
    } /* fi */

	/* call the compiler */

if (CallCompiler)
   { if (VerbosityLevel >= 1)
	{ fputs("Compiling...\n", stderr);
	 } /* fi */
     if (VerbosityLevel >= 2)
	{ fprintf(stderr, "   Input:  \"%s\"\n", FilenameCC.chars());
	  fprintf(stderr, "   Output: \"%s\"\n", FilenameO.chars());
	 } /* fi */
     if ((Error = CCD_Compiler(FilenameCC.chars(), FilenameO.chars())))
	{ GL_PrintError(Error, CCD_ErrorText(Error));
	  fputs("Aborted.\n", stderr);
	  CleanUp();
	  return(RESULT_ERROR);
	 } /* fi */
    } /* fi */

	/* call the linker */

if (CallLinker)
   { if (VerbosityLevel >= 1)
	{ fputs("Linking...\n", stderr);
	 } /* fi */
     if (VerbosityLevel >= 2)
	{ fprintf(stderr, "   Input:  \"%s\"\n", FilenameO.chars());
	  fprintf(stderr, "   Output: \"%s\"\n", FilenameOUT.chars());
	 } /* fi */
     if ((Error = CCD_Linker(FilenameO.chars(), FilenameOUT.chars())))
	{ GL_PrintError(Error, CCD_ErrorText(Error));
	  fputs("Aborted.\n", stderr);
	  CleanUp();
	  return(RESULT_ERROR);
	 } /* fi */
    } /* fi */

	/* call the deparser */

if (CallDeparser)
   { if (VerbosityLevel >= 1)
	{ fputs("Deparsing...\n", stderr);
	 } /* fi */
     if (VerbosityLevel >= 2)
	{ fprintf(stderr, "   Input:  <internal representation>\n");
	  fprintf(stderr, "   Output: \"%s\"\n", FilenameSCout.chars());
	 } /* fi */
     if (!(IO = GL_IO::OpenFile(FilenameSCout.chars(), GL_IO_WRITE)))
	{ GL_PrintError(GL_Error, GL_ErrorText(GL_Error));
	  fputs("Aborted.\n", stderr);
	  CleanUp();
	  return(RESULT_ERROR);
	 } /* fi */
     IO->SetTabStep(TabStep);
     assert(GL_Error == GL_ERROR_NO_ERROR);
     IO->SetSystemTabStep(SysTabStep);
     assert(GL_Error == GL_ERROR_NO_ERROR);
     IO->SetLineWrap(LineWrapping);
     assert(GL_Error == GL_ERROR_NO_ERROR);
     assert(TheDesign != NULL);
     if ((Error = TheDesign->WriteSC(IO,
			FilenameSCout.chars(),
			GenerateNotes, GenerateLines)))
	{ GL_PrintError(Error, SIR_ErrorText(Error));
	  fputs("Aborted.\n", stderr);
	  IO->Close();
	  CleanUp();
	  return(RESULT_ERROR);
	 } /* fi */
     if ((Error = IO->CloseWithCheck()))
	{ GL_PrintError(Error, GL_ErrorText(Error));
	  fputs("Aborted.\n", stderr);
	  CleanUp();
	  return(RESULT_ERROR);
	 } /* fi */
    } /* fi */

if (VerbosityLevel >= 1)
   { fputs("Done.\n", stderr);
    } /* fi */

CleanUp();

return(RESULT_OK);	/* success! */

} /* end of main */


/* EOF Compiler.c */
