// semaphore.sc: example for use of c_semaphore
//
// author: Rainer Doemer
//
// modifications: (most recent first)
//
// 02/13/02 RD	initial version (based on semaphore_test.sc)


#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <sim.sh>

import "c_semaphore";	// import the standard channel

const unsigned long N = 3;	// number of resources available


int Rnd(int Range)		// random number in range [0...Range-1]
{
    return((int)(drand48() * (double)Range));
}


interface i_monitor
{
    int ObtainResource(int User);
    void ReleaseResource(int Resource);
};

channel monitor(		// monitor to protect resource allocation table
    inout int   ResourceUser[N]) implements i_monitor
{
    int ObtainResource(int User)
    {
	unsigned int i;

	for(i=0; i<N; i++)
	{
	    if (ResourceUser[i] == 0)
	    {
		ResourceUser[i] = User;		// obtain it
		return(i);
	    }
	}
	assert(false);	// no resource left!
	return(-1);
    }

    void ReleaseResource(int Resource)
    {
	assert(ResourceUser[Resource] != 0);	// somebody uses it
	ResourceUser[Resource] = 0;
    }
};


behavior Competitor(		// competitor for resources
    i_semaphore Semaphore,
    inout int   SharedResource[N],
    in    int   ResourceUser[N],
    i_monitor   ResourceAlloc,
    in    int   ID)
{
    void UseResource(int Resource, int User, int Duration)
    {
	assert(ResourceUser[Resource] == User);	// nobody else must use this
	SharedResource[Resource]++;
	printf("Time%4s: Competitor %d using resource %d"
		" for %d time units: data = %d.\n",
		time2str(now()), User, Resource, Duration,
		SharedResource[Resource]);
	if (Duration)
	    waitfor(Duration);
	assert(ResourceUser[Resource] == User);	// nobody else must use this
    }

    void ObtainAndUseResource(void)
    {
	int Duration;
	int Resource;

	printf("Time%4s: Competitor %d trying to use a resource.\n",
		time2str(now()), ID);
	if (Semaphore.attempt())
	{
	    printf("Time%4s: Attempt immediately successful!\n",
			time2str(now()));
	}
	else
	{
	    printf("Time%4s: Attempt not successful, waiting.\n",
			time2str(now()));
	    Semaphore.acquire();
	}
	Duration = Rnd(10);
	Resource = ResourceAlloc.ObtainResource(ID);
	UseResource(Resource, ID, Duration);
	ResourceAlloc.ReleaseResource(Resource);
	Semaphore.release();
    }

    void Sleep(void)
    {
	int Duration;

	printf("Time%4s: Competitor %d done, ",
		time2str(now()), ID);
	Duration = Rnd(10);
	printf("going to sleep for %d time units.\n", Duration);
	if (Duration)
	    waitfor(Duration);
    }

    void main(void)
    {
	int i;

	for(i=0; i<10; i++)
	{
	    ObtainAndUseResource();
	    Sleep();
	}
    }
};


behavior Main			// let 5 behaviors compete for 3 resources!
{
    c_semaphore Semaphore((N));
    int         SharedResource[N];
    int         ResourceUser[N];
    monitor     ResourceAlloc(ResourceUser);

    Competitor c1(Semaphore, SharedResource, ResourceUser, ResourceAlloc, 1);
    Competitor c2(Semaphore, SharedResource, ResourceUser, ResourceAlloc, 2);
    Competitor c3(Semaphore, SharedResource, ResourceUser, ResourceAlloc, 3);
    Competitor c4(Semaphore, SharedResource, ResourceUser, ResourceAlloc, 4);
    Competitor c5(Semaphore, SharedResource, ResourceUser, ResourceAlloc, 5);

    int main(void)
    {
	printf("Time%4s: semaphore: Starting...\n",
		time2str(now()));
	par {	c1.main();
		c2.main();
		c3.main();
		c4.main();
		c5.main();
		}
	printf("Time%4s: semaphore: Done.\n",
		time2str(now()));
	return 0;
    }
};


// EOF semaphore.sc
