// ============================================================================
// bit.h: Bit Vector Library, user level
// ============================================================================
// 
//
// 09/26/06 PC  Adjustments for scrc 2.1
// 04/05/06 AG	Added overloaded copy and copy-assignment (for subclassing)
// 01/12/06 RD	added virtual destructor to template class 'bit'
// 11/28/01 <gerstl>	Removed constructors from shadow _longlong_
// 08/10/01 <gerstl>	Introduced len_t, bnd_t; changed shift cnt to unsigned
// 05/24/01 <gerstl>	Initial version
//
// ----------------------------------------------------------------------------

#ifndef __BIT
#define __BIT

#include <sys/bit.h>


/* ------------------------------------------------------------------------ */
/* Bit Vectors */


// Forward declarations
template<_bit::len_t,bool> class _bit_;


/* --- Bit vector representation --- */

template<_bit::len_t len> class _bitrep
{
public:
  // Array of chunks
  _bit::chunk  vec[_BITCHUNKS(len)];
};


/* --- Bit vector class --- */

template<_bit::len_t len, bool usign=false> class bit: protected _bitrep<len>
{
  typedef _bit::len_t len_t;  
  typedef _bit::bnd_t bnd_t;
  
public:
  // Constructors
  bit()                                                                     { }
  bit(const _bit& c)                                             { ref() = c; }
  bit(const _bitslice& s)                                        { ref() = s; }
  bit(const bit& c)                                       { ref() = c; }
  template<len_t l, bool us> bit(const bit<l,us>& c)      { ref() = c; }
  template<len_t l, bool us> bit(const _bit_<l,us>& c)    { ref() = c; }
  bit(const char* s)                                             { ref() = s; }
  bit(int i)                                                     { ref() = i; }
  bit(unsigned int i)                                            { ref() = i; }
  bit(long i)                                                    { ref() = i; }
  bit(unsigned long i)                                           { ref() = i; }
  bit(long long i)                                               { ref() = i; }
  bit(unsigned long long i)                                      { ref() = i; }
  
  // Destructor (01/12/06, RD)
  virtual ~bit()                                                            { }

  // Assignments
  bit& operator=(const _bit& o)                    { ref() = o; return put(); }
  bit& operator=(const _bitslice& o)               { ref() = o; return put(); }
  bit& operator=(const bit& c)              { ref() = c; return put(); }
  template<len_t l, bool us> 
    bit& operator=(const bit<l,us>& c)             { ref() = c; return put(); }
  template<len_t l, bool us> 
    bit& operator=(const _bit_<l,us>& c)           { ref() = c; return put(); }
  bit& operator=(const char* s)                    { ref() = s; return put(); }
  bit& operator=(int i)                            { ref() = i; return put(); }
  bit& operator=(unsigned int i)                   { ref() = i; return put(); }
  bit& operator=(long i)                           { ref() = i; return put(); }
  bit& operator=(unsigned long i)                  { ref() = i; return put(); }
  bit& operator=(long long i)                      { ref() = i; return put(); }
  bit& operator=(unsigned long long i)             { ref() = i; return put(); }
  bit& operator=(double d)                         { ref() = d; return put(); }
  bit& operator=(long double d)                    { ref() = d; return put(); }

  // Bit operators
  bit& operator^=(const _bit& op)         { get(); ref() ^= op; return put(); }
  bit& operator&=(const _bit& op)         { get(); ref() &= op; return put(); }
  bit& operator|=(const _bit& op)         { get(); ref() |= op; return put(); }

  // Arithmetic operators
  bit& operator+=(const _bit& op)         { get(); ref() += op; return put(); }
  bit& operator-=(const _bit& op)         { get(); ref() -= op; return put(); }
  bit& operator*=(const _bit& op)         { get(); ref() *= op; return put(); }
  bit& operator/=(const _bit& op)         { get(); ref() /= op; return put(); }
  bit& operator%=(const _bit& op)         { get(); ref() %= op; return put(); }

  // Shift operators
  bit& operator<<=(unsigned int sh)      { get(); ref() <<= sh; return put(); }
  bit& operator>>=(unsigned int sh)      { get(); ref() >>= sh; return put(); }

  // Flip the bits in the vector
  bit& flip(void)                        { get(); ref().flip(); return put(); }

  // Increment & Decrement
  bit& operator++ ()                        { get(); ++(ref()); return put(); }
  bit& operator-- ()                        { get(); --(ref()); return put(); }
  bit  operator++ (int)           { bit r(get()); ++(ref()); put(); return r; }
  bit  operator-- (int)           { bit r(get()); --(ref()); put(); return r; }

  // Test for != 0
  bool test(void) const                                { return val().test(); }

  // Explicit conversions
  int toInt(void) const                           { return val().toInt();     }
  unsigned int toUInt(void) const                 { return val().toUInt();    }
  long toLong(void) const                         { return val().toLong();    }
  unsigned long toULong(void) const               { return val().toULong();   }
  long long toLLong(void) const                   { return val().toLLong();   }
  unsigned long long toULLong(void) const         { return val().toULLong();  }
  double toDouble(void) const                     { return val().toDouble();  }
  long double toLDouble(void) const               { return val().toLDouble(); }
                                  
  // Slicing
  _bitslice operator[](len_t i)                   { return slice(i, i, true); }
  _bitslice operator()(len_t sl, len_t sr)            { return slice(sl, sr); }
  const _bitslice operator[](len_t i) const       { return slice(i, i, true); }
  const _bitslice operator()(len_t sl, len_t sr) const {return slice(sl, sr); }

  // Mapped slicing 
  _bitslice operator()(bnd_t l, bnd_t r, bnd_t i)    
              { ASSERT(_BITLEN(l,r)==len); return operator[](_BITMAP(l,r,i)); }
  _bitslice operator()(bnd_t l, bnd_t r, bnd_t sl, bnd_t sr) 
    {ASSERT(_BITLEN(l,r)==len); return slice(_BITMAP(l,r,sl),_BITMAP(l,r,sr));}
  const _bitslice operator()(bnd_t l, bnd_t r, bnd_t i) const 
              { ASSERT(_BITLEN(l,r)==len); return operator[](_BITMAP(l,r,i)); }
  const _bitslice operator()(bnd_t l, bnd_t r, bnd_t sl, bnd_t sr) const 
    {ASSERT(_BITLEN(l,r)==len); return slice(_BITMAP(l,r,sl),_BITMAP(l,r,sr));}
                              
  // Implicit cast to working (rhs) bit vector representation
  operator const _bit() const                { const_cast<bit*>(this)->get();
                      return _bit(const_cast<_bit::chunk*>(this->vec), len, usign); }
  
  // Create a piece for a bit vector bus
  virtual _bitbus_element* bus_slice(len_t sl, len_t sr) 
                       { return new _bitbus_element(this->vec, len, usign, sl, sr); }
  _bitbus_element* bus_slice(bnd_t l, bnd_t r, bnd_t sl, bnd_t sr) 
                        { return bus_slice(_BITMAP(l,r,sl), _BITMAP(l,r,sr)); }

protected:
  // Convert to _bit base class
  _bit ref(void)                              { return _bit(this->vec, len, usign); }
  const _bit val(void) const                  { return operator const _bit(); }
  
  // Create a slice
  virtual _bitslice slice(_bit::len_t l, _bit::len_t r, bool usgn = usign)
                             { return _bitslice(this->vec, len, usign, l, r, usgn); }
  virtual const _bitslice slice(len_t l, len_t r, bool usgn = usign) const
   { return _bitslice(const_cast<_bit::chunk*>(this->vec), len, usign, l, r, usgn); }
  
  // Functions for synchronization to be overloaded by bit busses
  virtual bit& put()                                          { return *this; }
  virtual bit& get()                                          { return *this; }
};



/* --- Bit vector bus (concatenated bit slices) --- */

template<_bit::len_t len, bool usign=false> class _bitbus
  : public bit<len,usign>
{
  typedef _bit::len_t len_t;  
  typedef _bit::bnd_t bnd_t;
  
protected:
  _bitbus_element* lst;

public:
  // Constructor
  _bitbus(_bitbus_element* list): lst(list)                                 { }
  
  // Desctructor
  virtual ~_bitbus() 
     { for(_bitbus_element* el=lst; el; el=lst)  { lst=el->next; delete el; } }
  
  // Create a piece for a bit vector bus
  virtual _bitbus_element* bus_slice(len_t sl, len_t sr) 
                                                 { return lst->slice(sl, sr); }
protected:  
  // Create a slice
  virtual _bitslice slice(len_t l, len_t r, bool usgn = usign)
                        { return _bitslice(this->vec, len, usign, l, r, usgn, lst); }
  virtual const _bitslice slice(len_t l, len_t r, bool usgn = usign) const
                            { return _bitslice(const_cast<_bit::chunk*>(this->vec), 
                    		                len, usign, l, r, usgn, lst); }
  // Sync with original vectors
  virtual _bitbus& put()                     { lst->put(this->ref()); return *this; }
  virtual _bitbus& get()                     { lst->get(this->ref()); return *this; }
};



/* --- Shadow class for use inside unions --- */

template<_bit::len_t len, bool usign=false> class _bit_
{
  typedef _bit::len_t len_t;  
  typedef _bit::bnd_t bnd_t;
  
public:  
  /* Cast to real bit vector
     Will fail miserably, cast from non-virtual to virtual class!
  operator const bit<len,usign>& (void) const 
                   { return *reinterpret_cast<const bit<len,usign>*>(this); }
  */
  
  // Cast to working bit vector
  operator const _bit (void) const 
               { return _bit(const_cast<_bit::chunk*>(data.vec), len, usign); }
  
  // Assignments
  _bit_& operator=(const _bit& c)                  { ref() = c; return *this; }
  _bit_& operator=(const _bitslice& s)             { ref() = s; return *this; }
  template<len_t l, bool us> 
    _bit_& operator=(const bit<l,us>& c)           { ref() = c; return *this; }
  template<len_t l, bool us> 
    _bit_& operator=(const _bit_<l,us>& c)         { ref() = c; return *this; }
  _bit_& operator=(const char* s)                  { ref() = s; return *this; }
  _bit_& operator=(int i)                          { ref() = i; return *this; }
  _bit_& operator=(unsigned int i)                 { ref() = i; return *this; }
  _bit_& operator=(long i)                         { ref() = i; return *this; }
  _bit_& operator=(unsigned long i)                { ref() = i; return *this; }
  _bit_& operator=(long long i)                    { ref() = i; return *this; }
  _bit_& operator=(unsigned long long i)           { ref() = i; return *this; }
  _bit_& operator=(double d)                       { ref() = d; return *this; }
  _bit_& operator=(long double d)                  { ref() = d; return *this; }

  // Bit operators
  _bit_& operator^=(const _bit& op)              { ref() ^= op; return *this; }
  _bit_& operator&=(const _bit& op)              { ref() &= op; return *this; }
  _bit_& operator|=(const _bit& op)              { ref() |= op; return *this; }

  // Arithmetic operators
  _bit_& operator+=(const _bit& op)              { ref() += op; return *this; }
  _bit_& operator-=(const _bit& op)              { ref() -= op; return *this; }
  _bit_& operator*=(const _bit& op)              { ref() *= op; return *this; }
  _bit_& operator/=(const _bit& op)              { ref() /= op; return *this; }
  _bit_& operator%=(const _bit& op)              { ref() %= op; return *this; }

  // Shift operators
  _bit_& operator<<=(unsigned int sh)           { ref() <<= sh; return *this; }
  _bit_& operator>>=(unsigned int sh)           { ref() >>= sh; return *this; }

  // Flip the bits in the vector
  _bit_& flip(void)                             { ref().flip(); return *this; }

  // Increment & Decrement
  _bit_& operator++ ()                             { ++(ref()); return *this; }
  _bit_& operator-- ()                             { --(ref()); return *this; }
  _bit_  operator++ (int)              { _bit_ r(*this); ++(ref()); return r; }
  _bit_  operator-- (int)              { _bit_ r(*this); --(ref()); return r; }
  
  // Test for != 0
  bool test(void) const                                { return val().test(); }

  // Explicit conversions
  int toInt(void) const                           { return val().toInt();     }
  unsigned int toUInt(void) const                 { return val().toUInt();    }
  long toLong(void) const                         { return val().toLong();    }
  unsigned long toULong(void) const               { return val().toULong();   }
  long long toLLong(void) const                   { return val().toLLong();   }
  unsigned long long toULLong(void) const         { return val().toULLong();  }
  double toDouble(void) const                     { return val().toDouble();  }
  long double toLDouble(void) const               { return val().toLDouble(); }
                                  
  // Slicing
  _bitslice operator[](len_t i)                   { return slice(i, i, true); }
  _bitslice operator()(len_t sl, len_t sr)            { return slice(sl, sr); }
  const _bitslice operator[](len_t i) const       { return slice(i, i, true); }
  const _bitslice operator()(len_t sl, len_t sr) const {return slice(sl, sr); }

  // Mapped slicing 
  _bitslice operator()(bnd_t l, bnd_t r, bnd_t i)    
              { ASSERT(_BITLEN(l,r)==len); return operator[](_BITMAP(l,r,i)); }
  _bitslice operator()(bnd_t l, bnd_t r, bnd_t sl, bnd_t sr) 
    {ASSERT(_BITLEN(l,r)==len); return slice(_BITMAP(l,r,sl),_BITMAP(l,r,sr));}
  const _bitslice operator()(bnd_t l, bnd_t r, bnd_t i) const 
              { ASSERT(_BITLEN(l,r)==len); return operator[](_BITMAP(l,r,i)); }
  const _bitslice operator()(bnd_t l, bnd_t r, bnd_t sl, bnd_t sr) const 
    {ASSERT(_BITLEN(l,r)==len); return slice(_BITMAP(l,r,sl),_BITMAP(l,r,sr));}
  
protected:
  // Convert to _bit base class
  _bit ref(void)                         { return _bit(data.vec, len, usign); }
  const _bit val(void) const                  { return operator const _bit(); }

  // Create a slice
  _bitslice slice(len_t l, len_t r, bool usgn = usign) 
                        { return _bitslice(data.vec, len, usign, l, r, usgn); }
  const _bitslice slice(len_t l, len_t r, bool usgn = usign) const 
                      { return _bitslice(const_cast<_bit::chunk*>(data.vec),
                                                     len, usign, l, r, usgn); }

private:
#ifdef GL_SYSTEM_VPTR_IS_FIRST_IN_CLASS
  // Simulate room needed for virtual method table pointer in real class
  void*  vptr;
#endif
  
  // Data area
  _bitrep<len> data;
  
#ifdef GL_SYSTEM_VPTR_IS_LAST_IN_CLASS
  // Simulate room needed for virtual method table pointer in real class
  void*  vptr;
#endif
};



#ifdef _BIT_TEMPLATE_OPERATORS


/* ------------------------------------------------------------------------ */
/* Result of an operation */


template<_bit::len_t len, bool usign> class _bitres: public _bit_<len,usign>
{
public:
  // Constructors
  _bitres(const _bit& op1, char type) 
                                          { _bit(vec, len, usign, op1, type); }
  _bitres(const _bit& op1, char type, unsigned int sh)
                                      { _bit(vec, len, usign, op1, type, sh); }
  _bitres(const _bit& op1, char type, const _bit& op2)
                                     { _bit(vec, len, usign, op1, type, op2); }
  _bitres(const _bit& op1, const _bit& op2)
                                           { _bit(vec, len, usign, op1, op2); }
};



/* ------------------------------------------------------------------------- */
/* Operators */


#define _BIT_UNARY(len,usign)	\
  _bitres<_BITLEN_UNARY(len,usign),_BITUSGN_UNARY(len,usign)>

#define _BIT_BINARY(len1,usign1,len2,usign2)	\
  _bitres<_BITLEN_BINARY(len1,usign1,len2,usign2),  \
          _BITUSGN_BINARY(len1,usign1,len2,usign2)>


template<_bit::len_t len1, bool usign1>
  const bit<len1,usign1>&
    operator+(const bit<len1,usign1>& op1) {
      return op1;
}

template<_bit::len_t len1, bool usign1>
  const _BIT_UNARY(len1,usign1)
    operator~(const bit<len1,usign1>& op1) {
      return _BIT_UNARY(len1,usign1)(op1, '~');
}

template<_bit::len_t len1, bool usign1>
  const _BIT_UNARY(len1,usign1)
    operator-(const bit<len1,usign1>& op1) {
      return _BIT_UNARY(len1,usign1)(op1, '-');
}

template<_bit::len_t len1, bool usign1>
  const _BIT_UNARY(len1,usign1)
    operator>>(const bit<len1,usign1>& op1, unsigned int sh) {
      return _BIT_UNARY(len1,usign1)(op1, '>', sh);
}

template<_bit::len_t len1, bool usign1>
  const _BIT_UNARY(len1,usign1)
    operator<<(const bit<len1,usign1>& op1, unsigned int sh) {
      return _BIT_UNARY(len1,usign1)(op1, '<', sh);
}
  
template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2>
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator+(const bit<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '+', op2);
}

template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2>
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator-(const bit<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '-', op2);
}

template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2>
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator*(const bit<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '*', op2);
}

template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2>
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator/(const bit<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '/', op2);
}

template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2>
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator%(const bit<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '%', op2);
}

template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2>
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator&(const bit<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '&', op2);
}

template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2>
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator|(const bit<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '|', op2);
}

template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2>
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator^(const bit<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '^', op2);
}



/* ------------------------------------------------------------------------- */
/* Operators with shadow class*/

template<_bit::len_t len1, bool usign1>
  const _bit_<len1,usign1>&
    operator+(const _bit_<len1,usign1>& op1) {
      return op1;
}

template<_bit::len_t len1, bool usign1>
  const _BIT_UNARY(len1,usign1)
    operator~(const _bit_<len1,usign1>& op1) {
      return _BIT_UNARY(len1,usign1)(op1, '~');
}

template<_bit::len_t len1, bool usign1>
  const _BIT_UNARY(len1,usign1)
    operator-(const _bit_<len1,usign1>& op1) {
      return _BIT_UNARY(len1,usign1)(op1, '-');
}

template<_bit::len_t len1, bool usign1>
  const _BIT_UNARY(len1,usign1)
    operator<<(const _bit_<len1,usign1>& op1, unsigned int sh) {
      return _BIT_UNARY(len1,usign1)(op1, '<', sh);
}

template<_bit::len_t len1, bool usign1>
  const _BIT_UNARY(len1,usign1)
    operator>>(const _bit_<len1,usign1>& op1, unsigned int sh) {
      return _BIT_UNARY(len1,usign1)(op1, '>', sh);
}

template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator+(const _bit_<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '+', op2);
}
template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator+(const bit<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '+', op2);
}
template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator+(const _bit_<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '+', op2);
}

template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator-(const _bit_<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '-', op2);
}
template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator-(const bit<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '-', op2);
}
template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator-(const _bit_<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '-', op2);
}

template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator*(const _bit_<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '*', op2);
}
template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator*(const bit<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '*', op2);
}
template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator*(const _bit_<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '*', op2);
}

template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator/(const _bit_<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '/', op2);
}
template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator/(const bit<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '/', op2);
}
template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator/(const _bit_<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '/', op2);
}

template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator%(const _bit_<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '%', op2);
}
template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator%(const bit<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '%', op2);
}
template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator%(const _bit_<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '%', op2);
}

template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator&(const _bit_<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '&', op2);
}
template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator&(const bit<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '&', op2);
}
template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator&(const _bit_<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '&', op2);
}

template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator|(const _bit_<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '|', op2);
}
template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator|(const bit<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '|', op2);
}
template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator|(const _bit_<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '|', op2);
}

template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator^(const _bit_<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '^', op2);
}
template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator^(const bit<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '^', op2);
}
template<_bit::len_t len1, bool usign1, _bit::len_t len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator^(const _bit_<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '^', op2);
}

#endif // #ifdef _BIT_TEMPLATE_OPERATORS


/* ------------------------------------------------------------------------- */
/* Global functions */


// Conversion to string
char *bit2str(unsigned int base, char *endptr, const _bit& b);
char *ubit2str(unsigned int base, char *endptr, const _bit& b);

// Conversion from string
void str2bit(unsigned int base, const char *str, _bit *bptr);
void str2ubit(unsigned int base, const char *str, _bit *ubptr);


#endif






