// c_semaphore.sc: provides protected access to shared resources
// (only a specified number of threads can use the resources at a time)
//
// author: Rainer Doemer
//
// modifications: (most recent first)
//
// 10/04/02 RD	added rule about safety of exceptions
// 02/13/02 RD	made the type of the count port 'const'
// 02/12/02 RD	applied naming convention, integrated with distribution
// 01/23/02 RD	added method 'attempt', minor changes
// 01/23/02 RD	bug fix: allow to wake up multiple threads at the same time
// 01/22/02 RD	bug fix: only call 'notifyone' if any threads are waiting
// 01/22/02 RD	brush up, moved into separate file "semaphore.sc"
// 12/27/01 RD	initial version
//
// interface rules:
//
// - see file i_semaphore.sc
//
// channel rules:
//
// - one channel instance is required for each set of shared resources
// - up to N threads may use the same channel instance, N=2**32-1
// - the number of available resources is given as an external count
//   which must be specified at the time of the channel instantiation
// - up to M resources can be controlled, M=2**32-1
// - no guarantees are given for fairness of resource usage
// - calling acquire() may suspend the calling thread indefinitely
// - this channel is only safe with respect to exceptions, if any exceptions
//   are guaranteed to occur only for all communicating threads simultaneously;
//   the behavior is undefined, if any exceptions occur only for a subset
//   of the communicating threads
// - no restrictions exist for use of 'waitfor'
// - no restrictions exist for use of 'wait', 'notify', 'notifyone'


import "i_semaphore";


channel c_semaphore(in const unsigned long c) implements i_semaphore
{
    event         e;
    unsigned long n = 0,
                  m = 0;

    void acquire(void)	// Dijkstra's "P" operation
    {
	if (++n > c)
	{
	    do wait e; while (!m);
	    m--;
	}
    }

    void release(void)	// Dijkstra's "V" operation
    {
	if (n-- > c)
	{
	    m++;
	    notify e;
	}
    }

    bool attempt(void)
    {
	if (n < c)
	{
	    n++;
	    return(true);
	}
	else
	{
	    return(false);
	}
    }
};


// EOF c_semaphore.sc
