/************************************************************************/
/* IntRep/PortMap.h: SpecC Internal Representation, PortMapping Classes	*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 06/24/98 */
/************************************************************************/

/* last update: 09/26/06 */

/* modifications: (most recent first)
 *
 * 09/26/06 PC  Adjustments for scrc 2.1
 * 06/03/05 RD	reorganized and renamed global type names
 * 11/08/01 RD	switched code generation to use GL_IO layer
 * 05/31/01 RD	eliminated level 2 of SIR API
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 02/14/01 RD	added member function IsOpenMapping to SIR_PortMap;
 *		adjusted parameters of WriteCC* methods
 * 02/13/01 RD	added member Constant to SIR_BitSlice in order to support
 *		port mapping to constants and open port mappings, as
 *		introduced with SpecC LRM1.0
 * 02/13/01 RD	created modifications list (last update was 04/12/99)
 */


#ifndef INTREP_PORTMAP_H
#define INTREP_PORTMAP_H


#include "Global.h"
#include "IntRep/List.h"
#include "IntRep/Constant.h"


/*** enumeration types **************************************************/


	/* none */


/*** type definitions ***************************************************/


typedef class SIR_PortMap		sir_portmap;
typedef SIR_List<sir_portmap>		sir_portmap_list;
typedef class SIR_PortMaps		sir_portmaps;

typedef class SIR_BitSlice		sir_bitslice;
typedef SIR_List<sir_bitslice>		sir_slice_list;
typedef class SIR_BitSlices		sir_bitslices;

typedef class SIR_Symbol		sir_symbol;	/* cyclic link */
typedef class SIR_Type			sir_type;	/* cyclic link */
typedef class SIR_TypePtr		sir_type_ptr;	/* cyclic link */
typedef class SIR_TypePtrs		sir_type_ptrs;	/* cyclic link */


/*** class declarations *************************************************/


	/*******************/
	/*** SIR_PortMap ***/
	/*******************/


class SIR_PortMap :		/* port mapping (for one port) */
	public SIR_ListElem<SIR_PortMap>	/* is a list element */
{
public:
sir_bitslices	*BitSlices;	/* list of bitslices */


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_PortMap(			/* constructor #1 (general) */
	sir_bitslices	*BitSlices);

SIR_PortMap(			/* constructor #3 (duplicator) */
	sir_portmap	*Original);

~SIR_PortMap(void);		/* destructor */


static sir_portmap *New(	/* create a new direct port mapping (#2b) */
	sir_symbol	*DirectMapping);/* (returns NULL if SIR_Error) */


ERROR DFS_ForAllNodes(		/* iterator over all nodes (depth first) */
	sir_node_mptr	MemberFct,
	sir_node_marg	MemberFctArg);


void UnAlias(void);		/* unalias all type, usertype, symbol links */


void WriteCC_Decl(		/* generates C++ source code (part 1) */
	gl_io		*IO,
	sir_type	*PortType,
	int		*BitBusNum,
	int		*OpenPortNum,
	int		*ConstPortNum);

const char *WriteCC_Init(	/* generates C++ source code (part 2) */
	sir_type	*PortType,  /* (returns NULL for "nothing to do") */
	int		*BitBusNum,
	int		*OpenPortNum,
	int		*ConstPortNum);

void WriteCC(			/* generates C++ source code (part 3) */
	gl_io		*IO,
	sir_type	*PortType,
	int		*BitBusNum,
	int		*OpenPortNum,
	int		*ConstPortNum);

bool IsOpenMapping(void);	/* determines whether mapping is open */

bool IsDirectlyMappable(	/* determines port mapping type */
	sir_type	*PortType);

ERROR Check(			/* semantically check the port mapping */
	sir_type_ptr	*Port,
	unsigned int	PortNum,
	const char	*SourceFile = NULL,	/* default: no warnings */
	unsigned int	SourceLine = 0);
};


	/********************/
	/*** SIR_PortMaps ***/
	/********************/


class SIR_PortMaps :		/* list of port mappings */
	public SIR_List<SIR_PortMap>	/* is simply a list of portmaps */
{					/* with additional methods */
public:

//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_PortMaps(			/* constructor #1 */
	sir_portmap	*FirstEntry = NULL);

SIR_PortMaps(			/* constructor #2 (duplicator) */
	sir_portmaps	*Original);

~SIR_PortMaps(void);		/* destructor */


ERROR DFS_ForAllNodes(		/* iterator over all nodes (depth first) */
	sir_node_mptr	MemberFct,
	sir_node_marg	MemberFctArg);


void UnAlias(void);		/* unalias all type, usertype, symbol links */


ERROR WriteSC(			/* (re-) generates SpecC source code */
	gl_io	*IO);


ERROR Check(			/* semantically check the port mappings */
	sir_type_ptrs	*PortList,
	const char	*SourceFile = NULL,	/* default: no warnings */
	unsigned int	SourceLine = 0);
};


	/********************/
	/*** SIR_BitSlice ***/
	/********************/


class SIR_BitSlice :		/* bitslice in port mapping */
	public SIR_ListElem<SIR_BitSlice>	/* is a list element */
{
public:
sir_symbol	*Symbol;	/* link to symbol table entry (NULL if open) */
sir_constant	*Constant;	/* constant mapping (or NULL if open) */
int		LeftBound,	/* bounds of slice (if BITVECTOR, else 0) */
		RightBound;


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_BitSlice(			/* constructor #1 (standard) */
	sir_symbol	*Symbol,	/* NULL for open connection */
	int		LeftBound = 0,
	int		RightBound = 0);

SIR_BitSlice(			/* constructor #2 (constant) */
	sir_constant	*Constant);

SIR_BitSlice(			/* constructor #4 (duplicator) */
	sir_bitslice	*Original);

~SIR_BitSlice(void);		/* destructor */


static sir_bitslice *New(	/* create a new bitslice #1b */
	sir_symbol	*Mapping,	/* (returns NULL if SIR_Error) */
	int		LeftBound = 0,	/* bounds for bitvectors */
	int		RightBound = 0);


ERROR DFS_ForAllNodes(		/* iterator over all nodes (depth first) */
	sir_node_mptr	MemberFct,
	sir_node_marg	MemberFctArg);
};


	/*********************/
	/*** SIR_BitSlices ***/
	/*********************/


class SIR_BitSlices :		/* list of (concatenated) bitslices */
	public SIR_List<SIR_BitSlice>	/* is simply a list of bitslices */
{					/* with additional methods */
public:

//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_BitSlices(			/* constructor #1 */
	sir_bitslice	*FirstEntry = NULL);

SIR_BitSlices(			/* constructor #2 (duplicator) */
	sir_bitslices	*Original);

~SIR_BitSlices(void);		/* destructor */


ERROR DFS_ForAllNodes(		/* iterator over all nodes (depth first) */
	sir_node_mptr	MemberFct,
	sir_node_marg	MemberFctArg);


void UnAlias(void);		/* unalias all type, usertype, symbol links */
};


/*** exported functions *************************************************/


	/* none */


#endif /* INTREP_PORTMAP_H */

/* EOF IntRep/PortMap.h */
