/************************************************************************/
/* GL_String.h: Replacement for GNU's libg++ string class...		*/
/************************************************************************/
/* Author: Andreas Gerstlauer <gerstl>		first version: 03/02/00 */
/************************************************************************/

/* last update: 09/26/06 */

/* modifications: (most recent first)
 *
 * 09/26/06 PC  Adjustments for scrc 2.1
 * 02/20/06 AG	changed internal string length from 'unsigned short' to 'size_t'
 * 06/03/05 RD	reorganized and renamed global type names
 * 01/18/05 RD	added SIR_INTERNAL sections
 * 10/05/04 RD	adjustments for compilation on 64-bit architectures
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 05/03/04 PC	SubString constructors made public tp fix compilation errors
 *		in gcc 3.4.0
 * 06/21/02 RD	replaced [io]stream with std::[io]stream
 * 06/20/02 RD	replaced deprecated header <iostream.h> with <iostream>
 * 11/15/01 RD	added two 'const' qualifiers to avoid overload ambiguities
 * 11/15/01 RD	started this header (last change was 11/05/01 by AG)
 */

#ifndef GL_STRING_H
#define GL_STRING_H

#include <stdlib.h>
#include <string.h>
#include <iostream>
#include <stdarg.h>
#include <assert.h>


/*** type definitions ***************************************************/


typedef class GL_String		gl_string;	/* use our own strings */
typedef class GL_SubString	gl_substring;

typedef struct GL_StringList	gl_stringlist;

struct GL_StringList		/* simple string list */
{
gl_stringlist	*Succ;
const char	*String;
};


/*** class definitions **************************************************/


class GL_String
{
   friend class GL_SubString;

protected:
   size_t	_len;	// string length
   size_t	_size;	// allocated size
   char*	_str;

public:
   // constructors & assignment
   GL_String();
   GL_String(const GL_String& x);
   GL_String(const GL_SubString& x);
   GL_String(const char* t);
   GL_String(const char* t, size_t length);
   GL_String(char c);

   ~GL_String();

   // assignment
   GL_String&	operator =  (const GL_String&	  y);
   GL_String&	operator =  (const GL_SubString&  y);
   GL_String&	operator =  (const char* y);
   GL_String&	operator =  (char	 c);

   // format string
   // Note: only plain %%, %c, %s, and %[d,u,i,o,x,X] are supported!
   GL_String&	form  (const char* format, ...);
   GL_String&	vform (const char* format, va_list args);

   // concatenation
   GL_String&	operator += (const GL_String&	  y); 
   GL_String&	operator += (const GL_SubString&  y);
   GL_String&	operator += (const char* t);
   GL_String&	operator += (char	 c);

   void		prepend(const GL_String&	  y); 
   void		prepend(const GL_SubString&  y);
   void		prepend(const char* t);
   void		prepend(char	 c);

   //  return number of occurences of target in string
   int		freq(char	 c) const; 
   int		freq(const char* t) const;

   // Substring extraction
   GL_SubString	at(int pos, size_t len);
   GL_SubString	at(size_t pos, size_t len);
   GL_SubString	at(const char* t, size_t startpos = 0);
   GL_SubString	at(char	 c, size_t startpos = 0);

   GL_SubString	before(int pos);
   GL_SubString	before(size_t pos);
   GL_SubString	before(const char* t, size_t startpos = 0);
   GL_SubString	before(char	   c, size_t startpos = 0);
   GL_SubString	beforelast(char	   c, size_t startpos = 0);

   GL_SubString	after(int pos);
   GL_SubString	after(size_t pos);
   GL_SubString	after(const char* t, size_t startpos = 0);
   GL_SubString	after(char	  c, size_t startpos = 0);
   GL_SubString	afterlast(char	  c, size_t startpos = 0);

   // deletion
   void		del(const char* t);
   void		del(char	c);

   // conversion
   operator const char*() const;
   const char*	chars() const;

   // element extraction  
   char&	operator [] (int i);
   const char&	operator [] (int i) const;
   char&	operator [] (size_t i);
   const char&	operator [] (size_t i) const;
   
   // status
   size_t	length() const;
   int		empty() const;

   // IO
   friend inline std::ostream& operator<<(std::ostream& s, const GL_String& x);
   friend inline std::ostream& operator<<(std::ostream& s, const GL_SubString& x);
   friend std::istream& operator>>(std::istream& s, GL_String& x);

protected:
   char* salloc (size_t len, size_t keep = 0, bool force = false);
   void	 scopy	(const char* t, size_t startpos, size_t len);
   void	 sinsert(const char* t, size_t pos, size_t len, size_t repl = 0);
   void	 sinsert(char c, size_t pos, size_t repl = 0);
};


class GL_SubString
{
  friend class GL_String;
  
protected:
  GL_String&	_str;	// The string I'm a substring of
  size_t	_pos;	// starting position in S's rep
  size_t	_len;	// length of substring

public:
  //PC 05/02/04
   GL_SubString(GL_String& x, size_t p, size_t l);
   GL_SubString(const GL_SubString& x);
  ~GL_SubString();

  GL_SubString&		operator =  (const GL_String&	 y);
  GL_SubString&		operator =  (const GL_SubString&	 y);
  GL_SubString&		operator =  (const char* t);
  GL_SubString&		operator =  (char	c);

  // status
  size_t		length() const;
  int			empty() const;
  const char*		chars() const;
  operator const	char*() const;
  // Note that these don't necessarily return a null-terminated char* !
  // Only a pointer into the parent string, valid up to length()

  // I/O
  friend inline std::ostream& operator<<(std::ostream& s, const GL_SubString& x);
};


extern const char* const _nullString;

// inlined constructors & assignment

inline GL_String::GL_String() 
 : _len(0), _size(0), _str((char*) _nullString) { }
inline GL_String::GL_String(const GL_String& x) 
 : _len(0), _size(0), _str((char*) _nullString) { operator= (x); }
inline GL_String::GL_String(const GL_SubString& x) 
 : _len(0), _size(0), _str((char*) _nullString) { operator= (x); }
inline GL_String::GL_String(const char* t) 
 : _len(0), _size(0), _str((char*) _nullString) { operator= (t); }
inline GL_String::GL_String(char c) 
 : _len(0), _size(0), _str((char*) _nullString) { operator= (c); }
inline GL_String::GL_String(const char* t, size_t length)
 : _len(0), _size(0), _str((char*) _nullString) {
  salloc(length);
  t? sinsert(t, 0, length, _len) : sinsert("", 0, 0, _len);
}
inline GL_String::~GL_String() {
   if (_str != _nullString) free (_str);
}
inline GL_String& GL_String::operator= (const GL_String& y) {
  sinsert(y.chars(), 0, y.length(), _len);
  return *this;
}
inline GL_String& GL_String::operator= (const GL_SubString& y) {
  sinsert(y.chars(), 0, y.length(), _len);
  return *this;
}
inline GL_String& GL_String::operator= (const char* y) {
  y? sinsert(y, 0, strlen(y), _len) : sinsert("", 0, 0, _len);
  return *this;
}
inline GL_String& GL_String::operator= (const char c) {
  sinsert(c, 0, _len);
  return *this;
}


// inlined methods

inline GL_String& GL_String::operator+= (const GL_String& y) {
  sinsert(y.chars(), _len, y.length());
  return *this;
}
inline GL_String& GL_String::operator+= (const GL_SubString& y) {
  sinsert(y.chars(), _len, y.length());
  return *this;
}
inline GL_String& GL_String::operator+= (const char* y) {
  if (y) sinsert(y, _len, strlen(y));
  return *this;
}
inline GL_String& GL_String::operator+= (const char c) {
  sinsert(c, _len);
  return *this;
}

inline void GL_String::prepend(const GL_String&	    y) {
  sinsert(y.chars(), 0, y.length());
}
inline void GL_String::prepend(const GL_SubString&  y) {
  sinsert(y.chars(), 0, y.length());
}
inline void GL_String::prepend(const char* t) {
  if (t) sinsert(t, 0, strlen(t));
}
inline void GL_String::prepend(char	   c) {
  sinsert(c, 0);
}

inline GL_SubString GL_String::at(int pos, size_t len) {
  assert(pos >= 0);
  return GL_SubString(*this, (size_t)pos, len);
}
inline GL_SubString GL_String::at(size_t pos, size_t len) {
  return GL_SubString(*this, pos, len);
}

inline GL_SubString GL_String::before(int pos) {
  assert(pos >= 0);
  assert((unsigned)pos < _len);
  return GL_SubString(*this, 0, (size_t)pos);
}
inline GL_SubString GL_String::before(size_t pos) {
  assert(pos < _len);
  return GL_SubString(*this, 0, pos);
}

inline GL_SubString GL_String::after(int pos) {
  assert(pos >= 0);
  assert((unsigned)pos < _len);
  return GL_SubString(*this, (size_t)pos + 1, _len - pos - 1);
}
inline GL_SubString GL_String::after(size_t pos) {
  assert(pos < _len);
  return GL_SubString(*this, pos + 1, _len - pos - 1);
}

inline void GL_String::del(const char* t) {
  at(t).operator= ((const char*)0);
}
inline void GL_String::del(char	       c) {
  at(c).operator= ((const char*)0);
}

inline const char* GL_String::chars() const {
   return _str;
}
inline GL_String::operator const char*() const {
   return chars();
}

inline size_t GL_String::length() const {
   return _len;
}
inline int GL_String::empty() const {
   return (_len == 0);
}

inline char&  GL_String::operator [] (int i)
{
  assert (i >= 0);
  assert (((size_t)i) < length());
  return _str[i];
}
inline char&  GL_String::operator [] (size_t i)
{ 
  assert (i < length());
  return _str[i];
}

inline const char&  GL_String::operator [] (int i) const
{ 
  assert (i >= 0);
  assert (((size_t)i) < length());
  return _str[i];
}
inline const char&  GL_String::operator [] (size_t i) const
{ 
  assert (i < length());
  return _str[i];
}


// ------------------- GL_SubString implementation ---------------------

// inlined constructors & assignment


inline GL_SubString::GL_SubString(GL_String& x, size_t p, size_t l)
  : _str(x), _pos(p), _len(l) 
{
  if (_pos + _len > x._len) _len = x._len - _pos;
  assert((p >= 0) && (p <= x._len));
  assert((l >= 0) && (p + _len <= x._len));
}
inline GL_SubString::GL_SubString(const GL_SubString& x) 
  : _str(x._str), _pos(x._pos), _len(x._len) {
}

inline GL_SubString::~GL_SubString() {
}


// inlined methods

inline const char* GL_SubString::chars() const {
   return &(_str.chars())[_pos];
}
inline GL_SubString::operator const char*() const {
   return chars();
}

inline size_t GL_SubString::length() const {
   return _len;
}
inline int GL_SubString::empty() const {
   return (_len == 0);
}


// ------------------- Global functions ---------------------

int compare(const GL_String& x, const GL_String& y);
int compare(const GL_String& x, const char* t);
int compare(const GL_SubString& x, const GL_String& y);
int compare(const GL_String& x, const GL_SubString&  y);
int compare(const GL_SubString& x, const GL_SubString&	y);
int compare(const GL_SubString& x, const char* t);

 
// addition operators

inline GL_String operator + (const GL_String& x, const GL_String& y) {
   GL_String r(x); r += y; return r;
}
inline GL_String operator + (const GL_String& x, const GL_SubString& y) {
   GL_String r(x); r += y; return r;
}
inline GL_String operator + (const GL_String& x, const char* y) {
   GL_String r(x); r += y; return r;
}
inline GL_String operator + (const char* x, const GL_String& y) {
   GL_String r(x); r += y; return r;
}
inline GL_String operator + (const GL_String& x, char y) {
   GL_String r(x); r += y; return r;
}
inline GL_String operator + (const GL_SubString& x, const GL_String& y) {
   GL_String r(x); r += y; return r;
}
inline GL_String operator + (const GL_SubString& x, const GL_SubString& y) {
   GL_String r(x); r += y; return r;
}
inline GL_String operator + (const GL_SubString& x, const char* y) {
   GL_String r(x); r += y; return r;
}
inline GL_String operator + (const char* x, const GL_SubString& y) {
   GL_String r(x); r += y; return r;
}
inline GL_String operator + (const GL_SubString& x, char y) {
   GL_String r(x); r += y; return r;
}


// I/O operators  
  
inline std::ostream& operator<<(std::ostream& s, const GL_String& x) {
   s << x.chars(); return s;
}
inline std::ostream& operator<<(std::ostream& s, const GL_SubString& x) {
  s.write(x.chars(), x.length());
  return s;
}


// a zillion comparison operators

inline int operator==(const GL_String& x, const GL_String& y) 
{
  return compare(x, y) == 0; 
}

inline int operator!=(const GL_String& x, const GL_String& y)
{
  return compare(x, y) != 0; 
}

inline int operator>(const GL_String& x, const GL_String& y)
{
  return compare(x, y) > 0; 
}

inline int operator>=(const GL_String& x, const GL_String& y)
{
  return compare(x, y) >= 0; 
}

inline int operator<(const GL_String& x, const GL_String& y)
{
  return compare(x, y) < 0; 
}

inline int operator<=(const GL_String& x, const GL_String& y)
{
  return compare(x, y) <= 0; 
}

inline int operator==(const GL_String& x, const GL_SubString&  y) 
{
  return compare(x, y) == 0; 
}

inline int operator!=(const GL_String& x, const GL_SubString&  y)
{
  return compare(x, y) != 0; 
}

inline int operator>(const GL_String& x, const GL_SubString&  y)      
{
  return compare(x, y) > 0; 
}

inline int operator>=(const GL_String& x, const GL_SubString&  y)
{
  return compare(x, y) >= 0; 
}

inline int operator<(const GL_String& x, const GL_SubString&  y) 
{
  return compare(x, y) < 0; 
}

inline int operator<=(const GL_String& x, const GL_SubString&  y)
{
  return compare(x, y) <= 0; 
}

inline int operator==(const GL_String& x, const char* t) 
{
  return compare(x, t) == 0; 
}

inline int operator!=(const GL_String& x, const char* t) 
{
  return compare(x, t) != 0; 
}

inline int operator>(const GL_String& x, const char* t)	 
{
  return compare(x, t) > 0; 
}

inline int operator>=(const GL_String& x, const char* t) 
{
  return compare(x, t) >= 0; 
}

inline int operator<(const GL_String& x, const char* t)	 
{
  return compare(x, t) < 0; 
}

inline int operator<=(const GL_String& x, const char* t) 
{
  return compare(x, t) <= 0; 
}

inline int operator==(const GL_SubString& x, const GL_String& y) 
{
  return compare(y, x) == 0; 
}

inline int operator!=(const GL_SubString& x, const GL_String& y)
{
  return compare(y, x) != 0;
}

inline int operator>(const GL_SubString& x, const GL_String& y)	     
{
  return compare(y, x) < 0;
}

inline int operator>=(const GL_SubString& x, const GL_String& y)     
{
  return compare(y, x) <= 0;
}

inline int operator<(const GL_SubString& x, const GL_String& y)	     
{
  return compare(y, x) > 0;
}

inline int operator<=(const GL_SubString& x, const GL_String& y)     
{
  return compare(y, x) >= 0;
}

inline int operator==(const GL_SubString& x, const GL_SubString&  y) 
{
  return compare(x, y) == 0; 
}

inline int operator!=(const GL_SubString& x, const GL_SubString&  y)
{
  return compare(x, y) != 0;
}

inline int operator>(const GL_SubString& x, const GL_SubString&	 y)	 
{
  return compare(x, y) > 0;
}

inline int operator>=(const GL_SubString& x, const GL_SubString&  y)
{
  return compare(x, y) >= 0;
}

inline int operator<(const GL_SubString& x, const GL_SubString&	 y) 
{
  return compare(x, y) < 0;
}

inline int operator<=(const GL_SubString& x, const GL_SubString&  y)
{
  return compare(x, y) <= 0;
}

inline int operator==(const GL_SubString& x, const char* t) 
{
  return compare(x, t) == 0; 
}

inline int operator!=(const GL_SubString& x, const char* t) 
{
  return compare(x, t) != 0;
}

inline int operator>(const GL_SubString& x, const char* t)  
{
  return compare(x, t) > 0; 
}

inline int operator>=(const GL_SubString& x, const char* t) 
{
  return compare(x, t) >= 0; 
}

inline int operator<(const GL_SubString& x, const char* t)  
{
  return compare(x, t) < 0; 
}

inline int operator<=(const GL_SubString& x, const char* t) 
{
  return compare(x, t) <= 0; 
}
  
  





#endif
