/************************************************************************/
/* IntRep/Constant.h: SpecC Internal Representation, Constant Class	*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 09/24/97 */
/************************************************************************/

/* last update: 09/26/06 */

/* modification: (most recent first)
 *
 * 09/26/06 PC  Adjustments for scrc 2.1
 * 06/03/05 RD	reorganized and renamed global type names
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 08/13/01 RD	switched type of bitvector length to unsigned
 * 05/31/01 RD	eliminated level 2 of SIR API
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 04/25/01 RD	inserted "union-safe" versions of LONG_LONG types
 * 04/19/01 RD	eliminated T_Value from the union in SIR_Constant
 * 04/17/01 RD	added missing length argument to Convert[ed]() functions
 * 04/06/01 RD	changed time type from 'long long' to 'unsigned long long'
 * 04/06/01 RD	added support for non-native 'long long' type
 * 04/06/01 RD	started this header (last update was 04/12/99)
 */

#ifndef INTREP_CONSTANT_H
#define INTREP_CONSTANT_H


#include "sys/bit.h"
#include "Global.h"
#include "IntRep/Node.h"


/*** enumeration types **************************************************/


enum SIR_ConstType	/* (order and offset are important!) */
{
SIR_CONST_BOOL,				// New() #1 or #2
SIR_CONST_CHAR,				// New() #1 or #2
SIR_CONST_UCHAR,			// New() #1 or #2
SIR_CONST_SHORT,			// New() #1 or #2
SIR_CONST_USHORT,			// New() #1 or #2
SIR_CONST_INT,				// New() #1 or #2
SIR_CONST_UINT,				// New() #1 or #2
SIR_CONST_LONG,				// New() #1 or #2
SIR_CONST_ULONG,			// New() #1 or #2
SIR_CONST_LONGLONG,			// New() #1 or #2
SIR_CONST_ULONGLONG,			// New() #1 or #2
SIR_CONST_FLOAT,			// New() #3
SIR_CONST_DOUBLE,			// New() #3
SIR_CONST_LONGDOUBLE,			// New() #3
SIR_CONST_BIT,				// New() #1 or #4 or #5
SIR_CONST_UBIT,				// New() #2 or #4 or #5
SIR_CONST_CHARSTRING,	/* max! */	// New() #5

SIR_CONST_TIME	= SIR_CONST_ULONGLONG	/* 'time' is 'unsigned long long' */
};


/*** type definitions ***************************************************/


typedef enum SIR_ConstType	SIR_CONSTTYPE;

typedef class SIR_Constant	sir_constant;

typedef class _bit		sir_bit;
typedef class _bitop		sir_bitop;


/*** class declarations *************************************************/


	/********************/
	/*** SIR_Constant ***/
	/********************/


class SIR_Constant :		/* constant representation */
	public SIR_Unit,		/* is a unit of the design */
	public SIR_Node			/* and a node */
{
public:
SIR_CONSTTYPE		Type;		/* type of the constant */
union
{
bool			B_Value;	/* boolean constants */
char			C_Value;	/* character constants */
unsigned char		UC_Value;
short int		S_Value;	/* integer constants */
unsigned short int	US_Value;
int			I_Value;
unsigned int		UI_Value;
long int		L_Value;
unsigned long int	UL_Value;
LONG_LONG_		LL_Value;
UNSIGNED_LONG_LONG_	ULL_Value;
float			F_Value;	/* floating constants */
double			D_Value;
long double		LD_Value;
sir_bit			*BIT_Value;	/* bit vector constant */
gl_string		*CS_Value;	/* character string constant */
};


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Constant(				/* constructor #1 */
	SIR_CONSTTYPE	Type,
	int		Value,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL);

SIR_Constant(				/* constructor #2 */
	SIR_CONSTTYPE	Type,
	LONG_LONG	Value,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL);

SIR_Constant(				/* constructor #3 */
	SIR_CONSTTYPE	Type,
	UNSIGNED_LONG_LONG	Value,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL);

SIR_Constant(				/* constructor #4 */
	SIR_CONSTTYPE	Type,
	long double	Value,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL);

SIR_Constant(				/* constructor #5 */
	SIR_CONSTTYPE	Type,		/* (the string is consumed!) */
	gl_string	*Value,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL);

SIR_Constant(				/* constructor #6 */
	SIR_CONSTTYPE	Type,		/* (the string is copied!) */
	const char	*Value,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL);

SIR_Constant(				/* constructor #7 (for bitvectors) */
	sir_bit		Value,		/* (the bitvector is copied) */
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL);

SIR_Constant(				/* constructor #8 */
	sir_constant	*Original);	/* (duplicator) */

~SIR_Constant(void);			/* destructor */


ERROR DFS_ForAllNodes(		/* iterator over all nodes (depth first) */
	sir_node_mptr	MemberFct,
	sir_node_marg	MemberFctArg);


sir_constant *Promoted(void);	/* promoted constant */

sir_constant *Converted(	/* converted constant */
	SIR_CONSTTYPE	Type,	/* (supports all legal conversions) */
	unsigned int	Length = 0);	/* for bitvecs; 0 is natural length */


static const char *PrettyChar(	/* returns a C-style character */
	const char	Char);

static void PrettyString(	/* returns a C-style string */
	gl_string	*Result,
	const char	*String);

static sir_constant *New(       /* create a new integer constant (#1) */
        LONG_LONG       Value,  /* (returns NULL if SIR_Error) */
        SIR_CONSTTYPE   CType = SIR_CONST_INT); /* default: integer */

sir_constant *Copy(             /* creates a copy of the constant */
        bool            Strip = true);  /* (returns NULL if SIR_Error) */

void Delete(void);              /* deletes the constant */

const char *Print(		/* return text string of the constants value */
	bool		CppNotation = false);	/* default: SpecC notation */

ERROR Compare(                  /* compare this constant with another one (#1)*/
        sir_constant    *Const2,        /* (returns error if not comparable) */
        int             *Result);       /* Result is {-1,0,1} as in strcmp() */

static int Compare(             /* compare two comparable constants (#2) */
        sir_constant    *Const1,        /* result is {-1,0,1} as in strcmp() */
        sir_constant    *Const2);
};


#endif /* INTREP_CONSTANT_H */

/* EOF IntRep/Constant.h */
