/************************************************************************/
/* GL_FileIO.h: Global utility layer for file IO			*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 11/02/01 */
/************************************************************************/

/* last update: 09/26/06 */

/* modifications: (most recent first)
 *
 * 09/26/06 PC  Adjustments for scrc 2.1
 * 06/03/05 RD	reorganized and renamed global type names
 * 12/16/04 RD	added Force option to GL_FileIsCreatable() (bug fix)
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 02/22/03 RD	added a useful comment
 * 06/24/02 RD	made global functions link as "C" style
 *		to avoid cross-platform linker problems
 * 11/09/01 RD	added support for line wrapping
 * 11/07/01 RD	completed implementation of class GL_IO
 * 11/02/01 RD	initial version (extract from Global.h)
 */

#ifndef GL_FILEIO_H
#define GL_FILEIO_H

#include <stdio.h>

#include "GL_String.h"


/*** constants and macros ***********************************************/


#define GL_TAB_STEP_DEFAULT	4
#define GL_TAB_STEP_SYSTEM	8	/* set to 0 to disable use of '\t' */
#define GL_LINE_WRAP_DEFAULT	70


/*** enumeration types **************************************************/


enum GL_IO_Mode		/* I/O mode */
{
GL_IO_NONE,
GL_IO_READ,
GL_IO_WRITE,
GL_IO_RW
};


/*** type definitions ***************************************************/


typedef int			ERROR;

typedef enum GL_IO_Mode		GL_IO_MODE;

typedef class GL_IO		gl_io;


/*** class declarations *************************************************/


class GL_IO			/* convenience input/output layer */
{
public:
GL_IO_MODE	Mode;		/* operation mode (see above) */
FILE		*FileHandle;	/* handle for actual file I/O (or NULL) */
gl_string	*StringHandle;	/* handle for string I/O (or NULL) */
unsigned int	LineNo;		/* current line number (starts with 1) */
unsigned int	ColNo;		/* current column number (starts with 0) */
unsigned int	Pos;		/* current byte position (starts with 0) */
int		Tab;		/* tabulator (current indent) */
int		TabStep;	/* tabulator stepping (increment/decrement) */
static int	SystemTabStep;	/* system tabulator stepping ('\t') (0=off) */
unsigned int	LineWrap;	/* column for line wrapping */


GL_IO(				/* constructor #1 */
	GL_IO_MODE	Mode,		/* (for file I/O) */
	FILE		*FileHandle);

GL_IO(				/* constructor #2 */
	GL_IO_MODE	Mode,		/* (for string I/O) */
	gl_string	*StringHandle);

~GL_IO(void);			/* destructor */


//++++++++++++++++++++++++++++ User API ++++++++++++++++++++++++++++++++//


static gl_io *OpenFile(		/* open a file */
	const char	*Filename,	/* NULL for tmp. file, "-" for stdio */
	GL_IO_MODE	Mode = GL_IO_WRITE);

static gl_io *OpenString(	/* open a string */
	gl_string	*String,	/* caller-supplied string */
	GL_IO_MODE	Mode = GL_IO_WRITE);

ERROR Check(void);		/* check for any I/O errors */

ERROR ReadEOF(void);		/* read (and check) EOF condition */

ERROR Rewind(void);		/* rewind an RW file/string */

void Close(void);		/* close the file/string I/O */

ERROR CloseWithCheck(void);	/* close the file/string I/O, check error */


unsigned int GetLineNo(void);	/* get current line number */

unsigned int GetColNo(void);	/* get current column number */

unsigned int GetPos(void);	/* get current byte position */

int GetTab(void);		/* get current tabulator setting */

int GetTabStep(void);		/* get tabulator stepping */

int GetSystemTabStep(void);	/* get system tabulator stepping */

unsigned int GetLineWrap(void);	/* get column of line wrapping */


void SetTab(			/* set new tabulator setting */
	int		Tab);

void SetTabStep(		/* set new tabulator stepping */
	int		TabStep);

void SetSystemTabStep(		/* set new system tabulator stepping */
	int		SystemTabStep);

void SetLineWrap(		/* set column of line wrapping */
	unsigned int	LineWrap);	/* (use 0 to turn off) */


void TabStepUp(void);		/* increment tabulator by one step */

void TabStepDown(void);		/* decrement tabulator by one step */

void Add2Tab(			/* increase/decrease tabulator setting */
	int		Chars);


ERROR Read(			/* read (binary) data from file/string */
	void		*Bytes,
	size_t		NumBytes);

char GetC(void);		/* getc() replacement */

char *GetS(			/* gets() replacement (#1) */
	char		*String,
	int		MaxLen);

gl_string *GetS(		/* gets() replacement (#2) */
	gl_string	*String);


ERROR Write(			/* write (binary) data to file/string */
	const void	*Bytes,
	size_t		NumBytes);

void PrintF(			/* printf() replacement */
	const char	*Format,
	...);

void PutS(			/* puts() replacement (#1) */
	const char	*String);

void PutS(			/* puts() replacement (#2) */
	const gl_string	String);

void PutC(			/* putc() replacement */
	const char	Char);

void PutNL(void);		/* start a new line */

void PutTAB(void);		/* put a tabulator */

void PutNLTAB(void);		/* start a new line and put a tabulator */

void WrapLine(void);		/* wrap line around if too long */

void WrapLineTAB(void);		/* wrap line around and put TAB if too long */
};


/*** exported variables *************************************************/


/* (none) */


/************************************************************************/
/*** prototype definitions for exported functions		      ***/
/************************************************************************/


	/* support functions for file I/O */

extern "C" FILE *GL_OpenFileForReading(	/* open a file for read access */
	const char	*Filename);	/* (supports "-" for stdin) */

extern "C" FILE *GL_OpenFileForWriting(	/* open a (new) file for write access */
	const char	*Filename);	/* (supports "-" for stdout) */

extern "C" FILE *GL_OpenFile(		/* open a file for read/write access */
	const char	*Filename);

extern "C" FILE *GL_OpenTmpFile(	/* open a new temp. file for writing */
	void);				/* (temp. file is deleted on close) */

extern "C" ERROR GL_RewindTmpFile(	/* rewind the temp. file to beginning */
	FILE		*FileHandle);	/* (allow read or write from start) */

extern "C" void GL_CloseFile(		/* close a file */
	FILE		*FileHandle);

extern "C" ERROR GL_CloseFileWithCheck(	/* close a file and check for errors */
	FILE		*FileHandle);

extern "C" ERROR GL_DeleteFile(		/* delete a file */
	const char	*Filename);

extern "C" ERROR GL_FileIsReadable(	/* check if file is read-accessable */
	const char	*Filename);	/* (supports "-" for stdin) */

extern "C" ERROR GL_FileIsCreatable(	/* check if file is write-accessable */
	const char	*Filename,	/* (supports "-" for stdout) */
	bool		Force = false);	/* (force: unlink if existing) */

extern "C" ERROR GL_ReadBytes(		/* read (binary) data from file */
	FILE		*FileHandle,
	void		*Bytes,
	size_t		NumBytes);

extern "C" ERROR GL_WriteBytes(		/* write (binary) data to file */
	FILE		*FileHandle,
	const void	*Bytes,
	size_t		NumBytes);

extern "C" ERROR GL_ReadEOF(		/* read (and check) EOF from file */
	FILE		*FileHandle);

extern "C" const char *GL_FindFileInDirList(/* find a file for read access */
	const char	*Filename,	/* (applying DirList and Suffix) */
	const char	*Suffix = NULL,	/* (returns NULL if not found) */
	gl_stringlist	*DirList = NULL);


#endif /* GL_FILEIO_H */

/* EOF GL_FileIO.h */
