// SigVectors.sc: (based on examples/simple/BitVectors.sc)
// --------------
//
// author:	Rainer Doemer
// last update:	01/06/03 (initial version 07/17/02)
//
// note:	this SpecC example demonstrates the use of signal bitvectors
//		(without making much more sense)


#include <stdio.h>
#include <stdlib.h>

#include "../../inc/sim.sh"	// #include <sim.sh>


behavior Reverse16(
	in  signal bit[15:0] In,
	out signal bit[15:0] Out)
{
    void main(void)
    {
	while(true)
	{
	    wait In;
	    Out = In[0:15];	// reverse slice access
	}
    }
};


behavior Adder8x16(
	in  signal bit[8] a,
	in  signal bit[16] b,
	out signal bit[16] c)
{
    void main(void)
    {
	while(true)
	{
	    wait a, b;
	    c = a + b;	// automatic extension from 8 to 16 bits for a
	}
    }
};


behavior Multiplier16x16(
	in  signal bit[16] a,
	in  signal bit[16] b,
	out signal bit[32] c)
{
    void main(void)
    {
	bit[32]		tmp_a,
			tmp_b;

	while(true)
	{
	    wait a, b;
	    tmp_a = a;		// automatic signed extension
	    tmp_b = b;
	    c = tmp_a * tmp_b;	// return full 32 bit result
	}
    }
};


behavior OnesCounter64(
	in  signal bit[64] b,
	out signal int Ones)
{
    void main(void)
    {
	int	i, n;

	while(true)
	{
	    wait b;
	    n = 0;
	    for(i=0; i<64; i++)
		n += b[i];
	    Ones = n;
	}
    }
};


behavior Encoder32(
	in  signal bit[31:0] In,
	out signal bit[31:0] Out,
	in  signal int Key)
{
    bit[7:0] ByteMagic(bit[3:0] Nibble1, bit[3:0] Nibble2)
    {
	bit[7:0] Tmp;

	Tmp = Nibble1[0:1] @ Nibble2[2:3] @ Nibble2[0:1] @ Nibble1[2:3];
	Tmp += 42;
	Tmp[3] = (Tmp[5] ^ 1) | (Tmp[1:2] @ Tmp[5:6])[2];
	Tmp[3:6] ^= Tmp[1:4];
	return(Tmp ^ 01010101b);
    }

    void main(void)
    {
	bit[31:0]	Tmp;

	while(true)
	{
	    wait In, Key;
	    Tmp = In + Key;
	    Tmp[ 0: 7] = ByteMagic(Tmp[ 7: 4], Tmp[ 3: 0]);
	    Tmp[ 8:15] = ByteMagic(Tmp[15:12], Tmp[11: 8]);
	    Tmp[16:23] = ByteMagic(Tmp[23:20], Tmp[19:16]);
	    Tmp[24:31] = ByteMagic(Tmp[31:28], Tmp[27:24]);
	    Out = Tmp;
	}
    }
};


behavior Design(
	in  signal bit[8]  InB8,
	in  signal bit[8]  InC8,
	in  signal bit[16] InA16,
	in  signal bit[64] InD64,
	out signal bit[32] Out32)
{
signal bit[16]	Bus16A, Bus16B, Bus16C;
signal bit[32]	Bus32;
signal int	I;

Reverse16	R1(InA16, Bus16A);
Adder8x16	A1(InB8, Bus16A, Bus16B),
		A2(InC8, Bus16A, Bus16C);
Multiplier16x16	M1(Bus16B, Bus16C, Bus32);
OnesCounter64	O1(InD64, I);
Encoder32	E1(Bus32, Out32, I);

    void main(void)
    {
	par
	{
	    R1.main();
	    A1.main();
	    A2.main();
	    M1.main();
	    O1.main();
	    E1.main();
	}
    }
};


// testbench

behavior Stimulator(
	out signal bit[8]  InB8,
	out signal bit[8]  InC8,
	out signal bit[16] InA16,
	out signal bit[64] InD64,
	inout long int Result)
{
    void main(void)
    {
	puts("Starting...");
	waitfor 1;

	InA16 = 1111000011110000b;
	waitfor 1;

	InB8  = 01110001b;
	waitfor 1;

	InC8  = 10000001b;
	waitfor 1;

	InD64 = 010101010000011111000000111110101010111110001111b;
	waitfor 1;

	puts("Exiting...");
	exit(Result != 42);
    }
};


behavior Monitor(
	in  signal bit[32] Out32,
	inout long int Result)
{
    void main(void)
    {
	const long int	Offset = 31819831 + 42;  // don't ask why!  :-)
	char		Buffer[33];

	while(true)
	{
	    wait Out32;
	    Result = Out32 + Offset;
	    printf("Time %s: Out32 = %s, Result = %ld\n",
			time2str(now()),
			ubit2str(2, &Buffer[32], Out32),
			Result);
	}
    }
};


behavior Main
{
    signal bit[8]  InB8, InC8;
    signal bit[16] InA16;
    signal bit[32] Out32;
    signal bit[64] InD64;
    long int       Result;

    Stimulator	s1(InB8, InC8, InA16, InD64, Result);
    Design	d1(InB8, InC8, InA16, InD64, Out32);
    Monitor	m1(Out32, Result);

    int main(void)
    {
	par
	{
	    s1.main();
	    d1.main();
	    m1.main();
	}
	return(0);	// never reached
    }
};

// EOF
