/* Statement.cc: SpecC Internal Representation, Statement Class		*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 10/20/97 */
/************************************************************************/

/* last update: 09/26/06 */

/* modifications: (most recent first)
 *
 * 09/26/06 PC  Adjustments for scrc 2.1
 * ....		.... allow timing labels in nested hierarchy!
 * 06/03/05 RD	reorganized and renamed global type names
 * 01/14/05 RD	added inclusion of IntRep/DelydAssgn.h
 * 10/26/04 RD	pass ((void*)0) instead of 0 as NULL for variable arguments
 *		(int and void* are different size on 64-bit architecture!)
 * 08/27/04 RD	bug fix: empty sensitivity in FSMD generates bad C++ code
 *		if asynchronous reset is specified
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 12/23/03 RD	bug fix in SIR_Statement::CheckFsmd()
 * 11/13/03 HY  added HAVE_METHOD support
 * 11/12/03 RD	bug fix: ParentStmnt pointer invalid in WrapInCompound()
 * 11/05/03 RD	added method SIR_FsmdStmnt::GetScope()
 * 11/04/03 RD	added semantic checking of 'fsmd' (CheckFsmd())
 * 11/03/03 RD	reworked fsmd state transition analysis
 * 10/30/03 RD	added method GetLoopOrSwitch() and some control-flow analysis
 * 10/29/03 RD	added member ParentStmnt to SIR_Statement
 * 10/24/03 RD	use SIR_Expression::IsInvalidInScope() to validate
 *		attachments of expressions to the statement tree
 * 10/23/03 RD	completed support for 'after' and 'piped' in fsmd statement
 * 10/03/03 RD	added member TmpVars to SIR_Statement
 * 09/12/03 RD	added support for 'after' and 'piped' in fsmd statement
 * 05/29/03 RD	bug fix: made iterator SIR_Statement::DFS_ForAllStatements()
 *		safe with respect to the deletion of the current leaf object
 *		(since it is used with conventions of API layer 2 iterators)
 * 04/02/03 RD	fixed CompoundSubBlocks() for 'fsm' and 'fsmd' statements
 * 04/01/03 RD	fixed code generation for 'fsmd' with internal clock
 * 02/26/03 RD	refined SIR_Statement::Print() for use with the GUI
 * 02/21/03 RD	fixed the dependency analysis among symbols with respect to
 *		the now supported local variables in functions
 * 02/18/03 RD	fixed deletion of default and reset states in 'fsmd' stmnt.
 * 02/13/03 RD	bug fix for annotations at labeled if branches before else
 * 02/11/03 RD	added C++ code generation for 'fsmd' statement
 * 02/07/03 RD	added SIR_STMNT_FSMD and related members and methods
 * 02/04/03 RD	bug fix in New() #3 for labeled statement
 * 01/03/03 RD	fixed SIR_RELAXED macro so that conversion from floating point
 *		types to bit vector and longlong types is handled explicitly
 * 01/03/03 RD	extended SIR_Expression::WriteSC() calls
 * 12/27/02 RD	extended SIR_Statement::IsIndirectDependant() to cover
 *		dependency of 'buffered' variables on clock/reset
 * 11/26/02 RD	added C++ code generation for edges of signals
 * 11/25/02 RD	added support for triggering edges of signals
 * 11/22/02 RD	switched event list from SIR_SymbolPtrs to SIR_Events
 * 09/19/02 RD	support for adding source code info during code generation
 * 08/14/02 RD	bug fix in C++ code generation for naming forked behaviors
 * 07/18/02 RD	added support for signals and buffers in code generation
 * 07/17/02 RD	fixed SIR_Statement::DFS_ForAllExpressions(), and
 *		SIR_Statement::DFS_FindDependant() for local states
 * 07/16/02 RD	fixed SIR_Statement::DFS_ForAllStatements() for local states
 * 07/16/02 RD	added handling for SIR_TYPE_SIGNAL and SIR_TYPE_BUFFER
 * 07/10/02 RD	adjusted code generation for FSMs with local states
 * 07/09/02 RD	added StateScope argument to FSM statement constructor
 * 06/26/02 RD	added 'wait' statement with AND semantics (SIR_STMNT_WAITAND)
 * 06/25/02 RD	reworked CheckCompoundBehavior() for SpecC version 2.0
 * 01/07/02 RD	bug fix: array copy code was missing some parentheses
 * 11/29/01 RD	bugfix with long long to bit vector conversions
 * 11/29/01 RD	bugfix with signed vs. unsigned long long conversions
 * 11/21/01 RD	took out default arguments from function definitions
 * 11/08/01 RD	switched code generation to use GL_IO layer
 * 11/01/01 RD	added indentation in code generation for '(' and ')'
 * 10/11/01 RD	improved overall formatting of generated code
 * 10/04/01 RD	improved indentation of generated code
 * 08/14/01 RD	added missing port-access checking at statement creation
 * 05/31/01 RD	eliminated level 2 of SIR API
 * 05/27/01 RD	added semantic checking for event port accesses
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 05/21/01 RD	added support for (native, non-native) array assignment
 * 05/12/01 RD	bug fix: generate correct C++ code for empty 'par' and 'pipe'
 * 04/30/01 RD	replaced use of obsolete form() from "stream.h" with own one
 * 04/18/01 RD	adjusted 'long long' constants for non-native longlong
 * 04/05/01 RD	added support for new 'pipe' construct (with flushing)
 * 03/23/01 RD	added SIR_Statement::WrapInCompound() and
 *		SIR_Statement::CompoundSubBlocks()
 * 03/20/01 RD	extended pipe construct to support termination
 * 01/30/01 RD	fixed a potential FMR problem in iterators
 *		SIR_Statements::DFS_ForAllXxxxx
 */


#include "IntRep/Statement.h"
#include "IntRep/Design.h"
#include "IntRep/Extern.h"

#include <assert.h>


/*** constants and macros ***********************************************/


	/* macro for behavior main calls */

#define SIR_CALLED_BEHAVIOR(Stmnt)	\
	((Stmnt)->Expression1->Arg1->Arg1->Symbol)


	/* macro for 'relaxed' type conversions */

// note: any change to SIR_RELAXED must also be applied to Expression.cc!
// (see SIR_RELAXED in Expression.cc for more info)

#ifdef HAVE_LLONG

#define SIR_RELAXED(expected, given)				\
	( (  (  ((expected) == SIR_TYPE_BIT)			\
	      ||((expected) == SIR_TYPE_UBIT))			\
	   &&(  ((given) != SIR_TYPE_FLOAT)			\
	      &&((given) != SIR_TYPE_DOUBLE)			\
	      &&((given) != SIR_TYPE_LONGDOUBLE)))		\
	 ? (SIR_TYPE_ANY_TYPE) : (expected))

#else /* !HAVE_LLONG */

#define SIR_RELAXED(expected, given)				\
	(  (  (  ((expected) == SIR_TYPE_BIT)			\
	       ||((expected) == SIR_TYPE_UBIT))			\
	    &&(  ((given) != SIR_TYPE_FLOAT)			\
	       &&((given) != SIR_TYPE_DOUBLE)			\
	       &&((given) != SIR_TYPE_LONGDOUBLE)		\
	       &&((given) != SIR_TYPE_LONGLONG)			\
	       &&((given) != SIR_TYPE_ULONGLONG)))		\
	 ||(  ((expected) == SIR_TYPE_LONGLONG)			\
	    &&(  ((given) != SIR_TYPE_BIT)			\
	       &&((given) != SIR_TYPE_UBIT)			\
	       &&((given) != SIR_TYPE_FLOAT)			\
	       &&((given) != SIR_TYPE_DOUBLE)			\
	       &&((given) != SIR_TYPE_LONGDOUBLE)		\
	       &&((given) != SIR_TYPE_ULONGLONG)))		\
	 ||(  ((expected) == SIR_TYPE_ULONGLONG)		\
	    &&(  ((given) != SIR_TYPE_BIT)			\
	       &&((given) != SIR_TYPE_UBIT)			\
	       &&((given) != SIR_TYPE_FLOAT)			\
	       &&((given) != SIR_TYPE_DOUBLE)			\
	       &&((given) != SIR_TYPE_LONGDOUBLE)		\
	       &&((given) != SIR_TYPE_LONGLONG)))		\
	 ? (SIR_TYPE_ANY_TYPE) : (expected))

#endif /* HAVE_LLONG */


/*** internal type declarations *****************************************/


	/* (none) */

/*** class implementations **********************************************/


	/*********************/
	/*** SIR_Statement ***/
	/*********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Statement::SIR_Statement(		/* constructor #1 */
	SIR_STMNT_TYPE	StmntType,
	sir_symbol	*Function,
	sir_expression	*Expression1 /* = NULL */,
	sir_statement	*Statement1 /* = NULL */,
	sir_statement	*Statement2 /* = NULL */,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */,
	sir_expression	*Expression2 /* = NULL */,
	sir_expression	*Expression3 /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(Function != NULL);

switch(StmntType)
   { case SIR_STMNT_LABELED:
     case SIR_STMNT_COMPOUND:
	{ assert(false);	/* wrong constructor */
	 }
     case SIR_STMNT_EXPRESSION:
	{ assert(Expression1 != NULL);
	  break;
	 }
     case SIR_STMNT_IF:
	{ assert(Expression1 != NULL);
	  assert(Statement1 != NULL);
	  break;
	 }
     case SIR_STMNT_IF_ELSE:
	{ assert(Expression1 != NULL);
	  assert(Statement1 != NULL);
	  assert(Statement2 != NULL);
	  break;
	 }
     case SIR_STMNT_SWITCH:
	{ assert(Expression1 != NULL);
	  assert(Statement1 != NULL);
	  break;
	 }
     case SIR_STMNT_CASE:
	{ assert(false);	/* wrong constructor */
	 }
     case SIR_STMNT_DEFAULT:
	{ assert(Statement1 != NULL);
	  break;
	 }
     case SIR_STMNT_WHILE:
     case SIR_STMNT_DO_WHILE:
	{ assert(Expression1 != NULL);
	  assert(Statement1 != NULL);
	  break;
	 }
     case SIR_STMNT_FOR:
	{ assert(Statement1 != NULL);
	  assert(Expression1 != NULL);
	  assert(Expression2 != NULL);
	  assert(Expression3 != NULL);
	  break;
	 }
     case SIR_STMNT_GOTO:
	{ assert(false);	/* wrong constructor */
	 }
     case SIR_STMNT_CONTINUE:
     case SIR_STMNT_BREAK:
	{ break;
	 }
     case SIR_STMNT_RETURN:
	{ assert(Expression1 != NULL);
	  break;
	 }
     case SIR_STMNT_PAR:
     case SIR_STMNT_PIPE:
	{ assert(Statement1 != NULL);
	  assert(  (  (Expression1 == NULL)
		    &&(Expression2 == NULL)
		    &&(Expression3 == NULL))
		 ||(  (Expression1 != NULL)
		    &&(Expression2 != NULL)
		    &&(Expression3 != NULL)));
	  break;
	 }
     case SIR_STMNT_EXCEPTION:
     case SIR_STMNT_TIMING:
     case SIR_STMNT_FSM:
     case SIR_STMNT_FSMD:
     case SIR_STMNT_WAIT:
     case SIR_STMNT_WAITAND:
	{ assert(false);	/* wrong constructor */
	 }
     case SIR_STMNT_WAITFOR:
	{ assert(Expression1 != NULL);
	  break;
	 }
     case SIR_STMNT_NOTIFY:
     case SIR_STMNT_NOTIFYONE:
	{ assert(false);	/* wrong constructor */
	 }
     default:
	{ assert(false);	/* bad StmntType */
	 }
    } /* hctiws */

SIR_Statement::StmntType	= StmntType;
SIR_Statement::Function		= Function;
SIR_Statement::ParentStmnt	= NULL;
SIR_Statement::Expression1	= Expression1;
SIR_Statement::Expression2	= Expression2;
SIR_Statement::Expression3	= Expression3;
SIR_Statement::Constant		= NULL;
SIR_Statement::Statement1	= Statement1;
SIR_Statement::Statement2	= Statement2;
SIR_Statement::Label		= NULL;
SIR_Statement::Scope		= NULL;
SIR_Statement::Statements	= NULL;
SIR_Statement::Events		= NULL;
SIR_Statement::Exceptions	= NULL;
SIR_Statement::Constraints	= NULL;
SIR_Statement::Transitions	= NULL;
SIR_Statement::Sensitivity	= NULL;
SIR_Statement::ResetSignal	= NULL;
SIR_Statement::ResetActiveHi	= false;
SIR_Statement::ResetState	= NULL;
SIR_Statement::DefaultStmnts	= NULL;
SIR_Statement::States		= NULL;
SIR_Statement::TmpVars		= NULL;
SIR_Statement::TmpVars		= NULL;

if (Statement1)
   { Statement1->ParentStmnt = this;	// automatically set the parent link
    } /* fi */
if (Statement2)
   { Statement2->ParentStmnt = this;	// automatically set the parent link
    } /* fi */

} /* end of SIR_Statement::SIR_Statement #1 */


SIR_Statement::SIR_Statement(		/* constructor #2 */
	SIR_STMNT_TYPE	StmntType,	/* (for compound statements) */
	sir_symbol	*Function,
	sir_symbols	*Scope,
	sir_statements	*Statements,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{
sir_statement	*Stmnt;

assert(StmntType == SIR_STMNT_COMPOUND);
assert(Function != NULL);
assert(Scope != NULL);
assert(Scope->ParentSymbol != NULL);	/* we always want a parent function */
/* Statements is temorary NULL in Parser */

SIR_Statement::StmntType	= StmntType;
SIR_Statement::Function		= Function;
SIR_Statement::ParentStmnt	= NULL;
SIR_Statement::Expression1	= NULL;
SIR_Statement::Expression2	= NULL;
SIR_Statement::Expression3	= NULL;
SIR_Statement::Constant		= NULL;
SIR_Statement::Statement1	= NULL;
SIR_Statement::Statement2	= NULL;
SIR_Statement::Label		= NULL;
SIR_Statement::Scope		= Scope;
SIR_Statement::Statements	= Statements;
SIR_Statement::Events		= NULL;
SIR_Statement::Exceptions	= NULL;
SIR_Statement::Constraints	= NULL;
SIR_Statement::Transitions	= NULL;
SIR_Statement::Sensitivity	= NULL;
SIR_Statement::ResetSignal	= NULL;
SIR_Statement::ResetActiveHi	= false;
SIR_Statement::ResetState	= NULL;
SIR_Statement::DefaultStmnts	= NULL;
SIR_Statement::States		= NULL;
SIR_Statement::TmpVars		= NULL;

Scope->ParentStmnt = this;	/* automatically fill in the back pointer */

if (Statements)
   { Statements->CmpndScope = Scope;	/* fill in compound scope link */
     Stmnt = Statements->First();
     while(Stmnt)
	{ Stmnt->ParentStmnt = this;	// automatically set the parent link
	  Stmnt = Stmnt->Succ();
	 } /* elihw */
    } /* fi */

} /* end of SIR_Statement::SIR_Statement #2 */


SIR_Statement::SIR_Statement(		/* constructor #3 */
	SIR_STMNT_TYPE	StmntType,	/* (for case statements) */
	sir_symbol	*Function,
	sir_constant	*Constant,
	sir_statement	*Statement1 /* = NULL */,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(StmntType == SIR_STMNT_CASE);
assert(Function != NULL);
assert(Constant != NULL);

SIR_Statement::StmntType	= StmntType;
SIR_Statement::Function		= Function;
SIR_Statement::ParentStmnt	= NULL;
SIR_Statement::Expression1	= NULL;
SIR_Statement::Expression2	= NULL;
SIR_Statement::Expression3	= NULL;
SIR_Statement::Constant		= Constant;
SIR_Statement::Statement1	= Statement1;
SIR_Statement::Statement2	= NULL;
SIR_Statement::Label		= NULL;
SIR_Statement::Scope		= NULL;
SIR_Statement::Statements	= NULL;
SIR_Statement::Events		= NULL;
SIR_Statement::Exceptions	= NULL;
SIR_Statement::Constraints	= NULL;
SIR_Statement::Transitions	= NULL;
SIR_Statement::Sensitivity	= NULL;
SIR_Statement::ResetSignal	= NULL;
SIR_Statement::ResetActiveHi	= false;
SIR_Statement::ResetState	= NULL;
SIR_Statement::DefaultStmnts	= NULL;
SIR_Statement::States		= NULL;
SIR_Statement::TmpVars		= NULL;

if (Statement1)
   { Statement1->ParentStmnt = this;	// automatically set the parent link
    } /* fi */

} /* end of SIR_Statement::SIR_Statement #3 */


SIR_Statement::SIR_Statement(		/* constructor #4 */
	SIR_STMNT_TYPE	StmntType,	/* (for statements with labels) */
	sir_symbol	*Function,
	sir_label	*Label,
	sir_statement	*Statement1 /* = NULL */,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(  (StmntType == SIR_STMNT_GOTO)
       ||(StmntType == SIR_STMNT_LABELED));
assert(Function != NULL);
assert(Label != NULL);

SIR_Statement::StmntType	= StmntType;
SIR_Statement::Function		= Function;
SIR_Statement::ParentStmnt	= NULL;
SIR_Statement::Expression1	= NULL;
SIR_Statement::Expression2	= NULL;
SIR_Statement::Expression3	= NULL;
SIR_Statement::Constant		= NULL;
SIR_Statement::Statement1	= Statement1;
SIR_Statement::Statement2	= NULL;
SIR_Statement::Label		= Label;
SIR_Statement::Scope		= NULL;
SIR_Statement::Statements	= NULL;
SIR_Statement::Events		= NULL;
SIR_Statement::Exceptions	= NULL;
SIR_Statement::Constraints	= NULL;
SIR_Statement::Transitions	= NULL;

SIR_Statement::Sensitivity	= NULL;
SIR_Statement::ResetSignal	= NULL;
SIR_Statement::ResetActiveHi	= false;
SIR_Statement::ResetState	= NULL;
SIR_Statement::DefaultStmnts	= NULL;
SIR_Statement::States		= NULL;
SIR_Statement::TmpVars		= NULL;

if (Statement1)
   { Statement1->ParentStmnt = this;	// automatically set the parent link
    } /* fi */

} /* end of SIR_Statement::SIR_Statement #4 */


SIR_Statement::SIR_Statement(		/* constructor #5 */
	SIR_STMNT_TYPE	StmntType,	/* (for statements with events) */
	sir_symbol	*Function,
	sir_events	*Events,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(  (StmntType == SIR_STMNT_WAIT)
       ||(StmntType == SIR_STMNT_WAITAND)
       ||(StmntType == SIR_STMNT_NOTIFY)
       ||(StmntType == SIR_STMNT_NOTIFYONE));
assert(Function != NULL);
assert(Events != NULL);

SIR_Statement::StmntType	= StmntType;
SIR_Statement::Function		= Function;
SIR_Statement::ParentStmnt	= NULL;
SIR_Statement::Expression1	= NULL;
SIR_Statement::Expression2	= NULL;
SIR_Statement::Expression3	= NULL;
SIR_Statement::Constant		= NULL;
SIR_Statement::Statement1	= NULL;
SIR_Statement::Statement2	= NULL;
SIR_Statement::Label		= NULL;
SIR_Statement::Scope		= NULL;
SIR_Statement::Statements	= NULL;
SIR_Statement::Events		= Events;
SIR_Statement::Exceptions	= NULL;
SIR_Statement::Constraints	= NULL;
SIR_Statement::Transitions	= NULL;
SIR_Statement::Sensitivity	= NULL;
SIR_Statement::ResetSignal	= NULL;
SIR_Statement::ResetActiveHi	= false;
SIR_Statement::ResetState	= NULL;
SIR_Statement::DefaultStmnts	= NULL;
SIR_Statement::States		= NULL;
SIR_Statement::TmpVars		= NULL;

} /* end of SIR_Statement::SIR_Statement #5 */


SIR_Statement::SIR_Statement(		/* constructor #6 */
	SIR_STMNT_TYPE	StmntType,	/* (for exception statements) */
	sir_symbol	*Function,
	sir_statement	*Statement1,
	sir_exceptions	*Exceptions,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{
sir_exception	*Except;

assert(StmntType == SIR_STMNT_EXCEPTION);
assert(Function != NULL);
assert(Statement1 != NULL);
assert(Exceptions != NULL);

SIR_Statement::StmntType	= StmntType;
SIR_Statement::Function		= Function;
SIR_Statement::ParentStmnt	= NULL;
SIR_Statement::Expression1	= NULL;
SIR_Statement::Expression2	= NULL;
SIR_Statement::Expression3	= NULL;
SIR_Statement::Constant		= NULL;
SIR_Statement::Statement1	= Statement1;
SIR_Statement::Statement2	= NULL;
SIR_Statement::Label		= NULL;
SIR_Statement::Scope		= NULL;
SIR_Statement::Statements	= NULL;
SIR_Statement::Events		= NULL;
SIR_Statement::Exceptions	= Exceptions;
SIR_Statement::Constraints	= NULL;
SIR_Statement::Transitions	= NULL;
SIR_Statement::Sensitivity	= NULL;
SIR_Statement::ResetSignal	= NULL;
SIR_Statement::ResetActiveHi	= false;
SIR_Statement::ResetState	= NULL;
SIR_Statement::DefaultStmnts	= NULL;
SIR_Statement::States		= NULL;
SIR_Statement::TmpVars		= NULL;

if (Statement1)
   { Statement1->ParentStmnt = this;	// automatically set the parent link
    } /* fi */
if (Exceptions)
   { Except = Exceptions->First();
     while(Except)
	{ if (Except->Handler)
	     { Except->Handler->ParentStmnt = this;	// set the parent link
	      } /* fi */
	  Except = Except->Succ();
	 } /* elihw */
    } /* fi */

} /* end of SIR_Statement::SIR_Statement #6 */


SIR_Statement::SIR_Statement(		/* constructor #7 */
	SIR_STMNT_TYPE	StmntType,	/* (for timing statement) */
	sir_symbol	*Function,
	sir_statement	*Statement1,
	sir_constraints	*Constraints,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(StmntType == SIR_STMNT_TIMING);
assert(Function != NULL);
assert(Statement1 != NULL);
assert(Constraints != NULL);

SIR_Statement::StmntType	= StmntType;
SIR_Statement::Function		= Function;
SIR_Statement::ParentStmnt	= NULL;
SIR_Statement::Expression1	= NULL;
SIR_Statement::Expression2	= NULL;
SIR_Statement::Expression3	= NULL;
SIR_Statement::Constant		= NULL;
SIR_Statement::Statement1	= Statement1;
SIR_Statement::Statement2	= NULL;
SIR_Statement::Label		= NULL;
SIR_Statement::Scope		= NULL;
SIR_Statement::Statements	= NULL;
SIR_Statement::Events		= NULL;
SIR_Statement::Exceptions	= NULL;
SIR_Statement::Constraints	= Constraints;
SIR_Statement::Transitions	= NULL;
SIR_Statement::Sensitivity	= NULL;
SIR_Statement::ResetSignal	= NULL;
SIR_Statement::ResetActiveHi	= false;
SIR_Statement::ResetState	= NULL;
SIR_Statement::DefaultStmnts	= NULL;
SIR_Statement::States		= NULL;
SIR_Statement::TmpVars		= NULL;

if (Statement1)
   { Statement1->ParentStmnt = this;	// automatically set the parent link
    } /* fi */

} /* end of SIR_Statement::SIR_Statement #7 */


SIR_Statement::SIR_Statement(		/* constructor #8 */
	SIR_STMNT_TYPE	StmntType,	/* (for fsm statement) */
	sir_symbol	*Function,
	sir_transitions	*Transitions,
	sir_symbols	*StateScope /* = NULL */,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(StmntType == SIR_STMNT_FSM);
assert(Function != NULL);
assert(Transitions != NULL);

SIR_Statement::StmntType	= StmntType;
SIR_Statement::Function		= Function;
SIR_Statement::ParentStmnt	= NULL;
SIR_Statement::Expression1	= NULL;
SIR_Statement::Expression2	= NULL;
SIR_Statement::Expression3	= NULL;
SIR_Statement::Constant		= NULL;
SIR_Statement::Statement1	= NULL;
SIR_Statement::Statement2	= NULL;
SIR_Statement::Label		= NULL;
SIR_Statement::Scope		= StateScope;
SIR_Statement::Statements	= NULL;
SIR_Statement::Events		= NULL;
SIR_Statement::Exceptions	= NULL;
SIR_Statement::Constraints	= NULL;
SIR_Statement::Transitions	= Transitions;
SIR_Statement::Sensitivity	= NULL;
SIR_Statement::ResetSignal	= NULL;
SIR_Statement::ResetActiveHi	= false;
SIR_Statement::ResetState	= NULL;
SIR_Statement::DefaultStmnts	= NULL;
SIR_Statement::States		= NULL;
SIR_Statement::TmpVars		= NULL;

if (StateScope)
   { StateScope->ParentStmnt = this;	/* fill in the back pointer */
    } /* fi */

} /* end of SIR_Statement::SIR_Statement #8 */


SIR_Statement::SIR_Statement(		/* constructor #9 */
	SIR_STMNT_TYPE	StmntType,	/* (for fsmd statement) */
	sir_symbol	*Function,
	sir_constant	*Constant,
	sir_events	*Events,
	sir_events	*Sensitivity,
	sir_symbol	*ResetSignal,
	bool		ResetActiveHi,
	sir_symbols	*Scope /* = NULL */,
	sir_expression	*Expression1 /*= NULL*/,
	sir_fsmdstmnt	*ResetState /* = NULL*/,
	sir_fsmdstmnt	*DefaultStmnts /*= NULL*/,
	sir_fsmdstates	*States /* = NULL */,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(StmntType == SIR_STMNT_FSMD);
assert(Function != NULL);

SIR_Statement::StmntType	= StmntType;
SIR_Statement::Function		= Function;
SIR_Statement::ParentStmnt	= NULL;
SIR_Statement::Expression1	= Expression1;
SIR_Statement::Expression2	= NULL;
SIR_Statement::Expression3	= NULL;
SIR_Statement::Constant		= Constant;
SIR_Statement::Statement1	= NULL;
SIR_Statement::Statement2	= NULL;
SIR_Statement::Label		= NULL;
SIR_Statement::Scope		= Scope;
SIR_Statement::Statements	= NULL;
SIR_Statement::Events		= Events;
SIR_Statement::Exceptions	= NULL;
SIR_Statement::Constraints	= NULL;
SIR_Statement::Transitions	= NULL;
SIR_Statement::Sensitivity	= Sensitivity;
SIR_Statement::ResetSignal	= ResetSignal;
SIR_Statement::ResetActiveHi	= ResetActiveHi;
SIR_Statement::ResetState	= ResetState;
SIR_Statement::DefaultStmnts	= DefaultStmnts;
SIR_Statement::States		= States;
SIR_Statement::TmpVars		= NULL;

if (Scope)
   { Scope->ParentStmnt = this;	/* fill in the back pointer */
    } /* fi */

} /* end of SIR_Statement::SIR_Statement #9 */


SIR_Statement::SIR_Statement(		/* constructor #11 (duplicator) */
	sir_statement	*Original,
	sir_statement	*ParentStmnt /* = NULL */) :
		SIR_Node(Original)
{

SIR_Statement::StmntType	= Original->StmntType;
SIR_Statement::Function		= Original->Function;
SIR_Statement::ParentStmnt	= ParentStmnt;
SIR_Statement::Expression1	= Original->Expression1 ?
					new SIR_Expression(
						Original->Expression1) :
					NULL;
SIR_Statement::Expression2	= Original->Expression2 ?
					new SIR_Expression(
						Original->Expression2) :
					NULL;
SIR_Statement::Expression3	= Original->Expression3 ?
					new SIR_Expression(
						Original->Expression3) :
					NULL;
SIR_Statement::Constant		= Original->Constant ?
					new SIR_Constant(
						Original->Constant) :
					NULL;
SIR_Statement::Statement1	= Original->Statement1 ?
					new SIR_Statement(
						Original->Statement1, this) :
					NULL;
SIR_Statement::Statement2	= Original->Statement2 ?
					new SIR_Statement(
						Original->Statement2, this) :
					NULL;
SIR_Statement::Label		= Original->Label;
SIR_Statement::Scope		= Original->Scope ?
					new SIR_Symbols(
						Original->Scope) :
					NULL;
SIR_Statement::Statements	= Original->Statements ?
					new SIR_Statements(
						Original->Statements, this) :
					NULL;
SIR_Statement::Events		= Original->Events ?
					new SIR_Events(
						Original->Events) :
					NULL;
SIR_Statement::Exceptions	= Original->Exceptions ?
					new SIR_Exceptions(
						Original->Exceptions, this) :
					NULL;
SIR_Statement::Constraints	= Original->Constraints ?
					new SIR_Constraints(
						Original->Constraints) :
					NULL;
SIR_Statement::Transitions	= Original->Transitions ?
					new SIR_Transitions(
						Original->Transitions) :
					NULL;
SIR_Statement::Sensitivity	= Original->Sensitivity ?
					new SIR_Events(
						Original->Sensitivity) :
					NULL;
SIR_Statement::ResetSignal	= Original->ResetSignal;
SIR_Statement::ResetActiveHi	= Original->ResetActiveHi;
SIR_Statement::ResetState	= Original->ResetState ?
					new SIR_FsmdStmnt(
						Original->ResetState) :
					NULL;
SIR_Statement::DefaultStmnts	= Original->DefaultStmnts ?
					new SIR_FsmdStmnt(
						Original->DefaultStmnts) :
					NULL;
SIR_Statement::States		= Original->States ?
					new SIR_FsmdStates(
						Original->States) :
					NULL;
SIR_Statement::TmpVars		= NULL;

} /* end of SIR_Statement::SIR_Statement #11 */


SIR_Statement::~SIR_Statement(void)	/* destructor */
{

delete Expression1;
delete Expression2;
delete Expression3;
delete Constant;
delete Statement1;
delete Statement2;
delete Scope;
delete Statements;
delete Events;
delete Exceptions;
delete Constraints;
delete Transitions;
delete Sensitivity;
delete ResetState;
delete DefaultStmnts;
delete States;

} /* end of SIR_Statement::~SIR_Statement */


bool SIR_Statement::IsCleanBehaviorCall( /* checks for "clean" behavior call */
	void)					/* (e.g. "b1.main();") */
{
bool	Test;

Test = (  (StmntType == SIR_STMNT_EXPRESSION)
	&&(Expression1->ExprType == SIR_EXPR_FUNCTION_CALL)
	&&(Expression1->Arg1->ExprType == SIR_EXPR_MEMBER_ACCESS)
	&&(Expression1->Arg1->Arg1->Type->Type == SIR_TYPE_BEHAVIOR)
	&&(Expression1->Arg1->Arg1->ExprType == SIR_EXPR_IDENTIFIER)
	&&(Expression1->Arg1->Symbol == NULL)	/* means: "main"! */
	&&(Expression1->Args->Empty()));

if (Test)
   { assert(SIR_CALLED_BEHAVIOR(this) != NULL);
    } /* fi */

return(Test);

} /* end of SIR_Statement::IsCleanBehaviorCall */


bool SIR_Statement::IsCleanListOfBehaviorCalls(	/* check "clean" comp. bhvr. */
	bool	OneCallMax /* = false */,     /* "{ b1.main(); b2.main(); }" */
	bool	OneCallMin /* = true */)
{
sir_statement	*Stmnt;

if (StmntType != SIR_STMNT_COMPOUND)
   { return(false);
    } /* fi */

assert(Scope != NULL);
assert(Statements != NULL);

if (  (! Scope->Empty())
    ||(! Scope->UserTypes->Empty()))
   { return(false);
    } /* fi */

if (  (OneCallMin)
    &&(Statements->NumElements() < 1))
   { return(false);
    } /* fi */
if (  (OneCallMax)
    &&(Statements->NumElements() > 1))
   { return(false);
    } /* fi */

Stmnt = Statements->First();
while(Stmnt)
   { if (! Stmnt->IsCleanBehaviorCall())
	{ return(false);
	 } /* fi */
     Stmnt = Stmnt->Succ();
    } /* elihw */

return(true);	/* looks clean */

} /* end of SIR_Statement::IsCleanListOfBehaviorCalls */


ERROR SIR_Statement::CheckCompoundBehavior( /* checks for "clean" beh. calls */
	bool	OneCallMax /* = false */,   /* "{ b1.main(); b2.main(); }" */
	bool	OneCallMin /* = true */)    /* (returns error if not clean) */
{

assert(StmntType == SIR_STMNT_COMPOUND);

if (! IsCleanListOfBehaviorCalls(OneCallMax, OneCallMin))
   { if (OneCallMax)
	{ if (OneCallMin)
	     { SIR_ErrMsg = "Compound statement restricted to "
				"exactly one behavior instance call";
	      } /* fi */
	  else
	     { SIR_ErrMsg = "Compound statement restricted to "
				"zero or one behavior instance calls";
	      } /* esle */
	 } /* fi */
     else
	{ if (OneCallMin)
	     { SIR_ErrMsg = "Compound statement restricted to "
				"a non-empty set of behavior instance calls";
	      } /* fi */
	  else
	     { SIR_ErrMsg = "Compound statement restricted to "
				"a set of behavior instance calls";
	      } /* esle */
	 } /* esle */
     return(SIR_ERROR_COMPOUND_STMNT_NOT_CLEAN);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::CheckCompoundBehavior */


bool SIR_Statement::IsDirectDependant(	/* checks if stmnt. depends on symbol */
	sir_symbol	*ThatSymbol)	/* (directly, as statement) */
{
sir_event	*Event;
sir_exception	*Except;
sir_transition	*Trans;

/* Function is a _back_ link, so ignore it */

if (Events)
   { Event = Events->First();
     while(Event)
	{ if (Event->Symbol == ThatSymbol)
	     { return(true);		/* references that event */
	      } /* fi */
	  Event = Event->Succ();
	 } /* elihw */
    } /* fi */

if (Exceptions)
   { Except = Exceptions->First();
     while(Except)
	{ assert(Except->Events != NULL);
	  Event = Except->Events->First();
	  while(Event)
	     { if (Event->Symbol == ThatSymbol)
		  { return(true);	/* references that event */
		   } /* fi */
	       Event = Event->Succ();
	      } /* elihw */
	  Except = Except->Succ();
	 } /* elihw */
    } /* fi */

if (Transitions)
   { Trans = Transitions->First();
     while(Trans)
	{ if (  (Trans->CurrState == ThatSymbol)
	      ||(Trans->NextState == ThatSymbol))
	     { return(true);		/* references that state */
	      } /* fi */
	  Trans = Trans->Succ();
	 } /* elihw */
    } /* fi */

if (Sensitivity)
   { Event = Sensitivity->First();
     while(Event)
	{ if (Event->Symbol == ThatSymbol)
	     { return(true);		/* references that event */
	      } /* fi */
	  Event = Event->Succ();
	 } /* elihw */
    } /* fi */

if (ResetSignal == ThatSymbol)
   { return(true);
    } /* fi */

return(false);	/* independent! */

} /* end of SIR_Statement::IsDirectDependant */


bool SIR_Statement::IsIndirectDependant(/* checks if stmnt. depends on symbol */
	sir_symbol	*ThatSymbol,	/* (indirectly, through expression) */
	SIR_DEPENDENCY	*Reason /* = NULL */,
	sir_expression	**DepExpr /* = NULL */)
{
sir_transition	*Trans;

if (Reason)
   { *Reason = SIR_DEP_NONE;
    } /* fi */
if (DepExpr)
   { *DepExpr = NULL;
    } /* fi */

if (Expression1)
   { if (Expression1->DFS_FindDependant(ThatSymbol, DepExpr))
	{ if (Reason)
	     { *Reason = SIR_DEP_EXPRESSION;
	      } /* fi */
	  return(true);
	 } /* fi */
    } /* fi */
if (Expression2)
   { if (Expression2->DFS_FindDependant(ThatSymbol, DepExpr))
	{ if (Reason)
	     { *Reason = SIR_DEP_EXPRESSION;
	      } /* fi */
	  return(true);
	 } /* fi */
    } /* fi */
if (Expression3)
   { if (Expression3->DFS_FindDependant(ThatSymbol, DepExpr))
	{ if (Reason)
	     { *Reason = SIR_DEP_EXPRESSION;
	      } /* fi */
	  return(true);
	 } /* fi */
    } /* fi */
if (Transitions)
   { Trans = Transitions->First();
     while(Trans)
	{ if (Trans->Condition)
	     { if (Trans->Condition->DFS_FindDependant(ThatSymbol, DepExpr))
		  { if (Reason)
		       { *Reason = SIR_DEP_EXPRESSION;
			} /* fi */
		    return(true);
		   } /* fi */
	      } /* fi */
	  Trans = Trans->Succ();
	 } /* elihw */
    } /* fi */

return(false);	/* independent! */

} /* end of SIR_Statement::IsIndirectDependant */


ERROR SIR_Statement::CheckFsmd(		/* semantically checks 'fsmd' stmnt. */
	void)				/* (FSMD only) */
{
sir_fsmdstate	*State;
sir_state_ptrs	*NextStates;
sir_state_ptr	*StatePtr;
gl_string	Location;
ERROR		Error;

assert(StmntType == SIR_STMNT_FSMD);	// FSMD only!

/* first make sure control flow is clean, otherwise code below will fail */
/* (bug fix 12/23/03, RD) */

if ((Error = DFS_ForAllFsmdStmnts(&SIR_FsmdStmnt::CheckControlFlow, NULL)))
   { return(Error);
    } /* fi */

if (ResetState)
   { NextStates = ResetState->ReachableStates(new SIR_StatePtrs());
     if (NextStates->Empty())
	{ if (ResetState->LineInfo)
	     { Location.form("line %u, file \"%s\"",
			ResetState->LineInfo->Line,
			ResetState->LineInfo->File->Filename.chars());
	      } /* fi */
	  else
	     { Location = "line <unknown>, file <unknown>";
	      } /* esle */
	  GL_PrintWarningFmt(GL_WARN_STANDARD,
		"'fsmd' reset state has no next state specified"
							GL_WARN_MSG_NEWLINE
		"(%s)",
		Location.chars());
	 } /* fi */
     delete NextStates;
    } /* fi */

if (  (!States)
    ||(States->Empty()))
   { if (LineInfo)
	{ Location.form("line %u, file \"%s\"",
			LineInfo->Line, LineInfo->File->Filename.chars());
	 } /* fi */
     else
	{ Location = "line <unknown>, file <unknown>";
	 } /* esle */
     GL_PrintWarningFmt(GL_WARN_STANDARD,
		"'fsmd' statement has no regular states" GL_WARN_MSG_NEWLINE
		"(%s)",
		Location.chars());
    } /* fi */
else
   {
#ifndef NDEBUG
     State = States->First();	// ensure that everybody is unused
     while(State)
	{ assert(State->NextStates == NULL);
	  State = State->Succ();
	 } /* elihw */
#endif /* NDEBUG */
     NextStates = new SIR_StatePtrs(		// starting states are
			new SIR_StatePtr(	// initial state and ...
				States->First()));
     if (ResetState)
	{ NextStates = NextStates->Join(	// ... any states after reset
			ResetState->ReachableStates(
				new SIR_StatePtrs()));
	 } /* fi */
     StatePtr = NextStates->First();
     while(StatePtr)				// create reachable trees
	{ if (StatePtr->State)			// from these starting points
	     { StatePtr->State->CreateReachableTree();
	      } /* fi */
	  StatePtr = StatePtr->Succ();
	 } /* elihw */
     delete NextStates;
     State = States->First();	// check if every state has been reached
     while(State)
	{ if (State->NextStates)
	     { // state has been reached!
	       delete State->NextStates;	// clean up
	       State->NextStates = NULL;
	      } /* fi */
	  else
	     { // state has not been reached!
	       if (State->LineInfo)
		  { Location.form("line %u, file \"%s\"",
				State->LineInfo->Line,
				State->LineInfo->File->Filename.chars());
		   } /* fi */
	       else
		  { Location = "line <unknown>, file <unknown>";
		   } /* esle */
	       GL_PrintWarningFmt(GL_WARN_STANDARD,
			"'fsmd' state %s is never reached" GL_WARN_MSG_NEWLINE
			"(%s)",
			State->StateName.chars(),
			Location.chars());
	      } /* esle */
	  State = State->Succ();
	 } /* elihw */
    } /* esle */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::CheckFsmd */


sir_statements *SIR_Statement::GetList(	/* determines the list of this stmnt. */
	void)
{

return((sir_statements*) Head());

} /* end of SIR_Statement::GetList */


ERROR SIR_Statement::DFS_ForAllNodes(	/* iterator over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{

if ((SIR_Error = (this->*MemberFct)(MemberFctArg)))	/* process this node */
   { return(SIR_Error);
    } /* fi */

if (Expression1)
   { if ((SIR_Error = Expression1->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Expression2)
   { if ((SIR_Error = Expression2->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Expression3)
   { if ((SIR_Error = Expression3->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Constant)
   { if ((SIR_Error = Constant->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statement1)
   { if ((SIR_Error = Statement1->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statement2)
   { if ((SIR_Error = Statement2->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Scope)
   { if ((SIR_Error = Scope->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statements)
   { if ((SIR_Error = Statements->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Exceptions)
   { if ((SIR_Error = Exceptions->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Constraints)
   { if ((SIR_Error = Constraints->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Transitions)
   { if ((SIR_Error = Transitions->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (ResetState)
   { if ((SIR_Error = ResetState->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (DefaultStmnts)
   { if ((SIR_Error = DefaultStmnts->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (States)
   { if ((SIR_Error = States->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::DFS_ForAllNodes */


ERROR SIR_Statement::DFS_ForAllSymbols(	/* iterator over all symbols */
	sir_symbol_mptr	MemberFct,	/* (depth first) */
	sir_symbol_marg	MemberFctArg)
{

/* this is not a symbol */

/* there are no symbols in Expression1 */
/* there are no symbols in Expression2 */
/* there are no symbols in Expression3 */
/* there are no symbols in Constant */
if (Statement1)
   { if ((SIR_Error = Statement1->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statement2)
   { if ((SIR_Error = Statement2->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Scope)
   { if ((SIR_Error = Scope->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statements)
   { if ((SIR_Error = Statements->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no symbols in Events */
if (Exceptions)
   { if ((SIR_Error = Exceptions->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no symbols in Constraints */
/* there are no symbols in Transitions */
/* there are no symbols in Sensitivity */
if (ResetState)
   { if ((SIR_Error = ResetState->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (DefaultStmnts)
   { if ((SIR_Error = DefaultStmnts->DFS_ForAllSymbols(MemberFct,MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (States)
   { if ((SIR_Error = States->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::DFS_ForAllSymbols */


ERROR SIR_Statement::DFS_ForAllUserTypes( /* iterator over all usertypes */
	sir_usertp_mptr	MemberFct,	/* (depth first) */
	sir_usertp_marg	MemberFctArg)
{

/* this is not a usertype */

/* there are no usertypes in Expression1 */
/* there are no usertypes in Expression2 */
/* there are no usertypes in Expression3 */
/* there are no usertypes in Constant */
if (Statement1)
   { if ((SIR_Error = Statement1->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statement2)
   { if ((SIR_Error = Statement2->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Scope)
   { if ((SIR_Error = Scope->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statements)
   { if ((SIR_Error = Statements->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no usertypes in Events */
if (Exceptions)
   { if ((SIR_Error = Exceptions->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no usertypes in Constraints */
/* there are no usertypes in Transitions */
/* there are no usertypes in Sensitivity */
if (ResetState)
   { if ((SIR_Error = ResetState->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (DefaultStmnts)
   { if ((SIR_Error = DefaultStmnts->DFS_ForAllUserTypes(MemberFct,
								MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (States)
   { if ((SIR_Error = States->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::DFS_ForAllUserTypes */


ERROR SIR_Statement::DFS_ForAllNotes( /* iterator over all notes */
	sir_note_mptr	MemberFct,	/* (depth first) */
	sir_note_marg	MemberFctArg)
{

/* this is not a note */

/* there are no notes in Expression1 */
/* there are no notes in Expression2 */
/* there are no notes in Expression3 */
/* there are no notes in Constant */
if (Statement1)
   { if ((SIR_Error = Statement1->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statement2)
   { if ((SIR_Error = Statement2->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Scope)
   { if ((SIR_Error = Scope->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statements)
   { if ((SIR_Error = Statements->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no notes in Events */
if (Exceptions)
   { if ((SIR_Error = Exceptions->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no notes in Constraints */
/* there are no notes in Transitions */
/* there are no notes in Sensitivity */
if (ResetState)
   { if ((SIR_Error = ResetState->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (DefaultStmnts)
   { if ((SIR_Error = DefaultStmnts->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (States)
   { if ((SIR_Error = States->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::DFS_ForAllNotes */


ERROR SIR_Statement::DFS_ForAllStatements( /* iterator over all statements */
	sir_stmnt_mptr	MemberFct,	/* (depth first) */
	sir_stmnt_marg	MemberFctArg)
{
sir_symbol	*LocalState;
bool		IsLeaf;

IsLeaf = !(Statement1 || Statement2 || Scope || Statements || Exceptions);

if ((SIR_Error = (this->*MemberFct)(MemberFctArg)))	/* process this node */
   { return(SIR_Error);
    } /* fi */

if (IsLeaf)
   { // it is allowed to delete leaf objects within this iterator;
     // in this case, we must not access any members any more,
     // because that would be a free-memory-access violation
     return(SIR_ERROR_NO_ERROR);
    } /* fi */

/* there are no statements in Expression1 */
/* there are no statements in Expression2 */
/* there are no statements in Expression3 */
/* there are no statements in Constant */
if (Statement1)
   { if ((SIR_Error = Statement1->DFS_ForAllStatements(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statement2)
   { if ((SIR_Error = Statement2->DFS_ForAllStatements(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Scope)
   { if (StmntType == SIR_STMNT_FSM)
	{ /* special case: FSM with local states */
	  LocalState = Scope->First();
	  while(LocalState)
	     { assert(LocalState->Type->Type == SIR_TYPE_VOID);
	       assert(LocalState->FctBody != NULL);
	       if ((SIR_Error = LocalState->FctBody->DFS_ForAllStatements(
						MemberFct, MemberFctArg)))
		  { return(SIR_Error);
		   } /* fi */
	       LocalState = LocalState->Succ();
	      } /* elihw */
	 } /* fi */
     else
	{ /* there are no statements in Scope */
	 } /* esle */
    } /* fi */
if (Statements)
   { if ((SIR_Error = Statements->DFS_ForAllStatements(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no statements in Events */
if (Exceptions)
   { if ((SIR_Error = Exceptions->DFS_ForAllStatements(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no statements in Constraints */
/* there are no statements in Transitions */
/* there are no statements in Sensitivity */
/* there are no statements in ResetState (only _fsmd_ statements) */
/* there are no statements in DefaultStmnts (only _fsmd_ statements) */
/* there are no statements in States (only _fsmd_ statements) */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::DFS_ForAllStatements */


ERROR SIR_Statement::DFS_ForAllFsmdStmnts( /* iterator over all fsmd stmnts. */
	sir_fsmdstmnt_mptr	MemberFct,	/* (depth first) */
	sir_fsmdstmnt_marg	MemberFctArg)	/* (SIR_STMNT_FSMD only!) */
{

assert(StmntType == SIR_STMNT_FSMD);	/* see restriction above */

// note: to iterate over all fsmd statements in, say, a design,
//       you can iterate first over all general statements and then,
//       whenever a SIR_STMNT_FSMD is found, iterate further down
//       using this iterator

/* there are no direct fsmd statements in Expression1 */
/* there are no direct fsmd statements in Expression2 */
/* there are no direct fsmd statements in Expression3 */
/* there are no direct fsmd statements in Constant */
/* there are no direct fsmd statements in Statement1 */
/* there are no direct fsmd statements in Statement2 */
/* there are no direct fsmd statements in Scope */
/* there are no direct fsmd statements in Statements */
/* there are no direct fsmd statements in Events */
/* there are no direct fsmd statements in Exceptions */
/* there are no direct fsmd statements in Constraints */
/* there are no direct fsmd statements in Transitions */
/* there are no direct fsmd statements in Sensitivity */

if (ResetState)
   { if ((SIR_Error = ResetState->DFS_ForAllFsmdStmnts(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (DefaultStmnts)
   { if ((SIR_Error = DefaultStmnts->DFS_ForAllFsmdStmnts(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (States)
   { if ((SIR_Error = States->DFS_ForAllFsmdStmnts(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::DFS_ForAllFsmdStmnts */


ERROR SIR_Statement::DFS_ForAllExpressions( /* iterator over all expressions */
	sir_expr_mptr	MemberFct,		/* (depth first) */
	sir_expr_marg	MemberFctArg)
{
sir_symbol	*LocalState;

/* this is not an expression */

if (Expression1)
   { if ((SIR_Error = Expression1->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Expression2)
   { if ((SIR_Error = Expression2->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Expression3)
   { if ((SIR_Error = Expression3->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no expressions in Constant */
if (Statement1)
   { if ((SIR_Error = Statement1->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statement2)
   { if ((SIR_Error = Statement2->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Scope)
   { if (StmntType == SIR_STMNT_FSM)
	{ /* special case: FSM with local states */
	  LocalState = Scope->First();
	  while(LocalState)
	     { assert(LocalState->Type->Type == SIR_TYPE_VOID);
	       assert(LocalState->FctBody != NULL);
	       if ((SIR_Error = LocalState->FctBody->DFS_ForAllExpressions(
						MemberFct, MemberFctArg)))
		  { return(SIR_Error);
		   } /* fi */
	       LocalState = LocalState->Succ();
	      } /* elihw */
	 } /* fi */
     else
	{ /* there are no expressions in Scope */
	 } /* esle */
    } /* fi */
if (Statements)
   { if ((SIR_Error = Statements->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no expressions in Events */
if (Exceptions)
   { if ((SIR_Error = Exceptions->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no expressions in Constraints */
if (Transitions)
   { if ((SIR_Error = Transitions->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no expressions in Sensitivity */
if (ResetState)
   { if ((SIR_Error = ResetState->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (DefaultStmnts)
   { if ((SIR_Error = DefaultStmnts->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (States)
   { if ((SIR_Error = States->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::DFS_ForAllExpressions */


bool SIR_Statement::DFS_FindDirectDependant(	/* search for direct deps. */
	sir_symbol	**SkipSymbol,
	sir_symbol	*ThatSymbol,
	sir_symbol	**DepSymbol,
	SIR_DEPENDENCY	*Reason)
{
sir_symbol	*Symbol,
		*LocalState;
sir_fsmdstate	*State;

assert(SkipSymbol != NULL);
assert(ThatSymbol != NULL);

if (  (Scope)
    &&(*SkipSymbol)
    &&(! Scope->IsAncestorOf(*SkipSymbol)))
   { return(false);	/* skip this symbol table (prune the subtree) */
    } /* fi */

if (Scope)
   { if (StmntType != SIR_STMNT_FSM)
	{ /* check dependency of 'buffered' variables in local scopes */
	  Symbol = Scope->First();
	  while(Symbol)
	     { if (*SkipSymbol)
		  { if (Symbol == *SkipSymbol)
		       { *SkipSymbol = NULL;	/* no more skipping */
			} /* fi */
		   } /* fi */
	       else
		  { if (Symbol->IsDirectDependant(ThatSymbol, Reason))
		       { assert(  (! Reason)
				||(*Reason == SIR_DEP_CLOCK)
				||(*Reason == SIR_DEP_RESET));
			 if (DepSymbol)
			    { *DepSymbol = Symbol;
			     } /* fi */
			 return(true);
			} /* fi */
		   } /* esle */
	       Symbol = Symbol->Succ();
	      } /* elihw */
	 } /* fi */
     else
	{ /* symbols in scope are local FSM states */
	 } /* esle */
    } /* fi */

/* there are no statements in Expression1 */
/* there are no statements in Expression2 */
/* there are no statements in Expression3 */
/* there are no statements in Constant */
if (Statement1)
   { if (Statement1->DFS_FindDirectDependant(SkipSymbol, ThatSymbol,
						DepSymbol, Reason))
	{ return(true);
	 } /* fi */
    } /* fi */
if (Statement2)
   { if (Statement2->DFS_FindDirectDependant(SkipSymbol, ThatSymbol,
						DepSymbol, Reason))
	{ return(true);
	 } /* fi */
    } /* fi */
if (Scope)
   { if (StmntType == SIR_STMNT_FSM)
	{ /* special case: FSM with local states */
	  LocalState = Scope->First();
	  while(LocalState)
	     { assert(LocalState->Type->Type == SIR_TYPE_VOID);
	       assert(LocalState->FctBody != NULL);
	       if (LocalState->FctBody->DFS_FindDirectDependant(
						SkipSymbol, ThatSymbol,
						DepSymbol, Reason))
		  { return(true);
		   } /* fi */
	       LocalState = LocalState->Succ();
	      } /* elihw */
	 } /* fi */
     else
	{ /* there are no statements in Scope */
	 } /* esle */
    } /* fi */
if (Statements)
   { if (Statements->DFS_FindDirectDependant(SkipSymbol, ThatSymbol,
						DepSymbol, Reason))
	{ return(true);
	 } /* fi */
    } /* fi */
/* there are no statements in Events */
/* there are statements, but no direct dependants in Exceptions */
/* there are no statements in Constraints */
/* there are no statements in Transitions */
/* there are no statements in Sensitivity */
if (ResetState)
   { if (ResetState->DFS_FindDirectDependant(SkipSymbol, ThatSymbol,
						DepSymbol, Reason))
	{ return(true);
	 } /* fi */
    } /* fi */
if (DefaultStmnts)
   { if (DefaultStmnts->DFS_FindDirectDependant(SkipSymbol, ThatSymbol,
						DepSymbol, Reason))
	{ return(true);
	 } /* fi */
    } /* fi */
if (States)
   { State = States->First();
     while(State)
	{ if (State->CmpndStmnt->DFS_FindDirectDependant(SkipSymbol, ThatSymbol,
						DepSymbol, Reason))
	     { return(true);
	      } /* fi */
	  State = State->Succ();
	 } /* elihw */
    } /* fi */

return(false);

} /* end of SIR_Statement::DFS_FindDirectDependant */


bool SIR_Statement::DFS_FindIndirectDependant(	/* searches indirect deps. */
	sir_symbol	*ThatSymbol,
	sir_statement	**DepStmnt,
	sir_fsmdstmnt	**DepFsmdStmnt,
	sir_expression	**DepExpr,
	SIR_DEPENDENCY	*Reason)
{
sir_exception	*Except;
sir_symbol	*LocalState;
sir_fsmdstate	*State;

assert(ThatSymbol != NULL);

if (IsDirectDependant(ThatSymbol))	/* check this statement */
   { if (DepStmnt)			/* (for statement dep.) */
	{ *DepStmnt = this;
	 } /* fi */
     if (DepFsmdStmnt)
	{ *DepFsmdStmnt = NULL;
	 } /* fi */
     if (DepExpr)
	{ *DepExpr = NULL;
	 } /* fi */
     if (Reason)
	{ *Reason = SIR_DEP_STATEMENT;
	 } /* fi */
     return(true);
    } /* fi */

if (IsIndirectDependant(ThatSymbol, Reason,	/* check this statement */
			DepExpr))		/* (for expression dep.) */
   { if (DepStmnt)
	{ *DepStmnt = this;
	 } /* fi */
     if (DepFsmdStmnt)
	{ *DepFsmdStmnt = NULL;
	 } /* fi */
     return(true);
    } /* fi */

/* there are no statements in Expression1 */
/* there are no statements in Expression2 */
/* there are no statements in Expression3 */
/* there are no statements in Constant */
if (Statement1)
   { if (Statement1->DFS_FindIndirectDependant(ThatSymbol, DepStmnt,
						DepFsmdStmnt, DepExpr,
						Reason))
	{ return(true);
	 } /* fi */
    } /* fi */
if (Statement2)
   { if (Statement2->DFS_FindIndirectDependant(ThatSymbol, DepStmnt,
						DepFsmdStmnt, DepExpr,
						Reason))
	{ return(true);
	 } /* fi */
    } /* fi */
if (Scope)
   { if (StmntType == SIR_STMNT_FSM)
	{ /* special case: FSM with local states */
	  LocalState = Scope->First();
	  while(LocalState)
	     { assert(LocalState->Type->Type == SIR_TYPE_VOID);
	       assert(LocalState->FctBody != NULL);
	       if (LocalState->FctBody->DFS_FindIndirectDependant(ThatSymbol,
						DepStmnt, DepFsmdStmnt,
						DepExpr, Reason))
		  { return(true);
		   } /* fi */
	       LocalState = LocalState->Succ();
	      } /* elihw */
	 } /* fi */
     else
	{ /* there are no statements in Scope */
	 } /* esle */
    } /* fi */
if (Statements)
   { if (Statements->DFS_FindIndirectDependant(ThatSymbol, DepStmnt,
						DepFsmdStmnt, DepExpr,
						Reason))
	{ return(true);
	 } /* fi */
    } /* fi */
/* there are no statements in Events */
if (Exceptions)
   { Except = Exceptions->First();
     while(Except)
	{ assert(Except->Handler != NULL);
	  if (Except->Handler->DFS_FindIndirectDependant(ThatSymbol,
						DepStmnt, DepFsmdStmnt,
						DepExpr, Reason))
	     { return(true);
	      } /* fi */
	  Except = Except->Succ();
	 } /* elihw */
    } /* fi */
/* there are no statements in Constraints */
/* there are no statements in Transitions */
/* there are no statements in Sensitivity */
if (ResetState)
   { if (ResetState->DFS_FindIndirectDependant(ThatSymbol,
						DepFsmdStmnt, DepExpr, Reason))
	{ if (DepStmnt)
	     { *DepStmnt = this;
	      } /* fi */
	  return(true);
	 } /* fi */
    } /* fi */
if (DefaultStmnts)
   { if (DefaultStmnts->DFS_FindIndirectDependant(ThatSymbol,
						DepFsmdStmnt, DepExpr, Reason))
	{ if (DepStmnt)
	     { *DepStmnt = this;
	      } /* fi */
	  return(true);
	 } /* fi */
    } /* fi */
if (States)
   { State = States->First();
     while(State)
	{ if (State->CmpndStmnt->DFS_FindIndirectDependant(ThatSymbol,
						DepFsmdStmnt, DepExpr, Reason))
	     { if (DepStmnt)
		  { *DepStmnt = this;
		   } /* fi */
	       return(true);
	      } /* fi */
	  State = State->Succ();
	 } /* elihw */
    } /* fi */

return(false);

} /* end of SIR_Statement::DFS_FindIndirectDependant */

ERROR SIR_Statement::MarkYourselfAndYourLabel(	/* iterator for color marking */
	sir_stmnt_marg	ColorArg)
{
SIR_COLOR	MyColor;

MyColor = *((SIR_COLOR*) ColorArg);

Color = MyColor;			/* mark yourself */
if (StmntType == SIR_STMNT_LABELED)
   { assert(Label != NULL);
     Label->Color = MyColor;		/* mark your label */
    } /* fi */
if (StmntType == SIR_STMNT_FSMD)	/* mark also labels in 'fsmd' */
   { if ((SIR_Error = DFS_ForAllFsmdStmnts(
				&SIR_FsmdStmnt::MarkYourLabel,
				(sir_fsmdstmnt_marg) &MyColor)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::MarkYourselfAndYourLabel */


ERROR SIR_Statement::CheckRequiredLabel(	/* iterator for label check */
	sir_stmnt_marg	/* Unused */)
{
sir_constraint	*Constr;

if (StmntType == SIR_STMNT_GOTO)
   { assert(Label != NULL);
     if (Label->Color == SIR_RED)
	{ SIR_ErrMsg.form("Label '%s' is required",
				Label->LabelName.chars());
	  return(SIR_ERROR_LABEL_REQUIRED);
	 } /* fi */
    } /* fi */
if (StmntType == SIR_STMNT_TIMING)
   { Constr = Constraints->First();
     while(Constr)
	{ assert(Constr->Label1 != NULL);
	  if (Constr->Label1->Color == SIR_RED)
	     { SIR_ErrMsg.form("Label '%s' is required",
					Constr->Label1->LabelName.chars());
	       return(SIR_ERROR_LABEL_REQUIRED);
	      } /* fi */
	  assert(Constr->Label2 != NULL);
	  if (Constr->Label2->Color == SIR_RED)
	     { SIR_ErrMsg.form("Label '%s' is required",
					Constr->Label2->LabelName.chars());
	       return(SIR_ERROR_LABEL_REQUIRED);
	      } /* fi */
	  Constr = Constr->Succ();
	 } /* elihw */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::CheckRequiredLabel */


ERROR SIR_Statement::CheckLabeledStmnt(	/* iterator for label check */
	sir_stmnt_marg	/* Unused */)
{

if (StmntType == SIR_STMNT_LABELED)
   { assert(Label != NULL);
     SIR_ErrMsg.form("Double definition of label '%s'",
				Label->LabelName.chars());
     return(SIR_ERROR_DOUBLE_DEFINITION_OF_LABEL);
    } /* fi */

if (StmntType == SIR_STMNT_FSMD)
   { if ((SIR_Error = DFS_ForAllFsmdStmnts(
			&SIR_FsmdStmnt::CheckLabeledStmnt,
			NULL)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::CheckLabeledStmnt */


ERROR SIR_Statement::CheckReturnType(	/* iterator for return type checking */
	sir_stmnt_marg	TypeArg)
{
sir_type	*ReturnType;

ReturnType = (sir_type*) TypeArg;

if (StmntType == SIR_STMNT_RETURN)
   { if (!(Expression1->Type->IsConvertableTo(ReturnType)))
	{ return(SIR_ERROR_RETURN_TYPE_MISMATCH);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::CheckReturnType */


ERROR SIR_Statement::CheckControlFlow(	/* iterator for control flow checking */
	sir_stmnt_marg	/* Unused */)
{

if (StmntType == SIR_STMNT_BREAK)
   { if (!(GetLoopOrSwitch()))
	{ if (LineInfo)
	     { SIR_ErrMsg.form("'break' statement outside loop"
				" or 'switch' statement" GL_ERROR_MSG_NEWLINE
				"(line %u, file \"%s\")",
				LineInfo->Line,
				LineInfo->File->Filename.chars());
	      } /* fi */
	  else
	     { SIR_ErrMsg = "'break' statement outside loop"
				" or 'switch' statement";
	      } /* esle */
	  return(SIR_ERROR_BREAK_OUTSIDE_LOOP_OR_SWITCH);
	 } /* fi */
     return(SIR_ERROR_NO_ERROR);
    } /* fi */

if (StmntType == SIR_STMNT_CONTINUE)
   { if (!(GetLoopOrSwitch()))
	{ if (LineInfo)
	     { SIR_ErrMsg.form("'continue' statement outside loop"
				" statement" GL_ERROR_MSG_NEWLINE
				"(line %u, file \"%s\")",
				LineInfo->Line,
				LineInfo->File->Filename.chars());
	      } /* fi */
	  else
	     { SIR_ErrMsg = "'continue' statement outside loop"
				" statement";
	      } /* esle */
	  return(SIR_ERROR_CONTINUE_OUTSIDE_LOOP);
	 } /* fi */
     return(SIR_ERROR_NO_ERROR);
    } /* fi */

if (StmntType == SIR_STMNT_CASE)
   { if (!(GetLoopOrSwitch()))
	{ if (LineInfo)
	     { SIR_ErrMsg.form("'case' statement outside"
				" 'switch' statement" GL_ERROR_MSG_NEWLINE
				"(line %u, file \"%s\")",
				LineInfo->Line,
				LineInfo->File->Filename.chars());
	      } /* fi */
	  else
	     { SIR_ErrMsg = "'case' statement outside"
				" 'switch' statement";
	      } /* esle */
	  return(SIR_ERROR_CASE_OUTSIDE_SWITCH);
	 } /* fi */
     return(SIR_ERROR_NO_ERROR);
    } /* fi */

if (StmntType == SIR_STMNT_DEFAULT)
   { if (!(GetLoopOrSwitch()))
	{ if (LineInfo)
	     { SIR_ErrMsg.form("'default' statement outside"
				" 'switch' statement" GL_ERROR_MSG_NEWLINE
				"(line %u, file \"%s\")",
				LineInfo->Line,
				LineInfo->File->Filename.chars());
	      } /* fi */
	  else
	     { SIR_ErrMsg = "'default' statement outside"
				" 'switch' statement";
	      } /* esle */
	  return(SIR_ERROR_DEFAULT_OUTSIDE_SWITCH);
	 } /* fi */
     return(SIR_ERROR_NO_ERROR);
    } /* fi */

if (StmntType == SIR_STMNT_FSMD)
   { return(DFS_ForAllFsmdStmnts(&SIR_FsmdStmnt::CheckControlFlow, NULL));
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::CheckControlFlow */


ERROR SIR_Statement::NeedsRangeChecker(	/* iterator for range checking */
	sir_stmnt_marg	/* Unused */)
{

if (  (StmntType == SIR_STMNT_TIMING)
    &&(!(Constraints->Empty())))
   { return(SIR_ERROR_FLAG_YES);
    } /* fi */

return(SIR_ERROR_FLAG_NO);

} /* end of SIR_Statement::NeedsRangeChecker */

void SIR_Statement::SetAlias(   /* sets all type, usertype, symbol alias' */
        sir_statement   *Alias) /* (iterates over symbols and usertypes) */
{

assert(Alias != NULL);

/* no need to process this node */

/* there are no symbols or usertypes in Expression1 */
/* there are no symbols or usertypes in Expression2 */
/* there are no symbols or usertypes in Expression3 */
/* there are no symbols or usertypes in Constant */
if (Statement1)
   { Statement1->SetAlias(Alias->Statement1);
    } /* fi */
if (Statement2)
   { Statement2->SetAlias(Alias->Statement2);
    } /* fi */
if (Scope)
   { Scope->SetAlias(Alias->Scope);
    } /* fi */
if (Statements)
   { Statements->SetAlias(Alias->Statements);
    } /* fi */
/* there are no symbols or usertypes in Events */
if (Exceptions)
   { Exceptions->SetAlias(Alias->Exceptions);
    } /* fi */
/* there are no symbols or usertypes in Constraints */
/* there are no symbols or usertypes in Transitions */
/* there are no symbols or usertypes in Sensitivity */
if (ResetState)
   { ResetState->SetAlias(Alias->ResetState);
    } /* fi */
if (DefaultStmnts)
   { DefaultStmnts->SetAlias(Alias->DefaultStmnts);
    } /* fi */
if (States)
   { States->SetAlias(Alias->States);
    } /* fi */

} /* end of SIR_Statement::SetAlias */

void SIR_Statement::UnAlias(    /* unalias all type, usertype, symbol links */
        sir_symbols     *GlobalSymbols)
{
sir_statement   *Stmnt;

if (  (Function)
    &&(Function->Alias))
   { Function = Function->Alias;
    } /* fi */
if (  (ResetSignal)
    &&(ResetSignal->Alias))
   { ResetSignal = ResetSignal->Alias;
    } /* fi */

if (Expression1)
   { Expression1->UnAlias();
    } /* fi */
if (Expression2)
   { Expression2->UnAlias();
    } /* fi */
if (Expression3)
   { Expression3->UnAlias();
    } /* fi */
/* there is nothing to unalias in Constant */
if (Statement1)
   { Statement1->UnAlias(GlobalSymbols);
    } /* fi */
if (Statement2)
   { Statement2->UnAlias(GlobalSymbols);
    } /* fi */
if (Scope)
   { Scope->UnAlias(GlobalSymbols);
    } /* fi */
if (Statements)
   { Stmnt = Statements->First();
     while(Stmnt)
        { Stmnt->UnAlias(GlobalSymbols);
          Stmnt = Stmnt->Succ();
         } /* elihw */
    } /* fi */
if (Events)
   { Events->UnAlias();
    } /* fi */
if (Exceptions)
   { Exceptions->UnAlias(GlobalSymbols);
    } /* fi */
/* there is nothing to unalias in Constraints */
if (Transitions)
   { Transitions->UnAlias();
    } /* fi */
if (Sensitivity)
   { Sensitivity->UnAlias();
    } /* fi */
if (ResetState)
   { ResetState->UnAlias(GlobalSymbols);
    } /* fi */
if (DefaultStmnts)
   { DefaultStmnts->UnAlias(GlobalSymbols);
    } /* fi */
if (States)
   { States->UnAlias(GlobalSymbols);
    } /* fi */

} /* end of SIR_Statement::UnAlias */

ERROR SIR_Statement::WritePipeStmnt(    /* generates C++ code for 'pipe' */
        gl_io       *IO,
        bool            WriteNotes)
{
sir_statement   *Stmnt;
sir_symbol      *PipedSymbol;
sir_symbols     *PipedScope;
int             ForkNum;

if (Expression1)        /* terminating (new) version */
   { assert(Expression2 != NULL);
     assert(Expression3 != NULL);
     IO->PutS("unsigned int " SIR_CXX_FIRST_NAME "=1, "
                SIR_CXX_LAST_NAME "=1;");
     SIR_LineInfo::WriteSPACE(IO);
     IO->PutS("for(");
     IO->TabStepUp();
     if ((SIR_Error = Expression1->WriteSC(IO, WriteNotes, true)))
        { return(SIR_Error);
         } /* fi */
     IO->PutS("; ");
     if ((SIR_Error = Expression2->WriteSC(IO, WriteNotes, true,
                                                SIR_TYPE_BOOL, 0, true)))
        { return(SIR_Error);
         } /* fi */
     IO->PutS("; ");
     if ((SIR_Error = Expression3->WriteSC(IO, WriteNotes, true)))
        { return(SIR_Error);
         } /* fi */
   //IO->TabStepDown();
     IO->PutC(')');
   //IO->TabStepUp();
     SIR_LineInfo::WriteSPACE(IO, true); // IO->PutC('{');
     IO->TabStepUp();
     IO->PrintF(SIR_SIM_PIPE_NAME "(%u, "
                        SIR_CXX_FIRST_NAME ", "
                        SIR_CXX_LAST_NAME ",",
                Statement1->Statements->NumElements());
     SIR_LineInfo::WriteSPACE(IO);
     Stmnt = Statement1->Statements->First();   /* threads */
     ForkNum = 0;
     while(Stmnt)
        { assert(SIR_CALLED_BEHAVIOR(Stmnt) != NULL);
          if (Stmnt->LineInfo)
             { if ((SIR_Error = Stmnt->LineInfo->WriteSC(IO)))
                  { return(SIR_Error);
                   } /* fi */
              } /* fi */
          else
             { SIR_LineInfo::WriteNewSC(IO, true);
              } /* esle */
          IO->PrintF("&" SIR_CXX_FORK_VARIABLE, ForkNum++);
          IO->PutC(',');
          SIR_LineInfo::WriteSPACE(IO);
          Stmnt = Stmnt->Succ();
         } /* elihw */
     PipedScope = Statement1->Scope->Parent;    /* list of piped symbols */
     while(  (PipedScope)                       /* in scope              */
           &&(PipedScope->ScopeInfo != SIR_SCOPE_GLOBAL))
        { PipedSymbol = PipedScope->First();
          while(PipedSymbol)
             { if (PipedSymbol->StorageClass == SIR_STORAGE_PIPED)
                  { IO->PutC('&');
                    IO->PutS(PipedSymbol->Name);
                    IO->PutC(',');
                    SIR_LineInfo::WriteSPACE(IO);
                   } /* fi */
               PipedSymbol = PipedSymbol->Succ();
              } /* elihw */
          PipedScope = PipedScope->Parent;
         } /* elihw */
     IO->PutS("((void*)0));");
     IO->TabStepDown();
     SIR_LineInfo::WriteSPACE(IO);
     IO->PrintF("if (" SIR_CXX_LAST_NAME "<%u)",
                Statement1->Statements->NumElements());
     IO->TabStepUp();
     SIR_LineInfo::WriteSPACE(IO);
     IO->PutS(SIR_CXX_LAST_NAME "++;");
     IO->TabStepDown();
     SIR_LineInfo::WriteNL(IO, false, true); // IO->PutC('}');
     IO->TabStepDown();
     SIR_LineInfo::WriteSPACE(IO);
     IO->PutS("while (" SIR_CXX_FIRST_NAME "++<" SIR_CXX_LAST_NAME ")");
     IO->TabStepUp();
     SIR_LineInfo::WriteSPACE(IO, true); // IO->PutC('{');
     IO->TabStepUp();
     IO->PrintF(SIR_SIM_PIPE_NAME "(%u, "
                                SIR_CXX_FIRST_NAME ", "
                                SIR_CXX_LAST_NAME ",",
                Statement1->Statements->NumElements());
     SIR_LineInfo::WriteSPACE(IO);
     Stmnt = Statement1->Statements->First();   /* threads */
     ForkNum = 0;
     while(Stmnt)
        { assert(SIR_CALLED_BEHAVIOR(Stmnt) != NULL);
          IO->PrintF("&" SIR_CXX_FORK_VARIABLE, ForkNum++);
          IO->PutC(',');
          SIR_LineInfo::WriteSPACE(IO);
          Stmnt = Stmnt->Succ();
         } /* elihw */
     PipedScope = Statement1->Scope->Parent;    /* list of piped symbols */
     while(  (PipedScope)                       /* in scope              */
           &&(PipedScope->ScopeInfo != SIR_SCOPE_GLOBAL))
        { PipedSymbol = PipedScope->First();
          while(PipedSymbol)
             { if (PipedSymbol->StorageClass == SIR_STORAGE_PIPED)
                  { IO->PutC('&');
                    IO->PutS(PipedSymbol->Name);
                    IO->PutC(',');
                    SIR_LineInfo::WriteSPACE(IO);
                   } /* fi */
               PipedSymbol = PipedSymbol->Succ();
              } /* elihw */
          PipedScope = PipedScope->Parent;
         } /* elihw */
     IO->PutS("((void*)0));");
     IO->TabStepDown();
     SIR_LineInfo::WriteSPACE(IO);
     IO->PrintF("if (" SIR_CXX_LAST_NAME "<%u)",
                Statement1->Statements->NumElements());
     IO->TabStepUp();
     SIR_LineInfo::WriteSPACE(IO);
     IO->PutS(SIR_CXX_LAST_NAME "++;");
     IO->TabStepDown();
     SIR_LineInfo::WriteNL(IO, false, true); // IO->PutC('}');
     IO->TabStepDown();
    } /* fi */
else                    /* non-terminating (old) version */
   { assert(Expression2 == NULL);
     assert(Expression3 == NULL);
     IO->TabStepUp();
     IO->PutS(SIR_SIM_PIPE_NAME "(");
     Stmnt = Statement1->Statements->First();
     ForkNum = 0;
     while(Stmnt)
        { assert(SIR_CALLED_BEHAVIOR(Stmnt) != NULL);
          if (Stmnt->LineInfo)
             { if ((SIR_Error = Stmnt->LineInfo->WriteSC(IO)))
                  { return(SIR_Error);
                   } /* fi */
              } /* fi */
          else
             { SIR_LineInfo::WriteNewSC(IO);
              } /* esle */
          IO->PrintF("&" SIR_CXX_FORK_VARIABLE, ForkNum++);
          IO->PutC(',');
          SIR_LineInfo::WriteSPACE(IO);
          Stmnt = Stmnt->Succ();
         } /* elihw */
     IO->PutS("((" SIR_SIM_FORK_NAME "*)0),");
     SIR_LineInfo::WriteSPACE(IO);	/* list of piped symbols in scope */
     PipedScope = Statement1->Scope->Parent;
     while(  (PipedScope)
           &&(PipedScope->ScopeInfo != SIR_SCOPE_GLOBAL))
        { PipedSymbol = PipedScope->First();
          while(PipedSymbol)
             { if (PipedSymbol->StorageClass == SIR_STORAGE_PIPED)
                  { IO->PutC('&');
                    IO->PutS(PipedSymbol->Name);
                    IO->PutC(',');
                    SIR_LineInfo::WriteSPACE(IO);
                   } /* fi */
               PipedSymbol = PipedSymbol->Succ();
              } /* elihw */
          PipedScope = PipedScope->Parent;
         } /* elihw */
     IO->PutS("((void*)0));");
     IO->TabStepDown();
    } /* esle */

return(SIR_ERROR_NO_ERROR);

} /* end of WritePipeStmnt */


ERROR SIR_Statement::WriteSC(	/* (re-) generates SpecC source code */
	gl_io	*IO,
	bool		WriteNotes,
	bool		CplusplusMode /* = false */,
	bool		PutNewLine /* = true */,
	bool		PutArgNotes /* = true */)
{
sir_statement	*Stmnt;
sir_exception	*Except;
sir_constraint	*Constr;
sir_transition	*Trans;
sir_symbol	*ParamSymbol;
sir_constant	*TmpConst;
int		ForkNum;
int		ExceptNum;
int		LabelNum;
gl_string		TmpString;
bool		PutBraces;

PutBraces = (StmntType == SIR_STMNT_COMPOUND);

if (LineInfo)
   { if ((SIR_Error = LineInfo->WriteSC(IO, true, PutBraces)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
else
   { if (PutNewLine)
	{ SIR_LineInfo::WriteNewSC(IO, true, PutBraces, &LineInfo);
	 } /* fi */
     else
	{ SIR_LineInfo::WriteSPACE(IO, PutBraces);
	 } /* fi */
    } /* esle */

switch(StmntType)
   { case SIR_STMNT_LABELED:
	{ IO->PutS(Label->LabelName);
	  IO->PutC(':');
	  if (Label->Color == SIR_RED)	/* is it a timing label? */
	     { IO->PutS(" " SIR_CXX_TIMESTAMP_PREFX);
	       IO->PutS(Label->LabelName);
	       IO->PutS(" = " SIR_SIM_CURR_TIME_NAME ";");
	      } /* fi */
	  IO->TabStepUp();
	  if ((SIR_Error = Statement1->WriteSC(IO, WriteNotes,
						CplusplusMode, false)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->TabStepDown();
	  // note: notes at labels are written at the end of each function
	  break;
	 }
     case SIR_STMNT_COMPOUND:
	{ assert(PutBraces);	// IO->PutC('{');
	  assert(Scope != NULL);	/* must have local scope */
	  if (  (PutArgNotes)
	      &&(Scope->Parent)	/* at highest level in function body? */
	      &&(Scope->Parent->ScopeInfo == SIR_SCOPE_PARAMETER))
	     { ParamSymbol = Scope->Parent->First();
	       while(ParamSymbol)
		  { if (ParamSymbol->Notes)	/* put notes at arguments */
		       { if ((SIR_Error = ParamSymbol->Notes->WriteSC(IO,
						WriteNotes, CplusplusMode)))
			    { return(SIR_Error);
			     } /* fi */
			} /* fi */
		    ParamSymbol = ParamSymbol->Succ();
		   } /* elihw */
	      } /* fi */
	  if (  (! Scope->Empty())
	      ||(! Scope->UserTypes->Empty()))
	     { if (CplusplusMode)
		  { if ((SIR_Error = Scope->WriteH(IO, WriteNotes)))
		       { return(SIR_Error);
			} /* fi */
		    if ((SIR_Error = Scope->WriteCC(IO, WriteNotes)))
		       { return(SIR_Error);
			} /* fi */
		   } /* fi */
	       else
		  { if ((SIR_Error = Scope->WriteSC(IO, WriteNotes)))
		       { return(SIR_Error);
			} /* fi */
		   } /* esle */
	       if (  (Statements)
		   &&(! Statements->Empty()))
		  { SIR_LineInfo::WriteVSPACE(IO);
		   } /* fi */
	      } /* fi */
	  if (Statements)
	     { Stmnt = Statements->First();
	       while(Stmnt)
		  { if ((SIR_Error = Stmnt->WriteSC(IO, WriteNotes,
							CplusplusMode, true)))
		       { return(SIR_Error);
			} /* fi */
		    Stmnt = Stmnt->Succ();
		   } /* elihw */
	      } /* fi */
	  if (  (PutArgNotes)	// here means: PutLabelNotes
	      &&(Scope->Parent)	/* at highest level in function body? */
	      &&(Scope->Parent->ScopeInfo == SIR_SCOPE_PARAMETER))
	     { sir_label	*Label;
	       assert(Scope->ParentSymbol != NULL);
	       assert(Scope->ParentSymbol->Labels != NULL);
	       Label = Scope->ParentSymbol->Labels->First();
	       while(Label)
		  { if (Label->Notes)	/* put notes at labels */
		       { if ((SIR_Error = Label->Notes->WriteSC(IO,
						WriteNotes, CplusplusMode)))
			    { return(SIR_Error);
			     } /* fi */
			} /* fi */
		    Label = Label->Succ();
		   } /* elihw */
	      } /* fi */
	  SIR_LineInfo::WriteNL(IO, false, true);	// IO->PutC('}');
	  break;
	 }
     case SIR_STMNT_EXPRESSION:
	{
#ifdef HAVE_ARYASGN
	  if ((SIR_Error = Expression1->WriteSC(IO, WriteNotes,
						CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->PutC(';');
#else /* ! HAVE_ARYASGN */
	  if (  (CplusplusMode)		/* array assignment handling */
	      &&(Expression1->ExprType == SIR_EXPR_ASSIGNMENT)
	      &&(Expression1->Arg1->Type->Type == SIR_TYPE_ARRAY)
	      &&(Expression1->Arg2->Type->Type == SIR_TYPE_ARRAY))
	     { if ((SIR_Error = Expression1->WriteArrayAssignment(IO,
								WriteNotes)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  else
	     { if ((SIR_Error = Expression1->WriteSC(IO, WriteNotes,
							CplusplusMode)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->PutC(';');
	      } /* esle */
#endif /* HAVE_ARYASGN */
	  break;
	 }
     case SIR_STMNT_IF:
	{ IO->PutS("if (");
	  IO->TabStepUp();
	  if ((SIR_Error = Expression1->WriteSC(IO, WriteNotes, CplusplusMode,
						SIR_TYPE_BOOL, 0, true)))
	     { return(SIR_Error);
	      } /* fi */
	//IO->TabStepDown();
	  IO->PutC(')');
	//IO->TabStepUp();
	  if ((SIR_Error = Statement1->WriteSC(IO, WriteNotes,
						CplusplusMode, true)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->TabStepDown();
	  break;
	 }
     case SIR_STMNT_IF_ELSE:
	{ IO->PutS("if (");
	  IO->TabStepUp();
	  if ((SIR_Error = Expression1->WriteSC(IO, WriteNotes, CplusplusMode,
						SIR_TYPE_BOOL, 0, true)))
	     { return(SIR_Error);
	      } /* fi */
	//IO->TabStepDown();
	  IO->PutC(')');
	//IO->TabStepUp();
	  if ((SIR_Error = Statement1->WriteSC(IO, WriteNotes,
						CplusplusMode, true)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->TabStepDown();
	  SIR_LineInfo::WriteNL(IO);
	  IO->PutS("else ");
	  IO->TabStepUp();
	  if ((SIR_Error = Statement2->WriteSC(IO, WriteNotes,
						CplusplusMode, true)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->TabStepDown();
	  break;
	 }
     case SIR_STMNT_SWITCH:
	{ IO->PutS("switch(");
	  IO->TabStepUp();
	  if ((SIR_Error = Expression1->WriteSC(IO, WriteNotes, CplusplusMode,
						SIR_TYPE_INT, 0, true)))
	     { return(SIR_Error);
	      } /* fi */
	//IO->TabStepDown();
	  IO->PutC(')');
	//IO->TabStepUp();
	  if ((SIR_Error = Statement1->WriteSC(IO, WriteNotes,
						CplusplusMode, true)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->TabStepDown();
	  break;
	 }
     case SIR_STMNT_CASE:
	{ IO->PutS("case ");
	  IO->PutS(Constant->Print());
	  IO->PutC(':');
	  IO->TabStepUp();
	  if ((SIR_Error = Statement1->WriteSC(IO, WriteNotes,
						CplusplusMode, false)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->TabStepDown();
	  break;
	 }
     case SIR_STMNT_DEFAULT:
	{ IO->PutS("default:");
	  IO->TabStepUp();
	  if ((SIR_Error = Statement1->WriteSC(IO, WriteNotes,
						CplusplusMode, false)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->TabStepDown();
	  break;
	 }
     case SIR_STMNT_WHILE:
	{ IO->PutS("while(");
	  IO->TabStepUp();
	  if ((SIR_Error = Expression1->WriteSC(IO, WriteNotes, CplusplusMode,
						SIR_TYPE_BOOL, 0, true)))
	     { return(SIR_Error);
	      } /* fi */
	//IO->TabStepDown();
	  IO->PutC(')');
	//IO->TabStepUp();
	  if ((SIR_Error = Statement1->WriteSC(IO, WriteNotes,
						CplusplusMode, true)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->TabStepDown();
	  break;
	 }
     case SIR_STMNT_DO_WHILE:
	{ IO->PutS("do ");
	  IO->TabStepUp();
	  if ((SIR_Error = Statement1->WriteSC(IO, WriteNotes,
						CplusplusMode, true)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->TabStepDown();
	  SIR_LineInfo::WriteNL(IO);
	  IO->PutS("while(");
	  IO->TabStepUp();
	  if ((SIR_Error = Expression1->WriteSC(IO, WriteNotes, CplusplusMode,
						SIR_TYPE_BOOL, 0, true)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->TabStepDown();
	  IO->PutS(");");
	  break;
	 }
     case SIR_STMNT_FOR:
	{ IO->PutS("for(");
	  IO->TabStepUp();
	  if ((SIR_Error = Expression1->WriteSC(IO, WriteNotes,
						CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->PutS("; ");
	  if ((SIR_Error = Expression2->WriteSC(IO, WriteNotes, CplusplusMode,
						SIR_TYPE_BOOL, 0, true)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->PutS("; ");
	  if ((SIR_Error = Expression3->WriteSC(IO, WriteNotes,
						CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	//IO->TabStepDown();
	  IO->PutC(')');
	//IO->TabStepUp();
	  if ((SIR_Error = Statement1->WriteSC(IO, WriteNotes,
						CplusplusMode, true)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->TabStepDown();
	  break;
	 }
     case SIR_STMNT_GOTO:
	{ IO->PutS("goto ");
	  IO->PutS(Label->LabelName);
	  IO->PutC(';');
	  break;
	 }
     case SIR_STMNT_CONTINUE:
	{ IO->PutS("continue;");
	  break;
	 }
     case SIR_STMNT_BREAK:
	{ IO->PutS("break;");
	  break;
	 }
     case SIR_STMNT_RETURN:
	{ sir_type	*ExpectType;

	  IO->PutS("return ");
	  assert(  (Function->Type->Type == SIR_TYPE_FUNCTION)
		 ||(Function->Type->Type == SIR_TYPE_METHOD));
	  ExpectType = Function->Type->SubType;
	  if ((SIR_Error = Expression1->WriteSC(IO,
					WriteNotes, CplusplusMode,
					SIR_RELAXED(ExpectType->Type,
							Expression1->Type),
					_BITLEN(ExpectType->LeftBound,
							ExpectType->RightBound),
					true)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->PutC(';');
	  break;
	 }
     case SIR_STMNT_PAR:
	{ if (CplusplusMode)
	     { assert(SIR_ERROR_NO_ERROR ==
			Statement1->CheckCompoundBehavior(false, false));
	       IO->TabStepUp();
	       IO->PutC('{');
	       SIR_LineInfo::WriteSPACE(IO);
	       if (! Statement1->Statements->Empty())
		  { IO->PutS(SIR_SIM_FORK_NAME " ");
		    Stmnt = Statement1->Statements->First();
		    ForkNum = 0;
		    while(Stmnt)
		       { assert(SIR_CALLED_BEHAVIOR(Stmnt) != NULL);
			 IO->PrintF(SIR_CXX_FORK_VARIABLE, ForkNum++);
			 IO->PutS("(&");
			 IO->PutS(SIR_CALLED_BEHAVIOR(Stmnt)->Name);
			 IO->PutC(')');
			 Stmnt = Stmnt->Succ();
			 if (Stmnt)
			    { IO->PutS(", ");
			     } /* fi */
			} /* elihw */
		    IO->PutC(';');
		    SIR_LineInfo::WriteSPACE(IO);
		   } /* fi */
	       IO->TabStepUp();
	       IO->PutS(SIR_SIM_PAR_NAME "(");
	       Stmnt = Statement1->Statements->First();
	       ForkNum = 0;
	       while(Stmnt)
		  { assert(SIR_CALLED_BEHAVIOR(Stmnt) != NULL);
		    if (Stmnt->LineInfo)
		       { if ((SIR_Error = Stmnt->LineInfo->WriteSC(IO)))
			    { return(SIR_Error);
			     } /* fi */
		        } /* fi */
		    else
		       { SIR_LineInfo::WriteNewSC(IO, false, false,
						&Stmnt->LineInfo);
			} /* esle */
		    IO->PrintF("&" SIR_CXX_FORK_VARIABLE, ForkNum++);
		    IO->PutC(',');
		    SIR_LineInfo::WriteSPACE(IO);
		    Stmnt = Stmnt->Succ();
		   } /* elihw */
               IO->PutS("((" SIR_SIM_FORK_NAME "*)0));");
	       IO->TabStepDown();
	       SIR_LineInfo::WriteNL(IO, false, true); // IO->PutC('}');
	       IO->TabStepDown();
	      } /* fi */
	  else
	     { IO->PutS("par");
	       IO->TabStepUp();
	       if ((SIR_Error = Statement1->WriteSC(IO, WriteNotes,
							CplusplusMode, false)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->TabStepDown();
	      } /* esle */
	  break;
	 }
     case SIR_STMNT_PIPE:
	{ if (CplusplusMode)
	     { assert(SIR_ERROR_NO_ERROR ==
			Statement1->CheckCompoundBehavior(false, false));
	       IO->TabStepUp();
	       IO->PutC('{');
	       SIR_LineInfo::WriteSPACE(IO);
	       if (! Statement1->Statements->Empty())
		  { IO->PutS(SIR_SIM_FORK_NAME " ");
		    Stmnt = Statement1->Statements->First();
		    ForkNum = 0;
		    while(Stmnt)
		       { assert(SIR_CALLED_BEHAVIOR(Stmnt) != NULL);
			 IO->PrintF(SIR_CXX_FORK_VARIABLE, ForkNum++);
			 IO->PutS("(&");
			 IO->PutS(SIR_CALLED_BEHAVIOR(Stmnt)->Name);
			 IO->PutC(')');
			 Stmnt = Stmnt->Succ();
			 if (Stmnt)
			    { IO->PutS(", ");
			     } /* fi */
			} /* elihw */
		    IO->PutC(';');
		    SIR_LineInfo::WriteSPACE(IO);
		   } /* fi */
	       if ((SIR_Error = WritePipeStmnt(IO, WriteNotes)))
		  { return(SIR_Error);
		   } /* fi */
	       SIR_LineInfo::WriteNL(IO, false, true); // IO->PutC('}');
	       IO->TabStepDown();
	      } /* fi */
	  else
	     { IO->PutS("pipe");
	       if (Expression1)
		  { assert(Expression2 != NULL);
		    assert(Expression3 != NULL);
		    IO->PutC('(');
		    if ((SIR_Error = Expression1->WriteSC(IO, WriteNotes,
							CplusplusMode)))
		       { return(SIR_Error);
			} /* fi */
		    IO->PutS("; ");
		    if ((SIR_Error = Expression2->WriteSC(IO, WriteNotes,
							CplusplusMode,
							SIR_TYPE_BOOL, 0,true)))
		       { return(SIR_Error);
			} /* fi */
		    IO->PutS("; ");
		    if ((SIR_Error = Expression3->WriteSC(IO, WriteNotes,
							CplusplusMode)))
		       { return(SIR_Error);
			} /* fi */
		    IO->PutC(')');
		   } /* fi */
	       else
		  { assert(Expression2 == NULL);
		    assert(Expression3 == NULL);
		   } /* esle */
	       IO->TabStepUp();
	       if ((SIR_Error = Statement1->WriteSC(IO, WriteNotes,
							CplusplusMode, false)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->TabStepDown();
	      } /* esle */
	  break;
	 }
     case SIR_STMNT_EXCEPTION:
	{ if (CplusplusMode)
	     { assert(SIR_ERROR_NO_ERROR ==
			Statement1->CheckCompoundBehavior(true, false));
	       IO->TabStepUp();
	       IO->PutC('{');
	       SIR_LineInfo::WriteSPACE(IO);
	       IO->PutS(SIR_SIM_TRY_BLOCK " " SIR_CXX_TRY_VARIABLE);
	       if (Statement1->Statements->Empty())
		  { IO->PutS("0(0);");	/* empty try block */
		   } /* fi */
	       else
		  { IO->PutS(SIR_CALLED_BEHAVIOR(
				Statement1->Statements->First())->Name);
		    IO->PutS("(&");
		    IO->PutS(SIR_CALLED_BEHAVIOR(
				Statement1->Statements->First())->Name);
		    IO->PutS(");");
		   } /* esle */
	       Except = Exceptions->First();
	       ExceptNum = 0;
	       while(Except)
		  { assert(SIR_ERROR_NO_ERROR ==
			Except->Handler->CheckCompoundBehavior(true, false));
		    if (Except->Pred())
		       { IO->PutS(", ");
			} /* fi */
		    else
		       { SIR_LineInfo::WriteSPACE(IO);
			 IO->PutS(SIR_SIM_EXCEPT_BLOCK " ");
			} /* esle */
		    IO->PrintF(SIR_CXX_EXCEPT_VARIABLE, ExceptNum++);
		    if (Except->ExceptionType == SIR_EXCEPTION_TRAP)
		       { IO->PutS("(" SIR_SIM_TRAP_FLAG);
			} /* fi */
		    else /* SIR_EXCEPTION_INTERRUPT */
		       { IO->PutS("(" SIR_SIM_INTERRUPT_FLAG);
			} /* esle */
		    if (Except->Handler->Statements->Empty())
		       { IO->PutS(", ((" SIR_SIM_BEHAVIOR_NAME "*)0)");
			} /* fi */
		    else
		       { IO->PutS(", &");
			 IO->PutS(SIR_CALLED_BEHAVIOR(
				Except->Handler->Statements->First())
						->Name);
			} /* esle */
		    if (Except->Events->NumElements())
		       { IO->PutS(", ");
			} /* fi */
		    if ((SIR_Error = Except->Events->WriteSC(IO, ",", true)))
		       { return(SIR_Error);
			} /* fi */
                    IO->PutS(", ((void*)0))");
		    Except = Except->Succ();
		   } /* elihw */
	       IO->PutC(';');
	       SIR_LineInfo::WriteSPACE(IO);
	       IO->PutS(SIR_SIM_EXCEPTION_NAME "(");
	       IO->TabStepUp();
	       IO->PutS("&" SIR_CXX_TRY_VARIABLE);
	       if (Statement1->Statements->Empty())
		  { IO->PutS("0");
		   } /* fi */
	       else
		  { IO->PutS(SIR_CALLED_BEHAVIOR(
				Statement1->Statements->First())->Name);
		   } /* esle */
	       Except = Exceptions->First();
	       ExceptNum = 0;
	       while(Except)
		  { IO->PutC(',');
		    if (Except->LineInfo)
		       { if ((SIR_Error =
				Except->LineInfo->WriteSC(IO, true)))
			    { return(SIR_Error);
			     } /* fi */
			} /* fi */
		    else
		       { SIR_LineInfo::WriteNewSC(IO, true, false,
						&Except->LineInfo);
			} /* esle */
		    IO->PrintF("&" SIR_CXX_EXCEPT_VARIABLE, ExceptNum++);
		    Except = Except->Succ();
		   } /* elihw */
	       IO->PutS(", 0);");
	       IO->TabStepDown();
	       SIR_LineInfo::WriteNL(IO, false, true); // IO->PutC('}');
	       IO->TabStepDown();
	      } /* fi */
	  else	/* not CplusplusMode */
	     { IO->PutS("try");
	       IO->TabStepUp();
	       if ((SIR_Error = Statement1->WriteSC(IO, WriteNotes,
							CplusplusMode, false)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->TabStepDown();
	       Except = Exceptions->First();
	       while(Except)
		  { if (Except->LineInfo)
		       { if ((SIR_Error =
				Except->LineInfo->WriteSC(IO, true)))
			    { return(SIR_Error);
			     } /* fi */
			} /* fi */
		    else
		       { SIR_LineInfo::WriteNewSC(IO, true, false,
						&Except->LineInfo);
			} /* esle */
		    switch(Except->ExceptionType)
		       { case SIR_EXCEPTION_TRAP:
			    { IO->PutS("trap(");
			      break;
			     }
			 case SIR_EXCEPTION_INTERRUPT:
			    { IO->PutS("interrupt(");
			      break;
			     }
			 default:
			    { assert(false);	/* bad exception type */
			     }
			} /* hctiws */
		    if ((SIR_Error = Except->Events->WriteSC(IO)))
		       { return(SIR_Error);
			} /* fi */
		    IO->PutC(')');
		    IO->TabStepUp();
		    if ((SIR_Error = Except->Handler->WriteSC(IO, WriteNotes,
							CplusplusMode, false)))
		       { return(SIR_Error);
			} /* fi */
		    IO->TabStepDown();
		    Except = Except->Succ();
		   } /* elihw */
	      } /* esle */
	  break;
	 }
     case SIR_STMNT_TIMING:
	{ if (CplusplusMode)
	     { IO->TabStepUp();
	       IO->PutC('{');
	       SIR_LineInfo::WriteSPACE(IO);
	       assert(Statement1->Statements != NULL);
	       Stmnt = Statement1->Statements->First();
	       LabelNum = 0;
	       while(Stmnt)
		  { if (Stmnt->StmntType == SIR_STMNT_LABELED)
		       { LabelNum++;
			 if (LabelNum == 1)
			    { IO->PutS(SIR_CXX_TIME_TYPE " ");
			     } /* fi */
			 else
			    { IO->PutS(", ");
			     } /* esle */
			 IO->PutS(SIR_CXX_TIMESTAMP_PREFX);
			 IO->PutS(Stmnt->Label->LabelName);
			 assert(Stmnt->Label->Color == SIR_WHITE);
			 Stmnt->Label->Color = SIR_RED;	/* mark timing labels */
			} /* fi */
		    Stmnt = Stmnt->Succ(); // .... flat list of labels only??
		   } /* elihw */
	       if (LabelNum > 0)
		  { IO->PutC(';');
		   } /* fi */
	       IO->TabStepUp();
	       if ((SIR_Error = Statement1->WriteSC(IO, WriteNotes,
						CplusplusMode, true)))
		  { // .... unmark timing labels!
		    return(SIR_Error);
		   } /* fi */
	       IO->TabStepDown();
	       Constr = Constraints->First();
	       while(Constr)
		  { if (Constr->LineInfo)
		       { if ((SIR_Error = Constr->LineInfo->WriteSC(
								IO, true)))
			    { // .... unmark timing labels!
			      return(SIR_Error);
			     } /* fi */
			} /* fi */
		    else
		       { SIR_LineInfo::WriteNewSC(IO, true, false,
						&Constr->LineInfo);
			} /* esle */
		    IO->PutS(SIR_CXX_RANGE_NAME "(" SIR_CXX_TIMESTAMP_PREFX);
		    IO->PutS(Constr->Label1->LabelName);
		    IO->PutS(", " SIR_CXX_TIMESTAMP_PREFX);
		    IO->PutS(Constr->Label2->LabelName);
		    IO->PutS(", ");
		    if (Constr->MinTime)
		       { TmpConst = (new SIR_Constant(Constr->MinTime))
					->Converted(SIR_CONST_LONGLONG);
			 IO->PutS(TmpConst->Print(true));
			 delete TmpConst;
			 IO->PutS(", true, ");
			} /* fi */
		    else
		       { IO->PutS("0, false, ");
			} /* esle */
		    if (Constr->MaxTime)
		       { TmpConst = (new SIR_Constant(Constr->MaxTime))
					->Converted(SIR_CONST_LONGLONG);
			 IO->PutS(TmpConst->Print(true));
			 delete TmpConst;
			 IO->PutS(", true, ");
			} /* fi */
		    else
		       { IO->PutS("0, false, ");
			} /* esle */
		    IO->PutC('"');
		    IO->PutS(Constr->Label1->LabelName);
		    IO->PutS("\", \"");
		    IO->PutS(Constr->Label2->LabelName);
		    IO->PutS("\", ");
		    if (Constr->LineInfo)
		       { SIR_Constant::PrettyString(&TmpString,
				Constr->LineInfo->File->Filename.chars());
			 IO->PutS(TmpString);
			 IO->PrintF(", %d", Constr->LineInfo->Line);
			} /* fi */
		    else
		       { if (SIR_LineInfo::CurrentFile)
			    { SIR_Constant::PrettyString(&TmpString,
						SIR_LineInfo::CurrentFile);
			      IO->PutS(TmpString);
			      IO->PrintF(", %d", IO->GetLineNo());
			     } /* fi */
			 else
			    { IO->PutS("\"<unknown>\", 0");
			     } /* esle */
			} /* esle */
		    IO->PutS(");");
		    Constr = Constr->Succ();
		   } /* elihw */
	       Stmnt = Statement1->Statements->First();
	       while(Stmnt)
		  { if (Stmnt->StmntType == SIR_STMNT_LABELED)
		       { assert(Stmnt->Label->Color == SIR_RED);
			 Stmnt->Label->Color = SIR_WHITE;	/* unmark */
			} /* fi */
		    Stmnt = Stmnt->Succ();
		   } /* elihw */
	       SIR_LineInfo::WriteNL(IO, false, true); // IO->PutC('}');
	       IO->TabStepDown();
	      } /* fi */
	  else
	     { IO->PutS("do ");
	       IO->TabStepUp();
	       if ((SIR_Error = Statement1->WriteSC(IO, WriteNotes,
						CplusplusMode, true)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->TabStepDown();
	       SIR_LineInfo::WriteNL(IO);
	       IO->PutS("timing{");
	       IO->TabStepUp();
	       Constr = Constraints->First();
	       while(Constr)
		  { if (Constr->LineInfo)
		       { if ((SIR_Error = Constr->LineInfo->WriteSC(
								IO, true)))
			    { return(SIR_Error);
			     } /* fi */
			} /* fi */
		    else
		       { SIR_LineInfo::WriteNewSC(IO, true, false,
						&Constr->LineInfo);
			} /* esle */
		    IO->PutS("range(");
		    IO->PutS(Constr->Label1->LabelName);
		    IO->PutS("; ");
		    IO->PutS(Constr->Label2->LabelName);
		    IO->PutS("; ");
		    if (Constr->MinTime)
		       { IO->PutS(Constr->MinTime->Print(CplusplusMode));
			} /* fi */
		    IO->PutS("; ");
		    if (Constr->MaxTime)
		       { IO->PutS(Constr->MaxTime->Print(CplusplusMode));
			} /* fi */
		    IO->PutS(");");
		    Constr = Constr->Succ();
		   } /* elihw */
	       SIR_LineInfo::WriteNL(IO, false, true);// IO->PutC('}');
	       IO->TabStepDown();
	      } /* esle */
	  break;
	 }
     case SIR_STMNT_FSM:
	{ if (CplusplusMode)
	     { IO->TabStepUp();
	       IO->PutC('{');
	       SIR_LineInfo::WriteSPACE(IO);
	       IO->PutS("enum { " SIR_CXX_STATE_PREFIX "0");
	       Trans = Transitions->First();
	       while(Trans)
		  { if (  (Trans->Pred() == NULL)
			||(Trans->Pred()->CurrState != Trans->CurrState))
		       { IO->PutS(", ");
			 IO->PutS(SIR_CXX_STATE_PREFIX);
			 IO->PutS(Trans->CurrState->Name);
			} /* fi */
		    Trans = Trans->Succ();
		   } /* elihw */
	       IO->PutS(" } " SIR_CXX_STATE_VARIABLE
				" = " SIR_CXX_STATE_PREFIX);
	       if (Transitions->First())
		  { IO->PutS(Transitions->First()->CurrState->Name);
		   } /* fi */
	       else
		  { IO->PutS("0");
		   } /* esle */
	       IO->PutC(';');
	       SIR_LineInfo::WriteSPACE(IO);
	       IO->PutS("do switch(" SIR_CXX_STATE_VARIABLE ")");
	       SIR_LineInfo::WriteSPACE(IO);
	       IO->TabStepUp();
	       IO->PutC('{');
	       Trans = Transitions->First();
	       while(Trans)
		  { assert(  (Trans->CurrState->Type->Type
					== SIR_TYPE_VOID)      /* local state */
			   ||(Trans->CurrState->Type->Type
					== SIR_TYPE_BEHAVIOR));/* bhvr.state */
		    if (Trans->CurrState->Type->Type
					== SIR_TYPE_VOID) /* dummy state type */
		       { assert(Trans->CurrState->FctBody != NULL);
			 if (Trans->CurrState->LineInfo)
			    { if ((SIR_Error = Trans->CurrState->LineInfo
							->WriteSC(IO, true)))
				 { return(SIR_Error);
				  } /* fi */
			     } /* fi */
			 else
			    { SIR_LineInfo::WriteNewSC(IO, true, false,
						&Trans->CurrState->LineInfo);
			     } /* esle */
			 IO->PutS("case " SIR_CXX_STATE_PREFIX);
			 IO->PutS(Trans->CurrState->Name);
			 IO->PutS(": { ");
			 IO->TabStepUp();
			 if ((SIR_Error = Trans->CurrState->FctBody->WriteSC(
						IO, WriteNotes,
						CplusplusMode, false)))
			    { return(SIR_Error);
			     } /* fi */
			 IO->TabStepDown();
			 if (Trans->LineInfo)
			    { if ((SIR_Error = Trans->LineInfo
							->WriteSC(IO, true)))
				 { return(SIR_Error);
				  } /* fi */
			     } /* fi */
			 else
			    { SIR_LineInfo::WriteNewSC(IO, true, false,
						&Trans->LineInfo);
			     } /* esle */
			} /* fi */
		    else
		       { if (Trans->LineInfo)
			    { if ((SIR_Error = Trans->LineInfo
							->WriteSC(IO, true)))
				 { return(SIR_Error);
				  } /* fi */
			     } /* fi */
			 else
			    { SIR_LineInfo::WriteNewSC(IO, true, false,
						&Trans->LineInfo);
			     } /* esle */
			 IO->PutS("case " SIR_CXX_STATE_PREFIX);
			 IO->PutS(Trans->CurrState->Name);
			 IO->PutS(": { ");
			 IO->PutS(Trans->CurrState->Name);
			 IO->PutS("." GL_MAIN_METHOD_NAME "();");
			} /* esle */
		    IO->TabStepUp();
		    SIR_LineInfo::WriteSPACE(IO);
		    if (Trans->Condition)
		       { IO->PutS("if (");
			 IO->TabStepUp();
			 if ((SIR_Error = Trans->Condition->WriteSC(IO,
						WriteNotes, true,
						SIR_TYPE_BOOL, 0, true)))
			    { return(SIR_Error);
			     } /* fi */
			 IO->TabStepDown();
			 IO->PutS(") ");
			} /* fi */
		    IO->PutS("{ " SIR_CXX_STATE_VARIABLE " = "
					SIR_CXX_STATE_PREFIX);
		    if (Trans->NextState)
		       { IO->PutS(Trans->NextState->Name);
			} /* fi */
		    else
		       { IO->PutC('0');
			} /* else */
		    IO->PutS("; break; }");
		    while(  (Trans->Succ())
			  &&(Trans->Succ()->CurrState == Trans->CurrState))
		       { Trans = Trans->Succ();	/* combine same CurrStates */
			 if (Trans->LineInfo)
			    { if ((SIR_Error = Trans->LineInfo->WriteSC(IO,
									true)))
				 { return(SIR_Error);
				  } /* fi */
			     } /* fi */
			 else
			    { SIR_LineInfo::WriteNewSC(IO, true, false,
							&Trans->LineInfo);
			     } /* esle */
			 if (Trans->Condition)
			    { IO->PutS("if (");
			      IO->TabStepUp();
			      if ((SIR_Error = Trans->Condition->WriteSC(IO,
						WriteNotes, true,
						SIR_TYPE_BOOL, 0, true)))
				 { return(SIR_Error);
				  } /* fi */
			      IO->TabStepDown();
			      IO->PutS(") ");
			     } /* fi */
			 IO->PutS("{ " SIR_CXX_STATE_VARIABLE " = "
					SIR_CXX_STATE_PREFIX);
			 if (Trans->NextState)
			    { IO->PutS(Trans->NextState->Name);
			     } /* fi */
			 else
			    { IO->PutC('0');
			     } /* esle */
			 IO->PutS("; break; }");
			} /* elihw */
		    SIR_LineInfo::WriteSPACE(IO);
		    IO->PutC('}');
		    IO->TabStepDown();
		    Trans = Trans->Succ();
		   } /* elihw */
	       SIR_LineInfo::WriteSPACE(IO);
	       IO->PutS("case " SIR_CXX_STATE_PREFIX "0: { "
			SIR_CXX_STATE_VARIABLE " = " SIR_CXX_STATE_PREFIX "0; "
			"break; }");
	       IO->TabStepDown();
	       SIR_LineInfo::WriteSPACE(IO);
	       IO->PutC('}');
	       SIR_LineInfo::WriteSPACE(IO);
	       IO->PutS("while(" SIR_CXX_STATE_VARIABLE " != "
					SIR_CXX_STATE_PREFIX "0);");
	       SIR_LineInfo::WriteNL(IO, false, true); // IO->PutC('}');
	       IO->TabStepDown();
	      } /* fi */
	  else	/* SpecC FSM code */
	     { IO->PutS("fsm{");
	       IO->TabStepUp();
	       Trans = Transitions->First();
	       while(Trans)
		  { assert(  (Trans->CurrState->Type->Type
					== SIR_TYPE_VOID)      /* local state */
			   ||(Trans->CurrState->Type->Type
					== SIR_TYPE_BEHAVIOR));/* bhvr.state */
		    if (Trans->CurrState->Type->Type
					== SIR_TYPE_VOID) /* dummy state type */
		       { assert(Trans->CurrState->FctBody != NULL);
			 if (Trans->CurrState->LineInfo)
			    { if ((SIR_Error = Trans->CurrState->LineInfo
							->WriteSC(IO, true)))
				 { return(SIR_Error);
				  } /* fi */
			     } /* fi */
			 else
			    { SIR_LineInfo::WriteNewSC(IO, true, false,
						&Trans->CurrState->LineInfo);
			     } /* esle */
			 IO->PutS(Trans->CurrState->Name);
			 IO->TabStepUp();
			 if ((SIR_Error = Trans->CurrState->FctBody->WriteSC(
						IO, WriteNotes,
						CplusplusMode, false)))
			    { return(SIR_Error);
			     } /* fi */
			 IO->TabStepDown();
			 IO->PutS(" : ");
			 if (Trans->LineInfo)
			    { if ((SIR_Error = Trans->LineInfo->WriteSC(
								IO, true)))
				 { return(SIR_Error);
				  } /* fi */
			     } /* fi */
			 else
			    { SIR_LineInfo::WriteNewSC(IO, true, false,
							&Trans->LineInfo);
			     } /* esle */
			} /* fi */
		    else
		       { if (Trans->LineInfo)
			    { if ((SIR_Error = Trans->LineInfo->WriteSC(
								IO, true)))
				 { return(SIR_Error);
				  } /* fi */
			     } /* fi */
			 else
			    { SIR_LineInfo::WriteNewSC(IO, true, false,
							&Trans->LineInfo);
			     } /* esle */
			 IO->PutS(Trans->CurrState->Name);
			 IO->PutS(": ");
			} /* esle */
		    if (Trans->Condition)
		       { IO->PutS("if (");
			 IO->TabStepUp();
			 if ((SIR_Error = Trans->Condition->WriteSC(IO,
								WriteNotes)))
			    { return(SIR_Error);
			     } /* fi */
			 IO->TabStepDown();
			 IO->PutS(") ");
			} /* fi */
		    if (Trans->NextState)
		       { IO->PutS("goto ");
			 IO->PutS(Trans->NextState->Name);
			 IO->PutC(';');
			} /* fi */
		    else
		       { IO->PutS("break;");
			} /* else */
		    IO->TabStepUp();
		    while(  (Trans->Succ())
			  &&(Trans->Succ()->CurrState == Trans->CurrState))
		       { Trans = Trans->Succ();	/* combine same CurrStates */
			 if (Trans->LineInfo)
			    { if ((SIR_Error = Trans->LineInfo->WriteSC(IO,
									true)))
				 { return(SIR_Error);
				  } /* fi */
			     } /* fi */
			 else
			    { SIR_LineInfo::WriteNewSC(IO, true, false,
							&Trans->LineInfo);
			     } /* esle */
			 if (Trans->Condition)
			    { IO->PutS("if (");
			      IO->TabStepUp();
			      if ((SIR_Error = Trans->Condition->WriteSC(IO,
								WriteNotes)))
				 { return(SIR_Error);
				  } /* fi */
			      IO->TabStepDown();
			      IO->PutS(") ");
			     } /* fi */
			 if (Trans->NextState)
			    { IO->PutS("goto ");
			      IO->PutS(Trans->NextState->Name);
			      IO->PutC(';');
			     } /* fi */
			 else
			    { IO->PutS("break;");
			     } /* esle */
			} /* elihw */
		    IO->TabStepDown();
		    Trans = Trans->Succ();
		   } /* elihw */
	       SIR_LineInfo::WriteNL(IO, false, true);// IO->PutC('}');
	       IO->TabStepDown();
	      } /* esle */
	  break;
	 }
     case SIR_STMNT_FSMD:
	{ if (CplusplusMode)
	     { sir_fsmdstate	*State;
	       if ((SIR_Error = CreateDelayedAssignments()))
		  { return(SIR_Error);
		   } /* fi */
	       IO->TabStepUp();
	       IO->PutC('{');
	       SIR_LineInfo::WriteSPACE(IO);
	       IO->PutS("enum " SIR_CXX_FSMD_STATE_TYPE " { ");
	       State = (States ? States->First() : NULL);
	       while(State)
		  { IO->PutS(SIR_CXX_FSMD_STATE_PFX);
		    IO->PutS(State->StateName);
		    IO->PutS(", ");
		    State = State->Succ();
		   } /* elihw */
	       IO->PutS(SIR_CXX_FSMD_BREAKSTATE " };");
	       SIR_LineInfo::WriteSPACE(IO);
	       if (Constant)
		  { IO->PutS(SIR_SIM_AUTO_EVENT_NAME " "
					SIR_CXX_FSMD_CLOCK_NAME "(");
		    IO->PutS(Constant->Print(true));
		    IO->PutS(");");
		    SIR_LineInfo::WriteSPACE(IO);
		   } /* fi */
	       IO->PutS(SIR_SIM_BUFFERED_NAME "<int> "
				SIR_CXX_FSMD_STATE_REG "(");
	       if (Constant)
		  { IO->PutS(SIR_SIM_ANY_SIG_PFX SIR_CXX_FSMD_CLOCK_NAME ")");
		   } /* fi */
	       else
		  { assert(Events != NULL);
		    if ((SIR_Error = Events->WriteCC(IO)))
		       { DeleteDelayedAssignments();
			 return(SIR_Error);
			} /* fi */
		   } /* esle */
	       if (ResetSignal)
		  { IO->PutS(", &");
		    IO->PutS(ResetSignal->Name);
		    IO->PutS(", ");
		    if (ResetActiveHi)
		       { IO->PutS("true");
			} /* fi */
		    else
		       { IO->PutS("false");
			} /* esle */
		   } /* fi */
	       IO->PutS(", ");
	       if (  (States)
		   &&(States->First()))
		  { IO->PutS(SIR_CXX_FSMD_STATE_PFX);
		    IO->PutS(States->First()->StateName);
		   } /* fi */
	       else
		  { IO->PutS(SIR_CXX_FSMD_BREAKSTATE);
		   } /* esle */
	       IO->PutS(");");
	       SIR_LineInfo::WriteSPACE(IO);
	       if ((SIR_Error = Scope->WriteH(IO, WriteNotes)))
		  { DeleteDelayedAssignments();
		    return(SIR_Error);
		   } /* fi */
	       if ((SIR_Error = Scope->WriteCC(IO, WriteNotes)))
		  { DeleteDelayedAssignments();
		    return(SIR_Error);
		   } /* fi */
	       SIR_LineInfo::WriteVSPACE(IO);
	       SIR_LineInfo::WriteSPACE(IO);
	       IO->PutS("while(" SIR_CXX_FSMD_STATE_REG " != "
					SIR_CXX_FSMD_BREAKSTATE ")");
	       SIR_LineInfo::WriteSPACE(IO);
	       IO->TabStepUp();
	       IO->PutC('{');
	       SIR_LineInfo::WriteSPACE(IO);
	       IO->PutS(SIR_SIM_WAIT_NAME "(");
	       if (Constant)
		  { IO->PutS(SIR_SIM_ANY_SIG_PFX SIR_CXX_FSMD_CLOCK_NAME ")");
		   } /* fi */
	       else
		  { if ((SIR_Error = Events->WriteSC(IO, ",", true)))
		       { DeleteDelayedAssignments();
			 return(SIR_Error);
			} /* fi */
		   } /* esle */
	       if (Sensitivity && !Sensitivity->Empty())	// bug fix 08/27/04, RD
		  { IO->PutS(", ");
		    if ((SIR_Error = Sensitivity->WriteSC(IO, ",", true)))
		       { DeleteDelayedAssignments();
			 return(SIR_Error);
			} /* fi */
		   } /* fi */
	       IO->PutS(", ((void*)0));");
	       SIR_LineInfo::WriteSPACE(IO);
	       if (ResetState)
		  { sir_lineinfo	*LineInfo;
		    assert(Expression1 != NULL);
		    if ((LineInfo = Expression1->GetFirstLineInfo()))
		       { if ((SIR_Error = LineInfo->WriteSC(IO, true)))
			    { DeleteDelayedAssignments();
			      return(SIR_Error);
			     } /* fi */
			} /* fi */
		    else
		       { SIR_LineInfo::WriteSPACE(IO);
			} /* esle */
		    IO->PutS("if (");
		    IO->TabStepUp();
		    if ((SIR_Error = Expression1->WriteSC(IO, WriteNotes,
						CplusplusMode,
						SIR_TYPE_BOOL, 0, true)))
		       { DeleteDelayedAssignments();
			 return(SIR_Error);
			} /* fi */
		  //IO->TabStepDown();
		    IO->PutC(')');
		  //IO->TabStepUp();
		    if ((SIR_Error = ResetState->WriteSC(IO, WriteNotes,
							CplusplusMode, true)))
		       { DeleteDelayedAssignments();
			 return(SIR_Error);
			} /* fi */
		    IO->TabStepDown();
		    SIR_LineInfo::WriteNL(IO);
		    IO->PutS("else ");
		    IO->TabStepUp();
		    SIR_LineInfo::WriteSPACE(IO, true);	// IO->PutC('{');
		   } /* fi */
	       if (DefaultStmnts)
		  { IO->TabStepUp();
		    if ((SIR_Error = DefaultStmnts->WriteSC(IO, WriteNotes,
							CplusplusMode, true)))
		       { DeleteDelayedAssignments();
			 return(SIR_Error);
			} /* fi */
		    IO->TabStepDown();
		   } /* fi */
	       SIR_LineInfo::WriteSPACE(IO);
	       IO->PutS("switch(" SIR_CXX_FSMD_STATE_REG ")");
	       IO->TabStepUp();
	       SIR_LineInfo::WriteSPACE(IO, true);	// IO->PutC('{');
	       State = (States ? States->First() : NULL);
	       while(State)
		  { if ((SIR_Error = State->WriteSC(IO, WriteNotes,
							CplusplusMode, false)))
		       { DeleteDelayedAssignments();
			 return(SIR_Error);
			} /* fi */
		    State = State->Succ();
		   } /* elihw */
	       SIR_LineInfo::WriteSPACE(IO);
	       IO->PutS("default: break; // never reached");
	       SIR_LineInfo::WriteNL(IO, false, true); // IO->PutC('}');
	       IO->TabStepDown();
	       if (ResetState)
		  { SIR_LineInfo::WriteNL(IO, false, true); // IO->PutC('}');
		    IO->TabStepDown();
		   } /* fi */
	       IO->TabStepDown();
	       SIR_LineInfo::WriteSPACE(IO);
	       IO->PutC('}');
	       SIR_LineInfo::WriteSPACE(IO);
	       SIR_LineInfo::WriteNL(IO, false, true); // IO->PutC('}');
	       IO->TabStepDown();
	       DeleteDelayedAssignments();
	      } /* fi */
	  else	/* SpecC code for FSMD */
	     { IO->PutS("fsmd(");
	       if (Constant)			/* internal clock */
		  { IO->PutS(Constant->Print());
		   } /* fi */
	       else
		  { assert(Events != NULL);	/* external clock */
		    if ((SIR_Error = Events->WriteSC(IO)))
		       { return(SIR_Error);
			} /* fi */
		   } /* esle */
	       if (  (Sensitivity)
		   ||(ResetSignal))
		  { IO->PutS("; ");
		   } /* fi */
	       if (Sensitivity)
		  { if ((SIR_Error = Sensitivity->WriteSC(IO)))
		       { return(SIR_Error);
			} /* fi */
		   } /* fi */
	       if (ResetSignal)
		  { IO->PutS("; ");
		    if (!ResetActiveHi)
		       { IO->PutC('!');
			} /* fi */
		    IO->PutS(ResetSignal->Name);
		   } /* fi */
	       IO->PutC(')');
	       IO->TabStepUp();
	       SIR_LineInfo::WriteSPACE(IO, true);	// IO->PutC('{');
	       if (  (! Scope->Empty())
		   ||(! Scope->UserTypes->Empty()))
		  { if ((SIR_Error = Scope->WriteSC(IO, WriteNotes)))
		       { return(SIR_Error);
			} /* fi */
		    if (  (ResetState)
			||(DefaultStmnts)
			||(  (States)
			   &&(! States->Empty())))
		       { SIR_LineInfo::WriteVSPACE(IO);
			} /* fi */
		   } /* fi */
	       if (ResetState)
		  { sir_lineinfo	*LineInfo;
		    assert(Expression1 != NULL);
		    if ((LineInfo = Expression1->GetFirstLineInfo()))
		       { if ((SIR_Error = LineInfo->WriteSC(IO, true)))
			    { return(SIR_Error);
			     } /* fi */
			} /* fi */
		    else
		       { SIR_LineInfo::WriteSPACE(IO);
			} /* esle */
		    IO->PutS("if (");
		    IO->TabStepUp();
		    if ((SIR_Error = Expression1->WriteSC(IO, WriteNotes,
								CplusplusMode,
								SIR_TYPE_BOOL,
								0, true)))
		       { return(SIR_Error);
			} /* fi */
		  //IO->TabStepDown();
		    IO->PutC(')');
		  //IO->TabStepUp();
		    if ((SIR_Error = ResetState->WriteSC(IO, WriteNotes,
							CplusplusMode, true)))
		       { return(SIR_Error);
			} /* fi */
		    IO->TabStepDown();
		   } /* fi */
	       if (DefaultStmnts)
		  { IO->TabStepUp();
		    if ((SIR_Error = DefaultStmnts->WriteSC(IO, WriteNotes,
							CplusplusMode, true)))
		       { return(SIR_Error);
			} /* fi */
		    IO->TabStepDown();
		   } /* fi */
	       if (States)
		  { sir_fsmdstate	*State;
		    State = States->First();
		    while(State)
		       { if ((SIR_Error = State->WriteSC(IO, WriteNotes)))
			    { return(SIR_Error);
			     } /* fi */
			 State = State->Succ();
			} /* elihw */
		   } /* fi */
	       SIR_LineInfo::WriteNL(IO, false, true);	// IO->PutC('}');
	       IO->TabStepDown();
	      } /* esle */
	  break;
	 }
     case SIR_STMNT_WAIT:
	{ if (CplusplusMode)
	     { IO->PutS(SIR_SIM_WAIT_NAME "(");
	       if ((SIR_Error = Events->WriteSC(IO, ",", true)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->PutS(", ((void*)0));");
	      } /* fi */
	  else
	     { IO->PutS("wait(");
	       if ((SIR_Error = Events->WriteSC(IO)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->PutS(");");
	      } /* esle */
	  break;
	 }
     case SIR_STMNT_WAITAND:
	{ if (CplusplusMode)
	     { IO->PutS(SIR_SIM_WAITAND_NAME "(");
	       if ((SIR_Error = Events->WriteSC(IO, ",", true)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->PutS(", ((void*)0));");
	      } /* fi */
	  else
	     { IO->PutS("wait(");
	       if ((SIR_Error = Events->WriteSC(IO, " &&")))
		  { return(SIR_Error);
		   } /* fi */
	       IO->PutS(");");
	      } /* esle */
	  break;
	 }
     case SIR_STMNT_WAITFOR:
	{ if (CplusplusMode)
	     { IO->PutS(SIR_SIM_WAITFOR_NAME "(");
	      } /* fi */
	  else
	     { IO->PutS("waitfor ");
	      } /* esle */
	  if ((SIR_Error = Expression1->WriteSC(IO, WriteNotes, CplusplusMode,
						SIR_TYPE_TIME, 0, true)))
	     { return(SIR_Error);
	      } /* fi */
	  if (CplusplusMode)
	     { IO->PutC(')');
	      } /* fi */
	  IO->PutC(';');
	  break;
	 }
     case SIR_STMNT_NOTIFY:
	{ if (CplusplusMode)
	     { IO->PutS(SIR_SIM_NOTIFY_NAME "(");
	       if ((SIR_Error = Events->WriteSC(IO, ",", true)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->PutS(", ((void*)0));");
	      } /* fi */
	  else
	     { IO->PutS("notify(");
	       if ((SIR_Error = Events->WriteSC(IO)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->PutS(");");
	      } /* esle */
	  break;
	 }
     case SIR_STMNT_NOTIFYONE:
	{ if (CplusplusMode)
	     { IO->PutS(SIR_SIM_NOTIFYONE_NAME "(");
	       if ((SIR_Error = Events->WriteSC(IO, ",", true)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->PutS(", ((void*)0));");
	      } /* fi */
	  else
	     { IO->PutS("notifyone(");
	       if ((SIR_Error = Events->WriteSC(IO)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->PutS(");");
	      } /* esle */
	  break;
	 }
     default:
	{ assert(false);	/* bad StmntType */
	 }
    } /* hctiws */

if ((SIR_Error = IO->Check()))
   { return(SIR_Error);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::WriteSC */


sir_statement *SIR_Statement::WrapInCompound(	/* wrap in a compound stmnt.*/
	sir_symbols	*ParentScope)	/* (caller must fix father pointers) */
{
sir_statement	*NewCompound;

assert(this != NULL);		/* must exist */
assert(Function != NULL);	/* must know its function */
assert(GetList() == NULL);	/* must not be in a list */
assert(ParentScope != NULL);	/* must be given */

NewCompound = new SIR_Statement(SIR_STMNT_COMPOUND,
				Function,
				new SIR_Symbols(ParentScope,
						SIR_SCOPE_STATEMENT,
						Function),
				new SIR_Statements((sir_symbols*)NULL, this));
ParentStmnt = NewCompound;

/* caller must fix all scopes below to take on this new father */

return(NewCompound);

} /* end of SIR_Statement::WrapInCompound */


void SIR_Statement::CompoundSubBlocks(	/* compound all blocks below */
	void)				/* (insert missing compound stmnts.) */
{
sir_statement	*Stmnt,
		*NewCompound;

assert(this != NULL);			/* must exist */
assert(StmntType == SIR_STMNT_COMPOUND); /* must start from compound block */
assert(Statements != NULL);		/* must have statement list */
assert(Scope != NULL);			/* must have a scope to start with */

Stmnt = Statements->First();
while(Stmnt)
   { assert(Stmnt->ParentStmnt == this);
     if (Stmnt->Statement1)
	{ if (Stmnt->Statement1->StmntType != SIR_STMNT_COMPOUND)
	     { NewCompound = Stmnt->Statement1->WrapInCompound(Scope);
	       Stmnt->Statement1 = NewCompound;		/* plug it in */
	       NewCompound->ParentStmnt = Stmnt;
	      } /* fi */
	  else
	     { Stmnt->Statement1->Scope->Parent = Scope;	/* fix scope */
	       Stmnt->Statement1->Scope->UserTypes->Parent =	/* hierarchy */
					Scope->UserTypes;
	       assert(Stmnt->Statement1->ParentStmnt == Stmnt);
	      } /* esle */
	  Stmnt->Statement1->CompoundSubBlocks();	/* recursive! */
	 } /* fi */
     if (Stmnt->Statement2)
	{ if (Stmnt->Statement2->StmntType != SIR_STMNT_COMPOUND)
	     { NewCompound = Stmnt->Statement2->WrapInCompound(Scope);
	       Stmnt->Statement2 = NewCompound;		/* plug it in */
	       NewCompound->ParentStmnt = Stmnt;
	      } /* fi */
	  else
	     { Stmnt->Statement2->Scope->Parent = Scope;	/* fix scope */
	       Stmnt->Statement2->Scope->UserTypes->Parent =	/* hierarchy */
					Scope->UserTypes;
	       assert(Stmnt->Statement2->ParentStmnt == Stmnt);
	      } /* esle */
	  Stmnt->Statement2->CompoundSubBlocks();	/* recursive! */
	 } /* fi */
     if (Stmnt->Statements)
	{ Stmnt->CompoundSubBlocks();	/* recursive! */
	 } /* fi */
     if (Stmnt->Scope)	/* 'fsm' and 'fsmd' statements have local scope too! */
	{ Stmnt->Scope->Parent = Scope;			/* fix scope */
	  Stmnt->Scope->UserTypes->Parent =		/* hierarchy */
					Scope->UserTypes;
	 } /* fi */
     if (ResetState)
	{ assert(ResetState->StmntType == SIR_FSMDSTMNT_COMPOUND);
	  assert(ResetState->Scope->Parent == Stmnt->Scope); // no need to fix
	  ResetState->CompoundSubBlocks();
	 } /* fi */
     if (DefaultStmnts)
	{ assert(DefaultStmnts->StmntType == SIR_FSMDSTMNT_COMPOUND);
	  assert(DefaultStmnts->Scope->Parent == Stmnt->Scope);// no need to fix
	  DefaultStmnts->CompoundSubBlocks();
	 } /* fi */
     if (States)
	{ sir_fsmdstate	*State;
	  State = States->First();
	  while(State)
	     { assert(State->CmpndStmnt != NULL);
	       assert(State->CmpndStmnt->StmntType == SIR_FSMDSTMNT_COMPOUND);
	       assert(State->CmpndStmnt->Scope->Parent ==
						Stmnt->Scope); // no need to fix
	       State->CmpndStmnt->CompoundSubBlocks();
	       State = State->Succ();
	      } /* elihw */
	 } /* fi */
     Stmnt = Stmnt->Succ();
    } /* elihw */

} /* end of SIR_Statement::CompoundSubBlocks */


ERROR SIR_Statement::CreateDelayedAssignments(	/* create TmpVars */
	void)
{
sir_fsmdstate	*State;
sir_state_ptrs	*NextStates;

assert(StmntType == SIR_STMNT_FSMD);	// must be 'fsmd'
assert(Scope != NULL);			// 'fsmd' has local scope
assert(TmpVars == NULL);		// doesn't exist yet

// create list of temporary variables
// which will be temporarily defined in the 'fsmd' scope;
// also, delayed assignments and reachable next states
// will be computed and attached to each fsmd state;

TmpVars = new SIR_TmpVars(Scope);

// preparation: collect reachable next states from each state

if (States)
   { State = States->First();
     while(State)
	{ assert(State->NextStates == NULL);
	  State->NextStates = State->CollectReachableStates();
	  State = State->Succ();
	 } /* elihw */
    } /* fi */

// create delayed assignments and all needed temp. variables
// starting from every state

if (ResetState)
   { // compute the reachable states from reset state
     NextStates = new SIR_StatePtrs();
     if (States)
	{ State = States->First();
	  while(State)
	     { NextStates->Add(State);	// by default, every state can be
	       State = State->Succ();	// reached by returning from reset
	      } /* elihw */		// (if no 'goto' or 'break' is
	 } /* fi */			// present in the reset state)
     NextStates = ResetState->ReachableStates(	// apply 'goto' and 'break'
				NextStates);	// in the reset state
     // now create the delayed assignments
     // originating in the reset state
     if ((SIR_Error = ResetState->CreateDelayedAssignments(TmpVars,
						NextStates, "<reset>")))
	{ DeleteDelayedAssignments();
	  delete NextStates;
	  return(SIR_Error);
	 } /* fi */
     delete NextStates;
    } /* fi */

if (DefaultStmnts)
   { // compute the reachable states from default state
     NextStates = new SIR_StatePtrs();
     if (States)
	{ State = States->First();
	  while(State)			// all states' next states are reachable
	     { NextStates->Join(new SIR_StatePtrs(State->NextStates));
	       State = State->Succ();
	      } /* elihw */
	 } /* fi */
     // now create the delayed assignments
     // originating in the default state
     if ((SIR_Error = DefaultStmnts->CreateDelayedAssignments(TmpVars,
						NextStates, "<default>")))
	{ DeleteDelayedAssignments();
	  delete NextStates;
	  return(SIR_Error);
	 } /* fi */
     delete NextStates;
    } /* fi */
if (States)
   { State = States->First();
     while(State)
	{ // now create the delayed assignments
	  // originating from this single state
	  if ((SIR_Error = State->CmpndStmnt->CreateDelayedAssignments(
						TmpVars, State->NextStates,
						State->StateName.chars())))
	     { DeleteDelayedAssignments();
	       return(SIR_Error);
	      } /* fi */
	  State = State->Succ();
	 } /* elihw */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::CreateDelayedAssignments */


void SIR_Statement::DeleteDelayedAssignments(	/* delete TmpVars */
	void)
{
sir_fsmdstate	*State;

assert(StmntType == SIR_STMNT_FSMD);
assert(TmpVars != NULL);		// must exist

if (ResetState)
   { ResetState->DeleteDelayedAssignments();
    } /* fi */
if (DefaultStmnts)
   { DefaultStmnts->DeleteDelayedAssignments();
    } /* fi */
if (States)
   { State = States->First();
     while(State)
	{ State->CmpndStmnt->DeleteDelayedAssignments();
	  delete State->DelayedAssgns;
	  State->DelayedAssgns = NULL;
	  delete State->NextStates;
	  State->NextStates = NULL;
	  State = State->Succ();
	 } /* elihw */
    } /* fi */

delete TmpVars;
TmpVars = NULL;

} /* end of SIR_Statement::DeleteDelayedAssignments */



	/**********************/
	/*** SIR_Statements ***/
	/**********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Statements::SIR_Statements(		/* constructor #1 */
	sir_symbols	*CmpndScope /* = NULL */,
	sir_statement	*FirstEntry /* = NULL */) :
		SIR_List<SIR_Statement>(FirstEntry)
{

SIR_Statements::CmpndScope = CmpndScope;

} /* end of SIR_Statements::SIR_Statements #1 */


SIR_Statements::SIR_Statements(		/* constructor #3 (duplicator) */
	sir_statements	*Original,
	sir_statement	*ParentStmnt /* = NULL */)
{
sir_statement	*Curr;

Curr = Original->First();
while(Curr)
   { Append(new SIR_Statement(Curr, ParentStmnt));
     Curr = Curr->Succ();
    } /* elihw */

SIR_Statements::CmpndScope	= Original->CmpndScope;

} /* end of SIR_Statements::SIR_Statements #3 */


SIR_Statements::~SIR_Statements(void)	/* destructor */
{

/* nothing to do */

} /* end of SIR_Statements::~SIR_Statements */


ERROR SIR_Statements::DFS_ForAllNodes(	/* iterator over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{
sir_statement	*Stmnt,
		*Succ;

/* this is not a node, but there are nodes below */

Stmnt = First();
while(Stmnt)
   { Succ = Stmnt->Succ();
     if ((SIR_Error = Stmnt->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Stmnt = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statements::DFS_ForAllNodes */


ERROR SIR_Statements::DFS_ForAllSymbols( /* iterator over all symbols */
	sir_symbol_mptr	MemberFct,	/* (depth first) */
	sir_symbol_marg	MemberFctArg)
{
sir_statement	*Stmnt,
		*Succ;

/* this is not a symbol, but there are symbols below */

Stmnt = First();
while(Stmnt)
   { Succ = Stmnt->Succ();
     if ((SIR_Error = Stmnt->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Stmnt = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statements::DFS_ForAllSymbols */


ERROR SIR_Statements::DFS_ForAllUserTypes( /* iterator over all usertypes */
	sir_usertp_mptr	MemberFct,	/* (depth first) */
	sir_usertp_marg	MemberFctArg)
{
sir_statement	*Stmnt,
		*Succ;

/* this is not a usertype, but there are usertypes below */

Stmnt = First();
while(Stmnt)
   { Succ = Stmnt->Succ();
     if ((SIR_Error = Stmnt->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Stmnt = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statements::DFS_ForAllUserTypes */


ERROR SIR_Statements::DFS_ForAllNotes(	/* iterator over all notes */
	sir_note_mptr	MemberFct,	/* (depth first) */
	sir_note_marg	MemberFctArg)
{
sir_statement	*Stmnt,
		*Succ;

/* this is not a note, but there are notes below */

Stmnt = First();
while(Stmnt)
   { Succ = Stmnt->Succ();
     if ((SIR_Error = Stmnt->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Stmnt = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statements::DFS_ForAllNotes */


ERROR SIR_Statements::DFS_ForAllStatements(	/* iterator over all stmnts. */
	sir_stmnt_mptr	MemberFct,		/* (depth first) */
	sir_stmnt_marg	MemberFctArg)
{
sir_statement	*Stmnt,
		*Succ;

/* this is not a statement, but there are statements below */

Stmnt = First();
while(Stmnt)
   { Succ = Stmnt->Succ();
     if ((SIR_Error = Stmnt->DFS_ForAllStatements(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Stmnt = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statements::DFS_ForAllStatements */


ERROR SIR_Statements::DFS_ForAllExpressions( /* iterator over all expressions */
	sir_expr_mptr	MemberFct,		/* (depth first) */
	sir_expr_marg	MemberFctArg)
{
sir_statement	*Stmnt,
		*Succ;

/* this is not an expression, but there are expressions below */

Stmnt = First();
while(Stmnt)
   { Succ = Stmnt->Succ();
     if ((SIR_Error = Stmnt->DFS_ForAllExpressions(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Stmnt = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statements::DFS_ForAllExpressions */


bool SIR_Statements::DFS_FindDirectDependant(	/* searches direct deps. */
	sir_symbol	**SkipSymbol,
	sir_symbol	*ThatSymbol,
	sir_symbol	**DepSymbol,
	SIR_DEPENDENCY	*Reason)
{
sir_statement	*Stmnt;

assert(ThatSymbol != NULL);

/* this is not a statement, but there are statements below */

Stmnt = First();
while(Stmnt)
   { if (Stmnt->DFS_FindDirectDependant(SkipSymbol, ThatSymbol,
					DepSymbol, Reason))
	{ return(true);
	 } /* fi */
     Stmnt = Stmnt->Succ();
    } /* elihw */

return(false);

} /* end of SIR_Statements::DFS_FindDirectDependant */


bool SIR_Statements::DFS_FindIndirectDependant(	/* searches indirect deps. */
	sir_symbol	*ThatSymbol,
	sir_statement	**DepStmnt,
	sir_fsmdstmnt	**DepFsmdStmnt,
	sir_expression	**DepExpr,
	SIR_DEPENDENCY	*Reason)
{
sir_statement	*Stmnt;

assert(ThatSymbol != NULL);

/* this is not a statement, but there are statements below */

Stmnt = First();
while(Stmnt)
   { if (Stmnt->DFS_FindIndirectDependant(ThatSymbol, DepStmnt,
						DepFsmdStmnt, DepExpr,
						Reason))
	{ return(true);
	 } /* fi */
     Stmnt = Stmnt->Succ();
    } /* elihw */

return(false);

} /* end of SIR_Statements::DFS_FindDependant */


void SIR_Statements::SetAlias(	/* sets all type, usertype, symbol alias' */
	sir_statements	*Alias)	/* (iterates over symbols and usertypes) */
{
sir_statement	*Statement,
		*StatementAlias;

assert(Alias != NULL);

/* no need to process this node */

Statement = First();
StatementAlias = Alias->First();
while(Statement)
   { Statement->SetAlias(StatementAlias);
     Statement = Statement->Succ();
     StatementAlias = StatementAlias->Succ();
    } /* elihw */

} /* end of SIR_Statements::SetAlias */


sir_statement *SIR_Statements::FindLabeledStmnt(/* find a labeled statement */
	sir_label	*Label)		/* (returns NULL if not found) */
{
sir_statement	*Stmnt;

Stmnt = First();
while(Stmnt)
   { if (  (Stmnt->StmntType == SIR_STMNT_LABELED)
	 &&(Stmnt->Label == Label))
	{ break;
	 } /* fi */
     Stmnt = Stmnt->Succ();
    } /* elihw */

return(Stmnt);

} /* end of SIR_Statements::FindLabeledStmnt */


sir_statement *SIR_Statement::GetLoopOrSwitch(/* obtain control-flow parent */
        void)                        /* (for BREAK, CONTINUE, CASE, DEFAULT)*/
{
sir_statement   *Ancestor;

if (StmntType == SIR_STMNT_CONTINUE)    // for CONTINUE, ...
   { Ancestor = ParentStmnt;
     while(Ancestor)
        { if (  (Ancestor->StmntType == SIR_STMNT_WHILE)
              ||(Ancestor->StmntType == SIR_STMNT_DO_WHILE)
              ||(Ancestor->StmntType == SIR_STMNT_FOR))
             { return(Ancestor);        // ... return innermost loop
              } /* fi */
          Ancestor = Ancestor->ParentStmnt;
         } /* elihw */
     return(NULL);      // illegal: 'continue' outside loop!
    } /* fi */

if (  (StmntType == SIR_STMNT_CASE)     // for CASE and DEFAULT, ...
    ||(StmntType == SIR_STMNT_DEFAULT))
   { Ancestor = ParentStmnt;
     while(Ancestor)
        { if (Ancestor->StmntType == SIR_STMNT_SWITCH)
             { return(Ancestor);        // ... return innermost switch
              } /* fi */
          Ancestor = Ancestor->ParentStmnt;
         } /* elihw */
     return(NULL);      // illegal: 'case' or 'default' outside 'switch'!
    } /* fi */

/* BREAK and all others */              // for BREAK (and all others), ...
Ancestor = ParentStmnt;
while(Ancestor)
   { if (  (Ancestor->StmntType == SIR_STMNT_SWITCH)
         ||(Ancestor->StmntType == SIR_STMNT_WHILE)
         ||(Ancestor->StmntType == SIR_STMNT_DO_WHILE)
         ||(Ancestor->StmntType == SIR_STMNT_FOR))
        { return(Ancestor);             // ... return innermost loop or switch
         } /* fi */
     Ancestor = Ancestor->ParentStmnt;
    } /* elihw */

return(NULL);           // illegal: 'break' outside loop or 'switch'!

} /* end of SIR_Statement::GetLoopOrSwitch */


/*** exported functions *************************************************/


	/* none */


/* EOF Statement.cc */
