/************************************************************************/
/* DelydAssgn.cc: SpecC Internal Representation, DelydAssgn Class	*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 09/12/03 */
/************************************************************************/

/* last update: 09/26/06 */

/* modifications: (most recent first)
 *
 * 09/26/06 PC  Adjustments for scrc 2.1
 * 06/03/05 RD	reorganized and renamed global type names
 * 01/14/05 RD	added inclusion of IntRep/Extern.h
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 10/24/03 RD	added comment about future optimizations
 * 10/23/03 RD	added origin and state infos to delayed assignment
 * 10/22/03 RD	completed functionality
 * 10/22/03 RD	added collision checking
 * 10/06/03 RD	inserted even more functionality
 * 10/03/03 RD	inserted more functionality
 * 09/12/03 RD	initial version
 */

/* NOTE: This file provides special support for the C++ code generator   */
/*       when generating delayed assignments from/to temporary variables */
/*       in order to implement 'after' and 'piped' statements in 'fsmd'; */
/*       these data structures are used only temporarily during code     */
/*       generation; thus, this is internal only!                        */

#include "IntRep/DelydAssgn.h"
#include "IntRep/Design.h"

#include <assert.h>


/*** constants and macros ***********************************************/


	/* (none) */


/*** internal type declarations *****************************************/


	/* (none) */


/*** prototypes of internal functions ***********************************/


	/* (none) */


/*** class implementations **********************************************/


	/**********************/
	/*** SIR_DelydAssgn ***/
	/**********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_DelydAssgn::SIR_DelydAssgn(		/* constructor #1 */
	SIR_DELYDASSGN	Type,
	sir_tmpvar	*TmpVar,
	sir_expression	*Condition,
	int		Cycles,
	sir_fsmdstmnt	*Origin,
	const char	*OriginState,
	sir_fsmdstate	*State /* = NULL */)
{

SIR_DelydAssgn::Type		= Type;
SIR_DelydAssgn::TmpVar		= TmpVar;
SIR_DelydAssgn::Condition	= Condition;
SIR_DelydAssgn::Cycles		= Cycles;
SIR_DelydAssgn::Origin		= Origin;
SIR_DelydAssgn::OriginState	= OriginState;
SIR_DelydAssgn::State		= State;

} /* end of SIR_DelydAssgn::SIR_DelydAssgn #1 */


SIR_DelydAssgn::SIR_DelydAssgn(		/* constructor #2 (duplicator) */
	sir_delydassgn	*Original)
{

assert(Original != NULL);

SIR_DelydAssgn::Type		= Original->Type;
SIR_DelydAssgn::TmpVar		= Original->TmpVar;
SIR_DelydAssgn::Condition	= Original->Condition ?
					new SIR_Expression(
						Original->Condition)
					: NULL;
SIR_DelydAssgn::Cycles		= Original->Cycles;
SIR_DelydAssgn::Origin		= Original->Origin;
SIR_DelydAssgn::OriginState	= Original->OriginState;
SIR_DelydAssgn::State		= Original->State;

} /* end of IR_DelydAssgn::SIR_DelydAssgn #2 */


SIR_DelydAssgn::~SIR_DelydAssgn(void)	/* destructor */
{

delete Condition;

} /* end of SIR_DelydAssgn::~SIR_DelydAssgn */


int SIR_DelydAssgn::Cmp(		/* compare to entries (like strcmp) */
	sir_delydassgn	*Assgn1,
	sir_delydassgn	*Assgn2)
{
int		CmpVal;

assert(Assgn1 != NULL);
assert(Assgn2 != NULL);

// [0] quick identity check for efficiency

if (Assgn1 == Assgn2)
   { return(0);
    } /* fi */

// [1] condition has highest priority for comparison

if (Assgn1->Condition)
   { if (Assgn2->Condition)
	{ if (0 != (CmpVal = SIR_Expression::Cmp(Assgn1->Condition,
							Assgn2->Condition)))
	     { return(CmpVal);
	      } /* fi */
	 } /* fi */
     else
	{ return(-1);	// only 1 has condition, make it less than 2
	 } /* esle */
    } /* fi */
else
   { if (Assgn2->Condition)
	{ return(1);	// only 2 has condition, make it greater than 1
	 } /* fi */
     else
	{ // both have no condition, continue comparison
	 } /* esle */
    } /* esle */

// [2] target has second-highest priority for comparison

if (0 != (CmpVal = strcmp(Assgn1->TmpVar->Symbol->Name.chars(),
				Assgn2->TmpVar->Symbol->Name.chars())))
   { return(CmpVal);
    } /* fi */

// [3] PIPED/AFTER type has lower priority for comparison

// note that the following type-comparison has low
// comparison priority here; this is intended
// as it ensures that AFTER/PIPED assignments with
// similar parameters (target, condition)
// are grouped together, i.e. first a block of
// AFTER assignments, then the PIPED assignments;
// this makes the collision check (see the next method!)
// more efficient

if (0 != (CmpVal = Assgn1->Type - Assgn2->Type))
   { return(CmpVal);
    } /* fi */

// [4] cycle count has lowest priority for comparison

// group similar AFTER/PIPED assignments together
// in order of increasing cycle count;
// note that this order is important (!)
// as it determines the order of temp. assignments
// in the same state; in particular, this makes
// the 'piped' assignments shift in pipeline fashion!

return(Assgn1->Cycles - Assgn2->Cycles);

} /* end of SIR_DelydAssgn::Cmp */


bool SIR_DelydAssgn::CheckCollision(	/* check for unintended collision */
	bool	IssueWarning /* = true */)	/* (and warn about it) */
{
sir_delydassgn	*Other;

// note that because of the special ordering defined
// by the Cmp() method above, colliding PIPED/AFTER
// assignments are located in two groups next to each other
// in the list; in particular, a colliding AFTER-entry is
// located in a group before the PIPED entry, and vice versa;
// therefore, the collision check can be done here with
// a minimum of searching

if (Type == SIR_DELYDASSGN_AFTER)
   { if (Cycles < 1)
	{ return(false);		// (a >= 1)
	 } /* fi */
     // search forward in the list
     Other = Succ();
     while(  (Other)
	   &&(Other->Type == Type)
	   &&(Other->TmpVar == TmpVar)
	   &&(  (  (Other->Condition == NULL)
		 &&(Condition == NULL))
	      ||(  (Other->Condition != NULL)
		 &&(Condition != NULL)
		 &&(0 == SIR_Expression::Cmp(Condition, Other->Condition)))))
	{ Other = Other->Succ(); // skip all in same group
	 } /* elihw */
     while(  (Other)
	   &&(Other->Type == SIR_DELYDASSGN_PIPED)
	   &&(Other->TmpVar == TmpVar)
	   &&(  (  (Other->Condition == NULL)
		 &&(Condition == NULL))
	      ||(  (Other->Condition != NULL)
		 &&(Condition != NULL)
		 &&(0 == SIR_Expression::Cmp(Condition, Other->Condition)))))
	{ if (Other->Cycles == 1)	// (p == 1)
	     { if (IssueWarning)
		  { sir_lineinfo	*LineInfo;
		    gl_string		Location,
					OtherLocation;

		    if ((LineInfo = Origin->LineInfo))
		       { Location.form("line %u, file \"%s\"",
					LineInfo->Line,
					LineInfo->File->Filename.chars());
			} /* fi */
		    else
		       { Location = "line <unknown>, file <unknown>";
			} /* esle */
		    if ((LineInfo = Other->Origin->LineInfo))
		       { OtherLocation.form("line %u, file \"%s\"",
					LineInfo->Line,
					LineInfo->File->Filename.chars());
			} /* fi */
		    else
		       { OtherLocation = "line <unknown>, file <unknown>";
			} /* esle */
		    GL_PrintWarningFmt(GL_WARN_STANDARD,
			"In state %s, a 'piped' assignment from state %s"
							GL_WARN_MSG_NEWLINE
			"(%s)" GL_WARN_MSG_NEWLINE
			"invalidates an 'after' assignment in state %s"
							GL_WARN_MSG_NEWLINE
			"(%s)." GL_WARN_MSG_NEWLINE
			"Both assignments target the same variable %s",
				State->StateName.chars(),
				Other->OriginState,
				OtherLocation.chars(),
				OriginState,
				Location.chars(),
				TmpVar->Target->Print(false, 0, 0, 0));
		   } /* fi */
	       return(true);	// collision found!
	      } /* fi */
	  Other = Other->Succ(); // check next candidate in group
	 } /* elihw */
    } /* fi */
else
   { assert(Type == SIR_DELYDASSGN_PIPED);
     if (Cycles != 1)
	{ return(false);		// (p == 1)
	 } /* fi */
     // search backward in the list
     Other = Pred();
     while(  (Other)
	   &&(Other->Type == Type)
	   &&(Other->TmpVar == TmpVar)
	   &&(  (  (Other->Condition == NULL)
		 &&(Condition == NULL))
	      ||(  (Other->Condition != NULL)
		 &&(Condition != NULL)
		 &&(0 == SIR_Expression::Cmp(Condition, Other->Condition)))))
	{ Other = Other->Pred(); // skip all in same group
	 } /* elihw */
     while(  (Other)
	   &&(Other->Type == SIR_DELYDASSGN_AFTER)
	   &&(Other->TmpVar == TmpVar)
	   &&(  (  (Other->Condition == NULL)
		 &&(Condition == NULL))
	      ||(  (Other->Condition != NULL)
		 &&(Condition != NULL)
		 &&(0 == SIR_Expression::Cmp(Condition, Other->Condition)))))
	{ if (Other->Cycles > 0)	// (a >= 1)
	     { if (IssueWarning)
		  { sir_lineinfo	*LineInfo;
		    gl_string		Location,
					OtherLocation;

		    if ((LineInfo = Origin->LineInfo))
		       { Location.form("line %u, file \"%s\"",
					LineInfo->Line,
					LineInfo->File->Filename.chars());
			} /* fi */
		    else
		       { Location = "line <unknown>, file <unknown>";
			} /* esle */
		    if ((LineInfo = Other->Origin->LineInfo))
		       { OtherLocation.form("line %u, file \"%s\"",
					LineInfo->Line,
					LineInfo->File->Filename.chars());
			} /* fi */
		    else
		       { OtherLocation = "line <unknown>, file <unknown>";
			} /* esle */
		    GL_PrintWarningFmt(GL_WARN_STANDARD,
			"In state %s, a 'piped' assignment from state %s"
							GL_WARN_MSG_NEWLINE
			"(%s)" GL_WARN_MSG_NEWLINE
			"invalidates an 'after' assignment in state %s"
							GL_WARN_MSG_NEWLINE
			"(%s)." GL_WARN_MSG_NEWLINE
			"Both assignments target the same variable %s",
				State->StateName.chars(),
				OriginState,
				Location.chars(),
				Other->OriginState,
				OtherLocation.chars(),
				TmpVar->Target->Print(false, 0, 0, 0));
		   } /* fi */
	       return(true);	// collision found!
	      } /* fi */
	  Other = Other->Pred(); // check next candidate in group
	 } /* elihw */
    } /* esle */

return(false);			// no collision found

} /* end of SIR_DelydAssgn::CheckCollision */


ERROR SIR_DelydAssgn::WriteSC(		/* generates C++ source code */
	gl_io		*IO,
	bool		WriteNotes)
{
sir_expression	*LHS,
		*RHS,
		*Assignment;
ERROR		Error;

assert(TmpVar != NULL);
assert(TmpVar->Symbol != NULL);
assert(TmpVar->Target != NULL);

if (Condition)
   { SIR_LineInfo::WriteSPACE(IO);
     IO->PutS("if (");
     IO->TabStepUp();
     if ((Error = Condition->WriteSC(IO, WriteNotes, true,
					SIR_TYPE_BOOL, 0, true)))
	{ return(Error);
	 } /* fi */
     //IO->TabStepDown();
     IO->PutC(')');
     //IO->TabStepUp();
    } /* fi */

if (Cycles > 0)
   { if (!(LHS = SIR_Expression::New(SIR_EXPR_ARRAY_ACCESS,
			SIR_Expression::New(TmpVar->Symbol),
			SIR_Expression::New(
				SIR_Constant::New((LONG_LONG) Cycles-1),
				TmpVar->Symbol->Type->GetTable()))))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
else
   { LHS = new SIR_Expression(TmpVar->Target);	// copy!
     assert(LHS != NULL);
    } /* esle */
if (!(RHS = SIR_Expression::New(SIR_EXPR_ARRAY_ACCESS,
			SIR_Expression::New(TmpVar->Symbol),
			SIR_Expression::New(
				SIR_Constant::New((LONG_LONG) Cycles),
				TmpVar->Symbol->Type->GetTable()))))
   { delete LHS;
     return(SIR_Error);
    } /* fi */

if (!(Assignment = SIR_Expression::New(SIR_EXPR_ASSIGNMENT, LHS, RHS)))
   { return(SIR_Error);
    } /* fi */

SIR_LineInfo::WriteSPACE(IO);

#ifdef HAVE_ARYASGN

if ((Error = Assignment->WriteSC(IO, WriteNotes, true)))
   { delete Assignment;
     return(Error);
    } /* fi */
IO->PutC(';');

#else /* ! HAVE_ARYASGN */

assert(Assignment->ExprType == SIR_EXPR_ASSIGNMENT);
if (  (Assignment->Arg1->Type->Type == SIR_TYPE_ARRAY)
    &&(Assignment->Arg2->Type->Type == SIR_TYPE_ARRAY))
   { if ((SIR_Error = Assignment->WriteArrayAssignment(IO, WriteNotes)))
	{ delete Assignment;
	  return(SIR_Error);
	 } /* fi */
    } /* fi */
else
   { if ((SIR_Error = Assignment->WriteSC(IO, WriteNotes, true)))
	{ delete Assignment;
	  return(SIR_Error);
	 } /* fi */
     IO->PutC(';');
    } /* esle */

#endif /* HAVE_ARYASGN */

delete Assignment;

if (Condition)
   { IO->TabStepDown();
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_DelydAssgn::WriteSC */


	/**********************/
	/*** SIR_DlydAssgns ***/
	/**********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_DlydAssgns::SIR_DlydAssgns(		/* constructor #1 */
	sir_delydassgn	*FirstEntry /* = NULL */) :
		SIR_List<SIR_DelydAssgn>(FirstEntry)
{

/* nothing else to do */

} /* end of SIR_DlydAssgns::SIR_DlydAssgns */


SIR_DlydAssgns::~SIR_DlydAssgns(void)	/* destructor */
{

/* nothing to do */

} /* end of SIR_DlydAssgns::~SIR_DlydAssgns */


sir_delydassgn *SIR_DlydAssgns::Find(	/* find an assignment in the list */
	sir_delydassgn	*Assign)	/* (returns NULL if not found) */
{
int		CmpVal;

First();	/* search for the entry */
while(Curr())
   { if (0 == (CmpVal = SIR_DelydAssgn::Cmp(Assign, Curr())))
	{ return(Curr());	/* found the entry */
	 } /* fi */
     if (CmpVal < 0)
	{ break;	/* not found (insert here) */
	 } /* fi */
     Next();
    } /* elihw */

return(NULL);	/* not found */

} /* end of SIR_DlydAssgns::Find */


sir_delydassgn *SIR_DlydAssgns::Insert(	/* insert (copy of) assignment */
	sir_delydassgn	*Assignment)	/* (returns NULL if already exists) */
{

assert(Assignment != NULL);

if (Find(Assignment))
   { return(NULL);	// already there, nothing to do
    } /* fi */

if (Curr())
   { return(InsertBefore(new SIR_DelydAssgn(Assignment)));
    } /* fi */
else
   { return(Append(new SIR_DelydAssgn(Assignment)));
    } /* esle */

} /* end of SIR_DlydAssgns::Insert */


ERROR SIR_DlydAssgns::WriteSC(		/* generates C++ source code */
	gl_io		*IO,
	bool		WriteNotes)
{
sir_delydassgn	*Assign;
ERROR		Error;

// note:
// for now, delayed assignments are simply printed one by one,
// each with its own condition; however, there is a lot of room
// for improvement and optimization here, as many conditions
// are usually the same or very much alike, so they should be
// combined in order to eliminate repeated evaluations;
// this is especially true because the same conditions are already
// grouped together in the list (see Cmp() method above)!
// due to lack of time, however, this optimization is left
// to be done in the future;			(RD, 10/24/03)

Assign = First();
while(Assign)
   { if (  (Assign->Type == SIR_DELYDASSGN_PIPED)
	 ||(Assign->Cycles == 0))	// skip intermediate AFTER assignments
	{ if ((Error = Assign->WriteSC(IO, WriteNotes)))
	     { return(Error);
	      } /* fi */
	 } /* fi */
     Assign = Assign->Succ();
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_DlydAssgns::WriteSC */


	/******************/
	/*** SIR_TmpVar ***/
	/******************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_TmpVar::SIR_TmpVar(			/* constructor #1 */
	sir_symbol	*Symbol,
	sir_expression	*Target)
{

SIR_TmpVar::Symbol	= Symbol;
SIR_TmpVar::Target	= Target;

} /* end of SIR_TmpVar::SIR_TmpVar */


SIR_TmpVar::~SIR_TmpVar(void)		/* destructor */
{

if (Symbol)		// although this is a link (not a pointer),
   { Symbol->Remove();	// we do want to remove this symbol upon cleanup
     delete Symbol;
    } /* fi */

} /* end of SIR_TmpVar::~SIR_TmpVar */


	/*******************/
	/*** SIR_TmpVars ***/
	/*******************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_TmpVars::SIR_TmpVars(		/* constructor #1 */
	sir_symbols	*Scope,
	sir_tmpvar	*FirstEntry /* = NULL */) :
		SIR_List<SIR_TmpVar>(FirstEntry)
{

assert(Scope != NULL);

SIR_TmpVars::Scope	= Scope;

} /* end of SIR_TmpVars::SIR_TmpVars */


SIR_TmpVars::~SIR_TmpVars(void)	/* destructor */
{

/* nothing to do */

} /* end of SIR_TmpVars::~SIR_TmpVars */


sir_tmpvar *SIR_TmpVars::Find(		/* find an entry by target expr. */
	sir_expression	*Target)	/* (where Expr::Cmp() == 0) */
{
int		CmpVal;

First();	/* search for the entry */
while(Curr())
   { if (0 == (CmpVal = SIR_Expression::Cmp(Target, Curr()->Target)))
	{ return(Curr());	/* found the entry */
	 } /* fi */
     if (CmpVal < 0)
	{ break;	/* not found (insert here) */
	 } /* fi */
     Next();
    } /* elihw */

return(NULL);	/* not found */

} /* end of SIR_TmpVars::Find */


sir_tmpvar *SIR_TmpVars::FindOrInsert(	/* find old or insert new temp. var. */
	sir_expression	*Target,	/* for this target expression */
	int		Size)
{
const char	*TmpName;
sir_type	*TmpType;
sir_symbol	*TmpSymbol;
sir_tmpvar	*TmpVar;


assert(Target != NULL);
assert(Size > 0);
assert(Scope != NULL);

if ((TmpVar = Find(Target)))
   { assert(TmpVar->Symbol != NULL);
     assert(TmpVar->Symbol->Type->Type == SIR_TYPE_ARRAY);
     if (TmpVar->Symbol->Type->Size < Size)
	{ /* resize the temp. variable (extend the array size) */
	  TmpType = TmpVar->Symbol->Type->SubType;
	  TmpType = TmpType->GetTable()->FindOrInsert(
			SIR_TYPE_ARRAY, TmpType, NULL, Size);
	  TmpVar->Symbol->Type = TmpType;	/* just overwrite the type! */
	 } /* fi */
     return(TmpVar);
    } /* fi */

TmpName = Scope->CreateNewName(SIR_CXX_FSMD_TMPVAR_PFX);
TmpType = Target->Type->GetTable()->FindOrInsert(
			SIR_TYPE_ARRAY, Target->Type, NULL, Size);
TmpSymbol = new SIR_Symbol(SIR_SYMBOL_IDENTIFIER, TmpName, TmpType);
TmpVar = new SIR_TmpVar(Scope->Insert(TmpSymbol), Target);

if (Curr())
   { return(InsertBefore(TmpVar));
    } /* fi */
else
   { return(Append(TmpVar));
    } /* esle */

} /* end of SIR_TmpVars::FindOrInsert */


/*** exported functions *************************************************/


	/* (none) */


/************************************************************************/
/*** implementation of internal functions			      ***/
/************************************************************************/


	/* (none) */


/* EOF DelydAssgn.cc */
