/************************************************************************/
/* IntRep/Note.h: SpecC Internal Representation, Note Classes		*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 10/08/97 */
/************************************************************************/

/* last update: 09/26/06 */

/* modifications: (most recent first)
 *
 * 09/26/06 PC  Adjustments for scrc 2.1
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 02/13/03 RD	added support for notes at fsmd states
 * 01/15/02 RD	made SIR_Note inherit from SIR_Node to store location info
 * 01/14/02 RD	added support for composite annotations
 * 11/08/01 RD	switched code generation to use GL_IO layer
 * 05/31/01 RD	eliminated level 2 of SIR API
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 05/25/01 RD	introduced this header (last change was 04/13/99)
 */

#ifndef INTREP_NOTE_H
#define INTREP_NOTE_H


#include "Global.h"
#include "IntRep/Initializer.h"
#include "IntRep/Import.h"


/*** constants and macros ***********************************************/


	/* none */


/*** enumeration types **************************************************/


	/* none */


/*** type definitions ***************************************************/


typedef class SIR_Note			sir_note;
typedef SIR_List<sir_note>		sir_note_list;
typedef class SIR_Notes			sir_notes;
typedef void *				sir_note_marg;
typedef ERROR (SIR_Note::*sir_note_mptr)(void*);

typedef class SIR_Symbol		sir_symbol;	/* cyclic link */
typedef class SIR_UserType		sir_usertype;	/* cyclic link */
typedef class SIR_Label			sir_label;	/* cyclic link */
typedef class SIR_Design		sir_design;	/* cyclic link */
typedef class SIR_FsmdState		sir_fsmdstate;	/* cyclic link */


/*** class declarations *************************************************/


	/****************/
	/*** SIR_Note ***/
	/****************/


class SIR_Note :		/* note (entry in a list of notes) */
	public SIR_Node,		/* is a node, and    */
	public SIR_ListElem<SIR_Note>	/* is a list element */
{
public:
gl_string	Name;		/* name of this note */
sir_initializer	*Contents;	/* contents of this note */
sir_symbol	*Symbol;	/* link back to symbol (or NULL) */
sir_usertype	*UserType;	/* link back to usertype (or NULL) */
sir_label	*Label;		/* link back to label (or NULL) */
sir_fsmdstate	*State;		/* link back to state (or NULL) */
sir_import	*Imported;	/* link to import file (NULL if not imported) */


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Note(			/* constructor #1 */
	const char	*Name,
	sir_initializer	*Contents,
	sir_symbol	*Symbol = NULL,		/* max. one */
	sir_usertype	*UserType = NULL,	/* of these */
	sir_label	*Label = NULL,		/* can be   */
	sir_fsmdstate	*State = NULL,		/* not NULL */
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL);

SIR_Note(			/* constructor #3 (duplicator) */
	sir_note	*Original);

~SIR_Note(void);		/* destructor */


ERROR DFS_ForAllNodes(		/* iterator over all nodes (depth first) */
	sir_node_mptr	MemberFct,
	sir_node_marg	MemberFctArg);

ERROR DFS_ForAllNotes(		/* iterator over all notes (depth first) */
	sir_note_mptr	MemberFct,
	sir_note_marg	MemberFctArg);


ERROR SetImported(		/* sets the Imported pointer (if unset) */
	sir_note_marg	ImportPtr);	/* or resets it if NULL */

ERROR TakeOverImport(		/* takes over an imported note */
	sir_note_marg	ImportPtr);

ERROR MarkImportEntry(		/* marks its import entry being used */
	sir_note_marg	/* Unused */);

ERROR RemoveDoubleImport(	/* deletes itself if it is double-imported */
	sir_note_marg	/* Unused */);


const char *Print(		/* print SpecC code (#1) */
	gl_string	*Buffer,	/* (Buffer must be initialized) */
	bool		CplusplusMode = false);

ERROR WriteSC(			/* (re-) generates SpecC source code */
	gl_io		*IO,
	bool		CplusplusMode = false);

};


	/*****************/
	/*** SIR_Notes ***/
	/*****************/


class SIR_Notes :		/* notes */
	public SIR_List<SIR_Note>	/* is simply a list of notes */
{					/* with additional member(function)s */
public:

//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Notes(			/* constructor #1 */
	sir_note	*FirstEntry = NULL);

SIR_Notes(			/* constructor #2 (duplicator) */
	sir_notes	*Original);

~SIR_Notes(void);		/* destructor */


ERROR DFS_ForAllNodes(		/* iterator over all nodes (depth first) */
	sir_node_mptr	MemberFct,
	sir_node_marg	MemberFctArg);

ERROR DFS_ForAllNotes(		/* iterator over all notes (depth first) */
	sir_note_mptr	MemberFct,
	sir_note_marg	MemberFctArg);


sir_note *Find(				/* find an entry */
	const char	*Name);		/* (returns NULL if not found) */

sir_note *Insert(			/* inserts a new note */
	sir_note	*NewNote);	/* (may return NULL and SIR_Error) */

static ERROR Merge(			/* merges two note-tables */
	sir_notes	**Notes1,	/* (merges 2 into 1) */
	sir_notes	**Notes2);	/* (supports automatic unaliasing) */


const char *Print(		/* print SpecC code (#1) */
	bool		DoIt,
	bool		CplusplusMode = false);

ERROR WriteSC(			/* (re-) generates SpecC source code */
	gl_io		*IO,
	bool		DoIt,
	bool		CplusplusMode = false);

};


#endif /* INTREP_NOTE_H */

/* EOF IntRep/Note.h */
