/************************************************************************/
/* PosixThread.cc: SpecC thread implementation based on Posix Threads	*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 03/26/01 */
/************************************************************************/

/* last update: 06/15/04 */

/* modifications: (most recent first)
 *
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 04/12/04 RD	moved RunningThread into the thread_base class
 * 05/16/01 RD	resolved name space competition with user by moving more
 *		definitions into _specc name space and adding prefixes
 * 05/16/01 RD	separated common thread code (in thread.[h|cc]) from
 *		platform-specific thread code (in PosixThread.[h|cc])
 * 03/26/01 RD	initial version
 */

#include "specc.h"
#include "thread.h"
#include "PosixThread.h"

#include <assert.h>


/*** constants and macros ***********************************************/


//#define SIM_PTHREAD_DEBUG	/* enable/disable debugging */


/*** internal type declarations *****************************************/


typedef int		PosixError;	/* pthreads error type */


/*** internal function prototypes ***************************************/


static void *PThreadWrapper(	/* wrapper around the actual task */
	void		*Arg);

static void CleanUpHandler(	/* clean-up handler for dying threads */
	void	* /* DyingThread */);


/*** internal variables *************************************************/


static pthread_attr_t	Attributes;	/* attributes for new threads */

static pthread_mutex_t	TheMutex;	/* the mutex */
					/* (locked by the running thread) */

#ifdef SIM_PTHREAD_DEBUG

/* regarding spurious wakeups, quote from "man pthread_cond_wait":
 *
 *   When using condition variables there  is  always  a  boolean
 *   predicate,   an  invariant,   associated with each condition
 *   wait that must be true before  the  thread  should  proceed.
 *   Spurious    wakeups    from   the   pthread_cond_wait()   or
 *   pthread_cond_timedwait()  functions  may  occur.  Since  the
 *   return  from pthread_cond_wait() or pthread_cond_timedwait()
 *   does not imply anything about the value of  this  predicate,
 *   the predicate should always be re-evaluated.
 */
static unsigned long	SpuriousWakeups = 0;

#endif /* SIM_PTHREAD_DEBUG */


/*** exported variables *************************************************/


	/* none */


/************************************************************************/
/*** implementation of internal functions			      ***/
/************************************************************************/


static void *PThreadWrapper(	/* wrapper around the actual task */
	void		*Arg)
{
_specc::thread_base	*Myself;
PosixError		Error;

assert(Arg != NULL);
Myself = (_specc::thread_base*) Arg;

#ifdef SIM_PTHREAD_DEBUG
puts("SIM_PTHREAD_DEBUG: Thread born, initializing...");
#endif /* SIM_PTHREAD_DEBUG */

if ((Error = pthread_mutex_lock(	/* first obtain the mutex */
			&TheMutex)))
   { _specc::fatalError("Child born cannot lock the mutex", Error);
    } /* fi */

pthread_cleanup_push(&CleanUpHandler,	/* install clean-up handler */
			(void*)Myself);	/* for clean cancellation   */

if ((Error = pthread_setcancelstate(	/* enable cancellation */
			PTHREAD_CANCEL_ENABLE, NULL)))
   { _specc::fatalError("Child born cannot enable cancellation", Error);
    } /* fi */

#ifdef SIM_PTHREAD_DEBUG
puts("SIM_PTHREAD_DEBUG: New thread signalling the creator...");
#endif /* SIM_PTHREAD_DEBUG */

assert(_specc::thread_base::RunningThread == Myself);	/* it's my initial turn now */
assert(Myself->ThreadCreator != NULL);	/* this is my creator */

Myself->ThreadRun(		/* transfer control back to the creator */
	Myself->ThreadCreator);	/* (the schedule will wake me up later) */

Myself->ThreadCreator = NULL;	/* my creator probably has died by now */

#ifdef SIM_PTHREAD_DEBUG
puts("SIM_PTHREAD_DEBUG: New thread starts its work...");
#endif /* SIM_PTHREAD_DEBUG */

Myself->ThreadFunction(Myself->ThreadArg);	/* perform the work */

#ifdef SIM_PTHREAD_DEBUG
puts("SIM_PTHREAD_DEBUG: Thread is done, unlocking the mutex and dying...");
#endif /* SIM_PTHREAD_DEBUG */

pthread_cleanup_pop(true);	/* de-install and call the clean-up handler */

return(NULL);	/* dummy result */

} /* end of PThreadWrapper */


static void CleanUpHandler(	/* clean-up handler for dying threads */
	void	* /* DyingThread */)
{
PosixError	Error;

if ((Error = pthread_mutex_unlock(&TheMutex)))
   { _specc::fatalError("Cannot unlock the mutex to clean up", Error);
    } /* fi */

} /* end of CleanUpHandler */


/*** class implementations **********************************************/


	/***************************/
	/*** _specc::thread_base ***/
	/***************************/


_specc::thread_base *_specc::thread_base::RunningThread = NULL;	/* the running thread */


_specc::thread_base::thread_base(	/* constructor #1 */
	thread_base	*Creator)
{
PosixError	Error;

_specc::thread_base::ThreadFunction	= NULL;
_specc::thread_base::ThreadArg	= NULL;
_specc::thread_base::ThreadCreator	= Creator;
_specc::thread_base::ThreadHandle	= 0;

if ((Error = pthread_cond_init(&CondVar,
				NULL)))	/* default is fine */
   { fatalError("Cannot initialize condition variable", Error);
    } /* fi */

} /* end of _specc::thread_base::thread_base */


_specc::thread_base::~thread_base(void)	/* destructor */
{

/* nothing to do */

} /* end of _specc::thread_base::~thread_base */


void _specc::thread_base::ThreadStart(	/* initialize thread usage */
	void)
{
PosixError	Error;

#ifdef SIM_PTHREAD_DEBUG
puts("SIM_PTHREAD_DEBUG: Initializing thread attributes...");
#endif /* SIM_PTHREAD_DEBUG */
if ((Error = pthread_attr_init(&Attributes)))
   { fatalError("Cannot initialize thread attributes", Error);
    } /* fi */
if ((Error = pthread_attr_setdetachstate(&Attributes,
				PTHREAD_CREATE_JOINABLE)))
   { fatalError("Cannot set thread attributes to joinable", Error);
    } /* fi */
/* we don't care about other attributes, any defaults are fine */

#ifdef SIM_PTHREAD_DEBUG
puts("SIM_PTHREAD_DEBUG: Initializing the mutex...");
#endif /* SIM_PTHREAD_DEBUG */
if ((Error = pthread_mutex_init(&TheMutex,
				NULL /* default attributes */)))
   { fatalError("Cannot initialize the mutex", Error);
    } /* fi */

RunningThread = NULL;	/* initialization */

} /* end of _specc::thread_base::ThreadStart */


void _specc::thread_base::ThreadEnd(	/* clean up after thread usage */
	void)
{
PosixError	Error;

assert(RunningThread == NULL);	/* clean exit */

#ifdef SIM_PTHREAD_DEBUG
printf("SIM_PTHREAD_DEBUG: Number of spurious wakeups = %lu.\n",
			(unsigned long)SpuriousWakeups);
#endif /* SIM_PTHREAD_DEBUG */

#ifdef SIM_PTHREAD_DEBUG
puts("SIM_PTHREAD_DEBUG: Destroying thread attributes...");
#endif /* SIM_PTHREAD_DEBUG */
if ((Error = pthread_attr_destroy(&Attributes)))
   { fatalError("Cannot destroy thread attributes", Error);
    } /* fi */

#ifdef SIM_PTHREAD_DEBUG
puts("SIM_PTHREAD_DEBUG: Destroying the mutex...");
#endif /* SIM_PTHREAD_DEBUG */
if ((Error = pthread_mutex_destroy(&TheMutex)))
   { fatalError("Cannot destroy the mutex", Error);
    } /* fi */

} /* end of _specc::thread_base::ThreadEnd */


void _specc::thread_base::ThreadCreate(	/* create this new thread */
	thread_fct	Function,	/* (NULL for root thread) */
	thread_arg	Arg)		/* (NULL for root thread) */
{
thread_base	*Creator;
PosixError	Error;

#ifdef SIM_PTHREAD_DEBUG
puts("SIM_PTHREAD_DEBUG: Creating a new thread...");
#endif /* SIM_PTHREAD_DEBUG */

assert(ThreadFunction == NULL);	/* must be unused */
assert(ThreadArg == NULL);
ThreadFunction	= Function;	/* store the function */
ThreadArg	= Arg;		/* and the argument   */

if (Function)			/* create new child thread */
   { if ((Error = pthread_create(&ThreadHandle, &Attributes,
				&PThreadWrapper, (void*)this)))
	{ fatalError("Cannot create new thread", Error);
	 } /* fi */
#ifdef SIM_PTHREAD_DEBUG
     puts("SIM_PTHREAD_DEBUG: Get new thread in starting position...");
#endif /* SIM_PTHREAD_DEBUG */
     Creator = RunningThread;
     assert(Creator != NULL);
     RunningThread = this;	/* temporary only */
     do { if ((Error = pthread_cond_wait(&Creator->CondVar, &TheMutex)))
	     { fatalError("Cannot wait on child to get started", Error);
	      } /* fi */
#ifdef SIM_PTHREAD_DEBUG
	  if (RunningThread != Creator)
	     { SpuriousWakeups++;
	      } /* fi */
#endif /* SIM_PTHREAD_DEBUG */
	 } while(RunningThread != Creator);
     assert(RunningThread == Creator);
    } /* fi */
else				/* create root thread */
   { assert(RunningThread == NULL);	/* not initialized yet */
     /* we are already running */
     RunningThread = this;
     if ((Error = pthread_mutex_lock(&TheMutex)))	/* lock the mutex */
	{ fatalError("Cannot lock the mutex for root", Error);
	 } /* fi */
    } /* esle */

#ifdef SIM_PTHREAD_DEBUG
printf("SIM_PTHREAD_DEBUG: new thread ID = %lu.\n",
			(unsigned long)ThreadHandle);
#endif /* SIM_PTHREAD_DEBUG */

} /* end of _specc::thread_base::ThreadCreate */


void _specc::thread_base::ThreadDelete(	/* delete this thread */
	void)
{
PosixError	Error;

if (ThreadFunction)
   { assert(RunningThread != this);
     ThreadFunction = NULL;
     ThreadArg = NULL;
    } /* fi */
else	/* delete the root thread */
   { assert(RunningThread == this);
     if ((Error = pthread_mutex_unlock(&TheMutex)))	/* unlock the mutex */
	{ fatalError("Cannot unlock the mutex for root", Error);
	 } /* fi */
     ThreadArg = NULL;
     RunningThread = NULL;
    } /* esle */

} /* end of _specc::thread_base::ThreadDelete */


void _specc::thread_base::ThreadJoin(	/* wait for a thread to complete */
	thread_base	*Thread)
{
void		*ExitStatus;
PosixError	Error;

assert(RunningThread == this);

#ifdef SIM_PTHREAD_DEBUG
puts("SIM_PTHREAD_DEBUG: Start joining, unlocking the mutex...");
#endif /* SIM_PTHREAD_DEBUG */
if ((Error = pthread_mutex_unlock(&TheMutex)))
   { fatalError("Cannot unlock the mutex", Error);
    } /* fi */

RunningThread = Thread;	/* temporary only */
if ((Error = pthread_cond_signal(&Thread->CondVar)))
   { fatalError("Cannot signal thread to terminate", Error);
    } /* fi */
if ((Error = pthread_join(Thread->ThreadHandle, &ExitStatus)))
   { fatalError("Cannot join with thread", Error);
    } /* fi */
RunningThread = this;	/* restore */

#ifdef SIM_PTHREAD_DEBUG
puts("SIM_PTHREAD_DEBUG: Joining done, locking the mutex...");
#endif /* SIM_PTHREAD_DEBUG */
if ((Error = pthread_mutex_lock(&TheMutex)))	/* obtain the mutex again */
   { fatalError("Cannot lock the mutex", Error);
    } /* fi */

} /* end of _specc::thread_base::ThreadJoin */


void _specc::thread_base::ThreadAbort(	/* abort this thread */
	void)
{
PosixError	Error;
void		*ExitStatus;

#ifdef SIM_PTHREAD_DEBUG
puts("SIM_PTHREAD_DEBUG: Start thread abortion...");
#endif /* SIM_PTHREAD_DEBUG */

assert(RunningThread != this);	/* must not commit suicide! */

if ((Error = pthread_cancel(ThreadHandle)))
   { fatalError("Cannot cancel thread", Error);
    } /* fi */
if ((Error = pthread_mutex_unlock(&TheMutex)))
   { fatalError("Cannot unlock the mutex for thread to die", Error);
    } /* fi */
if ((Error = pthread_join(ThreadHandle, &ExitStatus)))
   { fatalError("Cannot join with cancelled thread", Error);
    } /* fi */
if ((Error = pthread_mutex_lock(&TheMutex)))	/* obtain the mutex again */
   { fatalError("Cannot get the mutex back from dead thread", Error);
    } /* fi */

#ifdef SIM_PTHREAD_DEBUG
puts("SIM_PTHREAD_DEBUG: Abortion successful.");
#endif /* SIM_PTHREAD_DEBUG */

} /* end of _specc::thread_base::ThreadAbort */


void _specc::thread_base::ThreadRun(	/* transfer control to another thread */
	thread_base	*Thread)
{
PosixError	Error;

assert(Thread != NULL);
assert(RunningThread == this);

#ifdef SIM_PTHREAD_DEBUG
printf("SIM_PTHREAD_DEBUG: Transferring control from ID %lu to ID %lu...\n",
			(unsigned long)ThreadHandle,
			(unsigned long)Thread->ThreadHandle);
#endif /* SIM_PTHREAD_DEBUG */

if (Thread == this)	/* continue to run myself? */
   { return;		/* nothing to change */
    } /* fi */

if ((Error = pthread_cond_signal(&Thread->CondVar)))
   { fatalError("Cannot signal next thread", Error);
    } /* fi */
RunningThread = Thread;
do { if ((Error = pthread_cond_wait(&CondVar, &TheMutex)))
	{ fatalError("Cannot block myself", Error);
	 } /* fi */
#ifdef SIM_PTHREAD_DEBUG
     if (RunningThread != this)
	{ SpuriousWakeups++;
	 } /* fi */
#endif /* SIM_PTHREAD_DEBUG */
    } while(RunningThread != this);

#ifdef SIM_PTHREAD_DEBUG
printf("SIM_PTHREAD_DEBUG: Thread ID %lu continuing...\n",
			(unsigned long)ThreadHandle);
#endif /* SIM_PTHREAD_DEBUG */

} /* end of _specc::thread_base::ThreadRun */


/************************************************************************/
/*** implementation of exported functions			      ***/
/************************************************************************/


	/* none */


/* EOF PosixThread.cc */
