// ============================================================================
// longlong.cc: "long long" data type, signed & unsigned
// ============================================================================
// 
//
// 11/21/01 <gerstl>	Cleanup of test case
// 05/24/01 <gerstl>	Initial version
//
// ----------------------------------------------------------------------------


#ifdef DEBUG
#undef HAVE_LLONG            // Test library in any case!
#include "longlong.in.h"
#else
#include "longlong.h"
#endif


#include <ctype.h>


#ifdef HAVE_LLONG

/* ------------------------------------------------------------------------- */
/* Native, built-in long long type */


/* --- conversion to string --- */

char *ull2str(unsigned int base, char *endptr, unsigned long long ull) {
  unsigned int digit;
  *endptr = '\0';
  do {
    switch( base ) {
      case  2: digit = ull & 0x1;
               ull >>= 1;
               break;
      case  8: digit = ull & 0x111;
               ull >>= 3;
               break;
      case 16: digit = ull & 0x11111;
               ull >>= 4;
               break;
      default: digit = (ull % base);
               ull /= base;
    }
    *(--endptr) = (digit <= 9)? (digit + '0') : ((digit-10) + 'A');
  } while( ull );
  return endptr;
}

char *ll2str(unsigned int base, char *endptr, long long ll) {
  bool neg = (ll < 0);
  endptr = ull2str(base, endptr, neg? -ll : ll);
  if( neg ) *(--endptr) = '-';
  return endptr;
}


/* --- conversion from string --- */

unsigned long long str2ull(unsigned int base, const char *str) 
{
  bool neg = false;
  unsigned long long res;
  unsigned int digit;

  // Eat leading whitespace
  while( isspace(*str) ) str++;

  // Eat leading sign
  switch( *str ) {
    case '-': neg = true;
    case '+': str++;
  }
  
  // Base given?
  if( base < 2 )
  {
    // Nope, default is 10
    base = 10;
    
    // Octal or hexadecimal constant?
    if( *str == '0' ) {
      base = 8;
      str++;
      if( *str == 'x' || *str == 'X' ) {
	base = 16;
	str++;
      }
    }
  }
  
  // Loop over digits
  res = 0;
  for( ; *str >= '0'; str++)
  {
    if( *str > '9' ) {
      if( *str < 'A' ) break;
      digit = toupper(*str) - 'A' + 10;
    } else {
      digit = *str - '0';
    }
    
    if( digit >= base ) break;
        
    switch(base) {
      case 2:  res <<= 1; break;
      case 8:  res <<= 3; break;
      case 16: res <<= 4; break;
      default: res *= base;
    }
    
    res += digit;
  };
  
  // Possibly negate...
  if( neg ) return -res; else return res;
}

long long str2ll(unsigned int base, const char *str) 
{
  return str2ull(base, str);
}
 
#else // #ifdef HAVE_LLONG


/* ------------------------------------------------------------------------- */
/* Non-native _longlong support */


// Number of bits in a "long"
#ifndef LONG_BIT
#define LONG_BIT  	(sizeof(long) * CHAR_BIT)
#endif

// Number of bits in one half of a "long"
#define HALFLONG_BIT	(LONG_BIT >> 1)



/* --- External, global functions --- */


// conversion to string

char *ull2str(unsigned int base, char *endptr, _longlong<true> ull) {
  unsigned int digit;
  *endptr = '\0';
  do {
    switch( base ) {
      case  2: digit = ull.toUInt() & 0x1;
               ull >>= 1;
               break;
      case  8: digit = ull.toUInt() & 0x111;
               ull >>= 3;
               break;
      case 16: digit = ull.toUInt() & 0x11111;
               ull >>= 4;
               break;
      default: digit = (ull % base).toUInt();
               ull /= base;
    }
    *(--endptr) = (digit <= 9)? (digit + '0') : ((digit-10) + 'A');
  } while( ull.test() );
  return endptr;
}

char *ll2str(unsigned int base, char *endptr, _longlong<false> ll) {
  bool neg = (ll < 0);
  endptr = ull2str(base, endptr, neg? -ll : ll);
  if( neg ) *(--endptr) = '-';
  return endptr;
}


// conversion from string

_longlong<true> str2ull(unsigned int base, const char *str) {
  return _longlong<true>(str, base);
}

_longlong<false> str2ll(unsigned int base, const char *str) {
  return _longlong<false>(str, base);
}



/* ------------------------------------------------------------------------- */
/* Class __longlong */


/* --- Assignement --- */

// Double, Long Double
__longlong& __longlong::operator= (double d)
{
   s1 = (long)(d / ((double)ULONG_MAX + 1));
   d -= (double)s1 * ((double)ULONG_MAX + 1);
   u0 = (unsigned long)d;
   
   return *this;
}

__longlong& __longlong::operator= (long double d)
{
   s1 = (long)(d / ((long double)ULONG_MAX + 1));
   d -= (long double)s1 * ((long double)ULONG_MAX + 1);
   u0 = (unsigned long)d;
   
   return *this;
}


// String assignement
void __longlong::set(const char* str, unsigned int base)
{
  bool neg = false;
  unsigned int digit;

  // Eat leading whitespace
  while( isspace(*str) ) str++;
  
  // Eat leading sign
  switch( *str ) {
    case '-': neg = true;
    case '+': str++;
  }
  
  // Base given?
  if( base < 2 ) 
  {
    // Nope, default is 10
    base = 10;
    
    // Octal or hexadecimal constant?
    if( *str == '0' ) {
      base = 8;
      str++;
      if( *str == 'x' || *str == 'X' ) {
	base = 16;
	str++;
      }
    }
  }

  // Loop over digits
  u1 = u0 = 0;
  for( ; *str >= '0'; str++)
  {
    if( *str > '9' ) {
      if( *str < 'A' ) break;
      digit = toupper(*str) - 'A' + 10;
    } else {
      digit = *str - '0';
    }
    
    if( digit >= base ) break;
        
    switch(base) {
      case 2:  operator<<=(1); break;
      case 8:  operator<<=(3); break;
      case 16: operator<<=(4); break;
      default: operator*=( _longlong<true>(base) );
    }
    
    u0 += digit;
    if( u0 < digit ) u1++;
  };

  // Possibly negate
  if( neg ) this->neg();
}


/* --- Multiplication --- */

void __longlong::mul(const __longlong& ll) {
  unsigned long tmp;
  int carry = 0;
  
  // Split u0 and ll.u0 into halves
  unsigned long a0 = u0    & (ULONG_MAX >> HALFLONG_BIT);
  unsigned long b0 = ll.u0 & (ULONG_MAX >> HALFLONG_BIT);
  unsigned long a1 = u0    >> HALFLONG_BIT;
  unsigned long b1 = ll.u0 >> HALFLONG_BIT;
    
  // Calculate overflow of lower chunk into upper chunk
  tmp = a0 * b0;
  tmp >>= HALFLONG_BIT;
  tmp += a0 * b1;  // carry += (tmp < (a0*b1));  
  tmp += a1 * b0;  carry += (tmp < (a1*b0));
  tmp >>= HALFLONG_BIT;
  tmp += ((unsigned long)carry << HALFLONG_BIT);
  tmp += a1 * b1;
  
  // Calculate upper chunk
  u1 = tmp + u0 * ll.u1 + u1 * ll.u0;
  
  // Calculate lower chunk result
  u0 *= ll.u0;
}


/* --- Division --- */

// Reference: Knuth's "The Art of Computer Programming", 
//            Volume 2, Seminumerical Algorithms


#define NORM(hi,lo,sh)		(((hi)<<(sh)) | ((lo)>>(HALFLONG_BIT-(sh))))
#define MNORM(hi,lo,sh)		((NORM(hi,lo,sh)) & HALFLONG_BIT)


// Calculate quotient (q is estimated from n/d)
unsigned long __longlong::calc_q( unsigned long n1, unsigned long n2,
				  unsigned long d1, unsigned long d2 )
{
  unsigned long q, r;
  
  q = n1 / d1;
  r = n1 % d1;
  
  // Adjust
  while( q * d2 > (r << HALFLONG_BIT) + n2) {  
    q--;
    r += d1;
    if( r < d1 ) break;  // overflow
  }
  return q;
}

  
// Multiplty & subtract (*this -= q * d)
void __longlong::mul_sub( unsigned long& q,
			  unsigned long d1, unsigned long d2, 
			  unsigned long d3, unsigned long d4 )
{
  unsigned long sav0 = u0;
  unsigned long sav1 = u1;
  unsigned long mul;
  
  // Multiply & subtract
  mul   = q * d4;
  mul >>= HALFLONG_BIT;
  mul  += q * d3;
  mul >>= HALFLONG_BIT;
  mul  += q * d2;  
  
  u0   -= q * d4 + ((q * d3) << HALFLONG_BIT);
  u1   -= mul + ((q * d1) << HALFLONG_BIT) + (u0 > sav0);
      
  // Test remainder
  if( (u1 > sav1) || ((q * d1) > (ULONG_MAX>>HALFLONG_BIT)) ) {
    // Adjust, add back
    q--;
    sav0 = u0;
    u0 += d4 + (d3 << HALFLONG_BIT);
    u1 += d2 + (d1 << HALFLONG_BIT) + (u0 < sav0);
  }
}


// Divide by ll, all values must be positive, unsigned.
// Remainder is returned in the lower chunks of the long long,
// quotient in the return value and the upper chunk.
// How much of in-place data is remainder and how much is quotient 
// depends on the size of the divisor (remainder <= divisor).
unsigned long __longlong::div( const __longlong& ll )
{
  int norm = 0;
  
  // Split the denominator into chunks
  unsigned long d1 = ll.u1 >> HALFLONG_BIT;
  unsigned long d2 = ll.u1 & (ULONG_MAX >> HALFLONG_BIT);
  unsigned long d3 = ll.u0 >> HALFLONG_BIT;
  unsigned long d4 = ll.u0 & (ULONG_MAX >> HALFLONG_BIT);
  
  // Split the nominator into chunks
  unsigned long n1 = u1    >> HALFLONG_BIT;
  unsigned long n2 = u1    & (ULONG_MAX >> HALFLONG_BIT);
  unsigned long n3 = u0    >> HALFLONG_BIT;
  
  unsigned long dhi, dlo;
  unsigned long q, q2, q3;
  
  // Check size of divisor
  if( d1 )
  {
    // Determine normalization
    while( !( (ULONG_MAX << (HALFLONG_BIT-norm-1)) & d1 ) ) norm++;
    
    // Normalize divisor
    dhi = NORM(d1,d2,norm);
    dlo = MNORM(d2,d3,norm);
    
    q = calc_q( NORM(n1,n2,norm), MNORM(n2,n3,norm), dhi, dlo );
    mul_sub( q, d1, d2, d3, d4 );
    return q;
  }
  else if( d2 )
  {
    // Determine normalization
    while( !( (ULONG_MAX << (HALFLONG_BIT-norm-1)) & d2 ) ) norm++;
    
    // Normalize divisor
    dhi = NORM(d2,d3,norm);
    dlo = MNORM(d3,d4,norm);
    
    q = calc_q( NORM(n1,n2,norm), MNORM(n2,n3,norm), dhi, dlo );
    mul_sub( q, d2, d3, d4, 0 );
    
    n1 = u1;
    n2 = u0 >> HALFLONG_BIT;
    n3 = u0 & (ULONG_MAX >> HALFLONG_BIT);
    q2 = calc_q( NORM(n1,n2,norm), MNORM(n2,n3,norm), dhi, dlo );
    mul_sub(q2, 0, d2, d3, d4);
    
    return (q << HALFLONG_BIT) + q2;
  }
  else if( d3 )
  {
    // Determine normalization
    while( !( (ULONG_MAX << (HALFLONG_BIT-norm-1)) & d3 ) ) norm++;
    
    // Normalize divisor
    dhi = NORM(d3,d4,norm);
    dlo = MNORM(d4, 0,norm);
    
    q = calc_q( NORM(n1,n2,norm), MNORM(n2,n3,norm), dhi, dlo );
    mul_sub( q, d3, d4, 0, 0 );
    
    n1 = u1;
    n2 = u0 >> HALFLONG_BIT;
    n3 = u0 & (ULONG_MAX >> HALFLONG_BIT);
    q2 = calc_q( NORM(n1,n2,norm), MNORM(n2,n3,norm), dhi, dlo );
    mul_sub(q2, 0, d3, d4, 0 );
        
    n1 = (u1 << HALFLONG_BIT) + (u0 >> HALFLONG_BIT);
    n2 = u0 & (ULONG_MAX >> HALFLONG_BIT);
    q3 = calc_q( NORM(n1,n2,norm), MNORM(n2, 0,norm), dhi, dlo );
    mul_sub(q3, 0, 0, d3, d4 );
    
    u1 = q;
    return (q2 << HALFLONG_BIT) + q3;
  }
  
  // Single-chunk divisor
  q = n1 / d4;
  n1 = ((n1 % d4) << HALFLONG_BIT) + n2;
  q2 = n1 / d4;
  n1 = ((n1 % d4) << HALFLONG_BIT) + (u0 >> HALFLONG_BIT);
  q3 = n1 / d4;
  n1 = ((n1 % d4) << HALFLONG_BIT) + (u0 & (ULONG_MAX >> HALFLONG_BIT));

  u0 = n1 % d4;
  
  u1 = (q << HALFLONG_BIT) + q2;
  return (q3 << HALFLONG_BIT) + (n1 / d4);
}

#endif  // #ifdef HAVE_LLONG


/* ------------------------------------------------------------------------- */
/* Main */


// Only when debugging...
#ifdef DEBUG

#include <stdio.h>
#include <stdlib.h>


#if defined(LLONG_MAX) || defined(LONG_LONG_MAX) || defined(QUAD_MAX)
 
/* --- Error messages --- */

void operror(const char* o, 
	     unsigned long long a, unsigned long long b,
	     unsigned long long c, const _longlong<true>& ll)
{
  printf("Error in operation: %lluULL %s %lluULL = %lluULL != %lluULL\n",
	  a, o, b, ll.toULLong(), c);
  exit(1);
}

void operror(const char* o, 
	     long long a, long long b,
	     long long c, const _longlong<>& ll)
{
  printf("Error in operation: %lldLL %s %lldLL = %lldLL != %lldLL\n",
	  a, o, b, ll.toLLong(), c);
  exit(1);
}


/* --- Test operations --- */

void testOp(unsigned long long au, unsigned long long bu,
	    _longlong<true> xu,    _longlong<true> yu)
{ 
  long long a = (long long)au;
  long long b = (long long)bu;
  long long c;
  unsigned long long cu;

  _longlong<> x = (_longlong<>)xu;
  _longlong<> y = (_longlong<>)yu;
  _longlong<true>    zu;  
  _longlong<> z;
    
  // signed
  c = a + b;  z = x + y;
  if( c != z.toLLong() ) operror("+", a, b, c, z);
  c = a - b;  z = x - y;
  if( c != z.toLLong() ) operror("-", a, b, c, z);
  c = a * b;  z = x * y;
  if( c != z.toLLong() ) operror("*", a, b, c, z);
  if( b ) {
    c = a / b;  z = x / y;
    if( c != z.toLLong() ) operror("/", a, b, c, z);
    c = a % b;  z = x % y;
    if( c != z.toLLong() ) operror("%", a, b, c, z);
  }
  c = a & b;  z = x & y;
  if( c != z.toLLong() ) operror("&", a, b, c, z);
  c = a | b;  z = x | y;
  if( c != z.toLLong() ) operror("|", a, b, c, z);
  c = a ^ b;  z = x ^ y;
  if( c != z.toLLong() ) operror("^", a, b, c, z);
  c = (a == b);  z = (x == y);
  if( c != z.toLLong() ) operror("==", a, b, c, z);
  c = (a != b);  z = (x != y);
  if( c != z.toLLong() ) operror("!=", a, b, c, z);
  c = (a > b);  z = (x > y);
  if( c != z.toLLong() ) operror(">", a, b, c, z);
  c = (a >= b);  z = (x >= y);
  if( c != z.toLLong() ) operror(">=", a, b, c, z);
  c = (a < b);  z = (x < y);
  if( c != z.toLLong() ) operror("<", a, b, c, z);
  c = (a <= b);  z = (x <= y);
  if( c != z.toLLong() ) operror("<=", a, b, c, z);
  c = ~a; z = ~x;
  if( c != z.toLLong() ) operror("~", 0, a, c, z);
  c = -a; z = -x;
  if( c != z.toLLong() ) operror("-", 0, a, c, z);
  
  // unsigned 
  cu = au + bu;  zu = xu + yu;
  if( cu != zu.toULLong() ) operror("+", au, bu, cu, zu);
  cu = au - bu;  zu = xu - yu;
  if( cu != zu.toULLong() ) operror("-", au, bu, cu, zu);
  cu = au * bu;  zu = xu * yu;
  if( cu != zu.toULLong() ) operror("*", au, bu, cu, zu);
  if( bu ) {
    cu = au / bu;  zu = xu / yu;
    if( cu != zu.toULLong() ) operror("/", au, bu, cu, zu);
    cu = au % bu;  zu = xu % yu;
    if( cu != zu.toULLong() ) operror("%", au, bu, cu, zu);
  }
  cu = au & bu;  zu = xu & yu;
  if( cu != zu.toULLong() ) operror("&", au, bu, cu, zu);
  cu = au | bu;  zu = xu | yu;
  if( cu != zu.toULLong() ) operror("|", au, bu, cu, zu);
  cu = au ^ bu;  zu = xu ^ yu;
  if( cu != zu.toULLong() ) operror("^", au, bu, cu, zu);
  cu = (au == bu);  zu = (xu == yu);
  if( cu != zu.toULLong() ) operror("==", au, bu, cu, zu);
  cu = (au != bu);  zu = (xu != yu);
  if( cu != zu.toULLong() ) operror("!=", au, bu, cu, zu);
  cu = (au > bu);  zu = (xu > yu);
  if( cu != zu.toULLong() ) operror(">", au, bu, cu, zu);
  cu = (au >= bu);  zu = (xu >= yu);
  if( cu != zu.toULLong() ) operror(">=", au, bu, cu, zu);
  cu = (au < bu);  zu = (xu < yu);
  if( cu != zu.toULLong() ) operror("<", au, bu, cu, zu);
  cu = (au <= bu);  zu = (xu <= yu);
  if( cu != zu.toULLong() ) operror("<=", au, bu, cu, zu);
  cu = ~au; zu = ~xu;
  if( cu != zu.toULLong() ) operror("~", 0, au, cu, zu);
  cu = -au; zu = -xu;
  if( cu != zu.toULLong() ) operror("-", 0, au, cu, zu);
}


// Test operations with shadow class
void testOpShadow(unsigned long long au, unsigned long long bu,
	          _longlong<true> xsu,  _longlong<true> ysu)
{ 
  long long a = (long long)au;
  
  union {
    unsigned long long cu;
    long long          b;
  };
  b = bu;

  long long c;

  // Make sure, our special shadow _longlong_ can be used in a union!
  union {
    _longlong_<>        x;
    _longlong_<true>    xu;
  };
  xu = xsu;
  union {
    _longlong_<true>    zu;
    _longlong_<>        y;
  };
  y = ysu;
  union {
    _longlong_<>        z;
    _longlong_<true>    yu;
  };
  yu = ysu;
      
  // signed
  c = a + b;  z = x + y;
  if( c != z.toLLong() ) operror("+", a, b, c, z);
  c = a - b;  z = x - y;
  if( c != z.toLLong() ) operror("-", a, b, c, z);
  c = a * b;  z = x * y;
  if( c != z.toLLong() ) operror("*", a, b, c, z);
  if( b ) {
    c = a / b;  z = x / y;
    if( c != z.toLLong() ) operror("/", a, b, c, z);
    c = a % b;  z = x % y;
    if( c != z.toLLong() ) operror("%", a, b, c, z);
  }
  c = a & b;  z = x & y;
  if( c != z.toLLong() ) operror("&", a, b, c, z);
  c = a | b;  z = x | y;
  if( c != z.toLLong() ) operror("|", a, b, c, z);
  c = a ^ b;  z = x ^ y;
  if( c != z.toLLong() ) operror("^", a, b, c, z);
  c = (a == b);  z = (x == y);
  if( c != z.toLLong() ) operror("==", a, b, c, z);
  c = (a != b);  z = (x != y);
  if( c != z.toLLong() ) operror("!=", a, b, c, z);
  c = (a > b);  z = (x > y);
  if( c != z.toLLong() ) operror(">", a, b, c, z);
  c = (a >= b);  z = (x >= y);
  if( c != z.toLLong() ) operror(">=", a, b, c, z);
  c = (a < b);  z = (x < y);
  if( c != z.toLLong() ) operror("<", a, b, c, z);
  c = (a <= b);  z = (x <= y);
  if( c != z.toLLong() ) operror("<=", a, b, c, z);
  c = ~a; z = ~x;
  if( c != z.toLLong() ) operror("~", 0, a, c, z);
  c = -a; z = -x;
  if( c != z.toLLong() ) operror("-", 0, a, c, z);
  
  // unsigned
  yu = ysu;
  cu = au + bu;  zu = xu + yu;
  if( cu != zu.toULLong() ) operror("+", au, bu, cu, zu);
  cu = au - bu;  zu = xu - yu;
  if( cu != zu.toULLong() ) operror("-", au, bu, cu, zu);
  cu = au * bu;  zu = xu * yu;
  if( cu != zu.toULLong() ) operror("*", au, bu, cu, zu);
  if( bu ) {
    cu = au / bu;  zu = xu / yu;
    if( cu != zu.toULLong() ) operror("/", au, bu, cu, zu);
    cu = au % bu;  zu = xu % yu;
    if( cu != zu.toULLong() ) operror("%", au, bu, cu, zu);
  }
  cu = au & bu;  zu = xu & yu;
  if( cu != zu.toULLong() ) operror("&", au, bu, cu, zu);
  cu = au | bu;  zu = xu | yu;
  if( cu != zu.toULLong() ) operror("|", au, bu, cu, zu);
  cu = au ^ bu;  zu = xu ^ yu;
  if( cu != zu.toULLong() ) operror("^", au, bu, cu, zu);
  cu = (au == bu);  zu = (xu == yu);
  if( cu != zu.toULLong() ) operror("==", au, bu, cu, zu);
  cu = (au != bu);  zu = (xu != yu);
  if( cu != zu.toULLong() ) operror("!=", au, bu, cu, zu);
  cu = (au > bu);  zu = (xu > yu);
  if( cu != zu.toULLong() ) operror(">", au, bu, cu, zu);
  cu = (au >= bu);  zu = (xu >= yu);
  if( cu != zu.toULLong() ) operror(">=", au, bu, cu, zu);
  cu = (au < bu);  zu = (xu < yu);
  if( cu != zu.toULLong() ) operror("<", au, bu, cu, zu);
  cu = (au <= bu);  zu = (xu <= yu);
  if( cu != zu.toULLong() ) operror("<=", au, bu, cu, zu);
  cu = ~au; zu = ~xu;
  if( cu != zu.toULLong() ) operror("~", 0, au, cu, zu);
  cu = -au; zu = -xu;
  if( cu != zu.toULLong() ) operror("-", 0, au, cu, zu);
}

// Test a mix of shadow and normal classes
void testOpMix(unsigned long long au, unsigned long long bu,
	       _longlong<true> xu,    _longlong<true> ysu)
{ 
  long long a = (long long)au;
  long long b = (long long)bu;
  long long c;
  unsigned long long cu;

  _longlong<> x = xu;
  
  _longlong_<>     y;	y  = ysu;
  _longlong_<true> yu;  yu = ysu;
  
  _longlong<>  z;
  _longlong<true> zu;

  // signed
  c = a + b;  z = x + y;
  if( c != z.toLLong() ) operror("+", a, b, c, z);
  c = a - b;  z = x - y;
  if( c != z.toLLong() ) operror("-", a, b, c, z);
  c = a * b;  z = x * y;
  if( c != z.toLLong() ) operror("*", a, b, c, z);
  if( b ) {
    c = a / b;  z = x / y;
    if( c != z.toLLong() ) operror("/", a, b, c, z);
    c = a % b;  z = x % y;
    if( c != z.toLLong() ) operror("%", a, b, c, z);
  }
  c = a & b;  z = x & y;
  if( c != z.toLLong() ) operror("&", a, b, c, z);
  c = a | b;  z = x | y;
  if( c != z.toLLong() ) operror("|", a, b, c, z);
  c = a ^ b;  z = x ^ y;
  if( c != z.toLLong() ) operror("^", a, b, c, z);
  c = (a == b);  z = (x == y);
  if( c != z.toLLong() ) operror("==", a, b, c, z);
  c = (a != b);  z = (x != y);
  if( c != z.toLLong() ) operror("!=", a, b, c, z);
  c = (a > b);  z = (x > y);
  if( c != z.toLLong() ) operror(">", a, b, c, z);
  c = (a >= b);  z = (x >= y);
  if( c != z.toLLong() ) operror(">=", a, b, c, z);
  c = (a < b);  z = (x < y);
  if( c != z.toLLong() ) operror("<", a, b, c, z);
  c = (a <= b);  z = (x <= y);
  if( c != z.toLLong() ) operror("<=", a, b, c, z);
  c = ~a; z = ~x;
  if( c != z.toLLong() ) operror("~", 0, a, c, z);
  c = -a; z = -x;
  if( c != z.toLLong() ) operror("-", 0, a, c, z);
  
  // unsigned
  cu = au + bu;  zu = xu + yu;
  if( cu != zu.toULLong() ) operror("+", au, bu, cu, zu);
  cu = au - bu;  zu = xu - yu;
  if( cu != zu.toULLong() ) operror("-", au, bu, cu, zu);
  cu = au * bu;  zu = xu * yu;
  if( cu != zu.toULLong() ) operror("*", au, bu, cu, zu);
  if( bu ) {
    cu = au / bu;  zu = xu / yu;
    if( cu != zu.toULLong() ) operror("/", au, bu, cu, zu);
    cu = au % bu;  zu = xu % yu;
    if( cu != zu.toULLong() ) operror("%", au, bu, cu, zu);
  }
  cu = au & bu;  zu = xu & yu;
  if( cu != zu.toULLong() ) operror("&", au, bu, cu, zu);
  cu = au | bu;  zu = xu | yu;
  if( cu != zu.toULLong() ) operror("|", au, bu, cu, zu);
  cu = au ^ bu;  zu = xu ^ yu;
  if( cu != zu.toULLong() ) operror("^", au, bu, cu, zu);
  cu = (au == bu);  zu = (xu == yu);
  if( cu != zu.toULLong() ) operror("==", au, bu, cu, zu);
  cu = (au != bu);  zu = (xu != yu);
  if( cu != zu.toULLong() ) operror("!=", au, bu, cu, zu);
  cu = (au > bu);  zu = (xu > yu);
  if( cu != zu.toULLong() ) operror(">", au, bu, cu, zu);
  cu = (au >= bu);  zu = (xu >= yu);
  if( cu != zu.toULLong() ) operror(">=", au, bu, cu, zu);
  cu = (au < bu);  zu = (xu < yu);
  if( cu != zu.toULLong() ) operror("<", au, bu, cu, zu);
  cu = (au <= bu);  zu = (xu <= yu);
  if( cu != zu.toULLong() ) operror("<=", au, bu, cu, zu);
  cu = ~au; zu = ~xu;
  if( cu != zu.toULLong() ) operror("~", 0, au, cu, zu);
  cu = -au; zu = -xu;
  if( cu != zu.toULLong() ) operror("-", 0, au, cu, zu);
}
    

/* --- Main --- */

int main(int argc, char** argv)
{
  unsigned int i, j;
  long long a, c;
  _longlong<> x, z;
  _longlong_<> xs, zs;
  
  unsigned long long au, bu, cu;
  _longlong<true> xu, yu, zu;
  _longlong_<true> xsu, ysu, zsu;

    
  // Test operations
  for(i = 0, au = 0, xu = 0; i <= LONG_BIT*2; au = (au<<1) + ++i%2, xu = (xu<<1) + i%2)
  {
    for(j = 0, bu = 0, yu = 0; j <= LONG_BIT*2; bu = (bu<<1) + ++j%2, yu = (yu<<1) + j%2) {
      testOp(au, bu, xu, yu);
      testOpShadow(au, bu, xu, yu);
      testOpMix(au, bu, xu, yu);
    }
    
    a = (long long)au;

    x = (_longlong<>)xu;
        
    // shifting
    for(unsigned int k = 0; k < LONG_BIT*2; k++)
    {
      c = a << k;  z = x << k;
      if( c != z.toLLong() ) operror("<<", a, k, c, z);
      c = a >> k;  z = x >> k;
      if( c != z.toLLong() ) operror(">>", a, k, c, z);
      cu = au << k;  zu = xu << k;
      if( cu != zu.toULLong() ) operror("<<", au, k, cu, zu);
      cu = au >> k;  zu = xu >> k;
      if( cu != zu.toULLong() ) operror(">>", au, k, cu, zu);
    }
  }
  for(i = 0, au = 0, xu = 0; i <= LONG_BIT*2; i++, au = (au<<1) + 1, xu = (xu<<1) + 1)
  {
    for(j = 0, bu = 0, yu = 0; j <= LONG_BIT*2; j++, bu = (bu<<1) + 1, yu = (yu<<1) + 1) {
      testOp(au, bu, xu, yu);
      testOpShadow(au, bu, xu, yu);
      testOpMix(au, bu, xu, yu);
    }
  }

  // Now with shadow class
  for(i = 0, au = 0, xsu = 0; i <= LONG_BIT*2; au = (au<<1) + ++i%2, xsu = (xsu<<1) + i%2)
  {
    for(j = 0, bu = 0, ysu = 0; j <= LONG_BIT*2; bu = (bu<<1) + ++j%2, ysu = (ysu<<1) + j%2) {
      testOp(au, bu, xsu, ysu);
      testOpShadow(au, bu, xsu, ysu);
      testOpMix(au, bu, xsu, ysu);
    }
    
    a = (long long)au;

    xs = (_longlong<>)xsu;
        
    // shifting
    for(unsigned int k = 0; k < LONG_BIT*2; k++)
    {
      c = a << k;  zs = xs << k;
      if( c != zs.toLLong() ) operror("<<", a, k, c, zs);
      c = a >> k;  zs = xs >> k;
      if( c != zs.toLLong() ) operror(">>", a, k, c, zs);
      cu = au << k;  zsu = xsu << k;
      if( cu != zsu.toULLong() ) operror("<<", au, k, cu, zsu);
      cu = au >> k;  zsu = xsu >> k;
      if( cu != zsu.toULLong() ) operror(">>", au, k, cu, zsu);
    }
  }
  for(i = 0, au = 0, xsu = 0; i <= LONG_BIT*2; i++, au = (au<<1) + 1, xsu = (xsu<<1) + 1)
  {
    for(j = 0, bu = 0, ysu = 0; j <= LONG_BIT*2; j++, bu = (bu<<1) + 1, ysu = (ysu<<1) + 1) {
      testOp(au, bu, xsu, ysu);
      testOpShadow(au, bu, xsu, ysu);
      testOpMix(au, bu, xsu, ysu);
    }
  }

  return 0;    
}

#else // #if defined(LLONG_MAX) || defined(LONG_LONG_MAX) || defined(QUAD_MAX)
 
int main(int argc, char** argv)
{
  return 0;
}

#endif


#endif



