/************************************************************************/
/* Label.cc: SpecC Internal Representation, Label Class			*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 10/20/97 */
/************************************************************************/

/* last update: 06/15/04 */

/* modifications: (most recent first)
 *
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 04/08/03 RD	include missing <ctype.h>
 * 02/28/03 RD	bug fix: Statement OR FsmdStmnt pointer exists at layer 2
 * 02/17/03 RD	allow renaming to same name
 * 02/13/03 RD	added support for notes at fsmd states
 * 02/04/03 RD	added support for labels at fsmd statements
 * 01/15/02 RD	added support for composite annotations
 * 11/21/01 RD	took out default arguments from function definitions
 * 05/31/01 RD	eliminated level 2 of SIR API
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 04/30/01 RD	replaced use of obsolete form() from "stream.h" with own one
 * 01/30/01 RD	fixed a potential FMR problem in iteratorx
 *		SIR_Labels::DFS_ForAllXxxxx
 */

#include "IntRep/Label.h"
#include "IntRep/Symbol.h"

#include <ctype.h>
#include <assert.h>


/*** constants and macros ***********************************************/


	/* (none) */


/*** internal type declarations *****************************************/


	/* (none) */


/*** class implementations **********************************************/


	/*****************/
	/*** SIR_Label ***/
	/*****************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Label::SIR_Label(			/* constructor #1 */
	const char	*LabelName,
	sir_statement	*Statement /* = NULL */,
	sir_fsmdstmnt	*FsmdStmnt /* = NULL */,
	sir_notes	*Notes /* = NULL */)
{

assert(LabelName != NULL);
assert(*LabelName != 0);

SIR_Label::LabelName	= LabelName;
SIR_Label::Statement	= Statement;
SIR_Label::FsmdStmnt	= FsmdStmnt;
SIR_Label::Notes	= Notes;

} /* end of SIR_Label::SIR_Label #1 */


SIR_Label::SIR_Label(			/* constructor #3 (duplicator) */
	sir_label	*Original)
{

SIR_Label::LabelName	= Original->LabelName;
SIR_Label::Statement	= Original->Statement;
SIR_Label::FsmdStmnt	= Original->FsmdStmnt;
SIR_Label::Notes	= Original->Notes ?
				new SIR_Notes(Original->Notes) :
				NULL;

} /* end of SIR_Label::SIR_Label #3 */


SIR_Label::~SIR_Label(void)		/* destructor */
{

delete Notes;

} /* end of SIR_Label::~SIR_Label */


ERROR SIR_Label::DFS_ForAllNodes(	/* iterator for over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{

/* this is not a node, but there are nodes below */

if (Notes)
   { if ((SIR_Error = Notes->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Label::DFS_ForAllNodes */


ERROR SIR_Label::DFS_ForAllNotes(	/* iterator for over all nodes */
	sir_note_mptr	MemberFct,	/* (depth first) */
	sir_note_marg	MemberFctArg)
{

/* this is not a note, but there are notes below */

if (Notes)
   { if ((SIR_Error = Notes->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Label::DFS_ForAllNotes */



ERROR SIR_Label::Annotate(	/* attach a note to the label (#1) */
	sir_note	*NewNote)	/* (consumes NewNote) */
{

assert(NewNote != NULL);

NewNote->Symbol = NULL;		/* label note */
NewNote->UserType = NULL;
NewNote->Label = this;
NewNote->State = NULL;

if (Notes)
   { if (! Notes->Insert(NewNote))
	{ delete NewNote;
	  return(SIR_Error);
	 } /* fi */
    } /* fi */
else
   { Notes = new SIR_Notes(NewNote);
    } /* esle */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Label::Annotate */

	/******************/
	/*** SIR_Labels ***/
	/******************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Labels::SIR_Labels(void)	/* constructor #1 (empty list) */
{

/* nothing to do */

} /* end of SIR_Labels::SIR_Labels #1 */


SIR_Labels::SIR_Labels(		/* constructor #2 (duplicator) */
	sir_labels	*Original)
{
sir_label	*Curr;

Curr = Original->First();
while(Curr)
   { Append(new SIR_Label(Curr));
     Curr = Curr->Succ();
    } /* elihw */

} /* end of SIR_Labels::SIR_Labels #2 */


SIR_Labels::~SIR_Labels(void)	/* destructor */
{

/* nothing to do */

} /* end of SIR_Labels::~SIR_Labels */


ERROR SIR_Labels::DFS_ForAllNodes(	/* iterator over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{
sir_label	*Label,
		*Succ;

/* this is not a node, but there are nodes below */

Label = First();
while(Label)
   { Succ = Label->Succ();
     if ((SIR_Error = Label->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Label = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Labels::DFS_ForAllNodes */


ERROR SIR_Labels::DFS_ForAllNotes(	/* iterator over all nodes */
	sir_note_mptr	MemberFct,	/* (depth first) */
	sir_note_marg	MemberFctArg)
{
sir_label	*Label,
		*Succ;

/* this is not a note, but there are notes below */

Label = First();
while(Label)
   { Succ = Label->Succ();
     if ((SIR_Error = Label->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Label = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Labels::DFS_ForAllNotes */


sir_label *SIR_Labels::Define(		/* define a label */
	const char	*LabelName,	/* (might return NULL and SIR_Error) */
	sir_statement	*Statement)
{
sir_label	*Label;

assert(LabelName != NULL);

Label = FindOrInsert(LabelName);
assert(Label != NULL);

if (  (Label->Statement)	/* already defined? */
    ||(Label->FsmdStmnt))
   { SIR_ErrMsg.form("Redefinition of label '%s'", LabelName);
     SIR_Error = SIR_ERROR_REDEFINITION_OF_LABEL;
     return(NULL);
    } /* fi */

Label->Statement = Statement;	/* set the statement */

return(Label);

} /* end of SIR_Labels::Define #1 */


sir_label *SIR_Labels::Define(		/* define a label at a fsmd stmnt.(#2)*/
	const char	*LabelName,	/* (might return NULL and SIR_Error) */
	sir_fsmdstmnt	*FsmdStmnt)
{
sir_label	*Label;

assert(LabelName != NULL);

if (  (Label = Find(LabelName))		/* already declared? */
    &&(Label->Statement == NULL)
    &&(Label->FsmdStmnt == NULL))
   { // note: this could only be declared by an outside 'goto'
     //       whose target label wasn't defined at that point
     SIR_ErrMsg.form("Target label '%s' for 'goto' defined inside 'fsmd'",
			LabelName);
     SIR_Error = SIR_ERROR_GOTO_LABEL_IN_FSMD;
     return(NULL);
    } /* fi */

Label = FindOrInsert(LabelName);
assert(Label != NULL);

if (  (Label->Statement)	/* already defined? */
    ||(Label->FsmdStmnt))
   { SIR_ErrMsg.form("Redefinition of label '%s'", LabelName);
     SIR_Error = SIR_ERROR_REDEFINITION_OF_LABEL;
     return(NULL);
    } /* fi */

Label->FsmdStmnt = FsmdStmnt;	/* set the statement */

return(Label);

} /* end of SIR_Labels::Define #2 */


ERROR SIR_Labels::CheckDefStatements(	/* check if all labels are defined */
	void)
{
sir_label	*Label;

Label = First();
while(Label)
   { if (  (! Label->Statement)
	 &&(! Label->FsmdStmnt))
	{ SIR_ErrMsg.form("Undefined label '%s'", Label->LabelName.chars());
	  return(SIR_ERROR_UNDEFINED_LABEL);
	 } /* fi */
     assert(  (Label->Statement && !Label->FsmdStmnt)
	    ||(Label->FsmdStmnt && !Label->Statement));
     Label = Label->Succ();
    } /* elihw */

return(SIR_ERROR_NO_ERROR);	/* everything ok */

} /* end of SIR_Labels::CheckDefStatements */


void SIR_Labels::DeleteMarkedLabels(	/* deletes the marked labels */
	void)
{
sir_label	*Label,
		*Succ;

Label = First();
while(Label)
   { Succ = Label->Succ();
     if (Label->Color == SIR_RED)
	{ Remove(Label);
	  delete Label;
	 } /* fi */
     Label = Succ;
    } /* elihw */

} /* end of SIR_Labels::DeleteMarkedLabels */


sir_label *SIR_Labels::FindOrInsert(	/* find or insert a label */
	const char	*LabelName)
{
int		CmpVal;
sir_label	*NewLabel;

First();
while(Curr())
   { if (0 == (CmpVal = strcmp(LabelName, Curr()->LabelName.chars())))
	{ return(Curr());	/* found */
	 } /* fi */
     if (CmpVal < 0)
	{ break;
	 } /* fi */
     Next();
    } /* elihw */

NewLabel = new SIR_Label(LabelName);	/* not found, insert new one */

if (Curr())
   { return(InsertBefore(NewLabel));
    } /* fi */
else
   { return(Append(NewLabel));
    } /* esle */

} /* end of SIR_Labels::FindOrInsert */


sir_label *SIR_Labels::Insert(		/* insert a new label */
	sir_label	*Label)
{
int		CmpVal;

First();
while(Curr())
   { if (0 == (CmpVal = strcmp(Label->LabelName.chars(),
				Curr()->LabelName.chars())))
	{ assert(FALSE);	/* already exists */
	 } /* fi */
     if (CmpVal < 0)
	{ break;
	 } /* fi */
     Next();
    } /* elihw */

if (Curr())
   { return(InsertBefore(Label));
    } /* fi */
else
   { return(Append(Label));
    } /* esle */

} /* end of SIR_Labels::Insert */


sir_label *SIR_Labels::Find(		/* find a label */
	const char	*LabelName)	/* (returns NULL if not found) */
{
int		CmpVal;

First();
while(Curr())
   { if (0 == (CmpVal = strcmp(LabelName, Curr()->LabelName.chars())))
	{ return(Curr());	/* found */
	 } /* fi */
     if (CmpVal < 0)
	{ break;
	 } /* fi */
     Next();
    } /* elihw */

return(NULL);	/* not found */

} /* end of SIR_Labels::Find */


/*** exported functions *************************************************/


	/* none */


/* EOF Label.cc */
