/************************************************************************/
/* BasicType.cc: SpecC Internal Representation, Basic Type Class	*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 10/07/97 */
/************************************************************************/

/* last update: 11/21/01 */

/* modifications: (most recent first)
 *
 * 11/21/01 RD	took out default arguments from function definitions
 * 11/21/01 RD	started this header (last update was 10/02/98)
 */

/* NOTE: This file contains special support for the SpecC parser. */
/*       It is not used in a standard SpecC data structure.	  */


#include "IntRep/BasicType.h"

#include <assert.h>


/*** constants and macros ***********************************************/


	/* none */


/*** class implementations **********************************************/


	/*********************/
	/*** SIR_BasicType ***/
	/*********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_BasicType::SIR_BasicType(		/* constructor #1 */
	SIR_BASICTYPE	BasicType,
	bool		IsSigned /* = false */,
	bool		IsUnsigned /* = false */,
	bool		IsConst /* = false */,
	bool		IsVolatile /* = false */,
	int		LeftBound /* = 0 */,
	int		RightBound /* = 0 */)
{

assert(!(IsSigned && IsUnsigned));	/* check parameters for plausability */

SIR_BasicType::BasicType	= BasicType;
SIR_BasicType::IsSigned		= IsSigned;
SIR_BasicType::IsUnsigned	= IsUnsigned;
SIR_BasicType::IsConst		= IsConst;
SIR_BasicType::IsVolatile	= IsVolatile;
SIR_BasicType::LeftBound	= LeftBound;
SIR_BasicType::RightBound	= RightBound;
SIR_BasicType::DefinedInt	= (BasicType == SIR_BASICTYPE_INT);
SIR_BasicType::StorageClass	= SIR_STORAGE_NONE;
SIR_BasicType::PipeStages	= 0;

} /* end of SIR_BasicType::SIR_BasicType #1 */


SIR_BasicType::SIR_BasicType(		/* constructor #2 */
	SIR_STORAGE	StorageClass,
	int		PipeStages /* = 0 */,	/* (auto 1) */
	bool		IsConst /* = false */,
	bool		IsVolatile /* = false */)
{

assert(  (StorageClass == SIR_STORAGE_PIPED && PipeStages >= 0)
       ||(StorageClass != SIR_STORAGE_PIPED && PipeStages == 0));

SIR_BasicType::BasicType	= SIR_BASICTYPE_UNKNOWN;
SIR_BasicType::IsSigned		= false;
SIR_BasicType::IsUnsigned	= false;
SIR_BasicType::IsConst		= IsConst;
SIR_BasicType::IsVolatile	= IsVolatile;
SIR_BasicType::LeftBound	= 0;
SIR_BasicType::RightBound	= 0;
SIR_BasicType::DefinedInt	= false;
SIR_BasicType::StorageClass	= StorageClass;
SIR_BasicType::PipeStages	= ((StorageClass == SIR_STORAGE_PIPED) ?
						MAX(1, PipeStages) : 0);

} /* end of SIR_BasicType::SIR_BasicType #2 */


SIR_BasicType::~SIR_BasicType(void)	/* destructor */
{

/* nothing to do */

} /* end of SIR_BasicType::~SIR_BasicType */


sir_basic_type *SIR_BasicType::Merge(	/* merges the argument into itself */
	sir_basic_type	*Other)		/* (returns NULL if SIR_Error) */
{

switch(Other->BasicType)
   { case SIR_BASICTYPE_BOOL:
	{ if (this->BasicType == SIR_BASICTYPE_UNKNOWN)
	     { this->BasicType = Other->BasicType;
	       if (this->IsSigned || this->IsUnsigned)
		  { SIR_Error = SIR_ERROR_TYPE_SIGNED_BOOL;
		    return(NULL);
		   } /* fi */
	      } /* fi */
	  else
	     { SIR_Error = SIR_ERROR_TYPE_CONFLICT;
	       return(NULL);
	      } /* esle */
	  break;
	 }
     case SIR_BASICTYPE_CHAR:
	{ if (this->BasicType == SIR_BASICTYPE_UNKNOWN)
	     { this->BasicType = Other->BasicType;
	      } /* fi */
	  else
	     { SIR_Error = SIR_ERROR_TYPE_CONFLICT;
	       return(NULL);
	      } /* esle */
	  break;
	 }
     case SIR_BASICTYPE_SHORT:
	{ if (  (this->BasicType == SIR_BASICTYPE_UNKNOWN)
	      ||(this->BasicType == SIR_BASICTYPE_INT))
	     { this->BasicType = Other->BasicType;
	      } /* fi */
	  else
	     { SIR_Error = SIR_ERROR_TYPE_CONFLICT;
	       return(NULL);
	      } /* esle */
	  break;
	 }
     case SIR_BASICTYPE_INT:
	{ if (this->BasicType == SIR_BASICTYPE_UNKNOWN)
	     { this->BasicType = Other->BasicType;
	      } /* fi */
	  else
	     { if (  (this->BasicType != SIR_BASICTYPE_SHORT)
		   &&(this->BasicType != SIR_BASICTYPE_LONG)
		   &&(this->BasicType != SIR_BASICTYPE_LONGLONG))
		  { SIR_Error = SIR_ERROR_TYPE_CONFLICT;
		    return(NULL);
		   } /* fi */
	      } /* esle */
	  break;
	 }
     case SIR_BASICTYPE_LONG:
	{ if (  (this->BasicType == SIR_BASICTYPE_UNKNOWN)
	      ||(this->BasicType == SIR_BASICTYPE_INT))
	     { this->BasicType = Other->BasicType;
	      } /* fi */
	  else
	     { if (this->BasicType == SIR_BASICTYPE_LONG)
		  { this->BasicType = SIR_BASICTYPE_LONGLONG;
		   } /* fi */
	       else
		  { if (this->BasicType == SIR_BASICTYPE_DOUBLE)
		       { this->BasicType = SIR_BASICTYPE_LONGDOUBLE;
			 if (Other->IsSigned || Other->IsUnsigned)
			    { SIR_Error = SIR_ERROR_TYPE_SIGNED_LONGDOUBLE;
			      return(NULL);
			     } /* fi */
			 if (Other->DefinedInt)
			    { SIR_Error = SIR_ERROR_TYPE_INTLONGDOUBLE;
			      return(NULL);
			     } /* fi */
			} /* fi */
		    else
		       { SIR_Error = SIR_ERROR_TYPE_CONFLICT;
			 return(NULL);
			} /* esle */
		   } /* esle */
	      } /* esle */
	  break;
	 }
     case SIR_BASICTYPE_LONGLONG:
	{ if (  (this->BasicType == SIR_BASICTYPE_UNKNOWN)
	      ||(this->BasicType == SIR_BASICTYPE_INT))
	     { this->BasicType = Other->BasicType;
	      } /* fi */
	  else
	     { SIR_Error = SIR_ERROR_TYPE_CONFLICT;
	       return(NULL);
	      } /* esle */
	  break;
	 }
     case SIR_BASICTYPE_FLOAT:
	{ if (this->BasicType == SIR_BASICTYPE_UNKNOWN)
	     { this->BasicType = Other->BasicType;
	       if (this->IsSigned || this->IsUnsigned)
		  { SIR_Error = SIR_ERROR_TYPE_SIGNED_FLOAT;
		    return(NULL);
		   } /* fi */
	      } /* fi */
	  else
	     { SIR_Error = SIR_ERROR_TYPE_CONFLICT;
	       return(NULL);
	      } /* esle */
	  break;
	 }
     case SIR_BASICTYPE_DOUBLE:
	{ if (this->BasicType == SIR_BASICTYPE_UNKNOWN)
	     { this->BasicType = Other->BasicType;
	       if (this->IsSigned || this->IsUnsigned)
		  { SIR_Error = SIR_ERROR_TYPE_SIGNED_DOUBLE;
		    return(NULL);
		   } /* fi */
	      } /* fi */
	  else
	     { if (this->BasicType == SIR_BASICTYPE_LONG)
		  { this->BasicType = SIR_BASICTYPE_LONGDOUBLE;
		    if (this->IsSigned || this->IsUnsigned)
		       { SIR_Error = SIR_ERROR_TYPE_SIGNED_LONGDOUBLE;
			 return(NULL);
			} /* fi */
		    if (this->DefinedInt)
		       { SIR_Error = SIR_ERROR_TYPE_INTLONGDOUBLE;
			 return(NULL);
			} /* fi */
		   } /* fi */
	       else
		  { SIR_Error = SIR_ERROR_TYPE_CONFLICT;
		    return(NULL);
		   } /* esle */
	      } /* esle */
	  break;
	 }
     case SIR_BASICTYPE_LONGDOUBLE:
	{ if (this->BasicType == SIR_BASICTYPE_UNKNOWN)
	     { this->BasicType = Other->BasicType;
	       if (this->IsSigned || this->IsUnsigned)
		  { SIR_Error = SIR_ERROR_TYPE_SIGNED_LONGDOUBLE;
		    return(NULL);
		   } /* fi */
	      } /* fi */
	  else
	     { SIR_Error = SIR_ERROR_TYPE_CONFLICT;
	       return(NULL);
	      } /* esle */
	  break;
	 }
     case SIR_BASICTYPE_BIT:
	{ if (this->BasicType == SIR_BASICTYPE_UNKNOWN)
	     { this->BasicType = Other->BasicType;
	       this->LeftBound = Other->LeftBound;
	       this->RightBound = Other->RightBound;
	      } /* fi */
	  else
	     { SIR_Error = SIR_ERROR_TYPE_CONFLICT;
	       return(NULL);
	      } /* esle */
	  break;
	 }
     case SIR_BASICTYPE_VOID:
	{ if (this->BasicType == SIR_BASICTYPE_UNKNOWN)
	     { this->BasicType = Other->BasicType;
	       if (this->IsSigned || this->IsUnsigned)
		  { SIR_Error = SIR_ERROR_TYPE_SIGNED_VOID;
		    return(NULL);
		   } /* fi */
	      } /* fi */
	  else
	     { SIR_Error = SIR_ERROR_TYPE_CONFLICT;
	       return(NULL);
	      } /* esle */
	  break;
	 }
     case SIR_BASICTYPE_EVENT:
	{ if (this->BasicType == SIR_BASICTYPE_UNKNOWN)
	     { this->BasicType = Other->BasicType;
	       if (this->IsSigned || this->IsUnsigned)
		  { SIR_Error = SIR_ERROR_TYPE_SIGNED_EVENT;
		    return(NULL);
		   } /* fi */
	       if (this->IsConst || this->IsVolatile)
		  { SIR_Error = SIR_ERROR_TYPE_CONST_OR_VOLATILE_EVENT;
		    return(NULL);
		   } /* fi */
	       if (this->StorageClass == SIR_STORAGE_PIPED)
		  { SIR_Error = SIR_ERROR_TYPE_PIPED_EVENT;
		    return(NULL);
		   } /* fi */
	      } /* fi */
	  else
	     { SIR_Error = SIR_ERROR_TYPE_CONFLICT;
	       return(NULL);
	      } /* esle */
	  break;
	 }
     case SIR_BASICTYPE_UNKNOWN:
	{ /* nothing to change for this->BasicType */
	  break;
	 }
     default:
	{ assert(false);	/* bad BasicType */
	 }
    } /* hctiws */

if (Other->IsSigned)
   { if (this->IsSigned)
	{ SIR_Error = SIR_ERROR_TYPE_SIGNED_TWICE;
	  return(NULL);
	 } /* fi */
     else
	{ if (this->IsUnsigned)
	     { SIR_Error = SIR_ERROR_TYPE_SIGNED_AND_UNSIGNED;
	       return(NULL);
	      } /* fi */
	  else
	     { this->IsSigned = true;
	      } /* esle */
	 } /* esle */
    } /* fi */

if (Other->IsUnsigned)
   { if (this->IsUnsigned)
	{ SIR_Error = SIR_ERROR_TYPE_UNSIGNED_TWICE;
	  return(NULL);
	 } /* fi */
     else
	{ if (this->IsSigned)
	     { SIR_Error = SIR_ERROR_TYPE_SIGNED_AND_UNSIGNED;
	       return(NULL);
	      } /* fi */
	  else
	     { this->IsUnsigned = true;
	      } /* esle */
	 } /* esle */
    } /* fi */

if (Other->IsConst)
   { if (this->IsConst)
	{ SIR_Error = SIR_ERROR_TYPE_CONST_TWICE;
	  return(NULL);
	 } /* fi */
     else
	{ this->IsConst = true;
	 } /* esle */
    } /* fi */

if (Other->IsVolatile)
   { if (this->IsVolatile)
	{ SIR_Error = SIR_ERROR_TYPE_VOLATILE_TWICE;
	  return(NULL);
	 } /* fi */
     else
	{ this->IsVolatile = true;
	 } /* esle */
    } /* fi */

if (Other->DefinedInt)
   { if (this->DefinedInt)
	{ SIR_Error = SIR_ERROR_TYPE_INT_TWICE;
	  return(NULL);
	 } /* fi */
     else
	{ this->DefinedInt = true;
	 } /* esle */
    } /* fi */

if (Other->StorageClass == SIR_STORAGE_PIPED)
   { if (  (this->StorageClass != SIR_STORAGE_NONE)
	 &&(this->StorageClass != SIR_STORAGE_PIPED))
	{ SIR_Error = SIR_ERROR_TYPE_MULTIPLE_STORAGE_CLASSES;
	  return(NULL);
	 } /* fi */
     this->StorageClass = SIR_STORAGE_PIPED;
     this->PipeStages += Other->PipeStages;
    } /* fi */
else
   { if (Other->StorageClass != SIR_STORAGE_NONE)
	{ if (this->StorageClass != SIR_STORAGE_NONE)
	     { SIR_Error = SIR_ERROR_TYPE_MULTIPLE_STORAGE_CLASSES;
	       return(NULL);
	      } /* fi */
	  this->StorageClass = Other->StorageClass;
	 } /* fi */
    } /* esle */

return(this);	/* return myself */

} /* end of SIR_BasicType::Merge */


/* EOF BasicType.cc */
