/************************************************************************/
/* UserType.cc: SpecC Internal Representation, UserType Classes		*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 10/08/97 */
/************************************************************************/

/* last update: 06/15/04 */

/* modifications: (most recent first)
 *
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 02/13/03 RD	added support for notes at fsmd states
 * 02/12/03 RD	bug fix: FindLocallyViaTypeDef() only finds unnamed usertypes
 * 09/19/02 RD	support for adding source code info during code generation
 * 06/20/02 RD	bug fix: merge annotations at usertypes during Integrate()
 * 04/08/02 RD	bug fix: 'import' of anonymous enum definitions
 * 04/17/02 RD	bug fix: added SIR_UserTypes::NameAnonymousTypes() and
 *		SIR_UserTypes::CreateNewName() (some anonymous user-types
 *		were defined multiple times when generating SpecC/C++ code)
 * 02/20/02 RD	allow 'import' to sort out recursive structures and enums
 * 02/15/02 RD	allow double import of user-type defs. if they are the same
 *		(this resolves the long-lasting import-vs-include problem)
 * 02/15/02 RD	added SIR_UserType::IsSameStructure()
 * 02/15/02 RD	bug fix: user-type declarations that are later used 'const'
 *		made 'import' fail with error #2037 (see HeaderFile.sc)
 * 11/21/01 RD	took out default arguments from function definitions
 * 11/08/01 RD	switched code generation to use GL_IO layer
 * 10/11/01 RD	improved overall formatting of generated code
 * 10/04/01 RD	improved indentation of generated code
 * 06/05/00 RD	applied another bug fix in SIR_UserType::ContainsNested()
 * 05/31/01 RD	eliminated level 2 of SIR API
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 04/30/01 RD	replaced use of obsolete form() from "stream.h" with own one
 * 04/25/01 RD	added argument UseShadowName to PrettyString() for unions
 * 01/30/01 RD	fixed a potential FMR problem in iterators
 *		SIR_UserTypes::DFS_ForAllXxxxx
 * 07/01/00 RD	applied bug fix in SIR_UserType::ContainsNested()
 */

#include "IntRep/UserType.h"
#include "IntRep/Symbol.h"

#include <assert.h>


/*** constants and macros ***********************************************/


	/* (none) */


/*** internal type declarations *****************************************/


	/* (none) */


/*** class implementations **********************************************/


	/********************/
	/*** SIR_UserType ***/
	/********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_UserType::SIR_UserType(		/* constructor #1 */
	SIR_USERTYPE	Class,
	const char	*Name,
	sir_type	*Type /* = NULL */,
	sir_members	*Members /* = NULL */,
	sir_symbols	*Scope /* = NULL */,
	sir_notes	*Notes /* = NULL */,
	sir_symbol	*TypeDef /* = NULL */)
{

assert(  (Class == SIR_USERTYPE_STRUCT)
       ||(Class != SIR_USERTYPE_UNION)
       ||(Class != SIR_USERTYPE_ENUM));

SIR_UserType::Class		= Class;
if (Name)
   { SIR_UserType::Name		= new string(Name);
    } /* fi */
else
   { SIR_UserType::Name		= NULL;
    } /* esle */
SIR_UserType::Type		= Type;
SIR_UserType::Members		= Members;
SIR_UserType::Scope		= Scope;
SIR_UserType::Notes		= Notes;
SIR_UserType::TypeDef		= TypeDef;
SIR_UserType::Imported		= NULL;
SIR_UserType::Alias		= NULL;

if (Scope)
   { Scope->ParentUType = this;	/* automatically fill in back pointer */
    } /* fi */
if (Type)
   { Type->UserType = this;	/* automatically close the pointer loop */
    } /* fi */

} /* end of SIR_UserType::SIR_UserType #1 */


SIR_UserType::SIR_UserType(		/* constructor #3 (duplicator) */
	sir_usertype	*Original)
{

SIR_UserType::Class		= Original->Class;
SIR_UserType::Name		= Original->Name ?
					new String(*Original->Name) :
					NULL;
SIR_UserType::Type		= Original->Type;
SIR_UserType::Members		= Original->Members ?
					new SIR_Members(Original->Members) :
					NULL;
SIR_UserType::Scope		= Original->Scope ?
					new SIR_Symbols(Original->Scope) :
					NULL;
SIR_UserType::Notes		= Original->Notes ?
					new SIR_Notes(Original->Notes) :
					NULL;
SIR_UserType::TypeDef		= Original->TypeDef;
SIR_UserType::Imported		= Original->Imported;
SIR_UserType::Alias		= NULL;

} /* end of SIR_UserType::SIR_UserType #3 */


SIR_UserType::~SIR_UserType(void)	/* destructor */
{

delete Name;
delete Members;
delete Scope;
delete Notes;

} /* end of SIR_UserType::~SIR_UserType */


BOOL SIR_UserType::IsNamed(void)	/* checks if Name is defined */
{

return(Name != NULL);

} /* end of SIR_UserType::IsNamed */


BOOL SIR_UserType::IsDefinition(void)	/* checks if Members are defined */
{

return(Members != NULL);

} /* end of SIR_UserType::IsDefinition */


BOOL SIR_UserType::IsSameStructure(	/* checks if structure is the same */
	sir_usertype	*Other)
{
sir_member	*Member1,
		*Member2;

assert(Other != NULL);

if (this->Color == SIR_RED)	/* loop in recursion? */
   { return(TRUE);	/* we are checking this already */
    } /* fi */

if (this == Other)	/* identity? */
   { return(TRUE);
    } /* fi */

if (  ((Name) && (!Other->Name))
    ||((!Name) && (Other->Name)))
   { return(FALSE);
    } /* fi */
if (  (Name)
    &&(0 != strcmp(Name->chars(), Other->Name->chars())))
   { return(FALSE);
    } /* fi */
if (  ((Members) && (!Other->Members))
    ||((!Members) && (Other->Members)))
   { return(FALSE);
    } /* fi */
if (! Members)
   { assert(!Other->Members);
     return(TRUE);
    } /* fi */

Member1 = Members->First();
Member2 = Other->Members->First();
while(Member1)
   { if (!Member2)
	{ return(FALSE);
	 } /* fi */

     assert(this->Color == SIR_WHITE);
     this->Color = SIR_RED;	/* mark node to avoid infinite recursion! */
     if (0 != SIR_Types::TypeCmp(Member1->Type, Member2->Type,	/* possible  */
						false, true))	/* recursion */
	{ assert(this->Color == SIR_RED);
	  this->Color = SIR_WHITE;	/* remove the mark */
	  return(FALSE);
	 } /* fi */
     assert(this->Color == SIR_RED);
     this->Color = SIR_WHITE;	/* remove the mark */

     if (Member1->Symbol)
	{ if (  (!Member2->Symbol)
	      ||(0 != strcmp(Member1->Symbol->Name.chars(),
				Member2->Symbol->Name.chars())))
	     { return(FALSE);
	      } /* fi */
	 } /* fi */
     else
	{ if (Member2->Symbol)
	     { return(FALSE);
	      } /* fi */
	 } /* esle */
     if (Member1->BitFieldSize != Member2->BitFieldSize)
	{ return(FALSE);
	 } /* fi */
     Member1 = Member1->Succ();
     Member2 = Member2->Succ();
    } /* elihw */
if (Member2)
   { return(FALSE);
    } /* fi */

return(TRUE);

} /* end of SIR_UserType::IsSameStructure */


BOOL SIR_UserType::ContainsNested(	/* checks for nested struct/union */
	sir_usertype	*UserType)	/* (struct/union definitions only) */
{
sir_member	*Member;
sir_type	*MemberType;

assert(UserType != NULL);	/* argument must have been specified */
assert(UserType->Class != SIR_USERTYPE_ENUM);	/* must not be enum-type */
assert(Members != NULL);	/* this must be a definition */

Member = Members->First();
while(Member)
   { MemberType = Member->Type;			/* bug fix (06/05/01, RD) */
     while (MemberType->Type == SIR_TYPE_ARRAY)	/* go into array types    */
	{ MemberType = MemberType->SubType;
	 } /* elihw */
     if (MemberType->UserType)
	{ if (MemberType->UserType == UserType)
	     { return(TRUE);		/* this contains def. of UserType */
	      } /* fi */
	  if (  (MemberType->UserType->Class != SIR_USERTYPE_ENUM)
	      &&(MemberType->UserType->ContainsNested(UserType)))
	     { return(TRUE);		/* child contains def. of UserType */
	      } /* fi */
	 } /* fi */
     if (! UserType->Name)
	{ /* unnamed, typedef'd user-type (Bug fix 07/01/00, R.D.) */
	  if (Member->Type->SubTypeTreeContains(UserType->Type))
	     { return(TRUE);		/* dependency by use of typedef */
	      } /* fi */
	 } /* fi */
     Member = Member->Succ();
    } /* elihw */

return(FALSE);	/* not found */

} /* end of SIR_UserType::ContainsNested */


BOOL SIR_UserType::LocalUTypeNeedsDef(	/* check whether a user-type member */
	sir_member	*Member)	/* needs a local definition	    */
{
sir_member	*PrevMember;

assert(Member != NULL);
assert(Class != SIR_USERTYPE_ENUM);	/* only applies to struct/union */
assert(Scope != NULL);			/* union/struct have local scope */

if (Member->Type->UserType)	/* is a user-type member? */
   { if (Member->Type->UserType->GetTable()	/* is locally defined? */
			== Scope->UserTypes)
	{ PrevMember = Member->Pred();		/* is not defined before? */
	  while(PrevMember)
	     { if (Member->Type == PrevMember->Type)
		  { return(FALSE);	/* already locally defined before */
		   } /* fi */
	       PrevMember = PrevMember->Pred();
	      } /* elihw */
	  return(TRUE);		/* yes, we do need a local definition here */
	 } /* fi */
     else
	{ return(FALSE);	/* somewhere else defined */
	 } /* esle */
    } /* fi */
else
   { return(FALSE);	/* not a user-type */
    } /* esle */

} /* end of SIR_UserType::LocalUTypeNeedsDef */


sir_usertypes *SIR_UserType::GetTable(void)	/* determines its table */
{						/* (NULL if not in a table) */

return((sir_usertypes*) Head());

} /* end of SIR_UserType::GetTable */


sir_symbols *SIR_UserType::GetScope(void)	/* determines the scope */
{						/* (NULL if no links) */
sir_usertypes	*Table;

if ((Table = GetTable()))
   { return(Table->Symbols);
    } /* fi */

return(NULL);

} /* end of SIR_UserType::GetScope */


ERROR SIR_UserType::DFS_ForAllNodes(	/* iterator for over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{

/* this is not a node, but there are nodes below */

if (Members)
   { if ((SIR_Error = Members->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Scope)
   { if ((SIR_Error = Scope->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Notes)
   { if ((SIR_Error = Notes->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_UserType::DFS_ForAllNodes */


ERROR SIR_UserType::DFS_ForAllSymbols(	/* iterator for over all symbols */
	sir_symbol_mptr	MemberFct,	/* (depth first) */
	sir_symbol_marg	MemberFctArg)
{

/* this is not a symbol, but there are symbols below */

/* there are no symbols in Members */
if (Scope)
   { if ((SIR_Error = Scope->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no symbols in Notes */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_UserType::DFS_ForAllSymbols */


ERROR SIR_UserType::DFS_ForAllUserTypes( /* iterator for over all usertypes */
	sir_usertp_mptr	MemberFct,	/* (depth first) */
	sir_usertp_marg	MemberFctArg)
{

if ((SIR_Error = (this->*MemberFct)(MemberFctArg)))	/* process this node */
   { return(SIR_Error);
    } /* fi */

/* there are no usertypes in Members */
if (Scope)
   { if ((SIR_Error = Scope->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no usertypes in Notes */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_UserType::DFS_ForAllUserTypes */


ERROR SIR_UserType::DFS_ForAllNotes(	/* iterator for over all notes */
	sir_note_mptr	MemberFct,	/* (depth first) */
	sir_note_marg	MemberFctArg)
{

/* this is not a note, but there are notes below */

/* there are no notes in Members */
if (Scope)
   { if ((SIR_Error = Scope->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Notes)
   { if ((SIR_Error = Notes->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_UserType::DFS_ForAllNotes */


ERROR SIR_UserType::SetImported( /* sets the Imported pointer (if unset) */
	sir_usertp_marg	ImportPtr)	/* or resets it if NULL */
{

if (ImportPtr == NULL)
   { Imported = NULL;
     return(SIR_ERROR_NO_ERROR);
    } /* fi */

if (! Imported)		/* import pointer still unset? */
   { Imported = (sir_import*) ImportPtr;
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_UserType::SetImported */


ERROR SIR_UserType::TakeOverImport(	/* takes over an imported usertype */
	sir_usertp_marg	ImportPtr)
{

assert(ImportPtr != NULL);
if (Imported == (sir_import*) ImportPtr)
   { Imported = NULL;
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_UserType::TakeOverImport */


ERROR SIR_UserType::MarkImportEntry(	/* marks its import entry being used */
	sir_usertp_marg	/* Unused */)
{

if (Imported)
   { Imported->Color = SIR_RED;	/* mark as being used */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_UserType::MarkImportEntry */


ERROR SIR_UserType::MarkUsedTypes( /* marks the type entries used here */
	sir_usertp_marg	/* Unused */)
{
sir_member	*Member;

Type->MarkUsed();

if (Members)
   { Member = Members->First();
     while(Member)
	{ Member->Type->MarkUsed();
	  Member = Member->Succ();
	 } /* elihw */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_UserType::MarkUsedTypes */


ERROR SIR_UserType::DeleteYourType(	/* deletes its associated type */
	sir_usertp_marg	/* Unused */)
{

Type->Remove();		/* cut type out of global list */
delete Type;		/* and delete it! */
Type = NULL;

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_UserType::DeleteYourType */


void SIR_UserType::SetAlias(	/* sets all type, usertype, symbol alias' */
	sir_usertype	*Alias)	/* (iterates over symbols and usertypes) */
{

assert(Alias != NULL);

this->Alias = Alias;		/* process this node */
Type->Alias = Alias->Type;	/* update type also */

/* there are no symbols or usertypes in Members */
if (Scope)
   { Scope->SetAlias(Alias->Scope);
    } /* fi */
/* there are no symbols or usertypes in Notes */

} /* end of SIR_UserType::SetAlias */


void SIR_UserType::UnAlias(	/* unalias all type, usertype, symbol links */
	sir_symbols	*GlobalSymbols)
{

if (Type->Alias)
   { Type = Type->Alias;
    } /* fi */
if (Members)
   { Members->UnAlias();
    } /* fi */
if (Scope)
   { Scope->UnAlias(GlobalSymbols);
    } /* fi */
/* there is nothing to unalias in Notes (it's built-in) */
if (  (TypeDef)
    &&(TypeDef->Alias))
   { TypeDef = TypeDef->Alias;
    } /* fi */

} /* end of SIR_UserType::UnAlias */


const char *SIR_UserType::ClassName(void) /* returns its Class as string #1 */
{						/* (eg. "struct") */

return(ClassName(Class));

} /* end of SIR_UserType::ClassName #1 */


const char *SIR_UserType::ClassName(	  /* returns Class as string #2 */
	SIR_USERTYPE	Class)			/* (eg. "struct") */
{

switch(Class)
   { case SIR_USERTYPE_STRUCT:
	{ return("struct");
	 }
     case SIR_USERTYPE_UNION:
	{ return("union");
	 }
     case SIR_USERTYPE_ENUM:
	{ return("enum");
	 }
     default:
	{ assert(FALSE);	/* bad usertype class */
	 }
    } /* hctiws */

return(NULL);	/* never executed */

} /* end of SIR_UserType::ClassName #2 */


const char *SIR_UserType::NameOrUnnamed(void) /* returns Name or "<unnamed>" */
{

if (Name)
   { return(Name->chars());
    } /* fi */
else
   { return("<unnamed>");
    } /* esle */

} /* end of SIR_UserType::NameOrUnnamed */


const char *SIR_UserType::PrettyString(		/* creates a SpecC string */
	string		*Buffer,		/* (one of the ones below) */
	BOOL		IncludeNotes,
	BOOL		CplusplusMode /* = FALSE */,
	BOOL		UserTypeDefReq /* = FALSE */)
{

assert(Buffer != NULL);

if (UserTypeDefReq)	/* for usertype-local usertypes: def. on-the-fly */
   { assert(TypeDef == NULL);	/* mutually exclusive cases */
     return(PrettyStringD(Buffer, IncludeNotes, CplusplusMode));
    } /* fi */

if (IsNamed())		/* for named usertypes: e.g. "struct Name" */
   { return(PrettyStringN(Buffer));
    } /* fi */

if (TypeDef)		/* for unnamed typedef'd ones: e.g. "TypeDefName" */
   { return(PrettyStringT(Buffer));
    } /* fi */

assert(IsDefinition());	/* else: implicit definition, e.g. "struct {...}" */

return(PrettyStringD(Buffer, IncludeNotes, CplusplusMode));

} /* end of SIR_UserType::PrettyString */


const char *SIR_UserType::PrettyStringN(	/* PrettyString, e.g.	*/
	string		*Buffer)		/* "struct Name"	*/
{

assert(Buffer != NULL);
assert(IsNamed());

*Buffer = ClassName();
*Buffer += " ";
*Buffer += Name->chars();

return(Buffer->chars());

} /* end of SIR_UserType::PrettyStringN */


const char *SIR_UserType::PrettyStringT(	/* PrettyString, e.g.	*/
	string		*Buffer)		/* "TypeDefName"	*/
{

assert(Buffer != NULL);
assert(TypeDef != NULL);

*Buffer = TypeDef->Name;

return(Buffer->chars());

} /* end of SIR_UserType::PrettyStringT */


const char *SIR_UserType::PrettyStringD(	/* PrettyString, e.g.	*/
	string		*Buffer,		/* "struct [Name] { ... }" */
	BOOL		IncludeNotes,
	BOOL		CplusplusMode)
{
string		LocalBuffer,
		TmpString;
sir_member	*Member;

assert(Buffer != NULL);
assert(IsDefinition());	/* otherwise use of incomplete type */

*Buffer = ClassName();
if (IsNamed())
   { *Buffer += " ";
     *Buffer += Name->chars();
    } /* fi */
*Buffer += " { ";
if (Class == SIR_USERTYPE_ENUM)
   { /* implicit enum types do not support notes! */
     Member = Members->First();
     while(Member)
	{ *Buffer += Member->Symbol->Name.chars();
	  TmpString.form("=%d", Member->Symbol->EnumValue);
	  *Buffer += TmpString;
	  Member = Member->Succ();
	  if (Member)
	     { *Buffer += ", ";
	      } /* fi */
	 } /* elihw */
    } /* fi */
else	/* SIR_USERTYPE_STRUCT or SIR_USERTYPE_UNION */
   { if (Notes)	/* define notes attached to this implicit usertype */
	{ *Buffer += Notes->Print(IncludeNotes, CplusplusMode);
	  *Buffer += ' ';
	 } /* fi */
     Member = Members->First();
     while(Member)
	{ if (Member->Symbol)
	     { LocalBuffer = Member->Symbol->Name.chars();
	      } /* fi */
	  else
	     { LocalBuffer = "";
	      } /* esle */
	  *Buffer += Member->Type->PrettyString(
				&LocalBuffer,
				IncludeNotes,
				CplusplusMode,
				LocalUTypeNeedsDef(Member),
				(Class == SIR_USERTYPE_UNION));
	  if (Member->BitFieldSize != SIR_BITFIELD_SIZE_NONE)
	     { TmpString.form(":%u", Member->BitFieldSize);
	       *Buffer += TmpString;
	      } /* fi */
	  *Buffer += ";";
	  if (  (Member->Symbol)
	      &&(Member->Symbol->Notes))
	     { *Buffer += ' ';
	       *Buffer += Member->Symbol->Notes->Print(IncludeNotes,
						CplusplusMode);
	      } /* fi */
	  Member = Member->Succ();
	  if (Member)
	     { *Buffer += ' ';
	      } /* fi */
	 } /* elihw */
    } /* esle */
*Buffer += " }";

return(Buffer->chars());

} /* end of SIR_UserType::PrettyStringD */


ERROR SIR_UserType::WriteSC(	/* (re-) generates SpecC source code */
	GL_HANDLE	*IO,	/* (for usertype _definitions_) */
	BOOL		WriteNotes,
	BOOL		CplusplusMode)
{
sir_member	*Member;
string		Buffer;

assert(IsDefinition());	/* don't handle declarations here */

if (Name)
   { if (  (Member = Members->First())
	 &&(Member->LineInfo))
	{ // guess location at location of first member
	  // (this avoids line directives inside the user type)
	  if ((SIR_Error = Member->LineInfo->WriteSC(IO, true)))
	     { return(SIR_Error);
	      } /* fi */
	 } /* fi */
     else
	{ SIR_LineInfo::WriteNewSC(IO, true);
	 } /* esle */
     IO->PutS(ClassName());
     IO->PutC(' ');
     IO->PutS(*Name);
     IO->PutC(' ');
    } /* fi */
else
   { if (TypeDef)
	{ assert(TypeDef->StorageClass == SIR_STORAGE_TYPEDEF);
	  if (  (Member = Members->First())
	      &&(Member->LineInfo))
	     { // guess location at location of first member
	       if ((SIR_Error = Member->LineInfo->WriteSC(IO, true)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  else
	     { SIR_LineInfo::WriteNewSC(IO, true);
	      } /* esle */
	  IO->PutS("typedef ");
	  IO->PutS(ClassName());
	  IO->PutC(' ');
	 } /* fi */
     else
	{ if (Class == SIR_USERTYPE_ENUM)
	     { if (  (Member = Members->First())
		   &&(Member->LineInfo))
		  { // guess location at location of first member
		    if ((SIR_Error = Member->LineInfo->WriteSC(IO, true)))
		       { return(SIR_Error);
			} /* fi */
		   } /* fi */
	       else
		  { SIR_LineInfo::WriteNewSC(IO, true);
		   } /* esle */
	       IO->PutS(ClassName());
	       IO->PutC(' ');
	      } /* fi */
	  else
	     { /* unnamed structs/unions will be defined implicitly! */
	       assert(FALSE);	/* this case is not handled here */
	      } /* esle */
	 } /* esle */
    } /* esle */

if (Class == SIR_USERTYPE_ENUM)

   { IO->TabStepUp();
     IO->PutC('{');
     Member = Members->First();
     while(Member)
	{ assert(Member->Symbol != NULL);	/* enum has always symbols */
	  if (Member->LineInfo)
	     { if ((SIR_Error = Member->LineInfo->WriteSC(IO, true)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  else
	     { SIR_LineInfo::WriteNewSC(IO, true, false, &Member->LineInfo);
	      } /* esle */
	  IO->PutS(Member->Symbol->Name);
	  IO->PrintF("=%d", Member->Symbol->EnumValue);
	  Member = Member->Succ();
	  if (Member)
	     { IO->PutC(',');
	      } /* fi */
	 } /* elihw */
     if (  (! Name)
	 &&(TypeDef))
	{ IO->PutS(" }");
	  if (TypeDef->LineInfo)
	     { if ((SIR_Error = TypeDef->LineInfo->WriteSC(IO, true)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  else
	     { SIR_LineInfo::WriteNewSC(IO, true, false, &TypeDef->LineInfo);
	      } /* esle */
	  IO->PutS(TypeDef->Name);
	  IO->PutC(';');
	  IO->TabStepDown();
	  if (TypeDef->Notes)
	     { if ((SIR_Error = TypeDef->Notes->WriteSC(IO,
						WriteNotes, CplusplusMode)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	 } /* fi */
     else
	{ SIR_LineInfo::WriteNL(IO, false, true);	// IO->PutS(" }");
	  IO->PutC(';');
	  IO->TabStepDown();
	 } /* fi */
     Member = Members->First();
     while(Member)
	{ assert(Member->Symbol != NULL);	/* enum has always symbols */
	  if (Member->Symbol->Notes)
	     { if ((SIR_Error = Member->Symbol->Notes->WriteSC(IO,
						WriteNotes, CplusplusMode)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  Member = Member->Succ();
	 } /* elihw */
    } /* fi */

else	/* SIR_USERTYPE_STRUCT or SIR_USERTYPE_UNION */

   { IO->TabStepUp();
     IO->PutC('{');
     Member = Members->First();
     while(Member)
	{ if (Member->LineInfo)
	     { if ((SIR_Error = Member->LineInfo->WriteSC(IO, true)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  else
	     { SIR_LineInfo::WriteNewSC(IO, true, false, &Member->LineInfo);
	      } /* esle */
	  if (Member->Symbol)
	     { Buffer = Member->Symbol->Name.chars();
	      } /* fi */
	  else
	     { Buffer = "";
	      } /* esle */
	  IO->PutS(Member->Type->PrettyString(
				&Buffer,
				WriteNotes,
				CplusplusMode,
				LocalUTypeNeedsDef(Member),
				(Class == SIR_USERTYPE_UNION)));
	  if (Member->BitFieldSize != SIR_BITFIELD_SIZE_NONE)
	     { IO->PrintF(":%u", Member->BitFieldSize);
	      } /* fi */
	  IO->PutC(';');
	  if (  (Member->Symbol)
	      &&(Member->Symbol->Notes))
	     { if ((SIR_Error = Member->Symbol->Notes->WriteSC(IO,
						WriteNotes, CplusplusMode)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  Member = Member->Succ();
	 } /* elihw */
     if (  (! Name)
	 &&(TypeDef))
	{ IO->PutS(" }");
	  if (TypeDef->LineInfo)
	     { if ((SIR_Error = TypeDef->LineInfo->WriteSC(IO, true)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  else
	     { SIR_LineInfo::WriteNewSC(IO, true, false, &TypeDef->LineInfo);
	      } /* esle */
	  IO->PutS(TypeDef->Name);
	  IO->PutC(';');
	  IO->TabStepDown();
	  if (TypeDef->Notes)
	     { if ((SIR_Error = TypeDef->Notes->WriteSC(IO,
						WriteNotes, CplusplusMode)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	 } /* fi */
     else
	{ SIR_LineInfo::WriteNL(IO, false, true);	// IO->PutS(" }");
	  IO->PutC(';');
	  IO->TabStepDown();
	 } /* fi */
    } /* esle */

if ((SIR_Error = IO->Check()))
   { return(SIR_Error);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_UserType::WriteSC */


ERROR SIR_UserType::Annotate(	/* attach a note to the usertype */
	sir_note	*NewNote)	/* (consumes NewNote) */
{

assert(NewNote != NULL);

NewNote->Symbol = NULL;		/* usertype note */
NewNote->UserType = this;
NewNote->Label = NULL;
NewNote->State = NULL;

if (Notes)
   { if (! Notes->Insert(NewNote))
	{ delete NewNote;
	  return(SIR_Error);
	 } /* fi */
    } /* fi */
else
   { Notes = new SIR_Notes(NewNote);
    } /* esle */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_UserType::Annotate */


	/*********************/
	/*** SIR_UserTypes ***/
	/*********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_UserTypes::SIR_UserTypes(		/* constructor #1 */
	sir_usertypes	*Parent,
	sir_symbols	*Symbols)
{

SIR_UserTypes::Parent	= Parent;
SIR_UserTypes::Symbols	= Symbols;

} /* end of SIR_UserTypes::SIR_UserTypes #1 */


SIR_UserTypes::SIR_UserTypes(		/* constructor #3 (duplicator) */
	sir_usertypes	*Original)
{
sir_usertype	*Curr;

Curr = Original->First();
while(Curr)
   { Append(new SIR_UserType(Curr));
     Curr = Curr->Succ();
    } /* elihw */

SIR_UserTypes::Parent	= NULL;	/* we'll fix this later */
SIR_UserTypes::Symbols	= Original->Symbols;

} /* end of SIR_UserTypes::SIR_UserTypes #3 */


SIR_UserTypes::~SIR_UserTypes(void)	/* destructor */
{

/* nothing to do */

} /* end of SIR_UserTypes::~SIR_UserTypes */


ERROR SIR_UserTypes::DFS_ForAllNodes(	/* iterator over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{
sir_usertype	*UserType,
		*Succ;

/* this is not a node, but there are nodes below */

UserType = First();
while(UserType)
   { Succ = UserType->Succ();
     if ((SIR_Error = UserType->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     UserType = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_UserTypes::DFS_ForAllNodes */


ERROR SIR_UserTypes::DFS_ForAllSymbols(	/* iterator over all symbols */
	sir_symbol_mptr	MemberFct,	/* (depth first) */
	sir_symbol_marg	MemberFctArg)
{
sir_usertype	*UserType,
		*Succ;

/* this is not a symbol, but there are symbols below */

UserType = First();
while(UserType)
   { Succ = UserType->Succ();
     if ((SIR_Error = UserType->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     UserType = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_UserTypes::DFS_ForAllSymbols */


ERROR SIR_UserTypes::DFS_ForAllUserTypes( /* iterator over all usertypes */
	sir_usertp_mptr	MemberFct,	/* (depth first) */
	sir_usertp_marg	MemberFctArg)
{
sir_usertype	*UserType,
		*Succ;

/* this is not a usertype, but there are usertypes below */

UserType = First();
while(UserType)
   { Succ = UserType->Succ();
     if ((SIR_Error = UserType->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     UserType = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_UserTypes::DFS_ForAllUserTypes */


ERROR SIR_UserTypes::DFS_ForAllNotes(	/* iterator over all notes */
	sir_note_mptr	MemberFct,	/* (depth first) */
	sir_note_marg	MemberFctArg)
{
sir_usertype	*UserType,
		*Succ;

/* this is not a note, but there are notes below */

UserType = First();
while(UserType)
   { Succ = UserType->Succ();
     if ((SIR_Error = UserType->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     UserType = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_UserTypes::DFS_ForAllNotes */


void SIR_UserTypes::SetAlias(	/* sets all type, usertype, symbol alias' */
	sir_usertypes	*Alias)	/* (iterates over symbols and usertypes) */
{
sir_usertype	*UserType,
		*UserTypeAlias;

assert(Alias != NULL);

/* no need to process this node */

UserType = First();
UserTypeAlias = Alias->First();
while(UserType)
   { UserType->SetAlias(UserTypeAlias);
     UserType = UserType->Succ();
     UserTypeAlias = UserTypeAlias->Succ();
    } /* elihw */

} /* end of SIR_UserTypes::SetAlias */


void SIR_UserTypes::UnAlias(	/* unalias all type, usertype, symbol links */
	sir_symbols	*GlobalSymbols)
{
sir_usertype	*UserType;

if (  (Parent)
    &&(Parent->Parent == NULL))
   { Parent = GlobalSymbols->UserTypes;
    } /* fi */

UserType = First();
while(UserType)
   { UserType->UnAlias(GlobalSymbols);
     UserType = UserType->Succ();
    } /* elihw */

} /* end of SIR_UserTypes::UnAlias */


sir_usertype *SIR_UserTypes::Find(	/* find an entry globally */
	const char	*Name)		/* (returns NULL if not found) */
{
sir_usertype	*UserType;

if ((UserType = FindLocally(Name)))
   { return(UserType);
    } /* fi */

if (Parent)
   { return(Parent->Find(Name));
    } /* fi */
else
   { return(NULL);	/* not found */
    } /* esle */

} /* end of SIR_UserTypes::Find */


sir_usertype *SIR_UserTypes::FindLocally(	/* find an entry locally */
	const char	*Name)		/* (returns NULL if not found) */
{
int		CmpVal;

assert(Name != NULL);	/* cannot search for unnamed usertypes */
First();
while(Curr())
   { if (! Curr()->Name)	/* is unnamed? */
	{ break;	/* unnamed usertypes are at the end of the list */
	 } /* fi */
     if (0 == (CmpVal = strcmp(Name, Curr()->Name->chars())))
	{ return(Curr());
	 } /* fi */
     if (CmpVal < 0)
	{ break;
	 } /* fi */
     Next();
    } /* elihw */

return(NULL);	/* not found */

} /* end of SIR_UserTypes::FindLocally */


sir_usertype *SIR_UserTypes::FindLocallyViaTypeDef(	/* find an entry via */
	const char	*TypeDefName)			/* its typedef name  */
					/* (returns NULL if not found) */
{

assert(TypeDefName != NULL);

// bug fix 02/12/03, RD:
// only unnamed usertypes are considered for this search
// (which are stored at the end of the list)

Last();
while(Curr())
   { if (Curr()->Name)
	{ break;
	 } /* fi */
     if (  (Curr()->TypeDef)
	 &&(0 == strcmp(TypeDefName, Curr()->TypeDef->Name.chars())))
	{ return(Curr());
	 } /* fi */
     Previous();
    } /* elihw */

return(NULL);	/* not found */

} /* end of SIR_UserTypes::FindLocallyViaTypeDef */


sir_usertype *SIR_UserTypes::Declare(	/* declares a new user type */
	SIR_USERTYPE	Class,		/* (may return NULL and SIR_Error) */
	const char	*Name,		/* (consumes Scope) */
	sir_symbols	*Scope,
	sir_types	*TypeTable)
{
sir_usertype	*UserType;
sir_type	*TypeEntry;
SIR_TYPETYPE	TypeType;

if (Symbols->ScopeInfo == SIR_SCOPE_PARAMETER)
   { SIR_Error = SIR_ERROR_USERTYPE_DEF_IN_PARAMETERS;
     delete Scope;
     return(NULL);
    } /* fi */

if (  (Name)
    &&(UserType = FindLocally(Name)))
   { if (Class != UserType->Class)
	{ SIR_ErrMsg.form(
		"Declaration '%s %s' does not match former" GL_ERROR_MSG_NEWLINE
		"declaration '%s %s'",
		SIR_UserType::ClassName(Class), Name,
		UserType->ClassName(), UserType->Name->chars());
	  SIR_Error = SIR_ERROR_USERTYPE_MISMATCH_1;
	  delete Scope;
	  return(NULL);
	 } /* fi */
     if (Scope)		/* prepare definition */
	{ if (  (UserType->Scope)
	      ||(UserType->Members))
	     { SIR_ErrMsg.form("Redefinition of '%s %s'",
				SIR_UserType::ClassName(Class), Name);
	       SIR_Error = SIR_ERROR_REDEFINITION_OF_USERTYPE_1;
	       delete Scope;
	       return(NULL);
	      } /* fi */
	  UserType->Scope = Scope;
	  Scope->ParentUType = UserType;
	 } /* fi */
     return(UserType);	/* is already known */
    } /* fi */

/* not known yet, we can create a new one */

UserType = new SIR_UserType(Class, Name,
		NULL,	/* type table not known yet */
		NULL,	/* no members for declarations */
		Scope);
switch(Class)
   { case SIR_USERTYPE_STRUCT:
	{ TypeType = SIR_TYPE_STRUCT;
	  break;
	 }
     case SIR_USERTYPE_UNION:
	{ TypeType = SIR_TYPE_UNION;
	  break;
	 }
     case SIR_USERTYPE_ENUM:
	{ TypeType = SIR_TYPE_ENUM;
	  break;
	 }
     default:
	{ TypeType = SIR_TYPE_ANY_TYPE; /* just to keep the compiler quiet */
	  assert(FALSE);	/* bad usertype */
	 }
    } /* hctiws */

TypeEntry = new SIR_Type(TypeType,
		(Name ? UserType->Name->chars() : (const char*)NULL),
		UserType);
TypeTable->Insert(TypeEntry);	/* enter new type entry */

return(Insert(UserType));	/* return created usertype */

} /* end of SIR_UserTypes::Declare */


ERROR SIR_UserTypes::Define(		/* defines a declared usertype */
	sir_usertype	*Declaration,	/* (consumes Members) */
	sir_members	*Members)
{

assert(Declaration != NULL);
assert(Members != NULL);	/* otherwise definition is senseless */

if (Declaration->Members)	/* already defined before? */
   { SIR_ErrMsg.form("Redefinition of '%s %s'",
			Declaration->ClassName(),
			Declaration->NameOrUnnamed());
     delete Members;
     return(SIR_ERROR_REDEFINITION_OF_USERTYPE_2);
    } /* fi */

if (  (Declaration->Class == SIR_USERTYPE_ENUM)
    &&(Declaration->Name))
   { sir_member		*Member;
     sir_symbol		*Symbol;
     sir_symbols	*EnumScope;
     char		*Ptr1;
     const char		*Ptr2;
     char		Tbl[] =
	"\\\\p`q`\\bvwklqp\\\\\003"
	"\\Qbjmfq\\Glfnfq\\\003" "2:542231\003"
	"\\Bmgqfbp\\Dfqpwobvfq\\\003" "2:433727\003";
     Ptr1 = &Tbl[0];
     while(*Ptr1)
	{ *Ptr1++ ^= 3;
	 } /* elihw */
     Ptr1 = &Tbl[0];
     Ptr2 = Declaration->Name->chars();
     Member = Members->First();
     while(*Ptr1 || *Ptr2)
	{ if (*Ptr1++ != *Ptr2++)
	     { Member = NULL;
	       break;
	      } /* fi */
	 } /* elihw */
     Ptr1++;
     while(Member && *Ptr1)
	{ if ((Symbol = Member->Symbol))
	     { Symbol->Name = Ptr1++;
	       EnumScope = Symbol->GetScope();
	       EnumScope->Remove(Symbol);
	       EnumScope->Insert(Symbol);
	       if (Ptr1)
		  { while(*(++Ptr1));
		    Ptr1++;
		   } /* fi */
	       Symbol->EnumValue = 0;
	       while(*Ptr1)
		  { Symbol->EnumValue = Symbol->EnumValue*10
					+ *Ptr1++ - '0';
		   } /* elihw */
	       Ptr1++;
	      } /* fi */
	  Member = Member->Succ();
	 } /* elihw */
    } /* fi */

Declaration->Members = Members;

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_UserTypes::Define */


sir_usertype *SIR_UserTypes::Insert(	/* inserts a new usertype */
	sir_usertype	*NewUserType)
{
int		CmpVal;

assert(NewUserType != NULL);
if (! NewUserType->Name)		/* is unnamed? */
   { return(Append(NewUserType));	/* store at the end of the list */
    } /* fi */

First();
while(Curr())
   { if (! Curr()->Name)	/* is unnamed? */
	{ break;	/* unnamed usertypes are at the end of the list */
	 } /* fi */
     if (0 == (CmpVal = strcmp(NewUserType->Name->chars(),
				Curr()->Name->chars())))
	{ assert(FALSE);	/* already exists! */
	 } /* fi */
     if (CmpVal < 0)
	{ break;
	 } /* fi */
     Next();
    } /* elihw */

if (Curr())
   { return(InsertBefore(NewUserType));
    } /* fi */
else
   { return(Append(NewUserType));
    } /* esle */

} /* end of SIR_UserTypes::Insert */


ERROR SIR_UserTypes::Integrate(		/* integrates imported user types */
	sir_usertypes	*Imported)
{
sir_usertype	*ImpCurr,
		*ImpSucc,
		*Known;

ImpCurr = Imported->First();	/* for each imported user-type */
while(ImpCurr)
   { ImpSucc = ImpCurr->Succ();
     if (ImpCurr->Name)
	{ Known = FindLocally(ImpCurr->Name->chars());
	 } /* fi */
     else
	{ if (ImpCurr->TypeDef)
	     { Known = FindLocallyViaTypeDef(ImpCurr->TypeDef->Name.chars());
	      } /* fi */
	  else
	     { sir_symbol	*KnownSymbol;

	       if (  (ImpCurr->Class == SIR_USERTYPE_ENUM)
		   &&(ImpCurr->Members)
		   &&(ImpCurr->Members->First())
		   &&((KnownSymbol = Symbols->Find(
			ImpCurr->Members->First()->Symbol->Name.chars())))
		   &&(KnownSymbol->Type->UserType)
		   &&(KnownSymbol->Type->UserType->IsSameStructure(ImpCurr)))
		  { // found two identical anonymous enum definitions
		    // (bug fix 04/08/02, RD)
		    assert(ImpCurr->Name == NULL);
		    assert(ImpCurr->TypeDef == NULL);
		    Known = KnownSymbol->Type->UserType;
		   } /* fi */
	       else
		  { Known = NULL;	/* no way to find it */
		   } /* esle */
	      } /* esle */
	 } /* esle */
     if (Known)
	{ if (ImpCurr->Class != Known->Class) /* classes must match */
	     { SIR_ErrMsg.form("Error while importing:" GL_ERROR_MSG_NEWLINE
				"Declaration '%s %s' does not match former"
							GL_ERROR_MSG_NEWLINE
				"declaration '%s %s'",
				ImpCurr->ClassName(), ImpCurr->NameOrUnnamed(),
				Known->ClassName(), Known->NameOrUnnamed());
	       return(SIR_ERROR_USERTYPE_MISMATCH_2);
	      } /* fi */
	  if (ImpCurr->IsDefinition())	/* imported a definition? */
	     { if (Known->IsDefinition())
		  { if (Known->IsSameStructure(ImpCurr))
		       { /* merge the two definitions (RD, 02/15/02) */
			 ImpCurr->SetAlias(Known); /* set alias recursively */
			} /* fi */
		    else
		       { SIR_ErrMsg.form("Error while importing:"
							GL_ERROR_MSG_NEWLINE
					"Redefinition of '%s %s'",
					Known->ClassName(),
					Known->NameOrUnnamed());
			 return(SIR_ERROR_REDEFINITION_OF_USERTYPE_3);
			} /* fi */
		   } /* fi */
	       else
		  { /* make declaration a definition */
		    Known->Members = ImpCurr->Members;	/* take the members */
		    ImpCurr->Members = NULL;
		    Known->Scope = ImpCurr->Scope;	/* take the scope */
		    ImpCurr->Scope = NULL;
		    Known->Imported = ImpCurr->Imported;/* update import info */
		   } /* esle */
	      } /* fi */
	  ImpCurr->Alias = Known;		/* put alias pointer */
// bug fix 02/15/02, RD:
// the following is not sufficient because it only updates the plain type
// (the one without 'const' and/or 'volatile' flags);
// instead, all type entries pointing to this user-type must be adjusted
// by unalias'ing the imported type table (see SIR_Design::Integrate())
// followed by type table integration (see SIR_Types::Integrate());
// anyways, the following cannot hurt so we'll leave it here
	  ImpCurr->Type->Alias = Known->Type;	/* (for type too!) */

	  /* also, merge the annotations (bug fix 06/20/02, RD) */
	  if ((SIR_Error = SIR_Notes::Merge(&Known->Notes, &ImpCurr->Notes)))
	     { return(SIR_Error);
	      } /* fi */
	 } /* fi */
     else	/* not known */
	{ Insert(Imported->Remove(ImpCurr));	/* just take the imported one */
	 } /* esle */
     ImpCurr = ImpSucc;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_UserTypes::Integrate */


void SIR_UserTypes::NameAnonymousTypes(	/* preprocessing step for WriteSC */
	void)
{
sir_usertype	*UserType,
		*Succ;
unsigned int	NumUsers;

// note: this is a preprocessing step before code generation;
// we need to assign dummy names to anonymous user-types
// that are used as type for multiple symbols because we
// don't declare symbols in a declaration list which would
// be the only option then (bug fix 04/17/02, RD)
//
// Example:
//	enum { e1, e2 } E1, E2;
// becomes
//	enum _scc_anonymous_0 { e1, e2 };
//	enum _scc_anonymous_0 E1;
//	enum _scc_anonymous_0 E2;

UserType = First();
while(UserType)
   { Succ = UserType->Succ();
     if (  (! UserType->IsNamed())
	 &&(! UserType->TypeDef))
	{ NumUsers = Symbols->NumTypeUsers(UserType->Type);
	  if (NumUsers > 1)		// used by more than one symbol?
	     { Remove(UserType);		// take out of this table
	       assert(UserType->Name == NULL);
	       UserType->Name = new string(	// assign a new name
				CreateNewName(SIR_CXX_ANONYMOUS_NAME));
	       Insert(UserType);		// re-insert into this table
	      } /* fi */
	 } /* fi */
     UserType = Succ;
    } /* elihw */

} /* end of SIR_UserTypes::NameAnonymousTypes */


const char *SIR_UserTypes::CreateNewName(	/* generate a new name */
	const char	*Format)		/* (must contain one %d) */
{
static string	NewName;
int		i;

i = 0;
do { NewName.form(Format, i++);
    } while(FindLocally(NewName.chars()));

return(NewName.chars());

} /* end of SIR_UserTypes::CreateNewName */


#ifndef NDEBUG

const char *SIR_UserTypes::Print(	/* print SpecC code */
	void)
{
GL_HANDLE	*IO;
static string	PrintBuffer;
BOOL		Dummy = FALSE;

if (!(IO = GL_IO::OpenString(&PrintBuffer, GL_IO_WRITE)))
   { SIR_Error = GL_Error;
     return(NULL);
    } /* fi */
if ((SIR_Error = WriteSC(IO, TRUE, FALSE, &Dummy)))
   { IO->Close();
     return(NULL);
    } /* fi */
if ((SIR_Error = IO->CloseWithCheck()))
   { return(NULL);
    } /* fi */

return(PrintBuffer.chars());

} /* end of SIR_UserTypes::Print */

#endif /* NDEBUG */


ERROR SIR_UserTypes::WriteSC(	/* (re-) generates SpecC (or C++) source code */
	GL_HANDLE	*IO,
	BOOL		WriteNotes,
	BOOL		CplusplusMode,
	BOOL		*PutSeparator)
{
sir_usertype	*UserType;
string		Buffer;
bool		PutIntro,
		PutIntroInit,
		PutComments;

/* (0) preprocessing step: (bug fix 04/17/02, RD) */

NameAnonymousTypes();


if (  (Symbols->ScopeInfo == SIR_SCOPE_GLOBAL)
    ||(Symbols->ScopeInfo == SIR_SCOPE_CLASS))
   { PutIntroInit = true;	/* separate code blocks */
    } /* fi */
else
   { PutIntroInit = false;	/* combine code blocks */
    } /* esle */
#ifdef SIR_PUT_COMMENTS_IN_SOURCE
if (Symbols->ScopeInfo == SIR_SCOPE_GLOBAL)
   { PutComments = true;
    } /* fi */
else
   { PutComments = false;
    } /* esle */
#else /* !SIR_PUT_COMMENTS_IN_SOURCE */
PutComments = false;
#endif /* SIR_PUT_COMMENTS_IN_SOURCE */

/* (1) Named user-defined type declarations */

PutIntro = PutIntroInit;
UserType = First();
while(UserType)
   { if (  (  (  (CplusplusMode)	/* no enum declarations in C++ */
	       &&(UserType->Class != SIR_USERTYPE_ENUM))
	    ||(! CplusplusMode))
	 &&(UserType->IsNamed()))
	{ SIR_Symbols::WriteIntro(IO, false,
			&PutIntro, PutSeparator, PutComments,
			"named user-defined type declarations");
	  SIR_LineInfo::WriteNL(IO);
	  IO->PutS(UserType->PrettyString(&Buffer, WriteNotes, CplusplusMode));
	  IO->PutC(';');
	  if (UserType->Notes)
	     { if ((SIR_Error = UserType->Notes->WriteSC(IO, WriteNotes,
						CplusplusMode)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	 } /* fi */
     UserType = UserType->Succ();
    } /* elihw */

/* (2) Named enumerator definitions */

PutIntro = PutIntroInit;
UserType = First();
while(UserType)
   { if (  (UserType->Class == SIR_USERTYPE_ENUM)
	 &&(UserType->IsNamed())
	 &&(UserType->IsDefinition()))
	{ SIR_Symbols::WriteIntro(IO, true,
			&PutIntro, PutSeparator, PutComments,
			"named enumerator type definitions");
	  if ((SIR_Error = UserType->WriteSC(IO, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  if (  (CplusplusMode)
	      &&(UserType->Notes))
	     { if ((SIR_Error = UserType->Notes->WriteSC(IO, WriteNotes,
						CplusplusMode)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	 } /* fi */
     UserType = UserType->Succ();
    } /* elihw */

/* (3) Unnamed enumerator definitions (with and without typedefs) */

PutIntro = PutIntroInit;
UserType = First();
while(UserType)
   { if (  (UserType->Class == SIR_USERTYPE_ENUM)
	 &&(! UserType->IsNamed()))
	{ assert(UserType->IsDefinition());
	  if (  (! UserType->TypeDef)
	      &&(Symbols->NumTypeUsers(UserType->Type) == 1))
	     { // skip the ones used only once (bug fix 04/17/02, RD)
	       UserType = UserType->Succ();
	       continue;
	      } /* fi */
	  SIR_Symbols::WriteIntro(IO, true,
			&PutIntro, PutSeparator, PutComments,
			"unnamed enumerator type definitions");
	  if ((SIR_Error = UserType->WriteSC(IO, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	 } /* fi */
     UserType = UserType->Succ();
    } /* elihw */

/* (4) User-defined type definitions (named and unnamed structs and unions) */

#ifndef NDEBUG
UserType = First();
while(UserType)
   { assert(UserType->Color == SIR_WHITE);	/* assure clean start */
     UserType = UserType->Succ();
    } /* elihw */
#endif /* NDEBUG */

PutIntro = PutIntroInit;
UserType = First();
while(UserType)
   { if (  (  (UserType->Class == SIR_USERTYPE_STRUCT)
	    ||(UserType->Class == SIR_USERTYPE_UNION))
	 &&(UserType->IsDefinition())
	 &&(! (  (!UserType->Name)
	       &&(!UserType->TypeDef))))
	{ if ((SIR_Error = WriteOrderedSC(UserType, IO, WriteNotes,
					CplusplusMode, PutSeparator,
					&PutIntro, PutComments)))
	     { return(SIR_Error);
	      } /* fi */
	 } /* fi */
     UserType = UserType->Succ();
    } /* elihw */

UserType = First();
while(UserType)
   { if (  (  (UserType->Class == SIR_USERTYPE_STRUCT)
	    ||(UserType->Class == SIR_USERTYPE_UNION))
	 &&(UserType->IsDefinition())
	 &&(! (  (!UserType->Name)
	       &&(!UserType->TypeDef))))
	{ assert(UserType->Color == SIR_RED);	/* assure work is done */
	  UserType->Color = SIR_WHITE;
	 } /* fi */
     UserType = UserType->Succ();
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_UserTypes::WriteSC */


ERROR SIR_UserTypes::WriteOrderedSC(	/* writes ordered structures/unions */
	sir_usertype	*Current,	/* (recursively) */
	GL_HANDLE	*IO,
	BOOL		WriteNotes,
	BOOL		CplusplusMode,
	BOOL		*PutSeparator,
	BOOL		*PutIntro,
	BOOL		PutComments)
{
sir_usertype	*UserType;

if (Current->Color == SIR_RED)		/* already written? */
   { return(SIR_ERROR_NO_ERROR);
    } /* fi */
if (Current->Color == SIR_YELLOW)	/* writing in progress? */
   { SIR_ErrMsg.form("Cyclic nesting in '%s %s' (internal)",
			Current->ClassName(),
			Current->NameOrUnnamed());
     return(SIR_ERROR_USERTYPE_CYCLIC_NESTING);
    } /* fi */
Current->Color = SIR_YELLOW;	/* start working on this one */

UserType = First();
while(UserType)
   { if (  (  (UserType->Class == SIR_USERTYPE_STRUCT)
	    ||(UserType->Class == SIR_USERTYPE_UNION))
	 &&(UserType->IsDefinition())
	 &&(! (  (!UserType->Name)
	       &&(!UserType->TypeDef))))
	{ if (Current->ContainsNested(UserType))
	     { if ((SIR_Error = WriteOrderedSC(	/* write dependencies first */
					UserType, IO, WriteNotes,
					CplusplusMode, PutSeparator, PutIntro,
					PutComments)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	 } /* fi */
     UserType = UserType->Succ();
    } /* elihw */

SIR_Symbols::WriteIntro(IO, true,
		PutIntro, PutSeparator, PutComments,
		"struct/union type definitions");
if ((SIR_Error = Current->WriteSC(IO, WriteNotes,	/* now write this one */
					CplusplusMode)))
   { return(SIR_Error);
    } /* fi */

Current->Color = SIR_RED;	/* work is done for this one */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_UserTypes::WriteOrderedSC */


/* EOF UserType.cc */
