/************************************************************************/
/* FileInfo.cc: SpecC Internal Representation, FileInfo Classes		*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 09/24/97 */
/************************************************************************/

/* last update: 06/21/04 */

/* modifications: (most recent first)
 *
 * 06/21/04 PC  Introduced Macro EOL
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 11/21/01 RD	took out default arguments from function definitions
 * 11/13/01 RD	added support for line wrapping
 * 11/08/01 RD	switched code generation to use GL_IO layer
 * 11/01/01 RD	adjusted SIR_LineInfo::WriteVSPACE()
 * 10/04/01 RD	added SIR_LineInfo::WriteVSPACE()
 * 10/03/01 RD	improved indentation of generated code
 * 05/30/01 RD	eliminated level 2 of SIR API
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 04/30/01 RD	replaced use of obsolete form() from "stream.h" with own one
 * 04/30/01 RD	created this header (last change was 12/16/98)
 */

#include "IntRep/FileInfo.h"
#include "IntRep/Design.h"

#include <stdlib.h>
#include <assert.h>


/*** constants and macros ***********************************************/


//#define SIR_DEBUG_FILE_GARBAGE_COLLECTOR	/* debug garbage collector */


/*** internal type declarations *****************************************/


	/* (none) */


/*** internal function prototypes ***************************************/


static void PrintTab(		/* print a tabulator (indent the code) */
	GL_HANDLE	*IO,
	bool		PutOpenBracket = FALSE,
	bool		PutCloseBracket = FALSE);


/*** class implementations **********************************************/


	/********************/
	/*** SIR_LineInfo ***/
	/********************/


BOOL		SIR_LineInfo::Enabled = TRUE;	/* init static variables */
unsigned int	SIR_LineInfo::LastLine = 0;
sir_fileinfo	*SIR_LineInfo::LastFile = NULL;		/* THIS File */
const char	*SIR_LineInfo::CurrentFile = NULL;	/* unknown Name */
BOOL            SIR_LineInfo::SetSourceInfo = FALSE;
sir_fileinfo    *SIR_LineInfo::CurrentFileInfo = NULL;



//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_LineInfo::SIR_LineInfo(		/* constructor #1 */
	unsigned int	Line,
	sir_fileinfo	*File)
{

assert((Line != 0) && (File != NULL));

SIR_LineInfo::Line = Line;
SIR_LineInfo::File = File;

} /* end of SIR_LineInfo::SIR_LineInfo #1 */


SIR_LineInfo::~SIR_LineInfo(void)	/* destructor */
{

/* nothing to do */

} /* end of SIR_LineInfo::~SIR_LineInfo */

void SIR_LineInfo::InitWriteSC(	/* prepares for writing new SpecC line infos */
	bool		Enable,
	const char	*ThisFilename /* = NULL */)	/* default location */
{

LastLine	= 0;		/* reset */
LastFile	= NULL;
Enabled		= Enable;
CurrentFile	= ThisFilename;

} /* end of SIR_LineInfo::InitWriteSC */

//PC 06/21/04
ERROR SIR_LineInfo::WriteSC(	/* writes line and file in SpecC source */
	GL_HANDLE	*IO,
	bool		PutSpace /* = FALSE */,
	bool		PutOpenBracket /* = FALSE */)
{
unsigned int	i;

if (! Enabled)
   { if (PutSpace)
	{ IO->PutNL();
	  PrintTab(IO, PutOpenBracket);
	 } /* fi */
     else
	{ if (PutOpenBracket)
	     { IO->PutC('{');
	      } /* fi */
	 } /* esle */
     return(SIR_ERROR_NO_ERROR);
    } /* fi */

if (LastFile == File)	/* are we still in the last file? */
   { if (  (Line < LastLine)
	 ||(Line > LastLine + SIR_MAX_EMPTY_LINES))
	{ if (IO->GetColNo() > 0)
	     { IO->PutNL();
	      } /* fi */
	  IO->PutNL();
	  IO->PrintF("#line %d \"%s\"" EOL,
			Line, File->Filename.chars());
	  PrintTab(IO, PutOpenBracket);
	 } /* fi */
     else
	{ if (Line == LastLine)
	     { if (PutSpace)
		  { IO->PutC(' ');	/* output separator */
		   } /* fi */
	       if (PutOpenBracket)
		  { IO->PutC('{');
		   } /* fi */
	      } /* fi */
	  else
	     { for(i = Line - LastLine; i>0; i--)
		  { IO->PutNL();	/* output newlines */
		   } /* rof */
	       PrintTab(IO, PutOpenBracket);
	      } /* esle */
	 } /* esle */
    } /* fi */
else	/* it's another file */
   { if (IO->GetColNo() > 0)
	{ IO->PutNL();
	 } /* fi */
     IO->PutNL();
     IO->PrintF("#line %d \"%s\"" EOL,
			Line, File->Filename.chars());
     PrintTab(IO, PutOpenBracket);
     LastFile = File;
    } /* esle */

if ((SIR_Error = IO->Check()))
   { return(SIR_Error);
    } /* fi */

LastLine = Line;

return(SIR_ERROR_NO_ERROR);	/* writing successful */

} /* end of SIR_LineInfo::WriteSC */


void SIR_LineInfo::WriteSPACE(		/* writes a space in SpecC source */
	GL_HANDLE	*IO,		/* (regardless of any line info) */
	bool		PutOpenBracket /* = FALSE */)
{

if (Enabled)
   { if (PutOpenBracket)
	{ IO->PutC('{');
	 } /* fi */
     IO->PutC(' ');
     return;
    } /* fi */

IO->PutNL();
PrintTab(IO, PutOpenBracket);

} /* end of SIR_LineInfo::WriteSPACE */


void SIR_LineInfo::WriteVSPACE(		/* writes optional vertical space */
	GL_HANDLE	*IO,
	BOOL		PutTAB /* = FALSE */)
{

if (  (! Enabled)		/* switched off? */
    ||(LastFile == NULL))	/* still in THIS current file? */
   { if (PutTAB)
	{ IO->PutNLTAB();
	 } /* fi */
     else
	{ IO->PutNL();
	 } /* esle */
    } /* fi */

} /* end of SIR_LineInfo::WriteVSPACE */

//PC 06/21/04
void SIR_LineInfo::WriteNewSC(          /* writes unknown line info */
        GL_HANDLE       *IO,            /* (probably newly created stuff) */
        bool            PutSpace /* = FALSE */,
        bool            PutOpenBracket /* = FALSE */,
        sir_lineinfo    **NewLineInfo /* = NULL */)
{
unsigned int    CurrentLine;

if (  (! Enabled)               /* switched off? */
    ||(CurrentFile == NULL)     /* current file name unknown? */
    ||(LastFile == NULL))       /* still in THIS current file? */
   { if (PutSpace)
        { IO->PutNL();
          PrintTab(IO, PutOpenBracket);
         } /* fi */
     else
        { if (PutOpenBracket)
             { IO->PutC('{');
              } /* fi */
         } /* esle */
     if (  (SetSourceInfo)              /* global switch on? */
         &&(NewLineInfo)                /* node available? */
         &&(*NewLineInfo == NULL))      /* no source info yet? */
        { CurrentLine = IO->GetLineNo();
          assert(CurrentFileInfo != NULL); // must have been set by InitWriteSC
          *NewLineInfo = new SIR_LineInfo(CurrentLine, CurrentFileInfo);
         } /* fi */
     return;
    } /* fi */

/* switch to THIS file */

if (IO->GetColNo() > 0)
   { IO->PutNL();
    } /* fi */
IO->PutNL();
CurrentLine = IO->GetLineNo() + 1;
IO->PrintF("#line %d \"%s\"" EOL, CurrentLine, CurrentFile);
PrintTab(IO, PutOpenBracket);
LastFile = NULL;                /* THIS File */
LastLine = CurrentLine;         /* THIS Line */

if (  (SetSourceInfo)           /* global switch on? */
    &&(NewLineInfo)             /* node available? */
    &&(*NewLineInfo == NULL))   /* no source info yet? */
   { assert(CurrentFileInfo != NULL);   // must have been set by InitWriteSC
     *NewLineInfo = new SIR_LineInfo(CurrentLine, CurrentFileInfo);
    } /* fi */

} /* end of SIR_LineInfo::WriteNewSC */



void SIR_LineInfo::WriteNL(		/* writes a newline in SpecC source */
	GL_HANDLE	*IO,		/* (regardless of any line info) */
	bool		PutOpenBracket /* = FALSE */,
	bool		PutCloseBracket /* = FALSE */)
{

IO->PutNL();
PrintTab(IO, PutOpenBracket, PutCloseBracket);

if (Enabled)
   { LastLine++;
    } /* fi */

} /* end of SIR_LineInfo::WriteNL */


void SIR_LineInfo::WrapLine(	/* try to wrap line around if too long */
	GL_HANDLE	*IO)
{

if (  (! Enabled)		/* switched off? */
    ||(LastFile == NULL))	/* still in THIS current file? */
   { IO->WrapLineTAB();
    } /* fi */

} /* end of SIR_LineInfo::WrapLine */


	/********************/
	/*** SIR_FileInfo ***/
	/********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_FileInfo::SIR_FileInfo(		/* constructor #1 */
	const char	*Filename)
{

assert(Filename != NULL);

SIR_FileInfo::Filename	= Filename;
SIR_FileInfo::Alias	= NULL;

} /* end of SIR_FileInfo::SIR_FileInfo #1 */


SIR_FileInfo::SIR_FileInfo(		/* constructor #3 (duplicator) */
	sir_fileinfo	*Original)
{

SIR_FileInfo::Filename	= Original->Filename;
SIR_FileInfo::Alias	= NULL;

} /* end of SIR_FileInfo::SIR_FileInfo #3 */


SIR_FileInfo::~SIR_FileInfo(void)	/* destructor */
{

/* nothing to do */

} /* end of SIR_FileInfo::~SIR_FileInfo */


	/********************/
	/*** SIR_FileList ***/
	/********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_FileList::SIR_FileList(		/* constructor #1 */
	sir_fileinfo	*FirstEntry /* = NULL */) :
		SIR_List<SIR_FileInfo>(FirstEntry)
{

/* nothing else to do */

} /* end of SIR_FileList::SIR_FileList #1 */


SIR_FileList::SIR_FileList(		/* constructor #2 (duplicator) */
	sir_file_list	*Original)
{
sir_fileinfo	*Curr;

Curr = Original->First();
while(Curr)
   { Append(new SIR_FileInfo(Curr));
     Curr = Curr->Succ();
    } /* elihw */

} /* end of SIR_FileList::SIR_FileList #2 */


SIR_FileList::~SIR_FileList(void)	/* destructor */
{

/* nothing to do */

} /* end of SIR_FileList::~SIR_FileList */


ERROR SIR_FileList::Integrate(		/* integrates imported filelist */
	sir_file_list	*Imported)
{
sir_fileinfo	*ImpCurr,
		*ImpSucc;

ImpCurr = Imported->First();
while(ImpCurr)
   { ImpSucc = ImpCurr->Succ();
     if (!(ImpCurr->Alias = Find(ImpCurr->Filename.chars())))
	{ if (Curr())
	     { InsertBefore(Imported->Remove(ImpCurr));
	      } /* fi */
	  else
	     { Append(Imported->Remove(ImpCurr));
	      } /* esle */
	 } /* fi */
     ImpCurr = ImpSucc;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FileList::Integrate */


void SIR_FileList::GarbageCollection(		/* garbage collector */
	sir_design	*Design)
{
sir_fileinfo	*Curr,
		*Succ;

#ifdef SIR_DEBUG_FILE_GARBAGE_COLLECTOR
fprintf(stderr, "GARBAGE_COLLECTOR: Starting with %d file entries...\n",
			NumElements());
#endif /* SIR_DEBUG_FILE_GARBAGE_COLLECTOR */

#ifndef NDEBUG
Curr = First();		/* make sure every entry is unmarked */
while(Curr)
   { assert(Curr->Color == SIR_WHITE);
     Curr = Curr->Succ();
    } /* elihw */
#endif /* NDEBUG */

Design->DFS_ForAllNodes(&SIR_Node::MarkUsedFiles, NULL);

Curr = First();		/* remove every unused entry (unmarked) */
while(Curr)
   { Succ = Curr->Succ();
     if (Curr->Color != SIR_WHITE)
	{ assert(Curr->Color == SIR_RED);	/* marked = used */
	  Curr->Color = SIR_WHITE;
	 } /* fi */
     else					/* unmarked = not used */
	{
#ifdef SIR_DEBUG_FILE_GARBAGE_COLLECTOR
	  fprintf(stderr, "GARBAGE_COLLECTOR: Detected unused file: \"%s\"\n",
				Curr->Filename.chars());
#endif /* SIR_DEBUG_FILE_GARBAGE_COLLECTOR */
	  Remove(Curr);
	  delete Curr;
	 } /* esle */
     Curr = Succ;
    } /* elihw */

#ifdef SIR_DEBUG_FILE_GARBAGE_COLLECTOR
fprintf(stderr, "GARBAGE_COLLECTOR: Ending with %d file entries...\n",
			NumElements());
#endif /* SIR_DEBUG_FILE_GARBAGE_COLLECTOR */

} /* end of SIR_FileList::GarbageCollection */


sir_fileinfo *SIR_FileList::FindOrInsert(	/* find an entry or insert it */
	const char	*Filename)		/* if it not exists           */
{
sir_fileinfo	*Entry;

if ((Entry = Find(Filename)))
   { return(Entry);
    } /* fi */

if (Curr())
   { return(InsertBefore(new SIR_FileInfo(Filename)));
    } /* fi */
else
   { return(Append(new SIR_FileInfo(Filename)));
    } /* esle */

} /* end of SIR_FileList::FindOrInsert */


sir_fileinfo *SIR_FileList::Find(		/* find an entry */
	const char	*Filename)	/* (returns NULL if not found) */
{
int		CmpVal;

First();	/* search for the entry */
while(Curr())
   { CmpVal = strcmp(Filename, Curr()->Filename.chars());
     if (CmpVal == 0)
	{ return(Curr());	/* found the entry */
	 } /* fi */
     if (CmpVal < 0)
	{ break;	/* not found (insert here) */
	 } /* fi */
     Next();
    } /* elihw */

return(NULL);	/* not found */

} /* end of SIR_FileList::Find */


/*** internal functions *************************************************/


static void PrintTab(		/* print a tabulator (indent the code) */
	GL_HANDLE	*IO,
	bool		PutOpenBracket /* = FALSE */,
	bool		PutCloseBracket /* = FALSE */)
{
int	n;

assert(  (PutOpenBracket == FALSE)	/* not both! */
       ||(PutCloseBracket == FALSE));

if (PutOpenBracket || PutCloseBracket)
   { if (IO->GetTab() < IO->GetTabStep())
	{ if (PutOpenBracket)
	     { IO->PutC('{');
	      } /* fi */
	  if (PutCloseBracket)
	     { IO->PutC('}');
	      } /* fi */
	  return;
	 } /* fi */
     IO->TabStepDown();
     IO->PutTAB();
     if (PutOpenBracket)
	{ IO->PutC('{');
	  for(n=1; n<IO->GetTabStep(); n++)
	     { IO->PutC(' ');
	      } /* rof */
	 } /* fi */
     else
	{ assert(PutCloseBracket);
	  IO->PutC('}');
	 } /* esle */
     IO->TabStepUp();
    } /* fi */
else
   { IO->PutTAB();
    } /* esle */

} /* end of PrintTab */


/* EOF FileInfo.cc */
