/************************************************************************/
/* GL_String.cc: Replacement for GNU's libg++ String class...      	*/
/************************************************************************/
/* Author: Andreas Gerstlauer <gerstl>		first version: 03/02/00 */
/************************************************************************/

/* last update: 06/15/04 */

/* modifications: (most recent first)
 *
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 04/19/02 AG	fixed a bug in string comparing, added regression test
 * 11/21/01 RD	took out default arguments from function definitions
 * 11/21/01 RD	started this header (last change was 11/05/01 by AG)
 */

#include "Global.h"

#include <stdio.h>
#include <limits.h>


const char* const _nullString = "";


// ------------------- String methods ---------------------


// format string
String& String::vform(const char* format, va_list args)
{
  int dummy;
  char* old_str;
  va_list arglst = args;
  const char* s = format;
  int len = strlen(format);
  
  // Loop over format string to determine length needed
  while( (s = strchr(s, '%')) )
  {
    s++;
    len -= 2;

    // what's the parameter type?
    switch(s[0])
    {
      case 'd':
      case 'u':
      case 'i':
      case 'x':
      case 'X':
      case 'o':
	dummy = va_arg(arglst, int);
        // leave room for max. length (octal)
        len += ((sizeof(int) * CHAR_BIT) / 3) + 1;
        break;

      case 'c':
        dummy = va_arg(arglst, int);
        len++;
        break;
      
      case 's':
        // room according to string length
        len += strlen( va_arg(arglst, char*) );
        break;
      
      case '%':
        s++;
        len++;
        break;
	
      default: assert(false);
    }
  }
  va_end(arglst);
  
  // Allocate memory
  old_str = salloc(len);
  if ((old_str != _str) && (old_str != _nullString)) GL_free(old_str);
  
  // Assign string
  _len = vsprintf(_str, format, args);
  assert(_len < _size);
  
  return *this;
}

String& String::form(const char* format, ...)
{  
  va_list  arglst;  
  va_start(arglst, format);
  vform(format, arglst);
  va_end(arglst);  
  return *this;
}


//  return number of occurences of target in String
int String::freq(char c) const
{
   int   count = 0;
   char* cur = _str;
   
   while ((cur = strchr(cur, c))) {
      count++;
      cur++;
   }
   return count;   
}

int String::freq(const char* t) const
{
  int   count = 0;
  
  if (t) {
    char* cur = _str;
    unsigned int t_len = strlen(t);
   
    while ((cur = strstr(cur, t))) {
      count++;
      cur += t_len;
    }
  }
  
  return count;   
}


// Substring extraction

SubString String::at(const char* t, int startpos /* = 0 */)
{
  assert ((startpos >= 0) && (startpos <= _len));
  
  char* p = (_len && t)? strstr((startpos > 0)? &_str[startpos] : _str, t) : 0;

  return p? at(p - _str, strlen(t)) : at(0, 0);
}

SubString String::at(char c, int startpos /* = 0 */) 
{
  assert ((startpos >= 0) && (startpos <= _len));
  
  char* p = _len? strchr((startpos > 0)? &_str[startpos] : _str, c) : 0;

  return p? at(p - _str, 1) : at(0, 0);
}


SubString String::before(const char* t, int startpos /* = 0 */)
{
  assert (startpos <= _len);
  
  char* p = (_len && t)? strstr((startpos > 0)? &_str[startpos] : _str, t) : 0;

  return p? before(p - _str) : before(0);
}

SubString String::before(char c, int startpos /* = 0 */) 
{
  assert (startpos <= _len);
  
  char* p = _len? strchr((startpos > 0)? &_str[startpos] : _str, c) : 0;

  return p? before(p - _str) : before(0);
}

SubString String::after(const char* t, int startpos /* = 0 */)
{
  assert (startpos <= _len);
  
  char* p = (_len && t)? strstr((startpos > 0)? &_str[startpos] : _str, t) : 0;
  int pos = p? ((p - _str) + (strlen(t) - 1)) : (_len - 1);
   
  return after(pos);
}

SubString String::after(char c, int startpos /* = 0 */)
{
  assert (startpos <= _len);
  
  char* p = _len? strchr((startpos > 0)? &_str[startpos] : _str, c) : 0;

  return p? after(p - _str) : after(_len - 1);
}

// Helper routines for allocation, copying, etc.

char* String::salloc(int len, int keep, bool force)
{
  assert (len >= 0);
  assert ((keep >= 0) && (keep <= _len));
  
  char* old_str = _str;
  
  // enough room?
  if ( (len >= _size) || force )
  {
    // grow
    if (!_size) _size = 16;
    while (_size <= len) _size <<= 1;
    
    _str = (char*) GL_malloc (_size * sizeof(char));
    _str[len] = '\0';
    
    // Are we keeping a copy of current string?
    if (keep) {
      strncpy(_str, old_str, keep);
    }
  }
   
  // set new length
  _len = len;
  
  return old_str;
}
  
void String::scopy(const char* t, int startpos, int len)
{
  assert (t);
  assert ((startpos >= 0) && (startpos <= _len));
  assert (len + startpos  <= _len + 1);
     
  // fill in with given data
  // strings might overlap -> use memmove() !
  memmove(&_str[startpos], t, len * sizeof(char));
}


void String::sinsert(const char* t, int pos, int len, int repl)
{
  assert ((pos + repl >= 0) && (pos + repl <= _len));
  // if t and _str overlap we have to make a copy!
  char* old_str = salloc(_len + len - repl, pos, !(t+len<_str) || (t>_str+_len));
  scopy(&old_str[pos + repl], pos + len, _len - (pos + len) + 1);
  scopy(t, pos, len);
  if ((old_str != _str) && (old_str != _nullString)) GL_free(old_str);
}

void String::sinsert(char c, int pos, int repl)
{
  assert ((pos + repl >= 0) && (pos + repl <= _len));
  char* old_str = salloc(_len + 1 - repl, pos);
  scopy(&old_str[pos + repl], pos + 1, _len - pos);
  _str[pos] = c;
  if ((old_str != _str) && (old_str != _nullString)) GL_free(old_str);
}


// ------------------- SubString methods ---------------------


SubString& SubString::operator =  (const String&     y) 
{
  _str.sinsert(y.chars(), _pos, y.length(), _len); 
  _len = y.length();
  return *this;
}

SubString& SubString::operator =  (const SubString&  y) 
{
  _str.sinsert(y.chars(), _pos, y.length(), _len); 
  _len = y.length();
  return *this;
}

SubString& SubString::operator =  (const char* t) 
{
  if (t) {
     int t_len = strlen(t);
     _str.sinsert(t, _pos, t_len, _len);
     _len = t_len;
  } else {
     _str.sinsert("", _pos, 0, _len);
  }
  return *this;
}

SubString& SubString::operator =  (char        c) 
{
  _str.sinsert(c, _pos, _len); 
  _len = 1;
  return *this;
}



// ------------------- Global functions ---------------------


int compare(const String& x, const String& y) 
{
  return strncmp(x.chars(), y.chars(), MAX(x.length(), y.length()));
}

int compare(const String& x, const char* t) 
{
  int res = strncmp(x.chars(), t, x.length());
  if (!res) {
    return x.length() - strlen(t);
  }
  return res;
}

int compare(const SubString& x, const String& y) 
{
  int res = strncmp(x.chars(), y.chars(), x.length());
  if (!res) {
    return x.length() - y.length();
  }
  return res;
}

int compare(const String& x, const SubString&  y) 
{
  int res = strncmp(y.chars(), x.chars(), y.length());
  if (!res) {
    return x.length() - y.length();
  }
  return -res;
}

int compare(const SubString& x, const SubString&  y) 
{
  int res;
  int diff = x.length() - y.length();
  
  if (diff < 0) {
    res = strncmp(x.chars(), y.chars(), x.length());
  } else {
    res = -strncmp(y.chars(), x.chars(), y.length());
  }
  if (!res) {
    return diff;
  }
  return res;
}

int compare(const SubString& x, const char* t) 
{
  int res = strncmp(x.chars(), t, x.length());
  if (!res) {
    return x.length() - strlen(t);
  }
  return res;
}


/************************************************************************/
/*** main (for debugging only)					      ***/
/************************************************************************/


#ifdef DEBUG	/* module self test */

void String_Error(String S, const char* msg)
{
   fprintf(stderr, "String error: %s is %s\n", msg, (const char*)S );
   exit(1);
}

int main(int argc, char **argv)
{
   String x = "Hello";
   String y = "world";
   String n = "123";
   String z(0, 31);
   char*  s = ",";
   String lft, mid, rgt;
   String words[10];
   words[0] = "a";
   words[1] = "b";
   words[2] = "c";
   
   printf("DEBUGGING: GL_String ('%s')\n\n", argv[0]);
   if (argc != 1)
     { puts("WARNING: Arguments will be ignored!\n");
     } /* fi */
   
   printf("Testing String class...\n");
   if( z.length() != 0 ) String_Error(z, "z.length() = 0");  
  
   /* The following tests were taken from the specification of the String class in the
    * GNU libg++ library. At this point only very few selected methods are really 
    * implemented (and hence tested). This will be extended as need arises...
    */
   
   /* Comparing, Searching, and Matching */
   
   /*  x.index("lo") 
           returns the zero-based index of the leftmost occurrence of substring "lo" (3, in this case). The argument may be a String, SubString, char, char*, or
           Regex. */   
   /*  x.index("l", 2) 
             returns the index of the first of the leftmost occurrence of "l" found starting the search at position x[2], or 2 in this case. */
   /*  x.index("l", -1) 
             returns the index of the rightmost occurrence of "l", or 3 here. */
   /*  x.index("l", -3) 
             returns the index of the rightmost occurrence of "l" found by starting the search at the 3rd to the last position of x, returning 2 in this case. */
   /*  pos = r.search("leo", 3, len, 0) 
             returns the index of r in the char* string of length 3, starting at position 0, also placing the length of the match in reference parameter len.  */
   /*  x.contains("He") 
             returns nonzero if the String x contains the substring "He". The argument may be a String, SubString, char, char*, or Regex.  */
   /*  x.contains("el", 1) 
             returns nonzero if x contains the substring "el" at position 1. As in this example, the second argument to contains, if present, means to match the
           substring only at that position, and not to search elsewhere in the string.  */
   /*  x.contains(RXwhite); 
         returns nonzero if x contains any whitespace (space, tab, or newline). Recall that RXwhite is a global whitespace Regex. */
   /*  x.matches("lo", 3) 
             returns nonzero if x starting at position 3 exactly matches "lo", with no trailing characters (as it does in this example). */
   /*  x.matches(r) 
             returns nonzero if String x as a whole matches Regex r.  */
   int f = x.freq("l");		if( f != 2 ) String_Error( x, "x.freq(\"l\")" );
          /* returns the number of distinct, nonoverlapping matches to the argument (2 in this case). */
   
   /* Substring extraction */
   
   z = x.at(2, 3);		if( z != "llo" ) String_Error( z, "x.at(2,3)" );
	 /* sets String z to be equal to the length 3 substring of String x
	    starting at zero-based position 2, setting z to "llo" in this case. A
	    nil String is returned if the arguments don't make sense. */

   x.at(2, 2) = "r";		if( x != "Hero" ) String_Error( x, "x.at(2,2) = \"r\"" );
	 /* Sets what was in positions 2 to 3 of x to "r", setting x to "Hero" in
	    this case. As indicated here, SubString assignments may be of different
	    lengths. */
            x = "Hello";
   x.at("He") = "je";		if( x != "jello" ) String_Error( x, "x.at(\"He\") = \"je\"" );
	 /*  x("He") is the substring of x that matches the first occurrence of it's
	     argument. The substitution sets x to "jello". If "He" did not occur,
	     the substring would be nil, and the assignment would have no effect. */
             x = "Hello";
   /* x.at("l", -1) = "i"; */
          /* replaces the rightmost occurrence of "l" with "i", setting x to
	     "Helio". */
   /* z = x.at(r); */
	  /* sets String z to the first match in x of Regex r, or "ello" in this
	     case. A nil String is returned if there is no match. */
   z = x.before("o");		if( z != "Hell" ) String_Error( z, "x.before(\"o\")"  );
	  /* sets z to the part of x to the left of the first occurrence of "o", or
	     "Hell" in this case. The argument may also be a String, SubString, or
	     Regex. (If there is no match, z is set to "".) */
   x.before("ll") = "Bri";	if( x != "Brillo" ) String_Error( x, "x.before(\"ll\") = \"Bri\"" );
	  /* sets the part of x to the left of "ll" to "Bri", setting x to "Brillo". */
             x = "Hello";
   z = x.before(2);		if( z != "He" ) String_Error( z, "x.before(2)" );
	  /* sets z to the part of x to the left of x[2], or "He" in this case. */
   z = x.after("Hel");		if( z != "lo" ) String_Error( z, "x.after(\"Hel\")" );
	  /* sets z to the part of x to the right of "Hel", or "lo" in this case. */
   /* z = x.through("el")
	     sets z to the part of x up and including "el", or "Hel" in this case. */
   /* z = x.from("el")
	     sets z to the part of x from "el" to the end, or "ello" in this case. */
   x.after("Hel") = "p";	if( x != "Help" ) String_Error( x, "x.after(\"Hel\") = \"p\"" );
	  /* sets x to "Help"; */
             x = "Hello";
   z = x.after(3);		if( z != "o" ) String_Error( z, "x.after(3)" );
          /* sets z to the part of x to the right of x[3] or "o" in this case. */
   /* z = " ab c"; z = z.after(RXwhite)
	     sets z to the part of its old string to the right of the first group of
	     whitespace, setting z to "ab c"; Use gsub(below) to strip out multiple
	     occurrences of whitespace or any pattern. */
   x[0] = 'J';			if( x != "Jello" ) String_Error( x, "x[0] = J" );
	  /* sets the first element of x to 'J'. x[i] returns a reference to the ith
	     element of x, or triggers an error if i is out of range. */
             x = "Hello";									  
   /* common_prefix(x, "Help")
	     returns the String containing the common prefix of the two Strings or
	     "Hel" in this case. */
   /* common_suffix(x, "to")
	     returns the String containing the common suffix of the two Strings or
	     "o" in this case. */
   
   
   /* Concatenation */

   z = x + s + ' ' + y.at("w") + y.after("w") + "."; if( z != "Hello, world." ) String_Error( z, "z = ..." );
	 /*  sets z to "Hello, world." */
   x += y;		if( x != "Helloworld" ) String_Error( x, "x += y" );
	 /*  sets x to "Helloworld" */
         x = "Hello";
   /* cat(x, y, z)
	     A faster way to say z = x + y. */
   /* cat(z, y, x, x)
	     Double concatenation; A faster way to say x = z + y + x. */
   y.prepend(x);	if( y != "Helloworld" ) String_Error( y, "y.prepend(x)" );
	 /*  A faster way to say y = x + y. */
             y = "world";
   /* z = replicate(x, 3);
	     sets z to "HelloHelloHello". */
   /* z = join(words, 3, "/")
	     sets z to the concatenation of the first 3 Strings in String array
	     words, each separated by "/", setting z to "a/b/c" in this case. The
	     last argument may be "" or 0, indicating no separation. */

   /* Other manipulations */

   /* z = "this string has five words"; i = split(z, words, 10, RXwhite);
	     sets up to 10 elements of String array words to the parts of z
	     separated by whitespace, and returns the number of parts actually
	     encountered (5 in this case). Here, words[0] = "this", words[1] =
	     "string", etc. The last argument may be any of the usual. If there is
	     no match, all of z ends up in words[0]. The words array is not
	     dynamically created by split. */
   /* int nmatches x.gsub("l","ll")
	     substitutes all original occurrences of "l" with "ll", setting x to
	     "Hellllo". The first argument may be any of the usual, including Regex.
	     If the second argument is "" or 0, all occurrences are deleted. gsub
	     returns the number of matches that were replaced. */
   z = x + y; z.del("loworl");		if( z != "Held" ) String_Error( z, "z.del()" );
	 /*  deletes the leftmost occurrence of "loworl" in z, setting z to "Held". */
   /* z = reverse(x)
	     sets z to the reverse of x, or "olleH". */
   /* z = upcase(x)
	     sets z to x, with all letters set to uppercase, setting z to "HELLO" */
   /* z = downcase(x)
	     sets z to x, with all letters set to lowercase, setting z to "hello" */
   /* z = capitalize(x)
	     sets z to x, with the first letter of each word set to uppercase, and
	     all others to lowercase, setting z to "Hello" */
   /* x.reverse(), x.upcase(), x.downcase(), x.capitalize()
	     in-place, self-modifying versions of the above. */

   
   /* Some regression test */

   String s1("ab");
   String s2("abc");
   if( s1 == s2 ) String_Error("TRUE", "'ab' == 'abc'");
  
   String pes("proc1 proc2");
   pes += ' ';   
   pes = pes.after(' ');
   if( pes != "proc2 " ) String_Error(pes, "pes = pes.after()");      

   pes.form("%c1%s3%d5%u7%x9%o%%", '%', "2", 4, 6, 8, 0);
   if( pes != "%1234567890%" ) String_Error(pes, "pes.form() = '%1234567890%'");

   String tst("Test\0Test", 9);
   if( tst.length() != 9 ) String_Error(tst, "tst.length() = 9");
   if( tst[4] != '\0' || tst[5] != 'T' ) String_Error(tst, "tst = 'Test\\0Test'");
     
   exit(0);
}

#endif
