/************************************************************************/
/* piped.h: SpecC run-time simulation library, 'piped' template impl.	*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 03/28/01 */
/************************************************************************/

/* last update: 06/15/04 */

/* modifications: (most recent first)
 *
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 05/16/04 PC  piped class was included into '_specc' scope
 * 05/19/03 RD	bug fix: added missing copy constructor for _specc::pipslice
 * 05/16/03 RD	do not include <string.h> on CygWin platform
 * 03/04/03 RD	fixed right operand for shifts, should be 'unsigned int'
 * 01/03/03 RD	added copy constructors and fixed post-in/de-crement operators
 * 05/10/02 RD	added methods 'member' for struct/union member access
 * 04/10/02 RD	added handling of bitvector slices (_specc::pipslice)
 * 04/09/02 RD	added special pass-through functions for bitvectors, longlongs
 * 04/02/02 RD	added conversion ops for 'piped' arrays
 * 03/28/02 RD	allow 'piped' variables to be initialized, bug fixes
 * 03/27/02 RD	added template specialization for 'piped' arrays (bug fix)
 * 09/07/01 RD	bug fix: modified 'piped' template to use static memory
 * 04/25/01 RD	added copy-assignment method (bug fix)
 * 04/04/01 RD	completed function bodies
 * 04/03/01 RD	inserted initial implementation
 * 03/28/01 RD	initial version
 */

#ifndef __PIPED_H
#define __PIPED_H

/* NOTE:							*/
/*	this "header" file contains the _implementation_ of	*/
/*	the 'piped' template class				*/

#ifndef __GNUC__	// CygWin platform must not use this header file
#include <string.h>
#endif

/*** class implementations **********************************************/


	/****************************************/
	/*** piped template (general version) ***/
	/****************************************/


template<class T, unsigned int d> inline
_specc::piped<T,d>::piped(void)			/* constructor #1 */
{

/* nothing to do */

} /* end of piped<T,d>::piped #1 */


template<class T, unsigned int d> inline
_specc::piped<T,d>::piped(		/* constructor #2 (with init) */
	const T		Init)		/* initializer */
{
unsigned int	i;

for(i=0; i<=d; i++)
   { Value[i] = Init;
    } /* rof */

} /* end of piped<T,d>::piped #2 */


template<class T, unsigned int d> inline
_specc::piped<T,d>::piped(			/* copy constructor (#3) */
	const _specc::piped<T,d> &Orig)
{
unsigned int	i;

for(i=0; i<=d; i++)
   { Value[i] = Orig.Value[i];
    } /* rof */

} /* end of piped<T,d>::piped #3 */


template<class T, unsigned int d> inline
_specc::piped<T,d>::~piped(void)		/* destructor */
{

/* nothing to do */

} /* end of piped<T,d>::~piped */


template<class T, unsigned int d> inline
_specc::piped<T,d>::operator T() const	/* conversion operator (read access) */
{

return(Value[0]);

} /* end of piped<T,d>::operator T */


template<class T, unsigned int d> inline
_specc::piped<T,d> &_specc::piped<T,d>::operator=(/* assignment operator #1 */
	T	Var)
{

Value[d] = Var;

return *this;

} /* end of piped<T,d>::operator= #1 */


template<class T, unsigned int d> inline
_specc::piped<T,d> &_specc::piped<T,d>::operator=(/* assignment operator #1b */
	const _specc::piped<T,d> &PipedVar) /* (needed, otherwise compiler  */
{					/* will generate 'bad' default) */

Value[d] = PipedVar.Value[0];

return *this;

} /* end of piped<T,d>::operator= #1b */


template<class T, unsigned int d> inline
_specc::piped<T,d> &_specc::piped<T,d>::operator+=(/* assignment operator #2 */
	T	Var)
{

Value[d] = Value[0] + Var;

return *this;

} /* end of piped<T,d>::operator+= */


template<class T, unsigned int d> inline
_specc::piped<T,d> &_specc::piped<T,d>::operator-=(/* assignment operator #3 */
	T	Var)
{

Value[d] = Value[0] - Var;

return *this;

} /* end of piped<T,d>::operator-= */


template<class T, unsigned int d> inline
_specc::piped<T,d> &_specc::piped<T,d>::operator*=(/* assignment operator #4 */
	T	Var)
{

Value[d] = Value[0] * Var;

return *this;

} /* end of piped<T,d>::operator*= */


template<class T, unsigned int d> inline
_specc::piped<T,d> &_specc::piped<T,d>::operator/=(/* assignment operator #5 */
	T	Var)
{

Value[d] = Value[0] / Var;

return *this;

} /* end of piped<T,d>::operator/= */


template<class T, unsigned int d> inline
_specc::piped<T,d> &_specc::piped<T,d>::operator%=( /* assignment operator #6 */
	T	Var)
{

Value[d] = Value[0] % Var;

return *this;

} /* end of piped<T,d>::operator%= */


template<class T, unsigned int d> inline
_specc::piped<T,d> &_specc::piped<T,d>::operator^=( /* assignment operator #7 */
	T	Var)
{

Value[d] = Value[0] ^ Var;

return *this;

} /* end of piped<T,d>::operator^= */


template<class T, unsigned int d> inline
_specc::piped<T,d> &_specc::piped<T,d>::operator&=( /* assignment operator #8 */
	T	Var)
{

Value[d] = Value[0] & Var;

return *this;

} /* end of piped<T,d>::operator&= */


template<class T, unsigned int d> inline
_specc::piped<T,d> &_specc::piped<T,d>::operator|=( /* assignment operator #9 */
	T	Var)
{

Value[d] = Value[0] | Var;

return *this;

} /* end of piped<T,d>::operator|= */


template<class T, unsigned int d> inline
_specc::piped<T,d> &_specc::piped<T,d>::operator<<=( 
                                                /* assignment operator #10 */
	unsigned int	Var)
{

Value[d] = Value[0] << Var;

return *this;

} /* end of piped<T,d>::operator<<= */


template<class T, unsigned int d> inline
_specc::piped<T,d> &_specc::piped<T,d>::operator>>=(
                                               /* assignment operator #11 */
	unsigned int	Var)
{

Value[d] = Value[0] >> Var;

return *this;

} /* end of piped<T,d>::operator>>= */


template<class T, unsigned int d> inline
_specc::piped<T,d> &_specc::piped<T,d>::operator++()	
                                             /* increment operator #1 (pre-) */
{

Value[d] = Value[0] + 1;

return *this;

} /* end of piped<T,d>::operator++ */


template<class T, unsigned int d> inline
_specc::piped<T,d> _specc::piped<T,d>::operator++(int)	
                                            /* increment operator #2 (post-) */
{
_specc::piped<T,d>	Tmp(*this);

Value[d] = Value[0] + 1;

return Tmp;

} /* end of piped<T,d>::operator++ */


template<class T, unsigned int d> inline
_specc::piped<T,d> &_specc::piped<T,d>::operator--()	
                                          /* decrement operator #1 (pre-) */
{

Value[d] = Value[0] - 1;

return *this;

} /* end of piped<T,d>::operator-- */


template<class T, unsigned int d> inline
_specc::piped<T,d> _specc::piped<T,d>::operator--(int)	
                                         /* decrement operator #2 (post-) */
{
_specc::piped<T,d>	Tmp(*this);

Value[d] = Value[0] - 1;

return Tmp;

} /* end of piped<T,d>::operator-- */


template<class T, unsigned int d>
void _specc::piped<T,d>::update(	/* shift the data in the pipeline */
	void)
{
unsigned int	i;

for(i=0; i<d; i++)
   { Value[i] = Value[i+1];
    } /* rof */

} /* end of piped<T,d>::update */


template<class T, unsigned int d> inline
bool _specc::piped<T,d>::test(void) const	/* read access to bool */
{

return(Value[0].test());

} /* end of piped<T,d>::test */


template<class T, unsigned int d> inline
int _specc::piped<T,d>::toInt(void) const	/* read access to int */
{

return(Value[0].toInt());

} /* end of piped<T,d>::toInt */


template<class T, unsigned int d> inline
unsigned int _specc::piped<T,d>::toUInt(void) const /* read access to u. int */
{

return(Value[0].toUInt());

} /* end of piped<T,d>::toUInt */


template<class T, unsigned int d> inline
long _specc::piped<T,d>::toLong(void) const	/* read access to long */
{

return(Value[0].toLong());

} /* end of piped<T,d>::toLong */


template<class T, unsigned int d> inline
unsigned long _specc::piped<T,d>::toULong(void) const
                                               /* read access to u. long */
{

return(Value[0].toULong());

} /* end of piped<T,d>::toULong */


template<class T, unsigned int d> inline
_LONGLONG _specc::piped<T,d>::toLLong(void) const /* read access to long long */
{

return(Value[0].toLLong());

} /* end of piped<T,d>::toLLong */


template<class T, unsigned int d> inline
_ULONGLONG _specc::piped<T,d>::toULLong(void) const	/* read access to u.long long */
{

return(Value[0].toULLong());

} /* end of piped<T,d>::toULLong */


template<class T, unsigned int d> inline
double _specc::piped<T,d>::toDouble(void) const	/* read access to double */
{

return(Value[0].toDouble());

} /* end of piped<T,d>::toDouble */


template<class T, unsigned int d> inline
long double _specc::piped<T,d>::toLDouble(void) const	
                                           /* read access to long double */
{

return(Value[0].toLDouble());

} /* end of piped<T,d>::toLDouble */


template<class T, unsigned int d> inline
_specc::pipslice<d> _specc::piped<T,d>::operator()(
                                 	/* single bit of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	i)
{
_bit::chunk	*VecAddr[d+1];
unsigned int	j;

for(j=0; j<=d; j++)
   { VecAddr[j] = _bit(Value[j]).addr();
    } /* rof */

return _specc::pipslice<d>(VecAddr,
			_bit(Value[0]).isUnsigned(), _bit(Value[0]).length(),
			_BITMAP(l, r, i), _BITMAP(l, r, i),
			true);	/* single bit is always unsigned */

} /* end of piped<T,d>::operator() */


template<class T, unsigned int d> inline
_specc::pipslice<d> _specc::piped<T,d>::operator()( /* slice of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr)
{
_bit::chunk	*VecAddr[d+1];
unsigned int	j;

for(j=0; j<=d; j++)
   { VecAddr[j] = _bit(Value[j]).addr();
    } /* rof */

return _specc::pipslice<d>(VecAddr,
			_bit(Value[0]).isUnsigned(), _bit(Value[0]).length(),
			_BITMAP(l, r, sl), _BITMAP(l, r, sr),
			(_bit(Value[0])).isUnsigned());

} /* end of piped<T,d>::operator() */


template<class T, unsigned int d> inline
const _specc::pipslice<d> _specc::piped<T,d>::operator()( 
                                    /* const bit of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	i) const
{
_bit::chunk	*VecAddr[d+1];
unsigned int	j;

for(j=0; j<=d; j++)
   { VecAddr[j] = _bit(Value[j]).addr();
    } /* rof */

return _specc::pipslice<d>(VecAddr,
			_bit(Value[0]).isUnsigned(), _bit(Value[0]).length(),
			_BITMAP(l, r, i), _BITMAP(l, r, i),
			true);	/* single bit is always unsigned */

} /* end of piped<T,d>::operator() */


template<class T, unsigned int d> inline
const _specc::pipslice<d> _specc::piped<T,d>::operator()( 
                               /* const slice of a vector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr) const
{
_bit::chunk	*VecAddr[d+1];
unsigned int	j;

for(j=0; j<=d; j++)
   { VecAddr[j] = _bit(Value[j]).addr();
    } /* rof */

return _specc::pipslice<d>(VecAddr,
			_bit(Value[0]).isUnsigned(), _bit(Value[0]).length(),
			_BITMAP(l, r, sl), _BITMAP(l, r, sr),
			(_bit(Value[0])).isUnsigned());

} /* end of piped<T,d>::operator() */


template<class T, unsigned int d> template<class MT> inline
_specc::pip_elem<MT,d> _specc::piped<T,d>::member(/* member access "operator" */
	int		CharOffset)
{
void		*ValuePtrs[d+1];
unsigned int	j;

for(j=0; j<=d; j++)
   { ValuePtrs[j] = (void*)(((char*)(&Value[j])) + CharOffset);
    } /* rof */

return _specc::pip_elem<MT,d>(ValuePtrs);

} /* end of piped<T,d>::member */


template<class T, unsigned int d> template<class MT> inline
const _specc::pip_elem<MT,d> _specc::piped<T,d>::member( 
                                            /* const member access "op." */
	int		CharOffset) const
{
void		*ValuePtrs[d+1];
unsigned int	j;

for(j=0; j<=d; j++)
   { ValuePtrs[j] = (void*)(((char*)(&Value[j])) + CharOffset);
    } /* rof */

return _specc::pip_elem<MT,d>(ValuePtrs);

} /* end of piped<T,d>::member */


	/*******************************************************/
	/*** piped template (specialization for array types) ***/
	/*******************************************************/


template<int i, class T, unsigned int d> inline
_specc::piped<T[i],d>::piped(void)		/* constructor #1 */
{

/* nothing to do */

} /* end of piped<T[i],d>::piped #1 */


template<int i, class T, unsigned int d> inline
_specc::piped<T[i],d>::piped(		/* constructor #2 (with init) */
	const T		Init[i])	/* initializer */
{
unsigned int	j;

for(j=0; j<=d; j++)
   { memcpy(Value[j], Init, sizeof(T[i]));
    } /* rof */

} /* end of piped<T[i],d>::piped #2 */


template<int i, class T, unsigned int d> inline
_specc::piped<T[i],d>::piped(			/* copy constructor (#3) */
	const _specc::piped<T[i],d> &Orig)
{
unsigned int	j;

for(j=0; j<=d; j++)
   { memcpy(Value[j], Orig.Value[j], sizeof(T[i]));
    } /* rof */

} /* end of piped<T[i],d>::piped #3 */


template<int i, class T, unsigned int d> inline
_specc::piped<T[i],d>::~piped(void)		/* destructor */
{

/* nothing to do */

} /* end of piped<T[i],d>::~piped */


template<int i, class T, unsigned int d> inline
_specc::piped<T[i],d>::operator const T*() const 
                                      /* conversion operator (read access) */
{

return(&Value[0][0]);

} /* end of piped<T[i],d>::operator const T* */


template<int i, class T, unsigned int d> inline
_specc::piped<T[i],d> &_specc::piped<T[i],d>::operator=(
                                     /* assignment operator #1 */
	T	Var[i])
{

memcpy(Value[d], Var, sizeof(T[i]));

return *this;

} /* end of piped<T[i],d>::operator= #1 */


template<int i, class T, unsigned int d> inline
_specc::piped<T[i],d> &_specc::piped<T[i],d>::operator=(	
                                        /* assignment operator #1b */
      const _specc::piped<T[i],d> &PipedVar)/* (needed, otherwise compiler  */
{					/* will generate 'bad' default) */

memcpy(Value[d], PipedVar.Value[0], sizeof(T[i]));

return *this;

} /* end of piped<T[i],d>::operator= #1b */


template<int i, class T, unsigned int d> inline
_specc::piped<T[i],d> &_specc::piped<T[i],d>::operator=(
                                       /* assignment operator #1c */
	const _specc::pip_elem<T[i],d> &PipElemVar)
{

memcpy(Value[d], PipElemVar.Value[0], sizeof(T[i]));

return *this;

} /* end of piped<T[i],d>::operator= #1c */


template<int i, class T, unsigned int d> inline
_specc::pip_elem<T,d> _specc::piped<T[i],d>::operator[](
                                            /* array access operator */
	int		Index)
{
void		*ValuePtrs[d+1];
unsigned int	j;

for(j=0; j<=d; j++)
   { ValuePtrs[j] = &Value[j][Index];
    } /* rof */

return _specc::pip_elem<T,d>(ValuePtrs);

} /* end of piped<T[i],d>::operator[] */


template<int i, class T, unsigned int d>
void _specc::piped<T[i],d>::update(	/* shift the data in the pipeline */
	void)
{
unsigned int	j;

for(j=0; j<d; j++)
   { memcpy(Value[j], Value[j+1], sizeof(T[i]));
    } /* rof */

} /* end of piped<T[i],d>::update */


	/*******************************************/
	/*** pip_elem template (general version) ***/
	/*******************************************/


template<class T, unsigned int d> inline
_specc::pip_elem<T,d>::pip_elem(		/* constructor #1 */
	void	*ValuePtrs[d+1])
{
unsigned int	j;

for(j=0; j<=d; j++)
   { _specc::pip_elem<T,d>::ValuePtrs[j] = (T*) ValuePtrs[j];
    } /* rof */

} /* end of _specc::pip_elem<T,d>::pip_elem #1 */


template<class T, unsigned int d> inline
_specc::pip_elem<T,d>::~pip_elem(void)		/* destructor */
{

/* nothing to do */

} /* end of _specc::pip_elem<T,d>::~pip_elem */


template<class T, unsigned int d> inline
_specc::pip_elem<T,d>::operator T() const	/* conversion operator (read) */
{

return(*ValuePtrs[0]);

} /* end of _specc::pip_elem<T,d>::operator T */


template<class T, unsigned int d> inline
_specc::pip_elem<T,d> &_specc::pip_elem<T,d>::operator=( /* assignment op. #1 */
	T	Var)
{

*ValuePtrs[d] = Var;

return *this;

} /* end of _specc::pip_elem<T,d>::operator= */


template<class T, unsigned int d> inline
_specc::pip_elem<T,d> &_specc::pip_elem<T,d>::operator=( /* assignment op.#1b */
	const pip_elem<T,d> &PipElem)	/* (needed, otherwise compiler  */
{					/* will generate 'bad' default) */

*ValuePtrs[d] = *PipElem.ValuePtrs[0];

return *this;

} /* end of _specc::pip_elem<T,d>::operator= */


template<class T, unsigned int d> inline
_specc::pip_elem<T,d> &_specc::pip_elem<T,d>::operator+=( /* assignment op.#2 */
	T	Var)
{

*ValuePtrs[d] = *ValuePtrs[0] + Var;

return *this;

} /* end of _specc::pip_elem<T,d>::operator+= */


template<class T, unsigned int d> inline
_specc::pip_elem<T,d> &_specc::pip_elem<T,d>::operator-=( /* assignment op.#3 */
	T	Var)
{

*ValuePtrs[d] = *ValuePtrs[0] - Var;

return *this;

} /* end of _specc::pip_elem<T,d>::operator-= */


template<class T, unsigned int d> inline
_specc::pip_elem<T,d> &_specc::pip_elem<T,d>::operator*=( /* assignment op.#4 */
	T	Var)
{

*ValuePtrs[d] = *ValuePtrs[0] * Var;

return *this;

} /* end of _specc::pip_elem<T,d>::operator*= */


template<class T, unsigned int d> inline
_specc::pip_elem<T,d> &_specc::pip_elem<T,d>::operator/=( /* assignment op.#5 */
	T	Var)
{

*ValuePtrs[d] = *ValuePtrs[0] / Var;

return *this;

} /* end of _specc::pip_elem<T,d>::operator/= */


template<class T, unsigned int d> inline
_specc::pip_elem<T,d> &_specc::pip_elem<T,d>::operator%=( /* assignment op.#6 */
	T	Var)
{

*ValuePtrs[d] = *ValuePtrs[0] % Var;

return *this;

} /* end of _specc::pip_elem<T,d>::operator%= */


template<class T, unsigned int d> inline
_specc::pip_elem<T,d> &_specc::pip_elem<T,d>::operator^=( /* assignment op.#7 */
	T	Var)
{

*ValuePtrs[d] = *ValuePtrs[0] ^ Var;

return *this;

} /* end of _specc::pip_elem<T,d>::operator^= */


template<class T, unsigned int d> inline
_specc::pip_elem<T,d> &_specc::pip_elem<T,d>::operator&=( /* assignment op.#8 */
	T	Var)
{

*ValuePtrs[d] = *ValuePtrs[0] & Var;

return *this;

} /* end of _specc::pip_elem<T,d>::operator&= */


template<class T, unsigned int d> inline
_specc::pip_elem<T,d> &_specc::pip_elem<T,d>::operator|=( /* assignment op.#9 */
	T	Var)
{

*ValuePtrs[d] = *ValuePtrs[0] | Var;

return *this;

} /* end of _specc::pip_elem<T,d>::operator|= */


template<class T, unsigned int d> inline
_specc::pip_elem<T,d> &_specc::pip_elem<T,d>::operator<<=( /* assignment #10 */
	unsigned int	Var)
{

*ValuePtrs[d] = *ValuePtrs[0] << Var;

return *this;

} /* end of _specc::pip_elem<T,d>::operator<<= */


template<class T, unsigned int d> inline
_specc::pip_elem<T,d> &_specc::pip_elem<T,d>::operator>>=( /* assignment #11 */
	unsigned int	Var)
{

*ValuePtrs[d] = *ValuePtrs[0] >> Var;

return *this;

} /* end of _specc::pip_elem<T,d>::operator>>= */


template<class T, unsigned int d> inline
_specc::pip_elem<T,d> &_specc::pip_elem<T,d>::operator++() /* incr.op. #1 */
{

*ValuePtrs[d] = *ValuePtrs[0] + 1;

return *this;

} /* end of _specc::pip_elem<T,d>::operator++ */


template<class T, unsigned int d> inline
_specc::pip_elem<T,d> _specc::pip_elem<T,d>::operator++(int) /* incr.op. #2 */
{
pip_elem<T,d>	Tmp(*this);

*ValuePtrs[d] = *ValuePtrs[0] + 1;

return Tmp;

} /* end of _specc::pip_elem<T,d>::operator++ */


template<class T, unsigned int d> inline
_specc::pip_elem<T,d> &_specc::pip_elem<T,d>::operator--() /* decr.op. #1 */
{

*ValuePtrs[d] = *ValuePtrs[0] - 1;

return *this;

} /* end of _specc::pip_elem<T,d>::operator-- */


template<class T, unsigned int d> inline
_specc::pip_elem<T,d> _specc::pip_elem<T,d>::operator--(int) /* decr.op. #2 */
{
pip_elem<T,d>	Tmp(*this);

*ValuePtrs[d] = *ValuePtrs[0] - 1;

return Tmp;

} /* end of _specc::pip_elem<T,d>::operator-- */


template<class T, unsigned int d> inline
bool _specc::pip_elem<T,d>::test(void) const		/* read bool */
{

return(ValuePtrs[0]->test());

} /* end of _specc::pip_elem<T,d>::test */


template<class T, unsigned int d> inline
int _specc::pip_elem<T,d>::toInt(void) const		/* read int */
{

return(ValuePtrs[0]->toInt());

} /* end of _specc::pip_elem<T,d>::toInt */


template<class T, unsigned int d> inline
unsigned int _specc::pip_elem<T,d>::toUInt(void) const	/* read u. int */
{

return(ValuePtrs[0]->toUInt());

} /* end of _specc::pip_elem<T,d>::toUInt */


template<class T, unsigned int d> inline
long _specc::pip_elem<T,d>::toLong(void) const		/* read long */
{

return(ValuePtrs[0]->toLong());

} /* end of _specc::pip_elem<T,d>::toLong */


template<class T, unsigned int d> inline
unsigned long _specc::pip_elem<T,d>::toULong(void) const /* read u. long */
{

return(ValuePtrs[0]->toULong());

} /* end of _specc::pip_elem<T,d>::toULong */


template<class T, unsigned int d> inline
_LONGLONG _specc::pip_elem<T,d>::toLLong(void) const	/* read long long */
{

return(ValuePtrs[0]->toLLong());

} /* end of _specc::pip_elem<T,d>::toLLong */


template<class T, unsigned int d> inline
_ULONGLONG _specc::pip_elem<T,d>::toULLong(void) const	/* read u.long long */
{

return(ValuePtrs[0]->toULLong());

} /* end of _specc::pip_elem<T,d>::toULLong */


template<class T, unsigned int d> inline
double _specc::pip_elem<T,d>::toDouble(void) const	/* read double */
{

return(ValuePtrs[0]->toDouble());

} /* end of _specc::pip_elem<T,d>::toDouble */


template<class T, unsigned int d> inline
long double _specc::pip_elem<T,d>::toLDouble(void) const /* read long double */
{

return(ValuePtrs[0]->toLDouble());

} /* end of _specc::pip_elem<T,d>::toLDouble */


template<class T, unsigned int d> inline
_specc::pipslice<d> _specc::pip_elem<T,d>::operator()(/* single bit of vector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	i)
{
_bit::chunk	*VecAddr[d+1];
unsigned int	j;

for(j=0; j<=d; j++)
   { VecAddr[j] = _bit(*ValuePtrs[j]).addr();
    } /* rof */

return _specc::pipslice<d>(VecAddr,
			_bit(*ValuePtrs[0]).isUnsigned(),
			_bit(*ValuePtrs[0]).length(),
			_BITMAP(l, r, i), _BITMAP(l, r, i),
			true);	/* single bit is always unsigned */

} /* end of _specc::pip_elem<T,d>::operator() */


template<class T, unsigned int d> inline
_specc::pipslice<d> _specc::pip_elem<T,d>::operator()( /* slice of a vector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr)
{
_bit::chunk	*VecAddr[d+1];
unsigned int	j;

for(j=0; j<=d; j++)
   { VecAddr[j] = _bit(*ValuePtrs[j]).addr();
    } /* rof */

return _specc::pipslice<d>(VecAddr,
			_bit(*ValuePtrs[0]).isUnsigned(),
			_bit(*ValuePtrs[0]).length(),
			_BITMAP(l, r, sl), _BITMAP(l, r, sr),
			(_bit(*ValuePtrs[0])).isUnsigned());

} /* end of _specc::pip_elem<T,d>::operator() */


template<class T, unsigned int d> inline
const _specc::pipslice<d> _specc::pip_elem<T,d>::operator()( /* const bit */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	i) const
{
_bit::chunk	*VecAddr[d+1];
unsigned int	j;

for(j=0; j<=d; j++)
   { VecAddr[j] = _bit(*ValuePtrs[j]).addr();
    } /* rof */

return _specc::pipslice<d>(VecAddr,
			_bit(*ValuePtrs[0]).isUnsigned(),
			_bit(*ValuePtrs[0]).length(),
			_BITMAP(l, r, i), _BITMAP(l, r, i),
			true);	/* single bit is always unsigned */

} /* end of _specc::pip_elem<T,d>::operator() */


template<class T, unsigned int d> inline
const _specc::pipslice<d> _specc::pip_elem<T,d>::operator()( /* const slice */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr) const
{
_bit::chunk	*VecAddr[d+1];
unsigned int	j;

for(j=0; j<=d; j++)
   { VecAddr[j] = _bit(*ValuePtrs[j]).addr();
    } /* rof */

return _specc::pipslice<d>(VecAddr,
			_bit(*ValuePtrs[0]).isUnsigned(),
			_bit(*ValuePtrs[0]).length(),
			_BITMAP(l, r, sl), _BITMAP(l, r, sr),
			(_bit(*ValuePtrs[0])).isUnsigned());

} /* end of _specc::pip_elem<T,d>::operator() */


template<class T, unsigned int d> template<class MT> inline
_specc::pip_elem<MT,d> _specc::pip_elem<T,d>::member(	/* member "operator" */
	int		CharOffset)
{
void		*NewValuePtrs[d+1];
unsigned int	j;

for(j=0; j<=d; j++)
   { NewValuePtrs[j] = (void*)(((char*)(&ValuePtrs[j])) + CharOffset);
    } /* rof */

return _specc::pip_elem<MT,d>(NewValuePtrs);

} /* end of _specc::pip_elem<T,d>::member */


template<class T, unsigned int d> template<class MT> inline
const _specc::pip_elem<MT,d> _specc::pip_elem<T,d>::member( /* cnst.mbr."op." */
	int		CharOffset) const
{
void		*NewValuePtrs[d+1];
unsigned int	j;

for(j=0; j<=d; j++)
   { NewValuePtrs[j] = (void*)(((char*)(&ValuePtrs[j])) + CharOffset);
    } /* rof */

return _specc::pip_elem<MT,d>(NewValuePtrs);

} /* end of _specc::pip_elem<T,d>::member */


	/**********************************************************/
	/*** pip_elem template (specialization for array types) ***/
	/**********************************************************/


template<int i, class T, unsigned int d> inline
_specc::pip_elem<T[i],d>::pip_elem(		/* constructor #1 */
	void	*ValuePtrs[d+1])
{
unsigned int	j;

for(j=0; j<=d; j++)
   { _specc::pip_elem<T[i],d>::ValuePtrs[j] = (T*) ValuePtrs[j];
    } /* rof */

} /* end of _specc::pip_elem<T[i],d>::pip_elem #1 */


template<int i, class T, unsigned int d> inline
_specc::pip_elem<T[i],d>::~pip_elem(void)		/* destructor */
{

/* nothing to do */

} /* end of _specc::pip_elem<T[i],d>::~pip_elem */


template<int i, class T, unsigned int d> inline
_specc::pip_elem<T[i],d>::operator const T*() const /* conv.op. (read access) */
{

return(ValuePtrs[0]);

} /* end of _specc::pip_elem<T[i],d>::operator const T* */


template<int i, class T, unsigned int d> inline
_specc::pip_elem<T[i],d> &_specc::pip_elem<T[i],d>::operator=( /* assign. #1 */
	T	Var[i])
{

memcpy(ValuePtrs[d], Var, sizeof(T[i]));

return *this;

} /* end of _specc::pip_elem<T[i],d>::operator= #1 */


template<int i, class T, unsigned int d> inline
_specc::pip_elem<T[i],d> &_specc::pip_elem<T[i],d>::operator=( /* assign. #1b */
	const pip_elem<T[i],d> &PipElem) /* (needed, otherwise compiler  */
{					 /* will generate 'bad' default) */

memcpy(ValuePtrs[d], PipElem.ValuePtrs[0], sizeof(T[i]));

return *this;

} /* end of _specc::pip_elem<T[i],d>::operator= #1b */


template<int i, class T, unsigned int d> inline
_specc::pip_elem<T[i],d> &_specc::pip_elem<T[i],d>::operator=( /* assign. #1c */
	const _specc::piped<T[i],d> &PipedVar)
{

memcpy(ValuePtrs[d], PipedVar.Value[0], sizeof(T[i]));

return *this;

} /* end of _specc::pip_elem<T[i],d>::operator= #1c */


template<int i, class T, unsigned int d> inline
_specc::pip_elem<T,d> _specc::pip_elem<T[i],d>::operator[]( /* array access */
	int		Index)
{
void		*NewValuePtrs[d+1];
unsigned int	j;

for(j=0; j<=d; j++)
   { NewValuePtrs[j] = &ValuePtrs[j][Index];
    } /* rof */

return _specc::pip_elem<T,d>(NewValuePtrs);

} /* end of _specc::pip_elem<T[i],d>::operator[] */


	/*********************************/
	/*** _specc::pipslice template ***/
	/*********************************/


template<unsigned int d> inline
_specc::pipslice<d>::pipslice(			/* constructor #1 */
	_bit::chunk	*VecAddr[d+1],
	bool		VecUnsigned,
	_bit::len_t	VecLen,
	_bit::len_t	Left,
	_bit::len_t	Right,
	bool		Unsigned)
{
unsigned int	j;

// Note: We need to construct this array of bitslices dynamically
//       because C++ doesn't support array initialization without
//       default constructors (and a default constructor for _bitslice
//       doesn't make any sense!).

for(j=0; j<=d; j++)
   { Slices[j] = new _bitslice(VecAddr[j], VecLen, VecUnsigned,
				Left, Right, Unsigned);
    } /* rof */

} /* end of _specc::pipslice<d>::pipslice #1 */


template<unsigned int d> inline
_specc::pipslice<d>::pipslice(			/* copy constructor (#2) */
	const _specc::pipslice<d> &Orig)
{
unsigned int	j;

for(j=0; j<=d; j++)
   { Slices[j] = new _bitslice(Orig.Slices[j]->ref,
				Orig.Slices[j]->sl,
				Orig.Slices[j]->sr,
				Orig.Slices[j]->isUnsigned());
    } /* rof */

} /* end of _specc::pipslice<d>::pipslice #2 */


template<unsigned int d> inline
_specc::pipslice<d>::~pipslice(void)		/* destructor */
{
unsigned int	j;

for(j=0; j<=d; j++)
   { delete Slices[j];
    } /* rof */

} /* end of _specc::pipslice<d>::~pipslice */


template<unsigned int d> inline
_specc::pipslice<d>::operator _bit() const	/* conversion operator (read) */
{

return(*Slices[0]);

} /* end of _specc::pipslice<d>::operator _bit */


template<unsigned int d> inline
_specc::pipslice<d> &_specc::pipslice<d>::operator=(	/* assignment op. #1 */
	_bit	Var)
{

*Slices[d] = Var;

return *this;

} /* end of _specc::pipslice<d>::operator= */


template<unsigned int d> inline
_specc::pipslice<d> &_specc::pipslice<d>::operator=(	/* assignment op. #1b */
	const pipslice<d> &PipSlice)	/* (needed, otherwise compiler  */
{					/* will generate 'bad' default) */

*Slices[d] = *PipSlice.Slices[0];

return *this;

} /* end of _specc::pipslice<d>::operator= */


template<unsigned int d> inline
_specc::pipslice<d> &_specc::pipslice<d>::operator+=(	/* assignment op. #2 */
	_bit	Var)
{

*Slices[d] = *Slices[0] + Var;

return *this;

} /* end of _specc::pipslice<d>::operator+= */


template<unsigned int d> inline
_specc::pipslice<d> &_specc::pipslice<d>::operator-=(	/* assignment op. #3 */
	_bit	Var)
{

*Slices[d] = *Slices[0] - Var;

return *this;

} /* end of _specc::pipslice<d>::operator-= */


template<unsigned int d> inline
_specc::pipslice<d> &_specc::pipslice<d>::operator*=(	/* assignment op. #4 */
	_bit	Var)
{

*Slices[d] = *Slices[0] * Var;

return *this;

} /* end of _specc::pipslice<d>::operator*= */


template<unsigned int d> inline
_specc::pipslice<d> &_specc::pipslice<d>::operator/=(	/* assignment op. #5 */
	_bit	Var)
{

*Slices[d] = *Slices[0] / Var;

return *this;

} /* end of _specc::pipslice<d>::operator/= */


template<unsigned int d> inline
_specc::pipslice<d> &_specc::pipslice<d>::operator%=(	/* assignment op. #6 */
	_bit	Var)
{

*Slices[d] = *Slices[0] % Var;

return *this;

} /* end of _specc::pipslice<d>::operator%= */


template<unsigned int d> inline
_specc::pipslice<d> &_specc::pipslice<d>::operator^=(	/* assignment op. #7 */
	_bit	Var)
{

*Slices[d] = *Slices[0] ^ Var;

return *this;

} /* end of _specc::pipslice<d>::operator^= */


template<unsigned int d> inline
_specc::pipslice<d> &_specc::pipslice<d>::operator&=(	/* assignment op. #8 */
	_bit	Var)
{

*Slices[d] = *Slices[0] & Var;

return *this;

} /* end of _specc::pipslice<d>::operator&= */


template<unsigned int d> inline
_specc::pipslice<d> &_specc::pipslice<d>::operator|=(	/* assignment op. #9 */
	_bit	Var)
{

*Slices[d] = *Slices[0] | Var;

return *this;

} /* end of _specc::pipslice<d>::operator|= */


template<unsigned int d> inline
_specc::pipslice<d> &_specc::pipslice<d>::operator<<=(	/* assignment op. #10 */
	unsigned int	Var)
{

*Slices[d] = *Slices[0] << Var;

return *this;

} /* end of _specc::pipslice<d>::operator<<= */


template<unsigned int d> inline
_specc::pipslice<d> &_specc::pipslice<d>::operator>>=(	/* assignment op. #11 */
	unsigned int	Var)
{

*Slices[d] = *Slices[0] >> Var;

return *this;

} /* end of _specc::pipslice<d>::operator>>= */


template<unsigned int d> inline
_specc::pipslice<d> &_specc::pipslice<d>::operator++()	/* incr.op. #1 (pre-) */
{

*Slices[d] = *Slices[0] + 1;

return *this;

} /* end of _specc::pipslice<d>::operator++ */


template<unsigned int d> inline
_specc::pipslice<d> _specc::pipslice<d>::operator++(int) /* incr.op. #2(post-)*/
{
pipslice<d>	Tmp(*this);

*Slices[d] = *Slices[0] + 1;

return Tmp;

} /* end of _specc::pipslice<d>::operator++ */


template<unsigned int d> inline
_specc::pipslice<d> &_specc::pipslice<d>::operator--()	/* decr.op. #1 (pre-) */
{

*Slices[d] = *Slices[0] - 1;

return *this;

} /* end of _specc::pipslice<d>::operator-- */


template<unsigned int d> inline
_specc::pipslice<d> _specc::pipslice<d>::operator--(int) /* decr.op. #2(post-)*/
{
pipslice<d>	Tmp(*this);

*Slices[d] = *Slices[0] - 1;

return Tmp;

} /* end of _specc::pipslice<d>::operator-- */


template<unsigned int d> inline
bool _specc::pipslice<d>::test(void) const		/* read bool */
{

return(Slices[0]->test());

} /* end of _specc::pipslice<d>::test */


template<unsigned int d> inline
int _specc::pipslice<d>::toInt(void) const		/* read int */
{

return(Slices[0]->toInt());

} /* end of _specc::pipslice<d>::toInt */


template<unsigned int d> inline
unsigned int _specc::pipslice<d>::toUInt(void) const	/* read u. int */
{

return(Slices[0]->toUInt());

} /* end of _specc::pipslice<d>::toUInt */


template<unsigned int d> inline
long _specc::pipslice<d>::toLong(void) const		/* read long */
{

return(Slices[0]->toLong());

} /* end of _specc::pipslice<d>::toLong */


template<unsigned int d> inline
unsigned long _specc::pipslice<d>::toULong(void) const	/* read u. long */
{

return(Slices[0]->toULong());

} /* end of _specc::pipslice<d>::toULong */


template<unsigned int d> inline
_LONGLONG _specc::pipslice<d>::toLLong(void) const	/* read long long */
{

return(Slices[0]->toLLong());

} /* end of _specc::pipslice<d>::toLLong */


template<unsigned int d> inline
_ULONGLONG _specc::pipslice<d>::toULLong(void) const	/* read u.long long */
{

return(Slices[0]->toULLong());

} /* end of _specc::pipslice<d>::toULLong */


template<unsigned int d> inline
double _specc::pipslice<d>::toDouble(void) const	/* read double */
{

return(Slices[0]->toDouble());

} /* end of _specc::pipslice<d>::toDouble */


template<unsigned int d> inline
long double _specc::pipslice<d>::toLDouble(void) const	/* read long double */
{

return(Slices[0]->toLDouble());

} /* end of _specc::pipslice<d>::toLDouble */


template<unsigned int d> inline
_specc::pipslice<d> _specc::pipslice<d>::operator()(	/* single bit */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	i)
{
_bit::chunk	*VecAddr[d+1];
unsigned int	j;

for(j=0; j<=d; j++)
   { VecAddr[j] = Slices[j]->addr();
    } /* rof */

return _specc::pipslice<d>(VecAddr,
			Slices[0]->isUnsigned(), Slices[0]->length(),
			_BITMAP(l, r, i), _BITMAP(l, r, i),
			true);	/* single bit is always unsigned */

} /* end of _specc::pipslice<d>::operator() */


template<unsigned int d> inline
_specc::pipslice<d> _specc::pipslice<d>::operator()(	/* slice of a vector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr)
{
_bit::chunk	*VecAddr[d+1];
unsigned int	j;

for(j=0; j<=d; j++)
   { VecAddr[j] = Slices[j]->addr();
    } /* rof */

return _specc::pipslice<d>(VecAddr,
			Slices[0]->isUnsigned(), Slices[0]->length(),
			_BITMAP(l, r, sl), _BITMAP(l, r, sr),
			Slices[0]->isUnsigned());

} /* end of _specc::pipslice<d>::operator() */


template<unsigned int d> inline
const _specc::pipslice<d> _specc::pipslice<d>::operator()( /* const bit */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	i) const
{
_bit::chunk	*VecAddr[d+1];
unsigned int	j;

for(j=0; j<=d; j++)
   { VecAddr[j] = Slices[j]->addr();
    } /* rof */

return _specc::pipslice<d>(VecAddr,
			Slices[0]->isUnsigned(), Slices[0]->length(),
			_BITMAP(l, r, i), _BITMAP(l, r, i),
			true);	/* single bit is always unsigned */

} /* end of _specc::pipslice<d>::operator() */


template<unsigned int d> inline
const _specc::pipslice<d> _specc::pipslice<d>::operator()( /* const slice */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr) const
{
_bit::chunk	*VecAddr[d+1];
unsigned int	j;

for(j=0; j<=d; j++)
   { VecAddr[j] = Slices[j]->addr();
    } /* rof */

return _specc::pipslice<d>(VecAddr,
			Slices[0]->isUnsigned(), Slices[0]->length(),
			_BITMAP(l, r, sl), _BITMAP(l, r, sr),
			Slices[0]->isUnsigned());

} /* end of _specc::pipslice<d>::operator() */


#endif /* __PIPED_H */

/* EOF piped.h */
