// c_token.sc: token channel for use with 1 to N threads
// (consumer and producer model in Petri nets)
//
// author: Rainer Doemer
//
// modifications: (most recent first)
//
// 10/04/02 RD	added rule about safety of exceptions
// 02/12/02 RD	applied naming convention, integrated with distribution
// 02/06/02 RD	initial version (based on generalized handshake.sc)
//
// interface rules:
//
// - see files i_token.sc, i_producer.sc, i_consumer.sc
//
// channel rules:
//
// - this channel implements a token channel as known from Petri nets
// - one channel instance may be used multiple times and with multiple
//   consumers and/or producers
// - if used for production and consumption by the same thread, the thread
//   may consume its own tokens
// - up to N threads may use the same channel instance, N=2**32-1
// - up to M tokens can be stored in the channel at a time, M=2**32-1
// - each connected thread may act as a consumer, producer, or both
//   (depending on the interface used)
// - if multiple consumers are waiting for tokens, the consumers to consume
//   the tokens and resume their execution are selected non-deterministically
// - calling consume() may suspend the calling thread indefinitely
// - no guarantees are given for fairness of access
// - this channel is only safe with respect to exceptions, if any exceptions
//   are guaranteed to occur only for all communicating threads simultaneously;
//   the behavior is undefined, if any exceptions occur only for a subset
//   of the communicating threads
// - no restrictions exist for use of 'waitfor'
// - no restrictions exist for use of 'wait', 'notify', 'notifyone'


import "i_producer";
import "i_consumer";
import "i_token";


channel c_token implements i_producer, i_consumer, i_token
{
    event         e;
    unsigned long t = 0,
                  w = 0;

    void consume(unsigned long n)
    {
	while(n > t)
	{
	    w++;
	    wait e;
	    w--;
	}
	t -= n;
    }

    void produce(unsigned long n)
    {
	t += n;
	if (w)
	{
	    notify e;
	}
    }
};


// EOF c_token.sc
