/************************************************************************/
/* IntRep/Type.h: SpecC Internal Representation, Type Classes		*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 10/07/97 */
/************************************************************************/

/* last update: 06/15/04 */

/* modifications: (most recent first)
 *
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 04/12/04 RD	added SIR_TYPE_NUM_TYPES (for profiler)
 * 12/04/03 HY	added SIR_TYPE_METHOD to distinguish from SIR_TYPE_FUNCTION
 * 02/21/03 RD	moved GetTable() method into API layer 2
 * 12/12/02 RD	added SIR_Type::Offset(sir_symbol*)
 * 07/16/02 RD	added SIR_Type::IsAssignableTo()
 * 07/15/02 RD	extended SIR_Types::FindOrInsertDeclaratorType()
 * 07/13/02 RD	strengthened type checking by SIR_Type::Check()
 * 07/13/02 RD	added SIR_TYPE_SIGNAL and SIR_TYPE_BUFFER
 * 07/05/02 RD	added SIR_Type::GetAlignment(), SIR_Type::Offset() and
 *		SIR_Type::GetMemberOffset();
 * 05/06/02 RD	added SIR_Types::Obtain(Text)
 * 04/02/02 RD	added AutoArraySize parameter to PrettyString() methods
 * 02/20/02 RD	allow 'import' to sort out recursive structures and enums
 * 02/15/02 RD	eliminated sizeof() difference between SpecC and C++
 *		(otherwise bitvectors cannot be malloc()ed)
 * 10/31/01 RD	added SIR_Type::PrettyString() with separated pre/post-fix
 * 05/31/01 RD	eliminated level 2 of SIR API
 * 05/26/01 RD	eliminated support for binary SIR files (import/export)
 * 04/27/01 RD	added ConvertInitializer() to perform explicit type
 *		conversion for initializers to expected type
 * 04/25/01 RD	added parameter UseShadowName to PrettyString()
 * 04/20/01 RD	added static version of ConstType()
 * 04/16/01 RD	added support for non-native 'long long' type;
 *		changed use of 'long long int' to TIME type
 * 04/16/01 RD	started this header (last update was 07/01/00)
 */

#ifndef INTREP_TYPE_H
#define INTREP_TYPE_H


#include "Global.h"
#include "IntRep/Constant.h"
#include "IntRep/Parameter.h"


/*** constants and macros ***********************************************/


#define SIR_UNKNOWN_ARRAY_SIZE	(-1)
#define SIR_UNKNOWN_SUBTYPE	((sir_type*)NULL)


/*** enumeration types **************************************************/


enum SIR_TypeType		/* supported types */
{				/* (see class SIR_Type) */
SIR_TYPE_BOOL		= SIR_CONST_BOOL,
SIR_TYPE_CHAR		= SIR_CONST_CHAR,
SIR_TYPE_UCHAR		= SIR_CONST_UCHAR,	/* SIR_TYPE_CHAR + 1 */
SIR_TYPE_SHORT		= SIR_CONST_SHORT,
SIR_TYPE_USHORT		= SIR_CONST_USHORT,	/* SIR_TYPE_SHORT + 1 */
SIR_TYPE_INT		= SIR_CONST_INT,
SIR_TYPE_UINT		= SIR_CONST_UINT,	/* SIR_TYPE_INT + 1 */
SIR_TYPE_LONG		= SIR_CONST_LONG,
SIR_TYPE_ULONG		= SIR_CONST_ULONG,	/* SIR_TYPE_LONG + 1 */
SIR_TYPE_LONGLONG	= SIR_CONST_LONGLONG,
SIR_TYPE_ULONGLONG	= SIR_CONST_ULONGLONG,	/* SIR_TYPE_LONGLONG + 1 */
SIR_TYPE_FLOAT		= SIR_CONST_FLOAT,
SIR_TYPE_DOUBLE		= SIR_CONST_DOUBLE,
SIR_TYPE_LONGDOUBLE	= SIR_CONST_LONGDOUBLE,
SIR_TYPE_BIT		= SIR_CONST_BIT,
SIR_TYPE_UBIT		= SIR_CONST_UBIT,	/* SIR_TYPE_BIT + 1 */
SIR_TYPE_VOID,
SIR_TYPE_EVENT,
SIR_TYPE_POINTER,
SIR_TYPE_STRUCT,
SIR_TYPE_UNION,
SIR_TYPE_ENUM,
SIR_TYPE_ARRAY,
SIR_TYPE_FUNCTION,
SIR_TYPE_METHOD,
SIR_TYPE_ANY_TYPE,	/* function parameter ellipsis (...) */
SIR_TYPE_BEHAVIOR,	/* actually a class */
SIR_TYPE_CHANNEL,	/* actually a class */
SIR_TYPE_INTERFACE,	/* actually a class */
SIR_TYPE_SIGNAL,
SIR_TYPE_BUFFER,

SIR_TYPE_TIME		= SIR_CONST_TIME	/* type 'time' equivalence */
};

#define SIR_TYPE_LAST_TYPE	SIR_TYPE_BUFFER
#define SIR_TYPE_NUM_TYPES	(SIR_TYPE_LAST_TYPE + 1)

enum SIR_TypeClass		/* classification of types */
{
SIR_TYPECLASS_INTEGRAL,
SIR_TYPECLASS_FLOATING,
SIR_TYPECLASS_BITVECTOR,
SIR_TYPECLASS_CLASS,
SIR_TYPECLASS_OTHER
};

enum SIR_PortDirection		/* port direction qualifier */
{
SIR_PORT_NONE,
SIR_PORT_IN,
SIR_PORT_OUT,
SIR_PORT_INOUT
};


/*** type definitions ***************************************************/


typedef TIME				sir_time;	/* type 'time' alias */

typedef enum SIR_TypeType		SIR_TYPETYPE;
typedef enum SIR_TypeClass		SIR_TYPECLASS;
typedef enum SIR_PortDirection		SIR_DIRECTION;

typedef class SIR_Type			sir_type;
typedef SIR_List<sir_type>		sir_type_list;
typedef class SIR_Types			sir_types;

typedef class SIR_TypePtr		sir_type_ptr;
typedef SIR_List<sir_type_ptr>		sir_type_ptr_list;
typedef class SIR_TypePtrs		sir_type_ptrs;

typedef class SIR_UserType		sir_user_type;	/* cyclic link */
typedef class SIR_UserTypes		sir_usertypes;	/* cyclic link */
typedef class SIR_Symbol		sir_symbol;	/* cyclic link */
typedef class SIR_Symbols		sir_symbols;	/* cyclic link */
typedef class SIR_Member		sir_member;	/* cyclic link */

typedef class SIR_Declarator		sir_declarator;	/* cyclic link */
typedef class SIR_BasicType		sir_basic_type;	/* cyclic link */
typedef class SIR_Initializer		sir_initializer;/* cyclic link */

typedef class SIR_Design		sir_design;	/* cyclic link */


/*** class declarations *************************************************/


	/****************/
	/*** SIR_Type ***/
	/****************/


class SIR_Type :		/* type (entry in the list of types) */
	public SIR_ListElem<SIR_Type>	/* is a list element */
{
public:
SIR_TYPETYPE	Type;		/* type of this type (see above) */
const char	*Name;		/* pointer to type name (NULL if unnamed) */
sir_type	*SubType;	/* subtype link (PTR/ARY/FCT/METH/SIG/BUF)*/
sir_user_type	*UserType;	/* link to user type (if STRUCT, UNION, ENUM) */
sir_symbol	*ClassSymbol;	/* link to class symbol (BEH/CHAN/INT/METH) */
sir_type_ptrs	*Parameters;	/* parameters or ports (FUNC., BEH., CHAN.) */
int		LeftBound,	/* bounds (if BIT or UBIT) */
		RightBound;
int		Size;		/* array size (if ARRAY) */
bool		Const;		/* const qualifier */
bool		Volatile;	/* volatile qualifier */
SIR_DIRECTION	Direction;	/* in/out/inout qualifier (for ports) */

sir_type	*Alias;		/* alias pointer (temporary only) */


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Type(			/* constructor #1 (for built-in types) */
	SIR_TYPETYPE	Type,
	bool		Const = false,
	bool		Volatile = false,
	int		LeftBound = 0,
	int		RightBound = 0,
	SIR_DIRECTION	Direction = SIR_PORT_NONE);

SIR_Type(			/* constructor #2 (for composite types) */
	sir_type	*SubType,
	SIR_TYPETYPE	Type,
	sir_type_ptrs	*Parameters = NULL,
	int		Size = 0,
	bool		Const = false,
	bool		Volatile = false,
	SIR_DIRECTION	Direction = SIR_PORT_NONE);

SIR_Type(			/* constructor #3 (for class types) */
	SIR_TYPETYPE	Type,
	const char	*NamePtr,
	sir_symbol	*ClassSymbol = NULL,
	sir_type_ptrs	*Parameters = NULL);

SIR_Type(                       /* constructor #4 (for method types) */
	sir_type	*SubType,
        sir_type_ptrs   *Parameters,
        sir_symbol      *ClassSymbol);

SIR_Type(			/* constructor #5 (duplicator) */
	sir_type	*Original);

SIR_Type(			/* constructor #6 (for user-defined types) */
	SIR_TYPETYPE	Type,
	const char	*NamePtr,
	sir_user_type	*UserType,
	bool		Const = false,
	bool		Volatile = false,
	SIR_DIRECTION	Direction = SIR_PORT_NONE);

~SIR_Type(void);		/* destructor */


BOOL IsPort(void);		/* determines if this is a port type */

BOOL VoidParameters(void);	/* determines if there are no parameters */

BOOL IsConvertableTo(		/* determines possibility of conversion */
	sir_type	*TargetType);	/* (silent standard conversions) */

BOOL IsAssignableTo(		/* determines possibility of assignment */
	sir_type	*TargetType);

BOOL SubTypeTreeContains(	/* determines special dependency */
	sir_type	*TargetType);	/* (Bug fix 07/01/00, R.D.) */

static SIR_CONSTTYPE ConstType(	/* determines the constant type of the type */
	SIR_TYPETYPE	TypeType);

SIR_CONSTTYPE ConstType(void);	/* determines the constant type of this type */

ERROR Check(			/* performs general legal checks on this type */
	bool		VoidIsOK = false,
	bool		IsPort = false,
	bool		IsParameter = false,
	bool		IsMember = false,
	bool		IsExternOrTypedef = false,
	bool		IsPointer = false,	/* used in recursion */
	bool		IsFunction = false,	/* used in recursion */
	bool		IsArray = false,	/* used in recursion */
	bool		IsBuffered = false);	/* used in recursion */

ERROR CheckInitializer(		/* check if initializer is acceptable */
	sir_initializer	*Initializer);	/* (works recursively) */

void ConvertInitializer(	/* convert initializer to expected type */
	sir_initializer	*Initializer);	/* (works recursively) */

sir_types *GetTable(void);      /* determines the type table this type is in */


sir_type *Modified(		/* returns a (probably new) modified type */
	bool		Const = false,	/* (must be in a table) */
	bool		Volatile = false,
	SIR_DIRECTION	Direction = SIR_PORT_NONE);


unsigned int SizeOf(		/* compute the size of this type */
	unsigned int	*Alignment = NULL);	/* obtain alignment */

unsigned int Offset(		/* compute the offset of a member (#1) */
	sir_member	*Member);	/* member of struct/union */

unsigned int Offset(		/* compute the offset of a member (#2) */
	sir_symbol	*Symbol);	/* symbol as member of struct/union */


static const char *PrintDirection(	/* prints a SpecC port direction */
	SIR_DIRECTION	Direction,	/* (eg. "out" or "out ") */
	BOOL		AppendSpace = false);

const char *PrettyString(		/* prints a SpecC type (#1) */
	string	*Buffer,		/* (Buffer must be initialized */
	BOOL	IncludeNotes = FALSE,	/* with SymbolName or "" for   */
	BOOL	CplusplusMode = FALSE,	/* abstract types)             */
	BOOL	UserTypeDefReq = FALSE,
	BOOL	UseShadowName = FALSE,
	int	AutoArraySize = SIR_UNKNOWN_ARRAY_SIZE);

void PrettyString(			/* prints a SpecC type (#2) */
	string	*PrefixReturn,		/* (both Prefix- and Postfix- */
	string	*PostfixReturn,		/* Return must be initialized */
	BOOL	IncludeNotes = FALSE,	/* with an empty string ("")  */
	BOOL	CplusplusMode = FALSE,
	BOOL	UserTypeDefReq = FALSE,
	BOOL	UseShadowName = FALSE,
	int	AutoArraySize = SIR_UNKNOWN_ARRAY_SIZE);


void MarkUsed(void);		/* mark this type as being used */

SIR_TYPECLASS TypeClass(void);	/* determines the class of this type */
};


	/*****************/
	/*** SIR_Types ***/
	/*****************/


class SIR_Types :		/* list of all types */
	public SIR_List<SIR_Type>	/* is basically a list */
{					/* with additional members/methods */
public:
sir_design	*ThisDesign;	/* pointer back to the whole design */


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Types(			/* constructor #1 */
	sir_design	*ThisDesign);

SIR_Types(			/* constructor #3 (duplicator) */
	sir_types	*Original,
	sir_design	*ThisDesign);

~SIR_Types(void);		/* destructor */


void UnAlias(void);		/* unalias all type, usertype, symbol links */

ERROR Integrate(			/* integrates imported type table */
	sir_types	*Imported);

void GarbageCollection(void);	/* garbage collector */


sir_type *FindOrInsert(			/* find an entry or insert it */
	SIR_CONSTTYPE	ConstType,	/* if it not exists (#1)      */
	int		LeftBound = 0,	/* (for constant types)       */
	int		RightBound = 0);

sir_type *FindOrInsert(			/* find an entry or insert it */
	SIR_TYPETYPE	Type,		/* if it not exists (#2)      */
	bool		Const = false,	/* (for simple types)         */
	bool		Volatile = false,
	int		LeftBound = 0,
	int		RightBound = 0,
	SIR_DIRECTION	Direction = SIR_PORT_NONE);

sir_type *FindOrInsert(			/* find an entry or insert it */
	sir_basic_type	*BasicType);	/* if it not exists (#3)      */
					/* (support function for parser!) */

sir_type *FindOrInsert(			/* find an entry or insert it */
	SIR_TYPETYPE	Type,		/* if it not exists (#4)      */
	sir_type	*SubType,	/* (for composite types)      */
	sir_type_ptrs	*Parameters = NULL,
	int		Size = 0,
	bool		Const = false,
	bool		Volatile = false,
	SIR_DIRECTION	Direction = SIR_PORT_NONE,
	sir_symbol      *ClassSymbol = NULL);


sir_type *FindOrInsert(			/* find an entry or insert it */
	sir_type	*NewType);	/* if it not exists (#5)      */
					/* (consumes NewType!) */

sir_type *FindOrInsert(			/* find an entry or insert it */
	sir_constant	*Const);	/* if it not exists (#6)      */
					/* (for constant types)       */

sir_type *FindOrInsertModifiedType(	/* find or insert a modified type */
	sir_type	*OrigType,
	bool		Const = false,
	bool		Volatile = false,
	SIR_DIRECTION	Direction = SIR_PORT_NONE);

sir_type *FindOrInsertDeclaratorType(	/* find or insert a declarator type */
	sir_type	*TypeEntry,	/* (support function for parser!) */
	sir_type_list	*DeclTypes,	/* (consumes the list elements) */
	SIR_TYPETYPE	SignalClass = SIR_TYPE_VOID);

sir_type *FindOrInsertClassType(	/* find or insert a class type */
	SIR_TYPETYPE	Type,
	const char	*Name,
	sir_type_ptrs	*Parameters,
	sir_symbol	*ClassSymbol = NULL);

sir_type *Find(				/* find an equivalent entry */
	sir_type	*Type);		/* (where TypeCmp() == 0) */

sir_type *Insert(			/* insert a new entry */
	sir_type	*NewType);

static int MethodTypeCmp(		/* method type comparision	*/
        sir_type        *Type1,
        sir_type        *Type2);

static int TypeCmp(			/* type comparision (like strcmp) */
	sir_type	*Type1,
	sir_type	*Type2,
	bool		IgnoreDirection = false,
	bool		XCmpUTypes = false);

};


	/*******************/
	/*** SIR_TypePtr ***/
	/*******************/


class SIR_TypePtr :		/* type pointer */
	public SIR_ListElem<SIR_TypePtr>	/* is a list element */
{
public:
sir_type	*Type;		/* link to the type */


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_TypePtr(			/* constructor #1 */
	sir_type	*Type);

SIR_TypePtr(			/* constructor #3 (duplicator) */
	sir_type_ptr	*Original);

~SIR_TypePtr(void);		/* destructor */
};


	/********************/
	/*** SIR_TypePtrs ***/
	/********************/


class SIR_TypePtrs :		/* list of type pointers */
	public SIR_List<SIR_TypePtr>	/* is basically a list */
{					/* with additional functions */
public:

//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_TypePtrs(			/* constructor #1 */
	sir_type_ptr	*FirstEntry = NULL);

SIR_TypePtrs(			/* constructor #2 (duplicator) */
	sir_type_ptrs	*Original);

SIR_TypePtrs(			/* constructor #3 (from Parameters) */
	sir_parameters	*Parameters);

~SIR_TypePtrs(void);		/* destructor */


void UnAlias(void);		/* unalias all type, usertype, symbol links */

static int TypeCmp(		/* type comparision (like strcmp) */
	sir_type_ptrs	*TypeList1,
	sir_type_ptrs	*TypeList2,
	bool		IgnoreDirection = false,
	bool		XCmpUTypes = false);


};


#endif /* INTREP_TYPE_H */

/* EOF IntRep/Type.h */
