/************************************************************************/
/* IntRep/FileInfo.h: SpecC Internal Representation, File Info Classes	*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 09/24/97 */
/************************************************************************/

/* last update: 06/15/04 */

/* modifications: (most recent first)
 *
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 09/19/02 RD	added SetSourceInfo parameter to code generation
 * 11/13/01 RD	added support for line wrapping
 * 11/08/01 RD	switched code generation to use GL_IO layer
 * 10/04/01 RD	added SIR_LineInfo::WriteVSPACE()
 * 10/03/01 RD	improved indentation of generated code
 * 05/31/01 RD	eliminated level 2 of SIR API
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 05/25/01 RD	introduced this header (last change was 12/29/98)
 */

#ifndef INTREP_FILEINFO_H
#define INTREP_FILEINFO_H


#include "Global.h"
#include "IntRep/List.h"


/*** constants and macros ***********************************************/


#define SIR_MAX_EMPTY_LINES	5	/* max. 5 successive empty lines */


/*** enumeration types **************************************************/


	/* none */


/*** type definitions ***************************************************/


typedef class SIR_LineInfo		sir_lineinfo;

typedef class SIR_FileInfo		sir_fileinfo;
typedef SIR_List<sir_fileinfo>		sir_fileinfo_list;
typedef class SIR_FileList		sir_file_list;

typedef class SIR_Design		sir_design;	/* cyclic link */


/*** class declarations *************************************************/


	/********************/
	/*** SIR_LineInfo ***/
	/********************/


class SIR_LineInfo :		/* (source) line information */
	public SIR_Unit			/* is a unit of the design */
{
public:
static BOOL		Enabled;	/* internal stuff */
static unsigned int	LastLine;
static sir_fileinfo	*LastFile;
static const char	*CurrentFile;
static BOOL		SetSourceInfo;
static sir_fileinfo	*CurrentFileInfo;


unsigned int	Line;		/* line number */
sir_fileinfo	*File;		/* link to file information */


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_LineInfo(				/* constructor #1 */
	unsigned int	Line,
	sir_fileinfo	*File);

~SIR_LineInfo(void);			/* destructor */


static void InitWriteSC(        /* prepares for writing new SpecC line infos */
        bool            Enable,
        const char      *ThisFilename = NULL);  /* default location */

ERROR WriteSC(			/* writes line and file in SpecC source */
	GL_HANDLE	*IO,
	bool		PutSpace = FALSE,
	bool		PutOpenBracket = FALSE);

static void WriteSPACE(		/* writes a space in SpecC source */
	GL_HANDLE	*IO,	/* (regardless of any line info) */
	bool		PutOpenBracket = FALSE);

static void WriteVSPACE(	/* writes optional vertical space */
	GL_HANDLE	*IO,
	BOOL		PutTAB = FALSE);

static void WriteNewSC(         /* writes unknown line info */
        GL_HANDLE       *IO,    /* (probably newly created stuff) */
        bool            PutSpace = FALSE,
        bool            PutOpenBracket = FALSE,
        sir_lineinfo    **NewLineInfo = NULL);

static void WriteNL(		/* writes a newline in SpecC source */
	GL_HANDLE	*IO,	/* (regardless of any line info) */
	bool		PutOpenBracket = FALSE,
	bool		PutCloseBracket = FALSE);

static void WrapLine(		/* try to wrap line around if too long */
	GL_HANDLE	*IO);
};


	/********************/
	/*** SIR_FileInfo ***/
	/********************/


class SIR_FileInfo :		/* (source) file information */
	public SIR_ListElem<SIR_FileInfo>	/* is a list element */
{
public:
string		Filename;	/* file name */

sir_fileinfo	*Alias;		/* alias pointer (temporary only) */


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_FileInfo(			/* constructor #1 */
	const char	*Filename);

SIR_FileInfo(			/* constructor #3 (duplicator) */
	sir_fileinfo	*Original);

~SIR_FileInfo(void);		/* destructor */
};


	/********************/
	/*** SIR_FileList ***/
	/********************/


class SIR_FileList :		/* list of (source) files */
	public SIR_List<SIR_FileInfo>	/* is a basically a list */
{					/* with additional functions */
public:

//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_FileList(			/* constructor #1 */
	sir_fileinfo	*NewFirstEntry = NULL);

SIR_FileList(			/* constructor #2 (duplicator) */
	sir_file_list	*Original);

~SIR_FileList(void);		/* destructor */


ERROR Integrate(			/* integrates imported filelist */
	sir_file_list	*Imported);

void GarbageCollection(			/* garbage collector */
	sir_design	*Design);


sir_fileinfo *FindOrInsert(		/* find an entry or insert it */
	const char	*Filename);	/* if it not exists           */

sir_fileinfo *Find(			/* find an entry */
	const char	*Filename);	/* (returns NULL if not found) */
};


#endif /* INTREP_FILEINFO_H */

/* EOF IntRep/FileInfo.h */
