/************************************************************************/
/* signals.h: SpecC run-time simulation library, 'signal' template impl.*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 02/26/02 */
/************************************************************************/

/* last update: 06/15/04 */

/* modifications: (most recent first)
 *
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 05/16/04 PC  buffered and piped class was included into '_specc' scope
 * 05/14/04 PC  signal class was included into '_specc' scope
 * 05/02/04 PC	Fixes to fix the errors in compiling using gcc-3.4.0
 * 04/13/04 RD	fix special case for CygWin platform (declare memcpy())
 * 05/16/03 RD	do not include <string.h> on CygWin platform
 * 05/12/03 RD	fixed a "non-lvalue in unary &" condition as reported by AG
 * 03/20/03 RD	added support for signal/buffered bit vectors in port mapping
 * 03/07/03 RD	introduced template specialization of buffered class
 *		for bitvectors (which needs put() and get() methods)
 * 03/04/03 RD	fixed right operand for shifts, should be 'unsigned int'
 * 01/03/03 RD	added copy constructors and fixed post-in/de-crement operators
 * 12/13/02 RD	added support for asynchronous reset of 'buffered' variables
 * 12/06/02 RD	distinguish 'signal'/'buffered' variables in separate lists
 * 11/29/02 RD	class redesign: signal _is_ an event (not, _has_ an event);
 *		also, added support for signal edges in event lists
 * 05/10/02 RD	added methods 'member' for struct/union member access
 * 04/10/02 RD	added handling of bitvector slices (_specc::bufslice,
 *		_specc::sigslice)
 * 04/09/02 RD	added special pass-through functions for bitvectors, longlongs
 * 04/02/02 RD	added conversion ops for 'buffered', 'signal' arrays
 * 03/26/02 RD	added template classes for 'signal' array accesses
 * 03/26/02 RD	added missing array-assignment operators
 * 03/22/02 RD	added template classes for 'buffered' array accesses
 * 03/08/02 RD	added template specialization for array types
 * 03/07/02 RD	changed 'buffered' constructors to fixed number of parameters
 *		in order to avoid ambiguities
 * 03/06/02 RD	added event access method to 'signal'
 * 03/05/02 RD	refined 'signal' and 'buffered' templates (take out auto-
 *		conversion to event pointer; add explicit constructors for
 *		initialization; make 'signal' methods call the 'buffered'
 *		methods)
 * 02/27/02 RD	completed and integrated into 'base' distribution
 * 02/26/02 RD	initial version (based on piped.h)
 */

#ifndef __SIGNALS_H
#define __SIGNALS_H

/* NOTE:							*/
/*	this "header" file contains the _implementation_ of	*/
/*	the 'buffered' and 'signal' template classes		*/

#ifndef __GNUC__	// CygWin platform must not use this header file
#include <string.h>
#else
extern "C" void *memcpy(void*, const void*, unsigned int);
#endif
#include <assert.h>


/*** helper functions ***************************************************/


// NOTE: for the method
//
//	bool signal<T>::IsValidEdge(_specc::EDGE Edge)
//
// we need different implementations depending on the template type T;
// template specializations come in handy to accomplish this;
//
// more specifically, there is six sets of types:
//
// (a) built-in types ('int', 'float', etc.)
// (b) bit vectors (bit<l,s>)
// (c) longlong types (in case of non-native longlong)
// (d) pointer types
// (e) other types (such as 'struct', etc.)
// (f) array types (handled by specialization class signal<T[i]>)
//
// now, the types in (a), (b), (c), and (d) support an is-zero test
// (that is, they can be used with 'rising' and 'falling');
// however, the types in (e) and (f) do not support an is-zero test
// (that is, they cannot be used with 'rising' and 'falling');
//
// since the class signal<T> is very large and already consists of
// two specializations, we define here helper functions that can
// be template-specialized locally;
//
// further, template specialization only allows one default
// implementation that we need for the case (e) because we cannot
// enumerate those types; so, the following general helper version is
// for those "ugly" types:
//
// case (e): general version for other types


template <class T>		/* (general version) */
inline bool _signal_ValidEdge(	/* helper for signal edge event filter */
	_specc::EDGE		Edge,
	const _specc::signal<T>		& /* Sig */)
{

assert(Edge == _specc::EDGE_ANY);	// cannot apply 'rising' or 'falling'
					// to a default type (type checked)!

return(true);	/* every event is valid */

} /* end of _signal_ValidEdge */


// now, after the general version, we can define the specializations:
//
// case (a): specialization for built-in types
//
// note that all built-in types use the very same code, except for the
// type specifier; for example, the code for 'int' is as follows:


template <>			/* (specialization for 'int' type) */
inline bool _signal_ValidEdge(	/* helper for signal edge event filter */
	_specc::EDGE		Edge,
	const _specc::signal<int>	&Sig)
{

if (Edge == _specc::EDGE_ANY)
   { return(true);	/* every event is valid */
    } /* fi */

if (Edge == _specc::EDGE_RISING)
   { return((!Sig.CurrValue) && (Sig.NextValue));
    } /* fi */
else
   { assert(Edge == _specc::EDGE_FALLING);
     return((Sig.CurrValue) && (!Sig.NextValue));
    } /* esle */

} /* end of _signal_ValidEdge */

// in order to save us some typing, we'll use the preprocessor to define
// the specializations for the rest of the built-in types:
// (this is surely an issue that could be improved in C++!)

#define _SIGNAL_VALID_EDGE_CODE(type)					\
  template <>			/* (specialization for 'type' type) */	\
  inline bool _signal_ValidEdge(/* helper for signal edge event filter*/\
	_specc::EDGE		Edge,					\
	const _specc::signal<type>	&Sig)					\
{									\
									\
if (Edge == _specc::EDGE_ANY)						\
   { return(true);	/* every event is valid */			\
    } /* fi */								\
									\
if (Edge == _specc::EDGE_RISING)					\
   { return((!Sig.CurrValue) && (Sig.NextValue));			\
    } /* fi */								\
else									\
   { assert(Edge == _specc::EDGE_FALLING);				\
     return((Sig.CurrValue) && (!Sig.NextValue));			\
    } /* esle */							\
									\
} /* end of _signal_ValidEdge */

_SIGNAL_VALID_EDGE_CODE(bool)

_SIGNAL_VALID_EDGE_CODE(char)
_SIGNAL_VALID_EDGE_CODE(unsigned char)

_SIGNAL_VALID_EDGE_CODE(short)
_SIGNAL_VALID_EDGE_CODE(unsigned short)

// _SIGNAL_VALID_EDGE_CODE(int)	/* see example above */
_SIGNAL_VALID_EDGE_CODE(unsigned int)

_SIGNAL_VALID_EDGE_CODE(long)
_SIGNAL_VALID_EDGE_CODE(unsigned long)

#ifdef HAVE_LLONG
_SIGNAL_VALID_EDGE_CODE(long long)
_SIGNAL_VALID_EDGE_CODE(unsigned long long)
#endif /* HAVE_LLONG */

_SIGNAL_VALID_EDGE_CODE(float)
_SIGNAL_VALID_EDGE_CODE(double)
_SIGNAL_VALID_EDGE_CODE(long double)

#undef _SIGNAL_VALID_EDGE_CODE


// case (b): specialization for bit vectors


template <_bit::len_t l,bool s>	/* (specialization for bit vectors) */
inline bool _signal_ValidEdge(	/* helper for signal edge event filter */
	_specc::EDGE		Edge,
	const _specc::signal<bit<l,s> >	&Sig)
{

if (Edge == _specc::EDGE_ANY)
   { return(true);	/* every event is valid */
    } /* fi */

if (Edge == _specc::EDGE_RISING)
   { return((!Sig.CurrValue.test()) && (Sig.NextValue.test()));
    } /* fi */
else
   { assert(Edge == _specc::EDGE_FALLING);
     return((Sig.CurrValue.test()) && (!Sig.NextValue.test()));
    } /* esle */

} /* end of _signal_ValidEdge */


// case (c): specialization for longlong type


#ifndef HAVE_LLONG
template <bool s>		/* (specialization for longlong class) */
inline bool _signal_ValidEdge(	/* helper for signal edge event filter */
	_specc::EDGE			Edge,
	const _specc::signal<_longlong<s> >	&Sig)
{

if (Edge == _specc::EDGE_ANY)
   { return(true);	/* every event is valid */
    } /* fi */

if (Edge == _specc::EDGE_RISING)
   { return((!Sig.CurrValue.test()) && (Sig.NextValue.test()));
    } /* fi */
else
   { assert(Edge == _specc::EDGE_FALLING);
     return((Sig.CurrValue.test()) && (!Sig.NextValue.test()));
    } /* esle */

} /* end of _signal_ValidEdge */
#endif /* HAVE_LLONG */


// case (d): specialization for pointer types


template <class T>		/* (specialization for pointer types) */
inline bool _signal_ValidEdge(	/* helper for signal edge event filter */
	_specc::EDGE		Edge,
	const _specc::signal<T*>	&Sig)
{

if (Edge == _specc::EDGE_ANY)
   { return(true);	/* every event is valid */
    } /* fi */

if (Edge == _specc::EDGE_RISING)
   { return((!Sig.CurrValue) && (Sig.NextValue));	// tests for NULL!
    } /* fi */
else
   { assert(Edge == _specc::EDGE_FALLING);
     return((Sig.CurrValue) && (!Sig.NextValue));	// tests for NULL!
    } /* esle */

} /* end of _signal_ValidEdge */


// case (f): specialization for array types


template <class T, int i>	/* (specialization for array types) */
inline bool _signal_ValidEdge(	/* helper for signal edge event filter */
	_specc::EDGE		Edge,
	const _specc::signal<T[i]>	&Sig)
{

assert(Edge == _specc::EDGE_ANY);	// cannot apply 'rising' or 'falling'
					// to an array type (type checked)!

return(true);	/* every event is valid */

} /* end of _signal_ValidEdge */


// NOTE: for the method
//
//	bool signal<T>::ResetIsActive(bool ResetActiveHi)
//
// the very same situation arises as above;
// so, we'll play the very same game...
//
// case (e): general version for other types


template <class T>		/* (general version) */
inline bool _signal_ResetActive(	/* helper for reset signal check */
	bool		/* ResetActiveHi */,
	const _specc::signal<T>	& /* Sig */)
{

assert(false);	// cannot use a default type as reset signal (type checked)!

return(false);	// should never be reached!

} /* end of _signal_ResetActive */


// case (a): specialization for built-in types


#define _SIGNAL_RESET_CODE(type)					\
  template <>			/* (specialization for 'type' type) */	\
  inline bool _signal_ResetActive( /* helper for reset signal check */	\
	bool			ResetActiveHi,				\
	const _specc::signal<type>	&Sig)					\
{									\
									\
if (ResetActiveHi)							\
   { return(Sig.NextValue);						\
    } /* fi */								\
else									\
   { return(!Sig.NextValue);						\
    } /* esle */							\
									\
} /* end of _signal_ResetActive */

_SIGNAL_RESET_CODE(bool)

_SIGNAL_RESET_CODE(char)
_SIGNAL_RESET_CODE(unsigned char)

_SIGNAL_RESET_CODE(short)
_SIGNAL_RESET_CODE(unsigned short)

_SIGNAL_RESET_CODE(int)
_SIGNAL_RESET_CODE(unsigned int)

_SIGNAL_RESET_CODE(long)
_SIGNAL_RESET_CODE(unsigned long)

#ifdef HAVE_LLONG
_SIGNAL_RESET_CODE(long long)
_SIGNAL_RESET_CODE(unsigned long long)
#endif /* HAVE_LLONG */

_SIGNAL_RESET_CODE(float)
_SIGNAL_RESET_CODE(double)
_SIGNAL_RESET_CODE(long double)

#undef _SIGNAL_RESET_CODE


// case (b): specialization for bit vectors


template <_bit::len_t l,bool s>	/* (specialization for bit vectors) */
inline bool _signal_ResetActive( /* helper for reset signal check */
	bool			ResetActiveHi,
	const _specc::signal<bit<l,s> >	&Sig)
{

if (ResetActiveHi)
   { return(Sig.NextValue.test());
    } /* fi */
else
   { return(!Sig.NextValue.test());
    } /* esle */

} /* end of _signal_ResetActive */


// case (c): specialization for longlong type


#ifndef HAVE_LLONG
template <bool s>		/* (specialization for longlong class) */
inline bool _signal_ResetActive( /* helper for reset signal check */
	bool				ResetActiveHi,
	const _specc::signal<_longlong<s> >	&Sig)
{

if (ResetActiveHi)
   { return(Sig.NextValue.test());
    } /* fi */
else
   { return(!Sig.NextValue.test());
    } /* esle */

} /* end of _signal_ResetActive */
#endif /* HAVE_LLONG */


// case (d): specialization for pointer types


template <class T>		/* (specialization for pointer types) */
inline bool _signal_ResetActive( /* helper for reset signal check */
	bool			ResetActiveHi,
	const _specc::signal<T*>	&Sig)
{

if (ResetActiveHi)
   { return(Sig.NextValue);	// test for NULL!
    } /* fi */
else
   { return(!Sig.NextValue);	// test for NULL!
    } /* esle */

} /* end of _signal_ResetActive */


// case (f): specialization for array types


template <class T, int i>	/* (specialization for array types) */
inline bool _signal_ResetActive( /* helper for reset signal check */
	bool			/* ResetActiveHi */,
	const _specc::signal<T[i]>	& /* Sig */)
{

assert(false);	// cannot use an array type as reset signal (type checked)!

return(false);	// should never be reached!

} /* end of _signal_ResetActive */


/*** class implementations **********************************************/


	/*******************************************/
	/*** buffered template (general version) ***/
	/*******************************************/

//PC 05/16/04
template<class T> inline
_specc::buffered<T>::buffered(			/* constructor #1 */
	_specc::event_ptr *UpdateEvents)	/* list of updating events */
	: _specc::buffered_base(UpdateEvents)
{

/* nothing else to do */

} /* end of buffered<T>::buffered #1 */


template<class T> inline
_specc::buffered<T>::buffered(			/* constructor #2 */
	_specc::event_ptr *UpdateEvents,	/* list of updating events */
	_specc::event	*ResetSignal,		/* async. reset specifier */
	bool		ResetActiveHi)
	: _specc::buffered_base(UpdateEvents, ResetSignal, ResetActiveHi)
{

/* nothing else to do */

} /* end of buffered<T>::buffered #2 */


template<class T> inline
_specc::buffered<T>::buffered(			/* constructor #3 */
	_specc::event_ptr *UpdateEvents,	/* list of updating events */
	const T		Init)			/* initializer */
	: _specc::buffered_base(UpdateEvents)
{

InitValue = Init;
CurrValue = Init;
NextValue = Init;

} /* end of buffered<T>::buffered #3 */


template<class T> inline
_specc::buffered<T>::buffered(			/* constructor #4 */
	_specc::event_ptr *UpdateEvents,	/* list of updating events */
	_specc::event	*ResetSignal,		/* async. reset specifier */
	bool		ResetActiveHi,
	const T		Init)			/* initializer */
	: _specc::buffered_base(UpdateEvents, ResetSignal, ResetActiveHi)
{

InitValue = Init;
CurrValue = Init;
NextValue = Init;

} /* end of buffered<T>::buffered #4 */


template<class T> inline
_specc::buffered<T>::buffered(			/* constructor #5 (for signals) */
	_specc::event	*SignalEvent)		/* signal updating event */
	: buffered_base(SignalEvent)
{

/* nothing else to do */

} /* end of buffered<T>::buffered #5 */


template<class T> inline
_specc::buffered<T>::buffered(			/* constructor #6 (for signals) */
	_specc::event	*SignalEvent,		/* signal updating event */
	const T		Init)			/* initializer */
	: _specc::buffered_base(SignalEvent)
{

InitValue = Init;
CurrValue = Init;
NextValue = Init;

} /* end of buffered<T>::buffered #6 */


template<class T> inline
_specc::buffered<T>::buffered(			/* copy constructor (#7) */
	const _specc::buffered<T> &Orig)
	: _specc::buffered_base(NULL, NULL, false)
{

InitValue = Orig.InitValue;
CurrValue = Orig.CurrValue;
NextValue = Orig.NextValue;

} /* end of buffered<T>::buffered #7 */


template<class T> inline
_specc::buffered<T>::~buffered(void)		/* destructor */
{

/* nothing to do */

} /* end of buffered<T>::~buffered */


template<class T> inline
_specc::buffered<T>::operator T() const	/* conversion operator (read access) */
{

return(CurrValue);

} /* end of buffered<T>::operator T */


template<class T> inline
_specc::buffered<T> &_specc::buffered<T>::operator=(/* assignment operator #1 */
	T	Var)
{

NextValue = Var;

return *this;

} /* end of buffered<T>::operator= */


template<class T> inline
_specc::buffered<T> &_specc::buffered<T>::operator=(
                                                /* assignment operator #1b */
	const _specc::buffered<T> &BufferedVar)	
                                             /* (needed, otherwise compiler  */
{					/* will generate 'bad' default) */

NextValue = BufferedVar.CurrValue;

return *this;

} /* end of buffered<T>::operator= */


template<class T> inline
_specc::buffered<T> &_specc::buffered<T>::operator+=(
                                               /* assignment operator #2 */
	T	Var)
{

NextValue = CurrValue + Var;

return *this;

} /* end of buffered<T>::operator+= */


template<class T> inline
_specc::buffered<T> &_specc::buffered<T>::operator-=(	
                                               /* assignment operator #3 */
	T	Var)
{

NextValue = CurrValue - Var;

return *this;

} /* end of buffered<T>::operator-= */


template<class T> inline
_specc::buffered<T> &_specc::buffered<T>::operator*=(	
                                             /* assignment operator #4 */
	T	Var)
{

NextValue = CurrValue * Var;

return *this;

} /* end of buffered<T>::operator*= */


template<class T> inline
_specc::buffered<T> &_specc::buffered<T>::operator/=(	
                                           /* assignment operator #5 */
	T	Var)
{

NextValue = CurrValue / Var;

return *this;

} /* end of buffered<T>::operator/= */


template<class T> inline
_specc::buffered<T> &_specc::buffered<T>::operator%=(
                                           /* assignment operator #6 */
	T	Var)
{

NextValue = CurrValue % Var;

return *this;

} /* end of buffered<T>::operator%= */


template<class T> inline
_specc::buffered<T> &_specc::buffered<T>::operator^=(	
                                            /* assignment operator #7 */
	T	Var)
{

NextValue = CurrValue ^ Var;

return *this;

} /* end of buffered<T>::operator^= */


template<class T> inline
_specc::buffered<T> &_specc::buffered<T>::operator&=(	
                                          /* assignment operator #8 */
	T	Var)
{

NextValue = CurrValue & Var;

return *this;

} /* end of buffered<T>::operator&= */


template<class T> inline
_specc::buffered<T> &_specc::buffered<T>::operator|=(	
                                        /* assignment operator #9 */
	T	Var)
{

NextValue = CurrValue | Var;

return *this;

} /* end of buffered<T>::operator|= */


template<class T> inline
_specc::buffered<T> &_specc::buffered<T>::operator<<=(	
                                      /* assignment operator #10 */
	unsigned int	Var)
{

NextValue = CurrValue << Var;

return *this;

} /* end of buffered<T>::operator<<= */


template<class T> inline
_specc::buffered<T> &_specc::buffered<T>::operator>>=(	
                                          /* assignment operator #11 */
	unsigned int	Var)
{

NextValue = CurrValue >> Var;

return *this;

} /* end of buffered<T>::operator>>= */


template<class T> inline
_specc::buffered<T> &_specc::buffered<T>::operator++()	
                                          /* increment operator #1 (pre-) */
{

NextValue = CurrValue + 1;

return *this;

} /* end of buffered<T>::operator++ */


template<class T> inline
_specc::buffered<T> _specc::buffered<T>::operator++(int) 
                                         /* increment operator #2 (post-) */
{
_specc::buffered<T>	Tmp(*this);

NextValue = CurrValue + 1;

return Tmp;

} /* end of buffered<T>::operator++ */


template<class T> inline
_specc::buffered<T> &_specc::buffered<T>::operator--()	
                                         /* decrement operator #1 (pre-) */
{

NextValue = CurrValue - 1;

return *this;

} /* end of buffered<T>::operator-- */


template<class T> inline
_specc::buffered<T> _specc::buffered<T>::operator--(int) 
                                        /* decrement operator #2 (post-) */
{
_specc::buffered<T>	Tmp(*this);

NextValue = CurrValue - 1;

return Tmp;

} /* end of buffered<T>::operator-- */


template<class T>
void _specc::buffered<T>::update(/* make the next value the current value */
	void)
{

CurrValue = NextValue;

} /* end of buffered<T>::update */


template<class T>
void _specc::buffered<T>::reset(/* make the next value the initial value */
	void)
{

CurrValue = InitValue;
NextValue = InitValue;

} /* end of buffered<T>::reset */


template<class T> inline
bool _specc::buffered<T>::test(void) const	/* read access to bool */
{

return(CurrValue.test());

} /* end of buffered<T>::test */


template<class T> inline
int _specc::buffered<T>::toInt(void) const	/* read access to int */
{

return(CurrValue.toInt());

} /* end of buffered<T>::toInt */


template<class T> inline
unsigned int _specc::buffered<T>::toUInt(void) const/* read access to u. int */
{

return(CurrValue.toUInt());

} /* end of buffered<T>::toUInt */


template<class T> inline
long _specc::buffered<T>::toLong(void) const	/* read access to long */
{

return(CurrValue.toLong());

} /* end of buffered<T>::toLong */


template<class T> inline
unsigned long _specc::buffered<T>::toULong(void) const	
                                              /* read access to u. long */
{

return(CurrValue.toULong());

} /* end of buffered<T>::toULong */


template<class T> inline
double _specc::buffered<T>::toDouble(void) const /* read access to double */
{

return(CurrValue.toDouble());

} /* end of buffered<T>::toDouble */


template<class T> inline
long double _specc::buffered<T>::toLDouble(void) const	
                                            /* read access to long double */
{

return(CurrValue.toLDouble());

} /* end of buffered<T>::toLDouble */


template<class T> template<class MT> inline
_specc::buf_elem<MT> _specc::buffered<T>::member(/* member access "operator" */
	int		CharOffset)
{

return _specc::buf_elem<MT>(	(void*)(((char*)&CurrValue) + CharOffset),
				(void*)(((char*)&NextValue) + CharOffset));

} /* end of buffered<T>::member */


template<class T> template<class MT> inline
const _specc::buf_elem<MT> _specc::buffered<T>::member(	
                                            /* const member access "op." */
	int		CharOffset) const
{

return _specc::buf_elem<MT>(	(void*)(((char*)&CurrValue) + CharOffset),
				(void*)(((char*)&NextValue) + CharOffset));

} /* end of buffered<T>::member */


	/*********************************************************/
	/*** buffered template (specialization for bitvectors) ***/
	/*********************************************************/


template<_bit::len_t l, bool s> inline
_specc::buffered<bit<l,s> >::buffered(			/* constructor #1 */
	_specc::event_ptr *UpdateEvents)	/* list of updating events */
	: _specc::buffered_base(UpdateEvents)
{

/* nothing else to do */

} /* end of buffered<bit<l,s> >::buffered #1 */


template<_bit::len_t l, bool s> inline
_specc::buffered<bit<l,s> >::buffered(			/* constructor #2 */
	_specc::event_ptr *UpdateEvents,	/* list of updating events */
	_specc::event	*ResetSignal,		/* async. reset specifier */
	bool		ResetActiveHi)
	: _specc::buffered_base(UpdateEvents, ResetSignal, ResetActiveHi)
{

/* nothing else to do */

} /* end of buffered<bit<l,s> >::buffered #2 */


template<_bit::len_t l, bool s> inline
_specc::buffered<bit<l,s> >::buffered(			/* constructor #3 */
	_specc::event_ptr *UpdateEvents,	/* list of updating events */
	const bit<l,s>		Init)		/* initializer */
	: _specc::buffered_base(UpdateEvents)
{

InitValue = Init;
CurrValue = Init;
NextValue = Init;

} /* end of buffered<bit<l,s> >::buffered #3 */


template<_bit::len_t l, bool s> inline
_specc::buffered<bit<l,s> >::buffered(			/* constructor #4 */
	_specc::event_ptr *UpdateEvents,	/* list of updating events */
	_specc::event	*ResetSignal,		/* async. reset specifier */
	bool		ResetActiveHi,
	const bit<l,s>	Init)			/* initializer */
	: _specc::buffered_base(UpdateEvents, ResetSignal, ResetActiveHi)
{

InitValue = Init;
CurrValue = Init;
NextValue = Init;

} /* end of buffered<bit<l,s> >::buffered #4 */


template<_bit::len_t l, bool s> inline
_specc::buffered<bit<l,s> >::buffered(		/* constructor #5 (signals) */
	_specc::event	*SignalEvent)		/* signal updating event */
	: buffered_base(SignalEvent)
{

/* nothing else to do */

} /* end of buffered<bit<l,s> >::buffered #5 */


template<_bit::len_t l, bool s> inline
_specc::buffered<bit<l,s> >::buffered(		/* constructor #6 (signals) */
	_specc::event	*SignalEvent,		/* signal updating event */
	const bit<l,s>	Init)			/* initializer */
	: _specc::buffered_base(SignalEvent)
{

InitValue = Init;
CurrValue = Init;
NextValue = Init;

} /* end of buffered<bit<l,s> >::buffered #6 */


template<_bit::len_t l, bool s> inline
_specc::buffered<bit<l,s> >::buffered(		/* copy constructor (#7) */
	const _specc::buffered<bit<l,s> > &Orig)
	: _specc::buffered_base(NULL, NULL, false)
{

InitValue = Orig.InitValue;
CurrValue = Orig.CurrValue;
NextValue = Orig.NextValue;

} /* end of buffered<bit<l,s> >::buffered #7 */


template<_bit::len_t l, bool s> inline
_specc::buffered<bit<l,s> >::~buffered(void)		/* destructor */
{

/* nothing to do */

} /* end of buffered<bit<l,s> >::~buffered */


template<_bit::len_t l, bool s> inline
_specc::buffered<bit<l,s> >::operator bit<l,s>() const	
                                           /* conversion operator (read) */
{

Get();
return(CurrValue);

} /* end of buffered<bit<l,s> >::operator bit<l,s> */


template<_bit::len_t l, bool s> inline
_specc::buffered<bit<l,s> > &_specc::buffered<bit<l,s> >::operator=(
                                        	/* assignment op. #1 */
	bit<l,s>	Var)
{

NextValue = Var;
Put();

return *this;

} /* end of buffered<bit<l,s> >::operator= */


template<_bit::len_t l, bool s> inline
_specc::buffered<bit<l,s> > &_specc::buffered<bit<l,s> >::operator=(	
                                              /* assignment op. #1b */
	const _specc::buffered<bit<l,s> > &BufferedVar)	
                                        /* (needed, otherwise comp. */
{					/* will generate 'bad' default) */

BufferedVar.Get();
NextValue = BufferedVar.CurrValue;
Put();

return *this;

} /* end of buffered<bit<l,s> >::operator= */


template<_bit::len_t l, bool s> inline
_specc::buffered<bit<l,s> > &_specc::buffered<bit<l,s> >::operator+=(	
                                                 /* assignment op. #2 */
	bit<l,s>	Var)
{

Get();
NextValue = CurrValue + Var;
Put();

return *this;

} /* end of buffered<bit<l,s> >::operator+= */


template<_bit::len_t l, bool s> inline
_specc::buffered<bit<l,s> > &_specc::buffered<bit<l,s> >::operator-=(	
                                                 /* assignment op. #3 */
	bit<l,s>	Var)
{

Get();
NextValue = CurrValue - Var;
Put();

return *this;

} /* end of buffered<bit<l,s> >::operator-= */


template<_bit::len_t l, bool s> inline
_specc::buffered<bit<l,s> > &_specc::buffered<bit<l,s> >::operator*=(	
                                             /* assignment op. #4 */
	bit<l,s>	Var)
{

Get();
NextValue = CurrValue * Var;
Put();

return *this;

} /* end of buffered<bit<l,s> >::operator*= */


template<_bit::len_t l, bool s> inline
_specc::buffered<bit<l,s> > &_specc::buffered<bit<l,s> >::operator/=(	
                                            /* assignment op. #5 */
	bit<l,s>	Var)
{

Get();
NextValue = CurrValue / Var;
Put();

return *this;

} /* end of buffered<bit<l,s> >::operator/= */


template<_bit::len_t l, bool s> inline
_specc::buffered<bit<l,s> > &_specc::buffered<bit<l,s> >::operator%=(	
                                                /* assignment op. #6 */
	bit<l,s>	Var)
{

Get();
NextValue = CurrValue % Var;
Put();

return *this;

} /* end of buffered<bit<l,s> >::operator%= */


template<_bit::len_t l, bool s> inline
_specc::buffered<bit<l,s> > &_specc::buffered<bit<l,s> >::operator^=(	
                                               /* assignment op. #7 */
	bit<l,s>	Var)
{

Get();
NextValue = CurrValue ^ Var;
Put();

return *this;

} /* end of buffered<bit<l,s> >::operator^= */


template<_bit::len_t l, bool s> inline
_specc::buffered<bit<l,s> > &_specc::buffered<bit<l,s> >::operator&=(	
                                              /* assignment op. #8 */
	bit<l,s>	Var)
{

Get();
NextValue = CurrValue & Var;
Put();

return *this;

} /* end of buffered<bit<l,s> >::operator&= */


template<_bit::len_t l, bool s> inline
_specc::buffered<bit<l,s> > &_specc::buffered<bit<l,s> >::operator|=(	
                                                  /* assignment op. #9 */
	bit<l,s>	Var)
{

Get();
NextValue = CurrValue | Var;
Put();

return *this;

} /* end of buffered<bit<l,s> >::operator|= */


template<_bit::len_t l, bool s> inline
_specc::buffered<bit<l,s> > &_specc::buffered<bit<l,s> >::operator<<=(	
                                                  /* assignment op. #10 */
	unsigned int	Var)
{

Get();
NextValue = CurrValue << Var;
Put();

return *this;

} /* end of buffered<bit<l,s> >::operator<<= */


template<_bit::len_t l, bool s> inline
_specc::buffered<bit<l,s> > &_specc::buffered<bit<l,s> >::operator>>=(	
                                                 /* assignment op. #11 */
	unsigned int	Var)
{

Get();
NextValue = CurrValue >> Var;
Put();

return *this;

} /* end of buffered<bit<l,s> >::operator>>= */


template<_bit::len_t l, bool s> inline
_specc::buffered<bit<l,s> > &_specc::buffered<bit<l,s> >::operator++()	
                                                /* incr. op. #1 (pre-)*/
{

Get();
NextValue = CurrValue + 1;
Put();

return *this;

} /* end of buffered<bit<l,s> >::operator++ */


template<_bit::len_t l, bool s> inline
_specc::buffered<bit<l,s> > _specc::buffered<bit<l,s> >::operator++(int) 
                                               /* incr.op.#2 (post-)*/
{
_specc::buffered<bit<l,s> >	Tmp(*this);

Get();
NextValue = CurrValue + 1;
Put();

return Tmp;

} /* end of buffered<bit<l,s> >::operator++ */


template<_bit::len_t l, bool s> inline
_specc::buffered<bit<l,s> > &_specc::buffered<bit<l,s> >::operator--()	
                                               /* decr.op. #1 (pre-) */
{

Get();
NextValue = CurrValue - 1;
Put();

return *this;

} /* end of buffered<bit<l,s> >::operator-- */


template<_bit::len_t l, bool s> inline
_specc::buffered<bit<l,s> > _specc::buffered<bit<l,s> >::operator--(int) 
                                                    /* decr.op.#2(post-)*/
{
_specc::buffered<bit<l,s> >	Tmp(*this);

Get();
NextValue = CurrValue - 1;
Put();

return Tmp;

} /* end of buffered<bit<l,s> >::operator-- */


template<_bit::len_t l, bool s>
void _specc::buffered<bit<l,s> >::update(/* make next value the current value */
	void)
{

// note: we don't need any Get() or Put() here, because this method
//       will never be called for any _specc::bufbus;
//       (_specc::bufbus objects don't have updating clocks)

CurrValue = NextValue;

} /* end of buffered<bit<l,s> >::update */


template<_bit::len_t l, bool s>
void _specc::buffered<bit<l,s> >::reset(/* make next value the initial value */
	void)
{

// note: we don't need any Get() or Put() here, because this method
//       will never be called for any _specc::bufbus;
//       (_specc::bufbus objects don't have any resets)

CurrValue = InitValue;
NextValue = InitValue;

} /* end of buffered<bit<l,s> >::reset */


template<_bit::len_t l, bool s> inline
bool _specc::buffered<bit<l,s> >::test(void) const /* read access to bool */
{

Get();
return(CurrValue.test());

} /* end of buffered<bit<l,s> >::test */


template<_bit::len_t l, bool s> inline
int _specc::buffered<bit<l,s> >::toInt(void) const /* read access to int */
{

Get();
return(CurrValue.toInt());

} /* end of buffered<bit<l,s> >::toInt */


template<_bit::len_t l, bool s> inline
unsigned int _specc::buffered<bit<l,s> >::toUInt(void) const /* read to u. int */
{

Get();
return(CurrValue.toUInt());

} /* end of buffered<bit<l,s> >::toUInt */


template<_bit::len_t l, bool s> inline
long _specc::buffered<bit<l,s> >::toLong(void) const /* read access to long */
{

Get();
return(CurrValue.toLong());

} /* end of buffered<bit<l,s> >::toLong */


template<_bit::len_t l, bool s> inline
unsigned long _specc::buffered<bit<l,s> >::toULong(void) const	
                                                  /* read to u. long */
{

Get();
return(CurrValue.toULong());

} /* end of buffered<bit<l,s> >::toULong */


template<_bit::len_t l, bool s> inline
_LONGLONG _specc::buffered<bit<l,s> >::toLLong(void) const 
                                                /* read to long long */
{

Get();
return(CurrValue.toLLong());

} /* end of buffered<bit<l,s> >::toLLong */


template<_bit::len_t l, bool s> inline
_ULONGLONG _specc::buffered<bit<l,s> >::toULLong(void) const
           	                              /* read to u.longlong */
{

Get();
return(CurrValue.toULLong());

} /* end of buffered<bit<l,s> >::toULLong */


template<_bit::len_t l, bool s> inline
double _specc::buffered<bit<l,s> >::toDouble(void) const /* read to double */
{

Get();
return(CurrValue.toDouble());

} /* end of buffered<bit<l,s> >::toDouble */


template<_bit::len_t l, bool s> inline
long double _specc::buffered<bit<l,s> >::toLDouble(void) const	
                                                   /* read to long double*/
{

Get();
return(CurrValue.toLDouble());

} /* end of buffered<bit<l,s> >::toLDouble */


template<_bit::len_t l, bool s> inline
_specc::bufslice _specc::buffered<bit<l,s> >::buf_slice( /* single bit */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	i)
{

return _specc::bufslice(CurrValue, NextValue,
			_BITMAP(ll, r, i), _BITMAP(ll, r, i),
			true);	/* single bit is always unsigned */

} /* end of buffered<bit<l,s> >::buf_slice */


template<_bit::len_t l, bool s> inline
_specc::bufslice _specc::buffered<bit<l,s> >::buf_slice(	/* slice */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr)
{

return _specc::bufslice(CurrValue, NextValue,
			_BITMAP(ll, r, sl), _BITMAP(ll, r, sr),
			s);

} /* end of buffered<bit<l,s> >::buf_slice */


template<_bit::len_t l, bool s> inline
const _specc::bufslice _specc::buffered<bit<l,s> >::buf_slice(	/* const bit */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	i) const
{

return _specc::bufslice(CurrValue, NextValue,
			_BITMAP(ll, r, i), _BITMAP(ll, r, i),
			true);	/* single bit is always unsigned */

} /* end of buffered<bit<l,s> >::buf_slice */


template<_bit::len_t l, bool s> inline
const _specc::bufslice _specc::buffered<bit<l,s> >::buf_slice(	/* const slice*/
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr) const
{

return _specc::bufslice(CurrValue, NextValue,
			_BITMAP(ll, r, sl), _BITMAP(ll, r, sr),
			s);

} /* end of buffered<bit<l,s> >::buf_slice */


template<_bit::len_t l, bool s> inline
_specc::bufbus_elem* _specc::buffered<bit<l,s> >::bufbus_slice(	
                                                      /* bus slice #1 */
	_bit::len_t	sl,
	_bit::len_t	sr)
{

return(new _specc::bufbus_elem(
			CurrValue,
			NextValue,
			sl, sr));

} /* end of buffered<bit<l,s> >::bufbus_slice #1 */


template<_bit::len_t l, bool s> inline
_specc::bufbus_elem* _specc::buffered<bit<l,s> >::bufbus_slice(	
                                                    /* bus slice #2 */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr)
{

return(bufbus_slice(_BITMAP(ll,r,sl), _BITMAP(ll,r,sr)));

} /* end of buffered<bit<l,s> >::bufbus_slice #2 */


template<_bit::len_t l, bool s> inline
void _specc::buffered<bit<l,s> >::Put(void) const /* store value for bufbus */
{						/* (dummy to be overloaded) */

/* nothing to do for plain bitvectors */

} /* end of buffered<bit<l,s> >::Put */


template<_bit::len_t l, bool s> inline
void _specc::buffered<bit<l,s> >::Get(void) const /* load value for bufbus */
{						/* (dummy to be overloaded) */

/* nothing to do for plain bitvectors */

} /* end of buffered<bit<l,s> >::Get */


	/**********************************************************/
	/*** buffered template (specialization for array types) ***/
	/**********************************************************/


template<int i, class T> inline
_specc::buffered<T[i]>::buffered(			/* constructor #1 */
	_specc::event_ptr *UpdateEvents)/* list of updating events */
	: _specc::buffered_base(UpdateEvents)
{

/* nothing else to do */

} /* end of buffered<T[i]>::buffered #1 */


template<int i, class T> inline
_specc::buffered<T[i]>::buffered(			/* constructor #2 */
	_specc::event_ptr *UpdateEvents,/* list of updating events */
	_specc::event	*ResetSignal,	/* async. reset specifier */
	bool		ResetActiveHi)
	: _specc::buffered_base(UpdateEvents, ResetSignal, ResetActiveHi)
{

/* nothing else to do */

} /* end of buffered<T[i]>::buffered #2 */


template<int i, class T> inline
_specc::buffered<T[i]>::buffered(			/* constructor #3 */
	_specc::event_ptr *UpdateEvents,/* list of updating events */
	T		Init[i])	/* initializer */
	: _specc::buffered_base(UpdateEvents)
{

memcpy(InitValue, Init, sizeof(T[i]));
memcpy(CurrValue, Init, sizeof(T[i]));
memcpy(NextValue, Init, sizeof(T[i]));

} /* end of buffered<T[i]>::buffered #3 */


template<int i, class T> inline
_specc::buffered<T[i]>::buffered(			/* constructor #4 */
	_specc::event_ptr *UpdateEvents,/* list of updating events */
	_specc::event	*ResetSignal,	/* async. reset specifier */
	bool		ResetActiveHi,
	T		Init[i])	/* initializer */
	: _specc::buffered_base(UpdateEvents, ResetSignal, ResetActiveHi)
{

memcpy(InitValue, Init, sizeof(T[i]));
memcpy(CurrValue, Init, sizeof(T[i]));
memcpy(NextValue, Init, sizeof(T[i]));

} /* end of buffered<T[i]>::buffered #4 */


template<int i, class T> inline
_specc::buffered<T[i]>::buffered(		/* constructor #5 (signals) */
	_specc::event	*SignalEvent)	/* signal updating event */
	: buffered_base(SignalEvent)
{

/* nothing else to do */

} /* end of buffered<T[i]>::buffered #5 */


template<int i, class T> inline
_specc::buffered<T[i]>::buffered(		/* constructor #6 (signals) */
	_specc::event	*SignalEvent,	/* signal updating event */
	T		Init[i])	/* initializer */
	: _specc::buffered_base(SignalEvent)
{

memcpy(InitValue, Init, sizeof(T[i]));
memcpy(CurrValue, Init, sizeof(T[i]));
memcpy(NextValue, Init, sizeof(T[i]));

} /* end of buffered<T[i]>::buffered #6 */


template<int i, class T> inline
_specc::buffered<T[i]>::buffered(		/* copy constructor (#7) */
	const _specc::buffered<T[i]> &Orig)
	: _specc::buffered_base(NULL, NULL, false)
{

memcpy(InitValue, Orig.InitValue, sizeof(T[i]));
memcpy(CurrValue, Orig.CurrValue, sizeof(T[i]));
memcpy(NextValue, Orig.NextValue, sizeof(T[i]));

} /* end of buffered<T[i]>::buffered #7 */


template<int i, class T> inline
_specc::buffered<T[i]>::~buffered(void)			/* destructor */
{

/* nothing to do */

} /* end of buffered<T[i]>::~buffered */


template<int i, class T> inline
_specc::buffered<T[i]>::operator const T*() const /* conv. op. (read access) */
{

return(&CurrValue[0]);

} /* end of buffered<T[i]>::operator const T* */


template<int i, class T> inline
_specc::buffered<T[i]> &_specc::buffered<T[i]>::operator=(
                                               /* assignment operator #1 */
	T	Var[i])
{

memcpy(NextValue, Var, sizeof(T[i]));

return *this;

} /* end of buffered<T[i]>::operator= #1 */


template<int i, class T> inline
_specc::buffered<T[i]> &_specc::buffered<T[i]>::operator=(	
                                         /* assignment operator #1b */
	const _specc::buffered<T[i]> &BufferedVar) 
                                         /* (needed, otherwise compiler  */
{					 /* will generate 'bad' default) */

memcpy(NextValue, BufferedVar.CurrValue, sizeof(T[i]));

return *this;

} /* end of buffered<T[i]>::operator= #1b */


template<int i, class T> inline
_specc::buffered<T[i]> &_specc::buffered<T[i]>::operator=(	
                                         /* assignment operator #1c */
	const _specc::buf_elem<T[i]> &BufElemVar)
{

memcpy(NextValue, BufElemVar.CurrPtr, sizeof(T[i]));

return *this;

} /* end of buffered<T[i]>::operator= #1c */


template<int i, class T> inline
_specc::buf_elem<T> _specc::buffered<T[i]>::operator[](	
                                        /* array access operator */
	int		Index)
{

return _specc::buf_elem<T>((void*)(&CurrValue[Index]),
				(void*)(&NextValue[Index]));

} /* end of buffered<T[i]>::operator[] */


template<int i, class T>
void _specc::buffered<T[i]>::update( /* make the next value the current value */
	void)
{

memcpy(CurrValue, NextValue, sizeof(T[i]));

} /* end of buffered<T[i]>::update */


template<int i, class T>
void _specc::buffered<T[i]>::reset( /* make the next value the initial value */
	void)
{

memcpy(CurrValue, InitValue, sizeof(T[i]));
memcpy(NextValue, InitValue, sizeof(T[i]));

} /* end of buffered<T[i]>::reset */


	/*******************************************/
	/*** buf_elem template (general version) ***/
	/*******************************************/


template<class T> inline
_specc::buf_elem<T>::buf_elem(		/* constructor #1 */
	void	*CurrPtr,
	void	*NextPtr)
{

_specc::buf_elem<T>::CurrPtr = (T*) CurrPtr;
_specc::buf_elem<T>::NextPtr = (T*) NextPtr;

} /* end of _specc::buf_elem<T>::buf_elem #1 */


template<class T> inline
_specc::buf_elem<T>::~buf_elem(void)	/* destructor */
{

/* nothing to do */

} /* end of _specc::buf_elem<T>::~buf_elem */


template<class T> inline
_specc::buf_elem<T>::operator T() const	/* conversion operator (read access) */
{

return(*CurrPtr);

} /* end of _specc::buf_elem<T>::operator T */


template<class T> inline
_specc::buf_elem<T> &_specc::buf_elem<T>::operator=(	/* assignment op. #1 */
	T	Var)
{

*NextPtr = Var;

return *this;

} /* end of _specc::buf_elem<T>::operator= */


template<class T> inline
_specc::buf_elem<T> &_specc::buf_elem<T>::operator=(	/* assignment op. #1b */
	const buf_elem<T> &BufElem)	/* (needed, otherwise compiler  */
{					/* will generate 'bad' default) */

*NextPtr = *BufElem.CurrPtr;

return *this;

} /* end of _specc::buf_elem<T>::operator= */


template<class T> inline
_specc::buf_elem<T> &_specc::buf_elem<T>::operator+=(	/* assignment op. #2 */
	T	Var)
{

*NextPtr = *CurrPtr + Var;

return *this;

} /* end of _specc::buf_elem<T>::operator+= */


template<class T> inline
_specc::buf_elem<T> &_specc::buf_elem<T>::operator-=(	/* assignment op. #3 */
	T	Var)
{

*NextPtr = *CurrPtr - Var;

return *this;

} /* end of _specc::buf_elem<T>::operator-= */


template<class T> inline
_specc::buf_elem<T> &_specc::buf_elem<T>::operator*=(	/* assignment op. #4 */
	T	Var)
{

*NextPtr = *CurrPtr * Var;

return *this;

} /* end of _specc::buf_elem<T>::operator*= */


template<class T> inline
_specc::buf_elem<T> &_specc::buf_elem<T>::operator/=(	/* assignment op. #5 */
	T	Var)
{

*NextPtr = *CurrPtr / Var;

return *this;

} /* end of _specc::buf_elem<T>::operator/= */


template<class T> inline
_specc::buf_elem<T> &_specc::buf_elem<T>::operator%=(	/* assignment op. #6 */
	T	Var)
{

*NextPtr = *CurrPtr % Var;

return *this;

} /* end of _specc::buf_elem<T>::operator%= */


template<class T> inline
_specc::buf_elem<T> &_specc::buf_elem<T>::operator^=(	/* assignment op. #7 */
	T	Var)
{

*NextPtr = *CurrPtr ^ Var;

return *this;

} /* end of _specc::buf_elem<T>::operator^= */


template<class T> inline
_specc::buf_elem<T> &_specc::buf_elem<T>::operator&=(	/* assignment op. #8 */
	T	Var)
{

*NextPtr = *CurrPtr & Var;

return *this;

} /* end of _specc::buf_elem<T>::operator&= */


template<class T> inline
_specc::buf_elem<T> &_specc::buf_elem<T>::operator|=(	/* assignment op. #9 */
	T	Var)
{

*NextPtr = *CurrPtr | Var;

return *this;

} /* end of _specc::buf_elem<T>::operator|= */


template<class T> inline
_specc::buf_elem<T> &_specc::buf_elem<T>::operator<<=(	/* assignment op. #10 */
	unsigned int	Var)
{

*NextPtr = *CurrPtr << Var;

return *this;

} /* end of _specc::buf_elem<T>::operator<<= */


template<class T> inline
_specc::buf_elem<T> &_specc::buf_elem<T>::operator>>=(	/* assignment op. #11 */
	unsigned int	Var)
{

*NextPtr = *CurrPtr >> Var;

return *this;

} /* end of _specc::buf_elem<T>::operator>>= */


template<class T> inline
_specc::buf_elem<T> &_specc::buf_elem<T>::operator++()	/* incr.op. #1 (pre-) */
{

*NextPtr = *CurrPtr + 1;

return *this;

} /* end of _specc::buf_elem<T>::operator++ */


template<class T> inline
_specc::buf_elem<T> _specc::buf_elem<T>::operator++(int) /* incr.op. #2(post-)*/
{
buf_elem<T>	Tmp(*this);

*NextPtr = *CurrPtr + 1;

return Tmp;

} /* end of _specc::buf_elem<T>::operator++ */


template<class T> inline
_specc::buf_elem<T> &_specc::buf_elem<T>::operator--()	/* decr.op. #1 (pre-) */
{

*NextPtr = *CurrPtr - 1;

return *this;

} /* end of _specc::buf_elem<T>::operator-- */


template<class T> inline
_specc::buf_elem<T> _specc::buf_elem<T>::operator--(int) /* decr.op. #2(post-)*/
{
buf_elem<T>	Tmp(*this);

*NextPtr = *CurrPtr - 1;

return Tmp;

} /* end of _specc::buf_elem<T>::operator-- */


template<class T> inline
bool _specc::buf_elem<T>::test(void) const		/* read bool */
{

return(CurrPtr->test());

} /* end of _specc::buf_elem<T>::test */


template<class T> inline
int _specc::buf_elem<T>::toInt(void) const		/* read int */
{

return(CurrPtr->toInt());

} /* end of _specc::buf_elem<T>::toInt */


template<class T> inline
unsigned int _specc::buf_elem<T>::toUInt(void) const	/* read u. int */
{

return(CurrPtr->toUInt());

} /* end of _specc::buf_elem<T>::toUInt */


template<class T> inline
long _specc::buf_elem<T>::toLong(void) const		/* read long */
{

return(CurrPtr->toLong());

} /* end of _specc::buf_elem<T>::toLong */


template<class T> inline
unsigned long _specc::buf_elem<T>::toULong(void) const	/* read u. long */
{

return(CurrPtr->toULong());

} /* end of _specc::buf_elem<T>::toULong */


template<class T> inline
_LONGLONG _specc::buf_elem<T>::toLLong(void) const	/* read long long */
{

return(CurrPtr->toLLong());

} /* end of _specc::buf_elem<T>::toLLong */


template<class T> inline
_ULONGLONG _specc::buf_elem<T>::toULLong(void) const	/* read u.long long */
{

return(CurrPtr->toULLong());

} /* end of _specc::buf_elem<T>::toULLong */


template<class T> inline
double _specc::buf_elem<T>::toDouble(void) const	/* read double */
{

return(CurrPtr->toDouble());

} /* end of _specc::buf_elem<T>::toDouble */


template<class T> inline
long double _specc::buf_elem<T>::toLDouble(void) const	/* read long double */
{

return(CurrPtr->toLDouble());

} /* end of _specc::buf_elem<T>::toLDouble */


template<class T> inline
_specc::bufslice _specc::buf_elem<T>::buf_slice(/* single bit of a bitvec. */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	i)
{

return _specc::bufslice(*CurrPtr, *NextPtr,
			_BITMAP(l, r, i), _BITMAP(l, r, i),
			true);	/* single bit is always unsigned */

} /* end of _specc::buf_elem<T>::buf_slice */


template<class T> inline
_specc::bufslice _specc::buf_elem<T>::buf_slice( /* slice of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr)
{

return _specc::bufslice(*CurrPtr, *NextPtr,
			_BITMAP(l, r, sl), _BITMAP(l, r, sr),
			(_bit(*CurrPtr)).isUnsigned());

} /* end of _specc::buf_elem<T>::buf_slice */


template<class T> inline
const _specc::bufslice _specc::buf_elem<T>::buf_slice( /* const bit of vec.*/
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	i) const
{

return _specc::bufslice(*CurrPtr, *NextPtr,
			_BITMAP(l, r, i), _BITMAP(l, r, i),
			true);	/* single bit is always unsigned */

} /* end of _specc::buf_elem<T>::buf_slice */


template<class T> inline
const _specc::bufslice _specc::buf_elem<T>::buf_slice( /* const slice */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr) const
{

return _specc::bufslice(*CurrPtr, *NextPtr,
			_BITMAP(l, r, sl), _BITMAP(l, r, sr),
			(_bit(*CurrPtr)).isUnsigned());

} /* end of _specc::buf_elem<T>::buf_slice */


template<class T> template<class MT> inline
_specc::buf_elem<MT> _specc::buf_elem<T>::member( /* member access "operator" */
	int		CharOffset)
{

return _specc::buf_elem<MT>(	(void*)(((char*)CurrPtr) + CharOffset),
				(void*)(((char*)NextPtr) + CharOffset));

} /* end of _specc::buf_elem<T>::member */


template<class T> template<class MT> inline
const _specc::buf_elem<MT> _specc::buf_elem<T>::member(	/* const member "op." */
	int		CharOffset) const
{

return _specc::buf_elem<MT>(	(void*)(((char*)CurrPtr) + CharOffset),
				(void*)(((char*)NextPtr) + CharOffset));

} /* end of _specc::buf_elem<T>::member */


	/**********************************************************/
	/*** buf_elem template (specialization for array types) ***/
	/**********************************************************/


template<int i, class T> inline
_specc::buf_elem<T[i]>::buf_elem(		/* constructor #1 */
	void	*CurrPtr,
	void	*NextPtr)
{

_specc::buf_elem<T[i]>::CurrPtr = (T*) CurrPtr;
_specc::buf_elem<T[i]>::NextPtr = (T*) NextPtr;

} /* end of _specc::buf_elem<T[i]>::buf_elem #1 */


template<int i, class T> inline
_specc::buf_elem<T[i]>::~buf_elem(void)		/* destructor */
{

/* nothing to do */

} /* end of _specc::buf_elem<T[i]>::~buf_elem */


template<int i, class T> inline
_specc::buf_elem<T[i]>::operator const T*() const /* conv.op. (read access) */
{

return(CurrPtr);

} /* end of _specc::buf_elem<T[i]>::operator const T* */


template<int i, class T> inline
_specc::buf_elem<T[i]> &_specc::buf_elem<T[i]>::operator=( /* assign.op. #1 */
	T	Var[i])
{
//PC 05/02/04 
memcpy(NextPtr, Var, sizeof(T[i]));

return *this;

} /* end of _specc::buf_elem<T[i]>::operator= #1 */


template<int i, class T> inline
_specc::buf_elem<T[i]> &_specc::buf_elem<T[i]>::operator=( /* assign.op. #1b */
	const _specc::buf_elem<T[i]> &BufElem) /* (needed, otherwise compiler */
{					       /*  generates 'bad' default)   */

memcpy(NextPtr, BufElem.CurrPtr, sizeof(T[i]));

return *this;

} /* end of _specc::buf_elem<T[i]>::operator= #1b */

//PC 05/02/04 
template<int i, class T> inline
_specc::buf_elem<T[i]> &_specc::buf_elem<T[i]>::operator=( /* assign.op. #1c */
	const _specc::buffered<T[i]> &BufferedVar)
{

memcpy(NextPtr, BufferedVar.CurrValue, sizeof(T[i]));

return *this;

} /* end of _specc::buf_elem<T[i]>::operator= #1c */


template<int i, class T> inline
_specc::buf_elem<T> _specc::buf_elem<T[i]>::operator[](	/* array access op. */
	int		Index)
{

return _specc::buf_elem<T>((void*)(&CurrPtr[Index]),
				(void*)(&NextPtr[Index]));

} /* end of _specc::buf_elem<T[i]>::operator[] */


	/************************/
	/*** _specc::bufslice ***/
	/************************/


inline
_specc::bufslice::bufslice(			/* constructor #1 */
	const _bit	&CurrVec,
	const _bit	&NextVec,
	_bit::len_t	Left,
	_bit::len_t	Right,
	bool		Unsigned,
	bufbus_elem	*Slices /* = NULL */) /* only for slice of bufbus */
	:	CurrSlice(CurrVec.addr(),
				CurrVec.length(),
				CurrVec.isUnsigned(),
				Left, Right, Unsigned),
		NextSlice(NextVec.addr(),
				NextVec.length(),
				NextVec.isUnsigned(),
				Left, Right, Unsigned),
		BusSlices(Slices)
{

/* nothing else to do */

} /* end of _specc::bufslice::bufslice #1 */


inline
_specc::bufslice::bufslice(			/* copy constructor (#2) */
	const _specc::bufslice	&Orig)
	:	CurrSlice(Orig.CurrSlice),	// (dummy init, see below)
		NextSlice(Orig.NextSlice),	// (dummy init, see below)
		BusSlices(Orig.BusSlices)	// (dummy init, see below)
{

assert(false);	// this copy constructor should never be called
		// because all bitvector slices will be converted
		// to working _bit objects before use

} /* end of _specc::bufslice::bufslice #2 */


inline
_specc::bufslice::~bufslice(void)		/* destructor */
{

delete BusSlices;

} /* end of _specc::bufslice::~bufslice */


inline
_specc::bufslice::operator _bit() const		/* conversion operator (read) */
{

Get();
return(CurrSlice);

} /* end of _specc::bufslice::operator _bit */


inline
_specc::bufslice &_specc::bufslice::operator=(		/* assignment op. #1 */
	_bit	Var)
{

NextSlice = Var;
Put();

return *this;

} /* end of _specc::bufslice::operator= */


inline
_specc::bufslice &_specc::bufslice::operator=(		/* assignment op. #1b */
	const bufslice &BufSlice)	/* (needed, otherwise compiler  */
{					/* will generate 'bad' default) */

Get();
NextSlice = BufSlice.CurrSlice;
Put();

return *this;

} /* end of _specc::bufslice::operator= */


inline
_specc::bufslice &_specc::bufslice::operator+=(		/* assignment op. #2 */
	_bit	Var)
{

Get();
NextSlice = CurrSlice + Var;
Put();

return *this;

} /* end of _specc::bufslice::operator+= */


inline
_specc::bufslice &_specc::bufslice::operator-=(		/* assignment op. #3 */
	_bit	Var)
{

Get();
NextSlice = CurrSlice - Var;
Put();

return *this;

} /* end of _specc::bufslice::operator-= */


inline
_specc::bufslice &_specc::bufslice::operator*=(		/* assignment op. #4 */
	_bit	Var)
{

Get();
NextSlice = CurrSlice * Var;
Put();

return *this;

} /* end of _specc::bufslice::operator*= */


inline
_specc::bufslice &_specc::bufslice::operator/=(		/* assignment op. #5 */
	_bit	Var)
{

Get();
NextSlice = CurrSlice / Var;
Put();

return *this;

} /* end of _specc::bufslice::operator/= */


inline
_specc::bufslice &_specc::bufslice::operator%=(		/* assignment op. #6 */
	_bit	Var)
{

Get();
NextSlice = CurrSlice % Var;
Put();

return *this;

} /* end of _specc::bufslice::operator%= */


inline
_specc::bufslice &_specc::bufslice::operator^=(		/* assignment op. #7 */
	_bit	Var)
{

Get();
NextSlice = CurrSlice ^ Var;
Put();

return *this;

} /* end of _specc::bufslice::operator^= */


inline
_specc::bufslice &_specc::bufslice::operator&=(		/* assignment op. #8 */
	_bit	Var)
{

Get();
NextSlice = CurrSlice & Var;
Put();

return *this;

} /* end of _specc::bufslice::operator&= */


inline
_specc::bufslice &_specc::bufslice::operator|=(		/* assignment op. #9 */
	_bit	Var)
{

Get();
NextSlice = CurrSlice | Var;
Put();

return *this;

} /* end of _specc::bufslice::operator|= */


inline
_specc::bufslice &_specc::bufslice::operator<<=(	/* assignment op. #10 */
	unsigned int	Var)
{

Get();
NextSlice = CurrSlice << Var;
Put();

return *this;

} /* end of _specc::bufslice::operator<<= */


inline
_specc::bufslice &_specc::bufslice::operator>>=(	/* assignment op. #11 */
	unsigned int	Var)
{

Get();
NextSlice = CurrSlice >> Var;
Put();

return *this;

} /* end of _specc::bufslice::operator>>= */


inline
_specc::bufslice &_specc::bufslice::operator++()	/* incr.op. #1 (pre-) */
{

Get();
NextSlice = CurrSlice + 1;
Put();

return *this;

} /* end of _specc::bufslice::operator++ */


inline
_specc::bufslice _specc::bufslice::operator++(int)	 /* incr.op. #2(post-)*/
{
bufslice	Tmp(*this);

Get();
NextSlice = CurrSlice + 1;
Put();

return Tmp;

} /* end of _specc::bufslice::operator++ */


inline
_specc::bufslice &_specc::bufslice::operator--()	/* decr.op. #1 (pre-) */
{

Get();
NextSlice = CurrSlice - 1;
Put();

return *this;

} /* end of _specc::bufslice::operator-- */


inline
_specc::bufslice _specc::bufslice::operator--(int)	/* decr.op. #2(post-)*/
{
bufslice	Tmp(*this);

Get();
NextSlice = CurrSlice - 1;
Put();

return Tmp;

} /* end of _specc::bufslice::operator-- */


inline
bool _specc::bufslice::test(void) const			/* read bool */
{

Get();
return(CurrSlice.test());

} /* end of _specc::bufslice::test */


inline
int _specc::bufslice::toInt(void) const			/* read int */
{

Get();
return(CurrSlice.toInt());

} /* end of _specc::bufslice::toInt */


inline
unsigned int _specc::bufslice::toUInt(void) const	/* read u. int */
{

Get();
return(CurrSlice.toUInt());

} /* end of _specc::bufslice::toUInt */


inline
long _specc::bufslice::toLong(void) const		/* read long */
{

Get();
return(CurrSlice.toLong());

} /* end of _specc::bufslice::toLong */


inline
unsigned long _specc::bufslice::toULong(void) const	/* read u. long */
{

Get();
return(CurrSlice.toULong());

} /* end of _specc::bufslice::toULong */


inline
_LONGLONG _specc::bufslice::toLLong(void) const		/* read long long */
{

Get();
return(CurrSlice.toLLong());

} /* end of _specc::bufslice::toLLong */


inline
_ULONGLONG _specc::bufslice::toULLong(void) const	/* read u.long long */
{

Get();
return(CurrSlice.toULLong());

} /* end of _specc::bufslice::toULLong */


inline
double _specc::bufslice::toDouble(void) const		/* read double */
{

Get();
return(CurrSlice.toDouble());

} /* end of _specc::bufslice::toDouble */


inline
long double _specc::bufslice::toLDouble(void) const	/* read long double */
{

Get();
return(CurrSlice.toLDouble());

} /* end of _specc::bufslice::toLDouble */


inline
_specc::bufslice _specc::bufslice::buf_slice(	/* single bit of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	i)
{

return _specc::bufslice(CurrSlice, NextSlice,
			_BITMAP(l, r, i), _BITMAP(l, r, i),
			true,	/* single bit is always unsigned */
			(BusSlices ? BusSlices->slice(_BITMAP(l, r, i),
							_BITMAP(l, r, i))
					: NULL));

} /* end of _specc::bufslice::buf_slice */


inline
_specc::bufslice _specc::bufslice::buf_slice(	/* slice of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr)
{

return _specc::bufslice(CurrSlice, NextSlice,
			_BITMAP(l, r, sl), _BITMAP(l, r, sr),
			CurrSlice.isUnsigned(),
			(BusSlices ? BusSlices->slice(_BITMAP(l, r, sl),
							_BITMAP(l, r, sr))
					: NULL));

} /* end of _specc::bufslice::buf_slice */


inline
const _specc::bufslice _specc::bufslice::buf_slice(	/* const bit of vec.*/
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	i) const
{

return _specc::bufslice(CurrSlice, NextSlice,
			_BITMAP(l, r, i), _BITMAP(l, r, i),
			true,	/* single bit is always unsigned */
			(BusSlices ? BusSlices->slice(_BITMAP(l, r, i),
							_BITMAP(l, r, i))
					: NULL));

} /* end of _specc::bufslice::buf_slice */


inline
const _specc::bufslice _specc::bufslice::buf_slice(	/* const slice */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr) const
{

return _specc::bufslice(CurrSlice, NextSlice,
			_BITMAP(l, r, sl), _BITMAP(l, r, sr),
			CurrSlice.isUnsigned(),
			(BusSlices ? BusSlices->slice(_BITMAP(l, r, sl),
							_BITMAP(l, r, sr))
					: NULL));

} /* end of _specc::bufslice::buf_slice */


inline
void _specc::bufslice::Put(void) const	/* store value for bufbus */
{

if (BusSlices)
   { BusSlices->Put(NextSlice.ref);
    } /* fi */

} /* end of _specc::bufslice::Put */


inline
void _specc::bufslice::Get(void) const	/* load value for bufbus */
{

if (BusSlices)
   { BusSlices->Get(CurrSlice.ref);
    } /* fi */

} /* end of _specc::bufslice::Get */


	/**********************/
	/*** _specc::bufbus ***/
	/**********************/


template <_bit::len_t len, bool usign> inline
_specc::bufbus<len,usign>::bufbus(	/* constructor #1 */
	bufbus_elem	*Slices)
	: _specc::buffered<bit<len,usign> >((_specc::event_ptr*)NULL)
{

_specc::bufbus<len,usign>::Slices = Slices;

} /* end of _specc::bufbus::bufbus */


template <_bit::len_t len, bool usign> inline
_specc::bufbus<len,usign>::~bufbus(	/* destructor */
	void)
{

delete Slices;

} /* end of _specc::bufbus::~bufbus */


template <_bit::len_t len, bool usign> inline
_specc::bufslice _specc::bufbus<len,usign>::buf_slice(	/* single bit */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	i)
{

//PC 05/02/04 
return _specc::bufslice(this->CurrValue, this->NextValue,
			_BITMAP(ll, r, i), _BITMAP(ll, r, i),
			true,	/* single bit is always unsigned */
			Slices->slice(_BITMAP(ll, r, i), _BITMAP(ll, r, i)));

} /* end of _specc::bufbus<len,usign>::buf_slice */


template <_bit::len_t len, bool usign> inline
_specc::bufslice _specc::bufbus<len,usign>::buf_slice(	/* slice */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr)
{

return _specc::bufslice(this->CurrValue, this->NextValue,
			_BITMAP(ll, r, sl), _BITMAP(ll, r, sr),
			usign,
			Slices->slice(_BITMAP(ll, r, sl), _BITMAP(ll, r, sr)));

} /* end of _specc::bufbus<len,usign>::buf_slice */


template <_bit::len_t len, bool usign> inline
const _specc::bufslice _specc::bufbus<len,usign>::buf_slice(	/* const bit */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	i) const
{

return _specc::bufslice(this->CurrValue, this->NextValue,
			_BITMAP(ll, r, i), _BITMAP(ll, r, i),
			true,	/* single bit is always unsigned */
			Slices->slice(_BITMAP(ll, r, i), _BITMAP(ll, r, i)));

} /* end of _specc::bufbus<len,usign>::buf_slice */


template <_bit::len_t len, bool usign> inline
const _specc::bufslice _specc::bufbus<len,usign>::buf_slice( /* const slice */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr) const
{

return _specc::bufslice(this->CurrValue, this->NextValue,
			_BITMAP(ll, r, sl), _BITMAP(ll, r, sr),
			usign,
			Slices->slice(_BITMAP(ll, r, sl), _BITMAP(ll, r, sr)));

} /* end of _specc::bufbus<len,usign>::buf_slice */


template <_bit::len_t len, bool usign> inline
_specc::bufbus_elem* _specc::bufbus<len,usign>::bufbus_slice( /* bus slice #1 */
	_bit::len_t	sl,
	_bit::len_t	sr)
{

return(Slices->slice(sl, sr));

} /* end of _specc::bufbus<len,usign>::bufbus_slice #1 */


template <_bit::len_t len, bool usign> inline
_specc::bufbus_elem* _specc::bufbus<len,usign>::bufbus_slice( /* bus slice #2 */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr)
{

return(bufbus_slice(_BITMAP(ll,r,sl), _BITMAP(ll,r,sr)));

} /* end of _specc::bufbus<len,usign>::bufbus_slice #2 */


template <_bit::len_t len, bool usign> inline
void _specc::bufbus<len,usign>::Put(void) const	/* store value for bufbus */
{						/* (override dummy method) */

Slices->Put(this->NextValue);

} /* end of _specc::bufbus<len,usign>::Put */


template <_bit::len_t len, bool usign> inline
void _specc::bufbus<len,usign>::Get(void) const	/* load value for bufbus */
{						/* (override dummy method) */

Slices->Get(this->CurrValue);

} /* end of _specc::bufbus<len,usign>::Get */


	/***************************/
	/*** _specc::bufbus_elem ***/
	/***************************/


inline
_specc::bufbus_elem::bufbus_elem(	/* constructor #1 */
	const _bit	&CurrVal,
	const _bit	&NextVal,
	_bit::len_t	sl,
	_bit::len_t	sr,
	bufbus_elem	*Next /* = NULL */)
	: CurrSlice(CurrVal, sl, sr),
	  NextSlice(NextVal, sl, sr)
{

_specc::bufbus_elem::Next = Next;

} /* end of _specc::bufbus_elem::bufbus_elem */


inline
_specc::bufbus_elem::~bufbus_elem(	/* destructor */
	void)
{

delete Next;

} /* end of _specc::bufbus_elem::~bufbus_elem */


inline
_specc::bufbus_elem *_specc::bufbus_elem::concat(	/* concatenation */
	_specc::bufbus_elem	*List)
{
_specc::bufbus_elem	*Elem;

// note: this is equivalent to _bitbus_element::concat() in file "bit.cc"

assert(List != NULL);

Elem = List;
while(Elem->Next)
   { Elem = Elem->Next;
    } /* elihw */

Elem->Next = this;

/* no merging optimization at this time */

return List;

} /* end of _specc::bufbus_elem::concat */


inline
_specc::bufbus_elem *_specc::bufbus_elem::slice(	/* slicing */
	_bit::len_t	l,
	_bit::len_t	r)
{
_bit::len_t		i,
			NewSl,
			NewSr,
			ElemLen,
			Max,
			Min;
_specc::bufbus_elem	*Head,
			**Tmp,
			*Elem;

// note: this is equivalent to _bitbus_element::slice() in file "bit.cc"

assert(CurrSlice.sl == NextSlice.sl);
assert(CurrSlice.sr == NextSlice.sr);

ElemLen = _BITLEN(CurrSlice.sl, CurrSlice.sr);
Max = _MAXLEN(l,r);
Min = _MINLEN(l,r);
Head = NULL;
Tmp = &Head;

// Walk through the list until we hit parts of the slice

for(Elem = this, i = ElemLen; Min >= i; i += ElemLen)
   { Elem = Elem->Next;
     ElemLen = _BITLEN(Elem->CurrSlice.sl, Elem->CurrSlice.sr);
    } /* rof */
i -= ElemLen;

// Copy and transform elements until we leave slice

for( ; i <= Max; i += ElemLen)
   { // Map slice coordinates
     if (Elem->CurrSlice.sl > Elem->CurrSlice.sr)
	{ NewSl = Elem->CurrSlice.sl - _MAXLEN(i + ElemLen - 1 - Max, 0);
	  NewSr = Elem->CurrSlice.sr + _MAXLEN(Min - i, 0);
	 } /* fi */
     else
	{ NewSl = Elem->CurrSlice.sl + _MAXLEN(i + ElemLen - 1 - Max, 0);
	  NewSr = Elem->CurrSlice.sr - _MAXLEN(Min - i, 0);
	 } /* esle */
     if (l > r)
	{ // Add elements at tail of list
	  *Tmp = new _specc::bufbus_elem(Elem->CurrSlice.ref,
						Elem->NextSlice.ref,
						NewSl, NewSr);
	  Tmp = &((*Tmp)->Next);
	 } /* fi */
     else
	{ // Add elements at the head of list
	  Head = new _specc::bufbus_elem(Elem->CurrSlice.ref,
						Elem->NextSlice.ref,
						NewSr, NewSl, Head);
	 } /* esle */
     Elem = Elem->Next;
     ASSERT(Elem || ((i+ElemLen) > Max)); // Tried to slice outside of range...
     if (!Elem)
	{ break;
	 } /* fi */
     ElemLen = _BITLEN(Elem->CurrSlice.sl, Elem->CurrSlice.sr);
    } /* rof */

ASSERT(Head);	// Did we cut anything?

return(Head);

} /* end of _specc::bufbus_elem::slice */


inline
void _specc::bufbus_elem::Put(		/* synchronization with original */
	const _bit	&Buf)		/* (store tmp. value into next value) */
{
_bit::len_t		i,
			len;
_specc::bufbus_elem	*Elem;

// note: this is equivalent to _bitbus_element::put() in file "bit.cc"

Elem = this;
for(i = 0; Elem; i += len, Elem = Elem->Next)
   { len = _BITLEN(Elem->NextSlice.sl, Elem->NextSlice.sr);
     Elem->NextSlice.ref.copyBits(Buf, i + len - 1, i,
					Elem->NextSlice.sl,
					Elem->NextSlice.sr);
    } /* rof */

} /* end of _specc::bufbus_elem::Put */


inline
void _specc::bufbus_elem::Get(		/* synchronization with original */
	const _bit	&Buf)		/* (load tmp. value from curr. value) */
{
_bit::len_t		i,
			len;
const _specc::bufbus_elem *Elem;

// note: this is equivalent to _bitbus_element::get() in file "bit.cc"

Elem = this;
for(i = 0; Elem; i += len, Elem = Elem->Next)
   { len = _BITLEN(Elem->CurrSlice.sl, Elem->CurrSlice.sr);
     const_cast<_bit&>(Buf).copyBits(Elem->CurrSlice.ref,
					Elem->CurrSlice.sl,
					Elem->CurrSlice.sr,
					i + len - 1, i);
    } /* rof */

} /* end of _specc::bufbus_elem::Get */


inline
void _specc::bufbus_elem::GetNextValue(	/* synchronization with original */
	const _bit	&Buf)		/* (load tmp. value from next value) */
{
_bit::len_t		i,
			len;
const _specc::bufbus_elem *Elem;

// note: this is equivalent to _bitbus_element::get() in file "bit.cc"

Elem = this;
for(i = 0; Elem; i += len, Elem = Elem->Next)
   { len = _BITLEN(Elem->NextSlice.sl, Elem->NextSlice.sr);
     const_cast<_bit&>(Buf).copyBits(Elem->NextSlice.ref,
					Elem->NextSlice.sl,
					Elem->NextSlice.sr,
					i + len - 1, i);
    } /* rof */

} /* end of _specc::bufbus_elem::GetNextValue */


	/*****************************************/
	/*** signal template (general version) ***/
	/*****************************************/


template<class T> inline
_specc::signal<T>::signal(void)			/* constructor #1 */
	: _specc::buffered<T>(this)
{

/* nothing to do */

} /* end of signal<T>::signal #1 */


template<class T> inline
_specc::signal<T>::signal(			/* constructor #2 */
	const T		Init)
	: _specc::buffered<T>(this, Init)
{

/* nothing to do */

} /* end of signal<T>::signal #2 */


template<class T> inline
_specc::signal<T>::signal(			/* copy constructor (#3) */
	const _specc::signal<T> &Orig)
	: _specc::buffered<T>(Orig)
{

/* nothing to do */

} /* end of signal<T>::signal #3 */


template<class T> inline
_specc::signal<T>::~signal(void)		/* destructor */
{

/* nothing to do */

} /* end of signal<T>::~signal */


template<class T> inline
bool _specc::signal<T>::IsValidEdge(		/* signal edge event filter */
	_specc::EDGE	Edge)		/* (overloads dummy in event class) */
{

return(_signal_ValidEdge(Edge, *this));	// (see helper functions at top!)

} /* end of signal<T>::IsValidEdge */


template<class T> inline
bool _specc::signal<T>::ResetIsActive(	/* asynchronous reset signal check */
	bool	ResetActiveHi)		/* (overloads dummy in event class) */
{

return(_signal_ResetActive(ResetActiveHi,
			*this));	// (see helper functions at top!)

} /* end of signal<T>::ResetIsActive */


template<class T> inline
_specc::signal<T> &_specc::signal<T>::operator=(/* assignment operator #1 */
	T	Var)
{

Notify();
_specc::buffered<T>::operator=(Var);

return *this;

} /* end of signal<T>::operator= */


template<class T> inline
_specc::signal<T> &_specc::signal<T>::operator=(/* assignment operator #1b */
	const signal<T> &SignalVar)	/* (needed, otherwise compiler  */
{					/* will generate 'bad' default) */

Notify();
_specc::buffered<T>::operator=(SignalVar);

return *this;

} /* end of signal<T>::operator= */


template<class T> inline
_specc::signal<T> &_specc::signal<T>::operator+=(/* assignment operator #2 */
	T	Var)
{

Notify();
_specc::buffered<T>::operator+=(Var);

return *this;

} /* end of signal<T>::operator+= */


template<class T> inline
_specc::signal<T> &_specc::signal<T>::operator-=(/* assignment operator #3 */
	T	Var)
{

Notify();
_specc::buffered<T>::operator-=(Var);

return *this;

} /* end of signal<T>::operator-= */


template<class T> inline
_specc::signal<T> &_specc::signal<T>::operator*=(/* assignment operator #4 */
	T	Var)
{

Notify();
_specc::buffered<T>::operator*=(Var);

return *this;

} /* end of signal<T>::operator*= */


template<class T> inline
_specc::signal<T> &_specc::signal<T>::operator/=(/* assignment operator #5 */
	T	Var)
{

Notify();
_specc::buffered<T>::operator/=(Var);

return *this;

} /* end of signal<T>::operator/= */


template<class T> inline
_specc::signal<T> &_specc::signal<T>::operator%=(	/* assignment operator #6 */
	T	Var)
{

Notify();
_specc::buffered<T>::operator%=(Var);

return *this;

} /* end of signal<T>::operator%= */


template<class T> inline
_specc::signal<T> &_specc::signal<T>::operator^=(/* assignment operator #7 */
	T	Var)
{

Notify();
_specc::buffered<T>::operator^=(Var);

return *this;

} /* end of signal<T>::operator^= */


template<class T> inline
_specc::signal<T> &_specc::signal<T>::operator&=(/* assignment operator #8 */
	T	Var)
{

Notify();
_specc::buffered<T>::operator&=(Var);

return *this;

} /* end of signal<T>::operator&= */


template<class T> inline
_specc::signal<T> &_specc::signal<T>::operator|=(/* assignment operator #9 */
	T	Var)
{

Notify();
_specc::buffered<T>::operator|=(Var);

return *this;

} /* end of signal<T>::operator|= */


template<class T> inline
_specc::signal<T> &_specc::signal<T>::operator<<=(/* assignment operator #10 */
	unsigned int	Var)
{

Notify();
_specc::buffered<T>::operator<<=(Var);

return *this;

} /* end of signal<T>::operator<<= */


template<class T> inline
_specc::signal<T> &_specc::signal<T>::operator>>=(/* assignment operator #11 */
	unsigned int	Var)
{

Notify();
_specc::buffered<T>::operator>>=(Var);

return *this;

} /* end of signal<T>::operator>>= */


template<class T> inline
_specc::signal<T> &_specc::signal<T>::operator++()
                                   /* increment operator #1 (pre-) */
{

Notify();
_specc::buffered<T>::operator++();

return *this;

} /* end of signal<T>::operator++ */


template<class T> inline
_specc::signal<T> _specc::signal<T>::operator++(int)	
                                   /* increment operator #2 (post-) */
{
_specc::signal<T>	Tmp(*this);

Notify();
_specc::buffered<T>::operator++(1);

return Tmp;

} /* end of signal<T>::operator++ */


template<class T> inline
_specc::signal<T> &_specc::signal<T>::operator--()	
                                   /* decrement operator #1 (pre-) */
{

Notify();
_specc::buffered<T>::operator--();

return *this;

} /* end of signal<T>::operator-- */


template<class T> inline
_specc::signal<T> _specc::signal<T>::operator--(int)	
                                  /* decrement operator #2 (post-) */
{
_specc::signal<T>	Tmp(*this);

Notify();
_specc::buffered<T>::operator--(1);

return Tmp;

} /* end of signal<T>::operator-- */

//PC 05/02/04
template<class T> template<class MT> inline
_specc::sig_elem<MT> _specc::signal<T>::member(		
                                  /* member access "operator" */
	int		CharOffset)
{

return _specc::sig_elem<MT>(this,
			(void*)(((char*)&this->CurrValue) + CharOffset),
			(void*)(((char*)&this->NextValue) + CharOffset));

} /* end of signal<T>::member */


template<class T> template<class MT> inline
const _specc::sig_elem<MT> _specc::signal<T>::member(	
                                  /* const member access "op." */
	int		CharOffset) const
{

return _specc::sig_elem<MT>(const_cast<_specc::signal<T>*>(this),
			(void*)(((char*)&this->CurrValue) + CharOffset),
			(void*)(((char*)&this->NextValue) + CharOffset));

} /* end of signal<T>::member */


	/*******************************************************/
	/*** signal template (specialization for bitvectors) ***/
	/*******************************************************/


template<_bit::len_t l, bool s> inline
_specc::signal<bit<l,s> >::signal(void)			/* constructor #1 */
	: _specc::buffered<bit<l,s> >(this)
{

/* nothing to do */

} /* end of signal<bit<l,s> >::signal #1 */


template<_bit::len_t l, bool s> inline
_specc::signal<bit<l,s> >::signal(			/* constructor #2 */
	const bit<l,s>	Init)
	: _specc::buffered<bit<l,s> >(this, Init)
{

/* nothing to do */

} /* end of signal<bit<l,s> >::signal #2 */


template<_bit::len_t l, bool s> inline
_specc::signal<bit<l,s> >::signal(		/* copy constructor (#3) */
	const _specc::signal<bit<l,s> > &Orig)
	: _specc::buffered<bit<l,s> >(Orig)
{

/* nothing to do */

} /* end of signal<bit<l,s> >::signal #3 */


template<_bit::len_t l, bool s> inline
_specc::signal<bit<l,s> >::~signal(void)		/* destructor */
{

/* nothing to do */

} /* end of signal<bit<l,s> >::~signal */


template<_bit::len_t l, bool s> inline
bool _specc::signal<bit<l,s> >::IsValidEdge(	/* signal edge event filter */
	_specc::EDGE	Edge)		/* (overloads dummy in event class) */
{

return(_signal_ValidEdge(Edge, *this));	// (see helper functions at top!)

} /* end of signal<bit<l,s> >::IsValidEdge */


template<_bit::len_t l, bool s> inline
bool _specc::signal<bit<l,s> >::ResetIsActive(	
                                        /* asynchronous reset signal check */
	bool	ResetActiveHi)		/* (overloads dummy in event class) */
{

return(_signal_ResetActive(ResetActiveHi,
			*this));	// (see helper functions at top!)

} /* end of signal<bit<l,s> >::ResetIsActive */


template<_bit::len_t l, bool s> inline
_specc::signal<bit<l,s> > &_specc::signal<bit<l,s> >::operator=(
                                       /* assignment operator #1 */
	bit<l,s>	Var)
{

Notify();
_specc::buffered<bit<l,s> >::operator=(Var);

return *this;

} /* end of signal<bit<l,s> >::operator= */


template<_bit::len_t l, bool s> inline
_specc::signal<bit<l,s> > &_specc::signal<bit<l,s> >::operator=(
                                               /* assignment operator #1b */
	const _specc::signal<bit<l,s> > &SignalVar)
                                         /* (needed, otherwise comp. */
{					/* will gen. 'bad' default) */

Notify();
_specc::buffered<bit<l,s> >::operator=(SignalVar);

return *this;

} /* end of signal<bit<l,s> >::operator= */


template<_bit::len_t l, bool s> inline
_specc::signal<bit<l,s> > &_specc::signal<bit<l,s> >::operator+=(
                                               /* assignment operator #2 */
	bit<l,s>	Var)
{

Notify();
_specc::buffered<bit<l,s> >::operator+=(Var);

return *this;

} /* end of signal<bit<l,s> >::operator+= */


template<_bit::len_t l, bool s> inline
_specc::signal<bit<l,s> > &_specc::signal<bit<l,s> >::operator-=(
                                             /* assignment operator #3 */
	bit<l,s>	Var)
{

Notify();
_specc::buffered<bit<l,s> >::operator-=(Var);

return *this;

} /* end of signal<bit<l,s> >::operator-= */


template<_bit::len_t l, bool s> inline
_specc::signal<bit<l,s> > &_specc::signal<bit<l,s> >::operator*=(
                                           /* assignment operator #4 */
	bit<l,s>	Var)
{

Notify();
_specc::buffered<bit<l,s> >::operator*=(Var);

return *this;

} /* end of signal<bit<l,s> >::operator*= */


template<_bit::len_t l, bool s> inline
_specc::signal<bit<l,s> > &_specc::signal<bit<l,s> >::operator/=(
                                        /* assignment operator #5 */
	bit<l,s>	Var)
{

Notify();
_specc::buffered<bit<l,s> >::operator/=(Var);

return *this;

} /* end of signal<bit<l,s> >::operator/= */


template<_bit::len_t l, bool s> inline
_specc::signal<bit<l,s> > &_specc::signal<bit<l,s> >::operator%=(
                                       /* assignment operator #6 */
	bit<l,s>	Var)
{

Notify();
_specc::buffered<bit<l,s> >::operator%=(Var);

return *this;

} /* end of signal<bit<l,s> >::operator%= */


template<_bit::len_t l, bool s> inline
_specc::signal<bit<l,s> > &_specc::signal<bit<l,s> >::operator^=(
                                            /* assignment operator #7 */
	bit<l,s>	Var)
{

Notify();
_specc::buffered<bit<l,s> >::operator^=(Var);

return *this;

} /* end of signal<bit<l,s> >::operator^= */


template<_bit::len_t l, bool s> inline
_specc::signal<bit<l,s> > &_specc::signal<bit<l,s> >::operator&=(
                                            /* assignment operator #8 */
	bit<l,s>	Var)
{

Notify();
_specc::buffered<bit<l,s> >::operator&=(Var);

return *this;

} /* end of signal<bit<l,s> >::operator&= */


template<_bit::len_t l, bool s> inline
_specc::signal<bit<l,s> > &_specc::signal<bit<l,s> >::operator|=(
                                            /* assignment operator #9 */
	bit<l,s>	Var)
{

Notify();
_specc::buffered<bit<l,s> >::operator|=(Var);

return *this;

} /* end of signal<bit<l,s> >::operator|= */


template<_bit::len_t l, bool s> inline
_specc::signal<bit<l,s> > &_specc::signal<bit<l,s> >::operator<<=(
                                             /* assignment operator #10 */
	unsigned int	Var)
{

Notify();
_specc::buffered<bit<l,s> >::operator<<=(Var);

return *this;

} /* end of signal<bit<l,s> >::operator<<= */


template<_bit::len_t l, bool s> inline
_specc::signal<bit<l,s> > &_specc::signal<bit<l,s> >::operator>>=(
                                            /* assignment operator #11 */
	unsigned int	Var)
{

Notify();
_specc::buffered<bit<l,s> >::operator>>=(Var);

return *this;

} /* end of signal<bit<l,s> >::operator>>= */


template<_bit::len_t l, bool s> inline
_specc::signal<bit<l,s> > &_specc::signal<bit<l,s> >::operator++()
                                           /* incr. op. #1 (pre-) */
{

Notify();
_specc::buffered<bit<l,s> >::operator++();

return *this;

} /* end of signal<bit<l,s> >::operator++ */


template<_bit::len_t l, bool s> inline
_specc::signal<bit<l,s> > _specc::signal<bit<l,s> >::operator++(int)
                                         /* incr. op. #2 (post-) */
{
_specc::signal<bit<l,s> >	Tmp(*this);

Notify();
_specc::buffered<bit<l,s> >::operator++(1);

return Tmp;

} /* end of signal<bit<l,s> >::operator++ */


template<_bit::len_t l, bool s> inline
_specc::signal<bit<l,s> > &_specc::signal<bit<l,s> >::operator--()
                                          /* decr. op. #1 (pre-) */
{

Notify();
_specc::buffered<bit<l,s> >::operator--();

return *this;

} /* end of signal<bit<l,s> >::operator-- */


template<_bit::len_t l, bool s> inline
_specc::signal<bit<l,s> > _specc::signal<bit<l,s> >::operator--(int)
                                           /* decr. op. #2 (post-) */
{
_specc::signal<bit<l,s> >	Tmp(*this);

Notify();
_specc::buffered<bit<l,s> >::operator--(1);

return Tmp;

} /* end of signal<bit<l,s> >::operator-- */

//PC 05/02/04
template<_bit::len_t l, bool s> inline
_specc::sigslice _specc::signal<bit<l,s> >::sig_slice(	
                                            /* single bit of a bitvector */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	i)
{

return _specc::sigslice(this,
			this->CurrValue, this->NextValue,
			_BITMAP(ll, r, i), _BITMAP(ll, r, i),
			true);	/* single bit is always unsigned */

} /* end of signal<bit<l,s> >::sig_slice */


template<_bit::len_t l, bool s> inline
_specc::sigslice _specc::signal<bit<l,s> >::sig_slice(/* slice of a bitvector */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr)
{

return _specc::sigslice(this,
			this->CurrValue, this->NextValue,
			_BITMAP(ll, r, sl), _BITMAP(ll, r, sr),
			(_bit(this->CurrValue)).isUnsigned());

} /* end of signal<bit<l,s> >::sig_slice */


template<_bit::len_t l, bool s> inline
const _specc::sigslice _specc::signal<bit<l,s> >::sig_slice(
                                              /* const bit of a bitvec.*/
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	i) const
{

return _specc::sigslice(const_cast<_specc::signal<bit<l,s> >*>(this),
			this->CurrValue, this->NextValue,
			_BITMAP(ll, r, i), _BITMAP(ll, r, i),
			true);	/* single bit is always unsigned */

} /* end of signal<bit<l,s> >::sig_slice */


template<_bit::len_t l, bool s> inline
const _specc::sigslice _specc::signal<bit<l,s> >::sig_slice(
                               /* const slice of a bitvec*/
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr) const
{

return _specc::sigslice(const_cast<_specc::signal<bit<l,s> >*>(this),
			this->CurrValue, this->NextValue,
			_BITMAP(ll, r, sl), _BITMAP(ll, r, sr),
			(_bit(this->CurrValue)).isUnsigned());

} /* end of signal<bit<l,s> >::sig_slice */

//PC 05/02/04
template<_bit::len_t l, bool s> inline
_specc::sigbus_elem* _specc::signal<bit<l,s> >::sigbus_slice(	
                                                 /* bus slice #1 */
	_bit::len_t	sl,
	_bit::len_t	sr)
{

return(new _specc::sigbus_elem(
			this->CurrValue,
			this->NextValue,
			sl, sr,
			this));

} /* end of signal<bit<l,s> >::sigbus_slice #1 */


template<_bit::len_t l, bool s> inline
_specc::sigbus_elem* _specc::signal<bit<l,s> >::sigbus_slice(	
                                                 /* bus slice #2 */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr)
{

return(sigbus_slice(_BITMAP(ll,r,sl), _BITMAP(ll,r,sr)));

} /* end of signal<bit<l,s> >::sigbus_slice #2 */


	/********************************************************/
	/*** signal template (specialization for array types) ***/
	/********************************************************/


template<int i, class T> inline
_specc::signal<T[i]>::signal(void)		/* constructor #1 */
	: _specc::buffered<T[i]>(this)
{

/* nothing to do */

} /* end of signal<T[i]>::signal #1 */


template<int i, class T> inline
_specc::signal<T[i]>::signal(			/* constructor #2 */
	T		Init[i])
	: _specc::buffered<T[i]>(this, Init)
{

/* nothing to do */

} /* end of signal<T[i]>::signal #2 */


template<int i, class T> inline
_specc::signal<T[i]>::signal(			/* copy constructor (#3) */
	const _specc::signal<T[i]> &Orig)
	: _specc::buffered<T[i]>(Orig)
{

/* nothing to do */

} /* end of signal<T[i]>::signal #3 */


template<int i, class T> inline
_specc::signal<T[i]>::~signal(void)		/* destructor */
{

/* nothing to do */

} /* end of signal<T[i]>::~signal */


template<int i, class T> inline
bool _specc::signal<T[i]>::IsValidEdge(		/* signal edge event filter */
	_specc::EDGE	Edge)		/* (overloads dummy in event class) */
{

return(_signal_ValidEdge(Edge, *this));	// (see helper functions at top!)

} /* end of signal<T[i]>::IsValidEdge */


template<int i, class T> inline
bool _specc::signal<T[i]>::ResetIsActive(/* asynchronous reset signal check */
	bool		ResetActiveHi)	/* (overloads dummy in event class) */
{

return(_signal_ResetActive(ResetActiveHi,
			*this));	// (see helper functions at top!)

} /* end of signal<T[i]>::ResetIsActive */


template<int i, class T> inline
_specc::signal<T[i]> &_specc::signal<T[i]>::operator=(
                                         /* assignment operator #1 */
	T	Var[i])
{

Notify();
_specc::buffered<T[i]>::operator=(Var);

return *this;

} /* end of signal<T[i]>::operator= #1 */


template<int i, class T> inline
_specc::signal<T[i]> &_specc::signal<T[i]>::operator=(	
                                                /* assignment operator #1b */
	const _specc::signal<T[i]> &SignalVar)/* (needed, otherwise compiler  */
{					/* will generate 'bad' default) */

Notify();
_specc::buffered<T[i]>::operator=(SignalVar);

return *this;

} /* end of signal<T[i]>::operator= #1b */


template<int i, class T> inline
_specc::signal<T[i]> &_specc::signal<T[i]>::operator=(	
                                         /* assignment operator #1c */
	const _specc::sig_elem<T[i]> &SigElemVar)
{

Notify();
_specc::buffered<T[i]>::operator=((const _specc::buf_elem<T[i]>&)SigElemVar);

return *this;

} /* end of signal<T[i]>::operator= #1c */


template<int i, class T> inline
_specc::sig_elem<T> _specc::signal<T[i]>::operator[](	
                                         /* array access operator */
	int		Index)
{

return _specc::sig_elem<T>(this,
				(void*)(&this->CurrValue[Index]),
				(void*)(&this->NextValue[Index]));

} /* end of signal<T[i]>::operator[] */


	/*******************************************/
	/*** sig_elem template (general version) ***/
	/*******************************************/


template<class T> inline
_specc::sig_elem<T>::sig_elem(		/* constructor #1 */
	event	*SigEventPtr,
	void	*CurrPtr,
	void	*NextPtr)
	: _specc::buf_elem<T>(CurrPtr, NextPtr)
{

_specc::sig_elem<T>::SigEventPtr = SigEventPtr;

} /* end of _specc::sig_elem<T>::sig_elem #1 */


template<class T> inline
_specc::sig_elem<T>::~sig_elem(void)	/* destructor */
{

/* nothing to do */

} /* end of _specc::sig_elem<T>::~sig_elem */


template<class T> inline
_specc::sig_elem<T> &_specc::sig_elem<T>::operator=(	/* assignment op. #1 */
	T	Var)
{

SigEventPtr->Notify();
_specc::buf_elem<T>::operator=(Var);

return *this;

} /* end of _specc::sig_elem<T>::operator= */


template<class T> inline
_specc::sig_elem<T> &_specc::sig_elem<T>::operator=(	/* assignment op. #1b */
	const sig_elem<T> &SigElem)	/* (needed, otherwise compiler  */
{					/* will generate 'bad' default) */

SigEventPtr->Notify();
_specc::buf_elem<T>::operator=(SigElem);

return *this;

} /* end of _specc::sig_elem<T>::operator= */


template<class T> inline
_specc::sig_elem<T> &_specc::sig_elem<T>::operator+=(	/* assignment op. #2 */
	T	Var)
{

SigEventPtr->Notify();
_specc::buf_elem<T>::operator+=(Var);

return *this;

} /* end of _specc::sig_elem<T>::operator+= */


template<class T> inline
_specc::sig_elem<T> &_specc::sig_elem<T>::operator-=(	/* assignment op. #3 */
	T	Var)
{

SigEventPtr->Notify();
_specc::buf_elem<T>::operator-=(Var);

return *this;

} /* end of _specc::sig_elem<T>::operator-= */


template<class T> inline
_specc::sig_elem<T> &_specc::sig_elem<T>::operator*=(	/* assignment op. #4 */
	T	Var)
{

SigEventPtr->Notify();
_specc::buf_elem<T>::operator*=(Var);

return *this;

} /* end of _specc::sig_elem<T>::operator*= */


template<class T> inline
_specc::sig_elem<T> &_specc::sig_elem<T>::operator/=(	/* assignment op. #5 */
	T	Var)
{

SigEventPtr->Notify();
_specc::buf_elem<T>::operator/=(Var);

return *this;

} /* end of _specc::sig_elem<T>::operator/= */


template<class T> inline
_specc::sig_elem<T> &_specc::sig_elem<T>::operator%=(	/* assignment op. #6 */
	T	Var)
{

SigEventPtr->Notify();
_specc::buf_elem<T>::operator%=(Var);

return *this;

} /* end of _specc::sig_elem<T>::operator%= */


template<class T> inline
_specc::sig_elem<T> &_specc::sig_elem<T>::operator^=(	/* assignment op. #7 */
	T	Var)
{

SigEventPtr->Notify();
_specc::buf_elem<T>::operator^=(Var);

return *this;

} /* end of _specc::sig_elem<T>::operator^= */


template<class T> inline
_specc::sig_elem<T> &_specc::sig_elem<T>::operator&=(	/* assignment op. #8 */
	T	Var)
{

SigEventPtr->Notify();
_specc::buf_elem<T>::operator&=(Var);

return *this;

} /* end of _specc::sig_elem<T>::operator&= */


template<class T> inline
_specc::sig_elem<T> &_specc::sig_elem<T>::operator|=(	/* assignment op. #9 */
	T	Var)
{

SigEventPtr->Notify();
_specc::buf_elem<T>::operator|=(Var);

return *this;

} /* end of _specc::sig_elem<T>::operator|= */


template<class T> inline
_specc::sig_elem<T> &_specc::sig_elem<T>::operator<<=(	/* assignment op. #10 */
	unsigned int	Var)
{

SigEventPtr->Notify();
_specc::buf_elem<T>::operator<<=(Var);

return *this;

} /* end of _specc::sig_elem<T>::operator<<= */


template<class T> inline
_specc::sig_elem<T> &_specc::sig_elem<T>::operator>>=(	/* assignment op. #11 */
	unsigned int	Var)
{

SigEventPtr->Notify();
_specc::buf_elem<T>::operator>>=(Var);

return *this;

} /* end of _specc::sig_elem<T>::operator>>= */


template<class T> inline
_specc::sig_elem<T> &_specc::sig_elem<T>::operator++()	/* incr.op. #1 (pre-) */
{

SigEventPtr->Notify();
_specc::buf_elem<T>::operator++();

return *this;

} /* end of _specc::sig_elem<T>::operator++ */


template<class T> inline
_specc::sig_elem<T> _specc::sig_elem<T>::operator++(int) /* incr.op. #2(post-)*/
{
sig_elem<T>	Tmp(*this);

SigEventPtr->Notify();
_specc::buf_elem<T>::operator++(1);

return Tmp;

} /* end of _specc::sig_elem<T>::operator++ */


template<class T> inline
_specc::sig_elem<T> &_specc::sig_elem<T>::operator--()	/* decr.op. #1 (pre-) */
{

SigEventPtr->Notify();
_specc::buf_elem<T>::operator--();

return *this;

} /* end of _specc::sig_elem<T>::operator-- */


template<class T> inline
_specc::sig_elem<T> _specc::sig_elem<T>::operator--(int) /* decr.op. #2(post-)*/
{
sig_elem<T>	Tmp(*this);

SigEventPtr->Notify();
_specc::buf_elem<T>::operator--(1);

return Tmp;

} /* end of _specc::sig_elem<T>::operator-- */

//PC 05/02/04
template<class T> inline
_specc::sigslice _specc::sig_elem<T>::sig_slice(/* single bit of a bitvec. */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	i)
{

return _specc::sigslice(SigEventPtr,
			*this->CurrPtr, *this->NextPtr,
			_BITMAP(l, r, i), _BITMAP(l, r, i),
			true);	/* single bit is always unsigned */

} /* end of _specc::sig_elem<T>::sig_slice */


template<class T> inline
_specc::sigslice _specc::sig_elem<T>::sig_slice( /* slice of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr)
{

return _specc::sigslice(SigEventPtr,
			*this->CurrPtr, *this->NextPtr,
			_BITMAP(l, r, sl), _BITMAP(l, r, sr),
			(_bit(*this->CurrPtr)).isUnsigned());

} /* end of _specc::sig_elem<T>::sig_slice */


template<class T> inline
const _specc::sigslice _specc::sig_elem<T>::sig_slice( /* const bit of vec.*/
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	i) const
{

return _specc::sigslice(SigEventPtr,
			*this->CurrPtr, *this->NextPtr,
			_BITMAP(l, r, i), _BITMAP(l, r, i),
			true);	/* single bit is always unsigned */

} /* end of _specc::sig_elem<T>::sig_slice */


template<class T> inline
const _specc::sigslice _specc::sig_elem<T>::sig_slice( /* const slice */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr) const
{

return _specc::sigslice(SigEventPtr,
			*this->CurrPtr, *this->NextPtr,
			_BITMAP(l, r, sl), _BITMAP(l, r, sr),
			(_bit(*this->CurrPtr)).isUnsigned());

} /* end of _specc::sig_elem<T>::sig_slice */


template<class T> template<class MT> inline
_specc::sig_elem<MT> _specc::sig_elem<T>::member( /* member access "operator" */
	int		CharOffset)
{

return _specc::sig_elem<MT>(SigEventPtr,
				(void*)(((char*)this->CurrPtr) + CharOffset),
				(void*)(((char*)this->NextPtr) + CharOffset));

} /* end of _specc::sig_elem<T>::member */


template<class T> template<class MT> inline
const _specc::sig_elem<MT> _specc::sig_elem<T>::member(	/* const member "op." */
	int		CharOffset) const
{

return _specc::sig_elem<MT>(SigEventPtr,
				(void*)(((char*)this->CurrPtr) + CharOffset),
				(void*)(((char*)this->NextPtr) + CharOffset));

} /* end of _specc::sig_elem<T>::member */


	/**********************************************************/
	/*** sig_elem template (specialization for array types) ***/
	/**********************************************************/


template<int i, class T> inline
_specc::sig_elem<T[i]>::sig_elem(		/* constructor #1 */
	event	*SigEventPtr,
	void	*CurrPtr,
	void	*NextPtr)
	: _specc::buf_elem<T[i]>(CurrPtr, NextPtr)
{

_specc::sig_elem<T[i]>::SigEventPtr = SigEventPtr;

} /* end of _specc::sig_elem<T[i]>::sig_elem #1 */


template<int i, class T> inline
_specc::sig_elem<T[i]>::~sig_elem(void)		/* destructor */
{

/* nothing to do */

} /* end of _specc::sig_elem<T[i]>::~sig_elem */


template<int i, class T> inline
_specc::sig_elem<T[i]> &_specc::sig_elem<T[i]>::operator=( /* assign.op. #1 */
	T	Var[i])
{

SigEventPtr->Notify();
_specc::buf_elem<T[i]>::operator=(Var);

return *this;

} /* end of _specc::sig_elem<T[i]>::operator= #1 */


template<int i, class T> inline
_specc::sig_elem<T[i]> &_specc::sig_elem<T[i]>::operator=( /* assign.op. #1b */
	const _specc::sig_elem<T[i]> &SigElem) /* (needed, otherwise compiler */
{					       /*  generates 'bad' default)   */

SigEventPtr->Notify();
_specc::buf_elem<T[i]>::operator=(SigElem);

return *this;

} /* end of _specc::sig_elem<T[i]>::operator= #1b */


template<int i, class T> inline
_specc::sig_elem<T[i]> &_specc::sig_elem<T[i]>::operator=( /* assign.op. #1c */
	const _specc::signal<T[i]> &SignalVar)
{

SigEventPtr->Notify();
_specc::buf_elem<T[i]>::operator=(SignalVar);

return *this;

} /* end of _specc::sig_elem<T[i]>::operator= #1c */

//PC 05/02/04
template<int i, class T> inline
_specc::sig_elem<T> _specc::sig_elem<T[i]>::operator[](	/* array access op. */
	int		Index)
{

return _specc::sig_elem<T>(SigEventPtr,
				(void*)(&this->CurrPtr[Index]),
				(void*)(&this->NextPtr[Index]));

} /* end of _specc::sig_elem<T[i]>::operator[] */


	/************************/
	/*** _specc::sigslice ***/
	/************************/


inline
_specc::sigslice::sigslice(			/* constructor #1 */
	event		*SigEventPtr,
	const _bit	&CurrVec,
	const _bit	&NextVec,
	_bit::len_t	Left,
	_bit::len_t	Right,
	bool		Unsigned,
	sigbus_elem	*Slices /* = NULL */) : /* only for slice of sigbus */
		bufslice(CurrVec, NextVec, Left, Right, Unsigned, Slices),
		SigEventPtr(SigEventPtr)
{

/* nothing else to do */

} /* end of _specc::sigslice::sigslice #1 */


inline
_specc::sigslice::sigslice(			/* copy constructor (#2) */
	const _specc::sigslice	&Orig)
	:	_specc::bufslice(Orig),
		SigEventPtr(SigEventPtr)
{

assert(false);	// this copy constructor should never be called
		// because all bitvector slices will be converted
		// to working _bit objects before use

} /* end of _specc::sigslice::sigslice #2 */


inline
_specc::sigslice::~sigslice(void)		/* destructor */
{

delete ((sigbus_elem*) BusSlices);
BusSlices = NULL;	// don't let base class delete these again!

} /* end of _specc::sigslice::~sigslice */


inline
_specc::sigslice &_specc::sigslice::operator=(		/* assignment op. #1 */
	_bit	Var)
{

Notify();
_specc::bufslice::operator=(Var);

return *this;

} /* end of _specc::sigslice::operator= */


inline
_specc::sigslice &_specc::sigslice::operator=(		/* assignment op. #1b */
	const sigslice &SigSlice)	/* (needed, otherwise compiler  */
{					/* will generate 'bad' default) */

Notify();
_specc::bufslice::operator=(SigSlice);

return *this;

} /* end of _specc::sigslice::operator= */


inline
_specc::sigslice &_specc::sigslice::operator+=(		/* assignment op. #2 */
	_bit	Var)
{

Notify();
_specc::bufslice::operator+=(Var);

return *this;

} /* end of _specc::sigslice::operator+= */


inline
_specc::sigslice &_specc::sigslice::operator-=(		/* assignment op. #3 */
	_bit	Var)
{

Notify();
_specc::bufslice::operator-=(Var);

return *this;

} /* end of _specc::sigslice::operator-= */


inline
_specc::sigslice &_specc::sigslice::operator*=(		/* assignment op. #4 */
	_bit	Var)
{

Notify();
_specc::bufslice::operator*=(Var);

return *this;

} /* end of _specc::sigslice::operator*= */


inline
_specc::sigslice &_specc::sigslice::operator/=(		/* assignment op. #5 */
	_bit	Var)
{

Notify();
_specc::bufslice::operator/=(Var);

return *this;

} /* end of _specc::sigslice::operator/= */


inline
_specc::sigslice &_specc::sigslice::operator%=(		/* assignment op. #6 */
	_bit	Var)
{

Notify();
_specc::bufslice::operator%=(Var);

return *this;

} /* end of _specc::sigslice::operator%= */


inline
_specc::sigslice &_specc::sigslice::operator^=(		/* assignment op. #7 */
	_bit	Var)
{

Notify();
_specc::bufslice::operator^=(Var);

return *this;

} /* end of _specc::sigslice::operator^= */


inline
_specc::sigslice &_specc::sigslice::operator&=(		/* assignment op. #8 */
	_bit	Var)
{

Notify();
_specc::bufslice::operator&=(Var);

return *this;

} /* end of _specc::sigslice::operator&= */


inline
_specc::sigslice &_specc::sigslice::operator|=(		/* assignment op. #9 */
	_bit	Var)
{

Notify();
_specc::bufslice::operator|=(Var);

return *this;

} /* end of _specc::sigslice::operator|= */


inline
_specc::sigslice &_specc::sigslice::operator<<=(	/* assignment op. #10 */
	unsigned int	Var)
{

Notify();
_specc::bufslice::operator<<=(Var);

return *this;

} /* end of _specc::sigslice::operator<<= */


inline
_specc::sigslice &_specc::sigslice::operator>>=(	/* assignment op. #11 */
	unsigned int	Var)
{

Notify();
_specc::bufslice::operator>>=(Var);

return *this;

} /* end of _specc::sigslice::operator>>= */


inline
_specc::sigslice &_specc::sigslice::operator++()	/* incr.op. #1 (pre-) */
{

Notify();
_specc::bufslice::operator++();

return *this;

} /* end of _specc::sigslice::operator++ */


inline
_specc::sigslice _specc::sigslice::operator++(int)	 /* incr.op. #2(post-)*/
{
sigslice	Tmp(*this);

Notify();
_specc::bufslice::operator++(1);

return Tmp;

} /* end of _specc::sigslice::operator++ */


inline
_specc::sigslice &_specc::sigslice::operator--()	/* decr.op. #1 (pre-) */
{

Notify();
_specc::bufslice::operator--();

return *this;

} /* end of _specc::sigslice::operator-- */


inline
_specc::sigslice _specc::sigslice::operator--(int)	/* decr.op. #2(post-)*/
{
sigslice	Tmp(*this);

Notify();
_specc::bufslice::operator--(1);

return Tmp;

} /* end of _specc::sigslice::operator-- */


inline
_specc::sigslice _specc::sigslice::sig_slice(	/* single bit of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	i)
{

return _specc::sigslice(SigEventPtr,
			CurrSlice, NextSlice,
			_BITMAP(l, r, i), _BITMAP(l, r, i),
			true,	/* single bit is always unsigned */
			(BusSlices ? ((sigbus_elem*)BusSlices)->slice(
							_BITMAP(l, r, i),
							_BITMAP(l, r, i))
					: NULL));

} /* end of _specc::sigslice::sig_slice */


inline
_specc::sigslice _specc::sigslice::sig_slice(	/* slice of a bitvector */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr)
{

return _specc::sigslice(SigEventPtr,
			CurrSlice, NextSlice,
			_BITMAP(l, r, sl), _BITMAP(l, r, sr),
			CurrSlice.isUnsigned(),
			(BusSlices ? ((sigbus_elem*)BusSlices)->slice(
							_BITMAP(l, r, sl),
							_BITMAP(l, r, sr))
					: NULL));

} /* end of _specc::sigslice::sig_slice */


inline
const _specc::sigslice _specc::sigslice::sig_slice(	/* const bit of vec.*/
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	i) const
{

return _specc::sigslice(SigEventPtr,
			CurrSlice, NextSlice,
			_BITMAP(l, r, i), _BITMAP(l, r, i),
			true,	/* single bit is always unsigned */
			(BusSlices ? ((sigbus_elem*)BusSlices)->slice(
							_BITMAP(l, r, i),
							_BITMAP(l, r, i))
					: NULL));

} /* end of _specc::sigslice::sig_slice */


inline
const _specc::sigslice _specc::sigslice::sig_slice(	/* const slice */
	_bit::bnd_t	l,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr) const
{

return _specc::sigslice(SigEventPtr,
			CurrSlice, NextSlice,
			_BITMAP(l, r, sl), _BITMAP(l, r, sr),
			CurrSlice.isUnsigned(),
			(BusSlices ? ((sigbus_elem*)BusSlices)->slice(
							_BITMAP(l, r, sl),
							_BITMAP(l, r, sr))
					: NULL));

} /* end of _specc::sigslice::sig_slice */


inline
void _specc::sigslice::Notify(void) const	/* notify a signal slice */
{

if (BusSlices)					// if this is a bus,
   { ((sigbus_elem*)BusSlices)->Notify();	// notify all connected slices
    } /* fi */					// (which fall into this slice);
else						// otherwise,
   { SigEventPtr->_specc::event::Notify();	// notify the event associated
    } /* esle */				// with this signal (slice)

} /* end of _specc::sigslice::Notify */


	/**********************/
	/*** _specc::sigbus ***/
	/**********************/


template <_bit::len_t len, bool usign> inline
_specc::sigbus<len,usign>::sigbus(	/* constructor #1 */
	sigbus_elem	*Slices)
{

_specc::sigbus<len,usign>::Slices = Slices;

} /* end of _specc::sigbus::sigbus */


template <_bit::len_t len, bool usign> inline
_specc::sigbus<len,usign>::~sigbus(	/* destructor */
	void)
{

delete Slices;

} /* end of _specc::sigbus::~sigbus */

//PC 05/02/04
template<_bit::len_t len, bool usign> inline
bool _specc::sigbus<len,usign>::IsValidEdge(	/* signal edge event filter */
	_specc::EDGE	Edge)		/* (overrides event and signal class) */
{

// note: for signal busses, we need special Put() and Get() equivalents
//       which provide us with the actual CurrValue and NextValue data;
//       also, for performance reasons, we only want to Get/Put the data
//       if we actually need it;
//       so, the following code is the manually merged and optimized result
//       of Get(), Put(), and _signal_ValidEdge() (see at top) methods
//       for this special case of signal busses (of bitvectors)

if (Edge == _specc::EDGE_ANY)
   { return(true);	/* every event is valid */
    } /* fi */

Slices->Get(this->CurrValue);

if (Edge == _specc::EDGE_RISING)
   { if (this->CurrValue.test())
	{ return(false);
	 } /* fi */
     else
	{ Slices->GetNextValue(this->NextValue);
	  return(this->NextValue.test());
	 } /* esle */
    } /* fi */
else
   { assert(Edge == _specc::EDGE_FALLING);
     if (this->CurrValue.test())
	{ Slices->GetNextValue(this->NextValue);
	  return(!this->NextValue.test());
	 } /* fi */
     else
	{ return(false);
	 } /* esle */
    } /* esle */

} /* end of _specc::sigbus<len,usign>::IsValidEdge */


template<_bit::len_t len, bool usign> inline
bool _specc::sigbus<len,usign>::ResetIsActive(	/* asynchronous reset check */
	bool	ResetActiveHi)		/* (overrides event and signal class) */
{

// note: this is basically the same as above, see note above!

Slices->GetNextValue(this->NextValue);

if (ResetActiveHi)
   { return(this->NextValue.test());
    } /* fi */
else
   { return(!this->NextValue.test());
    } /* esle */

} /* end of _specc::sigbus<len,usign>::ResetIsActive */


template <_bit::len_t len, bool usign> inline
_specc::sigslice _specc::sigbus<len,usign>::sig_slice(	/* single bit */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	i)
{

return _specc::sigslice(this, this->CurrValue, this->NextValue,
			_BITMAP(ll, r, i), _BITMAP(ll, r, i),
			true,	/* single bit is always unsigned */
			Slices->slice(_BITMAP(ll, r, i), _BITMAP(ll, r, i)));

} /* end of _specc::sigbus<len,usign>::sig_slice */


template <_bit::len_t len, bool usign> inline
_specc::sigslice _specc::sigbus<len,usign>::sig_slice(	/* slice */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr)
{

return _specc::sigslice(this, this->CurrValue, this->NextValue,
			_BITMAP(ll, r, sl), _BITMAP(ll, r, sr),
			usign,
			Slices->slice(_BITMAP(ll, r, sl), _BITMAP(ll, r, sr)));

} /* end of _specc::sigbus<len,usign>::sig_slice */


template <_bit::len_t len, bool usign> inline
const _specc::sigslice _specc::sigbus<len,usign>::sig_slice(	/* const bit */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	i) const
{

return _specc::sigslice(const_cast<_specc::sigbus<len, usign>*>(this),
			this->CurrValue, this->NextValue,
			_BITMAP(ll, r, i), _BITMAP(ll, r, i),
			true,	/* single bit is always unsigned */
			Slices->slice(_BITMAP(ll, r, i), _BITMAP(ll, r, i)));

} /* end of _specc::sigbus<len,usign>::sig_slice */


template <_bit::len_t len, bool usign> inline
const _specc::sigslice _specc::sigbus<len,usign>::sig_slice( /* const slice */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr) const
{

return _specc::sigslice(const_cast<_specc::sigbus<len, usign>*>(this),
			this->CurrValue, this->NextValue,
			_BITMAP(ll, r, sl), _BITMAP(ll, r, sr),
			usign,
			Slices->slice(_BITMAP(ll, r, sl), _BITMAP(ll, r, sr)));

} /* end of _specc::sigbus<len,usign>::sig_slice */


template <_bit::len_t len, bool usign> inline
_specc::sigbus_elem* _specc::sigbus<len,usign>::sigbus_slice( /* bus slice #1 */
	_bit::len_t	sl,
	_bit::len_t	sr)
{

return(Slices->slice(sl, sr));

} /* end of _specc::sigbus<len,usign>::sigbus_slice #1 */


template <_bit::len_t len, bool usign> inline
_specc::sigbus_elem* _specc::sigbus<len,usign>::sigbus_slice( /* bus slice #2 */
	_bit::bnd_t	ll,
	_bit::bnd_t	r,
	_bit::bnd_t	sl,
	_bit::bnd_t	sr)
{

return(sigbus_slice(_BITMAP(ll,r,sl), _BITMAP(ll,r,sr)));

} /* end of _specc::sigbus<len,usign>::sigbus_slice #2 */


template <_bit::len_t len, bool usign> inline
void _specc::sigbus<len,usign>::Put(void) const	/* store value for sigbus */
{						/* (override dummy method) */

Slices->Put(this->NextValue);

} /* end of _specc::sigbus<len,usign>::Put */


template <_bit::len_t len, bool usign> inline
void _specc::sigbus<len,usign>::Get(void) const	/* load value for sigbus */
{						/* (override dummy method) */

Slices->Get(this->CurrValue);

} /* end of _specc::sigbus<len,usign>::Get */


template <_bit::len_t len, bool usign> inline
void _specc::sigbus<len,usign>::Notify(void)	/* notify all bus events */
{						/* (override event method) */

Slices->Notify();		// instead of this bus signal,
				// notify all connected slices

} /* end of _specc::sigbus<len,usign>::Notify */

//PC 05/02/04
template <_bit::len_t len, bool usign> inline
bool _specc::sigbus<len,usign>::IsNotified(void)/* check if this is notified */
{						/* (override event method) */
_specc::sigbus_elem	*Slice;

// the event of this bus is never used
assert(this->Notified == false);

// instead, we need to check the events of all mapped signals
Slice = Slices;
while(Slice)
   { if (Slice->EventPtr->Notified)	// we know what we're doing here, so
	{ return(true);			// we'll access the member directly
	 } /* fi */
     Slice = (_specc::sigbus_elem*) Slice->Next;
    } /* elihw */

return(false);

} /* end of _specc::sigbus<len,usign>::IsNotified */


template <_bit::len_t len, bool usign> inline
_specc::event_ptr *_specc::sigbus<len,usign>::MappedEvents(
	void)					/* get port-mapped events */
{						/* (override event method) */
_specc::sigbus_elem	*Slice;
_specc::event_ptr	*NewList,
			**NextPtr;

NewList = NULL;
NextPtr = &NewList;
Slice = Slices;
while(Slice)
   { *NextPtr = new _specc::event_ptr(Slice->EventPtr);
     NextPtr = &(*NextPtr)->Next;
     Slice = (_specc::sigbus_elem*) Slice->Next;
    } /* elihw */

return(NewList);

} /* end of _specc::sigbus<len,usign>::MappedEvents */


	/***************************/
	/*** _specc::sigbus_elem ***/
	/***************************/


inline
_specc::sigbus_elem::sigbus_elem(	/* constructor #1 */
	const _bit	&CurrVal,
	const _bit	&NextVal,
	_bit::len_t	sl,
	_bit::len_t	sr,
	_specc::event	*EventPtr,
	sigbus_elem	*Next /* = NULL */)
	: _specc::bufbus_elem(CurrVal, NextVal, sl, sr, Next)
{

_specc::sigbus_elem::EventPtr = EventPtr;

} /* end of _specc::sigbus_elem::sigbus_elem */


inline
_specc::sigbus_elem::~sigbus_elem(	/* destructor */
	void)
{

delete ((_specc::sigbus_elem*)Next);
Next = NULL;		// don't let base class delete these again!

} /* end of _specc::sigbus_elem::~sigbus_elem */


inline
_specc::sigbus_elem *_specc::sigbus_elem::concat(	/* concatenation */
	_specc::sigbus_elem	*List)
{

_specc::bufbus_elem::concat(List);

return List;

} /* end of _specc::sigbus_elem::concat */


inline
_specc::sigbus_elem *_specc::sigbus_elem::slice(	/* slicing */
	_bit::len_t	l,
	_bit::len_t	r)
{
_bit::len_t		i,
			NewSl,
			NewSr,
			ElemLen,
			Max,
			Min;
_specc::sigbus_elem	*Head,
			**Tmp,
			*Elem;

// note: this is equivalent to _bitbus_element::slice() in file "bit.cc"

assert(CurrSlice.sl == NextSlice.sl);
assert(CurrSlice.sr == NextSlice.sr);

ElemLen = _BITLEN(CurrSlice.sl, CurrSlice.sr);
Max = _MAXLEN(l,r);
Min = _MINLEN(l,r);
Head = NULL;
Tmp = &Head;

// Walk through the list until we hit parts of the slice

for(Elem = this, i = ElemLen; Min >= i; i += ElemLen)
   { Elem = (_specc::sigbus_elem*) Elem->Next;
     ElemLen = _BITLEN(Elem->CurrSlice.sl, Elem->CurrSlice.sr);
    } /* rof */
i -= ElemLen;

// Copy and transform elements until we leave slice

for( ; i <= Max; i += ElemLen)
   { // Map slice coordinates
     if (Elem->CurrSlice.sl > Elem->CurrSlice.sr)
	{ NewSl = Elem->CurrSlice.sl - _MAXLEN(i + ElemLen - 1 - Max, 0);
	  NewSr = Elem->CurrSlice.sr + _MAXLEN(Min - i, 0);
	 } /* fi */
     else
	{ NewSl = Elem->CurrSlice.sl + _MAXLEN(i + ElemLen - 1 - Max, 0);
	  NewSr = Elem->CurrSlice.sr - _MAXLEN(Min - i, 0);
	 } /* esle */
     if (l > r)
	{ // Add elements at tail of list
	  *Tmp = new _specc::sigbus_elem(Elem->CurrSlice.ref,
						Elem->NextSlice.ref,
						NewSl, NewSr,
						Elem->EventPtr);
	  Tmp = (_specc::sigbus_elem**)(&((*Tmp)->Next));
	 } /* fi */
     else
	{ // Add elements at the head of list
	  Head = new _specc::sigbus_elem(Elem->CurrSlice.ref,
						Elem->NextSlice.ref,
						NewSr, NewSl,
						Elem->EventPtr, Head);
	 } /* esle */
     Elem = (_specc::sigbus_elem*) Elem->Next;
     ASSERT(Elem || ((i+ElemLen) > Max)); // Tried to slice outside of range...
     if (!Elem)
	{ break;
	 } /* fi */
     ElemLen = _BITLEN(Elem->CurrSlice.sl, Elem->CurrSlice.sr);
    } /* rof */

ASSERT(Head);	// Did we cut anything?

return(Head);

} /* end of _specc::sigbus_elem::slice */


inline
void _specc::sigbus_elem::Notify(void)	/* notify all events in this list */
{
_specc::sigbus_elem	*Elem;

Elem = this;
while(Elem)
   { Elem->EventPtr->_specc::event::Notify();
     Elem = (_specc::sigbus_elem*) Elem->Next;
    } /* elihw */

} /* end of _specc::sigbus_elem::Notify */


#endif /* __SIGNALS_H */

/* EOF signals.h */
