// ============================================================================
// longlong.h: "long long" data type support
// ============================================================================
// 
// 04/08/02 <doemer>	added typedefs _LONGLONG and _ULONGLONG
// 11/28/01 <gerstl>	Removed ambiguities, move more casting into scc
// 05/24/01 <gerstl>	Initial version
//
// ----------------------------------------------------------------------------

#ifdef FIRST_PASS
_CPP_ifndef __LONGLONG
_CPP_define __LONGLONG
#endif

#ifndef __LONGLONG
#define __LONGLONG
  

#ifdef HAVE_LLONG

/* ------------------------------------------------------------------------- */
/* Native (built-in) long long data type */

#ifdef FIRST_PASS
_CPP_include <limits.h>

#undef HAVE_LLONG
// Native long long version of library
_CPP_undef  HAVE_LLONG  
_CPP_define HAVE_LLONG  
#define HAVE_LLONG

// Provide constants
_CPP_ifndef LONG_LONG_MAX
_CPP_ifdef LLONG_MAX
_CPP_define LONG_LONG_MAX     LLONG_MAX    // SOLARIS et al.
_CPP_elif defined(QUAD_MAX)
_CPP_define LONG_LONG_MAX     QUAD_MAX     // NETBSD
_CPP_else
_CPP_define LONG_LONG_MAX     0x7fffffffffffffffLL
_CPP_endif
_CPP_endif
  
_CPP_ifndef ULONG_LONG_MAX
_CPP_ifdef ULLONG_MAX
_CPP_define ULONG_LONG_MAX    ULLONG_MAX   // SOLARIS et al.
_CPP_elif defined(UQUAD_MAX)
_CPP_define ULONG_LONG_MAX    UQUAD_MAX    // NETBSD
_CPP_else
_CPP_define ULONG_LONG_MAX    0xffffffffffffffffULL
_CPP_endif
_CPP_endif

_CPP_ifndef LLONG_MAX
_CPP_define LLONG_MAX         LONG_LONG_MAX
_CPP_endif

_CPP_ifndef ULLONG_MAX
_CPP_define ULLONG_MAX        ULONG_LONG_MAX
_CPP_endif
#endif
  

/* --- Provide typedefs --- */

typedef long long             _LONGLONG;
typedef unsigned long long    _ULONGLONG;

  
/* ---  Global functions --- */
  
extern "C" {
  
// conversion of long long to string (see sim.sh)  
char *ll2str(unsigned int base, char *endptr, long long ll);
char *ull2str(unsigned int base, char *endptr, unsigned long long ull);

// conversion of string to long long (see sim.sh)
long long str2ll(unsigned int base, const char* str);
unsigned long long str2ull(unsigned int base, const char* str);
    
}

#else

/* ------------------------------------------------------------------------- */
/* Non-native long long data type support */

#ifdef FIRST_PASS
#ifdef _NO_LONGLONG
#undef _NO_LONGLONG
// Disable native long long on Solaris
_CPP_undef   _NO_LONGLONG
_CPP_define  _NO_LONGLONG     
#define _NO_LONGLONG
#endif

// Non-native long long version of library    
_CPP_undef  HAVE_LLONG        

  
_CPP_include <limits.h>
    
// Provide constants
_CPP_undef  LONG_LONG_MAX
_CPP_define LONG_LONG_MAX    (_longlong<false>(LONG_MAX, ULONG_MAX))
_CPP_undef  ULONG_LONG_MAX
_CPP_define ULONG_LONG_MAX   (_longlong<true>(ULONG_MAX, ULONG_MAX))
    
_CPP_undef  LLONG_MAX    
_CPP_define LLONG_MAX         LONG_LONG_MAX
_CPP_undef  ULLONG_MAX    
_CPP_define ULLONG_MAX        ULONG_LONG_MAX
    
#else
#include <limits.h>
#endif
  
  
/* --- Forward declaration of signed/unsigned class --- */

template<bool> class _longlong;
template<bool> class _longlong_;


/* --- Provide typedefs --- */

typedef _longlong<false>      _LONGLONG;
typedef _longlong<true>       _ULONGLONG;


/* --- Global functions --- */

extern "C" {
  
// conversion of long long to string (see sim.sh)  
char *ll2str(unsigned int base, char *endptr, _longlong<false> ll);
char *ull2str(unsigned int base, char *endptr, _longlong<true> ull);

// conversion of string to long long (see sim.sh)
_longlong<false> str2ll(unsigned int base, const char* str);
_longlong<true> str2ull(unsigned int base, const char* str);
    
}


/* ------------------------------------------------------------------------- */
/* Class for type "long long", signed/unsigned */


/* --- Base class, long long representation --- */

class ___longlong
{
protected:
  // long long representation
  union {
    unsigned long u0;
    signed   long s0;
  };
  union {
    unsigned long u1;
    signed   long s1;
  };
  
public:  
  // Test
  bool test(void) const                                  { return u0 || u1; }
  
  // Explicit conversions
  int toInt(void) const         	                       { return s0; }
  unsigned int toUInt(void) const	                       { return u0; }
  long toLong(void) const   		                       { return s0; }
  unsigned long toULong(void) const 	                       { return u0; }
  double toDouble(void) const {
    return (double)u0 + ((double)s1 * ((double)ULONG_MAX + 1));
  }    	
  long double toLDouble(void) const {
    return (long double)u0 + ((long double)s1 * ((long double)ULONG_MAX + 1));
  }
#ifdef DEBUG
#if defined(LLONG_MAX) || defined(LONG_LONG_MAX) || defined(QUAD_MAX)
  unsigned long long toULLong() const { 
    return (unsigned long long)u0 + ((unsigned long long)u1 << sizeof(long)*CHAR_BIT);
  }
  long long toLLong() const { 
    return (long long)u0 + ((long long)s1 << sizeof(long)*CHAR_BIT);
  }
#endif
#endif
};


/* --- Working long long class --- */

class __longlong: public ___longlong
{
public:
  // Initialization / Construction
  __longlong()                                                             { }
  __longlong(unsigned long hi, unsigned long lo)         { u1 = hi; u0 = lo; }
  __longlong(const char* s, unsigned int base = 0)	     { set(s, base); }
  __longlong(int i)                                             { *this = i; }
  __longlong(unsigned int i)                                    { *this = i; }
  __longlong(long i)                                            { *this = i; }
  __longlong(unsigned long i)                                   { *this = i; }
  __longlong(double d, bool usgn)                               { *this = d; }
  __longlong(long double d, bool usgn)                          { *this = d; }
  
  // Assignement
  __longlong& operator=(const char* s)       { set(s);                   return *this; }
  __longlong& operator=(int i)               { s0 = i; s1 = i<0? -1 : 0; return *this; }
  __longlong& operator=(unsigned int i)      { u0 = i; u1 = 0;           return *this; }
  __longlong& operator=(long i)              { s0 = i; s1 = i<0? -1 : 0; return *this; }
  __longlong& operator=(unsigned long i)     { u0 = i; u1 = 0;           return *this; }
  __longlong& operator=(double d);
  __longlong& operator=(long double d);
   
  // Arithmetic
  inline __longlong& operator+=(const __longlong& ll);
  inline __longlong& operator-=(const __longlong& ll);
  inline __longlong& operator*=(const __longlong& ll);
  inline __longlong& operator/=(const __longlong& ll);
  inline __longlong& operator%=(const __longlong& ll);
    
  // Increment & decrement
  inline __longlong& operator++();
  inline __longlong& operator--();
  
  // Logic
  inline __longlong& operator&=(const __longlong& ll);
  inline __longlong& operator|=(const __longlong& ll);
  inline __longlong& operator^=(const __longlong& ll);
  
  // Shifting
  inline __longlong& operator<<=(unsigned int sh);
  
  // Comparison
  friend bool operator==(const __longlong&, const __longlong&);
  friend bool operator!=(const __longlong&, const __longlong&);
  friend bool operator> (const _longlong<true>&, const _longlong<true>&);
  friend bool operator>=(const _longlong<true>&, const _longlong<true>&);
  friend bool operator< (const _longlong<true>&, const _longlong<true>&);
  friend bool operator<=(const _longlong<true>&, const _longlong<true>&);
  friend bool operator> (const _longlong<false>&, const _longlong<false>&);
  friend bool operator>=(const _longlong<false>&, const _longlong<false>&);
  friend bool operator< (const _longlong<false>&, const _longlong<false>&);
  friend bool operator<=(const _longlong<false>&, const _longlong<false>&);
  
protected:  
  // negation
  inline __longlong&   neg();

  // string assignement
  void          set( const char* str, unsigned int base = 0 );
  
  // multiplication helper
  void          mul( const __longlong& ll );
  
  // division helpers
  unsigned long div( const __longlong& ll );
  unsigned long calc_q( unsigned long n1, unsigned long n2,
		        unsigned long d1, unsigned long d2 );
  void          mul_sub( unsigned long& q,
			 unsigned long d1, unsigned long d2, 
			 unsigned long d3, unsigned long d4 );
};



/* --- signed/unsigned long long --- */

template<bool usign=false> class _longlong : public __longlong
{
public:  
  // Initialization / Construction
  _longlong()                                    : __longlong()         { }
  _longlong(unsigned long hi, unsigned long lo)  : __longlong(hi, lo)   { }
  _longlong(const char* s, unsigned int base = 0): __longlong(s, base)  { }
  _longlong(const __longlong& ll)                : __longlong(ll)       { }
  _longlong(int i)                               : __longlong(i)        { }
  _longlong(unsigned int i)                      : __longlong(i)        { }
  _longlong(long i)                              : __longlong(i)        { }
  _longlong(unsigned long i)                     : __longlong(i)        { }
  _longlong(double d, bool usgn)                 : __longlong(d, usgn)  { }
  _longlong(long double d, bool usgn)            : __longlong(d, usgn)  { }
  
  // Type-true assignments
  _longlong& operator=(const char* s)         { return (_longlong&)__longlong::operator=(s);  }
  _longlong& operator=(const __longlong& ll)  { return (_longlong&)__longlong::operator=(ll); }
  _longlong& operator=(const _longlong_<usign>& ll)  
                                              { return (_longlong&)__longlong::operator=(ll); }
  _longlong& operator=(int i)                 { return (_longlong&)__longlong::operator=(i);  }
  _longlong& operator=(unsigned int i)        { return (_longlong&)__longlong::operator=(i);  }
  _longlong& operator=(long i)                { return (_longlong&)__longlong::operator=(i);  }
  _longlong& operator=(unsigned long i)       { return (_longlong&)__longlong::operator=(i);  }
  _longlong& operator=(double d)              { return (_longlong&)__longlong::operator=(d);  }
  _longlong& operator=(long double d)         { return (_longlong&)__longlong::operator=(d);  }

  // Type-true arithmetic
  _longlong& operator+=(const __longlong& ll) { return (_longlong&)__longlong::operator+=(ll); }
  _longlong& operator-=(const __longlong& ll) { return (_longlong&)__longlong::operator-=(ll); }
  _longlong& operator*=(const __longlong& ll) { return (_longlong&)__longlong::operator*=(ll); }
  inline _longlong& operator/=(const __longlong& ll);
  inline _longlong& operator%=(const __longlong& ll);
  template<bool us>
    inline _longlong& operator/=(const _longlong<us>& ll);
  template<bool us>
    inline _longlong& operator%=(const _longlong<us>& ll);
  inline const _longlong  operator-() const;
  inline const _longlong& operator+() const; 
  
  // Type-true increment & decrement
  _longlong& operator++()                     { return (_longlong&)__longlong::operator++(); }
  _longlong& operator--()                     { return (_longlong&)__longlong::operator--(); }
  inline _longlong operator++(int);
  inline _longlong operator--(int);
  
  // Type-true logic
  _longlong& operator&=(const __longlong& ll) { return (_longlong&)__longlong::operator&=(ll); }
  _longlong& operator|=(const __longlong& ll) { return (_longlong&)__longlong::operator|=(ll); }
  _longlong& operator^=(const __longlong& ll) { return (_longlong&)__longlong::operator^=(ll); }
  inline _longlong operator~() const;
  
  // Type-true shift
  inline _longlong& operator>>=(unsigned int sh);  
  inline _longlong& operator<<=(unsigned int sh) { return (_longlong&)__longlong::operator<<=(sh); }
};




/* --- Shadow class for long long inside a union --- */

template<bool usign=false> class _longlong_: public ___longlong
{
public:
  // No constructors allowed inside a union!
  
  // Cast to real, working _longlong class
  operator const _longlong<usign>& (void) const {
    return *reinterpret_cast<const _longlong<usign>*>(this);
  }

  // assignements
  _longlong<usign>& operator=(const char* s)        { return (_longlong<usign>&)(*this) = s; }
  _longlong<usign>& operator=(const __longlong& ll) { return (_longlong<usign>&)(*this) = ll; }
  _longlong<usign>& operator=(int i)                { return (_longlong<usign>&)(*this) = i; }
  _longlong<usign>& operator=(unsigned int i)       { return (_longlong<usign>&)(*this) = i; }
  _longlong<usign>& operator=(long i)               { return (_longlong<usign>&)(*this) = i; }
  _longlong<usign>& operator=(unsigned long i)      { return (_longlong<usign>&)(*this) = i; }
  _longlong<usign>& operator=(double d)             { return (_longlong<usign>&)(*this) = d; }
  _longlong<usign>& operator=(long double d)        { return (_longlong<usign>&)(*this) = d; }

  // arithmetic
  _longlong<usign>& operator+=(const __longlong& ll) { return (_longlong<usign>&)(*this) += ll; }
  _longlong<usign>& operator-=(const __longlong& ll) { return (_longlong<usign>&)(*this) -= ll; }
  _longlong<usign>& operator*=(const __longlong& ll) { return (_longlong<usign>&)(*this) *= ll; }
  _longlong<usign>& operator/=(const __longlong& ll) { return (_longlong<usign>&)(*this) /= ll; }
  _longlong<usign>& operator%=(const __longlong& ll) { return (_longlong<usign>&)(*this) %= ll; }
  template<bool us>
    _longlong<usign>& operator/=(const _longlong<us>& ll)  { return (_longlong<usign>&)(*this) /= ll; }
  template<bool us>
    _longlong<usign>& operator%=(const _longlong<us>& ll)  { return (_longlong<usign>&)(*this) %= ll; }
  template<bool us>
    _longlong<usign>& operator/=(const _longlong_<us>& ll) { return (_longlong<usign>&)(*this) /= (_longlong<us>&)ll; }
  template<bool us>
    _longlong<usign>& operator%=(const _longlong_<us>& ll) { return (_longlong<usign>&)(*this) %= (_longlong<us>&)ll; }
  
  _longlong<usign>  operator-() const { return -((_longlong<usign>&)(*this)); }
  _longlong<usign>& operator+() const { return +((_longlong<usign>&)(*this)); }
  
  // increment & decrement
  _longlong<usign>& operator++()     { return ++((_longlong<usign>&)(*this)); }
  _longlong<usign>& operator--()     { return --((_longlong<usign>&)(*this)); }
  _longlong<usign> operator++(int)   { return ((_longlong<usign>&)(*this))++; }
  _longlong<usign> operator--(int)   { return ((_longlong<usign>&)(*this))--; }
  
  // logic
  _longlong<usign>& operator&=(const __longlong& ll) { return (_longlong<usign>&)(*this) &= ll; }
  _longlong<usign>& operator|=(const __longlong& ll) { return (_longlong<usign>&)(*this) |= ll; }
  _longlong<usign>& operator^=(const __longlong& ll) { return (_longlong<usign>&)(*this) ^= ll; }
  _longlong<usign> operator~() const { return ~((_longlong<usign>&)(*this)); }
  
  // shift
  _longlong<usign>& operator>>=(unsigned int sh) { return (_longlong<usign>&)(*this) >>= sh; }
  _longlong<usign>& operator<<=(unsigned int sh) { return (_longlong<usign>&)(*this) <<= sh; }  
};



/* ------------------------------------------------------------------------- */
/* External operators */


/* --- Addition --- */

// signed
inline _longlong<false> operator+( const _longlong<false>& ll1, const _longlong<false>& ll2 ) {
  _longlong<false> ll( ll1 );
  return ll += ll2;
}
  
// unsigned
inline _longlong<true> operator+( const _longlong<true>& ll1, const _longlong<true>& ll2 ) {
  _longlong<true> ll( ll1 );
  return ll += ll2;
}


/* --- Subtraction --- */

// signed
inline _longlong<false> operator-( const _longlong<false>& ll1, const _longlong<false>& ll2 ) {
  _longlong<false> ll( ll1 );
  return ll -= ll2;
}

// unsigned
inline _longlong<true> operator-( const _longlong<true>& ll1, const _longlong<true>& ll2 ) {
  _longlong<true> ll( ll1 );
  return ll -= ll2;
}


/* --- Multiplication --- */

// signed
inline _longlong<false> operator*( const _longlong<false>& ll1, const _longlong<false>& ll2 ) {
  _longlong<false> ll( ll1 );
  return ll *= ll2;
}

// unsigned
inline _longlong<true> operator*( const _longlong<true>& ll1, const _longlong<true>& ll2 ) {
  _longlong<true> ll( ll1 );
  return ll *= ll2;
}


/* --- Division --- */

// signed
inline _longlong<false> operator/( const _longlong<false>& ll1, const _longlong<false>& ll2 ) {
  _longlong<false> ll( ll1 );
  return ll /= ll2;
}

// unsigned
inline _longlong<true> operator/( const _longlong<true>& ll1, const _longlong<true>& ll2 ) {
  _longlong<true> ll( ll1 );
  return ll /= ll2;
}


/* --- Modulo */

// signed
inline _longlong<false> operator%( const _longlong<false>& ll1, const _longlong<false>& ll2 ) {
  _longlong<false> ll( ll1 );
  return ll %= ll2;
}

// unsigned
inline _longlong<true> operator%( const _longlong<true>& ll1, const _longlong<true>& ll2 ) {
  _longlong<true> ll( ll1 );
  return ll %= ll2;
}


/* --- And */

// signed
inline _longlong<false> operator&( const _longlong<false>& ll1, const _longlong<false>& ll2 ) {
  _longlong<false> ll( ll1 );
  return ll &= ll2;
}

// unsigned
inline _longlong<true> operator&( const _longlong<true>& ll1, const _longlong<true>& ll2 ) {
  _longlong<true> ll( ll1 );
  return ll &= ll2;
}


/* --- Or */

// signed
inline _longlong<false> operator|( const _longlong<false>& ll1, const _longlong<false>& ll2 ) {
  _longlong<false> ll( ll1 );
  return ll |= ll2;
}

// unsigned
inline _longlong<true> operator|( const _longlong<true>& ll1, const _longlong<true>& ll2 ) {
  _longlong<true> ll( ll1 );
  return ll |= ll2;
}


/* --- Xor */

// signed
inline _longlong<false> operator^( const _longlong<false>& ll1, const _longlong<false>& ll2 ) {
  _longlong<false> ll( ll1 );
  return ll ^= ll2;
}

// unsigned
inline _longlong<true> operator^( const _longlong<true>& ll1, const _longlong<true>& ll2 ) {
  _longlong<true> ll( ll1 );
  return ll ^= ll2;
}


/* --- Shift */

// signed
inline _longlong<false> operator<<( const _longlong<false>& ll, unsigned int sh ) {
  _longlong<false> r( ll );
  return r <<= sh;
}

inline _longlong<false> operator>>( const _longlong<false>& ll, unsigned int sh ) {
  _longlong<false> r( ll );
  return r >>= sh;
}

// unsigned
inline _longlong<true> operator<<( const _longlong<true>& ll, unsigned int sh ) {
  _longlong<true> r( ll );
  return r <<= sh;
}

inline _longlong<true> operator>>( const _longlong<true>& ll, unsigned int sh ) {
  _longlong<true> r( ll );
  return r >>= sh;
}


/* --- Comparison */

inline bool operator==( const __longlong& a, const __longlong& b ) {
  return (a.u0 == b.u0) && (a.u1 == b.u1);
}

inline bool operator!=( const __longlong& a, const __longlong& b ) {
  return (a.u0 != b.u0) || (a.u1 != b.u1);
}


// signed
inline bool operator>( const _longlong<false>& a, const _longlong<false>& b ) {
  if( a.s1 > b.s1 ) return true;
  if( a.s1 < b.s1 ) return false;
  return (a.u0 > b.u0);
}

inline bool operator>=( const _longlong<false>& a, const _longlong<false>& b ) {
  if( a.s1 > b.s1 ) return true;
  if( a.s1 < b.s1 ) return false;
  return (a.u0 >= b.u0);
}

inline bool operator<( const _longlong<false>& a, const _longlong<false>& b ) {
  if( a.s1 < b.s1 ) return true;
  if( a.s1 > b.s1 ) return false;
  return (a.u0 < b.u0);
}

inline bool operator<=( const _longlong<false>& a, const _longlong<false>& b ) {
  if( a.s1 < b.s1 ) return true;
  if( a.s1 > b.s1 ) return false;
  return (a.u0 <= b.u0);
}

// unsigned
inline bool operator>( const _longlong<true>& a, const _longlong<true>& b ) {
  if( a.u1 > b.u1 ) return true;
  if( a.u1 < b.u1 ) return false;
  return (a.u0 > b.u0);
}

inline bool operator>=( const _longlong<true>& a, const _longlong<true>& b ) {
  if( a.u1 > b.u1 ) return true;
  if( a.u1 < b.u1 ) return false;
  return (a.u0 >= b.u0);
}

inline bool operator<( const _longlong<true>& a, const _longlong<true>& b ) {
  if( a.u1 < b.u1 ) return true;
  if( a.u1 > b.u1 ) return false;
  return (a.u0 < b.u0);
}

inline bool operator<=( const _longlong<true>& a, const _longlong<true>& b ) {
  if( a.u1 < b.u1 ) return true;
  if( a.u1 > b.u1) return false;
  return (a.u0 <= b.u0);
}



/* ------------------------------------------------------------------------- */
/* Deferred inline operators */


/* --- Arithmetic --- */

// Addition
inline __longlong& __longlong::operator+=(const __longlong& ll) {
  u0 += ll.u0;
  u1 += ll.u1 + (u0 < ll.u0);
  return *this;
}

// Subtraction
inline __longlong& __longlong::operator-=(const __longlong& ll) {
  u1 -= ll.u1 + (ll.u0 > u0);
  u0 -= ll.u0;
  return *this;
}

// Multiplication
inline __longlong& __longlong::operator*=(const __longlong& ll) {
  mul( ll );
  return *this;
}

// Division
inline __longlong& __longlong::operator/=(const __longlong& ll) {
  u0 = div( ll );
  if( ll.u1 ) u1 = 0;    // clear any part of remainder in upper chunk
  return *this;
}

// Modulo
inline __longlong& __longlong::operator%=(const __longlong& ll) {
  div( ll );
  if( ! ll.u1 ) u1 = 0;  // clear any part of quotient left in upper chunk
  return *this;
}

// Type-true division
template<bool usign>
  inline _longlong<usign>& _longlong<usign>::operator/=(const __longlong& ll)
{    
  // Signed division?
  if( !usign )
  {
    const _longlong<usign>& op = ll;
    
    // Make operands positive
    bool nNeg = (s1 < 0);
    bool dNeg = (op < 0);
    if( nNeg ) neg();
    // Divide
    __longlong::operator/=( dNeg? -op : op );
    // Adjust result
    if( dNeg ^ nNeg ) neg();
    return *this;
  }
  return (_longlong<usign>&)__longlong::operator/=( ll );
}

template<bool usign> template<bool us>
  inline _longlong<usign>& _longlong<usign>::operator/=(const _longlong<us>& ll)
{
  if( !usign && !us ) 
    return operator/=( (__longlong&)ll );
  else
    return (_longlong<usign>&)__longlong::operator/=( ll );
}


// Type-true modulo
template<bool usign>
  inline _longlong<usign>& _longlong<usign>::operator%=(const __longlong& ll)
{
  // Signed division?
  if( !usign )
  {
    const _longlong<usign>& op = ll;
    
    // Make operands positive
    bool nNeg = (s1 < 0);
    bool dNeg = (op < 0);
    if( nNeg ) neg();
    // Divide
    __longlong::operator%=( dNeg? -op : op );
    // Adjust result
    if( nNeg ) neg();
    return *this;
  }
  return (_longlong<usign>&)__longlong::operator%=( ll );
}

template<bool usign> template<bool us>
  inline _longlong<usign>& _longlong<usign>::operator%=(const _longlong<us>& ll)
{
  if( !usign && !us ) 
    return operator%=( (__longlong&)ll );
  else
    return (_longlong<usign>&)__longlong::operator%=( ll );
}

// Negation
inline __longlong& __longlong::neg() {
  u1 = -u1 - (u0 != 0);
  u0 = -u0;
  return *this;
}


// Type-true negation
template<bool usign>
  inline const _longlong<usign> _longlong<usign>::operator-() const {
    _longlong<usign> ll( *this );
    return ll.neg();
}

// Type-true identity
template<bool usign>
  inline const _longlong<usign>& _longlong<usign>::operator+() const {
    return *this;
}


/* --- Logic --- */

// And
inline __longlong& __longlong::operator&=(const __longlong& ll) {
  u0 &= ll.u0;
  u1 &= ll.u1;
  return *this;
}

// Or
inline __longlong& __longlong::operator|=(const __longlong& ll) {
  u0 |= ll.u0;
  u1 |= ll.u1;
  return *this;
}

// Xor
inline __longlong& __longlong::operator^=(const __longlong& ll) {
  u0 ^= ll.u0;
  u1 ^= ll.u1;
  return *this;
}


// Inversion
template<bool usign>
  inline _longlong<usign> _longlong<usign>::operator~() const {
    _longlong<usign> ll;
    ll.u0 = ~u0;
    ll.u1 = ~u1;
    return ll;
}


/* --- Shifting --- */

// Left shift
inline __longlong& __longlong::operator<<=(unsigned int sh)
{
  if( sh >= (sizeof(long)*CHAR_BIT) ) {
    u1 = u0 << (sh - (sizeof(long)*CHAR_BIT));
    u0 = 0;
  } 
  else {
    u1 <<= sh;
    if( sh ) u1 |= (u0 >> (sizeof(long)*CHAR_BIT - sh));
    u0 <<= sh;
  }
  return *this;
}

// Right shift
template<bool usign>
  inline _longlong<usign>& _longlong<usign>::operator>>=(unsigned int sh) 
{
  if( sh >= (sizeof(long)*CHAR_BIT) ) {
    if( usign ) 
      u0 = u1 >> (sh - sizeof(long)*CHAR_BIT);
    else
      s0 = s1 >> (sh - sizeof(long)*CHAR_BIT);
    s1 = usign? 0 : s1 >>= (sizeof(long)*CHAR_BIT - 1);
  }
  else {
    u0 >>= sh;
    if( sh ) u0 |= (u1 << (sizeof(long)*CHAR_BIT - sh));
    if( usign ) u1 >>= sh; else s1 >>= sh;
  }
  return *this;
}



/* --- Increment & decrement --- */

// Prefix
inline __longlong& __longlong::operator++() {
  u1 += (++u0 == 0);
  return *this;
}

inline __longlong& __longlong::operator--() {
  u1 -= (u0-- == 0);
  return *this;
}

// Postfix
template<bool usign>
  inline _longlong<usign> _longlong<usign>::operator++(int) {
  _longlong<usign> ll( *this );
  this->operator++();
  return ll;
}

template<bool usign>
  inline _longlong<usign> _longlong<usign>::operator--(int) {
  _longlong<usign> ll( *this );
  this->operator--();
  return ll;
}


#endif

#ifdef FIRST_PASS
_CPP_endif
#endif
  
#endif
