/************************************************************************/
/* Constant.cc: SpecC Internal Representation, Constant Class		*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 09/24/97 */
/************************************************************************/

/* last update: 06/15/04 */

/* modifications: (most recent first)
 *
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 11/29/01 RD	provided isascii() macro for all platforms
 * 11/21/01 RD	took out default arguments from function definitions
 * 08/13/01 RD	switched type of bitvector length to unsigned
 * 05/31/01 RD	eliminated level 2 of SIR API
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 05/05/01 RD	print bitvector constants as template types
 * 04/30/01 RD	changed use of len member to length() method for bitvectors
 * 04/30/01 RD	replaced use of obsolete form() from "stream.h" with own one
 * 04/20/01 RD	adjusted bitvector constructor to changed arguments
 * 04/19/01 RD	changed bit2str to ubit2str to avoid leading '-'
 * 04/19/01 RD	eliminated an assumption about data location in unions
 * 04/19/01 RD	adjusted some arguments to "ambigous" bit constructor
 * 04/18/01 RD	fixed C++ generation for non-native 'long long' type
 * 04/18/01 RD	added missing length argument to Convert[ed]() functions
 * 04/17/01 RD	switched bitstring generation from toBin() to ubit2str()
 * 04/16/01 RD	fixed non-native output of (unsigned) long long values
 * 04/06/01 RD	added support for non-native 'long long' type
 * 04/06/01 RD	started this header (last update was 05/19/99)
 */

#include "IntRep/Constant.h"
#include "IntRep/Extern.h"
#include "bit.h"

#include <stdlib.h>
#include <assert.h>
#include <limits.h>
#include <string.h>
#include <ctype.h>


/*** constants and macros ***********************************************/


	/* precision of floating point values */

#ifndef FLT_DIG
#define FLT_DIG		6
#endif /* FLT_DIG */

#ifndef DBL_DIG
#define DBL_DIG		15
#endif /* DBL_DIG */

#ifndef LDBL_DIG
#define LDBL_DIG	24
#endif /* LDBL_DIG */


	/* maximum length of any kind of number in decimal format */

#define MAX_NUM_LENGTH	40	/* (large enough for all numbers) */


	/* options to output 'long long' types */

#ifdef HAVE_LLONG
//#define LONGLONG_USE_NATIVE_OUTPUT_ROUTINES
#define LONGLONG_USE_SPECC_OUTPUT_ROUTINES
#else /* !HAVE_LLONG */
#define LONGLONG_USE_SPECC_OUTPUT_ROUTINES
#endif /* HAVE_LLONG */


	/* support for non-ASCII characters */

// #define SUPPORT_KANJI	/* Japanese kanji encoding (2-bytes!) */
// #define SUPPORT_ISO		/* ISO encoding (codes 128 to 255) */
				/* (ISO includes EUC kanji encoding) */
#ifdef KANJI
#define SUPPORT_KANJI
#else
#define SUPPORT_ISO
#endif

#ifndef isascii			/* isascii() not available on some platforms */
#define isascii(i) ((unsigned char)(i) <= 127)
#endif /* isascii */


/*** internal type declarations *****************************************/


	/* (none) */


/*** internal function prototypes ***************************************/


static void SIR_ConvertLL(
	sir_constant	*Const,
	LONG_LONG	LL_Value,
	SIR_CONSTTYPE	Type);

static void SIR_ConvertULL(
	sir_constant		*Const,
	UNSIGNED_LONG_LONG	ULL_Value,
	SIR_CONSTTYPE		Type);

static void SIR_ConvertBIT(
	sir_constant	*Const,
	SIR_CONSTTYPE	Type);

static void SIR_ConvertLD(
	sir_constant	*Const,
	long double	LD_Value,
	SIR_CONSTTYPE	Type);


/*** class implementation ***********************************************/


	/********************/
	/*** SIR_Constant ***/
	/********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Constant::SIR_Constant(		/* constructor #1 */
	SIR_CONSTTYPE	Type,
	int		Value,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

SIR_Constant::Type = Type;

switch(Type)
   { case SIR_CONST_BOOL:
	{ B_Value = (Value ? true : false);
	  break;
	 }
     case SIR_CONST_CHAR:
	{ C_Value = Value;
	  break;
	 }
     case SIR_CONST_UCHAR:
	{ UC_Value = Value;
	  break;
	 }
     case SIR_CONST_SHORT:
	{ S_Value = Value;
	  break;
	 }
     case SIR_CONST_USHORT:
	{ US_Value = Value;
	  break;
	 }
     case SIR_CONST_INT:
	{ I_Value = Value;
	  break;
	 }
     case SIR_CONST_BIT:
	{ BIT_Value = new sir_bit(Value);
	  break;
	 }
     default:
	{ assert(FALSE);	/* bad Type */
	 }
    } /* hctiws */

} /* end of SIR_Constant::SIR_Constant #1 */


SIR_Constant::SIR_Constant(		/* constructor #2 */
	SIR_CONSTTYPE	Type,
	LONG_LONG	Value,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

SIR_Constant::Type = Type;

switch(Type)
   { case SIR_CONST_BOOL:
	{ if (Value != 0)
	     { B_Value = true;
	      } /* fi */
	  else
	     { B_Value = false;
	      } /* esle */
	  break;
	 }
     case SIR_CONST_CHAR:
	{ C_Value = SIR_LLONG2I(Value);
	  break;
	 }
     case SIR_CONST_UCHAR:
	{ UC_Value = SIR_LLONG2I(Value);
	  break;
	 }
     case SIR_CONST_SHORT:
	{ S_Value = SIR_LLONG2I(Value);
	  break;
	 }
     case SIR_CONST_USHORT:
	{ US_Value = SIR_LLONG2I(Value);
	  break;
	 }
     case SIR_CONST_INT:
	{ I_Value = SIR_LLONG2I(Value);
	  break;
	 }
     case SIR_CONST_UINT:
	{ UI_Value = SIR_LLONG2UI(Value);
	  break;
	 }
     case SIR_CONST_LONG:
	{ L_Value = SIR_LLONG2L(Value);
	  break;
	 }
     case SIR_CONST_ULONG:
	{ UL_Value = SIR_LLONG2UL(Value);
	  break;
	 }
     case SIR_CONST_LONGLONG:
	{ LL_Value = Value;
	  break;
	 }
     case SIR_CONST_ULONGLONG:
	{ ULL_Value = Value;
	  break;
	 }
     case SIR_CONST_BIT:
	{ BIT_Value = new sir_bit(Value);
	  break;
	 }
     default:
	{ assert(FALSE);	/* bad Type */
	 }
    } /* hctiws */

} /* end of SIR_Constant::SIR_Constant #2 */


SIR_Constant::SIR_Constant(		/* constructor #3 */
	SIR_CONSTTYPE	Type,
	UNSIGNED_LONG_LONG	Value,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

SIR_Constant::Type = Type;

switch(Type)
   { case SIR_CONST_BOOL:
	{ if (Value != 0)
	     { B_Value = true;
	      } /* fi */
	  else
	     { B_Value = false;
	      } /* esle */
	  break;
	 }
     case SIR_CONST_CHAR:
	{ C_Value = SIR_LLONG2I(Value);
	  break;
	 }
     case SIR_CONST_UCHAR:
	{ UC_Value = SIR_LLONG2I(Value);
	  break;
	 }
     case SIR_CONST_SHORT:
	{ S_Value = SIR_LLONG2I(Value);
	  break;
	 }
     case SIR_CONST_USHORT:
	{ US_Value = SIR_LLONG2I(Value);
	  break;
	 }
     case SIR_CONST_INT:
	{ I_Value = SIR_LLONG2I(Value);
	  break;
	 }
     case SIR_CONST_UINT:
	{ UI_Value = SIR_LLONG2UI(Value);
	  break;
	 }
     case SIR_CONST_LONG:
	{ L_Value = SIR_LLONG2L(Value);
	  break;
	 }
     case SIR_CONST_ULONG:
	{ UL_Value = SIR_LLONG2UL(Value);
	  break;
	 }
     case SIR_CONST_LONGLONG:
	{ LL_Value = Value;
	  break;
	 }
     case SIR_CONST_ULONGLONG:
	{ ULL_Value = Value;
	  break;
	 }
     case SIR_CONST_UBIT:
	{ BIT_Value = new sir_bit(Value);
	  break;
	 }
     default:
	{ assert(FALSE);	/* bad Type */
	 }
    } /* hctiws */

} /* end of SIR_Constant::SIR_Constant #3 */


SIR_Constant::SIR_Constant(		/* constructor #4 */
	SIR_CONSTTYPE	Type,
	long double	Value,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

SIR_Constant::Type = Type;

switch(Type)
   { case SIR_CONST_FLOAT:
	{ F_Value = Value;
	  break;
	 }
     case SIR_CONST_DOUBLE:
	{ D_Value = Value;
	  break;
	 }
     case SIR_CONST_LONGDOUBLE:
	{ LD_Value = Value;
	  break;
	 }
     default:
	{ assert(FALSE);	/* bad Type */
	 }
    } /* hctiws */

} /* end of SIR_Constant::SIR_Constant #4 */


SIR_Constant::SIR_Constant(		/* constructor #5 */
	SIR_CONSTTYPE	Type,
	string		*Value,	/* (the string is consumed!) */
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

SIR_Constant::Type = Type;

switch(Type)
   { case SIR_CONST_CHARSTRING:
	{ CS_Value = Value;
	  break;
	 }
     default:
	{ assert(FALSE);	/* bad Type */
	 }
    } /* hctiws */

} /* end of SIR_Constant::SIR_Constant #5 */


SIR_Constant::SIR_Constant(		/* constructor #6 */
	SIR_CONSTTYPE	Type,
	const char	*Value,	/* (the string is copied!) */
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

SIR_Constant::Type = Type;

switch(Type)
   { case SIR_CONST_BIT:
	{ BIT_Value = new sir_bit(Value, strlen(Value), SIR_BIT_SIGNED);
	  break;
	 }
     case SIR_CONST_UBIT:
	{ BIT_Value = new sir_bit(Value, strlen(Value), SIR_BIT_UNSIGNED);
	  break;
	 }
     case SIR_CONST_CHARSTRING:
	{ CS_Value = new string(Value);
	  break;
	 }
     default:
	{ assert(FALSE);	/* bad Type */
	 }
    } /* hctiws */

} /* end of SIR_Constant::SIR_Constant #6 */


SIR_Constant::SIR_Constant(		/* constructor #7 (for bitvectors) */
	sir_bit		Value,		/* (the bitvector is copied) */
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

SIR_Constant::Type = ((Value.isUnsigned()) ? SIR_CONST_UBIT : SIR_CONST_BIT);

BIT_Value = new sir_bit(Value);

} /* end of SIR_Constant::SIR_Constant #7 */


SIR_Constant::SIR_Constant(		/* constructor #8 */
	sir_constant	*Original) :	/* (duplicator) */
		SIR_Node(Original)
{

SIR_Constant::Type = Original->Type;

switch(Original->Type)
   { case SIR_CONST_BOOL:
	{ B_Value = Original->B_Value;
	  break;
	 }
      case SIR_CONST_CHAR:
	{ C_Value = Original->C_Value;
	  break;
	 }
      case SIR_CONST_UCHAR:
	{ UC_Value = Original->UC_Value;
	  break;
	 }
     case SIR_CONST_SHORT:
	{ S_Value = Original->S_Value;
	  break;
	 }
     case SIR_CONST_USHORT:
	{ US_Value = Original->US_Value;
	  break;
	 }
     case SIR_CONST_INT:
	{ I_Value = Original->I_Value;
	  break;
	 }
     case SIR_CONST_UINT:
	{ UI_Value = Original->UI_Value;
	  break;
	 }
     case SIR_CONST_LONG:
	{ L_Value = Original->L_Value;
	  break;
	 }
     case SIR_CONST_ULONG:
	{ UL_Value = Original->UL_Value;
	  break;
	 }
     case SIR_CONST_LONGLONG:
	{ LL_Value = Original->LL_Value;
	  break;
	 }
     case SIR_CONST_ULONGLONG:
	{ ULL_Value = Original->ULL_Value;
	  break;
	 }
     case SIR_CONST_FLOAT:
	{ F_Value = Original->F_Value;
	  break;
	 }
     case SIR_CONST_DOUBLE:
	{ D_Value = Original->D_Value;
	  break;
	 }
     case SIR_CONST_LONGDOUBLE:
	{ LD_Value = Original->LD_Value;
	  break;
	 }
     case SIR_CONST_BIT:
     case SIR_CONST_UBIT:
	{ BIT_Value = new sir_bit(*Original->BIT_Value);
	  break;
	 }
     case SIR_CONST_CHARSTRING:
	{ CS_Value = new string(*Original->CS_Value);
	  break;
	 }
     default:
	{ assert(FALSE);	/* bad Type */
	 }
    } /* hctiws */

} /* end of SIR_Constant::SIR_Constant #8 */


SIR_Constant::~SIR_Constant(void)	/* destructor */
{

if (  (Type == SIR_CONST_BIT)
    ||(Type == SIR_CONST_UBIT))
   { delete BIT_Value;
    } /* fi */

if (Type == SIR_CONST_CHARSTRING)
   { delete CS_Value;
    } /* fi */

} /* end of SIR_Constant::~SIR_Constant */


ERROR SIR_Constant::DFS_ForAllNodes(	/* iterator for over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{

if ((SIR_Error = (this->*MemberFct)(MemberFctArg)))	/* process this node */
   { return(SIR_Error);
    } /* fi */

/* there are no nodes below */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Constant::DFS_ForAllNodes */


sir_constant *SIR_Constant::Promoted(void)	/* promoted constant */
{

switch(Type)
   { case SIR_CONST_BOOL:
	{ Type = SIR_CONST_INT;		/* bool becomes int */
	  I_Value = B_Value;
	  break;
	 }
     case SIR_CONST_CHAR:
	{ Type = SIR_CONST_INT;		/* char becomes int */
	  I_Value = C_Value;
	  break;
	 }
     case SIR_CONST_UCHAR:
	{ Type = SIR_CONST_INT;		/* unsigned char becomes int */
	  I_Value = UC_Value;
	  break;
	 }
     case SIR_CONST_SHORT:
	{ Type = SIR_CONST_INT;		/* short becomes int */
	  I_Value = S_Value;
	  break;
	 }
     case SIR_CONST_USHORT:
	{ Type = SIR_CONST_INT;		/* unsigned short becomes int */
	  I_Value = US_Value;
	  break;
	 }
     case SIR_CONST_INT:
     case SIR_CONST_UINT:
     case SIR_CONST_LONG:
     case SIR_CONST_ULONG:
     case SIR_CONST_LONGLONG:
     case SIR_CONST_ULONGLONG:
	{ break;			/* nothing to promote */
	 }
     case SIR_CONST_FLOAT:
	{ Type = SIR_CONST_DOUBLE;	/* float becomes double */
	  D_Value = F_Value;
	  break;
	 }
     case SIR_CONST_DOUBLE:
     case SIR_CONST_LONGDOUBLE:
     case SIR_CONST_BIT:
     case SIR_CONST_UBIT:
     case SIR_CONST_CHARSTRING:
	{ break;			/* nothing to promote */
	 }
     default:
	{ assert(FALSE);	/* bad Type */
	 }
    } /* hctiws */

return(this);	/* return myself after promotion */

} /* end of SIR_Constant::Promoted */


sir_constant *SIR_Constant::Converted(	/* converted constant */
	SIR_CONSTTYPE	Type,		/* (supports all legal conversions) */
	unsigned int	Length /* = 0 */) /* for bitvecs; 0 is natural length */
{
sir_bit		*BitVec;

if (  (Type == SIR_CONST_BIT)	/* first handle conversions to bitvectors */
    ||(Type == SIR_CONST_UBIT))
   { switch(this->Type)
	{ case SIR_CONST_BOOL:
	     { BIT_Value = new sir_bit(B_Value, Length, Type==SIR_CONST_UBIT);
	       break;
	      }
	  case SIR_CONST_CHAR:
	     { BIT_Value = new sir_bit(C_Value, Length, Type==SIR_CONST_UBIT);
	       break;
	      }
	  case SIR_CONST_UCHAR:
	     { BIT_Value = new sir_bit(UC_Value, Length, Type==SIR_CONST_UBIT);
	       break;
	      }
	  case SIR_CONST_SHORT:
	     { BIT_Value = new sir_bit(S_Value, Length, Type==SIR_CONST_UBIT);
	       break;
	      }
	  case SIR_CONST_USHORT:
	     { BIT_Value = new sir_bit(US_Value, Length, Type==SIR_CONST_UBIT);
	       break;
	      }
	  case SIR_CONST_INT:
	     { BIT_Value = new sir_bit(I_Value, Length, Type==SIR_CONST_UBIT);
	       break;
	      }
	  case SIR_CONST_UINT:
	     { BIT_Value = new sir_bit(UI_Value, Length, Type==SIR_CONST_UBIT);
	       break;
	      }
	  case SIR_CONST_LONG:
	     { BIT_Value = new sir_bit(L_Value, Length, Type==SIR_CONST_UBIT);
	       break;
	      }
	  case SIR_CONST_ULONG:
	     { BIT_Value = new sir_bit(UL_Value, Length, Type==SIR_CONST_UBIT);
	       break;
	      }
	  case SIR_CONST_LONGLONG:
	     { BIT_Value = new sir_bit(LL_Value, Length, Type==SIR_CONST_UBIT);
	       break;
	      }
	  case SIR_CONST_ULONGLONG:
	     { BIT_Value = new sir_bit(ULL_Value, Length, Type==SIR_CONST_UBIT);
	       break;
	      }
	  case SIR_CONST_BIT:
	  case SIR_CONST_UBIT:
	     { BitVec = BIT_Value;	/* don't overwrite yourself! */
	       if (Length > 0)
		  { BIT_Value = new sir_bit(Length,
						(bool)(Type==SIR_CONST_UBIT));
		   } /* fi */
	       else
		  { BIT_Value = new sir_bit(BitVec->length(),
						(bool)(Type==SIR_CONST_UBIT));
		   } /* esle */
	       *BIT_Value = *BitVec;	/* assign bits (and extend/truncate) */
	       delete BitVec;
	       break;
	      }
	  case SIR_CONST_FLOAT:
	     { assert(Length > 0);	/* must be specified */
	       BIT_Value = new sir_bit(F_Value, Length, Type==SIR_CONST_UBIT);
	       break;
	      }
	  case SIR_CONST_DOUBLE:
	     { assert(Length > 0);	/* must be specified */
	       BIT_Value = new sir_bit(D_Value, Length, Type==SIR_CONST_UBIT);
	       break;
	      }
	  case SIR_CONST_LONGDOUBLE:
	     { assert(Length > 0);	/* must be specified */
	       BIT_Value = new sir_bit(LD_Value, Length, Type==SIR_CONST_UBIT);
	       break;
	      }
	  case SIR_CONST_CHARSTRING:
	     { assert(FALSE);	/* not convertable to bitvector */
	      }
	  default:
	     { assert(FALSE);	/* bad constant type */
	      }
	 } /* hctiws */
     assert((Type == SIR_CONST_UBIT) == (BIT_Value->isUnsigned()));
     this->Type = Type;
     return(this);
    } /* fi */

/* now handle the other conversions (not to bitvectors) */

switch(this->Type)
   { case SIR_CONST_BOOL:
	{ SIR_ConvertULL(this, B_Value, Type);
	  break;
	 }
     case SIR_CONST_CHAR:
	{ SIR_ConvertLL(this, C_Value, Type);
	  break;
	 }
     case SIR_CONST_UCHAR:
	{ SIR_ConvertULL(this, UC_Value, Type);
	  break;
	 }
     case SIR_CONST_SHORT:
	{ SIR_ConvertLL(this, S_Value, Type);
	  break;
	 }
     case SIR_CONST_USHORT:
	{ SIR_ConvertULL(this, US_Value, Type);
	  break;
	 }
     case SIR_CONST_INT:
	{ SIR_ConvertLL(this, I_Value, Type);
	  break;
	 }
     case SIR_CONST_UINT:
	{ SIR_ConvertULL(this, UI_Value, Type);
	  break;
	 }
     case SIR_CONST_LONG:
	{ SIR_ConvertLL(this, L_Value, Type);
	  break;
	 }
     case SIR_CONST_ULONG:
	{ SIR_ConvertULL(this, UL_Value, Type);
	  break;
	 }
     case SIR_CONST_LONGLONG:
	{ SIR_ConvertLL(this, LL_Value, Type);
	  break;
	 }
     case SIR_CONST_ULONGLONG:
	{ SIR_ConvertULL(this, ULL_Value, Type);
	  break;
	 }
     case SIR_CONST_FLOAT:
	{ SIR_ConvertLD(this, F_Value, Type);
	  break;
	 }
     case SIR_CONST_DOUBLE:
	{ SIR_ConvertLD(this, D_Value, Type);
	  break;
	 }
     case SIR_CONST_LONGDOUBLE:
	{ SIR_ConvertLD(this, LD_Value, Type);
	  break;
	 }
     case SIR_CONST_BIT:
     case SIR_CONST_UBIT:
	{ SIR_ConvertBIT(this, Type);
	  break;
	 }
     case SIR_CONST_CHARSTRING:
	{ assert(  (Type == SIR_CONST_BOOL)	/* conversion only to bool */
		 ||(Type == SIR_CONST_CHARSTRING));	/* (or no-conversion) */
	  if (Type == SIR_CONST_BOOL)
	     { delete CS_Value;
	       B_Value = true;	/* const char pointer evaluates to true */
	       this->Type = Type;
	      } /* fi */
	  break;
	 }
     default:
	{ assert(FALSE);	/* bad Type */
	 }
    } /* hctiws */

return(this);	/* return myself after conversion */

} /* end of SIR_Constant::Converted */


const char *SIR_Constant::PrettyChar(
	const char	Char)
{
static char	Pretty[10];

switch(Char)
   { case '\n':
	{ strcpy(&Pretty[0], "\\n");
	  break;
	 }
     case '\t':
	{ strcpy(&Pretty[0], "\\t");
	  break;
	 }
     case '\v':
	{ strcpy(&Pretty[0], "\\v");
	  break;
	 }
     case '\b':
	{ strcpy(&Pretty[0], "\\b");
	  break;
	 }
     case '\r':
	{ strcpy(&Pretty[0], "\\r");
	  break;
	 }
     case '\f':
	{ strcpy(&Pretty[0], "\\f");
	  break;
	 }
     case '\a':
	{ strcpy(&Pretty[0], "\\a");
	  break;
	 }
     case '\\':
	{ strcpy(&Pretty[0], "\\\\");
	  break;
	 }
     case '\'':
	{ strcpy(&Pretty[0], "\\'");
	  break;
	 }
     default:
	{
#ifdef SUPPORT_ISO
	  if (  (isascii(Char) && isprint(Char))
	      ||(((unsigned char)Char) >= 0xa1))
#else /* NO SUPPORT_ISO */
	  if (isascii(Char) && isprint(Char))
#endif /* SUPPORT_ISO */
	     { Pretty[0] = Char;
	       Pretty[1] = 0;
	      } /* fi */
	  else
	     { sprintf(&Pretty[0], "\\%03o", (unsigned char)Char);
	      } /* esle */
	 }
    } /* hctiws */

return(&Pretty[0]);

} /* end of SIR_Constant::PrettyChar */


void SIR_Constant::PrettyString(	/* returns a C-style string */
	string		*Result,
	const char	*String)
{
const char	*CharPtr;
#ifdef SUPPORT_KANJI
unsigned char	c1, c2;
#endif /* SUPPORT_KANJI */

assert(Result != NULL);
assert(String != NULL);

*Result = "\"";
CharPtr = String;
while(*CharPtr)
   {
#ifdef SUPPORT_KANJI
     c1 = *CharPtr;
     c2 = *(CharPtr+1);
     if (  (  (c1 >= 0x81 && c1 <= 0x9f)
	    ||(c1 >= 0xe0 && c1 <= 0xfc))
	 &&(  (c2 >= 0x40 && c2 <= 0xfc && c2 != 0x7f)))
	{ /* this looks like a (2-byte) S-JIS code */
	  *Result += c1;
	  *Result += c2;
	  CharPtr += 2;
	  continue;
	 } /* fi */
     if (  (c1 >= 0xa1 && c1 <= 0xfe)
	 &&(c2 >= 0xa1 && c2 <= 0xfe))
	{ /* this looks like a (2-byte) EUC code */
	  *Result += c1;
	  *Result += c2;
	  CharPtr += 2;
	  continue;
	 } /* fi */
#endif /* SUPPORT_KANJI */

     if (*CharPtr == '\'')
	{ *Result += "'";
	 } /* fi */
     else
	{ if (*CharPtr == '"')
	     { *Result += "\\\"";
	      } /* fi */
	  else
	     { *Result += PrettyChar(*CharPtr);
	      } /* esle */
	 } /* esle */
     CharPtr++;
    } /* elihw */
*Result += "\"";

} /* end of SIR_Constant::PrettyString */

/* PC No Error Check*/
sir_constant *SIR_Constant::New(        /* create a new integer constant (#1) */
        LONG_LONG       Value,          /* (returns NULL if SIR_Error) */
        SIR_CONSTTYPE   CType /* = SIR_CONST_INT */)    /* default: integer */
{
sir_constant    *NewConst;

switch(CType)
   { case SIR_CONST_BOOL:
     case SIR_CONST_CHAR:
     case SIR_CONST_UCHAR:
     case SIR_CONST_SHORT:
     case SIR_CONST_USHORT:
     case SIR_CONST_INT:
     case SIR_CONST_UINT:
     case SIR_CONST_LONG:
     case SIR_CONST_ULONG:
     case SIR_CONST_LONGLONG:
     case SIR_CONST_ULONGLONG:
     case SIR_CONST_BIT:
        { NewConst = new SIR_Constant(CType, Value);
          break;
         }
     case SIR_CONST_FLOAT:
     case SIR_CONST_DOUBLE:
     case SIR_CONST_LONGDOUBLE:
     case SIR_CONST_UBIT:
     case SIR_CONST_CHARSTRING:
     default:
        { 
          return(NULL);
         }
    } /* hctiws */

return(NewConst);

} /* end of SIR_Constant::New (#1) */

sir_constant *SIR_Constant::Copy(       /* creates a copy of the constant */
        BOOL            Strip /* = TRUE */) /* (returns NULL if SIR_Error) */
{
sir_constant    *Const;

Const = new SIR_Constant(this);

return(Const);

} /* end of SIR_Constant::Copy */


void SIR_Constant::Delete(void)         /* deletes a constant */
{

delete this;

} /* end of SIR_Constant::Delete */

const char *SIR_Constant::Print(	/* return text string of value */
	BOOL		CppNotation /* = FALSE */) /* default: SpecC notation */
{
static string	PrintedValue;
char		NumberBuffer[MAX_NUM_LENGTH];
char		*BitBuffer,
		*TmpValue;

switch(Type)
   { case SIR_CONST_BOOL:
	{ PrintedValue = (B_Value ? "true" : "false");
	  break;
	 }
     case SIR_CONST_CHAR:
	{ PrintedValue.form("'%s'", PrettyChar(C_Value));
	  break;
	 }
     case SIR_CONST_UCHAR:
	{ PrintedValue.form("((unsigned char)'%s')",
				PrettyChar((char)UC_Value));
	  break;
	 }
     case SIR_CONST_SHORT:
	{ PrintedValue.form("((short int)%d)", (int)S_Value);
	  break;
	 }
     case SIR_CONST_USHORT:
	{ PrintedValue.form("((unsigned short int)%u)", (unsigned)US_Value);
	  break;
	 }
     case SIR_CONST_INT:
	{ PrintedValue.form("%d", I_Value);
	  break;
	 }
     case SIR_CONST_UINT:
	{ PrintedValue.form("%uu", UI_Value);
	  break;
	 }
     case SIR_CONST_LONG:	/* (form() doesn't support %l formats) */
	{ sprintf(&NumberBuffer[0], "%ldl", L_Value);
	  PrintedValue = &NumberBuffer[0];
	  break;
	 }
     case SIR_CONST_ULONG:	/* (form() doesn't support %l formats) */
	{ sprintf(&NumberBuffer[0], "%luul", UL_Value);
	  PrintedValue = &NumberBuffer[0];
	  break;
	 }
     case SIR_CONST_LONGLONG:	/* (form() doesn't support %l formats) */
	{
#ifdef LONGLONG_USE_NATIVE_OUTPUT_ROUTINES

#ifdef SOLARIS
	  sprintf(&NumberBuffer[0], "%lldll", LL_Value);
#endif /* SOLARIS */
#ifdef SUNOS4
	  sprintf(&NumberBuffer[0], "%ldll", (long)LL_Value);
	  GL_PrintWarning(GL_WARN_IMPORTANT,
		"'long long' constant written with 'long' precision");
#endif /* SUNOS4 */
#ifdef NETBSD
	  sprintf(&NumberBuffer[0], "%qdll", LL_Value);
#endif /* NETBSD */
#ifdef LINUX
	  sprintf(&NumberBuffer[0], "%qdll", LL_Value);
#endif /* LINUX */
#ifdef GNUWIN32
	  sprintf(&NumberBuffer[0], "%qdll", LL_Value);
#endif /* GNUWIN32 */
#ifndef SOLARIS
#   ifndef SUNOS4
#      ifndef NETBSD
#         ifndef LINUX
#            ifndef GNUWIN32
#error "Unknown system! Don't know how to print 'long long int' constants!"
#            endif /* GNUWIN32 */
#         endif /* LINUX */
#      endif /* NETBSD */
#   endif /* SUNOS4 */
#endif /* SOLARIS */

	  PrintedValue = &NumberBuffer[0];

#endif /* LONGLONG_USE_NATIVE_OUTPUT_ROUTINES */

#ifdef LONGLONG_USE_SPECC_OUTPUT_ROUTINES

	  TmpValue = ll2str(10, &NumberBuffer[MAX_NUM_LENGTH-1], LL_Value);
#ifdef HAVE_LLONG
	  PrintedValue = TmpValue;
	  PrintedValue += "ll";
#else /* !HAVE_LLONG */
	  if (CppNotation)
	     { PrintedValue.form(SIR_LLONG_CLASS_NAME
				"<" SIR_LLONG_SIGNED_FLAG ">(\"%s\", 10)",
								TmpValue);
	      } /* fi */
	  else
	     { PrintedValue = TmpValue;
	       PrintedValue += "ll";
	      } /* esle */
#endif /* HAVE_LLONG */

#endif /* LONGLONG_USE_SPECC_OUTPUT_ROUTINES */

	  break;
	 }
     case SIR_CONST_ULONGLONG:	/* (form() doesn't support %l formats) */
	{
#ifdef LONGLONG_USE_NATIVE_OUTPUT_ROUTINES

#ifdef SOLARIS
	  sprintf(&NumberBuffer[0], "%lluull", ULL_Value);
#endif /* SOLARIS */
#ifdef SUNOS4
	  sprintf(&NumberBuffer[0], "%luull", (unsigned long)ULL_Value);
	  GL_PrintWarning(GL_WARN_IMPORTANT,
			"'unsigned long long' constant written "
			"with 'unsigned long' precision");
#endif /* SUNOS4 */
#ifdef NETBSD
	  sprintf(&NumberBuffer[0], "%quull", ULL_Value);
#endif /* NETBSD */
#ifdef LINUX
	  sprintf(&NumberBuffer[0], "%quull", ULL_Value);
#endif /* LINUX */
#ifdef GNUWIN32
	  sprintf(&NumberBuffer[0], "%quull", ULL_Value);
#endif /* GNUWIN32 */
#ifndef SOLARIS
#   ifndef SUNOS4
#      ifndef NETBSD
#         ifndef LINUX
#            ifndef GNUWIN32
#error "Unknown system! Don't know how to print 'unsigned long long int'!"
#            endif /* GNUWIN32 */
#         endif /* LINUX */
#      endif /* NETBSD */
#   endif /* SUNOS4 */
#endif /* SOLARIS */

	  PrintedValue = &NumberBuffer[0];

#endif /* LONGLONG_USE_NATIVE_OUTPUT_ROUTINES */

#ifdef LONGLONG_USE_SPECC_OUTPUT_ROUTINES

	  TmpValue = ull2str(10, &NumberBuffer[MAX_NUM_LENGTH-1], ULL_Value);
#ifdef HAVE_LLONG
	  PrintedValue = TmpValue;
	  PrintedValue += "ull";
#else /* !HAVE_LLONG */
	  if (CppNotation)
	     { PrintedValue.form(SIR_LLONG_CLASS_NAME
				"<" SIR_LLONG_UNSIGNED_FLAG ">(\"%s\", 10)",
								TmpValue);
	      } /* fi */
	  else
	     { PrintedValue = TmpValue;
	       PrintedValue += "ull";
	      } /* esle */
#endif /* HAVE_LLONG */

#endif /* LONGLONG_USE_SPECC_OUTPUT_ROUTINES */

	  break;
	 }
     case SIR_CONST_FLOAT:	/* (form() doesn't support %.* formats) */
	{ sprintf(&NumberBuffer[0], "%.*ef", FLT_DIG, F_Value);
	  PrintedValue = &NumberBuffer[0];
	  break;
	 }
     case SIR_CONST_DOUBLE:	/* (form() doesn't support %.* formats) */
	{ sprintf(&NumberBuffer[0], "%.*e", DBL_DIG, D_Value);
	  PrintedValue = &NumberBuffer[0];
	  break;
	 }
     case SIR_CONST_LONGDOUBLE:	/* (form() doesn't support %.* formats) */
	{
#ifdef SOLARIS
	  sprintf(&NumberBuffer[0], "%.*Lel", LDBL_DIG, LD_Value);
#endif /* SOLARIS */
#ifdef SUNOS4
	  sprintf(&NumberBuffer[0], "%.*el", LDBL_DIG, (double)LD_Value);
	  GL_PrintWarning(GL_WARN_IMPORTANT,
		"'long double' constant written with 'double' precision");
#endif /* SUNOS4 */
#ifdef NETBSD
	  sprintf(&NumberBuffer[0], "%.*Lel", LDBL_DIG, LD_Value);
#endif /* NETBSD */
#ifdef LINUX
	  sprintf(&NumberBuffer[0], "%.*Lel", LDBL_DIG, LD_Value);
#endif /* LINUX */
#ifdef GNUWIN32
	  sprintf(&NumberBuffer[0], "%.*Lel", LDBL_DIG, LD_Value);
#endif /* GNUWIN32 */
#ifndef SOLARIS
#   ifndef SUNOS4
#      ifndef NETBSD
#         ifndef LINUX
#            ifndef GNUWIN32
#error "Unknown system! Don't know how to handle 'long double' constants!"
#            endif /* GNUWIN32 */
#         endif /* LINUX */
#      endif /* NETBSD */
#   endif /* SUNOS4 */
#endif /* SOLARIS */
	  PrintedValue = &NumberBuffer[0];
	  break;
	 }
     case SIR_CONST_BIT:
	{ BitBuffer = new char[BIT_Value->length() + 1];
	  ubit2str(2, BitBuffer+(BIT_Value->length()), *BIT_Value);
	  if (CppNotation)
	     { PrintedValue.form(SIR_BIT_CLASS_NAME "<%d,"
					SIR_BIT_SIGNED_FLAG ">(\"%s\")",
					BIT_Value->length(), BitBuffer);
	      } /* fi */
	  else
	     { PrintedValue.form("%sb", BitBuffer);
	      } /* esle */
	  delete[] BitBuffer;
	  break;
	 }
     case SIR_CONST_UBIT:
	{ BitBuffer = new char[BIT_Value->length() + 1];
	  ubit2str(2, BitBuffer+(BIT_Value->length()), *BIT_Value);
	  if (CppNotation)
	     { PrintedValue.form(SIR_BIT_CLASS_NAME "<%d,"
					SIR_BIT_UNSIGNED_FLAG ">(\"%s\")",
					BIT_Value->length(), BitBuffer);
	      } /* fi */
	  else
	     { PrintedValue.form("%sub", BitBuffer);
	      } /* esle */
	  delete[] BitBuffer;
	  break;
	 }
     case SIR_CONST_CHARSTRING:
	{ PrettyString(&PrintedValue, CS_Value->chars());
	  break;
	 }
     default:
	{ assert(FALSE);	/* bad Type */
	 }
    } /* hctiws */

return(PrintedValue.chars());

} /* end of SIR_Constant::Print */


ERROR SIR_Constant::Compare(    /* compare this constant with another one (#1)*/
        sir_constant    *Const2,        /* (returns error if not comparable) */
        int             *Result)        /* Result is {-1,0,1} as in strcmp() */
{
sir_constant    *C1,
                *C2;

assert(Const2 != NULL);
assert(Result != NULL);

        /* first, handle special case of string comparisons */

if (  (Type == SIR_CONST_CHARSTRING)
    ||(Const2->Type == SIR_CONST_CHARSTRING))
   { 
     /* perform a STRING-type comparison */
     *Result = strcmp(CS_Value->chars(), Const2->CS_Value->chars());
     return(SIR_ERROR_NO_ERROR);
    } /* fi */

        /* second, handle floating point comparisons */

if (  (Type == SIR_CONST_FLOAT)
    ||(Type == SIR_CONST_DOUBLE)
    ||(Type == SIR_CONST_LONGDOUBLE)
    ||(Const2->Type == SIR_CONST_FLOAT)
    ||(Const2->Type == SIR_CONST_DOUBLE)
    ||(Const2->Type == SIR_CONST_LONGDOUBLE))
   { /* perform a LD-type comparison */
     C1 = new SIR_Constant(this);
     if (C1->Type != SIR_CONST_LONGDOUBLE)
        { C1->Converted(SIR_CONST_LONGDOUBLE);
         } /* fi */
     C2 = new SIR_Constant(Const2);
     if (C2->Type != SIR_CONST_LONGDOUBLE)
        { C2->Converted(SIR_CONST_LONGDOUBLE);
         } /* fi */
     if (C1->LD_Value > C2->LD_Value)
        { *Result = 1;
         } /* fi */
     else
        { if (C1->LD_Value < C2->LD_Value)
             { *Result = -1;
              } /* fi */
          else
             { *Result = 0;
              } /* esle */
         } /* esle */
     delete C1;
     delete C2;
     return(SIR_ERROR_NO_ERROR);
    } /* fi */

        /* third, handle bit vector comparisons */

if (  (Type == SIR_CONST_BIT)
    ||(Type == SIR_CONST_UBIT)
    ||(Const2->Type == SIR_CONST_BIT)
    ||(Const2->Type == SIR_CONST_UBIT))
   { /* perform a BIT-type comparison */
     C1 = new SIR_Constant(this);
     if (  (C1->Type == SIR_CONST_CHAR)
         ||(C1->Type == SIR_CONST_SHORT)
         ||(C1->Type == SIR_CONST_INT)
         ||(C1->Type == SIR_CONST_LONG)
         ||(C1->Type == SIR_CONST_LONGLONG))
        { C1->Converted(SIR_CONST_BIT);
         } /* fi */
     else
        { if (  (C1->Type == SIR_CONST_BOOL)
              ||(C1->Type == SIR_CONST_UCHAR)
              ||(C1->Type == SIR_CONST_USHORT)
              ||(C1->Type == SIR_CONST_UINT)
              ||(C1->Type == SIR_CONST_ULONG)
              ||(C1->Type == SIR_CONST_ULONGLONG))
             { C1->Converted(SIR_CONST_UBIT);
              } /* fi */
         } /* esle */
     C2 = new SIR_Constant(Const2);
     if (  (C2->Type == SIR_CONST_CHAR)
         ||(C2->Type == SIR_CONST_SHORT)
         ||(C2->Type == SIR_CONST_INT)
         ||(C2->Type == SIR_CONST_LONG)
         ||(C2->Type == SIR_CONST_LONGLONG))
        { C2->Converted(SIR_CONST_BIT);
         } /* fi */
     else
        { if (  (C2->Type == SIR_CONST_BOOL)
              ||(C2->Type == SIR_CONST_UCHAR)
              ||(C2->Type == SIR_CONST_USHORT)
              ||(C2->Type == SIR_CONST_UINT)
              ||(C2->Type == SIR_CONST_ULONG)
              ||(C2->Type == SIR_CONST_ULONGLONG))
             { C2->Converted(SIR_CONST_UBIT);
              } /* fi */
         } /* esle */
     if (*C1->BIT_Value > *C2->BIT_Value)
        { *Result = 1;
         } /* fi */
     else
        { if (*C1->BIT_Value < *C2->BIT_Value)
             { *Result = -1;
              } /* fi */
          else
             { *Result = 0;
              } /* esle */
         } /* esle */
     delete C1;
     delete C2;
     return(SIR_ERROR_NO_ERROR);
    } /* fi */

        /* forth, handle unsigned long long comparisons */

if (  (Type == SIR_CONST_ULONGLONG)
    ||(Const2->Type == SIR_CONST_ULONGLONG))
   { /* perform a ULL-type comparison */
     C1 = new SIR_Constant(this);
     if (C1->Type != SIR_CONST_ULONGLONG)
        { C1->Converted(SIR_CONST_ULONGLONG);
         } /* fi */
     C2 = new SIR_Constant(Const2);
     if (C2->Type != SIR_CONST_ULONGLONG)
        { C2->Converted(SIR_CONST_ULONGLONG);
         } /* fi */
     if (C1->ULL_Value > C2->ULL_Value)
        { *Result = 1;
         } /* fi */
     else
        { if (C1->ULL_Value < C2->ULL_Value)
             { *Result = -1;
              } /* fi */
          else
             { *Result = 0;
              } /* esle */
         } /* esle */
     delete C1;
     delete C2;
     return(SIR_ERROR_NO_ERROR);
    } /* fi */

        /* finally, handle the other integral comparisons */

/* perform a LL-type comparison */
C1 = new SIR_Constant(this);
if (C1->Type != SIR_CONST_LONGLONG)
   { C1->Converted(SIR_CONST_LONGLONG);
    } /* fi */
C2 = new SIR_Constant(Const2);
if (C2->Type != SIR_CONST_LONGLONG)
   { C2->Converted(SIR_CONST_LONGLONG);
    } /* fi */
if (C1->LL_Value > C2->LL_Value)
   { *Result = 1;
    } /* fi */
else
   { if (C1->LL_Value < C2->LL_Value)
        { *Result = -1;
         } /* fi */
     else
        { *Result = 0;
         } /* esle */
    } /* esle */
delete C1;
delete C2;

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Constant::Compare #1 */


int SIR_Constant::Compare(      /* compare two comparable constants (#2) */
        sir_constant    *Const1,        /* result is {-1,0,1} as in strcmp() */
        sir_constant    *Const2)
{
ERROR           Error;
int             Result;

assert(Const1 != NULL);
assert(Const2 != NULL);

Error = Const1->Compare(Const2, &Result);
assert(Error == SIR_ERROR_NO_ERROR);

return(Result);

} /* end of SIR_Constant::Compare #2 */


/*** internal functions *************************************************/


static void SIR_ConvertLL(
	sir_constant	*Const,
	LONG_LONG	LL_Value,
	SIR_CONSTTYPE	Type)
{

switch(Type)
   { case SIR_CONST_BOOL:
	{ Const->B_Value = SIR_LLONG2B(LL_Value);
	  break;
	 }
     case SIR_CONST_CHAR:
	{ Const->C_Value = SIR_LLONG2I(LL_Value);
	  break;
	 }
     case SIR_CONST_UCHAR:
	{ Const->UC_Value = SIR_LLONG2I(LL_Value);
	  break;
	 }
     case SIR_CONST_SHORT:
	{ Const->S_Value = SIR_LLONG2I(LL_Value);
	  break;
	 }
     case SIR_CONST_USHORT:
	{ Const->US_Value = SIR_LLONG2I(LL_Value);
	  break;
	 }
     case SIR_CONST_INT:
	{ Const->I_Value = SIR_LLONG2I(LL_Value);
	  break;
	 }
     case SIR_CONST_UINT:
	{ Const->UI_Value = SIR_LLONG2UI(LL_Value);
	  break;
	 }
     case SIR_CONST_LONG:
	{ Const->L_Value = SIR_LLONG2L(LL_Value);
	  break;
	 }
     case SIR_CONST_ULONG:
	{ Const->UL_Value = SIR_LLONG2UL(LL_Value);
	  break;
	 }
     case SIR_CONST_LONGLONG:
	{ Const->LL_Value = LL_Value;
	  break;
	 }
     case SIR_CONST_ULONGLONG:
	{ Const->ULL_Value = LL_Value;
	  break;
	 }
     case SIR_CONST_FLOAT:
	{ Const->F_Value = SIR_LLONG2D(LL_Value);
	  break;
	 }
     case SIR_CONST_DOUBLE:
	{ Const->D_Value = SIR_LLONG2D(LL_Value);
	  break;
	 }
     case SIR_CONST_LONGDOUBLE:
	{ Const->LD_Value = SIR_LLONG2LD(LL_Value);
	  break;
	 }
     case SIR_CONST_BIT:
     case SIR_CONST_UBIT:
	{ assert(FALSE);	/* is handled elsewhere */
	 }
     case SIR_CONST_CHARSTRING:
	{ assert(FALSE);	/* cannot be converted */
	 }
     default:
	{ assert(FALSE);	/* bad Type */
	 }
    } /* hctiws */

Const->Type = Type;

} /* end of SIR_ConvertLL */


static void SIR_ConvertULL(
	sir_constant		*Const,
	UNSIGNED_LONG_LONG	ULL_Value,
	SIR_CONSTTYPE		Type)
{

switch(Type)
   { case SIR_CONST_BOOL:
	{ Const->B_Value = SIR_LLONG2B(ULL_Value);
	  break;
	 }
     case SIR_CONST_CHAR:
	{ Const->C_Value = SIR_LLONG2I(ULL_Value);
	  break;
	 }
     case SIR_CONST_UCHAR:
	{ Const->UC_Value = SIR_LLONG2I(ULL_Value);
	  break;
	 }
     case SIR_CONST_SHORT:
	{ Const->S_Value = SIR_LLONG2I(ULL_Value);
	  break;
	 }
     case SIR_CONST_USHORT:
	{ Const->US_Value = SIR_LLONG2I(ULL_Value);
	  break;
	 }
     case SIR_CONST_INT:
	{ Const->I_Value = SIR_LLONG2I(ULL_Value);
	  break;
	 }
     case SIR_CONST_UINT:
	{ Const->UI_Value = SIR_LLONG2UI(ULL_Value);
	  break;
	 }
     case SIR_CONST_LONG:
	{ Const->L_Value = SIR_LLONG2L(ULL_Value);
	  break;
	 }
     case SIR_CONST_ULONG:
	{ Const->UL_Value = SIR_LLONG2UL(ULL_Value);
	  break;
	 }
     case SIR_CONST_LONGLONG:
	{ Const->LL_Value = ULL_Value;
	  break;
	 }
     case SIR_CONST_ULONGLONG:
	{ Const->ULL_Value = ULL_Value;
	  break;
	 }
     case SIR_CONST_FLOAT:
	{ Const->F_Value = SIR_LLONG2D(ULL_Value);
	  break;
	 }
     case SIR_CONST_DOUBLE:
	{ Const->D_Value = SIR_LLONG2D(ULL_Value);
	  break;
	 }
     case SIR_CONST_LONGDOUBLE:
	{ Const->LD_Value = SIR_LLONG2LD(ULL_Value);
	  break;
	 }
     case SIR_CONST_BIT:
     case SIR_CONST_UBIT:
	{ assert(FALSE);	/* is handled elsewhere */
	 }
     case SIR_CONST_CHARSTRING:
	{ assert(FALSE);	/* cannot be converted */
	 }
     default:
	{ assert(FALSE);	/* bad Type */
	 }
    } /* hctiws */

Const->Type = Type;

} /* end of SIR_ConvertULL */


static void SIR_ConvertBIT(
	sir_constant	*Const,
	SIR_CONSTTYPE	Type)
{
sir_bit		*Saved_BIT_Value;

Saved_BIT_Value = Const->BIT_Value;

switch(Type)
   { case SIR_CONST_BOOL:
	{ Const->B_Value = Saved_BIT_Value->test();
	  break;
	 }
     case SIR_CONST_CHAR:
	{ Const->C_Value = Saved_BIT_Value->toInt();
	  break;
	 }
     case SIR_CONST_UCHAR:
	{ Const->UC_Value = Saved_BIT_Value->toUInt();
	  break;
	 }
     case SIR_CONST_SHORT:
	{ Const->S_Value = Saved_BIT_Value->toInt();
	  break;
	 }
     case SIR_CONST_USHORT:
	{ Const->US_Value = Saved_BIT_Value->toUInt();
	  break;
	 }
     case SIR_CONST_INT:
	{ Const->I_Value = Saved_BIT_Value->toInt();
	  break;
	 }
     case SIR_CONST_UINT:
	{ Const->UI_Value = Saved_BIT_Value->toUInt();
	  break;
	 }
     case SIR_CONST_LONG:
	{ Const->L_Value = Saved_BIT_Value->toLong();
	  break;
	 }
     case SIR_CONST_ULONG:
	{ Const->UL_Value = Saved_BIT_Value->toULong();
	  break;
	 }
     case SIR_CONST_LONGLONG:
	{ Const->LL_Value = Saved_BIT_Value->toLLong();
	  break;
	 }
     case SIR_CONST_ULONGLONG:
	{ Const->ULL_Value = Saved_BIT_Value->toULLong();
	  break;
	 }
     case SIR_CONST_FLOAT:
	{ Const->F_Value = Saved_BIT_Value->toDouble();
	  break;
	 }
     case SIR_CONST_DOUBLE:
	{ Const->D_Value = Saved_BIT_Value->toDouble();
	  break;
	 }
     case SIR_CONST_LONGDOUBLE:
	{ Const->LD_Value = Saved_BIT_Value->toLDouble();
	  break;
	 }
     case SIR_CONST_BIT:
     case SIR_CONST_UBIT:
	{ assert(FALSE);	/* is handled elsewhere */
	 }
     case SIR_CONST_CHARSTRING:
	{ assert(FALSE);	/* cannot be converted */
	 }
     default:
	{ assert(FALSE);	/* bad Type */
	 }
    } /* hctiws */

Const->Type = Type;

delete Saved_BIT_Value;

} /* end of SIR_ConvertBIT */


static void SIR_ConvertLD(
	sir_constant	*Const,
	long double	LD_Value,
	SIR_CONSTTYPE	Type)
{

switch(Type)
   { case SIR_CONST_BOOL:
	{ Const->B_Value = (bool) LD_Value;
	  break;
	 }
     case SIR_CONST_CHAR:
	{ Const->C_Value = (char) LD_Value;
	  break;
	 }
     case SIR_CONST_UCHAR:
	{ Const->UC_Value = (unsigned char) LD_Value;
	  break;
	 }
     case SIR_CONST_SHORT:
	{ Const->S_Value = (short) LD_Value;
	  break;
	 }
     case SIR_CONST_USHORT:
	{ Const->US_Value = (unsigned short) LD_Value;
	  break;
	 }
     case SIR_CONST_INT:
	{ Const->I_Value = (int) LD_Value;
	  break;
	 }
     case SIR_CONST_UINT:
	{ Const->UI_Value = (unsigned int) LD_Value;
	  break;
	 }
     case SIR_CONST_LONG:
	{ Const->L_Value = (long) LD_Value;
	  break;
	 }
     case SIR_CONST_ULONG:
	{ Const->UL_Value = (unsigned long) LD_Value;
	  break;
	 }
     case SIR_CONST_LONGLONG:
	{
#ifdef HAVE_LLONG
	  Const->LL_Value = (LONG_LONG) LD_Value;
#else /* !HAVE_LLONG */
	  Const->LL_Value = LD_Value;
#endif /* HAVE_LLONG */
	  break;
	 }
     case SIR_CONST_ULONGLONG:
	{
#ifdef HAVE_LLONG
	  Const->ULL_Value = (UNSIGNED_LONG_LONG) LD_Value;
#else /* !HAVE_LLONG */
	  Const->ULL_Value = LD_Value;
#endif /* HAVE_LLONG */
	  break;
	 }
     case SIR_CONST_FLOAT:
	{ Const->F_Value = LD_Value;
	  break;
	 }
     case SIR_CONST_DOUBLE:
	{ Const->D_Value = LD_Value;
	  break;
	 }
     case SIR_CONST_LONGDOUBLE:
	{ Const->LD_Value = LD_Value;
	  break;
	 }
     case SIR_CONST_BIT:
     case SIR_CONST_UBIT:
	{ assert(FALSE);	/* handled earlier */
	 }
     case SIR_CONST_CHARSTRING:
	{ assert(FALSE);	/* cannot be converted */
	 }
     default:
	{ assert(FALSE);	/* bad Type */
	 }
    } /* hctiws */

Const->Type = Type;

} /* end of SIR_ConvertLD */


/* EOF Constant.cc */
