/************************************************************************/
/* IntRep/UserType.h: SpecC Internal Representation, User-Type Classes	*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 10/08/97 */
/************************************************************************/

/* last update: 06/15/04 */

/* modifications: (most recent first)
 *
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 04/17/02 RD	added SIR_UserTypes::NameAnonymousTypes() and
 *		SIR_UserTypes::CreateNewName() (bug fix)
 * 02/15/02 RD	added SIR_UserType::IsSameStructure()
 * 11/08/01 RD	switched code generation to use GL_IO layer
 * 10/10/01 RD	improved overall formatting of generated code
 * 05/31/01 RD	eliminated level 2 of SIR API
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 05/25/01 RD	introduced this header (last change was 04/13/99)
 */

#ifndef INTREP_USERTYPE_H
#define INTREP_USERTYPE_H


#include "Global.h"
#include "IntRep/Member.h"
#include "IntRep/Note.h"


/*** constants and macros ***********************************************/


	/* none */


/*** enumeration types **************************************************/


enum SIR_UserTypeClass		/* supported user types */
{				/* (see class SIR_UserType) */
SIR_USERTYPE_STRUCT,
SIR_USERTYPE_UNION,
SIR_USERTYPE_ENUM
};


/*** type definitions ***************************************************/


typedef enum SIR_UserTypeClass		SIR_USERTYPE;

typedef class SIR_UserType		sir_usertype;
typedef SIR_List<sir_usertype>		sir_usertype_list;
typedef class SIR_UserTypes		sir_usertypes;
typedef void *				sir_usertp_marg;
typedef ERROR (SIR_UserType::*sir_usertp_mptr)(void*);

typedef class SIR_Symbol		sir_symbol;	/* cyclic link */
typedef class SIR_Symbols		sir_symbols;	/* cyclic link */
typedef void *				sir_symbol_marg;/* cyclic link */
typedef ERROR (SIR_Symbol::*sir_symbol_mptr)(void*);	/* cyclic link */


/*** class declarations *************************************************/


	/********************/
	/*** SIR_UserType ***/
	/********************/


class SIR_UserType :		/* usertype (entry in a usertype table) */
	public SIR_ListElem<SIR_UserType>	/* is a list element */
{
public:
SIR_USERTYPE	Class;		/* class of this usertype (see above) */
string		*Name;		/* name of this usertype (NULL if unnamed) */
sir_type	*Type;		/* link to the type table entry */
sir_members	*Members;	/* list of members of this usertype */
sir_symbols	*Scope;		/* local scope (NULL if none) */
sir_notes	*Notes;		/* notes for this usertype (NULL if none) */
sir_symbol	*TypeDef;	/* link to type definition (NULL if none) */
sir_import	*Imported;	/* link to import file (NULL if not imported) */

sir_usertype	*Alias;		/* alias pointer (temporary only) */


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_UserType(			/* constructor #1 */
	SIR_USERTYPE	Class,
	const char	*Name,
	sir_type	*Type = NULL,
	sir_members	*Members = NULL,
	sir_symbols	*Scope = NULL,
	sir_notes	*Notes = NULL,
	sir_symbol	*TypeDef = NULL);

SIR_UserType(			/* constructor #3 (duplicator) */
	sir_usertype	*Original);

~SIR_UserType(void);		/* destructor */


BOOL IsNamed(void);		/* checks if Name is defined */

BOOL IsDefinition(void);	/* checks if Members are defined */

BOOL IsSameStructure(		/* checks if structure is the same */
	sir_usertype	*Other);

BOOL ContainsNested(		/* checks for nested struct/union */
	sir_usertype	*UserType);	/* (struct/union definitions only) */

BOOL LocalUTypeNeedsDef(		/* check whether a user-type member */
	sir_member	*Member);	/* needs a local definition	    */


sir_usertypes *GetTable(void);	/* determines the table where this is in */
				/* (returns NULL if not in a table) */

sir_symbols *GetScope(void);	/* determines the scope of this usertype */
				/* (returns NULL if links not available) */


ERROR DFS_ForAllNodes(		/* iterator over all nodes (depth first) */
	sir_node_mptr	MemberFct,
	sir_node_marg	MemberFctArg);

ERROR DFS_ForAllSymbols(	/* iterator over all symbols (depth first) */
	sir_symbol_mptr	MemberFct,
	sir_symbol_marg	MemberFctArg);

ERROR DFS_ForAllUserTypes(	/* iterator over all usertypes (depth first) */
	sir_usertp_mptr	MemberFct,
	sir_usertp_marg	MemberFctArg);

ERROR DFS_ForAllNotes(		/* iterator over all notes (depth first) */
	sir_note_mptr	MemberFct,
	sir_note_marg	MemberFctArg);


ERROR SetImported(		/* sets the Imported pointer (if unset) */
	sir_usertp_marg	ImportPtr);	/* or resets it if NULL */

ERROR TakeOverImport(		/* takes over an imported usertype */
	sir_usertp_marg	ImportPtr);

ERROR MarkImportEntry(		/* marks its import entry being used */
	sir_usertp_marg	/* Unused */);

ERROR MarkUsedTypes(		/* marks the type entries used here */
	sir_usertp_marg	/* Unused */);

ERROR DeleteYourType(		/* deletes its associated type */
	sir_usertp_marg	/* Unused */);


void SetAlias(			/* sets all type, usertype, symbol alias' */
	sir_usertype	*Alias);/* (iterates over symbols and usertypes) */

void UnAlias(			/* unalias all type, usertype, symbol links */
	sir_symbols	*GlobalSymbols);


const char *ClassName(void);	/* returns its Class as string (eg. "struct") */

static const char *ClassName(	/* returns Class as string (eg. "struct") */
	SIR_USERTYPE	Class);

const char *NameOrUnnamed(void);/* returns Name or "<unnamed>" */

const char *PrettyString(		/* creates a SpecC string */
	string		*Buffer,	/* (one of the ones below) */
	BOOL		IncludeNotes,
	BOOL		CplusplusMode = FALSE,
	BOOL		UserTypeDefReq = FALSE);

const char *PrettyStringN(		/* PrettyString with name */
	string		*Buffer);	/* e.g. "struct Name" */

const char *PrettyStringT(		/* PrettyString with typedef */
	string		*Buffer);	/* e.g. "TypeDefName" */

const char *PrettyStringD(		/* PrettyString with definition */
	string		*Buffer,	/* e.g. "struct [Name] { ... }" */
	BOOL		IncludeNotes,
	BOOL		CplusplusMode);

#ifndef NDEBUG
const char *Print(void);	/* print SpecC code */
#endif /* NDEBUG */

ERROR WriteSC(				/* (re-) generates SpecC source code */
	GL_HANDLE	*IO,		/* (for usertype _definitions_) */
	BOOL		WriteNotes,
	BOOL		CplusplusMode);


ERROR Annotate(				/* attach a note to the usertype */
	sir_note	*NewNote);	/* (consumes NewNote) */
};


	/*********************/
	/*** SIR_UserTypes ***/
	/*********************/


class SIR_UserTypes :		/* user-defined type "table" */
	public SIR_List<SIR_UserType>	/* is simply a list of user types */
{					/* with additional member(function)s */
public:
sir_usertypes	*Parent;	/* link to parent UserType table (or NULL) */
sir_symbols	*Symbols;	/* link back to symbol table on same level */


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_UserTypes(			/* constructor #1 */
	sir_usertypes	*Parent,
	sir_symbols	*Symbols);

SIR_UserTypes(			/* constructor #3 (duplicator) */
	sir_usertypes	*Original);

~SIR_UserTypes(void);		/* destructor */


ERROR DFS_ForAllNodes(		/* iterator over all nodes (depth first) */
	sir_node_mptr	MemberFct,
	sir_node_marg	MemberFctArg);

ERROR DFS_ForAllSymbols(	/* iterator over all symbols (depth first) */
	sir_symbol_mptr	MemberFct,
	sir_symbol_marg	MemberFctArg);

ERROR DFS_ForAllUserTypes(	/* iterator over all usertypes (depth first) */
	sir_usertp_mptr	MemberFct,
	sir_usertp_marg	MemberFctArg);

ERROR DFS_ForAllNotes(		/* iterator over all notes (depth first) */
	sir_note_mptr	MemberFct,
	sir_note_marg	MemberFctArg);


void SetAlias(			/* sets all type, usertype, symbol alias' */
	sir_usertypes	*Alias);/* (iterates over symbols and usertypes) */

void UnAlias(			/* unalias all type, usertype, symbol links */
	sir_symbols	*GlobalSymbols);


sir_usertype *Find(		/* find an entry globally (here and above) */
	const char	*Name);	/* (returns NULL if not found) */

sir_usertype *FindLocally(	/* find an entry locally (only in this table) */
	const char	*Name);	/* (returns NULL if not found) */

sir_usertype *FindLocallyViaTypeDef(	/* find an entry via its typedef name */
	const char	*TypeDefName);	/* (returns NULL if not found) */

sir_usertype *Declare(		/* declares a user type */
	SIR_USERTYPE	Class,	/* (may return NULL and SIR_Error) */
	const char	*Name,	/* (consumes Scope) */
	sir_symbols	*Scope,
	sir_types	*TypeTable);

ERROR Define(			/* defines a declared user type */
	sir_usertype	*Declaration,	/* (consumes Members) */
	sir_members	*Members);

sir_usertype *Insert(		/* inserts a new user type */
	sir_usertype	*NewUserType);	/* (the argument is consumed!) */

ERROR Integrate(		/* integrates imported user types */
	sir_usertypes	*Imported);


void NameAnonymousTypes(void);	/* preprocessing step for WriteSC */

const char *CreateNewName(	/* generate a new name in this table */
	const char	*Format);	/* (must contain one %d) */


#ifndef NDEBUG
const char *Print(void);	/* print SpecC code */
#endif /* NDEBUG */

ERROR WriteSC(			/* (re-) generates SpecC (or C++) source code */
	GL_HANDLE	*IO,
	BOOL		WriteNotes,
	BOOL		CplusplusMode,
	BOOL		*PutSeparator);

ERROR WriteOrderedSC(		/* writes ordered structures/unions */
	sir_usertype	*Current,	/* (recursively) */
	GL_HANDLE	*IO,
	BOOL		WriteNotes,
	BOOL		CplusplusMode,
	BOOL		*PutSeparator,
	BOOL		*PutIntro,
	BOOL		PutComments);
};


#endif /* INTREP_USERTYPE_H */

/* EOF IntRep/UserType.h */
