/************************************************************************/
/* Global.h: global definitions, error handling, memory management...	*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 07/02/97 */
/************************************************************************/

/* last update: 06/15/04 */

/* modifications: (most recent first)
 *
 * 06/15/04 PC  Adjustments for scrc 2.0
 * 12/17/03 RD	increased version to 2.2.c, extended copyright to 2004
 * 12/16/02 RD	increased version to 2.2.b, extended copyright to 2003
 * 10/23/02 RD	reallocated error ranges, added entry for OS tool
 * 09/16/02 RD	downgraded the version number to 2.2.a for an alpha release
 * 07/19/02 RD	increased version to 2.2.0
 * 07/05/02 RD	increased SIR_ERROR_RANGE to 499
 * 06/20/02 RD	replaced deprecated header <new.h> with <new>
 * 05/27/02 RD	updated error ranges for current tool set
 * 01/30/02 RD	increased range of SIR error codes (SIR_ERROR_RANGE)
 * 01/03/02 RD	updated copyright range to 2002
 * 11/20/01 RD	added support for 'new'/'delete' exceptions
 *		(conditional compilation with GL_MEM_USE_EXCEPTIONS)
 * 11/15/01 RD	added inclusion of GL_System.h
 * 11/02/01 RD	moved IO support functions into separate file GL_FileIO.h
 * 08/08/01 RD	adjusted version to 1.2
 * 08/06/01 RD	adjusted version to 1.1
 * 05/26/01 RD	eliminated support for binary SIR files (import/export)
 * 05/25/01 RD	removed code not needed for the SCRC
 * 05/24/01 RD	added copyright string definition
 * 05/03/01 RD	replaced GL_OpenFileWithDirList() by GL_FindFileInDirList()
 * 04/25/01 RD	added "union-safe" types LONG_LONG_ and UNSIGNED_LONG_LONG_
 * 04/06/01 RD	added type definition TIME
 * 04/06/01 AG	added type definitions LONG_LONG and UNSIGNED_LONG_LONG
 * 04/06/01 RD	started this header (last update was 06/18/00)
 */

#ifndef GLOBAL_H
#define GLOBAL_H

#include <stdio.h>
#include <new>
#include "GL_String.h"
#include "GL_FileIO.h"
#include "GL_System.h"
#include "longlong.h"


/*** type definitions ***************************************************/


typedef bool		BOOL;
typedef double		FLOAT;	/* use double precision floating point */

typedef String          string; /* use our own Strings */

#ifdef HAVE_LLONG
/* use the built-in "long long" */
typedef long long		LONG_LONG;
typedef unsigned long long	UNSIGNED_LONG_LONG;
typedef long long		LONG_LONG_;		// type inside unions
typedef unsigned long long	UNSIGNED_LONG_LONG_;	// type inside unions
#else
/* use our own long long implementation */
typedef _longlong<false>	LONG_LONG;
typedef _longlong<true>		UNSIGNED_LONG_LONG;
typedef _longlong_<false>	LONG_LONG_;		// type inside unions
typedef _longlong_<true>	UNSIGNED_LONG_LONG_;	// type inside unions
#endif

typedef UNSIGNED_LONG_LONG	TIME;	/* time is unsigned long long */


/*** constants and macros ***********************************************/


	/* flag for DEBUG mode */

/* #define GL_DEBUG */
/* #define MEM_DEBUG */


	/* System's name, version number, and copyright */

#define GL_SPECC_NAME			"SCRC"
#define GL_SPECC_VERSION_STRING		"2.0"
#define GL_SPECC_COPYRIGHT_STRING	\
		"(c) 1997-2004 CECS, University of California, Irvine"


	/* SpecC language definitions */

#define GL_MAIN_BEHAVIOR_NAME	"Main"
#define GL_MAIN_METHOD_NAME	"main"
#define GL_MAIN_FUNCTION_NAME	"main"


	/* function macros ABS(), MIN() and MAX() */
	/* (caution! side effects possible!) */

#define MIN(a, b)	((a) < (b) ? (a) : (b))
#define MAX(a, b)	((a) > (b) ? (a) : (b))
#define ABS(a)		((a) > (0) ? (a) : (-(a)))


	/* boolean values */

#ifndef FALSE
#define FALSE		false
#endif /* FALSE */
#ifndef TRUE
#define TRUE		true
#endif /* TRUE */


	/* result values at program termination */

#define RESULT_OK	 0	/* everything ok */
#define RESULT_ERROR	10	/* fatal error */


	/* memory management */

#define GL_MEM_NUMBER_RETRYS	5	/* number of malloc() retrys */
#ifdef __STRICT_ANSI__
#define GL_MEM_USE_EXCEPTIONS		/* 'new' and 'delete' do 'throw' */
#endif /* __STRICT_ANSI__ */


	/* floating point precision */

#define GL_FLOAT_OUT_FMT	"% 16.6f"	/* format string for printf() */
#define GL_FLOAT_IN_FMT		"%lf"		/* format string for scanf() */
#define GL_EPSILON	((FLOAT)0.000001)	/* precision for comparisons */


	/* common warning levels */

#define GL_WARN_ALWAYS		0	/* (better use error for this) */
#define GL_WARN_IMPORTANT	1	/* important warnings */
#define GL_WARN_STANDARD	2	/* standard warnings */
#define GL_WARN_INFORMATIVE	3	/* informative warnings */
#define GL_WARN_PEDANTIC	4	/* pedantic warnings */


	/* error code allocation for all modules */

#define NO_ERROR	0	/* return value reserved for "no error" */
#define FLAG_YES	1	/* return value reserved for "yes" */
#define FLAG_NO		0	/* return value reserved for "no" */

#define GL_ERROR_BASE	1000	/* global		libGlobal */
#define GL_ERROR_RANGE	  99
#define SIR_ERROR_BASE	2000	/* int. representation	libIntRep */
#define SIR_ERROR_RANGE	 499
#define CCD_ERROR_BASE	3000	/* compiler driver	libCcDriver */
#define CCD_ERROR_RANGE	  99
#define PRS_ERROR_BASE	3100	/* SpecC parser		libParser */
#define PRS_ERROR_RANGE	  99
#define SCC_ERROR_BASE	3200	/* SpecC compiler	scc */
#define SCC_ERROR_RANGE	  99


	/* error message output */

#define GL_ERROR_MSG_NEWLINE	"\n             "
#define GL_WARN_MSG_NEWLINE	"\n         "


	/* error codes for this module */

#define GL_ERROR_NO_ERROR			(NO_ERROR)
#define GL_ERROR_OUT_OF_MEMORY			(GL_ERROR_BASE +  1)
#define GL_ERROR_FILE_R_OPEN_FAILED		(GL_ERROR_BASE +  2)
#define GL_ERROR_FILE_W_OPEN_FAILED		(GL_ERROR_BASE +  3)
#define GL_ERROR_PROBLEM_WHILE_CLOSING_FILE	(GL_ERROR_BASE +  4)
#define GL_ERROR_FILE_NOT_READABLE		(GL_ERROR_BASE +  5)
#define GL_ERROR_UNLINK_FAILED			(GL_ERROR_BASE +  6)
#define GL_ERROR_READ_BYTES_FAILED		(GL_ERROR_BASE +  7)
#define GL_ERROR_WRITE_BYTES_FAILED		(GL_ERROR_BASE +  8)
#define GL_ERROR_NOT_AT_END_OF_FILE		(GL_ERROR_BASE +  9)
#define GL_ERROR_NOT_A_LEGAL_IDENTIFIER		(GL_ERROR_BASE + 10)
#define GL_ERROR_NOT_A_LEGAL_IDENTIFIER_2	(GL_ERROR_BASE + 11)
#define GL_ERROR_NOT_A_LEGAL_IDENTIFIER_3	(GL_ERROR_BASE + 12)
#define GL_ERROR_NOT_A_LEGAL_FILENAME		(GL_ERROR_BASE + 13)
#define GL_ERROR_NOT_A_LEGAL_FILE_PATH		(GL_ERROR_BASE + 14)
#define GL_ERROR_TMP_FILE_CREATE_FAILED		(GL_ERROR_BASE + 15)
#define GL_ERROR_TMP_FILE_REWIND_FAILED		(GL_ERROR_BASE + 16)
#define GL_ERROR_FILE_DELETE_FAILED		(GL_ERROR_BASE + 17)
#define GL_ERROR_ILLEGAL_ARGUMENTS		(GL_ERROR_BASE + 18)
#define GL_ERROR_RW_MODE_VIOLATION		(GL_ERROR_BASE + 19)
#define GL_ERROR_FILE_RW_OPEN_FAILED		(GL_ERROR_BASE + 20)
#define GL_ERROR_FILE_IO_ERROR			(GL_ERROR_BASE + 21)
#define GL_ERROR_NOT_AT_END_OF_STRING		(GL_ERROR_BASE + 22)


/*** exported variables *************************************************/


extern ERROR	GL_Error;		/* last error of this module */
extern string	GL_ErrorMessage;

extern int	GL_VerbosityLevel;	/* (published by main module) */
extern int	GL_WarningLevel;	/* (published by main module) */


/************************************************************************/
/*** prototype definitions for exported functions		      ***/
/************************************************************************/


/*** error handling *****************************************************/


	/* recover from error status */

extern void GL_Reset(void);


	/* error code to error message conversion */

extern const char *GL_ErrorText(
	ERROR	ErrorNumber);


	/* determine error message from the OS (errno, etc.) */

extern const char *GL_SystemErrorMessage(void);


	/* output error and warning messages */

extern void GL_PrintError(
	int		ErrorNumber,
	const char	*ErrorMsg);

extern void GL_PrintErrorFmt(
	int		ErrorNumber,
	const char	*Format,
	...);

extern void GL_PrintWarning(
	int		TestWarnLevel,
	const char	*WarnMsg);

extern void GL_PrintWarningFmt(
	int		TestWarnLevel,
	const char	*Format,
	...);


/*** memory management **************************************************/


	/* overloaded C++ memory management */

extern void *operator new (
	size_t	NumBytes)
#ifdef GL_MEM_USE_EXCEPTIONS
	throw (std::bad_alloc)
#endif /* GL_MEM_USE_EXCEPTIONS */
	;

extern void *operator new[] (
	size_t		NumBytes)
#ifdef GL_MEM_USE_EXCEPTIONS
	throw (std::bad_alloc)
#endif /* GL_MEM_USE_EXCEPTIONS */
	;

extern void operator delete (
	void		*MemAddress)
#ifdef GL_MEM_USE_EXCEPTIONS
	throw ( )
#endif /* GL_MEM_USE_EXCEPTIONS */
	;

extern void operator delete[] (
	void		*MemAddress)
#ifdef GL_MEM_USE_EXCEPTIONS
	throw ( )
#endif /* GL_MEM_USE_EXCEPTIONS */
	;


	/* smarter versions of malloc() and free() */

extern void *GL_malloc(
	size_t		NumBytes);

extern void GL_free(
	void		*MemAddress);


	/* spare memory handling */

extern ERROR GL_ReserveSpareMemory(
	size_t		NumBytes);

extern void GL_FreeSpareMemory(void);


/*** general support ****************************************************/


extern ERROR GL_CheckIdentifier(	/* checks for a legal identifier */
	const char	*Name);

extern ERROR GL_CheckSpecCIdentifier(	/* checks for a legal identifier */
	const char	*Name);		/* and not being a SpecC keyword */

extern ERROR GL_CheckFilename(		/* checks for a legal filename */
	const char	*Filename);

extern ERROR GL_CheckPathFilename(	/* checks for a legal filename */
	const char	*PathFile);	/* with local or absolute path */

extern const char *GL_GetFileFromPath(	/* searches file name in a path */
	const char	*Path);

extern const char *GL_GetBaseName(	/* obtains the base name from a path */
	const char	*Path);		/* (file name without suffix) */


#endif /* GLOBAL_H */

/* EOF Global.h */
