// ============================================================================
// bit.cc: Bit Vector Library
// ============================================================================
//
// 09/27/01 <gerstl>	Fixed bug in ubit2str() with base != 2
// 08/10/01 <gerstl>	Introduced len_t, bnd_t; changed shift cnt to unsigned
// 06/01/01 <gerstl>	Resolved conflict of method names with ANSI (keywords?)
// 05/24/01 <gerstl>	Initial version
//
// ----------------------------------------------------------------------------


// Enable assertions in library in any case 
// (default for library is "on" but disabled with -DNDEBUG ...)
#define _BITASSERT


#ifdef HAVE_LLONG

/* use the built-in "long long" */
typedef long long		LONG_LONG;
typedef unsigned long long	UNSIGNED_LONG_LONG;

#define ll2int(ll)		((int)(ll))
#define ll2uint(ll)		((unsigned int)(ll))

#else

/* use our own long long implementation */
#include <longlong.h>
typedef _longlong<false>	LONG_LONG;
typedef _longlong<true>		UNSIGNED_LONG_LONG;

#define ll2int(ll)		((ll).toInt())
#define ll2uint(ll)		((ll).toUInt())

#endif


#include <stdarg.h>
#include <string.h>
#include <ctype.h>

#include "sys/bit.h"


/* ------------------------------------------------------------------------- */
/* Class _bitbus_element */


// Add ourselves at the tail of the given list and return list.
_bitbus_element* _bitbus_element::concat(_bitbus_element* list)
{
  _bitbus_element* el;

  // Walk through list to end;
  for (el = list; el->next != NULL; el = el->next) ;

  // Merge the guys
  el->next = this;

  // Now test for merging..

  // Same vector?
  if (el->ref.addr() != ref.addr()) return list;

  // Check directions 
  int rsign = _BITSIGN(el->sl,   el->sr);
  int lsign = _BITSIGN(sl, sr);

  // Check sign
  if (lsign == rsign) {
    // Same sign, both no direction?
    if (lsign) {
      // Abort if not connected
      if (sr != el->sl + lsign) return list;
    }
    else {
      // No direction, check if connected in any direction
      if (_BITLEN(sr, el->sl) != 2) return list;
    }
  }
  else {
    // Not the same sign, check if any one has no direction
    if (lsign) {
      if ((rsign) || (sr != el->sl + lsign)) return list;
    } else {
      if (sr != el->sl + rsign) return list;
    }
  }

  // Merge the guys!
  el->sl = sl;
  el->next = next;
  delete this;

  // ... and exit finally
  return list;
}


// Return newly allocated list that is a slice of this list.
_bitbus_element* _bitbus_element::slice(len_t l, len_t r)
{
  len_t i, newsl, newsr;
  len_t ellen = _BITLEN(sl, sr);
  len_t max = _MAXLEN(l,r);
  len_t min = _MINLEN(l,r);
  _bitbus_element* head = NULL;
  _bitbus_element** tmp = &head;
  _bitbus_element* el;

  // Walk through the list until we hit parts of the slice
  for (el = this, i = ellen; min >= i; i += ellen) {
    el = el->next;
    ellen = _BITLEN(el->sl, el->sr);    
  }
  i -= ellen;
  
  // Copy and transform elements until we leave slice
  for ( ; i <= max; i += ellen) {

    // Map slice coordinates
    if (el->sl > el->sr) {
      newsl = el->sl - _MAXLEN(i + ellen - 1 - max, 0);
      newsr = el->sr + _MAXLEN(min - i, 0);
    } else {
      newsl = el->sl + _MAXLEN(i + ellen - 1 - max, 0);
      newsr = el->sr - _MAXLEN(min - i, 0);
    }

    if (l > r) {
      // Add elements at tail of list
      *tmp = new _bitbus_element(el->ref, newsl, newsr);
      tmp = &((*tmp)->next);
    } else {
      // Add elements at the head of list
      head = new _bitbus_element(el->ref, newsr, newsl, head);
    }

    el = el->next;
    ASSERT(el || ((i+ellen) > max));  // Tried to slice outside of range...
    if (!el) break;
    ellen = _BITLEN(el->sl, el->sr);    
  }

  ASSERT(head);  // Did we cut anything?
  return head;
}


// Sync with original vectors
void _bitbus_element::put(const _bit& buf)
{
  len_t i, len;
  _bitbus_element *el = this;
  
  for (i = 0; el; i += len, el = el->next) {
    len = _BITLEN(el->sl, el->sr);
    el->ref.copyBits(buf, i + len - 1, i, el->sl, el->sr);
  };
}

void _bitbus_element::get(const _bit& buf)
{
  // ## We could do some checksumming here to find out if the originals
  //    really have changed (room for optimization)
  len_t i, len;
  const _bitbus_element *el = this;
  
  for(i = 0; el; i += len, el = el->next) {
    len = _BITLEN(el->sl, el->sr);
    const_cast<_bit&>(buf).copyBits(el->ref, el->sl, el->sr, i + len - 1, i);
  };
}


/* ------------------------------------------------------------------------- */
/* Class _bit */


/* --- Shifting --- */

_bit& _bit::shlOp(const _bit& op1, unsigned int sh, bool mask)
{
  ASSERT((len_t)(op1.len+sh) >= len);
  
  // Divide shift in chunks
  int c = (sh / _BITCHUNK_SIZE);
  int s = sh % _BITCHUNK_SIZE;

  // Loop over chunks
  for (int i = _BITCHUNKS(len)-1; i >= 0; i--) 
  {
    // Masked?
    if (mask && (i == c)) {
      // Leave shifted in part untouched!
      vec[i].uval &= _BITMASK_LSB_SET(s);
      vec[i].uval |= op1.vec[i-c].uval << s;  
      break;
    }

    // Shifting in zeros or lower chunk?
    if (i >= c) {
      vec[i].uval = op1.vec[i-c].uval << s;  
    } else {
      vec[i].uval = 0;
    }

    // Set bits shifted in correctly
    if ((s) && (i > c))
      vec[i].uval |= op1.vec[i-c-1].uval >> (_BITCHUNK_SIZE - s);    
  }
  
  return fill(_BITCHUNK(len-1));
}


_bit& _bit::shrOp(const _bit& op1, unsigned int sh)
{
  int i;
  
  // Divide shift in chunks
  int c = sh / _BITCHUNK_SIZE;
  int s = sh % _BITCHUNK_SIZE;

  // Loop over chunks of operand and result
  for (i = 0; (i < _BITCHUNKS(len)) && ((i+c+1) < _BITCHUNKS(op1.length())); i++) 
  {
    // Shifting in higher chunk?
    vec[i].uval = op1.vec[i+c].uval >> s; 

    // Set bits shifted in correctly
    if (s) vec[i].uval |= op1.vec[i+c+1].uval << (_BITCHUNK_SIZE - s);
  }
  
  // Did we reach the end of the result already?
  if(i == _BITCHUNKS(len)) return fill(i-1);
  
  // Get sign extension of operand
  _bit::chunk ex = op1.sign();
  
  // Any operand chunk at all?
  if( (i+c) < _BITCHUNKS(op1.length()) )
  {
    // Shift in last operand chunk...
    vec[i].uval = op1.vec[i+c].uval >> s;
  
    // ... and set shifted in bits
    if (s) vec[i].uval |= ex.uval << (_BITCHUNK_SIZE - s);
    
    i++;
  }
  
  // Finally, fill any remaining chunks
  for( ; i < _BITCHUNKS(len); i++) vec[i] = ex;
  
  return fill(i-1);
}


/* --- Multiplication --- */

inline _bit& _bit::mulOp(const _bit& op1, const _bit& op2) 
{
  ASSERT(op1.len >= len);
  
  UNSIGNED_LONG_LONG tmp = 0;

  // Work on a copy in case we are one of the operands
  if ((op1.addr() == addr()) || (op2.addr() == addr())) {    
    _bit r(length(), isUnsigned());
    r.mulOp(op1, op2);    
    return operator=(r);
  }
  
  int i;
  for (i = 0; i < _BITCHUNKS(len); i++) {
    for (int j = 0; j <= i; j++) 
    {
      tmp += (UNSIGNED_LONG_LONG)op1.vec[j].uval * 
             (UNSIGNED_LONG_LONG)op2.extend(i-j).uval;
    }
    vec[i].uval = ll2uint(tmp);
    tmp >>= _BITCHUNK_SIZE;
  }
  return fill(i-1);
}



/* --- Division --- */

// Reference: Knuth's "The Art of Computer Programming", 
//            Volume 2, Seminumerical Algorithms

// If r is non-NULL no quotient is calculated, i.e. this->vec is untouched.
_bit& _bit::divOp(const _bit& n, const _bit& d, _bit* r) 
{
  int i, j;
  chunk q, d0;
  UNSIGNED_LONG_LONG nX;

  // Length, pointers
  int nsize = _BITCHUNKS(n.len);
  int dsize = _BITCHUNKS(d.len);
  const chunk* dp = d.vec;
  const chunk* np = n.vec;
  chunk* qp = r? 0 : vec;

  // Skip leading sign extensions
  while ((dp[dsize-1].uval == d.sign().uval) && (dsize > 1)) dsize--;
  while ((np[nsize-1].uval == n.sign().uval) && (nsize > 1)) nsize--;

  const bool dNeg = (d.sign().sval < 0);
  const bool nNeg = (n.sign().sval < 0); 
  bool rNeg = nNeg;

  // Anything to do?
  if (dsize > nsize) {
    if (r) {
      if (np != r->vec) *r = n;
    } else {
      *this = 0;
    }
    return *this;
  }

  // Special treatment for divider with just one chunk
  switch (dsize)
  {
    case 1:
    {
      // Make denominator positive
      if (dNeg) d0.sval = -dp[0].sval; else d0 = dp[0];

      // Initialize partial remainder
      nX = (LONG_LONG)n.sign().sval;

      for (i = nsize - 1; i >= 0; i--) 
      {
	// Set nominator for division
	nX = (nX << _BITCHUNK_SIZE) + np[i].uval;

	// Divide, get remainder
	if (rNeg) {
	  // Special handling for first chunk of a negative numerator
	  nX = -nX;
	  q.uval = ll2uint( -(nX / d0.uval) );
	  nX     =   nX % d0.uval;
	  if (nX != 0) { q.uval--;  nX = d0.uval - nX; }
	} else {
	  q.uval = ll2uint( nX / d0.uval );
	  nX     = nX % d0.uval;
	}

	// Set result
	if (qp && (i < _BITCHUNKS(len))) { 
	  if (dNeg) q.uval = ~q.uval; 
	  qp[i] = q; 
	}
	
	// Remainder now positive
	rNeg = false;
      }

      // Correction?
      rNeg = nNeg && (nX != 0);
      
      // Output values
      if (r) {
	// Set remainder, final adjust to make negative in case of n < 0
	if (rNeg) nX = (LONG_LONG)nX - d0.uval; 
	*r = (LONG_LONG)(nX);
      }
    }
    break;

    default:
    {
      int   norm = 0;
      chunk r0, n2, d1;
      chunk *rtmp = 0;
      chunk *rp;
      UNSIGNED_LONG_LONG dX, rX;

      // Get denominator
      dX   = ((UNSIGNED_LONG_LONG)dp[dsize-1].uval << _BITCHUNK_SIZE);
      dX  += dp[dsize-2].uval;
      
      // Determine normalization of denominator
      d0.sval = ll2uint((dNeg? -dX : dX) >> _BITCHUNK_SIZE);
      while( !(_BITMASK_MSB_SET(norm+1) & d0.uval) ) norm++;
            
      // Normalize it
      if( norm ) {
	dX <<= norm;
	if(dsize > 2) dX |= dp[dsize-3].uval >> (_BITCHUNK_SIZE - norm);
      }

      // Finally, make positive and get chunks
      if (dNeg) dX = -dX;
      d1.uval = ll2uint(dX);
      d0.uval = ll2uint(dX >> _BITCHUNK_SIZE);

      // Where do we put the remainder?
      if ((!r) || (_BITCHUNKS(r->len) < nsize) || (dp == r->vec)) {
	// No remainder given or doesn't fit -> make copy
	rtmp = rp = new chunk[nsize];
      } else {
	// Otherwise keep remainder in r
	rp = r->vec;
      }

      // Create bit vector around remainder array
      _bit rv(rp, nsize * _BITCHUNK_SIZE, n.isUnsigned());
      
      // Make a copy of numerator
      rv = n;

      // Don't overwrite denominator: put quotient in upper part of remainder
      if (dp == qp) qp = &rp[dsize-1];

      // Initialize first numerator piece
      n2 = n.sign();
      if (nNeg) 
	nX = (LONG_LONG)rp[nsize - 1].sval; 
      else 
	nX = rp[nsize - 1].uval;
      
      // Normalize that, too
      if( norm ) {
	nX <<= norm;
	nX  |= rp[nsize-2].uval >> (_BITCHUNK_SIZE - norm);
      }
      
      // And make positive
      if( nNeg ) nX = -nX;

      // Loop over quotient chunks
      for (i = nsize - dsize; i >= 0; i--)
      {
	// Divivde by most significant divisor chunk
	q.uval  = ll2uint( nX / d0.uval );
	r0.uval = ll2uint( nX % d0.uval );

	// Product with consideration of next chunk
	nX = (UNSIGNED_LONG_LONG)q.uval * (UNSIGNED_LONG_LONG)d1.uval;

	// Prepare extended remainder
	rX = rp[i+dsize-2].uval << norm;
	if( norm && ((i + dsize) > 2) ) {
	  rX &= rp[i+dsize-3].uval >> (_BITCHUNK_SIZE - norm);
	}
	
	// Adjust for negative effects
	if (rNeg) {
	  rX = -(rX + (2 - dNeg));

	  // Take care of underflow: rX has to be positive below for comparison
	  while ((((UNSIGNED_LONG_LONG)r0.uval << _BITCHUNK_SIZE) + rX)
		 > ((UNSIGNED_LONG_LONG)r0.uval << _BITCHUNK_SIZE)) 
	  {
	    q.uval--; 
	    r0.uval += d0.uval; 
	  }
	} else if (dNeg) {
	  rX++;
	  // Check for overflow
	  if ((((UNSIGNED_LONG_LONG)r0.uval << _BITCHUNK_SIZE) + rX) == 0) 
	    nX = 0;
	}	  

	// Did the division exceed the goal?
	while (nX > (((UNSIGNED_LONG_LONG)r0.uval << _BITCHUNK_SIZE) + rX))
	{
	  // Yes, it did -> back one notch
	  q.uval--;
	  r0.uval += d0.uval;
	  if (r0.uval < d0.uval) break;    // overflow?
	  nX -= d1.uval;
	}
	
	// Negative: Correct to make sure we are exactly one notch above
	if (rNeg) q.uval++;

	// Ok, subract the partial product
	// _bit(rp[i], dsize*_BITCHUNK_SIZE, true, false) -= d*q;
	for (nX = 0, j = i; j < dsize + i; j++) {
	  // Multiplication
	  nX += (UNSIGNED_LONG_LONG)d.extend(j-i).uval 
                          	               * (UNSIGNED_LONG_LONG)q.uval;
	  r0.uval = ll2uint(nX);
	  nX >>= _BITCHUNK_SIZE;

	  // Subtract it
	  if (nNeg ^ dNeg) {
	    rp[j].uval += r0.uval;
	    if (rp[j].uval < r0.uval) nX++;  // Carry out
	  } else {
	    if (r0.uval > rp[j].uval) nX++;  // Borrow for next subtraction
	    rp[j].uval -= r0.uval;
	  }
	}

	// Get output chunk of "submul"
	r0.uval = ll2uint(nX);
	if (dNeg) r0.uval = q.uval - r0.uval;
	if (rNeg) r0.uval = -r0.uval;

	// Quotient is negative if negative remainder
	if (rNeg) q.sval = -q.sval;

	// Overflow?
	if (n2.uval != r0.uval) 
	{
	  // Correct, i.e. up one notch again
	  _bit dummy(&rp[i], dsize * _BITCHUNK_SIZE, true);
	  if (dNeg)  dummy -= d; else dummy += d;

	  //if (nNeg ^ dNeg) nX = 1; else nX = 0;
	  //for (j = i; j < dsize + i; j++) {
	  //  nX += rp[j].uval;
	  //  if (nNeg ^ dNeg) {
	  //    nX += (~d.extend(j-i).uval);
	  //  } else {
	  //    nX += d.extend(j-i).uval;
	  //  }
	  //  rp[j].uval = nX;
	  //  nX >>= _BITCHUNK_SIZE;
	  //}
	  q.uval--;
	}

	// Construct numerator for next iteration
	n2 = rp[i + dsize - 1];
	nX = ((UNSIGNED_LONG_LONG )n2.uval << _BITCHUNK_SIZE)
                                                      + rp[i + dsize - 2].uval;

	// And normalize it
	if( norm ) {
	  nX <<= norm;
	  if( (i + dsize) > 2 ) {
	    nX |= rp[i + dsize - 3].uval >> (_BITCHUNK_SIZE - norm);
	  }
	}
      
	// Remainder is certainly positive now
	rNeg = false;

	// Assign quotient chunk
	if (qp && (i < _BITCHUNKS(len))) {
	  if (dNeg) q.uval = ~q.uval;
	  qp[i] = q;
	}
      }    

      // Do we have to do final correction in case of negative remainder?
      if (nNeg) rNeg = rv.test();
      
      // Assign remainder, if requested
      if (r) 
      {
	// Extend remainder if necessary
	for (i = dsize; i < nsize; i++) rp[i].uval = 0;

	// Do we have to copy?
	if (rp != r->vec) 
	{
	  // Assign & Correct if necessary
	  if (rNeg) {
	    if (dNeg) *r = rv + d; else *r = rv - d;
	  } else {
	    *r = rv;
	  }
	} else {
	  // No: perform operations in place
	  if (rNeg) {
	    if (dNeg) rv += d; else rv -= d;
	  }

	  // Extend to full length
	  for (i = nsize; i < _BITCHUNKS(r->len); i++) rp[i] = rv.sign();
	}
      }
      if( rtmp ) delete[] rtmp;
    }
  }

  // Quotient output
  if (qp) 
  {
    chunk qsign;
        
    // Determine sign extension of quotient
    if(dNeg ^ nNeg) qsign.sval = -1; else qsign.uval = 0;
    
    // Adjust negative result?
    if( dNeg ^ rNeg ) {
      // Increment by one
      nX = 1;
      
      // Loop over computed quotient chunks and increment
      for(i = 0; (i < _BITCHUNKS(len)) && (i < (nsize - dsize + 1)); i++) {
	nX += qp[i].uval;
	vec[i].uval = ll2uint(nX);
	nX >>= _BITCHUNK_SIZE;
      }
      
      // Loop over any remaining chunks, sign extend quotient
      for( ; i < _BITCHUNKS(len); i++) {
	nX += qsign.uval;
	vec[i].uval = ll2uint(nX);
	nX >>= _BITCHUNK_SIZE;
      }
    }
    else {
      // Do we have to copy the result? 
      if (qp != vec) {
	// Yes: make a copy of quotient
	for(i = 0; (i < _BITCHUNKS(len)) && (i < (nsize - dsize + 1)); i++)
	  vec[i] = qp[i];
      }
      
      // Initialize any quotient chunks we didn't visit
      for (i = nsize - dsize + 1; i < _BITCHUNKS(len); i++) vec[i] = qsign;
    }
    
    fill(i-1);
  }

  return *this;
}



/* --- String assignment --- */

_bit& _bit::set (const char* str, bool usign, unsigned int base)
{
  unsigned int digit;
  len_t  l = 0;
  bool neg = false;

  // Eat leading whitespace
  while( isspace(*str) ) str++;
  
  // Eat leading sign
  switch( *str ) {
    case '-': neg = true;
    case '+': usign = true;
              str++;
  }
  
  // Base given?
  if( base < 2 ) 
  {
    // Nope, default is 2
    base = 2;
    
    // Hexadecimal constant?
    if( *str == '0' && ( *str == 'x' || *str == 'X' ) ) {
      base = 16;
      str += 2;
    }
  }

  // Skip leading zeros
  while( *str == '0' ) { str++; l++; }
  
  // Loop over digits
  operator=(0);
  for( ; *str >= '0'; str++)
  {
    if( *str > '9' ) {
      if( *str < 'A' ) break;
      digit = toupper(*str) - 'A' + 10;
    } else {
      digit = *str - '0';
    }
    
    if( digit >= base ) break;
        
    switch(base) 
    {
      case 2:  operator<<=(1); 
               vec[0].uval += digit;
               l++;
               break;
      case 8:  operator<<=(3); 
               vec[0].uval += digit;
               l+= 3;
               break;
      case 16: operator<<=(4); 
               vec[0].uval += digit;
               l+= 4;
               break;
      default: operator*=( base );
               operator+=( digit );
    }
  };

  // Is this string unsigned?
  if( *str == 'u' || *str == 'U' ) usign = true;
  
  // If signed, do sign extension
  if( base != 2 ) usign = true;
  if( (!usign) && (l < len) ) {
    _bit dummy(vec, l, false);
    int i = _BITCHUNK(l-1);
    dummy.fill(i);
    for(i += 1; i <= _BITCHUNK(len-1); i++)
      vec[i] = dummy.extend(i);
  }
  
  // Possibly negate
  if( neg ) this->negOp(*this);

  return *this;
}


/* --- Assignment of bit slice --- */

void _bit::fill(len_t l, len_t r, int val)
{
  ASSERT( l >= r );
  ASSERT( _BITCHUNK(l) < _BITCHUNKS(len) );
  
  int lchnk = _BITCHUNK(l);
  int rchnk = _BITCHUNK(r);
  int lsh = (l % _BITCHUNK_SIZE) + 1;
  int rsh = (r % _BITCHUNK_SIZE);
  
  // Keep a copy of last chunk
  _bit::chunk rsav = vec[rchnk];
  
  // Check whether we have to sign extend result
  if( l >= (len-1) ) {
    // Do we sign extend with given value?
    if( usgn && val ) {
      // Nope, sign extend with zeros
      vec[lchnk].uval = _BITMASK_LSB_SET(len % _BITCHUNK_SIZE);
    } else {
      // Yes, just fill whole first chunk with value
      lchnk++;
    }
  }
  else {
    // Fill given part of first chunk
    if( val )
      vec[lchnk].uval |= _BITMASK_LSB_SET(lsh);
    else
      vec[lchnk].uval &= _BITMASK_LSB_CLR(lsh);
  }

  // Fill intermediate chunks up to end
  if( val ) val = -1;  
  while( lchnk != rchnk ) {
    vec[--lchnk].sval = val;
  }
    
  // Repair bits in the last chunk
  vec[rchnk].uval &= _BITMASK_LSB_CLR(rsh);
  vec[rchnk].uval |= rsav.uval & ~(_BITMASK_LSB_CLR(rsh));
}


void _bit::copyBits( const _bit& src, len_t sl, len_t sr, len_t dl, len_t dr )
{
  _bit::chunk* srcvec, *tmp = 0;
  
  ASSERT(dl < len);
  ASSERT(dr < len);

  len_t slen = _BITLEN(sl,sr);
  len_t dlen = _BITLEN(dl,dr);

  // Make a copy of source if source and destination are the same
  if( src.addr() == addr() )
  {
    int lchnk, rchnk;
    
    // Determine chunks to copy and adjust source indices
    if(sl > sr) {
      lchnk = _BITCHUNK(sl);
      rchnk = _BITCHUNK(sr);
      
      sr %= _BITCHUNK_SIZE;
      sl  = sr + slen - 1;
    } else {
      lchnk = _BITCHUNK(sr);
      rchnk = _BITCHUNK(sl);
      
      sl %= _BITCHUNK_SIZE;
      sr  = sl + slen - 1;
    }
    
    // Allocate chunks
    tmp = srcvec = new _bit::chunk[lchnk - rchnk + 1];
    
    // And copy them
    while(lchnk >= rchnk) {
      srcvec[lchnk] = src.vec[lchnk];
      --lchnk;
    }        
  } else {
    srcvec = src.vec;
  }
  
  // Make lengths equal
  if( slen > dlen )
  {
    // Shrink source
    sl -= (sl > sr)? (slen - dlen) : (dlen - slen);
  }
  else if( dlen > slen )
  {
    len_t rfill, lfill = dl;
    
    // Shrink destination
    dl -= (dl > dr)? (dlen - slen) : (slen - dlen);
    
    // Determine bounds of remaining bits
    if( dl < lfill ) {
      rfill = dl + 1;
    }
    else {
      rfill = lfill;
      lfill = dl - 1;
    }
    
    // Fill the remaining bits with sign extension of source
    if( src.isUnsigned() )
      fill(lfill, rfill, 0);
    else 
      fill(lfill, rfill, 
	   (srcvec[_BITCHUNK(sl)].uval >> (sl % _BITCHUNK_SIZE)) & 0x01);
  }
  
  // Make the destination canonical
  if (dl < dr) {
    len_t t;
    t = dr; dr = dl; dl = t;
    t = sr; sr = sl; sl = t;
  }

  int schnk = sr / _BITCHUNK_SIZE;
  int ssh   = sr % _BITCHUNK_SIZE;
  int dchnk = dr / _BITCHUNK_SIZE;
  int dsh   = dr % _BITCHUNK_SIZE;

  // Just shift?
  if (sl > sr) {
    _bit::chunk rsav = vec[dchnk];
    _bit::chunk lsav = vec[_BITCHUNK(dl)];
    
    // Determine shift
    int sh = (ssh - dsh);

    // Shifting in first chunk
    if (sh >= 0) {      
      vec[dchnk].uval = srcvec[schnk++].uval >> sh;
    } else {
      vec[dchnk].uval = 0;
    }
	 
    // Loop over chunks
    while (dchnk != _BITCHUNK(dl)) {
      // Set bits shifted in correctly
      if (sh > 0) 
	vec[dchnk].uval |= srcvec[schnk].uval << (_BITCHUNK_SIZE - sh);
      else if (sh < 0)
	vec[dchnk].uval |= srcvec[schnk].uval << -sh;
	
      dchnk++; schnk++;

      // Shifting in LSBs
      vec[dchnk].uval = srcvec[schnk-1].uval;
      if (sh > 0)
	vec[dchnk].uval >>= sh;
      else if (sh < 0)
	vec[dchnk].uval >>= (_BITCHUNK_SIZE + sh);
    }
    
    // How many bits to fill in last chunk?
    int dlsh = (dl % _BITCHUNK_SIZE) + 1;
    
    // How many bits left to copy in source?
    if (sh >= 0) {
      sh = _BITCHUNK_SIZE - sh;
    } else {
      sh = -sh;
    }

    // Copy bits if some are missing...
    if (sh < dlsh) vec[dchnk].uval |= srcvec[schnk].uval << sh;
    
    // Reset LSBs in first chunk to old dest bits
    if (dsh) {
      vec[_BITCHUNK(dr)].uval &= _BITMASK_LSB_CLR(dsh);
      vec[_BITCHUNK(dr)].uval |= (rsav.uval & ~(_BITMASK_LSB_CLR(dsh)));
    }

    // Set remaining bits of last chunk
    if (dl == (len - 1)) {
      // Sign extend if sign bit set
      fill(_BITCHUNK(len-1));
    } else {
      // keep old dest bits
      if (dlsh != _BITCHUNK_SIZE) {
	vec[dchnk].uval &= _BITMASK_LSB_SET(dlsh);
	vec[dchnk].uval |= (lsav.uval & _BITMASK_LSB_CLR(dlsh));
      }
    }
  }
  else {
    // Swap bits, copy bit by bit
    unsigned sptr = 0x01 << ssh;
    unsigned dptr = 0x01 << dsh;

    for (len_t i = 0; i <= (dl - dr); i++) 
    {
      // Set new bit according to source bit
      if (srcvec[schnk].uval & sptr) 
	vec[dchnk].uval |= dptr;
      else
	vec[dchnk].uval &= (~dptr);

      // Move src pointer
      sptr >>= 1;
      if (!sptr) { sptr = 0x01 << (_BITCHUNK_SIZE-1);  schnk--; }

      // Move dest pointer
      dptr <<= 1;
      if (!dptr) { dptr = 0x01;  dchnk++; }
    }

    if (dl == (len - 1)) fill(_BITCHUNK(dl));
  }
  
  // Delete temporary copy of src
  if(tmp) delete[] tmp;
}


/* --- Other assignments --- */
  
_bit& _bit::set(double d)
{
  for(int i = _BITCHUNK(len-1); i >= 0; i--) {
    double t = 1;
    for(int j = 0; j < i; j++) t *= ((double)UINT_MAX+1);
    vec[i].uval = (unsigned int)(d/t);
    d -= (double)(vec[i].uval) * t;
  }
  return fill(_BITCHUNK(len-1));
}

_bit& _bit::set(long double d) 
{
  for(int i = _BITCHUNK(len-1); i >= 0; i--) {
    long double t = 1;
    for(int j = 0; j < i; j++) t *= ((long double)UINT_MAX+1);
    vec[i].uval = (unsigned int)(d/t);
    d -= (long double)(vec[i].uval) * t;
  }
  return fill(_BITCHUNK(len-1));
}


/* --- Conversion --- */

long double _bit::toLDouble(void) const 
{
  long double res;
  int i = _BITCHUNK(len-1);

  if(usgn) res = vec[i].uval; else res = vec[i].sval;
  for (i = i - 1; i >= 0; i--) {
    res *= ((long double)(~(unsigned)0)) + 1;
    res += vec[i].uval;
  }
  return res;
}

double _bit::toDouble(void) const 
{
  double res;
  int i = _BITCHUNK(len-1);

  if(usgn) res = vec[i].uval; else res = vec[i].sval;
  for (i = i - 1; i >= 0; i--) {
    res *= ((double)(~(unsigned)0)) + 1;
    res += vec[i].uval;
  }
  return res;
}




/* ------------------------------------------------------------------------- */
/* Global, external functions */


// Conversion to string

char *ubit2str(unsigned int base, char *endptr, const _bit& b)
{
  unsigned int digit;
  _bit::len_t l = b.length();

  _bit ub(l, true);   // create local copy, unsigned!
  ub = b;
   
  // Loop over digits
  *endptr = '\0';
  do {
    switch( base )
    {
      case  2: digit = ub.toUInt() & 0x1;
               ub >>= 1;
               l--;
               break;
      case  4: digit = ub.toUInt() & 0x3;
               ub >>= 2;
               if(l > 2) l -= 2; else l = 0;
               break;
      case  8: digit = ub.toUInt() & 0x7;
               ub >>= 3;
               if(l > 3) l -= 3; else l = 0;
               break;
      case 16: digit = ub.toUInt() & 0xF;
               ub >>= 4;
               if(l > 4) l -= 4; else l = 0;
               break;
      case 32: digit = ub.toUInt() & 0x1F;
               ub >>= 5;
               if(l > 5) l -= 5; else l = 0;
               break;
      default: digit = (ub % base).toUInt();
               ub /= base;
               if( ! ub.test() ) l = 0;
    }
    *(--endptr) = (digit <= 9)? (digit + '0') : ((digit-10) + 'A');
  } while( l > 0 );
    
  return endptr;
}
  
char *bit2str(unsigned int base, char *endptr, const _bit& b)
{
  bool neg = (b < 0);
  endptr = ubit2str(base, endptr, neg? -b : b);
  if( neg ) *(--endptr) = '-';
  return endptr;
}


// Conversion from string

void str2bit(unsigned int base, const char *str, _bit *bptr) {
  bptr->set(str, false, base);
}

void str2ubit(unsigned int base, const char *str, _bit *ubptr) {
  ubptr->set(str, true, base);
}


/* ------------------------------------------------------------------------- */
/* Main */


#ifdef DEBUG

#include <bit.h>
#include <stdio.h>


bool testOp (short as, short bs, long ai, long bi, LONG_LONG al, LONG_LONG bl)
{
  short cs;
  long  ci;
  LONG_LONG cl;
  
  bit<16>  a16 = as, b16 = bs, c16;
  bit<32>  a32 = ai, b32 = bi, c32;
  bit<64>  a64 = al, b64 = bl, c64;
  
  cs = as + bs; c16 = a16 + b16; if (cs != c16) return false;
  ci = ai + bi; c32 = a32 + b32; if (ci != c32) return false;
  cl = al + bl; c64 = a64 + b64; if (cl != c64) return false;

  cs = as - bs; c16 = a16 - b16; if (cs != c16) return false;
  ci = ai - bi; c32 = a32 - b32; if (ci != c32) return false;
  cl = al - bl; c64 = a64 - b64; if (cl != c64) return false;

  cs = as * bs; c16 = a16 * b16; if (cs != c16) return false;
  ci = ai * bi; c32 = a32 * b32; if (ci != c32) return false;
  cl = al * bl; c64 = a64 * b64; if (cl != c64) return false;
   
  cs = as / bs; c16 = a16 / b16; if (cs != c16) return false;
  ci = ai / bi; c32 = a32 / b32; if (ci != c32) return false;
  cl = al / bl; c64 = a64 / b64; if (cl != c64) return false;
   
  cs = as % bs; c16 = a16 % b16; if (cs != c16) return false;
  ci = ai % bi; c32 = a32 % b32; if (ci != c32) return false;
  cl = al % bl; c64 = a64 % b64; if (cl != c64) return false;

  if( (bs >= 0) && (bs <= 16) ) {
    cs = as >> bs; c16 = a16 >> b16.toInt();   if (cs != c16) return false;
  }
  if( (bi >= 0) && (bi <= 32) ) {
    ci = ai >> bi; c32 = a32 >> b32.toLong();  if (ci != c32) return false;
  }
  if( (bl >= 0) && (bl <= 64) ) {
    cl = al >> ll2uint(bl); c64 = a64 >> b64.toUInt();  
    if (cl != c64) return false;
  }

  if( (bs >= 0) && (bs <= 16) ) {
    cs = as << bs; c16 = a16 << b16.toInt();   if (cs != c16) return false;
  }
  if( (bi >= 0) && (bi <= 32) ) {
    ci = ai << bi; c32 = a32 << b32.toLong();  if (ci != c32) return false;
  }
  if( (bl >= 0) && (bl <= 64) ) {
    cl = al << ll2uint(bl); c64 = a64 << b64.toUInt();  
    if (cl != c64) return false;
  }
   
  cs = as & bs; c16 = a16 & b16; if (cs != c16) return false;
  ci = ai & bi; c32 = a32 & b32; if (ci != c32) return false;
  cl = al & bl; c64 = a64 & b64; if (cl != c64) return false;

  cs = as | bs; c16 = a16 | b16; if (cs != c16) return false;
  ci = ai | bi; c32 = a32 | b32; if (ci != c32) return false;
  cl = al | bl; c64 = a64 | b64; if (cl != c64) return false;
   
  cs = as ^ bs; c16 = a16 ^ b16; if (cs != c16) return false;
  ci = ai ^ bi; c32 = a32 ^ b32; if (ci != c32) return false;
  cl = al ^ bl; c64 = a64 ^ b64; if (cl != c64) return false;
   
  cs = as > bs; c16 = a16 > b16; if (cs != c16) return false;
  ci = ai > bi; c32 = a32 > b32; if (ci != c32) return false;
  cl = al > bl; c64 = a64 > b64; if (cl != c64) return false;
   
  cs = as < bs; c16 = a16 < b16; if (cs != c16) return false;
  ci = ai < bi; c32 = a32 < b32; if (ci != c32) return false;
  cl = al < bl; c64 = a64 < b64; if (cl != c64) return false;

  cs = as == bs; c16 = a16 == b16; if (cs != c16) return false;
  ci = ai == bi; c32 = a32 == b32; if (ci != c32) return false;
  cl = al == bl; c64 = a64 == b64; if (cl != c64) return false;
   
  cs = as <= bs; c16 = a16 <= b16; if (cs != c16) return false;
  ci = ai <= bi; c32 = a32 <= b32; if (ci != c32) return false;
  cl = al <= bl; c64 = a64 <= b64; if (cl != c64) return false;

  cs = as >= bs; c16 = a16 >= b16; if (cs != c16) return false;
  ci = ai >= bi; c32 = a32 >= b32; if (ci != c32) return false;
  cl = al >= bl; c64 = a64 >= b64; if (cl != c64) return false;
   
  cs = as != bs; c16 = a16 != b16; if (cs != c16) return false;
  ci = ai != bi; c32 = a32 != b32; if (ci != c32) return false;
  cl = al != bl; c64 = a64 != b64; if (cl != c64) return false;
     
  return true;
}

bool testOpShadow (short as, short bs, long ai, long bi, 
		   LONG_LONG al, LONG_LONG bl)
{
  short cs;
  long  ci;
  LONG_LONG cl;

  union {
    _bit_<16> c16;
    _bit_<32> c32;
    _bit_<64> c64;
  };
    
  _bit_<16>  a16, b16;  a16 = as;  b16 = bs;
  _bit_<32>  a32, b32;  a32 = ai;  b32 = bi;
  _bit_<64>  a64, b64;  a64 = al;  b64 = bl;
  
  cs = as + bs; c16 = a16 + b16; if (cs != c16) return false;
  ci = ai + bi; c32 = a32 + b32; if (ci != c32) return false;
  cl = al + bl; c64 = a64 + b64; if (cl != c64) return false;

  cs = as - bs; c16 = a16 - b16; if (cs != c16) return false;
  ci = ai - bi; c32 = a32 - b32; if (ci != c32) return false;
  cl = al - bl; c64 = a64 - b64; if (cl != c64) return false;

  cs = as * bs; c16 = a16 * b16; if (cs != c16) return false;
  ci = ai * bi; c32 = a32 * b32; if (ci != c32) return false;
  cl = al * bl; c64 = a64 * b64; if (cl != c64) return false;
   
  cs = as / bs; c16 = a16 / b16; if (cs != c16) return false;
  ci = ai / bi; c32 = a32 / b32; if (ci != c32) return false;
  cl = al / bl; c64 = a64 / b64; if (cl != c64) return false;
   
  cs = as % bs; c16 = a16 % b16; if (cs != c16) return false;
  ci = ai % bi; c32 = a32 % b32; if (ci != c32) return false;
  cl = al % bl; c64 = a64 % b64; if (cl != c64) return false;

  if( (bs >= 0) && (bs <= 16) ) {
    cs = as >> bs; c16 = a16 >> b16.toInt();   if (cs != c16) return false;
  }
  if( (bi >= 0) && (bi <= 32) ) {
    ci = ai >> bi; c32 = a32 >> b32.toLong();  if (ci != c32) return false;
  }
  if( (bl >= 0) && (bl <= 64) ) {
    cl = al >> ll2uint(bl); c64 = a64 >> b64.toUInt();  
    if (cl != c64) return false;
  }

  if( (bs >= 0) && (bs <= 16) ) {
    cs = as << bs; c16 = a16 << b16.toInt();   if (cs != c16) return false;
  }
  if( (bi >= 0) && (bi <= 32) ) {
    ci = ai << bi; c32 = a32 << b32.toLong();  if (ci != c32) return false;
  }
  if( (bl >= 0) && (bl <= 64) ) {
    cl = al << ll2uint(bl); c64 = a64 << b64.toUInt();  
    if (cl != c64) return false;
  }
   
  cs = as & bs; c16 = a16 & b16; if (cs != c16) return false;
  ci = ai & bi; c32 = a32 & b32; if (ci != c32) return false;
  cl = al & bl; c64 = a64 & b64; if (cl != c64) return false;

  cs = as | bs; c16 = a16 | b16; if (cs != c16) return false;
  ci = ai | bi; c32 = a32 | b32; if (ci != c32) return false;
  cl = al | bl; c64 = a64 | b64; if (cl != c64) return false;
   
  cs = as ^ bs; c16 = a16 ^ b16; if (cs != c16) return false;
  ci = ai ^ bi; c32 = a32 ^ b32; if (ci != c32) return false;
  cl = al ^ bl; c64 = a64 ^ b64; if (cl != c64) return false;
   
  cs = as > bs; c16 = a16 > b16; if (cs != c16) return false;
  ci = ai > bi; c32 = a32 > b32; if (ci != c32) return false;
  cl = al > bl; c64 = a64 > b64; if (cl != c64) return false;
   
  cs = as < bs; c16 = a16 < b16; if (cs != c16) return false;
  ci = ai < bi; c32 = a32 < b32; if (ci != c32) return false;
  cl = al < bl; c64 = a64 < b64; if (cl != c64) return false;

  cs = as == bs; c16 = a16 == b16; if (cs != c16) return false;
  ci = ai == bi; c32 = a32 == b32; if (ci != c32) return false;
  cl = al == bl; c64 = a64 == b64; if (cl != c64) return false;
   
  cs = as <= bs; c16 = a16 <= b16; if (cs != c16) return false;
  ci = ai <= bi; c32 = a32 <= b32; if (ci != c32) return false;
  cl = al <= bl; c64 = a64 <= b64; if (cl != c64) return false;

  cs = as >= bs; c16 = a16 >= b16; if (cs != c16) return false;
  ci = ai >= bi; c32 = a32 >= b32; if (ci != c32) return false;
  cl = al >= bl; c64 = a64 >= b64; if (cl != c64) return false;
   
  cs = as != bs; c16 = a16 != b16; if (cs != c16) return false;
  ci = ai != bi; c32 = a32 != b32; if (ci != c32) return false;
  cl = al != bl; c64 = a64 != b64; if (cl != c64) return false;
     
  return true;
}

bool testOpMix (short as, short bs, long ai, long bi, 
		LONG_LONG al, LONG_LONG bl)
{
  short cs;
  long  ci;
  LONG_LONG cl;
  
  _bit_<16>  a16, b16, c16;  a16 = as;  b16 = bs;
  bit<32>    a32, b32, c32;  a32 = ai;  b32 = bi;
  _bit_<64>  a64, b64, c64;  a64 = al;  b64 = bl;
  
  cs = as + bs; c16 = a16 + b16; if (cs != c16) return false;
  ci = ai + bi; c32 = a32 + b32; if (ci != c32) return false;
  cl = al + bl; c64 = a64 + b64; if (cl != c64) return false;

  cs = as - bs; c16 = a16 - b16; if (cs != c16) return false;
  ci = ai - bi; c32 = a32 - b32; if (ci != c32) return false;
  cl = al - bl; c64 = a64 - b64; if (cl != c64) return false;

  cs = as * bs; c16 = a16 * b16; if (cs != c16) return false;
  ci = ai * bi; c32 = a32 * b32; if (ci != c32) return false;
  cl = al * bl; c64 = a64 * b64; if (cl != c64) return false;
   
  cs = as / bs; c16 = a16 / b16; if (cs != c16) return false;
  ci = ai / bi; c32 = a32 / b32; if (ci != c32) return false;
  cl = al / bl; c64 = a64 / b64; if (cl != c64) return false;
   
  cs = as % bs; c16 = a16 % b16; if (cs != c16) return false;
  ci = ai % bi; c32 = a32 % b32; if (ci != c32) return false;
  cl = al % bl; c64 = a64 % b64; if (cl != c64) return false;

  if( (bs >= 0) && (bs <= 16) ) {
    cs = as >> bs; c16 = a16 >> b16.toInt();   if (cs != c16) return false;
  }
  if( (bi >= 0) && (bi <= 32) ) {
    ci = ai >> bi; c32 = a32 >> b32.toLong();  if (ci != c32) return false;
  }
  if( (bl >= 0) && (bl <= 64) ) {
    cl = al >> ll2uint(bl); c64 = a64 >> b64.toUInt();  
    if (cl != c64) return false;
  }

  if( (bs >= 0) && (bs <= 16) ) {
    cs = as << bs; c16 = a16 << b16.toInt();   if (cs != c16) return false;
  }
  if( (bi >= 0) && (bi <= 32) ) {
    ci = ai << bi; c32 = a32 << b32.toLong();  if (ci != c32) return false;
  }
  if( (bl >= 0) && (bl <= 64) ) {
    cl = al << ll2uint(bl); c64 = a64 << b64.toUInt();  
    if (cl != c64) return false;
  }
   
  cs = as & bs; c16 = a16 & b16; if (cs != c16) return false;
  ci = ai & bi; c32 = a32 & b32; if (ci != c32) return false;
  cl = al & bl; c64 = a64 & b64; if (cl != c64) return false;

  cs = as | bs; c16 = a16 | b16; if (cs != c16) return false;
  ci = ai | bi; c32 = a32 | b32; if (ci != c32) return false;
  cl = al | bl; c64 = a64 | b64; if (cl != c64) return false;
   
  cs = as ^ bs; c16 = a16 ^ b16; if (cs != c16) return false;
  ci = ai ^ bi; c32 = a32 ^ b32; if (ci != c32) return false;
  cl = al ^ bl; c64 = a64 ^ b64; if (cl != c64) return false;
   
  cs = as > bs; c16 = a16 > b16; if (cs != c16) return false;
  ci = ai > bi; c32 = a32 > b32; if (ci != c32) return false;
  cl = al > bl; c64 = a64 > b64; if (cl != c64) return false;
   
  cs = as < bs; c16 = a16 < b16; if (cs != c16) return false;
  ci = ai < bi; c32 = a32 < b32; if (ci != c32) return false;
  cl = al < bl; c64 = a64 < b64; if (cl != c64) return false;

  cs = as == bs; c16 = a16 == b16; if (cs != c16) return false;
  ci = ai == bi; c32 = a32 == b32; if (ci != c32) return false;
  cl = al == bl; c64 = a64 == b64; if (cl != c64) return false;
   
  cs = as <= bs; c16 = a16 <= b16; if (cs != c16) return false;
  ci = ai <= bi; c32 = a32 <= b32; if (ci != c32) return false;
  cl = al <= bl; c64 = a64 <= b64; if (cl != c64) return false;

  cs = as >= bs; c16 = a16 >= b16; if (cs != c16) return false;
  ci = ai >= bi; c32 = a32 >= b32; if (ci != c32) return false;
  cl = al >= bl; c64 = a64 >= b64; if (cl != c64) return false;
   
  cs = as != bs; c16 = a16 != b16; if (cs != c16) return false;
  ci = ai != bi; c32 = a32 != b32; if (ci != c32) return false;
  cl = al != bl; c64 = a64 != b64; if (cl != c64) return false;
     
  return true;
}


bool testMisc(_bit_<71> a, bit<6,true> b)
{
  char buf1[128], buf2[128];
  
  a = (char)1;
  a = -a;  
  
  if (a.toInt() != -1) return false;
  a = (long)-1;
  if (a.toLong() != -1) return false;
  a = (LONG_LONG)-1;
  if (a.toLLong() != -1) return false;

  // Test string assignments
  if (b.toInt() != 63) return false;
  b = "00000000000000000010u";
  a = "1";  // -1

  if (a >= b) return false;
  if (a < -2) return false;
  if (-a > b) return false;
  
  a = a + b;

  if (++a != 2) return false;

  if (a != b) return false;

  b = a - b;

  if (b.test()) return false;

  b = b--;
  b.flip();
  if (b <= 3) return false;

  b >>= 2;
  b <<= 2;

  if (b(1,0).test()) return false;

  b(5,4) = b(0,1);
  if (b.toInt() != 12) return false;

  b(5,4) = b(4,1)(0,0);
  if (b.toInt() != 12) return false;

  b(4,4) = b(0,1);
  if (b.toInt() != 12) return false;

  b(5,4) = (b(3,2) - 3)(1,0);
  if (b.toInt() != 12) return false;

  b(5,4) = (b(3,2) - 3)(1,0)[0];
  if (b.toInt() != 12) return false;
  
  b(5,4) *= 1;
  if (b.toInt() != 12) return false;

  b(5,4)++;
  if (b.toInt() != 28) return false;

  --b(5,4);
  if (b.toInt() != 12) return false;
  
  a = b(3,2) * _bit("1111") * -5;

  if ((~b) != (a ^ _bit("0111100"))) return false;
  if ((b | 1) != (a & 13)) return false;

  if (_bit::concat(a, b) != _bit("1111001100u")) return false;

  b %= a/2 + a%2;
  a /= -2;
  a *= -2;

  if ((a / -42).toInt()) return false;

  // ### Removed as gcc-2.96 has trouble with that?!
  //     As scc will never generate such code the test is not needed anyway
  //b = b.test() ? b + 1 : b(5,0);
  //b--;
  
  printf("Done, results (should be 14, 4):\n");
  printf("a[63:0] = %s = %s\n", bit2str(10, &buf1[127], a(63,0)),
	                         bit2str(2, &buf2[127], a(63,0)));
  printf("b[5:0]  = %d = %s\n",  b.toInt(), bit2str(2, &buf1[127], b));

  if (a != 14) return false;
  if (b != 4)  return false;

  b = (b + a) + a;
  
  return true;
}


int errSignExt(void)
{
  printf("Error: Problem with sign extended right shifts!!\n");
  printf("       Manually assure that _RSH_IS_SIGN_EXTENDING in file <sys/bit.h> is undefined!\n");
  return 8;
}


int errSizeOp(void)
{
  printf("Error: Calculation of bit class size doesn't match sizeof() result!!\n");
  return 2;
}

int err(const char * s)
{
  printf("Error: %s\n", s);
  return 1;
}

int main(void)
{
  char* p;
  char buf[128], buf1[64], buf2[64];
  _bit_<71>   a;
  bit<6,true> b = "111111";

  printf("Testing bit vector library:\n");

  // If these tests fail, undefine _RSH_IS_SIGN_EXTENDING
  a = -1;
  if (a.toInt() != -1) return errSignExt();
  a = -1l;
  if (a.toLong() != -1l) return errSignExt();
#ifdef HAVE_LLONG  
  a = -1ll;
  if (a.toLLong() != -1ll) return errSignExt();
#else
  a = -LONG_LONG(1);
  if (a.toLLong() != -LONG_LONG(1)) return errSignExt();
#endif  
  
  // Test for correct size calculations
  if ( (sizeof(a) != _SIZEOFBITCLASS(71)) || (sizeof(b) != _SIZEOFBITCLASS(6)) )
    return errSizeOp();

  // Test output routines
  if(strcmp(p = ubit2str(2, &buf1[63], b), "111111") != 0) {
    sprintf(buf, "ubit2str(2, '111111') returns %s\n", p);
    return err(buf);
  }
  if(strcmp(p = ubit2str(4, &buf1[63], b), "333") != 0) {
    sprintf(buf, "ubit2str(4, '333') returns %s\n", p);
    return err(buf);
  }
  if(strcmp(p = ubit2str(8, &buf1[63], b), "77") != 0) {
    sprintf(buf, "ubit2str(8, '077') returns %s\n", p);
    return err(buf);
  }
  if(strcmp(p = ubit2str(16, &buf1[63], b), "3F") != 0) {
    sprintf(buf, "ubit2str(16, '0x3f') returns %s\n", p);
    return err(buf);
  }
  if(strcmp(p = ubit2str(32, &buf1[63], b), "1V") != 0) {
    sprintf(buf, "ubit2str(32, '1V') returns %s\n", p);
    return err(buf);
  }
  
  printf("Doing a test of miscellaneous bit operations...\n");
  if( ! testMisc(a, b) ) return err("Error in calculation");
  
  printf("Testing bit vector operators...\n");

  if( !testOp( -1, -1, -1, -1, -1, -1 ) ) return err("Op (-1, -1)");
  if( !testOp( -1, 1, -1, 1, -1, 1 ) ) return err("Op (-1, 1)");
  if( !testOp( 1, -1, 1, -1, 1, -1 ) ) return err("Op (1, -1)");

  if( !testOpShadow( -1, -1, -1, -1, -1, -1 ) ) return err("Shadow (-1, -1)");
  if( !testOpShadow( -1, 1, -1, 1, -1, 1 ) ) return err("Shadow (-1, 1)");
  if( !testOpShadow( 1, -1, 1, -1, 1, -1 ) ) return err("Shadow (1, -1)");
  
  if( !testOpMix( -1, -1, -1, -1, -1, -1 ) ) return err("Mix (-1, -1)");
  if( !testOpMix( -1, 1, -1, 1, -1, 1 ) ) return err("Mix (-1, 1)");
  if( !testOpMix( 1, -1, 1, -1, 1, -1 ) ) return err("Mix (1, -1)");
  
  int i, j;
  UNSIGNED_LONG_LONG x, y;
  for(i = 0, x = 1; i <= 64; i++, x = (x<<1)+1)
    for(j = 0, y = 1; j <= 64; j++, y = (y<<1)+1) {
      if(!testOp(ll2uint(x), ll2uint(y), ll2uint(x), ll2uint(y), x, y)) {
	sprintf(buf, "Op (%hd, %hd, %d, %d, %s, %s)",
		     ll2uint(x), ll2uint(y), ll2uint(x), ll2uint(y),
		     ll2str(10, &buf1[63], x), ll2str(10, &buf2[63], y));
        return err(buf);
      }
      if(!testOpShadow(ll2uint(x), ll2uint(y), ll2uint(x), ll2uint(y), x, y)) {
	sprintf(buf, "Shadow (%hd, %hd, %d, %d, %s, %s)",
		     ll2uint(x), ll2uint(y), ll2uint(x), ll2uint(y),
		     ll2str(10, &buf1[63], x), ll2str(10, &buf2[63], y));
        return err(buf);
      }
      if(!testOpMix(ll2uint(x), ll2uint(y), ll2uint(x), ll2uint(y), x, y)) {
	sprintf(buf, "Mix (%hd, %hd, %d, %d, %s, %s)",
		     ll2uint(x), ll2uint(y), ll2uint(x), ll2uint(y),
		     ll2str(10, &buf1[63], x), ll2str(10, &buf2[63], y));
        return err(buf);
      }
    }
  
#ifdef HAVE_LLONG  
  if (!testOp(0x5555, 0x5, 0x55555555l, 0x15l, 0x5555555555555555ll, 0x1All) ||
      !testOp(0xAAAA, 0x5, 0xAAAAAAAAl, 0x15l, 0xAAAAAAAAAAAAAAAAll, 0x1All) ||
      !testOp(0x5555, 0xA, 0x55555555l, 0x1Al, 0x5555555555555555ll, 0x15ll) ||
      !testOp(0xAAAA, 0xA, 0xAAAAAAAAl, 0x1Al, 0xAAAAAAAAAAAAAAAAll, 0x15ll))
#else    
  if (!testOp(0x5555, 0x5, 0x55555555l, 0x15l, "0x5555555555555555", "0x1A") ||
      !testOp(0xAAAA, 0x5, 0xAAAAAAAAl, 0x15l, "0xAAAAAAAAAAAAAAAA", "0x1A") ||
      !testOp(0x5555, 0xA, 0x55555555l, 0x1Al, "0x5555555555555555", "0x15") ||
      !testOp(0xAAAA, 0xA, 0xAAAAAAAAl, 0x1Al, "0xAAAAAAAAAAAAAAAA", "0x15"))
#endif
      return err("Op (0x55.55, 0xAA..AA)");
			 
#ifdef HAVE_LLONG  
  if (!testOpShadow(0x5555, 0x5, 0x55555555l, 0x15l, 
		    0x5555555555555555ll,     0x1All) ||
      !testOpShadow(0xAAAA, 0x5, 0xAAAAAAAAl, 0x15l, 
		    0xAAAAAAAAAAAAAAAAll,     0x1All) ||
      !testOpShadow(0x5555, 0xA, 0x55555555l, 0x1Al, 
		    0x5555555555555555ll,     0x15ll) ||
      !testOpShadow(0xAAAA, 0xA, 0xAAAAAAAAl, 0x1Al, 
		    0xAAAAAAAAAAAAAAAAll,     0x15ll))
#else    
  if (!testOpShadow(0x5555, 0x5, 0x55555555l, 0x15l, 
		    "0x5555555555555555",    "0x1A") ||
      !testOpShadow(0xAAAA, 0x5, 0xAAAAAAAAl, 0x15l, 
		    "0xAAAAAAAAAAAAAAAA",    "0x1A") ||
      !testOpShadow(0x5555, 0xA, 0x55555555l, 0x1Al, 
		    "0x5555555555555555",    "0x15") ||
      !testOpShadow(0xAAAA, 0xA, 0xAAAAAAAAl, 0x1Al, 
		    "0xAAAAAAAAAAAAAAAA",    "0x15"))
#endif
      return err("Shadow (0x55.55, 0xAA..AA)");

#ifdef HAVE_LLONG  
  if (!testOpMix(0x5555, 0x5, 0x55555555l, 0x15l, 
		 0x5555555555555555ll,     0x1All) ||
      !testOpMix(0xAAAA, 0x5, 0xAAAAAAAAl, 0x15l, 
		 0xAAAAAAAAAAAAAAAAll,     0x1All) ||
      !testOpMix(0x5555, 0xA, 0x55555555l, 0x1Al, 
		 0x5555555555555555ll,     0x15ll) ||
      !testOpMix(0xAAAA, 0xA, 0xAAAAAAAAl, 0x1Al, 
		 0xAAAAAAAAAAAAAAAAll,     0x15ll))
#else    
  if (!testOpMix(0x5555, 0x5, 0x55555555l, 0x15l, 
		 "0x5555555555555555",    "0x1A") ||
      !testOpMix(0xAAAA, 0x5, 0xAAAAAAAAl, 0x15l, 
		 "0xAAAAAAAAAAAAAAAA",    "0x1A") ||
      !testOpMix(0x5555, 0xA, 0x55555555l, 0x1Al, 
		 "0x5555555555555555",    "0x15") ||
      !testOpMix(0xAAAA, 0xA, 0xAAAAAAAAl, 0x1Al, 
		 "0xAAAAAAAAAAAAAAAA",    "0x15"))
#endif
      return err("Mix (0x55.55, 0xAA..AA)");
    
  printf("Done, everything fine.\n");
  return 0;
}

#endif



