/************************************************************************/
/* Node.cc: SpecC Internal Representation, Node Base Class		*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 09/24/97 */
/************************************************************************/

/* last update: 10/16/01 */

/* modifications: (most recent first)
 *
 * 10/16/01 RD	added SIR_Node::CmpLineInfo() for generating ordered code
 * 05/31/01 RD	eliminated level 2 of SIR API
 * 05/31/01 RD	introduced this header (last change was 04/12/99)
 */

#include "IntRep/Node.h"

#include <stdlib.h>
#include <assert.h>


/*** constants and macros ***********************************************/


	/* none */


/*** class implementations **********************************************/


	/****************/
	/*** SIR_Node ***/
	/****************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Node::SIR_Node(void)	/* constructor #1 (nodes without lineinfo) */
{

LineInfo = NULL;	/* not known */

} /* end of SIR_Node::SIR_Node #1 */


SIR_Node::SIR_Node(		/* constructor #2 (nodes with lineinfo) */
	unsigned int	Line,
	sir_fileinfo	*FileInfo)
{

assert(  ((Line == 0) && (FileInfo == NULL))
       ||((Line != 0) && (FileInfo != NULL)));

if (Line)
   { LineInfo = new SIR_LineInfo(Line, FileInfo);
    } /* fi */
else
   { LineInfo = NULL;
    } /* esle */

} /* end of SIR_Node::SIR_Node #2 */


SIR_Node::SIR_Node(		/* constructor #3 (duplicator) */
	sir_node	*Original)
{
unsigned int	Line;
sir_fileinfo	*FileInfo;

if (Original->LineInfo)
   { Line = Original->LineInfo->Line;
     FileInfo = Original->LineInfo->File;
     LineInfo = new SIR_LineInfo(Line, FileInfo);
    } /* fi */
else
   { LineInfo = NULL;
    } /* esle */

} /* end of SIR_Node::SIR_Node #3 */


SIR_Node::~SIR_Node(void)		/* destructor */
{

delete LineInfo;

} /* end of SIR_Node::~SIR_Node */


ERROR SIR_Node::UnaliasFilePointer(	/* unalias the file info (if any) */
	sir_node_marg	/* Unused */)
{

if (  (LineInfo)
    &&(LineInfo->File->Alias))
   { LineInfo->File = LineInfo->File->Alias;
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Node::UnaliasFilePointer */


ERROR SIR_Node::MarkUsedFiles(		/* marks the file entry used here */
	sir_node_marg	/* Unused */)
{

if (LineInfo)
   { assert(LineInfo->File != NULL);
     LineInfo->File->Color = SIR_RED;	/* mark being used */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Node::MarkUsedFiles */


ERROR SIR_Node::Strip(		/* removes the source information #1 */
	sir_node_marg	/* Unused */)	/* (iterator version) */
{

delete LineInfo;
LineInfo = NULL;

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Node::Strip #1 */


void SIR_Node::Strip(void)	/* removes the source information #2 */
{

delete LineInfo;
LineInfo = NULL;

} /* end of SIR_Node::Strip #2 */


void SIR_Node::UpdateLineInfo(	/* update the source info (if any) */
	unsigned int	Line,	/* (keeps old info if no new is specified) */
	sir_fileinfo	*FileInfo)
{

assert(  (Line == 0) && (FileInfo == NULL)
       ||(Line != 0) && (FileInfo != NULL));

if (Line)
   { delete LineInfo;	/* delete old info (if any) */
     LineInfo = new SIR_LineInfo(Line, FileInfo);
    } /* fi */

} /* end of SIR_Node::UpdateLineInfo */


void SIR_Node::SetLineInfo(	/* sets the same source info */
	sir_node	*Node)
{

assert(Node != NULL);

if (Node->LineInfo)
   { delete LineInfo;	/* delete old info (if any) */
     LineInfo = new SIR_LineInfo(
			Node->LineInfo->Line,
			Node->LineInfo->File);
    } /* fi */
else
   { delete LineInfo;	/* delete old info (if any) */
     LineInfo = NULL;
    } /* esle */

} /* end of SIR_Node::SetLineInfo */


int SIR_Node::CmpLineInfo(	/* compare two line infos (like strcmp()) */
	sir_node	*Node1,
	sir_node	*Node2)
{
int	Result;

assert(Node1 != NULL);
assert(Node2 != NULL);

if (Node1->LineInfo == Node2->LineInfo)
   { return(0);		// identical!
    } /* fi */
if (! Node1->LineInfo)
   { return(1);		// known stuff comes first, then new stuff
    } /* fi */
if (! Node2->LineInfo)
   { return(-1);	// known stuff comes first, then new stuff
    } /* fi */

if (Node1->LineInfo->File == Node2->LineInfo->File)
   { if (Node1->LineInfo->Line > Node2->LineInfo->Line)
	{ return(1);	// in same file, Node1 is after Node2
	 } /* fi */
     if (Node1->LineInfo->Line < Node2->LineInfo->Line)
	{ return(-1);	// in same file, Node1 is before Node2
	 } /* fi */
     return(0);		// in same file, in same line
    } /* fi */

Result = strcmp(Node1->LineInfo->File->Filename.chars(),
		Node2->LineInfo->File->Filename.chars());

assert(Result != 0);	/* that case was checked before */

return(Result);		// in different files

} /* end of SIR_Node::CmpLineInfo */


/* EOF Node.cc */
