/************************************************************************/
/* IntRep/Expression.h: SpecC Internal Representation, Expression Class	*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 09/24/97 */
/************************************************************************/

/* last update: 11/08/01 by RD */

/* modifications: (most recent first)
 *
 * 11/08/01 RD	switched code generation to use GL_IO layer
 * 05/31/01 RD	eliminated level 2 of SIR API
 * 05/27/01 RD	added semantic checking for port accesses, etc.
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 05/25/01 RD	removed code not needed for the SCRC
 * 05/22/01 RD	added SIR_Expression::FixTypePtr()
 * 04/25/01 RD	extended SIR_Expression::ExplicitTypeConv() for floats;
 *		added ExpectedLength argument to SIR_Expression::WriteSC[2]
 * 04/20/01 RD	added SIR_Expression::ExplicitTypeConv()
 * 03/08/01 RD	created this modifications list (last change was 04/13/99)
 */


#ifndef INTREP_EXPRESSION_H
#define INTREP_EXPRESSION_H


#include "Global.h"
#include "IntRep/Type.h"
#include "IntRep/Extern.h"


/*** enumeration types **************************************************/


enum SIR_ExprType		/* supported expression types */
{				/* (see class SIR_Expression) */
SIR_EXPR_VOID		= 0,	// New() #8
SIR_EXPR_CONSTANT,		// New() #1
SIR_EXPR_IDENTIFIER,		// New() #2
SIR_EXPR_PARENTHESES,		// New() #3, 1 argument
SIR_EXPR_THIS,			// New() #8
SIR_EXPR_ARRAY_ACCESS,		// New() #3, 2 arguments
SIR_EXPR_FUNCTION_CALL,		// New() #4
SIR_EXPR_MEMBER_ACCESS,		// New() #5
SIR_EXPR_MEMBER_POINTER,	// New() #5
SIR_EXPR_POST_INCREMENT,	// New() #3, 1 argument
SIR_EXPR_POST_DECREMENT,	// New() #3, 1 argument
SIR_EXPR_BITSLICE,		// New() #7
SIR_EXPR_PRE_INCREMENT,		// New() #3, 1 argument
SIR_EXPR_PRE_DECREMENT,		// New() #3, 1 argument
SIR_EXPR_ADDRESS_OF,		// New() #3, 1 argument
SIR_EXPR_CONTENT_OF,		// New() #3, 1 argument
SIR_EXPR_POSITIVE,		// New() #3, 1 argument
SIR_EXPR_NEGATIVE,		// New() #3, 1 argument
SIR_EXPR_NOT,			// New() #3, 1 argument
SIR_EXPR_LOGICAL_NOT,		// New() #3, 1 argument
SIR_EXPR_SIZEOF_EXPR,		// New() #3, 1 argument
SIR_EXPR_SIZEOF_TYPE,		// New() #6
SIR_EXPR_TYPE_CONVERSION,	// New() #6
SIR_EXPR_CONCATENATION,		// New() #3, 2 arguments
SIR_EXPR_MULTIPLY,		// New() #3, 2 arguments
SIR_EXPR_DIVIDE,		// New() #3, 2 arguments
SIR_EXPR_MODULO,		// New() #3, 2 arguments
SIR_EXPR_ADD,			// New() #3, 2 arguments
SIR_EXPR_SUBTRACT,		// New() #3, 2 arguments
SIR_EXPR_SHIFT_LEFT,		// New() #3, 2 arguments
SIR_EXPR_SHIFT_RIGHT,		// New() #3, 2 arguments
SIR_EXPR_LESS,			// New() #3, 2 arguments
SIR_EXPR_GREATER,		// New() #3, 2 arguments
SIR_EXPR_LESS_EQUAL,		// New() #3, 2 arguments
SIR_EXPR_GREATER_EQUAL,		// New() #3, 2 arguments
SIR_EXPR_EQUAL,			// New() #3, 2 arguments
SIR_EXPR_NOT_EQUAL,		// New() #3, 2 arguments
SIR_EXPR_AND,			// New() #3, 2 arguments
SIR_EXPR_EOR,			// New() #3, 2 arguments
SIR_EXPR_OR,			// New() #3, 2 arguments
SIR_EXPR_LOGICAL_AND,		// New() #3, 2 arguments
SIR_EXPR_LOGICAL_OR,		// New() #3, 2 arguments
SIR_EXPR_CONDITION,		// New() #3, 3 arguments
SIR_EXPR_ASSIGNMENT,		// New() #3, 2 arguments
SIR_EXPR_MUL_ASSIGN,		// New() #3, 2 arguments
SIR_EXPR_DIV_ASSIGN,		// New() #3, 2 arguments
SIR_EXPR_MOD_ASSIGN,		// New() #3, 2 arguments
SIR_EXPR_ADD_ASSIGN,		// New() #3, 2 arguments
SIR_EXPR_SUB_ASSIGN,		// New() #3, 2 arguments
SIR_EXPR_SHL_ASSIGN,		// New() #3, 2 arguments
SIR_EXPR_SHR_ASSIGN,		// New() #3, 2 arguments
SIR_EXPR_AND_ASSIGN,		// New() #3, 2 arguments
SIR_EXPR_EOR_ASSIGN,		// New() #3, 2 arguments
SIR_EXPR_OR_ASSIGN,		// New() #3, 2 arguments
SIR_EXPR_COMMA			// New() #3, 2 arguments
};


/*** type definitions ***************************************************/


typedef enum SIR_ExprType		SIR_EXPRTYPE;

typedef class SIR_Expression		sir_expression;
typedef SIR_List<sir_expression>	sir_expr_list;
typedef class SIR_Expressions		sir_expressions;
typedef void *				sir_expr_marg;
typedef ERROR (SIR_Expression::*sir_expr_mptr)(void*);
typedef struct SIR_FixTypePtrArgs
	{
	sir_type	*OldType,
			*NewType;
	}				sir_fix_types;

typedef class SIR_Symbol		sir_symbol;	/* cyclic link */


/*** class declarations *************************************************/


	/**********************/
	/*** SIR_Expression ***/
	/**********************/


class SIR_Expression :		/* expression node */
	public SIR_Node,			/* is a node */
	public SIR_ListElem<SIR_Expression>	/* can be a list element */
{
public:
SIR_EXPRTYPE	ExprType;	/* type of this expression */
sir_constant	*Constant;	/* constant value (for constants) */
sir_symbol	*Symbol;	/* link to symbol entry (for identifiers) */
sir_expression	*Arg1;		/* first argument (eg. unary expr.) */
sir_expression	*Arg2;		/* second argument (eg. right argument) */
sir_expression	*Arg3;		/* third argument (for ternary expr.) */
sir_expressions	*Args;		/* list of arguments (for n-ary expr.) */
sir_type	*TypeArg;	/* link to type argument (sizeof() or cast) */
sir_type	*Type;		/* link to result type of this expression */
int		LeftBound,	/* slice bounds (for bitvector slicing) */
		RightBound;


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Expression(				/* constructor #1 (for constants) */
	SIR_EXPRTYPE	ExprType,
	sir_constant	*Constant,
	sir_type	*Type,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL);

SIR_Expression(				/* constructor #2 (for identifiers) */
	SIR_EXPRTYPE	ExprType,
	sir_symbol	*Symbol,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL);

SIR_Expression(				/* constructor #3 */
	SIR_EXPRTYPE	ExprType,	/* (for std. expr. with 0-3 args.) */
	sir_type	*Type,
	sir_expression	*Arg1 = NULL,
	sir_expression	*Arg2 = NULL,
	sir_expression	*Arg3 = NULL,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL);

SIR_Expression(				/* constructor #4 */
	SIR_EXPRTYPE	ExprType,	/* (for function calls) */
	sir_type	*Type,
	sir_expression	*Arg1,
	sir_expressions	*Args,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL);

SIR_Expression(				/* constructor #5 */
	SIR_EXPRTYPE	ExprType,	/* (for member access) */
	sir_type	*Type,
	sir_expression	*Arg1,
	sir_symbol	*Symbol,	/* (Symbol==NULL for bhvr. main()) */
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL);

SIR_Expression(				/* constructor #6 */
	SIR_EXPRTYPE	ExprType,	/* (for sizeof(type) or casting) */
	sir_type	*Type,
	sir_type	*TypeArg,
	sir_expression	*Arg1 = NULL,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL);

SIR_Expression(				/* constructor #7 */
	SIR_EXPRTYPE	ExprType,	/* (for bit slicing) */
	sir_type	*Type,
	sir_expression	*Arg1,
	int		LeftBound,
	int		RightBound,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL);

SIR_Expression(				/* constructor #9 (duplicator) */
	sir_expression	*Original);	/* (recursive!) */

~SIR_Expression(void);			/* destructor */


static sir_expression *New(	/* creates an identifier expression (#2b) */
	sir_symbol	*Symbol);	/* (returns NULL if SIR_Error) */

static sir_expression *New(	/* creates a special expression (#8b) */
	SIR_EXPRTYPE	ExprType,	/* (supports VOID, THIS) */
	sir_types	*TypeTable,	/* (returns NULL if SIR_Error) */
	sir_symbol	*ClassSymbol = NULL);	/* class for THIS (or NULL) */


BOOL IsLvalue(void);		/* checks if this expr. is an lvalue */

BOOL IsModifiableLvalue(void);	/* checks if this expr. is a modifiable lv. */

BOOL IsWritable(void);		/* checks if this expr. allows write access */

BOOL IsReadable(void);		/* checks if this expr. allows read access */

ERROR CheckWriteAccess(void);	/* checks for error regarding write access */

ERROR CheckReadAccess(void);	/* checks for error regarding read access */

BOOL IsDependant(		/* checks if this expr. depends on symbol */
	sir_symbol	*ThatSymbol);


ERROR DFS_ForAllNodes(		/* iterator over all nodes (depth first) */
	sir_node_mptr	MemberFct,
	sir_node_marg	MemberFctArg);

ERROR DFS_ForAllExpressions(	/* iterator over all expressions (dfs) */
	sir_expr_mptr	MemberFct,
	sir_expr_marg	MemberFctArg);

BOOL DFS_FindDependant(		/* searches for dependants (depth first) */
	sir_symbol	*ThatSymbol,
	sir_expression	**DepExpr);


ERROR MarkUsedTypes(		/* marks the type entries used here */
	sir_expr_marg	/* Unused */);


void UnAlias(void);		/* unalias all type, usertype, symbol links */


static void ExplicitTypeConv(	/* perform explicit type conversion */
	SIR_CONSTTYPE	FromType,
	SIR_CONSTTYPE	ToType,
	const char	**Prefix,
	const char	**Suffix);	/* (caller must substitute length!) */

ERROR WriteSC(			/* (re-) generates SpecC source code */
	GL_HANDLE	*IO,
	BOOL		WriteNotes,
	BOOL		CplusplusMode = FALSE,
	SIR_TYPETYPE	ExpectedType = SIR_TYPE_ANY_TYPE,
	int		ExpectedLength = 0);	/* for expected bitvectors */

ERROR WriteSC2(			/* (re-) generates SpecC source code */
	const char	*Op,	/* (for binary ops only) */
	GL_HANDLE	*IO,
	BOOL		WriteNotes,
	BOOL		CplusplusMode = FALSE,
	SIR_TYPETYPE	ExpectedType1 = SIR_TYPE_ANY_TYPE,
	SIR_TYPETYPE	ExpectedType2 = SIR_TYPE_ANY_TYPE,
	int		ExpectedLength1 = 0,	/* for expected bitvectors */
	int		ExpectedLength2 = 0);	/* for expected bitvectors */


sir_constant *Eval(void);	/* evaluate a constant expression */
				/* (returns a new Constant */
				/*  or NULL and SIR_Error) */

int IntegerEval(void);		/* evaluate a constant expression to integer */
				/* (may set SIR_Error!) */

sir_lineinfo *GetFirstLineInfo(	/* obtain first line info of the tree */
	void);


static sir_expression *New(	/* creates a constant expression (#1) */
	sir_constant	*NewConst,	/* (returns NULL if SIR_Error) */
	sir_types	*TypeTable);

static sir_expression *New(	/* creates an expr. with 1-3 arguments (#3) */
	SIR_EXPRTYPE	ExprType,	/* (returns NULL if SIR_Error) */
	sir_expression	*NewArg1,
	sir_expression	*NewArg2 = NULL,
	sir_expression	*NewArg3 = NULL);

static sir_expression *New(	/* creates a function call expression (#4) */
	sir_expression	*NewArg1,	/* (returns NULL if SIR_Error) */
	sir_expressions	*NewArgs);

static sir_expression *New(	/* creates a member access expression (#5) */
	SIR_EXPRTYPE	ExprType,	/* (returns NULL if SIR_Error) */
	sir_expression	*NewArg1,
	const char	*MemberName);

static sir_expression *New(	/* creates a sizeof(type) or cast expr. (#6) */
	SIR_EXPRTYPE	ExprType,	/* (returns NULL if SIR_Error) */
	sir_type	*TypeArg,
	sir_expression	*NewArg1 = NULL);

static sir_expression *New(	/* creates a bit slice expression (#7) */
	sir_expression	*NewArg1,	/* (returns NULL if SIR_Error) */
	int		LeftBound,
	int		RightBound);
};


	/***********************/
	/*** SIR_Expressions ***/
	/***********************/


class SIR_Expressions :		/* list of expressions */
	public SIR_List<SIR_Expression>/* is simply a list of expressions */
{					/* with additional methods */
public:

//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Expressions(		/* constructor #1 */
	sir_expression	*NewFirstEntry = NULL);

SIR_Expressions(		/* constructor #2 (duplicator) */
	sir_expressions	*Original);	/* (recursive!) */

~SIR_Expressions(void);		/* destructor */


ERROR DFS_ForAllNodes(		/* iterator over all nodes (depth first) */
	sir_node_mptr	MemberFct,
	sir_node_marg	MemberFctArg);

ERROR DFS_ForAllExpressions(	/* iterator over all expressions (dfs) */
	sir_expr_mptr	MemberFct,
	sir_expr_marg	MemberFctArg);
};


/*** exported functions *************************************************/


	/* none */


#endif /* INTREP_EXPRESSION_H */

/* EOF IntRep/Expression.h */
