/************************************************************************/
/* Type.cc: SpecC Internal Representation, Type Classes			*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 10/07/97 */
/************************************************************************/

/* last update: 11/29/01 */

/* modifications: (most recent first)
 *
 * 11/29/01 RD	inserted a missing 'const' qualifier
 * 11/21/01 RD	took out default arguments from function definitions
 * 11/01/01 RD	eliminated trailing space when printing abstract types
 * 10/31/01 RD	added SIR_Type::PrettyString() with separated pre/post-fix
 * 05/31/01 RD	eliminated level 2 of SIR API
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 05/05/01 RD	changed use of 'long long' to 'LONG_LONG' (forgotten case)
 * 04/30/01 RD	changed use of len member to length() method for bitvectors
 * 04/30/01 RD	replaced use of obsolete form() from "stream.h" with own one
 * 04/27/01 RD	added ConvertInitializer() to perform explicit type
 *		conversion for initializers to expected type
 * 04/26/01 RD	allow conversion of any floating point type to any bit vector
 *		if the target size is known (which is the case for all
 *		uses of IsConvertableTo() at this time)
 * 04/25/01 RD	added parameter UseShadowName to PrettyString()
 * 04/20/01 RD	added static version of ConstType()
 * 04/18/01 RD	fixed C++ generation for non-native 'long long' type
 * 04/16/01 RD	added support for non-native 'long long' type;
 *		changed use of 'long long int' to TIME type
 * 01/11/01 RD	bug fix: don't forget direction for array ports
 * 07/01/00 RD	added Sir_Type::SubTypeTreeContains() (Bug fix)
 * 06/27/00 RD	added code for 12 and 16 byte alignment
 */

#include "IntRep/Type.h"
#include "IntRep/Design.h"
#include "IntRep/BasicType.h"
#include "IntRep/Extern.h"

#include "Alignment.h"

#include <assert.h>


/*** constants and macros ***********************************************/


//#define SIR_DEBUG_TYPE_GARBAGE_COLLECTOR	/* debug garbage collector */


/*** internal type declarations *****************************************/


	/* (none) */


/*** internal variables *************************************************/


static const char *TypeName[] =	/* table of built-in type names */
{				/* (requires well-defined enum SIR_TypeType) */

"bool",				/* SIR_TYPE_BOOL */
"char",				/* SIR_TYPE_CHAR */
"unsigned char",		/* SIR_TYPE_UCHAR */
"short int",			/* SIR_TYPE_SHORT */
"unsigned short int",		/* SIR_TYPE_USHORT */
"int",				/* SIR_TYPE_INT */
"unsigned int",			/* SIR_TYPE_UINT */
"long int",			/* SIR_TYPE_LONG */
"unsigned long int",		/* SIR_TYPE_ULONG */
"long long int",		/* SIR_TYPE_LONGLONG */
"unsigned long long int",	/* SIR_TYPE_ULONGLONG */
"float",			/* SIR_TYPE_FLOAT */
"double",			/* SIR_TYPE_DOUBLE */
"long double",			/* SIR_TYPE_LONGDOUBLE */
"bit",				/* SIR_TYPE_BIT */	/* [l:u] */
"unsigned bit",			/* SIR_TYPE_UBIT */	/* [l:u] */
"void",				/* SIR_TYPE_VOID */
"event",			/* SIR_TYPE_EVENT */
"*",				/* SIR_TYPE_POINTER */
NULL,				/* SIR_TYPE_STRUCT */	/* set dynamically */
NULL,				/* SIR_TYPE_UNION */	/* set dynamically */
NULL,				/* SIR_TYPE_ENUM */	/* set dynamically */
"[]",				/* SIR_TYPE_ARRAY */	/* not really used */
"()",				/* SIR_TYPE_FUNCTION */	/* not really used */
"...",				/* SIR_TYPE_ANY_TYPE */
NULL,				/* SIR_TYPE_BEHAVIOR */	/* set dynamically */
NULL,				/* SIR_TYPE_CHANNEL */	/* set dynamically */
NULL				/* SIR_TYPE_INTERFACE *//* set dynamically */
};


/*** class implementations **********************************************/


	/****************/
	/*** SIR_Type ***/
	/****************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Type::SIR_Type(		/* constructor #1 (for built-in types) */
	SIR_TYPETYPE	Type,
	bool		Const /* = false */,
	bool		Volatile /* = false */,
	int		LeftBound /* = 0 */,
	int		RightBound /* = 0 */,
	SIR_DIRECTION	Direction /* = SIR_PORT_NONE */)
{

SIR_Type::Type = Type;

switch(Type)		/* type-specific checks */
   { case SIR_TYPE_BOOL:
     case SIR_TYPE_CHAR:
     case SIR_TYPE_UCHAR:
     case SIR_TYPE_SHORT:
     case SIR_TYPE_USHORT:
     case SIR_TYPE_INT:
     case SIR_TYPE_UINT:
     case SIR_TYPE_LONG:
     case SIR_TYPE_ULONG:
     case SIR_TYPE_LONGLONG:
     case SIR_TYPE_ULONGLONG:
     case SIR_TYPE_FLOAT:
     case SIR_TYPE_DOUBLE:
     case SIR_TYPE_LONGDOUBLE:
	{ assert((LeftBound == 0) && (RightBound == 0));
	  break;
	 }
     case SIR_TYPE_BIT:
     case SIR_TYPE_UBIT:
	{ break;
	 }
     case SIR_TYPE_VOID:
	{ assert((LeftBound == 0) && (RightBound == 0));
	  break;
	 }
     case SIR_TYPE_EVENT:
	{ assert((LeftBound == 0) && (RightBound == 0));
	  assert((Const == false) && (Volatile == false));
	  break;
	 }
     case SIR_TYPE_POINTER:
     case SIR_TYPE_STRUCT:
     case SIR_TYPE_UNION:
     case SIR_TYPE_ENUM:
     case SIR_TYPE_ARRAY:
     case SIR_TYPE_FUNCTION:
	{ assert(FALSE);	/* not supported by this constructor */
	 }
     case SIR_TYPE_ANY_TYPE:
	{ assert((LeftBound == 0) && (RightBound == 0));
	  assert((Const == false) && (Volatile == false));
	  assert(Direction == SIR_PORT_NONE);
	  break;
	 }
     case SIR_TYPE_BEHAVIOR:
     case SIR_TYPE_CHANNEL:
     case SIR_TYPE_INTERFACE:
	{ assert(FALSE);	/* not supported by this constructor */
	 }
     default:
	{ assert(FALSE);	/* bad Type */
	 }
    } /* hctiws */

/* validate the table TypeName[] */
assert(0 == strcmp(TypeName[SIR_TYPE_BOOL], "bool"));
assert(0 == strcmp(TypeName[SIR_TYPE_EVENT], "event"));
assert(0 == strcmp(TypeName[SIR_TYPE_ANY_TYPE], "..."));

SIR_Type::Name		= TypeName[Type];
SIR_Type::SubType	= NULL;
SIR_Type::UserType	= NULL;
SIR_Type::ClassSymbol	= NULL;
SIR_Type::Parameters	= NULL;
SIR_Type::LeftBound	= LeftBound;
SIR_Type::RightBound	= RightBound;
SIR_Type::Size		= 0;
SIR_Type::Const		= Const;
SIR_Type::Volatile	= Volatile;
SIR_Type::Direction	= Direction;
SIR_Type::Alias		= NULL;

} /* end of SIR_Type::SIR_Type #1 */


SIR_Type::SIR_Type(		/* constructor #2 (for composite types) */
	sir_type	*SubType,	/* might be NULL (for abstract types) */
	SIR_TYPETYPE	Type,
	sir_type_ptrs	*Parameters /* = NULL */,
	int		Size /* = 0 */,
	bool		Const /* = false */,
	bool		Volatile /* = false */,
	SIR_DIRECTION	Direction /* = SIR_PORT_NONE */)
{

SIR_Type::Type = Type;

switch(Type)		/* type-specific checks */
   { case SIR_TYPE_BOOL:
     case SIR_TYPE_CHAR:
     case SIR_TYPE_UCHAR:
     case SIR_TYPE_SHORT:
     case SIR_TYPE_USHORT:
     case SIR_TYPE_INT:
     case SIR_TYPE_UINT:
     case SIR_TYPE_LONG:
     case SIR_TYPE_ULONG:
     case SIR_TYPE_LONGLONG:
     case SIR_TYPE_ULONGLONG:
     case SIR_TYPE_FLOAT:
     case SIR_TYPE_DOUBLE:
     case SIR_TYPE_LONGDOUBLE:
     case SIR_TYPE_BIT:
     case SIR_TYPE_UBIT:
     case SIR_TYPE_VOID:
     case SIR_TYPE_EVENT:
	{ assert(FALSE);	/* wrong constructor */
	 }
     case SIR_TYPE_POINTER:
	{ assert(Size == 0);
	  break;
	 }
     case SIR_TYPE_STRUCT:
     case SIR_TYPE_UNION:
     case SIR_TYPE_ENUM:
	{ assert(FALSE);	/* wrong constructor */
	 }
     case SIR_TYPE_ARRAY:
     case SIR_TYPE_FUNCTION:
	{ break;
	 }
     case SIR_TYPE_ANY_TYPE:
     case SIR_TYPE_BEHAVIOR:
     case SIR_TYPE_CHANNEL:
     case SIR_TYPE_INTERFACE:
	{ assert(FALSE);	/* wrong constructor */
	 }
     default:
	{ assert(FALSE);	/* bad Type */
	 }
    } /* hctiws */

SIR_Type::Name		= TypeName[Type];
SIR_Type::SubType	= SubType;
SIR_Type::UserType	= NULL;
SIR_Type::ClassSymbol	= NULL;
SIR_Type::Parameters	= Parameters;
SIR_Type::LeftBound	= 0;
SIR_Type::RightBound	= 0;
SIR_Type::Size		= Size;
SIR_Type::Const		= Const;
SIR_Type::Volatile	= Volatile;
SIR_Type::Direction	= Direction;
SIR_Type::Alias		= NULL;

} /* end of SIR_Type::SIR_Type #2 */


SIR_Type::SIR_Type(		/* constructor #3 (for class types) */
	SIR_TYPETYPE	Type,
	const char	*NamePtr,
	sir_symbol	*ClassSymbol /* = NULL */,
	sir_type_ptrs	*Parameters /* = NULL */)
{

assert(  (Type == SIR_TYPE_BEHAVIOR)
       ||(Type == SIR_TYPE_CHANNEL)
       ||(Type == SIR_TYPE_INTERFACE));
assert(NamePtr != NULL);

SIR_Type::Type		= Type;
SIR_Type::Name		= NamePtr;
SIR_Type::SubType	= NULL;
SIR_Type::UserType	= NULL;
SIR_Type::ClassSymbol	= ClassSymbol;
SIR_Type::Parameters	= Parameters;
SIR_Type::LeftBound	= 0;
SIR_Type::RightBound	= 0;
SIR_Type::Size		= 0;
SIR_Type::Const		= false;
SIR_Type::Volatile	= false;
SIR_Type::Direction	= SIR_PORT_NONE;
SIR_Type::Alias		= NULL;

} /* end of SIR_Type::SIR_Type #3 */


SIR_Type::SIR_Type(		/* constructor #4 (duplicator) */
	sir_type	*Original)
{

assert(Original != NULL);

SIR_Type::Type		= Original->Type;
SIR_Type::Name		= Original->Name;
SIR_Type::SubType	= Original->SubType;
SIR_Type::UserType	= Original->UserType;
SIR_Type::ClassSymbol	= Original->ClassSymbol;
SIR_Type::Parameters	= Original->Parameters ?
				new SIR_TypePtrs(Original->Parameters) :
				NULL;
SIR_Type::LeftBound	= Original->LeftBound;
SIR_Type::RightBound	= Original->RightBound;
SIR_Type::Size		= Original->Size;
SIR_Type::Const		= Original->Const;
SIR_Type::Volatile	= Original->Volatile;
SIR_Type::Direction	= Original->Direction;
SIR_Type::Alias		= NULL;

} /* end of SIR_Type::SIR_Type #4 */


SIR_Type::SIR_Type(		/* constructor #5 (for user-defined types) */
	SIR_TYPETYPE	Type,
	const char	*NamePtr,
	sir_user_type	*UserType,
	bool		Const /* = false */,
	bool		Volatile /* = false */,
	SIR_DIRECTION	Direction /* = SIR_PORT_NONE */)
{

assert(  (Type == SIR_TYPE_STRUCT)
       ||(Type == SIR_TYPE_UNION)
       ||(Type == SIR_TYPE_ENUM));

SIR_Type::Type		= Type;
SIR_Type::Name		= NamePtr;
SIR_Type::SubType	= NULL;
SIR_Type::UserType	= UserType;
SIR_Type::ClassSymbol	= NULL;
SIR_Type::Parameters	= NULL;
SIR_Type::LeftBound	= 0;
SIR_Type::RightBound	= 0;
SIR_Type::Size		= 0;
SIR_Type::Const		= Const;
SIR_Type::Volatile	= Volatile;
SIR_Type::Direction	= Direction;
SIR_Type::Alias		= NULL;

if (UserType)
   { UserType->Type = this;	/* automatically close the pointer loop */
    } /* fi */

} /* end of SIR_Type::SIR_Type #5 */


SIR_Type::~SIR_Type(void)	/* destructor */
{

delete Parameters;

} /* end of SIR_Type::~SIR_Type */


BOOL SIR_Type::IsPort(void)	/* determines if this is a port type */
{

return(  (Type == SIR_TYPE_INTERFACE)
       ||(Direction != SIR_PORT_NONE));

} /* end of SIR_Type::IsPort */


BOOL SIR_Type::VoidParameters(void)	/* determines if there are no params. */
{

assert(Parameters != NULL);	/* must be a class or function type */

if (  (Parameters->NumElements() == 0)
    ||(  (Parameters->NumElements() == 1)
       &&(Parameters->First()->Type->Type == SIR_TYPE_VOID)))
   { return(TRUE);
    } /* fi */

return(FALSE);

} /* end of SIR_Type::VoidParameters */


BOOL SIR_Type::IsConvertableTo(	/* determines possibility of conversion */
	sir_type	*TargetType)	/* (silent standard conversions) */
{

assert(TargetType != NULL);

switch(Type)
   { case SIR_TYPE_BOOL:
     case SIR_TYPE_CHAR:
     case SIR_TYPE_UCHAR:
     case SIR_TYPE_SHORT:
     case SIR_TYPE_USHORT:
     case SIR_TYPE_UINT:
     case SIR_TYPE_ULONG:
     case SIR_TYPE_LONGLONG:
     case SIR_TYPE_ULONGLONG:
     case SIR_TYPE_BIT:
     case SIR_TYPE_UBIT:
     case SIR_TYPE_ENUM:
	{ switch (TargetType->Type)
	     { case SIR_TYPE_VOID:
	       case SIR_TYPE_EVENT:
	       case SIR_TYPE_POINTER:
	       case SIR_TYPE_STRUCT:
	       case SIR_TYPE_UNION:
	       case SIR_TYPE_ARRAY:
	       case SIR_TYPE_FUNCTION:
	       case SIR_TYPE_ANY_TYPE:
	       case SIR_TYPE_BEHAVIOR:
	       case SIR_TYPE_CHANNEL:
	       case SIR_TYPE_INTERFACE:
		  { return(FALSE);
		   }
	       case SIR_TYPE_ENUM:
		  { return(UserType == TargetType->UserType);
		   }
	       default:
		  { return(TRUE);
		   }
	      } /* hctiws */
	 }
     case SIR_TYPE_INT:
     case SIR_TYPE_LONG:
	{ switch (TargetType->Type)
	     { case SIR_TYPE_VOID:
	       case SIR_TYPE_EVENT:
	       case SIR_TYPE_STRUCT:
	       case SIR_TYPE_UNION:
	       case SIR_TYPE_ENUM:
	       case SIR_TYPE_ARRAY:
	       case SIR_TYPE_FUNCTION:
	       case SIR_TYPE_ANY_TYPE:
	       case SIR_TYPE_BEHAVIOR:
	       case SIR_TYPE_CHANNEL:
	       case SIR_TYPE_INTERFACE:
		  { return(FALSE);
		   }
	       case SIR_TYPE_POINTER:	/* 0 or 0l is same as (void*)0 */
	       default:
		  { return(TRUE);
		   }
	      } /* hctiws */
	 }
     case SIR_TYPE_FLOAT:
     case SIR_TYPE_DOUBLE:
     case SIR_TYPE_LONGDOUBLE:
	{ switch (TargetType->Type)
	     { case SIR_TYPE_VOID:
	       case SIR_TYPE_EVENT:
	       case SIR_TYPE_POINTER:
	       case SIR_TYPE_STRUCT:
	       case SIR_TYPE_UNION:
	       case SIR_TYPE_ENUM:
	       case SIR_TYPE_ARRAY:
	       case SIR_TYPE_FUNCTION:
	       case SIR_TYPE_ANY_TYPE:
	       case SIR_TYPE_BEHAVIOR:
	       case SIR_TYPE_CHANNEL:
	       case SIR_TYPE_INTERFACE:
		  { return(FALSE);
		   }
	       default:
		  { return(TRUE);
		   }
	      } /* hctiws */
	 }
     case SIR_TYPE_VOID:
     case SIR_TYPE_EVENT:
	{ return(TargetType->Type == Type);	/* only to itself */
	 }
     case SIR_TYPE_POINTER:
     case SIR_TYPE_ARRAY:
	{ switch (TargetType->Type)
	     { case SIR_TYPE_BOOL:
	       case SIR_TYPE_INT:
	       case SIR_TYPE_LONG:
		  { return(TRUE);
		   }
	       case SIR_TYPE_POINTER:
	       case SIR_TYPE_ARRAY:
		  { if (  (SubType->Type == TargetType->SubType->Type)
			||(SubType->Type == SIR_TYPE_VOID)
			||(TargetType->SubType->Type == SIR_TYPE_VOID))
		       { return(TRUE);
			} /* fi */
		    else
		       { return(FALSE);
			} /* esle */
		   }
	       default:
		  { return(FALSE);
		   }
	      } /* hctiws */
	 }
     case SIR_TYPE_STRUCT:
     case SIR_TYPE_UNION:
	{ if (  (TargetType->Type == SIR_TYPE_STRUCT)
	      ||(TargetType->Type == SIR_TYPE_UNION))
	     { return(UserType == TargetType->UserType);
	      } /* fi */
	  return(FALSE);
	 }
     case SIR_TYPE_FUNCTION:
	{ if (TargetType == this)
	     { return(TRUE);
	      } /* fi */
	  if (  (TargetType->Type == SIR_TYPE_POINTER)
	      &&(TargetType->SubType == this))
	     { return(TRUE);
	      } /* fi */
	  return(FALSE);
	 }
     case SIR_TYPE_ANY_TYPE:
	{ return(FALSE);
	 }
     case SIR_TYPE_BEHAVIOR:
     case SIR_TYPE_CHANNEL:
	{ if (  (TargetType->Type == SIR_TYPE_INTERFACE)
	      &&(ClassSymbol->Interfaces->Find(TargetType->ClassSymbol)))
	     { return(TRUE);
	      } /* fi */
	  return(FALSE);
	 }
     case SIR_TYPE_INTERFACE:
	{ return(TargetType == this);
	 }
     default:
	{ assert(FALSE);	/* bad Type */
	 }
    } /* hctiws */

return(FALSE);	/* to keep the compiler quiet */

} /* end of SIR_Type::IsConvertableTo */


BOOL SIR_Type::SubTypeTreeContains(	/* determines special dependency */
	sir_type	*TargetType)	/* (Bug fix 07/01/00, R.D.) */
{
sir_type_ptr	*Param;

assert(TargetType);	/* must be given */

if (SubType == TargetType)
   { return(TRUE);
    } /* fi */

if (Parameters)
   { Param = Parameters->First();
     while(Param)
	{ if (Param->Type == TargetType)
	     { return(TRUE);
	      } /* fi */
	  if (Param->Type->SubTypeTreeContains(TargetType)) /* recursion */
	    { return(TRUE);
	     } /* fi */
	  Param = Param->Succ();
	 } /* elihw */
    } /* fi */

if (SubType)
   { return(SubType->SubTypeTreeContains(TargetType));	/* recursion */
    } /* fi */

return(FALSE);

} /* end of SIR_Type::SubTypeTreeContains */


SIR_CONSTTYPE SIR_Type::ConstType(	/* determines the const type */
	SIR_TYPETYPE	TypeType)
{

switch(TypeType)
   { case SIR_TYPE_BOOL:
     case SIR_TYPE_CHAR:
     case SIR_TYPE_UCHAR:
     case SIR_TYPE_SHORT:
     case SIR_TYPE_USHORT:
     case SIR_TYPE_INT:
     case SIR_TYPE_UINT:
     case SIR_TYPE_LONG:
     case SIR_TYPE_ULONG:
     case SIR_TYPE_LONGLONG:
     case SIR_TYPE_ULONGLONG:
     case SIR_TYPE_FLOAT:
     case SIR_TYPE_DOUBLE:
     case SIR_TYPE_LONGDOUBLE:
     case SIR_TYPE_BIT:
     case SIR_TYPE_UBIT:
	{ return((SIR_CONSTTYPE)TypeType);	/* exploit smart encoding! */
	 }
     case SIR_TYPE_VOID:
     case SIR_TYPE_EVENT:
	{ assert(FALSE);	/* no equivalent const type */
	 }
     case SIR_TYPE_POINTER:
	{ /* assert(SubType->Type == SIR_TYPE_CHAR); */
	  return(SIR_CONST_CHARSTRING);
	 }
     case SIR_TYPE_STRUCT:
     case SIR_TYPE_UNION:
	{ assert(FALSE);	/* no equivalent const type */
	 }
     case SIR_TYPE_ENUM:
	{ return(SIR_CONST_INT);
	 }
     case SIR_TYPE_ARRAY:
     case SIR_TYPE_FUNCTION:
     case SIR_TYPE_ANY_TYPE:
     case SIR_TYPE_BEHAVIOR:
     case SIR_TYPE_CHANNEL:
     case SIR_TYPE_INTERFACE:
	{ assert(FALSE);	/* no equivalent const type */
	 }
     default:
	{ assert(FALSE);	/* bad Type */
	 }
    } /* hctiws */

return(SIR_CONST_BOOL);	/* to keep the compiler quiet */

} /* end of SIR_Type::ConstType */


SIR_CONSTTYPE SIR_Type::ConstType(void)	/* determines the const type */
{

return(ConstType(Type));

} /* end of SIR_Type::ConstType */


ERROR SIR_Type::Check(		/* performs general legal checks on this type */
	bool		VoidIsOK /* = false */,
	bool		IsPort /* = false */,
	bool		IsExternOrTypedef /* = false */,
	bool		IsPointer /* = false */,	/* used in recursion */
	bool		IsFunction /* = false */,	/* used in recursion */
	bool		IsArray /* = false */)		/* used in recursion */
{
string		TmpErrMsg;

if (IsPort)
   { if (  (Direction == SIR_PORT_NONE)
	 &&(Type != SIR_TYPE_INTERFACE)	/* interface is ok */
	 &&(Type != SIR_TYPE_VOID))	/* void is probably ok */
	{ return(SIR_ERROR_PORT_DIRECTION_REQUIRED);
	 } /* fi */
    } /* fi */
else
   { if (Direction != SIR_PORT_NONE)
	{ return(SIR_ERROR_ILLEGAL_PORT_DIRECTION);
	 } /* fi */
    } /* fi */

switch (Type)
   { case SIR_TYPE_BOOL:
     case SIR_TYPE_CHAR:
     case SIR_TYPE_UCHAR:
     case SIR_TYPE_SHORT:
     case SIR_TYPE_USHORT:
     case SIR_TYPE_INT:
     case SIR_TYPE_UINT:
     case SIR_TYPE_LONG:
     case SIR_TYPE_ULONG:
     case SIR_TYPE_LONGLONG:
     case SIR_TYPE_ULONGLONG:
     case SIR_TYPE_FLOAT:
     case SIR_TYPE_DOUBLE:
     case SIR_TYPE_LONGDOUBLE:
     case SIR_TYPE_BIT:
     case SIR_TYPE_UBIT:
	{ break;
	 }
     case SIR_TYPE_VOID:
	{ if (VoidIsOK)
	     { break;
	      } /* fi */
	  else
	     { return(SIR_ERROR_TYPE_VOID_NOT_ALLOWED);
	      } /* esle */
	 }
     case SIR_TYPE_EVENT:
	{ if (IsPointer || IsFunction || IsArray)
	     { return(SIR_ERROR_TYPE_EVENT_NOT_PLAIN);
	      } /* fi */
	  break;
	 }
     case SIR_TYPE_POINTER:
	{ return(SubType->Check(true, false, IsExternOrTypedef,
				true, false, false));
	 }
     case SIR_TYPE_STRUCT:
	{ if (  (! IsPointer)
	      &&(! IsExternOrTypedef)
	      &&(UserType->Members == NULL))
	     { SIR_ErrMsg.form(
		"Aggregate type 'struct %s' declared, but not defined", Name);
	       return(SIR_ERROR_TYPE_AGGREGATE_NOT_DEFINED);
	      } /* fi */
	  break;
	 }
     case SIR_TYPE_UNION:
	{ if (  (! IsPointer)
	      &&(! IsExternOrTypedef)
	      &&(UserType->Members == NULL))
	     { SIR_ErrMsg.form(
		"Aggregate type 'union %s' declared, but not defined", Name);
	       return(SIR_ERROR_TYPE_AGGREGATE_NOT_DEFINED);
	      } /* fi */
	  break;
	 }
     case SIR_TYPE_ENUM:
	{ break;
	 }
     case SIR_TYPE_ARRAY:
	{ if (IsFunction)
	     { return(SIR_ERROR_TYPE_FUNCTION_RETURNS_ARRAY);
	      } /* fi */
	  return(SubType->Check(false, false, IsExternOrTypedef,
				false, false, true));
	 }
     case SIR_TYPE_FUNCTION:
	{ return(SubType->Check(true, false, IsExternOrTypedef,
				false, true, false));
	 }
     case SIR_TYPE_ANY_TYPE:
	{ break;
	 }
     case SIR_TYPE_BEHAVIOR:
     case SIR_TYPE_CHANNEL:
     case SIR_TYPE_INTERFACE:
	{ break;	/* checked by the grammar or during construction */
	 }
     default:
	{ assert(FALSE);	/* bad Type */
	 }
    } /* hctiws */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Type::Check */


void SIR_Type::ConvertInitializer( /* convert initializer to expected type */
	sir_initializer	*Initializer)	/* (works recursively) */
{
#ifndef NDEBUG
sir_type	*InitType;
#endif /* NDEBUG */
sir_types	*TypeTable;
sir_member	*Member;
sir_initializer	*Init;

/* note: all assertions in this function are based on the assumption
 *	 that CheckInitializer() has confirmed everything OK
 */

assert(Initializer != NULL);
assert(  (  (Initializer->InitList != NULL)
	  &&(Initializer->Initializer == NULL))
       ||(  (Initializer->InitList == NULL)
	  &&(Initializer->Initializer != NULL)));

TypeTable = GetTable();
assert(TypeTable != NULL);

switch(Type)
   { case SIR_TYPE_BOOL:
     case SIR_TYPE_CHAR:
     case SIR_TYPE_UCHAR:
     case SIR_TYPE_SHORT:
     case SIR_TYPE_USHORT:
     case SIR_TYPE_INT:
     case SIR_TYPE_UINT:
     case SIR_TYPE_LONG:
     case SIR_TYPE_ULONG:
     case SIR_TYPE_LONGLONG:
     case SIR_TYPE_ULONGLONG:
     case SIR_TYPE_FLOAT:
     case SIR_TYPE_DOUBLE:
     case SIR_TYPE_LONGDOUBLE:
     case SIR_TYPE_BIT:
     case SIR_TYPE_UBIT:
	{ assert(! Initializer->InitList);
#ifndef NDEBUG
	  InitType = TypeTable->FindOrInsert(Initializer->Initializer);
#endif /* NDEBUG */
	  assert(InitType != NULL);
	  assert(InitType->IsConvertableTo(this));
	  Initializer->Initializer->Converted(
			(SIR_CONSTTYPE) Type,	/* exploit smart enumerations */
			_BITLEN(LeftBound, RightBound));
	  break;
	 }
     case SIR_TYPE_VOID:
     case SIR_TYPE_EVENT:
	{ assert(FALSE);
	 }
     case SIR_TYPE_POINTER:
	{ assert(! Initializer->InitList);
#ifndef NDEBUG
	  InitType = TypeTable->FindOrInsert(Initializer->Initializer);
#endif /* NDEBUG */
	  assert(InitType != NULL);
	  assert(InitType->IsConvertableTo(this));
	  /* don't do anything; must not need conversion */
	  break;
	 }
     case SIR_TYPE_STRUCT:
	{ assert(! Initializer->Initializer);
	  assert(UserType->IsDefinition());
	  Member = UserType->Members->First();
	  Init = Initializer->InitList->First();
	  while(Init)
	     { assert(Member);
	       Member->Type->ConvertInitializer(Init);	/* recursion */
	       Member = Member->Succ();
	       Init = Init->Succ();
	      } /* elihw */
	  break;
	 }
     case SIR_TYPE_UNION:
     case SIR_TYPE_ENUM:	// .... no initializer allowed???
	{ assert(FALSE);
	 }
     case SIR_TYPE_ARRAY:
	{ if (  (  (SubType->Type == SIR_TYPE_CHAR) /* special strings init. */
		 ||(SubType->Type == SIR_TYPE_UCHAR))
	      &&(Initializer->Initializer)
	      &&(Initializer->Initializer->Type == SIR_CONST_CHARSTRING))
	     { /* nothing to convert */
	       break;
	      } /* fi */
	  assert(! Initializer->Initializer);
	  Init = Initializer->InitList->First();
	  while(Init)
	     { SubType->ConvertInitializer(Init);	/* recursion */
	       Init = Init->Succ();
	      } /* elihw */
	  break;
	 }
     case SIR_TYPE_FUNCTION:
     case SIR_TYPE_ANY_TYPE:
     case SIR_TYPE_BEHAVIOR:
     case SIR_TYPE_CHANNEL:
     case SIR_TYPE_INTERFACE:
	{ assert(FALSE);	/* should not be possible */
	 }
     default:
	{ assert(FALSE);	/* bad Type */
	 }
    } /* hctiws */

} /* end of SIR_Type::ConvertInitializer */


ERROR SIR_Type::CheckInitializer(	/* check if initializer is acceptable */
	sir_initializer	*Initializer)	/* (works recursively) */
{
sir_type	*InitType;
sir_types	*TypeTable;
sir_member	*Member;
sir_initializer	*Init;
int		NumInits;

assert(Initializer != NULL);
assert(  (  (Initializer->InitList != NULL)
	  &&(Initializer->Initializer == NULL))
       ||(  (Initializer->InitList == NULL)
	  &&(Initializer->Initializer != NULL)));

TypeTable = GetTable();
assert(TypeTable != NULL);

switch(Type)
   { case SIR_TYPE_BOOL:
     case SIR_TYPE_CHAR:
     case SIR_TYPE_UCHAR:
     case SIR_TYPE_SHORT:
     case SIR_TYPE_USHORT:
     case SIR_TYPE_INT:
     case SIR_TYPE_UINT:
     case SIR_TYPE_LONG:
     case SIR_TYPE_ULONG:
     case SIR_TYPE_LONGLONG:
     case SIR_TYPE_ULONGLONG:
     case SIR_TYPE_FLOAT:
     case SIR_TYPE_DOUBLE:
     case SIR_TYPE_LONGDOUBLE:
     case SIR_TYPE_BIT:
     case SIR_TYPE_UBIT:
	{ if (Initializer->InitList)
	     { SIR_ErrMsg.form("Complex initializer for type '%s'",
				Name);
	       return(SIR_ERROR_COMPLEX_INIT_FOR_PLAIN_TYPE);
	      } /* fi */
	  InitType = TypeTable->FindOrInsert(Initializer->Initializer);
	  assert(InitType != NULL);
	  if (!(InitType->IsConvertableTo(this)))
	     { SIR_ErrMsg.form("Initializer type mismatch:"
							GL_ERROR_MSG_NEWLINE
				"cannot convert '%s' to type '%s'",
				Initializer->Initializer->Print(),
				Name);
	       return(SIR_ERROR_INITIALIZER_TYPE_MISMATCH);
	      } /* fi */
	  break;
	 }
     case SIR_TYPE_VOID:
	{ SIR_ErrMsg.form("Illegal initializer '%s':" GL_ERROR_MSG_NEWLINE
			"type 'void' cannot be initialized",
			(Initializer->InitList ? "{...}" :
				Initializer->Initializer->Print()));
	  return(SIR_ERROR_ILLEGAL_INITIALIZER);
	 }
     case SIR_TYPE_EVENT:
	{ SIR_ErrMsg.form("Illegal initializer '%s':" GL_ERROR_MSG_NEWLINE
			"type 'event' cannot be initialized",
			(Initializer->InitList ? "{...}" :
				Initializer->Initializer->Print()));
	  return(SIR_ERROR_ILLEGAL_INITIALIZER);
	 }
     case SIR_TYPE_POINTER:
	{ if (Initializer->InitList)
	     { return(SIR_ERROR_COMPLEX_INIT_FOR_POINTER_TYPE);
	      } /* fi */
	  InitType = TypeTable->FindOrInsert(Initializer->Initializer);
	  assert(InitType != NULL);
	  if (!(InitType->IsConvertableTo(this)))
	     { SIR_ErrMsg.form("Initializer type mismatch:"
							GL_ERROR_MSG_NEWLINE
				"cannot convert '%s' to pointer type",
				Initializer->Initializer->Print());
	       return(SIR_ERROR_INITIALIZER_TYPE_MISMATCH);
	      } /* fi */
	  break;
	 }
     case SIR_TYPE_STRUCT:
	{ if (Initializer->Initializer)
	     { SIR_ErrMsg.form("Plain initializer '%s' for structure type",
				Initializer->Initializer->Print());
	       return(SIR_ERROR_PLAIN_INIT_FOR_STRUCTURE);
	      } /* fi */
	  if (! UserType->IsDefinition())
	     { return(SIR_ERROR_INIT_INCOMPLETE_STRUCTURE);
	      } /* fi */
	  Member = UserType->Members->First();
	  Init = Initializer->InitList->First();
	  while(Init)
	     { if (! Member)
		  { return(SIR_ERROR_TOO_MANY_INITIALIZERS_1);
		   } /* fi */
	       if ((SIR_Error = Member->Type->CheckInitializer(Init)))
		  { return(SIR_Error);
		   } /* fi */
	       Member = Member->Succ();
	       Init = Init->Succ();
	      } /* elihw */
	  /* note: it's ok to have less initializers than members */
	  break;
	 }
     case SIR_TYPE_UNION:
	{ SIR_ErrMsg.form("Illegal initializer '%s':" GL_ERROR_MSG_NEWLINE
			"an union-type cannot be initialized",
			(Initializer->InitList ? "{...}" :
				Initializer->Initializer->Print()));
	  return(SIR_ERROR_ILLEGAL_INITIALIZER);
	 }
     case SIR_TYPE_ENUM:
	{ SIR_ErrMsg.form("Illegal initializer '%s':" GL_ERROR_MSG_NEWLINE
			"an enumerator-type cannot be initialized",
			(Initializer->InitList ? "{...}" :
				Initializer->Initializer->Print()));
	  return(SIR_ERROR_ILLEGAL_INITIALIZER);
	 }
     case SIR_TYPE_ARRAY:
	{ if (  (  (SubType->Type == SIR_TYPE_CHAR) /* special strings init. */
		 ||(SubType->Type == SIR_TYPE_UCHAR))
	      &&(Initializer->Initializer)
	      &&(Initializer->Initializer->Type == SIR_CONST_CHARSTRING))
	     { NumInits = Initializer->Initializer->CS_Value->length() + 1;
	       if (  (Size != SIR_UNKNOWN_ARRAY_SIZE)
		   &&(NumInits > Size))
		  { return(SIR_ERROR_TOO_MANY_INITIALIZERS_2);
		   } /* fi */
	       break;
	      } /* fi */
	  if (Initializer->Initializer)
	     { SIR_ErrMsg.form("Plain initializer '%s' for array type",
				Initializer->Initializer->Print());
	       return(SIR_ERROR_PLAIN_INIT_FOR_ARRAY);
	      } /* fi */
	  Init = Initializer->InitList->First();
	  NumInits = 0;
	  while(Init)
	     { NumInits++;
	       if ((SIR_Error = SubType->CheckInitializer(Init)))
		  { return(SIR_Error);
		   } /* fi */
	       Init = Init->Succ();
	      } /* elihw */
	  if (  (Size != SIR_UNKNOWN_ARRAY_SIZE)
	      &&(NumInits > Size))
	     { return(SIR_ERROR_TOO_MANY_INITIALIZERS_2);
	      } /* fi */
	  /* note: it's ok to have less initializers than elements */
	  break;
	 }
     case SIR_TYPE_FUNCTION:
     case SIR_TYPE_ANY_TYPE:
     case SIR_TYPE_BEHAVIOR:
     case SIR_TYPE_CHANNEL:
     case SIR_TYPE_INTERFACE:
	{ assert(FALSE);	/* should not be possible */
	 }
     default:
	{ assert(FALSE);	/* bad Type */
	 }
    } /* hctiws */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Type::CheckInitializer */


sir_types *SIR_Type::GetTable(void)	/* determines the table of this type */
{					/* (returns NULL if not in a table) */

return((sir_types*) Head());

} /* end of SIR_Type::GetTable */


sir_type *SIR_Type::Modified(	/* returns a (probably new) modified type */
	bool		Const /* = false */,	/* (must be in a table) */
	bool		Volatile /* = false */,
	SIR_DIRECTION	Direction /* = SIR_PORT_NONE */)
{

assert(GetTable() != NULL);

return(GetTable()->FindOrInsertModifiedType(this,
				Const, Volatile, Direction));

} /* end of SIR_Type::Modified */


unsigned int SIR_Type::SizeOf(		/* compute the size of this type */
	unsigned int	*Alignment /* = NULL */, /* obtain alignment (NULL) */
	BOOL		SpecCsize /* = TRUE */)	 /* SpecC or C++ size */
{
sir_member	*Member;
unsigned int	Size,
		MSize;
unsigned int	MemberAlign,
		MaxAlign,
		Align;
unsigned int	Padding;
unsigned int	NumBitsLeft;

/* note: for this computation to be correct it is necessary to	*/
/*       compile this code on the same machine architecture as	*/
/*       the target architecture!				*/

assert (sizeof(char) == 1);	/* smallest addressable unit */

switch(Type)
   { case SIR_TYPE_BOOL:
	{ if (Alignment)
	     { *Alignment = SIR_ALIGN_BOOL;
	      } /* fi */
	  return(sizeof(bool));
	 }
     case SIR_TYPE_CHAR:
     case SIR_TYPE_UCHAR:
	{ if (Alignment)
	     { *Alignment = SIR_ALIGN_CHAR;
	      } /* fi */
	  return(sizeof(char));
	 }
     case SIR_TYPE_SHORT:
     case SIR_TYPE_USHORT:
	{ if (Alignment)
	     { *Alignment = SIR_ALIGN_SHORT;
	      } /* fi */
	  return(sizeof(short));
	 }
     case SIR_TYPE_INT:
     case SIR_TYPE_UINT:
	{ if (Alignment)
	     { *Alignment = SIR_ALIGN_INT;
	      } /* fi */
	  return(sizeof(int));
	 }
     case SIR_TYPE_LONG:
     case SIR_TYPE_ULONG:
	{ if (Alignment)
	     { *Alignment = SIR_ALIGN_LONG;
	      } /* fi */
	  return(sizeof(long));
	 }
     case SIR_TYPE_LONGLONG:
     case SIR_TYPE_ULONGLONG:
	{ if (Alignment)
	     { *Alignment = SIR_ALIGN_LONGLONG;
	      } /* fi */
	  return(sizeof(LONG_LONG));
	 }
     case SIR_TYPE_FLOAT:
	{ if (Alignment)
	     { *Alignment = SIR_ALIGN_FLOAT;
	      } /* fi */
	  return(sizeof(float));
	 }
     case SIR_TYPE_DOUBLE:
	{ if (Alignment)
	     { *Alignment = SIR_ALIGN_DOUBLE;
	      } /* fi */
	  return(sizeof(double));
	 }
     case SIR_TYPE_LONGDOUBLE:
	{ if (Alignment)
	     { *Alignment = SIR_ALIGN_LONGDOUBLE;
	      } /* fi */
	  return(sizeof(long double));
	 }
     case SIR_TYPE_BIT:
     case SIR_TYPE_UBIT:
	{ if (Alignment)
	     { *Alignment = SIR_ALIGN_BITVECTOR;
	      } /* fi */
	  if (SpecCsize)
	     { return(_SIZEOFBIT(_BITLEN(LeftBound, RightBound)));
	      } /* fi */
	  else
	     { return(_SIZEOFBITCLASS(_BITLEN(LeftBound, RightBound)));
	      } /* esle */
	 }
     case SIR_TYPE_VOID:
	{ assert(FALSE);	/* should have been caught by type checking */
	 }
     case SIR_TYPE_EVENT:
	{ assert(SpecCsize == FALSE);	/* in SpecC, sizeof(event) is illegal */
	  if (Alignment)
	     { *Alignment = SIR_ALIGN_EVENT;
	      } /* fi */
	  return(SIR_SIZE_OF_EVENT);
	 }
     case SIR_TYPE_POINTER:
	{ if (Alignment)
	     { *Alignment = SIR_ALIGN_POINTER;
	      } /* fi */
	  return(sizeof(void*));
	 }
     case SIR_TYPE_STRUCT:
	{ struct OneBit {int b:1;};

	  Size = 0;
	  MaxAlign = 1;
	  NumBitsLeft = 0;
	  Member = UserType->Members->First();
	  while(Member)
	     { if (Member->BitFieldSize == SIR_BITFIELD_SIZE_NONE)
		  { MSize = Member->Type->SizeOf(&MemberAlign, FALSE);
		    NumBitsLeft = 0;
		   } /* fi */
	       else
		  { if (NumBitsLeft >= Member->BitFieldSize)
		       { MSize = 0;	/* we pack it into its predecessor */
			 NumBitsLeft -= Member->BitFieldSize;
			} /* fi */
		    else
		       { MSize = sizeof(struct OneBit);	/* allocate bits */
#ifdef SIR_BITFIELD_PACKING_TIGHT
			 NumBitsLeft += MSize * 8;
#else
#ifdef SIR_BITFIELD_PACKING_LOOSE
			 NumBitsLeft = MSize * 8;
#else
#error "Unknown bitfield packing type!"
#endif /* SIR_BITFIELD_PACKING_LOOSE */
#endif /* SIR_BITFIELD_PACKING_TIGHT */
			 while (NumBitsLeft < Member->BitFieldSize)
			    { MSize += sizeof(struct OneBit);
			      NumBitsLeft += sizeof(struct OneBit) * 8;
			     } /* elihw */
			 NumBitsLeft -= Member->BitFieldSize;
			} /* esle */
		    MemberAlign = SIR_ALIGN_INT;
		   } /* esle */
	       assert(SIR_ALIGN_MAX <= 16);	/* worst case alignment */
	       switch (MemberAlign)
		  { case 1:
		       { /* no alignment */
			 break;
			}
		    case 2:
		    case 4:
		    case 8:
		    case 12:	/* (this ugly value appears on Windows!) */
		    case 16:
		       { if (Size % MemberAlign)	/* align properly */
			    { Padding = MemberAlign - (Size % MemberAlign);
			      Size += Padding;
			     } /* fi */
			 MaxAlign = MAX(MaxAlign, MemberAlign);
			 break;
			}
		    default:
		       { assert(FALSE);	/* unsupported alignment */
			}
		   } /* hctiws */
	       Size += MSize;		/* add member's size */
	       Member = Member->Succ();
	      } /* elihw */
	  Align = MAX(MaxAlign, SIR_ALIGN_STRUCT);
	  assert(Align <= SIR_ALIGN_MAX);
	  if (Size % Align)	/* align properly, fill-up this structure */
	     { Size += Align - (Size % Align);
	      } /* fi */
	  if (Alignment)
	     { *Alignment = Align;
	      } /* fi */
	  return(Size);
	 }
     case SIR_TYPE_UNION:
	{ Size = 0;
	  MaxAlign = 1;
	  Member = UserType->Members->First();
	  while(Member)
	     { Size = MAX(Size, Member->Type->SizeOf(&MemberAlign, FALSE));
	       MaxAlign = MAX(MaxAlign, MemberAlign);
	       Member = Member->Succ();
	      } /* elihw */
	  if (Alignment)
	     { *Alignment = MAX(MaxAlign, SIR_ALIGN_UNION);
	      } /* fi */
	  assert(MaxAlign <= SIR_ALIGN_MAX);
	  return(Size);
	 }
     case SIR_TYPE_ENUM:
	{ if (Alignment)
	     { *Alignment = SIR_ALIGN_INT;
	      } /* fi */
	  return(sizeof(int));
	 }
     case SIR_TYPE_ARRAY:
	{ assert(this->Size != SIR_UNKNOWN_ARRAY_SIZE);	/* semantic check! */
	  Size = this->Size * SubType->SizeOf(&MemberAlign, FALSE);
	  if (Alignment)
	     { *Alignment = MemberAlign;
	      } /* fi */
	  return(Size);
	 }
     case SIR_TYPE_FUNCTION:
	{ assert(FALSE);	/* should have been caught by type checking */
	 }
     case SIR_TYPE_ANY_TYPE:
	{ assert(FALSE);	/* should not be possible */
	 }
     case SIR_TYPE_BEHAVIOR:
     case SIR_TYPE_CHANNEL:
     case SIR_TYPE_INTERFACE:
	{ assert(FALSE);	/* should have been caught by type checking */
	 }
     default:
	{ assert(FALSE);	/* bad Type */
	 }
    } /* hctiws */

return(0);	/* should never be executed */

} /* end of SIR_Type::SizeOf */


const char *SIR_Type::PrintDirection(	/* prints a SpecC port direction */
	SIR_DIRECTION	Direction,	/* (eg. "out" or "out ") */
	BOOL		AppendSpace /* = false */)
{

switch(Direction)
   { case SIR_PORT_NONE:
	{ return("");
	 }
     case SIR_PORT_IN:
	{ if (AppendSpace)
	     { return("in ");
	      } /* fi */
	  else
	     { return("in");
	      } /* esle */
	 }
     case SIR_PORT_OUT:
	{ if (AppendSpace)
	     { return("out ");
	      } /* fi */
	  else
	     { return("out");
	      } /* esle */
	 }
     case SIR_PORT_INOUT:
	{ if (AppendSpace)
	     { return("inout ");
	      } /* fi */
	  else
	     { return("inout");
	      } /* esle */
	 }
     default:
	{ assert(FALSE);	/* bad port direction */
	 }
    } /* hctiws */

return(NULL);	/* never reached */

} /* end of SIR_Type::PrintDirection */


const char *SIR_Type::PrettyString(	/* prints a SpecC type (#1) */
	string	*Buffer,		/* (Buffer must be initialized */
	BOOL	IncludeNotes /* = FALSE */,	/* with SymbolName or "" for */
	BOOL	CplusplusMode /* = FALSE */,	/* abstract types)           */
	BOOL	UserTypeDefReq /* = FALSE */,
	BOOL	UseShadowName /* = FALSE */)
{
string		Prefix,
		Postfix;
int		LastChar;

assert(Buffer != NULL);

Prefix = "";
Postfix = "";

PrettyString(&Prefix, &Postfix,
		IncludeNotes, CplusplusMode, UserTypeDefReq, UseShadowName);

if (  (Buffer->empty())
    &&(Postfix.empty()))
   { assert(Prefix.length() >= 1);
     LastChar = Prefix.length()-1;
     if (Prefix.chars()[LastChar] == ' ')	/* optimize away  */
	{ *Buffer = Prefix.before(LastChar);	/* trailing space */
	 } /* fi */
     else
	{ *Buffer = Prefix;
	 } /* esle */
    } /* fi */
else
   { Buffer->prepend(Prefix);
     *Buffer += Postfix;
    } /* esle */

return(Buffer->chars());

} /* end of SIR_Type::PrettyString #1 */


void SIR_Type::PrettyString(		/* prints a SpecC type (#2) */
	string	*PrefixReturn,			/* (both Prefix- and Postfix- */
	string	*PostfixReturn,			/* Return must be initialized */
	BOOL	IncludeNotes /* = FALSE */,	/* with an empty string ("")  */
	BOOL	CplusplusMode /* = FALSE */,
	BOOL	UserTypeDefReq /* = FALSE */,
	BOOL	UseShadowName /* = FALSE */)
{
sir_type_ptr	*Param;
string		ParamBuffer,
		TmpString;

assert(PrefixReturn != NULL);
assert(PostfixReturn != NULL);

switch(Type)
   { case SIR_TYPE_BOOL:
     case SIR_TYPE_CHAR:
     case SIR_TYPE_UCHAR:
     case SIR_TYPE_SHORT:
     case SIR_TYPE_USHORT:
     case SIR_TYPE_INT:
     case SIR_TYPE_UINT:
     case SIR_TYPE_LONG:
     case SIR_TYPE_ULONG:
	{ PrefixReturn->prepend(" ");
	  PrefixReturn->prepend(Name);
	  break;
	 }
     case SIR_TYPE_LONGLONG:
	{
#ifdef HAVE_LLONG
	  PrefixReturn->prepend(" ");
	  PrefixReturn->prepend(Name);
#else /* !HAVE_LLONG */
	  if (CplusplusMode)
	     { PrefixReturn->prepend("<" SIR_LLONG_SIGNED_FLAG "> ");
	       if (UseShadowName)
		  { PrefixReturn->prepend(SIR_LLONG_SHADOW_NAME);
		   } /* fi */
	       else
		  { PrefixReturn->prepend(SIR_LLONG_CLASS_NAME);
		   } /* esle */
	      } /* fi */
	  else
	     { PrefixReturn->prepend(" ");
	       PrefixReturn->prepend(Name);
	      } /* esle */
#endif /* HAVE_LLONG */
	  break;
	 }
     case SIR_TYPE_ULONGLONG:
	{
#ifdef HAVE_LLONG
	  PrefixReturn->prepend(" ");
	  PrefixReturn->prepend(Name);
#else /* !HAVE_LLONG */
	  if (CplusplusMode)
	     { PrefixReturn->prepend("<" SIR_LLONG_UNSIGNED_FLAG "> ");
	       if (UseShadowName)
		  { PrefixReturn->prepend(SIR_LLONG_SHADOW_NAME);
		   } /* fi */
	       else
		  { PrefixReturn->prepend(SIR_LLONG_CLASS_NAME);
		   } /* esle */
	      } /* fi */
	  else
	     { PrefixReturn->prepend(" ");
	       PrefixReturn->prepend(Name);
	      } /* esle */
#endif /* HAVE_LLONG */
	  break;
	 }
     case SIR_TYPE_FLOAT:
     case SIR_TYPE_DOUBLE:
     case SIR_TYPE_LONGDOUBLE:
	{ PrefixReturn->prepend(" ");
	  PrefixReturn->prepend(Name);
	  break;
	 }
     case SIR_TYPE_BIT:
	{ if (CplusplusMode)
	     { TmpString.form("%s<" SIR_BIT_LENGTH_NAME "(%d,%d),"
						SIR_BIT_SIGNED_FLAG "> ",
					(UseShadowName ? SIR_BIT_SHADOW_NAME
							: SIR_BIT_CLASS_NAME),
					LeftBound, RightBound);
	      } /* fi */
	  else
	     { TmpString.form("%s[%d:%d] ",
					Name, LeftBound, RightBound);
	      } /* esle */
	  PrefixReturn->prepend(TmpString);
	  break;
	 }
     case SIR_TYPE_UBIT:
	{ if (CplusplusMode)
	     { TmpString.form("%s<" SIR_BIT_LENGTH_NAME "(%d,%d),"
						SIR_BIT_UNSIGNED_FLAG "> ",
					(UseShadowName ? SIR_BIT_SHADOW_NAME
							: SIR_BIT_CLASS_NAME),
					LeftBound, RightBound);
	      } /* fi */
	  else
	     { TmpString.form("%s[%d:%d] ",
					Name, LeftBound, RightBound);
	      } /* esle */
	  PrefixReturn->prepend(TmpString);
	  break;
	 }
     case SIR_TYPE_VOID:
	{ PrefixReturn->prepend(" ");
	  PrefixReturn->prepend(Name);
	  break;
	 }
     case SIR_TYPE_EVENT:
	{ PrefixReturn->prepend(" ");
	  if (CplusplusMode)
	     { PrefixReturn->prepend(SIR_SIM_EVENT_NAME);
	      } /* fi */
	  else
	     { PrefixReturn->prepend(Name);
	      } /* esle */
	  break;
	 }
     case SIR_TYPE_POINTER:
	{ if (Volatile)
	     { PrefixReturn->prepend("volatile ");
	      } /* fi */
	  if (Const)
	     { PrefixReturn->prepend("const ");
	      } /* fi */
	  PrefixReturn->prepend(Name);
	  if (  (SubType->Type == SIR_TYPE_FUNCTION) /* pointer to function? */
	      ||(SubType->Type == SIR_TYPE_ARRAY))   /* pointer to array? */
	     { PrefixReturn->prepend("(");
	       *PostfixReturn += ")";
	      } /* fi */
	  SubType->PrettyString(PrefixReturn, PostfixReturn,
				IncludeNotes, CplusplusMode);
	  if (! CplusplusMode)
	     { PrefixReturn->prepend(PrintDirection(Direction, true));
	      } /* fi */
	  return;
	 }
     case SIR_TYPE_STRUCT:
     case SIR_TYPE_UNION:
     case SIR_TYPE_ENUM:
	{ PrefixReturn->prepend(" ");
	  PrefixReturn->prepend(UserType->PrettyString(
				&ParamBuffer, IncludeNotes,
				CplusplusMode, UserTypeDefReq));
	  break;
	 }
     case SIR_TYPE_ARRAY:
	{ if (Size != SIR_UNKNOWN_ARRAY_SIZE)
	     { TmpString.form("[%d]", Size);
	       *PostfixReturn += TmpString;
	      } /* fi */
	  else
	     { *PostfixReturn += "[]";
	      } /* esle */
	  SubType->PrettyString(PrefixReturn, PostfixReturn,
				IncludeNotes, CplusplusMode);
	  break;
	 }
     case SIR_TYPE_FUNCTION:
	{ *PostfixReturn += "(";
	  Param = Parameters->First();
	  while(Param)
	     { if (  (CplusplusMode)
		   &&(Param->Type->Type == SIR_TYPE_INTERFACE))
		  { ParamBuffer = "&";	/* interface parameter as reference */
		   } /* fi */
	       else
		  { ParamBuffer = "";
		   } /* esle */
	       *PostfixReturn += Param->Type->PrettyString(&ParamBuffer,
						IncludeNotes, CplusplusMode);
	       Param = Param->Succ();
	       if (Param)
		  { *PostfixReturn += ", ";
		   } /* fi */
	      } /* elihw */
	  *PostfixReturn += ")";
	  SubType->PrettyString(PrefixReturn, PostfixReturn,
				IncludeNotes, CplusplusMode);
	  break;
	 }
     case SIR_TYPE_ANY_TYPE:
	{ assert(PrefixReturn->empty());
	  assert(PostfixReturn->empty());
	  *PrefixReturn = Name;
	  assert(! Volatile);
	  assert(! Const);
	  assert(Direction == SIR_PORT_NONE);
	  break;
	 }
     case SIR_TYPE_BEHAVIOR:
     case SIR_TYPE_CHANNEL:
     case SIR_TYPE_INTERFACE:
	{ PrefixReturn->prepend(" ");
	  PrefixReturn->prepend(Name);
	  assert(! Volatile);
	  assert(! Const);
	  assert(Direction == SIR_PORT_NONE);
	  break;
	 }
     default:
	{ assert(FALSE);	/* bad Type */
	 }
    } /* hctiws */

if (Volatile)
   { PrefixReturn->prepend("volatile ");
    } /* fi */
if (Const)
   { PrefixReturn->prepend("const ");
    } /* fi */
if (! CplusplusMode)
   { PrefixReturn->prepend(PrintDirection(Direction, true));
    } /* fi */

} /* end of SIR_Type::PrettyString #2 */


void SIR_Type::MarkUsed(void)		/* mark this type as being used */
{
sir_type_ptr	*TypePtr;

if (Color == SIR_RED)
   { return;	/* already marked */
    } /* fi */

Color = SIR_RED;			/* mark this guy */
if (SubType)
   { SubType->MarkUsed();		/* mark its subtype */
    } /* fi */
if (Parameters)
   { TypePtr = Parameters->First();
     while(TypePtr)
	{ TypePtr->Type->MarkUsed();	/* mark its parameters */
	  TypePtr = TypePtr->Succ();
	 } /* elihw */
    } /* fi */

} /* end of SIR_Type::MarkUsed */


SIR_TYPECLASS SIR_Type::TypeClass(void)	/* determines the class of this type */
{

switch(Type)
   { case SIR_TYPE_BOOL:
     case SIR_TYPE_CHAR:
     case SIR_TYPE_UCHAR:
     case SIR_TYPE_SHORT:
     case SIR_TYPE_USHORT:
     case SIR_TYPE_INT:
     case SIR_TYPE_UINT:
     case SIR_TYPE_LONG:
     case SIR_TYPE_ULONG:
     case SIR_TYPE_LONGLONG:
     case SIR_TYPE_ULONGLONG:
	{ return(SIR_TYPECLASS_INTEGRAL);
	 }
     case SIR_TYPE_FLOAT:
     case SIR_TYPE_DOUBLE:
     case SIR_TYPE_LONGDOUBLE:
	{ return(SIR_TYPECLASS_FLOATING);
	 }
     case SIR_TYPE_BIT:
     case SIR_TYPE_UBIT:
	{ return(SIR_TYPECLASS_BITVECTOR);
	 }
     case SIR_TYPE_VOID:
     case SIR_TYPE_EVENT:
     case SIR_TYPE_POINTER:
     case SIR_TYPE_STRUCT:
     case SIR_TYPE_UNION:
	{ return(SIR_TYPECLASS_OTHER);
	 }
     case SIR_TYPE_ENUM:
	{ return(SIR_TYPECLASS_INTEGRAL);
	 }
     case SIR_TYPE_ARRAY:
     case SIR_TYPE_FUNCTION:
     case SIR_TYPE_ANY_TYPE:
	{ return(SIR_TYPECLASS_OTHER);
	 }
     case SIR_TYPE_BEHAVIOR:
     case SIR_TYPE_CHANNEL:
     case SIR_TYPE_INTERFACE:
	{ return(SIR_TYPECLASS_CLASS);
	 }
     default:
	{ assert(FALSE);	/* bad Type */
	 }
    } /* hctiws */

return(SIR_TYPECLASS_OTHER);	/* to keep the compiler quiet */

} /* end of SIR_Type::TypeClass */


	/*****************/
	/*** SIR_Types ***/
	/*****************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Types::SIR_Types(			/* constructor #1 */
	sir_design	*ThisDesign)
{

assert(ThisDesign != NULL);

SIR_Types::ThisDesign	= ThisDesign;

} /* end of SIR_Types::SIR_Types #1 */


SIR_Types::SIR_Types(			/* constructor #3 (duplicator) */
	sir_types	*Original,
	sir_design	*ThisDesign)
{
sir_type	*Curr;

assert(Original != NULL);
assert(ThisDesign != NULL);

SIR_Types::ThisDesign	= ThisDesign;

Curr = Original->First();
while(Curr)
   { Append(new SIR_Type(Curr));
     Curr = Curr->Succ();
    } /* elihw */

} /* end of SIR_Types::SIR_Types #3 */


SIR_Types::~SIR_Types(void)		/* destructor */
{

/* nothing to do */

} /* end of SIR_Types::~SIR_Types */


void SIR_Types::UnAlias(void)	/* unalias all type, usertype, symbol links */
{
sir_type	*Type;

Type = First();
while(Type)
   { if (  (Type->SubType)
	 &&(Type->SubType->Alias))
	{ Type->SubType = Type->SubType->Alias;
	 } /* fi */
     if (  (Type->UserType)
	 &&(Type->UserType->Alias))
	{ Type->UserType = Type->UserType->Alias;
	  if (Type->Name)
	     { assert(Type->UserType->Name != NULL);
	       Type->Name = Type->UserType->Name->chars();
	      } /* fi */
	 } /* fi */
     if (  (Type->ClassSymbol)
	 &&(Type->ClassSymbol->Alias))
	{ Type->ClassSymbol = Type->ClassSymbol->Alias;
	  Type->Name = Type->ClassSymbol->Name.chars();
	 } /* fi */
     if (Type->Parameters)
	{ Type->Parameters->UnAlias();
	 } /* fi */
     Type = Type->Succ();
    } /* elihw */

} /* end of SIR_Types::UnAlias */


ERROR SIR_Types::Integrate(		/* integrates imported type table */
	sir_types	*Imported)
{
sir_type	*ImpCurr,
		*ImpSucc;

ImpCurr = Imported->First();	/* for all imported types */
while(ImpCurr)
   { ImpSucc = ImpCurr->Succ();
     if (ImpCurr->Alias)	/* alias already set by usertype integration? */
	{ ImpCurr = ImpSucc;
	  continue;
	 } /* fi */
     if (  (ImpCurr->SubType)
	 &&(ImpCurr->SubType->Alias))	/* if already known speeds search up */
	{ ImpCurr->SubType = ImpCurr->SubType->Alias;
	 } /* fi */
     if (ImpCurr->Parameters)		/* same speed up here */
	{ ImpCurr->Parameters->UnAlias();
	 } /* fi */
     if (!(ImpCurr->Alias = Find(ImpCurr)))	/* set alias pointer  */
	{ if (Curr())				/* or move completely */
	     { InsertBefore(Imported->Remove(ImpCurr));
	      } /* fi */
	  else
	     { Append(Imported->Remove(ImpCurr));
	      } /* esle */
	 } /* fi */
     ImpCurr = ImpSucc;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Types::Integrate */


void SIR_Types::GarbageCollection(void)	/* garbage collector */
{
sir_type	*Curr,
		*Succ;
#ifdef SIR_DEBUG_TYPE_GARBAGE_COLLECTOR
string		Buffer;
#endif /* SIR_DEBUG_TYPE_GARBAGE_COLLECTOR */

#ifdef SIR_DEBUG_TYPE_GARBAGE_COLLECTOR
fprintf(stderr, "GARBAGE_COLLECTOR: Starting with %d type entries...\n",
			NumElements());
#endif /* SIR_DEBUG_TYPE_GARBAGE_COLLECTOR */

#ifndef NDEBUG
Curr = First();		/* make sure every entry is unmarked */
while(Curr)
   { assert(Curr->Color == SIR_WHITE);
     Curr = Curr->Succ();
    } /* elihw */
#endif /* NDEBUG */

ThisDesign->DFS_ForAllUserTypes(
		&SIR_UserType::MarkUsedTypes, NULL);

ThisDesign->DFS_ForAllSymbols(
		&SIR_Symbol::MarkUsedTypes, NULL);

#ifdef SIR_DEBUG_TYPE_GARBAGE_COLLECTOR
Curr = First();
while(Curr)
   { if (Curr->Color == SIR_WHITE)
	{ Buffer = "";
	  fprintf(stderr, "GARBAGE_COLLECTOR: Detected unused type: %s\n",
			Curr->PrettyString(&Buffer));
	 } /* fi */
     Curr = Curr->Succ();
    } /* elihw */
#endif /* SIR_DEBUG_TYPE_GARBAGE_COLLECTOR */

Curr = First();		/* remove every unused entry (unmarked) */
while(Curr)
   { Succ = Curr->Succ();
     if (Curr->Color != SIR_WHITE)
	{ assert(Curr->Color == SIR_RED);	/* marked = used */
	  Curr->Color = SIR_WHITE;
	 } /* fi */
     else					/* unmarked = not used */
	{ Remove(Curr);
	  delete Curr;
	 } /* esle */
     Curr = Succ;
    } /* elihw */

#ifdef SIR_DEBUG_TYPE_GARBAGE_COLLECTOR
fprintf(stderr, "GARBAGE_COLLECTOR: Ending with %d type entries...\n",
			NumElements());
#endif /* SIR_DEBUG_TYPE_GARBAGE_COLLECTOR */

} /* end of SIR_Types::GarbageCollection */


sir_type *SIR_Types::FindOrInsert(	/* find an entry or insert it */
	SIR_CONSTTYPE	ConstType,	/* if it not exists (#1)      */
	int		LeftBound /* = 0 */,
	int		RightBound /* = 0 */)
{
SIR_TYPETYPE	MyType;
sir_type	*TypeEntry;

switch(ConstType)
   { case SIR_CONST_BOOL:
     case SIR_CONST_CHAR:
     case SIR_CONST_UCHAR:
     case SIR_CONST_SHORT:
     case SIR_CONST_USHORT:
     case SIR_CONST_INT:
     case SIR_CONST_UINT:
     case SIR_CONST_LONG:
     case SIR_CONST_ULONG:
     case SIR_CONST_LONGLONG:
     case SIR_CONST_ULONGLONG:
     case SIR_CONST_FLOAT:
     case SIR_CONST_DOUBLE:
     case SIR_CONST_LONGDOUBLE:
	{ MyType = (SIR_TYPETYPE) ConstType;	/* is defined accordingly */
	  return(FindOrInsert(new SIR_Type(MyType, true)));
	 }
     case SIR_CONST_BIT:
     case SIR_CONST_UBIT:
	{ MyType = (SIR_TYPETYPE) ConstType;	/* is defined accordingly */
	  return(FindOrInsert(new SIR_Type(MyType, true, false,
						LeftBound, RightBound)));
	 }
     case SIR_CONST_CHARSTRING:
	{ TypeEntry = FindOrInsert(			/* subtype "char" */
			new SIR_Type(SIR_TYPE_CHAR, true));
	  return(FindOrInsert(				/* main type "*" */
			new SIR_Type(TypeEntry, SIR_TYPE_POINTER,
				 	NULL, 0, true, false)));
	 }
     default:
	{ assert(FALSE);	/* bad ConstType */
	 }
    } /* hctiws */

return(NULL);	/* never reached */

} /* end of SIR_Types::FindOrInsert #1 */


sir_type *SIR_Types::FindOrInsert(	/* find an entry or insert it */
	SIR_TYPETYPE	Type,		/* if it not exists (#2)      */
	bool		Const /* = false */,
	bool		Volatile /* = false */,
	int		LeftBound /* = 0 */,
	int		RightBound /* = 0 */,
	SIR_DIRECTION	Direction /* = SIR_PORT_NONE */)
{

return(FindOrInsert(
	new SIR_Type(Type, Const, Volatile,
			LeftBound, RightBound,
			Direction)));

} /* end of SIR_Types::FindOrInsert #2 */


sir_type *SIR_Types::FindOrInsert(	/* find an entry or insert it */
	sir_basic_type	*BasicType)	/* if it not exists (#3)      */
{
SIR_TYPETYPE	Type;

	/* note: the following conversion depends on smart  */
	/*       encoding of SIR_BasicTypeType (see Type.h) */

Type = SIR_TYPE_ANY_TYPE;	/* just to keep the compiler quiet */
switch(BasicType->BasicType)
   { case SIR_BASICTYPE_BOOL:
	{ Type = (SIR_TYPETYPE) BasicType->BasicType;
	  break;
	 }
     case SIR_BASICTYPE_CHAR:
     case SIR_BASICTYPE_SHORT:
     case SIR_BASICTYPE_INT:
     case SIR_BASICTYPE_LONG:
     case SIR_BASICTYPE_LONGLONG:
	{ Type = (SIR_TYPETYPE) BasicType->BasicType;
	  if (BasicType->IsUnsigned)
	     { Type = (SIR_TYPETYPE)(Type + 1);
	      } /* fi */
	  break;
	 }
     case SIR_BASICTYPE_FLOAT:
     case SIR_BASICTYPE_DOUBLE:
     case SIR_BASICTYPE_LONGDOUBLE:
	{ Type = (SIR_TYPETYPE) BasicType->BasicType;
	  break;
	 }
     case SIR_BASICTYPE_BIT:
	{ Type = (SIR_TYPETYPE) BasicType->BasicType;
	  if (BasicType->IsUnsigned)
	     { Type = (SIR_TYPETYPE)(Type + 1);
	      } /* fi */
	  break;
	 }
     case SIR_BASICTYPE_VOID:
     case SIR_BASICTYPE_EVENT:
	{ Type = (SIR_TYPETYPE) BasicType->BasicType;
	  break;
	 }
     case SIR_BASICTYPE_UNKNOWN:	/* defaults to type 'int' */
	{ Type = SIR_TYPE_INT;
	  if (BasicType->IsUnsigned)
	     { Type = (SIR_TYPETYPE)(Type + 1);
	      } /* fi */
	  break;
	 }
     default:
	{ assert(false);	/* bad BasicType */
	 }
    } /* hctiws */

return(FindOrInsert(
	new SIR_Type(Type, BasicType->IsConst, BasicType->IsVolatile,
			BasicType->LeftBound, BasicType->RightBound)));

} /* end of SIR_Types::FindOrInsert #3 */


sir_type *SIR_Types::FindOrInsert(	/* find an entry or insert it */
	SIR_TYPETYPE	Type,		/* if it not exists (#4)      */
	sir_type	*SubType,	/* (for composite types)      */
	sir_type_ptrs	*Parameters /* = NULL */,
	int		Size /* = 0 */,
	bool		Const /* = false */,
	bool		Volatile /* = false */,
	SIR_DIRECTION	Direction /* = SIR_PORT_NONE */)
{

return(FindOrInsert(
	new SIR_Type(SubType, Type, Parameters,
			Size, Const, Volatile, Direction)));

} /* end of SIR_Types::FindOrInsert #4 */


sir_type *SIR_Types::FindOrInsert(	/* find an entry or insert it */
	sir_type	*NewType)	/* if it not exists (#5)      */
{					/* (consumes NewType!) */
sir_type	*TypeEntry;

if ((TypeEntry = Find(NewType)))
   { delete NewType;
     return(TypeEntry);	/* found! */
    } /* fi */

if (Curr())
   { return(InsertBefore(NewType));	/* NewType inserted */
    } /* fi */
else
   { return(Append(NewType));		/* NewType appended */
    } /* esle */

} /* end of SIR_Types::FindOrInsert #5 */


sir_type *SIR_Types::FindOrInsert(	/* find an entry or insert it */
	sir_constant	*Const)		/* if it not exists (#6)      */
{					/* (for constant types)       */

if (  (Const->Type == SIR_CONST_BIT)
    ||(Const->Type == SIR_CONST_UBIT))
   { return(FindOrInsert(Const->Type,
			Const->BIT_Value->length() - 1, 0));
    } /* fi */
else
   { return(FindOrInsert(Const->Type));
    } /* esle */

} /* end of SIR_Types::FindOrInsert #6 */


sir_type *SIR_Types::FindOrInsertModifiedType( /* find/insert a modified type */
	sir_type	*OrigType,
	bool		Const /* = false */,
	bool		Volatile /* = false */,
	SIR_DIRECTION	Direction /* = SIR_PORT_NONE */)
{
sir_type	*NewType;

assert(OrigType != NULL);

if (  (Const == OrigType->Const)
    &&(Volatile == OrigType->Volatile)
    &&(Direction == OrigType->Direction))
   { return(OrigType);	/* nothing to modify */
    } /* fi */

NewType = new SIR_Type(OrigType);

NewType->Const = Const;
NewType->Volatile = Volatile;
NewType->Direction = Direction;

return(FindOrInsert(NewType));

} /* end of SIR_Types::FindOrInsertModifiedType */


sir_type *SIR_Types::FindOrInsertDeclaratorType(/* find or insert */
	sir_type	*TypeEntry,		/* a declarator type */
	sir_type_list	*DeclTypes)	/* (consumes the list elements) */
{
sir_type	*Entry,
		*DeclType;

assert(TypeEntry != NULL);	/* basic type entry */
assert(DeclTypes != NULL);	/* list of declarator types */

Entry = TypeEntry;	/* start from given type entry */
while((DeclType = DeclTypes->Last()))	/* work backwards */
   { DeclTypes->Remove(DeclType);	/* take prepared type out */
     DeclType->SubType = Entry;		/* now we know the subtype */
     Entry = FindOrInsert(DeclType);	/* find or create type table entry */
    } /* elihw */

return(Entry);

} /* end of SIR_Types::FindOrInsertDeclaratorType */


sir_type *SIR_Types::FindOrInsertClassType( /* find or insert a class type */
	SIR_TYPETYPE	Type,
	const char	*Name,
	sir_type_ptrs	*Parameters,
	sir_symbol	*ClassSymbol /* = NULL */)
{

assert(  (Type == SIR_TYPE_BEHAVIOR)
       ||(Type == SIR_TYPE_CHANNEL)
       ||(Type == SIR_TYPE_INTERFACE));
assert(Name != NULL);

return(FindOrInsert(
	new SIR_Type(Type, Name, ClassSymbol, Parameters)));

} /* end of SIR_Types::FindOrInsertClassType */


sir_type *SIR_Types::Find(		/* find an equivalent entry */
	sir_type	*Type)		/* (where TypeCmp() == 0) */
{
int		CmpVal;

First();	/* search for the entry */
while(Curr())
   { if (0 == (CmpVal = TypeCmp(Type, Curr())))
	{ return(Curr());	/* found the entry */
	 } /* fi */
     if (CmpVal < 0)
	{ break;	/* not found (insert here) */
	 } /* fi */
     Next();
    } /* elihw */

return(NULL);	/* not found */

} /* end of SIR_Types::Find */


sir_type *SIR_Types::Insert(		/* insert a new entry */
	sir_type	*NewType)
{
sir_type	*TypeEntry;

if ((TypeEntry = Find(NewType)))
   { assert(FALSE);	/* already exists! */
    } /* fi */

if (Curr())
   { return(InsertBefore(NewType));	/* NewType inserted */
    } /* fi */
else
   { return(Append(NewType));		/* NewType appended */
    } /* esle */

} /* end of SIR_Types::Insert */


int SIR_Types::TypeCmp(		/* type comparision (like strcmp) */
	sir_type	*Type1,	/* (recursive for complex types!) */
	sir_type	*Type2,
	bool		IgnoreDirection /* = false */)
{
int		CmpVal;

if (Type1 == Type2)	/* check for identity first */
   { return(0);
    } /* fi */

if (0 != (CmpVal = Type1->Type - Type2->Type))	/* compare the contents */
   { return(CmpVal);
    } /* fi */
if (0 != (CmpVal = Type1->Const - Type2->Const))
   { return(CmpVal);
    } /* fi */
if (0 != (CmpVal = Type1->Volatile - Type2->Volatile))
   { return(CmpVal);
    } /* fi */
if (! IgnoreDirection)
   { if (0 != (CmpVal = Type1->Direction - Type2->Direction))
	{ return(CmpVal);
	 } /* fi */
    } /* fi */

switch(Type1->Type)	/* type-specific comparison */
   { case SIR_TYPE_BOOL:
     case SIR_TYPE_CHAR:
     case SIR_TYPE_UCHAR:
     case SIR_TYPE_SHORT:
     case SIR_TYPE_USHORT:
     case SIR_TYPE_INT:
     case SIR_TYPE_UINT:
     case SIR_TYPE_LONG:
     case SIR_TYPE_ULONG:
     case SIR_TYPE_LONGLONG:
     case SIR_TYPE_ULONGLONG:
     case SIR_TYPE_FLOAT:
     case SIR_TYPE_DOUBLE:
     case SIR_TYPE_LONGDOUBLE:
	{ /* no further comparison necessary */
	  break;
	 }
     case SIR_TYPE_BIT:
     case SIR_TYPE_UBIT:
	{ if (0 != (CmpVal = Type1->LeftBound - Type2->LeftBound))
	     { return(CmpVal);
	      } /* fi */
	  if (0 != (CmpVal = Type1->RightBound - Type2->RightBound))
	     { return(CmpVal);
	      } /* fi */
	  break;
	 }
     case SIR_TYPE_VOID:
     case SIR_TYPE_EVENT:
	{ /* no further comparison necessary */
	  break;
	 }
     case SIR_TYPE_POINTER:
	{ return(TypeCmp(Type1->SubType, Type2->SubType));	/* recursion */
	 }
     case SIR_TYPE_STRUCT:
     case SIR_TYPE_UNION:
     case SIR_TYPE_ENUM:
	{ if (Type1->UserType == Type2->UserType)	/* unique! */
	     { return(0);
	      } /* fi */
	  else
	     { return(1);	/* means: "continue the search" */
	      } /* fi */
	 }
     case SIR_TYPE_ARRAY:
	{ if (0 != (CmpVal = Type1->Size - Type2->Size))
	     { return(CmpVal);
	      } /* fi */
	  return(TypeCmp(Type1->SubType, Type2->SubType));	/* recursion */
	 }
     case SIR_TYPE_FUNCTION:
	{ if (0 != (CmpVal = TypeCmp(Type1->SubType, Type2->SubType)))
	     { return(CmpVal);
	      } /* fi */
	  return(SIR_TypePtrs::TypeCmp(Type1->Parameters,	/* indirect  */
					Type2->Parameters));	/* recursion */
	 }
     case SIR_TYPE_ANY_TYPE:
	{ /* no further comparison necessary */
	  break;
	 }
     case SIR_TYPE_BEHAVIOR:
     case SIR_TYPE_CHANNEL:
	{ if (0 != (CmpVal =				/* match ports */
		SIR_TypePtrs::TypeCmp(Type1->Parameters,	/* indirect  */
					Type2->Parameters)))	/* recursion */
	     { return(CmpVal);
	      } /* fi */
	  return(strcmp(Type1->Name, Type2->Name));	/* match names! */
	 }
     case SIR_TYPE_INTERFACE:
	{ /* no ports */
	  return(strcmp(Type1->Name, Type2->Name));
	 }
     default:
	{ assert(FALSE);	/* bad Type */
	 }
    } /* hctiws */

return(0);	/* Type1 is the same as Type2 */

} /* end of SIR_Types::TypeCmp */


	/*******************/
	/*** SIR_TypePtr ***/
	/*******************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_TypePtr::SIR_TypePtr(			/* constructor #1 */
	sir_type	*Type)
{

assert(Type != NULL);

SIR_TypePtr::Type	= Type;

} /* end of SIR_TypePtr::SIR_TypePtr #1 */


SIR_TypePtr::SIR_TypePtr(		/* constructor #3 */
	sir_type_ptr	*Original)
{

assert(Original != NULL);

SIR_TypePtr::Type	= Original->Type;

} /* end of SIR_TypePtr::SIR_TypePtr #3 */


SIR_TypePtr::~SIR_TypePtr(void)		/* destructor */
{

/* nothing to do */

} /* end of SIR_TypePtr::~SIR_TypePtr */


	/********************/
	/*** SIR_TypePtrs ***/
	/********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_TypePtrs::SIR_TypePtrs(		/* constructor #1 */
	sir_type_ptr	*FirstEntry /* = NULL */) :
		SIR_List<SIR_TypePtr>(FirstEntry)
{

/* nothing else to do */

} /* end of SIR_TypePtrs::SIR_TypePtrs #1 */


SIR_TypePtrs::SIR_TypePtrs(		/* constructor #2 (duplicator) */
	sir_type_ptrs	*Original)
{
sir_type_ptr	*TypePtr;

assert(Original != NULL);

TypePtr = Original->First();
while(TypePtr)
   { Append(new SIR_TypePtr(TypePtr));
     TypePtr = TypePtr->Succ();
    } /* elihw */

} /* end of SIR_TypePtrs::SIR_TypePtrs #2 */


SIR_TypePtrs::SIR_TypePtrs(		/* constructor #3 (from Parameters) */
	sir_parameters	*Parameters)
{
sir_parameter	*Parameter;

assert(Parameters != NULL);

Parameter = Parameters->First();
while(Parameter)
   { Append(new SIR_TypePtr(Parameter->Type));
     Parameter = Parameter->Succ();
    } /* elihw */

} /* end of SIR_TypePtrs::SIR_TypePtrs #3 */


SIR_TypePtrs::~SIR_TypePtrs(void)	/* destructor */
{

/* nothing to do */

} /* end of SIR_TypePtrs::~SIR_TypePtrs */


void SIR_TypePtrs::UnAlias(void) /* unalias all type, usertype, symbol links */
{
sir_type_ptr	*TypePtr;

TypePtr = First();
while(TypePtr)
   { if (  (TypePtr->Type)
	 &&(TypePtr->Type->Alias))
	{ TypePtr->Type = TypePtr->Type->Alias;
	 } /* fi */
     TypePtr = TypePtr->Succ();
    } /* elihw */

} /* end of SIR_TypePtrs::UnAlias */


int SIR_TypePtrs::TypeCmp(		/* type comparision (like strcmp) */
	sir_type_ptrs	*TypeList1,
	sir_type_ptrs	*TypeList2)
{
sir_type_ptr	*TypePtr1,
		*TypePtr2;
int		CmpVal;

assert(TypeList1 != NULL);
assert(TypeList2 != NULL);

TypePtr1 = TypeList1->First();
TypePtr2 = TypeList2->First();
while(TypePtr1)
   { if (!(TypePtr2))
	{ return(1);	/* TypeList1 has more entries */
	 } /* fi */
     if ((CmpVal = SIR_Types::TypeCmp(TypePtr1->Type, TypePtr2->Type)))
	{ return(CmpVal);
	 } /* fi */
     TypePtr1 = TypePtr1->Succ();
     TypePtr2 = TypePtr2->Succ();
    } /* elihw */
if (TypePtr2)
   { return(-1);	/* TypeList2 has more entries */
    } /* fi */

return(0);	/* equal! */

} /* end of SIR_TypePtrs::TypeCmp */


/* EOF Type.cc */
