/************************************************************************/
/* Statement.cc: SpecC Internal Representation, Statement Class		*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 10/20/97 */
/************************************************************************/

/* last update: 11/29/01 */

/* modifications: (most recent first)
 *
 * 11/29/01 RD	bugfix with long long to bit vector conversions
 * 11/29/01 RD	bugfix with signed vs. unsigned long long conversions
 * 11/21/01 RD	took out default arguments from function definitions
 * 11/08/01 RD	switched code generation to use GL_IO layer
 * 11/01/01 RD	added indentation in code generation for '(' and ')'
 * 10/11/01 RD	improved overall formatting of generated code
 * 10/04/01 RD	improved indentation of generated code
 * 08/14/01 RD	added missing port-access checking at statement creation
 * 05/31/01 RD	eliminated level 2 of SIR API
 * 05/27/01 RD	added semantic checking for event port accesses
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 05/21/01 RD	added support for (native, non-native) array assignment
 * 05/12/01 RD	bug fix: generate correct C++ code for empty 'par' and 'pipe'
 * 04/30/01 RD	replaced use of obsolete form() from "stream.h" with own one
 * 04/18/01 RD	adjusted 'long long' constants for non-native longlong
 * 04/05/01 RD	added support for new 'pipe' construct (with flushing)
 * 03/23/01 RD	added SIR_Statement::WrapInCompound() and
 *		SIR_Statement::CompoundSubBlocks()
 * 03/20/01 RD	extended pipe construct to support termination
 * 01/30/01 RD	fixed a potential FMR problem in iterators
 *		SIR_Statements::DFS_ForAllXxxxx
 */


#include "IntRep/Statement.h"
#include "IntRep/Design.h"
#include "IntRep/Extern.h"

#include <assert.h>


/*** constants and macros ***********************************************/


	/* macro for behavior main calls */

#define SIR_CALLED_BEHAVIOR(Stmnt)	\
	((Stmnt)->Expression1->Arg1->Arg1->Symbol)


	/* macro for 'relaxed' type conversions */

// note: any change to SIR_RELAXED must also be applied to Expression.cc!
// (see SIR_RELAXED in Expression.cc for more info)

#ifdef HAVE_LLONG

#define SIR_RELAXED(expected, given)				\
	( (  ((expected) == SIR_TYPE_BIT)			\
	   ||((expected) == SIR_TYPE_UBIT))			\
	 ? (SIR_TYPE_ANY_TYPE) : (expected))

#else /* !HAVE_LLONG */

#define SIR_RELAXED(expected, given)				\
	(  (  (  ((expected) == SIR_TYPE_BIT)			\
	       ||((expected) == SIR_TYPE_UBIT))			\
	    &&(  ((given) != SIR_TYPE_LONGLONG)			\
	       &&((given) != SIR_TYPE_ULONGLONG)))		\
	 ||(  ((expected) == SIR_TYPE_LONGLONG)			\
	    &&(  ((given) != SIR_TYPE_BIT)			\
	       &&((given) != SIR_TYPE_UBIT)			\
	       &&((given) != SIR_TYPE_ULONGLONG)))		\
	 ||(  ((expected) == SIR_TYPE_ULONGLONG)		\
	    &&(  ((given) != SIR_TYPE_BIT)			\
	       &&((given) != SIR_TYPE_UBIT)			\
	       &&((given) != SIR_TYPE_LONGLONG)))		\
	 ? (SIR_TYPE_ANY_TYPE) : (expected))

#endif /* HAVE_LLONG */


/*** internal type declarations *****************************************/


	/* (none) */


/*** class implementations **********************************************/


	/*********************/
	/*** SIR_Statement ***/
	/*********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Statement::SIR_Statement(		/* constructor #1 */
	SIR_STMNT_TYPE	StmntType,
	sir_symbol	*Function,
	sir_expression	*Expression1 /* = NULL */,
	sir_statement	*Statement1 /* = NULL */,
	sir_statement	*Statement2 /* = NULL */,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */,
	sir_expression	*Expression2 /* = NULL */,
	sir_expression	*Expression3 /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(Function != NULL);

switch(StmntType)
   { case SIR_STMNT_LABELED:
     case SIR_STMNT_COMPOUND:
	{ assert(FALSE);	/* wrong constructor */
	 }
     case SIR_STMNT_EXPRESSION:
	{ assert(Expression1 != NULL);
	  break;
	 }
     case SIR_STMNT_IF:
	{ assert(Expression1 != NULL);
	  assert(Statement1 != NULL);
	  break;
	 }
     case SIR_STMNT_IF_ELSE:
	{ assert(Expression1 != NULL);
	  assert(Statement1 != NULL);
	  assert(Statement2 != NULL);
	  break;
	 }
     case SIR_STMNT_SWITCH:
	{ assert(Expression1 != NULL);
	  assert(Statement1 != NULL);
	  break;
	 }
     case SIR_STMNT_CASE:
	{ assert(FALSE);	/* wrong constructor */
	 }
     case SIR_STMNT_DEFAULT:
	{ assert(Statement1 != NULL);
	  break;
	 }
     case SIR_STMNT_WHILE:
     case SIR_STMNT_DO_WHILE:
	{ assert(Expression1 != NULL);
	  assert(Statement1 != NULL);
	  break;
	 }
     case SIR_STMNT_FOR:
	{ assert(Statement1 != NULL);
	  assert(Expression1 != NULL);
	  assert(Expression2 != NULL);
	  assert(Expression3 != NULL);
	  break;
	 }
     case SIR_STMNT_GOTO:
	{ assert(FALSE);	/* wrong constructor */
	 }
     case SIR_STMNT_CONTINUE:
     case SIR_STMNT_BREAK:
	{ break;
	 }
     case SIR_STMNT_RETURN:
	{ assert(Expression1 != NULL);
	  break;
	 }
     case SIR_STMNT_PAR:
     case SIR_STMNT_PIPE:
	{ assert(Statement1 != NULL);
	  assert(  (  (Expression1 == NULL)
		    &&(Expression2 == NULL)
		    &&(Expression3 == NULL))
		 ||(  (Expression1 != NULL)
		    &&(Expression2 != NULL)
		    &&(Expression3 != NULL)));
	  break;
	 }
     case SIR_STMNT_EXCEPTION:
     case SIR_STMNT_TIMING:
     case SIR_STMNT_FSM:
     case SIR_STMNT_WAIT:
	{ assert(FALSE);	/* wrong constructor */
	 }
     case SIR_STMNT_WAITFOR:
	{ assert(Expression1 != NULL);
	  break;
	 }
     case SIR_STMNT_NOTIFY:
     case SIR_STMNT_NOTIFYONE:
	{ assert(FALSE);	/* wrong constructor */
	 }
     default:
	{ assert(FALSE);	/* bad StmntType */
	 }
    } /* hctiws */

SIR_Statement::StmntType	= StmntType;
SIR_Statement::Function		= Function;
SIR_Statement::Expression1	= Expression1;
SIR_Statement::Expression2	= Expression2;
SIR_Statement::Expression3	= Expression3;
SIR_Statement::Constant		= NULL;
SIR_Statement::Statement1	= Statement1;
SIR_Statement::Statement2	= Statement2;
SIR_Statement::Label		= NULL;
SIR_Statement::Scope		= NULL;
SIR_Statement::Statements	= NULL;
SIR_Statement::Events		= NULL;
SIR_Statement::Exceptions	= NULL;
SIR_Statement::Constraints	= NULL;
SIR_Statement::Transitions	= NULL;

} /* end of SIR_Statement::SIR_Statement #1 */


SIR_Statement::SIR_Statement(		/* constructor #2 */
	SIR_STMNT_TYPE	StmntType,	/* (for compound statements) */
	sir_symbol	*Function,
	sir_symbols	*Scope,
	sir_statements	*Statements,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(StmntType == SIR_STMNT_COMPOUND);
assert(Function != NULL);
assert(Scope != NULL);
assert(Scope->ParentSymbol != NULL);	/* we always want a parent function */
/* Statements is temorary NULL in Parser */

SIR_Statement::StmntType	= StmntType;
SIR_Statement::Function		= Function;
SIR_Statement::Expression1	= NULL;
SIR_Statement::Expression2	= NULL;
SIR_Statement::Expression3	= NULL;
SIR_Statement::Constant		= NULL;
SIR_Statement::Statement1	= NULL;
SIR_Statement::Statement2	= NULL;
SIR_Statement::Label		= NULL;
SIR_Statement::Scope		= Scope;
SIR_Statement::Statements	= Statements;
SIR_Statement::Events		= NULL;
SIR_Statement::Exceptions	= NULL;
SIR_Statement::Constraints	= NULL;
SIR_Statement::Transitions	= NULL;

Scope->ParentStmnt = this;	/* automatically fill in the back pointer */

if (Statements)
   { Statements->CmpndScope = Scope;	/* fill in compound scope link */
    } /* fi */

} /* end of SIR_Statement::SIR_Statement #2 */


SIR_Statement::SIR_Statement(		/* constructor #3 */
	SIR_STMNT_TYPE	StmntType,	/* (for case statements) */
	sir_symbol	*Function,
	sir_constant	*Constant,
	sir_statement	*Statement1 /* = NULL */,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(StmntType == SIR_STMNT_CASE);
assert(Function != NULL);
assert(Constant != NULL);

SIR_Statement::StmntType	= StmntType;
SIR_Statement::Function		= Function;
SIR_Statement::Expression1	= NULL;
SIR_Statement::Expression2	= NULL;
SIR_Statement::Expression3	= NULL;
SIR_Statement::Constant		= Constant;
SIR_Statement::Statement1	= Statement1;
SIR_Statement::Statement2	= NULL;
SIR_Statement::Label		= NULL;
SIR_Statement::Scope		= NULL;
SIR_Statement::Statements	= NULL;
SIR_Statement::Events		= NULL;
SIR_Statement::Exceptions	= NULL;
SIR_Statement::Constraints	= NULL;
SIR_Statement::Transitions	= NULL;

} /* end of SIR_Statement::SIR_Statement #3 */


SIR_Statement::SIR_Statement(		/* constructor #4 */
	SIR_STMNT_TYPE	StmntType,	/* (for statements with labels) */
	sir_symbol	*Function,
	sir_label	*Label,
	sir_statement	*Statement1 /* = NULL */,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(  (StmntType == SIR_STMNT_GOTO)
       ||(StmntType == SIR_STMNT_LABELED));
assert(Function != NULL);
assert(Label != NULL);

SIR_Statement::StmntType	= StmntType;
SIR_Statement::Function		= Function;
SIR_Statement::Expression1	= NULL;
SIR_Statement::Expression2	= NULL;
SIR_Statement::Expression3	= NULL;
SIR_Statement::Constant		= NULL;
SIR_Statement::Statement1	= Statement1;
SIR_Statement::Statement2	= NULL;
SIR_Statement::Label		= Label;
SIR_Statement::Scope		= NULL;
SIR_Statement::Statements	= NULL;
SIR_Statement::Events		= NULL;
SIR_Statement::Exceptions	= NULL;
SIR_Statement::Constraints	= NULL;
SIR_Statement::Transitions	= NULL;

} /* end of SIR_Statement::SIR_Statement #4 */


SIR_Statement::SIR_Statement(		/* constructor #5 */
	SIR_STMNT_TYPE	StmntType,	/* (for statements with events) */
	sir_symbol	*Function,
	sir_symbol_ptrs	*Events,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(  (StmntType == SIR_STMNT_WAIT)
       ||(StmntType == SIR_STMNT_NOTIFY)
       ||(StmntType == SIR_STMNT_NOTIFYONE));
assert(Function != NULL);
assert(Events != NULL);

SIR_Statement::StmntType	= StmntType;
SIR_Statement::Function		= Function;
SIR_Statement::Expression1	= NULL;
SIR_Statement::Expression2	= NULL;
SIR_Statement::Expression3	= NULL;
SIR_Statement::Constant		= NULL;
SIR_Statement::Statement1	= NULL;
SIR_Statement::Statement2	= NULL;
SIR_Statement::Label		= NULL;
SIR_Statement::Scope		= NULL;
SIR_Statement::Statements	= NULL;
SIR_Statement::Events		= Events;
SIR_Statement::Exceptions	= NULL;
SIR_Statement::Constraints	= NULL;
SIR_Statement::Transitions	= NULL;

} /* end of SIR_Statement::SIR_Statement #5 */


SIR_Statement::SIR_Statement(		/* constructor #6 */
	SIR_STMNT_TYPE	StmntType,	/* (for exception statements) */
	sir_symbol	*Function,
	sir_statement	*Statement1,
	sir_exceptions	*Exceptions,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(StmntType == SIR_STMNT_EXCEPTION);
assert(Function != NULL);
assert(Statement1 != NULL);
assert(Exceptions != NULL);

SIR_Statement::StmntType	= StmntType;
SIR_Statement::Function		= Function;
SIR_Statement::Expression1	= NULL;
SIR_Statement::Expression2	= NULL;
SIR_Statement::Expression3	= NULL;
SIR_Statement::Constant		= NULL;
SIR_Statement::Statement1	= Statement1;
SIR_Statement::Statement2	= NULL;
SIR_Statement::Label		= NULL;
SIR_Statement::Scope		= NULL;
SIR_Statement::Statements	= NULL;
SIR_Statement::Events		= NULL;
SIR_Statement::Exceptions	= Exceptions;
SIR_Statement::Constraints	= NULL;
SIR_Statement::Transitions	= NULL;

} /* end of SIR_Statement::SIR_Statement #6 */


SIR_Statement::SIR_Statement(		/* constructor #7 */
	SIR_STMNT_TYPE	StmntType,	/* (for timing statement) */
	sir_symbol	*Function,
	sir_statement	*Statement1,
	sir_constraints	*Constraints,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(StmntType == SIR_STMNT_TIMING);
assert(Function != NULL);
assert(Statement1 != NULL);
assert(Constraints != NULL);

SIR_Statement::StmntType	= StmntType;
SIR_Statement::Function		= Function;
SIR_Statement::Expression1	= NULL;
SIR_Statement::Expression2	= NULL;
SIR_Statement::Expression3	= NULL;
SIR_Statement::Constant		= NULL;
SIR_Statement::Statement1	= Statement1;
SIR_Statement::Statement2	= NULL;
SIR_Statement::Label		= NULL;
SIR_Statement::Scope		= NULL;
SIR_Statement::Statements	= NULL;
SIR_Statement::Events		= NULL;
SIR_Statement::Exceptions	= NULL;
SIR_Statement::Constraints	= Constraints;
SIR_Statement::Transitions	= NULL;

} /* end of SIR_Statement::SIR_Statement #7 */


SIR_Statement::SIR_Statement(		/* constructor #8 */
	SIR_STMNT_TYPE	StmntType,	/* (for fsm statement) */
	sir_symbol	*Function,
	sir_transitions	*Transitions,
	unsigned int	Line /* = 0 */,
	sir_fileinfo	*FileInfo /* = NULL */) :
		SIR_Node(Line, FileInfo)
{

assert(StmntType == SIR_STMNT_FSM);
assert(Function != NULL);
assert(Transitions != NULL);

SIR_Statement::StmntType	= StmntType;
SIR_Statement::Function		= Function;
SIR_Statement::Expression1	= NULL;
SIR_Statement::Expression2	= NULL;
SIR_Statement::Expression3	= NULL;
SIR_Statement::Constant		= NULL;
SIR_Statement::Statement1	= NULL;
SIR_Statement::Statement2	= NULL;
SIR_Statement::Label		= NULL;
SIR_Statement::Scope		= NULL;
SIR_Statement::Statements	= NULL;
SIR_Statement::Events		= NULL;
SIR_Statement::Exceptions	= NULL;
SIR_Statement::Constraints	= NULL;
SIR_Statement::Transitions	= Transitions;

} /* end of SIR_Statement::SIR_Statement #8 */


SIR_Statement::SIR_Statement(		/* constructor #10 (duplicator) */
	sir_statement	*Original) :
		SIR_Node(Original)
{

SIR_Statement::StmntType	= Original->StmntType;
SIR_Statement::Function		= Original->Function;
SIR_Statement::Expression1	= Original->Expression1 ?
					new SIR_Expression(
						Original->Expression1) :
					NULL;
SIR_Statement::Expression2	= Original->Expression2 ?
					new SIR_Expression(
						Original->Expression2) :
					NULL;
SIR_Statement::Expression3	= Original->Expression3 ?
					new SIR_Expression(
						Original->Expression3) :
					NULL;
SIR_Statement::Constant		= Original->Constant ?
					new SIR_Constant(
						Original->Constant) :
					NULL;
SIR_Statement::Statement1	= Original->Statement1 ?
					new SIR_Statement(
						Original->Statement1) :
					NULL;
SIR_Statement::Statement2	= Original->Statement2 ?
					new SIR_Statement(
						Original->Statement2) :
					NULL;
SIR_Statement::Label		= Original->Label;
SIR_Statement::Scope		= Original->Scope ?
					new SIR_Symbols(
						Original->Scope) :
					NULL;
SIR_Statement::Statements	= Original->Statements ?
					new SIR_Statements(
						Original->Statements) :
					NULL;
SIR_Statement::Events		= Original->Events ?
					new SIR_SymbolPtrs(
						Original->Events) :
					NULL;
SIR_Statement::Exceptions	= Original->Exceptions ?
					new SIR_Exceptions(
						Original->Exceptions) :
					NULL;
SIR_Statement::Constraints	= Original->Constraints ?
					new SIR_Constraints(
						Original->Constraints) :
					NULL;
SIR_Statement::Transitions	= Original->Transitions ?
					new SIR_Transitions(
						Original->Transitions) :
					NULL;

} /* end of SIR_Statement::SIR_Statement #10 */


SIR_Statement::~SIR_Statement(void)	/* destructor */
{

delete Expression1;
delete Expression2;
delete Expression3;
delete Constant;
delete Statement1;
delete Statement2;
delete Scope;
delete Statements;
delete Events;
delete Exceptions;
delete Constraints;
delete Transitions;

} /* end of SIR_Statement::~SIR_Statement */


BOOL SIR_Statement::IsCleanBehaviorCall( /* checks for "clean" behavior call */
	void)					/* (e.g. "b1.main();") */
{
BOOL	Test;

Test = (  (StmntType == SIR_STMNT_EXPRESSION)
	&&(Expression1->ExprType == SIR_EXPR_FUNCTION_CALL)
	&&(Expression1->Arg1->ExprType == SIR_EXPR_MEMBER_ACCESS)
	&&(Expression1->Arg1->Arg1->Type->Type == SIR_TYPE_BEHAVIOR)
	&&(Expression1->Arg1->Arg1->ExprType == SIR_EXPR_IDENTIFIER)
	&&(Expression1->Arg1->Symbol == NULL)	/* means: "main"! */
	&&(Expression1->Args->Empty()));

if (Test)
   { assert(SIR_CALLED_BEHAVIOR(this) != NULL);
    } /* fi */

return(Test);

} /* end of SIR_Statement::IsCleanBehaviorCall */


BOOL SIR_Statement::IsCleanListOfBehaviorCalls(	/* check "clean" comp. bhvr. */
	BOOL	OneCallMax /* = FALSE */,     /* "{ b1.main(); b2.main(); }" */
	BOOL	OneCallMin /* = TRUE */)
{
sir_statement	*Stmnt;

if (StmntType != SIR_STMNT_COMPOUND)
   { return(FALSE);
    } /* fi */

assert(Scope != NULL);
assert(Statements != NULL);

if (  (! Scope->Empty())
    ||(! Scope->UserTypes->Empty()))
   { return(FALSE);
    } /* fi */

if (  (OneCallMin)
    &&(Statements->NumElements() < 1))
   { return(FALSE);
    } /* fi */
if (  (OneCallMax)
    &&(Statements->NumElements() > 1))
   { return(FALSE);
    } /* fi */

Stmnt = Statements->First();
while(Stmnt)
   { if (! Stmnt->IsCleanBehaviorCall())
	{ return(FALSE);
	 } /* fi */
     Stmnt = Stmnt->Succ();
    } /* elihw */

return(TRUE);	/* looks clean */

} /* end of SIR_Statement::IsCleanListOfBehaviorCalls */


ERROR SIR_Statement::CheckCompoundBehavior( /* checks for "clean" beh. calls */
	BOOL	OneCallMax /* = FALSE */,   /* "{ b1.main(); b2.main(); }" */
	BOOL	OneCallMin /* = TRUE */)    /* (returns error if not clean) */
{
string TmpString;

assert(StmntType == SIR_STMNT_COMPOUND);

if (! IsCleanListOfBehaviorCalls(OneCallMax, OneCallMin))
   { SIR_ErrMsg = "Behavioral hierarchy not clean in compound statement;"
							GL_ERROR_MSG_NEWLINE
			"unable to generate simulation model";
     if (LineInfo)
	{ TmpString.form(GL_ERROR_MSG_NEWLINE
			"(line %d, file \"%s\")",
			LineInfo->Line, LineInfo->File->Filename.chars());
	  SIR_ErrMsg += TmpString;
	 } /* fi */
     return(SIR_ERROR_COMPOUND_STMNT_NOT_CLEAN);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::CheckCompoundBehavior */


BOOL SIR_Statement::IsDirectDependant(	/* checks if stmnt. depends on symbol */
	sir_symbol	*ThatSymbol)	/* (directly) */
{
sir_symbol_ptr	*Event;
sir_exception	*Except;
sir_transition	*Trans;

/* Function is a _back_ link, so ignore it */

if (Events)
   { Event = Events->First();
     while(Event)
	{ if (Event->Symbol == ThatSymbol)
	     { return(TRUE);		/* references that event */
	      } /* fi */
	  Event = Event->Succ();
	 } /* elihw */
    } /* fi */

if (Exceptions)
   { Except = Exceptions->First();
     while(Except)
	{ assert(Except->Events != NULL);
	  Event = Except->Events->First();
	  while(Event)
	     { if (Event->Symbol == ThatSymbol)
		  { return(TRUE);	/* references that event */
		   } /* fi */
	       Event = Event->Succ();
	      } /* elihw */
	  Except = Except->Succ();
	 } /* elihw */
    } /* fi */

if (Transitions)
   { Trans = Transitions->First();
     while(Trans)
	{ if (  (Trans->CurrState == ThatSymbol)
	      ||(Trans->NextState == ThatSymbol))
	     { return(TRUE);		/* references that state */
	      } /* fi */
	  Trans = Trans->Succ();
	 } /* elihw */
    } /* fi */

return(FALSE);	/* independent! */

} /* end of SIR_Statement::IsDirectDependant */


BOOL SIR_Statement::IsIndirectDependant(/* checks if stmnt. depends on symbol */
	sir_symbol	*ThatSymbol,	/* (indirectly) */
	sir_expression	**DepExpr /* = NULL */)
{
sir_transition	*Trans;

if (Expression1)
   { if (Expression1->DFS_FindDependant(ThatSymbol, DepExpr))
	{ return(TRUE);
	 } /* fi */
    } /* fi */
if (Expression2)
   { if (Expression2->DFS_FindDependant(ThatSymbol, DepExpr))
	{ return(TRUE);
	 } /* fi */
    } /* fi */
if (Expression3)
   { if (Expression3->DFS_FindDependant(ThatSymbol, DepExpr))
	{ return(TRUE);
	 } /* fi */
    } /* fi */
if (Transitions)
   { Trans = Transitions->First();
     while(Trans)
	{ if (Trans->Condition)
	     { if (Trans->Condition->DFS_FindDependant(ThatSymbol, DepExpr))
		  { return(TRUE);
		   } /* fi */
	      } /* fi */
	  Trans = Trans->Succ();
	 } /* elihw */
    } /* fi */

return(FALSE);	/* independent! */

} /* end of SIR_Statement::IsIndirectDependant */


sir_statements *SIR_Statement::GetList(	/* determines the list of this stmnt. */
	void)
{

return((sir_statements*) Head());

} /* end of SIR_Statement::GetList */


ERROR SIR_Statement::DFS_ForAllNodes(	/* iterator over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{

if ((SIR_Error = (this->*MemberFct)(MemberFctArg)))	/* process this node */
   { return(SIR_Error);
    } /* fi */

if (Expression1)
   { if ((SIR_Error = Expression1->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Expression2)
   { if ((SIR_Error = Expression2->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Expression3)
   { if ((SIR_Error = Expression3->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Constant)
   { if ((SIR_Error = Constant->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statement1)
   { if ((SIR_Error = Statement1->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statement2)
   { if ((SIR_Error = Statement2->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Scope)
   { if ((SIR_Error = Scope->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statements)
   { if ((SIR_Error = Statements->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Exceptions)
   { if ((SIR_Error = Exceptions->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Constraints)
   { if ((SIR_Error = Constraints->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Transitions)
   { if ((SIR_Error = Transitions->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::DFS_ForAllNodes */


ERROR SIR_Statement::DFS_ForAllSymbols(	/* iterator over all symbols */
	sir_symbol_mptr	MemberFct,	/* (depth first) */
	sir_symbol_marg	MemberFctArg)
{

/* this is not a symbol */

/* there are no symbols in Expression1 */
/* there are no symbols in Expression2 */
/* there are no symbols in Expression3 */
/* there are no symbols in Constant */
if (Statement1)
   { if ((SIR_Error = Statement1->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statement2)
   { if ((SIR_Error = Statement2->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Scope)
   { if ((SIR_Error = Scope->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statements)
   { if ((SIR_Error = Statements->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Exceptions)
   { if ((SIR_Error = Exceptions->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no symbols in Constraints */
/* there are no symbols in Transitions */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::DFS_ForAllSymbols */


ERROR SIR_Statement::DFS_ForAllUserTypes( /* iterator over all usertypes */
	sir_usertp_mptr	MemberFct,	/* (depth first) */
	sir_usertp_marg	MemberFctArg)
{

/* this is not a usertype */

/* there are no usertypes in Expression1 */
/* there are no usertypes in Expression2 */
/* there are no usertypes in Expression3 */
/* there are no usertypes in Constant */
if (Statement1)
   { if ((SIR_Error = Statement1->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statement2)
   { if ((SIR_Error = Statement2->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Scope)
   { if ((SIR_Error = Scope->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statements)
   { if ((SIR_Error = Statements->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Exceptions)
   { if ((SIR_Error = Exceptions->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no usertypes in Constraints */
/* there are no usertypes in Transitions */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::DFS_ForAllUserTypes */


ERROR SIR_Statement::DFS_ForAllNotes( /* iterator over all notes */
	sir_note_mptr	MemberFct,	/* (depth first) */
	sir_note_marg	MemberFctArg)
{

/* this is not a note */

/* there are no notes in Expression1 */
/* there are no notes in Expression2 */
/* there are no notes in Expression3 */
/* there are no notes in Constant */
if (Statement1)
   { if ((SIR_Error = Statement1->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statement2)
   { if ((SIR_Error = Statement2->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Scope)
   { if ((SIR_Error = Scope->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statements)
   { if ((SIR_Error = Statements->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Exceptions)
   { if ((SIR_Error = Exceptions->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no notes in Constraints */
/* there are no notes in Transitions */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::DFS_ForAllNotes */


ERROR SIR_Statement::DFS_ForAllStatements( /* iterator over all statements */
	sir_stmnt_mptr	MemberFct,	/* (depth first) */
	sir_stmnt_marg	MemberFctArg)
{

if ((SIR_Error = (this->*MemberFct)(MemberFctArg)))	/* process this node */
   { return(SIR_Error);
    } /* fi */

/* there are no statements in Expression1 */
/* there are no statements in Expression2 */
/* there are no statements in Expression3 */
/* there are no statements in Constant */
if (Statement1)
   { if ((SIR_Error = Statement1->DFS_ForAllStatements(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statement2)
   { if ((SIR_Error = Statement2->DFS_ForAllStatements(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no statements in Scope */
if (Statements)
   { if ((SIR_Error = Statements->DFS_ForAllStatements(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Exceptions)
   { if ((SIR_Error = Exceptions->DFS_ForAllStatements(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no statements in Constraints */
/* there are no statements in Transitions */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::DFS_ForAllStatements */


ERROR SIR_Statement::DFS_ForAllExpressions( /* iterator over all expressions */
	sir_expr_mptr	MemberFct,		/* (depth first) */
	sir_expr_marg	MemberFctArg)
{

/* this is not an expression */

if (Expression1)
   { if ((SIR_Error = Expression1->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Expression2)
   { if ((SIR_Error = Expression2->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Expression3)
   { if ((SIR_Error = Expression3->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no expressions in Constant */
if (Statement1)
   { if ((SIR_Error = Statement1->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statement2)
   { if ((SIR_Error = Statement2->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no expressions in Scope */
if (Statements)
   { if ((SIR_Error = Statements->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Exceptions)
   { if ((SIR_Error = Exceptions->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no expressions in Constraints */
if (Transitions)
   { if ((SIR_Error = Transitions->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::DFS_ForAllExpressions */


BOOL SIR_Statement::DFS_FindDependant(	/* searches for dependants (DFS) */
	sir_symbol	*ThatSymbol,
	sir_statement	**DepStmnt,
	sir_expression	**DepExpr,
	SIR_DEPENDENCY	*Reason)
{
sir_exception	*Except;

assert(ThatSymbol != NULL);

if (IsDirectDependant(ThatSymbol))	/* check this statement */
   { if (DepStmnt)			/* (for direct dep.) */
	{ *DepStmnt = this;
	 } /* fi */
     if (DepExpr)
	{ *DepExpr = NULL;
	 } /* fi */
     if (Reason)
	{ *Reason = SIR_DEP_STATEMENT;
	 } /* fi */
     return(TRUE);
    } /* fi */

if (IsIndirectDependant(ThatSymbol, DepExpr))	/* check this statement */
   { if (DepStmnt)				/* (for indirect dep.) */
	{ *DepStmnt = this;
	 } /* fi */
     if (Reason)
	{ *Reason = SIR_DEP_EXPRESSION;
	 } /* fi */
     return(TRUE);
    } /* fi */

/* there are no statements in Expression1 */
/* there are no statements in Expression2 */
/* there are no statements in Expression3 */
/* there are no statements in Constant */
if (Statement1)
   { if (Statement1->DFS_FindDependant(ThatSymbol, DepStmnt, DepExpr, Reason))
	{ return(TRUE);
	 } /* fi */
    } /* fi */
if (Statement2)
   { if (Statement2->DFS_FindDependant(ThatSymbol, DepStmnt, DepExpr, Reason))
	{ return(TRUE);
	 } /* fi */
    } /* fi */
/* there are no statements in Scope */
if (Statements)
   { if (Statements->DFS_FindDependant(ThatSymbol, DepStmnt, DepExpr, Reason))
	{ return(TRUE);
	 } /* fi */
    } /* fi */
if (Exceptions)
   { Except = Exceptions->First();
     while(Except)
	{ assert(Except->Handler != NULL);
	  if (Except->Handler->DFS_FindDependant(ThatSymbol, DepStmnt,
							DepExpr, Reason))
	     { return(TRUE);
	      } /* fi */
	  Except = Except->Succ();
	 } /* elihw */
    } /* fi */
/* there are no statements in Constraints */
/* there are no statements in Transitions */

return(FALSE);

} /* end of SIR_Statement::DFS_FindDependant */


ERROR SIR_Statement::MarkYourselfAndYourLabel(	/* iterator for color marking */
	sir_stmnt_marg	ColorArg)
{
SIR_COLOR	MyColor;

MyColor = *((SIR_COLOR*) ColorArg);

Color = MyColor;		/* mark yourself */
if (StmntType == SIR_STMNT_LABELED)
   { assert(Label != NULL);
     Label->Color = MyColor;	/* mark the label */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::MarkYourselfAndYourLabel */


ERROR SIR_Statement::CheckRequiredLabel(	/* iterator for label check */
	sir_stmnt_marg	/* Unused */)
{
sir_constraint	*Constr;

if (StmntType == SIR_STMNT_GOTO)
   { assert(Label != NULL);
     if (Label->Color == SIR_RED)
	{ SIR_ErrMsg.form("Label '%s' is required",
				Label->LabelName.chars());
	  return(SIR_ERROR_LABEL_REQUIRED);
	 } /* fi */
    } /* fi */
if (StmntType == SIR_STMNT_TIMING)
   { Constr = Constraints->First();
     while(Constr)
	{ assert(Constr->Label1 != NULL);
	  if (Constr->Label1->Color == SIR_RED)
	     { SIR_ErrMsg.form("Label '%s' is required",
					Constr->Label1->LabelName.chars());
	       return(SIR_ERROR_LABEL_REQUIRED);
	      } /* fi */
	  assert(Constr->Label2 != NULL);
	  if (Constr->Label2->Color == SIR_RED)
	     { SIR_ErrMsg.form("Label '%s' is required",
					Constr->Label2->LabelName.chars());
	       return(SIR_ERROR_LABEL_REQUIRED);
	      } /* fi */
	  Constr = Constr->Succ();
	 } /* elihw */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::CheckRequiredLabel */


ERROR SIR_Statement::CheckLabeledStmnt(	/* iterator for label check */
	sir_stmnt_marg	/* Unused */)
{

if (StmntType == SIR_STMNT_LABELED)
   { assert(Label != NULL);
     SIR_ErrMsg.form("Double definition of label '%s'",
				Label->LabelName.chars());
     return(SIR_ERROR_DOUBLE_DEFINITION_OF_LABEL);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::CheckLabeledStmnt */


ERROR SIR_Statement::CheckReturnType(	/* iterator for return type checking */
	sir_stmnt_marg	TypeArg)
{
sir_type	*ReturnType;

ReturnType = (sir_type*) TypeArg;

if (StmntType == SIR_STMNT_RETURN)
   { if (!(Expression1->Type->IsConvertableTo(ReturnType)))
	{ return(SIR_ERROR_RETURN_TYPE_MISMATCH);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::CheckReturnType */


ERROR SIR_Statement::NeedsRangeChecker(	/* iterator for range checking */
	sir_stmnt_marg	/* Unused */)
{

if (  (StmntType == SIR_STMNT_TIMING)
    &&(!(Constraints->Empty())))
   { return(SIR_ERROR_FLAG_YES);
    } /* fi */

return(SIR_ERROR_FLAG_NO);

} /* end of SIR_Statement::NeedsRangeChecker */


void SIR_Statement::SetAlias(	/* sets all type, usertype, symbol alias' */
	sir_statement	*Alias)	/* (iterates over symbols and usertypes) */
{

assert(Alias != NULL);

/* no need to process this node */

/* there are no symbols or usertypes in Expression1 */
/* there are no symbols or usertypes in Expression2 */
/* there are no symbols or usertypes in Expression3 */
/* there are no symbols or usertypes in Constant */
if (Statement1)
   { Statement1->SetAlias(Alias->Statement1);
    } /* fi */
if (Statement2)
   { Statement2->SetAlias(Alias->Statement2);
    } /* fi */
if (Scope)
   { Scope->SetAlias(Alias->Scope);
    } /* fi */
if (Statements)
   { Statements->SetAlias(Alias->Statements);
    } /* fi */
if (Exceptions)
   { Exceptions->SetAlias(Alias->Exceptions);
    } /* fi */
/* there are no symbols or usertypes in Constraints */
/* there are no symbols or usertypes in Transitions */

} /* end of SIR_Statement::SetAlias */


void SIR_Statement::UnAlias(	/* unalias all type, usertype, symbol links */
	sir_symbols	*GlobalSymbols)
{
sir_statement	*Stmnt;

if (  (Function)
    &&(Function->Alias))
   { Function = Function->Alias;
    } /* fi */
if (Expression1)
   { Expression1->UnAlias();
    } /* fi */
if (Expression2)
   { Expression2->UnAlias();
    } /* fi */
if (Expression3)
   { Expression3->UnAlias();
    } /* fi */
/* there is nothing to unalias in Constant */
if (Statement1)
   { Statement1->UnAlias(GlobalSymbols);
    } /* fi */
if (Statement2)
   { Statement2->UnAlias(GlobalSymbols);
    } /* fi */
if (Scope)
   { Scope->UnAlias(GlobalSymbols);
    } /* fi */
if (Statements)
   { Stmnt = Statements->First();
     while(Stmnt)
	{ Stmnt->UnAlias(GlobalSymbols);
	  Stmnt = Stmnt->Succ();
	 } /* elihw */
    } /* fi */
if (Events)
   { Events->UnAlias();
    } /* fi */
if (Exceptions)
   { Exceptions->UnAlias(GlobalSymbols);
    } /* fi */
/* there is nothing to unalias in Constraints */
if (Transitions)
   { Transitions->UnAlias();
    } /* fi */

} /* end of SIR_Statement::UnAlias */


ERROR SIR_Statement::WritePipeStmnt(	/* generates C++ code for 'pipe' */
	GL_HANDLE	*IO,
	BOOL		WriteNotes)
{
sir_statement	*Stmnt;
sir_symbol	*PipedSymbol;
sir_symbols	*PipedScope;

if (Expression1)	/* terminating (new) version */
   { assert(Expression2 != NULL);
     assert(Expression3 != NULL);
     IO->PutS("unsigned int " SIR_CXX_FIRST_NAME "=1, "
		SIR_CXX_LAST_NAME "=1;");
     SIR_LineInfo::WriteSPACE(IO);
     IO->PutS("for(");
     IO->TabStepUp();
     if ((SIR_Error = Expression1->WriteSC(IO, WriteNotes, true)))
	{ return(SIR_Error);
	 } /* fi */
     IO->PutS("; ");
     if ((SIR_Error = Expression2->WriteSC(IO, WriteNotes, true,
							SIR_TYPE_BOOL)))
	{ return(SIR_Error);
	 } /* fi */
     IO->PutS("; ");
     if ((SIR_Error = Expression3->WriteSC(IO, WriteNotes, true)))
	{ return(SIR_Error);
	 } /* fi */
   //IO->TabStepDown();
     IO->PutC(')');
   //IO->TabStepUp();
     SIR_LineInfo::WriteSPACE(IO, true); // IO->PutC('{');
     IO->TabStepUp();
     IO->PrintF(SIR_SIM_PIPE_NAME "(%u, "
			SIR_CXX_FIRST_NAME ", "
			SIR_CXX_LAST_NAME ",",
		Statement1->Statements->NumElements());
     SIR_LineInfo::WriteSPACE(IO);
     Stmnt = Statement1->Statements->First();	/* threads */
     while(Stmnt)
	{ assert(SIR_CALLED_BEHAVIOR(Stmnt) != NULL);
	  if (Stmnt->LineInfo)
	     { if ((SIR_Error = Stmnt->LineInfo->WriteSC(IO)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  else
	     { SIR_LineInfo::WriteNewSC(IO, true);
	      } /* esle */
	  IO->PutS("&" SIR_CXX_FORK_VARIABLE);
	  IO->PutS(SIR_CALLED_BEHAVIOR(Stmnt)->Name);
	  IO->PutC(',');
	  SIR_LineInfo::WriteSPACE(IO);
	  Stmnt = Stmnt->Succ();
	 } /* elihw */
     PipedScope = Statement1->Scope->Parent;	/* list of piped symbols */
     while(  (PipedScope)			/* in scope		 */
	   &&(PipedScope->ScopeInfo != SIR_SCOPE_GLOBAL))
	{ PipedSymbol = PipedScope->First();
	  while(PipedSymbol)
	     { if (PipedSymbol->StorageClass == SIR_STORAGE_PIPED)
		  { IO->PutC('&');
		    IO->PutS(PipedSymbol->Name);
		    IO->PutC(',');
		    SIR_LineInfo::WriteSPACE(IO);
		   } /* fi */
	       PipedSymbol = PipedSymbol->Succ();
	      } /* elihw */
	  PipedScope = PipedScope->Parent;
	 } /* elihw */
     IO->PutS("0);");
     IO->TabStepDown();
     SIR_LineInfo::WriteSPACE(IO);
     IO->PrintF("if (" SIR_CXX_LAST_NAME "<%u)",
		Statement1->Statements->NumElements());
     IO->TabStepUp();
     SIR_LineInfo::WriteSPACE(IO);
     IO->PutS(SIR_CXX_LAST_NAME "++;");
     IO->TabStepDown();
     SIR_LineInfo::WriteNL(IO, false, true); // IO->PutC('}');
     IO->TabStepDown();
     SIR_LineInfo::WriteSPACE(IO);
     IO->PutS("while (" SIR_CXX_FIRST_NAME "++<" SIR_CXX_LAST_NAME ")");
     IO->TabStepUp();
     SIR_LineInfo::WriteSPACE(IO, true); // IO->PutC('{');
     IO->TabStepUp();
     IO->PrintF(SIR_SIM_PIPE_NAME "(%u, "
				SIR_CXX_FIRST_NAME ", "
				SIR_CXX_LAST_NAME ",",
		Statement1->Statements->NumElements());
     SIR_LineInfo::WriteSPACE(IO);
     Stmnt = Statement1->Statements->First();	/* threads */
     while(Stmnt)
	{ assert(SIR_CALLED_BEHAVIOR(Stmnt) != NULL);
	  IO->PutS("&" SIR_CXX_FORK_VARIABLE);
	  IO->PutS(SIR_CALLED_BEHAVIOR(Stmnt)->Name);
	  IO->PutC(',');
	  SIR_LineInfo::WriteSPACE(IO);
	  Stmnt = Stmnt->Succ();
	 } /* elihw */
     PipedScope = Statement1->Scope->Parent;	/* list of piped symbols */
     while(  (PipedScope)			/* in scope		 */
	   &&(PipedScope->ScopeInfo != SIR_SCOPE_GLOBAL))
	{ PipedSymbol = PipedScope->First();
	  while(PipedSymbol)
	     { if (PipedSymbol->StorageClass == SIR_STORAGE_PIPED)
		  { IO->PutC('&');
		    IO->PutS(PipedSymbol->Name);
		    IO->PutC(',');
		    SIR_LineInfo::WriteSPACE(IO);
		   } /* fi */
	       PipedSymbol = PipedSymbol->Succ();
	      } /* elihw */
	  PipedScope = PipedScope->Parent;
	 } /* elihw */
     IO->PutS("0);");
     IO->TabStepDown();
     SIR_LineInfo::WriteSPACE(IO);
     IO->PrintF("if (" SIR_CXX_LAST_NAME "<%u)",
		Statement1->Statements->NumElements());
     IO->TabStepUp();
     SIR_LineInfo::WriteSPACE(IO);
     IO->PutS(SIR_CXX_LAST_NAME "++;");
     IO->TabStepDown();
     SIR_LineInfo::WriteNL(IO, false, true); // IO->PutC('}');
     IO->TabStepDown();
    } /* fi */
else			/* non-terminating (old) version */
   { assert(Expression2 == NULL);
     assert(Expression3 == NULL);
     IO->TabStepUp();
     IO->PutS(SIR_SIM_PIPE_NAME "(");
     Stmnt = Statement1->Statements->First();
     while(Stmnt)
	{ assert(SIR_CALLED_BEHAVIOR(Stmnt) != NULL);
	  if (Stmnt->LineInfo)
	     { if ((SIR_Error = Stmnt->LineInfo->WriteSC(IO)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  else
	     { SIR_LineInfo::WriteNewSC(IO);
	      } /* esle */
	  IO->PutS("&" SIR_CXX_FORK_VARIABLE);
	  IO->PutS(SIR_CALLED_BEHAVIOR(Stmnt)->Name);
	  IO->PutC(',');
	  SIR_LineInfo::WriteSPACE(IO);
	  Stmnt = Stmnt->Succ();
	 } /* elihw */
     IO->PutS("0,");			/* list of piped symbols in scope */
     SIR_LineInfo::WriteSPACE(IO);
     PipedScope = Statement1->Scope->Parent;
     while(  (PipedScope)
	   &&(PipedScope->ScopeInfo != SIR_SCOPE_GLOBAL))
	{ PipedSymbol = PipedScope->First();
	  while(PipedSymbol)
	     { if (PipedSymbol->StorageClass == SIR_STORAGE_PIPED)
		  { IO->PutC('&');
		    IO->PutS(PipedSymbol->Name);
		    IO->PutC(',');
		    SIR_LineInfo::WriteSPACE(IO);
		   } /* fi */
	       PipedSymbol = PipedSymbol->Succ();
	      } /* elihw */
	  PipedScope = PipedScope->Parent;
	 } /* elihw */
     IO->PutS("0);");
     IO->TabStepDown();
    } /* esle */

return(SIR_ERROR_NO_ERROR);

} /* end of WritePipeStmnt */


ERROR SIR_Statement::WriteSC(	/* (re-) generates SpecC source code */
	GL_HANDLE	*IO,
	BOOL		WriteNotes,
	BOOL		CplusplusMode /* = FALSE */,
	BOOL		PutNewLine /* = TRUE */,
	BOOL		PutArgNotes /* = TRUE */)
{
sir_statement	*Stmnt;
sir_symbol_ptr	*Event;
sir_exception	*Except;
sir_constraint	*Constr;
sir_transition	*Trans;
sir_symbol	*ParamSymbol;
sir_constant	*TmpConst;
int		ExceptNum;
int		LabelNum;
string		TmpString;
BOOL		PutBraces;

PutBraces = (StmntType == SIR_STMNT_COMPOUND);

if (LineInfo)
   { if ((SIR_Error = LineInfo->WriteSC(IO, true, PutBraces)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
else
   { if (PutNewLine)
	{ SIR_LineInfo::WriteNewSC(IO, true, PutBraces);
	 } /* fi */
     else
	{ SIR_LineInfo::WriteSPACE(IO, PutBraces);
	 } /* fi */
    } /* esle */

switch(StmntType)
   { case SIR_STMNT_LABELED:
	{ IO->PutS(Label->LabelName);
	  IO->PutC(':');
	  if (Label->Color == SIR_RED)	/* is it a timing label? */
	     { IO->PutS(" " SIR_CXX_TIMESTAMP_PREFX);
	       IO->PutS(Label->LabelName);
	       IO->PutS(" = " SIR_SIM_CURR_TIME_NAME ";");
	      } /* fi */
	  IO->TabStepUp();
	  if ((SIR_Error = Statement1->WriteSC(IO, WriteNotes,
						CplusplusMode, FALSE)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->TabStepDown();
	  if (Label->Notes)
	     { if ((SIR_Error = Label->Notes->WriteSC(IO, WriteNotes,
							CplusplusMode)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  break;
	 }
     case SIR_STMNT_COMPOUND:
	{ assert(PutBraces);	// IO->PutC('{');
	  assert(Scope != NULL);	/* must have local scope */
	  if (  (PutArgNotes)
	      &&(Scope->Parent)	/* at highest level in function body? */
	      &&(Scope->Parent->ScopeInfo == SIR_SCOPE_PARAMETER))
	     { ParamSymbol = Scope->Parent->First();
	       while(ParamSymbol)
		  { if (ParamSymbol->Notes)	/* put notes at arguments */
		       { if ((SIR_Error = ParamSymbol->Notes->WriteSC(IO,
						WriteNotes, CplusplusMode)))
			    { return(SIR_Error);
			     } /* fi */
			} /* fi */
		    ParamSymbol = ParamSymbol->Succ();
		   } /* elihw */
	      } /* fi */
	  if (  (! Scope->Empty())
	      ||(! Scope->UserTypes->Empty()))
	     { if (CplusplusMode)
		  { if ((SIR_Error = Scope->WriteH(IO, WriteNotes)))
		       { return(SIR_Error);
			} /* fi */
		    if ((SIR_Error = Scope->WriteCC(IO, WriteNotes)))
		       { return(SIR_Error);
			} /* fi */
		   } /* fi */
	       else
		  { if ((SIR_Error = Scope->WriteSC(IO, WriteNotes)))
		       { return(SIR_Error);
			} /* fi */
		   } /* esle */
	       if (  (Statements)
		   &&(! Statements->Empty()))
		  { SIR_LineInfo::WriteVSPACE(IO);
		   } /* fi */
	      } /* fi */
	  if (Statements)
	     { Stmnt = Statements->First();
	       while(Stmnt)
		  { if ((SIR_Error = Stmnt->WriteSC(IO, WriteNotes,
							CplusplusMode, TRUE)))
		       { return(SIR_Error);
			} /* fi */
		    Stmnt = Stmnt->Succ();
		   } /* elihw */
	      } /* fi */
	  SIR_LineInfo::WriteNL(IO, false, true);	// IO->PutC('}');
	  break;
	 }
     case SIR_STMNT_EXPRESSION:
	{
#ifndef HAVE_ARYASGN
	  if (  (CplusplusMode)		/* array assignment handling */
	      &&(Expression1->ExprType == SIR_EXPR_ASSIGNMENT)
	      &&(Expression1->Arg1->Type->Type == SIR_TYPE_ARRAY)
	      &&(Expression1->Arg2->Type->Type == SIR_TYPE_ARRAY))
	     { sir_type		*ArrayType;
	       unsigned int	Depth,
				Index;
	       Depth = 0;
	       ArrayType = Expression1->Arg1->Type;
	       while(ArrayType->Type == SIR_TYPE_ARRAY)
		  { assert(ArrayType->Size > 0);	/* type checked */
		    IO->PrintF("{ unsigned int " SIR_CXX_INDEX_NAME "%u; "
					"for(" SIR_CXX_INDEX_NAME "%u=0;"
					SIR_CXX_INDEX_NAME "%u<%d;"
					SIR_CXX_INDEX_NAME "%u++) ",
				Depth, Depth, Depth, ArrayType->Size, Depth);
		    Depth++;
		    ArrayType = ArrayType->SubType;
		   } /* elihw */
	       if ((SIR_Error = Expression1->Arg1->WriteSC(IO, WriteNotes,
							CplusplusMode)))
		  { return(SIR_Error);
		   } /* fi */
	       for(Index=0; Index<Depth; Index++)
		  { IO->PrintF("[" SIR_CXX_INDEX_NAME "%u]", Index);
		   } /* rof */
	       IO->PutS(" = ");
	       if ((SIR_Error = Expression1->Arg2->WriteSC(IO, WriteNotes,
							CplusplusMode)))
		  { return(SIR_Error);
		   } /* fi */
	       for(Index=0; Index<Depth; Index++)
		  { IO->PrintF("[" SIR_CXX_INDEX_NAME "%u]", Index);
		   } /* rof */
	       IO->PutC(';');
	       for(Index=0; Index<Depth; Index++)
		  { IO->PutS(" }");
		   } /* rof */
	       break;
	      } /* fi */
#endif /* HAVE_ARYASGN */
	  if ((SIR_Error = Expression1->WriteSC(IO, WriteNotes,
						CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->PutC(';');
	  break;
	 }
     case SIR_STMNT_IF:
	{ IO->PutS("if (");
	  IO->TabStepUp();
	  if ((SIR_Error = Expression1->WriteSC(IO, WriteNotes,
						CplusplusMode, SIR_TYPE_BOOL)))
	     { return(SIR_Error);
	      } /* fi */
	//IO->TabStepDown();
	  IO->PutC(')');
	//IO->TabStepUp();
	  if ((SIR_Error = Statement1->WriteSC(IO, WriteNotes,
						CplusplusMode, TRUE)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->TabStepDown();
	  break;
	 }
     case SIR_STMNT_IF_ELSE:
	{ IO->PutS("if (");
	  IO->TabStepUp();
	  if ((SIR_Error = Expression1->WriteSC(IO, WriteNotes,
						CplusplusMode, SIR_TYPE_BOOL)))
	     { return(SIR_Error);
	      } /* fi */
	//IO->TabStepDown();
	  IO->PutC(')');
	//IO->TabStepUp();
	  if ((SIR_Error = Statement1->WriteSC(IO, WriteNotes,
						CplusplusMode, TRUE)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->TabStepDown();
	  SIR_LineInfo::WriteNL(IO);
	  IO->PutS("else ");
	  IO->TabStepUp();
	  if ((SIR_Error = Statement2->WriteSC(IO, WriteNotes,
						CplusplusMode, TRUE)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->TabStepDown();
	  break;
	 }
     case SIR_STMNT_SWITCH:
	{ IO->PutS("switch(");
	  IO->TabStepUp();
	  if ((SIR_Error = Expression1->WriteSC(IO, WriteNotes,
						CplusplusMode, SIR_TYPE_INT)))
	     { return(SIR_Error);
	      } /* fi */
	//IO->TabStepDown();
	  IO->PutC(')');
	//IO->TabStepUp();
	  if ((SIR_Error = Statement1->WriteSC(IO, WriteNotes,
						CplusplusMode, TRUE)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->TabStepDown();
	  break;
	 }
     case SIR_STMNT_CASE:
	{ IO->PutS("case ");
	  IO->PutS(Constant->Print());
	  IO->PutC(':');
	  IO->TabStepUp();
	  if ((SIR_Error = Statement1->WriteSC(IO, WriteNotes,
						CplusplusMode, FALSE)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->TabStepDown();
	  break;
	 }
     case SIR_STMNT_DEFAULT:
	{ IO->PutS("default:");
	  IO->TabStepUp();
	  if ((SIR_Error = Statement1->WriteSC(IO, WriteNotes,
						CplusplusMode, FALSE)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->TabStepDown();
	  break;
	 }
     case SIR_STMNT_WHILE:
	{ IO->PutS("while(");
	  IO->TabStepUp();
	  if ((SIR_Error = Expression1->WriteSC(IO, WriteNotes,
						CplusplusMode, SIR_TYPE_BOOL)))
	     { return(SIR_Error);
	      } /* fi */
	//IO->TabStepDown();
	  IO->PutC(')');
	//IO->TabStepUp();
	  if ((SIR_Error = Statement1->WriteSC(IO, WriteNotes,
						CplusplusMode, TRUE)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->TabStepDown();
	  break;
	 }
     case SIR_STMNT_DO_WHILE:
	{ IO->PutS("do ");
	  IO->TabStepUp();
	  if ((SIR_Error = Statement1->WriteSC(IO, WriteNotes,
						CplusplusMode, TRUE)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->TabStepDown();
	  SIR_LineInfo::WriteNL(IO);
	  IO->PutS("while(");
	  IO->TabStepUp();
	  if ((SIR_Error = Expression1->WriteSC(IO, WriteNotes,
						CplusplusMode, SIR_TYPE_BOOL)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->TabStepDown();
	  IO->PutS(");");
	  break;
	 }
     case SIR_STMNT_FOR:
	{ IO->PutS("for(");
	  IO->TabStepUp();
	  if ((SIR_Error = Expression1->WriteSC(IO, WriteNotes,
						CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->PutS("; ");
	  if ((SIR_Error = Expression2->WriteSC(IO, WriteNotes,
						CplusplusMode, SIR_TYPE_BOOL)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->PutS("; ");
	  if ((SIR_Error = Expression3->WriteSC(IO, WriteNotes,
						CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	//IO->TabStepDown();
	  IO->PutC(')');
	//IO->TabStepUp();
	  if ((SIR_Error = Statement1->WriteSC(IO, WriteNotes,
						CplusplusMode, TRUE)))
	     { return(SIR_Error);
	      } /* fi */
	  IO->TabStepDown();
	  break;
	 }
     case SIR_STMNT_GOTO:
	{ IO->PutS("goto ");
	  IO->PutS(Label->LabelName);
	  IO->PutC(';');
	  break;
	 }
     case SIR_STMNT_CONTINUE:
	{ IO->PutS("continue;");
	  break;
	 }
     case SIR_STMNT_BREAK:
	{ IO->PutS("break;");
	  break;
	 }
     case SIR_STMNT_RETURN:
	{ SIR_TYPETYPE	ExpectType;

	  IO->PutS("return ");
	  assert(Function->Type->Type == SIR_TYPE_FUNCTION);
	  ExpectType = Function->Type->SubType->Type;
	  if ((SIR_Error = Expression1->WriteSC(IO,
					WriteNotes, CplusplusMode,
					SIR_RELAXED(ExpectType,
						Expression1->Type))))
	     { return(SIR_Error);
	      } /* fi */
	  IO->PutC(';');
	  break;
	 }
     case SIR_STMNT_PAR:
	{ if (CplusplusMode)
	     { if ((SIR_Error = Statement1->CheckCompoundBehavior(
							FALSE, FALSE)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->TabStepUp();
	       IO->PutC('{');
	       SIR_LineInfo::WriteSPACE(IO);
	       if (! Statement1->Statements->Empty())
		  { IO->PutS(SIR_SIM_FORK_NAME " ");
		    Stmnt = Statement1->Statements->First();
		    while(Stmnt)
		       { assert(SIR_CALLED_BEHAVIOR(Stmnt) != NULL);
			 IO->PutS(SIR_CXX_FORK_VARIABLE);
			 IO->PutS(SIR_CALLED_BEHAVIOR(Stmnt)->Name);
			 IO->PutS("(&");
			 IO->PutS(SIR_CALLED_BEHAVIOR(Stmnt)->Name);
			 IO->PutC(')');
			 Stmnt = Stmnt->Succ();
			 if (Stmnt)
			    { IO->PutS(", ");
			     } /* fi */
			} /* elihw */
		    IO->PutC(';');
		    SIR_LineInfo::WriteSPACE(IO);
		   } /* fi */
	       IO->TabStepUp();
	       IO->PutS(SIR_SIM_PAR_NAME "(");
	       Stmnt = Statement1->Statements->First();
	       while(Stmnt)
		  { assert(SIR_CALLED_BEHAVIOR(Stmnt) != NULL);
		    if (Stmnt->LineInfo)
		       { if ((SIR_Error = Stmnt->LineInfo->WriteSC(IO)))
			    { return(SIR_Error);
			     } /* fi */
		        } /* fi */
		    else
		       { SIR_LineInfo::WriteNewSC(IO);
			} /* esle */
		    IO->PutS("&" SIR_CXX_FORK_VARIABLE);
		    IO->PutS(SIR_CALLED_BEHAVIOR(Stmnt)->Name);
		    IO->PutC(',');
		    SIR_LineInfo::WriteSPACE(IO);
		    Stmnt = Stmnt->Succ();
		   } /* elihw */
	       IO->PutS("0);");
	       IO->TabStepDown();
	       SIR_LineInfo::WriteNL(IO, false, true); // IO->PutC('}');
	       IO->TabStepDown();
	      } /* fi */
	  else
	     { IO->PutS("par");
	       IO->TabStepUp();
	       if ((SIR_Error = Statement1->WriteSC(IO, WriteNotes,
							CplusplusMode, FALSE)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->TabStepDown();
	      } /* esle */
	  break;
	 }
     case SIR_STMNT_PIPE:
	{ if (CplusplusMode)
	     { if ((SIR_Error = Statement1->CheckCompoundBehavior(
							FALSE, FALSE)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->TabStepUp();
	       IO->PutC('{');
	       SIR_LineInfo::WriteSPACE(IO);
	       if (! Statement1->Statements->Empty())
		  { IO->PutS(SIR_SIM_FORK_NAME " ");
		    Stmnt = Statement1->Statements->First();
		    while(Stmnt)
		       { assert(SIR_CALLED_BEHAVIOR(Stmnt) != NULL);
			 IO->PutS(SIR_CXX_FORK_VARIABLE);
			 IO->PutS(SIR_CALLED_BEHAVIOR(Stmnt)->Name);
			 IO->PutS("(&");
			 IO->PutS(SIR_CALLED_BEHAVIOR(Stmnt)->Name);
			 IO->PutC(')');
			 Stmnt = Stmnt->Succ();
			 if (Stmnt)
			    { IO->PutS(", ");
			     } /* fi */
			} /* elihw */
		    IO->PutC(';');
		    SIR_LineInfo::WriteSPACE(IO);
		   } /* fi */
	       if ((SIR_Error = WritePipeStmnt(IO, WriteNotes)))
		  { return(SIR_Error);
		   } /* fi */
	       SIR_LineInfo::WriteNL(IO, false, true); // IO->PutC('}');
	       IO->TabStepDown();
	      } /* fi */
	  else
	     { IO->PutS("pipe");
	       if (Expression1)
		  { assert(Expression2 != NULL);
		    assert(Expression3 != NULL);
		    IO->PutC('(');
		    if ((SIR_Error = Expression1->WriteSC(IO, WriteNotes,
							CplusplusMode)))
		       { return(SIR_Error);
			} /* fi */
		    IO->PutS("; ");
		    if ((SIR_Error = Expression2->WriteSC(IO, WriteNotes,
						CplusplusMode, SIR_TYPE_BOOL)))
		       { return(SIR_Error);
			} /* fi */
		    IO->PutS("; ");
		    if ((SIR_Error = Expression3->WriteSC(IO, WriteNotes,
							CplusplusMode)))
		       { return(SIR_Error);
			} /* fi */
		    IO->PutC(')');
		   } /* fi */
	       else
		  { assert(Expression2 == NULL);
		    assert(Expression3 == NULL);
		   } /* esle */
	       IO->TabStepUp();
	       if ((SIR_Error = Statement1->WriteSC(IO, WriteNotes,
							CplusplusMode, FALSE)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->TabStepDown();
	      } /* esle */
	  break;
	 }
     case SIR_STMNT_EXCEPTION:
	{ if (CplusplusMode)
	     { if ((SIR_Error = Statement1->CheckCompoundBehavior(TRUE, FALSE)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->TabStepUp();
	       IO->PutC('{');
	       SIR_LineInfo::WriteSPACE(IO);
	       IO->PutS(SIR_SIM_TRY_BLOCK " " SIR_CXX_TRY_VARIABLE);
	       if (Statement1->Statements->Empty())
		  { IO->PutS("0(0);");	/* empty try block */
		   } /* fi */
	       else
		  { IO->PutS(SIR_CALLED_BEHAVIOR(
				Statement1->Statements->First())->Name);
		    IO->PutS("(&");
		    IO->PutS(SIR_CALLED_BEHAVIOR(
				Statement1->Statements->First())->Name);
		    IO->PutS(");");
		   } /* esle */
	       Except = Exceptions->First();
	       ExceptNum = 0;
	       while(Except)
		  { if ((SIR_Error =
			Except->Handler->CheckCompoundBehavior(TRUE, FALSE)))
		       { return(SIR_Error);
			} /* fi */
		    if (Except->Pred())
		       { IO->PutS(", ");
			} /* fi */
		    else
		       { SIR_LineInfo::WriteSPACE(IO);
			 IO->PutS(SIR_SIM_EXCEPT_BLOCK " ");
			} /* esle */
		    IO->PutS(SIR_CXX_EXCEPT_VARIABLE);
		    if (Except->Handler->Statements->Empty())
		       { IO->PrintF("%d", ExceptNum);
			} /* fi */
		    else
		       { IO->PutS(SIR_CALLED_BEHAVIOR(
				Except->Handler->Statements->First())
						->Name);
			} /* esle */
		    if (Except->ExceptionType == SIR_EXCEPTION_TRAP)
		       { IO->PutS("(" SIR_SIM_TRAP_FLAG);
			} /* fi */
		    else /* SIR_EXCEPTION_INTERRUPT */
		       { IO->PutS("(" SIR_SIM_INTERRUPT_FLAG);
			} /* esle */
		    if (Except->Handler->Statements->Empty())
		       { IO->PutS(", 0");
			} /* fi */
		    else
		       { IO->PutS(", &");
			 IO->PutS(SIR_CALLED_BEHAVIOR(
				Except->Handler->Statements->First())
						->Name);
			} /* esle */
		    Event = Except->Events->First();
		    while(Event)
		       { IO->PutS(", &");
			 IO->PutS(Event->Symbol->Name);
			 Event = Event->Succ();
			} /* elihw */
		    IO->PutS(", 0)");
		    Except = Except->Succ();
		    ExceptNum++;
		   } /* elihw */
	       IO->PutC(';');
	       SIR_LineInfo::WriteSPACE(IO);
	       IO->PutS(SIR_SIM_EXCEPTION_NAME "(");
	       IO->TabStepUp();
	       IO->PutS("&" SIR_CXX_TRY_VARIABLE);
	       if (Statement1->Statements->Empty())
		  { IO->PutS("0");
		   } /* fi */
	       else
		  { IO->PutS(SIR_CALLED_BEHAVIOR(
				Statement1->Statements->First())->Name);
		   } /* esle */
	       Except = Exceptions->First();
	       ExceptNum = 0;
	       while(Except)
		  { IO->PutC(',');
		    if (Except->LineInfo)
		       { if ((SIR_Error =
				Except->LineInfo->WriteSC(IO, true)))
			    { return(SIR_Error);
			     } /* fi */
			} /* fi */
		    else
		       { SIR_LineInfo::WriteNewSC(IO, true);
			} /* esle */
		    IO->PutS("&" SIR_CXX_EXCEPT_VARIABLE);
		    if (Except->Handler->Statements->Empty())
		       { IO->PrintF("%d", ExceptNum);
			} /* fi */
		    else
		       { IO->PutS(SIR_CALLED_BEHAVIOR(
				Except->Handler->Statements->First())
						->Name);
			} /* esle */
		    Except = Except->Succ();
		    ExceptNum++;
		   } /* elihw */
	       IO->PutS(", 0);");
	       IO->TabStepDown();
	       SIR_LineInfo::WriteNL(IO, false, true); // IO->PutC('}');
	       IO->TabStepDown();
	      } /* fi */
	  else	/* not CplusplusMode */
	     { IO->PutS("try");
	       IO->TabStepUp();
	       if ((SIR_Error = Statement1->WriteSC(IO, WriteNotes,
							CplusplusMode, FALSE)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->TabStepDown();
	       Except = Exceptions->First();
	       while(Except)
		  { if (Except->LineInfo)
		       { if ((SIR_Error =
				Except->LineInfo->WriteSC(IO, true)))
			    { return(SIR_Error);
			     } /* fi */
			} /* fi */
		    else
		       { SIR_LineInfo::WriteNewSC(IO, true);
			} /* esle */
		    switch(Except->ExceptionType)
		       { case SIR_EXCEPTION_TRAP:
			    { IO->PutS("trap(");
			      break;
			     }
			 case SIR_EXCEPTION_INTERRUPT:
			    { IO->PutS("interrupt(");
			      break;
			     }
			 default:
			    { assert(FALSE);	/* bad exception type */
			     }
			} /* hctiws */
		    Event = Except->Events->First();
		    while(Event)
		       { IO->PutS(Event->Symbol->Name);
			 Event = Event->Succ();
			 if (Event)
			    { IO->PutS(", ");
			     } /* fi */
			} /* elihw */
		    IO->PutC(')');
		    IO->TabStepUp();
		    if ((SIR_Error = Except->Handler->WriteSC(IO, WriteNotes,
							CplusplusMode, FALSE)))
		       { return(SIR_Error);
			} /* fi */
		    IO->TabStepDown();
		    Except = Except->Succ();
		   } /* elihw */
	      } /* esle */
	  break;
	 }
     case SIR_STMNT_TIMING:
	{ if (CplusplusMode)
	     { IO->TabStepUp();
	       IO->PutC('{');
	       SIR_LineInfo::WriteSPACE(IO);
	       assert(Statement1->Statements != NULL);
	       Stmnt = Statement1->Statements->First();
	       LabelNum = 0;
	       while(Stmnt)
		  { if (Stmnt->StmntType == SIR_STMNT_LABELED)
		       { LabelNum++;
			 if (LabelNum == 1)
			    { IO->PutS(SIR_CXX_TIME_TYPE " ");
			     } /* fi */
			 else
			    { IO->PutS(", ");
			     } /* esle */
			 IO->PutS(SIR_CXX_TIMESTAMP_PREFX);
			 IO->PutS(Stmnt->Label->LabelName);
			 assert(Stmnt->Label->Color == SIR_WHITE);
			 Stmnt->Label->Color = SIR_RED;	/* mark timing labels */
			} /* fi */
		    Stmnt = Stmnt->Succ();
		   } /* elihw */
	       if (LabelNum > 0)
		  { IO->PutC(';');
		   } /* fi */
	       IO->TabStepUp();
	       if ((SIR_Error = Statement1->WriteSC(IO, WriteNotes,
						CplusplusMode, TRUE)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->TabStepDown();
	       Constr = Constraints->First();
	       while(Constr)
		  { if (Constr->LineInfo)
		       { if ((SIR_Error = Constr->LineInfo->WriteSC(
								IO, true)))
			    { return(SIR_Error);
			     } /* fi */
			} /* fi */
		    else
		       { SIR_LineInfo::WriteNewSC(IO, true);
			} /* esle */
		    IO->PutS(SIR_CXX_RANGE_NAME "(" SIR_CXX_TIMESTAMP_PREFX);
		    IO->PutS(Constr->Label1->LabelName);
		    IO->PutS(", " SIR_CXX_TIMESTAMP_PREFX);
		    IO->PutS(Constr->Label2->LabelName);
		    IO->PutS(", ");
		    if (Constr->MinTime)
		       { TmpConst = (new SIR_Constant(Constr->MinTime))
					->Converted(SIR_CONST_LONGLONG);
			 IO->PutS(TmpConst->Print(TRUE));
			 delete TmpConst;
			 IO->PutS(", true, ");
			} /* fi */
		    else
		       { IO->PutS("0, false, ");
			} /* esle */
		    if (Constr->MaxTime)
		       { TmpConst = (new SIR_Constant(Constr->MaxTime))
					->Converted(SIR_CONST_LONGLONG);
			 IO->PutS(TmpConst->Print(TRUE));
			 delete TmpConst;
			 IO->PutS(", true, ");
			} /* fi */
		    else
		       { IO->PutS("0, false, ");
			} /* esle */
		    IO->PutC('"');
		    IO->PutS(Constr->Label1->LabelName);
		    IO->PutS("\", \"");
		    IO->PutS(Constr->Label2->LabelName);
		    IO->PutS("\", ");
		    if (Constr->LineInfo)
		       { SIR_Constant::PrettyString(&TmpString,
				Constr->LineInfo->File->Filename.chars());
			 IO->PutS(TmpString);
			 IO->PrintF(", %d", Constr->LineInfo->Line);
			} /* fi */
		    else
		       { if (SIR_LineInfo::CurrentFile)
			    { SIR_Constant::PrettyString(&TmpString,
						SIR_LineInfo::CurrentFile);
			      IO->PutS(TmpString);
			      IO->PrintF(", %d", IO->GetLineNo());
			     } /* fi */
			 else
			    { IO->PutS("\"<unknown>\", 0");
			     } /* esle */
			} /* esle */
		    IO->PutS(");");
		    Constr = Constr->Succ();
		   } /* elihw */
	       Stmnt = Statement1->Statements->First();
	       while(Stmnt)
		  { if (Stmnt->StmntType == SIR_STMNT_LABELED)
		       { assert(Stmnt->Label->Color == SIR_RED);
			 Stmnt->Label->Color = SIR_WHITE;	/* unmark */
			} /* fi */
		    Stmnt = Stmnt->Succ();
		   } /* elihw */
	       SIR_LineInfo::WriteNL(IO, false, true); // IO->PutC('}');
	       IO->TabStepDown();
	      } /* fi */
	  else
	     { IO->PutS("do ");
	       IO->TabStepUp();
	       if ((SIR_Error = Statement1->WriteSC(IO, WriteNotes,
						CplusplusMode, TRUE)))
		  { return(SIR_Error);
		   } /* fi */
	       IO->TabStepDown();
	       SIR_LineInfo::WriteNL(IO);
	       IO->PutS("timing{");
	       IO->TabStepUp();
	       Constr = Constraints->First();
	       while(Constr)
		  { if (Constr->LineInfo)
		       { if ((SIR_Error = Constr->LineInfo->WriteSC(
								IO, true)))
			    { return(SIR_Error);
			     } /* fi */
			} /* fi */
		    else
		       { SIR_LineInfo::WriteNewSC(IO, true);
			} /* esle */
		    IO->PutS("range(");
		    IO->PutS(Constr->Label1->LabelName);
		    IO->PutS("; ");
		    IO->PutS(Constr->Label2->LabelName);
		    IO->PutS("; ");
		    if (Constr->MinTime)
		       { IO->PutS(Constr->MinTime->Print(CplusplusMode));
			} /* fi */
		    IO->PutS("; ");
		    if (Constr->MaxTime)
		       { IO->PutS(Constr->MaxTime->Print(CplusplusMode));
			} /* fi */
		    IO->PutS(");");
		    Constr = Constr->Succ();
		   } /* elihw */
	       SIR_LineInfo::WriteNL(IO, false, true);// IO->PutC('}');
	       IO->TabStepDown();
	      } /* esle */
	  break;
	 }
     case SIR_STMNT_FSM:
	{ if (CplusplusMode)
	     { IO->TabStepUp();
	       IO->PutC('{');
	       SIR_LineInfo::WriteSPACE(IO);
	       IO->PutS("enum { " SIR_CXX_STATE_PREFIX "0");
	       Trans = Transitions->First();
	       while(Trans)
		  { if (  (Trans->Pred() == NULL)
			||(Trans->Pred()->CurrState != Trans->CurrState))
		       { IO->PutS(", ");
			 IO->PutS(SIR_CXX_STATE_PREFIX);
			 IO->PutS(Trans->CurrState->Name);
			} /* fi */
		    Trans = Trans->Succ();
		   } /* elihw */
	       IO->PutS(" } " SIR_CXX_STATE_VARIABLE
				" = " SIR_CXX_STATE_PREFIX);
	       if (Transitions->First())
		  { IO->PutS(Transitions->First()->CurrState->Name);
		   } /* fi */
	       else
		  { IO->PutS("0");
		   } /* esle */
	       IO->PutC(';');
	       SIR_LineInfo::WriteSPACE(IO);
	       IO->PutS("do switch(" SIR_CXX_STATE_VARIABLE ")");
	       SIR_LineInfo::WriteSPACE(IO);
	       IO->TabStepUp();
	       IO->PutC('{');
	       Trans = Transitions->First();
	       while(Trans)
		  { if (Trans->LineInfo)
		       { if ((SIR_Error = Trans->LineInfo->WriteSC(IO, true)))
			    { return(SIR_Error);
			     } /* fi */
			} /* fi */
		    else
		       { SIR_LineInfo::WriteNewSC(IO, true);
			} /* esle */
		    IO->PutS("case " SIR_CXX_STATE_PREFIX);
		    IO->PutS(Trans->CurrState->Name);
		    IO->PutS(": { ");
		    IO->PutS(Trans->CurrState->Name);
		    IO->PutS("." GL_MAIN_METHOD_NAME "();");
		    IO->TabStepUp();
		    SIR_LineInfo::WriteSPACE(IO);
		    if (Trans->Condition)
		       { IO->PutS("if (");
			 IO->TabStepUp();
			 if ((SIR_Error = Trans->Condition->WriteSC(IO,
					WriteNotes, TRUE, SIR_TYPE_BOOL)))
			    { return(SIR_Error);
			     } /* fi */
			 IO->TabStepDown();
			 IO->PutS(") ");
			} /* fi */
		    IO->PutS("{ " SIR_CXX_STATE_VARIABLE " = "
					SIR_CXX_STATE_PREFIX);
		    if (Trans->NextState)
		       { IO->PutS(Trans->NextState->Name);
			} /* fi */
		    else
		       { IO->PutC('0');
			} /* else */
		    IO->PutS("; break; }");
		    while(  (Trans->Succ())
			  &&(Trans->Succ()->CurrState == Trans->CurrState))
		       { Trans = Trans->Succ();	/* combine same CurrStates */
			 if (Trans->LineInfo)
			    { if ((SIR_Error = Trans->LineInfo->WriteSC(IO,
									true)))
				 { return(SIR_Error);
				  } /* fi */
			     } /* fi */
			 else
			    { SIR_LineInfo::WriteNewSC(IO, true);
			     } /* esle */
			 if (Trans->Condition)
			    { IO->PutS("if (");
			      IO->TabStepUp();
			      if ((SIR_Error = Trans->Condition->WriteSC(IO,
					WriteNotes, TRUE, SIR_TYPE_BOOL)))
				 { return(SIR_Error);
				  } /* fi */
			      IO->TabStepDown();
			      IO->PutS(") ");
			     } /* fi */
			 IO->PutS("{ " SIR_CXX_STATE_VARIABLE " = "
					SIR_CXX_STATE_PREFIX);
			 if (Trans->NextState)
			    { IO->PutS(Trans->NextState->Name);
			     } /* fi */
			 else
			    { IO->PutC('0');
			     } /* esle */
			 IO->PutS("; break; }");
			} /* elihw */
		    SIR_LineInfo::WriteSPACE(IO);
		    IO->PutC('}');
		    IO->TabStepDown();
		    Trans = Trans->Succ();
		   } /* elihw */
	       SIR_LineInfo::WriteSPACE(IO);
	       IO->PutS("case " SIR_CXX_STATE_PREFIX "0: { "
			SIR_CXX_STATE_VARIABLE " = " SIR_CXX_STATE_PREFIX "0; "
			"break; }");
	       IO->TabStepDown();
	       SIR_LineInfo::WriteSPACE(IO);
	       IO->PutC('}');
	       SIR_LineInfo::WriteSPACE(IO);
	       IO->PutS("while(" SIR_CXX_STATE_VARIABLE " != "
					SIR_CXX_STATE_PREFIX "0);");
	       SIR_LineInfo::WriteNL(IO, false, true); // IO->PutC('}');
	       IO->TabStepDown();
	      } /* fi */
	  else
	     { IO->PutS("fsm{");
	       IO->TabStepUp();
	       Trans = Transitions->First();
	       while(Trans)
		  { if (Trans->LineInfo)
		       { if ((SIR_Error = Trans->LineInfo->WriteSC(IO, true)))
			    { return(SIR_Error);
			     } /* fi */
			} /* fi */
		    else
		       { SIR_LineInfo::WriteNewSC(IO, true);
			} /* esle */
		    IO->PutS(Trans->CurrState->Name);
		    IO->PutS(": ");
		    if (Trans->Condition)
		       { IO->PutS("if (");
			 IO->TabStepUp();
			 if ((SIR_Error = Trans->Condition->WriteSC(IO,
								WriteNotes)))
			    { return(SIR_Error);
			     } /* fi */
			 IO->TabStepDown();
			 IO->PutS(") ");
			} /* fi */
		    if (Trans->NextState)
		       { IO->PutS("goto ");
			 IO->PutS(Trans->NextState->Name);
			 IO->PutC(';');
			} /* fi */
		    else
		       { IO->PutS("break;");
			} /* else */
		    IO->TabStepUp();
		    while(  (Trans->Succ())
			  &&(Trans->Succ()->CurrState == Trans->CurrState))
		       { Trans = Trans->Succ();	/* combine same CurrStates */
			 if (Trans->LineInfo)
			    { if ((SIR_Error = Trans->LineInfo->WriteSC(IO,
									true)))
				 { return(SIR_Error);
				  } /* fi */
			     } /* fi */
			 else
			    { SIR_LineInfo::WriteNewSC(IO, true);
			     } /* esle */
			 if (Trans->Condition)
			    { IO->PutS("if (");
			      IO->TabStepUp();
			      if ((SIR_Error = Trans->Condition->WriteSC(IO,
								WriteNotes)))
				 { return(SIR_Error);
				  } /* fi */
			      IO->TabStepDown();
			      IO->PutS(") ");
			     } /* fi */
			 if (Trans->NextState)
			    { IO->PutS("goto ");
			      IO->PutS(Trans->NextState->Name);
			      IO->PutC(';');
			     } /* fi */
			 else
			    { IO->PutS("break;");
			     } /* esle */
			} /* elihw */
		    IO->TabStepDown();
		    Trans = Trans->Succ();
		   } /* elihw */
	       SIR_LineInfo::WriteNL(IO, false, true);// IO->PutC('}');
	       IO->TabStepDown();
	      } /* esle */
	  break;
	 }
     case SIR_STMNT_WAIT:
	{ if (CplusplusMode)
	     { IO->PutS(SIR_SIM_WAIT_NAME "(");
	       Event = Events->First();
	       while(Event)
		  { IO->PutC('&');
		    IO->PutS(Event->Symbol->Name);
		    Event = Event->Succ();
		    IO->PutS(", ");
		   } /* elihw */
	       IO->PutS("0);");
	      } /* fi */
	  else
	     { IO->PutS("wait(");
	       Event = Events->First();
	       while(Event)
		  { IO->PutS(Event->Symbol->Name);
		    Event = Event->Succ();
		    if (Event)
		       { IO->PutS(", ");
			} /* fi */
		   } /* elihw */
	       IO->PutS(");");
	      } /* esle */
	  break;
	 }
     case SIR_STMNT_WAITFOR:
	{ if (CplusplusMode)
	     { IO->PutS(SIR_SIM_WAITFOR_NAME "(");
	      } /* fi */
	  else
	     { IO->PutS("waitfor ");
	      } /* esle */
	  if ((SIR_Error = Expression1->WriteSC(IO, WriteNotes,
						CplusplusMode, SIR_TYPE_TIME)))
	     { return(SIR_Error);
	      } /* fi */
	  if (CplusplusMode)
	     { IO->PutC(')');
	      } /* fi */
	  IO->PutC(';');
	  break;
	 }
     case SIR_STMNT_NOTIFY:
	{ if (CplusplusMode)
	     { IO->PutS(SIR_SIM_NOTIFY_NAME "(");
	       Event = Events->First();
	       while(Event)
		  { IO->PutC('&');
		    IO->PutS(Event->Symbol->Name);
		    Event = Event->Succ();
		    IO->PutS(", ");
		   } /* elihw */
	       IO->PutS("0);");
	      } /* fi */
	  else
	     { IO->PutS("notify(");
	       Event = Events->First();
	       while(Event)
		  { IO->PutS(Event->Symbol->Name);
		    Event = Event->Succ();
		    if (Event)
		       { IO->PutS(", ");
			} /* fi */
		   } /* elihw */
	       IO->PutS(");");
	      } /* esle */
	  break;
	 }
     case SIR_STMNT_NOTIFYONE:
	{ if (CplusplusMode)
	     { IO->PutS(SIR_SIM_NOTIFYONE_NAME "(");
	       Event = Events->First();
	       while(Event)
		  { IO->PutC('&');
		    IO->PutS(Event->Symbol->Name);
		    Event = Event->Succ();
		    IO->PutS(", ");
		   } /* elihw */
	       IO->PutS("0);");
	      } /* fi */
	  else
	     { IO->PutS("notifyone(");
	       Event = Events->First();
	       while(Event)
		  { IO->PutS(Event->Symbol->Name);
		    Event = Event->Succ();
		    if (Event)
		       { IO->PutS(", ");
			} /* fi */
		   } /* elihw */
	       IO->PutS(");");
	      } /* esle */
	  break;
	 }
     default:
	{ assert(FALSE);	/* bad StmntType */
	 }
    } /* hctiws */

if ((SIR_Error = IO->Check()))
   { return(SIR_Error);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::WriteSC */


	/**********************/
	/*** SIR_Statements ***/
	/**********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Statements::SIR_Statements(		/* constructor #1 */
	sir_symbols	*CmpndScope /* = NULL */,
	sir_statement	*FirstEntry /* = NULL */) :
		SIR_List<SIR_Statement>(FirstEntry)
{

SIR_Statements::CmpndScope = CmpndScope;

} /* end of SIR_Statements::SIR_Statements #1 */


SIR_Statements::SIR_Statements(		/* constructor #3 (duplicator) */
	sir_statements	*Original)
{
sir_statement	*Curr;

Curr = Original->First();
while(Curr)
   { Append(new SIR_Statement(Curr));
     Curr = Curr->Succ();
    } /* elihw */

SIR_Statements::CmpndScope	= Original->CmpndScope;

} /* end of SIR_Statements::SIR_Statements #3 */


SIR_Statements::~SIR_Statements(void)	/* destructor */
{

/* nothing to do */

} /* end of SIR_Statements::~SIR_Statements */


ERROR SIR_Statements::DFS_ForAllNodes(	/* iterator over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{
sir_statement	*Stmnt,
		*Succ;

/* this is not a node, but there are nodes below */

Stmnt = First();
while(Stmnt)
   { Succ = Stmnt->Succ();
     if ((SIR_Error = Stmnt->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Stmnt = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statements::DFS_ForAllNodes */


ERROR SIR_Statements::DFS_ForAllSymbols( /* iterator over all symbols */
	sir_symbol_mptr	MemberFct,	/* (depth first) */
	sir_symbol_marg	MemberFctArg)
{
sir_statement	*Stmnt,
		*Succ;

/* this is not a symbol, but there are symbols below */

Stmnt = First();
while(Stmnt)
   { Succ = Stmnt->Succ();
     if ((SIR_Error = Stmnt->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Stmnt = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statements::DFS_ForAllSymbols */


ERROR SIR_Statements::DFS_ForAllUserTypes( /* iterator over all usertypes */
	sir_usertp_mptr	MemberFct,	/* (depth first) */
	sir_usertp_marg	MemberFctArg)
{
sir_statement	*Stmnt,
		*Succ;

/* this is not a usertype, but there are usertypes below */

Stmnt = First();
while(Stmnt)
   { Succ = Stmnt->Succ();
     if ((SIR_Error = Stmnt->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Stmnt = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statements::DFS_ForAllUserTypes */


ERROR SIR_Statements::DFS_ForAllNotes(	/* iterator over all notes */
	sir_note_mptr	MemberFct,	/* (depth first) */
	sir_note_marg	MemberFctArg)
{
sir_statement	*Stmnt,
		*Succ;

/* this is not a note, but there are notes below */

Stmnt = First();
while(Stmnt)
   { Succ = Stmnt->Succ();
     if ((SIR_Error = Stmnt->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Stmnt = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statements::DFS_ForAllNotes */


ERROR SIR_Statements::DFS_ForAllStatements(	/* iterator over all stmnts. */
	sir_stmnt_mptr	MemberFct,		/* (depth first) */
	sir_stmnt_marg	MemberFctArg)
{
sir_statement	*Stmnt,
		*Succ;

/* this is not a statement, but there are statements below */

Stmnt = First();
while(Stmnt)
   { Succ = Stmnt->Succ();
     if ((SIR_Error = Stmnt->DFS_ForAllStatements(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Stmnt = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statements::DFS_ForAllStatements */


ERROR SIR_Statements::DFS_ForAllExpressions( /* iterator over all expressions */
	sir_expr_mptr	MemberFct,		/* (depth first) */
	sir_expr_marg	MemberFctArg)
{
sir_statement	*Stmnt,
		*Succ;

/* this is not an expression, but there are expressions below */

Stmnt = First();
while(Stmnt)
   { Succ = Stmnt->Succ();
     if ((SIR_Error = Stmnt->DFS_ForAllExpressions(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Stmnt = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statements::DFS_ForAllExpressions */


BOOL SIR_Statements::DFS_FindDependant(	/* searches for dependants (DFS) */
	sir_symbol	*ThatSymbol,
	sir_statement	**DepStmnt,
	sir_expression	**DepExpr,
	SIR_DEPENDENCY	*Reason)
{
sir_statement	*Stmnt;

assert(ThatSymbol != NULL);

/* this is not a statement, but there are statements below */

Stmnt = First();
while(Stmnt)
   { if (Stmnt->DFS_FindDependant(ThatSymbol, DepStmnt, DepExpr, Reason))
	{ return(TRUE);
	 } /* fi */
     Stmnt = Stmnt->Succ();
    } /* elihw */

return(FALSE);

} /* end of SIR_Statements::DFS_FindDependant */


void SIR_Statements::SetAlias(	/* sets all type, usertype, symbol alias' */
	sir_statements	*Alias)	/* (iterates over symbols and usertypes) */
{
sir_statement	*Statement,
		*StatementAlias;

assert(Alias != NULL);

/* no need to process this node */

Statement = First();
StatementAlias = Alias->First();
while(Statement)
   { Statement->SetAlias(StatementAlias);
     Statement = Statement->Succ();
     StatementAlias = StatementAlias->Succ();
    } /* elihw */

} /* end of SIR_Statements::SetAlias */


sir_statement *SIR_Statements::FindLabeledStmnt(/* find a labeled statement */
	sir_label	*Label)		/* (returns NULL if not found) */
{
sir_statement	*Stmnt;

Stmnt = First();
while(Stmnt)
   { if (  (Stmnt->StmntType == SIR_STMNT_LABELED)
	 &&(Stmnt->Label == Label))
	{ break;
	 } /* fi */
     Stmnt = Stmnt->Succ();
    } /* elihw */

return(Stmnt);

} /* end of SIR_Statements::FindLabeledStmnt */


/*** exported functions *************************************************/


	/* none */


/* EOF Statement.cc */
